"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.down = exports.up = void 0;
const dal_db_armon_schema_1 = require("../../armon/dal.db.armon.schema");
const dal_utils_1 = require("../../../dal.utils");
const luxon_1 = require("luxon");
const dal_access_models_1 = require("../../../access/dal.access.models");
const app_enums_1 = require("../../../../app.enums");
const lodash_1 = __importDefault(require("lodash"));
const currentPartitionName = luxon_1.DateTime.now().toFormat("yLL");
const currentPartitionStart = luxon_1.DateTime.now().startOf("month").toFormat("y-LL") + "-01 00:00:00";
const currentPartitionEnd = luxon_1.DateTime.now().startOf("month").plus({ months: 1 }).toFormat("y-LL") + "-01 00:00:00";
const VisitorExcelImportResultNotification = 33;
const expectedDateField = {
    name: "expectedDate",
    extension: false,
    widthLevel: 2,
    required: false,
    reportOrder: 2,
    reportable: true,
    caption: "Ziyaret Tarihi",
    preRegistrationFieldSettings: { invisible: false, required: false },
    type: 10,
    invisible: false,
    order: -1,
    placeholder: "Tarih Seçiniz",
    options: {
        maxStartDynamicDate: 7,
        minStartDynamicDate: 3,
        startDynamicDate: 3,
        endDynamicDate: 5,
        showTimer: true,
        isStartRequired: false,
        isEndRequired: false,
        isEndVisible: true,
    },
    dependentField: null,
    hideOnReportFilter: true,
    listingInDetail: false,
    reportableInDetail: false,
    printHeading: 0,
    printOrder: 0,
    showInCaptionLines: false,
    triggers: [],
};
async function up(client, dbuser, dbsuperuser) {
    await csvSupportMigration.up(client);
    await weekDaysEnumMigration.up(client);
    await imageConvertSettingsMigration.up(client);
    await visitorForbiddanceMigration.up(client);
    await createVisitorScheduledJobTable.up(client);
    await updateScheduledJobs.up(client);
    await addExpectedDateFieldToAllOrganizationsAndDisableAccessControlPointsForPreregistration.up(client);
    await addVisitorImportNotificationTypesToVisitorModuleEnabledOrganizations.up(client);
}
exports.up = up;
async function down(client, dbuser, dbsuperuser) {
    await addVisitorImportNotificationTypesToVisitorModuleEnabledOrganizations.down(client);
    await addExpectedDateFieldToAllOrganizationsAndDisableAccessControlPointsForPreregistration.down(client);
    await updateScheduledJobs.down(client);
    await createVisitorScheduledJobTable.down(client);
    await visitorForbiddanceMigration.down(client);
    await imageConvertSettingsMigration.down(client);
    await weekDaysEnumMigration.down(client);
    await csvSupportMigration.down(client);
}
exports.down = down;
const csvSupportMigration = {
    up: async (client) => {
        const q = `

		ALTER TABLE "___ORGANIZATION_ID___"."preparedReports"
		DROP CONSTRAINT "preparedReports_format_check";
		
		ALTER TABLE "___ORGANIZATION_ID___"."preparedReports"
		ADD CONSTRAINT "preparedReports_format_check"
		CHECK (format = ANY ('{1,2,3}'::smallint[]));`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
    down: async (client) => {
        const q = `    
		ALTER TABLE "___ORGANIZATION_ID___"."preparedReports"
		DROP CONSTRAINT "preparedReports_format_check";
		
		ALTER TABLE "___ORGANIZATION_ID___"."preparedReports"
		ADD CONSTRAINT "preparedReports_format_check"
		CHECK (format = ANY ('{1,2}'::smallint[]));`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
    },
};
const weekDaysEnumMigration = {
    up: async (client) => {
        const query = `
		UPDATE "___ORGANIZATION_ID___"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessRules}"
		SET parameters = (parameters::jsonb || jsonb_build_object('days', REPLACE((parameters->'days')::text, '0', '7')::jsonb))::json
		WHERE parameters->'days' IS NOT NULL`;
        (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, query, "___ORGANIZATION_ID___");
    },
    down: async (client) => {
        const query = `
		UPDATE "___ORGANIZATION_ID___"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessRules}"
		SET parameters = (parameters::jsonb || jsonb_build_object('days', REPLACE((parameters->'days')::text, '7', '0')::jsonb))::json
		WHERE parameters->'days' IS NOT NULL`;
        (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, query, "___ORGANIZATION_ID___");
    },
};
const imageConvertSettingsMigration = {
    up: async (client) => {
        const sageOrgId = "1c5374ee-7141-44df-b5f0-0483c2ac499e";
        const hasSage = (await client.query(`SELECT COUNT(id) > 0 as "hasSage" FROM public."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationList}" WHERE id = $1`, [sageOrgId])).rows[0].hasSage;
        if (hasSage) {
            await client.query(`
            UPDATE "${sageOrgId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizations}"
            SET settings = settings || '{"snapshotSettings": {"quality": 100}}'::jsonb;
        `);
        }
        else {
            const query = `
        UPDATE "___ORGANIZATION_ID___"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizations}"
        SET settings = settings || '{"snapshotSettings": {"quality": 60, "width":180}}'::jsonb;`;
            await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, query, "___ORGANIZATION_ID___");
        }
    },
    down: async (client) => {
        const query = `
		UPDATE "___ORGANIZATION_ID___"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizations}"
		SET settings = settings - 'snapshotSettings'`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, query, "___ORGANIZATION_ID___");
    },
};
const visitorForbiddanceMigration = {
    up: async (client) => {
        const insertForcedRegistrationOfForbiddenVisitorsIntoSettings = `
			UPDATE "___ORGANIZATION_ID___"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorModuleSettings}"
			SET settings = jsonb_insert(settings::jsonb, '{enableToForceRegistrationOfForbiddenVisitors}', 'false');`;
        await (0, dal_utils_1.queryForOrganizationOnlyWithVisitorModule)(client, insertForcedRegistrationOfForbiddenVisitorsIntoSettings, "___ORGANIZATION_ID___");
        const removeForbiddenField = `
			DO
			$$
			DECLARE 
				field jsonb;
				"targetIndex" int;
				"visitorProfileFormFieldsCount" int;
			BEGIN 
				"targetIndex" = 0;
				
				SELECT jsonb_array_length((settings->'visitorProfileFormFields')::jsonb) INTO "visitorProfileFormFieldsCount" 
				FROM "___ORGANIZATION_ID___"."organizationVisitorModuleSettings";
				
				LOOP
					SELECT settings->'visitorProfileFormFields'->"targetIndex" INTO field FROM "___ORGANIZATION_ID___"."organizationVisitorModuleSettings";
					EXIT WHEN field->>'name' = 'forbidden';
					EXIT WHEN "visitorProfileFormFieldsCount" = "targetIndex";
					"targetIndex" = "targetIndex" + 1;
				END LOOP;
					
				UPDATE "___ORGANIZATION_ID___"."organizationVisitorModuleSettings"
				SET settings = jsonb_set(settings::jsonb, '{visitorProfileFormFields}', (settings->'visitorProfileFormFields')::jsonb #- ('{' || "targetIndex" || '}')::text[]);
			END
			$$;
		`;
        await (0, dal_utils_1.queryForOrganizationOnlyWithVisitorModule)(client, removeForbiddenField, "___ORGANIZATION_ID___");
        const forbiddanceEndUtcFix = `
			UPDATE "___ORGANIZATION_ID___"."organizationVisitorStates"
			SET "endUtc" = NULL
			WHERE state = 5  AND "endUtc" = '2099-02-01 00:00:00+03';
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, forbiddanceEndUtcFix, "___ORGANIZATION_ID___");
    },
    down: async (client) => {
        const removeForcedRegistrationOfForbiddenVisitorsIntoSettings = `
        UPDATE "___ORGANIZATION_ID___"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorModuleSettings}"
        SET settings = settings::jsonb - 'enableToForceRegistrationOfForbiddenVisitors';`;
        await (0, dal_utils_1.queryForOrganizationOnlyWithVisitorModule)(client, removeForcedRegistrationOfForbiddenVisitorsIntoSettings, "___ORGANIZATION_ID___");
    },
};
const createVisitorScheduledJobTable = {
    up: async (client) => {
        const query = `
            CREATE TABLE IF NOT EXISTS "___ORGANIZATION_ID___".visitor_scheduled_job
            (
                id uuid NOT NULL,
                "visitId" uuid NOT NULL,
                "scheduledJobId" uuid NOT NULL,
                CONSTRAINT "visitor_scheduled_job_pkey" PRIMARY KEY (id),
                CONSTRAINT "visitor_scheduled_job_scheduledJobId_visitId_uniqueness" UNIQUE ("visitId", "scheduledJobId"),
                CONSTRAINT "visitor_scheduled_job_visitId_fkey" FOREIGN KEY ("visitId")
                    REFERENCES "___ORGANIZATION_ID___"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationActiveVisits}" (id)
                    ON UPDATE NO ACTION
                    ON DELETE CASCADE,
                CONSTRAINT "visitor_scheduled_job_scheduledJobId_fkey" FOREIGN KEY ("scheduledJobId")
                    REFERENCES public."${dal_db_armon_schema_1.ArmonSchema.tableNames.scheduled_job}" (id)
                    ON UPDATE NO ACTION
                    ON DELETE CASCADE
            );
                
            --------------- HISTORY TABLE -----------------
            CREATE TABLE IF NOT EXISTS "___ORGANIZATION_ID___".zz_visitor_scheduled_job (
                id uuid NOT NULL,
                action smallint NOT NULL,
                "actionT" timestamp without time zone NOT NULL,
                "transactionId" bigint NOT NULL,
                "visitId" uuid NOT NULL,
                "scheduledJobId" uuid NOT NULL
            ) PARTITION BY RANGE ("actionT");
        
            CREATE TABLE IF NOT EXISTS "___ORGANIZATION_ID___".zz_visitor_scheduled_job_default PARTITION OF "___ORGANIZATION_ID___".zz_visitor_scheduled_job DEFAULT;	
            CREATE TABLE IF NOT EXISTS "___ORGANIZATION_ID___".zz_visitor_scheduled_job_p${currentPartitionName} PARTITION OF "___ORGANIZATION_ID___".zz_visitor_scheduled_job
                FOR VALUES FROM ('${currentPartitionStart}') TO ('${currentPartitionEnd}');
        
            ------------- HISTORY TRIGGER FUNC --------------
            CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".zz_visitor_scheduled_job()
                RETURNS trigger
                LANGUAGE 'plpgsql'
                COST 100
                VOLATILE NOT LEAKPROOF
            AS $BODY$
                DECLARE
                    _now timestamp without time zone;
                BEGIN   
                    _now := now();
                    IF (TG_OP = 'INSERT' AND NOT EXISTS (SELECT id FROM "___ORGANIZATION_ID___".visitor_scheduled_job WHERE id = NEW.id)) THEN
                        INSERT INTO "___ORGANIZATION_ID___".zz_visitor_scheduled_job (
                            id, action, "actionT", "transactionId", "visitId", "scheduledJobId")
                        VALUES(NEW.id, 1, _now, txid_current(), NEW."visitId", NEW."scheduledJobId");
                    ELSIF (TG_OP = 'UPDATE') THEN
                        INSERT INTO "___ORGANIZATION_ID___".zz_visitor_scheduled_job(
                            id,action,"actionT","transactionId", "visitId", "scheduledJobId")
                        VALUES( NEW.id,2,_now,txid_current(),NEW."visitId",NEW."scheduledJobId");
                    ELSIF (TG_OP = 'DELETE') THEN
                        INSERT INTO "___ORGANIZATION_ID___".zz_visitor_scheduled_job( 
                            id,action,"actionT","transactionId", "visitId", "scheduledJobId")
                        VALUES(OLD.id, 3, _now, txid_current(), OLD."visitId", OLD."scheduledJobId");
                    END IF;    
                    RETURN NULL;
                    END;
            $BODY$;
        
            -------- HISTORY TRIGGER FUNC TO TABLE ---------
        
            CREATE TRIGGER tr99_zz_visitor_scheduled_job
            AFTER INSERT OR DELETE OR UPDATE
            ON "___ORGANIZATION_ID___".visitor_scheduled_job
            FOR EACH ROW
            EXECUTE FUNCTION "___ORGANIZATION_ID___".zz_visitor_scheduled_job();
        `;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, query, "___ORGANIZATION_ID___");
    },
    down: async (client) => {
        const query = `
            DROP FUNCTION IF EXISTS "___ORGANIZATION_ID___".zz_visitor_scheduled_job CASCADE;
            DROP TRIGGER IF EXISTS tr99_zz_visitor_scheduled_job ON "___ORGANIZATION_ID___".visitor_scheduled_job;
            DROP TABLE IF EXISTS "___ORGANIZATION_ID___".zz_visitor_scheduled_job;
            DROP TABLE IF EXISTS "___ORGANIZATION_ID___".visitor_scheduled_job;
        `;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, query, "___ORGANIZATION_ID___");
    },
};
const updateScheduledJobs = {
    up: async (client) => {
        const query = `
            ALTER TABLE public."${dal_db_armon_schema_1.ArmonSchema.tableNames.scheduled_job}" DROP CONSTRAINT scheduled_job_interval_check;
            ALTER TABLE public."${dal_db_armon_schema_1.ArmonSchema.tableNames.scheduled_job}" ADD CONSTRAINT scheduled_job_interval_check 
                CHECK ("interval" = ANY ('{1,2,3,4,5,6,7,8,9,10}'::smallint[]));
                
            ALTER TABLE public.${dal_db_armon_schema_1.ArmonSchema.tableNames.scheduled_job}
            ADD COLUMN data jsonb;
        `;
        await client.query(query);
    },
    down: async (client) => {
        const removeOneTimeScheduledJobsQuery = `
            DELETE FROM public."${dal_db_armon_schema_1.ArmonSchema.tableNames.scheduled_job}"
            WHERE "interval" = 10;
        `;
        await client.query(removeOneTimeScheduledJobsQuery);
        const changeIntervalCheckQuery = `
            ALTER TABLE public."${dal_db_armon_schema_1.ArmonSchema.tableNames.scheduled_job}" DROP CONSTRAINT scheduled_job_interval_check;
            ALTER TABLE public."${dal_db_armon_schema_1.ArmonSchema.tableNames.scheduled_job}" ADD CONSTRAINT scheduled_job_interval_check 
                CHECK ("interval" = ANY ('{1,2,3,4,5,6,7,8,9}'::smallint[]));

            ALTER TABLE public.${dal_db_armon_schema_1.ArmonSchema.tableNames.scheduled_job}
            DROP COLUMN data;
        `;
        await client.query(changeIntervalCheckQuery);
    },
};
const addExpectedDateFieldToAllOrganizationsAndDisableAccessControlPointsForPreregistration = {
    up: async (client) => {
        const organizationIds = (await client.query(` SELECT id FROM public."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationList}";`)).rows.map((r) => r.id);
        const query = `
            UPDATE "___ORGANIZATION_ID___"."organizationVisitorModuleSettings"
            SET settings = $1;
        `;
        for (const organizationId of organizationIds) {
            const orgModules = await client.query(`SELECT * FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationModules}"`);
            const visitorModule = orgModules.rows.find((m) => m.module === app_enums_1.enums.ArmonApplicationModule.Visitor);
            if (!visitorModule) {
                continue;
            }
            const organizationVisitorModuleSettings = await (0, dal_utils_1.getOrganizationVisitorModuleSettings)(organizationId, client);
            if (!organizationVisitorModuleSettings.visitFormFields.find((field) => field.name == dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.expectedDate)) {
                expectedDateField.order = organizationVisitorModuleSettings.visitFormFields.length + 1;
                organizationVisitorModuleSettings.visitFormFields.push(expectedDateField);
            }
            const accessControlPointsField = organizationVisitorModuleSettings.visitFormFields.find((field) => field.name == dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.accessControlPoints);
            if (accessControlPointsField) {
                if (!accessControlPointsField.preRegistrationFieldSettings) {
                    accessControlPointsField.preRegistrationFieldSettings = {
                        invisible: true,
                        required: false,
                    };
                }
                else {
                    accessControlPointsField.preRegistrationFieldSettings.invisible = true;
                    accessControlPointsField.preRegistrationFieldSettings.required = false;
                }
            }
            await client.query(query.split("___ORGANIZATION_ID___").join(`${organizationId}`), [organizationVisitorModuleSettings]);
        }
    },
    down: async (client) => {
        const organizationIds = (await client.query(` SELECT id FROM public."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationList}";`)).rows.map((r) => r.id);
        for (const organizationId of organizationIds) {
            let organizationVisitorModuleSettings = await (0, dal_utils_1.getOrganizationVisitorModuleSettings)(organizationId, client);
            let expectedDate = organizationVisitorModuleSettings.visitFormFields.find((field) => field.name == dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.expectedDate);
            expectedDateField.order = organizationVisitorModuleSettings.visitFormFields.length;
            if (lodash_1.default.isEqual(expectedDateField, expectedDate)) {
                organizationVisitorModuleSettings.visitFormFields.splice(organizationVisitorModuleSettings.visitFormFields.indexOf(expectedDate), 1);
                const query = `
                    UPDATE "${organizationId}"."organizationVisitorModuleSettings"
                    SET settings = jsonb_set(to_jsonb(settings), '{visitFormFields}'::TEXT[], to_jsonb('${JSON.stringify(organizationVisitorModuleSettings.visitFormFields)}'::JSON), true);
                `;
                await client.query(query);
            }
        }
    },
};
const addVisitorImportNotificationTypesToVisitorModuleEnabledOrganizations = {
    up: async (client) => {
        const newNotificationQuery = `
            UPDATE "___ORGANIZATION_ID___".organizations
            SET settings = jsonb_insert(settings, '{notification, enabledTypes, -1}', '${VisitorExcelImportResultNotification}');

            INSERT INTO "___ORGANIZATION_ID___".notification
            (id, "createdT", "createdByUserId", type, "receiverFilterId", settings, email, sms, web, "pushNotification") 
            VALUES (uuid_generate_v4(), now(), NULL, ${VisitorExcelImportResultNotification}, NULL, NULL, TRUE, FALSE, TRUE, TRUE); 
        `;
        await (0, dal_utils_1.queryForOrganizationOnlyWithVisitorModule)(client, newNotificationQuery, "___ORGANIZATION_ID___");
        const notificationConstraint = `
			ALTER TABLE "___ORGANIZATION_ID___".notification_unsubscribe 
			DROP CONSTRAINT notification_unsubscribe_type_check;

			ALTER TABLE "___ORGANIZATION_ID___".notification_unsubscribe
			ADD CONSTRAINT notification_unsubscribe_type_check CHECK (type = ANY ('{1,2,3,4,6,8,9,10,11,15,18,19,24,25,27,28,31,32,33,34,35,36,37,38,39,40,41,42,43,44,45,46,47,999}'::smallint[]));
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, notificationConstraint, "___ORGANIZATION_ID___");
    },
    down: async (client) => {
        const newNotificationQuery = `
			UPDATE "___ORGANIZATION_ID___".organizations
			SET settings = jsonb_set(settings, '{notification, enabledTypes}',(
				SELECT
					array_to_json(array_agg("enabledTypes")::int[])::jsonb
				FROM 
				(
					SELECT
					jsonb_array_elements_text(settings -> 'notification' -> 'enabledTypes') as "enabledTypes" 
					FROM "___ORGANIZATION_ID___".organizations 
					ORDER BY jsonb_array_elements_text(settings -> 'notification' -> 'enabledTypes')::int
				) as T 
				WHERE ("enabledTypes" != '${VisitorExcelImportResultNotification}'))
			);

			DELETE FROM "___ORGANIZATION_ID___".notification
			WHERE type = ${VisitorExcelImportResultNotification};   
        `;
        await (0, dal_utils_1.queryForOrganizationOnlyWithVisitorModule)(client, newNotificationQuery, "___ORGANIZATION_ID___");
        const notificationConstraint = `
			ALTER TABLE "___ORGANIZATION_ID___".notification_unsubscribe 
			DROP CONSTRAINT notification_unsubscribe_type_check;

			ALTER TABLE "___ORGANIZATION_ID___".notification_unsubscribe
			ADD CONSTRAINT notification_unsubscribe_type_check CHECK (type = ANY ('{1,2,3,4,6,8,9,10,11,15,18,19,24,25,27,28,31,32,34,35,36,37,38,39,40,41,42,43,44,45,46,47,999}'::smallint[]));
		`;
        await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, notificationConstraint, "___ORGANIZATION_ID___");
    },
};
