"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.down = exports.up = void 0;
const dal_utils_1 = require("../../../dal.utils");
async function up(client, dbuser, dbsuperuser) {
    const q = `

    ------------------- cameras table fill empty "accessControlPoints" -----------------------------
    UPDATE "___ORGANIZATION_ID___".cameras
    SET "accessControlPointId" = (SELECT id FROM "___ORGANIZATION_ID___"."accessControlPoints"
        WHERE "deviceId" IN (SELECT id FROM "___ORGANIZATION_ID___"."devices"
            WHERE "model" IN ('V10', 'M3', 'AH40', 'V6', 'M5', 'DS-K1T606MF', 'DS-K1T804MF') AND "deletedAt" IS NULL)
        AND "deletedAt" IS NULL limit 1)
    WHERE "accessControlPointId" IS NULL;

    -------------------------- cameras table structural changes ------------------------------------
    ALTER TABLE "___ORGANIZATION_ID___".cameras
    ADD COLUMN "proxyTerminalId" uuid,
    ADD CONSTRAINT "cameras_proxyTerminalId_device_id_fk" 
        FOREIGN KEY ("proxyTerminalId") REFERENCES "___ORGANIZATION_ID___"."devices"(id) ON DELETE CASCADE;
    
    UPDATE "___ORGANIZATION_ID___".cameras c
    SET "proxyTerminalId" = (
        SELECT "deviceId" FROM "___ORGANIZATION_ID___"."accessControlPoints"
        WHERE id = c."accessControlPointId"
    );
        
    ALTER TABLE "___ORGANIZATION_ID___".cameras
        DROP COLUMN IF EXISTS "proxyUrl",
        DROP COLUMN IF EXISTS "streamUrl",
        ALTER COLUMN "proxyTerminalId" SET NOT NULL;


	------------------------------------------- CAMERAS_TRIGGER ----------------------------------------------

	CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".before_up_in_del_camera()
		RETURNS trigger
		LANGUAGE 'plpgsql'
		COST 100
		VOLATILE NOT LEAKPROOF
	AS $BODY$
	DECLARE
		transaction_id	uuid;
		camera_id 		uuid;
		proxy_terminal_id 		uuid;
		BEGIN
			transaction_id := uuid_generate_v4();
			IF (TG_OP = 'DELETE') THEN
				camera_id:= OLD."id";
				proxy_terminal_id:= OLD."proxyTerminalId";
			ELSE
				camera_id:= NEW."id";
				proxy_terminal_id:= NEW."proxyTerminalId";
			END IF;
			INSERT INTO "___ORGANIZATION_ID___"."terminalChanges"
			("id", "transactionId", "deviceId", "actionDateISO", "type", "data")
			SELECT uuid_generate_v4(), transaction_id, acp."deviceId", now(), 3, '{}'::json
			FROM "___ORGANIZATION_ID___"."cameras" AS c
			INNER JOIN "___ORGANIZATION_ID___"."cameraActions" AS ca ON c."id" = ca."cameraId" AND c."deletedAt" IS NULL AND c."id" = camera_id
			INNER JOIN "___ORGANIZATION_ID___"."accessControlPoints" AS acp ON ca."accessControlPointId" = acp."id" AND acp."deletedAt" IS NULL;

			INSERT INTO "___ORGANIZATION_ID___"."terminalChanges"
			("id", "transactionId", "deviceId", "actionDateISO", "type", "data")
			SELECT uuid_generate_v4(), transaction_id, proxy_terminal_id, now(), 3, '{}'::json;
			IF (TG_OP = 'DELETE') THEN
				RETURN OLD;
			ELSE
				RETURN NEW;
			END IF;

		END;
	$BODY$;

	ALTER FUNCTION "___ORGANIZATION_ID___".before_up_in_del_camera()
		OWNER TO ${dbsuperuser};

	GRANT EXECUTE ON FUNCTION "___ORGANIZATION_ID___".before_up_in_del_camera() TO ${dbsuperuser};

	GRANT EXECUTE ON FUNCTION "___ORGANIZATION_ID___".before_up_in_del_camera() TO PUBLIC;

	GRANT EXECUTE ON FUNCTION "___ORGANIZATION_ID___".before_up_in_del_camera() TO ${dbuser};

    ------------------------------------------- DEVICE_TRIGGER ------------------------------------------------

CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".after_up_del_device_state_and_notification()
    RETURNS trigger
    LANGUAGE 'plpgsql'
    COST 100
    VOLATILE NOT LEAKPROOF
    AS $BODY$
    DECLARE
        organization_id 	text;
        data json;
		updateDeviceStateCache boolean;
		updateDeviceNotificationInfoCache boolean;
    BEGIN
        IF (TG_OP = 'UPDATE') THEN
		   IF(OLD."deletedAt" IS NULL AND NEW."deletedAt" IS NOT NULL) THEN
				updateDeviceNotificationInfoCache := true;
				updateDeviceStateCache := true;
			END IF;
            IF ((NEW.health->>'isConnected')::boolean != (OLD.health->>'isConnected')::boolean OR
			(NEW.health->>'warnings')::text != (OLD.health->>'warnings')::text 
		   ) THEN
		   		updateDeviceStateCache := true;
			END IF;
			IF(NEW.brand != OLD.brand OR 
                NEW.model NOT LIKE OLD.model OR
                NEW.name != OLD.name OR
                NEW.location != OLD.location) THEN
				updateDeviceNotificationInfoCache := true;
			END IF;
		ELSIF (TG_OP = 'DELETE') THEN
				updateDeviceNotificationInfoCache := true;
				updateDeviceStateCache := true;
        END IF;
		IF(updateDeviceNotificationInfoCache = true) THEN
			PERFORM PG_NOTIFY('update_device_notification_info_cache', JSON_BUILD_OBJECT('id', NEW.id, 'organizationId', TG_TABLE_SCHEMA)::text);								  
		END IF;
		IF(updateDeviceStateCache = true) THEN
			PERFORM PG_NOTIFY('update_device_state_cache', JSON_BUILD_OBJECT('id', NEW.id, 'organizationId', TG_TABLE_SCHEMA)::text);								  
		END IF;
    RETURN NEW;
    END;
    $BODY$;

	ALTER FUNCTION "___ORGANIZATION_ID___".after_up_del_device_state_and_notification()
	OWNER TO ${dbsuperuser};

	GRANT EXECUTE ON FUNCTION "___ORGANIZATION_ID___".after_up_del_device_state_and_notification() TO ${dbsuperuser};

	GRANT EXECUTE ON FUNCTION "___ORGANIZATION_ID___".after_up_del_device_state_and_notification() TO PUBLIC;

	GRANT EXECUTE ON FUNCTION "___ORGANIZATION_ID___".after_up_del_device_state_and_notification() TO ${dbuser};

    CREATE TRIGGER trg_after_up_del_device_state_and_notification
        AFTER UPDATE OR DELETE
        ON "___ORGANIZATION_ID___".devices
        FOR EACH ROW
    	EXECUTE FUNCTION "___ORGANIZATION_ID___".after_up_del_device_state_and_notification();

    ------------------------------------------------- AccessPoint redis trigger --------------------------------------------- 

    CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".after_up_del_access_point()
    RETURNS trigger
    LANGUAGE 'plpgsql'
    COST 100
    VOLATILE NOT LEAKPROOF
    AS $BODY$
    BEGIN
        IF (TG_OP = 'UPDATE' AND (NEW.name != OLD.name OR NEW."deletedAt" != OLD."deletedAt")) THEN
             PERFORM PG_NOTIFY('update_access_point_info_cache', JSON_BUILD_OBJECT('id', NEW.id, 'organizationId', TG_TABLE_SCHEMA)::text);								  		
		ELSIF (TG_OP = 'DELETE') THEN
             PERFORM PG_NOTIFY('update_access_point_info_cache', JSON_BUILD_OBJECT('id', OLD.id, 'organizationId', TG_TABLE_SCHEMA)::text);								  		
        END IF;
    RETURN NEW;
    END;
    $BODY$;

	ALTER FUNCTION "___ORGANIZATION_ID___".after_up_del_access_point()
	OWNER TO ${dbsuperuser};

	GRANT EXECUTE ON FUNCTION "___ORGANIZATION_ID___".after_up_del_access_point() TO ${dbsuperuser};

	GRANT EXECUTE ON FUNCTION "___ORGANIZATION_ID___".after_up_del_access_point() TO PUBLIC;

	GRANT EXECUTE ON FUNCTION "___ORGANIZATION_ID___".after_up_del_access_point() TO ${dbuser};
        
    CREATE TRIGGER trg_after_up_del_access_point
        AFTER UPDATE OR DELETE
        ON "___ORGANIZATION_ID___"."accessControlPoints"
        FOR EACH ROW
        EXECUTE FUNCTION "___ORGANIZATION_ID___".after_up_del_access_point();

    ------------------------------------------------- Region redis trigger --------------------------------------------- 

    CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".after_up_del_region()
    RETURNS trigger
    LANGUAGE 'plpgsql'
    COST 100
    VOLATILE NOT LEAKPROOF
    AS $BODY$
    BEGIN
        IF (TG_OP = 'UPDATE' AND (
                NEW.name != OLD.name OR 
                NEW."visitorAccess" != OLD."visitorAccess" OR 
                NEW."terminateVisitOnExit" != OLD."terminateVisitOnExit")) THEN
            PERFORM PG_NOTIFY('update_region_info_cache', JSON_BUILD_OBJECT('id', NEW.id, 'organizationId', TG_TABLE_SCHEMA)::text);								  		
        ELSIF (TG_OP = 'DELETE') THEN
            PERFORM PG_NOTIFY('update_region_info_cache', JSON_BUILD_OBJECT('id', OLD.id, 'organizationId', TG_TABLE_SCHEMA)::text);								  		
        END IF;
    RETURN NEW;
    END;
    $BODY$;

	ALTER FUNCTION "___ORGANIZATION_ID___".after_up_del_region()
	OWNER TO ${dbsuperuser};

	GRANT EXECUTE ON FUNCTION "___ORGANIZATION_ID___".after_up_del_region() TO ${dbsuperuser};

	GRANT EXECUTE ON FUNCTION "___ORGANIZATION_ID___".after_up_del_region() TO PUBLIC;

	GRANT EXECUTE ON FUNCTION "___ORGANIZATION_ID___".after_up_del_region() TO ${dbuser};
        
    CREATE TRIGGER trg_after_up_del_region
        AFTER UPDATE OR DELETE
        ON "___ORGANIZATION_ID___"."regions"
        FOR EACH ROW
        EXECUTE FUNCTION "___ORGANIZATION_ID___".after_up_del_region();

    ------------------------------------------------- User redis trigger (users) --------------------------------------------- 

    CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".after_up_del_user()
    RETURNS trigger
    LANGUAGE 'plpgsql'
    COST 100
    VOLATILE NOT LEAKPROOF
    AS $BODY$
    BEGIN
        IF (TG_OP = 'UPDATE' AND (                    
                NEW."deletedAt" != OLD."deletedAt" OR 
                NEW."notificationToken" != OLD."notificationToken" OR 
                NEW."notificationTokenType" != OLD."notificationTokenType" OR 
                NEW.settings->>'locale' != OLD.settings->>'locale'
            )) THEN
            PERFORM PG_NOTIFY('update_user_info_cache', JSON_BUILD_OBJECT('id', NEW.id, 'organizationId', TG_TABLE_SCHEMA)::text);								  		
        ELSIF (TG_OP = 'DELETE') THEN
            PERFORM PG_NOTIFY('update_user_info_cache', JSON_BUILD_OBJECT('id', OLD.id, 'organizationId', TG_TABLE_SCHEMA)::text);								  		
        END IF;
    RETURN NEW;
    END;
    $BODY$;

	ALTER FUNCTION "___ORGANIZATION_ID___".after_up_del_user()
	OWNER TO ${dbsuperuser};

	GRANT EXECUTE ON FUNCTION "___ORGANIZATION_ID___".after_up_del_user() TO ${dbsuperuser};

	GRANT EXECUTE ON FUNCTION "___ORGANIZATION_ID___".after_up_del_user() TO PUBLIC;

	GRANT EXECUTE ON FUNCTION "___ORGANIZATION_ID___".after_up_del_user() TO ${dbuser};
        
    CREATE TRIGGER trg_after_up_del_user
        AFTER UPDATE OR DELETE
        ON "___ORGANIZATION_ID___"."users"
        FOR EACH ROW
        EXECUTE FUNCTION "___ORGANIZATION_ID___".after_up_del_user();

    ------------------------------------------------- User redis trigger (userOrganizations) --------------------------------------------- 
    CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".after_up_del_user_organization()
    RETURNS trigger
    LANGUAGE 'plpgsql'
    COST 100
    VOLATILE NOT LEAKPROOF
    AS $BODY$
    BEGIN
        IF (TG_OP = 'UPDATE' AND (                    
                NEW."deletedAt" != OLD."deletedAt" OR 
                NEW."roleId" != OLD."roleId" OR 
                NEW."isDisabled" != OLD."isDisabled" OR 
                NEW.settings->'settings' != OLD.settings->'settings'
                )) THEN
            PERFORM PG_NOTIFY('update_user_info_cache', JSON_BUILD_OBJECT('id', NEW."userId", 'organizationId', TG_TABLE_SCHEMA)::text);								  		
        ELSIF (TG_OP = 'DELETE') THEN
            PERFORM PG_NOTIFY('update_user_info_cache', JSON_BUILD_OBJECT('id', OLD."userId", 'organizationId', TG_TABLE_SCHEMA)::text);								  		
        END IF;
    RETURN NEW;
    END;
    $BODY$;

	ALTER FUNCTION "___ORGANIZATION_ID___".after_up_del_user_organization()
	OWNER TO ${dbsuperuser};

	GRANT EXECUTE ON FUNCTION "___ORGANIZATION_ID___".after_up_del_user_organization() TO ${dbsuperuser};

	GRANT EXECUTE ON FUNCTION "___ORGANIZATION_ID___".after_up_del_user_organization() TO PUBLIC;

	GRANT EXECUTE ON FUNCTION "___ORGANIZATION_ID___".after_up_del_user_organization() TO ${dbuser};
        
    CREATE TRIGGER trg_after_up_del_user_organization
        AFTER UPDATE OR DELETE
        ON "___ORGANIZATION_ID___"."userOrganizations"
        FOR EACH ROW
        EXECUTE FUNCTION "___ORGANIZATION_ID___".after_up_del_user_organization();

    --------------------------- User redis trigger (userProfiles) --------------------------------------------- 
    CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".after_up_del_user_profile()
        RETURNS trigger
    LANGUAGE 'plpgsql'
        COST 100
        VOLATILE NOT LEAKPROOF
        AS $BODY$
        BEGIN
            IF (TG_OP = 'UPDATE' AND (                    
                    NEW."deletedAt" != OLD."deletedAt" OR 
                    NEW."name" != OLD."name" OR 
                    NEW."surname" != OLD."surname" OR 
                    NEW."email" != OLD."email" OR
                    NEW."phoneNumber" != OLD."phoneNumber"
                    )) THEN
                PERFORM PG_NOTIFY('update_user_info_cache', JSON_BUILD_OBJECT('id', NEW."userId", 'organizationId', TG_TABLE_SCHEMA)::text);								  		
            ELSIF (TG_OP = 'DELETE') THEN
                PERFORM PG_NOTIFY('update_user_info_cache', JSON_BUILD_OBJECT('id', OLD."userId", 'organizationId', TG_TABLE_SCHEMA)::text);								  		
            END IF;
        RETURN NEW;
        END;
        $BODY$;
		
	ALTER FUNCTION "___ORGANIZATION_ID___".after_up_del_user_profile()
	OWNER TO ${dbsuperuser};

	GRANT EXECUTE ON FUNCTION "___ORGANIZATION_ID___".after_up_del_user_profile() TO ${dbsuperuser};

	GRANT EXECUTE ON FUNCTION "___ORGANIZATION_ID___".after_up_del_user_profile() TO PUBLIC;

	GRANT EXECUTE ON FUNCTION "___ORGANIZATION_ID___".after_up_del_user_profile() TO ${dbuser};
            
        CREATE TRIGGER trg_after_up_del_user_profile
            AFTER UPDATE OR DELETE
            ON "___ORGANIZATION_ID___"."userProfiles"
            FOR EACH ROW
            EXECUTE FUNCTION "___ORGANIZATION_ID___".after_up_del_user_profile();

    --------------------------- User redis trigger (userOrganizationProfiles) --------------------------------------------- 

    CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".after_up_del_user_organization_profile()
        RETURNS trigger
    LANGUAGE 'plpgsql'
        COST 100
        VOLATILE NOT LEAKPROOF
        AS $BODY$
        BEGIN
            IF (TG_OP = 'UPDATE' AND (                    
                    NEW."deletedAt" != OLD."deletedAt" OR 
                    NEW."name" != OLD."name" OR 
                    NEW."surname" != OLD."surname" OR 
                    NEW."email" != OLD."email" OR
                    NEW."phoneNumber" != OLD."phoneNumber" OR
                    NEW."uniqueId" != OLD."uniqueId"
                    )) THEN
                PERFORM PG_NOTIFY('update_user_info_cache', JSON_BUILD_OBJECT('id', NEW."userId", 'organizationId', TG_TABLE_SCHEMA)::text);								  		
            ELSIF (TG_OP = 'DELETE') THEN
                PERFORM PG_NOTIFY('update_user_info_cache', JSON_BUILD_OBJECT('id', OLD."userId", 'organizationId', TG_TABLE_SCHEMA)::text);								  		
            END IF;
        RETURN NEW;
        END;
        $BODY$;

				
		ALTER FUNCTION "___ORGANIZATION_ID___".after_up_del_user_organization_profile()
		OWNER TO ${dbsuperuser};

		GRANT EXECUTE ON FUNCTION "___ORGANIZATION_ID___".after_up_del_user_organization_profile() TO ${dbsuperuser};

		GRANT EXECUTE ON FUNCTION "___ORGANIZATION_ID___".after_up_del_user_organization_profile() TO PUBLIC;

		GRANT EXECUTE ON FUNCTION "___ORGANIZATION_ID___".after_up_del_user_organization_profile() TO ${dbuser};
            
        CREATE TRIGGER trg_after_up_del_user_organization_profile
            AFTER UPDATE OR DELETE
            ON "___ORGANIZATION_ID___"."userOrganizationProfiles"
            FOR EACH ROW
            EXECUTE FUNCTION "___ORGANIZATION_ID___".after_up_del_user_organization_profile();

    --------------------------- organization redis trigger (organization) --------------------------------------------- 

    CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".after_up_del_organization()
        RETURNS trigger
    LANGUAGE 'plpgsql'
        COST 100
        VOLATILE NOT LEAKPROOF
        AS $BODY$
        DECLARE
            transaction_id	uuid;
        BEGIN
            transaction_id := uuid_generate_v4();
            IF (TG_OP = 'UPDATE') THEN
				IF (NEW.settings->'webRtc' != OLD.settings->'webRtc') THEN
                    INSERT INTO "___ORGANIZATION_ID___"."terminalChanges"
                    ("id", "transactionId", "deviceId", "actionDateISO", "type", "data")
                    SELECT uuid_generate_v4(), transaction_id, T."proxyTerminalId", now(), 3, '{}'::json FROM
                    (SELECT DISTINCT "proxyTerminalId" FROM "___ORGANIZATION_ID___"."cameras") as T;
                	PERFORM PG_NOTIFY('webrtc_ice_servers', JSON_BUILD_OBJECT('id', NEW.id)::text);								  		
				END IF;	
			END IF;	
			PERFORM PG_NOTIFY('organization_settings_changed', TG_TABLE_SCHEMA);
        RETURN NEW;
        END;
        $BODY$;

		ALTER FUNCTION "___ORGANIZATION_ID___".after_up_del_organization()
		OWNER TO ${dbsuperuser};

		GRANT EXECUTE ON FUNCTION "___ORGANIZATION_ID___".after_up_del_organization() TO ${dbsuperuser};

		GRANT EXECUTE ON FUNCTION "___ORGANIZATION_ID___".after_up_del_organization() TO PUBLIC;

		GRANT EXECUTE ON FUNCTION "___ORGANIZATION_ID___".after_up_del_organization() TO ${dbuser};
            
        CREATE TRIGGER trg_after_up_del_organization
            AFTER UPDATE OR DELETE
            ON "___ORGANIZATION_ID___"."organizations"
            FOR EACH ROW
            EXECUTE FUNCTION "___ORGANIZATION_ID___".after_up_del_organization();

        DROP FUNCTION IF EXISTS after_update_organization_settings CASCADE;
        DROP TRIGGER IF EXISTS after_up_organization_settings ON "___ORGANIZATION_ID___".organizations;
            `;
    await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
}
exports.up = up;
async function down(client, dbuser, dbsuperuser) {
    const q = `
        
    -------------------------- cameras table structural changes ------------------------------------
    ALTER TABLE "___ORGANIZATION_ID___".cameras
        ADD COLUMN IF NOT EXISTS "proxyUrl" character varying(255),
        ADD COLUMN IF NOT EXISTS "streamUrl" character varying(255),
        DROP COLUMN IF EXISTS "proxyTerminalId",
        DROP CONSTRAINT IF EXISTS "cameras_proxyTerminalId_device_id_fk";
            
    ------------------------------------------- DEVICE_TRIGGER ------------------------------------------------

    DROP FUNCTION IF EXISTS "___ORGANIZATION_ID___".after_up_del_device_state_and_notification() CASCADE;
    DROP TRIGGER IF EXISTS trg_after_up_del_device_state_and_notification ON "___ORGANIZATION_ID___".devices;
    

    ------------------------------------------------- AccessPoint redis trigger --------------------------------------------- 

    DROP FUNCTION IF EXISTS "___ORGANIZATION_ID___".after_up_del_access_point() CASCADE;
    DROP TRIGGER IF EXISTS trg_after_up_del_access_point ON "___ORGANIZATION_ID___"."accessControlPoints";

    ------------------------------------------------- Region redis trigger --------------------------------------------- 

    DROP FUNCTION IF EXISTS "___ORGANIZATION_ID___".after_up_del_region() CASCADE;
    DROP TRIGGER IF EXISTS trg_after_up_del_region ON "___ORGANIZATION_ID___"."regions";

    ------------------------------------------------- User redis trigger (users) --------------------------------------------- 
    
    DROP FUNCTION IF EXISTS "___ORGANIZATION_ID___".after_up_del_user() CASCADE;
    DROP TRIGGER IF EXISTS trg_after_up_del_user ON "___ORGANIZATION_ID___"."users";

    ------------------------------------------------- User redis trigger (userOrganizations) --------------------------------------------- 

    DROP FUNCTION IF EXISTS "___ORGANIZATION_ID___".after_up_del_user_organization() CASCADE;
    DROP TRIGGER IF EXISTS trg_after_up_del_user_organization ON "___ORGANIZATION_ID___"."userOrganizations";

    --------------------------- User redis trigger (userProfiles) --------------------------------------------- 

    DROP FUNCTION IF EXISTS "___ORGANIZATION_ID___".after_up_del_user_profile() CASCADE;
    DROP TRIGGER IF EXISTS trg_after_up_del_user_profile ON "___ORGANIZATION_ID___"."userProfiles";

    --------------------------- User redis trigger (userOrganizationProfiles) --------------------------------------------- 

    DROP FUNCTION IF EXISTS "___ORGANIZATION_ID___".after_up_del_user_organization_profile() CASCADE;
    DROP TRIGGER IF EXISTS trg_after_up_del_user_organization_profile ON "___ORGANIZATION_ID___"."userOrganizationProfiles";

    --------------------------- organization redis trigger (organization) --------------------------------------------- 

    DROP FUNCTION IF EXISTS "___ORGANIZATION_ID___".after_up_del_organization() CASCADE;
    DROP TRIGGER IF EXISTS trg_after_up_del_organization ON "___ORGANIZATION_ID___"."organizations";

    
    CREATE OR REPLACE FUNCTION "___ORGANIZATION_ID___".after_update_organization_settings()
    RETURNS trigger
    LANGUAGE 'plpgsql'
    COST 100
    VOLATILE NOT LEAKPROOF
    AS $BODY$
    DECLARE
            organization_id 	text;
        BEGIN
            organization_id:= NEW.id:: text;
            PERFORM PG_NOTIFY('organization_settings_changed', organization_id);
            RETURN NULL;
        END;
    $BODY$;

    ALTER FUNCTION "___ORGANIZATION_ID___".after_update_organization_settings()
    OWNER TO ${dbsuperuser};

    GRANT EXECUTE ON FUNCTION "___ORGANIZATION_ID___".after_update_organization_settings() TO ${dbsuperuser};

    GRANT EXECUTE ON FUNCTION "___ORGANIZATION_ID___".after_update_organization_settings() TO PUBLIC;

    GRANT EXECUTE ON FUNCTION "___ORGANIZATION_ID___".after_update_organization_settings() TO ${dbuser};

    CREATE TRIGGER after_up_organization_settings
    AFTER UPDATE OF settings
    ON "___ORGANIZATION_ID___".organizations
    FOR EACH ROW
    EXECUTE PROCEDURE "___ORGANIZATION_ID___".after_update_organization_settings();

    `;
    await (0, dal_utils_1.queryForAllOrganizationSchemasPg)(client, q, "___ORGANIZATION_ID___");
}
exports.down = down;
