"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.dbManager = exports.DalManager = void 0;
const fs_1 = __importDefault(require("fs"));
const knex_1 = __importDefault(require("knex"));
const path_1 = __importDefault(require("path"));
const pg_1 = require("pg");
const redis_1 = require("redis");
const app_util_1 = require("../app.util");
const utils_1 = require("../components/cli/services/db-management/utils");
const dal_memcache_1 = require("./access/dal.memcache");
const dal_access_psql_accessControlPoint_1 = require("./access/psql/dal.access.psql.accessControlPoint");
const dal_access_psql_accessLog_1 = require("./access/psql/dal.access.psql.accessLog");
const dal_access_psql_camera_1 = require("./access/psql/dal.access.psql.camera");
const dal_access_psql_device_1 = require("./access/psql/dal.access.psql.device");
const dal_access_psql_functions_1 = require("./access/psql/dal.access.psql.functions");
const dal_access_psql_identity_1 = require("./access/psql/dal.access.psql.identity");
const dal_access_psql_log_1 = require("./access/psql/dal.access.psql.log");
const dal_access_psql_log_pacs_1 = require("./access/psql/dal.access.psql.log.pacs");
const dal_access_psql_mobileClient_1 = require("./access/psql/dal.access.psql.mobileClient");
const dal_access_psql_notifications_1 = require("./access/psql/dal.access.psql.notifications");
const dal_access_psql_oauth_1 = require("./access/psql/dal.access.psql.oauth");
const dal_access_psql_organization_1 = require("./access/psql/dal.access.psql.organization");
const dal_access_psql_organizationUnit_1 = require("./access/psql/dal.access.psql.organizationUnit");
const dal_access_psql_pacs_1 = require("./access/psql/dal.access.psql.pacs");
const dal_access_psql_pacs2_1 = require("./access/psql/dal.access.psql.pacs2");
const dal_access_psql_region_1 = require("./access/psql/dal.access.psql.region");
const dal_access_psql_report_1 = require("./access/psql/dal.access.psql.report");
const dal_access_psql_social_1 = require("./access/psql/dal.access.psql.social");
const dal_access_psql_system_1 = require("./access/psql/dal.access.psql.system");
const dal_access_psql_user_1 = require("./access/psql/dal.access.psql.user");
const dal_access_psql_userGroup_1 = require("./access/psql/dal.access.psql.userGroup");
const dal_access_psql_visitor_1 = require("./access/psql/dal.access.psql.visitor");
const dal_access_cache_redis_1 = require("./access/redis/dal.access.cache.redis");
const dal_logger_1 = require("./dal.logger");
const dal_access_psql_userfilter_1 = require("./access/psql/dal.access.psql.userfilter");
class DalManager {
    get accessRedisCache() {
        return this._dalRedisCache;
    }
    get accessAccessLog() {
        return this._dalAccessAccessLog;
    }
    get accessLog() {
        return this._dalAccessLog;
    }
    get accessFunctions() {
        return this._dalAccessFunctions;
    }
    get accessLogPacs() {
        return this._dalAccessLogPacs;
    }
    get accessVisitor() {
        return this._dalAccessvisitor;
    }
    get accessReport() {
        return this._dalAccessReport;
    }
    get accessSocial() {
        return this._dalAccessSocial;
    }
    get accessSystem() {
        return this._dalAccessSystem;
    }
    get accessOrganization() {
        return this._dalAccessOrganization;
    }
    get accessOrganizationUnit() {
        return this._dalAccessOrganizationUnit;
    }
    get accessOAuth() {
        return this._dalAccessOAuth;
    }
    get accessNotifications() {
        return this._dalAccessNotifications;
    }
    get accessCamera() {
        return this._dalAccessCamera;
    }
    get accessUser() {
        return this._dalAccessUser;
    }
    get accessDevice() {
        return this._dalAccessDevice;
    }
    get accessMobileClient() {
        return this._dalAccessMobileClient;
    }
    get accessAccessControlPoint() {
        return this._dalAccessAccessControlPoint;
    }
    get accessRegion() {
        return this._dalAccessRegion;
    }
    get accessUserGroup() {
        return this._dalAccessUserGroup;
    }
    get accessIdentity() {
        return this._dalAccessIdentity;
    }
    get accessPacs() {
        return this._dalAccessPacs;
    }
    get accessPacs2() {
        return this._dalAccessPacs2;
    }
    get accessUserFilter() {
        return this._dalAccessUserFilter;
    }
    get transaction() {
        return this._armonDBKnex.transaction.bind(this._armonDBKnex);
    }
    get armondb() {
        return this._armonDBKnex;
    }
    get dbConfig() {
        return this._config;
    }
    async info() {
    }
    onReady() {
        this._onReadyListeners.forEach((l) => l());
    }
    registerToOnReady(listener) {
        this._onReadyListeners.push(listener);
    }
    async init(config, byPassMigrationCheck, loadRedisCache, loadMemCache, listenPacsDbNotifications) {
        this._onReadyListeners = [];
        dal_logger_1.DBlogger.init(config.logDirectory);
        try {
            this._config = config;
            await this.initArmonDbClient();
            await this.instantiateAccessInterfaces(listenPacsDbNotifications);
            if (!fs_1.default.existsSync(path_1.default.resolve(__dirname, "db", "migrations", "postgresql"))) {
                (0, app_util_1.mkDirSyncRecursively)(path_1.default.resolve(__dirname, "db", "migrations", "postgresql"));
            }
            const client = await exports.dbManager.poolMain.connect();
            if (!byPassMigrationCheck) {
                const migrationCheckResult = await (0, utils_1.checkFileSystemAgainstMigrationsTable)(client);
                if (migrationCheckResult && migrationCheckResult.isDbCompatible && !migrationCheckResult.migrationNeeded) {
                    await this.initCache(loadRedisCache, loadMemCache);
                }
                else {
                    client.release();
                    throw Error("Db and file system migrations does not match! please migrate db first!");
                }
            }
            else {
                await this.initCache(false, false);
            }
            client.release();
        }
        catch (error) {
            dal_logger_1.dbLogger.error("Database initialization error %s", error);
            throw error;
        }
        this.onReady();
    }
    async instantiateAccessInterfaces(listenPacsDbNotifications) {
        this._dalAccessAccessControlPoint = new dal_access_psql_accessControlPoint_1.PSQLDalAccessAccessControlPoint(this._armonDBKnex, this._poolMain);
        this._dalAccessCamera = new dal_access_psql_camera_1.PSQLDalAccessCamera(this._armonDBKnex, this._poolMain);
        this._dalAccessIdentity = new dal_access_psql_identity_1.PSQLDalAccessIdentity(this._armonDBKnex, this._poolMain);
        this._dalAccessNotifications = new dal_access_psql_notifications_1.PSQLDalAccessNotifications(this._armonDBKnex, this._poolMain);
        this._dalAccessOAuth = new dal_access_psql_oauth_1.PSQLDalAccessOAuth(this._armonDBKnex, this._poolMain);
        this._dalAccessOrganization = new dal_access_psql_organization_1.PSQLDalAccessOrganization(this._armonDBKnex, this._poolMain);
        this._dalAccessOrganizationUnit = new dal_access_psql_organizationUnit_1.PSQLDalAccessOrganizationUnit(this._armonDBKnex, this._poolMain);
        this._dalAccessPacs = new dal_access_psql_pacs_1.PSQLDalAccessPacs(this._armonDBKnex, this._poolMain);
        this._dalAccessPacs2 = new dal_access_psql_pacs2_1.PSQLDalAccessPacs2(this._armonDBKnex, this._poolMain, listenPacsDbNotifications);
        this._dalAccessvisitor = new dal_access_psql_visitor_1.PSQLDalAccessVisitor(this._armonDBKnex, this._poolMain);
        this._dalAccessRegion = new dal_access_psql_region_1.PSQLDalAccessRegion(this._armonDBKnex, this._poolMain);
        this._dalAccessReport = new dal_access_psql_report_1.PSQLDalAccessReport(this._armonDBKnex, this._poolMain);
        this._dalAccessSocial = new dal_access_psql_social_1.PSQLDalAccessSocial(this._armonDBKnex, this._poolMain);
        this._dalAccessSystem = new dal_access_psql_system_1.PSQLDalAccessSystem(this._armonDBKnex, this._poolMain);
        this._dalAccessUserGroup = new dal_access_psql_userGroup_1.PSQLDalAccessUserGroup(this._armonDBKnex, this._poolMain);
        this._dalAccessUser = new dal_access_psql_user_1.PSQLDalAccessUser(this._armonDBKnex, this._poolMain);
        this._dalAccessDevice = new dal_access_psql_device_1.PSQLDalAccessDevice(this._armonDBKnex, this._poolMain);
        this._dalAccessFunctions = new dal_access_psql_functions_1.PSQLDalAccessFunctions(this._armonDBKnex, this._poolMain);
        this._dalAccessMobileClient = new dal_access_psql_mobileClient_1.PSQLDalAccessMobileClient(this._armonDBKnex, this._poolMain);
        this._dalAccessAccessLog = new dal_access_psql_accessLog_1.PSQLDalAccessAccessLog(this._armonDBKnex, this._poolMain);
        this._dalAccessLog = new dal_access_psql_log_1.PSQLDalAccessLog(this._armonDBKnex, this._poolMain);
        this._dalAccessLogPacs = new dal_access_psql_log_pacs_1.PSQLDalAccessLogPacs(this._armonDBKnex, this._poolMain);
        this._dalAccessUserFilter = new dal_access_psql_userfilter_1.PSQLDalAccessUserFilter(this._armonDBKnex, this._poolMain);
    }
    async initCache(loadRedisCache, loadMemCache) {
        try {
            this._redisClient = (0, redis_1.createClient)({
                url: `redis://${this._config.redis.host}:${this._config.redis.port}`,
            });
            await this._redisClient.connect();
            this._dalRedisCache = new dal_access_cache_redis_1.RedisCache(this._redisClient);
            if (loadRedisCache) {
                await this._dalRedisCache.loadCache();
            }
            if (loadMemCache) {
                await (0, dal_memcache_1.initMemCache)();
            }
        }
        catch (error) {
            dal_logger_1.dbLogger.error("init cache has problems %s", error);
            throw error;
        }
    }
    async initArmonDbClient() {
        let masterDBConnectionPromise = new Promise(async (resolve, reject) => {
            let knexConfigure = {
                debug: false,
                client: "",
                version: "",
                connection: {
                    host: this._config.main.host,
                    port: this._config.main.port,
                    user: this._config.main.username,
                    password: this._config.main.password,
                    database: this._config.main.name,
                },
                migrations: {
                    tableName: "migrations",
                    directory: path_1.default.resolve(__dirname, "db", "migrations"),
                },
                pool: { min: 0, max: this._config.main.knexPool?.max || 40, idleTimeoutMillis: 10000 },
                acquireConnectionTimeout: 20000,
                useNullAsDefault: true,
            };
            knexConfigure.client = "pg";
            knexConfigure.version = "8.6.0";
            this._armonDBKnex = (0, knex_1.default)(knexConfigure);
            this._poolMain = new pg_1.Pool({
                host: this._config.main.host,
                port: this._config.main.port,
                user: this._config.main.username,
                password: this._config.main.password,
                database: this._config.main.name,
                application_name: (process.env.PGAPPNAME || "app_web") + "_" + process.env.HOSTNAME,
                max: 1000,
                keepAlive: true,
            });
            this._armonDBKnex.on("query-error", (error, obj) => this.checkDatabaseErrors(error, obj));
            setTimeout(() => {
                this.checkConnectionStatus(resolve, reject, this._armonDBKnex, "Master DB");
            }, 1000);
        });
        return masterDBConnectionPromise;
    }
    checkDatabaseErrors(error, obj) {
        dal_logger_1.dbLogger.error(error);
        error.message = "Internal DB error";
        throw error;
    }
    async checkConnectionStatus(resolve, reject, knex, dbName) {
        let tableQuery = `SELECT tablename FROM pg_catalog.pg_tables WHERE schemaname='public'`;
        try {
            let result = await knex.raw(tableQuery);
            dal_logger_1.dbLogger.info(dbName + " table count is " + result.rows.length);
            return resolve();
        }
        catch (error) {
            reject(error);
        }
    }
    get poolMain() {
        return this._poolMain;
    }
    set poolMain(value) {
        this._poolMain = value;
    }
    async organizationTransaction(trx, userId, organizationId) {
        const client = await this._poolMain.connect();
        try {
            await client.query(`BEGIN`);
            await client.query(`INSERT INTO "${organizationId}".transaction (id, "userId") VALUES (txid_current(), $1)`, [userId]);
            const result = await trx(client);
            await client.query("COMMIT");
            client.release();
            return result;
        }
        catch (error) {
            await client.query("ROLLBACK");
            client.release(error);
            throw error;
        }
    }
    async systemTransaction(trx) {
        const client = await this._poolMain.connect();
        try {
            await client.query(`BEGIN`);
            const result = await trx(client);
            await client.query("COMMIT");
            client.release();
            return result;
        }
        catch (error) {
            await client.query("ROLLBACK");
            client.release(error);
            throw error;
        }
    }
    async pgTransactionMainDb(trx) {
        const client = await this._poolMain.connect();
        try {
            await client.query("BEGIN");
            const result = await trx(client);
            await client.query("COMMIT");
            client.release();
            return result;
        }
        catch (error) {
            await client.query("ROLLBACK");
            client.release(error);
            throw error;
        }
    }
}
exports.DalManager = DalManager;
exports.dbManager = new DalManager();
