"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getAutoShiftCache = exports.RedisCache = void 0;
const got = require("got");
const app_config_1 = require("../../../app.config");
const app_logs_1 = require("../../../app.logs");
const dal_constants_1 = require("../../dal.constants");
const dal_manager_1 = require("../../dal.manager");
const dal_access_cache_1 = require("../access-interfaces/dal.access.cache");
const cli_queries_1 = require("../psql/cli-queries");
const business_device_1 = require("../../../business/business.device");
const dal_access_psql_pacs2_1 = require("../psql/dal.access.psql.pacs2");
class RedisCache {
    constructor(redis) {
        this._redis = redis;
    }
    async flushDb() {
        try {
            const res = await this._redis.flushDb();
            if (res !== "OK") {
                app_logs_1.logger.error(`[dal.access.cache.redis.ts] flushDb result is not "OK"`);
            }
            return;
        }
        catch (err) {
            app_logs_1.logger.error(`[dal.access.cache.redis.ts] Error while redis flushDB`);
            app_logs_1.logger.error(err);
            throw err;
        }
    }
    setValue(key, value) {
        return new Promise(async (resolve) => {
            let convertedValue = Buffer.from(value).toString("base64");
            try {
                const res = await this._redis.set(key, convertedValue);
                if (res === "OK") {
                    resolve(true);
                }
                else {
                    app_logs_1.logger.error(`[dal.access.cache.redis.ts] setValue result is not "OK" for key ${key}`);
                    resolve(false);
                }
            }
            catch (err) {
                app_logs_1.logger.error("[dal.access.cache.redis.ts] error while setting key: " + err.toString());
                resolve(false);
            }
        });
    }
    async setValues(pairs) {
        try {
            const pipeline = this._redis.multi();
            for (const { key, value } of pairs) {
                const convertedValue = Buffer.from(value).toString("base64");
                pipeline.set(key, convertedValue);
            }
            const results = (await pipeline.exec());
            const allOk = results.every(([err, res]) => !err && res === "OK");
            if (!allOk) {
                app_logs_1.logger.error("[dal.access.cache.redis.ts] setValues: not all keys were set successfully");
            }
            return allOk;
        }
        catch (err) {
            app_logs_1.logger.error("[dal.access.cache.redis.ts] error while setting multiple keys: " + err.toString());
            return false;
        }
    }
    lock(key, expireInSeconds) {
        return new Promise(async (resolve) => {
            try {
                const res = await this._redis.setEx(key, expireInSeconds, "locked");
                if (res === "OK") {
                    resolve(true);
                }
                else {
                    app_logs_1.logger.error(`[dal.access.cache.redis.ts] lock result is not "OK" for key ${key}`);
                    resolve(false);
                }
            }
            catch (err) {
                app_logs_1.logger.error(`[dal.access.cache.redis.ts] Error while locking key ${key}`);
                app_logs_1.logger.error(err);
                resolve(false);
            }
        });
    }
    unlock(key) {
        return new Promise(async (resolve) => {
            try {
                const res = await this._redis.del(key);
                if (res) {
                    resolve(true);
                }
                else {
                    app_logs_1.logger.error(`[dal.access.cache.redis.ts] unlock result is falsy (${res}) for key ${key}, does not exists in redis`);
                    resolve(true);
                }
            }
            catch (err) {
                app_logs_1.logger.error(`[dal.access.cache.redis.ts] Error while unlocking key ${key}`);
                app_logs_1.logger.error(err);
                resolve(false);
            }
        });
    }
    existsLock(key) {
        return new Promise(async (resolve) => {
            try {
                const res = await this._redis.get(key);
                if (res && res === "locked") {
                    resolve(true);
                }
                else {
                    app_logs_1.logger.error(`[dal.access.cache.redis.ts] existsLock result is not "locked" for key ${key}`);
                    resolve(false);
                }
            }
            catch (err) {
                app_logs_1.logger.error(`[dal.access.cache.redis.ts] Error while checking lock for key ${key}`);
                app_logs_1.logger.error(err);
                resolve(false);
            }
        });
    }
    setExpireValue(key, value, expireInSeconds) {
        return new Promise(async (resolve) => {
            let convertedValue = Buffer.from(value).toString("base64");
            try {
                const res = await this._redis.setEx(key, expireInSeconds, convertedValue);
                if (res === "OK") {
                    resolve(true);
                }
                else {
                    app_logs_1.logger.error(`[dal.access.cache.redis.ts] setExpireValue result is not OK for key ${key}`);
                    resolve(false);
                }
            }
            catch (err) {
                app_logs_1.logger.error(`[dal.access.cache.redis.ts] Error while setting auto-expired key-value pair at redis for key ${key}`);
                app_logs_1.logger.error(err);
                resolve(false);
            }
        });
    }
    async getValue(key) {
        try {
            const res = await this._redis.get(key);
            if (!res)
                return null;
            return Buffer.from(res, "base64").toString("utf8");
        }
        catch (err) {
            app_logs_1.logger.error(`[dal.access.cache.redis.ts] Error while getting key-value pair at redis for key ${key}`);
            app_logs_1.logger.error(err);
            return null;
        }
    }
    async getValues(keys) {
        try {
            const results = await this._redis.mGet(keys);
            return results.map((res) => {
                if (!res)
                    return null;
                try {
                    return Buffer.from(res, "base64").toString("utf8");
                }
                catch (err) {
                    app_logs_1.logger.error(`[dal.access.cache.redis.ts] Failed to decode base64 for a key`);
                    app_logs_1.logger.error(err);
                    return null;
                }
            });
        }
        catch (err) {
            app_logs_1.logger.error(`[dal.access.cache.redis.ts] Error during mGet for keys: ${keys.join(", ")}`);
            app_logs_1.logger.error(err);
            return keys.map(() => null);
        }
    }
    delValue(key) {
        return new Promise(async (resolve) => {
            try {
                const res = await this._redis.del(key);
                if (res) {
                    resolve(true);
                }
                else {
                    resolve(true);
                }
            }
            catch (err) {
                app_logs_1.logger.error(`[dal.access.cache.redis.ts] Error while deleting key ${key}`);
                app_logs_1.logger.error(err);
                resolve(false);
            }
        });
    }
    removeUserBadge(organizationId, userId) {
        const key = `${dal_constants_1.DalConstants.NewDataCacheType.UserBadge}${organizationId}${userId}`;
        return this.delValue(key);
    }
    scanValue(pattern) {
        return new Promise(async (resolve, reject) => {
            try {
                const keys = [];
                for await (const key of this._redis.scanIterator({
                    MATCH: pattern,
                    TYPE: "string",
                    COUNT: 100,
                })) {
                    keys.push(key);
                }
                resolve(keys);
            }
            catch (err) {
                app_logs_1.logger.error(`[dal.access.cache.redis.ts] Error while scanValue for pattern ${pattern}`);
                app_logs_1.logger.error(err);
                reject(err);
            }
        });
    }
    scanValuesMatchingPatternAndDelete(key) {
        return new Promise(async (resolve) => {
            const keys = await this.scanValue(key);
            for (const key of keys) {
                await this.delValue(key);
            }
            resolve();
        });
    }
    async loadDeviceStatus(organizationId) {
        const deviceStatus = await dal_manager_1.dbManager.accessDevice.getDeviceStatus(organizationId);
        await dal_manager_1.dbManager.accessRedisCache.setValue(dal_access_cache_1.CacheConstantKeys.DeviceStatus + organizationId, JSON.stringify(deviceStatus));
    }
    async loadCache() {
        await this.flushDb();
        app_logs_1.logger.debug("load cache starts");
        if (!app_config_1.appConfig.doNotLoadCacheAtStart) {
            await this.loadOrganizationCache("", dal_constants_1.DalConstants.NewDataCacheType.HashedNotification);
            const organizationIds = await dal_manager_1.dbManager.accessOrganization.listOrganizations();
            for (const organizationId of organizationIds) {
                await this.loadOrganizationCache(organizationId, dal_constants_1.DalConstants.NewDataCacheType.AccessPoint);
                await this.loadOrganizationCache(organizationId, dal_constants_1.DalConstants.NewDataCacheType.CameraProxyTerminal);
                await this.loadOrganizationCache(organizationId, dal_constants_1.DalConstants.NewDataCacheType.CredentialOwner);
                await this.loadOrganizationCache(organizationId, dal_constants_1.DalConstants.NewDataCacheType.DeviceNotificationInfo);
                await this.loadOrganizationCache(organizationId, dal_constants_1.DalConstants.NewDataCacheType.DeviceState);
                await this.loadOrganizationCache(organizationId, dal_constants_1.DalConstants.NewDataCacheType.Region);
                await this.loadOrganizationCache(organizationId, dal_constants_1.DalConstants.NewDataCacheType.UserBadge);
                await this.loadOrganizationCache(organizationId, dal_constants_1.DalConstants.NewDataCacheType.UserNotificiation);
                await this.loadOrganizationCache(organizationId, dal_constants_1.DalConstants.NewDataCacheType.Organization);
                await this.loadOrganizationCache(organizationId, dal_constants_1.DalConstants.NewDataCacheType.SnapshotSettings);
            }
        }
        app_logs_1.logger.info("cache loaded");
        return Promise.resolve();
    }
    async loadOrganizationCache(organizationId, type) {
        switch (type) {
            case dal_constants_1.DalConstants.NewDataCacheType.HashedNotification:
                await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                    await dal_manager_1.dbManager.accessNotifications.getHashedNotifications(trx, async (rows) => {
                        if (rows) {
                            for (const row of rows) {
                                let cachedValue = await dal_manager_1.dbManager.accessRedisCache.getValue(row.hashKey);
                                if (cachedValue) {
                                    const cachedNotificationIds = JSON.parse(cachedValue);
                                    if (!cachedNotificationIds.includes(row.id)) {
                                        cachedNotificationIds.push(row.id);
                                    }
                                    cachedValue = JSON.stringify(cachedNotificationIds);
                                }
                                else {
                                    cachedValue = JSON.stringify([row.id]);
                                }
                                await dal_manager_1.dbManager.accessRedisCache.setValue(row.hashKey, cachedValue);
                            }
                        }
                    });
                });
                break;
            case dal_constants_1.DalConstants.NewDataCacheType.Organization:
                await dal_manager_1.dbManager.accessOrganization.getAllOrganizationRedisPersistentData({
                    organizationId,
                    onData: async (rows) => {
                        if (rows) {
                            await Promise.all(rows.map(async (row) => {
                                const { id, ...data } = row;
                                const key = `${dal_constants_1.DalConstants.NewDataCacheType.Organization}${organizationId}`;
                                await this.setValue(key, JSON.stringify(data));
                            }));
                        }
                    },
                });
                break;
            case dal_constants_1.DalConstants.NewDataCacheType.DeviceState:
                await dal_manager_1.dbManager.accessDevice.getAllDeviceStateRedisPersistentData({
                    organizationId,
                    onData: async (rows) => {
                        if (rows) {
                            await Promise.all(rows.map(async (row) => {
                                const { id, ...data } = row;
                                const key = `${dal_constants_1.DalConstants.NewDataCacheType.DeviceState}${organizationId}${id}`;
                                await this.setValue(key, JSON.stringify(data));
                            }));
                        }
                    },
                });
                break;
            case dal_constants_1.DalConstants.NewDataCacheType.DeviceNotificationInfo:
                await dal_manager_1.dbManager.accessDevice.getAllDeviceNotificationInfoRedisPersistentData({
                    organizationId,
                    onData: async (rows) => {
                        if (rows) {
                            await Promise.all(rows.map(async (row) => {
                                const { id, ...data } = row;
                                const key = `${dal_constants_1.DalConstants.NewDataCacheType.DeviceNotificationInfo}${organizationId}${id}`;
                                await this.setValue(key, JSON.stringify(data));
                            }));
                        }
                    },
                });
                break;
            case dal_constants_1.DalConstants.NewDataCacheType.CameraProxyTerminal:
                await dal_manager_1.dbManager.accessDevice.getAllCameraProxyTerminalsRedisPersistentData({
                    organizationId,
                    onData: async (rows) => {
                        if (rows) {
                            await Promise.all(rows.map(async (row) => {
                                const key = `${dal_constants_1.DalConstants.NewDataCacheType.CameraProxyTerminal}${organizationId}${row.c}`;
                                await this.setValue(key, row.t);
                            }));
                        }
                    },
                });
                break;
            case dal_constants_1.DalConstants.NewDataCacheType.AccessPoint:
                await dal_manager_1.dbManager.accessAccessControlPoint.getAllAccessControlPointsRedisPersistentData({
                    organizationId,
                    onData: async (rows) => {
                        if (rows) {
                            await Promise.all(rows.map(async (row) => {
                                const { id, ...data } = row;
                                const key = `${dal_constants_1.DalConstants.NewDataCacheType.AccessPoint}${organizationId}${id}`;
                                await this.setValue(key, JSON.stringify(data));
                            }));
                        }
                    },
                });
                break;
            case dal_constants_1.DalConstants.NewDataCacheType.Region:
                await dal_manager_1.dbManager.accessRegion.getAllRegionRedisPersistentData({
                    organizationId,
                    onData: async (rows) => {
                        if (rows) {
                            await Promise.all(rows.map(async (row) => {
                                const { id, ...data } = row;
                                const key = `${dal_constants_1.DalConstants.NewDataCacheType.Region}${organizationId}${id}`;
                                await this.setValue(key, JSON.stringify(data));
                            }));
                        }
                    },
                });
                break;
            case dal_constants_1.DalConstants.NewDataCacheType.UserNotificiation:
                await dal_manager_1.dbManager.accessUser.getAllUserNotificationRedisPersistentData({
                    organizationId,
                    onData: async (rows) => {
                        if (rows && rows.length > 0) {
                            await Promise.all(rows.map(async (row) => {
                                const { id, ...data } = row;
                                const key = `${dal_constants_1.DalConstants.NewDataCacheType.UserNotificiation}${organizationId}${id}`;
                                await this.setValue(key, JSON.stringify(data));
                            }));
                        }
                    },
                });
                break;
            case dal_constants_1.DalConstants.NewDataCacheType.UserBadge:
                await dal_manager_1.dbManager.accessUser.getAllUserOrganizationBadgeRedisPersistentData({
                    organizationId,
                    onData: async (rows) => {
                        if (rows) {
                            await Promise.all(rows.map(async (row) => {
                                const key = `${dal_constants_1.DalConstants.NewDataCacheType.UserBadge}${organizationId}${row.id}`;
                                await this.setValue(key, JSON.stringify(row));
                            }));
                        }
                    },
                });
                break;
            case dal_constants_1.DalConstants.NewDataCacheType.DeviceRabbitMQConnectionName:
                const response = await got.get(`http://${app_config_1.appConfig.amqpClient.hostname}:${app_config_1.appConfig.amqpClient.managementApiPort}/api/connections`, {
                    json: true,
                    headers: {
                        Authorization: app_config_1.appConfig.amqpManagementApiAuth,
                    },
                });
                const devices = await dal_manager_1.dbManager.accessDevice.getTerminalDeviceIdList();
                for (const device of devices) {
                    const rabbitConnectionName = response.body.find((r) => r.user === device.id)?.name;
                    if (rabbitConnectionName) {
                        await (0, business_device_1.updateDeviceConnectionStatus)({
                            organizationId: device.organizationId,
                            deviceId: device.id,
                            isConnected: true,
                            isRestart: true,
                        });
                        await dal_manager_1.dbManager.accessRedisCache.setValue(dal_constants_1.DalConstants.NewDataCacheType.DeviceRabbitMQConnectionName + "_" + rabbitConnectionName, JSON.stringify({ id: device.id, organizationId: device.organizationId }));
                    }
                    else {
                        await (0, business_device_1.updateDeviceConnectionStatus)({
                            organizationId: device.organizationId,
                            deviceId: device.id,
                            isConnected: false,
                            isRestart: true,
                        });
                    }
                }
                break;
            case dal_constants_1.DalConstants.NewDataCacheType.SnapshotSettings:
                const organizationSnapshotSettings = await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                    return dal_manager_1.dbManager.accessOrganization.getOrganizationSnapshotSettings({ organizationId, trx });
                });
                const key = `${dal_constants_1.DalConstants.NewDataCacheType.SnapshotSettings}${organizationId}`;
                await this.setValue(key, JSON.stringify(organizationSnapshotSettings));
                break;
            default:
                app_logs_1.logger.warn(`DataCacheType[${type}] should be used only lazy load mode!`);
                break;
        }
    }
    async deleteDirtyCachedData(organizationId, type, id) {
        let key = "";
        if (type === dal_constants_1.DalConstants.NewDataCacheType.Organization || type === dal_constants_1.DalConstants.NewDataCacheType.SnapshotSettings || type === dal_constants_1.DalConstants.NewDataCacheType.ShiftRule) {
            key = `${type}${organizationId}`;
        }
        else {
            key = `${type}${organizationId}${id}`;
        }
        await this.delValue(key);
    }
    async getAccessPointCache(params) {
        const key = `${dal_constants_1.DalConstants.NewDataCacheType.AccessPoint}${params.organizationId}${params.accessPointId}`;
        try {
            const cachedData = await this.getValue(key);
            if (cachedData) {
                return JSON.parse(cachedData);
            }
            else {
                const persistentData = params.trx
                    ? await (0, cli_queries_1.getAccessPointRedisPersistentData)({ organizationId: params.organizationId, accessPointId: params.accessPointId, trx: params.trx })
                    : await dal_manager_1.dbManager.accessAccessControlPoint.getAccessPointRedisPersistentData(params);
                await this.setValue(key, JSON.stringify(persistentData));
                return persistentData;
            }
        }
        catch (error) {
            app_logs_1.logger.error(`Error at getAccessPointCache organizationId[${params.organizationId}] accessPointId[${params.accessPointId}]" err: ${error}`);
        }
        return null;
    }
    async getProxyTerminalIdOfCameraCache(params) {
        const key = `${dal_constants_1.DalConstants.NewDataCacheType.CameraProxyTerminal}${params.organizationId}${params.cameraId}`;
        try {
            const cachedData = await this.getValue(key);
            if (cachedData) {
                return JSON.parse(cachedData);
            }
            else {
                const deviceState = await dal_manager_1.dbManager.accessDevice.getProxyTerminalIdCamera(params);
                await this.setValue(key, JSON.stringify(deviceState));
                return deviceState;
            }
        }
        catch (error) {
            app_logs_1.logger.error(`Error at getDeviceStateCache organizationId[${params.organizationId}] deviceId[${params.cameraId}] err: ${error}`);
        }
        return null;
    }
    async getDeviceStateCache(params) {
        const key = `${dal_constants_1.DalConstants.NewDataCacheType.DeviceState}${params.organizationId}${params.deviceId}`;
        try {
            const cachedData = await this.getValue(key);
            if (cachedData) {
                return JSON.parse(cachedData);
            }
            else {
                const deviceState = params.trx
                    ? await (0, cli_queries_1.getDeviceStatePersistentData)({ organizationId: params.organizationId, deviceId: params.deviceId, trx: params.trx })
                    : await dal_manager_1.dbManager.accessDevice.getDeviceStatePersistentData(params);
                await this.setValue(key, JSON.stringify(deviceState));
                return deviceState;
            }
        }
        catch (error) {
            app_logs_1.logger.error(`Error at getDeviceStateCache organizationId[${params.organizationId}] deviceId[${params.deviceId}] err: ${error}`);
        }
        return null;
    }
    async getDeviceNotificationInfoCache(params) {
        const key = `${dal_constants_1.DalConstants.NewDataCacheType.DeviceNotificationInfo}${params.organizationId}${params.deviceId}`;
        try {
            const cachedData = await this.getValue(key);
            if (cachedData) {
                return JSON.parse(cachedData);
            }
            else {
                const deviceState = params.trx
                    ? await (0, cli_queries_1.getDeviceNotificationInfoPersistentData)({ organizationId: params.organizationId, deviceId: params.deviceId, trx: params.trx })
                    : await dal_manager_1.dbManager.accessDevice.getDeviceNotificationInfoPersistentData(params);
                await this.setValue(key, JSON.stringify(deviceState));
                return deviceState;
            }
        }
        catch (error) {
            app_logs_1.logger.error(`Error at getDeviceStateCache organizationId[${params.organizationId}] deviceId[${params.deviceId}] err: ${error}`);
        }
        return null;
    }
    async getUserNotificationCache(params) {
        const key = `${dal_constants_1.DalConstants.NewDataCacheType.UserNotificiation}${params.organizationId}${params.userId}`;
        try {
            const cachedData = await this.getValue(key);
            if (cachedData) {
                return JSON.parse(cachedData);
            }
            else {
                const persistentData = params.trx
                    ? await (0, cli_queries_1.getUserOrganizationRedisPersistentData)({ organizationId: params.organizationId, userId: params.userId, trx: params.trx })
                    : await dal_manager_1.dbManager.accessUser.getUserOrganizationRedisPersistentData(params);
                await this.setValue(key, JSON.stringify(persistentData));
                return persistentData;
            }
        }
        catch (error) {
            app_logs_1.logger.error(`Error at get user cache organizationId[${params.organizationId}] userId[${params.userId}]" err: ${error}`);
        }
        return null;
    }
    async getUsersNotificationCache(params) {
        const keys = params.userIds.map((userId) => `${dal_constants_1.DalConstants.NewDataCacheType.UserNotificiation}${params.organizationId}${userId}`);
        try {
            const cachedData = await this.getValues(keys);
            const resultMap = new Map();
            const missingUserIds = [];
            cachedData.forEach((data, index) => {
                const userId = params.userIds[index];
                if (data) {
                    resultMap.set(userId, JSON.parse(data));
                }
                else {
                    missingUserIds.push(userId);
                }
            });
            if (missingUserIds.length === 0) {
                return resultMap;
            }
            const persistentData = await dal_manager_1.dbManager.accessUser.getUsersOrganizationRedisPersistentData({
                organizationId: params.organizationId,
                userIds: missingUserIds,
                trx: params.trx,
            });
            const pairs = [];
            for (const [userId, info] of persistentData.entries()) {
                const key = `${dal_constants_1.DalConstants.NewDataCacheType.UserNotificiation}${params.organizationId}${userId}`;
                pairs.push({ key, value: JSON.stringify(info) });
                resultMap.set(userId, info);
            }
            await this.setValues(pairs);
            return resultMap;
        }
        catch (error) {
            app_logs_1.logger.error(`Error at get user cache organizationId[${params.organizationId}] userIds[${params.userIds.join(",")}] err: ${error}`);
        }
        return new Map();
    }
    async getUserBadgeCache(params) {
        const key = `${dal_constants_1.DalConstants.NewDataCacheType.UserBadge}${params.organizationId}${params.userId}`;
        try {
            const cachedData = await this.getValue(key);
            if (cachedData) {
                return JSON.parse(cachedData);
            }
            else {
                app_logs_1.logger.warn("user badge fetched from db");
                const persistentData = await dal_manager_1.dbManager.accessUser.getUserOrganizationBadgeRedisPersistentData(params);
                await this.setValue(key, JSON.stringify(persistentData));
                return persistentData;
            }
        }
        catch (error) {
            app_logs_1.logger.error(`Error at getUserBadgeCache organizationId[${params.organizationId}] userId[${params.userId}]" err: ${error}`);
        }
        return null;
    }
    async getUserBadgeCacheBulk(params) {
        const { organizationId, userIds, trx } = params;
        const result = [];
        const keys = userIds.map((userId) => `${dal_constants_1.DalConstants.NewDataCacheType.UserBadge}${organizationId}${userId}`);
        const cachedResults = await this.getValues(keys);
        const missingUserIds = [];
        userIds.forEach((userId, index) => {
            const cachedValue = cachedResults[index];
            if (cachedValue) {
                const parsed = JSON.parse(cachedValue);
                result.push(parsed);
            }
            else {
                missingUserIds.push(userId);
            }
        });
        if (missingUserIds.length > 0) {
            app_logs_1.logger.warn(`User badges fetched from DB for ${missingUserIds.length} users`);
            const persistentList = await dal_manager_1.dbManager.accessUser.getUserOrganizationBadgeRedisPersistentDataBulk({
                organizationId,
                userIds: missingUserIds,
                trx,
            });
            for (const data of persistentList) {
                const key = `${dal_constants_1.DalConstants.NewDataCacheType.UserBadge}${organizationId}${data.id}`;
                await this.setValue(key, JSON.stringify(data));
                result.push(data.id, data);
            }
        }
        return result;
    }
    async getRegionCache(params) {
        const key = `${dal_constants_1.DalConstants.NewDataCacheType.Region}${params.organizationId}${params.regionId}`;
        try {
            const cachedData = await this.getValue(key);
            if (cachedData) {
                return JSON.parse(cachedData);
            }
            else {
                const persistentData = params.trx
                    ? await (0, cli_queries_1.getRegionRedisPersistentData)({ organizationId: params.organizationId, regionId: params.regionId, trx: params.trx })
                    : await dal_manager_1.dbManager.accessRegion.getRegionRedisPersistentData({ organizationId: params.organizationId, regionId: params.regionId, trx: params.trx });
                await this.setValue(key, JSON.stringify(persistentData));
                return persistentData;
            }
        }
        catch (error) {
            app_logs_1.logger.error(`Error at getRegionCache organizationId[${params.organizationId}] regionId[${params.regionId}]" err: ${error}`);
        }
        return null;
    }
    async getOrganizationCache(params) {
        const key = `${dal_constants_1.DalConstants.NewDataCacheType.Organization}${params.organizationId}`;
        try {
            const cachedData = await this.getValue(key);
            if (cachedData) {
                return JSON.parse(cachedData);
            }
            else {
                const persistentData = params.trx
                    ? await (0, cli_queries_1.getOrganizationRedisPersistentData)({ organizationId: params.organizationId, trx: params.trx })
                    : await dal_manager_1.dbManager.accessOrganization.getOrganizationRedisPersistentData({ organizationId: params.organizationId, trx: params.trx });
                await this.setValue(key, JSON.stringify(persistentData));
                return persistentData;
            }
        }
        catch (error) {
            app_logs_1.logger.error(`Error at getOrganizationCache organizationId[${params.organizationId}] err: ${error}`);
        }
        return null;
    }
    async getOrganizationSnapshotSettingsCache(params) {
        const key = `${dal_constants_1.DalConstants.NewDataCacheType.SnapshotSettings}${params.organizationId}`;
        try {
            const cachedData = await this.getValue(key);
            if (cachedData) {
                return JSON.parse(cachedData);
            }
            else {
                const persistentData = await dal_manager_1.dbManager.accessOrganization.getOrganizationSnapshotSettings({ organizationId: params.organizationId, trx: params.trx });
                await this.setValue(key, JSON.stringify(persistentData));
                return persistentData;
            }
        }
        catch (error) {
            app_logs_1.logger.error(`Error at getOrganizationSnapshotSettingsCache organizationId[${params.organizationId}] err: ${error}`);
        }
        return null;
    }
    async setVerificationToken(username, paramms) {
        if (paramms.ets) {
            return await this.setExpireValue(username, JSON.stringify(paramms), paramms.ets);
        }
        else {
            return await this.setValue(username, JSON.stringify(paramms));
        }
    }
    async getVerificationToken(userId) {
        const redisValue = JSON.parse(await this.getValue(userId));
        return redisValue;
    }
}
exports.RedisCache = RedisCache;
async function getAutoShiftCache(params) {
    const key = `${dal_constants_1.DalConstants.NewDataCacheType.ShiftRule}${params.organizationId}`;
    try {
        const cachedData = await params.redisCache.getValue(key);
        if (cachedData) {
            return JSON.parse(cachedData);
        }
        else {
            const persistentData = await (0, dal_access_psql_pacs2_1.checkAutoShiftRuleExistency)(params.organizationId, params.trx);
            await params.redisCache.setValue(key, JSON.stringify(persistentData));
            return persistentData;
        }
    }
    catch (error) {
        app_logs_1.logger.error(`Error at getAutoShiftCache organizationId[${params.organizationId}] err: ${error}`);
    }
    return null;
}
exports.getAutoShiftCache = getAutoShiftCache;
