"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PSQLDalAccessVisitor = void 0;
const i18n_1 = __importDefault(require("i18n"));
const lodash_1 = __importDefault(require("lodash"));
const moment_1 = __importDefault(require("moment"));
const luxon_1 = require("luxon");
const moment_range_1 = require("moment-range");
const uuid_1 = __importDefault(require("uuid"));
const api_error_1 = require("../../../api/api.error");
const app_config_1 = require("../../../app.config");
const app_enums_1 = require("../../../app.enums");
const app_logs_1 = require("../../../app.logs");
const business_report_export_1 = require("../../../business/report/business.report.export");
const report_util_1 = require("../../../business/report/report.util");
const restapi_1 = require("../../../lib/es/models/restapi");
const dal_constants_1 = require("../../dal.constants");
const dal_logger_1 = require("../../dal.logger");
const dal_manager_1 = require("../../dal.manager");
const dal_utils_1 = require("../../dal.utils");
const dal_db_armon_schema_1 = require("../../db/armon/dal.db.armon.schema");
const predefined_permissions_1 = require("../../db/predefined/predefined.permissions");
const predefined_roles_1 = require("../../db/predefined/predefined.roles");
const dal_access_error_1 = require("../dal.access.error");
const dal_access_models_1 = require("../dal.access.models");
const dal_memcache_1 = require("../dal.memcache");
const dal_visitor_utils_1 = require("../dal.visitor.utils");
const dal_access_rdb_visitor_1 = require("../rdb/dal.access.rdb.visitor");
const Cursor = require("pg-cursor");
class PSQLDalAccessVisitor extends dal_access_rdb_visitor_1.RDBDalAccessVisitor {
    constructor(knex, pgPool) {
        super(knex, pgPool);
    }
    async isLicensePlateRegisteredByVisitors(organizationId, licensePlate, trx) {
        let { rows } = await (trx ?? this._pgPool).query(`
			SELECT COUNT(*) as count
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorProfiles}" as ovp
			INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationActiveVisits}" as oav
				ON ovp.id = oav."organizationVisitorProfileId"
			WHERE ovp."deletedAt" is NULL AND ovp."extensionFields"->>'licencePlate' = $1 AND oav.state = 2;
		`, [licensePlate]);
        return rows[0].count > 0;
    }
    async terminateVisitNew(organizationId, visitId) {
        return await this.dbClient.transaction(async (trx) => {
            let visit = await trx
                .withSchema(organizationId)
                .table("organizationActiveVisits as uav")
                .innerJoin("organizationVisitorProfiles as ovp", "uav.organizationVisitorProfileId", "ovp.id")
                .innerJoin("userOrganizations as uo", "uo.id", "ovp.userOrganizationId")
                .where("uav.id", visitId)
                .where("uav.organizationId", organizationId)
                .first("uo.userId");
            if (!visit || !visit.userId)
                (0, dal_access_error_1.throwDbAccessNotFoundErrorTr)("ERRORS.VISITOR.VISITNOTFOUND");
            let opTime = new Date();
            await trx
                .withSchema(organizationId)
                .table(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials)
                .where("userId", visit.userId)
                .where("organizationId", organizationId)
                .whereNull("deletedAt")
                .update({
                deletedAt: opTime,
                updatedAt: opTime,
            });
            await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.userAccessRights).where("userId", visit.userId).where("access", true).whereNull("deletedAt").update({
                deletedAt: opTime,
                updatedAt: opTime,
            });
            await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationActiveVisits).where("id", visitId).delete();
            return Promise.resolve(visit.userId);
        });
    }
    async getVisitAndVisitorFormSettings(organizationId, trx) {
        let query = this.dbClient
            .withSchema(organizationId)
            .from(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorModuleSettings)
            .where("organizationId", organizationId)
            .first("settings", "settingsVersion");
        if (trx) {
            query.transacting(trx);
        }
        let settings = await query;
        if (!settings)
            (0, dal_access_error_1.throwDbAccessNotFoundErrorTr)("ERRORS.VISITOR.SETTINGSNOTFOUND");
        return Object.assign({}, settings.settings, { version: settings.settingsVersion });
    }
    prepareOrderForListActiveVisitors(options, qb) {
        let order = "";
        if (options.sortVisitFormField) {
            if (options.sortVisitFormField.extension) {
                order += `oav."extensionFields" ->> '` + options.sortVisitFormField.name + "'";
            }
            else {
                order += 'oav."' + options.sortVisitFormField.name + '"';
            }
        }
        else if (options.sortProfileFormField) {
            if (options.sortProfileFormField.extension) {
                order += `ovp."extensionFields"->>'` + options.sortProfileFormField.name + "'";
            }
            else {
                order += 'ovp."' + options.sortProfileFormField.name + '"';
            }
        }
        else {
            order += ' "startUtc"';
        }
        order += options.sortAscending ? " ASC" : " DESC";
        order += ' NULLS FIRST, CASE WHEN "startUtc" is null then ovp.name END DESC';
        return qb.orderByRaw(order);
    }
    prepareActiveVisitWhereOptionsForListActiveVisitors(organizationId, options, visitorModuleSettings, qb) {
        qb.where("oav.organizationId", "=", organizationId).whereNull("ovp.deletedAt");
        if (options.filter) {
            if (options.filter.profileFields && options.filter.profileFields.length > 0) {
                for (const filterField of options.filter.profileFields) {
                    for (const profileFormField of visitorModuleSettings.visitorProfileFormFields) {
                        if (profileFormField.reportable && filterField.name === profileFormField.name) {
                            if (profileFormField.extension) {
                                qb.where(this.dbClient.raw(`unaccent(ovp."extensionFields"->>'${filterField.name}') ilike unaccent('%${filterField.value.toUpperCase()}%')`));
                            }
                            else {
                                qb.where(this.dbClient.raw(`unaccent(ovp."${filterField.name}") ilike unaccent('%${filterField.value.toUpperCase()}%')`));
                            }
                            break;
                        }
                    }
                }
            }
            if (options.filter.visitFields && options.filter.visitFields.length > 0) {
                for (const filterField of options.filter.visitFields) {
                    for (const visitFormField of visitorModuleSettings.visitFormFields) {
                        if (visitFormField.reportable && filterField.name === visitFormField.name) {
                            if (visitFormField.extension) {
                                if (visitFormField.type == dal_constants_1.DalConstants.FormFieldType.Select) {
                                    let options = visitFormField.options;
                                    let targetValue = options.options.find((s) => s.captionLines.some((a) => lodash_1.default.deburr(a).toUpperCase().indexOf(lodash_1.default.deburr(filterField.value).toUpperCase()) > -1));
                                    if (targetValue) {
                                        qb.where(this.dbClient.raw(`oav."extensionFields"->>'${filterField.name}' = '${targetValue.value}'`));
                                    }
                                }
                                else {
                                    qb.where(this.dbClient.raw(`unaccent(oav."extensionFields"->>'${filterField.name}') ilike unaccent('%${filterField.value}%')`));
                                }
                            }
                            else {
                                switch (visitFormField.name) {
                                    case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.accessControlPoints:
                                        break;
                                    case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.escort:
                                        qb.andWhere(this.dbClient.raw(`unaccent(oav."escortPerson"->>'name') ilike unaccent('${filterField.value}%)'
                                        or unaccent(oav."escortPerson"->>'surname') ilike unaccent('${filterField.value}%)'
                                        or unaccent(oav."escortPerson"->>'uniqueId') ilike unaccent('${filterField.value}%')`));
                                        break;
                                    case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.visitedOrganizationUnit:
                                        qb.where(this.dbClient.raw(`unaccent(oav."visitedOrganizationUnit"->>'name') ilike unaccent('${filterField.value}%')`));
                                        break;
                                    case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.visitedPerson:
                                        qb.andWhere(this.dbClient.raw(`unaccent(vuop.name || ' ' ||vuop.surname) ilike unaccent('%${filterField.value}%')`));
                                        break;
                                    default:
                                        qb.where(this.dbClient.raw(`unaccent(oav."${filterField.name}") ilike unaccent('%${filterField.value}%')`));
                                        break;
                                }
                            }
                            break;
                        }
                    }
                }
            }
        }
        return qb;
    }
    prepareVisitColumnsForListActiveVisitors(visitorModuleSettings) {
        let result = [{ visitId: "oav.id" }, "organizationVisitorProfileId", "startUtc", "state", "expectedStartUtc", "expectedEndUtc"];
        for (const visitField of visitorModuleSettings.visitFormFields) {
            if (visitField.reportable) {
                if (visitField.extension) {
                    result.push({ "oav.extensionFields": "oav.extensionFields" });
                    return result;
                }
            }
        }
        return result;
    }
    prepareVisitorProfileReportableColumns(visitorModuleSettings) {
        let result = [{ "ovp.thumbnail": "ovp.thumbnail" }];
        let extensionAdded = false;
        for (const visitorProfileField of visitorModuleSettings.visitorProfileFormFields) {
            if (visitorProfileField.reportable) {
                if (visitorProfileField.extension) {
                    extensionAdded = true;
                }
                else {
                    switch (visitorProfileField.name) {
                        case dal_access_models_1.DbAccessModel.PredefinedVisitorProfileFormFields.name:
                            result.push({ "ovp.name": "ovp.name" });
                            break;
                        case dal_access_models_1.DbAccessModel.PredefinedVisitorProfileFormFields.surname:
                            result.push({ "ovp.surname": "ovp.surname" });
                            break;
                        case dal_access_models_1.DbAccessModel.PredefinedVisitorProfileFormFields.uniqueId:
                            result.push({ "ovp.uniqueId": "ovp.uniqueId" });
                            break;
                        default:
                            break;
                    }
                }
            }
        }
        if (extensionAdded) {
            result.push({ "ovp.extensionFields": "ovp.extensionFields" });
        }
        return result;
    }
    prepareVisitJoinsAndColumnsForListActiveVisitors(organizationId, visitorModuleSettings, qb) {
        let columns = [];
        for (const visitField of visitorModuleSettings.visitFormFields) {
            if (visitField.reportable) {
                switch (visitField.name) {
                    case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.accessControlPoints:
                        columns.push(this.dbClient.raw(`
                        (select to_json(array_agg(json_build_object(
                            'id', acp."id", 'name', acp."name"
                            ))) from "${organizationId}"."accessControlPoints" as acp
                        inner join "${organizationId}"."userAccessRights" as uar on acp."id" = uar."accessControlPointId"
                        inner join "${organizationId}"."userOrganizations" as uo on uar."userId" = uo."userId"
                        where uar."deletedAt" is null and acp."deletedAt" is null
                        and uo."deletedAt" is null
                        and uo."id" = ovp."userOrganizationId"
                        and uar.access = true
                        ) "oav.accessControlPoints"
                    `));
                        break;
                    case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.escort:
                        qb.leftOuterJoin("userOrganizationProfiles as euop", (join) => {
                            join.on("euop.userOrganizationId", "=", "oav.escortUserOrganizationId").andOnNotNull("oav.escortUserOrganizationId");
                        });
                        columns.push(this.dbClient.raw(`
                         json_build_object(
                         'userId', euop."userId",
                         'uniqueId', euop."uniqueId", 'name', euop."name",
                         'surname', euop."surname"
                         ) as "oav.escortPerson"
                     `));
                        break;
                    case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.visitedOrganizationUnit:
                        qb.leftOuterJoin("organizationUnits as vou", (join) => {
                            join.on("vou.id", "=", "oav.visitedOrganizationUnitId").andOnNotNull("oav.visitedOrganizationUnitId");
                        });
                        columns.push(this.dbClient.raw(`
                        json_build_object(
                         'id', vou."id",
                         'name', vou."name") as "oav.visitedOrganizationUnit"
                     `));
                        break;
                    case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.visitedPerson:
                        qb.leftOuterJoin("userOrganizationProfiles as vuop", (join) => {
                            join.on("vuop.userOrganizationId", "=", "oav.visitedUserOrganizationId").andOnNotNull("oav.visitedUserOrganizationId");
                        });
                        columns.push(this.dbClient.raw(`
                             json_build_object(
                             'userId', vuop."userId",
                             'uniqueId', vuop."uniqueId", 'name', vuop."name",
                             'surname', vuop."surname"
                             ) as "oav.visitedPerson"
                         `));
                        break;
                    case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.credentials:
                    default:
                        break;
                }
            }
        }
        return {
            qb: qb,
            columns: columns,
        };
    }
    async listUnterminatedVisits(organizationId, options, visitorModuleSettings, requestUserId) {
        let result = {
            options: options,
            total: 0,
            items: [],
        };
        let columns = [];
        columns = columns.concat(this.prepareVisitColumnsForListActiveVisitors(visitorModuleSettings));
        columns = columns.concat(this.prepareVisitorProfileReportableColumns(visitorModuleSettings));
        let qb = this.dbClient.withSchema(organizationId).from("organizationActiveVisits as oav");
        if (!requestUserId) {
            qb.whereRaw(` ((oav.state = ? and oav."visitorRegistrationPointId" is null ) or oav."visitorRegistrationPointId" = ?)`, [
                dal_constants_1.DalConstants.UnterminatedVisitState.Expected,
                options.visitorRegistrationPointId,
            ]);
        }
        let filterExists = options.filter && (options.filter.profileFields.length > 0 || options.filter.visitFields.length > 0);
        let qbCount = null;
        if (options.filter && options.filter.state) {
            if (options.filter.state !== dal_constants_1.DalConstants.UnterminatedVisitState.All) {
                qb.where("oav.state", options.filter.state);
            }
        }
        if (!requestUserId && visitorModuleSettings.maxExpectedDayCount && visitorModuleSettings.maxExpectedDayCount > 0) {
            let now = new Date();
            qb.andWhereRaw('(oav.state != ? or (oav.state = ? and oav."expectedStartUtc" <= ? and oav."expectedEndUtc" >= ?))', [
                dal_constants_1.DalConstants.UnterminatedVisitState.Expected,
                dal_constants_1.DalConstants.UnterminatedVisitState.Expected,
                now,
                now,
            ]);
        }
        if (requestUserId) {
            let requestUserOrganization = await this.dbClient
                .withSchema(organizationId)
                .table("userOrganizations")
                .where("userId", requestUserId)
                .where("organizationId", organizationId)
                .whereNull("deletedAt")
                .first("id");
            if (requestUserOrganization && requestUserOrganization.id)
                qb.where("oav.visitedUserOrganizationId", requestUserOrganization.id);
        }
        if (!filterExists) {
            qbCount = qb.clone();
            qbCount = qbCount.where("oav.organizationId", "=", organizationId);
            result.total = await qbCount.count("oav.id as c").then((rows) => {
                return parseInt(rows[0].c);
            });
            if (result.total === 0) {
                return Promise.resolve(result);
            }
        }
        qb.innerJoin("organizationVisitorProfiles as ovp", (join) => {
            join.on("ovp.id", "=", "oav.organizationVisitorProfileId");
        });
        let visitJoins = this.prepareVisitJoinsAndColumnsForListActiveVisitors(organizationId, visitorModuleSettings, qb);
        qb = visitJoins.qb;
        columns = columns.concat(visitJoins.columns);
        qb = this.prepareActiveVisitWhereOptionsForListActiveVisitors(organizationId, options, visitorModuleSettings, qb);
        if (filterExists) {
            qbCount = qb.clone();
            result.total = await qbCount.count("oav.id as c").then((rows) => {
                return parseInt(rows[0].c);
            });
            if (result.total === 0) {
                return Promise.resolve(result);
            }
        }
        qb = this.prepareOrderForListActiveVisitors(options, qb);
        qb.column(columns).select().offset(options.skip).limit(options.take);
        await qb.then((rows) => {
            result.items = rows.map((t) => {
                let visitState = t.state;
                let visitorProfileFields = {};
                for (const profileField of visitorModuleSettings.visitorProfileFormFields) {
                    if (visitState == dal_constants_1.DalConstants.UnterminatedVisitState.Expected) {
                        if (profileField.reportable && (!profileField.preRegistrationFieldSettings || !profileField.preRegistrationFieldSettings.invisible)) {
                            if (profileField.extension) {
                                visitorProfileFields[profileField.name] = t["ovp.extensionFields"][profileField.name];
                            }
                            else {
                                visitorProfileFields[profileField.name] = t["ovp." + profileField.name];
                            }
                        }
                    }
                    else {
                        if (profileField.reportable) {
                            if (profileField.extension) {
                                visitorProfileFields[profileField.name] = t["ovp.extensionFields"][profileField.name];
                            }
                            else {
                                visitorProfileFields[profileField.name] = t["ovp." + profileField.name];
                            }
                        }
                    }
                }
                let visitFields = {};
                for (const visitFormField of visitorModuleSettings.visitFormFields) {
                    if (requestUserId && visitFormField.name == dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.visitedPerson)
                        continue;
                    if (visitState == dal_constants_1.DalConstants.UnterminatedVisitState.Expected) {
                        if (visitFormField.reportable) {
                            if (visitFormField.extension) {
                                visitFields[visitFormField.name] = t["oav.extensionFields"][visitFormField.name];
                            }
                            else {
                                if (visitFormField.name === dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.expectedDate) {
                                    visitFields[visitFormField.name] = {
                                        startDate: t.expectedStartUtc,
                                        endDate: t.expectedEndUtc,
                                    };
                                }
                                else {
                                    visitFields[visitFormField.name] = t["oav." + visitFormField.name];
                                }
                            }
                        }
                    }
                    else {
                        if (visitFormField.reportable) {
                            if (visitFormField.extension) {
                                visitFields[visitFormField.name] = t["oav.extensionFields"][visitFormField.name];
                            }
                            else {
                                if (visitFormField.name === dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.expectedDate) {
                                    visitFields[visitFormField.name] = {
                                        startDate: t.expectedStartUtc,
                                        endDate: t.expectedEndUtc,
                                    };
                                }
                                else {
                                    visitFields[visitFormField.name] = t["oav." + visitFormField.name];
                                }
                            }
                        }
                    }
                }
                let thumbnail = t["ovp.thumbnail"];
                thumbnail = thumbnail ? Buffer.from(thumbnail).toString() : null;
                return {
                    thumbnail: thumbnail,
                    visitorProfileId: t.organizationVisitorProfileId,
                    visitId: t.visitId,
                    visitStartUtc: t.startUtc,
                    visitorProfileFields: visitorProfileFields,
                    visitFields: visitFields,
                    state: t.state,
                };
            });
        });
        return Promise.resolve(result);
    }
    async searchVisitorProfileFormField(organizationId, visitorModuleSettings, options, hasOrganizationWide, requestUserId) {
        if (options.formField.type === dal_constants_1.DalConstants.FormFieldType.Select) {
            return Promise.resolve(this.searchFieldInSelectOptions(options));
        }
        else if (options.formField.type === dal_constants_1.DalConstants.FormFieldType.Text || options.formField.type === dal_constants_1.DalConstants.FormFieldType.Number) {
            let qb = this.dbClient
                .withSchema(organizationId)
                .from("organizationVisitorProfiles as ovp")
                .innerJoin("userOrganizations as uo", (join) => {
                join.on("ovp.userOrganizationId", "=", "uo.id");
            });
            qb.where("uo.organizationId", "=", organizationId).whereNull("uo.deletedAt").whereNull("ovp.deletedAt");
            if (!hasOrganizationWide) {
                let userPreviousVisitors = await dal_manager_1.dbManager.accessLog.listPreviousVisitorProfileIds(organizationId, requestUserId);
                let currentActiveVisitors = (await this.dbClient
                    .withSchema(organizationId)
                    .from("organizationActiveVisits as oav")
                    .innerJoin("userOrganizations as uo", "uo.id", "oav.visitedUserOrganizationId")
                    .where("uo.userId", requestUserId)
                    .where("uo.organizationId", organizationId)
                    .whereNull("uo.deletedAt")
                    .select("oav.organizationVisitorProfileId")).map((u) => u.organizationVisitorProfileId);
                let previousVisitors = userPreviousVisitors.concat(currentActiveVisitors);
                if (previousVisitors && previousVisitors.length > 0) {
                    qb.whereIn("ovp.id", previousVisitors);
                }
                else {
                    return Promise.resolve({
                        total: 0,
                        items: [],
                    });
                }
            }
            return this.searchDistinctFieldInDbTable(options, qb, "ovp.");
        }
        else {
            return Promise.resolve({
                total: 0,
                items: [],
            });
        }
    }
    async searchVisitFormField(organizationId, visitorModuleSettings, options) {
        if (options.formField.type === dal_constants_1.DalConstants.FormFieldType.Select) {
            return Promise.resolve(this.searchFieldInSelectOptions(options));
        }
        else if (options.formField.type === dal_constants_1.DalConstants.FormFieldType.Text || options.formField.type === dal_constants_1.DalConstants.FormFieldType.Number) {
            let qb = this.dbClient.withSchema(organizationId).from("organizationActiveVisits as oav").where("oav.organizationId", "=", organizationId);
            return this.searchDistinctFieldInDbTable(options, qb, "oav.");
        }
        else {
            return Promise.resolve({
                total: 0,
                items: [],
            });
        }
    }
    async searchDistinctFieldInDbTable(options, qb, tablePrefix) {
        let result = {
            total: 0,
            items: [],
        };
        let filter = options.filter;
        let rawColumn = tablePrefix;
        if (options.formField.extension) {
            rawColumn += `"extensionFields"->>'` + options.formField.name + "'";
            if (options.formField.type == dal_constants_1.DalConstants.FormFieldType.Text) {
                let textOptions = options.formField.options;
                if (textOptions && textOptions.searchType && textOptions.searchType == dal_constants_1.DalConstants.FormTypeaheadSearchFilterOptionType.StartsWith) {
                    qb.whereRaw("unaccent(" + rawColumn + ")::TEXT ilike unaccent(?)", filter.toUpperCase() + "%");
                }
                else {
                    qb.whereRaw("unaccent(" + rawColumn + ")::TEXT ilike unaccent(?)", "%" + filter.toUpperCase() + "%");
                }
            }
        }
        else {
            rawColumn += '"' + options.formField.name + '"';
            if (options.formField.type == dal_constants_1.DalConstants.FormFieldType.Text) {
                let textOptions = options.formField.options;
                if (textOptions && textOptions.searchType && textOptions.searchType == dal_constants_1.DalConstants.FormTypeaheadSearchFilterOptionType.StartsWith) {
                    qb.whereRaw("unaccent(" + rawColumn + ")::TEXT ilike unaccent(?)", filter.toUpperCase() + "%");
                }
                else {
                    qb.whereRaw("unaccent(" + rawColumn + ")::TEXT ilike unaccent(?)", "%" + filter.toUpperCase() + "%");
                }
            }
        }
        qb.groupByRaw(rawColumn);
        let qbCount = qb.clone();
        result.total = await qbCount.count("* as c").then((rows) => {
            return rows && rows.length > 0 ? rows[0].c : 0;
        });
        if (result.total > 0) {
            qb.orderBy("f")
                .select(this.dbClient.raw(rawColumn + " as f"))
                .offset(options.skip)
                .limit(options.take);
            result.items = await qb.then((rows) => {
                return rows.map((t) => {
                    return {
                        captionLines: [t.f],
                    };
                });
            });
        }
        return Promise.resolve(result);
    }
    searchFieldInSelectOptions(options) {
        let result = {
            total: 0,
            items: [],
        };
        let selectFieldOptions = options.formField.options;
        let list = selectFieldOptions.options;
        let resultItems = [];
        let regex = new RegExp(options.filter, "ig");
        for (const item of list) {
            for (const line of item.captionLines) {
                let searchResult = line.search(regex);
                if ((searchResult === 0 && selectFieldOptions.searchType === dal_constants_1.DalConstants.FormTypeaheadSearchFilterOptionType.StartsWith) ||
                    (searchResult >= 0 && selectFieldOptions.searchType === dal_constants_1.DalConstants.FormTypeaheadSearchFilterOptionType.Similar)) {
                    resultItems.push(item);
                }
            }
        }
        result.total = resultItems.length;
        result.items = resultItems.slice(options.skip, options.skip + options.take).map((t) => {
            return {
                captionLines: t.captionLines,
                value: t.value,
            };
        });
        return result;
    }
    async searchVisitorProfileForm(organizationId, visitorModuleSettings, options, hasOrganizationWide, requestUserId, trx) {
        let result = {
            total: 0,
            items: [],
        };
        let qb = trx
            .withSchema(organizationId)
            .from("organizationVisitorProfiles as ovp")
            .innerJoin("userOrganizations as uo", (join) => {
            join.on("ovp.userOrganizationId", "=", "uo.id");
        })
            .joinRaw(`left join (select
                distinct ovp.id,
                CASE WHEN oav.state is not null THEN oav.state
                    WHEN oav.state is null and ovs1.state is not null THEN ovs1.state
                    WHEN oav.state is null and ovs1.state is null and ovs2.state  is not null THEN ovs2.state
                    WHEN oav.state is null and ovs1.state is null and ovs2.state  is  null THEN null
                    END as state
               from "${organizationId}"."organizationVisitorProfiles" as ovp
               left join "${organizationId}"."organizationActiveVisits" as oav
               on oav."organizationVisitorProfileId" = ovp.id
               and oav.state = ?
               and oav."expectedStartUtc" < now()
               and oav."expectedEndUtc" > now()
               left join "${organizationId}"."organizationVisitorStates" ovs1
               on ovs1."organizationVisitorProfileId" = ovp.id
                   and ovs1."startUtc" < now()
                   and ovs1."endUtc" > now()
                   and ovs1.state = ?
               left join "${organizationId}"."organizationVisitorStates" ovs2
               on ovs2."organizationVisitorProfileId" = ovp.id
                   and ovs2."startUtc" < now()
                   and ovs2."endUtc" > now()
                   and ovs2.state = ?
               where
               oav.state is not null
               or ovs1.state is not null
               or ovs2.state is not null) S on S.id = ovp.id`, [dal_constants_1.DalConstants.OrganizationVisitorStates.Expected, dal_constants_1.DalConstants.OrganizationVisitorStates.Forbidden, dal_constants_1.DalConstants.OrganizationVisitorStates.TemporaryPermit]);
        qb.where("uo.organizationId", "=", organizationId).whereNull("uo.deletedAt").whereNull("ovp.deletedAt");
        if (!hasOrganizationWide) {
            let userPreviousVisitors = await dal_manager_1.dbManager.accessLog.listPreviousVisitorProfileIds(organizationId, requestUserId);
            let currentActiveVisitors = (await trx
                .withSchema(organizationId)
                .from("organizationActiveVisits as oav")
                .innerJoin("userOrganizations as uo", "uo.id", "oav.visitedUserOrganizationId")
                .where("uo.userId", requestUserId)
                .where("uo.organizationId", organizationId)
                .whereNull("uo.deletedAt")
                .select("oav.organizationVisitorProfileId")).map((u) => u.organizationVisitorProfileId);
            let previousVisitors = userPreviousVisitors.concat(currentActiveVisitors);
            if (previousVisitors && previousVisitors.length > 0) {
                qb.whereIn("ovp.id", previousVisitors);
            }
            else {
                return Promise.resolve(result);
            }
        }
        let i = 0;
        let orders = [];
        let columns = ["ovp.id"];
        let captionLineRowNumbers = {};
        let captionMaxCaptionLineRowNumber = 0;
        let captionLineFields = [];
        for (const visitorProfileField of visitorModuleSettings.visitorProfileFormFields) {
            if (visitorProfileField.showInCaptionLines) {
                captionMaxCaptionLineRowNumber = Math.max(visitorProfileField.captionLineRowOrder, captionMaxCaptionLineRowNumber);
                captionLineRowNumbers[visitorProfileField.captionLineRowOrder + ""] = true;
                let captionLineField = {
                    name: visitorProfileField.name,
                    rowOrder: visitorProfileField.captionLineRowOrder,
                    colOrder: visitorProfileField.captionLineColumnOrder,
                    isExtension: visitorProfileField.extension,
                    colName: null,
                    isUnique: visitorProfileField.unique,
                };
                if (visitorProfileField.extension) {
                    captionLineField.colName = "c" + i++;
                    captionLineFields.push(captionLineField);
                    let fieldName = `ovp."extensionFields"->>'` + visitorProfileField.name + "'";
                    columns.push(trx.raw(fieldName + " as " + captionLineField.colName));
                }
                else {
                    captionLineField.colName = visitorProfileField.name;
                    switch (visitorProfileField.name) {
                        case dal_access_models_1.DbAccessModel.PredefinedVisitorProfileFormFields.name:
                            captionLineFields.push(captionLineField);
                            columns.push("ovp.name");
                            break;
                        case dal_access_models_1.DbAccessModel.PredefinedVisitorProfileFormFields.surname:
                            captionLineFields.push(captionLineField);
                            columns.push("ovp.surname");
                            break;
                        case dal_access_models_1.DbAccessModel.PredefinedVisitorProfileFormFields.uniqueId:
                            captionLineFields.push(captionLineField);
                            columns.push("ovp.uniqueId");
                            break;
                        default:
                            break;
                    }
                }
            }
        }
        let addedSearchFields = [];
        qb.andWhere((where) => {
            for (const filterField of options.filterFields) {
                let fieldName = "ovp.";
                let c = "c" + i++;
                if (filterField.formField.extension) {
                    fieldName += `"extensionFields"->>'` + filterField.name + "'";
                    where.orWhereRaw("unaccent(" + fieldName + ")::TEXT ilike unaccent(?)", "%" + filterField.filter + "%");
                }
                else {
                    fieldName += '"' + filterField.name + '"';
                    where.orWhereRaw("unaccent(" + fieldName + ")::TEXT ilike unaccent(?)", "%" + filterField.filter + "%");
                }
                if (!addedSearchFields.map((m) => m.name).includes(filterField.name)) {
                    columns.push(trx.raw(fieldName + " as " + c));
                    addedSearchFields.push({ name: filterField.name, colName: c });
                    orders.push(c);
                }
            }
        });
        let u = 0;
        let anyUniqueFieldMissing = false;
        let uniqueVisitorFormFields = visitorModuleSettings.visitorProfileFormFields.filter((t) => t.unique);
        for (const filterField of uniqueVisitorFormFields) {
            let fieldName = "ovp.";
            if (filterField.extension) {
                fieldName += `"extensionFields"->>'` + filterField.name + "'";
            }
            else {
                fieldName += '"' + filterField.name + '"';
            }
            let valueExist = options.filterFields.find((t) => t.name == filterField.name);
            if (valueExist) {
                columns.push(trx.raw(`CASE WHEN ? ilike ? THEN true ELSE FALSE END as unique` + u, [trx.raw(fieldName), valueExist.filter]));
                orders.push("unique" + u++ + " DESC");
            }
            else {
                if (filterField.required) {
                    anyUniqueFieldMissing = true;
                }
                else {
                    columns.push(trx.raw(`CASE WHEN ? is null or ? = '' THEN true ELSE FALSE END as unique` + u, [trx.raw(fieldName), trx.raw(fieldName)]));
                    orders.push("unique" + u++ + " DESC");
                }
            }
        }
        let qbCount = qb.clone();
        result.total = await qbCount.count("ovp.id as c").then((rows) => {
            return parseInt(rows[0].c);
        });
        if (result.total === 0) {
            return Promise.resolve(result);
        }
        orders.unshift(" S.state ASC ");
        qb.orderByRaw(orders.join(",")).column(columns).select().offset(options.skip).limit(options.take);
        let rows = await qb;
        for (const t of rows) {
            let captionLines = [];
            let isUnique = true;
            for (let rowNum = 1; rowNum <= captionMaxCaptionLineRowNumber; rowNum++) {
                if (captionLineRowNumbers[rowNum + ""]) {
                    let itemsOfCaptionLine = [];
                    let rowItems = captionLineFields
                        .filter((cl) => cl.rowOrder === rowNum)
                        .sort((n1, n2) => {
                        if (n1.colOrder > n2.colOrder) {
                            return 1;
                        }
                        if (n1.colOrder < n2.colOrder) {
                            return -1;
                        }
                        return 0;
                    });
                    for (let colIndex = 0; colIndex < rowItems.length; colIndex++) {
                        let val = t[rowItems[colIndex].colName];
                        if (val) {
                            itemsOfCaptionLine.push(val);
                        }
                    }
                    let line = itemsOfCaptionLine.join(" ").trim();
                    if (line && line.length > 0) {
                        captionLines.push(line);
                    }
                }
            }
            for (let uniqueIndex = 0; uniqueIndex < u; uniqueIndex++) {
                isUnique = isUnique && t["unique" + uniqueIndex.toString()];
            }
            let item = {
                id: t["id"],
                captionLines: captionLines,
                isUniqueMatch: isUnique && !anyUniqueFieldMissing,
                suggestionType: restapi_1.SuggestionType.Internal,
                states: [],
                searchFields: addedSearchFields.map((tt) => {
                    return {
                        name: tt.name,
                        value: t[tt.colName],
                    };
                }),
            };
            result.items.push(item);
        }
        let visitorProfileIds = result.items.map((r) => r.id);
        let visitorStates = await this.listVisitorStates(organizationId, visitorProfileIds, visitorModuleSettings);
        for (const visitorState of visitorStates) {
            let item = result.items.find((r) => r.id === visitorState.visitorProfileId);
            if (item) {
                item.states = visitorState.states;
            }
        }
        return Promise.resolve(result);
    }
    prepareVisitorProfileColumns(visitorModuleSettings) {
        let result = [this.dbClient.raw(`"oav"."id" as "oav.visitId"`), this.dbClient.raw(`"oav"."state" as "oav.state"`), this.dbClient.raw(`"ovp"."thumbnail" as "ovp.thumbnail"`)];
        for (const visitorProfileField of visitorModuleSettings.visitorProfileFormFields) {
            if (visitorProfileField.extension) {
                result.push(this.dbClient.raw(`"ovp"."extensionFields"->>'` + visitorProfileField.name + `' as "ovp.` + visitorProfileField.name + `"`));
            }
            else {
                result.push(this.dbClient.raw(`"ovp"."` + visitorProfileField.name + `" as "ovp.` + visitorProfileField.name + `"`));
            }
        }
        return result;
    }
    prepareVisitJoinsForVisitorDetailed(organizationId, visitorModuleSettings, qb) {
        let columns = [];
        for (const visitField of visitorModuleSettings.visitFormFields) {
            switch (visitField.name) {
                case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.accessControlPoints:
                    columns.push(this.dbClient.raw(`
                        (select to_json(array_agg(json_build_object(
                            'id', acp."id", 'name', acp."name"
                            ))) from "${organizationId}"."accessControlPoints" as acp
                        inner join "${organizationId}"."userAccessRights" as uar on acp."id" = uar."accessControlPointId"
                        inner join "${organizationId}"."userOrganizations" as uo on uar."userId" = uo."userId"
                        where uar."deletedAt" is null and acp."deletedAt" is null
                        and uo."deletedAt" is null
                        and uo."id" = ovp."userOrganizationId"
                        and uar.access = true
                        ) "oav.accessControlPoints"
                    `));
                    break;
                case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.escort:
                    qb.leftOuterJoin("userOrganizationProfiles as euop", (join) => {
                        join.on("euop.userOrganizationId", "=", "oav.escortUserOrganizationId").andOnNotNull("oav.escortUserOrganizationId");
                    });
                    columns.push(this.dbClient.raw(`
                         json_build_object(
                         'userId', euop."userId",
                         'uniqueId', euop."uniqueId", 'name', euop."name",
                         'surname', euop."surname"
                         ) as "oav.escortPerson"
                     `));
                    break;
                case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.visitedOrganizationUnit:
                    qb.leftOuterJoin("organizationUnits as vou", (join) => {
                        join.on("vou.id", "=", "oav.visitedOrganizationUnitId").andOnNotNull("oav.visitedOrganizationUnitId");
                    });
                    columns.push(this.dbClient.raw(`
                        json_build_object(
                         'id', vou."id",
                         'name', vou."name") as "oav.visitedOrganizationUnit"
                     `));
                    break;
                case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.visitedPerson:
                    qb.leftOuterJoin("userOrganizationProfiles as vuop", (join) => {
                        join.on("vuop.userOrganizationId", "=", "oav.visitedUserOrganizationId").andOnNotNull("oav.visitedUserOrganizationId");
                    });
                    columns.push(this.dbClient.raw(`
                             json_build_object(
                             'userId', vuop."userId",
                             'uniqueId', vuop."uniqueId", 'name', vuop."name",
                             'surname', vuop."surname"
                             ) as "oav.visitedPerson"
                         `));
                    break;
                case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.credentials:
                    columns.push(this.dbClient.raw(`
                        (SELECT to_json(array_agg(json_build_object(
                            'id', uoc."id",
                            'type', uoc."type",
                            'data', uoc."data",
                            'expirationUtc', uoc."expiresOn",
                            'credentialNumber', uoc."credentialNumber"
                            )))
                            FROM "${organizationId}"."userOrganizationCredentials" as uoc
                            INNER JOIN "${organizationId}"."userOrganizations" as uo ON uoc."userId" = uo."userId"
                            WHERE uo."organizationId" = ?
                            AND uo."id" = ovp."userOrganizationId"
                            AND uoc."deletedAt" is null
                            AND uo."deletedAt" is null)
                            "oav.credentials"
                    `, [organizationId]));
                default:
                    break;
            }
        }
        return {
            qb: qb,
            columns: columns,
        };
    }
    async findActiveVisitorByCredentialData(organizationId, data) {
        let qb = this.dbClient
            .withSchema(organizationId)
            .from("userOrganizationCredentials as uoc")
            .column({ visitorProfileId: "ovp.id" })
            .innerJoin("userOrganizations as uo", (join) => {
            join.on("uoc.userId", "=", "uo.userId");
        })
            .innerJoin("organizationVisitorProfiles as ovp", (join) => {
            join.on("ovp.userOrganizationId", "=", "uo.id");
        })
            .where("uoc.organizationId", "=", organizationId)
            .where("uo.organizationId", "=", organizationId)
            .whereNull("uoc.deletedAt")
            .whereNull("ovp.deletedAt")
            .whereNull("uo.deletedAt")
            .where("uoc.data", data);
        let visitorProfileId = await qb.then((rows) => {
            return rows.length > 0 ? rows[0].visitorProfileId : null;
        });
        if (!visitorProfileId) {
            return Promise.resolve(null);
        }
        return this.getVisitorDetailed(organizationId, visitorProfileId, await this.getVisitAndVisitorFormSettings(organizationId));
    }
    async getVisitorDetailed(organizationId, visitorProfileId, organizationVisitorModuleSettings) {
        let result = {
            profile: null,
            lastVisit: null,
        };
        let qb = this.dbClient
            .withSchema(organizationId)
            .from("organizationVisitorProfiles as ovp")
            .innerJoin("userOrganizations as uo", (join) => {
            join.on("ovp.userOrganizationId", "=", "uo.id");
        })
            .leftOuterJoin("organizationActiveVisits as oav", (join) => {
            join.on("ovp.id", "=", "oav.organizationVisitorProfileId");
        });
        qb.where("uo.organizationId", "=", organizationId).where("ovp.id", "=", visitorProfileId).whereNull("uo.deletedAt").whereNull("ovp.deletedAt");
        qb.select(this.prepareVisitorProfileColumns(organizationVisitorModuleSettings));
        let activeVisitId = null;
        result.profile = await qb.then(async (rows) => {
            if (!rows || rows.length === 0) {
                (0, dal_access_error_1.throwDbAccessNotFoundErrorTr)("ERRORS.VISITOR.VISITORNOTFOUND");
            }
            let visitor = rows[0];
            let fields = this.fillVisitorFormFieldFromDataRow(visitor, organizationVisitorModuleSettings);
            for (const row of rows) {
                if (row["oav.state"] === dal_constants_1.DalConstants.UnterminatedVisitState.Active)
                    activeVisitId = row["oav.visitId"];
            }
            let thumbnail = visitor["ovp.thumbnail"];
            let states = await this.getVisitorState(organizationId, visitorProfileId, organizationVisitorModuleSettings);
            return {
                id: visitorProfileId,
                fields: fields,
                states: states,
                thumbnail: thumbnail ? Buffer.from(thumbnail).toString() : null,
            };
        });
        if (!activeVisitId) {
            let lastTerminatedVisit = await dal_manager_1.dbManager.accessLog.getLastTerminatedVisit(organizationId, visitorProfileId, organizationVisitorModuleSettings);
            if (lastTerminatedVisit) {
                result.lastVisit = {
                    id: lastTerminatedVisit.id,
                    startUtc: lastTerminatedVisit.su,
                    endUtc: lastTerminatedVisit.eu,
                    fields: lastTerminatedVisit.v,
                };
                if (result.lastVisit.fields && result.lastVisit.fields.visitedPerson) {
                    let visitedUserId = result.lastVisit.fields.visitedPerson.userId;
                    if (visitedUserId) {
                        let visitedUser = await this.dbClient
                            .withSchema(organizationId)
                            .from("userOrganizations as uo")
                            .innerJoin("users as u", "u.id", "uo.userId")
                            .innerJoin("userOrganizationProfiles as uop", "uo.id", "uop.userOrganizationId")
                            .where("u.id", visitedUserId)
                            .where("organizationId", organizationId)
                            .whereNull("uo.deletedAt")
                            .whereNull("u.deletedAt")
                            .whereNull("uop.deletedAt")
                            .where("uo.isDisabled", false)
                            .first("u.id");
                        if (!visitedUser) {
                            result.lastVisit.fields.visitedPerson = null;
                        }
                    }
                }
            }
        }
        else {
            let columns = [
                { "oav.extensionFields": "oav.extensionFields" },
                { "oav.startUtc": "oav.startUtc" },
                { "oav.expectedStartUtc": "oav.expectedStartUtc" },
                { "oav.expectedEndUtc": "oav.expectedEndUtc" },
            ];
            qb = this.dbClient
                .withSchema(organizationId)
                .from("organizationActiveVisits as oav")
                .innerJoin("organizationVisitorProfiles as ovp", (join) => {
                join.on("ovp.id", "oav.organizationVisitorProfileId");
            })
                .where("oav.id", activeVisitId)
                .whereNull("ovp.deletedAt");
            let visitJoins = this.prepareVisitJoinsForVisitorDetailed(organizationId, organizationVisitorModuleSettings, qb);
            qb = visitJoins.qb;
            columns = columns.concat(visitJoins.columns);
            qb.column(columns).select();
            let lastVisit = {
                id: activeVisitId,
                startUtc: null,
                endUtc: undefined,
                fields: {},
            };
            await qb.then((rows) => {
                if (!rows || rows.length === 0) {
                    (0, dal_access_error_1.throwDbAccessNotFoundErrorTr)("ERRORS.VISITOR.VISITNOTFOUND");
                }
                let visit = rows[0];
                lastVisit.startUtc = visit["oav.startUtc"];
                lastVisit.fields = this.fillVisitFormFieldsFromDataRow(visit, organizationVisitorModuleSettings);
            });
            result.lastVisit = lastVisit;
        }
        return Promise.resolve(result);
    }
    fillVisitFormFieldsFromDataRow(visit, organizationVisitorModuleSettings) {
        let visitFields = {};
        for (const visitFormField of organizationVisitorModuleSettings.visitFormFields) {
            if (visitFormField.extension) {
                if (!visit["oav.extensionFields"][visitFormField.name]) {
                    switch (visitFormField.type) {
                        case dal_constants_1.DalConstants.FormFieldType.Checkbox:
                            let options = visitFormField.options;
                            visitFields[visitFormField.name] = options.defaultValue ? options.defaultValue : false;
                            break;
                        default:
                            visitFields[visitFormField.name] = null;
                    }
                }
                else {
                    visitFields[visitFormField.name] = visit["oav.extensionFields"][visitFormField.name];
                }
            }
            else {
                if (visitFormField.name === dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.expectedDate) {
                    visitFields[visitFormField.name] = {
                        startDate: visit["oav.expectedStartUtc"],
                        endDate: visit["oav.expectedEndUtc"],
                    };
                }
                else {
                    visitFields[visitFormField.name] = visit["oav." + visitFormField.name];
                }
            }
        }
        return visitFields;
    }
    fillVisitorFormFieldFromDataRow(visitor, organizationVisitorModuleSettings) {
        let fields = {};
        for (const visitorProfileField of organizationVisitorModuleSettings.visitorProfileFormFields) {
            if (!visitor["ovp." + visitorProfileField.name]) {
                switch (visitorProfileField.type) {
                    case dal_constants_1.DalConstants.FormFieldType.Checkbox:
                        fields[visitorProfileField.name] = false;
                        break;
                    default:
                        fields[visitorProfileField.name] = null;
                }
            }
            else
                fields[visitorProfileField.name] = visitor["ovp." + visitorProfileField.name];
        }
        return fields;
    }
    async getVisitorDetailedWithExpectedVisits(organizationId, visitorProfileId, hasVisitorWrite, requestUserId, organizationVisitorModuleSettings) {
        let result = {
            profile: null,
            lastVisit: null,
            expectedVisits: [],
        };
        let visitorDetailed = await this.getVisitorDetailed(organizationId, visitorProfileId, organizationVisitorModuleSettings);
        result.profile = visitorDetailed.profile;
        result.lastVisit = visitorDetailed.lastVisit;
        result.expectedVisits = await this.getProfileExpectedVisits(organizationId, visitorProfileId, hasVisitorWrite, requestUserId, organizationVisitorModuleSettings);
        return Promise.resolve(result);
    }
    async getProfileExpectedVisitsCountWithDateRange(organizationId, visitorProfileId, expectedStartDate, expectedEndDate, isMaxStartDynamicDateDefined, visitId, trx) {
        let qb = this.dbClient
            .withSchema(organizationId)
            .from("organizationActiveVisits as oav")
            .innerJoin("organizationVisitorProfiles as ovp", (join) => {
            join.on("ovp.id", "oav.organizationVisitorProfileId");
        })
            .where("oav.state", dal_constants_1.DalConstants.UnterminatedVisitState.Expected)
            .whereNull("oav.startUtc")
            .where("ovp.id", visitorProfileId);
        if (visitId) {
            qb.where("oav.id", "!=", visitId);
        }
        if (!isMaxStartDynamicDateDefined) {
            if (expectedStartDate == null || expectedEndDate == null) {
                qb.whereNull("oav.expectedStartUtc").whereNull("oav.expectedEndUtc");
            }
            else {
                qb.where("oav.expectedStartUtc", "<=", expectedEndDate).where("oav.expectedEndUtc", ">=", expectedStartDate);
            }
        }
        else {
            qb.where("oav.expectedStartUtc", "=", expectedStartDate);
        }
        let count;
        if (trx) {
            count = await qb
                .count("oav.id as c")
                .transacting(trx)
                .then((rows) => {
                return parseInt(rows[0].c);
            });
        }
        else {
            count = await qb.count("oav.id as c").then((rows) => {
                return parseInt(rows[0].c);
            });
        }
        return Promise.resolve(count);
    }
    async getProfileExpectedVisits(organizationId, visitorProfileId, hasVisitorWrite, requestUserId, organizationVisitorModuleSettings) {
        let result = [];
        let qb = this.dbClient
            .withSchema(organizationId)
            .from("organizationActiveVisits as oav")
            .innerJoin("organizationVisitorProfiles as ovp", (join) => {
            join.on("ovp.id", "oav.organizationVisitorProfileId");
        })
            .where("oav.state", dal_constants_1.DalConstants.UnterminatedVisitState.Expected)
            .whereNull("oav.startUtc")
            .where("ovp.id", visitorProfileId)
            .whereNull("ovp.deletedAt")
            .leftJoin("userOrganizationProfiles as vuop", (join) => {
            join.on("vuop.userOrganizationId", "=", "oav.visitedUserOrganizationId").andOnNotNull("oav.visitedUserOrganizationId");
        });
        if (organizationVisitorModuleSettings.maxExpectedDayCount && organizationVisitorModuleSettings.maxExpectedDayCount > 0) {
            let now = new Date();
            qb.where("oav.expectedStartUtc", "<=", now).where("oav.expectedEndUtc", ">=", now);
        }
        if (!hasVisitorWrite) {
            qb.innerJoin("userOrganizations as uo", "uo.id", "oav.visitedUserOrganizationId")
                .where("uo.userId", requestUserId)
                .where("uo.organizationId", organizationId)
                .whereNull("uo.deletedAt");
        }
        let visits = await qb.select("oav.id", "oav.expectedStartUtc", "oav.expectedEndUtc", "vuop.userId as visitedUserId", "vuop.uniqueId", "vuop.surname", "vuop.name");
        if (visits) {
            result = visits.map((v) => {
                return {
                    id: v.id,
                    expectedStartUtc: v.expectedStartUtc,
                    expectedEndUtc: v.expectedEndUtc,
                    visitedUserProfile: v.visitedUserId
                        ? {
                            id: v.visitedUserId,
                            fullname: (v.name ?? "") + " " + (v.surname ?? ""),
                            uniqueId: v.uniqueId,
                            captionLines: [],
                        }
                        : null,
                };
            });
        }
        return Promise.resolve(result);
    }
    async getVisitorDetailedByVisitId(organizationId, visitId, organizationVisitorModuleSettings) {
        let result = {
            profile: null,
            lastVisit: null,
        };
        let qb = this.dbClient
            .withSchema(organizationId)
            .from("organizationVisitorProfiles as ovp")
            .innerJoin("userOrganizations as uo", (join) => {
            join.on("ovp.userOrganizationId", "uo.id");
        })
            .innerJoin("organizationActiveVisits as oav", (join) => {
            join.on("ovp.id", "oav.organizationVisitorProfileId");
        });
        qb.where("uo.organizationId", organizationId).where("oav.id", visitId).whereNull("uo.deletedAt").whereNull("ovp.deletedAt");
        let columns = this.prepareVisitorProfileColumns(organizationVisitorModuleSettings);
        columns.push("ovp.id as visitorProfileId");
        columns.push("oav.state as visitState");
        qb.select(columns);
        let visitorProfileId = null;
        result.profile = await qb.then(async (rows) => {
            if (!rows || rows.length === 0) {
                (0, dal_access_error_1.throwDbAccessNotFoundErrorTr)("ERRORS.VISITOR.VISITORNOTFOUND");
            }
            let visitor = rows[0];
            let fields = this.fillVisitorFormFieldFromDataRow(visitor, organizationVisitorModuleSettings);
            visitorProfileId = visitor["visitorProfileId"];
            let states = await this.getVisitorState(organizationId, visitorProfileId, organizationVisitorModuleSettings);
            return {
                id: visitor["visitorProfileId"],
                fields: fields,
                states: states,
            };
        });
        if (visitId) {
            let columns = [
                { "oav.extensionFields": "oav.extensionFields" },
                { "oav.expectedStartUtc": "oav.expectedStartUtc" },
                { "oav.expectedEndUtc": "oav.expectedEndUtc" },
                { "oav.startUtc": "oav.startUtc" },
            ];
            qb = this.dbClient
                .withSchema(organizationId)
                .from("organizationActiveVisits as oav")
                .innerJoin("organizationVisitorProfiles as ovp", (join) => {
                join.on("ovp.id", "=", "oav.organizationVisitorProfileId");
            })
                .where("oav.id", "=", visitId);
            let visitJoins = this.prepareVisitJoinsForVisitorDetailed(organizationId, organizationVisitorModuleSettings, qb);
            qb = visitJoins.qb;
            columns = columns.concat(visitJoins.columns);
            qb.column(columns).select();
            let lastVisit = {
                id: visitId,
                startUtc: null,
                endUtc: undefined,
                fields: {},
            };
            await qb.then((rows) => {
                if (!rows || rows.length === 0) {
                    (0, dal_access_error_1.throwDbAccessNotFoundErrorTr)("ERRORS.VISITOR.VISITNOTFOUND");
                }
                let visit = rows[0];
                lastVisit.startUtc = visit["oav.startUtc"];
                lastVisit.fields = this.fillVisitFormFieldsFromDataRow(visit, organizationVisitorModuleSettings);
            });
            result.lastVisit = lastVisit;
        }
        return Promise.resolve(result);
    }
    async getActiveVisitDetailed(organizationId, visitId, organizationVisitorModuleSettings, trx) {
        let result = {
            id: visitId,
            userId: null,
            startUtc: null,
            endUtc: undefined,
            fields: {},
            profile: null,
            visitorRegistrationPointId: null,
            processTime: 0,
        };
        let columns = [
            { "oav.extensionFields": "oav.extensionFields" },
            { "oav.startUtc": "oav.startUtc" },
            { "oav.expectedStartUtc": "oav.expectedStartUtc" },
            { "oav.expectedEndUtc": "oav.expectedEndUtc" },
            { "ovp.id": "ovp.id" },
            { "oav.visitorRegistrationPointId": "oav.visitorRegistrationPointId" },
            { "oav.processTime": "oav.processTime" },
            { "uo.userId": "uo.userId" },
        ];
        columns = columns.concat(this.prepareVisitorProfileColumns(organizationVisitorModuleSettings));
        let qb = this.dbClient
            .withSchema(organizationId)
            .from("organizationActiveVisits as oav")
            .innerJoin("organizationVisitorProfiles as ovp", (join) => {
            join.on("ovp.id", "=", "oav.organizationVisitorProfileId");
        })
            .innerJoin("userOrganizations as uo", (join) => {
            join.on("ovp.userOrganizationId", "=", "uo.id");
        });
        qb.where("oav.organizationId", "=", organizationId).where("oav.id", "=", visitId);
        let visitJoins = this.prepareVisitJoinsForVisitorDetailed(organizationId, organizationVisitorModuleSettings, qb);
        qb = visitJoins.qb;
        columns = columns.concat(visitJoins.columns);
        qb.column(columns).select().limit(1);
        if (trx) {
            qb.transacting(trx);
        }
        await qb.then(async (rows) => {
            if (!rows || rows.length === 0) {
                (0, dal_access_error_1.throwDbAccessNotFoundErrorTr)("ERRORS.VISITOR.VISITNOTFOUND");
            }
            let visit = rows[0];
            let profileFields = {};
            for (const visitorProfileField of organizationVisitorModuleSettings.visitorProfileFormFields) {
                profileFields[visitorProfileField.name] = visit["ovp." + visitorProfileField.name] ? visit["ovp." + visitorProfileField.name] : null;
            }
            let states = await this.getVisitorState(organizationId, visit["ovp.id"], organizationVisitorModuleSettings, trx);
            result.userId = visit["uo.userId"];
            result.profile = {
                id: visit["ovp.id"],
                fields: profileFields,
                states: states,
            };
            result.startUtc = visit["oav.startUtc"];
            result.visitorRegistrationPointId = visit["oav.visitorRegistrationPointId"];
            result.processTime = visit["oav.processTime"];
            let visitFields = {};
            for (const visitFormField of organizationVisitorModuleSettings.visitFormFields) {
                if (visitFormField.extension) {
                    visitFields[visitFormField.name] = visit["oav.extensionFields"][visitFormField.name] ? visit["oav.extensionFields"][visitFormField.name] : null;
                }
                else {
                    if (visitFormField.name === dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.expectedDate) {
                        visitFields[visitFormField.name] = {
                            startDate: visit["oav.expectedStartUtc"],
                            endDate: visit["oav.expectedEndUtc"],
                        };
                    }
                    else {
                        visitFields[visitFormField.name] = visit["oav." + visitFormField.name] ? visit["oav." + visitFormField.name] : null;
                    }
                }
            }
            result.fields = visitFields;
        });
        return Promise.resolve(result);
    }
    async getExpectedVisitDetailed(organizationId, visitId, organizationVisitorModuleSettings, trx) {
        let result = {
            id: visitId,
            userId: null,
            startUtc: null,
            expectedStartUtc: null,
            expectedEndUtc: null,
            fields: {},
            profile: null,
            visitorRegistrationPointId: null,
            requestUserId: null,
        };
        let columns = [
            { "oav.extensionFields": "oav.extensionFields" },
            { "oav.startUtc": "oav.startUtc" },
            { "oav.state": "oav.state" },
            { "oav.expectedStartUtc": "oav.expectedStartUtc" },
            { "oav.expectedEndUtc": "oav.expectedEndUtc" },
            { "ovp.id": "ovp.id" },
            { "oav.visitorRegistrationPointId": "oav.visitorRegistrationPointId" },
            { "uo.userId": "uo.userId" },
        ];
        columns = columns.concat(this.prepareVisitorProfileColumns(organizationVisitorModuleSettings));
        let qb = this.dbClient
            .withSchema(organizationId)
            .from("organizationActiveVisits as oav")
            .innerJoin("organizationVisitorProfiles as ovp", (join) => {
            join.on("ovp.id", "=", "oav.organizationVisitorProfileId");
        })
            .innerJoin("userOrganizations as uo", (join) => {
            join.on("ovp.userOrganizationId", "=", "uo.id");
        });
        qb.where("oav.organizationId", "=", organizationId).where("oav.id", "=", visitId);
        qb.where("oav.state", dal_constants_1.DalConstants.UnterminatedVisitState.Expected);
        let visitJoins = this.prepareVisitJoinsForVisitorDetailed(organizationId, organizationVisitorModuleSettings, qb);
        qb.transacting(trx);
        qb = visitJoins.qb;
        columns = columns.concat(visitJoins.columns);
        qb.column(columns).select().limit(1);
        await qb.then(async (rows) => {
            if (!rows || rows.length === 0) {
                (0, dal_access_error_1.throwDbAccessNotFoundErrorTr)("ERRORS.VISITOR.VISITNOTFOUND");
            }
            let visit = rows[0];
            let profileFields = {};
            for (const visitorProfileField of organizationVisitorModuleSettings.visitorProfileFormFields) {
                profileFields[visitorProfileField.name] = visit["ovp." + visitorProfileField.name];
            }
            result.userId = visit["uo.userId"];
            result.profile = {
                id: visit["ovp.id"],
                fields: profileFields,
                states: await this.getVisitorState(organizationId, visit["ovp.id"], organizationVisitorModuleSettings),
            };
            result.expectedStartUtc = visit["oav.expectedStartUtc"];
            result.expectedEndUtc = visit["oav.expectedEndUtc"];
            result.visitorRegistrationPointId = visit["oav.visitorRegistrationPointId"];
            let visitFields = {};
            for (const visitFormField of organizationVisitorModuleSettings.visitFormFields) {
                if (visitFormField.extension) {
                    visitFields[visitFormField.name] = visit["oav.extensionFields"][visitFormField.name] ? visit["oav.extensionFields"][visitFormField.name] : null;
                }
                else {
                    if (visitFormField.name === dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.expectedDate) {
                        visitFields[visitFormField.name] = {
                            startDate: visit["oav.expectedStartUtc"],
                            endDate: visit["oav.expectedEndUtc"],
                        };
                    }
                    else {
                        visitFields[visitFormField.name] = visit["oav." + visitFormField.name] ? visit["oav." + visitFormField.name] : null;
                    }
                }
            }
            result.fields = visitFields;
        });
        return Promise.resolve(result);
    }
    async terminateVisit(organizationId, visitId, organizationVisitorModuleSettings, requestUserId) {
        let visitDetailed = await this.getActiveVisitDetailed(organizationId, visitId, organizationVisitorModuleSettings);
        const visitorUserId = await this.terminateVisitNew(organizationId, visitId);
        await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
            await dal_manager_1.dbManager.accessRegion.unlockUserStateForAllRegions(organizationId, visitorUserId, trx);
        }, requestUserId, organizationId);
        await dal_manager_1.dbManager.accessLog.insertActiveToNoSql(organizationId, visitDetailed, organizationVisitorModuleSettings);
        await this.updateDailyVisitSummary(organizationId, visitDetailed);
        await dal_manager_1.dbManager.accessLog.addUserActionHistoryItem({
            oId: organizationId,
            o: requestUserId,
            u: new Date(),
            c: dal_constants_1.DalConstants.UserActionCategory.Visitor,
            tg: ["Visitor", "TerminateVisit"],
            t: dal_constants_1.DalConstants.UserVisitorActionType.TerminateVisit,
            d: {
                visitId: visitId,
            },
        });
    }
    async listVisitorProfilesWithState(organizationId, visitorModuleSettings, take, skip, visitorProfileIds, state) {
        let result = {
            total: 0,
            items: [],
        };
        let qb = this.dbClient
            .withSchema(organizationId)
            .from("organizationVisitorProfiles as ovp")
            .innerJoin("userOrganizations as uo", (join) => {
            join.on("ovp.userOrganizationId", "=", "uo.id");
        });
        qb.where("uo.organizationId", "=", organizationId).whereNull("uo.deletedAt").whereNull("ovp.deletedAt").whereIn("ovp.id", visitorProfileIds);
        if (state == dal_constants_1.DalConstants.UnterminatedVisitState.Active || state == dal_constants_1.DalConstants.UnterminatedVisitState.Expected) {
            qb.innerJoin("organizationActiveVisits as oav", (join) => {
                join.on("ovp.id", "oav.organizationVisitorProfileId");
            }).where("oav.state", state);
        }
        let columns = ["ovp.id"];
        let name = false;
        let surname = false;
        let uniqueId = false;
        for (const visitorProfileField of visitorModuleSettings.visitorProfileFormFields) {
            let fieldName = "ovp.";
            let columnName = fieldName;
            if (visitorProfileField.reportable) {
                if (visitorProfileField.extension) {
                    fieldName += `"extensionFields"->>'` + visitorProfileField.name + "'";
                    columnName += "extension." + visitorProfileField.name;
                }
                else {
                    fieldName += '"' + visitorProfileField.name + '"';
                    columnName += visitorProfileField.name;
                    switch (visitorProfileField.name) {
                        case dal_access_models_1.DbAccessModel.PredefinedVisitorProfileFormFields.name:
                            name = true;
                            break;
                        case dal_access_models_1.DbAccessModel.PredefinedVisitorProfileFormFields.surname:
                            surname = true;
                            break;
                        case dal_access_models_1.DbAccessModel.PredefinedVisitorProfileFormFields.uniqueId:
                            uniqueId = true;
                            break;
                        default:
                            break;
                    }
                }
                columns.push(this.dbClient.raw(fieldName + ' as "' + columnName + '"'));
            }
        }
        let qbCount = qb.clone();
        result.total = await qbCount.count("ovp.id as c").then((rows) => {
            return parseInt(rows[0].c);
        });
        if (result.total === 0) {
            return Promise.resolve(result);
        }
        qb.orderByRaw("ovp.name").column(columns).select().offset(skip).limit(take);
        result.items = await qb.then((rows) => {
            return rows.map((t) => {
                let captionLines = [];
                if (name && surname) {
                    captionLines.push([t["ovp.name"], t["ovp.surname"]].join(" "));
                }
                else if (name) {
                    captionLines.push(t["ovp.name"]);
                }
                else if (surname) {
                    captionLines.push(t["ovp.surname"]);
                }
                return {
                    id: t["id"],
                    captionLines: captionLines,
                    matchItem: null,
                    uniqueId: uniqueId ? t["ovp.uniqueId"] : "",
                };
            });
        });
        return Promise.resolve(result);
    }
    async genericSearchVisitorProfile(organizationId, visitorModuleSettings, options, hasOrganizationWide, requestUserId) {
        let result = {
            total: 0,
            items: [],
        };
        let qb = this.dbClient
            .withSchema(organizationId)
            .from("organizationVisitorProfiles as ovp")
            .innerJoin("userOrganizations as uo", (join) => {
            join.on("ovp.userOrganizationId", "=", "uo.id");
        });
        qb.where("uo.organizationId", "=", organizationId).whereNull("uo.deletedAt").whereNull("ovp.deletedAt");
        if (options.state == dal_constants_1.DalConstants.UnterminatedVisitState.Active || options.state == dal_constants_1.DalConstants.UnterminatedVisitState.Expected) {
            qb.innerJoin("organizationActiveVisits as oav", (join) => {
                join.on("ovp.id", "oav.organizationVisitorProfileId");
            }).where("oav.state", options.state);
        }
        if (!hasOrganizationWide) {
            let userPreviousVisitors = await dal_manager_1.dbManager.accessLog.listPreviousVisitorProfileIds(organizationId, requestUserId);
            let currentActiveVisitors = (await this.dbClient
                .withSchema(organizationId)
                .from("organizationActiveVisits as oav")
                .innerJoin("userOrganizations as uo", "uo.id", "oav.visitedUserOrganizationId")
                .where("uo.userId", requestUserId)
                .where("uo.organizationId", organizationId)
                .whereNull("uo.deletedAt")
                .select("oav.organizationVisitorProfileId")).map((u) => u.organizationVisitorProfileId);
            let previousVisitors = userPreviousVisitors.concat(currentActiveVisitors);
            if (previousVisitors && previousVisitors.length > 0) {
                qb.whereIn("ovp.id", previousVisitors);
            }
            else {
                return Promise.resolve(result);
            }
        }
        let columns = ["ovp.id"];
        let name = false;
        let surname = false;
        let uniqueId = false;
        let filterFields = [];
        for (const visitorProfileField of visitorModuleSettings.visitorProfileFormFields) {
            let fieldName = "ovp.";
            let columnName = fieldName;
            if (visitorProfileField.reportable) {
                if (visitorProfileField.extension) {
                    fieldName += `"extensionFields"->>'` + visitorProfileField.name + "'";
                    columnName += "extension." + visitorProfileField.name;
                }
                else {
                    fieldName += '"' + visitorProfileField.name + '"';
                    columnName += visitorProfileField.name;
                    switch (visitorProfileField.name) {
                        case dal_access_models_1.DbAccessModel.PredefinedVisitorProfileFormFields.name:
                            name = true;
                            break;
                        case dal_access_models_1.DbAccessModel.PredefinedVisitorProfileFormFields.surname:
                            surname = true;
                            break;
                        case dal_access_models_1.DbAccessModel.PredefinedVisitorProfileFormFields.uniqueId:
                            uniqueId = true;
                            break;
                        default:
                            break;
                    }
                }
                columns.push(this.dbClient.raw(fieldName + ' as "' + columnName + '"'));
                filterFields.push("unaccent(upper(" + fieldName + "))::TEXT ilike unaccent(upper(?))");
            }
        }
        qb.whereWrapped((q) => {
            let filter = "%" + options.filter + "%";
            for (const w of filterFields) {
                q.orWhereRaw(w, filter);
            }
        });
        let qbCount = qb.clone();
        result.total = await qbCount.count("ovp.id as c").then((rows) => {
            return parseInt(rows[0].c);
        });
        if (result.total === 0) {
            return Promise.resolve(result);
        }
        qb.orderByRaw("ovp.name").column(columns).select().offset(options.skip).limit(options.take);
        result.items = await qb.then((rows) => {
            return rows.map((t) => {
                let captionLines = [];
                if (name && surname) {
                    captionLines.push([t["ovp.name"], t["ovp.surname"]].join(" "));
                }
                else if (name) {
                    captionLines.push(t["ovp.name"]);
                }
                else if (surname) {
                    captionLines.push(t["ovp.surname"]);
                }
                let matchItem = "";
                let regex = new RegExp((0, dal_utils_1.extendStringForUnaccent)(options.filter), "ig");
                for (const key of Object.keys(t)) {
                    matchItem = t[key] + "";
                    let searchResult = matchItem.search(regex);
                    if (searchResult >= 0) {
                        break;
                    }
                    matchItem = "";
                }
                return {
                    id: t["id"],
                    captionLines: captionLines,
                    matchItem: matchItem,
                };
            });
        });
        return Promise.resolve(result);
    }
    async genericUnterminatedVisitSearch(organizationId, visitorModuleSettings, options, hasOrganizationWide, requestUserId) {
        let result = {
            total: 0,
            items: [],
        };
        let qb = this.dbClient
            .withSchema(organizationId)
            .from("organizationActiveVisits as oav")
            .innerJoin("organizationVisitorProfiles as ovp", "oav.organizationVisitorProfileId", "ovp.id")
            .innerJoin("userOrganizations as uo", "ovp.userOrganizationId", "uo.id")
            .where("uo.organizationId", "=", organizationId)
            .where("oav.organizationId", "=", organizationId)
            .whereNull("uo.deletedAt")
            .whereNull("ovp.deletedAt");
        if (!hasOrganizationWide) {
            let currentActiveVisitors = (await this.dbClient
                .withSchema(organizationId)
                .from("organizationActiveVisits as oav")
                .innerJoin("userOrganizations as uo", "uo.id", "oav.visitedUserOrganizationId")
                .where("uo.userId", requestUserId)
                .where("uo.organizationId", organizationId)
                .whereNull("uo.deletedAt")
                .select("oav.organizationVisitorProfileId")).map((u) => u.organizationVisitorProfileId);
            if (currentActiveVisitors && currentActiveVisitors.length > 0) {
                qb.whereIn("ovp.id", currentActiveVisitors);
            }
            else {
                return Promise.resolve(result);
            }
        }
        let i = 0;
        let orders = [];
        let columns = ["ovp.id"];
        let captionLineRowNumbers = {};
        let captionMaxCaptionLineRowNumber = 0;
        let captionLineFields = [];
        for (const visitorProfileField of visitorModuleSettings.visitorProfileFormFields) {
            if (visitorProfileField.showInCaptionLines) {
                captionMaxCaptionLineRowNumber = Math.max(visitorProfileField.captionLineRowOrder, captionMaxCaptionLineRowNumber);
                captionLineRowNumbers[visitorProfileField.captionLineRowOrder + ""] = true;
                let captionLineField = {
                    name: visitorProfileField.name,
                    rowOrder: visitorProfileField.captionLineRowOrder,
                    colOrder: visitorProfileField.captionLineColumnOrder,
                    isExtension: visitorProfileField.extension,
                    colName: null,
                };
                if (visitorProfileField.extension) {
                    captionLineField.colName = "c" + i++;
                    captionLineFields.push(captionLineField);
                    let fieldName = `ovp."extensionFields"->>'` + visitorProfileField.name + "'";
                    columns.push(this.dbClient.raw(fieldName + " as " + captionLineField.colName));
                }
                else {
                    captionLineField.colName = visitorProfileField.name;
                    switch (visitorProfileField.name) {
                        case dal_access_models_1.DbAccessModel.PredefinedVisitorProfileFormFields.name:
                            captionLineFields.push(captionLineField);
                            columns.push("ovp.name");
                            break;
                        case dal_access_models_1.DbAccessModel.PredefinedVisitorProfileFormFields.surname:
                            captionLineFields.push(captionLineField);
                            columns.push("ovp.surname");
                            break;
                        case dal_access_models_1.DbAccessModel.PredefinedVisitorProfileFormFields.uniqueId:
                            captionLineFields.push(captionLineField);
                            columns.push("ovp.uniqueId");
                            break;
                        default:
                            break;
                    }
                }
            }
        }
        let addedSearchFields = [];
        let orWhereList = [];
        for (const filterField of visitorModuleSettings.visitorProfileFormFields) {
            let fieldName = "ovp.";
            let c = "c" + i++;
            if (filterField.extension) {
                fieldName += `"extensionFields"->>'` + filterField.name + "'";
                orWhereList.push(["unaccent(" + fieldName + ")::TEXT ilike unaccent(?)", "%" + options.filter + "%"]);
            }
            else {
                fieldName += '"' + filterField.name + '"';
                orWhereList.push(["unaccent(" + fieldName + ")::TEXT ilike unaccent(?)", "%" + options.filter + "%"]);
            }
            columns.push(this.dbClient.raw(fieldName + " as " + c));
            addedSearchFields.push({ name: filterField.name, colName: c });
            orders.push(c);
        }
        if (orWhereList.length > 0) {
            qb.where((qbx) => {
                qbx = qbx.whereRaw(orWhereList[0][0], orWhereList[0][1]);
                for (let index = 1; index < orWhereList.length; index++) {
                    qbx.orWhereRaw(orWhereList[index][0], orWhereList[index][1]);
                }
            });
        }
        let qbCount = qb.clone();
        result.total = await qbCount.count("ovp.id as c").then((rows) => {
            return parseInt(rows[0].c);
        });
        if (result.total === 0) {
            return Promise.resolve(result);
        }
        qb.orderByRaw(orders.join(",")).column(columns).select().offset(options.skip).limit(options.take);
        result.items = await qb.then((rows) => {
            return rows.map((t) => {
                let captionLines = [];
                for (let rowNum = 1; rowNum <= captionMaxCaptionLineRowNumber; rowNum++) {
                    if (captionLineRowNumbers[rowNum + ""]) {
                        let itemsOfCaptionLine = [];
                        let rowItems = captionLineFields
                            .filter((cl) => cl.rowOrder === rowNum)
                            .sort((n1, n2) => {
                            if (n1.colOrder > n2.colOrder) {
                                return 1;
                            }
                            if (n1.colOrder < n2.colOrder) {
                                return -1;
                            }
                            return 0;
                        });
                        for (let colIndex = 0; colIndex < rowItems.length; colIndex++) {
                            let val = t[rowItems[colIndex].colName];
                            if (val) {
                                itemsOfCaptionLine.push(val);
                            }
                        }
                        let line = itemsOfCaptionLine.join(" ").trim();
                        if (line && line.length > 0) {
                            captionLines.push(line);
                        }
                    }
                }
                return {
                    id: t["id"],
                    captionLines: captionLines,
                    searchFields: addedSearchFields.map((tt) => t[tt.colName]).join(" "),
                };
            });
        });
        return Promise.resolve(result);
    }
    async upsertVisitorProfile(organizationId, fields, organizationVisitorModuleSettings, requestUserId, visitorProfileId) {
        let result = {
            affectedDeviceIds: [],
            visitorProfileId: visitorProfileId,
            visitorUserId: null,
        };
        return this.dbClient.transaction(async (trx) => {
            if (visitorProfileId) {
                await this.updateVisitorProfile(organizationId, requestUserId, fields, organizationVisitorModuleSettings, visitorProfileId, trx);
            }
            else {
                const insertProfileResult = await this.insertVisitorProfile(organizationId, requestUserId, fields, organizationVisitorModuleSettings, trx, true);
                result.visitorProfileId = insertProfileResult.visitorProfileId;
                result.visitorUserId = insertProfileResult.visitorUserId;
            }
            return Promise.resolve(result);
        });
    }
    async updateVisitorProfile(organizationId, requestUserId, fields, organizationVisitorModuleSettings, visitorProfileId, trx) {
        let opTime = new Date();
        let visitorProfile = await this.dbClient
            .withSchema(organizationId)
            .table("organizationVisitorProfiles as ovp")
            .innerJoin("userOrganizations as uo", (join) => {
            join.on("ovp.userOrganizationId", "=", "uo.id");
        })
            .where("ovp.id", "=", visitorProfileId)
            .where("uo.organizationId", "=", organizationId)
            .select("ovp.*", "uo.userId")
            .transacting(trx)
            .then((rows) => {
            if (!rows || rows.length === 0) {
                (0, dal_access_error_1.throwDbAccessNotFoundError)("Visitor profile is not found");
            }
            return rows[0];
        });
        const visitorUserId = visitorProfile["userId"];
        let visitorProfileExtensionFields = visitorProfile.extensionFields || {};
        visitorProfile.updatedAt = opTime;
        for (const updateField of fields) {
            let profileField = organizationVisitorModuleSettings.visitorProfileFormFields.find((t) => t.name === updateField.name);
            if (profileField && profileField.extension) {
                visitorProfileExtensionFields[profileField.name] = updateField.value;
            }
            else if (profileField?.name) {
                visitorProfile[profileField.name] = updateField.value;
            }
        }
        visitorProfile.extensionFields = JSON.stringify(visitorProfileExtensionFields);
        delete visitorProfile["userId"];
        let qbUpdate = this.dbClient
            .withSchema(organizationId)
            .table("organizationVisitorProfiles")
            .where("id", "=", visitorProfileId)
            .whereNull("deletedAt")
            .update(visitorProfile, ["userOrganizationId"])
            .transacting(trx);
        await qbUpdate.then((rows) => {
            if (!rows || rows.length === 0) {
                (0, dal_access_error_1.throwDbAccessNotFoundError)("Visitor profile is not found");
            }
            return rows[0];
        });
        for (let i = 0; i < fields.length; i++) {
            if (fields[i].value === undefined) {
                fields[i].value = null;
            }
        }
        await dal_manager_1.dbManager.accessLog.addUserActionHistoryItem({
            oId: organizationId,
            o: requestUserId,
            u: new Date(),
            c: dal_constants_1.DalConstants.UserActionCategory.Visitor,
            tg: ["Visitor", "UpdateVisitorProfile"],
            t: dal_constants_1.DalConstants.UserVisitorActionType.UpdateVisitorProfile,
            d: {
                visitorProfileId: visitorProfileId,
                fields: fields,
            },
        });
        return visitorUserId;
    }
    async checkVisitorProfileUniqueness(organizationId, fields, organizationVisitorModuleSettings, trx, hasOrganizationWide, previousVisitors) {
        let qb = trx
            .withSchema(organizationId)
            .from("organizationVisitorProfiles as ovp")
            .innerJoin("userOrganizations as uo", (join) => {
            join.on("ovp.userOrganizationId", "=", "uo.id");
        });
        qb.where("uo.organizationId", "=", organizationId).whereNull("uo.deletedAt").whereNull("ovp.deletedAt");
        if (!hasOrganizationWide) {
            if (previousVisitors && previousVisitors.length > 0)
                qb.whereIn("ovp.id", previousVisitors);
            else {
                return Promise.resolve(true);
            }
        }
        let uniqueProfileFields = organizationVisitorModuleSettings.visitorProfileFormFields.filter((t) => t.unique);
        if (uniqueProfileFields.length < 1) {
            return Promise.resolve(true);
        }
        for (const profileField of uniqueProfileFields) {
            let fieldName = "ovp.";
            let targetValue = fields.find((t) => t.name == profileField.name);
            if (targetValue.value) {
                if (profileField.extension) {
                    fieldName += `"extensionFields"->>'` + profileField.name + "'";
                    qb.whereRaw(fieldName + "::TEXT = ?", targetValue.value);
                }
                else {
                    fieldName += '"' + profileField.name + '"';
                    qb.whereRaw(fieldName + "::TEXT = ?", targetValue.value);
                }
            }
            else {
                if (profileField.extension) {
                    fieldName += `"extensionFields"->>'` + profileField.name + "'";
                    qb.whereRaw(fieldName + "::TEXT = ? ", "");
                }
                else {
                    fieldName += '"' + profileField.name + '"';
                    qb.whereRaw(fieldName + " is null");
                }
            }
        }
        let results = await qb.select("ovp.id");
        if (results.length > 0) {
            return Promise.resolve(false);
        }
        return Promise.resolve(true);
    }
    async insertVisitorProfile(organizationId, requestUserId, fields, organizationVisitorModuleSettings, trx, hasOrganizationWide, previousVisitors) {
        let checkUniqueFields = await this.checkVisitorProfileUniqueness(organizationId, fields, organizationVisitorModuleSettings, trx, hasOrganizationWide, previousVisitors);
        if (!checkUniqueFields) {
            (0, dal_access_error_1.throwDbAccessConflictErrorTr)("ERRORS.VISITOR.PROFILENOTUNIQUE", null, false);
        }
        let opTime = new Date();
        let uniqueIdField = fields.find((t) => t.name === "uniqueId");
        if (uniqueIdField && uniqueIdField.value) {
            await this.dbClient
                .withSchema(organizationId)
                .table("organizationVisitorProfiles as ovp")
                .innerJoin("userOrganizations as uo", (join) => {
                join.on("ovp.userOrganizationId", "=", "uo.id");
            })
                .where("ovp.uniqueId", "=", uniqueIdField.value)
                .where("uo.organizationId", "=", organizationId)
                .whereNull("ovp.deletedAt")
                .select("ovp.*")
                .transacting(trx)
                .then((rows) => {
                if (rows && rows.length > 0) {
                    (0, dal_access_error_1.throwDbAccessBadRequestErrorTr)("ERRORS.VISITOR.PROFILEALREADYEXISTS", null, false);
                }
            });
        }
        let userId = await this.dbClient
            .withSchema("public")
            .table("userList")
            .insert({
            id: uuid_1.default.v4(),
            username: null,
        })
            .returning("id")
            .transacting(trx)
            .then((rows) => rows[0]);
        let userOrganizationId = await this.dbClient
            .withSchema("public")
            .table("userOrganizationMapping")
            .insert({
            id: uuid_1.default.v4(),
            userId: userId,
            organizationId: organizationId,
        })
            .returning("id")
            .transacting(trx)
            .then((rows) => rows[0]);
        await this.dbClient
            .withSchema(organizationId)
            .table("users")
            .insert({
            id: userId,
            createdAt: opTime,
            updatedAt: opTime,
            settings: {
                locale: (await (0, dal_memcache_1.getCacheOrganizationSettings)(organizationId)).locale,
            },
        })
            .transacting(trx);
        const organizationMediumSettings = (await (0, dal_memcache_1.getCacheOrganizationSettings)(organizationId)).notification.mediumSettings.general;
        await this.dbClient
            .withSchema(organizationId)
            .table("userOrganizations")
            .insert({
            id: userOrganizationId,
            createdAt: opTime,
            updatedAt: opTime,
            userId: userId,
            organizationId: organizationId,
            roleId: await dal_manager_1.dbManager.accessUser.getOrganizationVisitorTypedRoleIdOfOrganization(organizationId, trx),
            settings: {
                notification: {
                    mediumSettings: {
                        general: organizationMediumSettings,
                    },
                },
            },
        })
            .transacting(trx);
        let visitorProfileId = uuid_1.default.v4();
        let visitorProfile = {
            id: visitorProfileId,
            createdAt: opTime,
            updatedAt: opTime,
            userOrganizationId: userOrganizationId,
            extensionFields: null,
        };
        let visitorProfileExtensionFields = {};
        for (const updateField of fields) {
            let profileField = organizationVisitorModuleSettings.visitorProfileFormFields.find((t) => t.name === updateField.name);
            if (profileField && profileField.extension) {
                visitorProfileExtensionFields[profileField.name] = updateField.value;
            }
            else if (profileField?.name) {
                visitorProfile[profileField.name] = updateField.value;
            }
        }
        visitorProfile.extensionFields = JSON.stringify(visitorProfileExtensionFields);
        await this.dbClient.withSchema(organizationId).table("organizationVisitorProfiles").insert(visitorProfile).transacting(trx);
        await dal_manager_1.dbManager.accessLog.addUserActionHistoryItem({
            oId: organizationId,
            o: requestUserId,
            u: new Date(),
            c: dal_constants_1.DalConstants.UserActionCategory.Visitor,
            tg: ["Visitor", "InsertVisitorProfile"],
            t: dal_constants_1.DalConstants.UserVisitorActionType.InsertVisitorProfile,
            d: {
                visitorProfileId: visitorProfileId,
            },
        });
        return Promise.resolve({ visitorProfileId, visitorUserId: userId });
    }
    async replaceVisitorProfile(organizationId, requestUserId, visitorProfileId, replacedVisitorProfileId, organizationVisitorModuleSettings, trx) {
        if (visitorProfileId === replacedVisitorProfileId) {
            return Promise.resolve();
        }
        let opTime = new Date();
        let visitorProfile = await trx
            .withSchema(organizationId)
            .table("organizationVisitorProfiles as ovp")
            .innerJoin("userOrganizations as uo", (join) => {
            join.on("ovp.userOrganizationId", "=", "uo.id");
        })
            .where("ovp.id", "=", visitorProfileId)
            .where("uo.organizationId", "=", organizationId)
            .first("ovp.*");
        let replacedVisitorProfile = await trx
            .withSchema(organizationId)
            .table("organizationVisitorProfiles as ovp")
            .innerJoin("userOrganizations as uo", (join) => {
            join.on("ovp.userOrganizationId", "=", "uo.id");
        })
            .where("ovp.id", "=", replacedVisitorProfileId)
            .where("uo.organizationId", "=", organizationId)
            .first("ovp.*");
        if (!replacedVisitorProfile) {
            (0, dal_access_error_1.throwDbAccessNotFoundError)("replace visitor not found");
        }
        let visitorProfileExtensionFields = visitorProfile.extensionFields || {};
        let replacedVisitorProfileExtensionFields = replacedVisitorProfile.extensionFields || {};
        visitorProfile.updatedAt = opTime;
        for (const profileField of organizationVisitorModuleSettings.visitorProfileFormFields) {
            if (profileField && profileField.extension) {
                visitorProfileExtensionFields[profileField.name] = visitorProfileExtensionFields[profileField.name] || replacedVisitorProfileExtensionFields[profileField.name];
            }
            else {
                visitorProfile[profileField.name] = replacedVisitorProfile.value;
            }
        }
        visitorProfile.extensionFields = JSON.stringify(visitorProfileExtensionFields);
        await trx.withSchema(organizationId).table("organizationVisitorProfiles").where("id", visitorProfileId).whereNull("deletedAt").returning("userOrganizationId").update(visitorProfile);
        await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorProfiles).where("id", replacedVisitorProfile.id).update({
            deletedAt: opTime,
        });
        await trx
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationActiveVisits)
            .update({
            organizationVisitorProfileId: visitorProfile.id,
        })
            .where("organizationVisitorProfileId", replacedVisitorProfile.id);
        await dal_manager_1.dbManager.accessLog.addUserActionHistoryItem({
            oId: organizationId,
            o: requestUserId,
            u: new Date(),
            c: dal_constants_1.DalConstants.UserActionCategory.Visitor,
            tg: ["Visitor", "MergeVisitorProfile"],
            t: dal_constants_1.DalConstants.UserVisitorActionType.MergeVisitorProfile,
            d: {
                visitorProfileId: visitorProfileId,
                replacedVisitorProfileId: replacedVisitorProfile.id,
            },
        });
        await dal_manager_1.dbManager.accessLog.updateVisitorProfileId(organizationId, visitorProfile.id, replacedVisitorProfile.id);
        return Promise.resolve();
    }
    async verifyActiveVisit(organizationId, visitFields, credentials, visitorProfileFields, visitorProfileId, trx, replacedVisitorProfileId) {
        if (visitorProfileId) {
            await this.dbClient
                .withSchema(organizationId)
                .from("organizationActiveVisits as oav")
                .where("oav.organizationId", organizationId)
                .where("oav.organizationVisitorProfileId", visitorProfileId)
                .whereNotNull("startUtc")
                .select("id")
                .transacting(trx)
                .then((rows) => {
                if (rows && rows.length > 0) {
                    (0, dal_access_error_1.throwDbAccessBadRequestErrorTr)("ERRORS.VISITOR.ALREADYACTIVEVISIT", null, false);
                }
            });
        }
        if (visitorProfileId && replacedVisitorProfileId) {
            let replacedVisitorProfile = await trx
                .withSchema(organizationId)
                .table("organizationVisitorProfiles as ovp")
                .innerJoin("userOrganizations as uo", (join) => {
                join.on("ovp.userOrganizationId", "=", "uo.id");
            })
                .where("ovp.id", "=", replacedVisitorProfileId)
                .where("uo.organizationId", "=", organizationId)
                .first("ovp.*")
                .transacting(trx);
            if (!replacedVisitorProfile) {
                (0, dal_access_error_1.throwDbAccessNotFoundError)("replace visitor not found");
            }
        }
        if (visitorProfileFields && visitorProfileFields.length > 0 && visitorProfileId) {
            await this.dbClient
                .withSchema(organizationId)
                .table("organizationVisitorProfiles as ovp")
                .innerJoin("userOrganizations as uo", (join) => {
                join.on("ovp.userOrganizationId", "uo.id");
            })
                .where("ovp.id", visitorProfileId)
                .where("uo.organizationId", organizationId)
                .select("ovp.*", "uo.userId")
                .transacting(trx)
                .then((rows) => {
                if (!rows || rows.length === 0) {
                    (0, dal_access_error_1.throwDbAccessNotFoundError)("Visitor profile is not found");
                }
                return rows[0];
            });
        }
        let visitedPersonUser = visitFields.find((t) => t.name === dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.visitedPerson && t.value);
        if (visitedPersonUser) {
            let visitedUser = await this.dbClient
                .withSchema(organizationId)
                .from("userOrganizations as uo")
                .innerJoin("users as u", "u.id", "uo.userId")
                .innerJoin("userOrganizationProfiles as uop", "uo.id", "uop.userOrganizationId")
                .where("u.id", visitedPersonUser.value)
                .where("organizationId", organizationId)
                .whereNull("uo.deletedAt")
                .whereNull("u.deletedAt")
                .whereNull("uop.deletedAt")
                .where("uo.isDisabled", false)
                .first("u.id")
                .transacting(trx);
            if (!visitedUser) {
                throw (0, api_error_1.generateForbiddenError)({ message: "visited person does not exist" });
            }
        }
        if (credentials) {
            for (const credential of credentials) {
                let checkCredential = await trx
                    .withSchema(organizationId)
                    .table("userOrganizationCredentials")
                    .whereNull("deletedAt")
                    .where("data", credential.data)
                    .where("type", credential.type)
                    .first()
                    .transacting(trx);
                if (checkCredential) {
                    throw (0, api_error_1.generateConflictError)({ message: "Credential Already Exists", details: credential.data });
                }
            }
        }
    }
    async insertActiveVisit(organizationId, requestUserId, params, trx) {
        let visitorProfile;
        let { accessControlPointIds, credentials, organizationVisitorModuleSettings, visitFields, visitorProfileId, processTime, replacedVisitorProfileId, thumbnail, visitId, visitorProfileFields, visitorRegistrationPointId, } = params;
        if (visitorProfileId) {
            await this.dbClient
                .withSchema(organizationId)
                .from("organizationActiveVisits as oav")
                .where("oav.organizationId", "=", organizationId)
                .where("oav.organizationVisitorProfileId", "=", visitorProfileId)
                .whereNotNull("startUtc")
                .select("id")
                .then((rows) => {
                if (rows && rows.length > 0) {
                    (0, dal_access_error_1.throwDbAccessBadRequestErrorTr)("ERRORS.VISITOR.ALREADYACTIVEVISIT", null, false);
                }
            });
            await this.dbClient
                .withSchema(organizationId)
                .from(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorStates)
                .where("organizationVisitorProfileId", "=", visitorProfileId)
                .whereNotNull("startUtc")
                .where("state", "=", dal_constants_1.DalConstants.OrganizationVisitorStates.Forbidden)
                .whereNull("deletedAt")
                .where(function () {
                this.whereNull("endUtc").orWhere("endUtc", ">", "NOW()");
            })
                .select("id", "startUtc", "endUtc")
                .then((rows) => {
                if (rows && rows.length > 0) {
                    if (organizationVisitorModuleSettings.enableToForceRegistrationOfForbiddenVisitors) {
                        if (!params.force) {
                            if (rows[0].endUtc) {
                                throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.VISITOR_REGISTRATION_PARK_SLOT_IS_FULL, "ERRORS.VISITOR.FORBIDDEN_VISITOR.BETWEEN", {
                                    startUtc: new Date(rows[0].startUtc).toLocaleString(params.locale ?? "en"),
                                    endUtc: new Date(rows[0].endUtc).toLocaleString(params.locale ?? "en"),
                                }, true, true);
                            }
                            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.VISITOR_REGISTRATION_PARK_SLOT_IS_FULL, "ERRORS.VISITOR.FORBIDDEN_VISITOR.INDEFINITE", { startUtc: new Date(rows[0].startUtc).toLocaleString(params.locale ?? "en") }, true, true);
                        }
                    }
                    else {
                        if (rows[0].endUtc) {
                            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.BAD_REQUEST, "ERRORS.VISITOR.FORBIDDEN_VISITOR.BETWEEN", { startUtc: new Date(rows[0].startUtc).toLocaleString(params.locale ?? "en"), endUtc: new Date(rows[0].endUtc).toLocaleString(params.locale ?? "en") }, true, true);
                        }
                        throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.BAD_REQUEST, "ERRORS.VISITOR.FORBIDDEN_VISITOR.INDEFINITE", { startUtc: new Date(rows[0].startUtc).toLocaleString(params.locale ?? "en") }, true, true);
                    }
                }
            });
        }
        let opTime = new Date();
        let result = {
            visitorProfileId: visitorProfileId,
            visitId: null,
            visitedPersonUserId: null,
            visitorUserId: null,
            invitationInfo: null,
        };
        if (visitorProfileId && replacedVisitorProfileId) {
            await this.replaceVisitorProfile(organizationId, requestUserId, visitorProfileId, replacedVisitorProfileId, organizationVisitorModuleSettings, trx);
        }
        if (visitorProfileFields && visitorProfileFields.length > 0) {
            if (visitorProfileId) {
                result.visitorUserId = await this.updateVisitorProfile(organizationId, requestUserId, visitorProfileFields, organizationVisitorModuleSettings, visitorProfileId, trx);
            }
            else {
                const { visitorProfileId, visitorUserId } = await this.insertVisitorProfile(organizationId, requestUserId, visitorProfileFields, organizationVisitorModuleSettings, trx, true);
                result.visitorProfileId = visitorProfileId;
                result.visitorUserId = visitorUserId;
            }
        }
        else if (visitorProfileId) {
            visitorProfile = (await this.getVisitorDetailed(organizationId, visitorProfileId, organizationVisitorModuleSettings)).profile;
            result.visitorUserId = await trx
                .withSchema(organizationId)
                .table("organizationVisitorProfiles as ovp")
                .innerJoin("userOrganizations as uo", (join) => {
                join.on("ovp.userOrganizationId", "=", "uo.id");
            })
                .where("ovp.id", "=", visitorProfileId)
                .where("uo.organizationId", "=", organizationId)
                .select("uo.userId")
                .then((rows) => {
                if (!rows || rows.length === 0) {
                    (0, dal_access_error_1.throwDbAccessNotFoundError)("Visitor profile is not found");
                }
                return rows[0].userId;
            });
        }
        if (thumbnail && result.visitorProfileId) {
            await trx.withSchema(organizationId).table("organizationVisitorProfiles").where("id", result.visitorProfileId).update({
                thumbnail: thumbnail,
            });
        }
        let newVisit = {
            id: visitId ? visitId : uuid_1.default.v4(),
            organizationId: organizationId,
            startUtc: opTime,
            organizationVisitorProfileId: result.visitorProfileId,
            visitedOrganizationUnitId: undefined,
            visitedUserOrganizationId: undefined,
            escortUserOrganizationId: undefined,
            extensionFields: null,
            state: dal_constants_1.DalConstants.UnterminatedVisitState.Active,
            visitorRegistrationPointId: visitorRegistrationPointId,
            processTime: processTime,
        };
        let visitedPersonUser = visitFields.find((t) => t.name === dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.visitedPerson && t.value);
        let escortUser = visitFields.find((t) => t.name === dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.escort && t.value);
        let userIds = [];
        if (visitedPersonUser) {
            let visitedUser = await this.dbClient
                .withSchema(organizationId)
                .from("userOrganizations as uo")
                .innerJoin("users as u", "u.id", "uo.userId")
                .innerJoin("userOrganizationProfiles as uop", "uo.id", "uop.userOrganizationId")
                .where("u.id", visitedPersonUser.value)
                .where("organizationId", organizationId)
                .whereNull("uo.deletedAt")
                .whereNull("u.deletedAt")
                .whereNull("uop.deletedAt")
                .where("uo.isDisabled", false)
                .first("u.id");
            if (!visitedUser) {
                throw (0, api_error_1.generateForbiddenError)({ message: "visited person does not exist" });
            }
            userIds.push(visitedPersonUser.value);
            result.visitedPersonUserId = visitedPersonUser.value;
        }
        if (escortUser) {
            userIds.push(escortUser.value);
        }
        if (userIds.length > 0) {
            let users = await this.dbClient
                .withSchema(organizationId)
                .from("userOrganizations as uo")
                .whereIn("userId", userIds)
                .where("organizationId", organizationId)
                .whereNull("deletedAt")
                .column([{ userOrganizationId: "uo.id" }, "userId"])
                .select()
                .transacting(trx)
                .then((rows) => {
                return rows;
            });
            if (visitedPersonUser) {
                let user = users.find((t) => t.userId === visitedPersonUser.value);
                if (user) {
                    newVisit.visitedUserOrganizationId = user.userOrganizationId;
                }
            }
            if (escortUser) {
                let user = users.find((t) => t.userId === escortUser.value);
                if (user) {
                    newVisit.escortUserOrganizationId = user.userOrganizationId;
                }
            }
        }
        let visitExtensionFields = {};
        for (const newVisitField of visitFields) {
            let visitField = organizationVisitorModuleSettings.visitFormFields.find((t) => t.name === newVisitField.name);
            if (visitField.unique) {
                let existingUniqueQb = this.dbClient
                    .withSchema(organizationId)
                    .from("organizationActiveVisits as oav")
                    .where("oav.organizationId", "=", organizationId)
                    .select("oav.id")
                    .transacting(trx);
                if (visitorRegistrationPointId)
                    existingUniqueQb = existingUniqueQb.where("visitorRegistrationPointId", visitorRegistrationPointId);
                let col = visitField.name;
                if (visitField.extension) {
                    col = `"extensionFields"->>'` + visitField.name + "'";
                }
                let row = await existingUniqueQb.whereRaw(col + `= '${newVisitField.value}'`).first();
                if (row) {
                    (0, dal_access_error_1.throwDbAccessBadRequestErrorTr)("ERRORS.VISITOR.FIELDINUSE", { field: visitField.caption ? visitField.caption.toUpperCase() : "" }, false);
                }
            }
            if (visitField.extension) {
                visitExtensionFields[newVisitField.name] = newVisitField.value;
            }
            else {
                switch (newVisitField.name) {
                    case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.visitedOrganizationUnit:
                        newVisit.visitedOrganizationUnitId = newVisitField.value;
                        break;
                    default:
                        break;
                }
            }
        }
        newVisit.extensionFields = JSON.stringify(visitExtensionFields);
        if (visitId) {
            await this.dbClient.withSchema(organizationId).table("organizationActiveVisits").where("id", newVisit.id).update(newVisit).transacting(trx);
        }
        else {
            await this.dbClient.withSchema(organizationId).table("organizationActiveVisits").insert(newVisit).transacting(trx);
        }
        result.visitId = newVisit.id;
        let user = {
            userId: "",
            name: "",
            surname: "",
        };
        if (organizationVisitorModuleSettings.qrCodeEnabled) {
            credentials = credentials || [];
            credentials.push({
                data: JSON.stringify(dal_visitor_utils_1.DalVisitorUtils.generateIdBasedQrCodeContent(newVisit.id)),
                type: dal_constants_1.DalConstants.CredentialType.QrCode,
                expirationUtc: visitFields.find((f) => f.name === dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.qrTimeout)?.value ?? undefined,
            });
        }
        if (credentials || accessControlPointIds) {
            user = await this.dbClient
                .withSchema(organizationId)
                .from("organizationVisitorProfiles as ovp")
                .innerJoin("userOrganizations as uo", (join) => {
                join.on("ovp.userOrganizationId", "=", "uo.id");
            })
                .where("ovp.id", "=", result.visitorProfileId)
                .where("uo.organizationId", "=", organizationId)
                .whereNull("uo.deletedAt")
                .whereNull("ovp.deletedAt")
                .column([{ name: "ovp.name" }, { surname: "ovp.surname" }, { userId: "uo.userId" }])
                .select()
                .transacting(trx)
                .then((rows) => {
                return rows[0];
            });
        }
        if (credentials) {
            for (const credential of credentials) {
                let newCredential = {
                    id: uuid_1.default.v4(),
                    createdAt: opTime,
                    updatedAt: opTime,
                    userId: user.userId,
                    organizationId: organizationId,
                    expiresOn: credential.expirationUtc,
                    type: credential.type,
                    data: credential.data,
                    credentialNumber: credential.credentialNumber,
                };
                try {
                    await this.dbClient.withSchema(organizationId).table("userOrganizationCredentials").insert(newCredential).transacting(trx);
                }
                catch (error) {
                    if (error.message == "Internal DB error") {
                        const symbolKey = Reflect.ownKeys(error).find((key) => key.toString() === "Symbol(message)");
                        if (error[symbolKey].includes(app_enums_1.enums.ErrorCode.CredentialToInsertAlreadyExists.toString())) {
                            (0, dal_access_error_1.throwDbAccessConflictErrorTr)("ERRORS.IDENTITY.DUPLICATECREDENTIAL");
                        }
                    }
                }
            }
        }
        let insertedAccessRights = [];
        if (accessControlPointIds) {
            insertedAccessRights = await this.dbClient
                .withSchema(organizationId)
                .from("accessControlPoints")
                .whereIn("id", accessControlPointIds)
                .where("organizationId", organizationId)
                .whereNull("deletedAt")
                .whereNotNull("deviceId")
                .select("id as accessControlPointId", "deviceId")
                .transacting(trx)
                .then((rows) => {
                return rows;
            });
            for (const insertedAccessRight of insertedAccessRights) {
                insertedAccessRight.id = uuid_1.default.v4();
                let newAccessRight = {
                    id: insertedAccessRight.id,
                    createdAt: opTime,
                    updatedAt: opTime,
                    userId: user.userId,
                    accessControlPointId: insertedAccessRight.accessControlPointId,
                    remoteAccess: false,
                    access: true,
                };
                await this.dbClient.withSchema(organizationId).table("userAccessRights").insert(newAccessRight).transacting(trx);
            }
        }
        if (organizationVisitorModuleSettings.qrCodeEnabled && organizationVisitorModuleSettings.qrSettings && organizationVisitorModuleSettings.qrSettings.communicationMedium) {
            let qrCodeData = dal_visitor_utils_1.DalVisitorUtils.generateIdBasedQrCodeContent(newVisit.id);
            let extensionFieldResult;
            if (!visitorProfileFields) {
                extensionFieldResult = await trx
                    .withSchema(organizationId)
                    .table(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorProfiles)
                    .where("id", visitorProfileId)
                    .first("extensionFields")
                    .transacting(trx);
            }
            result.invitationInfo = {
                visitId: newVisit.id,
                organizationContactInfo: null,
                visitorProfile: visitorProfile || {
                    fields: visitorProfileFields || extensionFieldResult?.extensionFields,
                    id: visitorProfileId,
                    thumbnail,
                    states: [],
                },
                visitedOrganizationUnitName: null,
                visitedUser: null,
                visitorRegistrationPointName: null,
                qrCodeData: qrCodeData,
            };
            await trx
                .withSchema(organizationId)
                .table(dal_db_armon_schema_1.ArmonSchema.tableNames.organizations)
                .where("id", organizationId)
                .first("name", "contactInfo")
                .transacting(trx)
                .then((r) => {
                result.invitationInfo.organizationContactInfo = Object.assign(r.contactInfo || {}, { name: r.name });
            });
            if (newVisit.visitedUserOrganizationId) {
                await dal_manager_1.dbManager.accessUser.getBasicUserInfoWithUserOrganizationId(organizationId, newVisit.visitedUserOrganizationId, trx).then((r) => {
                    result.invitationInfo.visitedUser = r;
                });
            }
            if (newVisit.visitedOrganizationUnitId) {
                await dal_manager_1.dbManager.accessOrganizationUnit.getOrganizationUnitBasic({ organizationId, organizationUnitId: newVisit.visitedOrganizationUnitId }).then((r) => {
                    result.invitationInfo.visitedOrganizationUnitName = r.name;
                });
            }
            if (newVisit.visitorRegistrationPointId) {
                await trx
                    .withSchema(organizationId)
                    .table(dal_db_armon_schema_1.ArmonSchema.tableNames.visitorRegistrationPoints)
                    .where("id", newVisit.visitorRegistrationPointId)
                    .first("name")
                    .transacting(trx)
                    .then((r) => {
                    result.invitationInfo.visitorRegistrationPointName = r.name;
                });
            }
        }
        await dal_manager_1.dbManager.accessLog.addUserActionHistoryItem({
            oId: organizationId,
            o: requestUserId,
            u: opTime,
            c: dal_constants_1.DalConstants.UserActionCategory.Visitor,
            tg: ["Visitor", "NewVisit"],
            t: dal_constants_1.DalConstants.UserVisitorActionType.NewVisit,
            d: {
                visitId: newVisit.id,
            },
        });
        return result;
    }
    async upsertPreregisteredVisit(organizationId, requestUserId, visitedUserId, organizationVisitorModuleSettings, visitFields, visitorProfileFields, visitorProfileId, hasOrganizationWide, trx, visitId, locale) {
        let result = {
            visitorProfileId: visitorProfileId,
            visitorUserId: null,
        };
        const fn = async (trx) => {
            if (visitorProfileFields && visitorProfileFields.length > 0) {
                if (visitorProfileId) {
                    result.visitorUserId = await this.updateVisitorProfile(organizationId, requestUserId, visitorProfileFields, organizationVisitorModuleSettings, visitorProfileId, trx);
                    let expectedDate = visitFields.find((field) => field.name == dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.expectedDate)?.value ?? null;
                    await this.checkPreregistrationForbiddenceOfVisitor(organizationId, visitorProfileId, expectedDate, locale, trx);
                }
                else {
                    let previousVisitors = [];
                    if (!hasOrganizationWide) {
                        let userPreviousVisitors = await dal_manager_1.dbManager.accessLog.listPreviousVisitorProfileIds(organizationId, visitedUserId, trx);
                        let currentActiveVisitors = (await this.dbClient
                            .withSchema(organizationId)
                            .from("organizationActiveVisits as oav")
                            .innerJoin("userOrganizations as uo", "uo.id", "oav.visitedUserOrganizationId")
                            .where("uo.userId", visitedUserId)
                            .where("uo.organizationId", organizationId)
                            .whereNull("uo.deletedAt")
                            .select("oav.organizationVisitorProfileId")
                            .transacting(trx)).map((u) => u.organizationVisitorProfileId);
                        previousVisitors = userPreviousVisitors.concat(currentActiveVisitors);
                    }
                    const { visitorProfileId, visitorUserId } = await this.insertVisitorProfile(organizationId, requestUserId, visitorProfileFields, organizationVisitorModuleSettings, trx, hasOrganizationWide, previousVisitors);
                    result.visitorProfileId = visitorProfileId;
                    result.visitorUserId = visitorUserId;
                }
            }
            else if (visitorProfileId) {
                result.visitorUserId = await this.dbClient
                    .withSchema(organizationId)
                    .table("organizationVisitorProfiles as ovp")
                    .innerJoin("userOrganizations as uo", (join) => {
                    join.on("ovp.userOrganizationId", "=", "uo.id");
                })
                    .where("ovp.id", "=", visitorProfileId)
                    .where("uo.organizationId", "=", organizationId)
                    .select("ovp.*", "uo.userId")
                    .transacting(trx)
                    .then((rows) => {
                    if (!rows || rows.length === 0) {
                        (0, dal_access_error_1.throwDbAccessNotFoundError)("Visitor profile is not found");
                    }
                    return rows[0].userId;
                });
                let expectedDate = visitFields.find((field) => field.name == dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.expectedDate)?.value ?? null;
                await this.checkPreregistrationForbiddenceOfVisitor(organizationId, visitorProfileId, expectedDate, locale, trx);
            }
            let newVisit = {
                id: visitId ? visitId : uuid_1.default.v4(),
                organizationId: organizationId,
                startUtc: null,
                expectedStartUtc: null,
                expectedEndUtc: null,
                organizationVisitorProfileId: result.visitorProfileId,
                visitedOrganizationUnitId: undefined,
                visitedUserOrganizationId: undefined,
                escortUserOrganizationId: undefined,
                extensionFields: null,
                state: dal_constants_1.DalConstants.UnterminatedVisitState.Expected,
            };
            let escortUser = visitFields.find((t) => t.name === dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.escort && t.value);
            let userIds = [];
            if (visitedUserId) {
                userIds.push(visitedUserId);
                result.visitedPersonUserId = visitedUserId;
            }
            if (escortUser) {
                userIds.push(escortUser.value);
            }
            let expectedDate = visitFields.find((t) => t.name === dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.expectedDate && t.value);
            if (expectedDate) {
                newVisit.expectedStartUtc = expectedDate.value.startDate || null;
                newVisit.expectedEndUtc = expectedDate.value.endDate || null;
            }
            const visitFormFields = organizationVisitorModuleSettings.visitFormFields.find((f) => f.name == dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.expectedDate);
            let existingPreregisteredVisitCount;
            if (visitFormFields && expectedDate) {
                const dateRangeFormFieldOpts = visitFormFields.options;
                if (dateRangeFormFieldOpts.maxStartDynamicDate && dateRangeFormFieldOpts.maxStartDynamicDate === dal_constants_1.DalConstants.DateDynamicValue.EndOfDayAfterFiveDays) {
                    existingPreregisteredVisitCount = await this.getProfileExpectedVisitsCountWithDateRange(organizationId, result.visitorProfileId, newVisit.expectedStartUtc, newVisit.expectedEndUtc, true, visitId, trx);
                }
                else {
                    existingPreregisteredVisitCount = await this.getProfileExpectedVisitsCountWithDateRange(organizationId, result.visitorProfileId, newVisit.expectedStartUtc, newVisit.expectedEndUtc, false, visitId, trx);
                }
            }
            else {
                existingPreregisteredVisitCount = await this.getProfileExpectedVisitsCountWithDateRange(organizationId, result.visitorProfileId, newVisit.expectedStartUtc, newVisit.expectedEndUtc, false, visitId, trx);
            }
            if (existingPreregisteredVisitCount > 0) {
                (0, dal_access_error_1.throwDbAccessBadRequestErrorTr)("ERRORS.VISITOR.ALREADYPREGISTEREDVISIT");
            }
            if (userIds.length > 0) {
                let users = await this.dbClient
                    .withSchema(organizationId)
                    .from("userOrganizations as uo")
                    .whereIn("userId", userIds)
                    .where("organizationId", organizationId)
                    .whereNull("deletedAt")
                    .column([{ userOrganizationId: "uo.id" }, "userId"])
                    .select()
                    .transacting(trx)
                    .then((rows) => {
                    return rows;
                });
                if (visitedUserId) {
                    let user = users.find((t) => t.userId === visitedUserId);
                    if (user) {
                        newVisit.visitedUserOrganizationId = user.userOrganizationId;
                    }
                }
                if (escortUser) {
                    let user = users.find((t) => t.userId === escortUser.value);
                    if (user) {
                        newVisit.escortUserOrganizationId = user.userOrganizationId;
                    }
                }
            }
            let visitExtensionFields = {};
            for (const newVisitField of visitFields) {
                let visitField = organizationVisitorModuleSettings.visitFormFields.find((t) => t.name === newVisitField.name);
                if (visitField?.extension) {
                    visitExtensionFields[newVisitField.name] = newVisitField.value;
                }
                else {
                    switch (newVisitField.name) {
                        case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.visitedOrganizationUnit:
                            newVisit.visitedOrganizationUnitId = newVisitField.value;
                            break;
                        default:
                            break;
                    }
                }
            }
            newVisit.extensionFields = JSON.stringify(visitExtensionFields);
            if (visitId) {
                await this.dbClient.withSchema(organizationId).table("organizationActiveVisits").update(newVisit).where("id", visitId).transacting(trx);
            }
            else {
                await this.dbClient.withSchema(organizationId).table("organizationActiveVisits").insert(newVisit).transacting(trx);
            }
            result.visitId = newVisit.id;
            let expectedVisit = await this.getExpectedVisitDetailed(organizationId, newVisit.id, organizationVisitorModuleSettings, trx);
            expectedVisit.requestUserId = requestUserId;
            if (organizationVisitorModuleSettings.qrCodeEnabled && organizationVisitorModuleSettings.gdprSettings) {
                let qrCodeData = dal_visitor_utils_1.DalVisitorUtils.generateIdBasedQrCodeContent(newVisit.id);
                if (organizationVisitorModuleSettings.qrSettings.communicationMedium) {
                    result.invitationInfo = {
                        visitId: newVisit.id,
                        visitorProfile: expectedVisit.profile,
                        organizationContactInfo: null,
                        visitedOrganizationUnitName: null,
                        visitedUser: null,
                        expectedStartUtc: expectedVisit.expectedStartUtc,
                        expectedEndUtc: expectedVisit.expectedEndUtc,
                        visitorRegistrationPointName: null,
                        qrCodeData: qrCodeData,
                    };
                    await trx
                        .withSchema(organizationId)
                        .table(dal_db_armon_schema_1.ArmonSchema.tableNames.organizations)
                        .where("id", organizationId)
                        .first("name", "contactInfo")
                        .transacting(trx)
                        .then((r) => {
                        result.invitationInfo.organizationContactInfo = Object.assign(r.contactInfo || {}, { name: r.name });
                    });
                    if (newVisit.visitedUserOrganizationId) {
                        await dal_manager_1.dbManager.accessUser.getBasicUserInfoWithUserOrganizationId(organizationId, newVisit.visitedUserOrganizationId, trx).then((r) => {
                            result.invitationInfo.visitedUser = r;
                        });
                    }
                    if (newVisit.visitedOrganizationUnitId) {
                        await dal_manager_1.dbManager.accessOrganizationUnit.getOrganizationUnitBasic({ organizationId, organizationUnitId: newVisit.visitedOrganizationUnitId }).then((r) => {
                            result.invitationInfo.visitedOrganizationUnitName = r.name;
                        });
                    }
                    if (newVisit.visitorRegistrationPointId) {
                        await trx
                            .withSchema(organizationId)
                            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.visitorRegistrationPoints)
                            .where("id", newVisit.visitorRegistrationPointId)
                            .first("name")
                            .transacting(trx)
                            .then((r) => {
                            result.invitationInfo.visitorRegistrationPointName = r.name;
                        });
                    }
                }
            }
            await dal_manager_1.dbManager.accessLog.insertPreregisteredVisitToNoSql(organizationId, expectedVisit, trx);
            await this.updateDailyPreregisterSummary(organizationId, { date: new Date(), self: requestUserId == visitedUserId, turnedToVisit: false }, trx);
            await dal_manager_1.dbManager.accessLog.addUserActionHistoryItem({
                oId: organizationId,
                o: requestUserId,
                u: new Date(),
                c: dal_constants_1.DalConstants.UserActionCategory.Visitor,
                tg: ["Visitor", "NewExpectedVisit"],
                t: dal_constants_1.DalConstants.UserVisitorActionType.NewExpectedVisit,
                d: {
                    visitId: newVisit.id,
                    visitorProfileId: newVisit.organizationVisitorProfileId,
                },
            }, trx);
            return Promise.resolve(result);
        };
        if (trx) {
            return await fn(trx);
        }
        else {
            return await dal_manager_1.dbManager.transaction(fn);
        }
    }
    async findActiveVisitorByUniqueField(organizationId, organizationVisitorModuleSettings, fieldName, filter, visitorRegistrationPointId) {
        let qb = this.dbClient.withSchema(organizationId).from("organizationActiveVisits as oav").where("oav.organizationId", "=", organizationId);
        if (visitorRegistrationPointId)
            qb.where("oav.visitorRegistrationPointId", visitorRegistrationPointId);
        let column = fieldName;
        if (organizationVisitorModuleSettings.visitFormFields.find((vf) => vf.name === fieldName).extension) {
            column = `"extensionFields"->>'` + fieldName + "'";
        }
        let row = await qb.whereRaw(column + `= '${filter}'`).first();
        if (!row) {
            return Promise.resolve(null);
        }
        let visitorDetailed = await this.getVisitorDetailedByVisitId(organizationId, row.id, organizationVisitorModuleSettings);
        return Promise.resolve(visitorDetailed);
    }
    async findVisitorByCredentialData(organizationId, hasIdentityRead, credentialData, credentialType) {
        let result = {
            existance: dal_constants_1.DalConstants.CredentialExistance.NotExits,
            credentialType: credentialType,
            member: "",
            visitor: null,
        };
        let checkCredential = await dal_manager_1.dbManager.accessIdentity.findCredentialStatus(organizationId, credentialData, credentialType);
        result.existance = checkCredential.existance;
        result.credentialType = checkCredential.credentialType;
        if (checkCredential.existance == dal_constants_1.DalConstants.CredentialExistance.NotExits) {
            return Promise.resolve(result);
        }
        if (checkCredential.existance == dal_constants_1.DalConstants.CredentialExistance.Member) {
            if (hasIdentityRead) {
                let user = await dal_manager_1.dbManager.accessUser.getBasicUserInfo(organizationId, checkCredential.userId);
                result.member = user.fullname;
            }
            return Promise.resolve(result);
        }
        result.visitor = await this.findActiveVisitorByCredentialData(organizationId, credentialData);
        return Promise.resolve(result);
    }
    async disableVisitorModule(organizationId, trx) {
        await trx.raw(`DELETE FROM public."${dal_db_armon_schema_1.ArmonSchema.tableNames.scheduled_job}"
			WHERE type = ANY(?::int[]) AND "organizationId" = ?
		`, [[104], organizationId]);
        await trx.raw(`DELETE FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.notification}"
			WHERE type = ANY (?::int[])
		`, [dal_constants_1.DalConstants.organizationVisitorModuleNotificationTypes]);
        await trx.raw(`
			UPDATE "${organizationId}".organizations
			SET "settings" = jsonb_set ("settings", '{notification,enabledTypes}', (REPLACE( REPLACE ((
				SELECT COALESCE(ARRAY_AGG(elem), '{}')
				FROM UNNEST(REPLACE( REPLACE ("settings"->'notification'->>'enabledTypes', '[', '{'), ']', '}')::int[] ) elem
				WHERE elem <> ALL(?::int[]))::text,'{','['),'}',']'))::jsonb, true)
			WHERE id = '${organizationId}';
		`, [dal_constants_1.DalConstants.organizationVisitorModuleNotificationTypes]);
    }
    async enableVisitorModule(organizationId, trx) {
        let qb = this.dbClient.withSchema(organizationId).from("organizationVisitorModuleSettings").where("organizationId", organizationId);
        if (trx)
            qb.transacting(trx);
        let qbz = this.dbClient.withSchema(organizationId).from("roles").where("organizationId", organizationId);
        if (trx)
            qbz = qbz.transacting(trx);
        let roles = await qbz.select();
        for (const role of roles) {
            role.permissions = JSON.parse(role.permissions);
            if (role.typeId == predefined_roles_1.PredefinedRoles.StandartUser.id || role.typeId == predefined_roles_1.PredefinedRoles.StandartUserOrganization.id) {
                if (role.permissions.indexOf(predefined_permissions_1.Permissions.visitor.getPreregister()) < 0) {
                    role.permissions.push(predefined_permissions_1.Permissions.visitor.getPreregister());
                    let qbr = this.dbClient.withSchema(organizationId).from("roles");
                    if (trx)
                        qbr = qbr.transacting(trx);
                    await qbr.where("id", role.id).update("permissions", JSON.stringify(role.permissions));
                }
            }
        }
        let settingsExists = await qb.first("id");
        if (!settingsExists) {
            let qbx = this.dbClient.withSchema(organizationId).from("organizations").where("id", organizationId);
            if (trx)
                qbx = qbx.transacting(trx);
            let availableCredentialTypes = await qbx.first("availableCredentialTypes");
            let qby = this.dbClient.withSchema(organizationId).from("organizationVisitorModuleSettings");
            let visitorProfileFormFields = [
                {
                    name: dal_access_models_1.DbAccessModel.PredefinedVisitorProfileFormFields.name,
                    caption: "Adı Soyadı",
                    order: 1,
                    extension: false,
                    type: dal_constants_1.DalConstants.FormFieldType.Text,
                    required: true,
                    placeholder: "Ziyaretçi Adı ve Soyadı",
                    regex: undefined,
                    options: {
                        maxLength: 255,
                        minLength: 3,
                        type: dal_constants_1.DalConstants.TextFormFieldType.TypeaheadForm,
                        searchType: dal_constants_1.DalConstants.FormTypeaheadSearchFilterOptionType.StartsWith,
                        unique: false,
                    },
                    widthLevel: 2,
                    reportable: true,
                    reportOrder: 2,
                },
                {
                    name: dal_access_models_1.DbAccessModel.PredefinedVisitorProfileFormFields.uniqueId,
                    caption: "T.C. Kimlik No / Pasaport No.",
                    order: 2,
                    extension: false,
                    type: dal_constants_1.DalConstants.FormFieldType.Text,
                    required: true,
                    regex: undefined,
                    options: {
                        maxLength: 255,
                        minLength: 3,
                        type: dal_constants_1.DalConstants.TextFormFieldType.TypeaheadForm,
                        searchType: dal_constants_1.DalConstants.FormTypeaheadSearchFilterOptionType.StartsWith,
                        unique: true,
                    },
                    widthLevel: 1,
                    reportable: true,
                    reportOrder: 3,
                },
                {
                    name: "licencePlate",
                    caption: "Plaka",
                    order: 3,
                    extension: true,
                    type: dal_constants_1.DalConstants.FormFieldType.Text,
                    required: false,
                    placeholder: "Araç Plaka",
                    options: {
                        maxLength: 255,
                        minLength: 3,
                        type: dal_constants_1.DalConstants.TextFormFieldType.Text,
                        unique: false,
                    },
                    widthLevel: 1,
                    reportable: true,
                    reportOrder: 4,
                },
                {
                    name: "company",
                    caption: "Şirket",
                    order: 5,
                    extension: true,
                    placeholder: "Şirket / Kurum",
                    type: dal_constants_1.DalConstants.FormFieldType.Text,
                    required: false,
                    options: {
                        maxLength: 255,
                        minLength: 1,
                        type: dal_constants_1.DalConstants.TextFormFieldType.TypeaheadField,
                        unique: false,
                    },
                    widthLevel: 1,
                    reportable: false,
                },
            ];
            let visitFormFields = [
                {
                    name: "reason",
                    caption: "Ziyaret Sebebi",
                    order: 1,
                    extension: true,
                    type: dal_constants_1.DalConstants.FormFieldType.Select,
                    required: true,
                    options: {
                        options: [
                            {
                                captionLines: ["BAKIM ONARIM"],
                                value: 1,
                            },
                            {
                                captionLines: ["EVRAK TESLİM ETME-ALMA"],
                                value: 2,
                            },
                            {
                                captionLines: ["EĞİTİM"],
                                value: 3,
                            },
                            {
                                captionLines: ["İHALE"],
                                value: 4,
                            },
                            {
                                captionLines: ["İNŞAAT"],
                                value: 5,
                            },
                            {
                                captionLines: ["İŞ GÖRÜŞMESİ"],
                                value: 6,
                            },
                            {
                                captionLines: ["İŞ TAKİBİ"],
                                value: 7,
                            },
                            {
                                captionLines: ["KARGO TESLİM ETME-ALMA"],
                                value: 8,
                            },
                            {
                                captionLines: ["MALZEME TESLİM ETME-ALMA"],
                                value: 9,
                            },
                            {
                                captionLines: ["ÖZEL ZİYARET"],
                                value: 10,
                            },
                            {
                                captionLines: ["STAJYER"],
                                value: 11,
                            },
                            {
                                captionLines: ["TEST"],
                                value: 12,
                            },
                            {
                                captionLines: ["TOPLANTI"],
                                value: 13,
                            },
                            {
                                captionLines: ["YEMEKHANE"],
                                value: 14,
                            },
                        ],
                        type: dal_constants_1.DalConstants.SelectFormFieldType.TypeaheadLocal,
                        searchType: dal_constants_1.DalConstants.FormTypeaheadSearchFilterOptionType.Similar,
                        multiSelect: false,
                    },
                    widthLevel: 1,
                    reportable: true,
                    reportOrder: 8,
                },
                {
                    name: dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.escort,
                    caption: "Refakatçı",
                    order: 2,
                    extension: false,
                    type: dal_constants_1.DalConstants.FormFieldType.Select,
                    required: false,
                    widthLevel: 1,
                    reportable: false,
                    options: {
                        options: [],
                        type: dal_constants_1.DalConstants.SelectFormFieldType.TypeaheadServerField,
                        searchType: dal_constants_1.DalConstants.FormTypeaheadSearchFilterOptionType.Similar,
                        formFieldSearchEndPoint: "/u/v1/" + organizationId + "/user/genericsearch",
                        multiSelect: false,
                        extraSearchFilterOptions: {
                            isDisabled: false,
                        },
                    },
                },
                {
                    name: dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.visitedPerson,
                    caption: "Ziyaret Edilen",
                    order: 3,
                    extension: false,
                    type: dal_constants_1.DalConstants.FormFieldType.Select,
                    required: false,
                    widthLevel: 1,
                    reportable: true,
                    options: {
                        options: [],
                        type: dal_constants_1.DalConstants.SelectFormFieldType.TypeaheadServerField,
                        searchType: dal_constants_1.DalConstants.FormTypeaheadSearchFilterOptionType.Similar,
                        formFieldSearchEndPoint: "/u/v1/" + organizationId + "/user/genericsearch",
                        multiSelect: false,
                        extraSearchFilterOptions: {
                            isDisabled: false,
                        },
                    },
                    reportOrder: 1,
                },
                {
                    name: dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.visitedOrganizationUnit,
                    caption: "Ziyaret Edilen Birim",
                    order: 4,
                    extension: false,
                    type: dal_constants_1.DalConstants.FormFieldType.Select,
                    required: false,
                    widthLevel: 1,
                    reportable: true,
                    options: {
                        options: [],
                        type: dal_constants_1.DalConstants.SelectFormFieldType.TypeaheadServerField,
                        searchType: dal_constants_1.DalConstants.FormTypeaheadSearchFilterOptionType.Similar,
                        formFieldSearchEndPoint: "/u/v1/" + organizationId + "/organizationUnit/genericsearch",
                        multiSelect: false,
                    },
                    reportOrder: 6,
                },
                {
                    name: dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.accessControlPoints,
                    caption: "Yetkili Geçiş Kontrol Noktaları",
                    order: 5,
                    extension: false,
                    type: dal_constants_1.DalConstants.FormFieldType.Select,
                    required: true,
                    widthLevel: 1,
                    reportable: true,
                    options: {
                        defaultValue: [],
                        type: dal_constants_1.DalConstants.SelectFormFieldType.TypeaheadServerField,
                        searchType: dal_constants_1.DalConstants.FormTypeaheadSearchFilterOptionType.Similar,
                        formFieldSearchEndPoint: "/u/v1/" + organizationId + "/accesscontrolpoint/genericsearch",
                        multiSelect: true,
                    },
                    reportOrder: 7,
                },
                {
                    name: dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.credentials,
                    caption: "Kimlik Bilgileri",
                    order: 6,
                    extension: false,
                    type: dal_constants_1.DalConstants.FormFieldType.Custom,
                    required: true,
                    options: null,
                    widthLevel: 2,
                    reportable: false,
                },
            ];
            let visitorProfileFormFieldSearchEndPoint = "/v/v1/" + organizationId + "/profile/fieldsearch";
            let visitorProfileFormSearchEndPoint = "/v/v1/" + organizationId + "/profile/formsearch";
            let visitFormFieldSearchEndPoint = "/v/v1/" + organizationId + "/visit/fieldsearch";
            let visitStates = [
                { state: dal_constants_1.DalConstants.OrganizationVisitorStates.Active, priority: 2 },
                { state: dal_constants_1.DalConstants.OrganizationVisitorStates.Expected, priority: 3 },
                { state: dal_constants_1.DalConstants.OrganizationVisitorStates.None, priority: 5 },
                { state: dal_constants_1.DalConstants.OrganizationVisitorStates.Forbidden, priority: 1 },
                { state: dal_constants_1.DalConstants.OrganizationVisitorStates.TemporaryPermit, priority: 4 },
            ];
            if (trx)
                qby = qby.transacting(trx);
            await qby.insert({
                id: uuid_1.default.v4(),
                organizationId: organizationId,
                settings: JSON.stringify({
                    visitorProfileFormFields: visitorProfileFormFields,
                    availableCredentials: availableCredentialTypes,
                    visitFormFields: visitFormFields,
                    visitorProfileFormFieldSearchEndPoint: visitorProfileFormFieldSearchEndPoint,
                    visitorProfileFormSearchEndPoint: visitorProfileFormSearchEndPoint,
                    visitFormFieldSearchEndPoint: visitFormFieldSearchEndPoint,
                    preRegistrationAvailable: true,
                    forceToSelectRegistrationPoint: false,
                    maxExpectedDayCount: null,
                    visitorStates: visitStates,
                    visitFormFinalizedEndPoints: null,
                    gdprEnabled: false,
                    printableVisitorCard: false,
                    qrCodeEnabled: false,
                    enableToForceRegistrationOfForbiddenVisitors: false,
                }),
            });
        }
        let visitorPoint = await trx.withSchema(organizationId).from(dal_db_armon_schema_1.ArmonSchema.tableNames.visitorRegistrationPoints).where("organizationId", organizationId).whereNull("deletedAt").first();
        if (!visitorPoint) {
            await trx.withSchema(organizationId).from(dal_db_armon_schema_1.ArmonSchema.tableNames.visitorRegistrationPoints).insert({
                id: uuid_1.default.v4(),
                organizationId: organizationId,
                createdAt: new Date(),
                updatedAt: new Date(),
                deletedAt: null,
                name: "Ziyaretçi Kabul Noktası",
            });
        }
        await trx.raw(`INSERT INTO public.${dal_db_armon_schema_1.ArmonSchema.tableNames.scheduled_job}(
			id, "createdT", "organizationId", type, "createdByUserId", enabled, "interval", "firstExecutionDate", "nextExecutionDate", note, "notificationId") 
			SELECT uuid_generate_v4(), now(), '${organizationId}', UNNEST(?::int[]), null, true, ${3}, 
				(select date_trunc('day', now()) + interval '1 day'), (select date_trunc('day', now()) + interval '1 day'), null, null
		`, [[104]]);
        await trx.raw(`INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.notification}"
				(id, "createdT", type, "receiverFilterId", email, sms, "pushNotification", web, "settings")
				VALUES (uuid_generate_v4(), now(), UNNEST (?::int[]), null, true, false, true, true, null)`, [dal_constants_1.DalConstants.organizationVisitorModuleNotificationTypes]);
        await trx.raw(`
			UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizations}"
			SET "settings" = jsonb_set ("settings", '{notification,enabledTypes}', 
										(REPLACE( REPLACE ((REPLACE( REPLACE ("settings"->'notification'->>'enabledTypes', '[', '{'), ']', '}')::int[] 
															|| ?::int[])::text,'{','['),'}',']'))::jsonb, true)
			WHERE id = '${organizationId}';
		`, [dal_constants_1.DalConstants.organizationVisitorModuleNotificationTypes]);
    }
    async deleteRemovedVisitorStates(params) {
        const { rowCount } = await params.trx.query(`
			DELETE FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorStates}"
			WHERE "deletedAt" IS NOT NULL
			`);
        return rowCount;
    }
    async deletePastExpectedVisits(params) {
        const { rowCount } = await params.trx.query(`
					DELETE FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationActiveVisits}"
					WHERE state = $1
					AND "expectedEndUtc" < $2
					AND "expectedStartUtc" IS NOT NULL
					AND "expectedEndUtc" IS NOT NULL`, [dal_constants_1.DalConstants.UnterminatedVisitState.Expected, params.date]);
        return rowCount;
    }
    async deleteExpectedVisit(organizationId, requestUserId, visitId) {
        let visit = await this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationActiveVisits)
            .where("id", visitId)
            .where("state", dal_constants_1.DalConstants.UnterminatedVisitState.Expected)
            .first();
        if (!visit)
            (0, dal_access_error_1.throwDbAccessNotFoundError)(i18n_1.default.__({ phrase: "ERRORS.VISITOR.EXPECTEDVISITNOTFOUND" }));
        await this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationActiveVisits).where("id", visitId).delete();
        await dal_manager_1.dbManager.accessLog.addUserActionHistoryItem({
            oId: organizationId,
            o: requestUserId,
            u: new Date(),
            c: dal_constants_1.DalConstants.UserActionCategory.Visitor,
            tg: ["Visit", "ExpectedVisit"],
            t: dal_constants_1.DalConstants.UserVisitorActionType.DeleteExpectedVisit,
            d: {
                visitId: visitId,
            },
        });
    }
    async removeRegistrationPoint(organizationId, visitorRegistrationPointId) {
        await this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.visitorRegistrationPoints).where("id", visitorRegistrationPointId).update({
            deletedAt: new Date(),
        });
    }
    async getRegistrationPointByDeviceId(organizationId, deviceId) {
        let visitorRegistrationPoint = await this.dbClient.withSchema(organizationId).from(dal_db_armon_schema_1.ArmonSchema.tableNames.visitorRegistrationPoints).where("tabletDeviceId", deviceId).first();
        if (!visitorRegistrationPoint || !visitorRegistrationPoint.id) {
            return Promise.resolve(null);
        }
        let accessControlPoints = await this.dbClient
            .withSchema(organizationId)
            .from("visitorRegistrationPointAccessControlPoints as vrap")
            .innerJoin("accessControlPoints as acp", "acp.id", "vrap.accessControlPointId")
            .whereNull("acp.deletedAt")
            .where("vrap.visitorRegistrationPointId", visitorRegistrationPoint.id)
            .select("acp.id", "acp.name");
        let device = null;
        if (visitorRegistrationPoint.tabletDeviceId) {
            device = await dal_manager_1.dbManager.accessDevice.getDeviceBasic(organizationId, visitorRegistrationPoint.tabletDeviceId);
        }
        return Promise.resolve({
            id: visitorRegistrationPoint.id,
            accessControlPoints: accessControlPoints,
            name: visitorRegistrationPoint.name,
            deviceName: device ? device.name : null,
            isConnected: device && device.health ? device.health.isConnected : false,
            tabletDeviceId: device ? device.id : null,
        });
    }
    async getRegistrationPoint(organizationId, visitorRegistrationPointId, trx) {
        let visitorRegistrationPointQuery = this.dbClient
            .withSchema(organizationId)
            .from(dal_db_armon_schema_1.ArmonSchema.tableNames.visitorRegistrationPoints)
            .where("id", visitorRegistrationPointId)
            .whereNull("deletedAt")
            .first();
        if (trx) {
            visitorRegistrationPointQuery.transacting(trx);
        }
        let visitorRegistrationPoint = await visitorRegistrationPointQuery;
        if (!visitorRegistrationPoint || !visitorRegistrationPoint.id) {
            (0, dal_access_error_1.throwDbAccessConflictErrorTr)("ERRORS.VISITOR.NOREGISTRATIONPOINT");
        }
        let accessControlPointsQuery = this.dbClient
            .withSchema(organizationId)
            .from("visitorRegistrationPointAccessControlPoints as vrap")
            .innerJoin("accessControlPoints as acp", "acp.id", "vrap.accessControlPointId")
            .whereNull("acp.deletedAt")
            .where("vrap.visitorRegistrationPointId", visitorRegistrationPoint.id)
            .select("acp.id", "acp.name");
        if (trx) {
            accessControlPointsQuery.transacting(trx);
        }
        let accessControlPoints = await accessControlPointsQuery;
        let device = null;
        if (visitorRegistrationPoint.tabletDeviceId) {
            device = await dal_manager_1.dbManager.accessDevice.getDeviceBasic(organizationId, visitorRegistrationPoint.tabletDeviceId);
        }
        return Promise.resolve({
            id: visitorRegistrationPoint.id,
            accessControlPoints: accessControlPoints,
            name: visitorRegistrationPoint.name,
            deviceName: device ? device.name : null,
            isConnected: device && device.health ? device.health.isConnected : false,
            tabletDeviceId: device ? device.id : null,
        });
    }
    async listRegistrationPointsDetailed(organizationId, requestUserId, trx) {
        let items = [];
        if (!organizationId) {
            return items;
        }
        let relatedOrganizationUnitIds = [];
        const qFunction = async (trx) => {
            relatedOrganizationUnitIds = await dal_manager_1.dbManager.accessOrganizationUnit.getAncestorAndChildOrganizationUnitIds(organizationId, requestUserId, trx);
            const qParams = [];
            qParams.push(organizationId);
            if (relatedOrganizationUnitIds.length) {
                qParams.push(relatedOrganizationUnitIds);
            }
            const registrationPointsDbResult = await trx.query(`
				SELECT 
					vrp."id", 
					vrp."name", 
					vrp."tabletDeviceId", 
					acp.id as "accessControlPointId", 
					acp.name as "accessControlPointName", 
					d.name as "deviceName", 
					d."health"
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.visitorRegistrationPoints}" AS vrp
				LEFT JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.visitorRegistrationPointAccessControlPoints}" AS vrpacp
					ON vrpacp."visitorRegistrationPointId" = vrp."id" 
						AND vrp."deletedAt" IS NULL
				LEFT JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessControlPoints}" AS acp
					ON vrpacp."accessControlPointId" = acp.id
						AND acp."deletedAt" IS NULL
				LEFT JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.devices}" AS d
					ON vrp."tabletDeviceId" = d.id
						AND d."deletedAt" IS NULL
				LEFT JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" AS ou
					ON vrp."organizationUnitId" = ou.id
						AND ou."deletedAt" IS NULL
				WHERE vrp."organizationId" = $1 AND vrp."deletedAt" IS NULL
				${relatedOrganizationUnitIds.length ? ` AND (ou.id = ANY($2::UUID[]) OR ou.id IS NULL) ` : ``}
				ORDER BY vrp.name
			`, qParams);
            for (const row of registrationPointsDbResult.rows) {
                if (!items.some((s) => s.id === row.id)) {
                    items.push({
                        id: row.id,
                        name: row.name,
                        tabletDeviceId: row.tabletDeviceId,
                        accessControlPoints: registrationPointsDbResult.rows
                            .filter((r) => r.id === row.id && row.accessControlPointId)
                            .map((s) => {
                            return {
                                name: s.accessControlPointName,
                                id: s.accessControlPointId,
                            };
                        }),
                        deviceName: row.deviceName,
                        isConnected: row.health?.isConnected ?? false,
                    });
                }
            }
            return Promise.resolve(items);
        };
        if (trx) {
            return await qFunction(trx);
        }
        else {
            return await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                return qFunction(trx);
            });
        }
    }
    async listRegistrationPoints(organizationId, args, requestUserId) {
        if (!organizationId) {
            return {
                total: 0,
                items: [],
            };
        }
        let relatedOrganizationUnitIds = [];
        await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
            relatedOrganizationUnitIds = await dal_manager_1.dbManager.accessOrganizationUnit.getAncestorAndChildOrganizationUnitIds(organizationId, requestUserId, trx);
        }, requestUserId, organizationId);
        let qb = this.dbClient
            .withSchema(organizationId)
            .from("visitorRegistrationPoints as vrp")
            .leftJoin("visitorRegistrationPointAccessControlPoints as vrap", "vrap.visitorRegistrationPointId", "vrp.id")
            .leftJoin("accessControlPoints as acp", "acp.id", "vrap.accessControlPointId")
            .leftJoin("organizationUnits as ou", "ou.id", "vrp.organizationUnitId")
            .whereNull("vrp.deletedAt")
            .whereNull("acp.deletedAt")
            .whereNull("ou.deletedAt")
            .where("vrp.organizationId", organizationId)
            .where(function () {
            if (relatedOrganizationUnitIds.length > 0) {
                this.whereIn("ou.id", relatedOrganizationUnitIds).orWhereNull("ou.id");
            }
        });
        if (args.filter)
            qb.where("vrp.name", "ilike", args.filter + "%");
        let total = await qb
            .clone()
            .count()
            .first()
            .then((row) => parseInt(row.count));
        if (args.pagination.skip)
            qb.offset(args.pagination.skip);
        if (args.pagination.take)
            qb.limit(args.pagination.take);
        let items = [];
        await qb
            .orderBy("vrp.name")
            .select("vrp.id", "vrp.name", "vrp.tabletDeviceId", "acp.id as accessControlPointId", "acp.name as accessControlPointName")
            .then((rows) => {
            for (const row of rows) {
                if (!items.some((s) => s.id === row.id)) {
                    items.push({
                        id: row.id,
                        name: row.name,
                        tabletDeviceId: row.tabletDeviceId,
                        accessControlPoints: rows
                            .filter((r) => r.id === row.id)
                            .map((s) => {
                            return {
                                name: s.accessControlPointName,
                                id: s.accessControlPointId,
                            };
                        }),
                    });
                }
            }
        });
        return Promise.resolve({
            total: total,
            items: items,
        });
    }
    async upsertRegistrationPoint(organizationId, requestUserId, args) {
        return await this.dbClient.transaction(async (trx) => {
            let id = args.id || uuid_1.default.v4();
            if (args.id) {
                await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.visitorRegistrationPoints).where("id", args.id).update({
                    name: args.name,
                    updatedAt: new Date(),
                });
                await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.visitorRegistrationPointAccessControlPoints).where("visitorRegistrationPointId", args.id).delete();
            }
            else {
                await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.visitorRegistrationPoints).insert({
                    id: id,
                    name: args.name,
                    organizationId: organizationId,
                    createdAt: new Date(),
                    updatedAt: new Date(),
                    deletedAt: null,
                });
            }
            for (let accessControlPointId of args.accessControlPointIds) {
                await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.visitorRegistrationPointAccessControlPoints).insert({
                    id: uuid_1.default.v4(),
                    visitorRegistrationPointId: id,
                    accessControlPointId: accessControlPointId,
                });
            }
            await dal_manager_1.dbManager.accessLog.addUserActionHistoryItem({
                oId: organizationId,
                o: requestUserId,
                u: new Date(),
                c: dal_constants_1.DalConstants.UserActionCategory.Visitor,
                tg: ["Visitor", "VisitorRegistrationPoint"],
                t: dal_constants_1.DalConstants.UserVisitorActionType.UpsertVisitorRegistrationPoint,
                d: {
                    visitorRegistrationPointId: id,
                },
            });
            return id;
        });
    }
    async upsertVisitorState(organizationId, requestUserId, visitorProfileId, args) {
        let now = new Date();
        let existingState = await this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorStates)
            .whereNull("deletedAt")
            .where("state", args.state)
            .where("organizationVisitorProfileId", visitorProfileId)
            .first();
        let id = uuid_1.default.v4() || existingState.id;
        if (existingState && existingState.id) {
            await this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorStates).where("id", existingState.id).update({
                startUtc: args.startUtc,
                endUtc: args.endUtc,
                note: args.note,
                createdAt: now,
                createdBy: requestUserId,
            });
        }
        else {
            await this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorStates).insert({
                id: id,
                organizationVisitorProfileId: visitorProfileId,
                state: args.state,
                startUtc: args.startUtc,
                endUtc: args.endUtc,
                note: args.note,
                createdAt: now,
                createdBy: requestUserId,
                deletedAt: null,
                deletedBy: null,
            });
        }
        await dal_manager_1.dbManager.accessLog.addUserActionHistoryItem({
            oId: organizationId,
            o: requestUserId,
            u: new Date(),
            c: dal_constants_1.DalConstants.UserActionCategory.Visitor,
            tg: ["Visitor", "VisitorState"],
            t: args.state == dal_constants_1.DalConstants.OrganizationVisitorStates.Forbidden
                ? dal_constants_1.DalConstants.UserVisitorActionType.SetVisitorAsBlacklisted
                : dal_constants_1.DalConstants.UserVisitorActionType.SetVisitorAsTemporaryPermitted,
            d: {
                state: args.state,
                visitorProfileId: visitorProfileId,
                startUtc: args.startUtc,
                endUtc: args.endUtc,
            },
        });
        return Promise.resolve(id);
    }
    async removeVisitorState(organizationId, requestUserId, visitorProfileId, state) {
        let now = new Date();
        let existingState = await this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorStates)
            .whereNull("deletedAt")
            .where("state", state)
            .where("organizationVisitorProfileId", visitorProfileId)
            .first();
        if (!existingState || !existingState.id) {
            (0, dal_access_error_1.throwDbAccessConflictErrorTr)("ERRORS.VISITOR.NOVISITORSTATE");
        }
        await dal_manager_1.dbManager.accessLog.addUserActionHistoryItem({
            oId: organizationId,
            o: requestUserId,
            u: new Date(),
            c: dal_constants_1.DalConstants.UserActionCategory.Visitor,
            tg: ["Visitor", "UpsertVisitorState"],
            t: state == dal_constants_1.DalConstants.OrganizationVisitorStates.Forbidden
                ? dal_constants_1.DalConstants.UserVisitorActionType.UnsetVisitorAsBlacklisted
                : dal_constants_1.DalConstants.UserVisitorActionType.UnsetVisitorAsTemporaryPermitted,
            d: {
                state: state,
                visitorProfileId: visitorProfileId,
            },
        });
        await this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorStates).where("id", existingState.id).update({
            deletedAt: now,
            deletedBy: requestUserId,
        });
    }
    async listVisitors(organizationId, args, visitorModuleSettings) {
        let result = {
            pagination: {
                take: args.take,
                skip: args.skip,
                total: 0,
            },
            items: [],
        };
        let qb = this.dbClient
            .withSchema(organizationId)
            .from("organizationVisitorProfiles as ovp")
            .innerJoin("userOrganizations as uo", (join) => {
            join.on("ovp.userOrganizationId", "uo.id");
        });
        qb.where("uo.organizationId", organizationId).whereNull("uo.deletedAt").whereNull("ovp.deletedAt");
        if (args.visitorProfileId) {
            qb.where("ovp.id", args.visitorProfileId);
        }
        else {
            for (const filterField of args.visitorProfileFilterFields) {
                let field = visitorModuleSettings.visitorProfileFormFields.find((f) => f.name === filterField.name);
                if (field.extension) {
                    let fieldName = "ovp.";
                    fieldName += `"extensionFields"->>'` + filterField.name + "'";
                    if (field.type == dal_constants_1.DalConstants.FormFieldType.Text) {
                        let textOptions = field.options;
                        if (textOptions && textOptions.searchType && textOptions.searchType == dal_constants_1.DalConstants.FormTypeaheadSearchFilterOptionType.StartsWith) {
                            qb.whereRaw("unaccent(" + fieldName + ")::TEXT ilike unaccent(?)", filterField.value + "%");
                        }
                        else {
                            qb.whereRaw("unaccent(" + fieldName + ")::TEXT ilike unaccent(?)", "%" + filterField.value + "%");
                        }
                    }
                    else {
                        qb.whereRaw("unaccent(" + fieldName + ")::TEXT ilike unaccent(?)", "%" + filterField.value + "%");
                    }
                }
                else {
                    let fieldName = "ovp.";
                    fieldName += '"' + filterField.name + '"';
                    if (field.type == dal_constants_1.DalConstants.FormFieldType.Text) {
                        let textOptions = field.options;
                        if (textOptions && textOptions.searchType && textOptions.searchType == dal_constants_1.DalConstants.FormTypeaheadSearchFilterOptionType.StartsWith) {
                            qb.whereRaw("unaccent(" + fieldName + ")::TEXT ilike unaccent(?)", filterField.value + "%");
                        }
                        else {
                            qb.whereRaw("unaccent(" + fieldName + ")::TEXT ilike unaccent(?)", "%" + filterField.value + "%");
                        }
                    }
                    else {
                        qb.whereRaw("unaccent(" + fieldName + ")::TEXT ilike unaccent(?)", "%" + filterField.value + "%");
                    }
                }
            }
            let now = new Date();
            if (args.showOnlyBlacklist || args.showOnlyTemporaryPermitted) {
                let states = [];
                if (args.showOnlyBlacklist)
                    states.push(dal_constants_1.DalConstants.OrganizationVisitorStates.Forbidden);
                if (args.showOnlyTemporaryPermitted)
                    states.push(dal_constants_1.DalConstants.OrganizationVisitorStates.TemporaryPermit);
                qb.innerJoin("organizationVisitorStates as ovs", "ovs.organizationVisitorProfileId", "ovp.id")
                    .whereIn("ovs.state", states)
                    .whereNull("ovs.deletedAt")
                    .where("ovs.startUtc", "<", now)
                    .where(function () {
                    this.where("ovs.endUtc", ">", now).orWhereNull("ovs.endUtc");
                });
            }
        }
        result.pagination.total = await qb
            .clone()
            .count("* as c")
            .then((rows) => {
            return rows && rows.length > 0 ? parseInt(rows[0].c) : 0;
        });
        if (args.skip)
            qb.offset(args.skip);
        if (args.take)
            qb.limit(args.take);
        let columns = ["ovp.id"];
        for (const visitorProfileField of visitorModuleSettings.visitorProfileFormFields) {
            if (visitorProfileField.extension) {
                columns.push(this.dbClient.raw(`"ovp"."extensionFields"->>'` + visitorProfileField.name + `' as "ovp.` + visitorProfileField.name + `"`));
            }
            else {
                columns.push(this.dbClient.raw(`"ovp"."` + visitorProfileField.name + `" as "ovp.` + visitorProfileField.name + `"`));
            }
        }
        let visitors = await qb.orderBy("ovp.name").select(columns);
        let visitorProfileIds = visitors.map((v) => v.id);
        let visitStates = (await this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationActiveVisits)
            .whereIn("organizationVisitorProfileId", visitorProfileIds)
            .select("organizationVisitorProfileId", "state"));
        let customStates = (await this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorStates)
            .whereNull("deletedAt")
            .where("startUtc", "<=", new Date())
            .where(function () {
            this.where("endUtc", ">=", new Date()).orWhereNull("endUtc");
        })
            .whereIn("organizationVisitorProfileId", visitorProfileIds)
            .select("organizationVisitorProfileId", "state", "startUtc", "endUtc", "note"));
        for (const visitor of visitors) {
            let visitorFields = this.fillVisitorFormFieldFromDataRow(visitor, visitorModuleSettings);
            let visitStatesForVisitor = visitStates.filter((v) => v.organizationVisitorProfileId === visitor.id);
            let customStatesForVisitor = customStates.filter((v) => v.organizationVisitorProfileId === visitor.id);
            let visitorStates = [];
            for (const visitState of visitStatesForVisitor) {
                visitorStates.push({
                    state: visitState.state,
                });
            }
            for (const customState of customStatesForVisitor) {
                visitorStates.push({
                    state: customState.state,
                    note: customState.note,
                    startUtc: customState.startUtc.toISOString() == dal_constants_1.DalConstants.minDate.toISOString() ? null : customState.startUtc,
                    endUtc: customState.endUtc,
                });
            }
            visitorStates.sort((x, y) => visitorModuleSettings.visitorStates.find((f) => f.state == x.state).priority < visitorModuleSettings.visitorStates.find((f) => f.state == y.state).priority ? -1 : 1);
            result.items.push({
                id: visitor.id,
                fields: visitorFields,
                states: visitorStates,
            });
        }
        return Promise.resolve(result);
    }
    async getVisitor(organizationId, organizationVisitorProfileId, visitorModuleSettings) {
        let qb = this.dbClient
            .withSchema(organizationId)
            .from("organizationVisitorProfiles as ovp")
            .innerJoin("userOrganizations as uo", (join) => {
            join.on("ovp.userOrganizationId", "uo.id");
        })
            .where("uo.organizationId", organizationId)
            .whereNull("uo.deletedAt")
            .whereNull("ovp.deletedAt")
            .where("ovp.id", organizationVisitorProfileId);
        let columns = ["ovp.id"];
        for (const visitorProfileField of visitorModuleSettings.visitorProfileFormFields) {
            if (visitorProfileField.extension) {
                columns.push(this.dbClient.raw(`"ovp"."extensionFields"->>'` + visitorProfileField.name + `' as "ovp.` + visitorProfileField.name + `"`));
            }
            else {
                columns.push(this.dbClient.raw(`"ovp"."` + visitorProfileField.name + `" as "ovp.` + visitorProfileField.name + `"`));
            }
        }
        let visitor = await qb.first(columns);
        if (!visitor || !visitor.id) {
            (0, dal_access_error_1.throwDbAccessNotFoundErrorTr)("ERRORS.VISITOR.VISITORNOTFOUND");
        }
        let visitorFields = this.fillVisitorFormFieldFromDataRow(visitor, visitorModuleSettings);
        let visitStates = (await this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationActiveVisits)
            .where("organizationVisitorProfileId", visitor.id)
            .select("organizationVisitorProfileId", "state"));
        let customStates = (await this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorStates)
            .whereNull("deletedAt")
            .where("organizationVisitorProfileId", visitor.id)
            .where("startUtc", "<=", new Date())
            .where(function () {
            this.where("endUtc", ">=", new Date()).orWhereNull("endUtc");
        })
            .select("organizationVisitorProfileId", "state", "startUtc", "endUtc", "note"));
        let visitorStates = [];
        for (const visitState of visitStates) {
            visitorStates.push({
                state: visitState.state,
            });
        }
        for (const customState of customStates) {
            visitorStates.push({
                state: customState.state,
                note: customState.note,
                startUtc: customState.startUtc.toISOString() == dal_constants_1.DalConstants.minDate.toISOString() ? null : customState.startUtc,
                endUtc: customState.endUtc,
            });
        }
        visitorStates.sort((x, y) => visitorModuleSettings.visitorStates.find((f) => f.state == x.state).priority < visitorModuleSettings.visitorStates.find((f) => f.state == y.state).priority ? -1 : 1);
        let result = {
            id: visitor.id,
            fields: visitorFields,
            states: visitorStates,
        };
        return Promise.resolve(result);
    }
    async getVisitorState(organizationId, visitorProfileId, visitorModuleSettings, trx) {
        let now = new Date();
        const fn = async (trx) => {
            let visitStates = (await this.dbClient
                .withSchema(organizationId)
                .table(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationActiveVisits)
                .where("organizationVisitorProfileId", visitorProfileId)
                .transacting(trx)
                .select("organizationVisitorProfileId", "state"));
            let customStates = (await this.dbClient
                .withSchema(organizationId)
                .table(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorStates)
                .whereNull("deletedAt")
                .where("startUtc", "<", now)
                .where(function () {
                this.where("endUtc", ">=", new Date()).orWhereNull("endUtc");
            })
                .where("organizationVisitorProfileId", visitorProfileId)
                .transacting(trx)
                .select("organizationVisitorProfileId", "state", "startUtc", "endUtc", "note"));
            let visitorStates = [];
            for (const visitState of visitStates) {
                visitorStates.push({
                    state: visitState.state,
                });
            }
            for (const customState of customStates) {
                visitorStates.push({
                    state: customState.state,
                    note: customState.note,
                    startUtc: customState.startUtc.toISOString() == dal_constants_1.DalConstants.minDate.toISOString() ? null : customState.startUtc,
                    endUtc: customState.endUtc,
                });
            }
            visitorStates.sort((x, y) => visitorModuleSettings.visitorStates.find((f) => f.state == x.state).priority < visitorModuleSettings.visitorStates.find((f) => f.state == y.state).priority ? -1 : 1);
            return visitorStates;
        };
        if (trx) {
            return fn(trx);
        }
        else {
            return this.dbClient.transaction(fn);
        }
    }
    async listVisitorStates(organizationId, visitorProfileIds, visitorModuleSettings) {
        let now = new Date();
        let visitStates = (await this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationActiveVisits)
            .whereIn("organizationVisitorProfileId", visitorProfileIds)
            .select("organizationVisitorProfileId", "state"));
        let customStates = (await this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorStates)
            .whereNull("deletedAt")
            .where("startUtc", "<", now)
            .where(function () {
            this.where("endUtc", ">=", new Date()).orWhereNull("endUtc");
        })
            .whereIn("organizationVisitorProfileId", visitorProfileIds)
            .select("organizationVisitorProfileId", "state", "startUtc", "endUtc", "note"));
        let items = [];
        for (const visitor of visitorProfileIds) {
            let visitorStates = [];
            for (const visitState of visitStates.filter((v) => v.organizationVisitorProfileId == visitor)) {
                visitorStates.push({
                    state: visitState.state,
                });
            }
            for (const customState of customStates.filter((v) => v.organizationVisitorProfileId == visitor)) {
                visitorStates.push({
                    state: customState.state,
                    note: customState.note,
                    startUtc: customState.startUtc.toISOString() == dal_constants_1.DalConstants.minDate.toISOString() ? null : customState.startUtc,
                    endUtc: customState.endUtc,
                });
            }
            visitorStates.sort((x, y) => visitorModuleSettings.visitorStates.find((f) => f.state == x.state).priority < visitorModuleSettings.visitorStates.find((f) => f.state == y.state).priority ? -1 : 1);
            items.push({
                visitorProfileId: visitor,
                states: visitorStates,
            });
        }
        return Promise.resolve(items);
    }
    async getVisitorDashboardInfo(organizationId, requestUserId, date, visitorRegistrationPointId) {
        let result = {
            date: date,
            totalActiveVisit: 0,
            totalTerminatedVisit: 0,
            registrationPoints: [],
        };
        let startDate = (0, moment_1.default)(date).startOf("day").toDate();
        let endDate = (0, moment_1.default)(date).endOf("day").toDate();
        let activeQb = this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationActiveVisits)
            .where("startUtc", "<=", endDate)
            .where("startUtc", ">=", startDate)
            .where("organizationId", organizationId);
        let passiveQb = this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.visitorDailySummaries)
            .where("organizationId", organizationId)
            .where("date", ">=", startDate)
            .where("date", "<=", endDate);
        if (visitorRegistrationPointId) {
            activeQb = activeQb.where("visitorRegistrationPointId", visitorRegistrationPointId);
            passiveQb = passiveQb.where("visitorRegistrationPointId", visitorRegistrationPointId);
        }
        let passiveVisits = await passiveQb.groupBy("visitorRegistrationPointId").sum("count as count").select("visitorRegistrationPointId");
        let activeVisits = await activeQb.groupBy("visitorRegistrationPointId").count("*").select("visitorRegistrationPointId");
        let visitorRegistrationPoints = await this.listRegistrationPoints(organizationId, { pagination: { take: 1000, skip: 0 } }, requestUserId);
        if (visitorRegistrationPointId) {
            result.totalActiveVisit = activeVisits.reduce((a, b) => {
                return a + parseInt(b.count);
            }, 0);
            result.totalTerminatedVisit = passiveVisits.reduce((a, b) => {
                return a + parseInt(b.count);
            }, 0);
            let item = {
                id: visitorRegistrationPointId,
                name: visitorRegistrationPoints.items.find((v) => v.id == visitorRegistrationPointId).name,
                totalActiveVisit: result.totalActiveVisit,
                totalTerminatedVisit: result.totalTerminatedVisit,
            };
            result.registrationPoints.push(item);
        }
        else {
            result.totalActiveVisit = activeVisits.reduce((a, b) => {
                return a + parseInt(b.count);
            }, 0);
            result.totalTerminatedVisit = passiveVisits.reduce((a, b) => {
                return a + parseInt(b.count);
            }, 0);
            for (const registrationPoint of visitorRegistrationPoints.items) {
                let active = activeVisits.find((a) => a.visitorRegistrationPointId == registrationPoint.id);
                let passive = passiveVisits.find((a) => a.visitorRegistrationPointId == registrationPoint.id);
                let item = {
                    id: registrationPoint.id,
                    name: registrationPoint.name,
                    totalActiveVisit: active ? active.count : 0,
                    totalTerminatedVisit: passive ? passive.count : 0,
                };
                result.registrationPoints.push(item);
            }
        }
        return Promise.resolve(result);
    }
    async getVisitorDetailedDashboardInfo(organizationId, requestUserId, visitorProfileId) {
        let settings = await this.getVisitAndVisitorFormSettings(organizationId);
        let totalVisits = await dal_manager_1.dbManager.accessLog.getVisitorTotalVisitCounts(organizationId, [visitorProfileId]);
        let visitorProfile = await this.getVisitorDetailed(organizationId, visitorProfileId, settings);
        let totalVisitCount = totalVisits.find((v) => v.visitorProfileId == visitorProfileId);
        let visitorBasicInfo = await dal_manager_1.dbManager.accessUser.getBasicVisitorInfoWithRoleId(organizationId, visitorProfileId);
        let result = {
            profile: visitorProfile.profile,
            lastVisit: visitorProfile.lastVisit,
            totalVisitCount: totalVisitCount ? totalVisitCount.totalVisitCount : 0,
            userCaptions: visitorBasicInfo.userCaptions,
        };
        return Promise.resolve(result);
    }
    async listActiveVisitsForReport(organizationId, options, visitorRegistrationPointId, visitorModuleSettings, userId) {
        let result = {
            items: [],
            options: options,
            total: 0,
        };
        let columns = [
            { "oav.extensionFields": "oav.extensionFields" },
            { "oav.startUtc": "oav.startUtc" },
            { "ovp.id": "ovp.id" },
            { "oav.id": "oav.id" },
            { "oav.visitorRegistrationPointId": "oav.visitorRegistrationPointId" },
            { "uo.userId": "uo.userId" },
        ];
        columns = columns.concat(this.prepareVisitorProfileColumns(visitorModuleSettings));
        let qb = this.dbClient
            .withSchema(organizationId)
            .from("organizationActiveVisits as oav")
            .innerJoin("organizationVisitorProfiles as ovp", (join) => {
            join.on("ovp.id", "oav.organizationVisitorProfileId");
        })
            .innerJoin("userOrganizations as uo", (join) => {
            join.on("ovp.userOrganizationId", "uo.id");
        })
            .leftJoin("organizationUnits as ou", (join) => {
            join.on("oav.visitedOrganizationUnitId", "ou.id");
        })
            .leftJoin("visitorRegistrationPoints as vrp", (join) => {
            join.on("oav.visitorRegistrationPointId", "vrp.id");
        })
            .where("oav.organizationId", organizationId)
            .where("oav.state", dal_constants_1.DalConstants.UnterminatedVisitState.Active)
            .where("oav.startUtc", ">=", options.startUtc)
            .where("oav.startUtc", "<=", options.endUtc)
            .whereNull("vrp.deletedAt");
        let visitJoins = this.prepareVisitJoinsForVisitorDetailed(organizationId, visitorModuleSettings, qb);
        qb = visitJoins.qb;
        columns = columns.concat(visitJoins.columns);
        if (visitorRegistrationPointId)
            qb.where("oav.visitorRegistrationPointId", visitorRegistrationPointId);
        if (options.visitorProfileIds && options.visitorProfileIds.length > 0) {
            qb.whereIn("ovp.id", options.visitorProfileIds);
        }
        if (options.visitorProfileFilterFields) {
            for (const filterField of options.visitorProfileFilterFields) {
                let fieldDef = visitorModuleSettings.visitorProfileFormFields.find((f) => f.name == filterField.name);
                if (!fieldDef)
                    continue;
                if (fieldDef.extension) {
                    qb.where(this.dbClient.raw(`upper(unaccent(ovp."extensionFields"->>'${filterField.name}')) ilike upper(unaccent('%${filterField.value}%'))`));
                }
                else {
                    if (fieldDef.name === dal_access_models_1.DbAccessModel.PredefinedVisitorProfileFormFields.name && options.visitorProfileIds?.length) {
                        continue;
                    }
                    else {
                        qb.where(this.dbClient.raw(`upper(unaccent(ovp."${filterField.name}")) ilike upper(unaccent('%${filterField.value}%'))`));
                    }
                }
            }
        }
        if (options.visitFilterFields) {
            for (const filterField of options.visitFilterFields) {
                let visitFormField = visitorModuleSettings.visitFormFields.find((f) => f.name == filterField.name);
                if (!visitFormField)
                    continue;
                if (visitFormField.extension) {
                    if (visitFormField.type == dal_constants_1.DalConstants.FormFieldType.Select) {
                        qb.where(this.dbClient.raw(`oav."extensionFields"->>'${filterField.name}' = '${filterField.value}'`));
                    }
                    else {
                        qb.where(this.dbClient.raw(`unaccent(oav."extensionFields"->>'${filterField.name}') ilike unaccent('%${filterField.value}%')`));
                    }
                }
                else {
                    switch (visitFormField.name) {
                        case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.accessControlPoints:
                            break;
                        case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.escort:
                            qb.andWhere(this.dbClient.raw(`unaccent(oav."escortPerson"->>'name') ilike unaccent('${filterField.value}%)'
                                or unaccent(oav."escortPerson"->>'surname') ilike unaccent('${filterField.value}%)'
                                or unaccent(oav."escortPerson"->>'uniqueId') ilike unaccent('${filterField.value}%')`));
                            break;
                        case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.visitedOrganizationUnit:
                            qb.where(this.dbClient.raw(`unaccent(ou.name) ilike unaccent('${filterField.value}%')`));
                            break;
                        case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.visitedPerson:
                            if ((0, dal_utils_1.isUUID)(filterField.value)) {
                                qb.andWhere(this.dbClient.raw(`vuop."userId" = '${filterField.value}'`));
                            }
                            else {
                                qb.andWhere(this.dbClient.raw(`unaccent(vuop.name || ' ' ||vuop.surname) ilike unaccent('%${filterField.value}%')`));
                            }
                            break;
                        default:
                            qb.where(this.dbClient.raw(`unaccent(oav."${filterField.name}") ilike unaccent('%${filterField.value}%')`));
                            break;
                    }
                }
            }
        }
        let relatedOrganizationUnitIds = [];
        await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
            relatedOrganizationUnitIds = await dal_manager_1.dbManager.accessOrganizationUnit.getAncestorAndChildOrganizationUnitIds(organizationId, userId, trx);
        }, userId, organizationId);
        qb.where(function () {
            if (relatedOrganizationUnitIds.length > 0) {
                this.whereIn("vrp.organizationUnitId", relatedOrganizationUnitIds).orWhereNull("vrp.organizationUnitId");
            }
        });
        result.total = await qb
            .clone()
            .count()
            .then((c) => {
            return parseInt(c[0].count);
        });
        if (options.skip)
            qb.offset(options.skip);
        if (options.take)
            qb.limit(options.take);
        let activeVisits = (await qb.columns(columns).orderBy("oav.startUtc", "desc").select());
        for (const visit of activeVisits) {
            let profileFields = {};
            for (const visitorProfileField of visitorModuleSettings.visitorProfileFormFields) {
                profileFields[visitorProfileField.name] = visit["ovp." + visitorProfileField.name] ? visit["ovp." + visitorProfileField.name] : null;
            }
            let userId = visit["uo.userId"];
            let profileId = visit["ovp.id"];
            let startUtc = visit["oav.startUtc"];
            let visitorRegistrationPointId = visit["oav.visitorRegistrationPointId"];
            let visitFields = {};
            for (const visitFormField of visitorModuleSettings.visitFormFields) {
                if (visitFormField.extension) {
                    visitFields[visitFormField.name] = visit["oav.extensionFields"][visitFormField.name] ? visit["oav.extensionFields"][visitFormField.name] : null;
                }
                else {
                    if (visitFormField.name === dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.expectedDate) {
                        visitFields[visitFormField.name] = {
                            startDate: visit["oav.expectedStartUtc"],
                            endDate: visit["oav.expectedEndUtc"],
                        };
                    }
                    else {
                        visitFields[visitFormField.name] = visit["oav." + visitFormField.name] ? visit["oav." + visitFormField.name] : null;
                    }
                }
            }
            visitFields["visitorRegistrationPointId"] = visitorRegistrationPointId;
            result.items.push({
                visitEndUtc: null,
                visitFields: visitFields,
                visitId: visit["oav.id"],
                visitStartUtc: startUtc,
                visitorProfileFields: profileFields,
                visitorProfileId: profileId,
                visitorUserId: userId,
            });
        }
        return Promise.resolve(result);
    }
    async getCountOfActiveVisitsForReport(organizationId, options, visitorRegistrationPointId, visitorModuleSettings) {
        let columns = [
            { "oav.extensionFields": "oav.extensionFields" },
            { "oav.startUtc": "oav.startUtc" },
            { "ovp.id": "ovp.id" },
            { "oav.id": "oav.id" },
            { "oav.visitorRegistrationPointId": "oav.visitorRegistrationPointId" },
            { "uo.userId": "uo.userId" },
        ];
        columns = columns.concat(this.prepareVisitorProfileColumns(visitorModuleSettings));
        let qb = this.dbClient
            .withSchema(organizationId)
            .from("organizationActiveVisits as oav")
            .innerJoin("organizationVisitorProfiles as ovp", (join) => {
            join.on("ovp.id", "oav.organizationVisitorProfileId");
        })
            .innerJoin("userOrganizations as uo", (join) => {
            join.on("ovp.userOrganizationId", "uo.id");
        })
            .where("oav.organizationId", organizationId)
            .where("oav.state", dal_constants_1.DalConstants.UnterminatedVisitState.Active)
            .where("oav.startUtc", ">=", options.startUtc)
            .where("oav.startUtc", "<=", options.endUtc);
        let visitJoins = this.prepareVisitJoinsForVisitorDetailed(organizationId, visitorModuleSettings, qb);
        qb = visitJoins.qb;
        columns = columns.concat(visitJoins.columns);
        if (visitorRegistrationPointId)
            qb.where("oav.visitorRegistrationPointId", visitorRegistrationPointId);
        if (options.visitorProfileIds && options.visitorProfileIds.length > 0) {
            qb.whereIn("ovp.id", options.visitorProfileIds);
        }
        if (options.visitorProfileFilterFields) {
            for (const filterField of options.visitorProfileFilterFields) {
                let fieldDef = visitorModuleSettings.visitorProfileFormFields.find((f) => f.name == filterField.name);
                if (!fieldDef)
                    continue;
                if (fieldDef.extension) {
                    qb.where(this.dbClient.raw(`unaccent(ovp."extensionFields"->>'${filterField.name}') ilike unaccent('%${filterField.value}%')`));
                }
                else {
                    qb.where(this.dbClient.raw(`unaccent(ovp."${filterField.name}") ilike unaccent('%${filterField.value}%')`));
                }
            }
        }
        if (options.visitFilterFields) {
            for (const filterField of options.visitFilterFields) {
                let visitFormField = visitorModuleSettings.visitFormFields.find((f) => f.name == filterField.name);
                if (!visitFormField)
                    continue;
                if (visitFormField.extension) {
                    if (visitFormField.type == dal_constants_1.DalConstants.FormFieldType.Select) {
                        qb.where(this.dbClient.raw(`oav."extensionFields"->>'${filterField.name}' = '${filterField.value}'`));
                    }
                    else {
                        qb.where(this.dbClient.raw(`unaccent(oav."extensionFields"->>'${filterField.name}') ilike unaccent('%${filterField.value}%')`));
                    }
                }
                else {
                    switch (visitFormField.name) {
                        case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.accessControlPoints:
                            break;
                        case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.escort:
                            qb.andWhere(this.dbClient.raw(`unaccent(oav."escortPerson"->>'name') ilike unaccent('${filterField.value}%)'
                                or unaccent(oav."escortPerson"->>'surname') ilike unaccent('${filterField.value}%)'
                                or unaccent(oav."escortPerson"->>'uniqueId') ilike unaccent('${filterField.value}%')`));
                            break;
                        case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.visitedOrganizationUnit:
                            qb.where(this.dbClient.raw(`unaccent(oav."visitedOrganizationUnit"->>'name') ilike unaccent('${filterField.value}%')`));
                            break;
                        case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.visitedPerson:
                            qb.andWhere(this.dbClient.raw(`unaccent(vuop.name || ' ' ||vuop.surname) ilike unaccent('%${filterField.value}%')`));
                            break;
                        default:
                            qb.where(this.dbClient.raw(`unaccent(oav."${filterField.name}") ilike unaccent('%${filterField.value}%')`));
                            break;
                    }
                }
            }
        }
        let total = 0;
        total = await qb
            .clone()
            .count()
            .then((c) => {
            return parseInt(c[0].count);
        });
        return Promise.resolve(total);
    }
    async listActiveVisitsForReportWithCursor(organizationId, options, visitorRegistrationPointId, visitorModuleSettings, userId) {
        let columns = [
            { "oav.extensionFields": "oav.extensionFields" },
            { "oav.startUtc": "oav.startUtc" },
            { "ovp.id": "ovp.id" },
            { "oav.id": "oav.id" },
            { "oav.visitorRegistrationPointId": "oav.visitorRegistrationPointId" },
            { "uo.userId": "uo.userId" },
        ];
        columns = columns.concat(this.prepareVisitorProfileColumns(visitorModuleSettings));
        let qb = this.dbClient
            .withSchema(organizationId)
            .from("organizationActiveVisits as oav")
            .innerJoin("organizationVisitorProfiles as ovp", (join) => {
            join.on("ovp.id", "oav.organizationVisitorProfileId");
        })
            .innerJoin("userOrganizations as uo", (join) => {
            join.on("ovp.userOrganizationId", "uo.id");
        })
            .leftJoin("visitorRegistrationPoints as vrp", (join) => {
            join.on("oav.visitorRegistrationPointId", "vrp.id");
        })
            .where("oav.organizationId", organizationId)
            .where("oav.state", dal_constants_1.DalConstants.UnterminatedVisitState.Active)
            .where("oav.startUtc", ">=", options.startUtc)
            .where("oav.startUtc", "<=", options.endUtc)
            .whereNull("vrp.deletedAt");
        let visitJoins = this.prepareVisitJoinsForVisitorDetailed(organizationId, visitorModuleSettings, qb);
        qb = visitJoins.qb;
        columns = columns.concat(visitJoins.columns);
        if (visitorRegistrationPointId)
            qb.where("oav.visitorRegistrationPointId", visitorRegistrationPointId);
        if (options.visitorProfileIds && options.visitorProfileIds.length > 0) {
            qb.whereIn("ovp.id", options.visitorProfileIds);
        }
        if (options.visitorProfileFilterFields) {
            for (const filterField of options.visitorProfileFilterFields) {
                let fieldDef = visitorModuleSettings.visitorProfileFormFields.find((f) => f.name == filterField.name);
                if (!fieldDef)
                    continue;
                if (fieldDef.extension) {
                    qb.where(this.dbClient.raw(`unaccent(ovp."extensionFields"->>'${filterField.name}') ilike unaccent('%${filterField.value}%')`));
                }
                else {
                    qb.where(this.dbClient.raw(`unaccent(ovp."${filterField.name}") ilike unaccent('%${filterField.value}%')`));
                }
            }
        }
        if (options.visitFilterFields) {
            for (const filterField of options.visitFilterFields) {
                let visitFormField = visitorModuleSettings.visitFormFields.find((f) => f.name == filterField.name);
                if (!visitFormField)
                    continue;
                if (visitFormField.extension) {
                    if (visitFormField.type == dal_constants_1.DalConstants.FormFieldType.Select) {
                        qb.where(this.dbClient.raw(`oav."extensionFields"->>'${filterField.name}' = '${filterField.value}'`));
                    }
                    else {
                        qb.where(this.dbClient.raw(`unaccent(oav."extensionFields"->>'${filterField.name}') ilike unaccent('%${filterField.value}%')`));
                    }
                }
                else {
                    switch (visitFormField.name) {
                        case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.accessControlPoints:
                            break;
                        case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.escort:
                            qb.andWhere(this.dbClient.raw(`unaccent(oav."escortPerson"->>'name') ilike unaccent('${filterField.value}%)'
                                or unaccent(oav."escortPerson"->>'surname') ilike unaccent('${filterField.value}%)'
                                or unaccent(oav."escortPerson"->>'uniqueId') ilike unaccent('${filterField.value}%')`));
                            break;
                        case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.visitedOrganizationUnit:
                            qb.where(this.dbClient.raw(`unaccent(oav."visitedOrganizationUnit"->>'name') ilike unaccent('${filterField.value}%')`));
                            break;
                        case dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.visitedPerson:
                            qb.andWhere(this.dbClient.raw(`unaccent(vuop.name || ' ' ||vuop.surname) ilike unaccent('%${filterField.value}%')`));
                            break;
                        default:
                            qb.where(this.dbClient.raw(`unaccent(oav."${filterField.name}") ilike unaccent('%${filterField.value}%')`));
                            break;
                    }
                }
            }
        }
        let relatedOrganizationUnitIds = [];
        await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
            relatedOrganizationUnitIds = await dal_manager_1.dbManager.accessOrganizationUnit.getAncestorAndChildOrganizationUnitIds(organizationId, userId, trx);
        }, userId, organizationId);
        qb.where(function () {
            if (relatedOrganizationUnitIds.length > 0) {
                this.whereIn("vrp.organizationUnitId", relatedOrganizationUnitIds).orWhereNull("vrp.organizationUnitId");
            }
        });
        qb.columns(columns).orderBy("oav.startUtc", "desc");
        let str = qb.toSQL().toNative();
        return Promise.resolve({
            query: str.sql,
            queryParams: str.bindings,
        });
    }
    async listAllVisitsForVisitReport(organizationId, options, visitorRegistrationPointId, visitorModuleSettings, userId) {
        switch (options.type) {
            case dal_constants_1.DalConstants.ListVisitState.OnlyActive:
                return this.listActiveVisitsForReport(organizationId, options, visitorRegistrationPointId, visitorModuleSettings, userId);
            case dal_constants_1.DalConstants.ListVisitState.OnlyTerminated:
                return dal_manager_1.dbManager.accessLog.listTerminateVisits(organizationId, options, visitorRegistrationPointId, visitorModuleSettings, userId);
            case dal_constants_1.DalConstants.ListVisitState.All:
            default:
                break;
        }
        let skipCount = options.skip;
        options.take += options.skip;
        options.skip = 0;
        let terminatedVisits = await dal_manager_1.dbManager.accessLog.listTerminateVisits(organizationId, options, visitorRegistrationPointId, visitorModuleSettings, userId);
        let activeVisits = await this.listActiveVisitsForReport(organizationId, options, visitorRegistrationPointId, visitorModuleSettings, userId);
        let result = {
            items: [],
            options: options,
            total: terminatedVisits.total + activeVisits.total,
        };
        result.items = terminatedVisits.items.concat(activeVisits.items);
        result.items.sort((x, y) => ((0, moment_1.default)(x.visitStartUtc).isBefore(y.visitStartUtc) ? 1 : -1));
        result.items = result.items.slice(skipCount, options.take);
        result.options.skip = skipCount;
        result.options.take -= skipCount;
        return Promise.resolve(result);
    }
    async generateVisitRecordsExcelReport(organizationId, userId, request, settings, onData) {
        let result = {
            report: {
                options: null,
                total: null,
                items: null,
            },
            captionDefinitions: null,
        };
        result.captionDefinitions = await (0, business_report_export_1.getCaptionDefinitions)(organizationId, userId);
        result.organization = await dal_manager_1.dbManager.accessOrganization.getOrganizationLogoAndName(organizationId);
        if (request.type === dal_constants_1.DalConstants.ListVisitState.All || request.type === dal_constants_1.DalConstants.ListVisitState.OnlyActive) {
            let queryResult = await this.listActiveVisitsForReportWithCursor(organizationId, {
                sortStartUtcAsc: request.sortStartUtcAsc,
                take: app_config_1.appConfig.reportExportRowLimit + 1,
                skip: 0,
                startUtc: request.startUtc,
                endUtc: request.endUtc,
                visitorProfileIds: request.visitorProfileIds,
                visitorProfileFilterFields: request.visitorProfileFilterFields,
                visitFilterFields: request.visitFilterFields,
                type: request.type,
            }, request.visitorRegistrationPointId, settings, userId);
            let queryParams = queryResult.queryParams;
            let query = queryResult.query;
            const client = await this._pgPool.connect();
            const cursor = client.query(new Cursor(query, queryParams));
            let rows = [];
            while (true) {
                let items = [];
                try {
                    rows = await new Promise((resolve, reject) => {
                        cursor.read(500, async (err, rows) => {
                            if (err) {
                                reject(err);
                            }
                            else {
                                resolve(rows);
                            }
                        });
                    });
                    for (const visit of rows) {
                        let profileFields = {};
                        for (const visitorProfileField of settings.visitorProfileFormFields) {
                            profileFields[visitorProfileField.name] = visit["ovp." + visitorProfileField.name] ? visit["ovp." + visitorProfileField.name] : null;
                        }
                        let userId = visit["uo.userId"];
                        let profileId = visit["ovp.id"];
                        let startUtc = visit["oav.startUtc"];
                        let visitorRegistrationPointId = visit["oav.visitorRegistrationPointId"];
                        let visitFields = {};
                        for (const visitFormField of settings.visitFormFields) {
                            if (visitFormField.extension) {
                                visitFields[visitFormField.name] = visit["oav.extensionFields"][visitFormField.name] ? visit["oav.extensionFields"][visitFormField.name] : null;
                            }
                            else {
                                if (visitFormField.name === dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.expectedDate) {
                                    visitFields[visitFormField.name] = {
                                        startDate: visit["oav.expectedStartUtc"],
                                        endDate: visit["oav.expectedEndUtc"],
                                    };
                                }
                                else {
                                    visitFields[visitFormField.name] = visit["oav." + visitFormField.name] ? visit["oav." + visitFormField.name] : null;
                                }
                            }
                        }
                        visitFields["visitorRegistrationPointId"] = visitorRegistrationPointId;
                        items.push({
                            visitEndUtc: null,
                            visitFields: visitFields,
                            visitId: visit["oav.id"],
                            visitStartUtc: startUtc,
                            visitorProfileFields: profileFields,
                            visitorProfileId: profileId,
                            visitorUserId: userId,
                        });
                    }
                    if (request.sensitiveFields) {
                        (0, report_util_1.hideSensitiveFieldsOfVisitsForReport)(request.sensitiveFields.visitFields, request.sensitiveFields.visitorFields, items, {
                            hideUniqueIdForVisitor: request.sensitiveFields.hideUniqueIdForVisitor,
                            showNumberOfInitial: request.sensitiveFields.showNumberOfInitial,
                            showNumberOfLast: request.sensitiveFields.showNumberOfLast,
                        });
                    }
                    result.report.items = items;
                    await onData(result.report.items);
                }
                catch (error) {
                    app_logs_1.logger.error("Error while fetch visit records data with cursor!");
                }
                if (rows.length < 500) {
                    break;
                }
            }
            try {
                await new Promise((resolve, reject) => {
                    cursor.close((err) => {
                        if (err) {
                            reject(err);
                        }
                        else {
                            resolve();
                        }
                    });
                });
                client.release();
            }
            catch (error) {
                client?.release(error);
                app_logs_1.logger.error(error);
            }
        }
        if (request.type === dal_constants_1.DalConstants.ListVisitState.All || request.type === dal_constants_1.DalConstants.ListVisitState.OnlyTerminated) {
            await dal_manager_1.dbManager.accessLog.listTerminateVisitsWithCursor(organizationId, userId, {
                sortStartUtcAsc: request.sortStartUtcAsc,
                take: app_config_1.appConfig.reportExportRowLimit + 1,
                skip: 0,
                startUtc: request.startUtc,
                endUtc: request.endUtc,
                visitorProfileIds: request.visitorProfileIds,
                visitorProfileFilterFields: request.visitorProfileFilterFields,
                visitFilterFields: request.visitFilterFields,
                type: request.type,
            }, request, settings, onData.bind(this));
        }
        return Promise.resolve();
    }
    async updateDailyPreregisterSummary(organizationId, params, trx) {
        let dailyVisitStat;
        let dailyVisitStatQuery = this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.preregisterDailySummaries)
            .where("organizationId", organizationId)
            .where("date", ">=", (0, moment_1.default)(params.date).startOf("day").toDate())
            .where("date", "<=", (0, moment_1.default)(params.date).endOf("day").toDate())
            .first();
        if (trx) {
            dailyVisitStat = await dailyVisitStatQuery.transacting(trx);
        }
        else {
            dailyVisitStat = await dailyVisitStatQuery;
        }
        if (!dailyVisitStat) {
            let query = this.dbClient
                .withSchema(organizationId)
                .table(dal_db_armon_schema_1.ArmonSchema.tableNames.preregisterDailySummaries)
                .insert({
                id: uuid_1.default.v4(),
                organizationId: organizationId,
                date: (0, moment_1.default)(params.date).startOf("day").toDate(),
                totalCount: 1,
                selfPreregisterCount: params.self ? 1 : 0,
                totaltoVisitCount: params.turnedToVisit ? 1 : 0,
                selftoVisitCount: params.self && params.turnedToVisit ? 1 : 0,
            });
            if (trx) {
                await query.transacting(trx);
            }
            else {
                await query;
            }
        }
        else {
            let query = this.dbClient
                .withSchema(organizationId)
                .table(dal_db_armon_schema_1.ArmonSchema.tableNames.preregisterDailySummaries)
                .where("id", dailyVisitStat.id)
                .update({
                totalCount: dailyVisitStat.totalCount + (params.terminatingPreregistered ? 0 : 1),
                selfPreregisterCount: dailyVisitStat.selfPreregisterCount + (params.self && !params.terminatingPreregistered ? 1 : 0),
                totaltoVisitCount: params.turnedToVisit ? dailyVisitStat.totaltoVisitCount + 1 : dailyVisitStat.totaltoVisitCount,
                selftoVisitCount: params.self && params.turnedToVisit ? dailyVisitStat.selftoVisitCount + 1 : dailyVisitStat.selftoVisitCount,
            });
            if (trx) {
                await query.transacting(trx);
            }
            else {
                await query;
            }
        }
    }
    async updateDailyVisitSummary(organizationId, visitDetailed) {
        let preregisterLog = await dal_manager_1.dbManager.accessLog.getPreregisteredVisitToNoSql(organizationId, visitDetailed.id);
        let isSelfPreregistered = false;
        if (preregisterLog) {
            if (preregisterLog.v && preregisterLog.v.visitedPerson && preregisterLog.v.visitedPerson.userId) {
                isSelfPreregistered = preregisterLog.v.visitedPerson.userId == preregisterLog.r ? true : false;
                await this.updateDailyPreregisterSummary(organizationId, {
                    date: (0, moment_1.default)(preregisterLog.u).startOf("day").toDate(),
                    self: isSelfPreregistered,
                    turnedToVisit: true,
                    terminatingPreregistered: true,
                });
            }
            else {
                await this.updateDailyPreregisterSummary(organizationId, {
                    date: (0, moment_1.default)(preregisterLog.u).startOf("day").toDate(),
                    self: false,
                    turnedToVisit: true,
                    terminatingPreregistered: true,
                });
            }
        }
        let dailyVisitStat = await this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.visitorDailySummaries)
            .where("organizationId", organizationId)
            .where("date", ">=", (0, moment_1.default)(visitDetailed.startUtc).startOf("day").toDate())
            .where("date", "<=", (0, moment_1.default)(visitDetailed.startUtc).endOf("day").toDate())
            .where("visitorRegistrationPointId", visitDetailed.visitorRegistrationPointId)
            .where("reason", visitDetailed.fields["reason"])
            .first();
        if (!dailyVisitStat) {
            await this.dbClient
                .withSchema(organizationId)
                .table(dal_db_armon_schema_1.ArmonSchema.tableNames.visitorDailySummaries)
                .insert({
                id: uuid_1.default.v4(),
                organizationId: organizationId,
                date: (0, moment_1.default)(visitDetailed.startUtc).startOf("day").toDate(),
                visitorRegistrationPointId: visitDetailed.visitorRegistrationPointId,
                reason: visitDetailed.fields["reason"],
                count: 1,
                averageProcessTime: visitDetailed.processTime,
            });
        }
        else {
            let totalProcessTime = dailyVisitStat.averageProcessTime * dailyVisitStat.count;
            totalProcessTime += visitDetailed.processTime;
            let average = totalProcessTime / (dailyVisitStat.count + 1);
            await this.dbClient
                .withSchema(organizationId)
                .table(dal_db_armon_schema_1.ArmonSchema.tableNames.visitorDailySummaries)
                .where("id", dailyVisitStat.id)
                .update({
                count: dailyVisitStat.count + 1,
                averageProcessTime: average,
            });
        }
    }
    async getPreregistrationAnalysis(organizationId, args) {
        let result = [];
        let startDate = args.startDate;
        let endDate = args.endDate;
        switch (args.segmentType) {
            case dal_constants_1.DalConstants.SegmentType.Day:
                startDate = (0, moment_1.default)(startDate).startOf("day").toDate();
                endDate = (0, moment_1.default)(endDate).endOf("day").toDate();
                break;
            case dal_constants_1.DalConstants.SegmentType.Week:
                startDate = (0, moment_1.default)(startDate).startOf("isoWeek").toDate();
                endDate = (0, moment_1.default)(endDate).endOf("isoWeek").toDate();
                break;
            case dal_constants_1.DalConstants.SegmentType.Month:
                startDate = (0, moment_1.default)(startDate).startOf("month").toDate();
                endDate = (0, moment_1.default)(endDate).endOf("month").toDate();
                break;
            default:
                break;
        }
        let preregistrationSummaries = await this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.preregisterDailySummaries)
            .where("organizationId", organizationId)
            .where("date", ">=", startDate)
            .where("date", "<=", endDate)
            .groupBy("date")
            .sum("selfPreregisterCount as selfPreregisterCount")
            .sum("totalCount as totalCount")
            .sum("totaltoVisitCount as totaltoVisitCount")
            .sum("selftoVisitCount as selftoVisitCount")
            .orderBy("date", "desc")
            .select("date");
        let sumBetweenDates = (arr, startDate, endDate, field) => {
            return arr.filter((a) => (0, moment_1.default)(a.date).isSameOrAfter(startDate) && (0, moment_1.default)(a.date).isSameOrBefore(endDate)).reduce((a, b) => a + parseInt(b[field]), 0);
        };
        let countBetweenDates = (arr, startDate, endDate) => {
            return arr.filter((a) => (0, moment_1.default)(a.date).isSameOrAfter(startDate) && (0, moment_1.default)(a.date).isSameOrBefore(endDate)).length;
        };
        let dateIndexStart = (0, moment_1.default)(startDate);
        let dateIndexEnd = (0, moment_1.default)(endDate);
        let loopGuard = 0;
        while (dateIndexStart.isSameOrBefore(dateIndexEnd) && loopGuard < 1000) {
            loopGuard++;
            let lookupEndDate = (0, moment_1.default)(dateIndexStart).toDate();
            switch (args.segmentType) {
                case dal_constants_1.DalConstants.SegmentType.Day:
                    lookupEndDate = (0, moment_1.default)(dateIndexStart).endOf("day").toDate();
                    break;
                case dal_constants_1.DalConstants.SegmentType.Week:
                    lookupEndDate = (0, moment_1.default)(dateIndexStart).endOf("isoWeek").toDate();
                    break;
                case dal_constants_1.DalConstants.SegmentType.Month:
                    lookupEndDate = (0, moment_1.default)(dateIndexStart).endOf("month").toDate();
                    break;
                default:
                    break;
            }
            let visitInfo = await dal_manager_1.dbManager.accessLog.listVisitStats(organizationId, {
                startDateTime: dateIndexStart.toDate(),
                endDateTime: lookupEndDate,
            });
            let dataItems = {
                selfPreregistered: sumBetweenDates(preregistrationSummaries, dateIndexStart.toDate(), lookupEndDate, "selfPreregisterCount"),
                totalPreregistered: sumBetweenDates(preregistrationSummaries, dateIndexStart.toDate(), lookupEndDate, "totalCount"),
                selfPreregisteredIntoVisit: sumBetweenDates(preregistrationSummaries, dateIndexStart.toDate(), lookupEndDate, "selftoVisitCount"),
                turnedIntoVisit: sumBetweenDates(preregistrationSummaries, dateIndexStart.toDate(), lookupEndDate, "totaltoVisitCount"),
                totalVisit: countBetweenDates(visitInfo, dateIndexStart.toDate(), lookupEndDate),
                totalVisitFromPreregistration: countBetweenDates(visitInfo.filter((v) => v.fromPreregistration), dateIndexStart.toDate(), lookupEndDate),
            };
            let item = {
                startDate: dateIndexStart.toDate(),
                endDate: lookupEndDate,
                data: [dataItems],
            };
            result.push(item);
            switch (args.segmentType) {
                case dal_constants_1.DalConstants.SegmentType.Day:
                    dateIndexStart = (0, moment_1.default)(dateIndexStart).add(1, "day");
                    break;
                case dal_constants_1.DalConstants.SegmentType.Week:
                    dateIndexStart = (0, moment_1.default)(dateIndexStart).add(1, "week");
                    break;
                case dal_constants_1.DalConstants.SegmentType.Month:
                    dateIndexStart = (0, moment_1.default)(dateIndexStart).add(1, "month");
                    break;
                default:
                    break;
            }
        }
        return Promise.resolve(result);
    }
    async getVisitCountsPointReason(organizationId, args) {
        let result = [];
        let startDate = luxon_1.DateTime.fromISO(args.startDate.toString());
        let endDate = luxon_1.DateTime.fromISO(args.endDate.toString());
        switch (args.segmentType) {
            case dal_constants_1.DalConstants.SegmentType.Day:
                startDate = startDate.startOf("day");
                endDate = endDate.endOf("day");
                break;
            case dal_constants_1.DalConstants.SegmentType.Week:
                startDate = startDate.startOf("week");
                endDate = endDate.endOf("week");
                break;
            case dal_constants_1.DalConstants.SegmentType.Month:
                startDate = startDate.startOf("month");
                endDate = endDate.endOf("month");
                break;
            default:
                break;
        }
        let qb = this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.visitorDailySummaries)
            .where("organizationId", organizationId)
            .where("date", ">=", startDate.toISODate())
            .where("date", "<=", endDate.toISODate());
        if (args.visitorRegistrationPointId) {
            qb.where("visitorRegistrationPointId", args.visitorRegistrationPointId);
        }
        if (args.reason) {
            qb.where("reason", args.reason);
        }
        qb.groupBy("date", "visitorRegistrationPointId", "reason").sum("count as count").orderBy("date", "desc").select("date", "visitorRegistrationPointId", "reason");
        let dailyVisits = await qb;
        let keys = lodash_1.default.uniqBy(dailyVisits.map((d) => ({
            visitorRegistrationPointId: d.visitorRegistrationPointId,
            reason: d.reason,
        })), (key) => `${key.visitorRegistrationPointId}-${key.reason}`);
        let sumBetweenDates = (arr, startDate, endDate) => {
            return arr
                .filter((a) => {
                const date = luxon_1.DateTime.fromJSDate(a.date);
                return date.toMillis() >= startDate.toMillis() && date.toMillis() <= endDate.toMillis();
            })
                .reduce((a, b) => a + parseInt(b.count, 10), 0);
        };
        let dateIndexStart = startDate;
        let dateIndexEnd = endDate;
        let loopGuard = 0;
        while (dateIndexStart <= dateIndexEnd && loopGuard < 1000) {
            loopGuard++;
            let lookupEndDate = dateIndexStart;
            switch (args.segmentType) {
                case dal_constants_1.DalConstants.SegmentType.Day:
                    lookupEndDate = dateIndexStart.endOf("day");
                    break;
                case dal_constants_1.DalConstants.SegmentType.Week:
                    lookupEndDate = dateIndexStart.endOf("week");
                    break;
                case dal_constants_1.DalConstants.SegmentType.Month:
                    lookupEndDate = dateIndexStart.endOf("month");
                    break;
                default:
                    break;
            }
            let dataItems = [];
            for (const key of keys) {
                let dataItem = {
                    count: sumBetweenDates(dailyVisits.filter((d) => d.visitorRegistrationPointId === key.visitorRegistrationPointId && d.reason === key.reason), dateIndexStart, lookupEndDate),
                    visitorRegistrationPointId: key.visitorRegistrationPointId,
                    reason: key.reason,
                };
                dataItems.push(dataItem);
            }
            let item = {
                startDate: dateIndexStart.toJSDate(),
                endDate: lookupEndDate.toJSDate(),
                data: dataItems,
            };
            result.push(item);
            switch (args.segmentType) {
                case dal_constants_1.DalConstants.SegmentType.Day:
                    dateIndexStart = dateIndexStart.plus({ days: 1 });
                    break;
                case dal_constants_1.DalConstants.SegmentType.Week:
                    dateIndexStart = dateIndexStart.plus({ weeks: 1 });
                    break;
                case dal_constants_1.DalConstants.SegmentType.Month:
                    dateIndexStart = dateIndexStart.plus({ months: 1 });
                    break;
                default:
                    break;
            }
        }
        return Promise.resolve(result);
    }
    async getVisitProcessTimePointReason(organizationId, args) {
        let result = [];
        let startDate = luxon_1.DateTime.fromISO(args.startDate.toString());
        let endDate = luxon_1.DateTime.fromISO(args.endDate.toString());
        switch (args.segmentType) {
            case dal_constants_1.DalConstants.SegmentType.Day:
                startDate = startDate.startOf("day");
                endDate = endDate.endOf("day");
                break;
            case dal_constants_1.DalConstants.SegmentType.Week:
                startDate = startDate.startOf("week");
                endDate = endDate.endOf("week");
                break;
            case dal_constants_1.DalConstants.SegmentType.Month:
                startDate = startDate.startOf("month");
                endDate = endDate.endOf("month");
                break;
            default:
                break;
        }
        let qb = this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.visitorDailySummaries)
            .where("organizationId", organizationId)
            .where("date", ">=", startDate.toISODate())
            .where("date", "<=", endDate.toISODate());
        if (args.visitorRegistrationPointId) {
            qb.where("visitorRegistrationPointId", args.visitorRegistrationPointId);
        }
        if (args.reason) {
            qb.where("reason", args.reason);
        }
        qb.groupBy("date", "visitorRegistrationPointId", "reason")
            .select(this.dbClient.raw('avg(CASE WHEN "averageProcessTime" <> 0 THEN "averageProcessTime" ELSE NULL END) as "processTime"'))
            .orderBy("date", "desc")
            .select("date", "visitorRegistrationPointId", "reason");
        let dailyVisits = await qb;
        let avgBetweenDates = (arr, startDate, endDate) => {
            let filtered = arr.filter((a) => {
                const date = luxon_1.DateTime.fromJSDate(a.date);
                return date.toMillis() >= startDate.toMillis() && date.toMillis() <= endDate.toMillis() && a.processTime;
            });
            let sum = filtered.reduce((a, b) => a + parseFloat(b.processTime), 0);
            return filtered.length === 0 ? 0 : sum / filtered.length;
        };
        let keys = lodash_1.default.uniqWith(dailyVisits.map((d) => ({ visitorRegistrationPointId: d.visitorRegistrationPointId, reason: d.reason })), lodash_1.default.isEqual);
        let dateIndexStart = startDate;
        let dateIndexEnd = endDate;
        let loopGuard = 0;
        while (dateIndexStart <= dateIndexEnd && loopGuard < 1000) {
            loopGuard++;
            let lookupEndDate = dateIndexStart;
            switch (args.segmentType) {
                case dal_constants_1.DalConstants.SegmentType.Day:
                    lookupEndDate = dateIndexStart.endOf("day");
                    break;
                case dal_constants_1.DalConstants.SegmentType.Week:
                    lookupEndDate = dateIndexStart.endOf("week");
                    break;
                case dal_constants_1.DalConstants.SegmentType.Month:
                    lookupEndDate = dateIndexStart.endOf("month");
                    break;
                default:
                    break;
            }
            let dataItems = [];
            for (const key of keys) {
                let dataItem = {
                    averageProcessTime: avgBetweenDates(dailyVisits.filter((d) => d.visitorRegistrationPointId == key.visitorRegistrationPointId && d.reason == key.reason), dateIndexStart, lookupEndDate),
                    visitorRegistrationPointId: key.visitorRegistrationPointId,
                    reason: key.reason,
                };
                dataItems.push(dataItem);
            }
            let item = {
                startDate: dateIndexStart.toJSDate(),
                endDate: lookupEndDate.toJSDate(),
                data: dataItems,
            };
            result.push(item);
            switch (args.segmentType) {
                case dal_constants_1.DalConstants.SegmentType.Day:
                    dateIndexStart = dateIndexStart.plus({ days: 1 });
                    break;
                case dal_constants_1.DalConstants.SegmentType.Week:
                    dateIndexStart = dateIndexStart.plus({ weeks: 1 });
                    break;
                case dal_constants_1.DalConstants.SegmentType.Month:
                    dateIndexStart = dateIndexStart.plus({ months: 1 });
                    break;
                default:
                    break;
            }
        }
        return Promise.resolve(result);
    }
    async getVisitCountsByFilter(organizationId, args, byRegistrationPoint) {
        let result = [];
        let startDate = args.startDate;
        let endDate = args.endDate;
        switch (args.segmentType) {
            case dal_constants_1.DalConstants.SegmentType.Day:
                startDate = (0, moment_1.default)(startDate).startOf("day").toDate();
                endDate = (0, moment_1.default)(endDate).endOf("day").toDate();
                break;
            case dal_constants_1.DalConstants.SegmentType.Week:
                startDate = (0, moment_1.default)(startDate).startOf("isoWeek").toDate();
                endDate = (0, moment_1.default)(endDate).endOf("isoWeek").toDate();
                break;
            case dal_constants_1.DalConstants.SegmentType.Month:
                startDate = (0, moment_1.default)(startDate).startOf("month").toDate();
                endDate = (0, moment_1.default)(endDate).endOf("month").toDate();
                break;
            default:
                break;
        }
        let qb = this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.visitorDailySummaries)
            .where("organizationId", organizationId)
            .where("date", ">=", startDate)
            .where("date", "<=", endDate);
        if (byRegistrationPoint) {
            if (args.visitorRegistrationPointId) {
                qb.where("visitorRegistrationPointId", args.visitorRegistrationPointId);
            }
            qb.groupBy("date", "visitorRegistrationPointId").sum("count as count").orderBy("date", "desc").select("date", "visitorRegistrationPointId");
        }
        else {
            if (args.reason) {
                qb.where("reason", args.reason);
            }
            qb.groupBy("date", "reason").sum("count as count").orderBy("date", "desc").select("date", "reason");
        }
        let dailyVisits = await qb;
        let segments = {};
        let sumBetweenDates = (arr, startDate, endDate) => {
            return arr.filter((a) => (0, moment_1.default)(a.date).isSameOrAfter(startDate) && (0, moment_1.default)(a.date).isSameOrBefore(endDate)).reduce((a, b) => a + parseInt(b.count), 0);
        };
        let keys = [];
        if (byRegistrationPoint) {
            keys = lodash_1.default.uniq(dailyVisits.map((d) => d.visitorRegistrationPointId));
        }
        else {
            keys = lodash_1.default.uniq(dailyVisits.map((d) => d.reason));
        }
        let dateIndexStart = (0, moment_1.default)(startDate);
        let dateIndexEnd = (0, moment_1.default)(endDate);
        let loopGuard = 0;
        while (dateIndexStart.isSameOrBefore(dateIndexEnd) && loopGuard < 1000) {
            loopGuard++;
            let lookupEndDate = (0, moment_1.default)(dateIndexStart).toDate();
            switch (args.segmentType) {
                case dal_constants_1.DalConstants.SegmentType.Day:
                    lookupEndDate = (0, moment_1.default)(dateIndexStart).endOf("day").toDate();
                    break;
                case dal_constants_1.DalConstants.SegmentType.Week:
                    lookupEndDate = (0, moment_1.default)(dateIndexStart).endOf("isoWeek").toDate();
                    break;
                case dal_constants_1.DalConstants.SegmentType.Month:
                    lookupEndDate = (0, moment_1.default)(dateIndexStart).endOf("month").toDate();
                    break;
                default:
                    break;
            }
            if (byRegistrationPoint) {
                if (args.visitorRegistrationPointId) {
                    let dataItems = {
                        count: sumBetweenDates(dailyVisits.filter((d) => d.visitorRegistrationPointId == args.visitorRegistrationPointId), dateIndexStart.toDate(), lookupEndDate),
                        visitorRegistrationPointId: args.visitorRegistrationPointId,
                    };
                    let item = {
                        startDate: dateIndexStart.toDate(),
                        endDate: lookupEndDate,
                        data: [dataItems],
                    };
                    result.push(item);
                }
                else {
                    let dataItems = [];
                    for (const key of keys) {
                        let dataItem = {
                            count: sumBetweenDates(dailyVisits.filter((d) => d.visitorRegistrationPointId == key), dateIndexStart.toDate(), lookupEndDate),
                            visitorRegistrationPointId: key,
                        };
                        dataItems.push(dataItem);
                    }
                    let item = {
                        startDate: dateIndexStart.toDate(),
                        endDate: lookupEndDate,
                        data: dataItems,
                    };
                    result.push(item);
                }
            }
            else {
                if (args.reason) {
                    let dataItems = {
                        count: sumBetweenDates(dailyVisits.filter((d) => d.reason == args.reason), dateIndexStart.toDate(), lookupEndDate),
                        visitorRegistrationPointId: args.visitorRegistrationPointId,
                    };
                    let item = {
                        startDate: dateIndexStart.toDate(),
                        endDate: lookupEndDate,
                        data: [dataItems],
                    };
                    result.push(item);
                }
                else {
                    let dataItems = [];
                    for (const key of keys) {
                        let dataItem = {
                            count: sumBetweenDates(dailyVisits.filter((d) => d.reason == key), dateIndexStart.toDate(), lookupEndDate),
                            reason: key,
                        };
                        dataItems.push(dataItem);
                    }
                    let item = {
                        startDate: dateIndexStart.toDate(),
                        endDate: lookupEndDate,
                        data: dataItems,
                    };
                    result.push(item);
                }
            }
            switch (args.segmentType) {
                case dal_constants_1.DalConstants.SegmentType.Day:
                    dateIndexStart = (0, moment_1.default)(dateIndexStart).add(1, "day");
                    break;
                case dal_constants_1.DalConstants.SegmentType.Week:
                    dateIndexStart = (0, moment_1.default)(dateIndexStart).add(1, "week");
                    break;
                case dal_constants_1.DalConstants.SegmentType.Month:
                    dateIndexStart = (0, moment_1.default)(dateIndexStart).add(1, "month");
                    break;
                default:
                    break;
            }
        }
        return Promise.resolve(result);
    }
    async getVisitProcessTimeByFilter(organizationId, args, byRegistrationPoint) {
        let result = [];
        let startDate = args.startDate;
        let endDate = args.endDate;
        switch (args.segmentType) {
            case dal_constants_1.DalConstants.SegmentType.Day:
                startDate = (0, moment_1.default)(startDate).startOf("day").toDate();
                endDate = (0, moment_1.default)(endDate).endOf("day").toDate();
                break;
            case dal_constants_1.DalConstants.SegmentType.Week:
                startDate = (0, moment_1.default)(startDate).startOf("isoWeek").toDate();
                endDate = (0, moment_1.default)(endDate).endOf("isoWeek").toDate();
                break;
            case dal_constants_1.DalConstants.SegmentType.Month:
                startDate = (0, moment_1.default)(startDate).startOf("month").toDate();
                endDate = (0, moment_1.default)(endDate).endOf("month").toDate();
                break;
            default:
                break;
        }
        let qb = this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.visitorDailySummaries)
            .where("organizationId", organizationId)
            .where("date", ">=", startDate)
            .where("date", "<=", endDate);
        if (byRegistrationPoint) {
            if (args.visitorRegistrationPointId) {
                qb.where("visitorRegistrationPointId", args.visitorRegistrationPointId);
            }
            qb.groupBy("date", "visitorRegistrationPointId")
                .select(this.dbClient.raw('avg(CASE WHEN "averageProcessTime" <> 0 THEN "averageProcessTime" ELSE NULL END) as "processTime"'))
                .orderBy("date", "desc")
                .select("date", "visitorRegistrationPointId");
        }
        else {
            if (args.reason) {
                qb.where("reason", args.reason);
            }
            qb.groupBy("date", "reason")
                .select(this.dbClient.raw('avg(CASE WHEN "averageProcessTime" <> 0 THEN "averageProcessTime" ELSE NULL END) as "processTime"'))
                .orderBy("date", "desc")
                .select("date", "reason");
        }
        let dailyVisits = await qb;
        let avgBetweenDates = (arr, startDate, endDate) => {
            let sum = arr
                .filter((a) => a.processTime)
                .filter((a) => (0, moment_1.default)(a.date).isSameOrAfter(startDate) && (0, moment_1.default)(a.date).isSameOrBefore(endDate))
                .reduce((a, b) => a + parseFloat(b.processTime), 0);
            let count = arr.filter((a) => (0, moment_1.default)(a.date).isSameOrAfter(startDate) && (0, moment_1.default)(a.date).isSameOrBefore(endDate) && a.processTime && parseInt(a.processTime) > 0).length;
            if (count < 1) {
                return 0;
            }
            return sum / count;
        };
        let keys = [];
        if (byRegistrationPoint) {
            keys = lodash_1.default.uniq(dailyVisits.map((d) => d.visitorRegistrationPointId));
        }
        else {
            keys = lodash_1.default.uniq(dailyVisits.map((d) => d.reason));
        }
        let dateIndexStart = (0, moment_1.default)(startDate);
        let dateIndexEnd = (0, moment_1.default)(endDate);
        let loopGuard = 0;
        while (dateIndexStart.isSameOrBefore(dateIndexEnd) && loopGuard < 1000) {
            loopGuard++;
            let lookupEndDate = (0, moment_1.default)(dateIndexStart).toDate();
            switch (args.segmentType) {
                case dal_constants_1.DalConstants.SegmentType.Day:
                    lookupEndDate = (0, moment_1.default)(dateIndexStart).endOf("day").toDate();
                    break;
                case dal_constants_1.DalConstants.SegmentType.Week:
                    lookupEndDate = (0, moment_1.default)(dateIndexStart).endOf("isoWeek").toDate();
                    break;
                case dal_constants_1.DalConstants.SegmentType.Month:
                    lookupEndDate = (0, moment_1.default)(dateIndexStart).endOf("month").toDate();
                    break;
                default:
                    break;
            }
            if (byRegistrationPoint) {
                if (args.visitorRegistrationPointId) {
                    let dataItems = {
                        averageProcessTime: avgBetweenDates(dailyVisits.filter((d) => d.visitorRegistrationPointId == args.visitorRegistrationPointId), dateIndexStart.toDate(), lookupEndDate),
                        visitorRegistrationPointId: args.visitorRegistrationPointId,
                    };
                    let item = {
                        startDate: dateIndexStart.toDate(),
                        endDate: lookupEndDate,
                        data: [dataItems],
                    };
                    result.push(item);
                }
                else {
                    let dataItems = [];
                    for (const key of keys) {
                        let dataItem = {
                            averageProcessTime: avgBetweenDates(dailyVisits.filter((d) => d.visitorRegistrationPointId == key), dateIndexStart.toDate(), lookupEndDate),
                            visitorRegistrationPointId: key,
                        };
                        dataItems.push(dataItem);
                    }
                    let item = {
                        startDate: dateIndexStart.toDate(),
                        endDate: lookupEndDate,
                        data: dataItems,
                    };
                    result.push(item);
                }
            }
            else {
                if (args.reason) {
                    let dataItems = {
                        averageProcessTime: avgBetweenDates(dailyVisits.filter((d) => d.reason == args.reason), dateIndexStart.toDate(), lookupEndDate),
                        visitorRegistrationPointId: args.visitorRegistrationPointId,
                    };
                    let item = {
                        startDate: dateIndexStart.toDate(),
                        endDate: lookupEndDate,
                        data: [dataItems],
                    };
                    result.push(item);
                }
                else {
                    let dataItems = [];
                    for (const key of keys) {
                        let dataItem = {
                            averageProcessTime: avgBetweenDates(dailyVisits.filter((d) => d.reason == key), dateIndexStart.toDate(), lookupEndDate),
                            reason: key,
                        };
                        dataItems.push(dataItem);
                    }
                    let item = {
                        startDate: dateIndexStart.toDate(),
                        endDate: lookupEndDate,
                        data: dataItems,
                    };
                    result.push(item);
                }
            }
            switch (args.segmentType) {
                case dal_constants_1.DalConstants.SegmentType.Day:
                    dateIndexStart = (0, moment_1.default)(dateIndexStart).add(1, "day");
                    break;
                case dal_constants_1.DalConstants.SegmentType.Week:
                    dateIndexStart = (0, moment_1.default)(dateIndexStart).add(1, "week");
                    break;
                case dal_constants_1.DalConstants.SegmentType.Month:
                    dateIndexStart = (0, moment_1.default)(dateIndexStart).add(1, "month");
                    break;
                default:
                    break;
            }
        }
        return Promise.resolve(result);
    }
    async updateOrganizationVisitorModuleSettings(organizationId, field, form) {
        if (!form || !field) {
            return Promise.reject("invalid data");
        }
        let organizationVisitorModuleSettings = await this.getVisitAndVisitorFormSettings(organizationId);
        await this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorModuleSettings)
            .where("organizationId", organizationId)
            .update(field, JSON.stringify(form));
        return Promise.resolve(JSON.stringify(organizationVisitorModuleSettings[field]));
    }
    async listUserActiveVisits(organizationId, visitedUserOrganizationId) {
        let result = [];
        return this.dbClient
            .withSchema(organizationId)
            .from(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationActiveVisits)
            .where("organizationId", organizationId)
            .where("visitedUserOrganizationId", visitedUserOrganizationId)
            .select("id", "state", "expectedStartUtc");
    }
    async listActiveVisitsOfVisitorByUserId(organizationId, userId) {
        return (await this.dbClient
            .withSchema(organizationId)
            .from(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationActiveVisits + " as oav")
            .innerJoin(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorProfiles + " as ovp", "ovp.id", "oav.organizationVisitorProfileId")
            .innerJoin(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations + " as uo", "uo.id", "ovp.userOrganizationId")
            .where("oav.organizationId", organizationId)
            .where("oav.state", dal_constants_1.DalConstants.OrganizationVisitorStates.Active)
            .where("uo.organizationId", organizationId)
            .where("uo.userId", userId)
            .whereNull("ovp.deletedAt")
            .whereNull("uo.deletedAt")
            .select("oav.id")).map((v) => v.id);
    }
    async makeVisitorProfileAnonym(organizationId, visitorProfileId) {
        let visitorModuleSettings = await dal_manager_1.dbManager.accessVisitor.getVisitAndVisitorFormSettings(organizationId);
        await this.dbClient.transaction(async (trx) => {
            let qb = trx.withSchema(organizationId).from(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorProfiles).where("id", visitorProfileId).update({
                name: "****",
                surname: "****",
                uniqueId: "****",
                thumbnail: null,
                isAnonym: true,
            });
            for (const visitorField of visitorModuleSettings.visitorProfileFormFields) {
                if (visitorField.gdpr && visitorField.extension) {
                    qb = qb.update("extensionFields", trx.raw(` jsonb_set("extensionFields"::jsonb,'{"${visitorField.name}"}','"****"') `));
                }
            }
            await qb;
            await dal_manager_1.dbManager.accessLog.makeVisitorProfileHistoryAnonym(organizationId, [visitorProfileId]);
        });
    }
    async updateExpiredVisitorProfilesAsAnonym(params) {
        const visitorModuleSettings = await dal_manager_1.dbManager.accessVisitor.getVisitAndVisitorFormSettings(params.organizationId);
        const expiredIdsResult = (await params.trx.query(`
                SELECT ovp.id from "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorProfiles}" AS ovp
                INNER JOIN "${params.organizationId}"."userOrganizations" AS uo ON uo."id" = ovp."userOrganizationId"
                WHERE ovp."deletedAt" IS NULL AND
                    ovp."isAnonym" != TRUE AND
                    ovp."createdAt" < $1 AND
                    uo."organizationId" = $2
            `, [params.date, params.organizationId])).rows.map((r) => r.id);
        if (expiredIdsResult.length > 0) {
            let query = `
                UPDATE "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorProfiles}"
                SET name = '****', surname = '****', "uniqueId" = '****', "thumbnail" = NULL, "isAnonym" = TRUE,
                "extensionFields" = CASE WHEN "extensionFields" IS NULL THEN NULL ELSE
            `;
            let innerQuery = `"extensionFields"::jsonb`;
            for (const visitorField of visitorModuleSettings.visitorProfileFormFields) {
                if (visitorField.gdpr && visitorField.extension && visitorField.type === dal_constants_1.DalConstants.FormFieldType.Text) {
                    innerQuery = " jsonb_set(" + innerQuery + `, '{${visitorField.name}}' , '"****"', false)`;
                }
            }
            query +=
                innerQuery +
                    `
                END
                WHERE id = ANY($1::UUID[]);
            `;
            await params.trx.query(query, [expiredIdsResult]);
            await dal_manager_1.dbManager.accessLog.makeVisitorProfileHistoryAnonym(params.organizationId, expiredIdsResult);
        }
    }
    async deleteVisitorData(organizationId, visitorProfileId) {
        await this.dbClient.transaction(async (trx) => {
            await trx.withSchema(organizationId).from(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorProfiles).where("id", visitorProfileId).delete();
            await dal_manager_1.dbManager.accessLog.deleteVisitorProfileHistory(organizationId, visitorProfileId);
        });
    }
    async getUserIdsByVisitorProfileIds(organizationId, visitorProfileIds, trx) {
        let result = await dal_manager_1.dbManager.pgTransactionMainDb(async (trx) => {
            try {
                let resultOfMapping = (await trx.query(`
                    SELECT ovp."id" as "visitorProfileId", u."id" AS "visitorUserId"
                    FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorProfiles}" AS ovp
                    INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" AS uo
                        ON uo."id" = ovp."userOrganizationId" AND ovp."deletedAt" IS NULL AND uo."deletedAt" IS NULL
                    INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}" AS u ON u."id" = uo."userId" AND u."deletedAt" IS NULL
                    WHERE uo."organizationId" = $1 AND
                        ovp."id" = ANY($2::UUID[])
                `, [organizationId, visitorProfileIds])).rows;
                return resultOfMapping;
            }
            catch (err) {
                (0, dal_access_error_1.throwDbAccessBadRequestErrorTr)("DB Error while getUserIdsByVisitorProfileIds function!");
                dal_logger_1.dbLogger.error(err);
                return null;
            }
        });
        return result;
    }
    async getVisitorFullnameAndRegistrationPoint(organizationId, activeVisitId, trx) {
        const trxx = trx ?? this._pgPool;
        const { rows, rowCount } = await trxx.query(`SELECT 
				oav."startUtc",  (ovp.name || ' ' || CASE WHEN ovp.surname IS NULL THEN '' ELSE ovp.surname END) AS "visitorFullName", 
				vrp.name AS "visitorRegistrationPointName",
				vrp.id AS "visitorRegistrationPointId"
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationActiveVisits}" oav
            LEFT JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.visitorRegistrationPoints}" vrp
            ON oav."visitorRegistrationPointId" = vrp.id
            LEFT JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorProfiles}" ovp
            ON ovp.id = oav."organizationVisitorProfileId"
            WHERE oav.id = $1
            AND ovp."deletedAt" IS NULL
            AND vrp."deletedAt" IS NULL`, [activeVisitId]);
        if (rowCount > 0) {
            return {
                visitorFullname: rows[0].visitorFullName,
                visitorRegistrationPointName: rows[0].visitorRegistrationPointName,
                visitorRegistrationPointId: rows[0].visitorRegistrationPointId,
                date: rows[0].startUtc,
            };
        }
        else {
            const visitHistory = await dal_manager_1.dbManager.accessLog.getVisitorIdAndRegistrationPointIdFromHistory({ visitId: activeVisitId, organizationId });
            if (!visitHistory) {
                return null;
            }
            const visitorProfile = await this._pgPool.query(`SELECT (name || ' ' || CASE WHEN surname IS NULL THEN '' ELSE surname END) AS "visitorFullName" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorProfiles}"
				where id = $1`, [visitHistory.visitorProfileId]);
            const visitorRegionstrationPoint = await this._pgPool.query(`SELECT name AS "visitorRegistrationPointName", id AS "visitorRegistrationPointId" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.visitorRegistrationPoints}"
				where id = $1`, [visitHistory.visitorRegistrationPointId]);
            return {
                date: visitHistory.date,
                visitorFullname: visitorProfile.rows[0].visitorFullName,
                visitorRegistrationPointName: visitorRegionstrationPoint.rows[0].visitorRegistrationPointName,
                visitorRegistrationPointId: visitorRegionstrationPoint.rows[0].visitorRegistrationPointId,
            };
        }
    }
    async listActiveVisitsOfVisitorByUniqueId(organizationId, uniqueId, trx) {
        return (await trx
            .withSchema(organizationId)
            .from(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationActiveVisits + " as oav")
            .innerJoin(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorProfiles + " as ovp", "ovp.id", "oav.organizationVisitorProfileId")
            .where("oav.organizationId", organizationId)
            .whereNotIn("oav.state", [
            dal_constants_1.DalConstants.OrganizationVisitorStates.Forbidden,
            dal_constants_1.DalConstants.OrganizationVisitorStates.None,
            dal_constants_1.DalConstants.OrganizationVisitorStates.TemporaryPermit,
        ])
            .where("ovp.uniqueId", uniqueId)
            .whereNull("ovp.deletedAt")
            .select("oav.id", "oav.state", "oav.expectedStartUtc", "oav.expectedEndUtc")).map((v) => {
            return {
                id: v.id,
                state: v.state,
                expectedStartUtc: new Date(v.expectedStartUtc),
                expectedEndUtc: new Date(v.expectedEndUtc),
            };
        });
    }
    async listScheduledVisitsByExcel(organizationId, scheduledJobId, visitState, trx) {
        const { rows, rowCount } = await trx.query(`SELECT oav.id, oav."organizationVisitorProfileId"
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationActiveVisits}" as oav
            LEFT JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.visitorScheduledJob}" as vsj
            	ON oav.id = vsj."visitId"
            LEFT JOIN public."${dal_db_armon_schema_1.ArmonSchema.tableNames.scheduled_job}" as sj
           		ON sj.id = vsj."scheduledJobId"
            WHERE oav."organizationId" = $1
				AND sj.id = $2
				AND oav.state = $3
			`, [organizationId, scheduledJobId, visitState]);
        return rows.map((v) => {
            return {
                id: v.id,
                visitorProfileId: v.organizationVisitorProfileId,
                visitFields: v.visitFields,
            };
        });
    }
    async createVisitorExcelScheduledJob(organizationId, userId, params, trx) {
        return (await trx
            .withSchema("public")
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.scheduled_job)
            .insert({
            id: uuid_1.default.v4(),
            createdT: new Date(Date.now()),
            organizationId: organizationId,
            type: params.type,
            createdByUserId: userId,
            enabled: true,
            interval: params.interval,
            firstExecutionDate: params.executionDate,
            nextExecutionDate: params.executionDate,
            note: null,
            notificationId: null,
            data: params.data ?? null,
        })
            .returning("id"))[0];
    }
    async mapVisitToScheduledJob(organizationId, visitId, scheduledJobId, trx) {
        await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.visitorScheduledJob).insert({
            id: uuid_1.default.v4(),
            visitId: visitId,
            scheduledJobId: scheduledJobId,
        });
    }
    async checkPreregistrationForbiddenceOfVisitor(organizationId, visitorProfileId, expectedDate, locale, trx) {
        await this.dbClient
            .withSchema(organizationId)
            .from(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorStates)
            .where("organizationVisitorProfileId", "=", visitorProfileId)
            .whereNotNull("startUtc")
            .where("state", "=", dal_constants_1.DalConstants.OrganizationVisitorStates.Forbidden)
            .whereNull("deletedAt")
            .where(function () {
            this.whereNull("endUtc").orWhere("endUtc", ">", "NOW()");
        })
            .select("id", "startUtc", "endUtc")
            .transacting(trx)
            .then((rows) => {
            if (rows && rows.length > 0) {
                if (!expectedDate) {
                    if (rows[0].endUtc) {
                        throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.BAD_REQUEST, "ERRORS.VISITOR.FORBIDDEN_VISITOR.BETWEEN", { startUtc: new Date(rows[0].startUtc).toLocaleString(locale ?? "en"), endUtc: new Date(rows[0].endUtc).toLocaleString(locale ?? "en") }, true, true);
                    }
                    throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.BAD_REQUEST, "ERRORS.VISITOR.FORBIDDEN_VISITOR.INDEFINITE", { startUtc: new Date(rows[0].startUtc).toLocaleString(locale ?? "en") }, true, true);
                }
                for (let row of rows) {
                    if (!row.endUtc) {
                        if (new Date(expectedDate.startDate) > new Date(row.startUtc) || (expectedDate.endDate && new Date(expectedDate.startDate) > new Date(row.startUtc))) {
                            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.BAD_REQUEST, "ERRORS.VISITOR.FORBIDDEN_VISITOR.INDEFINITE", { startUtc: new Date(rows[0].startUtc).toLocaleString(locale ?? "en") }, true, true);
                        }
                    }
                    else if (!expectedDate.endDate && row.endUtc) {
                        if (new Date(expectedDate.startDate) > new Date(row.startUtc) && new Date(expectedDate.startDate) < new Date(row.endUtc)) {
                            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.BAD_REQUEST, "ERRORS.VISITOR.FORBIDDEN_VISITOR.BETWEEN", { startUtc: new Date(rows[0].startUtc).toLocaleString(locale ?? "en"), endUtc: new Date(rows[0].endUtc).toLocaleString(locale ?? "en") }, true, true);
                        }
                    }
                    else {
                        const expectedVisitRange = new moment_range_1.DateRange(expectedDate.startDate, expectedDate.endDate);
                        const forbiddenRange = new moment_range_1.DateRange((0, moment_1.default)(row.startUtc), (0, moment_1.default)(row.endUtc));
                        if (expectedVisitRange.intersect(forbiddenRange)) {
                            if (row.endUtc) {
                                throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.BAD_REQUEST, "ERRORS.VISITOR.FORBIDDEN_VISITOR.BETWEEN", { startUtc: new Date(rows[0].startUtc).toLocaleString(locale ?? "en"), endUtc: new Date(rows[0].endUtc).toLocaleString(locale ?? "en") }, true, true);
                            }
                            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.BAD_REQUEST, "ERRORS.VISITOR.FORBIDDEN_VISITOR.INDEFINITE", { startUtc: new Date(rows[0].startUtc).toLocaleString(locale ?? "en") }, true, true);
                        }
                    }
                }
            }
        });
    }
    async getActiveVisitState(organizationId, visitId, trx) {
        const visit = await trx.withSchema(organizationId).from(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationActiveVisits).where("id", visitId).select("state").first();
        if (!visit) {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.NOT_FOUND, "ERRORS.VISITOR.VISITNOTFOUND", { visitId: visitId }, true, false);
        }
        return visit.state;
    }
}
exports.PSQLDalAccessVisitor = PSQLDalAccessVisitor;
