"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PSQLDalAccessUserGroup = void 0;
const i18n_1 = __importDefault(require("i18n"));
const uuid_1 = __importDefault(require("uuid"));
const dal_access_error_1 = require("../dal.access.error");
const dal_access_rdb_userGroup_1 = require("../rdb/dal.access.rdb.userGroup");
const dal_manager_1 = require("../../dal.manager");
const api_error_1 = require("../../../api/api.error");
const predefined_permissions_1 = require("../../db/predefined/predefined.permissions");
const dal_db_armon_schema_1 = require("../../db/armon/dal.db.armon.schema");
const dal_constants_1 = require("../../dal.constants");
const app_enums_1 = require("../../../app.enums");
const dal_access_psql_common_1 = require("./dal.access.psql.common");
class PSQLDalAccessUserGroup extends dal_access_rdb_userGroup_1.RDBDalAccessUserGroup {
    constructor(knex, pgPool) {
        super(knex, pgPool);
    }
    async listUsersOfUserGroups(organizationId, userGroupIds) {
        let queryParamIndex = 1;
        let queryParams = [];
        let query = `SELECT DISTINCT uo."userId" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroups}" ug
        INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupUserOrganizations}" uguo
        ON ug.id = uguo."userGroupId"
        INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" uop
        ON uop."userOrganizationId" = uguo."userOrganizationId"
        INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo
        ON uo.id = uop."userOrganizationId"
        WHERE uo."deletedAt" IS NULL
        AND uo."deletedAt" IS NULL
        AND uguo."deletedAt" IS NULL
        AND uop."deletedAt" IS NULL
        AND ug."deletedAt" IS NULL
        AND ug.id = ANY($${queryParamIndex++}::uuid[])
        AND ug."organizationId" = $${queryParamIndex++}
        `;
        queryParams.push(userGroupIds, organizationId);
        let userIds = (await this._pgPool.query(query, queryParams)).rows.map((r) => r.userId);
        return userIds;
    }
    async listUsersOfUserGroup(organizationId, userGroupId, filter) {
        let qb = this.dbClient
            .withSchema(organizationId)
            .from("userGroups as ug")
            .innerJoin("userGroupUserOrganizations as uguo", "ug.id", "uguo.userGroupId")
            .innerJoin("userOrganizationProfiles as uop", "uguo.userOrganizationId", "uop.userOrganizationId")
            .whereNull("uguo.deletedAt")
            .whereNull("ug.deletedAt")
            .whereNull("uop.deletedAt")
            .where("ug.id", "=", userGroupId)
            .where("ug.organizationId", "=", organizationId)
            .distinct("uo.id", "uop.name", "uop.surname")
            .select()
            .orderBy("uop.name")
            .orderBy("uop.surname");
        if (filter.filter) {
            qb.where("uop.profileText", "LIKE", this.dbClient.raw("'%' || lower(unaccent(?)) || '%'", filter.filter));
        }
        let qbCount = await qb.clone().countDistinct("uo.id").first();
        let count = qbCount && qbCount.count ? parseInt(qbCount.count) : 0;
        let result = {
            pagination: {
                total: count,
                skip: 0,
                take: 0,
            },
            items: [],
        };
        if (count === 0) {
            return Promise.resolve(result);
        }
        result.pagination.take = filter.take || 0;
        result.pagination.skip = filter.skip || 0;
        qb.distinct("uo.id", "uop.name", "uop.surname").select().orderBy("uop.name").orderBy("uop.surname");
        if (filter.skip) {
            qb.offset(filter.skip);
        }
        if (filter.take) {
            qb.limit(filter.take);
        }
        let userOrganizationIds = (await qb).map((m) => m.id);
        result.items = await dal_manager_1.dbManager.accessIdentity.getSearchIdentityResultFromUserOrganizationIds(userOrganizationIds, organizationId);
        return Promise.resolve(result);
    }
    async listUserGroupsByIds(organizationId, userGroupIds) {
        let result = [];
        let qb = this.dbClient.withSchema(organizationId).from("userGroups as ug").whereIn("ug.id", userGroupIds).whereNull("ug.deletedAt");
        return await qb
            .orderByRaw("ug.name")
            .select("ug.id", "ug.name", "ug.colorCode", "ug.organizationUnitId")
            .then((rows) => {
            return rows.map((t) => {
                return {
                    id: t.id,
                    name: t.name,
                    colorCode: t.colorCode,
                    organizationUnitId: t.organizationUnitId,
                };
            });
        });
    }
    async listUserGroupsByFilter(params) {
        const trxx = params.trx || this._pgPool;
        let queryParamIndex = 1;
        const queryParams = [];
        const queryWheres = [];
        let qSelect = `
			SELECT ug.id, ug.name, ug."colorCode", ug."organizationUnitId"
		`;
        let qCount = `
			SELECT count(*) AS count
		`;
        let q = `
			FROM "${params.organizationId}"."userGroups" AS ug
		`;
        if (params.apiFilter.userGroupIds?.length > 0) {
            queryWheres.push(`
				ug.id = ANY ($${queryParamIndex++}::UUID[])
			`);
            queryParams.push(params.apiFilter.userGroupIds);
        }
        if (params.apiFilter.filter) {
            queryWheres.push(`
				ug.name ilike '%' || $${queryParamIndex++} || '%'
			`);
            queryParams.push(params.apiFilter.filter);
        }
        queryWheres.push(`ug."deletedAt" IS NULL`);
        q += ` WHERE ${queryWheres.join(" AND ")}`;
        const dbCountResult = await trxx.query(qCount + q, queryParams);
        q += " ORDER BY ug.name ";
        if (params.apiFilter.pagination.skip) {
            q += `OFFSET $${queryParamIndex++}`;
            queryParams.push(params.apiFilter.pagination.skip);
        }
        if (params.apiFilter.pagination.take) {
            q += `LIMIT $${queryParamIndex++}`;
            queryParams.push(params.apiFilter.pagination.take);
        }
        const dbResult = await trxx.query(qSelect + q, queryParams);
        return Promise.resolve({
            pagination: {
                total: parseInt(dbCountResult.rows[0].count),
                skip: params.apiFilter.pagination.skip || 0,
                take: params.apiFilter.pagination.take || 100,
            },
            items: dbResult.rows,
        });
    }
    async listUserGroupsForControlPanel(organizationId) {
        let result = [];
        let userGroups = await this.dbClient.withSchema(organizationId).from("userGroups as ug").where("ug.organizationId", organizationId).whereNull("ug.deletedAt").select();
        for (const userGroup of userGroups) {
            let users = await this.dbClient
                .withSchema(organizationId)
                .from("userGroupUserOrganizations as ug")
                .innerJoin("userOrganizations as uo", "uo.id", "ug.userOrganizationId")
                .where("ug.userGroupId", userGroup.id)
                .whereNull("ug.deletedAt")
                .whereNull("uo.deletedAt")
                .select("uo.userId");
            result.push({
                userGroupId: userGroup.id,
                userIds: users.map((u) => u.userId),
            });
        }
        return Promise.resolve(result);
    }
    async addUserGroup(organizationId, request, hasOrganizationWideRight, jwt) {
        if (!hasOrganizationWideRight && !request.organizationUnitId) {
            throw (0, api_error_1.generateForbiddenError)({ message: "You do not have permission to add user group for organization wide!" });
        }
        else if (request.organizationUnitId && !hasOrganizationWideRight && !jwt.isPermittedForUnit(organizationId, predefined_permissions_1.Permissions.userGroup.getWrite(), request.organizationUnitId)) {
            let units = await this.dbClient
                .withSchema(organizationId)
                .from("vW_UserOrganizationUnitHierarchys")
                .where("userId", jwt.userId)
                .where("organizationId", organizationId)
                .where("descenderOrganizationUnitId", request.organizationUnitId)
                .select();
            if (!units || units.length < 1) {
                throw (0, api_error_1.generateForbiddenError)({ message: "You do not have permission to add user group for the specified organization unit!" });
            }
        }
        let existingQb = this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.userGroups)
            .where("organizationId", organizationId)
            .where("name", request.name)
            .whereNull("deletedAt");
        if (request.organizationUnitId) {
            existingQb = existingQb.where("organizationUnitId", request.organizationUnitId);
        }
        let existing = await existingQb.first();
        if (existing && existing.id) {
            throw (0, api_error_1.generateBadRequestApiError)({ message: "This group is already exists!" });
        }
        let userGroup = {
            id: uuid_1.default.v4(),
            name: request.name,
            organizationUnitId: request.organizationUnitId ? request.organizationUnitId : null,
            colorCode: request.colorCode,
            organizationId: organizationId,
            deletedAt: null,
            createdAt: new Date(),
            updatedAt: new Date(),
        };
        await this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.userGroups).insert(userGroup);
        return Promise.resolve({
            id: userGroup.id,
            name: userGroup.name,
            organizationUnitId: userGroup.organizationUnitId,
            colorCode: userGroup.colorCode,
        });
    }
    async updateUserGroup(params) {
        if (params.request.organizationUnitId &&
            !params.hasOrganizationWideRight &&
            !params.jwt.isPermittedForUnit(params.organizationId, predefined_permissions_1.Permissions.userGroup.getWrite(), params.request.organizationUnitId)) {
            const { rows: unitRows } = await params.trx.query(`
				SELECT * FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.viewNames.vW_UserOrganizationUnitHierarchys}"
				WHERE "userId" = $1
				AND "descenderOrganizationUnitId" = $2`, [params.jwt.userId, params.request.organizationUnitId]);
            if (unitRows.length === 0) {
                throw (0, api_error_1.generateForbiddenError)({ message: "You do not have permission to add user group for the specified organization unit!" });
            }
        }
        let qx = 1;
        const qb = [];
        let query = `SELECT * FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroups}"
					WHERE name = $${qx++}
					AND id !=  $${qx++}
					AND "deletedAt" IS NULL`;
        qb.push(params.request.name, params.userGroupId);
        if (params.request.organizationUnitId) {
            query += ` "organizationUnitId" = $${qx++}`;
            qb.push(params.request.organizationUnitId);
        }
        const { rows: groupRows, rowCount: groupCountWithSameName } = await params.trx.query(query, qb);
        if (groupCountWithSameName > 0) {
            throw (0, api_error_1.generateBadRequestApiError)({ message: "This group that you specified is already exists!" });
        }
        const userGroup = await params.trx.query(`
			SELECT * FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroups}"
			WHERE id = $1
			AND "deletedAt" IS NULL
		`, [params.userGroupId]);
        if (userGroup.rowCount === 0) {
            throw (0, api_error_1.generateNotFoundApiError)({ message: "The group is not found" });
        }
        await params.trx.query(`UPDATE "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroups}"
		SET name = $1,
			"organizationUnitId" = $2,
			"colorCode" = $3,
			"organizationId" = $4,
			"updatedAt" = $5
		WHERE id = $6 `, [params.request.name, params.request.organizationUnitId, params.request.colorCode, params.organizationId, new Date(), params.userGroupId]);
        return {
            id: params.userGroupId,
            ...params.request,
        };
    }
    async listUserGroups(organizationId, hasOrganizationWideRight, jwt) {
        let permittedUnits = jwt.getPermittedUnitIdsFor(organizationId, predefined_permissions_1.Permissions.userGroup.getRead());
        let qb = this.dbClient.withSchema(organizationId).from("userGroups as ug").where("ug.organizationId", organizationId).whereNull("ug.deletedAt");
        if (!hasOrganizationWideRight) {
            let subquery = await this.dbClient
                .withSchema(organizationId)
                .select(this.dbClient.raw(` ou."id", 
                                        CASE WHEN ou."ancestorIds" = '[]' OR ou."ancestorIds" = '' OR ou."ancestorIds" IS null THEN ARRAY[]::uuid[] 
                                        ELSE regexp_split_to_array(ou."ancestorIds", E',')::uuid[] 
                                        END as "ancestorIds" `))
                .from("organizationUnits as ou")
                .where(this.dbClient.raw(` ou."deletedAt" IS NULL AND ou."organizationId" = ? AND ou."id" = ANY( ? ) `, [organizationId, permittedUnits]));
            let ancestorIds = [];
            for (let row of subquery) {
                ancestorIds = ancestorIds.concat(row.ancestorIds);
            }
            qb.innerJoin("organizationUnits as uo", "uo.id", "ug.organizationUnitId").whereNull("uo.deletedAt").whereIn("uo.id", ancestorIds).orWhereIn("uo.id", permittedUnits);
        }
        return Promise.resolve(await qb.select("ug.id", "ug.name", "ug.colorCode", "ug.organizationUnitId"));
    }
    async removeUserGroup(organizationId, userGroupId, hasOrganizationWideRight, jwt, trx) {
        const existanceCheckDbResult = await trx.query(`
			SELECT *
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroups}"
			WHERE "deletedAt" IS NULL
				AND id = $1
			`, [userGroupId]);
        if (!existanceCheckDbResult.rowCount) {
            throw (0, api_error_1.generateNotFoundApiError)({ message: "User group is not found!" });
        }
        if (!hasOrganizationWideRight) {
            let permittedUnits = jwt.getPermittedUnitIdsFor(organizationId, predefined_permissions_1.Permissions.userGroup.getWrite());
            const permittedUserGroupsDbResult = await trx.query(`
			SELECT ug.id
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroups}" ug
			WHERE ug."deletedAt" IS NULL
				AND ug.id = $1
				AND (
					ug."organizationUnitId" IN (
						SELECT DISTINCT (UNNEST(sq."ancestorIds")) AS "ancestors" FROM (
							SELECT
							CASE WHEN ou."ancestorIds" = '[]' OR ou."ancestorIds" = '' OR ou."ancestorIds" IS NULL THEN ARRAY[]::uuid[] 
								ELSE regexp_split_to_array(ou."ancestorIds", E',')::uuid[] 
								END AS "ancestorIds"
							FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" ou
							WHERE ou."deletedAt" IS NULL
								AND ou.id = ANY ($2::UUID[])
						) sq
					) OR ug."organizationUnitId" = ANY ($2::UUID[])
				)

			`, [userGroupId, permittedUnits]);
            if (!permittedUserGroupsDbResult.rowCount) {
                throw (0, api_error_1.generateForbiddenError)({ message: "You do not have permission to remove that group!" });
            }
        }
        let checkAutoShiftUsage = (await trx.query(`
			SELECT COUNT(asrsg.id)::integer, asrs.name
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.autoShiftRuleSetGroup}" asrsg
			INNER JOIN 
					(	SELECT "ruleSetId" 
						FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.autoShiftRuleSetGroup}"
						WHERE "groupId" = $1
					) as sq
			ON asrsg."ruleSetId" = sq."ruleSetId"
			INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.autoShiftRuleSet}" asrs
				ON asrsg."ruleSetId" = asrs."id"
			GROUP BY asrsg."ruleSetId", asrs.name
			`, [userGroupId])).rows?.filter((item) => item.count === 1);
        if (checkAutoShiftUsage && checkAutoShiftUsage.length > 0) {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.PRECONDITION_FAILED, "ERRORS.PACS.GROUPUSINGINAUTOSHIFT", {
                autoShiftName: checkAutoShiftUsage.map((elem) => elem.name).join(", "),
            });
        }
        await trx.query(`
			UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroups}" 
			SET "deletedAt" = now() 
			WHERE id = $1 
				AND "deletedAt" IS NULL
			`, [userGroupId]);
        await trx.query(`
			UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupUserOrganizations}" 
			SET "deletedAt" = now() 
			WHERE "userGroupId" = $1 
				AND "deletedAt" IS NULL
			`, [userGroupId]);
        await trx.query(`
			DELETE FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.autoShiftRuleSetGroup}" 
			WHERE "groupId" = $1
			`, [userGroupId]);
    }
    async getIdNamePairsOfUserGroups(organizationId, userGroupIds, trx) {
        let transactionScope = async (trx) => {
            let dbResult = await trx
                .withSchema(organizationId)
                .table("userGroups")
                .whereIn("id", userGroupIds)
                .where("organizationId", organizationId)
                .whereNull("deletedAt")
                .select("id", "name");
            return Promise.resolve(dbResult);
        };
        if (trx) {
            return transactionScope(trx);
        }
        else {
            return this.dbClient.transaction(transactionScope);
        }
    }
    async updateUserGroupUserMappings(organizationId, userGroupId, request, hasOrganizationWideRight, jwt) {
        let opTime = new Date();
        let removeUsers = await this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations)
            .whereNull("deletedAt")
            .where("organizationId", organizationId)
            .whereIn("userId", request.removedUserIds)
            .select("id", "userId");
        await this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupUserOrganizations)
            .whereIn("userOrganizationId", removeUsers.map((r) => r.id))
            .update({
            deletedAt: opTime,
        });
        let addUsers = await this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations)
            .whereNull("deletedAt")
            .where("organizationId", organizationId)
            .whereIn("userId", request.addedUserIds)
            .select("id", "userId");
        let addUserGroupMappings = [];
        for (const addUser of addUsers) {
            addUserGroupMappings.push({
                id: uuid_1.default.v4(),
                createdAt: opTime,
                updatedAt: opTime,
                deletedAt: null,
                userOrganizationId: addUser,
                userGroupId: userGroupId,
            });
        }
        const valuesPlaceholder = addUserGroupMappings.map(() => "(?, ?, ?, ?, ?, ?)").join(", ");
        const flattenedValues = addUserGroupMappings.flatMap((item) => [item.id, item.createdAt, item.updatedAt, item.deletedAt, item.userOrganizationId, item.userGroupId]);
        await this.dbClient.raw(`
			INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupUserOrganizations}"
			(id, "createdAt", "updatedAt", "deletedAt", "userOrganizationId", "userGroupId")
			VALUES ${valuesPlaceholder}
			ON CONFLICT ("userOrganizationId", "userGroupId") WHERE "deletedAt" IS NULL
			DO NOTHING;
			`, [...flattenedValues]);
        return Promise.resolve();
    }
    async getUserOrganizationAccessRules(organizationId, userOrganizationId) {
        let { rows } = await this._pgPool.query(`
        SELECT  ars.id, 
		ars.name, 
		ars."regionId", 
		array_agg(json_build_object(
                'id', ar.id,
                'accessRuleSetId', ar."accessRuleSetId",
                'direction', ar.direction,
                'type', ar.type,
                'acceptStatus', ar."acceptStatus",
                'parameters', ar.parameters)::jsonb)  AS rules,
		ars."isActive",
		ars."uncoveredAccessStatus" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessRuleSets}" ars
        INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessRules}" ar
        ON ars.id = ar."accessRuleSetId"
        WHERE ars.id = ANY(SELECT "accessRuleSetId" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupAccessRuleSets}"
            WHERE "userGroupId" = ANY(SELECT "userGroupId" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupUserOrganizations}"
                WHERE "userOrganizationId" = $1
                AND "deletedAt" IS NULL)
            )
        AND ars."deletedAt" IS NULL
        GROUP BY  (ars.id, ars.name, ars."regionId", ars."isActive", ars."uncoveredAccessStatus")
        `, [userOrganizationId]);
        return rows.map((r) => {
            return {
                id: r.id,
                name: r.name,
                regionId: r.regionId,
                rules: r.rules,
                isActive: r.isActive,
                uncoveredAccessStatus: r.uncoveredAccessStatus,
            };
        });
    }
    async updateUserGroupLimitRules(params) {
        const { accessControlPointId, rulesToUpdate, actionDateISO, trx, organizationId } = params;
        for (const rule of rulesToUpdate) {
            await trx.query(`
				UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupAccessRuleSets}" AS ugars
					SET "currentCapacityUsage" = $1, "lastUpdateAccessControlPointId" = $2, "lastActionDateISO" = $3
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessRules}" AS ar
				WHERE ugars."userGroupId" = $4 
					AND ugars."accessRuleSetId" = ar."accessRuleSetId"
					AND ar."id" = $5
					`, [rule.count, accessControlPointId, actionDateISO, rule.userGroupId, rule.accessRuleId]);
        }
    }
    async insertUsersIntoGroup(organizationId, requesterUserId, trx, params) {
        const userFilter = (0, dal_access_psql_common_1.getReportUserFilterForPgClient)({
            organizationId,
            requesterUserId,
            idBasedUserFilter: {
                userIds: params.userIds,
                applyOrganizationUnitFilterHierarchically: true,
                userOrganizationStatus: dal_constants_1.DalConstants.IdentityStatusType.All,
            },
            requiredOrganizationUnitWidePermissions: ["i:b"],
            requiredOrganizationWidePermissions: ["i:b"],
            bindingKeys: [],
            specificSelectItems: ["userId"],
        });
        const { rows: UserFilter } = await trx.query(userFilter.query, userFilter.bindingKeys);
        const allowedUserIds = UserFilter.map((filter) => filter.userId);
        for (const userId of params.userIds) {
            if (!allowedUserIds.includes(userId)) {
                (0, dal_access_error_1.throwDbAccessAuthorizationErrorTr)("ERRORS.IDENTITY.NOT_AUTH_FOR_SPECIFIED_USER", { id: userId });
            }
        }
        let qx = 1;
        const qb = [];
        const qValues = [];
        const [{ rows: userGroupRows }, { rows: userOrganizationRows }] = await Promise.all([
            trx.query(`
				SELECT id, name FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroups}"
				WHERE id = $1
				AND "deletedAt" IS NULL
				`, [params.groupId]),
            trx.query(`SELECT id as "userOrganizationId", "userId" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}"
					WHERE "userId" = ANY($1::uuid[])
					AND "deletedAt" IS NULL `, [params.userIds]),
        ]);
        const userOrganizationIds = userOrganizationRows.map((row) => row.userOrganizationId);
        if (userGroupRows.length === 0) {
            (0, dal_access_error_1.throwDbAccessNotFoundErrorTr)("ERRORS.GROUP.GROUP_NOT_FOUND");
        }
        if (userOrganizationRows.length < params.userIds.length) {
            const { l: locale } = await dal_manager_1.dbManager.accessRedisCache.getUserNotificationCache({ userId: requesterUserId, organizationId, trx });
            const unMatchedUserIds = params.userIds.filter((userId) => !userOrganizationRows.map((r) => r.userId).includes(userId));
            (0, dal_access_error_1.throwDbAccessNotFoundError)(i18n_1.default.__({ phrase: "ERRORS.IDENTITY.USER_NOT_FOUND", locale }, { id: JSON.stringify(unMatchedUserIds) }));
        }
        const { rows: usersThatAlreadyInGroup } = await trx.query(`SELECT "userOrganizationId" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupUserOrganizations}"
				WHERE "userGroupId" = $1
				AND "userOrganizationId" = ANY($2::uuid[])
				AND "deletedAt" IS NULL
				`, [params.groupId, userOrganizationIds]);
        const userOrganizationIdsThatShouldBeAdded = userOrganizationIds.filter((userOrganizationId) => usersThatAlreadyInGroup.map((r) => r.userOrganizationId).indexOf(userOrganizationId) < 0);
        for (const userOrganizationId of userOrganizationIdsThatShouldBeAdded) {
            qValues.push(`(uuid_generate_v4(), now(), now(), $${qx++}, $${qx++})`);
            qb.push(userOrganizationId, params.groupId);
        }
        if (userOrganizationIdsThatShouldBeAdded.length > 0) {
            await trx.query(`INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupUserOrganizations}"(
					id, "createdAt", "updatedAt", "userOrganizationId", "userGroupId")
					VALUES ${qValues.join(" , ")}
				ON CONFLICT ("userOrganizationId", "userGroupId") WHERE "deletedAt" IS NULL
    			DO NOTHING;`, qb);
            const { rows: userIdsThatShouldBeAddedRows } = await trx.query(`
				SELECT "userId" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}"
				WHERE id = ANY($1::uuid[])
			`, [userOrganizationIdsThatShouldBeAdded]);
            await dal_manager_1.dbManager.accessLog.addMultipleUserActionHistoryItemsPg(organizationId, requesterUserId, trx, userIdsThatShouldBeAddedRows.map((r) => {
                return {
                    oId: organizationId,
                    o: requesterUserId,
                    u: new Date(),
                    c: dal_constants_1.DalConstants.UserActionCategory.Main,
                    tg: ["Identity", "UserGroup"],
                    t: dal_constants_1.DalConstants.UserMainActionType.AssignUserToUserGroup,
                    d: {
                        id: r.userId,
                        c: JSON.stringify([
                            {
                                f: "userGroupName",
                                o: null,
                                n: userGroupRows[0].name,
                                i: true,
                            },
                        ]),
                    },
                };
            }));
        }
    }
    async removeUserFromGroup(organizationId, requesterUserId, trx, params) {
        const userFilter = (0, dal_access_psql_common_1.getReportUserFilterForPgClient)({
            organizationId,
            requesterUserId,
            idBasedUserFilter: {
                userIds: [params.userId],
                applyOrganizationUnitFilterHierarchically: true,
                userOrganizationStatus: dal_constants_1.DalConstants.IdentityStatusType.All,
            },
            requiredOrganizationUnitWidePermissions: ["i:b"],
            requiredOrganizationWidePermissions: ["i:b"],
            bindingKeys: [],
            specificSelectItems: ["userId"],
        });
        const { rows: UserFilter } = await trx.query(userFilter.query, userFilter.bindingKeys);
        const allowedUserIds = UserFilter.map((filter) => filter.userId);
        if (!allowedUserIds.includes(params.userId)) {
            (0, dal_access_error_1.throwDbAccessAuthorizationErrorTr)("ERRORS.IDENTITY.NOT_AUTH_FOR_SPECIFIED_USER");
        }
        const [{ rows: userGroupRows }, { rows: userOrganizationRows }] = await Promise.all([
            trx.query(`
				SELECT id, name FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroups}"
				WHERE id = $1
				AND "deletedAt" IS NULL
				`, [params.groupId]),
            trx.query(`SELECT id FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}"
					WHERE "userId" = $1
					AND "deletedAt" IS NULL`, [params.userId]),
        ]);
        if (userGroupRows.length === 0) {
            (0, dal_access_error_1.throwDbAccessNotFoundErrorTr)("ERRORS.GROUP.GROUP_NOT_FOUND");
        }
        if (userOrganizationRows.length === 0) {
            const { l: locale } = await dal_manager_1.dbManager.accessRedisCache.getUserNotificationCache({ userId: requesterUserId, organizationId, trx });
            (0, dal_access_error_1.throwDbAccessNotFoundError)(i18n_1.default.__({ phrase: "ERRORS.IDENTITY.USER_NOT_FOUND", locale }, { id: params.userId }));
        }
        const { rowCount } = await trx.query(`UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupUserOrganizations}"
				SET "deletedAt" = now(),
					"updatedAt" = now()
			 	WHERE "userOrganizationId" = $1
				AND "userGroupId" = $2
				AND "deletedAt" IS NULL
			`, [userOrganizationRows[0].id, params.groupId]);
        if (rowCount > 0) {
            dal_manager_1.dbManager.accessLog.addUserActionHistoryItemPg(organizationId, requesterUserId, trx, {
                oId: organizationId,
                o: requesterUserId,
                u: new Date(),
                c: dal_constants_1.DalConstants.UserActionCategory.Main,
                tg: ["Identity", "UserGroup"],
                t: dal_constants_1.DalConstants.UserMainActionType.RemoveUserFromUserGroup,
                d: {
                    id: params.userId,
                    c: JSON.stringify([
                        {
                            f: "userGroupName",
                            o: userGroupRows[0].name,
                            n: null,
                            i: true,
                        },
                    ]),
                },
            });
        }
    }
    async getUserGroupInfoUsingUserId(organizationId, userId) {
        return await dal_manager_1.dbManager.systemTransaction(async (client) => {
            const result = (await client.query(`
				SELECT "userGroupId" 
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupUserOrganizations}" as uguo
				INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" as uo
					ON uo."id" = uguo."userOrganizationId"
				WHERE	uo."userId" = $1 AND
						uo."isDisabled" = false AND
						uo."deletedAt" IS NULL AND
						uguo."deletedAt" IS  NULL
				;

				`, [userId])).rows;
            return result;
        });
    }
}
exports.PSQLDalAccessUserGroup = PSQLDalAccessUserGroup;
