"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PSQLDalAccessUser = void 0;
const lodash_1 = __importDefault(require("lodash"));
const luxon_1 = require("luxon");
const moment_1 = __importDefault(require("moment"));
const sharp_1 = __importDefault(require("sharp"));
const ua_parser_1 = __importDefault(require("ua-parser"));
const uuid_1 = __importDefault(require("uuid"));
const api_error_1 = require("../../../api/api.error");
const app_enums_1 = require("../../../app.enums");
const app_logs_1 = require("../../../app.logs");
const business_hooks_1 = require("../../../business/business.hooks");
const business_visitor_1 = require("../../../business/visitor/business.visitor");
const restapi_1 = require("../../../lib/es/models/restapi");
const dal_constants_1 = require("../../dal.constants");
const dal_manager_1 = require("../../dal.manager");
const dal_utils_1 = require("../../dal.utils");
const dal_db_armon_schema_1 = require("../../db/armon/dal.db.armon.schema");
const predefined_permissions_1 = require("../../db/predefined/predefined.permissions");
const predefined_roles_1 = require("../../db/predefined/predefined.roles");
const dal_access_error_1 = require("../dal.access.error");
const dal_memcache_1 = require("../dal.memcache");
const dal_access_rdb_user_1 = require("../rdb/dal.access.rdb.user");
const app_config_1 = require("../../../app.config");
const dynamicFormCurrent_1 = require("../../../lib/es/models/dynamicFormCurrent");
const deviceParser = require("device");
const Cursor = require("pg-cursor");
class PSQLDalAccessUser extends dal_access_rdb_user_1.RDBDalAccessUser {
    constructor(knex, pgPool) {
        super(knex, pgPool);
    }
    async getUserRoleId(params) {
        const { rows } = await (params.trx ?? this._pgPool).query(`SELECT "roleId"
				FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}"
				WHERE "deletedAt" is NULL 
					AND "userId" = $1`, [params.userId]);
        return rows[0].roleId;
    }
    async searchUsersByLicensePlate(params) {
        const { rows } = await (params.trx ?? this._pgPool).query(`SELECT id, "userId", type, data
				FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials}"
				WHERE "deletedAt" is NULL AND type = 6 AND data = $1`, [params.licensePlate]);
        return rows;
    }
    async getUserId(username) {
        let user = await this.dbClient.withSchema("public").from(dal_db_armon_schema_1.ArmonSchema.tableNames.userList).first("id").whereRaw("lower(username) = lower(?)", username);
        if (!user) {
            return null;
        }
        return user.id;
    }
    async getUserOrganizationCredential(organizationId, credentialId) {
        return await this.dbClient.withSchema(organizationId).from(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials).whereNull("deletedAt").where("id", credentialId).first();
    }
    async getOrganizationSystemAdminUserEmails(organizationId) {
        const { rows } = await this.pgPool.query(`SELECT uo."userId", uop.email AS "userOrganizationEmail", up.email AS "userEmail" from "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo
            LEFT JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" uop
            ON uo.id = uop."userOrganizationId"
            LEFT JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userProfiles}" up
            ON uo."userId" = up."userId"
            WHERE uo."roleId" = (
            SELECT id FROM "${organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.roles} r
            WHERE r."typeId" = $2
            AND r."organizationId" = $1)
            AND uo."organizationId"= $1
            AND uo."deletedAt" IS NULL
			AND uo."isDisabled" IS false
			AND uop."deletedAt" IS NULL
			AND up."deletedAt" IS NULL

        `, [organizationId, predefined_roles_1.PredefinedRoles.SystemAdministrator.id]);
        return rows;
    }
    async mapUserWithOrganization(organizationId, userId) {
        return this.dbClient.transaction(async (trx) => {
            let userOrganization = await trx
                .withSchema(organizationId)
                .from(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations)
                .where("userId", userId)
                .where("organizationId", organizationId)
                .first("id", "deletedAt", "roleId");
            if (userOrganization) {
                await trx.withSchema(organizationId).from(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations).where("userId", userId).where("organizationId", organizationId).update({
                    updatedAt: new Date(),
                    deletedAt: null,
                });
            }
            else {
                let role = await trx
                    .withSchema(organizationId)
                    .from(dal_db_armon_schema_1.ArmonSchema.tableNames.roles)
                    .where("typeId", predefined_roles_1.PredefinedRoles.StandartUser.id)
                    .where("organizationId", organizationId)
                    .first("id");
                if (!role) {
                    (0, dal_access_error_1.throwDbAccessNotFoundError)("StandartUser role does not exists for the organization organization Id: " + organizationId);
                }
                let now = new Date();
                let userOrganizationId = uuid_1.default.v4();
                let userProfile = await trx.withSchema(organizationId).from(dal_db_armon_schema_1.ArmonSchema.tableNames.userProfiles).where("userId", userId).first("name", "surname");
                if (!userProfile) {
                    (0, dal_access_error_1.throwDbAccessNotFoundError)("User has not a userProfile userId: " + userId);
                }
                await trx.withSchema(organizationId).from(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations).insert({
                    id: userOrganizationId,
                    createdAt: now,
                    updatedAt: now,
                    deletedAt: null,
                    userId: userId,
                    organizationId: organizationId,
                    roleId: role.id,
                    isDisabled: false,
                });
                await trx.withSchema(organizationId).from(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles).insert({
                    id: uuid_1.default.v4(),
                    createdAt: now,
                    updatedAt: now,
                    deletedAt: null,
                    userId: userId,
                    userOrganizationId: userOrganizationId,
                    name: userProfile.name,
                    surname: userProfile.surname,
                    uniqueId: uuid_1.default.v4(),
                });
            }
        });
    }
    async getUserInfoByCredentialData(params) {
        const { rows, rowCount } = await (params.trx ?? this._pgPool).query(`SELECT "credentialId", "userId", "organizationId", data, "fullName", "roleId", "profileId" 
				FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.viewNames.vW_CredentialUserProfiles}"
				WHERE data = $1`, [params.credentialData]);
        return rowCount > 0 ? rows[0] : null;
    }
    async obtainUserSelectionSession(organizationId, params, trx) {
        const { rows: existingSessionRows, rowCount: existingSessionRowCount } = await trx.query(`
			SELECT id, "userId", "expirationUtc" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userSelectionSessions}"
			WHERE "relatedItemId" = $1
			AND type = $2
		`, [params.relatedItemId, params.type]);
        const result = {
            upserted: false,
            sessionId: null,
            expirationUtc: null,
            justRefreshed: false,
            overwritten: null,
        };
        const expirationUtc = (0, moment_1.default)().add(20, "minutes").toDate();
        let create = false;
        if (existingSessionRowCount > 0) {
            const existingSession = existingSessionRows[0];
            if (existingSession.userId === params.requestUserId) {
                await trx.query(`
					UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userSelectionSessions}"
					SET "expirationUtc" = $2
					WHERE  id = $1
				`, [existingSession.id, expirationUtc]);
                result.justRefreshed = true;
                result.expirationUtc = expirationUtc;
                result.sessionId = existingSession.id;
                result.upserted = true;
            }
            else {
                const { rows: ownerRows, rowCount: ownerRowCount } = await trx.query(`
					SELECT id, name, surname, "uniqueId" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}"
					WHERE "userId" = $1
					AND "deletedAt" IS NULL`, [existingSession.userId]);
                const owner = ownerRowCount > 0 ? ownerRows[0] : null;
                const overwritten = {
                    ownerUserId: existingSession.userId,
                    sessionId: existingSession.id,
                    expirationUtc: existingSession.expirationUtc,
                    captionLines: !owner ? [] : [[owner.name, owner.surname].join(" "), owner.uniqueId],
                };
                result.overwritten = overwritten;
                if (params.forceToCreate) {
                    await trx.query(`DELETE FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userSelectionSessions}"
						WHERE id = $1`, [existingSession.id]);
                    create = true;
                }
                else {
                    result.upserted = false;
                }
            }
        }
        else {
            create = true;
        }
        if (create) {
            const session = {
                id: uuid_1.default.v4(),
                organizationId: organizationId,
                userId: params.requestUserId,
                relatedItemId: params.relatedItemId,
                expirationUtc: expirationUtc,
                type: params.type,
            };
            await trx.query(`INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userSelectionSessions}"
				(id, "organizationId", "userId", "relatedItemId", "expirationUtc", type)
				VALUES ($1, $2, $3, $4, $5, $6)`, [session.id, session.organizationId, session.userId, session.relatedItemId, session.expirationUtc, session.type]);
            let paramsIndex = 1;
            const queryParams = [];
            let insertQ = null;
            let whereQ = null;
            let groupByQ = "GROUP BY upp.id";
            switch (params.type) {
                case dal_constants_1.DalConstants.UserSelectionSessionType.UserGroup:
                    insertQ = `
					INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userSelectionSessionActions}"
					(id, "sessionId", "userId", "originalMappingId", "action")
					SELECT uuid_generate_v4(), $${paramsIndex++}, uo."userId", uguo.id, $${paramsIndex++}
					FROM "${session.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupUserOrganizations}" as "uguo"
					INNER JOIN "${session.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" as "uo"
						ON "uguo"."userOrganizationId" = "uo"."id" 
							AND uo."isDisabled" IS FALSE
							AND "uo"."deletedAt" IS NULL
						`;
                    queryParams.push(session.id);
                    queryParams.push(dal_constants_1.DalConstants.UserSelectionSessionAction.Unchanged);
                    whereQ = `
					WHERE "uguo"."userGroupId" = $${paramsIndex++}
						AND "uguo"."deletedAt" IS NULL 
					`;
                    queryParams.push(params.relatedItemId);
                    if (!params.isAuthorizedForAllUsers) {
                        insertQ += `
						INNER JOIN "${session.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}" as "uoou"
							ON "uoou"."userOrganizationId" = uo.id
						INNER JOIN "${session.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" as "ou"
							ON "uoou"."organizationUnitId" = ou.id
						`;
                        whereQ += `
							AND (ou.id = ANY($${paramsIndex}::UUID[])
								OR string_to_array(ou."ancestorIds", ',')::UUID[] && $${paramsIndex}::UUID[] )
							AND uoou."deletedAt" IS NULL
						`;
                        queryParams.push(params.authorizedOrganizationUnitIds);
                    }
                    await trx.query(insertQ + whereQ, queryParams);
                    break;
                case dal_constants_1.DalConstants.UserSelectionSessionType.PPermission:
                    insertQ = `
					INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userSelectionSessionActions}"
					(id, "sessionId", "userId", "originalMappingId", "action")
					SELECT uuid_generate_v4(), $${paramsIndex++}, upp."userId", upp.id, $${paramsIndex++}
						FROM "${session.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userPPermissions}" as "upp"
					INNER JOIN "${session.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" as "uo"
						ON "upp"."userId" = "uo"."userId" 
							AND uo."isDisabled" IS FALSE 
							AND "uo"."deletedAt" IS NULL
						`;
                    queryParams.push(session.id);
                    queryParams.push(dal_constants_1.DalConstants.UserSelectionSessionAction.Unchanged);
                    whereQ = `
					WHERE "upp"."ppermissionId" = $${paramsIndex++}
					`;
                    queryParams.push(params.relatedItemId);
                    if (!params.isAuthorizedForAllUsers) {
                        insertQ += `
						INNER JOIN "${session.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}" as "uoou"
							ON "uoou"."userOrganizationId" = uo.id
						INNER JOIN "${session.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" as "ou"
							ON "uoou"."organizationUnitId" = ou.id
						`;
                        whereQ += `
							AND (ou.id = ANY($${paramsIndex}::UUID[])
								OR string_to_array(ou."ancestorIds", ',')::UUID[] && $${paramsIndex}::UUID[] )
							AND uoou."deletedAt" IS NULL
						`;
                        queryParams.push(params.authorizedOrganizationUnitIds);
                    }
                    await trx.query(insertQ + whereQ + groupByQ, queryParams);
                    break;
                case dal_constants_1.DalConstants.UserSelectionSessionType.UserWorkPlanAdd:
                    break;
                case dal_constants_1.DalConstants.UserSelectionSessionType.UserWorkPlanRemove:
                    insertQ = `
					INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userSelectionSessionActions}"
					(id, "sessionId", "userId", "originalMappingId", "action")
					SELECT uuid_generate_v4(), $${paramsIndex++}, uwp."userId", uwp."userId", $${paramsIndex++}
						FROM "${session.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userWorkPlans}" as "uwp"
					INNER JOIN "${session.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" as "uo"
						ON "uwp"."userId" = "uo"."userId" 
							AND uo."isDisabled" IS FALSE
							AND "uo"."deletedAt" IS NULL
							AND (uwp."endDateTime" > now() OR uwp."endDateTime" IS NULL)
						`;
                    queryParams.push(session.id);
                    queryParams.push(dal_constants_1.DalConstants.UserSelectionSessionAction.Unchanged);
                    whereQ = `
					WHERE "uwp"."workPlanId" = $${paramsIndex++}
					`;
                    queryParams.push(params.relatedItemId);
                    if (!params.isAuthorizedForAllUsers) {
                        insertQ += `
						INNER JOIN "${session.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}" as "uoou"
							ON "uoou"."userOrganizationId" = uo.id
						INNER JOIN "${session.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" as "ou"
							ON "uoou"."organizationUnitId" = ou.id
						`;
                        whereQ += `
							AND (ou.id = ANY($${paramsIndex}::UUID[])
								OR string_to_array(ou."ancestorIds", ',')::UUID[] && $${paramsIndex}::UUID[] )
							AND uoou."deletedAt" IS NULL
						`;
                        queryParams.push(params.authorizedOrganizationUnitIds);
                    }
                    await trx.query(insertQ + whereQ + ` GROUP BY "uwp"."userId" `, queryParams);
                    break;
                case dal_constants_1.DalConstants.UserSelectionSessionType.AccessControlPointGrant:
                    insertQ = `
					INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userSelectionSessionActions}"
					(id, "sessionId", "userId", "originalMappingId", "action")
					SELECT uuid_generate_v4(), $${paramsIndex++}, uar."userId", uar.id, $${paramsIndex++}
						FROM "${session.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userAccessRights}" as "uar"
					INNER JOIN "${session.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" as "uo"
						ON "uar"."userId" = "uo"."userId" 
							AND uo."isDisabled" IS FALSE
							AND "uo"."deletedAt" IS NULL
						`;
                    queryParams.push(session.id);
                    queryParams.push(dal_constants_1.DalConstants.UserSelectionSessionAction.Unchanged);
                    whereQ = `
					WHERE "uar"."accessControlPointId" = $${paramsIndex++}
						AND "uar"."deletedAt" IS NULL
						AND (uar.grant = true OR uar.read = true OR uar.config = true OR uar."remoteAccess" = true OR uar.access = true OR uar.snapshot = true)
					`;
                    queryParams.push(params.relatedItemId);
                    if (!params.isAuthorizedForAllUsers) {
                        insertQ += `
						INNER JOIN "${session.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}" as "uoou"
							ON "uoou"."userOrganizationId" = uo.id
						INNER JOIN "${session.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" as "ou"
							ON "uoou"."organizationUnitId" = ou.id
						`;
                        whereQ += `
							AND (ou.id = ANY($${paramsIndex}::UUID[])
								OR string_to_array(ou."ancestorIds", ',')::UUID[] && $${paramsIndex}::UUID[] )
							AND uoou."deletedAt" IS NULL
						`;
                        queryParams.push(params.authorizedOrganizationUnitIds);
                    }
                    await trx.query(insertQ + whereQ, queryParams);
                    break;
                case dal_constants_1.DalConstants.UserSelectionSessionType.AddUsersToOrganizationUnit:
                    await trx.query(`INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userSelectionSessionActions}"
							(id, "sessionId", "userId", "originalMappingId", "action")
						SELECT uuid_generate_v4(), $1, uo."userId", uoou.id, $2 from "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}" uoou
						INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo
							ON uo.id = uoou."userOrganizationId" AND uo."isDisabled" IS FALSE
						WHERE uoou."organizationUnitId" = $3
							AND uoou."deletedAt" IS NULL 
							AND uo."deletedAt" IS NULL
							 `, [session.id, dal_constants_1.DalConstants.UserSelectionSessionAction.Unchanged, params.relatedItemId]);
                    break;
                case dal_constants_1.DalConstants.UserSelectionSessionType.RegionTicketTransaction:
                case dal_constants_1.DalConstants.UserSelectionSessionType.SystemMessage:
                case dal_constants_1.DalConstants.UserSelectionSessionType.WorkPlan:
                    break;
                default:
                    break;
            }
            result.upserted = true;
            result.sessionId = session.id;
        }
        return result;
    }
    async askUsersForSelectionSession(organizationId, requestUserId, sessionId, userIds) {
        let session = await this.validateUserSelectionSession(organizationId, requestUserId, sessionId);
        let result = {
            expirationUtc: null,
            items: [],
            added: 0,
            removed: 0,
        };
        await this.dbClient.transaction(async (trx) => {
            let totalCounts = await this.dbClient.withSchema(organizationId).from("userSelectionSessionActions as ussa").where("sessionId", "=", sessionId).transacting(trx).select("action");
            let filteredUsers = await this.dbClient
                .withSchema(organizationId)
                .from("userSelectionSessionActions as ussa")
                .where("sessionId", "=", sessionId)
                .whereIn("userId", userIds)
                .transacting(trx)
                .select("userId", "action");
            result.items = filteredUsers.map((u) => {
                return {
                    status: u.action,
                    userId: u.userId,
                };
            });
            result.added = totalCounts.filter((tc) => tc.action === dal_constants_1.DalConstants.UserSelectionSessionAction.Added).length;
            result.removed = totalCounts.filter((tc) => tc.action === dal_constants_1.DalConstants.UserSelectionSessionAction.Removed).length;
            result.expirationUtc = await this.snoozeExpirationOfUserSelectionSession(organizationId, sessionId, trx);
        });
        return Promise.resolve(result);
    }
    async cancelUserSelectionSession(organizationId, requestUserId, sessionId) {
        let session = await this.validateUserSelectionSession(organizationId, requestUserId, sessionId);
        await this.dbClient.withSchema(organizationId).table("userSelectionSessions").where("id", "=", sessionId).delete();
        return Promise.resolve();
    }
    async applyUserSelectionSession(params) {
        let session = await this.validateUserSelectionSession(params.organizationId, params.requestUserId, params.sessionId, params.trx);
        let now = new Date();
        let queryParamIndex = 0;
        switch (session.type) {
            case dal_constants_1.DalConstants.UserSelectionSessionType.UserGroup:
                queryParamIndex = 0;
                await params.trx.query(`INSERT INTO "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupUserOrganizations}"
					(id, "createdAt", "updatedAt", "userOrganizationId", "userGroupId")
					SELECT ussa."generatedMappingId", $${++queryParamIndex}, $${++queryParamIndex}, uo."id", $${++queryParamIndex}
					FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userSelectionSessionActions}" AS "ussa"
					INNER JOIN "${params.organizationId}"."userOrganizations" AS "uo" ON "ussa"."userId" = "uo"."userId"
					WHERE "ussa"."sessionId" = $${++queryParamIndex}
						AND "ussa"."action" = $${++queryParamIndex}
					ON CONFLICT ("userOrganizationId", "userGroupId") WHERE "deletedAt" is null
					DO NOTHING;`, [now, now, session.relatedItemId, session.id, dal_constants_1.DalConstants.UserSelectionSessionAction.Added]);
                queryParamIndex = 0;
                await params.trx.query(`UPDATE "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupUserOrganizations}" AS uguo
					SET "deletedAt" = $${++queryParamIndex}
					FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userSelectionSessionActions}" AS "ussa"
					WHERE uguo."id" = ussa."originalMappingId"
						AND uguo."userGroupId" = $${++queryParamIndex}
						AND "ussa"."sessionId" = $${++queryParamIndex}
						AND "ussa"."action" = $${++queryParamIndex}`, [now, session.relatedItemId, params.sessionId, dal_constants_1.DalConstants.UserSelectionSessionAction.Removed]);
                const hook = business_hooks_1.armonHookManager.getAfterUserGroupSessionApplyHook(params.organizationId);
                if (hook) {
                    const dbResult = (await params.trx.query(`SELECT
								array_agg(ussa."userId") FILTER (WHERE action = $1) as "addedUsers", 
								array_agg(ussa."userId") FILTER (WHERE action = $2) as "removedUsers"
							FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userSelectionSessionActions}" ussa
							WHERE ussa."sessionId" = $3;`, [dal_constants_1.DalConstants.UserSelectionSessionAction.Added, dal_constants_1.DalConstants.UserSelectionSessionAction.Removed, params.sessionId])).rows[0];
                    hook(session.relatedItemId, dbResult.addedUsers, dbResult.removedUsers);
                }
                break;
            case dal_constants_1.DalConstants.UserSelectionSessionType.PPermission:
                queryParamIndex = 0;
                await params.trx.query(`
                            INSERT INTO "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userPPermissions}"
                            (id, "userId", "ppermissionId")
                                SELECT ussa."generatedMappingId", ussa."userId", $${++queryParamIndex} FROM "${params.organizationId}"."userSelectionSessionActions" AS "ussa"
                                WHERE "ussa"."sessionId" = $${++queryParamIndex}
                                    AND "ussa"."action" = $${++queryParamIndex}
							ON CONFLICT ("userId", "ppermissionId")
							DO NOTHING;
                            `, [session.relatedItemId, session.id, dal_constants_1.DalConstants.UserSelectionSessionAction.Added]);
                queryParamIndex = 0;
                await params.trx.query(`
                            DELETE FROM "${params.organizationId}"."userPPermissions" AS upp
                            WHERE upp."ppermissionId" = $${++queryParamIndex} AND upp."userId" IN (
                                SELECT "userId" FROM
                                "${params.organizationId}"."userSelectionSessionActions" AS ussa
                                WHERE "ussa"."sessionId" = $${++queryParamIndex}
                                    AND "ussa"."action" = $${++queryParamIndex})
                            `, [session.relatedItemId, params.sessionId, dal_constants_1.DalConstants.UserSelectionSessionAction.Removed]);
                break;
            case dal_constants_1.DalConstants.UserSelectionSessionType.AccessControlPointGrant:
                break;
            case dal_constants_1.DalConstants.UserSelectionSessionType.WorkPlan:
                break;
            case dal_constants_1.DalConstants.UserSelectionSessionType.AddUsersToOrganizationUnit:
                break;
            case dal_constants_1.DalConstants.UserSelectionSessionType.UserWorkPlanAdd:
            case dal_constants_1.DalConstants.UserSelectionSessionType.UserWorkPlanRemove:
                break;
            case dal_constants_1.DalConstants.UserSelectionSessionType.RegionTicketTransaction:
                break;
            default:
                break;
        }
        await params.trx.query(`
                DELETE FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userSelectionSessions}"
                    WHERE "id" = $1
            `, [params.sessionId]);
    }
    async listUsersOfUserSelectionSession(organizationId, requestUserId, sessionId, take, skip, filter) {
        let session = await this.validateUserSelectionSession(organizationId, requestUserId, sessionId);
        let result = {
            total: 0,
            take: take,
            skip: skip,
            expirationUtc: null,
            items: [],
        };
        await this.dbClient.transaction(async (trx) => {
            let qbBase = this.dbClient
                .withSchema(organizationId)
                .from("userSelectionSessionActions as ussa")
                .where("sessionId", "=", sessionId)
                .innerJoin("userOrganizationProfiles as uop", (join) => {
                join.andOn("ussa.userId", "uop.userId");
            })
                .innerJoin("userOrganizations as uo", (join) => {
                join.andOn("ussa.userId", "uo.userId").andOn("uo.id", "uop.userOrganizationId");
            })
                .whereNull("uo.deletedAt")
                .whereNull("uop.deletedAt")
                .where("uo.organizationId", organizationId);
            switch (filter) {
                case dal_constants_1.DalConstants.UserSelectionSessionFilterType.JustAdded:
                    qbBase = qbBase.where("ussa.action", "=", dal_constants_1.DalConstants.UserSelectionSessionAction.Added);
                    break;
                case dal_constants_1.DalConstants.UserSelectionSessionFilterType.JustRemoved:
                    qbBase = qbBase.where("ussa.action", "=", dal_constants_1.DalConstants.UserSelectionSessionAction.Removed);
                    break;
                case dal_constants_1.DalConstants.UserSelectionSessionFilterType.JustUnchanged:
                    qbBase = qbBase.where("ussa.action", "=", dal_constants_1.DalConstants.UserSelectionSessionAction.Unchanged);
                    break;
                case dal_constants_1.DalConstants.UserSelectionSessionFilterType.JustChanged:
                    qbBase = qbBase.whereIn("ussa.action", [dal_constants_1.DalConstants.UserSelectionSessionAction.Added, dal_constants_1.DalConstants.UserSelectionSessionAction.Removed]);
                    break;
                case dal_constants_1.DalConstants.UserSelectionSessionFilterType.All:
                default:
                    break;
            }
            let qbTotalResult = await qbBase.clone().transacting(trx).count().first();
            result.total = qbTotalResult ? parseInt(qbTotalResult.count) : 0;
            if (result.total > 0) {
                let items = await qbBase
                    .orderBy("uop.name")
                    .orderBy("uop.surname")
                    .offset(skip)
                    .limit(take)
                    .transacting(trx)
                    .select("ussa.userId as userId", "ussa.action as status", "uop.name as name", "uop.surname as surname", "uop.uniqueId as uniqueId");
                result.items = await Promise.all(items.map(async (u) => {
                    const badge = await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId, userId: u.userId });
                    return {
                        userId: u.userId,
                        userCaptions: badge?.caption ?? [],
                        status: u.status,
                    };
                }));
            }
            result.expirationUtc = await this.snoozeExpirationOfUserSelectionSession(organizationId, sessionId, trx);
        });
        return Promise.resolve(result);
    }
    async addUserToUserSelectionSession(organizationId, requestUserId, sessionId, userId) {
        let session = await this.validateUserSelectionSession(organizationId, requestUserId, sessionId);
        let result = null;
        await this.dbClient.transaction(async (trx) => {
            let existingItem = await this.dbClient
                .withSchema(organizationId)
                .from("userSelectionSessionActions")
                .where("sessionId", "=", session.id)
                .where("userId", "=", userId)
                .transacting(trx)
                .first("id", "originalMappingId", "generatedMappingId", "action");
            if (existingItem && existingItem.action === dal_constants_1.DalConstants.UserSelectionSessionAction.Removed) {
                let action = existingItem.originalMappingId ? dal_constants_1.DalConstants.UserSelectionSessionAction.Unchanged : dal_constants_1.DalConstants.UserSelectionSessionAction.Added;
                await this.dbClient.withSchema(organizationId).table("userSelectionSessionActions").update("action", action).transacting(trx).where("id", "=", existingItem.id);
            }
            else if (!existingItem) {
                await this.dbClient.withSchema(organizationId).table("userSelectionSessionActions").transacting(trx).insert({
                    id: uuid_1.default.v4(),
                    sessionId: session.id,
                    userId: userId,
                    generatedMappingId: uuid_1.default.v4(),
                    action: dal_constants_1.DalConstants.UserSelectionSessionAction.Added,
                });
            }
            result = await this.snoozeExpirationAndGetStatusOfUserSelectionSession(organizationId, sessionId, trx);
        });
        return Promise.resolve(result);
    }
    async removeUserFromUserSelectionSession(organizationId, requestUserId, sessionId, userId) {
        let session = await this.validateUserSelectionSession(organizationId, requestUserId, sessionId);
        let result = null;
        await this.dbClient.transaction(async (trx) => {
            let existingItem = await this.dbClient
                .withSchema(organizationId)
                .from("userSelectionSessionActions")
                .where("sessionId", "=", session.id)
                .where("userId", "=", userId)
                .transacting(trx)
                .first("id", "originalMappingId", "generatedMappingId", "action");
            if (existingItem) {
                if (existingItem.action === dal_constants_1.DalConstants.UserSelectionSessionAction.Added) {
                    await this.dbClient.withSchema(organizationId).table("userSelectionSessionActions").delete().transacting(trx).where("id", "=", existingItem.id);
                }
                else if (existingItem.action === dal_constants_1.DalConstants.UserSelectionSessionAction.Unchanged) {
                    await this.dbClient
                        .withSchema(organizationId)
                        .table("userSelectionSessionActions")
                        .transacting(trx)
                        .update("action", dal_constants_1.DalConstants.UserSelectionSessionAction.Removed)
                        .where("id", "=", existingItem.id);
                }
            }
            result = await this.snoozeExpirationAndGetStatusOfUserSelectionSession(organizationId, sessionId, trx);
        });
        return Promise.resolve(result);
    }
    async removeAllUsersFromUserSelectionSession(organizationId, requestUserId, sessionId) {
        let session = await this.validateUserSelectionSession(organizationId, requestUserId, sessionId);
        let result = null;
        await this.dbClient.transaction(async (trx) => {
            await this.dbClient.withSchema(organizationId).table("userSelectionSessionActions").where("sessionId", "=", session.id).whereNull("originalMappingId").transacting(trx).delete();
            await this.dbClient
                .withSchema(organizationId)
                .table("userSelectionSessionActions")
                .where("sessionId", "=", session.id)
                .whereNotNull("originalMappingId")
                .transacting(trx)
                .update("action", dal_constants_1.DalConstants.UserSelectionSessionAction.Removed);
            result = await this.snoozeExpirationAndGetStatusOfUserSelectionSession(organizationId, sessionId, trx);
        });
        return Promise.resolve(result);
    }
    async addRemoveUsersToSelectionSessionBySearchResult(requestUserId, organizationId, sessionId, action, filter) {
        let session = await this.validateUserSelectionSession(organizationId, requestUserId, sessionId);
        let result = null;
        let qb = dal_manager_1.dbManager.accessIdentity.prepareCountableSearchIdentityRequest(filter);
        await this.dbClient.transaction(async (trx) => {
            if (action === dal_constants_1.DalConstants.AddRemoveAction.Remove) {
                let updateClone = qb.clone().select(this.dbClient.raw(`distinct uo."userId"`));
                let delQb = await trx.raw(`DELETE FROM "${organizationId}"."userSelectionSessionActions"
                     WHERE "originalMappingId" IS NULL AND "sessionId" = ? and "userId" IN ?`, [sessionId, updateClone]);
                let setDelQb = await trx.raw(`UPDATE "${organizationId}"."userSelectionSessionActions"
                    SET "action" = ? WHERE "originalMappingId" IS NOT NULL AND "sessionId" = ? and "userId" IN ?`, [dal_constants_1.DalConstants.UserSelectionSessionAction.Removed, sessionId, updateClone]);
            }
            else {
                let addQb = await trx.raw(`INSERT INTO "${organizationId}"."userSelectionSessionActions"
                    ("userId", "id", "sessionId", "generatedMappingId", "action")
                    ? ON CONFLICT DO NOTHING`, [
                    qb.clone().select(this.dbClient.raw(`distinct uo."userId",
                                    uuid_generate_v4(), ?::uuid,
                                uuid_generate_v4(), ?::integer
                    `, [sessionId, dal_constants_1.DalConstants.UserSelectionSessionAction.Added])),
                ]);
                let updateClone = qb.clone().select(this.dbClient.raw(`distinct uo."userId"`));
                let setAddQb = await trx.raw(`UPDATE "${organizationId}"."userSelectionSessionActions"
                    SET "action" = ? WHERE "originalMappingId" IS NULL AND "sessionId" = ? and "userId" IN ?`, [dal_constants_1.DalConstants.UserSelectionSessionAction.Added, sessionId, updateClone]);
                let setUnchangedQb = await trx.raw(`UPDATE "${organizationId}"."userSelectionSessionActions"
                    SET "action" = ? WHERE "originalMappingId" IS NOT NULL AND "sessionId" = ? and "userId" IN ?`, [dal_constants_1.DalConstants.UserSelectionSessionAction.Unchanged, sessionId, updateClone]);
            }
            result = await this.snoozeExpirationAndGetStatusOfUserSelectionSession(organizationId, sessionId, trx);
        });
        return Promise.resolve(result);
    }
    async snoozeExpirationOfUserSelectionSession(organizationId, id, trx) {
        let expirationUtc = (0, moment_1.default)().add(20, "minutes").toDate();
        let query = this.dbClient
            .withSchema(organizationId)
            .table("userSelectionSessions")
            .update("expirationUtc", expirationUtc)
            .where("id", "=", id);
        if (trx) {
            query = query.transacting(trx);
        }
        return Promise.resolve(expirationUtc);
    }
    async getCountsOfUserSelectionSession(organizationId, id, trx) {
        let countsQuery = this.dbClient
            .withSchema(organizationId)
            .from("userSelectionSessionActions")
            .select(this.dbClient.raw("count(*) total"), this.dbClient.raw('sum(case when "action" = ? then 1 else 0 end) added', dal_constants_1.DalConstants.UserSelectionSessionAction.Added), this.dbClient.raw('sum(case when "action" = ? then 1 else 0 end) removed', dal_constants_1.DalConstants.UserSelectionSessionAction.Removed))
            .where("sessionId", "=", id)
            .first();
        if (trx) {
            countsQuery = countsQuery.transacting(trx);
        }
        let counts = await countsQuery;
        let result = {
            total: parseInt(counts.total),
            added: parseInt(counts.added),
            removed: parseInt(counts.removed),
        };
        return Promise.resolve(result);
    }
    async snoozeExpirationAndGetStatusOfUserSelectionSession(organizationId, id, trx) {
        let expirationUtc = await this.snoozeExpirationOfUserSelectionSession(organizationId, id, trx);
        let counts = await this.getCountsOfUserSelectionSession(organizationId, id, trx);
        let result = {
            total: counts.total,
            added: counts.added,
            removed: counts.removed,
            expirationUtc: expirationUtc,
        };
        return Promise.resolve(result);
    }
    async validateUserSelectionSession(organizationId, requestUserId, sessionId, trx) {
        const trxx = trx ?? this._pgPool;
        const { rows, rowCount } = await trxx.query(`
            SELECT id, "userId", "expirationUtc", "organizationId", "relatedItemId", type
            FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userSelectionSessions}"
            WHERE id = $1`, [sessionId]);
        if (rowCount < 1) {
            (0, dal_access_error_1.throwDbAccessNotFoundError)("Specified session is not found, another user might closed the session please try again!");
        }
        if (rows[0].userId !== requestUserId) {
            (0, dal_access_error_1.throwDbAccessConflictError)("This session is owned by another user");
        }
        if (rows[0].expirationUtc < new Date()) {
            (0, dal_access_error_1.throwDbResourceExpiredError)("This session is expired");
        }
        return rows[0];
    }
    async deleteExpiredSessions(params) {
        const { rowCount } = await params.trx.query(`
			DELETE FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userSelectionSessions}"
			WHERE "expirationUtc" > $1
		`, [params.controlDateUTC]);
        return rowCount;
    }
    async getUserOrganizationProfilePersonnelNumber(organizationId, userId) {
        let result = "";
        await this.dbClient
            .withSchema(organizationId)
            .from("userOrganizationProfiles as uop")
            .innerJoin("userOrganizations as uo", "uop.userOrganizationId", "uo.id")
            .whereNull("uop.deletedAt")
            .where("uo.organizationId", organizationId)
            .where("uo.userId", userId)
            .first(this.dbClient.raw(`uop."extensionFields"->>'personnelNumber' as "personnelNumber"`))
            .then((row) => (result = row.personnelNumber));
        return Promise.resolve(result);
    }
    async getUserIdFromExtensionFieldInfo(organizationId, fieldName, fieldValue) {
        let result = "";
        let qb = this.dbClient
            .withSchema(organizationId)
            .from("userOrganizationProfiles as uop")
            .innerJoin("userOrganizations as uo", "uop.userOrganizationId", "uo.id")
            .whereNull("uop.deletedAt")
            .where("uo.organizationId", organizationId)
            .whereRaw('uop."extensionFields" ->> ? = ?', [fieldName, fieldValue]);
        await qb
            .select("uo.userId")
            .first()
            .then((row) => (result = row.userId));
        return Promise.resolve(result);
    }
    async listAllUsersWithExtensionFields(organizationId) {
        let result = [];
        let qb = this.dbClient
            .withSchema(organizationId)
            .from("userOrganizationProfiles as uop")
            .innerJoin("userOrganizations as uo", "uo.userId", "uop.userId")
            .where("uo.organizationId", organizationId)
            .whereNull("uop.deletedAt")
            .whereNull("uo.deletedAt");
        result = await qb.select(["uo.userId", "uop.name", "uop.surname", "uop.uniqueId", "uop.extensionFields"]);
        return Promise.resolve(result);
    }
    async getUserFullNameFromExtensionFieldInfo(organizationId, fieldName, fieldValue) {
        let result = "";
        let qb = this.dbClient
            .withSchema(organizationId)
            .from("userOrganizationProfiles as uop")
            .innerJoin("userOrganizations as uo", "uop.userOrganizationId", "uo.id")
            .whereNull("uop.deletedAt")
            .where("uo.organizationId", organizationId)
            .whereRaw('uop."extensionFields" ->> ? = ?', [fieldName, fieldValue]);
        await qb
            .select("uop.name", "uop.surname")
            .first()
            .then((row) => (result = (row.name ?? "") + " " + (row.surname ?? "")));
        return Promise.resolve(result);
    }
    async getUserOrganizationId(params) {
        const trxx = params.trx ?? this._pgPool;
        const { rows, rowCount } = await trxx.query(`SELECT id FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}"
            WHERE "userId" = $1
            AND "organizationId" = $2
            AND "deletedAt" IS NULL`, [params.userId, params.organizationId]);
        return rowCount > 0 ? rows[0].id : null;
    }
    async getUserOrganizationIdFromExtensionFieldInfo(organizationId, fieldName, fieldValue) {
        let result = "";
        let qb = this.dbClient
            .withSchema(organizationId)
            .from("userOrganizationProfiles as uop")
            .innerJoin("userOrganizations as uo", "uop.userOrganizationId", "uo.id")
            .whereNull("uop.deletedAt")
            .where("uo.organizationId", organizationId)
            .whereRaw('uop."extensionFields" ->> ? = ?', [fieldName, fieldValue]);
        await qb
            .select("uo.id")
            .first()
            .then((row) => (result = row.userId));
        return Promise.resolve(result);
    }
    async getUserOrganizationProfileExtensionFields(organizationId, userId) {
        let result;
        await this.dbClient
            .withSchema(organizationId)
            .from("userOrganizationProfiles as uop")
            .innerJoin("userOrganizations as uo", "uop.userOrganizationId", "uo.id")
            .whereNull("uop.deletedAt")
            .where("uo.userId", userId)
            .where("uo.organizationId", organizationId)
            .select("uop.extensionFields")
            .first()
            .then((row) => (result = row.extensionFields));
        return Promise.resolve(result);
    }
    async getUsersOrganizationProfileExtensionFields(organizationId, userIds, trx) {
        const query = `
			SELECT "userId", "extensionFields"
			FROM "${organizationId}"."userOrganizationProfiles"
			WHERE "deletedAt" IS NULL AND "userId" = ANY($1::uuid[])
		`;
        try {
            const { rows } = await trx.query(query, [userIds]);
            return rows;
        }
        catch (error) {
            app_logs_1.logger.error("Error fetching extension fields:", error);
        }
    }
    async getUserOrganizationProfileEmploymentInfo(organizationId, userId, trx) {
        let result;
        let qb = this.dbClient
            .withSchema(organizationId)
            .from("userOrganizationProfiles as uop")
            .innerJoin("userOrganizations as uo", "uop.userOrganizationId", "uo.id")
            .whereNull("uop.deletedAt")
            .where("uo.userId", userId)
            .where("uo.organizationId", organizationId);
        if (trx)
            qb.transacting(trx);
        await qb
            .select([
            "uop.previousServiceDuration",
            "uop.employmentStartUtc",
            "uop.employmentEndUtc",
            "uop.birthDateUtc",
            "uop.previousAnnualVacationRight",
            "pacsEnabledRemainedAnnualPPermission",
            "manuallyRemainedAnnualPermission",
        ])
            .first()
            .then((row) => {
            if (row) {
                result = {
                    previousServiceDuration: row.previousServiceDuration,
                    employmentStartUtc: row.employmentStartUtc ? new Date(row.employmentStartUtc) : null,
                    employmentEndUtc: row.employmentEndUtc ? new Date(row.employmentEndUtc) : null,
                    birthDateUtc: row.birthDateUtc ? new Date(row.birthDateUtc) : null,
                    previousAnnualVacationRight: row.previousAnnualVacationRight ? row.previousAnnualVacationRight : 0,
                    pacsEnabledRemainedAnnualPPermission: row.pacsEnabledRemainedAnnualPPermission ? row.pacsEnabledRemainedAnnualPPermission : 0,
                    manuallyRemainedAnnualPermission: row.manuallyRemainedAnnualPermission,
                };
            }
        });
        return Promise.resolve(result);
    }
    async getEmploymentsUserOrganizationProfileInfos(organizationId, userIds, trx) {
        const query = `
			SELECT 
				uo."userId",
				uop."previousServiceDuration",
				uop."employmentStartUtc",
				uop."employmentEndUtc",
				uop."birthDateUtc",
				uop."previousAnnualVacationRight",
				uop."pacsEnabledRemainedAnnualPPermission",
				uop."manuallyRemainedAnnualPermission"
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" as uop
			INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" as uo
				ON uo.id = uop."userOrganizationId"
			WHERE uop."deletedAt" IS NULL AND uo."userId" = ANY($1::uuid[])
		`;
        const rows = trx
            ? (await trx.query(query, [userIds])).rows
            : await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                return (await trx.query(query, [userIds])).rows;
            });
        return rows.map((r) => ({
            userId: r.userId,
            employeeProfileInfo: {
                previousServiceDuration: r.previousServiceDuration,
                employmentStartUtc: r.employmentStartUtc ? new Date(r.employmentStartUtc) : null,
                employmentEndUtc: r.employmentEndUtc ? new Date(r.employmentEndUtc) : null,
                birthDateUtc: r.birthDateUtc ? new Date(r.birthDateUtc) : null,
                previousAnnualVacationRight: r.previousAnnualVacationRight ?? 0,
                pacsEnabledRemainedAnnualPPermission: r.pacsEnabledRemainedAnnualPPermission ?? 0,
                manuallyRemainedAnnualPermission: r.manuallyRemainedAnnualPermission,
            },
        }));
    }
    async getUserIdFromUniqueId(organizationId, uniqueId) {
        let result = "";
        await this.dbClient
            .withSchema(organizationId)
            .from("userOrganizationProfiles as uop")
            .innerJoin("userOrganizations as uo", "uop.userOrganizationId", "uo.id")
            .whereNull("uop.deletedAt")
            .where("uo.organizationId", organizationId)
            .where("uop.uniqueId", uniqueId)
            .first("uo.userId")
            .then((row) => (result = row.userId));
        return Promise.resolve(result);
    }
    async getUniqueIdFromUserId(organizationId, userId, trx) {
        let result = "";
        const trxx = trx ?? this._pgPool;
        const { rows } = await trxx.query(`
			SELECT uop."uniqueId"
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" AS uop
			INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" AS uo ON uop."userOrganizationId" = uo.id
			WHERE uop."deletedAt" IS NULL
			AND uo."organizationId" = $1
			AND uop."userId" = $2
			LIMIT 1
			`, [organizationId, userId]);
        result = rows.length > 0 ? rows[0].uniqueId : null;
        return Promise.resolve(result);
    }
    async listAddedUsersOfSelectionSession(organizationId, userSelectionSessionId) {
        let addedUserIds = [];
        await this.dbClient
            .withSchema(organizationId)
            .table("userSelectionSessionActions")
            .where("sessionId", userSelectionSessionId)
            .where("action", dal_constants_1.DalConstants.UserSelectionSessionAction.Added)
            .select("userId")
            .then((rows) => (addedUserIds = rows.map((u) => u.userId)));
        return addedUserIds;
    }
    async listUsersWithRoleId(organizationId, roleTypeId) {
        let users = [];
        await this.dbClient
            .withSchema(organizationId)
            .from("userOrganizations as uo")
            .innerJoin("roles as r", "r.id", "uo.roleId")
            .whereNull("uo.deletedAt")
            .whereNull("r.deletedAt")
            .where("uo.organizationId", organizationId)
            .where("r.typeId", roleTypeId)
            .distinct("uo.userId")
            .select()
            .then((rows) => {
            if (rows) {
                users = rows.map((a) => a.userId);
            }
        });
        return Promise.resolve(users);
    }
    async listUsersWithPermissionPg(organizationId, permission) {
        let users = [];
        let queryParams = [organizationId, "%" + permission + "%"];
        let queryParamIndex = 1;
        let users1 = (await this._pgPool.query(`SELECT DISTINCT "userId" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo
                                INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}" uoou
                                ON uoou."userOrganizationId" = uo.id
                                INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.roles}" r 
                                ON r.id = uoou."roleId"
                                WHERE uo."deletedAt" IS NULL
                                AND uo."organizationId" = $${queryParamIndex++}
                                AND r.permissions ilike $${queryParamIndex++}
                                AND uo."isDisabled" IS False
                                `, queryParams)).rows;
        queryParamIndex = 1;
        let users2 = (await this._pgPool.query(`SELECT DISTINCT "userId" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo
                                INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.roles}" r 
                                ON r.id = uo."roleId"
                                WHERE uo."deletedAt" IS NULL
                                AND uo."organizationId" = $${queryParamIndex++}
                                AND r.permissions ilike $${queryParamIndex++}
                                AND uo."isDisabled" IS False
                                 `, queryParams)).rows;
        if (users1) {
            users.push(...users1.map((r) => r.userId));
        }
        if (users2) {
            users.push(...users2.map((r) => r.userId));
        }
        return Array.from(new Set(users));
    }
    async listUsersWithPermission(organizationId, permission) {
        let users = [];
        users = (await this._pgPool.query(`SELECT DISTINCT uo."userId" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo
        INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}" uoou 
        ON uo.id = uoou."userOrganizationId"
        INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.roles}" r
        ON r.id = uoou."roleId"
        WHERE uo."organizationId" = $1 
        AND r.permissions like $2
        AND uo."deletedAt" IS NULL
        AND NOT uo."isDisabled"`, [organizationId, "%" + permission + "%"])).rows.map((r) => r.userId);
        const { rows } = await this._pgPool.query(`SELECT DISTINCT uo."userId" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo
        INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.roles}" r
        ON r.id = uo."roleId"
        WHERE uo."organizationId" = $1 
        AND r.permissions like $2
        AND uo."deletedAt" IS NULL
        AND NOT uo."isDisabled"`, [organizationId, "%" + permission + "%"]);
        for (const row of rows) {
            if (!~users.indexOf(row.userId)) {
                users.push(row.userId);
            }
        }
        return users;
    }
    async updateRole(organizationId, roleId, name, permissions, trx) {
        const roleCheckDbResult = await trx.query(`
			SELECT id, "typeId", "permissions", "isOrganizationWide"
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.roles}"
			WHERE "deletedAt" IS NULL
				AND id = $1
			`, [roleId]);
        if (!roleCheckDbResult.rowCount) {
            (0, dal_access_error_1.throwDbAccessNotFoundError)("The role is not found!");
        }
        if (roleCheckDbResult.rows[0].typeId) {
            (0, dal_access_error_1.throwDbAccessBadRequestError)("You can not update the name of a typed role");
        }
        await trx.query(`
			UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.roles}"
			SET "name" = $2, "permissions" = $3
			WHERE id = $1
			`, [roleId, name, JSON.stringify(permissions)]);
        if (roleCheckDbResult.rows[0].permissions !== JSON.stringify(permissions)) {
            await this.revokeChangeRoleUsers(organizationId, roleId, trx);
        }
        return {
            id: roleCheckDbResult.rows[0].id,
            isOrganizationWide: roleCheckDbResult.rows[0].isOrganizationWide,
            name,
            permissions,
            typeId: roleCheckDbResult.rows[0].typeId,
        };
    }
    async revokeChangeRoleUsers(organizationId, roleId, trx) {
        await trx.query(`
			UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.oAuthTokens}"
			SET "deletedAt" = now(), "reason" = 'user role changed'
			WHERE id IN (
				SELECT oat."id"
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.oAuthTokens}" oat
				INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo
					ON uo."userId" = oat."userId" 
						AND uo."deletedAt" IS NULL
						AND oat."deletedAt" IS NULL
				WHERE uo."roleId" = $1

				UNION

				SELECT oat."id"
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.oAuthTokens}" oat
				INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo
					ON uo."userId" = oat."userId" 
						AND uo."deletedAt" IS NULL
						AND oat."deletedAt" IS NULL
				INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}" uoou
					ON uoou."userOrganizationId" = uo.id
						AND uoou."deletedAt" IS NULL
				WHERE uoou."roleId" = $1
			)
			`, [roleId]);
    }
    async revokeOAuthTokensPasswordReset(organizationId, requesterUserId, params) {
        const transactionScope = async (trx) => {
            await trx.query(`UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.oAuthTokens}"
				SET "deletedAt" = now(), reason = 'password reset'
				WHERE "userId" = $1 AND "deletedAt" IS NULL`, [params.userId]);
            await trx.query(`UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}"
				SET "publicKey" = NULL, "notificationToken" = NULL
				WHERE "id" = $1`, [params.userId]);
        };
        if (params.trx) {
            await transactionScope(params.trx);
        }
        else {
            await dal_manager_1.dbManager.organizationTransaction(transactionScope, requesterUserId, organizationId);
        }
        return Promise.resolve();
    }
    async updateTypedRole(organizationId, roleId, permissions, trx) {
        const roleCheckDbResult = await trx.query(`
			SELECT id, name, "typeId", "permissions", "isOrganizationWide", "minPermissions"
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.roles}"
			WHERE "deletedAt" IS NULL
				AND id = $1
			`, [roleId]);
        const role = roleCheckDbResult.rows[0];
        if (!role) {
            (0, dal_access_error_1.throwDbAccessNotFoundError)("The role is not found!");
        }
        if (!role.typeId) {
            (0, dal_access_error_1.throwDbAccessBadRequestError)("The role is not a typed role");
        }
        const minPermissions = role.minPermissions || [];
        for (const permission of minPermissions) {
            if (permissions.indexOf(permission) === -1) {
                permissions.push(permission);
            }
        }
        await trx.query(`
			UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.roles}"
			SET permissions = $2
			WHERE id = $1
				AND "deletedAt" IS NULL
			`, [roleId, JSON.stringify(permissions)]);
        await this.revokeChangeRoleUsers(organizationId, roleId, trx);
        return {
            id: roleId,
            isOrganizationWide: role.isOrganizationWide,
            name: role.name,
            permissions,
            typeId: role.typeId,
            minPermissions: role.minPermissions,
        };
    }
    async removeRole(organizationId, roleId, transferRoleId, trx) {
        const rolesCheckDbResult = await trx.query(`
			SELECT id, "typeId", "isOrganizationWide"
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.roles}"
			WHERE "deletedAt" IS NULL
				AND (id = $1 OR id = $2)
			`, [roleId, transferRoleId]);
        if (!rolesCheckDbResult.rows.find((f) => f.id === roleId)) {
            (0, dal_access_error_1.throwDbAccessNotFoundError)("The role is not found!");
        }
        else if (!rolesCheckDbResult.rows.find((f) => f.id === transferRoleId)) {
            (0, dal_access_error_1.throwDbAccessNotFoundError)("The transfer role is not found!");
        }
        if (rolesCheckDbResult.rows.find((f) => f.id === roleId).isOrganizationWide) {
            await trx.query(`
				UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}"
				SET "roleId" = $2, "updatedAt" = now()
				WHERE "deletedAt" IS NULL
					AND "roleId" = $1
				`, [roleId, transferRoleId]);
        }
        else {
            await trx.query(`
				UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}"
				SET "roleId" = $2, "updatedAt" = now()
				WHERE "deletedAt" IS NULL
					AND "roleId" = $1
				`, [roleId, transferRoleId]);
        }
        await trx.query(`
			UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.roles}"
			SET "deletedAt" = now()
			WHERE id = $1
			`, [roleId]);
        await this.revokeChangeRoleUsers(organizationId, roleId, trx);
    }
    async addRole(organizationId, role) {
        let roleId = uuid_1.default.v4();
        let opTime = new Date();
        await this.dbClient
            .withSchema(organizationId)
            .table("roles")
            .insert({
            id: roleId,
            createdAt: opTime,
            updatedAt: opTime,
            organizationId: organizationId,
            name: role.name,
            permissions: JSON.stringify(role.permissions),
            isOrganizationWide: role.isOrganizationWide,
        });
        return Promise.resolve({
            id: roleId,
            name: role.name,
            permissions: role.permissions,
            isOrganizationWide: role.isOrganizationWide,
        });
    }
    async listRolesOfOrganization(organizationId) {
        return this.dbClient
            .withSchema(organizationId)
            .from("roles")
            .where("organizationId", organizationId)
            .whereNull("deletedAt")
            .select("id", "name", this.dbClient.raw("permissions::json"), "isOrganizationWide", "typeId", "minPermissions")
            .orderBy("name");
    }
    async getTypedRoleIdsOfOrganizationPg(params) {
        const roles = (await params.trx.query(`
			SELECT id, "typeId" FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.roles}"
			WHERE "typeId" IS NOT NULL
		`)).rows;
        return {
            unitAdministratorId: roles.find((r) => r.typeId === predefined_roles_1.PredefinedRoles.UnitAdministrator.id)?.id,
            systemAdministratorId: roles.find((r) => r.typeId === predefined_roles_1.PredefinedRoles.SystemAdministrator.id)?.id,
            standartUserOrganizationId: roles.find((r) => r.typeId === predefined_roles_1.PredefinedRoles.StandartUserOrganization.id)?.id,
            standartUserId: roles.find((r) => r.typeId === predefined_roles_1.PredefinedRoles.StandartUser.id)?.id,
            organizationVisitorId: roles.find((r) => r.typeId === predefined_roles_1.PredefinedRoles.OrganizationVisitor.id)?.id,
            organizationWideHumanResourcesManagerId: roles.find((r) => r.typeId === predefined_roles_1.PredefinedRoles.OrganizationWideHumanResourcesManager.id)?.id,
        };
    }
    async getTypedRoleIdsOfOrganization(organizationId, trx) {
        let rolesQuery = this.dbClient.withSchema(organizationId).from("roles").select("id", "typeId").where("organizationId", organizationId).whereNotNull("typeId");
        if (trx) {
            rolesQuery = rolesQuery.transacting(trx);
        }
        let roles = await rolesQuery;
        return Promise.resolve({
            unitAdministratorId: roles.find((r) => r.typeId === predefined_roles_1.PredefinedRoles.UnitAdministrator.id)?.id,
            systemAdministratorId: roles.find((r) => r.typeId === predefined_roles_1.PredefinedRoles.SystemAdministrator.id)?.id,
            standartUserOrganizationId: roles.find((r) => r.typeId === predefined_roles_1.PredefinedRoles.StandartUserOrganization.id)?.id,
            standartUserId: roles.find((r) => r.typeId === predefined_roles_1.PredefinedRoles.StandartUser.id)?.id,
            organizationVisitorId: roles.find((r) => r.typeId === predefined_roles_1.PredefinedRoles.OrganizationVisitor.id)?.id,
            organizationWideHumanResourcesManagerId: roles.find((r) => r.typeId === predefined_roles_1.PredefinedRoles.OrganizationWideHumanResourcesManager.id)?.id,
        });
    }
    async getOrganizationVisitorTypedRoleIdOfOrganization(organizationId, trx) {
        let resultQuery = this.dbClient.withSchema(organizationId).from("roles").first("id").where("organizationId", organizationId).where("typeId", predefined_roles_1.PredefinedRoles.OrganizationVisitor.id);
        if (trx) {
            resultQuery = resultQuery.transacting(trx);
        }
        let result = await resultQuery;
        return result ? Promise.resolve(result.id) : null;
    }
    async getBasicUserInfoList(organizationId, userIds, trx) {
        let qb = this.dbClient
            .withSchema(organizationId)
            .from("userOrganizations as uo")
            .innerJoin("userOrganizationProfiles as uop", "uo.id", "uop.userOrganizationId")
            .where("uo.organizationId", organizationId)
            .whereIn("uo.userId", userIds)
            .whereNull("uo.deletedAt")
            .whereNull("uop.deletedAt")
            .select("uo.userId as id", "uop.name", "uop.surname", "uop.uniqueId");
        if (trx) {
            qb.transacting(trx);
        }
        let rows = await qb;
        let result = rows.map((t) => {
            let fullname = (t.name ?? "").trim();
            if (t.surname) {
                t.surname = t.surname.trim();
                if (t.surname.length > 0) {
                    fullname += " " + t.surname;
                }
            }
            return {
                captionLines: [fullname, t.uniqueId],
                fullname: fullname,
                id: t.id,
                uniqueId: t.uniqueId,
            };
        });
        return Promise.resolve(result);
    }
    async getReportableUserInfoList(organizationId, userIds, trx) {
        const client = trx || this._pgPool;
        try {
            const reportableFieldsQuery = `
				SELECT COALESCE(
					jsonb_agg(
						jsonb_build_object(
							'name', field->>'name',
							'caption', field->>'caption',
							'type', field->'type',
							'options', CASE
								WHEN field->'options' IS NOT NULL AND jsonb_typeof(field->'options') = 'object'
								THEN field->'options'->'options'
								ELSE NULL
							END
						)
					),
					'[]'::jsonb
				) AS reportablefields
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationForms}",
					jsonb_array_elements(fields::jsonb) AS field
				WHERE field->>'reportable' = 'true';
			`;
            const { rows: reportableFieldsRows } = await client.query(reportableFieldsQuery);
            const reportableFields = reportableFieldsRows[0]?.reportablefields ?? [];
            const query = `
				SELECT 
					uop.*
				FROM 
					"${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" AS uo
				INNER JOIN 
					"${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" AS uop
				ON 
					uo.id = uop."userOrganizationId"
				WHERE 
					uo."organizationId" = $1
					AND uo."userId" = ANY($2)
					AND uo."deletedAt" IS NULL
					AND uop."deletedAt" IS NULL
			`;
            const { rows } = await client.query(query, [organizationId, userIds]);
            if (rows.length === 0) {
                return [
                    {
                        id: uuid_1.default.v4(),
                        uniqueId: "",
                        fullname: "",
                        reportableFields: reportableFields.length
                            ? reportableFields.reduce((acc, field) => {
                                acc[field.name] = {
                                    value: null,
                                    caption: field.caption,
                                };
                                return acc;
                            }, {})
                            : null,
                    },
                ];
            }
            const result = rows.map(({ userId, name, surname, uniqueId, extensionFields = {}, ...otherFields }) => {
                const fullname = [name?.trim(), surname?.trim()].filter(Boolean).join(" ");
                const reportableData = reportableFields.length
                    ? reportableFields.reduce((acc, field) => {
                        const rawValue = extensionFields?.[field.name] ?? otherFields?.[field.name] ?? null;
                        const finalValue = rawValue !== null ? this.getFieldValue(field, rawValue) : null;
                        acc[field.name] = {
                            value: finalValue,
                            caption: field.caption,
                        };
                        return acc;
                    }, {})
                    : null;
                return { id: userId, fullname, uniqueId, reportableFields: reportableData };
            });
            return result;
        }
        catch (error) {
            console.error("Error fetching user details:", error);
            throw error;
        }
    }
    getFieldValue(field, value) {
        if (!value)
            return null;
        const formatDateValue = (value, format) => {
            if (value instanceof Date) {
                return luxon_1.DateTime.fromJSDate(value).toFormat(format);
            }
            return value.includes("T") ? luxon_1.DateTime.fromISO(value).toFormat(format) : luxon_1.DateTime.fromSQL(value).toFormat(format);
        };
        const getSelectOptionCaption = (value) => {
            if (typeof value !== "string")
                return null;
            const matchedOption = field.options?.find((option) => option.value === value);
            return matchedOption?.captionLines?.join(" ") ?? null;
        };
        switch (field.type) {
            case dynamicFormCurrent_1.FormFieldType.Date:
                return formatDateValue(value, "dd-MM-yyyy");
            case dynamicFormCurrent_1.FormFieldType.DateTime:
                return formatDateValue(value, "dd-MM-yyyy HH:mm");
            case dynamicFormCurrent_1.FormFieldType.Select:
                return getSelectOptionCaption(value);
            default:
                return value;
        }
    }
    async getBasicUserInfoWithUserOrganizationId(organizationId, userOrganizationId, trx) {
        let result = {
            id: "",
            fullname: "",
            uniqueId: "",
            captionLines: [],
        };
        let qb = this.dbClient
            .withSchema(organizationId)
            .from("userOrganizationProfiles as uop")
            .innerJoin("userOrganizations as uo", "uo.userId", "uop.userId")
            .where("uop.userOrganizationId", userOrganizationId)
            .whereNull("uop.deletedAt");
        if (trx)
            qb.transacting(trx);
        await qb.first("uo.userId", "uop.name", "uop.surname", "uop.uniqueId").then((row) => {
            if (row) {
                result.id = row.userId;
                result.fullname = (row.name ?? "") + " " + (row.surname ?? "");
                result.uniqueId = row.uniqueId;
                result.captionLines = [];
            }
        });
        return Promise.resolve(result);
    }
    async getBasicUserInfo(organizationId, userId, trx) {
        if (!userId)
            return Promise.resolve(null);
        let result = {
            id: "",
            fullname: "",
            uniqueId: "",
            captionLines: [],
        };
        let qb = this.dbClient
            .withSchema(organizationId)
            .from("userOrganizationProfiles as uop")
            .innerJoin("userOrganizations as uo", "uo.userId", "uop.userId")
            .where("uo.organizationId", organizationId)
            .where("uo.userId", userId)
            .whereNull("uop.deletedAt");
        if (trx)
            qb.transacting(trx);
        await qb
            .select(["uo.userId", "uop.name", "uop.surname", "uop.uniqueId"])
            .first()
            .then((row) => {
            if (row) {
                result.id = row.userId;
                result.fullname = (row.name ?? "") + " " + (row.surname ?? "");
                result.uniqueId = row.uniqueId;
                result.captionLines = [];
            }
        });
        return Promise.resolve(result);
    }
    async getBasicUserInfoPg(params) {
        if (!params.userIds.length)
            return Promise.resolve(null);
        let result = [];
        const dbResult = (await params.trx.query(`
			SELECT "userId", "name", "surname", "uniqueId" 
			FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}"
			WHERE "deletedAt" IS NULL AND
				"userId" =  ANY ($1::UUID[]);

		`, [params.userIds])).rows;
        for (const basicUserInfo of dbResult) {
            result.push({
                id: basicUserInfo.userId,
                fullname: (basicUserInfo.name ?? "") + " " + (basicUserInfo.surname ?? ""),
                uniqueId: basicUserInfo.uniqueId,
                captionLines: (await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId: params.organizationId, userId: basicUserInfo.userId, trx: params.trx })).caption,
            });
        }
        return Promise.resolve(result);
    }
    async getBasicVisitorInfoWithRoleId(organizationId, visitorProfileId, trx) {
        let result = {
            id: "",
            fullname: "",
            uniqueId: "",
            captionLines: [],
            roleId: "",
        };
        const { rows, rowCount } = await (trx ?? this._pgPool).query(` SELECT uo."userId", ovp.name, ovp.surname, ovp."uniqueId", uo."roleId", uo."isDisabled"
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorProfiles}" ovp
			INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo
			ON uo.id = ovp."userOrganizationId"
			WHERE ovp.id = $1
			AND ovp."deletedAt" IS NULL`, [visitorProfileId]);
        if (rowCount > 0) {
            const row = rows[0];
            return {
                id: row.userId,
                fullname: (row.name ?? "") + " " + (row.surname ?? ""),
                isDisabled: row.isDisabled,
                locale: (await (0, dal_memcache_1.getCacheOrganizationSettings)(organizationId, trx)).locale,
                userCaptions: await this.getSingleUserOrganizationCaptionLines(organizationId, row.userId),
                roleId: row.roleId,
                uniqueId: row.uniqueId,
            };
        }
        else {
            return null;
        }
    }
    async getBasicVisitorInfoWithRoleIdKnex(organizationId, visitorProfileId, trx) {
        let qb = this.dbClient
            .withSchema(organizationId)
            .select("uo.userId")
            .select("ovp.name")
            .select("ovp.surname")
            .select("ovp.uniqueId")
            .select("uo.roleId")
            .select("uo.isDisabled")
            .from(`${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorProfiles} as ovp`)
            .innerJoin(`${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations} as uo`, "ovp.userOrganizationId", "uo.id")
            .whereNull("ovp.deletedAt")
            .where("ovp.id", visitorProfileId);
        if (trx)
            qb.transacting(trx);
        const results = await qb;
        if (results.length > 0) {
            return {
                id: results[0].userId,
                fullname: (results[0].name ?? "") + " " + (results[0].surname || ""),
                isDisabled: results[0].isDisabled,
                locale: (await (0, dal_memcache_1.getCacheOrganizationSettings)(organizationId)).locale,
                userCaptions: await this.getSingleUserOrganizationCaptionLines(organizationId, results[0].userId),
                roleId: results[0].roleId,
                uniqueId: results[0].uniqueId,
            };
        }
        else {
            return null;
        }
    }
    async getUserOrganizationRedisPersistentData(params) {
        const trxx = params.trx ?? this._pgPool;
        const { rows, rowCount } = await trxx.query(`SELECT
              u.settings->>'locale' as locale,
              uo.settings->'notification' as "notificationSettings",
              uop.name || ' ' || uop.surname as fullname,
			  uop."uniqueId",
			  uo."roleId",
			  uo."isDisabled",
			  (CASE WHEN uop."email" IS NULL THEN up.email ELSE uop.email END) as email,
			  (CASE WHEN uop."phoneNumber" IS NULL THEN up."phoneNumber" ELSE uop."phoneNumber" END) as "phoneNumber",
			  u."notificationToken",
			  u."notificationTokenType"
              FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}" u
              INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo 
                ON uo."userId" = u."id" 
                AND u."deletedAt" IS NULL 
                AND uo."deletedAt" IS NULL
				AND u."id" = $1
			  LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userProfiles}" up 
                ON uo."userId" = up."userId" 
                AND up."deletedAt" IS NULL 
                AND uo."deletedAt" IS NULL                
              LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" uop 
                ON uop."userOrganizationId" = uo."id" 
                AND uop."deletedAt" IS NULL`, [params.userId]);
        if (rowCount > 0) {
            const row = rows[0];
            return {
                f: row.fullname,
                l: row.locale,
                s: row.notificationSettings,
                u: row.uniqueId,
                d: row.isDisabled,
                r: row.roleId,
                e: row.email,
                m: row.phoneNumber,
                n: row.notificationToken,
                z: row.notificationTokenType,
            };
        }
        else {
            return null;
        }
    }
    async getUsersOrganizationRedisPersistentData(params) {
        if (!params.userIds || params.userIds.length === 0)
            return new Map();
        const { rows } = await params.trx.query(`SELECT
			u.settings->>'locale' as locale,
			uo.settings->'notification' as "notificationSettings",
			uop.name || ' ' || uop.surname as fullname,
			uop."uniqueId",
			uo."roleId",
			uo."isDisabled",
			(CASE WHEN uop."email" IS NULL THEN up.email ELSE uop.email END) as email,
			(CASE WHEN uop."phoneNumber" IS NULL THEN up."phoneNumber" ELSE uop."phoneNumber" END) as "phoneNumber",
			u."notificationToken",
			u."notificationTokenType",
			u.id as "userId"
		FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}" u
		INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo 
			ON uo."userId" = u."id" 
			AND u."deletedAt" IS NULL 
			AND uo."deletedAt" IS NULL
		LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userProfiles}" up 
			ON uo."userId" = up."userId" 
			AND up."deletedAt" IS NULL
		LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" uop 
			ON uop."userOrganizationId" = uo."id" 
			AND uop."deletedAt" IS NULL
		WHERE u.id = ANY($1::uuid[])`, [params.userIds]);
        const resultMap = new Map();
        for (const row of rows) {
            const userId = row.userId;
            const info = {
                f: row.fullname,
                l: row.locale,
                s: row.notificationSettings,
                u: row.uniqueId,
                d: row.isDisabled,
                r: row.roleId,
                e: row.email,
                m: row.phoneNumber,
                n: row.notificationToken,
                z: row.notificationTokenType,
            };
            if (!resultMap.has(userId)) {
                resultMap.set(userId, info);
            }
        }
        return resultMap;
    }
    async removeFingerPrintCredentialFromDevice(organizationId, requesterDeviceId, fingerPrint, trx) {
        let transactionScope = async (trxIn) => {
            let existing = await trxIn
                .withSchema(organizationId)
                .from(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials)
                .first("id")
                .where("userId", fingerPrint.userId)
                .where("type", fingerPrint.type)
                .where("organizationId", organizationId)
                .whereRaw(`"extensionFields"->>'finger' = ?`, fingerPrint.finger);
            if (existing) {
                await trxIn.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials).where("id", existing.id).delete();
            }
            return Promise.resolve(existing ? existing.id : undefined);
        };
        if (!trx) {
            return this.dbClient.transaction(transactionScope);
        }
        else {
            return transactionScope(trx);
        }
    }
    async getExistingDataFromUserOrgCredentials(trxIn, type, credential, organizationId) {
        let result = null;
        switch (type) {
            case app_enums_1.enums.TerminalInsertDataFormat.StandartFingerPrint:
                result = await trxIn
                    .withSchema(organizationId)
                    .from(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials)
                    .first("id")
                    .where("userId", credential.userId)
                    .where("type", credential.type)
                    .where("organizationId", organizationId)
                    .whereRaw(`"extensionFields"->>'finger' = ?`, credential.finger);
                break;
            default:
                break;
        }
        if (result) {
            return result.id;
        }
        else {
            return null;
        }
    }
    async insertNewCredential(organizationId, requesterDeviceId, credential, trx) {
        let transactionScope = async (trxIn) => {
            let existingId;
            switch (credential.type) {
                case app_enums_1.enums.CredentialType.FingerPrintISO19794:
                    existingId = await this.getExistingDataFromUserOrgCredentials(trxIn, app_enums_1.enums.TerminalInsertDataFormat.StandartFingerPrint, credential, organizationId);
                    break;
                default:
                    break;
            }
            if (existingId) {
                await trxIn.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials).where("id", existingId).delete();
            }
            let now = new Date();
            await trxIn
                .withSchema(organizationId)
                .table(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials)
                .insert({
                id: credential.credentialId,
                createdAt: now,
                updatedAt: now,
                organizationId: organizationId,
                userId: credential.userId,
                data: credential.data,
                specialData: credential.specialData,
                specialDataSecondary: credential.specialDataSecondary,
                type: credential.type,
                extensionFields: {
                    finger: credential.finger,
                },
            });
            return Promise.resolve(existingId ? existingId : undefined);
        };
        if (!trx) {
            return this.dbClient.transaction(transactionScope);
        }
        else {
            return transactionScope(trx);
        }
    }
    async upsertFingerPrintCredentialFromDevice(organizationId, requesterDeviceId, fingerPrint, trx) {
        let transactionScope = async (trxIn) => {
            let existing = await trxIn
                .withSchema(organizationId)
                .from(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials)
                .first("id")
                .where("userId", fingerPrint.userId)
                .where("type", fingerPrint.type)
                .where("organizationId", organizationId)
                .whereRaw(`"extensionFields"->>'finger' = ?`, fingerPrint.finger);
            let now = new Date();
            if (existing) {
                await trxIn.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials).where("id", existing.id).delete();
            }
            await trxIn
                .withSchema(organizationId)
                .table(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials)
                .insert({
                id: fingerPrint.credentialId,
                createdAt: now,
                updatedAt: now,
                organizationId: organizationId,
                userId: fingerPrint.userId,
                data: fingerPrint.finger,
                specialData: fingerPrint.specialData,
                specialDataSecondary: fingerPrint.specialDataSecondary,
                type: fingerPrint.type,
                extensionFields: {
                    finger: fingerPrint.finger,
                },
            });
            return Promise.resolve(existing ? existing.id : undefined);
        };
        if (!trx) {
            return this.dbClient.transaction(transactionScope);
        }
        else {
            return transactionScope(trx);
        }
    }
    async removeFingerPrintCredentialsFromDevice(organizationId, requesterDeviceId, fingerPrints) {
        return this.dbClient.transaction(async (trx) => {
            let removedCredentialIds = [];
            for (const fingerPrint of fingerPrints) {
                let resultForFingerprint = await this.removeFingerPrintCredentialFromDevice(organizationId, requesterDeviceId, fingerPrint, trx);
                if (resultForFingerprint) {
                    removedCredentialIds.push(resultForFingerprint);
                }
            }
            return Promise.resolve(removedCredentialIds);
        });
    }
    async upsertFingerPrintCredentialsFromDevice(organizationId, requesterDeviceId, fingerPrints) {
        return this.dbClient.transaction(async (trx) => {
            let removedCredentialIds = [];
            let relatedDeviceIds = [];
            for (const fingerPrint of fingerPrints) {
                let resultForFingerprint = await this.upsertFingerPrintCredentialFromDevice(organizationId, requesterDeviceId, fingerPrint, trx);
                if (resultForFingerprint) {
                    removedCredentialIds.push(resultForFingerprint);
                }
            }
            return Promise.resolve(removedCredentialIds);
        });
    }
    async upsertFaceRecogniseCredentialFromDevice(organizationId, requesterDeviceId, faceRecogniseCredential, trx) {
        let transactionScope = async (trxIn) => {
            let existing = await trxIn
                .withSchema(organizationId)
                .from(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials)
                .first("id")
                .where("userId", faceRecogniseCredential.userId)
                .where("type", faceRecogniseCredential.type)
                .where("organizationId", organizationId);
            let now = new Date();
            if (existing) {
                await trxIn.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials).where("id", existing.id).delete();
            }
            let imageBuffer = Buffer.from(faceRecogniseCredential.specialData, "base64");
            let thumbnail = (await (0, sharp_1.default)(imageBuffer, { failOnError: false }).resize(48, 48).toFormat("jpeg").toBuffer()).toString("base64");
            await trxIn.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials).insert({
                id: faceRecogniseCredential.credentialId,
                createdAt: now,
                updatedAt: now,
                organizationId: organizationId,
                userId: faceRecogniseCredential.userId,
                data: "",
                specialData: faceRecogniseCredential.specialData,
                specialDataSecondary: thumbnail,
                type: faceRecogniseCredential.type,
                extensionFields: null,
            });
            return Promise.resolve(existing ? existing.id : undefined);
        };
        if (!trx) {
            return this.dbClient.transaction(transactionScope);
        }
        else {
            return transactionScope(trx);
        }
    }
    async removeFaceRecogniseCredentialFromDevice(organizationId, requesterDeviceId, faceRecogniseCredential, trx) {
        let transactionScope = async (trxIn) => {
            let existing = await trxIn
                .withSchema(organizationId)
                .from(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials)
                .first("id")
                .where("userId", faceRecogniseCredential.userId)
                .where("type", faceRecogniseCredential.type)
                .where("organizationId", organizationId);
            if (existing) {
                await trxIn.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials).where("id", existing.id).delete();
            }
            return Promise.resolve(existing ? existing.id : undefined);
        };
        if (!trx) {
            return this.dbClient.transaction(transactionScope);
        }
        else {
            return transactionScope(trx);
        }
    }
    async removeFaceRecogniseCredentialsFromDevice(organizationId, requesterDeviceId, faceRecognizeCredentials) {
        return this.dbClient.transaction(async (trx) => {
            let removedCredentialIds = [];
            for (const faceRecognizeCredential of faceRecognizeCredentials) {
                let resultForFaceRecognize = await this.removeFaceRecogniseCredentialFromDevice(organizationId, requesterDeviceId, faceRecognizeCredential, trx);
                if (resultForFaceRecognize) {
                    removedCredentialIds.push(resultForFaceRecognize);
                }
            }
            return Promise.resolve(removedCredentialIds);
        });
    }
    async upsertFaceRecogniseCredentialsFromDevice(organizationId, requesterDeviceId, faceRecognizeCredentials) {
        return this.dbClient.transaction(async (trx) => {
            let removedCredentialIds = [];
            for (const faceRecognizeCredential of faceRecognizeCredentials) {
                let resultForFaceRecognise = await this.upsertFaceRecogniseCredentialFromDevice(organizationId, requesterDeviceId, faceRecognizeCredential, trx);
                if (resultForFaceRecognise) {
                    removedCredentialIds.push(resultForFaceRecognise);
                }
            }
            return Promise.resolve(removedCredentialIds);
        });
    }
    async getBasicUserOrganizationProfileInfo(organizationId, userId, trx) {
        if (!userId)
            return Promise.resolve(null);
        let result = {
            id: "",
            fullname: "",
            uniqueId: "",
            isDisabled: false,
            employmentStartUtc: null,
            employmentEndUtc: null,
            extensionFields: null,
            captionLines: [],
        };
        let qb = this.dbClient
            .withSchema(organizationId)
            .from("userOrganizationProfiles as uop")
            .innerJoin("userOrganizations as uo", "uo.userId", "uop.userId")
            .where("uo.organizationId", organizationId)
            .where("uo.userId", userId)
            .whereNull("uop.deletedAt")
            .whereNull("uo.deletedAt");
        if (trx)
            qb.transacting(trx);
        await qb.first(["uo.userId", "uop.name", "uop.surname", "uop.uniqueId", "uo.isDisabled", "uop.employmentStartUtc", "uop.employmentEndUtc", "uop.extensionFields"]).then((row) => {
            if (row) {
                result.id = row.userId;
                result.fullname = (row.name ?? "") + " " + (row.surname ?? "");
                result.uniqueId = row.uniqueId;
                result.isDisabled = row.isDisabled;
                result.extensionFields = row.extensionFields;
                result.employmentStartUtc = row.employmentStartUtc;
                result.employmentEndUtc = row.employmentEndUtc;
            }
        });
        return Promise.resolve(result);
    }
    async upsertUserProfile(organizationId, userId, userProfile, trx) {
        const profile = (await trx.query(`
				SELECT * FROM "${organizationId}"."userProfiles"
				WHERE "userId" = $1
				`, [userId])).rows[0];
        if (!profile) {
            const user = (await trx.query(`
					SELECT * FROM "${organizationId}".users
					WHERE id = $1
					`, [userId])).rows[0];
            if (!user)
                (0, dal_access_error_1.throwDbAccessNotFoundError)("user not found");
            await trx.query(`
				INSERT INTO "${organizationId}"."userProfiles"(
					id, "createdAt", "updatedAt", "deletedAt", "userId", name, surname, email, "phoneNumber", thumbnail, "mfaSettings")
				VALUES (gen_random_uuid(), $1, $2, $3, $4, $5, $6, $7, $8, $9)
			`, [new Date(), new Date(), null, userId, userProfile.name, userProfile.surname, userProfile.email, userProfile.phoneNumber, null, userProfile.mfaSettings]);
        }
        else {
            await trx.query(`
				UPDATE "${organizationId}"."userProfiles"
				SET name = $1,
					surname = $2,
					"phoneNumber" = $3,
					email = $4,
					"mfaSettings" = $5,
					"updatedAt" = $6
				WHERE "userId" = $7
			`, [userProfile.name, userProfile.surname, userProfile.phoneNumber, userProfile.email, userProfile.mfaSettings, new Date(), userId]);
        }
    }
    async updateUserProfileMfaSettings(organizationId, userId, mfaSettings, trx) {
        let userCustomNotificationMediumSettings = {};
        if (mfaSettings.enabledStatus === true) {
            const organizationSettings = await dal_manager_1.dbManager.accessOrganization.getOrganizationSettings({ organizationId, trx });
            switch (organizationSettings?.multiFactorAuthenticationSettings?.method) {
                case app_enums_1.enums.MultifactorAuthenticationVerificationMethod.EMAIL:
                    userCustomNotificationMediumSettings[app_enums_1.enums.NotificationType.SendingVerificationToken] = {
                        email: true,
                        sms: false,
                        web: false,
                        pushNotification: false,
                    };
                    break;
                case app_enums_1.enums.MultifactorAuthenticationVerificationMethod.SMS:
                    userCustomNotificationMediumSettings[app_enums_1.enums.NotificationType.SendingVerificationToken] = {
                        sms: true,
                        email: false,
                        web: false,
                        pushNotification: false,
                    };
                    break;
            }
            await dal_manager_1.dbManager.accessUser.upsertUserOrganizationMulitFactorAuthNotificationSettings({ organizationId, userId, userCustomNotificationMediumSettings }, trx);
        }
        else {
            await dal_manager_1.dbManager.accessUser.upsertUserOrganizationMulitFactorAuthNotificationSettings({ organizationId, userId }, trx);
        }
        await trx.query(`
			UPDATE "${organizationId}"."userProfiles"
			SET "mfaSettings" = $1,
				"updatedAt" = $2
			WHERE "userId" = $3
			`, [mfaSettings, new Date(), userId]);
    }
    async getUserProfile(organizationId, userId) {
        let dbResult = await this.dbClient
            .withSchema(organizationId)
            .from("userProfiles as up")
            .innerJoin("users as u", "u.id", "up.userId")
            .where("up.userId", userId)
            .whereNull("up.deletedAt")
            .first("up.name", "up.surname", "u.username", "up.phoneNumber", "up.email", "up.mfaSettings");
        return dbResult
            ? Promise.resolve({
                name: dbResult.name,
                surname: dbResult.surname,
                phoneNumber: dbResult.phoneNumber,
                username: dbResult.username,
                email: dbResult.email,
                mfaSettings: dbResult.mfaSettings,
            })
            : null;
    }
    async getUserProfileThumbnail(organizationId, userId) {
        let result = await this.dbClient.withSchema(organizationId).from("userProfiles").where("userId", userId).whereNull("deletedAt").first("thumbnail");
        if (result)
            return Promise.resolve(result.thumbnail);
        return Promise.resolve("");
    }
    async upsertUserProfileThumbnail(organizationId, userId, thumbnail) {
        let affectedRowCount = 0;
        await this.dbClient
            .withSchema(organizationId)
            .from("userProfiles")
            .where("userId", userId)
            .whereNull("deletedAt")
            .update({
            thumbnail: thumbnail,
        })
            .then((result) => (affectedRowCount = result));
        return Promise.resolve(affectedRowCount > 0);
    }
    async revokeUserDeviceToken(organizationId, deviceTokenId, trx) {
        await trx.query(`
			UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.oAuthTokens}"
			SET "deletedAt" = now(), reason = $2
			WHERE id = $1
			`, [deviceTokenId, "user request"]);
        return;
    }
    async logoutUserWithTokenId(organizationId, tokenId, userAgentHeader, trx) {
        const tokenDbResult = await trx.query(`
				SELECT id, "userId"
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.oAuthTokens}"
				WHERE "deletedAt" IS NULL
					AND id = $1
			`, [tokenId]);
        if (!tokenDbResult.rowCount || !tokenDbResult.rows[0].id || !tokenDbResult.rows[0].userId) {
            (0, dal_access_error_1.throwDbAccessNotFoundError)("user token not found");
        }
        await this.revokeUserDeviceToken(organizationId, tokenDbResult.rows[0].id, trx);
        if (userAgentHeader.startsWith("mobilemanager")) {
            await trx.query(`
				UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}"
				SET "updatedAt" = now(), "notificationToken" = null, "notificationTokenType" = null, "publicKey" = null
				WHERE id = $1
				`, [tokenDbResult.rows[0].userId]);
        }
        return;
    }
    async listUserActiveDevices(organizationId, userId) {
        let result = {
            items: [],
        };
        let deviceTokens = await dal_manager_1.dbManager.pgTransactionMainDb(async (trx) => {
            return (await trx.query(`SELECT a.id, a."grantedAt", a."userAgent"
					FROM "${organizationId}"."oAuthTokens" AS a
					WHERE a."userId" = $1 AND
						a."deletedAt" IS NULL AND
						(a."refreshTokenExpiresOn" > now() OR a."accessTokenExpiresOn" > now());`, [userId])).rows;
        });
        for (let token of deviceTokens) {
            let device = deviceParser(token.userAgent, {
                parseUserAgent: true,
            });
            let deviceEnum = dal_constants_1.DalConstants.UserAgentDeviceType.Unknown;
            switch (device.type) {
                case "desktop":
                    deviceEnum = dal_constants_1.DalConstants.UserAgentDeviceType.Desktop;
                    break;
                case "tv":
                    deviceEnum = dal_constants_1.DalConstants.UserAgentDeviceType.TV;
                    break;
                case "phone":
                    deviceEnum = dal_constants_1.DalConstants.UserAgentDeviceType.Phone;
                    break;
                case "tablet":
                    deviceEnum = dal_constants_1.DalConstants.UserAgentDeviceType.Tablet;
                    break;
                case "bot":
                    deviceEnum = dal_constants_1.DalConstants.UserAgentDeviceType.Bot;
                    break;
                case "car":
                    deviceEnum = dal_constants_1.DalConstants.UserAgentDeviceType.Car;
                    break;
                default:
                    deviceEnum = dal_constants_1.DalConstants.UserAgentDeviceType.Unknown;
                    break;
            }
            let userAgent = ua_parser_1.default.parse(token.userAgent);
            let item = {
                id: token.id,
                lastGrantSessionUtc: token.grantedAt,
                os: userAgent.os.family,
                device: deviceEnum,
            };
            result.items.push(item);
        }
        return Promise.resolve(result);
    }
    async listUserUnitsHierarchically(organizationId, userId) {
        return await this.dbClient.transaction(async (trx) => {
            let oUnitIds = [];
            let result = [];
            await trx
                .withSchema(organizationId)
                .table("userOrganizations as uo")
                .innerJoin("userOrganizationOrganizationUnits as uoou", (join) => {
                join.on("uoou.userOrganizationId", "uo.id");
            })
                .innerJoin("organizationUnits as ou", (join) => {
                join.on("ou.id", "uoou.organizationUnitId");
            })
                .where("uo.userId", userId)
                .whereNull("uoou.deletedAt")
                .whereNull("ou.deletedAt")
                .whereNull("uo.deletedAt")
                .select(["ou.id", "ou.ancestorIds"])
                .then((rows) => {
                for (let row of rows) {
                    oUnitIds.push(row.id);
                    if (row.ancestorIds) {
                        oUnitIds = [...oUnitIds, ...row.ancestorIds.split(",")];
                    }
                }
            });
            let distinctUnits = Array.from(new Set(oUnitIds));
            result = await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits).whereNull("deletedAt").whereIn("id", distinctUnits).select("id", "name");
            return Promise.resolve(result);
        });
    }
    async listUserUnitsHierarchicallyOrdered(organizationId, userId, trx) {
        const userUnitIdsQuery = `
			SELECT ou.id, ou."ancestorIds"
			FROM "${organizationId}"."userOrganizations" as uo
			INNER JOIN "${organizationId}"."userOrganizationOrganizationUnits" as uoou
				ON uoou."userOrganizationId" = uo.id
			INNER JOIN "${organizationId}"."organizationUnits" as ou
				ON ou.id = uoou."organizationUnitId"
			WHERE uo."userId" = $1 AND uoou."deletedAt" IS NULL AND ou."deletedAt" IS NULL AND uo."deletedAt" IS NULL
		`;
        const userUnitIdsResult = await trx.query(userUnitIdsQuery, [userId]);
        let oUnitIds = [];
        userUnitIdsResult.rows.forEach((row) => {
            oUnitIds.push(row.id);
            if (row.ancestorIds) {
                oUnitIds = [...oUnitIds, ...row.ancestorIds.split(",")];
            }
        });
        let distinctUnits = Array.from(new Set(oUnitIds));
        if (distinctUnits.length === 0) {
            return [];
        }
        const organizationUnitsQuery = `
			WITH RECURSIVE hierarchicalunits AS (
				SELECT id, 1 as level
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}"
				WHERE "parentId" IS NULL  -- En üst seviyedeki birimler
				UNION ALL
				SELECT child.id, parent.level + 1
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" AS child
				JOIN hierarchicalunits AS parent ON child."parentId" = parent.id
			)
			SELECT ou.id, ou.name
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" ou
			INNER JOIN hierarchicalunits hu on hu.id = ou.id
			WHERE ou."deletedAt" IS NULL
			AND ou.id IN (${"'" + distinctUnits.join("','") + "'"})
			ORDER BY hu.level asc;
		`;
        const result = await trx.query(organizationUnitsQuery);
        return result.rows;
    }
    async listUsersUnitsHierarchicallyOrdered(organizationId, userIds, trx) {
        if (userIds.length === 0)
            return [];
        const userUnitIdsQuery = `
        SELECT uo."userId", ou.id as "unitId", ou."ancestorIds"
        FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" as uo
        INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}" as uoou
            ON uoou."userOrganizationId" = uo.id
        INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" as ou
            ON ou.id = uoou."organizationUnitId"
        WHERE uo."userId" = ANY($1)
        AND uo."deletedAt" IS NULL
        AND uoou."deletedAt" IS NULL
        AND ou."deletedAt" IS NULL
    `;
        const userUnitIdsResult = await trx.query(userUnitIdsQuery, [userIds]);
        const userUnitMap = new Map();
        for (const row of userUnitIdsResult.rows) {
            const userId = row.userId;
            if (!userUnitMap.has(userId)) {
                userUnitMap.set(userId, new Set());
            }
            const unitSet = userUnitMap.get(userId);
            unitSet.add(row.unitId);
            if (row.ancestorIds) {
                for (const ancestorId of row.ancestorIds.split(",")) {
                    unitSet.add(ancestorId);
                }
            }
        }
        const allUnitIds = Array.from(new Set([...userUnitMap.values()].flatMap((set) => Array.from(set))));
        if (allUnitIds.length === 0) {
            return userIds.map((id) => ({ userId: id, units: [] }));
        }
        const hierarchyQuery = `
        WITH RECURSIVE HierarchicalUnits AS (
            SELECT id, name, 1 as level
            FROM "${organizationId}"."organizationUnits"
            WHERE "parentId" IS NULL AND id = ANY($1)
            UNION ALL
            SELECT child.id, child.name, parent.level + 1
            FROM "${organizationId}"."organizationUnits" AS child
            JOIN HierarchicalUnits AS parent ON child."parentId" = parent.id
            WHERE child.id = ANY($1)
        )
        SELECT id, name, level
        FROM HierarchicalUnits
        ORDER BY level ASC, name ASC
    `;
        const hierarchyResult = await trx.query(hierarchyQuery, [allUnitIds]);
        const orderedUnits = hierarchyResult.rows.map((row) => ({ id: row.id, name: row.name }));
        const result = [];
        for (const userId of userIds) {
            const userUnitIds = userUnitMap.get(userId) || new Set();
            const userUnits = orderedUnits.filter((unit) => userUnitIds.has(unit.id));
            result.push({ userId, units: userUnits });
        }
        return result;
    }
    async getUserMobileNotificationParameters(organizationId, userId, client) {
        return (await client.query(`
            SELECT "publicKey", "notificationToken", "notificationTokenType" FROM "${organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.users}
            WHERE id = $1
            AND "deletedAt" IS NULL
            `, [userId])).rows[0];
    }
    async listUserUniqueIds(organizationId, userIds) {
        return await this.dbClient
            .withSchema(organizationId)
            .from("userOrganizationProfiles as uop")
            .innerJoin("userOrganizations as uo", "uop.userOrganizationId", "uo.id")
            .whereNull("uop.deletedAt")
            .whereNull("uo.deletedAt")
            .where("uo.organizationId", organizationId)
            .whereIn("uo.userId", userIds)
            .select("uo.userId", "uop.uniqueId");
    }
    async listUserIdsFromUniqueIds(organizationId, uniqueIds) {
        return await this.dbClient
            .withSchema(organizationId)
            .from("userOrganizationProfiles as uop")
            .whereNull("uop.deletedAt")
            .whereIn("uop.uniqueId", uniqueIds)
            .select("uop.userId", "uop.uniqueId", "uop.name", "uop.surname");
    }
    async getUserOrganizationProfileFullNames(organizationId, userIds, deletedAlso) {
        let qb = this.dbClient
            .withSchema(organizationId)
            .from("userOrganizationProfiles as uop")
            .innerJoin("userOrganizations as uo", "uop.userOrganizationId", "uo.id")
            .where("uo.organizationId", organizationId)
            .whereIn("uop.userId", userIds);
        if (deletedAlso !== true) {
            qb.whereNull("uop.deletedAt");
        }
        let dbResult = await qb.select("uop.userId as id", "uop.name", "uop.surname");
        return Promise.resolve(dbResult.map((m) => {
            return {
                id: m.id,
                name: [m.name, m.surname || ""].join(" ").trim(),
            };
        }));
    }
    async listLDAPNotUpdateUsers(organizationId, date) {
        return await this.dbClient
            .withSchema(organizationId)
            .table("users as u")
            .innerJoin("userOrganizationProfiles as uo", "uo.userId", "u.id")
            .whereNull("uo.deletedAt")
            .where("u.updatedAt", "<", date)
            .where("u.fromLDAP", true)
            .select("uo.userId", "uo.userOrganizationId", "uo.uniqueId", "uo.extensionFields");
    }
    async makeLDAPUsersPassive(organizationId, date) {
        let totalResult = 0;
        await this.dbClient.transaction(async (trx) => {
            let qb = trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.users).where("updatedAt", "<", date).where("fromLDAP", true);
            let now = new Date();
            totalResult = (await qb.clone().count().first()).count;
            await qb.clone().update({
                accountEnabled: false,
                updatedAt: now,
            });
            let userIds = (await qb.select("id")).map((r) => r.id);
            await trx.withSchema(organizationId).table("userOrganizations").where("organizationId", organizationId).whereIn("userId", userIds).update({
                isDisabled: true,
                updatedAt: now,
            });
            await trx
                .withSchema(organizationId)
                .table("userOrganizations")
                .where("organizationId", organizationId)
                .whereIn("userId", userIds)
                .where("organizationId", organizationId)
                .update({
                deletedAt: now,
            });
        });
        return Promise.resolve(totalResult);
    }
    async upsertUserByLDAPInfo(organizationId, organizationRoleId, username, uniqueId, userProfileInfo, userOrganizationProfileInfo, enableAccount, targetUserGroupId) {
        await this.dbClient.transaction(async (trx) => {
            let now = new Date();
            let user = await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.users).whereRaw("lower(username) = lower(?)", username).first();
            if (!user) {
                user = {
                    id: uuid_1.default.v4(),
                    createdAt: now,
                    updatedAt: now,
                    deletedAt: null,
                    accountEnabled: enableAccount,
                    fromLDAP: true,
                    salt: "37ffcc5c9ef2c4a3240673adb7831894",
                    hashedPassword: "a79aa523a80dea647bf6874b3a4516aade888ed96a264ea459d10ddf559336b8c7c94e0cd5e7de117c54ff375294e6c53817ad834b1be8224738dc2a7565aca3",
                    publicKey: "",
                    username: username,
                };
                await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.users).insert(user);
            }
            else {
                user.updatedAt = now;
                await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.users).where("id", user.id).update({
                    updatedAt: now,
                    accountEnabled: enableAccount,
                });
            }
            let userProfile = await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.userProfiles).where("userId", user.id).first();
            if (!userProfile) {
                userProfile = {
                    id: uuid_1.default.v4(),
                    createdAt: now,
                    updatedAt: now,
                    deletedAt: null,
                    userId: user.id,
                    name: userProfileInfo.name,
                    surname: userProfileInfo.surname,
                };
                await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.userProfiles).insert(userProfile);
            }
            else {
                await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.userProfiles).where("userId", user.id).update({
                    updatedAt: now,
                    name: userProfile.name,
                    surname: userProfile.surname,
                });
            }
            let userOrganization = await trx
                .withSchema(organizationId)
                .table(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations)
                .where("userId", user.id)
                .where("organizationId", organizationId)
                .first();
            if (!userOrganization) {
                userOrganization = {
                    id: uuid_1.default.v4(),
                    createdAt: now,
                    updatedAt: now,
                    deletedAt: null,
                    userId: user.id,
                    organizationId: organizationId,
                    roleId: organizationRoleId,
                    isDisabled: false,
                };
                await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations).insert(userOrganization);
            }
            else {
                await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations).where("id", userOrganization.id).update({ updatedAt: now });
            }
            let userOrganizationProfile = await trx.withSchema(organizationId).from(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles).where("userId", user.id).first();
            if (!userOrganizationProfile) {
                userOrganizationProfile = {
                    id: uuid_1.default.v4(),
                    createdAt: now,
                    updatedAt: now,
                    deletedAt: null,
                    userId: user.id,
                    userOrganizationId: userOrganization.id,
                    name: userOrganizationProfileInfo.name,
                    surname: userOrganizationProfileInfo.surname,
                    email: userOrganizationProfileInfo.email,
                    uniqueId: uniqueId,
                    extensionFields: userOrganizationProfileInfo.extensionFields,
                };
                await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles).insert(userOrganizationProfile);
            }
            else {
                await trx
                    .withSchema(organizationId)
                    .table(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles)
                    .where("id", userOrganizationProfile.id)
                    .update({
                    updatedAt: now,
                    name: userOrganizationProfileInfo.name,
                    surname: userOrganizationProfileInfo.surname,
                    email: userOrganizationProfileInfo.email ? userOrganizationProfileInfo.email : "",
                    uniqueId: uniqueId,
                    extensionFields: userOrganizationProfileInfo.extensionFields,
                });
            }
            if (targetUserGroupId) {
                let existsInGroup = await trx
                    .withSchema(organizationId)
                    .table(dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupUserOrganizations)
                    .where("userGroupId", targetUserGroupId)
                    .where("userOrganizationId", userOrganization.id)
                    .whereNull("deletedAt")
                    .first();
                if (!existsInGroup) {
                    await trx.raw(`
						INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupUserOrganizations}"
						(id, "createdAt", "updatedAt", "userOrganizationId", "userGroupId")
						VALUES(?, ?, ?, ?, ?, ?)
						ON CONFLICT ("userOrganizationId", "userGroupId") WHERE "deletedAt" IS NULL
						DO NOTHING;
					`, [uuid_1.default.v4(), now, now, userOrganization.id, targetUserGroupId]);
                }
            }
            let existingUserAccessRights = (await trx
                .withSchema(organizationId)
                .table(dal_db_armon_schema_1.ArmonSchema.tableNames.userAccessRights)
                .where("userId", user.id)
                .where("access", true)
                .whereNull("deletedAt")
                .select("accessControlPointId"));
            let acps = (await trx
                .withSchema(organizationId)
                .table(dal_db_armon_schema_1.ArmonSchema.tableNames.accessControlPoints)
                .where("organizationId", organizationId)
                .whereNull("deletedAt")
                .select("id"));
            let newAccessRights = [];
            for (const acp of acps) {
                if (!existingUserAccessRights.find((u) => u.accessControlPointId == acp.id)) {
                    newAccessRights.push({
                        id: uuid_1.default.v4(),
                        createdAt: now,
                        deletedAt: null,
                        updatedAt: now,
                        userId: user.id,
                        accessControlPointId: acp.id,
                        remoteAccess: false,
                        markedAsFavorite: false,
                        access: true,
                    });
                }
            }
            if (newAccessRights.length > 0) {
                await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.userAccessRights).insert(newAccessRights);
            }
            return Promise.resolve();
        });
        return Promise.resolve();
    }
    async getPredefinedSystemUserInfoForOrganization(organizationId) {
        let systemUser = await dal_manager_1.dbManager.accessPacs.getBasicUserInfo(organizationId, dal_constants_1.DalConstants.SystemUserId);
        if (!systemUser?.id) {
            await this.dbClient.transaction(async (trx) => {
                const orgSettings = await (0, dal_memcache_1.getCacheOrganizationSettings)(organizationId);
                let opTime = new Date();
                let user = await trx.withSchema(organizationId).from(dal_db_armon_schema_1.ArmonSchema.tableNames.users).where("id", dal_constants_1.DalConstants.SystemUserId).first();
                if (!user) {
                    let userListEntry = await trx.withSchema("public").from(dal_db_armon_schema_1.ArmonSchema.tableNames.userList).where("id", dal_constants_1.DalConstants.SystemUserId).first();
                    if (!userListEntry) {
                        await trx.withSchema("public").from(dal_db_armon_schema_1.ArmonSchema.tableNames.userList).insert({
                            id: dal_constants_1.DalConstants.SystemUserId,
                            username: dal_constants_1.DalConstants.SystemUserName,
                        });
                    }
                    await trx
                        .withSchema(organizationId)
                        .from("users")
                        .insert({
                        id: dal_constants_1.DalConstants.SystemUserId,
                        createdAt: opTime,
                        updatedAt: opTime,
                        accountEnabled: false,
                        username: dal_constants_1.DalConstants.SystemUserName,
                        fromLDAP: false,
                        settings: {
                            locale: orgSettings.locale,
                        },
                    });
                }
                let systemUserOrganization = await trx
                    .withSchema(organizationId)
                    .from(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations)
                    .where("userId", dal_constants_1.DalConstants.SystemUserId)
                    .where("organizationId", organizationId)
                    .first();
                if (!systemUserOrganization) {
                    let organizationRoles = await dal_manager_1.dbManager.accessUser.getTypedRoleIdsOfOrganization(organizationId, trx);
                    let userOrganizationId = uuid_1.default.v4();
                    let userOrganizationMappingEntry = await trx
                        .withSchema("public")
                        .from(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationMapping)
                        .where("userId", dal_constants_1.DalConstants.SystemUserId)
                        .where("organizationId", organizationId)
                        .first();
                    if (!userOrganizationMappingEntry) {
                        await trx.withSchema("public").from(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationMapping).insert({
                            id: userOrganizationId,
                            userId: dal_constants_1.DalConstants.SystemUserId,
                            organizationId: organizationId,
                        });
                    }
                    await trx
                        .withSchema(organizationId)
                        .from("userOrganizations")
                        .insert({
                        id: userOrganizationId,
                        createdAt: opTime,
                        updatedAt: opTime,
                        userId: dal_constants_1.DalConstants.SystemUserId,
                        organizationId: organizationId,
                        roleId: organizationRoles.standartUserId,
                        isDisabled: true,
                        extraPermissions: null,
                        settings: {
                            notification: {
                                mediumSettings: orgSettings.notification.mediumSettings,
                            },
                        },
                    });
                    let userOrganizationProfileId = uuid_1.default.v4();
                    await trx.withSchema(organizationId).from("userOrganizationProfiles").insert({
                        id: userOrganizationProfileId,
                        createdAt: opTime,
                        updatedAt: opTime,
                        userId: dal_constants_1.DalConstants.SystemUserId,
                        userOrganizationId: userOrganizationId,
                        name: "Sistem",
                        surname: "",
                        uniqueId: "Armon Sistem Kullanıcısı",
                    });
                }
            });
            systemUser = await dal_manager_1.dbManager.accessPacs.getBasicUserInfo(organizationId, dal_constants_1.DalConstants.SystemUserId);
        }
        return Promise.resolve(systemUser);
    }
    async updateUserOrganizationProfilePacsRemainedAnnualPPermission(organizationId, userId, pacsEnabledRemainedAnnualPPermission) {
        let userOrganizationId = await this.getUserOrganizationId({ organizationId, userId });
        await this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles).whereNull("deletedAt").where("userOrganizationId", userOrganizationId).update({
            pacsEnabledRemainedAnnualPPermission: pacsEnabledRemainedAnnualPPermission,
        });
    }
    async getUserRelatedDeviceIds(params, trx) {
        let qb = this.dbClient
            .withSchema(params.organizationId)
            .from("users as u")
            .innerJoin("userAccessRights as uar", "uar.userId", "u.id")
            .innerJoin("accessControlPoints as acp", "uar.accessControlPointId", "acp.id")
            .whereNull("acp.deletedAt")
            .where("uar.access", true)
            .whereNull("uar.deletedAt")
            .whereNull("u.deletedAt")
            .where("u.id", params.userId)
            .where("acp.organizationId", params.organizationId)
            .whereNotNull("acp.deviceId");
        if (params.authenticationFactor) {
            qb.whereRaw(`"authenticationFactors"::text SIMILAR TO '%("factor":${params.authenticationFactor})%'`);
        }
        if (params.exceptionDeviceIds && params.exceptionDeviceIds.length > 0) {
            qb.whereNotIn("acp.deviceId", params.exceptionDeviceIds);
        }
        if (trx)
            qb.transacting(trx);
        let deviceIds = (await qb.select("acp.deviceId")).map((u) => u.deviceId);
        return Promise.resolve(deviceIds);
    }
    async genericSearchUserFromFullName(organizationId, args) {
        let result = {
            items: [],
            total: 0,
        };
        let qb = this.dbClient
            .withSchema(organizationId)
            .from("users as u")
            .innerJoin("userOrganizationProfiles as uop", "uop.userId", "u.id")
            .innerJoin("userOrganizations as uo", "uop.userOrganizationId", "uo.id")
            .whereNull("uop.deletedAt")
            .whereNull("u.deletedAt")
            .whereNull("uo.deletedAt")
            .where("uo.organizationId", organizationId)
            .whereRaw(` unaccent(uop.name::text || ' ' || uop.surname::text)  ilike unaccent(upper(?))`, "%" + args.filter + "%");
        result.total = parseInt((await qb.clone().count())[0].count);
        if (args.skip)
            qb.offset(args.skip);
        if (args.take)
            qb.limit(args.take);
        let searchResult = await qb.select("u.id", "uop.name").orderBy("uop.name", "asc");
        let captions = await this.getUserOrganizationCaptionLines(organizationId, searchResult.map((s) => s.id));
        for (const row of searchResult) {
            let userCaptions = captions.find((c) => c.id == row.id);
            result.items.push({
                id: row.id,
                userCaptions: userCaptions ? userCaptions.captionLines : [],
            });
        }
        return Promise.resolve(result);
    }
    async genericSearchUserFromCaptionLines(organizationId, params, trx) {
        let result = {
            items: [],
            total: 0,
        };
        let paramIndex = 1;
        const queryParams = [];
        const selectInnerQ = `
		SELECT sq1."userId", string_agg(sq1."value", ' ')  FILTER (WHERE sq2."captionKey" IS NOT NULL) AS "searchText" 
		FROM
			(
				SELECT uop."userId", d.key, trim (BOTH d.value) AS value
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" uop
				LEFT JOIN 
				jsonb_each_text(
					jsonb_build_object (
						'fullname', uop.name || ' ' || uop.surname, 
						'uniqueId', uop."uniqueId",
						'email', uop.email,
						'phoneNumber', uop."phoneNumber"
					) || COALESCE (uop."extensionFields"::jsonb, '{}'::jsonb)
				) d 
					ON TRUE
			) sq1
		LEFT JOIN
			(
				SELECT clf->>'name' as "captionKey" FROM
				"${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizations}",
					json_array_elements ("userOrganizationProfileSettings"->'captionLineFields') AS clf
				WHERE clf->>'type' = '1'
			) sq2	
		ON sq1."key" = sq2."captionKey"
		INNER JOIN "${organizationId}"."userOrganizations" uo 
		ON uo."userId" = sq1."userId"
		WHERE sq1.value IS NOT NULL
			AND length(sq1.value) > 0
			AND uo."isDisabled" = false
		GROUP BY sq1."userId"
		`;
        let fromQuery = `
		FROM (${selectInnerQ}) sqi
		`;
        if (params.filter) {
            fromQuery += `
			WHERE UNACCENT(UPPER(sqi."searchText")) ILIKE UNACCENT(UPPER($${paramIndex++}))`;
            queryParams.push("%" + params.filter?.split(" ").join("%") + "%");
        }
        result.total = (await trx.query(`SELECT COUNT(sqi.*)::INTEGER AS c ` + fromQuery, queryParams)).rows[0].c;
        fromQuery += ` ORDER BY sqi."searchText" ASC `;
        if (result.total !== 0) {
            if (params.skip) {
                fromQuery += ` OFFSET $${paramIndex++}`;
                queryParams.push(params.skip);
            }
            if (params.take) {
                fromQuery += ` LIMIT $${paramIndex++}`;
                queryParams.push(params.take);
            }
            let searchResult = (await trx.query(`SELECT sqi."userId" ` + fromQuery, queryParams)).rows;
            let captions = await this.getUserOrganizationCaptionLines(organizationId, searchResult.map((s) => s.userId));
            for (const sr of searchResult) {
                let userCaptions = captions.find((c) => c.id == sr.userId);
                result.items.push({
                    id: sr.userId,
                    userCaptions: userCaptions ? userCaptions.captionLines : [],
                });
            }
        }
        return Promise.resolve(result);
    }
    async checkUserIsUnderAuthority(organizationId, requesterUserId, userId) {
        return await this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.viewNames.vW_UserHierarchys)
            .where("organizationId", organizationId)
            .where("userId", requesterUserId)
            .where("userIdUnderAuthority", userId)
            .select();
    }
    async checkUserUnderHierarchyOfAnotherUser(organizationId, requesterUserId, targetUserId, trx) {
        const result = [];
        let accessibleOrganizationUnits = (await (trx ?? this._pgPool).query(`
			SELECT cou.id AS "ancestorOrganizationUnitId", ou.id AS "descenderOrganizationUnitId"
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}" uoou
				JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo ON uoou."userOrganizationId" = uo.id AND uoou."deletedAt" IS NULL AND uo."deletedAt" IS NULL
				JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" ou ON ou.id = uoou."organizationUnitId" AND ou."deletedAt" IS NULL AND uoou."deletedAt" IS NULL
				JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" cou ON (ou.id = cou.id OR ou."ancestorIds"::text ~~ (('%'::text || cou.id) || '%'::text)) 
					AND uoou."deletedAt" IS NULL AND ou."deletedAt" IS NULL AND cou."deletedAt" IS NULL
				WHERE uo."userId" = $1;
			`, [targetUserId])).rows;
        const organizationUnitIdsForUser = new Set();
        accessibleOrganizationUnits.forEach((element) => {
            organizationUnitIdsForUser.add(element.ancestorOrganizationUnitId);
            organizationUnitIdsForUser.add(element.descenderOrganizationUnitId);
        });
        const { rows } = await (trx ?? this._pgPool).query(`
			SELECT uoou."organizationUnitId" AS "ancestorOrganizationUnitId"
   			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}" uoou
     			JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo ON uoou."userOrganizationId" = uo.id AND uoou."deletedAt" IS NULL AND uo."deletedAt" IS NULL
			WHERE uo."userId" = $1 AND uoou."organizationUnitId"::text = ANY($2)
			`, [requesterUserId, [...organizationUnitIdsForUser]]);
        if (rows.length) {
            accessibleOrganizationUnits.forEach((element) => {
                result.push({
                    organizationId: organizationId,
                    userId: requesterUserId,
                    userIdUnderAuthority: targetUserId,
                    ancestorOrganizationUnitId: element.ancestorOrganizationUnitId,
                    descenderOrganizationUnitId: element.descenderOrganizationUnitId,
                });
            });
        }
        return result;
    }
    async changePasswordOfIdentity(organizationId, identityChangePassword, hasOrganizationWideUserAccount, jwt) {
        let userOrganization = await this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations)
            .whereNull("deletedAt")
            .where("userId", identityChangePassword.id)
            .where("organizationId", organizationId)
            .first();
        let user = await this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.users)
            .whereNull("deletedAt")
            .where("id", identityChangePassword.id)
            .whereNotNull("username")
            .first("salt", "hashedPassword");
        if (!userOrganization) {
            throw (0, api_error_1.generateNotFoundApiError)({ message: "User is not found" });
        }
        if (!user) {
            throw (0, api_error_1.generateForbiddenError)({ message: "User has not an account!" });
        }
        if (!hasOrganizationWideUserAccount) {
            let userHierarchyItems = await this.checkUserUnderHierarchyOfAnotherUser(organizationId, jwt.userId, identityChangePassword.id);
            let underAuthority = false;
            for (let index = 0; index < userHierarchyItems.length; index++) {
                let userHierarchyItem = userHierarchyItems[index];
                if (jwt.isPermittedForUnit(organizationId, predefined_permissions_1.Permissions.identity.getUserAccount(), userHierarchyItem.ancestorOrganizationUnitId)) {
                    underAuthority = true;
                    break;
                }
            }
            if (!underAuthority) {
                throw (0, api_error_1.generateForbiddenError)({
                    message: "You can not set password of the user!",
                });
            }
        }
        let hash = (0, dal_utils_1.saltHashPassword)(identityChangePassword.password);
        await this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.users).whereNull("deletedAt").where("id", identityChangePassword.id).update({
            salt: hash.salt,
            hashedPassword: hash.passwordHash,
            updatedAt: new Date(),
        });
        await this.revokeOAuthTokensPasswordReset(organizationId, jwt.userId, { userId: identityChangePassword.id });
    }
    async changeMyPassword(organizationId, userId, changePasswordRequest) {
        let userOrganization = await this.dbClient
            .withSchema(organizationId)
            .table("userOrganizations as uo")
            .innerJoin("users as u", "u.id", "uo.userId")
            .whereNull("u.deletedAt")
            .whereNull("uo.deletedAt")
            .where("uo.userId", userId)
            .where("uo.organizationId", organizationId)
            .first("u.salt", "u.hashedPassword");
        if (!userOrganization) {
            (0, dal_access_error_1.throwDbAccessNotFoundError)("User is not found!");
        }
        let oldHash = (0, dal_utils_1.hashPasswordWithSalt)(changePasswordRequest.oldPassword, userOrganization.salt);
        if (oldHash.passwordHash !== userOrganization.hashedPassword) {
            (0, dal_access_error_1.throwDbAccessBadRequestError)("Old password is wrong!");
        }
        let newhash = (0, dal_utils_1.saltHashPassword)(changePasswordRequest.newPassword);
        await this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.users).whereNull("deletedAt").where("id", userId).update({
            salt: newhash.salt,
            hashedPassword: newhash.passwordHash,
            updatedAt: new Date(),
        });
        await this.revokeOAuthTokensPasswordReset(organizationId, userId, { userId });
        return Promise.resolve();
    }
    async getUsersFullnamesAndUniqueIds(organizationId, userIds, trx) {
        const { rows } = await (trx ?? this._pgPool).query(`
			SELECT "userId", name || ' ' || surname as fullname, "uniqueId" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}"
			WHERE "userId" = ANY($1::uuid[])
			`, [userIds]);
        return rows.map((row) => ({
            userId: row.userId,
            fullName: row.fullname,
            uniqueId: row.uniqueId,
        }));
    }
    async getUsersFullnamesAndUniqueIdsKnex(organizationId, userIds, trx) {
        const rows = await trx
            .select("userId", this._knex.raw(`name || ' ' || surname as fullname`), "uniqueId")
            .from(`${organizationId}.${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}`)
            .whereIn("userId", userIds);
        return rows.map((row) => ({
            userId: row.userId,
            fullName: row.fullname,
            uniqueId: row.uniqueId,
        }));
    }
    async getUserCard(organizationId, userId, trx) {
        let response = {};
        let captionLines = await this.getUserOrganizationCaptionLines(organizationId, [userId]);
        if (captionLines.some((t) => t.id == userId)) {
            const { rows } = await (trx ?? this._pgPool).query(`
				SELECT "uniqueId" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}"
				WHERE "userId" = $1
				`, [userId]);
            response.userCaptions = captionLines.find((t) => t.id == userId).captionLines;
            response.uniqueId = rows[0].uniqueId;
            return response;
        }
        return Promise.resolve(null);
    }
    async getSingleUserOrganizationCaptionLines(organizationId, userId, trx) {
        let captionLines = await this.getUserOrganizationCaptionLines(organizationId, [userId], trx);
        if (captionLines.some((t) => t.id == userId)) {
            return captionLines.find((t) => t.id == userId).captionLines;
        }
        return Promise.resolve(null);
    }
    async getUserOrganizationCaptionLines(organizationId, users, trx) {
        let result = [];
        let userProfileSettings = await dal_manager_1.dbManager.accessOrganization.getOrganizationUserProfileSettings(organizationId);
        let visitorModuleSettings;
        await dal_manager_1.dbManager.systemTransaction(async (trx) => {
            visitorModuleSettings = await (0, dal_utils_1.getOrganizationVisitorModuleSettings)(organizationId, trx);
        });
        const chunkSize = 1000;
        for (let i = 0; i < users.length; i += chunkSize) {
            const userChunk = users.slice(i, i + chunkSize);
            let qb = this.dbClient
                .withSchema(organizationId)
                .from("users as u")
                .innerJoin("userOrganizations as uo", "uo.userId", "u.id")
                .whereNull("u.deletedAt")
                .whereNull("uo.deletedAt")
                .where("uo.organizationId", organizationId)
                .whereIn("u.id", userChunk);
            let now = new Date();
            if (trx)
                qb.transacting(trx);
            let columns = [];
            columns.push("u.id");
            let fieldTypes = new Set(userProfileSettings.captionLineFields.map((c) => c.type));
            for (const field of fieldTypes) {
                switch (field) {
                    case dal_constants_1.DalConstants.OrganizationCaptionLineType.profilePhoto:
                        {
                            qb.leftJoin("userOrganizationProfiles as uop2", "uop2.userOrganizationId", "uo.id").whereNull("uop2.deletedAt");
                            columns.push("uop2.thumbnail");
                        }
                        break;
                    case dal_constants_1.DalConstants.OrganizationCaptionLineType.userOrganizationProfile:
                        {
                            qb.leftJoin("userOrganizationProfiles as uop", "uop.userOrganizationId", "uo.id").whereNull("uop.deletedAt");
                            columns.push(this.dbClient.raw(`
									json_build_object(
									'userId', uop."userId",
									'uniqueId', uop."uniqueId",
									'name', uop."name",
									'surname', uop."surname",
									'fullname',uop."name" || ' ' || uop."surname",
									'email',uop."email",
									'extensionFields',uop."extensionFields"
									) as "profile"
								`));
                            qb.leftJoin("organizationVisitorProfiles as ovp", (join) => {
                                join.on("ovp.userOrganizationId", "uo.id").andOn(this.dbClient.raw('ovp."deletedAt" is null'));
                            });
                            columns.push(this.dbClient.raw(`
									json_build_object(
									'userId', uo."userId",
									'uniqueId', ovp."uniqueId",
									'name', ovp."name",
									'surname', ovp."surname",
									'fullname', COALESCE(ovp."name" || ' ', '')  || ' ' || COALESCE(ovp."surname" || ' ', ''),
									'email', '',
									'extensionFields',ovp."extensionFields"
									) as "visitorProfile"
								`));
                        }
                        break;
                    case dal_constants_1.DalConstants.OrganizationCaptionLineType.userGroup:
                        {
                            qb.leftJoin("userGroupUserOrganizations as uguo", "uguo.userOrganizationId", "uo.id").leftJoin("userGroups as ug", (join) => {
                                join.on("ug.id", "uguo.userGroupId").andOn(this.dbClient.raw('uguo."deletedAt" is null')).andOn(this.dbClient.raw('ug."deletedAt" is null'));
                            });
                            columns.push(this.dbClient.raw(`
						json_build_object(
						'id', ug."id",
						'name', ug."name",
						'colorCode', ug."colorCode"
						) as "userGroup"
					`));
                        }
                        break;
                    case dal_constants_1.DalConstants.OrganizationCaptionLineType.organizationUnit:
                        {
                            qb.leftJoin("userOrganizationOrganizationUnits as uoou", "uoou.userOrganizationId", "uo.id").leftJoin("organizationUnits as org", (join) => {
                                join.on("org.id", "uoou.organizationUnitId").andOn(this.dbClient.raw('uoou."deletedAt" IS NULL')).andOn(this.dbClient.raw('org."deletedAt" IS NULL'));
                            });
                            columns.push(this.dbClient.raw(`
					json_build_object(
					'id', org."id",
					'name', org."name"
					) as "organizationUnit"
				`));
                        }
                        break;
                    case dal_constants_1.DalConstants.OrganizationCaptionLineType.credentials:
                        {
                            qb.leftJoin("userOrganizationCredentials as uoc", (join) => {
                                join.on("uoc.userId", "uo.userId")
                                    .andOn("uoc.organizationId", "uo.organizationId")
                                    .andOn(this.dbClient.raw(' (uoc."expiresOn" >= ?  or uoc."expiresOn" is null)', now))
                                    .andOn(this.dbClient.raw('uoc."deletedAt" is null'));
                            });
                            columns.push(this.dbClient.raw(`
						json_build_object(
						'id', uoc."id",
						'type', uoc."type",
						'data', uoc."data",
						'extensionFields',uoc."extensionFields"
						) as "credential"
					`));
                        }
                        break;
                    case dal_constants_1.DalConstants.OrganizationCaptionLineType.warning:
                        {
                            qb.leftJoin("userOrganizationForbiddances as uof", (join) => {
                                join.on("uof.userId", "uo.userId")
                                    .andOn("uof.organizationId", "uo.organizationId")
                                    .andOn(this.dbClient.raw(' (uof."startUtc" <= ?  or uof."startUtc" is null) and (uof."endUtc" >= ? or uof."endUtc" is null)', [now, now]))
                                    .andOn(this.dbClient.raw('uof."deletedAt" is null'));
                            });
                            columns.push(this.dbClient.raw(`
							json_build_object(
							'id', uof."id"
							) as "forbiddance"
						`));
                            qb.leftJoin("organizationVisitorProfiles as ovp2", (join) => {
                                join.on("ovp2.userOrganizationId", "uo.id").andOn(this.dbClient.raw('ovp2."deletedAt" is null'));
                            });
                            qb.leftJoin("organizationVisitorStates as ovsf", (join) => {
                                join.on("ovp2.id", "ovsf.organizationVisitorProfileId")
                                    .andOn(this.dbClient.raw(' (ovsf."startUtc" <= ?  or ovsf."startUtc" is null) and (ovsf."endUtc" >= ? or ovsf."endUtc" is null)', [now, now]))
                                    .andOn(this.dbClient.raw('ovsf."deletedAt" is null'))
                                    .andOn(this.dbClient.raw(' ovsf."state" = ?', dal_constants_1.DalConstants.OrganizationVisitorStates.Forbidden));
                            });
                            columns.push(this.dbClient.raw(`
							json_build_object(
							'id', ovsf."id"
							) as "visitorForbiddance"
						`));
                            qb.leftJoin("organizationVisitorStates as ovsw", (join) => {
                                join.on("ovp2.id", "ovsw.organizationVisitorProfileId")
                                    .andOn(this.dbClient.raw(' (ovsw."startUtc" <= ?  or ovsw."startUtc" is null) and (ovsw."endUtc" >= ? or ovsw."endUtc" is null)', [now, now]))
                                    .andOn(this.dbClient.raw('ovsw."deletedAt" is null'))
                                    .andOn(this.dbClient.raw(' ovsw."state" = ?', dal_constants_1.DalConstants.OrganizationVisitorStates.TemporaryPermit));
                            });
                            columns.push(this.dbClient.raw(`
							json_build_object(
							'id', ovsw."id"
							) as "visitorPermit"
						`));
                        }
                        break;
                    default:
                        break;
                }
            }
            let captionResults = await qb.column(columns).select();
            for (const user of userChunk) {
                let captionResult = captionResults.filter((r) => r.id == user);
                if (!captionResult || captionResult.length < 1) {
                    result.push({ id: user, captionLines: [] });
                    continue;
                }
                let captions = [];
                for (let captionItem = 0; captionItem < userProfileSettings.captionLineFields.length; captionItem++) {
                    let captionField = userProfileSettings.captionLineFields[captionItem];
                    let texts = [];
                    let colors = [];
                    let profilePhoto = null;
                    let translate = false;
                    switch (captionField.type) {
                        case dal_constants_1.DalConstants.OrganizationCaptionLineType.profilePhoto:
                            {
                                let thumbnails = captionResult.filter((c) => c.thumbnail).map((c) => c.thumbnail);
                                if (thumbnails.length > 0) {
                                    profilePhoto = Buffer.from(thumbnails[0]).toString();
                                }
                            }
                            break;
                        case dal_constants_1.DalConstants.OrganizationCaptionLineType.userOrganizationProfile:
                            let profiles = lodash_1.default.uniqBy(captionResult
                                .map((c) => c.profile)
                                .concat(captionResult.map((c) => {
                                (0, business_visitor_1.hideVisitorUniqueId)(c.visitorProfile, visitorModuleSettings);
                                return c.visitorProfile;
                            })), function (x) {
                                return x.userId;
                            });
                            for (const profile of profiles) {
                                if (captionField.isExtension) {
                                    if (profile.extensionFields) {
                                        let field = profile.extensionFields[captionField.name];
                                        if (field && lodash_1.default.isArray(field)) {
                                            texts.push(field.join("-"));
                                        }
                                        else {
                                            texts.push(field);
                                        }
                                    }
                                }
                                else {
                                    profile[captionField.name] ? texts.push(profile[captionField.name]) : "";
                                }
                                if (captionField.colorCode) {
                                    colors.push(captionField.colorCode);
                                }
                            }
                            break;
                        case dal_constants_1.DalConstants.OrganizationCaptionLineType.userGroup:
                            let userGroups = lodash_1.default.uniqBy(captionResult.map((c) => c.userGroup), function (x) {
                                return x.id;
                            });
                            for (const userGroup of userGroups) {
                                texts.push(userGroup[captionField.name]);
                                if (captionField.colorCode) {
                                    colors.push(captionField.colorCode);
                                }
                                else {
                                    colors.push(userGroup.colorCode);
                                }
                            }
                            break;
                        case dal_constants_1.DalConstants.OrganizationCaptionLineType.organizationUnit:
                            let organizationUnits = lodash_1.default.uniqBy(captionResult.map((c) => c.organizationUnit), function (x) {
                                return x.id;
                            });
                            for (const organizationUnit of organizationUnits) {
                                texts.push(organizationUnit[captionField.name]);
                                if (captionField.colorCode) {
                                    colors.push(captionField.colorCode);
                                }
                            }
                            break;
                        case dal_constants_1.DalConstants.OrganizationCaptionLineType.credentials:
                            let credentials = lodash_1.default.uniqBy(captionResult.map((c) => c.credential), function (x) {
                                return x.id;
                            });
                            for (const credential of credentials) {
                                if (captionField.filterTypes && captionField.filterTypes.length > 0) {
                                    if (!captionField.filterTypes.some((a) => parseInt(a) == parseInt(credential["type"])))
                                        continue;
                                }
                                if (captionField.isExtension) {
                                    let field = credential.extensionFields ? credential.extensionFields[captionField.name] : null;
                                    if (field)
                                        texts.push(field.value);
                                }
                                else {
                                    texts.push(credential[captionField.name]);
                                }
                                if (captionField.colorCode) {
                                    colors.push(captionField.colorCode);
                                }
                            }
                            break;
                        case dal_constants_1.DalConstants.OrganizationCaptionLineType.warning:
                            {
                                let userForbiddance = lodash_1.default.uniqBy(captionResult.map((c) => c.forbiddance).filter((c) => c.id), function (x) {
                                    return x.id;
                                });
                                let userVisitorForbiddance = lodash_1.default.uniqBy(captionResult.map((c) => c.visitorForbiddance).filter((c) => c.id), function (x) {
                                    return x.id;
                                });
                                let userVisitorPermit = lodash_1.default.uniqBy(captionResult.map((c) => c.visitorPermit).filter((c) => c.id), function (x) {
                                    return x.id;
                                });
                                if (userForbiddance && userForbiddance.length > 0) {
                                    texts.push(dal_constants_1.DalConstants.UserCaptionKeys.USER_FORBIDDEN);
                                    colors.push(dal_constants_1.DalConstants.UserCaptionColorCode.code);
                                    translate = true;
                                }
                                if (userVisitorForbiddance && userVisitorForbiddance.length > 0) {
                                    texts.push(dal_constants_1.DalConstants.UserCaptionKeys.VISITOR_FORBIDDEN);
                                    colors.push(dal_constants_1.DalConstants.UserCaptionColorCode.code);
                                    translate = true;
                                }
                                if (userVisitorPermit && userVisitorPermit.length > 0) {
                                    texts.push(dal_constants_1.DalConstants.UserCaptionKeys.VISITOR_PERMITTED);
                                    colors.push(dal_constants_1.DalConstants.UserCaptionColorCode.code);
                                    translate = true;
                                }
                            }
                            break;
                        default:
                            break;
                    }
                    if (captionField.type != dal_constants_1.DalConstants.OrganizationCaptionLineType.profilePhoto) {
                        if (texts.length == 0) {
                            captions.push(null);
                        }
                        else {
                            let filteredTexts = lodash_1.default.compact(texts);
                            if (filteredTexts.length == 0) {
                                captions.push(null);
                            }
                            else {
                                if (translate) {
                                    captions.push({
                                        text: texts,
                                        colorCode: colors,
                                        translate: true,
                                    });
                                }
                                else {
                                    captions.push({
                                        text: texts,
                                        colorCode: colors,
                                    });
                                }
                            }
                        }
                    }
                    else {
                        captions.push({
                            avatar: profilePhoto ? profilePhoto : dal_constants_1.DalConstants.DefaultAvatar,
                            text: ["__avatar"],
                        });
                    }
                }
                for (let captionIndex = 0; captionIndex < captions.length; ++captionIndex) {
                    if (!captions[captionIndex]) {
                        captions[captionIndex] = {
                            text: [""],
                        };
                    }
                }
                captions = lodash_1.default.compact(captions);
                result.push({ id: user, captionLines: captions });
            }
        }
        return Promise.resolve(result);
    }
    async getUserOrganizationProfile(organizationId, users) {
        let result = [];
        let queryParamIndex = 1;
        let queryParams = [];
        let query = `
            SELECT
                u."id",
                json_build_object(
                    'userId', uop."userId",
                    'uniqueId', uop."uniqueId",
                    'name', uop."name",
                    'surname', uop."surname",
                    'email',uop."email",
                    'phoneNumber', uop."phoneNumber",
                    'extensionFields',uop."extensionFields"
                ) AS "profile",
                json_build_object(
                    'userId', uo."userId",
                    'uniqueId', ovp."uniqueId",
                    'name', ovp."name",
                    'surname', ovp."surname",
                    'email', '',
                    'extensionFields',ovp."extensionFields"
                ) AS "visitorProfile"
            FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}" u
            INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo ON uo."userId" = u."id" AND u."deletedAt" IS NULL AND uo."deletedAt" IS NULL AND uo."organizationId" = $${queryParamIndex++} AND u."id" = ANY ($${queryParamIndex++}::uuid[])
            LEFT JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" uop ON uop."userOrganizationId" = uo."id" AND uop."deletedAt" IS NULL
            LEFT JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorProfiles}" ovp ON ovp."userOrganizationId" = uo.id AND ovp."deletedAt" IS NULL
        `;
        queryParams.push(organizationId, users);
        const dbResult = await this._pgPool.query(query, queryParams);
        for (const user of users) {
            let userProfile = dbResult.rows.filter((r) => r.id == user);
            if (!userProfile || userProfile.length < 1) {
                result.push({ userId: user, userOrganizationProfile: null });
                continue;
            }
            let profile = lodash_1.default.uniqBy(userProfile.map((c) => c.profile).concat(userProfile.map((c) => c.visitorProfile)), function (x) {
                return x.userId;
            })[0];
            let extensionFields = [];
            if (profile.extensionFields) {
                let keys = Object.keys(profile.extensionFields);
                if (keys && keys.length > 0) {
                    for (const key of keys) {
                        let value = profile.extensionFields[key];
                        if (value && lodash_1.default.isArray(value)) {
                            extensionFields.push({
                                name: key,
                                value: value.join(", "),
                            });
                        }
                        else {
                            extensionFields.push({
                                name: key,
                                value: value,
                            });
                        }
                    }
                }
            }
            profile.extensionFields = extensionFields;
            result.push({ userId: user, userOrganizationProfile: profile });
        }
        return Promise.resolve(result);
    }
    async getUserApplicationProfile(organizationId, userId) {
        let profile = await this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations)
            .where("userId", userId)
            .where("organizationId", organizationId)
            .whereNull("deletedAt")
            .first("applicationprofile");
        if (profile)
            return Promise.resolve(profile.applicationprofile);
        (0, dal_access_error_1.throwDbAccessNotFoundErrorTr)("ERRORS.USER.APPLICATIONPROFILENOTFOUND");
    }
    async upsertUserApplicationProfile(organizationId, userId, profile) {
        await this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations).where("userId", userId).where("organizationId", organizationId).update({
            updatedAt: new Date(),
            applicationprofile: profile,
        });
        return Promise.resolve();
    }
    async listUserOrganizationAuthMethods(username, showHidden, trx) {
        const qParams = [];
        let q = `
			SELECT
				oam.*,
				o.code AS "organizationCode",
				o.name AS "organizationName",
				o.logo,
				o.id AS "organizationId",
				o.alias
			FROM public."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationList}" AS o
			INNER JOIN public."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationAuthenticationMethods}" AS oam
				ON oam."organizationId" = o.id
		`;
        let whereQ = `
			WHERE oam."deletedAt" IS NULL
		`;
        if (app_config_1.appConfig.isCloudServer) {
            q += `
			INNER JOIN public."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationMapping}" AS uom
				ON uom."organizationId" = o.id
			INNER JOIN public."${dal_db_armon_schema_1.ArmonSchema.tableNames.userList}" AS ul
				ON ul."id" = uom."userId"
			`;
            qParams.push(username);
            whereQ += `
				AND lower(ul.username) = lower ($${qParams.length})
			`;
        }
        if (!showHidden) {
            whereQ += `
				AND (oam."isHidden" IS FALSE OR oam."isHidden" IS NULL)
			`;
        }
        const finalQ = q + whereQ + ` ORDER BY o.name ASC`;
        const dbResult = await trx.query(finalQ, qParams);
        let result = [];
        let orgIds = lodash_1.default.uniq(dbResult.rows.map((r) => r.organizationId));
        for (const orgId of orgIds) {
            let authMethods = dbResult.rows.filter((r) => r.organizationId == orgId);
            result.push({
                alias: authMethods[0].alias,
                code: authMethods[0].organizationCode,
                id: orgId,
                logo: authMethods[0].logo,
                name: authMethods[0].organizationName,
                authentications: authMethods.map((m) => {
                    return {
                        authenticationMethod: m.authenticationMethod,
                        config: m.config,
                        grantType: m.grantType,
                        id: m.id,
                        isDefault: m.isDefault,
                        name: m.name,
                    };
                }),
            });
        }
        return Promise.resolve(result);
    }
    async canUserSeeOtherUsersAccess(params) {
        let transactionScope = async (trx) => {
            const result = await trx.query(`
            SELECT COUNT(*) AS "count"
            FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userAccessRights}" AS uar
            INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessControlPoints}" AS acp 
				ON acp."id" = uar."accessControlPointId" AND uar."read" = TRUE AND uar."deletedAt" IS NULL AND acp."deletedAt" IS NULL
            INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" AS uo 
				ON uo."userId" = uar."userId" AND uo."organizationId" = acp."organizationId" AND uo."deletedAt" IS NULL
            INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}" AS uoou 
				ON uoou."userOrganizationId" = uo."id" AND uoou."deletedAt" IS NULL
            INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.roles}" AS r 
				ON uoou."roleId" = r."id" AND POSITION('l:d' IN r."permissions") > 0
            INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" AS ou 
				ON uoou."organizationUnitId" = ou."id" AND ou."deletedAt" IS NULL 
            INNER JOIN (
				SELECT (oui."id" || ',' || (CASE WHEN oui."ancestorIds" IS NULL THEN '' ELSE oui."ancestorIds" END)) AS "opath" 
				FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" AS oui
				INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}" AS uooui 
					ON oui."id" = uooui."organizationUnitId" AND oui."deletedAt" IS NULL AND uooui."deletedAt" IS NULL
				INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" AS uoi 
					ON uooui."userOrganizationId" = uoi."id" AND uoi."deletedAt" IS NULL
				WHERE "uoi"."userId" = $1
			) AS T2 
				ON POSITION (ou."id"::text IN T2."opath") > 0
            WHERE
            uar."accessControlPointId" = $2
            AND uar."userId" = $3
            `, [params.userId, params.accessControlPointId, params.requesterUserId]);
            return result.rows && result.rows.length > 0 && parseInt(result.rows[0].count) > 0;
        };
        if (params.trx) {
            return transactionScope(params.trx);
        }
        else {
            return await dal_manager_1.dbManager.organizationTransaction(transactionScope, params.requesterUserId, params.organizationId);
        }
    }
    async getUsername(organizationId, userId) {
        const { rows } = await this._pgPool.query(`
            SELECT username FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}"
            WHERE id = $1`, [userId]);
        return rows[0].username;
    }
    async canManagerRemoveUserFromUnits(params) {
        const query = `
            SELECT * FROM "${params.organizationId}"."userOrganizationOrganizationUnits" AS uoou
            INNER JOIN "${params.organizationId}"."userOrganizations" AS uo ON uo.id = uoou."userOrganizationId"
            INNER JOIN "${params.organizationId}"."organizationUnits" As ou ON ou.id = uoou."organizationUnitId"
            INNER JOIN "${params.organizationId}".roles AS r ON "uoou"."roleId" = r."id"
            WHERE uo."userId" = $1
            AND POSITION ($2 IN r.permissions) > 0
            AND r."deletedAt" IS NULL
            AND uoou."deletedAt" IS NULL
            AND POSITION (
                uoou."organizationUnitId"::text IN (
                    SELECT string_agg(oui."id" || ',' || (CASE WHEN oui."ancestorIds" IS NULL THEN '' ELSE oui."ancestorIds" END), ',') AS "opath"
                        FROM "${params.organizationId}"."organizationUnits" AS oui
                        INNER JOIN "${params.organizationId}"."userOrganizationOrganizationUnits" AS uooui ON uooui."organizationUnitId" = oui.id
                        INNER JOIN "${params.organizationId}"."userOrganizations" AS uoi ON uoi.id = uooui."userOrganizationId"
                        WHERE (uoi."deletedAt" IS NULL AND
                            uooui."deletedAt" IS NULL AND
                            uooui."organizationUnitId" != ANY ($3::UUID[]) AND
                            uoi."userId" = $4)
                            OR oui.id = ANY ($5::UUID[])
                )
            ) > 0
        `;
        let countResult = await dal_manager_1.dbManager.pgTransactionMainDb(async (trx) => {
            let { rowCount } = await trx.query(query, [
                params.managerId,
                predefined_permissions_1.Permissions.identity.getHierarchy(),
                params.removedOrganizationUnitIds,
                params.userId,
                params.addedOrganizationUnitIds,
            ]);
            return rowCount;
        });
        if (countResult > 0)
            return true;
        return false;
    }
    async getUserSettings(params) {
        const { rows, rowCount } = await params.trx.query(`
			SELECT u.settings AS usettings, uo.settings as uosettings FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" AS uo
			INNER JOIN "${params.organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.users} AS u ON u.id = uo."userId"
            WHERE u.id = $1
        `, [params.userId]);
        if (rowCount < 1) {
            return null;
        }
        const row = rows[0];
        return {
            locale: row.usettings.locale,
            notification: row.uosettings.notification,
        };
    }
    async setUserSettings(params) {
        const userSettings = await this.getUserSettings({ organizationId: params.organizationId, userId: params.userId, trx: params.trx });
        if (params.forceCloseAllEmailMediums && userSettings.notification.mediumSettings.custom) {
            for (const key of Object.keys(userSettings.notification.mediumSettings.custom)) {
                userSettings.notification.mediumSettings.custom[key] = {
                    ...userSettings.notification.mediumSettings.custom[key],
                    email: false,
                };
            }
        }
        if (!params.updateMode) {
            userSettings.notification.mediumSettings.custom = {};
        }
        const customNotifications = {};
        if (params.settings.notification.mediumSettings.custom) {
            for (const key of Object.keys(params.settings.notification.mediumSettings.custom)) {
                customNotifications[key] = {
                    email: params.settings.notification.mediumSettings.custom[key].email ??
                        userSettings.notification.mediumSettings.custom[key]?.email ??
                        userSettings.notification.mediumSettings.general.email,
                    pushNotification: params.settings.notification.mediumSettings.custom[key].pushNotification ??
                        userSettings.notification.mediumSettings.custom[key]?.pushNotification ??
                        userSettings.notification.mediumSettings.general.pushNotification,
                    sms: params.settings.notification.mediumSettings.custom[key].sms ??
                        userSettings.notification.mediumSettings.custom[key]?.sms ??
                        userSettings.notification.mediumSettings.general.sms,
                    web: params.settings.notification.mediumSettings.custom[key].web ??
                        userSettings.notification.mediumSettings.custom[key]?.web ??
                        userSettings.notification.mediumSettings.general.web,
                };
            }
        }
        const updatedSettings = {
            locale: params.settings.locale ?? userSettings.locale,
            notification: {
                mediumSettings: {
                    general: {
                        email: params.settings?.notification?.mediumSettings?.general?.email ?? userSettings.notification.mediumSettings.general.email,
                        pushNotification: params.settings?.notification?.mediumSettings?.general?.pushNotification ?? userSettings.notification.mediumSettings.general.pushNotification,
                        sms: params.settings?.notification?.mediumSettings?.general?.sms ?? userSettings.notification.mediumSettings.general.sms,
                        web: params.settings?.notification?.mediumSettings?.general?.web ?? userSettings.notification.mediumSettings.general.web,
                    },
                    custom: { ...userSettings.notification.mediumSettings.custom, ...customNotifications },
                },
            },
        };
        if (params.settings.notification) {
            await params.trx.query(`
				UPDATE "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}"
				SET settings = $1
				WHERE "userId" = $2
				AND "deletedAt" IS NULL
				`, [{ notification: updatedSettings.notification }, params.userId]);
        }
        if (params.settings.locale) {
            await params.trx.query(`
				UPDATE "${params.organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.users}
				SET settings = $1
				WHERE "id" = $2
				AND "deletedAt" IS NULL
				`, [
                {
                    locale: updatedSettings.locale,
                },
                params.userId,
            ]);
        }
    }
    async getMobileUserIds(params) {
        const { rows, rowCount } = await params.trx.query(`SELECT id FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}" WHERE "notificationToken" IS NOT NULL`);
        return rowCount > 0 ? rows.map((r) => r.id) : null;
    }
    async getAllUserNotificationRedisPersistentData(params) {
        const query = `
			SELECT
			u.id,
			u.settings->>'locale' as l,
			uo.settings->'notification' as "s",
			uop.name || ' ' || uop.surname as f,
			uop."uniqueId" as u,
			uo."roleId" as r,
			uo."isDisabled" as d,
			(CASE WHEN uop."email" IS NULL THEN up.email ELSE uop.email END) as e,
			(CASE WHEN uop."phoneNumber" IS NULL THEN up."phoneNumber" ELSE uop."phoneNumber" END) as "m",
			u."notificationToken" as n,
			u."notificationTokenType" as z
			FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}" u
			INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo 
			ON uo."userId" = u."id" 
			AND u."deletedAt" IS NULL 
			AND uo."deletedAt" IS NULL
			LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userProfiles}" up 
			ON uo."userId" = up."userId" 
			AND up."deletedAt" IS NULL 
			AND uo."deletedAt" IS NULL                
			LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" uop 
			ON uop."userOrganizationId" = uo."id" 
			AND uop."deletedAt" IS NULL
	`;
        const trxx = params.trx ?? (await this._pgPool.connect());
        const cursor = trxx.query(new Cursor(query));
        let userNotificationRedisInfo = [];
        while (true) {
            try {
                userNotificationRedisInfo = await new Promise((resolve, reject) => {
                    cursor.read(100, async (err, rows) => {
                        if (err) {
                            reject(err);
                        }
                        else {
                            if (rows.length > 0) {
                                resolve(rows);
                            }
                            else {
                                resolve([]);
                            }
                        }
                    });
                });
                await params.onData(userNotificationRedisInfo);
            }
            catch (error) {
                app_logs_1.logger.error("Error while fetching user cache data with cursor!", error);
            }
            if (userNotificationRedisInfo.length < 100) {
                break;
            }
        }
        try {
            await new Promise((resolve, reject) => {
                cursor.close((err) => {
                    if (err) {
                        reject(err);
                    }
                    else {
                        resolve();
                    }
                });
            });
            if (!params.trx) {
                trxx.release();
            }
        }
        catch (error) {
            if (!params.trx) {
                trxx?.release(error);
            }
            app_logs_1.logger.error(error);
        }
    }
    async getAllUserOrganizationBadgeRedisPersistentData(params) {
        const userProfileSettings = await dal_manager_1.dbManager.accessOrganization.getOrganizationUserProfileSettingsPg({ organizationId: params.organizationId, trx: params.trx });
        const qBindings = [];
        let qx = 1;
        let qFrom = `
			FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}" u
			INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo
				ON uo."userId" = u.id AND uo."deletedAt" IS NULL AND u."deletedAt" IS NULL
		`;
        const now = new Date();
        const columns = [];
        columns.push("u.id");
        const fieldTypes = new Set(userProfileSettings.captionLineFields.map((c) => c.type));
        for (const field of fieldTypes) {
            switch (field) {
                case dal_constants_1.DalConstants.OrganizationCaptionLineType.profilePhoto:
                    {
                        qFrom += `
							LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" as uop2 
							ON uop2."userOrganizationId" = uo.id AND uop2."deletedAt" IS NULL
							`;
                        columns.push("uop2.thumbnail");
                    }
                    break;
                case dal_constants_1.DalConstants.OrganizationCaptionLineType.userOrganizationProfile:
                    {
                        qFrom += `
							LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" as uop 
								ON uop."userOrganizationId" = uo.id AND uop."deletedAt" IS NULL
								`;
                        qFrom += `
							LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorProfiles}" as ovp 
							ON ovp."userOrganizationId" = uo.id AND ovp."deletedAt" IS NULL
					`;
                        columns.push(`
                        JSON_BUILD_OBJECT(
							'userId', uop."userId",
							'uniqueId', uop."uniqueId",
							'name', uop."name",
							'surname', uop."surname",
							'fullname',uop."name" || ' ' || uop."surname",
							'email',uop."email",
							'extensionFields',uop."extensionFields"
                        ) as "profile" `);
                        columns.push(`
							JSON_BUILD_OBJECT(
								'userId', uo."userId",
								'uniqueId', ovp."uniqueId",
								'name', ovp."name",
								'surname', ovp."surname",
								'fullname', COALESCE(ovp."name" || ' ', '')  || ' ' || COALESCE(ovp."surname" || ' ', ''),
								'email', '',
								'extensionFields', ovp."extensionFields"
							) as "visitorProfile" `);
                    }
                    break;
                case dal_constants_1.DalConstants.OrganizationCaptionLineType.userGroup:
                    {
                        qFrom += `
						LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupUserOrganizations}" uguo 
							ON uguo."userOrganizationId" = uo.id AND uguo."deletedAt" IS NULL
						LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroups}" ug
							ON ug.id = uguo."userGroupId" AND ug."deletedAt" IS NULL 
						`;
                        columns.push(`
							JSON_BUILD_OBJECT(
								'id', ug."id",
								'name', ug."name",
								'colorCode', ug."colorCode"
							) as "userGroup" `);
                    }
                    break;
                case dal_constants_1.DalConstants.OrganizationCaptionLineType.organizationUnit:
                    {
                        qFrom += `
						LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}" uoou 
							ON uoou."userOrganizationId" = uo.id AND uoou."deletedAt" IS NULL
						LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" ou
							ON ou.id = uoou."organizationUnitId" AND ou."deletedAt" IS NULL
						`;
                        columns.push(`
							JSON_BUILD_OBJECT(
								'id', ou."id",
								'name', ou."name"
							) as "organizationUnit"
						`);
                    }
                    break;
                case dal_constants_1.DalConstants.OrganizationCaptionLineType.credentials:
                    {
                        qFrom += `
							LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials}" uoc 
								ON uoc."userId" = uo."userId"
								AND (uoc."expiresOn" >= $${qx++} OR uoc."expiresOn" IS NULL)
								AND uoc."deletedAt" IS NULL
						`;
                        qBindings.push(now);
                        columns.push(`
							JSON_BUILD_OBJECT(
								'id', uoc."id",
								'type', uoc."type",
								'data', uoc."data",
								'extensionFields',uoc."extensionFields"
							) as "credential"
              		  `);
                    }
                    break;
                case dal_constants_1.DalConstants.OrganizationCaptionLineType.warning:
                    {
                        qFrom += `
						LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationForbiddances}" uof 
							ON uof."userId" = uo."userId" 
							AND (uof."startUtc" <= $${qx}  OR uof."startUtc" IS NULL) 
							AND (uof."endUtc" >= $${qx++} OR uof."endUtc" IS NULL)
							AND uof."deletedAt" IS NULL
						`;
                        qBindings.push(now);
                        columns.push(`
							JSON_BUILD_OBJECT(
								'id', uof."id"
							) as "forbiddance"
                    	`);
                        qFrom += `
							LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorProfiles}" as ovp2
								ON ovp2."userOrganizationId" = uo.id AND ovp2."deletedAt" IS NULL
							`;
                        qFrom += `
						LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorStates}" as ovsf 
							ON ovp2."id" = uo.id AND ovsf."organizationVisitorProfileId" IS NULL
							AND (ovsf."startUtc" <= $${qx}  OR ovsf."startUtc" IS NULL) AND (ovsf."endUtc" >= $${qx++} OR ovsf."endUtc" IS NULL)
							AND  ovsf."deletedAt" IS NULL
							AND ovsf.state = $${qx++}
						`;
                        qBindings.push(now, dal_constants_1.DalConstants.OrganizationVisitorStates.Forbidden);
                        columns.push(`
							JSON_BUILD_OBJECT(
                     	   		'id', ovsf."id"
                        	) as "visitorForbiddance"
                    	`);
                        qFrom += `
						LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorStates}" as ovsw
							ON ovp2."id" = uo.id AND ovsw."organizationVisitorProfileId" IS NULL
							AND (ovsw."startUtc" <= $${qx}  OR ovsw."startUtc" IS NULL) AND (ovsw."endUtc" >= $${qx++} OR ovsw."endUtc" IS NULL)
							AND  ovsw."deletedAt" IS NULL
							AND ovsw.state = $${qx++}
						`;
                        qBindings.push(now, dal_constants_1.DalConstants.OrganizationVisitorStates.TemporaryPermit);
                        columns.push(`
							JSON_BUILD_OBJECT(
								'id', ovsw."id"
							) as "visitorPermit"
                   		 `);
                    }
                    break;
                default:
                    break;
            }
        }
        const q = `SELECT ${columns.join(" , ")} ${qFrom}`;
        const trxx = params.trx ?? (await this._pgPool.connect());
        const cursor = await trxx.query(new Cursor(q, qBindings));
        let fetchedData = [];
        while (true) {
            try {
                fetchedData = await new Promise((resolve, reject) => {
                    cursor.read(100, async (err, rows) => {
                        if (err) {
                            reject(err);
                        }
                        else {
                            const badges = [];
                            const userIds = rows.map((r) => r.id);
                            for (const userId of userIds) {
                                let captionResult = rows.filter((r) => r.id === userId);
                                let captions = [];
                                for (let captionItem = 0; captionItem < userProfileSettings.captionLineFields.length; captionItem++) {
                                    let captionField = userProfileSettings.captionLineFields[captionItem];
                                    let texts = [];
                                    let colors = [];
                                    let profilePhoto = null;
                                    let translate = false;
                                    switch (captionField.type) {
                                        case dal_constants_1.DalConstants.OrganizationCaptionLineType.profilePhoto:
                                            {
                                                let thumbnails = captionResult.filter((c) => c.thumbnail).map((c) => c.thumbnail);
                                                if (thumbnails.length > 0) {
                                                    profilePhoto = Buffer.from(thumbnails[0]).toString();
                                                }
                                            }
                                            break;
                                        case dal_constants_1.DalConstants.OrganizationCaptionLineType.userOrganizationProfile:
                                            let profiles = lodash_1.default.uniqBy(captionResult.map((c) => c.profile).concat(captionResult.map((c) => c.visitorProfile)), function (x) {
                                                return x.userId;
                                            });
                                            for (const profile of profiles) {
                                                if (captionField.isExtension) {
                                                    if (profile.extensionFields) {
                                                        let field = profile.extensionFields[captionField.name];
                                                        if (field && lodash_1.default.isArray(field)) {
                                                            texts.push(field.join("-"));
                                                        }
                                                        else {
                                                            texts.push(field);
                                                        }
                                                    }
                                                }
                                                else {
                                                    profile[captionField.name] ? texts.push(profile[captionField.name]) : "";
                                                }
                                                if (captionField.colorCode) {
                                                    colors.push(captionField.colorCode);
                                                }
                                            }
                                            break;
                                        case dal_constants_1.DalConstants.OrganizationCaptionLineType.userGroup:
                                            let userGroups = lodash_1.default.uniqBy(captionResult.map((c) => c.userGroup), function (x) {
                                                return x.id;
                                            });
                                            for (const userGroup of userGroups) {
                                                texts.push(userGroup[captionField.name]);
                                                if (captionField.colorCode) {
                                                    colors.push(captionField.colorCode);
                                                }
                                                else {
                                                    colors.push(userGroup.colorCode);
                                                }
                                            }
                                            break;
                                        case dal_constants_1.DalConstants.OrganizationCaptionLineType.organizationUnit:
                                            let organizationUnits = lodash_1.default.uniqBy(captionResult.map((c) => c.organizationUnit), function (x) {
                                                return x.id;
                                            });
                                            for (const organizationUnit of organizationUnits) {
                                                texts.push(organizationUnit[captionField.name]);
                                                if (captionField.colorCode) {
                                                    colors.push(captionField.colorCode);
                                                }
                                            }
                                            break;
                                        case dal_constants_1.DalConstants.OrganizationCaptionLineType.credentials:
                                            let credentials = lodash_1.default.uniqBy(captionResult.map((c) => c.credential), function (x) {
                                                return x.id;
                                            });
                                            for (const credential of credentials) {
                                                if (captionField.filterTypes && captionField.filterTypes.length > 0) {
                                                    if (!captionField.filterTypes.some((a) => parseInt(a) == parseInt(credential["type"])))
                                                        continue;
                                                }
                                                if (captionField.isExtension) {
                                                    let field = credential.extensionFields ? credential.extensionFields[captionField.name] : null;
                                                    if (field)
                                                        texts.push(field.value);
                                                }
                                                else {
                                                    texts.push(credential[captionField.name]);
                                                }
                                                if (captionField.colorCode) {
                                                    colors.push(captionField.colorCode);
                                                }
                                            }
                                            break;
                                        case dal_constants_1.DalConstants.OrganizationCaptionLineType.warning:
                                            {
                                                let userForbiddance = lodash_1.default.uniqBy(captionResult.map((c) => c.forbiddance).filter((c) => c.id), function (x) {
                                                    return x.id;
                                                });
                                                let userVisitorForbiddance = lodash_1.default.uniqBy(captionResult.map((c) => c.visitorForbiddance).filter((c) => c.id), function (x) {
                                                    return x.id;
                                                });
                                                let userVisitorPermit = lodash_1.default.uniqBy(captionResult.map((c) => c.visitorPermit).filter((c) => c.id), function (x) {
                                                    return x.id;
                                                });
                                                if (userForbiddance && userForbiddance.length > 0) {
                                                    texts.push(dal_constants_1.DalConstants.UserCaptionKeys.USER_FORBIDDEN);
                                                    colors.push(dal_constants_1.DalConstants.UserCaptionColorCode.code);
                                                    translate = true;
                                                }
                                                if (userVisitorForbiddance && userVisitorForbiddance.length > 0) {
                                                    texts.push(dal_constants_1.DalConstants.UserCaptionKeys.VISITOR_FORBIDDEN);
                                                    colors.push(dal_constants_1.DalConstants.UserCaptionColorCode.code);
                                                    translate = true;
                                                }
                                                if (userVisitorPermit && userVisitorPermit.length > 0) {
                                                    texts.push(dal_constants_1.DalConstants.UserCaptionKeys.VISITOR_PERMITTED);
                                                    colors.push(dal_constants_1.DalConstants.UserCaptionColorCode.code);
                                                    translate = true;
                                                }
                                            }
                                            break;
                                        default:
                                            break;
                                    }
                                    if (captionField.type != dal_constants_1.DalConstants.OrganizationCaptionLineType.profilePhoto) {
                                        if (texts.length == 0) {
                                            captions.push(null);
                                        }
                                        else {
                                            let filteredTexts = lodash_1.default.compact(texts);
                                            if (filteredTexts.length == 0) {
                                                captions.push(null);
                                            }
                                            else {
                                                if (translate) {
                                                    captions.push({
                                                        text: texts,
                                                        colorCode: colors,
                                                        translate: true,
                                                    });
                                                }
                                                else {
                                                    captions.push({
                                                        text: texts,
                                                        colorCode: colors,
                                                    });
                                                }
                                            }
                                        }
                                    }
                                    else {
                                        captions.push({
                                            avatar: profilePhoto ? profilePhoto : dal_constants_1.DalConstants.DefaultAvatar,
                                            text: ["__avatar"],
                                        });
                                    }
                                }
                                for (let captionIndex = 0; captionIndex < captions.length; ++captionIndex) {
                                    if (!captions[captionIndex]) {
                                        captions[captionIndex] = {
                                            text: [""],
                                        };
                                    }
                                }
                                captions = lodash_1.default.compact(captions);
                                badges.push({ id: userId, caption: captions });
                            }
                            resolve(badges);
                        }
                    });
                });
                await params.onData(fetchedData);
            }
            catch (error) {
                app_logs_1.logger.error("Error while fetching userbadge cache data with cursor!", error);
            }
            if (fetchedData.length < 100) {
                break;
            }
        }
        try {
            await new Promise((resolve, reject) => {
                cursor.close((err) => {
                    if (err) {
                        reject(err);
                    }
                    else {
                        resolve();
                    }
                });
            });
            if (!params.trx) {
                trxx?.release();
            }
        }
        catch (error) {
            trxx?.release(error);
            app_logs_1.logger.error(error);
        }
    }
    async getUserOrganizationBadgeRedisPersistentData(params) {
        const userProfileSettings = await dal_manager_1.dbManager.accessOrganization.getOrganizationUserProfileSettingsPg({ organizationId: params.organizationId, trx: params.trx });
        const qBindings = [];
        let qx = 1;
        let qFrom = `
			FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}" u
			INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo
				ON uo."userId" = u.id AND uo."deletedAt" IS NULL AND u."deletedAt" IS NULL
		`;
        const now = new Date();
        const columns = [];
        columns.push("u.id");
        const fieldTypes = new Set(userProfileSettings.captionLineFields.map((c) => c.type));
        for (const field of fieldTypes) {
            switch (field) {
                case dal_constants_1.DalConstants.OrganizationCaptionLineType.profilePhoto:
                    {
                        qFrom += `
							LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" as uop2
							ON uop2."userOrganizationId" = uo.id AND uop2."deletedAt" IS NULL
							`;
                        columns.push("uop2.thumbnail");
                    }
                    break;
                case dal_constants_1.DalConstants.OrganizationCaptionLineType.userOrganizationProfile:
                    {
                        qFrom += `
							LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" as uop 
								ON uop."userOrganizationId" = uo.id AND uop."deletedAt" IS NULL
								`;
                        qFrom += `
							LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorProfiles}" as ovp 
							ON ovp."userOrganizationId" = uo.id AND ovp."deletedAt" IS NULL
					`;
                        columns.push(`
                        JSON_BUILD_OBJECT(
							'userId', uop."userId",
							'uniqueId', uop."uniqueId",
							'name', uop."name",
							'surname', uop."surname",
							'fullname',uop."name" || ' ' || uop."surname",
							'email',uop."email",
							'extensionFields',uop."extensionFields"
                        ) as "profile" `);
                        columns.push(`
							JSON_BUILD_OBJECT(
								'userId', uo."userId",
								'uniqueId', ovp."uniqueId",
								'name', ovp."name",
								'surname', ovp."surname",
								'fullname', COALESCE(ovp."name" || ' ', '')  || ' ' || COALESCE(ovp."surname" || ' ', ''),
								'email', '',
								'extensionFields', ovp."extensionFields"
							) as "visitorProfile" `);
                    }
                    break;
                case dal_constants_1.DalConstants.OrganizationCaptionLineType.userGroup:
                    {
                        qFrom += `
						LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupUserOrganizations}" uguo 
							ON uguo."userOrganizationId" = uo.id AND uguo."deletedAt" IS NULL
						LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroups}" ug
							ON ug.id = uguo."userGroupId" AND ug."deletedAt" IS NULL
						`;
                        columns.push(`
							JSON_BUILD_OBJECT(
								'id', ug."id",
								'name', ug."name",
								'colorCode', ug."colorCode"
							) as "userGroup" `);
                    }
                    break;
                case dal_constants_1.DalConstants.OrganizationCaptionLineType.organizationUnit:
                    {
                        qFrom += `
						LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}" uoou 
							ON uoou."userOrganizationId" = uo.id AND uoou."deletedAt" IS NULL
						LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" ou
							ON ou.id = uoou."organizationUnitId" AND ou."deletedAt" IS NULL
						`;
                        columns.push(`
							JSON_BUILD_OBJECT(
								'id', ou."id",
								'name', ou."name"
							) as "organizationUnit"
						`);
                    }
                    break;
                case dal_constants_1.DalConstants.OrganizationCaptionLineType.credentials:
                    {
                        qFrom += `
							LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials}" uoc 
							ON uoc."userId" = uo."userId"
							AND (uoc."expiresOn" >= $${qx++} OR uoc."expiresOn" IS NULL)
							AND uoc."deletedAt" IS NULL
						`;
                        qBindings.push(now);
                        columns.push(`
							JSON_BUILD_OBJECT(
								'id', uoc."id",
								'type', uoc."type",
								'data', uoc."data",
								'extensionFields',uoc."extensionFields"
							) as "credential"
              		  `);
                    }
                    break;
                case dal_constants_1.DalConstants.OrganizationCaptionLineType.warning:
                    {
                        qFrom += `
						LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationForbiddances}" uof 
							ON uof."userId" = uo."userId" 
							AND (uof."startUtc" <= $${qx}  OR uof."startUtc" IS NULL) 
							AND (uof."endUtc" >= $${qx++} OR uof."endUtc" IS NULL)
							AND uof."deletedAt" IS NULL
						`;
                        qBindings.push(now);
                        columns.push(`
							JSON_BUILD_OBJECT(
								'id', uof."id"
							) as "forbiddance"
                    	`);
                        qFrom += `
							LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorProfiles}" as ovp2
								ON ovp2."userOrganizationId" = uo.id AND ovp2."deletedAt" IS NULL
							`;
                        qFrom += `
						LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorStates}" as ovsf
							ON ovp2."id" = uo.id AND ovsf."organizationVisitorProfileId" IS NULL
							AND (ovsf."startUtc" <= $${qx}  OR ovsf."startUtc" IS NULL) AND (ovsf."endUtc" >= $${qx++} OR ovsf."endUtc" IS NULL)
							AND  ovsf."deletedAt" IS NULL
							AND ovsf.state = $${qx++}
						`;
                        qBindings.push(now, dal_constants_1.DalConstants.OrganizationVisitorStates.Forbidden);
                        columns.push(`
							JSON_BUILD_OBJECT(
                     	   		'id', ovsf."id"
                        	) as "visitorForbiddance"
                    	`);
                        qFrom += `
						LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorStates}" as ovsw
							ON ovp2."id" = uo.id AND ovsw."organizationVisitorProfileId" IS NULL
							AND (ovsw."startUtc" <= $${qx}  OR ovsw."startUtc" IS NULL) AND (ovsw."endUtc" >= $${qx++} OR ovsw."endUtc" IS NULL)
							AND  ovsw."deletedAt" IS NULL
							AND ovsw.state = $${qx++}
						`;
                        qBindings.push(now, dal_constants_1.DalConstants.OrganizationVisitorStates.TemporaryPermit);
                        columns.push(`
							JSON_BUILD_OBJECT(
								'id', ovsw."id"
							) as "visitorPermit"
                   		 `);
                    }
                    break;
                default:
                    break;
            }
        }
        const q = `SELECT ${columns.join(" , ")} ${qFrom} WHERE uo."userId" = $${qx++}`;
        qBindings.push(params.userId);
        const { rows, rowCount } = await (params.trx ?? this._pgPool).query(q, qBindings);
        if (rowCount === 0) {
            return null;
        }
        else {
            let captionResult = rows.filter((r) => r.id === params.userId);
            let captions = [];
            for (let captionItem = 0; captionItem < userProfileSettings.captionLineFields.length; captionItem++) {
                let captionField = userProfileSettings.captionLineFields[captionItem];
                let texts = [];
                let colors = [];
                let profilePhoto = null;
                let translate = false;
                switch (captionField.type) {
                    case dal_constants_1.DalConstants.OrganizationCaptionLineType.profilePhoto:
                        {
                            let thumbnails = captionResult.filter((c) => c.thumbnail).map((c) => c.thumbnail);
                            if (thumbnails.length > 0) {
                                profilePhoto = Buffer.from(thumbnails[0]).toString();
                            }
                        }
                        break;
                    case dal_constants_1.DalConstants.OrganizationCaptionLineType.userOrganizationProfile:
                        let profiles = lodash_1.default.uniqBy(captionResult.map((c) => c.profile).concat(captionResult.map((c) => c.visitorProfile)), function (x) {
                            return x.userId;
                        });
                        for (const profile of profiles) {
                            if (captionField.isExtension) {
                                if (profile.extensionFields) {
                                    let field = profile.extensionFields[captionField.name];
                                    if (field && lodash_1.default.isArray(field)) {
                                        texts.push(field.join("-"));
                                    }
                                    else {
                                        texts.push(field);
                                    }
                                }
                            }
                            else {
                                profile[captionField.name] ? texts.push(profile[captionField.name]) : "";
                            }
                            if (captionField.colorCode) {
                                colors.push(captionField.colorCode);
                            }
                        }
                        break;
                    case dal_constants_1.DalConstants.OrganizationCaptionLineType.userGroup:
                        let userGroups = lodash_1.default.uniqBy(captionResult.map((c) => c.userGroup), function (x) {
                            return x.id;
                        });
                        for (const userGroup of userGroups) {
                            texts.push(userGroup[captionField.name]);
                            if (captionField.colorCode) {
                                colors.push(captionField.colorCode);
                            }
                            else {
                                colors.push(userGroup.colorCode);
                            }
                        }
                        break;
                    case dal_constants_1.DalConstants.OrganizationCaptionLineType.organizationUnit:
                        let organizationUnits = lodash_1.default.uniqBy(captionResult.map((c) => c.organizationUnit), function (x) {
                            return x.id;
                        });
                        for (const organizationUnit of organizationUnits) {
                            texts.push(organizationUnit[captionField.name]);
                            if (captionField.colorCode) {
                                colors.push(captionField.colorCode);
                            }
                        }
                        break;
                    case dal_constants_1.DalConstants.OrganizationCaptionLineType.credentials:
                        let credentials = lodash_1.default.uniqBy(captionResult.map((c) => c.credential), function (x) {
                            return x.id;
                        });
                        for (const credential of credentials) {
                            if (captionField.filterTypes && captionField.filterTypes.length > 0) {
                                if (!captionField.filterTypes.some((a) => parseInt(a) == parseInt(credential["type"])))
                                    continue;
                            }
                            if (captionField.isExtension) {
                                let field = credential.extensionFields ? credential.extensionFields[captionField.name] : null;
                                if (field)
                                    texts.push(field.value);
                            }
                            else {
                                texts.push(credential[captionField.name]);
                            }
                            if (captionField.colorCode) {
                                colors.push(captionField.colorCode);
                            }
                        }
                        break;
                    case dal_constants_1.DalConstants.OrganizationCaptionLineType.warning:
                        {
                            let userForbiddance = lodash_1.default.uniqBy(captionResult.map((c) => c.forbiddance).filter((c) => c.id), function (x) {
                                return x.id;
                            });
                            let userVisitorForbiddance = lodash_1.default.uniqBy(captionResult.map((c) => c.visitorForbiddance).filter((c) => c.id), function (x) {
                                return x.id;
                            });
                            let userVisitorPermit = lodash_1.default.uniqBy(captionResult.map((c) => c.visitorPermit).filter((c) => c.id), function (x) {
                                return x.id;
                            });
                            if (userForbiddance && userForbiddance.length > 0) {
                                texts.push(dal_constants_1.DalConstants.UserCaptionKeys.USER_FORBIDDEN);
                                colors.push(dal_constants_1.DalConstants.UserCaptionColorCode.code);
                                translate = true;
                            }
                            if (userVisitorForbiddance && userVisitorForbiddance.length > 0) {
                                texts.push(dal_constants_1.DalConstants.UserCaptionKeys.VISITOR_FORBIDDEN);
                                colors.push(dal_constants_1.DalConstants.UserCaptionColorCode.code);
                                translate = true;
                            }
                            if (userVisitorPermit && userVisitorPermit.length > 0) {
                                texts.push(dal_constants_1.DalConstants.UserCaptionKeys.VISITOR_PERMITTED);
                                colors.push(dal_constants_1.DalConstants.UserCaptionColorCode.code);
                                translate = true;
                            }
                        }
                        break;
                    default:
                        break;
                }
                if (captionField.type != dal_constants_1.DalConstants.OrganizationCaptionLineType.profilePhoto) {
                    if (texts.length == 0) {
                        captions.push(null);
                    }
                    else {
                        let filteredTexts = lodash_1.default.compact(texts);
                        if (filteredTexts.length == 0) {
                            captions.push(null);
                        }
                        else {
                            if (translate) {
                                captions.push({
                                    text: texts,
                                    colorCode: colors,
                                    translate: true,
                                });
                            }
                            else {
                                captions.push({
                                    text: texts,
                                    colorCode: colors,
                                });
                            }
                        }
                    }
                }
                else {
                    captions.push({
                        avatar: profilePhoto ? profilePhoto : dal_constants_1.DalConstants.DefaultAvatar,
                        text: ["__avatar"],
                    });
                }
            }
            for (let captionIndex = 0; captionIndex < captions.length; ++captionIndex) {
                if (!captions[captionIndex]) {
                    captions[captionIndex] = {
                        text: [""],
                    };
                }
            }
            captions = lodash_1.default.compact(captions);
            return { id: params.userId, caption: captions };
        }
    }
    async getUserOrganizationBadgeRedisPersistentDataBulk(params) {
        if (params.userIds.length === 0) {
            return [];
        }
        const userProfileSettings = await dal_manager_1.dbManager.accessOrganization.getOrganizationUserProfileSettingsPg({
            organizationId: params.organizationId,
            trx: params.trx,
        });
        if (!userProfileSettings || !userProfileSettings.captionLineFields) {
            return params.userIds.map((userId) => ({ id: userId, caption: [] }));
        }
        const qBindings = [];
        let qx = 1;
        let qFrom = `
			FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}" u
			INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo
				ON uo."userId" = u.id AND uo."deletedAt" IS NULL AND u."deletedAt" IS NULL
		`;
        const now = new Date();
        const columns = [];
        columns.push("u.id");
        const fieldTypes = new Set(userProfileSettings.captionLineFields.map((c) => c.type));
        for (const field of fieldTypes) {
            switch (field) {
                case dal_constants_1.DalConstants.OrganizationCaptionLineType.profilePhoto:
                    qFrom += `
						LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" as uop2
						ON uop2."userOrganizationId" = uo.id AND uop2."deletedAt" IS NULL
					`;
                    columns.push("uop2.thumbnail");
                    break;
                case dal_constants_1.DalConstants.OrganizationCaptionLineType.userOrganizationProfile:
                    qFrom += `
						LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" as uop 
							ON uop."userOrganizationId" = uo.id AND uop."deletedAt" IS NULL
					`;
                    qFrom += `
						LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorProfiles}" as ovp 
						ON ovp."userOrganizationId" = uo.id AND ovp."deletedAt" IS NULL
					`;
                    columns.push(`
						JSON_BUILD_OBJECT(
							'userId', uop."userId",
							'uniqueId', uop."uniqueId",
							'name', uop."name",
							'surname', uop."surname",
							'fullname',uop."name" || ' ' || uop."surname",
							'email',uop."email",
							'extensionFields',uop."extensionFields"
						) as "profile" 
					`);
                    columns.push(`
						JSON_BUILD_OBJECT(
							'userId', uo."userId",
							'uniqueId', ovp."uniqueId",
							'name', ovp."name",
							'surname', ovp."surname",
							'fullname', COALESCE(ovp."name" || ' ', '')  || ' ' || COALESCE(ovp."surname" || ' ', ''),
							'email', '',
							'extensionFields', ovp."extensionFields"
						) as "visitorProfile" 
					`);
                    break;
                case dal_constants_1.DalConstants.OrganizationCaptionLineType.userGroup:
                    qFrom += `
						LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupUserOrganizations}" uguo 
							ON uguo."userOrganizationId" = uo.id AND uguo."deletedAt" IS NULL
						LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroups}" ug
							ON ug.id = uguo."userGroupId" AND ug."deletedAt" IS NULL
					`;
                    columns.push(`
						JSON_BUILD_OBJECT(
							'id', ug."id",
							'name', ug."name",
							'colorCode', ug."colorCode"
						) as "userGroup" 
					`);
                    break;
                case dal_constants_1.DalConstants.OrganizationCaptionLineType.organizationUnit:
                    qFrom += `
						LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}" uoou 
							ON uoou."userOrganizationId" = uo.id AND uoou."deletedAt" IS NULL
						LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" ou
							ON ou.id = uoou."organizationUnitId" AND ou."deletedAt" IS NULL
					`;
                    columns.push(`
						JSON_BUILD_OBJECT(
							'id', ou."id",
							'name', ou."name"
						) as "organizationUnit"
					`);
                    break;
                case dal_constants_1.DalConstants.OrganizationCaptionLineType.credentials:
                    qFrom += `
						LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials}" uoc 
						ON uoc."userId" = uo."userId"
						AND (uoc."expiresOn" >= $${qx++} OR uoc."expiresOn" IS NULL)
						AND uoc."deletedAt" IS NULL
					`;
                    qBindings.push(now);
                    columns.push(`
						JSON_BUILD_OBJECT(
							'id', uoc."id",
							'type', uoc."type",
							'data', uoc."data",
							'extensionFields',uoc."extensionFields"
						) as "credential"
					`);
                    break;
                case dal_constants_1.DalConstants.OrganizationCaptionLineType.warning:
                    qFrom += `
						LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationForbiddances}" uof 
							ON uof."userId" = uo."userId" 
							AND (uof."startUtc" <= $${qx}  OR uof."startUtc" IS NULL) 
							AND (uof."endUtc" >= $${qx++} OR uof."endUtc" IS NULL)
							AND uof."deletedAt" IS NULL
					`;
                    qBindings.push(now);
                    columns.push(`
						JSON_BUILD_OBJECT(
							'id', uof."id"
						) as "forbiddance"
					`);
                    qFrom += `
						LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorProfiles}" as ovp2
							ON ovp2."userOrganizationId" = uo.id AND ovp2."deletedAt" IS NULL
					`;
                    qFrom += `
						LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorStates}" as ovsf
							ON ovp2."id" = uo.id AND ovsf."organizationVisitorProfileId" IS NULL
							AND (ovsf."startUtc" <= $${qx}  OR ovsf."startUtc" IS NULL) AND (ovsf."endUtc" >= $${qx++} OR ovsf."endUtc" IS NULL)
							AND ovsf."deletedAt" IS NULL
							AND ovsf.state = $${qx++}
					`;
                    qBindings.push(now, dal_constants_1.DalConstants.OrganizationVisitorStates.Forbidden);
                    columns.push(`
						JSON_BUILD_OBJECT(
							'id', ovsf."id"
						) as "visitorForbiddance"
					`);
                    qFrom += `
						LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorStates}" as ovsw
							ON ovp2."id" = uo.id AND ovsw."organizationVisitorProfileId" IS NULL
							AND (ovsw."startUtc" <= $${qx}  OR ovsw."startUtc" IS NULL) AND (ovsw."endUtc" >= $${qx++} OR ovsw."endUtc" IS NULL)
							AND ovsw."deletedAt" IS NULL
							AND ovsw.state = $${qx++}
					`;
                    qBindings.push(now, dal_constants_1.DalConstants.OrganizationVisitorStates.TemporaryPermit);
                    columns.push(`
						JSON_BUILD_OBJECT(
							'id', ovsw."id"
						) as "visitorPermit"
					`);
                    break;
                default:
                    break;
            }
        }
        const q = `SELECT ${columns.join(" , ")} ${qFrom} WHERE uo."userId" = ANY($${qx++})`;
        qBindings.push(params.userIds);
        const { rows } = await (params.trx ?? this._pgPool).query(q, qBindings);
        const userRowsMap = new Map();
        for (const row of rows) {
            if (!userRowsMap.has(row.id)) {
                userRowsMap.set(row.id, []);
            }
            userRowsMap.get(row.id)?.push(row);
        }
        const result = [];
        for (const userId of params.userIds) {
            const userRows = userRowsMap.get(userId);
            if (!userRows) {
                result.push({ id: userId, caption: [] });
                continue;
            }
            let captions = [];
            for (let captionItem = 0; captionItem < userProfileSettings.captionLineFields.length; captionItem++) {
                let captionField = userProfileSettings.captionLineFields[captionItem];
                let texts = [];
                let colors = [];
                let profilePhoto = null;
                let translate = false;
                switch (captionField.type) {
                    case dal_constants_1.DalConstants.OrganizationCaptionLineType.profilePhoto: {
                        let thumbnails = userRows.filter((c) => c.thumbnail).map((c) => c.thumbnail);
                        if (thumbnails.length > 0) {
                            profilePhoto = Buffer.from(thumbnails[0]).toString();
                        }
                        break;
                    }
                    case dal_constants_1.DalConstants.OrganizationCaptionLineType.userOrganizationProfile: {
                        let profiles = lodash_1.default.uniqBy(userRows.map((c) => c.profile).concat(userRows.map((c) => c.visitorProfile)), (x) => x.userId);
                        for (const profile of profiles) {
                            if (captionField.isExtension) {
                                if (profile.extensionFields) {
                                    let field = profile.extensionFields[captionField.name];
                                    if (field && lodash_1.default.isArray(field)) {
                                        texts.push(field.join("-"));
                                    }
                                    else {
                                        texts.push(field);
                                    }
                                }
                            }
                            else {
                                profile[captionField.name] && texts.push(profile[captionField.name]);
                            }
                            if (captionField.colorCode) {
                                colors.push(captionField.colorCode);
                            }
                        }
                        break;
                    }
                    case dal_constants_1.DalConstants.OrganizationCaptionLineType.userGroup: {
                        let userGroups = lodash_1.default.uniqBy(userRows.map((c) => c.userGroup), (x) => x.id);
                        for (const userGroup of userGroups) {
                            texts.push(userGroup[captionField.name]);
                            colors.push(captionField.colorCode || userGroup.colorCode);
                        }
                        break;
                    }
                    case dal_constants_1.DalConstants.OrganizationCaptionLineType.organizationUnit: {
                        let organizationUnits = lodash_1.default.uniqBy(userRows.map((c) => c.organizationUnit), (x) => x.id);
                        for (const organizationUnit of organizationUnits) {
                            texts.push(organizationUnit[captionField.name]);
                            if (captionField.colorCode) {
                                colors.push(captionField.colorCode);
                            }
                        }
                        break;
                    }
                    case dal_constants_1.DalConstants.OrganizationCaptionLineType.credentials: {
                        let credentials = lodash_1.default.uniqBy(userRows.map((c) => c.credential), (x) => x.id);
                        for (const credential of credentials) {
                            if (captionField.filterTypes && captionField.filterTypes.length > 0) {
                                if (!captionField.filterTypes.some((a) => parseInt(a) == parseInt(credential["type"])))
                                    continue;
                            }
                            if (captionField.isExtension) {
                                let field = credential.extensionFields ? credential.extensionFields[captionField.name] : null;
                                if (field)
                                    texts.push(field.value);
                            }
                            else {
                                texts.push(credential[captionField.name]);
                            }
                            if (captionField.colorCode) {
                                colors.push(captionField.colorCode);
                            }
                        }
                        break;
                    }
                    case dal_constants_1.DalConstants.OrganizationCaptionLineType.warning: {
                        let userForbiddance = lodash_1.default.uniqBy(userRows.map((c) => c.forbiddance).filter((c) => c.id), (x) => x.id);
                        let userVisitorForbiddance = lodash_1.default.uniqBy(userRows.map((c) => c.visitorForbiddance).filter((c) => c.id), (x) => x.id);
                        let userVisitorPermit = lodash_1.default.uniqBy(userRows.map((c) => c.visitorPermit).filter((c) => c.id), (x) => x.id);
                        if (userForbiddance.length > 0) {
                            texts.push(dal_constants_1.DalConstants.UserCaptionKeys.USER_FORBIDDEN);
                            colors.push(dal_constants_1.DalConstants.UserCaptionColorCode.code);
                            translate = true;
                        }
                        if (userVisitorForbiddance.length > 0) {
                            texts.push(dal_constants_1.DalConstants.UserCaptionKeys.VISITOR_FORBIDDEN);
                            colors.push(dal_constants_1.DalConstants.UserCaptionColorCode.code);
                            translate = true;
                        }
                        if (userVisitorPermit.length > 0) {
                            texts.push(dal_constants_1.DalConstants.UserCaptionKeys.VISITOR_PERMITTED);
                            colors.push(dal_constants_1.DalConstants.UserCaptionColorCode.code);
                            translate = true;
                        }
                        break;
                    }
                    default:
                        break;
                }
                if (captionField.type != dal_constants_1.DalConstants.OrganizationCaptionLineType.profilePhoto) {
                    if (texts.length == 0) {
                        captions.push(null);
                    }
                    else {
                        let filteredTexts = lodash_1.default.compact(texts);
                        if (filteredTexts.length == 0) {
                            captions.push(null);
                        }
                        else {
                            captions.push({
                                text: texts,
                                colorCode: colors,
                                ...(translate && { translate: true }),
                            });
                        }
                    }
                }
                else {
                    captions.push({
                        avatar: profilePhoto ? profilePhoto : dal_constants_1.DalConstants.DefaultAvatar,
                        text: ["__avatar"],
                    });
                }
            }
            for (let captionIndex = 0; captionIndex < captions.length; ++captionIndex) {
                if (!captions[captionIndex]) {
                    captions[captionIndex] = { text: [""] };
                }
            }
            captions = lodash_1.default.compact(captions);
            result.push({ id: userId, caption: captions });
        }
        return result;
    }
    async getUserIdOfUsersWhoInRegionDuringTheEmergency(params) {
        const { rows, rowCount } = await (params.trx ?? this._pgPool).query(`
			SELECT aps."userId" FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.antiPassbackStates}" aps
			INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}" u
			ON u.id = aps."userId" AND u."deletedAt" IS NULL
			INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo
			ON u.id = uo."userId" AND uo."deletedAt" IS NULL AND uo."isDisabled" = false
			WHERE aps.state = $1
			AND aps."regionId" = $2
		`, [dal_constants_1.DalConstants.AntiPassbackState.In, params.regionId]);
        return rowCount > 0 ? rows.map((r) => r.userId) : [];
    }
    async getCheckUserRights(params) {
        const { rows, rowCount } = await params.trx.query(`
			SELECT permissions FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo
			INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.roles}" r
			ON uo."roleId" = r.id
			WHERE "userId" = $1
		`, [params.userId]);
        const result = {};
        if (rowCount > 0) {
            const permissions = rows[0].permissions;
            params.filter.permissions.forEach((permission) => {
                result[permission] = permissions.includes(permission);
            });
            return { permissions: result };
        }
        return null;
    }
    async activateUser(params) {
        const { rowCount } = await params.trx.query(`
			UPDATE "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo
			SET "isDisabled" = FALSE
			WHERE "userId" = $1
		`, [params.userId]);
        if (rowCount === 0) {
            (0, dal_access_error_1.throwDbAccessNotFoundError)("Kullanıcı bulunamadı.");
        }
    }
    async getUserWorkPlanAndOrganizationProfilesInfo(params) {
        const organizationId = params.organizationId;
        const { rows, rowCount } = await (params.trx ?? this._pgPool).query(`
			SELECT wp."name" as name,
				   uop."employmentStartUtc" as "employeeWorkingStartTime",
				   uop."employmentEndUtc" as "employeeWorkingEndTime"
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userWorkPlans}" as uwp
			INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.workPlans}" as  wp
				ON uwp."workPlanId" = wp.id
			INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" as  uop
				ON uop."userId" = uwp."userId"
			WHERE uwp."userId" = $1 AND uwp."endDateTime" IS NULL;
			`, [params.userId]);
        const result = rows.map((elem) => {
            return {
                name: elem.name,
                employeeWorkingStartTime: elem.employeeWorkingStartTime ? luxon_1.DateTime.fromISO(elem.employeeWorkingStartTime.toISOString()) : null,
                employeeWorkingEndTime: elem.employeeWorkingEndTime ? luxon_1.DateTime.fromISO(elem.employeeWorkingEndTime.toISOString()) : null,
            };
        });
        return result;
    }
    async getFilteredUserIds(organizationId, params, trx) {
        let result = {
            pagination: {
                total: 0,
                skip: params.pagination?.skip,
                take: params.pagination?.take,
            },
            items: [],
        };
        const selectQuery = `
		SELECT	"uo"."id" as "userOrganizationId", 
				"uo"."userId", 
				"uop"."name", 
				"uop"."surname", 
				"uop"."uniqueId",
				"uop"."manuallyRemainedAnnualPermission" `;
        let baseQuery = `
		 
		FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" as "uo" 
		INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" as "uop" 
			ON "uo"."id" = "uop"."userOrganizationId" 
		WHERE	"uop"."deletedAt" IS NULL
				AND "uo"."deletedAt" IS NULL
		`;
        if (params.userOrganizationStatus && params.userOrganizationStatus !== dal_constants_1.DalConstants.IdentityStatusType.All) {
            baseQuery += `
				AND "uo"."isDisabled" = ${params.userOrganizationStatus === dal_constants_1.DalConstants.IdentityStatusType.Disabled}
			`;
        }
        let orQueryArray = [];
        let bindingKeys = [];
        let queryParams = 1;
        if (params.organizationUnitIds || params.userGroupIds || params.userIds) {
            if (params.organizationUnitIds && params.organizationUnitIds.length > 0) {
                if (params.applyOrganizationUnitFilterHierarchically === true) {
                    params.organizationUnitIds = (await trx.query(`
						SELECT	"id" 
						FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" 
						WHERE "deletedAt" is null 
							   AND ("id" = ANY($1::uuid[]) OR "ancestorIds" SIMILAR TO '%(${params.organizationUnitIds.join("|")})%')
						`, [params.organizationUnitIds])).rows.map((elem) => elem.id);
                }
                orQueryArray.push(`
				"uo"."id" IN (SELECT "userOrganizationId" 
							  FROM   "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}" 
							  WHERE "organizationUnitId" = ANY($${queryParams++}::uuid[]) AND "deletedAt" IS NULL)`);
                bindingKeys.push(params.organizationUnitIds);
            }
            if (params.userIds && params.userIds.length > 0) {
                orQueryArray.push(`
					"uop"."userId"=ANY($${queryParams++}::uuid[])
					`);
                bindingKeys.push(params.userIds);
            }
            if (params.userGroupIds && params.userGroupIds.length > 0) {
                orQueryArray.push(`
					"uo"."id" IN (SELECT "userOrganizationId" 
								  FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupUserOrganizations}" 
								  WHERE "userGroupId" = ANY($${queryParams++}) and "deletedAt" IS NULL)
					`);
                bindingKeys.push(params.userGroupIds);
            }
        }
        if (orQueryArray.length > 0) {
            const generatedOrCondition = orQueryArray.join(" OR ");
            baseQuery += "AND ( " + generatedOrCondition + " )";
        }
        result.pagination.total = parseInt((await trx.query(`SELECT COUNT(uo.id) ` + baseQuery, bindingKeys)).rows[0].count);
        if (!params.pagination) {
            params.pagination = {
                skip: 0,
                take: 100,
            };
            result.pagination.skip = params.pagination.skip;
            result.pagination.take = params.pagination.take;
        }
        baseQuery += ` ORDER BY "uop"."name" ASC,
								"uop"."surname" ASC
					   OFFSET ${params.pagination.skip}
					   LIMIT ${params.pagination.take}`;
        result.items = (await trx.query(selectQuery + baseQuery, bindingKeys)).rows.map((elem) => {
            let fullname = [elem.name.trim(), (elem.surname || "").trim()].join(" ");
            return {
                id: elem.userId,
                fullname: fullname,
                uniqueId: elem.uniqueId,
                captionLines: [fullname, elem.uniqueId],
                manuallyRemainedAnnualPermission: elem.manuallyRemainedAnnualPermission,
            };
        });
        return result;
    }
    async getUsersOrganizationUnits(organizationId, userId, trx) {
        const usersOrganizationUnitIds = (await trx.query(`
				SELECT	ou.id,
						ou."ancestorIds",
						ou."managerUserId"
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" as uo
				INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}" as uoou
					ON uoou."userOrganizationId" = uo.id
				INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" as ou
					ON ou.id = uoou."organizationUnitId"
				WHERE uo."userId" = $1 AND uoou."deletedAt" IS NULL AND ou."deletedAt" IS NULL
		`, [userId])).rows;
        return usersOrganizationUnitIds;
    }
    async getUserEmailAndPhoneNumberInfo(organizationId, userId) {
        const mailAndPhoneNumbers = (await this._pgPool.query(`
				SELECT	(CASE WHEN uop."email" IS NULL THEN up.email ELSE uop.email END) as email,
						(CASE WHEN uop."phoneNumber" IS NULL THEN up."phoneNumber" ELSE uop."phoneNumber" END) as "phoneNumber"
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}" u
				INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo 
					ON uo."userId" = u."id" 
					AND u."deletedAt" IS NULL 
					AND uo."deletedAt" IS NULL
					AND u."id" = $1
				LEFT JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userProfiles}" up 
					ON uo."userId" = up."userId" 
					AND up."deletedAt" IS NULL 
					AND uo."deletedAt" IS NULL                
				LEFT JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" uop 
					ON uop."userOrganizationId" = uo."id" 
					AND uop."deletedAt" IS NULL
			`, [userId])).rows[0];
        return mailAndPhoneNumbers;
    }
    async enableAllUsersMultiFactorAuthSettings(organizationId, mfaSettings, userCustomNotificationMediumSettings, trx) {
        await trx.query(`
			UPDATE "${organizationId}"."userProfiles"
			SET "mfaSettings" = $1;
			`, [mfaSettings]);
        await trx.query(`
			UPDATE "${organizationId}"."userOrganizations"
			SET settings = 
			CASE 
				WHEN settings->'notification'->'mediumSettings'->>'custom' IS NOT NULL
				THEN jsonb_set(settings, '{notification,mediumSettings,custom}',
					(settings->'notification'->'mediumSettings'->'custom')::jsonb || $1::jsonb   )
				ELSE jsonb_set(settings, '{notification,mediumSettings,custom}', $1::jsonb)
			END;
		`, [JSON.stringify(userCustomNotificationMediumSettings)]);
    }
    async upsertUserOrganizationMulitFactorAuthNotificationSettings(params, trx) {
        if (params.userCustomNotificationMediumSettings) {
            await trx.query(`
				UPDATE "${params.organizationId}"."userOrganizations"
				SET settings = 
				CASE 
					WHEN settings->'notification'->'mediumSettings'->>'custom' IS NOT NULL
					THEN jsonb_set(settings, '{notification,mediumSettings,custom}',
						(settings->'notification'->'mediumSettings'->'custom')::jsonb || $1::jsonb )
					ELSE jsonb_set(settings, '{notification,mediumSettings,custom}', $1::jsonb)
				END
				WHERE "userId" = $2;
			`, [JSON.stringify(params.userCustomNotificationMediumSettings), params.userId]);
        }
        else {
            await trx.query(`
				UPDATE "${params.organizationId}"."userOrganizations"
				SET settings =
				CASE
					WHEN settings->'notification'->'mediumSettings'->>'custom' IS NOT NULL
					THEN jsonb_set(settings, '{notification,mediumSettings,custom}',
						(settings->'notification'->'mediumSettings'->'custom')::jsonb - $1)
					ELSE settings
				END;
				`, [restapi_1.NotificationType.SendingVerificationToken]);
        }
    }
    async disableAllUsersMultFactorAuthSettings(organizationId, trx) {
        await trx.query(`
			UPDATE "${organizationId}"."userProfiles"
			SET "mfaSettings" = '{"enabledStatus":false}';
			`);
        await trx.query(`
			UPDATE "${organizationId}"."userOrganizations"
			SET settings =
			CASE
				WHEN settings->'notification'->'mediumSettings'->>'custom' IS NOT NULL
				THEN jsonb_set(settings, '{notification,mediumSettings,custom}',
					(settings->'notification'->'mediumSettings'->'custom')::jsonb - $1)
				ELSE settings
	   		END;
			`, [restapi_1.NotificationType.SendingVerificationToken]);
    }
}
exports.PSQLDalAccessUser = PSQLDalAccessUser;
