"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PSQLDalAccessSystem = void 0;
const moment_1 = __importDefault(require("moment"));
const uuid_1 = __importDefault(require("uuid"));
const dal_constants_1 = require("../../dal.constants");
const dal_db_armon_schema_1 = require("../../db/armon/dal.db.armon.schema");
const dal_access_error_1 = require("../dal.access.error");
const dal_access_rdb_system_1 = require("../rdb/dal.access.rdb.system");
const dal_manager_1 = require("../../dal.manager");
const app_enums_1 = require("../../../app.enums");
const dal_utils_1 = require("../../dal.utils");
const app_config_1 = require("../../../app.config");
const Cursor = require("pg-cursor");
class PSQLDalAccessSystem extends dal_access_rdb_system_1.RDBDalAccessSystem {
    constructor(knex, pgPool) {
        super(knex, pgPool);
    }
    async getOrganizationDefinitionFileContentV3(organizationId) {
        let organization = await this.dbClient
            .withSchema(organizationId)
            .from("organizations")
            .where("id", organizationId)
            .whereNull("deletedAt")
            .first("id", "name", "publicKey", "privateKey");
        if (!organization) {
            (0, dal_access_error_1.throwDbAccessNotFoundError)("Organization is not found");
        }
        let content = {
            id: organizationId,
            name: organization.name,
            publicKey: organization.publicKey,
            hostType: app_config_1.appConfig.isCloudServer ? 1 : 2,
            domainNames: [],
        };
        let result = {
            organizationPrivateKey: organization.privateKey,
            content: content,
        };
        return result;
    }
    async getSystemHealth(organizationId) {
        let emergencyState = await this.dbClient
            .withSchema(organizationId)
            .from("organizations")
            .where("id", organizationId)
            .first("emergencyState")
            .then((or) => (or ? or.emergencyState : dal_constants_1.DalConstants.EmergencyState.Normal));
        let armonControlPanelESeries = (await this.dbClient.withSchema(organizationId).from(dal_db_armon_schema_1.ArmonSchema.tableNames.controlPanelESeriesSettings).select("deviceId")).map((e) => e.deviceId);
        let devices = [];
        await this.dbClient
            .withSchema(organizationId)
            .from("devices")
            .whereNull("deletedAt")
            .where("organizationId", organizationId)
            .whereNotIn("brand", [dal_constants_1.DalConstants.DeviceBrand.HikVision, dal_constants_1.DalConstants.DeviceBrand.Impinj])
            .whereNotIn("id", armonControlPanelESeries)
            .select()
            .then((rows) => {
            if (rows) {
                for (const row of rows) {
                    let item = {
                        batteryLevel: null,
                        brand: row.brand,
                        model: row.model,
                        caption: dal_constants_1.DalConstants.DeviceBrand[row.brand] + " " + row.model,
                        freeSpace: null,
                        totalSpace: null,
                        emergencyState: null,
                        id: row.id,
                        onBattery: null,
                        ip: row.ip,
                        name: row.name,
                        isRealTimeClockBatteryOk: null,
                        temperature: null,
                        isTemperatureCritical: null,
                        lastSyncISO: row.lastSyncUtc,
                        location: row.location,
                    };
                    devices.push(item);
                }
            }
        });
        return Promise.resolve({
            devices: devices,
            emergencyState: emergencyState,
        });
    }
    async insertPredefinedJobs(organizationId, jobs, jobParams, trx) {
        let qbInsertJobs = this.dbClient.withSchema("public").table(dal_db_armon_schema_1.ArmonSchema.tableNames.scheduled_job).insert(jobs);
        if (trx) {
            qbInsertJobs.transacting(trx);
        }
        await qbInsertJobs;
        if (jobParams) {
            await this.dbClient
                .withSchema(organizationId)
                .table("organizations")
                .where("id", organizationId)
                .transacting(trx)
                .update({
                scheduledJobParameters: JSON.stringify(jobParams),
            });
        }
    }
    async disableScheduledJob(organizationId, jobType, trx) {
        let qb = this.dbClient.withSchema("public").table(dal_db_armon_schema_1.ArmonSchema.tableNames.scheduled_job).where("organizationId", organizationId).where("type", jobType).del();
        if (trx) {
            qb.transacting(trx);
        }
        await qb;
    }
    async checkScheduledJobByType(jobType) {
        let job = await this.dbClient.withSchema("public").table(dal_db_armon_schema_1.ArmonSchema.tableNames.scheduled_job).where("type", jobType).first("id");
        return Promise.resolve(job);
    }
    async upsertScheduledJobHistory(historyLog, trx) {
        let id = historyLog.id || uuid_1.default.v4();
        if (!historyLog.id) {
            await trx.query(`
				INSERT INTO public."scheduledJobHistoryLogs"
				(id,"jobId",name,"startDate","executionDate","endDate",message,success,"tryCount")
				VALUES ($1,$2,$3,$4,$5,$6,$7,$8,$9)
			`, [id, historyLog.jobId, historyLog.name, historyLog.startDate, historyLog.executionDate, historyLog.endDate, historyLog.message, historyLog.success, historyLog.tryCount]);
        }
        else {
            await trx.query(`
				UPDATE public."scheduledJobHistoryLogs"
				SET "jobId" = $1,name = $2,"startDate" = $3,"executionDate" = $4,"endDate" = $5,
				message = $6,success = $7,"tryCount" = $8
				WHERE id = $9
			`, [historyLog.jobId, historyLog.name, historyLog.startDate, historyLog.executionDate, historyLog.endDate, historyLog.message, historyLog.success, historyLog.tryCount, id]);
        }
        return id;
    }
    async getExistingScheduledJobLog(jobId, date) {
        let id = "";
        await this.dbClient
            .withSchema("public")
            .table("scheduledJobHistoryLogs")
            .where("jobId", jobId)
            .where("executionDate", "<=", (0, moment_1.default)(date).endOf("day").toDate())
            .where("executionDate", ">=", (0, moment_1.default)(date).startOf("day").toDate())
            .first("id")
            .then((row) => {
            if (row)
                id = row.id;
        });
        return Promise.resolve(id);
    }
    async updateScheduledJobHistoryFinalStatus(historyLogId, endDateTime, success, message, trx) {
        await trx.query(`
		UPDATE public."scheduledJobHistoryLogs"
		SET "endDate" = $1, message = $2, success = $3
		WHERE id = $4
		`, [endDateTime, message, success, historyLogId]);
    }
    async updateScheduledJobHistoryStatus(historyLogId, tryCount, trx) {
        await trx.query(`
		UPDATE public."scheduledJobHistoryLogs"
		SET "tryCount" = $1
		WHERE id = $2
		`, [tryCount, historyLogId]);
    }
    async listFailedScheduledJobs(tryCount) {
        let failedJobs = [];
        await this.dbClient
            .withSchema("public")
            .table("scheduledJobHistoryLogs")
            .where("tryCount", "<=", tryCount)
            .where("success", false)
            .select()
            .then((rows) => {
            if (rows) {
                for (let row of rows) {
                    failedJobs.push({
                        historyLogId: row.id,
                        jobId: row.jobId,
                        executionDate: row.executionDate,
                        tryCount: row.tryCount,
                    });
                }
            }
        });
        return Promise.resolve(failedJobs);
    }
    async clearOldScheduledJobHistoryLogs(date) {
        await this.dbClient.withSchema("public").table("scheduledJobHistoryLogs").where("startDate", "<=", date).del();
    }
    async getNumberOfOrganizations() {
        return this.dbClient
            .withSchema("public")
            .from(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationList)
            .count()
            .then((rows) => {
            return rows[0].count;
        });
    }
    async addOrganizationFeedback(organizationId, args, requestUserId) {
        return await dal_manager_1.dbManager.pgTransactionMainDb(async (trx) => {
            let caption = 12345;
            let id = uuid_1.default.v4();
            caption += (await trx.query(`SELECT count(*) FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationFeedbacks}"`)).rows[0].count;
            await trx.query(`INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationFeedbacks}"
            (id, "organizationId", "createdBy", "createdAt", caption, status, type) VALUES (
                $1, $2, $3, $4, $5, $6, $7
            )`, [id, organizationId, requestUserId, new Date(), caption, dal_constants_1.DalConstants.OrganizationFeedbackStatus.New, args.type]);
            await dal_manager_1.dbManager.accessLog.addOrganizationFeedback({
                oId: organizationId,
                o: requestUserId,
                tid: id,
                b: args.body,
                t: args.title,
                or: args.origin,
                c: caption,
                dt: args.data ? args.data : null,
                s: args.screenshot ? args.screenshot : null,
                u: new Date(),
            });
            return id;
        });
    }
    async updateOrganizationFeedback(organizationId, feedbackId, args, requestUserId) {
        return await this.dbClient.transaction(async (trx) => {
            await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationFeedbacks).where("id", feedbackId).update({
                organizationId: organizationId,
                status: args.status,
                type: args.type,
            });
            await dal_manager_1.dbManager.accessLog.addOrganizationFeedbackUpdate({
                oId: organizationId,
                o: requestUserId,
                tid: feedbackId,
                n: args.note,
                s: args.status,
                t: args.type,
                u: new Date(),
            });
        });
    }
    async getOrganizationFeedback(organizationId, feedbackId) {
        let feedbackLastStatus = await this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationFeedbacks).where("id", feedbackId).first();
        let feedbackData = await dal_manager_1.dbManager.accessLog.listOrganizationFeedback(organizationId, [feedbackId]);
        if (!feedbackData || feedbackData.length > 1) {
            Promise.reject("invalid operation");
        }
        let feedbackHistory = await dal_manager_1.dbManager.accessLog.listOrganizationFeedbackUpdates(organizationId, [feedbackId]);
        let updateUserIds = feedbackHistory.map((h) => h.o);
        let updateUsers = [];
        if (updateUserIds && updateUserIds.length > 0) {
            updateUsers = await dal_manager_1.dbManager.accessPacs.getBasicUserInfoList(organizationId, updateUserIds);
        }
        let feedback = {
            id: feedbackId,
            title: feedbackData[0].t,
            body: feedbackData[0].b,
            origin: feedbackData[0].or,
            caption: feedbackData[0].c,
            status: feedbackLastStatus.status,
            type: feedbackLastStatus.type,
            screenshot: feedbackData[0].s,
            data: feedbackData[0].dt,
            createdAt: feedbackData[0].u,
            requestUser: await dal_manager_1.dbManager.accessUser.getBasicUserInfo(organizationId, feedbackData[0].o),
            requestUserId: feedbackData[0].o,
            history: feedbackHistory.map((r) => {
                return {
                    id: r.id,
                    type: r.t,
                    status: r.s,
                    note: r.n,
                    updatedAt: r.u,
                    updatedUser: updateUsers.find((u) => u.id === r.o),
                };
            }),
        };
        return Promise.resolve(feedback);
    }
    async listOrganizationFeedback(organizationId, args) {
        let result = {
            pagination: {
                take: args.pagination.take,
                skip: args.pagination.skip,
                total: 0,
            },
            items: [],
        };
        let qb = this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationFeedbacks)
            .where("createdAt", "<", args.dateRange.endDateTime)
            .where("createdAt", ">", args.dateRange.startDateTime);
        if (args.requestUserId)
            qb.where("createdBy", args.requestUserId);
        if (args.status)
            qb.where("status", args.status);
        if (args.type)
            qb.where("type", args.type);
        if (args.caption)
            qb.where("caption", args.caption);
        result.pagination.total = parseInt((await qb
            .clone()
            .count()
            .then((result) => result[0].count)));
        if (args.pagination.skip) {
            qb.offset(args.pagination.skip);
        }
        if (args.pagination.take) {
            qb.limit(args.pagination.take);
        }
        let feedbackStatus = await qb.select().orderBy("createdAt", "desc");
        let feedBackDataList = await dal_manager_1.dbManager.accessLog.listOrganizationFeedback(organizationId, feedbackStatus.map((f) => f.id));
        let feedBackUpdates = await dal_manager_1.dbManager.accessLog.listOrganizationFeedbackUpdates(organizationId, feedbackStatus.map((f) => f.id));
        let updateUserIds = feedBackUpdates.map((h) => h.o);
        let updateUsers = [];
        if (updateUserIds && updateUserIds.length > 0) {
            updateUsers = await dal_manager_1.dbManager.accessPacs.getBasicUserInfoList(organizationId, updateUserIds);
        }
        for (const feedbackCurrentData of feedbackStatus) {
            let feedBackData = feedBackDataList.find((f) => f.tid === feedbackCurrentData.id);
            if (!feedBackData) {
                continue;
            }
            let feedback = {
                id: feedbackCurrentData.id,
                title: feedBackData.t,
                body: feedBackData.b,
                origin: feedBackData.or,
                caption: feedBackData.c,
                status: feedbackCurrentData.status,
                type: feedbackCurrentData.type,
                screenshot: feedBackData.s,
                data: feedBackData.dt,
                createdAt: feedBackData.u,
                requestUser: await dal_manager_1.dbManager.accessUser.getBasicUserInfo(organizationId, feedBackData.o),
                requestUserId: feedBackData.o,
                history: feedBackUpdates
                    .filter((f) => f.tid === feedbackCurrentData.id)
                    .map((r) => {
                    return {
                        id: r.id,
                        type: r.t,
                        status: r.s,
                        note: r.n,
                        updatedAt: r.u,
                        updatedUser: updateUsers.find((u) => u.id === r.o),
                    };
                }),
            };
            result.items.push(feedback);
        }
        result.items.sort((x, y) => (x.caption < y.caption ? -1 : 1));
        return Promise.resolve(result);
    }
    async getRemoteAccessCommandStructure(organizationId, userId, accessControlPointId, qrCode) {
        let q = `
			SELECT uar.id, acp."deviceId", acp.name, acp.location
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userAccessRights}" AS uar
			INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessControlPoints}" AS acp ON acp.id = uar."accessControlPointId"
			WHERE uar."deletedAt" IS NULL AND acp."deletedAt" IS NULL AND
				uar."userId" = $1 AND
				uar."accessControlPointId" = $2 
		`;
        if (qrCode) {
            q += ` AND uar."access" = TRUE`;
        }
        else {
            q += ` AND uar."remoteAccess" = TRUE AND acp."remoteAvailable" = TRUE`;
        }
        const accessRights = await this._pgPool.query(q, [userId, accessControlPointId]);
        if (!accessRights || !accessRights.rowCount) {
            return Promise.resolve(null);
        }
        let user = await dal_manager_1.dbManager.accessUser.getBasicUserInfo(organizationId, userId);
        if (!user || !user.id) {
            return Promise.resolve(null);
        }
        const accessRight = accessRights.rows[0];
        return Promise.resolve({
            userAccessRightId: accessRight.id,
            userId: userId,
            accessControlPointName: accessRight.name,
            accessControlPointId: accessControlPointId,
            userFullname: user.fullname,
            deviceId: accessRight.deviceId,
        });
    }
    async getOrganizationFloorPlanById(organizationId, floorPlanId) {
        let result = await this.dbClient.withSchema(organizationId).from(dal_db_armon_schema_1.ArmonSchema.tableNames.floorPlans).where("id", floorPlanId).whereNull("deletedAt").first("id", "svg");
        return Promise.resolve(result);
    }
    async getOrganizationFloorPlanList(organizationId, request) {
        let result = {
            pagination: {
                take: request.pagination.take,
                skip: request.pagination.skip,
                total: 0,
            },
            items: [],
        };
        let qb = this.dbClient.withSchema(organizationId).from(dal_db_armon_schema_1.ArmonSchema.tableNames.floorPlans).whereNull("deletedAt").where("organizationId", organizationId);
        if (request.name) {
            qb.where("name", "ilike", "%" + request.name + "%");
        }
        result.pagination.total = parseInt(await qb
            .clone()
            .count()
            .then((result) => result[0].count));
        if (result.pagination.total === 0) {
            return Promise.resolve(result);
        }
        if (request.pagination.skip) {
            qb.offset(request.pagination.skip);
        }
        if (request.pagination.take) {
            qb.limit(request.pagination.take);
        }
        await qb
            .select("id", "name", "createdAt", "updatedAt")
            .orderBy("name")
            .then((rows) => {
            for (let row of rows) {
                result.items.push({
                    id: row.id,
                    name: row.name,
                    createdAt: row.createdAt,
                    updatedAt: row.updatedAt,
                });
            }
        });
        return Promise.resolve(result);
    }
    async setScheduleJobNextExecutionTime(id, nextExecutionDate) {
        await this._pgPool.query(`UPDATE public."${dal_db_armon_schema_1.ArmonSchema.tableNames.scheduled_job}"
            SET "nextExecutionDate" = $2    
        WHERE id = $1 
            `, [id, nextExecutionDate]);
    }
    async listScheduledJobs(params) {
        const { filter, pagination, sorting, trx, organizationId } = params;
        let qx = 1;
        const qb = [];
        let qfrom = `                        
                FROM public."${dal_db_armon_schema_1.ArmonSchema.tableNames.scheduled_job}" AS sj
                LEFT OUTER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.reportScheduledJobs}" rsj ON sj.id = rsj.id
                LEFT OUTER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.pacsNotificationScheduledJobs}" pnsj ON sj.id = pnsj.id
				LEFT OUTER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.notification}" n ON n.id = sj."notificationId" 
				LEFT OUTER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.reportTemplates}" rt ON rsj."reportTemplateId" = rt.id
				LEFT OUTER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.customReports}" cr ON cr.id = rt."customReportId"
				LEFT OUTER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilter}" sef ON sef.id = pnsj."subjectedEmployeesFilterId"
				`;
        const qw = [`sj."organizationId" = $${qx++}`];
        qb.push(organizationId);
        if (filter?.types?.length > 0) {
            const qt = [];
            for (const t of filter.types) {
                qt.push(`$${qx++}`);
                qb.push(t);
            }
            qw.push(`sj.type IN (${qt.join(",")}::integer)`);
        }
        if (filter?.executionInRange) {
            if (filter.executionInRange.start) {
                qw.push(`sj."nextExecutionDate" >= $${qx++}`);
                qb.push(filter.executionInRange.start);
            }
            if (filter.executionInRange.end) {
                qw.push(`sj."nextExecutionDate" <= $${qx++}`);
                qb.push(filter.executionInRange.end);
            }
        }
        if (filter?.enabled !== undefined && filter?.enabled !== null) {
            qw.push(`sj.enabled = $${qx++}`);
            qb.push(filter.enabled);
        }
        if (filter?.createdBySystem) {
            qw.push(`sj."createdByUserId" IS NULL`);
        }
        else if (!filter?.createdBySystem && !filter?.createdByUserId) {
            if (params.userIdsUnderHierarchy?.length) {
                qw.push(`sj."createdByUserId" = ANY($${qx++}::UUID[])`);
                qb.push(params.userIdsUnderHierarchy);
            }
            else {
                qw.push(`sj."createdByUserId" IS NOT NULL`);
            }
        }
        else if (filter?.createdByUserId) {
            qw.push(`sj."createdByUserId" = $${qx++}`);
            qb.push(filter?.createdByUserId);
        }
        if (filter?.editable !== undefined && filter?.editable !== null) {
            const qt = [];
            for (const t of dal_constants_1.DalConstants.editableScheduledJobs) {
                qt.push(`$${qx++}`);
                qb.push(t);
            }
            qw.push(`sj.type ${filter.editable ? "" : "NOT"} IN (${qt.join(",")}::integer)`);
        }
        let qws = "";
        if (qw.length > 0) {
            qws =
                `
				WHERE ` + qw.join(" AND ");
        }
        const qtotal = `SELECT COUNT(*) AS C ` + qfrom + qws;
        const totalResult = await trx.query(qtotal, qb);
        const result = {
            pagination: {
                total: totalResult.rowCount > 0 ? parseInt(totalResult.rows[0].c) : 0,
                skip: pagination.skip || 0,
                take: pagination.take || 100,
            },
            items: [],
        };
        if (result.pagination.total === 0) {
            return result;
        }
        qfrom += `
		LEFT JOIN "${organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilter} AS f
		ON f.id = n."receiverFilterId"
		`;
        let q = qfrom + qws;
        const sortField = params.sorting?.field && params.sorting.field === "nextExecutionDate" ? `sj."nextExecutionDate"` : params.sorting?.field === "status" ? `sj.enabled` : `sj.type`;
        const sortOrder = ["ASC", "DESC"].includes(params.sorting?.order) ? params.sorting?.order : "DESC";
        q += `
			ORDER BY ${sortField} ${sortOrder}`;
        if (pagination?.skip > 0) {
            q += `
                OFFSET $${qx++}`;
            qb.push(pagination.skip);
        }
        q += `
                LIMIT $${qx++}`;
        qb.push(pagination.take || 100);
        q =
            `
			SELECT	sj.id, 
					sj.enabled,
					sj.type,
					sj."createdByUserId",
					sj.interval, 
					sj."firstExecutionDate", 
					sj."nextExecutionDate", 
					sj.note, 
					sj."notificationId", 
					rsj."reportTemplateId", 
					rsj."sendWhenEmpty", 
					rsj.format,
					CASE WHEN sef.id IS NOT NULL THEN jsonb_build_object('id', sef.id, 'name', sef.name) END  as "subjectedEmployeesFilter",
					CASE WHEN f.id IS NOT NULL THEN jsonb_build_object('id', f.id, 'name', f.name) END  as "filter",
					n.email, 
					n.sms, 
					n.web, 
					n."pushNotification", 
					rt.name as "reportTemplateName", 
					rt.type as "reportType", 
					cr.id as "customReportId"
		` + q;
        const { rows } = await trx.query(q, qb);
        for (const row of rows) {
            const createdBy = row.createdByUserId
                ? {
                    id: row.createdByUserId,
                    caption: (await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId, userId: row.createdByUserId })).caption,
                }
                : undefined;
            const notification = row.filter
                ? {
                    receiverFilter: row.filter,
                    mediums: {
                        email: row.email,
                        pushNotification: row.pushNotification,
                        sms: row.sms,
                        web: row.web,
                    },
                }
                : null;
            result.items.push({
                createdT: row.createdT,
                enabled: row.enabled,
                firstExecutionDate: row.firstExecutionDate,
                id: row.id,
                interval: row.interval,
                nextExecutionDate: row.enabled ? (0, dal_utils_1.calculateNextExecutionDateOfScheduledJob)(row.interval, row.firstExecutionDate) : row.nextExecutionDate,
                type: row.type,
                createdBy,
                note: row.note,
                notificationId: row.notificationId,
                settings: row.type === 19
                    ? {
                        format: row.format,
                        sendWhenEmpty: row.sendWhenEmpty,
                        templateId: row.reportTemplateId,
                        templateName: row.reportTemplateName ?? undefined,
                        reportType: row.reportType,
                        notification,
                        customReportId: row.customReportId ?? undefined,
                    }
                    : {
                        notification,
                        subjectedEmployees: row.subjectedEmployeesFilter,
                    },
            });
        }
        return result;
    }
    async addScheduledJob(params) {
        let notificationId = null;
        const createdT = new Date();
        if (params.settings?.notification?.receiverFilterId ||
            params.type === 31 ||
            params.type === 49 ||
            params.type === 32) {
            let receiverFilterId = null;
            if (params.type === 31 ||
                params.type === 49 ||
                params.type === 32) {
                receiverFilterId = params.settings.subjectedEmployeesFilterId;
            }
            else {
                receiverFilterId = params.settings?.notification?.receiverFilterId;
            }
            notificationId = uuid_1.default.v4();
            let instanceContext = null;
            await params.trx.query(`INSERT INTO "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.notification}"
					(id, "createdT", type, "createdByUserId", "receiverFilterId", email, sms, "pushNotification", web, "settings")
					VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10)`, [
                notificationId,
                createdT,
                params.type,
                params.userId,
                receiverFilterId,
                params.settings.notification.mediums.email || false,
                params.settings.notification.mediums.sms || false,
                params.settings.notification.mediums.pushNotification || false,
                params.settings.notification.mediums.web || false,
                instanceContext,
            ]);
        }
        const id = params.id || uuid_1.default.v4();
        await params.trx.query(`INSERT INTO public."${dal_db_armon_schema_1.ArmonSchema.tableNames.scheduled_job}" 
			(id, "createdT", "organizationId", type, enabled, "createdByUserId", interval, 
			"firstExecutionDate", "nextExecutionDate", note, "notificationId")
            VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11)`, [id, createdT, params.organizationId, params.type, true, params.userId, params.interval, params.firstExecutionDate, params.firstExecutionDate, params.note, notificationId]);
        if (params.type === 19) {
            const settings = params.settings;
            await params.trx.query(`INSERT INTO "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.reportScheduledJobs}" 
			(id, "reportTemplateId", format, "sendWhenEmpty")
			VALUES ($1, $2, $3, $4 )`, [id, settings.templateId, settings.format, settings.sendWhenEmpty]);
        }
        else if ([32, 31, 49].includes(params.type)) {
            const settings = params.settings;
            await params.trx.query(`
				INSERT INTO "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.pacsNotificationScheduledJobs}" 
				(id, "subjectedEmployeesFilterId")
				VALUES ($1, $2)`, [id, settings.subjectedEmployeesFilterId]);
        }
        const result = {
            createdT,
            enabled: true,
            firstExecutionDate: new Date(params.firstExecutionDate),
            id,
            interval: params.interval,
            nextExecutionDate: new Date(params.nextExecutionDate) || new Date(params.firstExecutionDate),
            settings: params.settings,
            type: params.type,
            createdBy: params.userId
                ? {
                    id: params.userId,
                    caption: (await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId: params.organizationId, userId: params.userId })).caption,
                }
                : null,
            notificationId: notificationId,
        };
        return result;
    }
    async deleteScheduledJob(params) {
        const { rows, rowCount } = await params.trx.query(`SELECT "organizationId", "notificationId" FROM "public".${dal_db_armon_schema_1.ArmonSchema.tableNames.scheduled_job}
		 WHERE id = $1`, [params.id]);
        if (rowCount > 0 && rows[0].notificationId) {
            const row = rows[0];
            await dal_manager_1.dbManager.accessNotifications.deleteNotification({ id: row.notificationId, organizationId: row.organizationId, trx: params.trx });
        }
        await params.trx.query(`DELETE FROM public."${dal_db_armon_schema_1.ArmonSchema.tableNames.scheduled_job}" WHERE id = $1`, [params.id]);
    }
    async deleteScheduledJobsOfDisabledUser(params) {
        const { rows, rowCount } = await params.trx.query(`SELECT "notificationId" FROM "public".${dal_db_armon_schema_1.ArmonSchema.tableNames.scheduled_job}
		 WHERE "createdByUserId" = $1 AND "organizationId" = $2`, [params.userId, params.organizationId]);
        if (rowCount > 0) {
            await params.trx.query(`
				DELETE FROM "${params.organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.notification}
				WHERE id = ANY($1::uuid[])
			`, [rows.filter((row) => row.notificationId).map((r) => r.notificationId)]);
        }
        await params.trx.query(`DELETE FROM public."${dal_db_armon_schema_1.ArmonSchema.tableNames.scheduled_job}" WHERE "createdByUserId" = $1 AND "organizationId" = $2`, [params.userId, params.organizationId]);
    }
    async setNextExecutionDateOfScheduledJob(params) {
        await params.trx.query(`
		UPDATE public."${dal_db_armon_schema_1.ArmonSchema.tableNames.scheduled_job}"
		SET "nextExecutionDate" = $1 WHERE id = $2
	`, [params.nextExecutionDate, params.id]);
    }
    async updateScheduledJob(params) {
        const sj = await params.trx.query(`SELECT type, "interval", "firstExecutionDate", "nextExecutionDate", enabled, "notificationId" FROM public."${dal_db_armon_schema_1.ArmonSchema.tableNames.scheduled_job}" 
			WHERE id = $1`, [params.id]);
        if (sj.rowCount < 1) {
            (0, dal_access_error_1.throwDbAccessNotFoundError)("Scheduled job is not found");
        }
        let sqp = [];
        let qb = [];
        let qx = 1;
        const oldJob = sj.rows[0];
        if (oldJob.type === 19) {
            const settings = params.settings;
            if (settings) {
                if (settings.templateId !== undefined) {
                    sqp.push(`"reportTemplateId" = $${qx++}`);
                    qb.push(settings.templateId);
                }
                if (settings.format !== undefined) {
                    sqp.push(`format = $${qx++}`);
                    qb.push(settings.format);
                }
                if (settings.sendWhenEmpty !== undefined) {
                    sqp.push(`"sendWhenEmpty" = $${qx++}`);
                    qb.push(settings.sendWhenEmpty);
                }
                if (sqp.length > 0) {
                    qb.push(params.id);
                    await params.trx.query(`UPDATE "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.reportScheduledJobs}" 
					SET ${sqp.join(",")}
					WHERE id = $${qx++}`, qb);
                }
                if (settings.notification) {
                    sqp = [];
                    qb = [];
                    qx = 1;
                    if (settings.notification.receiverFilterId !== undefined) {
                        const userFilterId = params.settings.notification.receiverFilterId;
                        sqp.push(`"receiverFilterId" = $${qx++}`);
                        qb.push(userFilterId);
                    }
                    if (settings.notification.mediums) {
                        if (settings.notification.mediums.email !== undefined) {
                            sqp.push(`"email" = $${qx++}`);
                            qb.push(settings.notification.mediums.email);
                        }
                        if (settings.notification.mediums.sms !== undefined) {
                            sqp.push(`"sms" = $${qx++}`);
                            qb.push(settings.notification.mediums.sms);
                        }
                        if (settings.notification.mediums.pushNotification !== undefined) {
                            sqp.push(`"pushNotification" = $${qx++}`);
                            qb.push(settings.notification.mediums.pushNotification);
                        }
                        if (settings.notification.mediums.web !== undefined) {
                            sqp.push(`"web" = $${qx++}`);
                            qb.push(settings.notification.mediums.web);
                        }
                    }
                    if (sqp.length > 0) {
                        qb.push(oldJob.notificationId);
                        await params.trx.query(`UPDATE "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.notification}"
							SET ${sqp.join(",")}
							WHERE id = $${qx++}`, qb);
                    }
                }
            }
        }
        else if ([32, 31, 49].includes(oldJob.type)) {
            const settings = params.settings;
            let subjectedEmployeesFilterId;
            if (settings) {
                if (settings.subjectedEmployeesFilterId !== undefined) {
                    subjectedEmployeesFilterId = settings.subjectedEmployeesFilterId;
                    sqp.push(`"subjectedEmployeesFilterId" = $${qx++}`);
                    qb.push(subjectedEmployeesFilterId);
                }
                if (sqp.length > 0) {
                    qb.push(params.id);
                    await params.trx.query(`UPDATE "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.pacsNotificationScheduledJobs}" 
					SET ${sqp.join(",")}
					WHERE id = $${qx++}`, qb);
                }
                if (settings.notification || subjectedEmployeesFilterId) {
                    sqp = [];
                    qb = [];
                    qx = 1;
                    if (settings.notification.receiverFilterId !== undefined || subjectedEmployeesFilterId) {
                        const recieverFilterId = subjectedEmployeesFilterId;
                        sqp.push(`"receiverFilterId" = $${qx++}`);
                        qb.push(recieverFilterId);
                    }
                    if (settings.notification.mediums) {
                        if (settings.notification.mediums.email !== undefined) {
                            sqp.push(`"email" = $${qx++}`);
                            qb.push(settings.notification.mediums.email);
                        }
                        if (settings.notification.mediums.sms !== undefined) {
                            sqp.push(`"sms" = $${qx++}`);
                            qb.push(settings.notification.mediums.sms);
                        }
                        if (settings.notification.mediums.pushNotification !== undefined) {
                            sqp.push(`"pushNotification" = $${qx++}`);
                            qb.push(settings.notification.mediums.pushNotification);
                        }
                        if (settings.notification.mediums.web !== undefined) {
                            sqp.push(`"web" = $${qx++}`);
                            qb.push(settings.notification.mediums.web);
                        }
                    }
                    if (sqp.length > 0) {
                        qb.push(oldJob.notificationId);
                        await params.trx.query(`UPDATE "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.notification}"
						SET ${sqp.join(",")}
						WHERE id = $${qx++}`, qb);
                    }
                }
            }
        }
        sqp = [];
        qb = [];
        qx = 1;
        if (params.enabled !== undefined && oldJob.enabled !== params.enabled) {
            sqp.push(`enabled = $${qx++}`);
            qb.push(params.enabled);
        }
        if (params.interval !== undefined && oldJob.interval !== params.interval) {
            sqp.push(`interval = $${qx++}`);
            qb.push(params.interval);
        }
        if (params.firstExecutionDate !== undefined && oldJob.firstExecutionDate.getTime() !== new Date(params.firstExecutionDate).getTime()) {
            sqp.push(`"firstExecutionDate" = $${qx++}`);
            qb.push(params.firstExecutionDate);
        }
        const nextExecutionDate = (0, dal_utils_1.calculateNextExecutionDateOfScheduledJob)(params.interval || oldJob.interval, new Date(params.firstExecutionDate) || new Date(oldJob.firstExecutionDate));
        if (nextExecutionDate.getTime() !== oldJob.nextExecutionDate.getTime()) {
            sqp.push(`"nextExecutionDate" = $${qx++}`);
            qb.push(nextExecutionDate);
        }
        if (params.note !== undefined) {
            sqp.push(`note = $${qx++}`);
            qb.push(params.note);
        }
        if (sqp.length > 0) {
            qb.push(params.id);
            await params.trx.query(`UPDATE public."${dal_db_armon_schema_1.ArmonSchema.tableNames.scheduled_job}" 
				SET ${sqp.join(",")}
				WHERE id = $${qx++}`, qb);
        }
    }
    async getScheduledJob(params) {
        const { rows, rowCount } = await params.trx.query(`
			SELECT id, "createdT", "organizationId", type, enabled,
			"createdByUserId", interval, "firstExecutionDate", "nextExecutionDate", note, "notificationId", "data"
			FROM public."${dal_db_armon_schema_1.ArmonSchema.tableNames.scheduled_job}"
			WHERE id = $1
		`, [params.id]);
        return rowCount > 0 ? rows[0] : null;
    }
    async getScheduledJobsCursor(params) {
        const q = `
			SELECT id, "interval", "nextExecutionDate", "firstExecutionDate" FROM public."${dal_db_armon_schema_1.ArmonSchema.tableNames.scheduled_job}"
			WHERE enabled = true
			`;
        const cursor = await params.trx.query(new Cursor(q, []));
        while (true) {
            const rows = await new Promise((resolve, reject) => {
                cursor.read(100, (err, rows) => {
                    if (err) {
                        reject(err);
                    }
                    else {
                        resolve(rows);
                    }
                });
            });
            await params.onData(rows);
            if (rows.length < 100) {
                break;
            }
        }
        cursor.close();
    }
    async getScheduledJobListItem(params) {
        const { rows, rowCount } = await params.trx.query(`
			SELECT	sj.id, 
					sj.enabled, 
					sj.type, 
					sj."createdByUserId",
					sj.interval, 
					sj."firstExecutionDate", 
					sj."nextExecutionDate", 
					sj.note, 
					sj."notificationId", 
					rsj."reportTemplateId", 
					rsj."sendWhenEmpty", 
					rsj.format,
					CASE WHEN sef.id IS NOT NULL THEN jsonb_build_object('id', sef.id, 'name', sef.name) END  as "subjectedEmployeesFilter",
					CASE WHEN f.id IS NOT NULL THEN jsonb_build_object('id', f.id, 'name', f.name) END  as "filter",
					n.email, 
					n.sms, 
					n.web, 
					n."pushNotification",
					rt.type as "reportType", 
					rt."customReportId"
			FROM public."${dal_db_armon_schema_1.ArmonSchema.tableNames.scheduled_job}" sj
			LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.reportScheduledJobs}" rsj
			ON sj.id = rsj.id
			LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.pacsNotificationScheduledJobs}" pnsj
			ON sj.id = pnsj.id
			LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.notification}" n
			ON sj."notificationId" = n.id
			LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.reportTemplates}" rt
			ON rsj."reportTemplateId" = rt.id
			LEFT JOIN "${params.organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilter} AS f
			ON f.id = n."receiverFilterId"
			LEFT JOIN "${params.organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilter} AS sef
			ON sef.id = pnsj."subjectedEmployeesFilterId"
			WHERE sj.id = $1
		`, [params.id]);
        if (rowCount === 0) {
            return null;
        }
        else {
            const row = rows[0];
            const createdBy = row.createdByUserId
                ? {
                    id: row.createdByUserId,
                    caption: (await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId: params.organizationId, userId: row.createdByUserId })).caption,
                }
                : undefined;
            const notification = row.filter
                ? {
                    receiverFilter: row.filter,
                    mediums: {
                        email: row.email,
                        pushNotification: row.pushNotification,
                        sms: row.sms,
                        web: row.web,
                    },
                }
                : null;
            return {
                id: row.id,
                createdT: row.createdT,
                enabled: row.enabled,
                firstExecutionDate: row.firstExecutionDate,
                interval: row.interval,
                nextExecutionDate: row.nextExecutionDate,
                settings: row.type === 19
                    ? {
                        format: row.format,
                        sendWhenEmpty: row.sendWhenEmpty,
                        templateId: row.reportTemplateId,
                        reportType: row.reportType,
                        customReportId: row.customReportId,
                        notification,
                    }
                    : {
                        notification,
                        subjectedEmployees: row.subjectedEmployeesFilter,
                    },
                type: row.type,
                createdBy: createdBy,
                note: row.note,
                notificationId: row.notificationId,
            };
        }
    }
    async getPACSNotificationScheduledJob(params) {
        const { rows, rowCount } = await params.trx.query(`
			SELECT sj.id, sj."notificationId", f."filter" as "userFilter" FROM public."${dal_db_armon_schema_1.ArmonSchema.tableNames.scheduled_job}" sj
			INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.pacsNotificationScheduledJobs}" pnsj
			ON sj.id = pnsj.id
			INNER JOIN "${params.organizationId}".${dal_db_armon_schema_1.ArmonSchema.viewNames.vwUserFilter} as f 
			ON f.id = pnsj."subjectedEmployeesFilterId"
			WHERE sj.id = $1`, [params.id]);
        return rowCount > 0 ? rows[0] : null;
    }
    async getDynamicForm(params) {
        const trxx = params.trx ?? this._pgPool;
        const { rows, rowCount } = await trxx.query(`
			SELECT form FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.dynamicForms}" 
			WHERE id = $1::uuid
        `, [params.dynamicFormId]);
        const row = rowCount > 0 ? rows[0].form : null;
        let userContent;
        if (row) {
            const userDependencies = row.userDependencies[0];
            userContent = {
                id: row.id,
                dataSchema: row.dataSchema,
                version: row.version,
                readonlyFields: userDependencies?.readonlyFields,
                form: userDependencies?.form,
                list: userDependencies?.list,
                print: userDependencies?.print,
            };
        }
        return userContent;
    }
    async getDynamicForms(params) {
        if (params.dynamicFormIds.length === 0)
            return [];
        const { rows } = await params.trx.query(`
				SELECT form FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.dynamicForms}"
				WHERE id = ANY($1::uuid[])
			`, [params.dynamicFormIds]);
        const results = [];
        for (const row of rows) {
            const form = row.form;
            if (!form)
                continue;
            const userDependencies = form.userDependencies?.[0];
            results.push({
                id: form.id,
                dataSchema: form.dataSchema,
                version: form.version,
                readonlyFields: userDependencies?.readonlyFields,
                form: userDependencies?.form,
                list: userDependencies?.list,
                print: userDependencies?.print,
            });
        }
        return results;
    }
}
exports.PSQLDalAccessSystem = PSQLDalAccessSystem;
