"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PSQLDalAccessSocial = void 0;
const uuid_1 = __importDefault(require("uuid"));
const dal_access_rdb_1 = require("../rdb/dal.access.rdb");
const dal_db_armon_schema_1 = require("../../db/armon/dal.db.armon.schema");
const api_error_1 = require("../../../api/api.error");
const dal_manager_1 = require("../../dal.manager");
const dal_constants_1 = require("../../dal.constants");
class PSQLDalAccessSocial extends dal_access_rdb_1.RDBAccess {
    constructor(knex, pgPool) {
        super(knex, pgPool);
    }
    async upsertRegionTicket(organizationId, userId, regionTicket) {
        let id = regionTicket.id || uuid_1.default.v4();
        await this.dbClient.transaction(async (trx) => {
            if (regionTicket.id) {
                await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.regionTickets).where("id", regionTicket.id).update({
                    regionId: regionTicket.regionId,
                    organizationId: organizationId,
                    name: regionTicket.name,
                    requiredUnits: regionTicket.requiredUnits,
                    updatedAt: new Date(),
                });
            }
            else {
                let existingRegionTicket = await trx
                    .withSchema(organizationId)
                    .table(dal_db_armon_schema_1.ArmonSchema.tableNames.regionTickets)
                    .where("regionId", regionTicket.regionId)
                    .whereNull("deletedAt")
                    .first("id");
                if (existingRegionTicket && existingRegionTicket.id) {
                    throw (0, api_error_1.generateConflictError)({ message: "a ticket for region exists", details: "region exists" });
                }
                await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.regionTickets).insert({
                    id: id,
                    regionId: regionTicket.regionId,
                    organizationId: organizationId,
                    name: regionTicket.name,
                    requiredUnits: regionTicket.requiredUnits,
                    createdAt: new Date(),
                    updatedAt: new Date(),
                });
            }
        });
        return Promise.resolve(id);
    }
    async deleteRegionTicket(organizationId, regionTicketId) {
        let regionTicket = await this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.regionTickets).where("id", regionTicketId).first();
        if (!regionTicket)
            (0, api_error_1.generateNotFoundApiError)({ message: "region ticket not found" });
        await this.dbClient.transaction(async (trx) => {
            await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.regionTickets).where("id", regionTicketId).update({
                deletedAt: new Date(),
            });
        });
    }
    async getRegionTicket(organizationId, regionTicketId) {
        let regionTicketDetailed = await this.dbClient
            .withSchema(organizationId)
            .table("regionTickets as rt")
            .innerJoin("regions as r", "r.id", "rt.regionId")
            .where("rt.id", regionTicketId)
            .whereNull("rt.deletedAt")
            .first("rt.id", "rt.name", "rt.organizationId", "r.name as regionName", "rt.requiredUnits", "r.id as regionId");
        return Promise.resolve(regionTicketDetailed);
    }
    async listRegionTickets(params) {
        const qb = [];
        const qw = [];
        let qx = 1;
        const qFrom = `
		FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.regionTickets}" rt
		INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.regions}" r
			ON r.id = rt."regionId"
		LEFT JOIN "${params.organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.region_administrators} ra
			ON r.id = ra."regionId" 
	`;
        qw.push(`rt."deletedAt" IS NULL`);
        if (params.regionIds) {
            qw.push(`r.id = ANY($${qx++}::uuid[])`);
            qb.push(params.regionIds);
        }
        if (params.regionTicketIds) {
            qw.push(`rt.id = ANY($${qx++}::uuid[])`);
            qb.push(params.regionTicketIds);
        }
        qw.push(`(ra."userId" IS NULL OR ra."userId" = $${qx++} AND "read" IS TRUE)`);
        qb.push(params.requesterUserId);
        let qWhere = `WHERE ${qw.join(" AND ")}`;
        const { rowCount, rows } = await params.trx.query(`SELECT count(*)::SMALLINT AS c ${qFrom} ${qWhere}`, qb);
        const result = {
            skip: params.pagination.skip,
            take: params.pagination.take,
            total: rowCount > 0 ? rows[0].c : 0,
            items: [],
        };
        if (result.total > 0) {
            let q = `
			SELECT rt.id, rt.name, rt."organizationId", r.name as "regionName", rt."requiredUnits", r.id as "regionId"
			${qFrom} ${qWhere}
			`;
            if (params.pagination?.skip > 0) {
                q += `
					OFFSET $${qx++}`;
                qb.push(params.pagination.skip);
            }
            q += `
                LIMIT $${qx++}`;
            qb.push(params.pagination.take || 100);
            result.items = (await params.trx.query(q, qb)).rows;
            return result;
        }
        return result;
    }
    async listUserRegionTicketUnits(organizationId, pagination, args) {
        let paginationResponse = {
            skip: pagination.skip,
            take: pagination.take,
            total: 0,
        };
        let items = [];
        let qb = this.dbClient
            .withSchema(organizationId)
            .table("userRegionTicketUnits as urc")
            .innerJoin("userOrganizations as uo", (join) => {
            join.on("uo.userId", "urc.userId").on("uo.organizationId", "urc.organizationId");
        })
            .innerJoin("userOrganizationProfiles as uop", (join) => {
            join.on("uop.userOrganizationId", "uo.id");
        })
            .where("uo.organizationId", organizationId)
            .where("urc.organizationId", organizationId);
        if (args.userIds)
            qb.whereIn("urc.userId", args.userIds);
        if (args.userGroupIds)
            qb.innerJoin("userGroupUserOrganizations as uguo", "uguo.userOrganizationId", "uo.id").whereNull("uguo.deletedAt").whereIn("uguo.userGroupId", args.userGroupIds);
        if (args.organizationUnitIds)
            qb.innerJoin("userOrganizationOrganizationUnits as uoou", "uoou.userOrganizationId", "uo.id")
                .whereNull("uoou.deletedAt")
                .whereIn("uoou.organizationUnitId", args.organizationUnitIds);
        await qb
            .clone()
            .count()
            .then((count) => {
            paginationResponse.total = parseInt(count[0].count);
        });
        if (pagination.skip)
            qb.offset(pagination.skip);
        if (pagination.take)
            qb.limit(pagination.take);
        let rows = await qb.orderBy("uop.name", "asc").select("urc.remainingUnits", "uop.name", "uop.surname", "uop.uniqueId", "urc.userId", "urc.totalSpentUnits");
        let userCaptions = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, rows.map((i) => i.userId));
        for (const row of rows) {
            let userCaption = userCaptions.find((u) => u.id == row.userId);
            let item = {
                totalSpentUnits: row.totalSpentUnits,
                remainingUnits: row.remainingUnits,
                user: {
                    fullname: (row.name ?? "") + " " + (row.surname ?? ""),
                    id: row.userId,
                    uniqueId: row.uniqueId,
                },
                userCaptions: userCaption ? userCaption.captionLines : [],
            };
            items.push(item);
        }
        return Promise.resolve({
            pagination: paginationResponse,
            items: items,
        });
    }
    async assignRegionTicketUserUnits(organizationId, params, trx) {
        let session = await dal_manager_1.dbManager.accessUser.validateUserSelectionSession(organizationId, params.requestUserId, params.userSelectionSessionId, trx);
        const userActions = await trx.query(`
			SELECT "userId" 
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userSelectionSessionActions}"
			WHERE "sessionId" = $1
				AND "action" = $2
		`, [session.id, dal_constants_1.DalConstants.UserSelectionSessionAction.Added]);
        const userIds = userActions.rows.map((m) => m.userId);
        const existingUsers = await trx.query(`
			SELECT * 
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userRegionTicketUnits}"
			WHERE "userId" = ANY ($1::UUID[])
		`, [userIds]);
        const newUsers = userIds.filter((f) => !existingUsers.rows.some((r) => r.userId === f));
        let newUserItems = [];
        let now = new Date();
        for (const newUser of newUsers) {
            let item = {
                createdAt: now,
                updatedAt: now,
                organizationId: organizationId,
                remainingUnits: params.amount,
                userId: newUser,
                lastUpdateUserId: params.requestUserId,
                lastUpdateAmount: params.amount,
            };
            newUserItems.push(item);
        }
        await trx.query(`
			INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userRegionTicketUnits}"
			(id, "createdAt", "updatedAt", "organizationId", "userId", "remainingUnits", "lastUpdateUserId", "lastUpdateAmount")
			SELECT gen_random_uuid(), (u->>'createdAt')::TIMESTAMP WITH TIME ZONE, (u->>'updatedAt')::TIMESTAMP WITH TIME ZONE, 
				(u->>'organizationId')::UUID, (u->>'userId')::UUID, (u->>'remainingUnits')::INTEGER,
				(u->>'lastUpdateUserId')::UUID, (u->>'lastUpdateAmount')::INTEGER
			FROM UNNEST ($1::JSONB[]) AS u
		`, [newUserItems]);
        const existingUserItems = existingUsers.rows.map((m) => {
            return {
                id: m.id,
                updatedAt: now,
                remainingUnits: +params.amount + +m.remainingUnits,
                lastUpdateUserId: params.requestUserId,
                lastUpdateAmount: params.amount,
            };
        });
        await trx.query(`
			UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userRegionTicketUnits}"
			SET "updatedAt" = (u->>'updatedAt')::TIMESTAMP WITH TIME ZONE,
				"remainingUnits" = (u->>'remainingUnits')::INTEGER,
				"lastUpdateUserId" = (u->>'lastUpdateUserId')::UUID,
				"lastUpdateAmount" = (u->>'lastUpdateAmount')::INTEGER
			FROM UNNEST($1::JSONB[]) AS u
			WHERE (u->>'id')::UUID = id
		`, [existingUserItems]);
        await trx.query(`
			DELETE FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userSelectionSessions}"
			WHERE id = $1
		`, [params.userSelectionSessionId]);
        let logs = [];
        logs.push({
            u: now,
            a: params.amount,
            o: userIds,
            oId: organizationId,
            r: params.requestUserId,
            n: params.note ?? null,
        });
        await dal_manager_1.dbManager.accessLog.uploadUserTicketTransactionHistory(organizationId, logs, trx);
    }
    async listUserRegionTicketsForControlPanel(organizationId) {
        return Promise.resolve(await this.dbClient
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.userRegionTicketUnits)
            .where("organizationId", organizationId)
            .select("userId", "remainingUnits as amount"));
    }
    async listRegionTicketUsageReport(organizationId, pagination, args) {
        let paginationResponse = {
            skip: pagination.skip,
            take: pagination.take,
            total: 0,
        };
        let items = [];
        let qb = this.dbClient
            .withSchema(organizationId)
            .table("userRegionTicketUnits as urc")
            .innerJoin("userOrganizations as uo", (join) => {
            join.on("uo.userId", "urc.userId").on("uo.organizationId", "urc.organizationId");
        })
            .innerJoin("userOrganizationProfiles as uop", (join) => {
            join.on("uop.userOrganizationId", "uo.id");
        })
            .where("uo.organizationId", organizationId);
        if (args.userIds)
            qb.whereIn("urc.userId", args.userIds);
        if (args.userGroupIds)
            qb.innerJoin("userGroupUserOrganizations as uguo", "uguo.userOrganizationId", "uo.id").whereNull("uguo.deletedAt").whereIn("uguo.userGroupId", args.userGroupIds);
        if (args.organizationUnitIds)
            qb.innerJoin("userOrganizationOrganizationUnits as uoou", "uoou.userOrganizationId", "uo.id")
                .whereNull("uoou.deletedAt")
                .whereIn("uoou.organizationUnitId", args.organizationUnitIds);
        let userIds = (await qb.select("uo.userId")).map((u) => u.userId);
        let logResult = await dal_manager_1.dbManager.accessLog.listRegionTicketReport(organizationId, pagination, args.dateRange, userIds, args.regionTicketIds);
        paginationResponse = logResult.pagination;
        let regionTickets = await this.dbClient.withSchema(organizationId).table("regionTickets").where("organizationId", organizationId).select("id", "name");
        let users = await dal_manager_1.dbManager.accessPacs.getBasicUserInfoList(organizationId, logResult.items.map((l) => l.o));
        for (const log of logResult.items) {
            for (const regionLog of log.rg) {
                let regionTicket = regionTickets.find((r) => regionLog.rti && regionLog.rti.i == r.id);
                if (regionTicket) {
                    items.push({
                        user: users.find((u) => u.id === log.o),
                        actionDate: new Date(log.u),
                        ticketName: regionTicket ? regionTicket.name : "",
                    });
                }
            }
        }
        return Promise.resolve({
            pagination: paginationResponse,
            items: items,
        });
    }
    async genericSearchRegionTicket(organizationId, options) {
        let result = {
            total: 0,
            items: [],
        };
        let qb = this.dbClient.withSchema(organizationId).from(dal_db_armon_schema_1.ArmonSchema.tableNames.regionTickets).where("organizationId", organizationId).whereNull("deletedAt");
        let filterExists = options.filter && options.filter.trim().length > 0;
        let regexList = [];
        if (filterExists) {
            qb.whereWrapped((q) => {
                let filterTokens = options.filter.split(" ");
                for (const filterToken of filterTokens) {
                    let token = filterToken.trim();
                    regexList.push(new RegExp(options.filter.replace(/[-[\]{}()*+?.,\\^$|#\s]/g, "\\$&"), "ig"));
                    if (token.length > 0) {
                        token = "%" + token + "%";
                        q.orWhereRaw("name ilike ?", token);
                    }
                }
            });
        }
        let columns = ["id", "name"];
        let qbCount = qb.clone();
        result.total = await qbCount.count("id as c").then((rows) => {
            return parseInt(rows[0].c);
        });
        if (result.total === 0) {
            return Promise.resolve(result);
        }
        qb.orderByRaw("name").column(columns).select().offset(options.skip).limit(options.take);
        result.items = await qb.then((rows) => {
            return rows.map((t) => {
                let name = t["name"];
                return {
                    id: t["id"],
                    captionLines: [name],
                    matchItem: name,
                };
            });
        });
        return Promise.resolve(result);
    }
    async getUserCurrentBalance(organizationId, userId) {
        return this.dbClient.withSchema(organizationId).from(dal_db_armon_schema_1.ArmonSchema.tableNames.userRegionTicketUnits).where("organizationId", organizationId).where("userId", userId).first();
    }
    async getActiveBalanceUserCount(organizationId, trx) {
        let result = 0;
        let qb = this.dbClient
            .withSchema(organizationId)
            .from(dal_db_armon_schema_1.ArmonSchema.tableNames.userRegionTicketUnits)
            .where("organizationId", organizationId)
            .whereNull("deletedAt")
            .where("remainingUnits", ">", 0)
            .count();
        if (trx) {
            qb.transacting(trx);
        }
        result = parseInt((await qb.first()).count);
        return Promise.resolve(result);
    }
}
exports.PSQLDalAccessSocial = PSQLDalAccessSocial;
