"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PSQLDalAccessReport = void 0;
const lodash_1 = __importDefault(require("lodash"));
const moment_1 = __importDefault(require("moment"));
const uuid_1 = __importDefault(require("uuid"));
const api_error_1 = require("../../../api/api.error");
const dal_constants_1 = require("../../dal.constants");
const dal_manager_1 = require("../../dal.manager");
const dal_db_armon_schema_1 = require("../../db/armon/dal.db.armon.schema");
const predefined_roles_1 = require("../../db/predefined/predefined.roles");
const dal_access_rdb_report_1 = require("../rdb/dal.access.rdb.report");
const dal_access_psql_common_1 = require("./dal.access.psql.common");
class PSQLDalAccessReport extends dal_access_rdb_report_1.RDBDalAccessReport {
    constructor(knex, pgPool) {
        super(knex, pgPool);
    }
    async collectMonthlyWorkSummaryFilterInfo(organizationId, organizationUnitIds, userGroupIds, userIds, workPlanIds) {
        let result = {
            user: null,
            userGroups: null,
            organizationUnits: null,
            workplans: null,
        };
        if (organizationId != null && userIds != null && userIds.length > 0) {
            result.user = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, userIds);
        }
        if (organizationUnitIds != null && organizationUnitIds.length > 0) {
            result.organizationUnits = await this.dbClient
                .withSchema(organizationId)
                .select("name")
                .from(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits)
                .whereIn("id", organizationUnitIds)
                .whereNull("deletedAt");
        }
        if (userGroupIds != null && userGroupIds.length > 0) {
            result.userGroups = await this.dbClient.withSchema(organizationId).select("name").from(dal_db_armon_schema_1.ArmonSchema.tableNames.userGroups).whereIn("id", userGroupIds).whereNull("deletedAt");
        }
        if (workPlanIds)
            result.workplans = await dal_manager_1.dbManager.accessPacs.listWorkPlanByIds(organizationId, workPlanIds);
        return result;
    }
    async collectForbiddenUsersFilterInfo(organizationId, userIds, regionIds) {
        let result = {
            userCaptions: null,
            regions: null,
        };
        if (organizationId != null && userIds != null && userIds.length > 0) {
            result.userCaptions = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, userIds);
        }
        if (regionIds != null && regionIds.length > 0) {
            result.regions = await this.dbClient.withSchema(organizationId).select("name").from(dal_db_armon_schema_1.ArmonSchema.tableNames.regions).whereIn("id", regionIds);
        }
        return result;
    }
    async collectFirstAndLastAccessLogsForSingleUserFilterInfo(organizationId, userIds, accessControlPointIds, regionIds) {
        let result = {
            userCaptions: null,
            accessControlPoints: null,
            regions: null,
        };
        if (organizationId != null && userIds != null && userIds.length > 0) {
            result.userCaptions = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, userIds);
        }
        if (organizationId != null && accessControlPointIds != null && accessControlPointIds.length > 0) {
            result.accessControlPoints = await this.dbClient
                .withSchema(organizationId)
                .select("name")
                .from(dal_db_armon_schema_1.ArmonSchema.tableNames.accessControlPoints + " as acp")
                .whereIn("id", accessControlPointIds)
                .where("acp.organizationId", "=", organizationId)
                .whereNull("acp.deletedAt");
        }
        if (regionIds != null && regionIds.length > 0) {
            result.regions = await this.dbClient.withSchema(organizationId).select("name").from(dal_db_armon_schema_1.ArmonSchema.tableNames.regions).whereIn("id", regionIds);
        }
        return result;
    }
    async collectFirstAndLastAccessLogsFilterInfo(organizationId, userIds, organizationUnitIds, userGroupIds, accessControlPointIds, regionIds) {
        let result = {
            userCaptions: null,
            organizationUnits: null,
            userGroups: null,
            accessControlPoints: null,
            regions: null,
        };
        if (organizationId != null && userIds != null && userIds.length > 0) {
            result.userCaptions = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, userIds);
        }
        if (organizationUnitIds != null && organizationUnitIds.length > 0) {
            result.organizationUnits = await this.dbClient
                .withSchema(organizationId)
                .select("name")
                .from(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits)
                .whereIn("id", organizationUnitIds)
                .whereNull("deletedAt");
        }
        if (userGroupIds != null && userGroupIds.length > 0) {
            result.userGroups = await this.dbClient.withSchema(organizationId).select("name").from(dal_db_armon_schema_1.ArmonSchema.tableNames.userGroups).whereIn("id", userGroupIds).whereNull("deletedAt");
        }
        if (organizationId != null && accessControlPointIds != null && accessControlPointIds.length > 0) {
            result.accessControlPoints = await this.dbClient
                .withSchema(organizationId)
                .select("name")
                .from(dal_db_armon_schema_1.ArmonSchema.tableNames.accessControlPoints + " as acp")
                .whereIn("id", accessControlPointIds)
                .where("acp.organizationId", "=", organizationId)
                .whereNull("acp.deletedAt");
        }
        if (regionIds != null && regionIds.length > 0) {
            result.regions = await this.dbClient.withSchema(organizationId).select("name").from(dal_db_armon_schema_1.ArmonSchema.tableNames.regions).whereIn("id", regionIds);
        }
        return result;
    }
    async collectCredentialFilterInfo(organizationId, filter) {
        let result = {
            userCaptions: null,
            organizationUnits: null,
            userGroups: null,
        };
        if (organizationId != null && filter.userIds != null && filter.userIds.length > 0) {
            result.userCaptions = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, filter.userIds);
        }
        if (filter.organizationUnitIds != null && filter.organizationUnitIds.length > 0) {
            result.organizationUnits = await this.dbClient
                .withSchema(organizationId)
                .select("name")
                .from(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits)
                .whereIn("id", filter.organizationUnitIds)
                .whereNull("deletedAt");
        }
        if (filter.userGroupIds != null && filter.userGroupIds.length > 0) {
            result.userGroups = await this.dbClient
                .withSchema(organizationId)
                .select("name")
                .from(dal_db_armon_schema_1.ArmonSchema.tableNames.userGroups)
                .whereIn("id", filter.userGroupIds)
                .whereNull("deletedAt");
        }
        return result;
    }
    async collectPermissionManagementFilterInfo(permissionTypeId, userIds, organizationId) {
        let result = { userCaptions: null, permissionType: null };
        if (organizationId != null && userIds != null && userIds.length > 0) {
            result.userCaptions = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, userIds);
        }
        if (permissionTypeId) {
            result.permissionType = await dal_manager_1.dbManager.accessPacs.getPPermissionType(organizationId, permissionTypeId);
        }
        return result;
    }
    async collectRegionTicketTransactionInfo(targetUserId, actionUserId, organizationId) {
        let result = {
            actionUsers: null,
            targetUsers: null,
        };
        if (organizationId != null && targetUserId != null) {
            result.actionUsers = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, [targetUserId]);
        }
        if (organizationId != null && actionUserId != null) {
            result.targetUsers = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, [actionUserId]);
        }
        return result;
    }
    async collectRegionTickerUserUnitFilterInfo(organizationId, organizationUnitIds, userGroupIds, userIds) {
        let result = {
            organizationUnits: null,
            userGroups: null,
            user: null,
        };
        if (organizationUnitIds != null && organizationUnitIds.length > 0) {
            result.organizationUnits = await this.dbClient
                .withSchema(organizationId)
                .select("name")
                .from(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits)
                .whereIn("id", organizationUnitIds)
                .whereNull("deletedAt");
        }
        if (userGroupIds != null && userGroupIds.length > 0) {
            result.userGroups = await this.dbClient.withSchema(organizationId).select("name").from(dal_db_armon_schema_1.ArmonSchema.tableNames.userGroups).whereIn("id", userGroupIds).whereNull("deletedAt");
        }
        if (organizationId != null && userIds != null && userIds.length > 0) {
            result.user = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, userIds);
        }
        return result;
    }
    async collectUserTicketUsageFilterInfo(regionTicketIds, organizationId, organizationUnitIds, userGroupIds, userIds) {
        let result = {
            organizationUnits: null,
            userGroups: null,
            user: null,
            tickets: null,
        };
        if (organizationUnitIds != null && organizationUnitIds.length > 0) {
            result.organizationUnits = await this.dbClient
                .withSchema(organizationId)
                .select("name")
                .from(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits)
                .whereIn("id", organizationUnitIds)
                .whereNull("deletedAt");
        }
        if (userGroupIds != null && userGroupIds.length > 0) {
            result.userGroups = await this.dbClient.withSchema(organizationId).select("name").from(dal_db_armon_schema_1.ArmonSchema.tableNames.userGroups).whereIn("id", userGroupIds).whereNull("deletedAt");
        }
        if (organizationId != null && userIds != null && userIds.length > 0) {
            result.user = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, userIds);
        }
        if (regionTicketIds != null && organizationId != null && regionTicketIds.length > 0) {
            let items = [];
            let regionIds = null;
            let qb = this.dbClient
                .withSchema(organizationId)
                .table("regionTickets as rt")
                .innerJoin("regions as r", "r.id", "rt.regionId")
                .whereNull("rt.deletedAt")
                .where("r.organizationId", organizationId)
                .where("rt.organizationId", organizationId);
            if (regionIds)
                qb.whereIn("r.id", regionIds);
            if (regionTicketIds)
                qb.whereIn("rt.id", regionTicketIds);
            await qb.select("rt.id", "rt.name", "rt.organizationId", "r.name as regionName", "rt.requiredUnits", "r.id as regionId").then((rows) => {
                for (const row of rows)
                    items.push(row);
            });
            result.tickets = items;
        }
        return result;
    }
    async collectWorkSummaryFilterInfo(userIds, userGroupIds, organizationUnitIds, workPlanIds, organizationId) {
        let result = { users: null, userGroups: null, organizationUnits: null, workplans: null };
        if (userGroupIds != null && userGroupIds.length > 0) {
            result.userGroups = await this.dbClient.withSchema(organizationId).select("name").from(dal_db_armon_schema_1.ArmonSchema.tableNames.userGroups).whereIn("id", userGroupIds).whereNull("deletedAt");
        }
        if (organizationUnitIds != null && organizationUnitIds.length > 0) {
            result.organizationUnits = await this.dbClient
                .withSchema(organizationId)
                .select("name")
                .from(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits)
                .whereIn("id", organizationUnitIds)
                .whereNull("deletedAt");
        }
        if (organizationId != null && userIds != null && userIds.length > 0) {
            result.users = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, userIds);
        }
        if (workPlanIds)
            result.workplans = await dal_manager_1.dbManager.accessPacs.listWorkPlanByIds(organizationId, workPlanIds);
        return result;
    }
    async collectAccessLogReportFilterInfo(userIds, userGroupIds, organizationUnitIds, accessControlPointIds, regionIds, organizationId) {
        let result = { userCaptions: null, userGroups: null, organizationUnits: null, accessControlPoints: null, regions: null };
        if (organizationId != null && userIds != null && userIds.length > 0) {
            result.userCaptions = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, userIds);
        }
        if (userGroupIds != null && userGroupIds.length > 0) {
            result.userGroups = await this.dbClient.withSchema(organizationId).select("name").from(dal_db_armon_schema_1.ArmonSchema.tableNames.userGroups).whereIn("id", userGroupIds).whereNull("deletedAt");
        }
        if (organizationUnitIds != null && organizationUnitIds.length > 0) {
            result.organizationUnits = await this.dbClient
                .withSchema(organizationId)
                .select("name")
                .from(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits)
                .whereIn("id", organizationUnitIds)
                .whereNull("deletedAt");
        }
        if (organizationId != null && accessControlPointIds != null && accessControlPointIds.length > 0) {
            result.accessControlPoints = await this.dbClient
                .withSchema(organizationId)
                .select("name")
                .from(dal_db_armon_schema_1.ArmonSchema.tableNames.accessControlPoints + " as acp")
                .whereIn("id", accessControlPointIds)
                .where("acp.organizationId", "=", organizationId)
                .whereNull("acp.deletedAt");
        }
        if (regionIds != null && regionIds.length > 0) {
            result.regions = await this.dbClient.withSchema(organizationId).select("name").from(dal_db_armon_schema_1.ArmonSchema.tableNames.regions).whereIn("id", regionIds);
        }
        return result;
    }
    async collectRegionStateFilterInfo(regionId, userIds, organizationId) {
        let result = { userCaptions: null, region: null };
        if (organizationId != null && userIds != null && userIds.length > 0) {
            result.userCaptions = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, userIds);
        }
        if (regionId != null) {
            result.region = await this.dbClient.withSchema(organizationId).select("name").from(dal_db_armon_schema_1.ArmonSchema.tableNames.regions).where("id", regionId);
        }
        return result;
    }
    async collectDurationInRegionFilterInfo(regionId, userIds, organizationId, organizationUnitIds, userGroupIds) {
        let result = {
            userCaptions: null,
            region: null,
            organizationUnits: null,
            userGroups: null,
        };
        if (organizationId != null && userIds != null && userIds.length > 0) {
            result.userCaptions = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, userIds);
        }
        if (regionId != null) {
            result.region = await this.dbClient.withSchema(organizationId).select("name").from(dal_db_armon_schema_1.ArmonSchema.tableNames.regions).where("id", regionId);
        }
        if (organizationUnitIds != null && organizationUnitIds.length > 0) {
            result.organizationUnits = await this.dbClient
                .withSchema(organizationId)
                .select("name")
                .from(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits)
                .whereIn("id", organizationUnitIds)
                .whereNull("deletedAt");
        }
        if (userGroupIds != null && userGroupIds.length > 0) {
            result.userGroups = await this.dbClient.withSchema(organizationId).select("name").from(dal_db_armon_schema_1.ArmonSchema.tableNames.userGroups).whereIn("id", userGroupIds).whereNull("deletedAt");
        }
        return result;
    }
    async collectSensorEventsInfo(organizationId, sensorFilter, terminalFilter) {
        let sensorFilterResult;
        let terminalFilterResult;
        if (sensorFilter != null && sensorFilter.length > 0) {
            let qb = this.dbClient.withSchema(organizationId).from("deviceDryContactInputs as dry").whereNull("dry.deletedAt").whereIn("dry.id", sensorFilter);
            let result = await qb.select("dry.deviceId", "dry.number", "dry.name");
            if (result != null && result.length > 0) {
                for (const res of result) {
                    if (res.name == null || res.name == "") {
                        let qb2 = this.dbClient.withSchema(organizationId).from("devices as d").whereNull("d.deletedAt").where("d.id", res.deviceId);
                        let name = await qb2.select("d.name").first();
                        if (name != null && name.name != null)
                            res.name = name.name + " #" + res.number;
                    }
                }
            }
            sensorFilterResult = result;
        }
        if (terminalFilter != null && terminalFilter.length > 0) {
            let qb = this.dbClient
                .withSchema(organizationId)
                .table(dal_db_armon_schema_1.ArmonSchema.tableNames.devices + " as d")
                .where("d.organizationId", organizationId)
                .whereNull("d.deletedAt")
                .whereIn("d.brand", [dal_constants_1.DalConstants.DeviceBrand.Armon, dal_constants_1.DalConstants.DeviceBrand.AssaAbloy, dal_constants_1.DalConstants.DeviceBrand.HikVision])
                .whereIn("d.model", dal_constants_1.DalConstants.terminalModels)
                .whereIn("d.id", terminalFilter);
            terminalFilterResult = (await qb.select("d.name"));
        }
        let result = {
            sensorFilterResult: sensorFilterResult,
            terminalFilterResult: terminalFilterResult,
        };
        return result;
    }
    async collectUsersForAccessReportFilter(organizationId, options) {
        let result = {
            pagination: {
                total: 0,
                skip: 0,
                take: 0,
            },
            items: [],
        };
        let filterOrganizationUnitIds = null;
        if (options.organizationUnitIds && options.organizationUnitIds.length > 0) {
            let qbOrganizationUnits = this.dbClient
                .withSchema(organizationId)
                .from(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits)
                .select("id")
                .where("organizationId", organizationId)
                .whereNull("deletedAt")
                .transacting(options.trx);
            if (options.filterOrganizationUnitMembersHierarchically) {
                qbOrganizationUnits = qbOrganizationUnits.where((qbOr) => {
                    qbOr.whereIn("id", options.organizationUnitIds).orWhereRaw(`"ancestorIds" similar to '%(${options.organizationUnitIds.join("|")})%'`);
                });
            }
            else {
                qbOrganizationUnits = qbOrganizationUnits.whereIn("id", options.organizationUnitIds);
            }
            filterOrganizationUnitIds = (await qbOrganizationUnits).map((m) => m.id);
            if (filterOrganizationUnitIds.length === 0) {
                filterOrganizationUnitIds = null;
            }
        }
        let qb = this.dbClient
            .withSchema(organizationId)
            .from(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations + " as uo")
            .innerJoin(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles + " as uop", "uo.id", "uop.userOrganizationId")
            .whereNull("uop.deletedAt")
            .whereNull("uo.deletedAt")
            .where("uo.organizationId", organizationId)
            .transacting(options.trx);
        let filterUserIds = options.userIds && options.userIds.length > 0 ? options.userIds : null;
        let filterUserGroupIds = options.userGroupIds && options.userGroupIds.length > 0 ? options.userGroupIds : null;
        if (filterOrganizationUnitIds || filterUserIds || filterUserGroupIds) {
            let ffun = (qbOr) => {
                let firstHandled = false;
                if (filterUserIds) {
                    qbOr = qbOr.whereIn("uop.userId", filterUserIds);
                    firstHandled = true;
                }
                if (filterUserGroupIds) {
                    let subquery = this.dbClient
                        .withSchema(organizationId)
                        .from(dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupUserOrganizations)
                        .whereIn("userGroupId", filterUserGroupIds)
                        .whereNull("deletedAt")
                        .select("userOrganizationId");
                    if (firstHandled) {
                        qbOr = qbOr.orWhereIn("uo.id", subquery);
                    }
                    else {
                        qbOr = qbOr.whereIn("uo.id", subquery);
                        firstHandled = true;
                    }
                }
                if (filterOrganizationUnitIds) {
                    let subquery = this.dbClient
                        .withSchema(organizationId)
                        .from(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits)
                        .whereIn("organizationUnitId", filterOrganizationUnitIds)
                        .whereNull("deletedAt")
                        .select("userOrganizationId");
                    if (firstHandled) {
                        qbOr = qbOr.orWhereIn("uo.id", subquery);
                    }
                    else {
                        qbOr = qbOr.whereIn("uo.id", subquery);
                    }
                }
            };
            qb.where(ffun);
        }
        if (options.status && options.status !== dal_constants_1.DalConstants.IdentityStatusType.All) {
            qb.where("uo.isDisabled", options.status === dal_constants_1.DalConstants.IdentityStatusType.Disabled);
        }
        if (options.pagination) {
            let qbc = qb.clone();
            qbc = qbc.transacting(options.trx).count();
            result.pagination.total = await qbc.then((r) => parseInt(r[0].count));
        }
        qb.select("uo.id as userOrganizationId", "uo.userId", "uop.name", "uop.surname", "uop.uniqueId").orderBy("uop.name").orderBy("uop.surname");
        if (options.pagination) {
            if (options.pagination.skip) {
                qb.offset(options.pagination.skip);
                result.pagination.skip = options.pagination.skip;
            }
            if (options.pagination.take && options.pagination.take > 0) {
                qb.limit(options.pagination.take);
                result.pagination.take = options.pagination.take;
            }
            else {
                qb.limit(100);
                result.pagination.take = 100;
            }
        }
        result.items = (await qb).map((m) => {
            let fullname = [m.name.trim(), (m.surname || "").trim()].join(" ");
            return {
                id: m.userId,
                fullname: fullname,
                uniqueId: m.uniqueId,
                captionLines: [fullname, m.uniqueId],
            };
        });
        return Promise.resolve(result);
    }
    async filterFirstAndtLastAccessLogs(organizationId, filter, requesterUserId, authorizeForFilter) {
        let filteredUsers = await this.dbClient.transaction(async (trx) => {
            if (authorizeForFilter) {
                let authResultForFilter = await dal_manager_1.dbManager.accessFunctions.dbFuncAuthorizeUserFor({
                    organizationId: organizationId,
                    userId: requesterUserId,
                    organizationUnitIds: filter.organizationUnitIds,
                    accessControlPointIds: filter.accessControlPointIds,
                    userIds: filter.userIds,
                    userGroupIds: filter.userGroupIds,
                });
                if (!authResultForFilter.result) {
                    throw (0, api_error_1.generateForbiddenError)({ message: "Unauthorized filter items" });
                }
            }
            const userFilter = (0, dal_access_psql_common_1.getReportUserFilterForPgClient)({
                organizationId: organizationId,
                requesterUserId: requesterUserId,
                idBasedUserFilter: {
                    userIds: filter?.userIds,
                    userGroupIds: filter?.userGroupIds,
                    organizationUnitIds: filter?.organizationUnitIds,
                    userOrganizationStatus: filter.status,
                    applyOrganizationUnitFilterHierarchically: filter.filterOrganizationUnitMembersHierarchically,
                },
                requiredOrganizationUnitWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
                requiredOrganizationWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
                bindingKeys: [],
            });
            let queryFrom = `
						FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" uop
						INNER JOIN(${userFilter.query}) uf ON uf."userId" = uop."userId"`;
            return await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                const qtotal = `SELECT COUNT(*) AS c ` + queryFrom;
                const totalResult = await trx.query(qtotal, userFilter.bindingKeys);
                queryFrom += ` ORDER BY uop."name" ASC, uop."surname" ASC`;
                if (filter.pagination?.skip > 0) {
                    queryFrom += ` OFFSET $${userFilter.bindingKeys.length + 1}`;
                    userFilter.bindingKeys.push(filter.pagination.skip);
                }
                queryFrom += ` LIMIT $${userFilter.bindingKeys.length + 1}`;
                userFilter.bindingKeys.push(filter.pagination?.take || 100);
                let userList = (await trx.query(`SELECT uop."userId" as id, uop."name" || ' ' || uop."surname" as fullname, uop."uniqueId" ${queryFrom}`, userFilter.bindingKeys)).rows;
                userList.forEach((row) => {
                    row["captionLines"] = [row.fullname, row.uniqueId];
                });
                const result = {
                    pagination: {
                        total: totalResult.rowCount > 0 ? parseInt(totalResult.rows[0].c) : 0,
                        skip: filter.pagination?.skip || 0,
                        take: filter.pagination?.take || 100,
                    },
                    items: userList,
                };
                return result;
            });
        });
        let acpIds = filter.accessControlPointIds ? filter.accessControlPointIds : [];
        if (filter.regionIds && filter.regionIds.length > 0) {
            await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                acpIds.push(...(await dal_manager_1.dbManager.accessAccessControlPoint.listRegionsAccesControlPointIds(organizationId, filter.regionIds, trx)));
            });
        }
        let userIds = filteredUsers.items.map((m) => m.id);
        let userCaptions = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, userIds);
        let logs = await dal_manager_1.dbManager.accessLog.getFirstAndLastAccessLogsForUsersWithAcpId(organizationId, filter.dateRange, userIds, acpIds);
        acpIds.push(...logs.map((l) => l.firstAcpId));
        acpIds.push(...logs.map((l) => l.lastAcpId));
        acpIds = lodash_1.default.uniq(lodash_1.default.compact(acpIds));
        let acpInfos = [];
        if (acpIds.length > 0) {
            acpInfos = await this.dbClient.withSchema(organizationId).from(dal_db_armon_schema_1.ArmonSchema.tableNames.accessControlPoints).whereNull("deletedAt").whereIn("id", acpIds).select("id", "name");
        }
        let result = {
            pagination: filteredUsers.pagination,
            items: filteredUsers.items.map((mm) => {
                let log = logs.find((ll) => ll.userId === mm.id);
                let caption = userCaptions.find((u) => u.id == mm.id);
                let firstAcp = null;
                let lastAcp = null;
                if (log && log.firstAcpId) {
                    firstAcp = acpInfos.find((a) => a.id == log.firstAcpId);
                }
                if (log && log.lastAcpId) {
                    lastAcp = acpInfos.find((a) => a.id == log.lastAcpId);
                }
                return {
                    user: mm,
                    userCaptions: caption ? caption.captionLines : [],
                    firstAccessLogId: log ? log.firstAccessLogId : null,
                    firstUtc: log ? log.firstUtc : null,
                    lastUtc: log ? log.lastUtc : null,
                    lastAccessLogId: log ? log.lastAccessLogId : null,
                    firstAccessControlPoint: firstAcp,
                    lastAccessControlPoint: lastAcp,
                };
            }),
        };
        return Promise.resolve(result);
    }
    async filterFirstAndtLastAccessLogsForSingleUser(organizationId, filter, pagination) {
        let result = {
            items: [],
            user: null,
            userCaptions: [],
            pagination: {
                take: pagination.take,
                skip: pagination.skip,
                total: 0,
            },
        };
        let userInfo = await dal_manager_1.dbManager.accessUser.getBasicUserInfo(organizationId, filter.userId);
        result.user = userInfo;
        let userCaptions = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, [filter.userId]);
        result.userCaptions = userCaptions[0] ? userCaptions[0].captionLines : [];
        let startDate = (0, moment_1.default)(filter.dateRange.startDateTime).startOf("day");
        let endDate = (0, moment_1.default)(filter.dateRange.endDateTime).endOf("day");
        result.pagination.total = Math.abs(endDate.diff(startDate, "day") + 1);
        if (pagination.skip) {
            startDate = (0, moment_1.default)(startDate).add(pagination.skip, "day");
        }
        if (pagination.take && pagination.take < result.pagination.total) {
            endDate = (0, moment_1.default)(startDate).add(pagination.take, "day");
            if (endDate.isAfter(filter.dateRange.endDateTime, "day")) {
                endDate = (0, moment_1.default)(filter.dateRange.endDateTime).endOf("day");
            }
        }
        let acpIds = filter.accessControlPointIds ? filter.accessControlPointIds : [];
        if (filter.regionIds && filter.regionIds.length > 0) {
            await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                acpIds.push(...(await dal_manager_1.dbManager.accessAccessControlPoint.listRegionsAccesControlPointIds(organizationId, filter.regionIds, trx)));
            });
        }
        let logs = await dal_manager_1.dbManager.accessLog.getAccessLogsReport(organizationId, {
            userIds: [filter.userId],
            accessControlPointIds: acpIds,
            startUtc: startDate.toDate(),
            endUtc: endDate.toDate(),
            accessResult: dal_constants_1.DalConstants.AccessReportFilterAccessResultType.Success,
            pagination: {
                take: 10000,
                skip: 0,
            },
            sortDateDesc: true,
        }, false);
        acpIds.push(...logs.logs.map((l) => l.a));
        acpIds = lodash_1.default.uniq(lodash_1.default.compact(acpIds));
        acpIds = lodash_1.default.uniq(acpIds);
        let acpInfos = [];
        if (acpIds.length > 0) {
            acpInfos = await this.dbClient.withSchema(organizationId).from(dal_db_armon_schema_1.ArmonSchema.tableNames.accessControlPoints).whereNull("deletedAt").whereIn("id", acpIds).select("id", "name");
        }
        let groupedResults = lodash_1.default.groupBy(logs.logs, (log) => {
            return (0, moment_1.default)(log.u).format("YYYYMMDD");
        });
        for (let i = 0; i < Math.min(result.pagination.take, result.pagination.total - result.pagination.skip); i++) {
            let targetDate = (0, moment_1.default)(startDate).add(i, "day");
            let logsOfDay = groupedResults[targetDate.format("YYYYMMDD")];
            if (!logsOfDay || logsOfDay.length < 1) {
                result.items.push({
                    date: targetDate.endOf("day").toDate(),
                    firstUtc: null,
                    lastUtc: null,
                });
            }
            else {
                let firstLog = logsOfDay[logsOfDay.length - 1];
                let lastLog = logsOfDay[0];
                let firstAcp = acpInfos.find((a) => a.id == firstLog.a);
                let lastAcp = acpInfos.find((a) => a.id == lastLog.a);
                result.items.push({
                    date: targetDate.endOf("day").toDate(),
                    firstUtc: firstLog.u,
                    lastUtc: lastLog.u,
                    firstAccessControlPoint: firstAcp,
                    lastAccessControlPoint: lastAcp,
                });
            }
        }
        return Promise.resolve(result);
    }
    async getEntryByUnitData(filter) {
        const userFilter = (0, dal_access_psql_common_1.getReportUserFilterForPgClient)({
            organizationId: filter.organizationId,
            requesterUserId: filter.requesterUserId,
            idBasedUserFilter: {
                userGroupIds: filter?.userGroupIds,
                organizationUnitIds: filter?.organizationUnitIds,
                userOrganizationStatus: dal_constants_1.DalConstants.IdentityStatusType.All,
                applyOrganizationUnitFilterHierarchically: true,
            },
            requiredOrganizationUnitWidePermissions: [],
            requiredOrganizationWidePermissions: [],
            bindingKeys: [],
        });
        let queryParams = [...userFilter.bindingKeys, (0, moment_1.default)(filter.dateRange.startDateTime).format("YYYY-MM-DD"), (0, moment_1.default)(filter.dateRange.endDateTime).format("YYYY-MM-DD")];
        let bindingIndex = userFilter.bindingKeys.length + 1;
        let query = `
			WITH filtered_users AS (
				SELECT users."userId", uoou."organizationUnitId" as "unitId"
				FROM (
					SELECT uop."userId", uop."userOrganizationId"
					FROM "${filter.organizationId}"."userOrganizationProfiles" uop
					INNER JOIN(${userFilter.query}) uf ON uf."userId" = uop."userId"
				) users
				INNER JOIN "${filter.organizationId}"."userOrganizationOrganizationUnits" uoou
					ON uoou."userOrganizationId" = users."userOrganizationId" AND uoou."deletedAt" IS NULL
			),
			filtered_logs AS (
				SELECT "userId", date_trunc('day', "actionUtc")::date as al_date
				FROM (
					SELECT (log->>'o'):: uuid as "userId", (jsonb_array_elements(COALESCE(log->>'rg', '[{}]')::jsonb)->>'i')::uuid as region_id, al."actionUtc"
					FROM "${filter.organizationId}"."access_logs" al
					WHERE al."actionUtc"::date >= $${bindingIndex++} AND al."actionUtc"::date <= $${bindingIndex++} AND log->>'o' IS NOT NULL AND (log->>'s')::boolean
		`;
        if (filter?.accessControlPointIds?.length > 0) {
            query += `AND (log->>'a')::uuid = ANY($${bindingIndex++}::uuid[])`;
            queryParams.push(filter.accessControlPointIds);
        }
        query += `
				) ap_filtered
		`;
        if (filter?.regionIds?.length > 0) {
            query += `WHERE ap_filtered.region_id IS NOT NULL AND ap_filtered.region_id = ANY($${bindingIndex++}::uuid[])`;
            queryParams.push(filter.regionIds);
        }
        query += `
				GROUP BY "userId", al_date
			),
			filtered_units AS (
				SELECT id as "unitId", unnest(string_to_array("ancestorIds", ',')::uuid[] || id) as count_for
				FROM "${filter.organizationId}"."organizationUnits" ou
				WHERE ou."deletedAt" IS NULL
		`;
        if (filter?.organizationUnitIds?.length > 0) {
            query += `AND (id = ANY($${bindingIndex}::uuid[]) OR string_to_array("ancestorIds", ',')::uuid[] && $${bindingIndex++}::uuid[])`;
            queryParams.push(filter.organizationUnitIds);
        }
        query += `
			),
			selected_data AS (
				SELECT count_for as "unitId", al_date, COUNT(DISTINCT logs."userId") as count
				FROM filtered_units units
				LEFT JOIN filtered_users users
					ON units."unitId" = "users"."unitId"
				LEFT JOIN filtered_logs logs
					ON logs."userId" = users."userId"
				GROUP BY count_for, al_date
			),
			total_data AS (
				SELECT null as "unitId", al_date, COUNT(DISTINCT logs."userId") as count
				FROM filtered_users users
				INNER JOIN filtered_logs logs
					ON logs."userId" = users."userId"
				GROUP BY al_date
			)
			SELECT null as "unitId", null as "parentId", 'totals' as name, null as "rootPath", SUM(count)::integer as total, jsonb_agg(jsonb_build_object('date', al_date, 'count', count)) as "dayResults"
			FROM total_data
			UNION ALL
			SELECT unit_results."unitId", ou."parentId", ou.name, (string_to_array("ancestorIds", ',')::uuid[] || id) as "rootPath", unit_results.total, unit_results."dayResults"
			FROM (
				SELECT "unitId", SUM(count)::integer as total, jsonb_agg(jsonb_build_object('date', al_date, 'count', count)) FILTER (WHERE count > 0) as "dayResults"
				FROM selected_data
				GROUP BY "unitId"
			) unit_results
			INNER JOIN "${filter.organizationId}"."organizationUnits" ou
				ON ou.id = unit_results."unitId"
			ORDER BY "rootPath" ASC NULLS FIRST
		`;
        const { rows } = await dal_manager_1.dbManager.systemTransaction(async (trx) => {
            return await trx.query(query, queryParams);
        });
        let currentDate = new Date(filter.dateRange.startDateTime);
        let endDate = new Date(filter.dateRange.endDateTime);
        rows.forEach((item) => {
            while (currentDate <= endDate) {
                const dateString = (0, moment_1.default)(currentDate).format("YYYY-MM-DD").toString();
                if (!item.dayResults) {
                    item.dayResults = [];
                    item.dayResults.push({
                        date: dateString,
                        count: 0,
                    });
                    currentDate.setDate(currentDate.getDate() + 1);
                    continue;
                }
                const index = item.dayResults.map((e) => e.date.toString()).indexOf(dateString);
                if (index === -1) {
                    item.dayResults.push({
                        date: dateString,
                        count: 0,
                    });
                }
                currentDate.setDate(currentDate.getDate() + 1);
            }
            item.dayResults.sort((a, b) => new Date(a.date).getTime() - new Date(b.date).getTime());
            currentDate = new Date(filter.dateRange.startDateTime);
        });
        return {
            items: rows,
        };
    }
    async getCustomReportList(organizationId) {
        let result = (await this._pgPool.query(`
            SELECT id, name, filters, description, "userFilterId", formats, "minPermissions"
            FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.customReports}"
            ORDER BY "name"; 
        		`)).rows;
        return result;
    }
    async getCustomReport(id, organizationId) {
        let { rows, rowCount } = await this._pgPool.query(`
            SELECT "id", "name", "filters", "description", "userFilterId", "formats", "minPermissions"
            FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.customReports}"
            WHERE "id" = $1; 
        `, [id]);
        if (rowCount > 0) {
            return rows[0];
        }
        return null;
    }
    async getExpiredPreparedReportList(params) {
        const { rows, rowCount } = await params.trx.query(`
            SELECT id             
            FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.preparedReports}"
            WHERE "expirationDt" < now()
        `);
        return rowCount > 0 ? rows.map((r) => r.id) : [];
    }
    async getPreparedReport(reportId, organizationId) {
        const { rows, rowCount } = await this._pgPool.query(`
		SELECT "createdDt", "expirationDt", f.filter AS "userFilter", format
		FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.preparedReports}" AS pr
		LEFT JOIN "${organizationId}".${dal_db_armon_schema_1.ArmonSchema.viewNames.vwUserFilter} as f 
		ON f.id = pr."userFilterId"
        WHERE pr.id = $1
        `, [reportId]);
        if (rowCount === 0) {
            return null;
        }
        return rows[0];
    }
    async insertPreparedReport(params) {
        await params.trx.query(`
            INSERT INTO "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.preparedReports}"
            (id, "createdDt", "expirationDt", "userFilterId", format)
                VALUES ($1, $2, $3, $4, $5);
            `, [params.reportId, new Date(), params.expirationDt, params.userFilterId, params.format]);
    }
    async getCardActivityDetailByLogId(organizationId, logId) {
        let logInfo = await dal_manager_1.dbManager.accessLog.getLogDetailInfoById(organizationId, logId);
        if (!logInfo)
            Promise.reject("not found");
        let accessControlPoint = await dal_manager_1.dbManager.accessAccessControlPoint.getAccessControlPointDetailesV2(organizationId, logInfo.log.a);
        let organizationUnitsColumn = this.dbClient
            .raw(`
            to_json( ARRAY(select json_build_object('id', ou.id, 'name', ou.name) 
            from "${organizationId}"."userOrganizationOrganizationUnits" as uoou 
            inner join "${organizationId}"."organizationUnits" as ou on uoou."organizationUnitId" = ou.id
            where uoou."userOrganizationId" = uo.id and uoou."deletedAt" is null
            order by ou.name))
            `)
            .wrap("(", ') "organizationUnits"');
        let qbResultItems = this.dbClient
            .withSchema(organizationId)
            .from("userOrganizations as uo")
            .innerJoin("users as u", "u.id", "uo.userId")
            .innerJoin("userOrganizationProfiles as uop", "uop.userOrganizationId", "uo.id")
            .where("u.id", logInfo.log.o)
            .whereNull("uo.deletedAt")
            .whereNull("uop.deletedAt")
            .first("uo.userId", "uop.name", "uop.surname", "uop.uniqueId", organizationUnitsColumn);
        let identity = await qbResultItems;
        let reader = null;
        if (logInfo.log.re !== null && accessControlPoint && accessControlPoint.wiegandReaders) {
            let wr = accessControlPoint.wiegandReaders.find((wr) => wr.deviceId === logInfo.log.cid && wr.readerWiegandInputNumber === logInfo.log.re);
            if (wr) {
                reader = {
                    id: wr.id,
                    name: wr.name,
                    location: wr.location,
                };
            }
        }
        let result = {
            id: logId,
            utc: new Date(logInfo.log.u),
            accessRightId: logInfo.log.i,
            credentialType: null,
            result: logInfo.log.r,
            direction: logInfo.log.d,
            isVisitor: logInfo.log.ul === predefined_roles_1.PredefinedRoles.OrganizationVisitor.id || logInfo.log.ul === predefined_roles_1.PredefinedRoles.OrganizationUnitVisitor.id,
            isManuallyLogged: logInfo.log.m !== null,
            accessControlPoint: {
                id: logInfo.log.a,
                name: accessControlPoint.name,
                location: accessControlPoint.location,
                reader: reader,
            },
            identity: identity
                ? {
                    id: identity.userId,
                    name: identity.name,
                    surname: identity.surname,
                    uniqueId: identity.uniqueId,
                    organizationUnits: identity.organizationUnits,
                }
                : null,
            isExitButton: logInfo.log.di !== null,
            isRemoteAccess: logInfo.log.ir !== null,
            logAccessControlPointInfo: logInfo.log.an || "",
            logUserInfo: (logInfo.log.on || "") + (logInfo.log.c ? " (" + logInfo.log.c + ")" : ""),
            snapshots: logInfo.snapshotInfo,
        };
        return Promise.resolve(result);
    }
    async listReportTemplates(params) {
        const { filter, sorting, organizationId, trx, pagination } = params;
        let qx = 1;
        const qb = [];
        const qw = [];
        let qfrom = `
            FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.reportTemplates}" rt
            LEFT JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.reportScheduledJobs}" rsj
			ON rt.id = rsj."reportTemplateId"
			`;
        if (filter) {
            if (filter.createdByUserId) {
                qw.push(`rt."userId" = $${qx++}`);
                qb.push(filter.createdByUserId);
            }
            if (filter.name) {
                qw.push(`rt.name ilike $${qx++}`);
                qb.push(filter.name);
            }
            if (filter.customReportId) {
                qw.push(`rt."customReportId" = $${qx++}`);
                qb.push(filter.customReportId);
            }
            if (filter.types?.length > 0) {
                const qt = [];
                for (const t of filter.types) {
                    qt.push(`$${qx++}`);
                    qb.push(t);
                }
                qw.push(`rt.type IN (${qt.join(",")}::integer)`);
            }
        }
        if (qw.length > 0) {
            qfrom +=
                `
			WHERE ` + qw.join(" AND ");
        }
        qfrom += `
			GROUP BY rt.id, rt."createdT", rt.name, rt."userId", rt.type, rt.filter
		`;
        const qtotal = `SELECT COUNT(*) AS c ` + qfrom;
        const totalResult = await trx.query(qtotal, qb);
        const result = {
            pagination: {
                total: totalResult.rowCount > 0 ? parseInt(totalResult.rows[0].c) : 0,
                skip: pagination?.skip || 0,
                take: pagination?.take || 100,
            },
            items: [],
        };
        if (result.pagination.total === 0) {
            return result;
        }
        if (sorting) {
            qfrom += `
			ORDER BY "${sorting.field}" ${sorting.order ? "DESC" : "ASC"}`;
        }
        if (pagination?.skip > 0) {
            qfrom += `
                OFFSET $${qx++}`;
            qb.push(pagination.skip);
        }
        qfrom += `
                LIMIT $${qx++}`;
        qb.push(pagination?.take || 100);
        const qlist = `SELECT rt.id, rt."createdT", rt.name, rt."userId", 
		rt.type, rt.filter, array_agg(rsj.id) as "scheduledJobIds" ` + qfrom;
        const { rows } = await trx.query(qlist, qb);
        result.items = await Promise.all(rows.map(async (row) => {
            try {
                let userCaption;
                if (row.userId) {
                    userCaption = await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId, userId: row.userId });
                }
                return {
                    id: row.id,
                    createdT: row.createdT,
                    name: row.name,
                    userId: row.userId
                        ? {
                            id: row.userId,
                            caption: userCaption.caption,
                        }
                        : null,
                    type: row.type,
                    filter: row.filter,
                    scheduledJobIds: row.scheduledJobIds.filter(Boolean),
                };
            }
            catch (error) {
                return null;
            }
        }));
        return result;
    }
    async getReportTemplateViaNotificationId(params) {
        const { rows, rowCount } = await (params.trx ?? this._pgPool).query(`
			SELECT rt.id, rt.type, rt."userId", rt."createdT", rt.name, rt.filter, sj.id as "scheduledJobId"
			FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.notification}" n
			INNER JOIN public."${dal_db_armon_schema_1.ArmonSchema.tableNames.scheduled_job}" sj
			ON sj."notificationId" = n.id
			INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.reportScheduledJobs}" rsj
			ON sj.id = rsj.id
			INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.reportTemplates}" rt
			ON rt.id = rsj."reportTemplateId"
			WHERE sj."notificationId" = $1`, [params.notificationId]);
        if (rowCount < 1) {
            return null;
        }
        else {
            return rows[0];
        }
    }
    async getReportTemplate(organizationId, reportTemplateId, trx) {
        const trxx = trx ?? this._pgPool;
        const { rows, rowCount } = await trxx.query(`
			SELECT rt.type, rt."userId", rt."createdT", rt.name, rt.filter, rsj.id as "scheduledJobId"
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.reportTemplates}" rt
			LEFT JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.reportScheduledJobs}" rsj
			ON rt.id = rsj."reportTemplateId"
			WHERE rt.id = $1`, [reportTemplateId]);
        if (rowCount < 1) {
            return null;
        }
        else {
            const row = rows[0];
            return {
                createdT: row.createdT,
                name: row.name,
                userId: row.userId,
                type: row.type,
                filter: row.filter,
                scheduledJobId: row.scheduledJobId,
            };
        }
    }
    async getReportTemplateForScheduledReport(params) {
        const { rows, rowCount } = await (params.trx ?? this._pgPool).query(`
		SELECT rsj.format, rsj."sendWhenEmpty", rt.type, rt.filter, rt.name, rt."userId", sj.interval, 
		rsj."reportTemplateId", sj."organizationId", n."receiverFilterId", sj."createdByUserId", rt."customReportId"
		FROM public."${dal_db_armon_schema_1.ArmonSchema.tableNames.scheduled_job}" sj
		INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.reportScheduledJobs}" rsj
		ON sj.id = rsj.id 
		INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.reportTemplates}" rt
		ON rsj."reportTemplateId" = rt.id
		INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.notification}" n
		ON n.id = sj."notificationId"
		WHERE sj.id = $1`, [params.scheduledJobId]);
        if (rowCount > 0) {
            return rows[0];
        }
        else {
            return null;
        }
    }
    async addReportTemplate(params) {
        const id = uuid_1.default.v4();
        await (params.trx ?? this._pgPool).query(`INSERT INTO "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.reportTemplates}"
				(id, "createdT", name, "userId", type, filter, "customReportId")
			VALUES($1, $2, $3, $4, $5, $6, $7)
				`, [id, new Date(), params.name, params.userId, params.type, params.filter, params.customReportId]);
        return id;
    }
    async updateReportTemplate(params) {
        const trxx = params.trx ?? this._pgPool;
        const reportTemplate = await this.getReportTemplate(params.organizationId, params.reportTemplateId, params.trx);
        if (params.filter) {
            reportTemplate.filter = params.filter;
        }
        if (params.name) {
            reportTemplate.name = params.name;
        }
        await trxx.query(`UPDATE "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.reportTemplates}"
			SET name = $2,
				filter = $3
			WHERE id = $1`, [params.reportTemplateId, reportTemplate.name, reportTemplate.filter]);
    }
    async deleteReportTemplate(params) {
        const { rows } = await params.trx.query(`SELECT id FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.reportScheduledJobs}"
			WHERE "reportTemplateId" = $1`, [params.reportTemplateId]);
        if (rows.length === 0) {
            await params.trx.query(`DELETE FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.reportTemplates}"
				WHERE id = $1`, [params.reportTemplateId]);
        }
        else if (rows.length > 0 && params.force) {
            await params.trx.query(`DELETE FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.reportTemplates}"
				WHERE id = $1`, [params.reportTemplateId]);
            await params.trx.query(`DELETE FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.notification}"
				WHERE id IN (SELECT "notificationId" FROM public.scheduled_job WHERE id = ANY($1::uuid[]))`, [rows.map((row) => row.id)]);
        }
        return rows.length;
    }
}
exports.PSQLDalAccessReport = PSQLDalAccessReport;
