"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PSQLDalAccessPacs = void 0;
const moment_1 = __importDefault(require("moment"));
const uuid_1 = __importDefault(require("uuid"));
const app_enums_1 = require("../../../app.enums");
const business_hooks_1 = require("../../../business/business.hooks");
const business_pacs_ppermission_1 = require("../../../business/pacs/business.pacs.ppermission");
const dal_constants_1 = require("../../dal.constants");
const dal_manager_1 = require("../../dal.manager");
const dal_utils_1 = require("../../dal.utils");
const dal_db_armon_schema_1 = require("../../db/armon/dal.db.armon.schema");
const predefined_permissions_1 = require("../../db/predefined/predefined.permissions");
const predefined_roles_1 = require("../../db/predefined/predefined.roles");
const dal_access_error_1 = require("../dal.access.error");
const dal_access_rdb_pacs_1 = require("../rdb/dal.access.rdb.pacs");
const dal_access_psql_common_1 = require("./dal.access.psql.common");
class PSQLDalAccessPacs extends dal_access_rdb_pacs_1.RDBDalAccessPacs {
    async upsertVacation(organizationId, args) {
        let id = args.vacationId || uuid_1.default.v4();
        let countOfIntersectionTstz;
        await this.dbClient.transaction(async (trx) => {
            let qb = this.dbClient
                .withSchema(organizationId)
                .table("vacations")
                .select(this.dbClient.raw("COUNT (*) > 0 as exists"))
                .where(this.dbClient.raw('tstzrange("startDateTime","endDateTime") && tstzrange(?,?)', [args.startDateTime, args.endDateTime]));
            if (args.vacationId) {
                qb.where("id", "<>", id);
            }
            await qb.then((row) => (countOfIntersectionTstz = row[0].exists));
            if (+countOfIntersectionTstz > 0) {
                throw (0, dal_access_error_1.throwDbAccessConflictErrorTr)("ERRORS.PACS.INTERSECTVACATIONDATES");
            }
            if (args.vacationId) {
                await this.dbClient.withSchema(organizationId).table("vacations").where("id", id).update({
                    organizationId: organizationId,
                    startDateTime: args.startDateTime,
                    endDateTime: args.endDateTime,
                    title: args.title,
                    type: args.type,
                });
            }
            else {
                await this.dbClient.withSchema(organizationId).table("vacations").insert({
                    id: id,
                    organizationId: organizationId,
                    startDateTime: args.startDateTime,
                    endDateTime: args.endDateTime,
                    title: args.title,
                    type: args.type,
                });
            }
        });
        return Promise.resolve({
            id: id,
            startDateTime: args.startDateTime,
            endDateTime: args.endDateTime,
            title: args.title,
            type: args.type,
        });
    }
    async getVacation(organizationId, vacationId, trx) {
        let result = {
            id: "",
            startDateTime: null,
            endDateTime: null,
            title: "",
            type: app_enums_1.enums.VacationType.National,
        };
        let qb = this.dbClient.withSchema(organizationId).table("vacations").where("organizationId", organizationId).where("id", vacationId);
        if (trx)
            qb.transacting(trx);
        await qb
            .select()
            .first()
            .then((row) => {
            if (row) {
                (result.id = row.id), (result.startDateTime = row.startDateTime), (result.endDateTime = row.endDateTime), (result.title = row.title), (result.type = row.type);
            }
        });
        return Promise.resolve(result);
    }
    async removeVacation(organizationId, vacationId) {
        let affectedRowCount = 0;
        await this.dbClient.transaction(async (trx) => {
            let vacation = await this.getVacation(organizationId, vacationId, trx);
            await trx
                .withSchema(organizationId)
                .table("vacations")
                .where("id", vacationId)
                .del()
                .then((result) => (affectedRowCount = result));
        });
        return Promise.resolve(affectedRowCount > 0);
    }
    async listAllVacations(organizationId, trx) {
        let vacationList = { items: [] };
        let qb = this.dbClient.withSchema(organizationId).table("vacations").where("organizationId", organizationId).orderBy("startDateTime", "asc");
        if (trx)
            qb.transacting(trx);
        await qb.select().then(function (result) {
            if (result) {
                result.forEach((element) => {
                    vacationList.items.push({
                        id: element.id,
                        title: element.title,
                        type: element.type,
                        startDateTime: element.startDateTime,
                        endDateTime: element.endDateTime,
                    });
                });
            }
        });
        return vacationList;
    }
    async listVacations(organizationId, year) {
        let vacationList = { items: [] };
        let startDateTime = (0, moment_1.default)(new Date(year, 0, 1)).startOf("year").toDate();
        let endDateTime = (0, moment_1.default)(new Date(year, 0, 1)).startOf("year").add(1, "year").toDate();
        await this.dbClient
            .withSchema(organizationId)
            .table("vacations")
            .where(this.dbClient.raw('tstzrange("startDateTime","endDateTime") && tstzrange(?,?)', [startDateTime, endDateTime]))
            .where("organizationId", organizationId)
            .orderBy("startDateTime", "asc")
            .select()
            .then(function (result) {
            if (result) {
                result.forEach((element) => {
                    vacationList.items.push({
                        id: element.id,
                        title: element.title,
                        type: element.type,
                        startDateTime: element.startDateTime,
                        endDateTime: element.endDateTime,
                    });
                });
            }
        });
        return vacationList;
    }
    async listWorkPlans(trx, organizationId, requestUserId, type, name) {
        let query = `
		SELECT wp.*, reg.name as "regionName", ou.name as "organizationUnitName",
			COALESCE(POSITION($2::text IN gr.permissions) > 0 OR POSITION($2::text IN r.permissions) > 0, false) as "isEditable"
		FROM "${organizationId}"."workPlans" wp
		INNER JOIN "${organizationId}"."userOrganizations" uo
			ON wp."deletedAt" IS NULL AND uo."deletedAt" IS NULL AND NOT uo."isDisabled" AND uo."userId" = $1
		INNER JOIN "${organizationId}"."roles" gr
			ON gr."deletedAt" IS NULL AND gr.id = uo."roleId"
		LEFT JOIN "${organizationId}"."regions" reg
			ON reg.id = wp."regionId"
		LEFT JOIN "${organizationId}"."organizationUnits" ou
			ON wp."organizationUnitId" = ou.id AND ou."deletedAt" IS NULL AND wp."deletedAt" IS NULL
		LEFT JOIN "${organizationId}"."userOrganizationOrganizationUnits" uoou
			ON uoou."deletedAt" IS NULL AND uoou."userOrganizationId" = uo.id AND (uoou."organizationUnitId" = ou.id OR POSITION(uoou."organizationUnitId"::text IN ou."ancestorIds") > 0)
		LEFT JOIN "${organizationId}"."roles" r
			ON r."deletedAt" IS NULL AND r.id = uoou."roleId"
		WHERE (POSITION($3::text IN gr.permissions) > 0 OR POSITION($3::text IN r.permissions) > 0)
		`;
        let bindings = [requestUserId, predefined_permissions_1.Permissions.attendancePlan.getWrite(), predefined_permissions_1.Permissions.attendancePlan.getRead()];
        if (name) {
            bindings.push(name);
            query += `AND wp.name ILIKE '%' || $${bindings.length} || '%'`;
        }
        if (type) {
            bindings.push(type);
            query += `
			AND wp.type = $${bindings.length}::smallint;`;
        }
        let dbResults = await trx.query(query, bindings);
        let workPlanList = {
            items: [],
        };
        for (let row of dbResults.rows) {
            let existingItem = workPlanList.items.find((i) => i.id === row.id);
            if (!existingItem) {
                let item = {
                    id: row.id,
                    name: row.name,
                    type: row.type,
                    assignedUserCount: await this.getWorkPlanAssignmentCount(organizationId, row.id),
                    regionId: row.regionId,
                    regionName: row.regionName,
                    accessCheckType: row.accessCheckType,
                    offset: row.offset,
                    ignoreHolidays: row.ignoreHolidays === null || row.ignoreHolidays ? true : row.ignoreHolidays,
                    allowMobileCheckins: row.allowMobileCheckins,
                    mobileCheckinRequiresLocation: row.mobileCheckinRequiresLocation,
                    allowUnreliableCheckins: row.allowUnreliableCheckins,
                    permissionRequiredForExtraWorking: row.permissionRequiredForExtraWorking,
                    geoLocationCount: row.typedGeoLocations?.length ?? 0,
                    organizationUnitId: row.organizationUnitId,
                    organizationUnitName: row.organizationUnitName,
                    isEditable: row.isEditable,
                    colorCode: row.colorCode,
                };
                workPlanList.items.push(item);
            }
            else {
                existingItem.isEditable = existingItem.isEditable || row.isEditable;
            }
        }
        const workPlanRegionIds = workPlanList.items.map((workPlan) => workPlan.regionId);
        const regionsWithAdministrators = await dal_manager_1.dbManager.accessRegion.getAdministratorsOfRegions({
            organizationId,
            regionIds: workPlanRegionIds,
            trx,
        });
        const regionIdsThatUserCanSee = [];
        for (const workPlanRegionId of workPlanRegionIds) {
            const regionWithAdmins = regionsWithAdministrators.find((r) => r.id === workPlanRegionId);
            if (regionWithAdmins) {
                const userRightsForRegion = regionWithAdmins.userRegionRights.find((urr) => urr.userId === requestUserId);
                if (!userRightsForRegion || !userRightsForRegion.userRights.read) {
                    continue;
                }
            }
            regionIdsThatUserCanSee.push(workPlanRegionId);
        }
        const filteredWorkPlanList = {
            items: [],
        };
        filteredWorkPlanList.items = workPlanList.items.filter((workPlan) => {
            if (workPlan.regionId && !regionIdsThatUserCanSee.includes(workPlan.regionId)) {
                return false;
            }
            return true;
        });
        return filteredWorkPlanList;
    }
    async listWorkPlanByIds(organizationId, workPlanIds) {
        let workPlanList = {
            items: [],
        };
        let qb = this.dbClient
            .withSchema(organizationId)
            .table("workPlans as wp")
            .leftJoin("regions as r", "r.id", "wp.regionId")
            .leftJoin("organizationUnits as ou", "ou.id", "wp.organizationUnitId")
            .whereIn("wp.id", workPlanIds)
            .whereNull("wp.deletedAt")
            .orderBy("wp.name", "asc");
        await qb
            .select("wp.id", "wp.name", "wp.type", "wp.type", "r.id as regionId", "r.name as regionName", "wp.offset", "wp.accessCheckType", "wp.ignoreHolidays", "wp.allowMobileCheckins", "wp.mobileCheckinRequiresLocation", "wp.allowUnreliableCheckins", "wp.permissionRequiredForExtraWorking", "wp.colorCode", "wp.typedGeoLocations", "wp.organizationUnitId", "ou.name as organizationUnitName")
            .then(async (rows) => {
            for (let row of rows) {
                let item = {
                    id: row.id,
                    name: row.name,
                    type: row.type,
                    assignedUserCount: await this.getWorkPlanAssignmentCount(organizationId, row.id),
                    regionId: row.regionId,
                    regionName: row.regionName,
                    accessCheckType: row.accessCheckType,
                    offset: row.offset,
                    ignoreHolidays: row.ignoreHolidays === null || row.ignoreHolidays ? true : row.ignoreHolidays,
                    allowMobileCheckins: row.allowMobileCheckins,
                    mobileCheckinRequiresLocation: row.mobileCheckinRequiresLocation,
                    allowUnreliableCheckins: row.allowUnreliableCheckins,
                    permissionRequiredForExtraWorking: row.permissionRequiredForExtraWorking,
                    geoLocationCount: row.typedGeoLocations?.length ?? 0,
                    colorCode: row.colorCode,
                    organizationUnitId: row.organizationUnitId,
                    organizationUnitName: row.organizationUnitName,
                    isEditable: false,
                };
                workPlanList.items.push(item);
            }
        });
        return workPlanList;
    }
    async getWorkPlanAssignmentCount(organizationId, workPlanId, at) {
        let count = 0;
        await this.dbClient
            .raw(`
        SELECT COUNT(*) as count FROM "${organizationId}"."userWorkPlans" uwp
        INNER JOIN "${organizationId}"."workPlans" wp ON wp.id = uwp."workPlanId" AND wp.id = ? AND wp."organizationId" = ?
        INNER JOIN "${organizationId}"."users" u ON u.id = uwp."userId" AND u."deletedAt" IS NULL
        INNER JOIN "${organizationId}"."userOrganizations" uo
            ON uo."deletedAt" IS NULL AND uo."isDisabled" = false AND uo."userId" = u.id
            AND uo."organizationId" = wp."organizationId"
        INNER JOIN "${organizationId}"."userOrganizationProfiles" uop ON uop."userOrganizationId" = uo.id and uop."deletedAt" IS NULL
        WHERE tstzrange(uwp."startDateTime", uwp."endDateTime") @> ?::timestamp with time zone
        `, [workPlanId, organizationId, at ? at : new Date()])
            .then((result) => {
            if (result.rows[0])
                count = parseInt(result.rows[0].count);
        });
        return Promise.resolve(count);
    }
    async getWorkPlan(trx, organizationId, requestUserId, workplanId) {
        let result = {
            id: workplanId,
            name: "",
            type: 0,
            workPlanFullFlexiblePeriods: [],
            workPlanHalfFlexiblePeriods: [],
            workPlanRegularPeriods: [],
            workPlanShiftPeriods: [],
            regionId: "",
            regionName: "",
            accessCheckType: dal_constants_1.DalConstants.WorkPlanAccessCheckType.Region,
            offset: 0,
            allowMobileCheckins: false,
            mobileCheckinRequiresLocation: false,
            allowUnreliableCheckins: false,
            permissionRequiredForExtraWorking: false,
            organizationUnitId: null,
            isEditable: false,
            colorCode: "000000",
            ignoreUnpairedAccesses: false,
        };
        let query = `
		SELECT wp.*, reg.name as "regionName", ou.name as "organizationUnitName",
			COALESCE(POSITION($3::text IN gr.permissions) > 0 OR POSITION($3::text IN r.permissions) > 0, false) as "isEditable"
		FROM "${organizationId}"."workPlans" wp
		INNER JOIN "${organizationId}"."userOrganizations" uo
			ON wp."deletedAt" IS NULL AND wp."id" = $1 AND uo."deletedAt" IS NULL AND NOT uo."isDisabled" AND uo."userId" = $2
		INNER JOIN "${organizationId}"."roles" gr
			ON gr."deletedAt" IS NULL AND gr.id = uo."roleId"
		LEFT JOIN "${organizationId}"."regions" reg
			ON reg.id = wp."regionId"
		LEFT JOIN "${organizationId}"."organizationUnits" ou
			ON wp."organizationUnitId" = ou.id AND ou."deletedAt" IS NULL AND wp."deletedAt" IS NULL
		LEFT JOIN "${organizationId}"."userOrganizationOrganizationUnits" uoou
			ON uoou."deletedAt" IS NULL AND uoou."userOrganizationId" = uo.id AND (uoou."organizationUnitId" = ou.id OR POSITION(uoou."organizationUnitId"::text IN ou."ancestorIds") > 0)
		LEFT JOIN "${organizationId}"."roles" r
			ON r."deletedAt" IS NULL AND r.id = uoou."roleId"
		WHERE POSITION($4::text IN gr.permissions) > 0 OR POSITION($4::text IN r.permissions) > 0
		`;
        let bindings = [workplanId, requestUserId, predefined_permissions_1.Permissions.attendancePlan.getWrite(), predefined_permissions_1.Permissions.attendancePlan.getRead()];
        let wpResult = (await trx.query(query, bindings)).rows[0];
        if (wpResult) {
            result.name = wpResult.name;
            result.type = wpResult.type;
            result.regionId = wpResult.regionId;
            result.regionName = wpResult.regionName;
            result.accessCheckType = wpResult.accessCheckType;
            result.offset = wpResult.offset;
            result.timezone = wpResult.timezone;
            result.ignoreHolidays = wpResult.ignoreHolidays;
            result.allowMobileCheckins = wpResult.allowMobileCheckins;
            result.mobileCheckinRequiresLocation = wpResult.mobileCheckinRequiresLocation;
            result.allowUnreliableCheckins = wpResult.allowUnreliableCheckins;
            result.permissionRequiredForExtraWorking = wpResult.permissionRequiredForExtraWorking;
            result.geoLocations = wpResult.typedGeoLocations
                ?.filter((g) => g.t === dal_constants_1.DalConstants.GeoLocationType.Circle)
                ?.map((g) => {
                return {
                    longitude: g.lo,
                    latitude: g.la,
                    radius: g.r,
                    name: g.n,
                };
            });
            (result.typedGeoLocations = wpResult.typedGeoLocations?.map((g) => {
                if (g.t === dal_constants_1.DalConstants.GeoLocationType.Circle) {
                    return {
                        type: g.t,
                        longitude: g.lo,
                        latitude: g.la,
                        radius: g.r,
                        name: g.n,
                    };
                }
                else {
                    return {
                        type: g.t,
                        name: g.n,
                        points: g.p.map((point) => {
                            return {
                                longitude: point.lo,
                                latitude: point.la,
                            };
                        }),
                    };
                }
            })),
                (result.organizationUnitId = wpResult.organizationUnitId);
            result.isEditable = wpResult.isEditable;
            result.colorCode = wpResult.colorCode;
            result.ignoreUnpairedAccesses = wpResult.ignoreUnpairedAccesses;
            let periods = await trx.query(`
			SELECT * FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.workPlanPeriods}"
			WHERE "workPlanId" = $1 AND "deletedAt" IS NULL
			ORDER BY "periodStartDateTime" DESC;
			`, [workplanId]);
            for (const element of periods.rows) {
                let detail = Object.assign({
                    id: element.id,
                    workPlanId: workplanId,
                    periodStartDateTime: element.periodStartDateTime,
                    offset: element.offset,
                }, element.details);
                switch (result.type) {
                    case dal_constants_1.DalConstants.WorkPlanType.SHIFT:
                        result.workPlanShiftPeriods.push(detail);
                        break;
                    case dal_constants_1.DalConstants.WorkPlanType.REGULAR:
                        result.workPlanRegularPeriods.push(detail);
                        break;
                    case dal_constants_1.DalConstants.WorkPlanType.HALF_FLEXIBLE:
                        result.workPlanHalfFlexiblePeriods.push(detail);
                        break;
                    case dal_constants_1.DalConstants.WorkPlanType.FULL_FLEXIBLE:
                        result.workPlanFullFlexiblePeriods.push(detail);
                        break;
                    default:
                }
            }
        }
        return Promise.resolve(result);
    }
    async getBasicUserInfo(organizationId, userId, trx) {
        if (!userId)
            return Promise.resolve(null);
        let result = {
            id: "",
            fullname: "",
            uniqueId: "",
            captionLines: [],
        };
        let qb = this.dbClient
            .withSchema(organizationId)
            .from("userOrganizationProfiles as uop")
            .innerJoin("userOrganizations as uo", "uo.id", "uop.userOrganizationId")
            .where("uo.organizationId", organizationId)
            .where("uo.userId", userId)
            .whereNull("uop.deletedAt");
        if (trx)
            qb.transacting(trx);
        let { caption: requestedUserCaption } = await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId: organizationId, userId: userId });
        await qb
            .select(["uo.userId", "uop.name", "uop.surname", "uop.uniqueId"])
            .first()
            .then(async (row) => {
            if (row) {
                result.id = row.userId;
                result.fullname = (row.name ?? "") + " " + (row.surname ?? "");
                result.uniqueId = row.uniqueId;
                result.captionLines = requestedUserCaption;
            }
        });
        return Promise.resolve(result);
    }
    async getBasicUserInfoList(organizationId, userIdList, trx) {
        if (!userIdList)
            return Promise.resolve(null);
        let result = [];
        let qb = this.dbClient
            .withSchema(organizationId)
            .from("userOrganizationProfiles as uop")
            .innerJoin("userOrganizations as uo", "uo.id", "uop.userOrganizationId")
            .where("uo.organizationId", organizationId)
            .whereIn("uo.userId", userIdList)
            .whereNull("uop.deletedAt");
        if (trx)
            qb.transacting(trx);
        await qb.select(["uo.userId", "uop.name", "uop.surname", "uop.uniqueId"]).then(async (rows) => {
            for (let row of rows) {
                let { caption: requestedUserCaption } = await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId: organizationId, userId: row.userId });
                result.push({
                    id: row.userId,
                    fullname: (row.name ?? "") + " " + (row.surname ?? ""),
                    uniqueId: row.uniqueId,
                    captionLines: requestedUserCaption,
                });
            }
        });
        return Promise.resolve(result);
    }
    async removePPermissionDeputyInfo(organizationId, ppermissionId, trx) {
        if (trx) {
            await this.dbClient.withSchema(organizationId).from("userOrganizationDeputies").where("ppermissionId", ppermissionId).transacting(trx).del();
        }
        else {
            await this.dbClient.withSchema(organizationId).from("userOrganizationDeputies").where("ppermissionId", ppermissionId).del();
        }
        return Promise.resolve();
    }
    async getUserOrganizationUnitList(organizationId, userId, rootOrganizationUnitId, trx) {
        let result = [];
        let qb = this.dbClient
            .withSchema(organizationId)
            .table("userOrganizations as uo")
            .innerJoin("userOrganizationOrganizationUnits as uoou", (join) => {
            join.on("uoou.userOrganizationId", "uo.id");
        })
            .innerJoin("organizationUnits as ou", (join) => {
            join.on("ou.id", "uoou.organizationUnitId");
        })
            .where("uo.userId", userId)
            .where("uo.organizationId", organizationId)
            .whereNull("uoou.deletedAt")
            .whereNull("ou.deletedAt")
            .whereNull("uo.deletedAt");
        if (rootOrganizationUnitId) {
            qb = qb.whereRaw('(ou."ancestorIds" like ?  or (ou."ancestorIds" is null and ou."id" = ?))', [rootOrganizationUnitId + "%", rootOrganizationUnitId]);
        }
        if (trx)
            qb.transacting(trx);
        await qb.select(["ou.id", "ou.ancestorIds", "ou.managerUserId"]).then((rows) => {
            for (let row of rows) {
                result.push({
                    organizationUnitId: row.id,
                    ancestorIds: row.ancestorIds,
                    managerUserId: row.managerUserId,
                });
            }
        });
        return Promise.resolve(result);
    }
    async getPairOfOrganizationUnitAndManager(organizationId, trx) {
        let result = [];
        let qb = this.dbClient.withSchema(organizationId).table("organizationUnits as ou").whereNotNull("ou.managerUserId").whereNull("ou.deletedAt");
        if (trx)
            qb.transacting(trx);
        await qb.select("ou.id", "ou.managerUserId").then((rows) => {
            for (let row of rows) {
                result.push({
                    managerUserId: row.managerUserId,
                    organizationUnitId: row.id,
                });
            }
        });
        return Promise.resolve(result);
    }
    async getPairOfOrganizationUnitAndManagerPg(organizationId, trx) {
        let result = (await trx.query(`
			SELECT	ou.id as "organizationUnitId",
					ou."managerUserId"
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" as ou
			WHERE ou."managerUserId" IS NOT NULL AND ou."deletedAt" IS NULL
		`)).rows;
        return result;
    }
    async getPairOfOrganizationUnitAndHRPg(organizationId, trx) {
        let result = {};
        (await trx.query(`
				SELECT	uo."userId" as "hrUserId",
						uoou."organizationUnitId"
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" as uo
				INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}" as uoou
					ON uo.id = uoou."userOrganizationId"
				INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.roles}" as r
					ON r.id = uoou."roleId"
				WHERE r.permissions ILIKE $1
				`, ["%" + predefined_permissions_1.Permissions.ppermission.getHRApprovement() + "%"])).rows.forEach((elem) => (result[elem.organizationUnitId] = elem.hrUserId));
        return result;
    }
    async getOrganizationWideHRsPg(organizationId, trx) {
        const hrIds = (await trx.query(`
			SELECT "userId"
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.roles}" as r
			INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" as uo
				ON r.id = uo."roleId"
			WHERE r."typeId" = '${predefined_roles_1.PredefinedRoles.OrganizationWideHumanResourcesManager.id}'
		`)).rows.map((elem) => elem.userId);
        return hrIds;
    }
    async getUserOrganizationUnitsOfLevel(organizationId, userId, order, rootOrganizationUnitId, trx) {
        let result = await this.getUserOrganizationUnitList(organizationId, userId, rootOrganizationUnitId, trx);
        if (!result) {
            return [];
        }
        let ancestorsOrganizationUnits = [];
        let targetOrder = order;
        for (let organizationUnit of result) {
            if (organizationUnit.managerUserId === userId)
                targetOrder = order + 1;
            else
                targetOrder = order;
            if (targetOrder < 1) {
                ancestorsOrganizationUnits.push(organizationUnit.organizationUnitId);
            }
            else {
                if (!organizationUnit.ancestorIds)
                    continue;
                else {
                    let ancestorIds = organizationUnit.ancestorIds.split(",");
                    if (ancestorIds.length >= targetOrder) {
                        ancestorsOrganizationUnits.push(ancestorIds[ancestorIds.length - targetOrder]);
                    }
                }
            }
        }
        return Promise.resolve(ancestorsOrganizationUnits);
    }
    async getOrganizationUnitsOfManager(organizationId, userId, trx) {
        let result = [];
        let qb = this.dbClient.withSchema(organizationId).from("organizationUnits").where("organizationId", organizationId).where("managerUserId", userId);
        if (trx)
            qb.transacting(trx);
        await qb.select("id").then((rows) => {
            for (let row of rows) {
                result.push(row.id);
            }
        });
        return Promise.resolve(result);
    }
    async getOrganizationUnitsOfManagerBulk(organizationId, userIds, trx) {
        const qb = this.dbClient.withSchema(organizationId).from("organizationUnits").where("organizationId", organizationId).whereIn("managerUserId", userIds).select("managerUserId", "id");
        if (trx)
            qb.transacting(trx);
        const rows = await qb;
        const result = new Map();
        for (const row of rows) {
            if (!result.has(row.managerUserId)) {
                result.set(row.managerUserId, []);
            }
            result.get(row.managerUserId).push(row.id);
        }
        return result;
    }
    async setDeputyOfManager(organizationId, userId, args) {
        let organizationUnitList = await this.getOrganizationUnitsOfManager(organizationId, userId, args.trx);
        for (let organizationUnit of organizationUnitList) {
            await this.setDeputyOfOrganizationUnit(organizationId, organizationUnit, {
                deputyUserId: args.deputyUserId,
                deputyStartDateTime: args.deputyStartDateTime,
                deputyEndDateTime: args.deputyEndDateTime,
                ppermissionId: args.ppermissionId,
                userId: userId,
                trx: args.trx,
            });
        }
        return Promise.resolve();
    }
    async setDeputyOfManagerBulk(organizationId, userAndDeputyList, args) {
        const validList = userAndDeputyList.filter((u) => u.deputyUserId);
        if (validList.length === 0)
            return;
        const userIds = validList.map((u) => u.userId);
        const unitMap = await this.getOrganizationUnitsOfManagerBulk(organizationId, userIds, args.trx);
        const bulkEntries = [];
        for (const { userId, deputyUserId } of validList) {
            const unitIds = unitMap.get(userId) || [];
            for (const unitId of unitIds) {
                bulkEntries.push({
                    deputyUserId,
                    organizationUnitId: unitId,
                    userId,
                });
            }
        }
        if (bulkEntries.length > 0) {
            await this.setDeputyOfOrganizationUnitBulk(organizationId, args.deputyStartDateTime, args.deputyEndDateTime, bulkEntries, args.ppermissionId, args.trx);
        }
        return;
    }
    async getUserOrganizationId(organizationId, userId, trx) {
        let userOrganizationId = "";
        let qb = this.dbClient.withSchema(organizationId).from("userOrganizations").where("userId", userId).where("organizationId", organizationId).whereNull("deletedAt");
        if (trx)
            qb.transacting(trx);
        await qb.first("id").then((result) => {
            if (result)
                userOrganizationId = result.id;
        });
        return Promise.resolve(userOrganizationId);
    }
    async getUserOrganizationIds(organizationId, userIds, trx) {
        if (userIds.length === 0)
            return {};
        let qb = this.dbClient
            .withSchema(organizationId)
            .from(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations)
            .whereIn("userId", userIds)
            .andWhere("organizationId", organizationId)
            .whereNull("deletedAt")
            .select("userId", "id as userOrganizationId");
        if (trx)
            qb = qb.transacting(trx);
        const rows = await qb;
        const result = {};
        for (const row of rows) {
            result[row.userId] = row.userOrganizationId;
        }
        return result;
    }
    async setDeputyOfOrganizationUnit(organizationId, organizationUnitId, args) {
        let existingDeputy = false;
        let qb = this.dbClient
            .withSchema(organizationId)
            .table("userOrganizationDeputies as uod")
            .innerJoin("userOrganizations as uo", (join) => {
            join.on("uo.id", "uod.deputyUserOrganizationId");
        })
            .where("uo.userId", args.deputyUserId)
            .where("uod.organizationUnitId", organizationUnitId)
            .where("uod.startDateTime", "<=", args.deputyStartDateTime)
            .where("uod.endDateTime", ">=", args.deputyEndDateTime)
            .whereNull("uo.deletedAt");
        if (args.ppermissionId)
            qb.where("uod.ppermissionId", args.ppermissionId);
        if (args.trx)
            qb.transacting(args.trx);
        await qb.count().then((row) => (existingDeputy = row[0].count > 0));
        if (existingDeputy)
            return Promise.resolve(true);
        let affectedRowCount = 0;
        let qbx = this.dbClient.withSchema(organizationId).table("userOrganizationDeputies");
        if (args.trx)
            qbx = qbx.transacting(args.trx);
        await qbx
            .insert({
            id: uuid_1.default.v4(),
            organizationUnitId: organizationUnitId,
            ppermissionId: args.ppermissionId,
            userOrganizationId: await this.getUserOrganizationId(organizationId, args.userId),
            deputyUserOrganizationId: await this.getUserOrganizationId(organizationId, args.deputyUserId),
            startDateTime: args.deputyStartDateTime,
            endDateTime: args.deputyEndDateTime,
            type: dal_constants_1.DalConstants.UserDeputyType.Management,
        })
            .then((result) => (affectedRowCount = result[0]));
        return Promise.resolve(affectedRowCount > 0);
    }
    async setDeputyOfOrganizationUnitBulk(organizationId, deputyStartDateTime, deputyEndDateTime, entries, ppermissionId, trx) {
        if (entries.length === 0)
            return true;
        const schema = this.dbClient.withSchema(organizationId);
        const uniqueUserIds = [...new Set(entries.map((e) => e.userId).filter(Boolean))];
        const uniqueDeputyUserIds = [...new Set(entries.map((e) => e.deputyUserId))];
        const userOrgIdMap = new Map();
        const deputyUserOrgIdMap = new Map();
        const userOrgRows = await schema
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations)
            .select("id", "userId")
            .whereIn("userId", [...uniqueUserIds, ...uniqueDeputyUserIds])
            .whereNull("deletedAt");
        for (const row of userOrgRows) {
            if (uniqueUserIds.includes(row.userId))
                userOrgIdMap.set(row.userId, row.id);
            if (uniqueDeputyUserIds.includes(row.userId))
                deputyUserOrgIdMap.set(row.userId, row.id);
        }
        let checkQuery = schema
            .table(`${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationDeputies} as uod`)
            .select("uo.userId as deputyUserId", "uod.organizationUnitId")
            .innerJoin(`${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations} as uo`, "uo.id", "uod.deputyUserOrganizationId")
            .whereBetween("uod.startDateTime", [deputyStartDateTime, deputyEndDateTime])
            .whereBetween("uod.endDateTime", [deputyStartDateTime, deputyEndDateTime])
            .whereNull("uo.deletedAt");
        if (ppermissionId)
            checkQuery = checkQuery.where("uod.ppermissionId", ppermissionId);
        if (trx)
            checkQuery = checkQuery.transacting(trx);
        const existing = await checkQuery;
        const existingSet = new Set(existing.map((e) => `${e.deputyUserId}-${e.organizationUnitId}`));
        const toInsert = entries
            .filter((e) => !existingSet.has(`${e.deputyUserId}-${e.organizationUnitId}`))
            .map((e) => ({
            id: uuid_1.default.v4(),
            organizationUnitId: e.organizationUnitId,
            ppermissionId: ppermissionId || null,
            userOrganizationId: e.userId ? userOrgIdMap.get(e.userId) : null,
            deputyUserOrganizationId: deputyUserOrgIdMap.get(e.deputyUserId),
            startDateTime: deputyStartDateTime,
            endDateTime: deputyEndDateTime,
            type: dal_constants_1.DalConstants.UserDeputyType.Management,
        }))
            .filter((e) => e.userOrganizationId && e.deputyUserOrganizationId);
        if (toInsert.length === 0)
            return true;
        let insertQuery = schema.table(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationDeputies);
        if (trx)
            insertQuery = insertQuery.transacting(trx);
        await insertQuery.insert(toInsert);
        return true;
    }
    async pgGetVisibleOrganizationUnitsAsUnitHR(organizationId, userId, listOrganizationUnitsHierarchically, trx) {
        const query = `
		SELECT ou.id FROM "${organizationId}"."organizationUnits" ou WHERE
		(
			SELECT array_agg(uoou."organizationUnitId") FROM "${organizationId}"."userOrganizationOrganizationUnits" uoou
			INNER JOIN "${organizationId}"."roles" r ON r.id = uoou."roleId"
			WHERE uoou."deletedAt" IS NULL AND uoou."userOrganizationId" = (
				SELECT uo.id
				FROM "${organizationId}"."userOrganizations" uo
				WHERE uo."deletedAt" IS NULL AND NOT uo."isDisabled" AND uo."userId" = $1
			) AND POSITION($2::text IN r.permissions) > 0
        ) && ${listOrganizationUnitsHierarchically ? `(string_to_array(ou."ancestorIds", ',')::uuid[] || ou.id)` : `array[ou.id]`}`;
        if (trx) {
            return (await trx.query(query, [userId, predefined_permissions_1.Permissions.ppermission.getHRApprovement()])).rows.map((r) => r.id);
        }
        else {
            return await dal_manager_1.dbManager.pgTransactionMainDb(async (trx) => {
                return (await trx.query(query, [organizationId, userId])).rows.map((r) => r.id);
            });
        }
    }
    async pgGetVisibleOrganizationUnitsAsManagerOrDeputy(organizationId, userId, listOrganizationUnitsHierarchically, date, trx) {
        let queryWithDeputy = `
        SELECT ou_2.id FROM "${organizationId}"."organizationUnits" ou_2 WHERE
        (
            SELECT array_agg(ou_1.id)
            FROM "${organizationId}"."organizationUnits" ou_1
            WHERE ou_1."managerUserOrganizationId" IN
            (
                SELECT unnest(array_remove(array_agg(DISTINCT uod."deputyUserOrganizationId") || uo.id, NULL)) as "managerOrDeputyUOID"
                FROM "${organizationId}"."userOrganizations" uo
                LEFT JOIN "${organizationId}"."userOrganizationDeputies" uod ON tstzrange(uod."startDateTime", uod."endDateTime") @> now() AND uod."deputyUserOrganizationId" = uo.id
                WHERE uo."deletedAt" IS NULL AND NOT uo."isDisabled" AND uo."organizationId" = $1 AND uo."userId" = $2
                GROUP BY uo.id
            )
        ) && ${listOrganizationUnitsHierarchically ? `(string_to_array(ou_2."ancestorIds", ',')::uuid[] || ou_2.id)` : `array[ou_2.id]`}`;
        let queryWithoutDeputy = `
        SELECT ou_2.id FROM "${organizationId}"."organizationUnits" ou_2 WHERE
        (
            SELECT array_agg(ou_1.id)
            FROM "${organizationId}"."organizationUnits" ou_1
            WHERE ou_1."managerUserOrganizationId" IN
            (
                SELECT uo.id
                FROM "${organizationId}"."userOrganizations" uo
                WHERE uo."deletedAt" IS NULL AND NOT uo."isDisabled" AND uo."organizationId" = $1 AND uo."userId" = $2
            )
        ) && ${listOrganizationUnitsHierarchically ? `(string_to_array(ou_2."ancestorIds", ',')::uuid[] || ou_2.id)` : `array[ou_2.id]`}`;
        if (trx) {
            return (await trx.query(queryWithoutDeputy, [organizationId, userId])).rows.map((r) => r.id);
        }
        else {
            return await dal_manager_1.dbManager.pgTransactionMainDb(async (trx) => {
                return (await trx.query(queryWithoutDeputy, [organizationId, userId])).rows.map((r) => r.id);
            });
        }
    }
    async getManagerOrDeputyOrganizationListWithChilds(organizationId, userId, date, trx) {
        let result = [];
        let qb = this.dbClient.withSchema(organizationId).from("organizationUnits").where("managerUserId", userId);
        if (trx)
            qb.transacting(trx);
        await qb.select("id").then((rows) => {
            for (let row of rows) {
                result.push(row.id);
            }
        });
        let userOrganizationId = await this.getUserOrganizationId(organizationId, userId);
        if (!userOrganizationId)
            return Promise.resolve(result);
        let qbx = this.dbClient
            .withSchema(organizationId)
            .from("organizationUnits as ou")
            .innerJoin("userOrganizationDeputies as oud", "ou.id", "oud.organizationUnitId")
            .where("oud.startDateTime", "<=", new Date(Date.now()))
            .where("oud.endDateTime", ">=", new Date(Date.now()))
            .where("oud.deputyUserOrganizationId", userOrganizationId);
        if (trx)
            qbx = qbx.transacting(trx);
        await qbx.select("ou.id").then((rows) => {
            for (let row of rows) {
                result.push(row.id);
            }
        });
        let childResult = [];
        for (let i = 0; i < result.length; i++) {
            const item = result[i];
            await this.fetchAllChildren(organizationId, item, childResult, trx);
        }
        result = [...result, ...childResult];
        return Promise.resolve(result);
    }
    async getManagerOrDeputyOrganizationList(organizationId, userId, date, trx) {
        let result = [];
        let qb = this.dbClient.withSchema(organizationId).from("organizationUnits").where("managerUserId", userId);
        if (trx)
            qb.transacting(trx);
        await qb.select("id").then((rows) => {
            for (let row of rows) {
                result.push(row.id);
            }
        });
        let userOrganizationId = await this.getUserOrganizationId(organizationId, userId);
        if (!userOrganizationId)
            return Promise.resolve(result);
        let qbx = this.dbClient
            .withSchema(organizationId)
            .from("organizationUnits as ou")
            .innerJoin("userOrganizationDeputies as oud", "ou.id", "oud.organizationUnitId")
            .where("oud.startDateTime", "<=", new Date(Date.now()))
            .where("oud.endDateTime", ">=", new Date(Date.now()))
            .where("oud.deputyUserOrganizationId", userOrganizationId);
        if (trx)
            qbx = qbx.transacting(trx);
        await qbx.select("ou.id").then((rows) => {
            for (let row of rows) {
                result.push(row.id);
            }
        });
        return Promise.resolve(result);
    }
    async fetchAllChildren(organizationId, parentId, result, trx) {
        let qb = this.dbClient.withSchema(organizationId).table("organizationUnits").where("ancestorIds", "like", `%${parentId}%`).whereNull("deletedAt");
        if (trx)
            qb = qb.transacting(trx);
        const rows = await qb.select("id");
        for (const row of rows) {
            result.push(row.id);
        }
    }
    async isUserManagerOrDeputyOfOrganizationUnit(organizationId, userId, organizationIds, date, trx) {
        let result = await this.getManagerOrDeputyOrganizationListWithChilds(organizationId, userId, date, trx);
        if (result) {
            let intersect = organizationIds.filter(function (item) {
                return result.indexOf(item) > -1;
            });
            if (intersect && intersect.length > 0) {
                return Promise.resolve(true);
            }
        }
        return Promise.resolve(false);
    }
    async listManagerOrDeputyOfOrganizationUnits(organizationId, organizationUnitIds, date, trx) {
        let result = [];
        let qb = this.dbClient.withSchema(organizationId).from("organizationUnits as wg").whereIn("wg.id", organizationUnitIds).whereNotNull("wg.managerUserId").whereNull("wg.deletedAt");
        if (trx)
            qb.transacting(trx);
        await qb.select("wg.managerUserId").then(async (rows) => {
            if (rows) {
                for (let row of rows) {
                    result.push(row.managerUserId);
                }
            }
        });
        let qbx = this.dbClient
            .withSchema(organizationId)
            .from("organizationUnits as wg")
            .innerJoin("userOrganizationDeputies as wgd", "wg.id", "wgd.organizationUnitId")
            .innerJoin("userOrganizations as uo", "uo.id", "wgd.deputyUserOrganizationId")
            .whereIn("wg.id", organizationUnitIds)
            .whereNull("uo.deletedAt")
            .where("uo.organizationId", organizationId)
            .where("wgd.startDateTime", "<=", date)
            .where("wgd.endDateTime", ">=", date);
        if (trx)
            qbx = qbx.transacting(trx);
        await qbx.select("uo.userId").then(async (rows) => {
            if (rows) {
                for (let row of rows) {
                    result.push(row.userId);
                }
            }
        });
        return Promise.resolve(result);
    }
    async listManagerOrDeputyOfOrganizationUnitBulk(organizationId, organizationUnitIds, date = new Date(), trx) {
        const trxx = trx ?? this._pgPool;
        const resultMap = new Map();
        const managerSql = `
			SELECT id as "organizationUnitId", "managerUserId"
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}"
			WHERE id = ANY($1)
				AND "managerUserId" IS NOT NULL
				AND "deletedAt" IS NULL
		`;
        const managerResult = await trxx.query(managerSql, [organizationUnitIds]);
        for (const row of managerResult.rows) {
            if (!resultMap.has(row.organizationUnitId)) {
                resultMap.set(row.organizationUnitId, new Set());
            }
            resultMap.get(row.organizationUnitId).add(row.managerUserId);
        }
        const deputySql = `
			SELECT wg.id as "organizationUnitId", uo."userId"
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" wg
			INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationDeputies}" wgd ON wg.id = wgd."organizationUnitId"
			INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo ON uo.id = wgd."deputyUserOrganizationId"
			WHERE wg.id = ANY($1)
				AND uo."organizationId" = $2
				AND uo."deletedAt" IS NULL
				AND wgd."startDateTime" <= $3
				AND wgd."endDateTime" >= $3
		`;
        const deputyResult = await trxx.query(deputySql, [organizationUnitIds, organizationId, date]);
        for (const row of deputyResult.rows) {
            if (!resultMap.has(row.organizationUnitId)) {
                resultMap.set(row.organizationUnitId, new Set());
            }
            resultMap.get(row.organizationUnitId).add(row.userId);
        }
        const result = [];
        for (const [organizationUnitId, userIdSet] of resultMap.entries()) {
            result.push({
                organizationUnitId,
                managerOrDeputyUserIds: Array.from(userIdSet),
            });
        }
        return result;
    }
    async upsertWorkPlanShiftImport(organizationId, args, trx) {
        const id = args.id || uuid_1.default.v4();
        if (args.defaultWorkPlanId) {
            const defaultWorkPlan = await trx.query(`
                SELECT * 
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.workPlans}"
                WHERE "id" = $1;`, [args.defaultWorkPlanId]);
            if (!defaultWorkPlan.rows[0]) {
                throw (0, dal_access_error_1.throwDbAccessBadRequestError)("Work Plan couldn't be found");
            }
        }
        if (args.id) {
            await trx.query(`
                UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.shiftTemplates}"
                SET ("name","defaultWorkplanId")
                = ($2,$3)
                WHERE "id" = $1;`, [args.id, args.name, args.defaultWorkPlanId]);
            await trx.query(`DELETE FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.workPlanImportRules}" WHERE "shiftTemplateId" = $1`, [args.id]);
        }
        else {
            await trx.query(`
                INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.shiftTemplates}"
                ("id","name","defaultWorkplanId", "createdAt")
                VALUES ($1,$2,$3,$4);`, [id, args.name, args.defaultWorkPlanId, new Date()]);
        }
        for (let rule of args.workPlanImportRules) {
            await trx.query(`
                INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.workPlanImportRules}"
                ("id","workplanId", "shiftTemplateId","keyword", "createdAt", "membershipRange") VALUES ($1,$2,$3,$4,$5,$6)`, [uuid_1.default.v4(), rule.workPlanId, id, rule.keyword, new Date(), rule.membershipRange]);
        }
        return id;
    }
    async listWorkPlanShiftImports(organizationId, args, userId, trx) {
        let result = {
            pagination: {
                total: 0,
                take: args.take,
                skip: args.skip,
            },
            items: [],
        };
        let qb = this.dbClient.withSchema(organizationId).from(`${dal_db_armon_schema_1.ArmonSchema.tableNames.shiftTemplates}`);
        result.pagination.take = args.take;
        result.pagination.skip = args.skip;
        if (args.take > 0)
            qb.limit(args.take);
        await dal_manager_1.dbManager.systemTransaction(async (trx) => {
            const workplanIdsThatUserCanSee = (await this.listWorkPlans(trx, organizationId, userId)).items.map((item) => item.id);
            let countQuery = `SELECT COUNT(*) FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.shiftTemplates}" st`;
            countQuery += `
						WHERE NOT EXISTS (
  						SELECT 1
  						FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.workPlanImportRules}" wirx
  						WHERE wirx."shiftTemplateId" = st."id"
						AND wirx."workplanId" NOT IN (SELECT UNNEST($1::uuid[]))
						) `;
            let countQueryParams = [];
            let dataQueryParams = [];
            countQueryParams.push(workplanIdsThatUserCanSee);
            if (args.name) {
                countQuery += ` AND st."name" ILIKE $2`;
                countQueryParams.push(`%${args.name}%`);
            }
            console.log("countQuery: ", countQuery);
            const countResult = await trx.query(countQuery, countQueryParams);
            result.pagination.total = parseInt(countResult.rows[0].count);
            let dataQuery = `SELECT st.* FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.shiftTemplates}" st`;
            dataQuery += `
						WHERE NOT EXISTS (
  						SELECT 1
  						FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.workPlanImportRules}" wirx
  						WHERE wirx."shiftTemplateId" = st."id"
						AND wirx."workplanId" NOT IN (SELECT UNNEST($1::uuid[]))
						) `;
            dataQueryParams.push(workplanIdsThatUserCanSee);
            if (args.name === undefined) {
                args.name = "";
            }
            dataQuery += ` AND st."name" ILIKE $2`;
            dataQueryParams.push(`%${args.name}%`);
            dataQuery += ` ORDER BY st."name" ASC`;
            if (args.take > 0) {
                dataQuery += ` LIMIT $3`;
                dataQueryParams.push(args.take);
            }
            if (args.skip > 0) {
                dataQuery += ` OFFSET $4`;
                dataQueryParams.push(args.skip);
            }
            let rows = (await trx.query(dataQuery, dataQueryParams)).rows;
            for (let row of rows) {
                let item = {
                    id: row.id,
                    name: row.name,
                    defaultWorkPlanName: (await this.getWorkPlan(trx, organizationId, userId, row.defaultWorkplanId)).name,
                    workPlanImportRuleCount: (await this.getWorkPlanShiftImportRules(organizationId, row.id)).length,
                };
                result.items.push(item);
            }
        });
        return result;
    }
    async getWorkPlanShiftImportRules(organizationId, shiftTemplateId, trx) {
        let results = [];
        let qb = this.dbClient.withSchema(organizationId).table(`${dal_db_armon_schema_1.ArmonSchema.tableNames.workPlanImportRules}`).where("shiftTemplateId", shiftTemplateId);
        if (trx)
            qb.transacting(trx);
        await qb.select().then((rows) => {
            results = rows.map((wir) => {
                return { id: wir.id, workPlanId: wir.workplanId, shiftTemplateId: wir.shiftTemplateId, keyword: wir.keyword, membershipRange: wir.membershipRange };
            });
        });
        return Promise.resolve(results);
    }
    async getWorkPlanShiftImport(organizationId, shiftTemplateId, trx) {
        let result = {
            id: "",
            name: "",
            defaultWorkPlanId: "",
            workPlanImportRules: [],
        };
        let qb = this.dbClient.withSchema(organizationId).table(`${dal_db_armon_schema_1.ArmonSchema.tableNames.shiftTemplates}`).where("id", shiftTemplateId);
        if (trx)
            qb.transacting(trx);
        await qb.first().then(async (row) => {
            if (!row)
                return Promise.resolve(result);
            (result.id = row.id),
                (result.name = row.name),
                (result.defaultWorkPlanId = row.defaultWorkplanId),
                (result.workPlanImportRules = await this.getWorkPlanShiftImportRules(organizationId, shiftTemplateId));
        });
        return Promise.resolve(result);
    }
    async deleteWorkPlanShiftImport(organizationId, params, trx) {
        await trx.query(`
            DELETE FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.shiftTemplates}" WHERE "id" = $1;
            `, [params.id]);
        return { id: params.id };
    }
    async upsertPPermissionTypePg(params) {
        const { args, organizationId, trx } = params;
        const id = args.id || uuid_1.default.v4();
        if (args.id) {
            await trx.query(`
                UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissionTypes}"
                SET ("organizationId","name","isPassive","description","hrCanInit","personnelCanInit","unitManagerCanInit",
                    "isDailyScheduled","minValue","maxValue","maxValueBoundaryInterval","method","sendNotification",
                    "userMustSign","needsManagerDeputy","needsUserDeputy","updatedAt","showRemainingCount","isUnpaidLeave","notifyHR","noteIsMandatory", "applyOnHolidays", "applyOnlyOnWorkingHours", "notifyManagers", "isNegativeBalanceAllowed")
                = ($2,$3,$4,$5,$6,$7,$8,$9,$10,$11,$12,$13,$14,$15,$16,$17,$18,$19,$20,$21,$22,$23,$24,$25,$26)
                WHERE "id" = $1;`, [
                args.id,
                organizationId,
                args.name,
                args.isPassive,
                args.description ? args.description : "",
                args.hrCanInit,
                args.personnelCanInit,
                args.unitManagerCanInit,
                args.isDailyScheduled,
                args.minValue,
                args.maxValue,
                args.maxValueBoundaryInterval,
                args.method,
                args.sendNotification,
                args.userMustSign,
                args.needsManagerDeputy,
                args.needsUserDeputy,
                new Date(),
                args.showRemainingCount,
                args.isUnpaidLeave,
                args.notifyHR,
                args.noteIsMandatory,
                args.applyOnHolidays,
                args.applyOnlyOnWorkingHours,
                args.notifyManagers,
                args.isNegativeBalanceAllowed,
            ]);
            await trx.query(`DELETE FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissionTypeApprovements}" WHERE "ppermissionTypeId" = $1`, [args.id]);
        }
        else {
            let nameExists = (await trx.query(`SELECT COUNT(*) > 0 as "typenameExists" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissionTypes}"
					WHERE "deletedAt" IS NULL AND name ilike $1;`, [args.name])).rows[0].typenameExists;
            if (nameExists) {
                (0, dal_access_error_1.throwDbAccessConflictError)("ppermission type with same name exists");
            }
            await trx.query(`
                INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissionTypes}"
                ("id","organizationId","name","isPassive","description","hrCanInit","personnelCanInit","unitManagerCanInit",
                "isDailyScheduled","minValue","maxValue","maxValueBoundaryInterval","method","sendNotification",
                "userMustSign","needsManagerDeputy","needsUserDeputy","createdAt","type","showRemainingCount","isUnpaidLeave","notifyHR","noteIsMandatory","applyOnHolidays","applyOnlyOnWorkingHours","notifyManagers","isNegativeBalanceAllowed")
                VALUES ($1,$2,$3,$4,$5,$6,$7,$8,$9,$10,$11,$12,$13,$14,$15,$16,$17,$18,$19,$20,$21,$22,$23,$24,$25,$26,$27);`, [
                id,
                organizationId,
                args.name,
                args.isPassive,
                args.description ? args.description : "",
                args.hrCanInit,
                args.personnelCanInit,
                args.unitManagerCanInit,
                args.isDailyScheduled,
                args.minValue,
                args.maxValue,
                args.maxValueBoundaryInterval,
                args.method,
                args.sendNotification,
                args.userMustSign,
                args.needsManagerDeputy,
                args.needsUserDeputy,
                new Date(),
                args.type,
                args.showRemainingCount,
                args.isUnpaidLeave,
                args.notifyHR,
                args.noteIsMandatory,
                args.applyOnHolidays,
                args.applyOnlyOnWorkingHours,
                args.notifyManagers,
                args.isNegativeBalanceAllowed,
            ]);
        }
        let order = 1;
        for (let approvement of args.approvementList) {
            await trx.query(`
                INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissionTypeApprovements}"
                	("id","ppermissionTypeId", "rootOrganizationUnitId","order") VALUES ($1,$2,$3,$4)`, [uuid_1.default.v4(), id, approvement ? approvement : null, order++]);
        }
        await trx.query(`DELETE FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissionTypeRules}"
			WHERE "ppermissionTypeId" = $1
			`, [id]);
        if (args.applicableDays?.length > 0) {
            for (const applicableDay of args.applicableDays) {
                await trx.query(`
				INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissionTypeRules}"
				("id","ppermissionTypeId", "userFilterId","applicableDays") 
				VALUES (gen_random_uuid(), $1, $2, $3)
				ON CONFLICT ("ppermissionTypeId", "userFilterId") 
				DO UPDATE SET "applicableDays" = $3;
				`, [id, applicableDay.userFilterId, applicableDay.days]);
            }
        }
        return id;
    }
    async removePPermissionType(organizationId, ppermissionTypeId) {
        let affectedRowCount = 0;
        await this.dbClient.transaction(async (trx) => {
            await trx
                .withSchema(organizationId)
                .table("ppermissionTypes")
                .where("id", ppermissionTypeId)
                .where("type", dal_constants_1.DalConstants.PredefinedPPermissionType.Other)
                .orWhereNull("type")
                .update({
                deletedAt: new Date(Date.now()),
            })
                .then((result) => (affectedRowCount = result));
            await trx.withSchema(organizationId).table("ppermissions").where("ppermissionTypeId", ppermissionTypeId).where("startDateTime", ">", new Date()).del();
        });
        return Promise.resolve(affectedRowCount > 0);
    }
    async getPPermissionTypeApprovements(organizationId, ppermissionTypeId, trx) {
        let results = [];
        let qb = this.dbClient.withSchema(organizationId).table("ppermissionTypeApprovements").where("ppermissionTypeId", ppermissionTypeId).orderBy("order", "asc");
        if (trx)
            qb.transacting(trx);
        await qb.select().then((rows) => {
            results = rows.map((pa) => {
                return pa.rootOrganizationUnitId ? pa.rootOrganizationUnitId : "";
            });
        });
        return Promise.resolve(results);
    }
    async getPPermissionType(organizationId, ppermissionTypeId, trx) {
        let result = {
            id: "",
            name: "",
            isPassive: false,
            description: "",
            hrCanInit: false,
            personnelCanInit: false,
            unitManagerCanInit: false,
            method: dal_constants_1.DalConstants.PPermissionType.OffTime,
            isDailyScheduled: false,
            minValue: 0,
            maxValue: 0,
            maxValueBoundaryInterval: 0,
            approvementList: [],
            sendNotification: false,
            userMustSign: false,
            needsManagerDeputy: false,
            needsUserDeputy: false,
            type: dal_constants_1.DalConstants.PredefinedPPermissionType.Other,
            showRemainingCount: false,
            isUnpaidLeave: false,
            notifyHR: false,
            noteIsMandatory: false,
            dynamicForm: null,
            applyOnHolidays: false,
            applyOnlyOnWorkingHours: false,
            notifyManagers: false,
            isNegativeBalanceAllowed: false,
        };
        let qb = this.dbClient.withSchema(organizationId).table("ppermissionTypes").where("id", ppermissionTypeId).whereNull("deletedAt");
        let qbApplicableDays = this.dbClient
            .withSchema(organizationId)
            .table("ppermission_type_rules as ptr")
            .innerJoin("user_filter as uf", "uf.id", "ptr.userFilterId")
            .where("ptr.ppermissionTypeId", ppermissionTypeId);
        if (trx)
            qbApplicableDays.transacting(trx);
        let applicableDaysRes = await qbApplicableDays.select("ptr.applicableDays", "uf.id");
        let applicableDays = applicableDaysRes.map((applicableDay) => {
            return {
                userFilterId: applicableDay.id,
                days: applicableDay.applicableDays,
            };
        });
        if (trx)
            qb.transacting(trx);
        await qb.first().then(async (row) => {
            if (!row)
                return Promise.resolve(result);
            (result.id = ppermissionTypeId),
                (result.name = row.name),
                (result.isPassive = row.isPassive),
                (result.description = row.description),
                (result.hrCanInit = row.hrCanInit),
                (result.personnelCanInit = row.personnelCanInit),
                (result.unitManagerCanInit = row.unitManagerCanInit),
                (result.method = row.method),
                (result.isDailyScheduled = row.isDailyScheduled),
                (result.minValue = row.minValue),
                (result.maxValue = row.maxValue),
                (result.maxValueBoundaryInterval = row.maxValueBoundaryInterval),
                (result.sendNotification = row.sendNotification),
                (result.needsManagerDeputy = row.needsManagerDeputy),
                (result.needsUserDeputy = row.needsUserDeputy),
                (result.userMustSign = row.userMustSign),
                (result.approvementList = await this.getPPermissionTypeApprovements(organizationId, ppermissionTypeId)),
                (result.type = row.type),
                (result.showRemainingCount = row.showRemainingCount),
                (result.isUnpaidLeave = row.isUnpaidLeave),
                (result.notifyHR = row.notifyHR),
                (result.notifyManagers = row.notifyManagers),
                (result.noteIsMandatory = row.noteIsMandatory),
                (result.isNegativeBalanceAllowed = row.isNegativeBalanceAllowed),
                (result.dynamicForm = row.dynamicFormId ? await dal_manager_1.dbManager.accessSystem.getDynamicForm({ organizationId: organizationId, dynamicFormId: row.dynamicFormId }) : null);
            result.applyOnHolidays = row.applyOnHolidays;
            result.applyOnlyOnWorkingHours = row.applyOnlyOnWorkingHours;
            result.applicableDays = applicableDays;
        });
        return Promise.resolve(result);
    }
    async getPPermissionTypes(organizationId, ppermissionTypeIds, trx) {
        if (ppermissionTypeIds.length === 0)
            return [];
        const client = trx ?? this._pgPool;
        const permissionTypesQuery = `
			SELECT * FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissionTypes}"
			WHERE id = ANY($1) AND "deletedAt" IS NULL
		`;
        const permissionTypesRes = await client.query(permissionTypesQuery, [ppermissionTypeIds]);
        if (permissionTypesRes.rows.length === 0)
            return [];
        const applicableDaysQuery = `
			SELECT ptr."ppermissionTypeId", ptr."applicableDays", ptr."userFilterId"
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissionTypeRules}" ptr
			WHERE ptr."ppermissionTypeId" = ANY($1)
		`;
        const applicableDaysRes = await client.query(applicableDaysQuery, [ppermissionTypeIds]);
        const applicableDaysMap = new Map();
        for (const row of applicableDaysRes.rows) {
            const entry = { userFilterId: row.userFilterId, days: row.applicableDays };
            if (!applicableDaysMap.has(row.ppermissionTypeId)) {
                applicableDaysMap.set(row.ppermissionTypeId, []);
            }
            applicableDaysMap.get(row.ppermissionTypeId).push(entry);
        }
        const approvementsQuery = `
			SELECT "ppermissionTypeId", "rootOrganizationUnitId"
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissionTypeApprovements}"
			WHERE "ppermissionTypeId" = ANY($1)
			ORDER BY "ppermissionTypeId", "order"
		`;
        const approvementsRes = await client.query(approvementsQuery, [ppermissionTypeIds]);
        const approvementsMap = new Map();
        for (const { ppermissionTypeId, rootOrganizationUnitId } of approvementsRes.rows) {
            if (!approvementsMap.has(ppermissionTypeId)) {
                approvementsMap.set(ppermissionTypeId, []);
            }
            approvementsMap.get(ppermissionTypeId).push(rootOrganizationUnitId);
        }
        const dynamicFormIds = permissionTypesRes.rows.filter((row) => row.dynamicFormId).map((row) => row.dynamicFormId);
        const dynamicFormsMap = new Map();
        if (dynamicFormIds.length > 0) {
            const dynamicFormsQuery = `
				SELECT id, form FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.dynamicForms}"
				WHERE id = ANY($1)
			`;
            const dynamicFormsRes = await client.query(dynamicFormsQuery, [dynamicFormIds]);
            for (const { id, form } of dynamicFormsRes.rows) {
                dynamicFormsMap.set(id, form);
            }
        }
        return permissionTypesRes.rows.map((row) => ({
            id: row.id,
            name: row.name,
            isPassive: row.isPassive,
            description: row.description,
            hrCanInit: row.hrCanInit,
            personnelCanInit: row.personnelCanInit,
            unitManagerCanInit: row.unitManagerCanInit,
            method: row.method,
            isDailyScheduled: row.isDailyScheduled,
            minValue: row.minValue,
            maxValue: row.maxValue,
            maxValueBoundaryInterval: row.maxValueBoundaryInterval,
            approvementList: approvementsMap.get(row.id) || [],
            sendNotification: row.sendNotification,
            userMustSign: row.userMustSign,
            needsManagerDeputy: row.needsManagerDeputy,
            needsUserDeputy: row.needsUserDeputy,
            type: row.type,
            showRemainingCount: row.showRemainingCount,
            isUnpaidLeave: row.isUnpaidLeave,
            notifyHR: row.notifyHR,
            noteIsMandatory: row.noteIsMandatory,
            dynamicForm: row.dynamicFormId ? dynamicFormsMap.get(row.dynamicFormId) : null,
            applyOnHolidays: row.applyOnHolidays,
            applyOnlyOnWorkingHours: row.applyOnlyOnWorkingHours,
            notifyManagers: row.notifyManagers,
            isNegativeBalanceAllowed: row.isNegativeBalanceAllowed,
            applicableDays: applicableDaysMap.get(row.id) || [],
        }));
    }
    async getPPermissionTypePoolClient(organizationId, ppermissionTypeId, trx) {
        const queryFunction = async (trx) => {
            let result = (await trx.query(`
			SELECT	id,
					type,
					"maxValueBoundaryInterval",
					"isDailyScheduled",
					"maxValue"
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissionTypes}"
			WHERE id = $1
		`, [ppermissionTypeId])).rows[0];
            return result;
        };
        if (trx) {
            return await queryFunction(trx);
        }
        else {
            return await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                return await queryFunction(trx);
            });
        }
    }
    async listPPermissionTypes(organizationId, args, trx) {
        let result = {
            pagination: {
                total: 0,
                take: args.take,
                skip: args.skip,
            },
            items: [],
        };
        let qb = this.dbClient.withSchema(organizationId).from("ppermissionTypes").where("organizationId", organizationId);
        if (!args.includeDeleted)
            qb.whereNull("deletedAt");
        if (trx)
            qb.transacting(trx);
        if (args.name)
            qb.where("name", "ilike", "%" + args.name + "%");
        if (args.isPassive)
            qb.where("isPassive", args.isPassive);
        if (args.hrStatus === false && args.unitManagerReadPermissionStatus === false) {
            qb.where("personnelCanInit", true);
        }
        else if (args.hrStatus === true && args.unitManagerReadPermissionStatus === false) {
            qb.whereWrapped((q) => {
                q.where("personnelCanInit", true).orWhere("hrCanInit", true);
            });
        }
        else if (args.hrStatus === false && args.unitManagerReadPermissionStatus === true) {
            qb.whereWrapped((q) => {
                q.where("personnelCanInit", true).orWhere("unitManagerCanInit", true);
            });
        }
        result.pagination.total = await qb
            .clone()
            .count()
            .then((rows) => {
            return parseInt(rows[0].count);
        });
        result.pagination.take = args.take;
        result.pagination.skip = args.skip;
        if (args.take > 0)
            qb.limit(args.take);
        await qb
            .offset(args.skip)
            .orderBy("name", "asc")
            .select()
            .then(async (rows) => {
            for (let row of rows) {
                let item = {
                    id: row.id,
                    name: row.name,
                    isPassive: row.isPassive,
                    description: row.description,
                    hrCanInit: row.hrCanInit,
                    personnelCanInit: row.personnelCanInit,
                    unitManagerCanInit: row.unitManagerCanInit,
                    method: row.method,
                    isDailyScheduled: row.isDailyScheduled,
                    minValue: row.minValue,
                    maxValue: row.maxValue,
                    maxValueBoundaryInterval: row.maxValueBoundaryInterval,
                    sendNotification: row.sendNotification,
                    needsManagerDeputy: row.needsManagerDeputy,
                    needsUserDeputy: row.needsUserDeputy,
                    userMustSign: row.userMustSign,
                    approvementList: await this.getPPermissionTypeApprovements(organizationId, row.id),
                    type: row.type,
                    showRemainingCount: row.showRemainingCount,
                    isUnpaidLeave: row.isUnpaidLeave,
                    notifyHR: row.notifyHR,
                    noteIsMandatory: row.noteIsMandatory,
                    dynamicForm: row.dynamicFormId ? await dal_manager_1.dbManager.accessSystem.getDynamicForm({ organizationId: organizationId, dynamicFormId: row.dynamicFormId }) : null,
                    applyOnHolidays: row.applyOnHolidays,
                    applyOnlyOnWorkingHours: row.applyOnlyOnWorkingHours,
                    notifyManagers: row.notifyManagers,
                    isNegativeBalanceAllowed: row.isNegativeBalanceAllowed,
                };
                result.items.push(item);
            }
        });
        return result;
    }
    async deletePPermissionType(organizationId, ppermissionTypeId, trx) {
        let qb = this.dbClient.withSchema(organizationId).table("ppermissionTypes").where("id", ppermissionTypeId);
        if (trx)
            qb.transacting(trx);
        return await qb.update({
            deletedAt: new Date(),
        });
    }
    async deletePPermissionTypeByType(organizationId, type, trx) {
        let qb = this.dbClient.withSchema(organizationId).table("ppermissionTypes").whereIn("type", type);
        if (trx)
            qb.transacting(trx);
        return await qb.update({
            deletedAt: new Date(),
        });
    }
    async getApplicableDaysForPPermissionType(organizationId, ppermissionTypeId, ppermissionType, isUnpaid, userId, trx) {
        const transactionScope = async (trx) => {
            const { rows: ruleRows } = await trx.query(`
				SELECT "applicableDays", "userFilterId" 
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissionTypeRules}"
				WHERE "ppermissionTypeId" = $1
				`, [ppermissionTypeId]);
            const userFilterIds = ruleRows.map((r) => r.userFilterId).filter((id) => id !== null);
            let matchingUserFilterIds = [];
            if (userId && userFilterIds.length > 0) {
                const userFilterChecks = await Promise.all(userFilterIds.map(async (ufid) => {
                    const bindingKeys = [];
                    const q = (0, dal_utils_1.generateUserFilterQuery)(organizationId, ufid, bindingKeys);
                    const filteredUserIds = (await trx.query(q, bindingKeys)).rows.map((r) => r.userId);
                    if (filteredUserIds.includes(userId)) {
                        return ufid;
                    }
                    else {
                        return null;
                    }
                }));
                matchingUserFilterIds = userFilterChecks.filter((id) => !!id);
            }
            const applicableDaysSet = new Set();
            for (const row of ruleRows) {
                if (!row.userFilterId || matchingUserFilterIds.includes(row.userFilterId)) {
                    row.applicableDays.forEach((d) => applicableDaysSet.add(d));
                }
            }
            if (applicableDaysSet.size === 0) {
                if (ppermissionType === app_enums_1.enums.PPermissionType.OffTime && !isUnpaid) {
                    [
                        app_enums_1.enums.DayOfWeekForWorking.Monday,
                        app_enums_1.enums.DayOfWeekForWorking.Tuesday,
                        app_enums_1.enums.DayOfWeekForWorking.Wednesday,
                        app_enums_1.enums.DayOfWeekForWorking.Thursday,
                        app_enums_1.enums.DayOfWeekForWorking.Friday,
                    ].forEach((d) => applicableDaysSet.add(d));
                }
                else {
                    [
                        app_enums_1.enums.DayOfWeekForWorking.Monday,
                        app_enums_1.enums.DayOfWeekForWorking.Tuesday,
                        app_enums_1.enums.DayOfWeekForWorking.Wednesday,
                        app_enums_1.enums.DayOfWeekForWorking.Thursday,
                        app_enums_1.enums.DayOfWeekForWorking.Friday,
                        app_enums_1.enums.DayOfWeekForWorking.Saturday,
                        app_enums_1.enums.DayOfWeekForWorking.Sunday,
                    ].forEach((d) => applicableDaysSet.add(d));
                }
            }
            return Array.from(applicableDaysSet);
        };
        if (trx) {
            return transactionScope(trx);
        }
        else {
            return await dal_manager_1.dbManager.organizationTransaction(transactionScope, userId, organizationId);
        }
    }
    async getApplicableDaysForPPermissionTypes(organizationId, ppermissionTypes, userId, trx) {
        const transactionScope = async (trx) => {
            const ppermissionTypeIds = Object.keys(ppermissionTypes);
            const { rows: ruleRows } = await trx.query(`
					SELECT "ppermissionTypeId", "applicableDays", "userFilterId"
					FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissionTypeRules}"
					WHERE "ppermissionTypeId" = ANY($1)
				`, [ppermissionTypeIds]);
            const allUserFilterIds = ruleRows.map((r) => r.userFilterId).filter((id) => id !== null);
            let matchingUserFilterIds = new Set();
            if (userId && allUserFilterIds.length > 0) {
                const userFilterChecks = await Promise.all(allUserFilterIds.map(async (ufid) => {
                    const bindingKeys = [];
                    const q = (0, dal_utils_1.generateUserFilterQuery)(organizationId, ufid, bindingKeys);
                    const filterUserIds = (await trx.query(q, bindingKeys)).rows.map((r) => r.userId);
                    const userMatch = filterUserIds.includes(userId);
                    return userMatch ? ufid : null;
                }));
                userFilterChecks.filter((id) => !!id).forEach((id) => matchingUserFilterIds.add(id));
            }
            const result = new Map();
            for (const typeId of ppermissionTypeIds) {
                const relatedRules = ruleRows.filter((r) => r.ppermissionTypeId === typeId && (!r.userFilterId || matchingUserFilterIds.has(r.userFilterId)));
                const days = new Set();
                for (const rule of relatedRules) {
                    rule.applicableDays.forEach((d) => days.add(d));
                }
                if (days.size === 0) {
                    const { type, isUnpaid } = ppermissionTypes[typeId];
                    const defaultDays = type === app_enums_1.enums.PPermissionType.OffTime && !isUnpaid
                        ? [
                            app_enums_1.enums.DayOfWeekForWorking.Monday,
                            app_enums_1.enums.DayOfWeekForWorking.Tuesday,
                            app_enums_1.enums.DayOfWeekForWorking.Wednesday,
                            app_enums_1.enums.DayOfWeekForWorking.Thursday,
                            app_enums_1.enums.DayOfWeekForWorking.Friday,
                        ]
                        : [
                            app_enums_1.enums.DayOfWeekForWorking.Monday,
                            app_enums_1.enums.DayOfWeekForWorking.Tuesday,
                            app_enums_1.enums.DayOfWeekForWorking.Wednesday,
                            app_enums_1.enums.DayOfWeekForWorking.Thursday,
                            app_enums_1.enums.DayOfWeekForWorking.Friday,
                            app_enums_1.enums.DayOfWeekForWorking.Saturday,
                            app_enums_1.enums.DayOfWeekForWorking.Sunday,
                        ];
                    defaultDays.forEach((d) => days.add(d));
                }
                result.set(typeId, Array.from(days));
            }
            return result;
        };
        if (trx) {
            return transactionScope(trx);
        }
        else {
            return dal_manager_1.dbManager.organizationTransaction(transactionScope, userId, organizationId);
        }
    }
    async updatePPermissionDateRangeByHR(organizationId, ppermissionId, requestUserId, startDateTime, endDateTime) {
        if ((0, moment_1.default)(startDateTime).isAfter((0, moment_1.default)(endDateTime)))
            return Promise.reject("invalid date range");
        let now = new Date();
        await this.dbClient.transaction(async (trx) => {
            let ppermission = await this.getPPermissionTrx(organizationId, ppermissionId, trx);
            if (!ppermission)
                return Promise.reject("invalid date ppermission");
            await this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissions).where("id", ppermissionId).update({
                startDateTime: startDateTime,
                endDateTime: endDateTime,
            });
            let approvementList = await this.listPPermissionUserApprovementsAndStatus(organizationId, ppermissionId, trx);
            await dal_manager_1.dbManager.accessLogPacs.insertPPermissionHistoryContentLog(organizationId, ppermissionId, requestUserId, new Date(), dal_constants_1.DalConstants.PPermissionHistoryActionType.UpdateDateTime, {
                typeId: ppermission.ppermissionType.id,
                startDate: startDateTime,
                endDate: endDateTime,
                status: ppermission.status,
                userAndDeputyList: ppermission.userAndDeputyList
                    ? ppermission.userAndDeputyList.map((m) => {
                        return {
                            userId: m.user.id,
                            deputyUserId: m.deputyUser ? m.deputyUser.id : null,
                        };
                    })
                    : null,
                approvementList: approvementList.userApprovementList
                    ? approvementList.userApprovementList.map((m) => {
                        return {
                            approverUserId: m.approverUserId,
                            organizationUnitId: m.organizationUnitId,
                            isHr: m.isHr,
                            order: m.order,
                            note: m.note,
                            isApproved: m.isApproved,
                        };
                    })
                    : null,
            });
        });
        return Promise.resolve();
    }
    async upsertPPermissionUserSignStatus(organizationId, ppermissionId, userId, status) {
        await this.dbClient.transaction(async (trx) => {
            await trx.withSchema(organizationId).table("userPPermissions").where("ppermissionId", ppermissionId).where("userId", userId).update({
                signedByUser: status,
            });
            let ppermission = await this.getPPermissionLogBasicTrx(organizationId, ppermissionId, trx);
            let approvementList = await this.listPPermissionUserApprovementsAndStatus(organizationId, ppermissionId, trx);
            await dal_manager_1.dbManager.accessLogPacs.insertPPermissionHistoryContentLog(organizationId, ppermissionId, userId, new Date(), dal_constants_1.DalConstants.PPermissionHistoryActionType.PPermissionSigned, {
                typeId: ppermission.ppermissionTypeId,
                startDate: ppermission.dateRange.startDateTime,
                endDate: ppermission.dateRange.endDateTime,
                status: ppermission.status,
                userAndDeputyList: ppermission.userAndDeputyList
                    ? ppermission.userAndDeputyList.map((m) => {
                        return {
                            userId: m.userId,
                            deputyUserId: m.deputyUserId,
                        };
                    })
                    : null,
                approvementList: approvementList.userApprovementList
                    ? approvementList.userApprovementList.map((m) => {
                        return {
                            approverUserId: m.approverUserId,
                            organizationUnitId: m.organizationUnitId,
                            isHr: m.isHr,
                            order: m.order,
                            note: m.note,
                            isApproved: m.isApproved,
                        };
                    })
                    : null,
            });
        });
    }
    async upsertPPermissionDeputyUsers(organizationId, ppermissionId, requestUserId, userAndDeputyList) {
        await this.dbClient.transaction(async (trx) => {
            let ppermission = await this.getPPermissionTrx(organizationId, ppermissionId, trx);
            await this.setDeputyOfManagerBulk(organizationId, userAndDeputyList, {
                deputyStartDateTime: ppermission.dateRange.startDateTime,
                deputyEndDateTime: ppermission.dateRange.endDateTime,
                ppermissionId: ppermission.id,
                trx: trx,
            });
            const userIds = userAndDeputyList.filter((item) => !item.deputyUserId).map((item) => item.userId);
            const userOrgMap = await this.getUserOrganizationIds(organizationId, userIds, trx);
            const userOrganizationIds = Object.values(userOrgMap);
            if (userOrganizationIds.length > 0) {
                await this.dbClient
                    .withSchema(organizationId)
                    .table(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationDeputies)
                    .where("ppermissionId", ppermissionId)
                    .whereIn("userId", userOrganizationIds)
                    .transacting(trx)
                    .del();
            }
        });
        return Promise.resolve();
    }
    async upsertFingerPrintCredential(organizationId, userId, finger, type, specialData) {
        return await this.dbClient.transaction(async (trx) => {
            let existing = await trx
                .withSchema(organizationId)
                .from(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials)
                .first("id")
                .where("userId", userId)
                .where("type", type)
                .where("organizationId", organizationId)
                .whereRaw(`"extensionFields"->>'finger' = ?`, finger);
            let now = new Date();
            if (existing) {
                await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials).where("id", existing.id).update({
                    specialData: specialData,
                    updatedAt: now,
                });
                return Promise.resolve(existing.id);
            }
            else {
                let id = uuid_1.default.v4();
                await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials).insert({
                    id: id,
                    createdAt: now,
                    updatedAt: now,
                    organizationId: organizationId,
                    userId: userId,
                    data: finger,
                    specialData: specialData,
                    type: type,
                });
                return Promise.resolve(id);
            }
        });
    }
    async upsertPPermissionUsersByUserSelectionSession(organizationId, ppermissionId, requestUserId, userSelectionSessionId, preApproved, upsertAs) {
        let addedUsers = [];
        let session = await dal_manager_1.dbManager.accessUser.validateUserSelectionSession(organizationId, requestUserId, userSelectionSessionId);
        await this.dbClient.transaction(async (trx) => {
            if (upsertAs === dal_constants_1.DalConstants.PPermissionInsertRole.Manager) {
                let existingPPermissionRequestRole = await trx.raw(`
                SELECT "requestRole" FROM "${organizationId}"."ppermissions" WHERE id = ?
                `, [ppermissionId]);
                if (existingPPermissionRequestRole.rows[0] && existingPPermissionRequestRole.rows[0].requestRole === dal_constants_1.DalConstants.PPermissionInsertRole.HR) {
                    (0, dal_access_error_1.throwDbAccessBadRequestError)("This permission was created by a user with higher privilege");
                }
                let hrApprovementCount = await trx.raw(`
                SELECT COUNT(*) as count
                FROM "${organizationId}"."userPPermissionApprovements" uppa 
                INNER JOIN "${organizationId}"."userSelectionSessionActions" ussa ON ussa."userId" = uppa."userId" AND ussa."sessionId" = ? AND ussa."action" = ? 
                WHERE uppa."ppermissionId" = ? AND uppa."organizationUnitId" IS NULL AND uppa."approverUserId" IS NOT NULL AND status = true
                `, [userSelectionSessionId, dal_constants_1.DalConstants.UserSelectionSessionAction.Removed, ppermissionId]);
                if (hrApprovementCount.rows[0].count > 0) {
                    (0, dal_access_error_1.throwDbAccessBadRequestError)("You can't remove users whose permission has already been approved by hr");
                }
            }
            let ppermissionPrevious = await this.getPPermissionLogBasicTrx(organizationId, ppermissionId, trx);
            let ppermissionType = await dal_manager_1.dbManager.accessPacs.getPPermissionType(organizationId, ppermissionPrevious.ppermissionTypeId, trx);
            let userActions = (await trx.raw(`
                SELECT action, array_agg("userId") as users FROM "${organizationId}"."userSelectionSessionActions" ussa
                WHERE ussa."sessionId" = ? GROUP BY action;
            `, [session.id])).rows;
            addedUsers = userActions.find((ua) => ua.action === dal_constants_1.DalConstants.UserSelectionSessionAction.Added)?.users;
            addedUsers = addedUsers ? addedUsers : [];
            let removedUsers = userActions.find((ua) => ua.action === dal_constants_1.DalConstants.UserSelectionSessionAction.Removed)?.users;
            removedUsers = removedUsers ? removedUsers : [];
            let unchangedUsers = userActions.find((ua) => ua.action === dal_constants_1.DalConstants.UserSelectionSessionAction.Unchanged)?.users;
            unchangedUsers = unchangedUsers ? unchangedUsers : [];
            if (ppermissionType.type === app_enums_1.enums.PredefinedPPermissionType.Annual) {
                const usersToRemove = [];
                const userPPermissionApprovementsMap = await dal_manager_1.dbManager.accessPacs.getUserPPermissionApprovementsBulk(organizationId, ppermissionId, removedUsers, trx);
                for (const userId of removedUsers) {
                    const userPPermissionApprovements = userPPermissionApprovementsMap.get(userId) ?? [];
                    const hasPending = userPPermissionApprovements.some((upa) => upa.status === false);
                    if (!hasPending) {
                        usersToRemove.push(userId);
                    }
                }
                if (usersToRemove.length > 0) {
                    await dal_manager_1.dbManager.accessPacs2.updateRemainingAnnualPPermissionOfUsersByApproval(organizationId, usersToRemove, ppermissionPrevious.ppermissionTypeId, ppermissionPrevious.dateRange.startDateTime, ppermissionPrevious.dateRange.endDateTime, trx, true);
                }
                if (preApproved && addedUsers.length > 0) {
                    await dal_manager_1.dbManager.accessPacs2.updateRemainingAnnualPPermissionOfUsersByApproval(organizationId, addedUsers, ppermissionPrevious.ppermissionTypeId, ppermissionPrevious.dateRange.startDateTime, ppermissionPrevious.dateRange.endDateTime, trx, false);
                }
            }
            await trx.raw(`
                DELETE FROM "${organizationId}"."userPPermissions" AS upp
                WHERE upp."ppermissionId" = ? AND upp."userId" IN (
                    SELECT "userId" FROM "${organizationId}"."userSelectionSessionActions" AS ussa 
                    WHERE "ussa"."sessionId" = ? AND "ussa"."action" = ?
                )`, [ppermissionId, userSelectionSessionId, dal_constants_1.DalConstants.UserSelectionSessionAction.Removed]);
            await trx.raw(`
                DELETE FROM "${organizationId}"."userPPermissionApprovements" AS upp
                WHERE upp."ppermissionId" = ? AND upp."userId" IN (
                    SELECT "userId" FROM "${organizationId}"."userSelectionSessionActions" AS ussa 
                    WHERE "ussa"."sessionId" = ? AND "ussa"."action" = ?
                )`, [ppermissionId, userSelectionSessionId, dal_constants_1.DalConstants.UserSelectionSessionAction.Removed]);
            await trx.raw(`INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userPPermissions}" 
                (id, "userId", "ppermissionId", "currentApprovementOrder")
                SELECT uuid_generate_v4(), ussa."userId", ? , '1' FROM "${organizationId}"."userSelectionSessionActions" AS "ussa" 
                where "ussa"."sessionId" = ? AND "ussa"."action" = ?
				ON CONFLICT ("userId", "ppermissionId")
				DO UPDATE
				SET
					"currentApprovementOrder" = EXCLUDED."currentApprovementOrder";
                `, [session.relatedItemId, session.id, dal_constants_1.DalConstants.UserSelectionSessionAction.Added]);
            let existingApprovals = (await trx.raw(`
                SELECT 
                SUM(CASE WHEN "approverUserId" IS NULL THEN 1 ELSE 0 END) as "waiting",
                SUM(CASE WHEN "approverUserId" IS NOT NULL AND status THEN 1 ELSE 0 END) as "accepted",
                SUM(CASE WHEN "approverUserId" IS NOT NULL AND NOT status THEN 1 ELSE 0 END) as "rejected"
                FROM "${organizationId}"."userPPermissionApprovements"
                WHERE "ppermissionId" = ?
            `, [ppermissionId])).rows[0];
            await this.dbClient.withSchema(organizationId).table("userSelectionSessions").where("id", userSelectionSessionId).transacting(trx).del();
            let createApprovementListResult = null;
            if (addedUsers.length > 0) {
                createApprovementListResult = await this.createApprovementListForPPermission(organizationId, ppermissionId, addedUsers, trx);
            }
            let ppermission = await this.getPPermissionLogBasicTrx(organizationId, ppermissionId, trx);
            await dal_manager_1.dbManager.accessLogPacs.insertPPermissionHistoryContentLog(organizationId, ppermissionId, requestUserId, new Date(), dal_constants_1.DalConstants.PPermissionHistoryActionType.EmployeeListChanged, {
                typeId: ppermission.ppermissionTypeId,
                startDate: ppermission.dateRange.startDateTime,
                endDate: ppermission.dateRange.endDateTime,
                status: ppermission.status,
                userAndDeputyList: ppermission.userAndDeputyList.map((m) => {
                    return {
                        userId: m.userId,
                        deputyUserId: m.deputyUserId,
                    };
                }),
                approvementList: createApprovementListResult && createApprovementListResult.userApprovementList
                    ? createApprovementListResult.userApprovementList.map((m) => {
                        return {
                            approverUserId: m.approverUserId,
                            organizationUnitId: m.organizationUnitId,
                            isHr: m.isHr,
                            order: m.order,
                            note: m.note,
                            isApproved: m.isApproved,
                        };
                    })
                    : null,
            });
            let approvalStatus = dal_constants_1.DalConstants.PPermissionStatus.Waiting;
            if (existingApprovals) {
                if (existingApprovals.waiting > 0) {
                    approvalStatus = dal_constants_1.DalConstants.PPermissionStatus.Waiting;
                }
                else if (existingApprovals.rejected > 0) {
                    if (addedUsers.length > 0 && !preApproved) {
                        approvalStatus = dal_constants_1.DalConstants.PPermissionStatus.Waiting;
                    }
                    else {
                        approvalStatus = dal_constants_1.DalConstants.PPermissionStatus.Rejected;
                    }
                }
                else {
                    if (addedUsers.length > 0 && !preApproved) {
                        approvalStatus = dal_constants_1.DalConstants.PPermissionStatus.Waiting;
                    }
                    else {
                        approvalStatus = dal_constants_1.DalConstants.PPermissionStatus.Approved;
                    }
                }
            }
            else {
                approvalStatus = preApproved ? dal_constants_1.DalConstants.PPermissionStatus.Approved : dal_constants_1.DalConstants.PPermissionStatus.Waiting;
            }
            await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissions).where("id", ppermissionId).update({
                status: approvalStatus,
            });
            if (preApproved) {
                const systemUserId = dal_constants_1.DalConstants.SystemUserId;
                await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.userPPermissionApprovements).where("ppermissionId", ppermissionId).whereIn("userId", addedUsers).update({
                    status: true,
                    approvementDate: new Date(),
                    approverUserId: systemUserId,
                    organizationUnitId: null,
                });
                await dal_manager_1.dbManager.accessLogPacs.insertPPermissionHistoryContentLog(organizationId, ppermissionId, requestUserId, new Date(), dal_constants_1.DalConstants.PPermissionHistoryActionType.Approved, {
                    typeId: ppermission.ppermissionTypeId,
                    startDate: ppermission.dateRange.startDateTime,
                    endDate: ppermission.dateRange.endDateTime,
                    status: dal_constants_1.DalConstants.PPermissionStatus.Approved,
                    userAndDeputyList: ppermission.userAndDeputyList.map((m) => {
                        return {
                            userId: m.userId,
                            deputyUserId: m.deputyUserId,
                        };
                    }),
                    approvementList: createApprovementListResult && createApprovementListResult.userApprovementList
                        ? createApprovementListResult.userApprovementList.map((m) => {
                            return {
                                approverUserId: systemUserId,
                                organizationUnitId: null,
                                isHr: true,
                                order: 0,
                                note: m.note,
                                isApproved: true,
                            };
                        })
                        : null,
                });
            }
        });
        (0, business_pacs_ppermission_1.sendNextPPermissionNotification)({ organizationId, ppermissionId, userIds: addedUsers, requesterUserId: requestUserId });
        return Promise.resolve();
    }
    async upsertPPermissionWithUsers(organizationId, args) {
        let id = args.id || uuid_1.default.v4();
        let now = new Date();
        await this.dbClient.transaction(async (trx) => {
            let type = await this.getPPermissionType(organizationId, args.ppermissionTypeId, trx);
            if (type.noteIsMandatory && (!args.note || args.note.trim() === "")) {
                (0, dal_access_error_1.throwDbAccessBadRequestError)("Note is mandatory for this type of permission.");
            }
            if (args.id) {
                const editable = await this.isPPermissionEditableByRole(organizationId, args.id, dal_constants_1.DalConstants.PPermissionInsertRole.HR, trx);
                if (editable) {
                    await this.updatePPermission(organizationId, args, trx);
                }
            }
            else {
                await this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissions).transacting(trx).insert({
                    id: id,
                    organizationId: organizationId,
                    ppermissionTypeId: args.ppermissionTypeId,
                    startDateTime: args.startDateTime,
                    endDateTime: args.endDateTime,
                    note: args.note,
                    status: dal_constants_1.DalConstants.PPermissionStatus.Waiting,
                    requestUserId: args.requestUserId,
                    requestRole: dal_constants_1.DalConstants.PPermissionInsertRole.HR,
                    requestDateTime: now,
                    dynamicFormId: args.dynamicFormId,
                    dynamicFormData: args.dynamicFormData,
                });
                if (args.userAndDeputyList.length > 0) {
                    const values = [];
                    for (const u of args.userAndDeputyList) {
                        values.push(uuid_1.default.v4(), u.userId, id, 1);
                    }
                    const placeholders = args.userAndDeputyList.map(() => `(?, ?, ?, ?)`).join(", ");
                    await trx.raw(`
							INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userPPermissions}"
							(id, "userId", "ppermissionId", "currentApprovementOrder")
							VALUES ${placeholders}
							ON CONFLICT ("userId", "ppermissionId")
							DO UPDATE SET "currentApprovementOrder" = EXCLUDED."currentApprovementOrder"
						`, values);
                    await this.setDeputyOfManagerBulk(organizationId, args.userAndDeputyList, {
                        deputyStartDateTime: args.startDateTime,
                        deputyEndDateTime: args.endDateTime,
                        ppermissionId: id,
                        trx,
                    });
                }
            }
            const createResult = await this.createApprovementListForPPermission(organizationId, id, [], trx);
            if (args.attachmentList?.length) {
                await this.attachFileToPPermission(organizationId, id, args.attachmentList, trx);
            }
            if (args.approve) {
                let ppermissionUpdated = await this.getPPermissionLogBasicTrx(organizationId, id, trx);
                let ppermissionType = await dal_manager_1.dbManager.accessPacs.getPPermissionType(organizationId, ppermissionUpdated.ppermissionTypeId, trx);
                if (ppermissionType.type === app_enums_1.enums.PredefinedPPermissionType.Annual) {
                    const userIds = args.userAndDeputyList.map((user) => user.userId);
                    if (userIds.length > 0) {
                        await dal_manager_1.dbManager.accessPacs2.updateRemainingAnnualPPermissionOfUsersByApproval(organizationId, userIds, ppermissionUpdated.ppermissionTypeId, ppermissionUpdated.dateRange.startDateTime, ppermissionUpdated.dateRange.endDateTime, trx, false);
                    }
                }
                const systemUserId = dal_constants_1.DalConstants.SystemUserId;
                await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissions).where("id", id).update({
                    status: dal_constants_1.DalConstants.PPermissionStatus.Approved,
                });
                await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.userPPermissionApprovements).where("ppermissionId", id).update({
                    status: true,
                    approvementDate: new Date(),
                    approverUserId: systemUserId,
                    organizationUnitId: null,
                });
            }
            await dal_manager_1.dbManager.accessLogPacs.insertPPermissionHistoryContentLog(organizationId, id, args.requestUserId, now, args.id ? dal_constants_1.DalConstants.PPermissionHistoryActionType.CreatedByManager : dal_constants_1.DalConstants.PPermissionHistoryActionType.UpdatedByManager, {
                typeId: args.ppermissionTypeId,
                startDate: args.startDateTime,
                endDate: args.endDateTime,
                status: createResult.status,
                userAndDeputyList: args.userAndDeputyList.map((m) => ({
                    userId: m.userId,
                    deputyUserId: m.deputyUserId,
                })),
                approvementList: createResult.userApprovementList?.map((m) => ({
                    approverUserId: m.approverUserId,
                    organizationUnitId: m.organizationUnitId,
                    isHr: m.isHr,
                    order: m.order,
                    note: m.note,
                    isApproved: args.approve,
                })),
            });
        });
        return Promise.resolve(id);
    }
    async insertMultiplePPermissionsForUser(organizationId, args) {
        let ids = [];
        const now = new Date();
        const userAndDeputy = args.userAndDeputy;
        await this.dbClient.transaction(async (trx) => {
            for (const range of args.ranges) {
                const ppermissionId = uuid_1.default.v4();
                ids.push(ppermissionId);
                await this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissions).transacting(trx).insert({
                    id: ppermissionId,
                    organizationId: organizationId,
                    ppermissionTypeId: args.ppermissionTypeId,
                    startDateTime: range.startDateTime,
                    endDateTime: range.endDateTime,
                    note: args.note,
                    status: dal_constants_1.DalConstants.PPermissionStatus.Waiting,
                    requestUserId: args.requestUserId,
                    requestDateTime: now,
                    requestRole: dal_constants_1.DalConstants.PPermissionInsertRole.HR,
                    dynamicFormId: args.dynamicFormId,
                    dynamicFormData: args.dynamicFormData,
                });
                await this.dbClient
                    .raw(`
						INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userPPermissions}" 
						(id, "userId", "ppermissionId", "currentApprovementOrder")
						VALUES (?, ?, ?, ?)
						ON CONFLICT ("userId", "ppermissionId")
						DO UPDATE 
						SET "currentApprovementOrder" = EXCLUDED."currentApprovementOrder";
					`, [uuid_1.default.v4(), userAndDeputy.userId, ppermissionId, 1])
                    .transacting(trx);
                if (userAndDeputy.deputyUserId) {
                    await this.setDeputyOfManager(organizationId, userAndDeputy.userId, {
                        deputyUserId: userAndDeputy.deputyUserId,
                        deputyStartDateTime: range.startDateTime,
                        deputyEndDateTime: range.endDateTime,
                        ppermissionId,
                        trx: trx,
                    });
                }
                const createResult = await this.createApprovementListForPPermission(organizationId, ppermissionId, [], trx);
                if (args.attachmentList?.length) {
                    await this.attachFileToPPermission(organizationId, ppermissionId, args.attachmentList, trx);
                }
                await dal_manager_1.dbManager.accessLogPacs.insertPPermissionHistoryContentLog(organizationId, ppermissionId, args.requestUserId, now, dal_constants_1.DalConstants.PPermissionHistoryActionType.CreatedByManager, {
                    typeId: args.ppermissionTypeId,
                    startDate: range.startDateTime,
                    endDate: range.endDateTime,
                    status: createResult.status,
                    userAndDeputyList: userAndDeputy ? [userAndDeputy] : null,
                    approvementList: createResult.userApprovementList
                        ? createResult.userApprovementList.map((m) => {
                            return {
                                approverUserId: m.approverUserId,
                                organizationUnitId: m.organizationUnitId,
                                isHr: m.isHr,
                                order: m.order,
                                note: m.note,
                                isApproved: m.isApproved,
                            };
                        })
                        : null,
                });
            }
        });
        return ids;
    }
    async isPPermissionEditableByRole(organizationId, ppermissionId, editorRole, trx) {
        let existingPPermissionRequestRole = await trx.raw(`
        SELECT "requestRole" FROM "${organizationId}"."ppermissions" WHERE id = ?
        `, [ppermissionId]);
        if (existingPPermissionRequestRole.rows[0] && existingPPermissionRequestRole.rows[0].requestRole < editorRole) {
            (0, dal_access_error_1.throwDbAccessBadRequestError)("This permission was created by a user with higher privilege");
        }
        const systemUserId = dal_constants_1.DalConstants.SystemUserId;
        let approvementCounts = await trx.raw(`
        SELECT "organizationUnitId", COUNT(*) as count, array_agg("approverUserId") as approvers
        FROM "${organizationId}"."userPPermissionApprovements" uppa 
        WHERE uppa."ppermissionId" = ? AND uppa."approverUserId" IS NOT NULL AND status = true
        GROUP BY uppa."organizationUnitId"
        `, [ppermissionId]);
        for (const row of approvementCounts.rows) {
            let areAllApprovementsSystemUser = row.approvers.reduce((acc, curr) => acc && curr === systemUserId, true);
            if (areAllApprovementsSystemUser) {
                continue;
            }
            else if ((row.organizationUnitId === null && row.count > 0 && editorRole > dal_constants_1.DalConstants.PPermissionInsertRole.HR) ||
                (row.organizationUnitId !== null && row.count > 0 && editorRole > dal_constants_1.DalConstants.PPermissionInsertRole.Manager)) {
                (0, dal_access_error_1.throwDbAccessBadRequestError)("This permission includes users whose permission is already approved by hr");
            }
        }
        return true;
    }
    async upsertPPermission(organizationId, args) {
        let id = args.id || uuid_1.default.v4();
        let now = new Date();
        await this.dbClient.transaction(async (trx) => {
            let ppermission = args.approve && args.id ? await this.getPPermissionTrx(organizationId, id, trx) : null;
            let type = await this.getPPermissionType(organizationId, args.ppermissionTypeId, trx);
            if (type.noteIsMandatory && (!args.note || args.note === "")) {
                (0, dal_access_error_1.throwDbAccessBadRequestError)("Note is mandatory for this type of permission.");
            }
            if (args.id) {
                if (await this.isPPermissionEditableByRole(organizationId, args.id, args.requestRole, trx)) {
                    await this.updatePPermission(organizationId, args, trx);
                }
            }
            else {
                await this.dbClient.withSchema(organizationId).table("ppermissions").transacting(trx).insert({
                    id: id,
                    organizationId: organizationId,
                    ppermissionTypeId: args.ppermissionTypeId,
                    startDateTime: args.startDateTime,
                    endDateTime: args.endDateTime,
                    note: args.note,
                    status: dal_constants_1.DalConstants.PPermissionStatus.Waiting,
                    requestUserId: args.requestUserId,
                    requestRole: args.requestRole,
                    requestDateTime: now,
                    dynamicFormId: args.dynamicFormId,
                    dynamicFormData: args.dynamicFormData,
                });
            }
            if (args.attachmentList && args.attachmentList.length > 0) {
                await this.attachFileToPPermission(organizationId, id, args.attachmentList, trx);
            }
            let createApprovementListResult = await this.createApprovementListForPPermission(organizationId, id, [], trx);
            if (args.approve && ppermission) {
                let ppermissionUpdated = await this.getPPermissionLogBasicTrx(organizationId, id, trx);
                let ppermissionType = await dal_manager_1.dbManager.accessPacs.getPPermissionType(organizationId, ppermissionUpdated.ppermissionTypeId, trx);
                if (ppermissionType.type === app_enums_1.enums.PredefinedPPermissionType.Annual) {
                    const userIds = ppermission.userAndDeputyList.map((user) => user.user.id);
                    if (userIds.length > 0) {
                        await dal_manager_1.dbManager.accessPacs2.updateRemainingAnnualPPermissionOfUsersByApproval(organizationId, userIds, ppermissionUpdated.ppermissionTypeId, ppermissionUpdated.dateRange.startDateTime, ppermissionUpdated.dateRange.endDateTime, trx, false);
                    }
                }
                await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissions).where("id", id).update({
                    status: dal_constants_1.DalConstants.PPermissionStatus.Approved,
                });
                const systemUserId = dal_constants_1.DalConstants.SystemUserId;
                await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.userPPermissionApprovements).where("ppermissionId", id).update({
                    status: true,
                    approvementDate: new Date(),
                    approverUserId: systemUserId,
                    organizationUnitId: null,
                });
                await dal_manager_1.dbManager.accessLogPacs.insertPPermissionHistoryContentLog(organizationId, ppermissionUpdated.id, args.requestUserId, new Date(), dal_constants_1.DalConstants.PPermissionHistoryActionType.EmployeeListChanged, {
                    typeId: ppermissionUpdated.ppermissionTypeId,
                    startDate: ppermissionUpdated.dateRange.startDateTime,
                    endDate: ppermissionUpdated.dateRange.endDateTime,
                    status: ppermissionUpdated.status,
                    userAndDeputyList: ppermissionUpdated.userAndDeputyList.map((m) => {
                        return {
                            userId: m.userId,
                            deputyUserId: m.deputyUserId,
                        };
                    }),
                    approvementList: createApprovementListResult.userApprovementList
                        ? createApprovementListResult.userApprovementList.map((m) => {
                            return {
                                approverUserId: systemUserId,
                                organizationUnitId: m.organizationUnitId,
                                isHr: m.isHr,
                                order: m.order,
                                note: m.note,
                                isApproved: true,
                            };
                        })
                        : null,
                });
            }
            else {
                await dal_manager_1.dbManager.accessLogPacs.insertPPermissionHistoryContentLog(organizationId, id, args.requestUserId, now, args.id ? dal_constants_1.DalConstants.PPermissionHistoryActionType.UpdatedByManager : dal_constants_1.DalConstants.PPermissionHistoryActionType.CreatedByManager, {
                    typeId: args.ppermissionTypeId,
                    startDate: args.startDateTime,
                    endDate: args.endDateTime,
                    status: dal_constants_1.DalConstants.PPermissionStatus.Waiting,
                    userAndDeputyList: null,
                    approvementList: null,
                });
            }
        });
        return Promise.resolve(id);
    }
    async upsertSelfPPermission(organizationId, args) {
        let id = args.id || uuid_1.default.v4();
        let now = new Date();
        await this.dbClient.transaction(async (trx) => {
            let type = await this.getPPermissionType(organizationId, args.ppermissionTypeId, trx);
            if (type.noteIsMandatory && (!args.note || args.note === "")) {
                (0, dal_access_error_1.throwDbAccessBadRequestError)("Note is mandatory for this type of permission.");
            }
            if (args.id) {
                let control = await this.updateSelfPPermission(organizationId, args, trx);
                if (!control)
                    return Promise.resolve("");
            }
            else {
                await this.dbClient.withSchema(organizationId).table("ppermissions").transacting(trx).insert({
                    id: id,
                    organizationId: organizationId,
                    ppermissionTypeId: args.ppermissionTypeId,
                    startDateTime: args.startDateTime,
                    endDateTime: args.endDateTime,
                    note: args.note,
                    status: dal_constants_1.DalConstants.PPermissionStatus.Waiting,
                    requestUserId: args.requestUserId,
                    requestRole: dal_constants_1.DalConstants.PPermissionInsertRole.Self,
                    requestDateTime: now,
                    dynamicFormId: args.dynamicFormId,
                    dynamicFormData: args.dynamicFormData,
                });
                if (args.userAndDeputyList?.length > 0) {
                    const values = [];
                    for (const u of args.userAndDeputyList) {
                        values.push(uuid_1.default.v4(), u.userId, id, 1);
                    }
                    const placeholders = args.userAndDeputyList.map(() => `(?, ?, ?, ?)`).join(", ");
                    await trx.raw(`
							INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userPPermissions}"
							(id, "userId", "ppermissionId", "currentApprovementOrder")
							VALUES ${placeholders}
							ON CONFLICT ("userId", "ppermissionId")
							DO UPDATE SET "currentApprovementOrder" = EXCLUDED."currentApprovementOrder"
						`, values);
                    await this.setDeputyOfManagerBulk(organizationId, args.userAndDeputyList, { deputyStartDateTime: args.startDateTime, deputyEndDateTime: args.endDateTime, trx });
                }
            }
            let createResult = await this.createApprovementListForPPermission(organizationId, id, [], trx);
            if (args.attachmentList && args.attachmentList.length > 0) {
                await this.attachFileToPPermission(organizationId, id, args.attachmentList, trx);
            }
            if ((0, moment_1.default)(args.startDateTime).diff(now, "days") < -1) {
            }
            if (type.type === app_enums_1.enums.PredefinedPPermissionType.Annual) {
                let organizationPACSModuleSettings = await dal_manager_1.dbManager.accessPacs.getOrganizationPACSModuleSettings(organizationId);
                if (organizationPACSModuleSettings.ppermissionClaimProfile === app_enums_1.enums.PPermissionClaimProfile.Manual && (!type.approvementList || type.approvementList.length < 1)) {
                    await dal_manager_1.dbManager.accessPacs2.updateRemainingAnnualPPermissionOfUserByApprove(organizationId, args.userAndDeputyList[0].userId, type.id, args.startDateTime, args.endDateTime);
                }
            }
            await dal_manager_1.dbManager.accessLogPacs.insertPPermissionHistoryContentLog(organizationId, id, args.requestUserId, now, args.id ? dal_constants_1.DalConstants.PPermissionHistoryActionType.UpdatedBySelf : dal_constants_1.DalConstants.PPermissionHistoryActionType.CreatedBySelf, {
                typeId: args.ppermissionTypeId,
                startDate: args.startDateTime,
                endDate: args.endDateTime,
                status: createResult.status,
                userAndDeputyList: args.userAndDeputyList
                    ? args.userAndDeputyList.map((m) => {
                        return {
                            userId: m.userId,
                            deputyUserId: m.deputyUserId,
                        };
                    })
                    : null,
                approvementList: createResult.userApprovementList
                    ? createResult.userApprovementList.map((m) => {
                        return {
                            approverUserId: m.approverUserId,
                            organizationUnitId: m.organizationUnitId,
                            isHr: m.isHr,
                            order: m.order,
                            note: m.note,
                            isApproved: m.isApproved,
                        };
                    })
                    : null,
            });
        });
        return Promise.resolve(id);
    }
    async updatePPermission(organizationId, args, trx) {
        let ppermission = await this.getPPermissionTrx(organizationId, args.id, trx);
        let qb = this.dbClient.withSchema(organizationId).table("ppermissions").where("id", args.id);
        if (trx)
            qb.transacting(trx);
        await qb.update({
            organizationId: organizationId,
            ppermissionTypeId: args.ppermissionTypeId,
            startDateTime: args.startDateTime,
            endDateTime: args.endDateTime,
            note: args.note,
            requestUserId: args.requestUserId,
            requestDateTime: new Date(),
            dynamicFormId: args.dynamicFormId,
            dynamicFormData: args.dynamicFormData,
        });
        let ppermissionType = await dal_manager_1.dbManager.accessPacs.getPPermissionType(organizationId, ppermission.ppermissionType.id, trx);
        if (ppermissionType.type === app_enums_1.enums.PredefinedPPermissionType.Annual) {
            const usersToRestore = [];
            const userIds = ppermission.userAndDeputyList.map((u) => u.user.id);
            const approvementsMap = await dal_manager_1.dbManager.accessPacs.getUserPPermissionApprovementsBulk(organizationId, ppermission.id, userIds, trx);
            for (const userId of userIds) {
                const approvements = approvementsMap.get(userId) ?? [];
                const hasPending = approvements.some((a) => a.status === false);
                if (!hasPending) {
                    usersToRestore.push(userId);
                }
            }
            if (usersToRestore.length > 0) {
                await dal_manager_1.dbManager.accessPacs2.updateRemainingAnnualPPermissionOfUsersByApproval(organizationId, usersToRestore, ppermission.ppermissionType.id, ppermission.dateRange.startDateTime, ppermission.dateRange.endDateTime, trx, true);
            }
        }
        await this.removePPermissionApprovements(organizationId, ppermission.id, trx);
        return Promise.resolve();
    }
    async updateSelfPPermission(organizationId, args, trx) {
        let ppermission = await this.getPPermissionTrx(organizationId, args.id, trx);
        if (ppermission && ppermission.userAndDeputyList.length > 1)
            return Promise.resolve(false);
        if (ppermission && ppermission.requestUserId && ppermission.requestUserId != args.userAndDeputyList[0].userId)
            return Promise.resolve(false);
        await this.dbClient.withSchema(organizationId).table("ppermissions").transacting(trx).where("id", args.id).update({
            organizationId: organizationId,
            ppermissionTypeId: args.ppermissionTypeId,
            startDateTime: args.startDateTime,
            endDateTime: args.endDateTime,
            note: args.note,
            requestUserId: args.requestUserId,
            requestDateTime: new Date(),
            status: dal_constants_1.DalConstants.PPermissionStatus.Waiting,
            dynamicFormId: args.dynamicFormId,
            dynamicFormData: args.dynamicFormData,
        });
        await this.removePPermissionDeputyInfo(organizationId, args.id, trx);
        await this.setDeputyOfManagerBulk(organizationId, args.userAndDeputyList, {
            deputyStartDateTime: args.startDateTime,
            deputyEndDateTime: args.endDateTime,
            ppermissionId: args.id,
            trx: trx,
        });
        await this.removePPermissionApprovementsForUser(organizationId, args.id, args.requestUserId, trx);
        if (ppermission.ppermissionType.type === app_enums_1.enums.PredefinedPPermissionType.Annual) {
            let organizationPACSModuleSettings = await dal_manager_1.dbManager.accessPacs.getOrganizationPACSModuleSettings(organizationId);
            if (organizationPACSModuleSettings.ppermissionClaimProfile === app_enums_1.enums.PPermissionClaimProfile.Manual &&
                (!ppermission.ppermissionType.approvementList || ppermission.ppermissionType.approvementList.length < 1)) {
                await dal_manager_1.dbManager.accessPacs2.updateRemainingAnnualPPermissionOfUserByApprove(organizationId, args.userAndDeputyList[0].userId, args.ppermissionTypeId, ppermission.dateRange.startDateTime, ppermission.dateRange.endDateTime, true);
            }
        }
        return Promise.resolve(true);
    }
    async removePPermissionApprovements(organizationId, ppermissionId, trx) {
        let qb = this.dbClient.withSchema(organizationId).from("userPPermissionApprovements").where("ppermissionId", ppermissionId);
        if (trx)
            qb.transacting(trx);
        await qb.del();
        let qbx = this.dbClient.withSchema(organizationId).from("userPPermissions").where("ppermissionId", ppermissionId);
        if (trx)
            qbx = qbx.transacting(trx);
        await qbx.update({
            currentApprovementOrder: 1,
        });
        return Promise.resolve();
    }
    async removePPermissionApprovementsForUser(organizationId, ppermissionId, userId, trx) {
        let qb = this.dbClient.withSchema(organizationId).from("userPPermissionApprovements").where("ppermissionId", ppermissionId).where("userId", userId);
        if (trx)
            qb.transacting(trx);
        await qb.del();
        let qbx = this.dbClient.withSchema(organizationId).from("userPPermissions").where("ppermissionId", ppermissionId).where("userId", userId);
        if (trx)
            qbx = qbx.transacting(trx);
        await qbx.update({
            currentApprovementOrder: 1,
        });
        return Promise.resolve();
    }
    async listPPermissionUserApprovementsAndStatus(organizationId, ppermissionId, trx) {
        let ppermission = await this.getPPermissionTrx(organizationId, ppermissionId, trx);
        let userApprovements = await trx
            .withSchema(organizationId)
            .table("ppermissions as p")
            .innerJoin("userPPermissions as up", "up.ppermissionId", "p.id")
            .innerJoin("userPPermissionApprovements as upa", (join) => {
            join.on("upa.ppermissionId", "p.id").andOn("up.userId", "upa.userId");
        })
            .where("p.id", ppermissionId)
            .select("upa.order", "upa.organizationUnitId", "upa.userId", "upa.note", "upa.status", "upa.approverUserId");
        let result = {
            status: ppermission.status,
            userApprovementList: userApprovements.map((m) => {
                return {
                    organizationUnitId: m.organizationUnitId,
                    order: m.order,
                    approverUserId: m.approverUserId,
                    isHr: m.organizationUnitId == null,
                    note: m.note,
                    isApproved: m.status,
                };
            }),
        };
        return Promise.resolve(result);
    }
    async createApprovementListForPPermissionUser(organizationId, ppermission, userId, trx) {
        let userApprovements = [];
        let approvementList = ppermission.ppermissionType.approvementList;
        const approvementOrderHook = business_hooks_1.armonHookManager.getChangePermissionApprovementOrderHook(organizationId);
        if (approvementOrderHook) {
            approvementList = await approvementOrderHook(organizationId, userId, approvementList, ppermission.ppermissionType.id);
        }
        const pairsOfManagersAndOrganizationUnit = await this.getPairOfOrganizationUnitAndManager(organizationId, trx);
        if (approvementList && approvementList.length > 0) {
            const hrApprovementStatus = approvementList.includes(null) ? (approvementList[0] === null ? "first" : "last") : "none";
            const unitApprovementCounts = {};
            approvementList.filter((a) => a).map((a) => (unitApprovementCounts[a] ? (unitApprovementCounts[a] = unitApprovementCounts[a] + 1) : (unitApprovementCounts[a] = 1)));
            const unitsOfUser = await this.getUserOrganizationUnitList(organizationId, userId, null, trx);
            const pairsOfManagersAndOrganizationUnit = await this.getPairOfOrganizationUnitAndManager(organizationId, trx);
            const approvementSteps = {};
            for (const unit of unitsOfUser) {
                let path = unit.ancestorIds ? unit.ancestorIds.split(",") : [];
                path.push(unit.organizationUnitId);
                const stepCount = (hrApprovementStatus === "first" ? unitApprovementCounts[path[0]] + 1 : unitApprovementCounts[path[0]]) ?? 0;
                path = path.reverse();
                let order = hrApprovementStatus === "first" ? 2 : 1;
                while (order <= stepCount && path.length > 0) {
                    const unitId = path.shift();
                    const unit = pairsOfManagersAndOrganizationUnit.find((f) => f.organizationUnitId === unitId);
                    if (unit && unit.managerUserId && unit.managerUserId !== userId) {
                        let previousOrder = approvementSteps[unitId] ?? -1;
                        let assignedOrder = order < previousOrder ? previousOrder : order;
                        approvementSteps[unitId] = assignedOrder;
                        order = assignedOrder + 1;
                    }
                }
            }
            if (hrApprovementStatus === "first") {
                approvementSteps["null"] = 1;
            }
            else if (hrApprovementStatus === "last") {
                approvementSteps["null"] = Math.max(...Object.values(approvementSteps), 0) + 1;
            }
            for (const key of Object.keys(approvementSteps)) {
                userApprovements.push({
                    id: uuid_1.default.v4(),
                    ppermissionId: ppermission.id,
                    userId: userId,
                    organizationUnitId: key === "null" ? null : key,
                    order: approvementSteps[key],
                    status: false,
                });
            }
        }
        if (userApprovements.length === 0) {
            const systemUserId = dal_constants_1.DalConstants.SystemUserId;
            userApprovements.push({
                id: uuid_1.default.v4(),
                ppermissionId: ppermission.id,
                userId: userId,
                organizationUnitId: null,
                order: 1,
                approverUserId: systemUserId,
                approvementDate: new Date(),
                note: "",
                status: true,
            });
        }
        const hook = business_hooks_1.armonHookManager.getPermissionApprovementLevelHook(organizationId);
        if (hook) {
            userApprovements = await hook(approvementList, userApprovements);
        }
        return Promise.resolve(userApprovements);
    }
    async createApprovementListForPPermission(organizationId, ppermissionId, forUsers, trx) {
        let ppermission = await this.getPPermissionTrx(organizationId, ppermissionId, trx);
        if (forUsers.length === 0) {
            forUsers = ppermission.userAndDeputyList.map((u) => u.user.id);
        }
        let userApprovements = [];
        for (let userId of forUsers) {
            let approvements = await this.createApprovementListForPPermissionUser(organizationId, ppermission, userId, trx);
            if (approvements)
                userApprovements = userApprovements.concat(approvements);
        }
        await this.dbClient.withSchema(organizationId).transacting(trx).table("userPPermissionApprovements").insert(userApprovements);
        let result = {
            status: dal_constants_1.DalConstants.PPermissionStatus.Waiting,
            userApprovementList: userApprovements.map((m) => {
                return {
                    organizationUnitId: m.organizationUnitId,
                    order: m.order,
                    approverUserId: m.approverUserId,
                    isHr: m.organizationUnitId == null,
                    note: m.note,
                    isApproved: m.status,
                };
            }),
        };
        if (userApprovements.filter((a) => a.status === true).length === userApprovements.length) {
            result.status = dal_constants_1.DalConstants.PPermissionStatus.Approved;
            await this.dbClient.withSchema(organizationId).transacting(trx).table("ppermissions").where("id", ppermissionId).update({
                status: result.status,
            });
        }
        return Promise.resolve(result);
    }
    async removePPermission(organizationId, ppermissionId, isHR, userId, isUnitManager, permittedUnitIds) {
        let affectedRowCount = 0;
        await this.dbClient.transaction(async (trx) => {
            if (await this.isPPermissionEditableByRole(organizationId, ppermissionId, isHR ? dal_constants_1.DalConstants.PPermissionInsertRole.HR : dal_constants_1.DalConstants.PPermissionInsertRole.Manager, trx)) {
                let ppermission = await this.getPPermissionTrx(organizationId, ppermissionId, trx);
                let ppermissionType = await dal_manager_1.dbManager.accessPacs.getPPermissionType(organizationId, ppermission.ppermissionType.id, trx);
                if (ppermissionType.type === app_enums_1.enums.PredefinedPPermissionType.Annual) {
                    const usersToRemove = [];
                    const userIds = ppermission.userAndDeputyList.map((u) => u.user.id);
                    const userPPermissionApprovementsMap = await dal_manager_1.dbManager.accessPacs.getUserPPermissionApprovementsBulk(organizationId, ppermissionId, userIds, trx);
                    for (const userId of userIds) {
                        const userPPermissionApprovements = userPPermissionApprovementsMap.get(userId) ?? [];
                        const hasPending = userPPermissionApprovements.some((upa) => upa.status === false);
                        if (!hasPending) {
                            usersToRemove.push(userId);
                        }
                    }
                    if (usersToRemove.length > 0) {
                        await dal_manager_1.dbManager.accessPacs2.updateRemainingAnnualPPermissionOfUsersByApproval(organizationId, usersToRemove, ppermission.ppermissionType.id, ppermission.dateRange.startDateTime, ppermission.dateRange.endDateTime, trx, true);
                    }
                }
                if (isHR) {
                    await trx
                        .withSchema(organizationId)
                        .table("ppermissions")
                        .where("id", ppermissionId)
                        .del()
                        .then((result) => (affectedRowCount = result));
                }
                else if (isUnitManager && ppermission.status == dal_constants_1.DalConstants.PPermissionStatus.Waiting) {
                    let permittedUserIdsResult = await trx
                        .withSchema(organizationId)
                        .from("userOrganizations as uo")
                        .innerJoin("userOrganizationOrganizationUnits as uoou", "uoou.userOrganizationId", "uo.id")
                        .whereIn("uoou.organizationUnitId", permittedUnitIds)
                        .select("uo.userId");
                    let permittedUserIds = permittedUserIdsResult.map((pui) => pui.userId);
                    for (const userAndDeputy of ppermission.userAndDeputyList) {
                        if (!permittedUserIds.includes(userAndDeputy.user.id)) {
                            return false;
                        }
                    }
                    await trx
                        .withSchema(organizationId)
                        .table("ppermissions")
                        .where("id", ppermissionId)
                        .del()
                        .then((result) => (affectedRowCount = result));
                }
                else {
                    await trx
                        .withSchema(organizationId)
                        .table("ppermissions")
                        .where("id", ppermissionId)
                        .where("requestUserId", userId)
                        .del()
                        .then((result) => (affectedRowCount = result));
                }
                await dal_manager_1.dbManager.accessLogPacs.insertPPermissionHistoryContentLog(organizationId, ppermissionId, userId, new Date(), dal_constants_1.DalConstants.PPermissionHistoryActionType.Deleted, {
                    typeId: ppermission.ppermissionType.id,
                    startDate: ppermission.dateRange.startDateTime,
                    endDate: ppermission.dateRange.endDateTime,
                    status: ppermission.status,
                    userAndDeputyList: ppermission.userAndDeputyList
                        ? ppermission.userAndDeputyList.map((m) => {
                            return {
                                userId: m.user.id,
                                deputyUserId: m.deputyUser ? m.deputyUser.id : null,
                            };
                        })
                        : null,
                    approvementList: null,
                });
            }
        });
        return Promise.resolve(affectedRowCount > 0);
    }
    async listUsersCurrentPPermissionInfoWithType(organizationId, userIds, startDate, endDate) {
        let result = [];
        await this.dbClient
            .withSchema(organizationId)
            .table("ppermissions as p")
            .innerJoin("userPPermissions as up", "up.ppermissionId", "p.id")
            .innerJoin("ppermissionTypes as pt", "pt.id", "p.ppermissionTypeId")
            .whereIn("up.userId", userIds)
            .whereRaw('tstzrange(p."startDateTime", p."endDateTime") && tstzrange(?, ?)', [startDate, endDate])
            .select("p.status", "p.startDateTime", "p.endDateTime", "pt.id", "up.userId", "p.id as ppermissionId")
            .then(async (rows) => {
            for (let row of rows) {
                result.push({
                    ppermissionId: row.ppermissionId,
                    userId: row.userId,
                    status: row.status,
                    startDateTime: row.startDateTime,
                    endDateTime: row.endDateTime,
                    ppermissionType: await this.getPPermissionType(organizationId, row.id),
                });
            }
        });
        return Promise.resolve(result);
    }
    async listOldPPermissionsForValidation(organizationId, userId, ppermissionTypeId, startDateTime, endDateTime, ppermissionId) {
        let ppermissionCount = 0;
        let qb = this.dbClient
            .withSchema(organizationId)
            .table("ppermissions as p")
            .innerJoin("userPPermissions as up", "up.ppermissionId", "p.id")
            .where("up.userId", userId)
            .where("p.ppermissionTypeId", ppermissionTypeId)
            .where("p.status", "!=", dal_constants_1.DalConstants.PPermissionStatus.Rejected)
            .whereWrapped((w) => {
            w.where("p.startDateTime", "<", endDateTime).andWhere("p.endDateTime", ">", startDateTime);
        });
        if (ppermissionId)
            qb.where("p.id", "!=", ppermissionId);
        await qb.count().then((result) => {
            ppermissionCount = parseInt(result[0].count);
        });
        return Promise.resolve(ppermissionCount > 0);
    }
    async listOldPPermissionsForValidationBulk(organizationId, userIds, ppermissionTypeId, startDateTime, endDateTime, ppermissionId) {
        const resultMap = new Map();
        if (userIds.length === 0)
            return resultMap;
        let qb = this.dbClient
            .withSchema(organizationId)
            .table("ppermissions as p")
            .innerJoin("userPPermissions as up", "up.ppermissionId", "p.id")
            .whereIn("up.userId", userIds)
            .andWhere("p.ppermissionTypeId", ppermissionTypeId)
            .andWhere("p.status", "!=", dal_constants_1.DalConstants.PPermissionStatus.Rejected)
            .andWhere((w) => {
            w.where("p.startDateTime", "<", endDateTime).andWhere("p.endDateTime", ">", startDateTime);
        });
        if (ppermissionId) {
            qb.andWhere("p.id", "!=", ppermissionId);
        }
        const rows = await qb.select("up.userId").groupBy("up.userId");
        for (const userId of userIds) {
            const hasConflict = rows.some((r) => r.userId === userId);
            resultMap.set(userId, hasConflict);
        }
        return resultMap;
    }
    async listPermissionUnified(organizationId, listUserId, listAs, args) {
        return await dal_manager_1.dbManager.pgTransactionMainDb(async (trx) => {
            if (listAs === dal_constants_1.DalConstants.PPermissionInsertRole.Self)
                args.userIds = [listUserId];
            let visibleUserOrganizationUnitIds;
            if (listAs === dal_constants_1.DalConstants.PPermissionInsertRole.HR) {
                visibleUserOrganizationUnitIds = await this.pgGetVisibleOrganizationUnitsAsUnitHR(organizationId, listUserId, true, trx);
            }
            else {
                visibleUserOrganizationUnitIds = await this.pgGetVisibleOrganizationUnitsAsManagerOrDeputy(organizationId, listUserId, true, new Date(), trx);
            }
            let response = {
                items: [],
                pagination: {
                    take: args.take,
                    skip: args.skip,
                    total: 0,
                },
            };
            let approveCheck = "";
            if (listAs === dal_constants_1.DalConstants.PPermissionInsertRole.HR) {
                approveCheck = `uppa_1.order = upp_2."currentApprovementOrder" AND uppa_1."approverUserId" IS NULL AND uppa_1."organizationUnitId" IS NULL`;
            }
            else if (listAs === dal_constants_1.DalConstants.PPermissionInsertRole.Manager) {
                let approvableOrganizationUnitIds = await this.getManagerOrDeputyOrganizationList(organizationId, listUserId, new Date());
                approveCheck = `uppa_1.order = upp_2."currentApprovementOrder" AND uppa_1."approverUserId" IS NULL AND uppa_1."organizationUnitId"::text = ANY(array['${approvableOrganizationUnitIds.join(`','`)}'])`;
            }
            else {
                approveCheck = `false`;
            }
            let bindings = [];
            let bindIndex = 0;
            let query = `
            SELECT
                pp.id,
                upp."userCount",
                pp."requestUserId",
                pp."requestRole",
                pp.note,
				pp."dynamicFormData",
				pp."dynamicFormId",
                ppt.data as "ppermissionType",
                json_build_object('startDateTime', pp."startDateTime", 'endDateTime', pp."endDateTime") as "dateRange",
                pp.status,
                upp."userPPermissionInfo",
                CASE WHEN patt.data IS NOT NULL THEN patt.data ELSE '[]'::jsonb END as "attachmentInfoList"
            FROM "${organizationId}"."ppermissions" pp 
            INNER JOIN (
                SELECT
                    up_1."ppermissionId",
                    jsonb_agg(jsonb_build_object(
                        'userAndDeputyInfo', jsonb_build_object('user', up_1."userId",'deputyUser', uod."deputyUserOrganizationId"),
                        'userSigned', up_1."signedByUser",
                        'canListerApprove', uppa."rejectCount" = 0 AND true = ANY(uppa."canListerApprove"),
                        'approvementInfo', uppa.data,
                        'currentApprovementOrder', up_1."currentApprovementOrder",
                        'status', 
                        CASE
							WHEN uppa."rejectCount" > 0 THEN 2
                            WHEN (uppa."totalCount" = uppa."acceptCount") OR (uppa.data IS NULL) THEN 1
                            WHEN (uppa."totalCount" = uppa."acceptCount" + uppa."rejectCount") AND (uppa."rejectCount" > 0) THEN 2
                            ELSE 0
                        END
                    ))
                    -- USER FILTERING COMES IN HERE
                    ${visibleUserOrganizationUnitIds.length > 0 || (args.userIds && args.userIds.length > 0) ? ` FILTER (WHERE ` : ``}
                    `;
            if (visibleUserOrganizationUnitIds.length > 0) {
                query +=
                    `uoou.ouids && array['` +
                        visibleUserOrganizationUnitIds.join(`','`) +
                        `']::uuid[]
						`;
            }
            if (args.userIds && args.userIds.length > 0) {
                if (visibleUserOrganizationUnitIds.length > 0) {
                    query += "AND ";
                }
                query +=
                    `up_1."userId" = ANY(array['` +
                        args.userIds.join(`','`) +
                        `']::uuid[])
                        `;
            }
            query += `
                    ${visibleUserOrganizationUnitIds.length > 0 || (args.userIds && args.userIds.length > 0) ? `)` : ``}
                    as "userPPermissionInfo",
                    COUNT(up_1."userId") AS "userCount"
                FROM "${organizationId}"."userPPermissions" up_1
                INNER JOIN "${organizationId}"."userOrganizations" uo ON uo."isDisabled" = false AND uo."deletedAt" IS NULL AND uo."userId" = up_1."userId" AND uo."organizationId" = $${++bindIndex}
                LEFT JOIN (
                    SELECT uppa_1."ppermissionId", uppa_1."userId",
                        jsonb_agg(
                            jsonb_build_object(
                                'approvementDate', uppa_1."approvementDate",
                                'approverUserId', uppa_1."approverUserId",
                                'organizationUnit', uppa_1."organizationUnitId",
                                'organizationUnitName', ounit."name",
                                'isHR', uppa_1."organizationUnitId" IS NULL,
                                'note', uppa_1.note,
                                'order', uppa_1.order,
                                'status', CASE WHEN uppa_1."approvementDate" IS NULL THEN 0 ELSE CASE WHEN uppa_1.status THEN 1 ELSE 2 END END,
                                'canListerApprove', ${approveCheck}
                            )
                        ) as data,
                        COUNT(*) as "totalCount",
                        COUNT(*) FILTER (WHERE uppa_1."approvementDate" IS NULL) as "waitingCount",
                        COUNT(*) FILTER (WHERE uppa_1."approvementDate" IS NOT NULL AND uppa_1.status) as "acceptCount",
                        COUNT(*) FILTER (WHERE uppa_1."approvementDate" IS NOT NULL AND NOT uppa_1.status) as "rejectCount",
                        COUNT(*) FILTER (WHERE uppa_1."approvementDate" IS NOT NULL AND uppa_1.status AND uppa_1."approverUserId" = $${++bindIndex}) as "acceptedByMeCount",
                        COUNT(*) FILTER (WHERE uppa_1."approvementDate" IS NOT NULL AND NOT uppa_1.status AND uppa_1."approverUserId" = $${++bindIndex}) as "rejectedByMeCount",
                        array_agg(${approveCheck}) as "canListerApprove"
                    FROM "${organizationId}"."userPPermissionApprovements" uppa_1
                    INNER JOIN "${organizationId}"."userPPermissions" upp_2 ON upp_2."ppermissionId" = uppa_1."ppermissionId" AND upp_2."userId" = uppa_1."userId"
                    LEFT JOIN "${organizationId}"."organizationUnits" ounit ON ounit.id = uppa_1."organizationUnitId"
                    GROUP BY uppa_1."ppermissionId", uppa_1."userId"
                ) uppa ON uppa."ppermissionId" = up_1."ppermissionId" AND uppa."userId" = up_1."userId"
                LEFT JOIN "${organizationId}"."userOrganizationDeputies" uod ON uod."ppermissionId" = up_1."ppermissionId" AND uod."userOrganizationId" = uo.id
                LEFT JOIN (
                    SELECT uoou_1."userOrganizationId", jsonb_agg(row_to_json(ou_1)) as data, array_agg(ou_1.id) as ouids
                    FROM "${organizationId}"."userOrganizationOrganizationUnits" uoou_1
                    INNER JOIN "${organizationId}"."organizationUnits" ou_1 ON ou_1."deletedAt" IS NULL AND ou_1.id = uoou_1."organizationUnitId"
                    WHERE uoou_1."deletedAt" IS NULL
                    GROUP BY uoou_1."userOrganizationId"
                ) uoou ON uoou."userOrganizationId" = uo.id
            `;
            if (args.status) {
                switch (args.status) {
                    case dal_constants_1.DalConstants.PPermissionFilterStatus.ApprovedByMe:
                        query += `WHERE uppa."acceptedByMeCount" > 0
                        `;
                        break;
                    case dal_constants_1.DalConstants.PPermissionFilterStatus.RejectedByMe:
                        query += `WHERE uppa."rejectedByMeCount" > 0
                        `;
                        break;
                    case dal_constants_1.DalConstants.PPermissionFilterStatus.WaitingMyApproval:
                        query += `WHERE uppa."rejectCount" = 0 AND true = ANY(uppa."canListerApprove")
                        `;
                        break;
                    case dal_constants_1.DalConstants.PPermissionFilterStatus.Approved:
                        query += `WHERE uppa."rejectCount" = 0 AND uppa."acceptCount" > 0
                        `;
                        break;
                    case dal_constants_1.DalConstants.PPermissionFilterStatus.Rejected:
                        query += `WHERE uppa."rejectCount" > 0
                        `;
                        break;
                    case dal_constants_1.DalConstants.PPermissionFilterStatus.Waiting:
                        query += `WHERE uppa."rejectCount" = 0 AND uppa."waitingCount" > 0
                        `;
                        break;
                    default:
                        break;
                }
            }
            query += `
                GROUP BY up_1."ppermissionId"
            ) upp ON upp."ppermissionId" = pp.id
            INNER JOIN (
                SELECT ppt_1.id, row_to_json(ppt_1)::jsonb || jsonb_build_object('approvementList', jsonb_agg(ppta."rootOrganizationUnitId")) as data
                FROM "${organizationId}"."ppermissionTypes" ppt_1
                LEFT JOIN "${organizationId}"."ppermissionTypeApprovements" ppta ON ppta."ppermissionTypeId" = ppt_1.id
                GROUP BY ppt_1.id
            ) ppt ON ppt.id = pp."ppermissionTypeId"
            LEFT JOIN (
                SELECT patt_1."ppermissionId", jsonb_agg(jsonb_build_object('id', patt_1.id,'name', patt_1.name)) as data
                FROM "${organizationId}"."ppermissionAttachments" patt_1
                GROUP BY patt_1."ppermissionId"
            ) patt ON patt."ppermissionId" = pp.id
            WHERE pp."organizationId" = $${++bindIndex} AND tstzrange(pp."startDateTime", pp."endDateTime") && tstzrange($${++bindIndex}, $${++bindIndex}) AND upp."userPPermissionInfo" IS NOT NULL
            `;
            bindings.push(organizationId, listUserId, listUserId, organizationId, args.startDateTime ? args.startDateTime : null, args.endDateTime ? args.endDateTime : null);
            if (args.ppermissionTypeId) {
                query += `
                AND ppt.id = $${++bindIndex}`;
                bindings.push(args.ppermissionTypeId);
            }
            if (args.ppermissionIds && args.ppermissionIds.length > 0) {
                query +=
                    `
                AND pp.id = ANY(array['` +
                        args.ppermissionIds.join(`','`) +
                        `']::uuid[])`;
            }
            response.pagination.total = parseInt((await trx.query(`SELECT COUNT(*) as cnt FROM (` + query + `) sq`, bindings)).rows[0].cnt);
            const systemUserId = dal_constants_1.DalConstants.SystemUserId;
            let sortType;
            let sortOrder = ["ASC", "DESC"].includes(args.sortOrder) ? args.sortOrder : "DESC";
            switch (args.sortType) {
                case app_enums_1.enums.PPermissionFilterSortType.PPermissionStart:
                    sortType = ' pp."startDateTime" ';
                    break;
                case app_enums_1.enums.PPermissionFilterSortType.PPermissionEnd:
                    sortType = ' pp."endDateTime" ';
                    break;
                case app_enums_1.enums.PPermissionFilterSortType.PPermissionType:
                    sortType = " ppt.data->'name' ";
                    break;
                case app_enums_1.enums.PPermissionFilterSortType.PPermissionApproval:
                    sortType = " pp.status ";
                    break;
                case app_enums_1.enums.PPermissionFilterSortType.PPermissionDuration:
                    sortType = ' pp."endDateTime"::timestamp - pp."startDateTime"::timestamp ';
                    break;
                default:
                    sortType = ' pp."requestDateTime" ';
                    break;
            }
            query += `
                ORDER BY ${sortType} ${sortOrder}
                OFFSET $${++bindIndex} LIMIT $${++bindIndex}
            `;
            bindings.push(args.skip, args.take);
            let dbResults = await trx.query(query, bindings);
            let userIds = new Set();
            for (const row of dbResults.rows) {
                for (const upp of row.userPPermissionInfo) {
                    userIds.add(upp.userAndDeputyInfo.user);
                    if (upp.userAndDeputyInfo.deputyUser) {
                        userIds.add(upp.userAndDeputyInfo.deputyUser);
                    }
                    if (upp.approvementInfo) {
                        for (const app of upp.approvementInfo) {
                            if (app.approverUserId) {
                                userIds.add(app.approverUserId);
                                app.approverUsers = [app.approverUserId];
                            }
                            else {
                                app.approverUsers = await this.listManagerOrDeputyOfOrganizationUnits(organizationId, [app.organizationUnit], new Date());
                                for (const waitingApprover of app.approverUsers) {
                                    userIds.add(waitingApprover);
                                }
                            }
                        }
                    }
                }
            }
            let userInfoList = await this.getBasicUserInfoList(organizationId, Array.from(userIds));
            for (const row of dbResults.rows) {
                delete row.ppermissionType.createdAt;
                delete row.ppermissionType.updatedAt;
                delete row.ppermissionType.deletedAt;
                delete row.ppermissionType.organziationId;
                const dynamicFormId = row.dynamicFormId ?? row.ppermissionType.dynamicFormId;
                row.ppermissionType.dynamicForm = dynamicFormId ? await dal_manager_1.dbManager.accessSystem.getDynamicForm({ organizationId: organizationId, dynamicFormId: dynamicFormId }) : null;
                delete row.ppermissionType.dynamicFormId;
                delete row.dynamicFormId;
                let areAllApprovementsSystemUser = true;
                let highestApproverRole = undefined;
                let canApproveUserPermissionIds = new Set();
                for (const upp of row.userPPermissionInfo) {
                    let basicUserInfo = userInfoList.find((ui) => ui.id === upp.userAndDeputyInfo.user);
                    if (basicUserInfo) {
                        basicUserInfo.userCaptions = userInfoList.find((uc) => uc.id === basicUserInfo.id).captionLines;
                    }
                    upp.userAndDeputyInfo.user = basicUserInfo;
                    upp.userAndDeputyInfo.userCaptions = basicUserInfo?.captionLines;
                    if (upp.userAndDeputyInfo.deputyUser) {
                        let basicDeputyInfo = userInfoList.find((ui) => ui.id === upp.userAndDeputyInfo.deputy);
                        if (basicDeputyInfo) {
                            basicDeputyInfo.userCaptions = userInfoList.find((uc) => uc.id === basicDeputyInfo.id).captionLines;
                        }
                        upp.userAndDeputyInfo.deputy = basicDeputyInfo;
                        upp.userAndDeputyInfo.deputyCaptions = basicDeputyInfo?.captionLines;
                    }
                    upp.userAndDeputyInfo.deputyUser = upp.userAndDeputyInfo.deputyUser ? userInfoList.find((ui) => ui.id === upp.userAndDeputyInfo.deputyUser) : null;
                    upp.userAndDeputyInfo.deputyCaptions = upp.userAndDeputyInfo.deputyUser
                        ? userInfoList.find((uc) => uc.id === upp.userAndDeputyInfo.deputyUser.id)?.captionLines
                        : null;
                    if (upp.approvementInfo) {
                        for (const app of upp.approvementInfo) {
                            if (app.approverUserId) {
                                app.approverUsers = [userInfoList.find((ui) => ui.id === app.approverUserId)];
                            }
                            else {
                                app.approverUsers = app.approverUsers.map((approver) => {
                                    let basicApproverInfo = userInfoList.find((ui) => ui.id === approver);
                                    if (basicApproverInfo) {
                                        basicApproverInfo.userCaptions = userInfoList.find((uc) => uc.id === basicApproverInfo.id).captionLines;
                                    }
                                    return basicApproverInfo;
                                });
                            }
                            if (app.canListerApprove) {
                                canApproveUserPermissionIds.add(upp.userAndDeputyInfo.user.id);
                            }
                            if (app.approverUserId) {
                                highestApproverRole = Math.min(highestApproverRole !== undefined ? highestApproverRole : dal_constants_1.DalConstants.PPermissionInsertRole.Manager, app.organizationUnit ? dal_constants_1.DalConstants.PPermissionInsertRole.Manager : dal_constants_1.DalConstants.PPermissionInsertRole.HR);
                                areAllApprovementsSystemUser = areAllApprovementsSystemUser && app.approverUserId === systemUserId;
                            }
                        }
                    }
                    else {
                        upp.approvementInfo = [];
                    }
                }
                row.editableStatus = dal_constants_1.DalConstants.PPermissionEditableStatus.Editable;
                if (listAs !== dal_constants_1.DalConstants.PPermissionInsertRole.HR) {
                    if (areAllApprovementsSystemUser) {
                        row.editableStatus = dal_constants_1.DalConstants.PPermissionEditableStatus.Editable;
                    }
                    else if (row.requestRole < listAs) {
                        row.editableStatus = dal_constants_1.DalConstants.PPermissionEditableStatus.CreatorHasHigherPrivilege;
                    }
                    else if (highestApproverRole !== undefined && highestApproverRole < listAs) {
                        row.editableStatus = dal_constants_1.DalConstants.PPermissionEditableStatus.AlreadyApprovedByHigherPrivilegedUser;
                    }
                    else if (row.userCount > row.userPPermissionInfo.length) {
                        row.editableStatus = dal_constants_1.DalConstants.PPermissionEditableStatus.PermissionHasUnlistedUsers;
                    }
                }
                row.approvementInfo = {
                    acceptCount: row.userPPermissionInfo.filter((u) => u.status == dal_constants_1.DalConstants.PPermissionStatus.Approved).length,
                    rejectCount: row.userPPermissionInfo.filter((u) => u.status == dal_constants_1.DalConstants.PPermissionStatus.Rejected).length,
                    waitingCount: row.userPPermissionInfo.filter((u) => u.status == dal_constants_1.DalConstants.PPermissionStatus.Waiting).length,
                    waitingMyApprovalCount: canApproveUserPermissionIds.size,
                };
                row.userCount = listAs === dal_constants_1.DalConstants.PPermissionInsertRole.Self ? 1 : parseInt(row.userCount);
            }
            response.items = dbResults.rows;
            return response;
        });
    }
    async getPPermission(organizationId, ppermissionId) {
        return await dal_manager_1.dbManager.systemTransaction(async (trx) => {
            const ppermissions = (await trx.query(`
					SELECT
						p.id as "leaveId",
						p."startDateTime" as "startDateTime",
						p."endDateTime" as "endDateTime",
						p.note as "note",
						p.status as "status",
						p."requestDateTime" as "requestDateTime",
						p."requestUserId" as "requestUserId",
						p."dynamicFormId" as "pDynamicFormId",
						ARRAY_AGG(DISTINCT up."userId") as "userIds",
						pt.*,
						COALESCE(
							ARRAY_AGG(pta."rootOrganizationUnitId" ORDER BY pta."order") FILTER (WHERE pta.id IS NOT NULL),
							ARRAY[]::UUID[]
						) AS "rootOrgUnitIds"
					FROM "${organizationId}"."ppermissions" AS p
					LEFT JOIN "${organizationId}"."ppermissionTypes" pt
						ON pt.id = p."ppermissionTypeId"
					LEFT JOIN "${organizationId}"."ppermissionTypeApprovements" pta
						ON pt.id = pta."ppermissionTypeId"
					LEFT JOIN "${organizationId}"."userPPermissions" up
    					ON up."ppermissionId" = p."id"
					WHERE p.id = $1
					GROUP BY p.id, pt.id
					`, [ppermissionId])).rows;
            if (ppermissions.length != 1) {
                (0, dal_access_error_1.throwDbAccessConflictError)("no ppermission found or many ppermissions");
            }
            const p = ppermissions[0];
            const usersInfo = (await dal_manager_1.dbManager.accessUser.getBasicUserInfoList(organizationId, p.userIds)).map((user) => {
                return {
                    user,
                };
            });
            const dynamicFormId = p.pdynamicFormId ?? p.dynamicFormId;
            let permissionDetailed = {
                id: p.leaveId,
                ppermissionType: {
                    id: p.id,
                    name: p.name,
                    isPassive: p.isPassive,
                    description: p.description,
                    hrCanInit: p.hrCanInit,
                    personnelCanInit: p.personnelCanInit,
                    isDailyScheduled: p.isDailyScheduled,
                    minValue: p.minValue,
                    maxValue: p.maxValue,
                    maxValueBoundaryInterval: p.maxValueBoundaryInterval,
                    method: p.method,
                    approvementList: p.rootOrgUnitIds,
                    sendNotification: p.sendNotification,
                    userMustSign: p.userMustSign,
                    needsManagerDeputy: p.needsManagerDeputy,
                    needsUserDeputy: p.needsUserDeputy,
                    type: p.type,
                    showRemainingCount: p.showRemainingCount,
                    isUnpaidLeave: p.isUnpaidLeave,
                    notifyHR: p.notifyHR,
                    dynamicForm: dynamicFormId ? await dal_manager_1.dbManager.accessSystem.getDynamicForm({ organizationId: organizationId, dynamicFormId: dynamicFormId }) : null,
                    notifyManagers: p.notifyManagers,
                },
                dateRange: {
                    startDateTime: p.startDateTime,
                    endDateTime: p.endDateTime,
                },
                userAndDeputyList: usersInfo,
                note: p.note,
                status: p.status,
                requestDateTime: p.requestDateTime,
                requestUserId: p.requestUserId,
                dynamicFormData: p.dynamicFormData,
            };
            return Promise.resolve(permissionDetailed);
        });
    }
    async getPPermissionTrx(organizationId, ppermissionId, trx) {
        let permissionDetailed = {
            id: null,
            ppermissionType: {
                id: null,
                name: null,
                isPassive: null,
                description: null,
                hrCanInit: null,
                personnelCanInit: null,
                isDailyScheduled: null,
                minValue: null,
                maxValue: null,
                maxValueBoundaryInterval: null,
                method: null,
                approvementList: null,
                sendNotification: null,
                userMustSign: null,
                needsManagerDeputy: null,
                needsUserDeputy: null,
                type: null,
                showRemainingCount: null,
                isUnpaidLeave: null,
                notifyHR: null,
                dynamicForm: null,
                notifyManagers: null,
            },
            dateRange: {
                startDateTime: null,
                endDateTime: null,
            },
            userAndDeputyList: null,
            note: null,
            status: null,
            requestDateTime: null,
            requestUserId: null,
            dynamicFormData: null,
        };
        let qb = this.dbClient.withSchema(organizationId).table("ppermissions as p");
        if (trx)
            qb.transacting(trx);
        const result = await qb
            .leftJoin("ppermissionTypes as pt", "pt.id", "p.ppermissionTypeId")
            .leftJoin("ppermissionTypeApprovements as pta", "pt.id", "pta.ppermissionTypeId")
            .leftJoin("userPPermissions as up", "p.id", "up.ppermissionId")
            .where("p.id", ppermissionId)
            .groupBy("p.id", "pt.id")
            .select(this.dbClient.raw(`
				p.id as "leaveId",
				p."startDateTime" as "startDateTime",
				p."endDateTime" as "endDateTime",
				p.note as "note",
				p.status as "status",
				p."requestDateTime" as "requestDateTime",
				p."requestUserId" as "requestUserId",
				p."dynamicFormId" as "pDynamicFormId",
				ARRAY_AGG(DISTINCT up."userId") as "userIds",
				pt.*,
				COALESCE(
					ARRAY_AGG(pta."rootOrganizationUnitId" ORDER BY pta."order") FILTER (WHERE pta.id IS NOT NULL),
					ARRAY[]::UUID[]
				) AS "rootOrgUnitIds"`));
        if (result.length > 0) {
            const p = result[0];
            const usersInfo = await dal_manager_1.dbManager.accessUser.getBasicUserInfoList(organizationId, p.userIds);
            const mappedUsersInfo = usersInfo.map((user) => ({ user }));
            const dynamicFormId = p.pDynamicFormId ?? p.dynamicFormId;
            const dynamicForm = dynamicFormId ? await dal_manager_1.dbManager.accessSystem.getDynamicForm({ organizationId, dynamicFormId }) : null;
            permissionDetailed = {
                id: p.leaveId,
                ppermissionType: {
                    id: p.id,
                    name: p.name,
                    isPassive: p.isPassive,
                    description: p.description,
                    hrCanInit: p.hrCanInit,
                    personnelCanInit: p.personnelCanInit,
                    isDailyScheduled: p.isDailyScheduled,
                    minValue: p.minValue,
                    maxValue: p.maxValue,
                    maxValueBoundaryInterval: p.maxValueBoundaryInterval,
                    method: p.method,
                    approvementList: p.rootOrgUnitIds,
                    sendNotification: p.sendNotification,
                    userMustSign: p.userMustSign,
                    needsManagerDeputy: p.needsManagerDeputy,
                    needsUserDeputy: p.needsUserDeputy,
                    type: p.type,
                    showRemainingCount: p.showRemainingCount,
                    isUnpaidLeave: p.isUnpaidLeave,
                    notifyHR: p.notifyHR,
                    dynamicForm: dynamicForm,
                    notifyManagers: p.notifyManagers,
                },
                dateRange: {
                    startDateTime: p.startDateTime,
                    endDateTime: p.endDateTime,
                },
                userAndDeputyList: mappedUsersInfo,
                note: p.note,
                status: p.status,
                requestDateTime: p.requestDateTime,
                requestUserId: p.requestUserId,
                dynamicFormData: p.dynamicFormData,
            };
        }
        return permissionDetailed;
    }
    async getPPermissionLogBasicTrx(organizationId, ppermissionId, trx) {
        let result = {
            id: "",
            ppermissionTypeId: null,
            note: "",
            dateRange: {
                startDateTime: null,
                endDateTime: null,
            },
            status: null,
            userAndDeputyList: null,
        };
        let qb = this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissions).where("id", ppermissionId);
        if (trx)
            qb.transacting(trx);
        await qb
            .select()
            .first()
            .then(async (row) => {
            if (row) {
                result.id = row.id;
                result.ppermissionTypeId = row.ppermissionTypeId;
                result.note = row.note;
                result.dateRange.startDateTime = row.startDateTime;
                result.dateRange.endDateTime = row.endDateTime;
                result.status = row.status;
            }
        });
        result.userAndDeputyList = await this.listPPermissionUserAndDeputyListLogBasic(organizationId, ppermissionId, trx);
        return Promise.resolve(result);
    }
    async getPPermissionStatusInfoForUser(organizationId, ppermissionId, userId, trx) {
        let status = dal_constants_1.DalConstants.PPermissionStatus.Waiting;
        let acceptCount = 0;
        let rejectCount = 0;
        let totalCount = 0;
        let qb = this.dbClient.withSchema(organizationId).table("userPPermissionApprovements").where("ppermissionId", ppermissionId).where("userId", userId);
        if (trx)
            qb.transacting(trx);
        let approvements = await qb.select("approverUserId", "status");
        if (!approvements)
            return Promise.resolve(status);
        acceptCount = approvements.filter((a) => a.approverUserId && a.status).length;
        rejectCount = approvements.filter((a) => a.approverUserId && !a.status).length;
        totalCount = approvements.length;
        if (totalCount == acceptCount)
            status = dal_constants_1.DalConstants.PPermissionStatus.Approved;
        else if (totalCount == acceptCount + rejectCount && rejectCount > 0)
            status = dal_constants_1.DalConstants.PPermissionStatus.Rejected;
        return Promise.resolve(status);
    }
    async getPPermissionStatusInfoForUserWithMyApproval(organizationId, ppermissionId, userId, requestUserId, forHR, trx) {
        let status = dal_constants_1.DalConstants.PPermissionExtendedStatus.Waiting;
        let acceptCount = 0;
        let rejectCount = 0;
        let totalCount = 0;
        let waitingMyApprovalCount = 0;
        let waitingHRApprovement = 0;
        let qbx = this.dbClient.withSchema(organizationId).table("userPPermissionApprovements").where("ppermissionId", ppermissionId).where("userId", userId);
        if (trx)
            qbx = qbx.transacting(trx);
        let approvements = await qbx.select("approverUserId", "status");
        if (!approvements)
            return Promise.resolve(status);
        acceptCount = approvements.filter((a) => a.approverUserId && a.status).length;
        rejectCount = approvements.filter((a) => a.approverUserId && !a.status).length;
        totalCount = approvements.length;
        let qb = this.dbClient
            .withSchema(organizationId)
            .table("userPPermissionApprovements as upa")
            .innerJoin("ppermissions as p", (join) => {
            join.on("p.id", "upa.ppermissionId");
        })
            .innerJoin("userPPermissions as up", (join) => {
            join.on("up.ppermissionId", "p.id").andOn("up.currentApprovementOrder", "upa.order");
        })
            .where("upa.ppermissionId", ppermissionId)
            .where("upa.userId", userId)
            .where("up.userId", userId);
        if (trx)
            qb.transacting(trx);
        if (!forHR) {
            let listUserOrganizationUnits = await this.getManagerOrDeputyOrganizationList(organizationId, requestUserId, new Date(Date.now()), trx);
            await qb
                .clone()
                .whereNull("upa.approverUserId")
                .whereNotNull("upa.organizationUnitId")
                .whereIn("upa.organizationUnitId", listUserOrganizationUnits)
                .count()
                .then((rows) => (waitingMyApprovalCount = parseInt(rows[0].count)));
        }
        else {
            await qb
                .clone()
                .whereNull("upa.approverUserId")
                .whereNull("upa.organizationUnitId")
                .count()
                .then((rows) => (waitingHRApprovement = parseInt(rows[0].count)));
        }
        if (totalCount == acceptCount)
            status = dal_constants_1.DalConstants.PPermissionExtendedStatus.Approved;
        else if (totalCount == acceptCount + rejectCount && rejectCount > 0)
            status = dal_constants_1.DalConstants.PPermissionExtendedStatus.Rejected;
        else if (waitingMyApprovalCount > 0)
            status = dal_constants_1.DalConstants.PPermissionExtendedStatus.WaitingMyApproval;
        else if (waitingHRApprovement > 0)
            status = dal_constants_1.DalConstants.PPermissionExtendedStatus.WaitingHRApproval;
        return Promise.resolve(status);
    }
    async getPPermissionStatusInfoWithMyApproval(organizationId, ppermissionId, requestUserId, forHR, trx) {
        let result = {
            acceptCount: 0,
            rejectCount: 0,
            waitingCount: 0,
            waitingMyApprovalCount: 0,
        };
        let qb = this.dbClient.withSchema(organizationId).table("userPPermissions").where("ppermissionId", ppermissionId);
        if (trx)
            qb.transacting(trx);
        let users = await qb.select("userId").then((rows) => rows.map((a) => a.userId));
        for (let user of users) {
            let userStat = await this.getPPermissionStatusInfoForUserWithMyApproval(organizationId, ppermissionId, user, requestUserId, forHR, trx);
            switch (userStat) {
                case dal_constants_1.DalConstants.PPermissionExtendedStatus.Approved:
                    result.acceptCount++;
                    break;
                case dal_constants_1.DalConstants.PPermissionExtendedStatus.Rejected:
                    result.rejectCount++;
                    break;
                case dal_constants_1.DalConstants.PPermissionExtendedStatus.WaitingMyApproval:
                    if (!forHR)
                        result.waitingMyApprovalCount++;
                case dal_constants_1.DalConstants.PPermissionExtendedStatus.WaitingHRApproval:
                    if (forHR)
                        result.waitingMyApprovalCount++;
                case dal_constants_1.DalConstants.PPermissionExtendedStatus.Waiting:
                    result.waitingCount++;
                    break;
            }
        }
        return Promise.resolve(result);
    }
    async listPPermissionUsers(organizationId, ppermissionId, listUserId, listAs, args, trx) {
        return await dal_manager_1.dbManager.pgTransactionMainDb(async (trx) => {
            let visibleUserOrganizationUnitIds;
            if (listAs === dal_constants_1.DalConstants.PPermissionInsertRole.HR) {
                visibleUserOrganizationUnitIds = await this.pgGetVisibleOrganizationUnitsAsUnitHR(organizationId, listUserId, true, trx);
            }
            else {
                visibleUserOrganizationUnitIds = await this.pgGetVisibleOrganizationUnitsAsManagerOrDeputy(organizationId, listUserId, true, new Date(), trx);
            }
            let response = {
                pagination: {
                    take: args.take,
                    skip: args.skip,
                    total: 0,
                },
                userInfoList: [],
            };
            let approveCheck = ``;
            if (listAs === dal_constants_1.DalConstants.PPermissionInsertRole.HR) {
                approveCheck = `uppa_1.order = upp_2."currentApprovementOrder" AND uppa_1."approverUserId" IS NULL AND uppa_1."organizationUnitId" IS NULL`;
            }
            else if (listAs === dal_constants_1.DalConstants.PPermissionInsertRole.Manager) {
                let approvableOrganizationUnitIds = await this.getManagerOrDeputyOrganizationList(organizationId, listUserId, new Date());
                approveCheck = `uppa_1.order = upp_2."currentApprovementOrder" AND uppa_1."approverUserId" IS NULL AND uppa_1."organizationUnitId"::text = ANY(array['${approvableOrganizationUnitIds.join(`','`)}'])`;
            }
            else {
                approveCheck = `false`;
            }
            let bindings = [];
            let bindIndex = 0;
            let query = `
                SELECT
                    up_1."ppermissionId",
                    jsonb_build_object('user', up_1."userId",'deputyUser', uod."deputyUserOrganizationId") as "userAndDeputyInfo",
                    up_1."signedByUser" as "userSigned",
                    uppa.data as "approvementInfo",
                    up_1."currentApprovementOrder" as "currentApprovementOrder",
                    uppa."rejectCount" = 0 AND true = ANY(uppa."canListerApprove") as "canListerApprove",
                    CASE
                        WHEN uppa."totalCount" = uppa."acceptCount" THEN 1
                        WHEN (uppa."totalCount" = uppa."acceptCount" + uppa."rejectCount") AND (uppa."rejectCount" > 0) THEN 2
                        ELSE 0
                    END as "status"
                FROM "${organizationId}"."userPPermissions" up_1
                INNER JOIN "${organizationId}"."userOrganizations" uo ON uo."isDisabled" = false AND uo."deletedAt" IS NULL AND uo."userId" = up_1."userId" AND uo."organizationId" = $${++bindIndex}
                INNER JOIN "${organizationId}"."userOrganizationProfiles" uop ON uop."deletedAt" IS NULL AND uop."userOrganizationId" = uo.id
                LEFT JOIN (
                    SELECT uppa_1."ppermissionId", uppa_1."userId",
                    jsonb_agg(
                        jsonb_build_object(
                            'approvementDate', uppa_1."approvementDate",
                            'approverUserId', uppa_1."approverUserId",
                            'organizationUnit', uppa_1."organizationUnitId",
                            'isHR', uppa_1."organizationUnitId" IS NULL,
                            'note', uppa_1.note,
                            'order', uppa_1.order,
                            'status', CASE WHEN uppa_1."approvementDate" IS NULL THEN 0 ELSE CASE WHEN uppa_1.status THEN 1 ELSE 2 END END,
                            'canListerApprove', ${approveCheck}
                        )
                    ) as data,
                    array_agg(${approveCheck}) as "canListerApprove",
                    COUNT(*) as "totalCount",
                    COUNT(*) FILTER (WHERE uppa_1."approvementDate" IS NULL) as "waitingCount",
                    COUNT(*) FILTER (WHERE uppa_1."approvementDate" IS NOT NULL AND uppa_1.status) as "acceptCount",
                    COUNT(*) FILTER (WHERE uppa_1."approvementDate" IS NOT NULL AND NOT uppa_1.status) as "rejectCount"
                    FROM "${organizationId}"."userPPermissionApprovements" uppa_1
                    INNER JOIN "${organizationId}"."userPPermissions" upp_2 ON upp_2."ppermissionId" = uppa_1."ppermissionId" AND upp_2."userId" = uppa_1."userId"
                    GROUP BY uppa_1."ppermissionId", uppa_1."userId"
                ) uppa ON uppa."ppermissionId" = up_1."ppermissionId" AND uppa."userId" = up_1."userId"
                LEFT JOIN "${organizationId}"."userOrganizationDeputies" uod ON uod."ppermissionId" = up_1."ppermissionId" AND uod."userOrganizationId" = uo.id
                LEFT JOIN (
                    SELECT uoou_1."userOrganizationId", jsonb_agg(row_to_json(ou_1)) as data, array_agg(ou_1.id) as ouids
                    FROM "${organizationId}"."userOrganizationOrganizationUnits" uoou_1
                    INNER JOIN "${organizationId}"."organizationUnits" ou_1 ON ou_1."deletedAt" IS NULL AND ou_1.id = uoou_1."organizationUnitId"
                    WHERE uoou_1."deletedAt" IS NULL
                    GROUP BY uoou_1."userOrganizationId"
                ) uoou ON uoou."userOrganizationId" = uo.id
                WHERE up_1."ppermissionId" = $${++bindIndex}
                `;
            bindings.push(organizationId, ppermissionId);
            if (visibleUserOrganizationUnitIds && visibleUserOrganizationUnitIds.length > 0) {
                query +=
                    `AND uoou.ouids && array['` +
                        visibleUserOrganizationUnitIds.join(`','`) +
                        `']::uuid[]
                `;
            }
            if (listAs === dal_constants_1.DalConstants.PPermissionInsertRole.Self) {
                query += `AND up_1."userId" = $${++bindIndex}
                `;
                bindings.push(listUserId);
            }
            if (args.userIds && args.userIds.length > 0) {
                query += `AND up_1."userId"::text = ANY(array['${args.userIds.join(`','`)}'])
                `;
            }
            response.pagination.total = parseInt(await (await trx.query(`SELECT COUNT(*) as cnt FROM (` + query + `) sq`, bindings)).rows[0].cnt);
            query += `ORDER BY uop.name || ' ' || uop.surname DESC
            OFFSET $${++bindIndex} LIMIT $${++bindIndex};`;
            bindings.push(args.skip, args.take);
            let dbResults = await trx.query(query, bindings);
            let userIds = new Set();
            for (const row of dbResults.rows) {
                userIds.add(row.userAndDeputyInfo.user);
                if (row.userAndDeputyInfo.deputyUser) {
                    userIds.add(row.userAndDeputyInfo.deputyUser);
                }
                if (row.approvementInfo) {
                    for (const app of row.approvementInfo) {
                        if (app.approverUserId) {
                            userIds.add(app.approverUserId);
                            app.approverUsers = [app.approverUserId];
                        }
                        else {
                            app.approverUsers = await this.listManagerOrDeputyOfOrganizationUnits(organizationId, [app.organizationUnit], new Date());
                            for (const waitingApprover of app.approverUsers) {
                                userIds.add(waitingApprover);
                            }
                        }
                    }
                }
            }
            let userInfoList = await this.getBasicUserInfoList(organizationId, Array.from(userIds));
            for (const row of dbResults.rows) {
                let upp = {
                    userAndDeputyInfo: {
                        user: userInfoList.find((ui) => ui.id === row.userAndDeputyInfo.user),
                        deputyUser: row.userAndDeputyInfo.deputyUserId ? userInfoList.find((ui) => ui.id === row.userAndDeputyInfo.deputyUserId) : null,
                    },
                    canListerApprove: row.canListerApprove,
                    status: row.status,
                    userSigned: row.userSigned,
                    approvementInfo: row.approvementInfo,
                };
                if (upp.approvementInfo) {
                    for (const app of upp.approvementInfo) {
                        if (app.approverUserId) {
                            app.approverUsers = [userInfoList.find((ui) => ui.id === app.approverUserId)];
                        }
                        else {
                            app.approverUsers = app.approverUsers.map((approver) => {
                                return userInfoList.find((ui) => ui.id === approver);
                            });
                        }
                    }
                }
                else {
                    upp.approvementInfo = [];
                }
                response.userInfoList.push(upp);
            }
            return response;
        });
    }
    async listPPermissionUserApprovementInfo(organizationId, ppermissionId, userId, requestUserId, trx) {
        let ppermission = await this.getPPermissionTrx(organizationId, ppermissionId, trx);
        if (ppermission && ppermission.ppermissionType && (!ppermission.ppermissionType.approvementList || ppermission.ppermissionType.approvementList.length == 0)) {
            return Promise.resolve({
                overallStatus: dal_constants_1.DalConstants.PPermissionStatus.Approved,
                items: [],
            });
        }
        let approvers = [];
        let approvements = [];
        let qb = this.dbClient
            .withSchema(organizationId)
            .from("userPPermissionApprovements as upa")
            .innerJoin("ppermissions as p", function (join) {
            join.on("p.id", "=", "upa.ppermissionId");
        })
            .innerJoin("userPPermissions as up", (join) => {
            join.on("up.ppermissionId", "p.id");
        })
            .where("upa.ppermissionId", ppermissionId)
            .where("upa.userId", userId)
            .where("up.userId", userId)
            .orderBy("order", "asc");
        if (trx)
            qb.transacting(trx);
        await qb.select(["upa.approvementDate", "upa.note", "upa.approverUserId", "upa.order", "upa.organizationUnitId", "upa.status", "up.currentApprovementOrder"]).then(async (rows) => {
            if (rows) {
                for (let row of rows) {
                    approvements.push({
                        approvementDate: row.approvementDate,
                        note: row.note,
                        approverUserId: row.approverUserId,
                        status: row.status,
                        order: row.order,
                        organizationUnitId: row.organizationUnitId,
                        currentApprovementOrder: row.currentApprovementOrder,
                    });
                }
            }
        });
        for (let approvement of approvements) {
            if (approvement.organizationUnitId) {
                if (approvement.approverUserId) {
                    approvers.push({
                        order: approvement.order,
                        approverUsers: [await this.getBasicUserInfo(organizationId, approvement.approverUserId, trx)],
                        isHR: false,
                        userOrganizationUnitName: approvement.organizationUnitId
                            ? (await dal_manager_1.dbManager.accessOrganizationUnit.getOrganizationUnitBasic({
                                organizationId,
                                organizationUnitId: approvement.organizationUnitId,
                                includingDeleted: true,
                            })).name
                            : null,
                        canListerApprove: false,
                        status: approvement.status ? dal_constants_1.DalConstants.PPermissionStatus.Approved : dal_constants_1.DalConstants.PPermissionStatus.Rejected,
                        note: approvement.note,
                        approvementDate: approvement.approvementDate,
                    });
                }
                else {
                    let managerOrDeputyList = await this.listManagerOrDeputyOfOrganizationUnits(organizationId, [approvement.organizationUnitId], new Date(Date.now()), trx);
                    approvers.push({
                        order: approvement.order,
                        approverUsers: await this.getBasicUserInfoList(organizationId, managerOrDeputyList, trx),
                        isHR: false,
                        userOrganizationUnitName: approvement.organizationUnitId
                            ? (await dal_manager_1.dbManager.accessOrganizationUnit.getOrganizationUnitBasic({
                                organizationId,
                                organizationUnitId: approvement.organizationUnitId,
                                includingDeleted: true,
                            })).name
                            : null,
                        canListerApprove: approvement.currentApprovementOrder == approvement.order && managerOrDeputyList.filter((a) => a === requestUserId).length > 0,
                        status: dal_constants_1.DalConstants.PPermissionStatus.Waiting,
                        note: approvement.note,
                        approvementDate: approvement.approvementDate,
                    });
                }
            }
            else {
                if (approvement.approverUserId) {
                    approvers.push({
                        order: approvement.order,
                        approverUsers: [await this.getBasicUserInfo(organizationId, approvement.approverUserId, trx)],
                        isHR: true,
                        userOrganizationUnitName: approvement.organizationUnitId
                            ? (await dal_manager_1.dbManager.accessOrganizationUnit.getOrganizationUnitBasic({
                                organizationId,
                                organizationUnitId: approvement.organizationUnitId,
                                includingDeleted: true,
                            })).name
                            : null,
                        canListerApprove: false,
                        status: approvement.status ? dal_constants_1.DalConstants.PPermissionStatus.Approved : dal_constants_1.DalConstants.PPermissionStatus.Rejected,
                        note: approvement.note,
                        approvementDate: approvement.approvementDate,
                    });
                }
                else {
                    approvers.push({
                        order: approvement.order,
                        approverUsers: [],
                        isHR: true,
                        userOrganizationUnitName: approvement.organizationUnitId
                            ? (await dal_manager_1.dbManager.accessOrganizationUnit.getOrganizationUnitBasic({
                                organizationId,
                                organizationUnitId: approvement.organizationUnitId,
                                includingDeleted: true,
                            })).name
                            : null,
                        canListerApprove: approvement.currentApprovementOrder == approvement.order && !requestUserId,
                        status: dal_constants_1.DalConstants.PPermissionStatus.Waiting,
                        note: approvement.note,
                        approvementDate: approvement.approvementDate,
                    });
                }
            }
        }
        let rejectCount = 0;
        let waitingCount = 0;
        for (const app of approvers) {
            rejectCount += app.status === dal_constants_1.DalConstants.PPermissionStatus.Rejected ? 1 : 0;
            waitingCount += app.status === dal_constants_1.DalConstants.PPermissionStatus.Waiting ? 1 : 0;
        }
        let overallStatus = dal_constants_1.DalConstants.PPermissionStatus.Approved;
        if (rejectCount > 0) {
            overallStatus = dal_constants_1.DalConstants.PPermissionStatus.Rejected;
        }
        else if (waitingCount > 0) {
            overallStatus = dal_constants_1.DalConstants.PPermissionStatus.Waiting;
        }
        return Promise.resolve({
            overallStatus: overallStatus,
            items: approvers,
        });
    }
    async listPPermissionUserAndDeputyListLogBasic(organizationId, ppermissionId, trx) {
        let qb = this.dbClient
            .withSchema(organizationId)
            .from("ppermissions as p")
            .innerJoin("userPPermissions as up", (join) => {
            join.on("p.id", "up.ppermissionId");
        })
            .innerJoin("userOrganizations as uo", (join) => {
            join.on("uo.userId", "up.userId").andOn("uo.organizationId", this.dbClient.raw("?", [organizationId]));
        })
            .leftJoin("userOrganizationDeputies as wgd", (join) => {
            join.on("wgd.ppermissionId", "p.id").andOn("wgd.userOrganizationId", "uo.id");
        })
            .leftJoin("userOrganizations as oud", (join) => {
            join.on("oud.id", "wgd.deputyUserOrganizationId");
        })
            .where("p.id", ppermissionId)
            .where("p.organizationId", organizationId);
        if (trx)
            qb.transacting(trx);
        return qb.select("uo.userId", "oud.userId as deputyUserId");
    }
    async getUserPPermissionApprovements(organizationId, ppermissionId, userId, trx) {
        let qb = this.dbClient.withSchema(organizationId).table("userPPermissionApprovements").where("ppermissionId", ppermissionId).where("userId", userId);
        if (trx)
            qb = qb.transacting(trx);
        return (await qb.select());
    }
    async getUserPPermissionApprovementsBulk(organizationId, ppermissionId, userIds, trx) {
        if (userIds.length === 0)
            return new Map();
        let qb = this.dbClient.withSchema(organizationId).table("userPPermissionApprovements").where("ppermissionId", ppermissionId).whereIn("userId", userIds);
        if (trx)
            qb = qb.transacting(trx);
        const rows = (await qb.select());
        const resultMap = new Map();
        for (const row of rows) {
            if (!resultMap.has(row.userId)) {
                resultMap.set(row.userId, []);
            }
            resultMap.get(row.userId).push(row);
        }
        return resultMap;
    }
    async listPPermissionWaitingApprovementsNew(organizationId, ppermissionId, userId, trx) {
        let result = [];
        let qb = this.dbClient
            .withSchema(organizationId)
            .table("userPPermissionApprovements as upa")
            .innerJoin("ppermissions as p", (join) => {
            join.on("p.id", "upa.ppermissionId");
        })
            .where("upa.ppermissionId", ppermissionId)
            .where("upa.userId", userId);
        if (trx)
            qb.transacting(trx);
        let userOrder = 1;
        await qb
            .clone()
            .whereNull("upa.approverUserId")
            .min("upa.order")
            .select()
            .then((row) => {
            if (row)
                userOrder = row[0].min;
        });
        let qbx = this.dbClient
            .withSchema(organizationId)
            .from("userPPermissionApprovements as upa")
            .innerJoin("ppermissions as p", function (join) {
            join.on("p.id", "=", "upa.ppermissionId");
        })
            .where("upa.ppermissionId", ppermissionId)
            .where("upa.userId", userId)
            .where("upa.order", userOrder);
        if (trx)
            qbx = qbx.transacting(trx);
        await qbx.select("upa.organizationUnitId").then(async (rows) => {
            if (rows) {
                for (let row of rows) {
                    result.push(row.organizationUnitId);
                }
            }
        });
        return Promise.resolve(result);
    }
    async approveAllApproveableStepsAsUser(organizationId, args, trx) {
        let result = await this.approvePPermissionForUserTrx(organizationId, args.ppermissionId, args.userId, {
            note: args.note,
            status: args.status,
            approveUserId: args.approveUserId,
            isHR: args.isHR,
            trx: trx,
        });
        if (result) {
            return await this.approveAllApproveableStepsAsUser(organizationId, {
                ppermissionId: args.ppermissionId,
                userId: args.userId,
                approveUserId: args.approveUserId,
                note: args.note,
                status: args.status,
                isHR: args.isHR,
                approvedCount: args.approvedCount + 1,
            }, trx);
        }
        else {
            if (args.approvedCount > 0) {
                return Promise.resolve(true);
            }
            else {
                return Promise.resolve(false);
            }
        }
        return false;
    }
    async approvePPermissionForUserTrx(organizationId, ppermissionId, userId, args) {
        let qb = this.dbClient
            .withSchema(organizationId)
            .from("userPPermissionApprovements as upa")
            .innerJoin("ppermissions as p", function (join) {
            join.on("p.id", "=", "upa.ppermissionId");
        })
            .innerJoin("userPPermissions as up", (join) => {
            join.on("p.id", "up.ppermissionId").andOn("up.currentApprovementOrder", "upa.order");
        })
            .where("upa.userId", userId)
            .where("upa.ppermissionId", ppermissionId)
            .whereNull("upa.approverUserId")
            .where("up.userId", userId)
            .transacting(args.trx)
            .orderBy("order", "asc");
        if (args.isHR)
            qb.whereNull("upa.organizationUnitId");
        else {
            let targetOrganizationUnits = (await qb.clone().select("upa.organizationUnitId")).map((row) => row.organizationUnitId);
            if (targetOrganizationUnits && targetOrganizationUnits.length > 0) {
                let hasRight = await this.isUserManagerOrDeputyOfOrganizationUnit(organizationId, args.approveUserId, targetOrganizationUnits, new Date(Date.now()), args.trx);
                if (!hasRight)
                    return Promise.resolve(false);
            }
            let userOrganizationUnits = await this.getManagerOrDeputyOrganizationListWithChilds(organizationId, args.approveUserId, new Date(Date.now()), args.trx);
            qb.whereIn("upa.organizationUnitId", userOrganizationUnits);
        }
        let approvement = await qb.clone().first().select(["upa.id", "upa.organizationUnitId", "upa.order"]);
        if (!approvement)
            return Promise.resolve(false);
        await this.dbClient
            .withSchema(organizationId)
            .table("userPPermissionApprovements")
            .where("organizationUnitId", approvement.organizationUnitId)
            .where("id", approvement.id)
            .transacting(args.trx)
            .update({
            status: args.status == dal_constants_1.DalConstants.PPermissionStatus.Approved,
            note: args.note,
            approverUserId: args.approveUserId,
            approvementDate: new Date(Date.now()),
        });
        let firstQB = this.dbClient
            .withSchema(organizationId)
            .transacting(args.trx)
            .table("userPPermissionApprovements as upa")
            .innerJoin("ppermissions as p", (join) => {
            join.on("p.id", "upa.ppermissionId");
        })
            .where("upa.ppermissionId", ppermissionId)
            .where("upa.userId", userId);
        let userOrder = 1;
        await firstQB
            .whereNull("upa.approverUserId")
            .min("upa.order")
            .select()
            .then((row) => {
            if (row)
                userOrder = row[0].min;
        });
        await this.dbClient.withSchema(organizationId).table("userPPermissions").where("ppermissionId", ppermissionId).where("userId", userId).transacting(args.trx).update({
            currentApprovementOrder: userOrder,
        });
        let ppermission = await this.getPPermissionTrx(organizationId, ppermissionId, args.trx);
        let approveQb = await this.dbClient.withSchema(organizationId).table("userPPermissionApprovements").where("ppermissionId", ppermissionId).transacting(args.trx).select();
        let targetApprovementCount = approveQb.length;
        let approvedPositiveCount = approveQb.filter((a) => a.approverUserId && a.status).length;
        let approvedNegativeCount = approveQb.filter((a) => a.approverUserId && !a.status).length;
        let status = dal_constants_1.DalConstants.PPermissionStatus.Waiting;
        if (targetApprovementCount == approvedPositiveCount)
            status = dal_constants_1.DalConstants.PPermissionStatus.Approved;
        if (targetApprovementCount == approvedNegativeCount + approvedPositiveCount && approvedNegativeCount > 0)
            status = dal_constants_1.DalConstants.PPermissionStatus.Rejected;
        await this.dbClient.withSchema(organizationId).table("ppermissions").where("id", ppermissionId).transacting(args.trx).update({
            status: status,
        });
        let ppermissionType = await dal_manager_1.dbManager.accessPacs.getPPermissionType(organizationId, ppermission.ppermissionType.id, args.trx);
        if (ppermissionType.type === app_enums_1.enums.PredefinedPPermissionType.Annual) {
            let userPPermissionApprovements = await this.getUserPPermissionApprovements(organizationId, ppermission.id, userId, args.trx);
            let checkPPermissionApprovements = userPPermissionApprovements.filter((upa) => upa.status === false);
            if (!checkPPermissionApprovements || checkPPermissionApprovements.length === 0) {
                await dal_manager_1.dbManager.accessPacs2.updateRemainingAnnualPPermissionOfUserByApprove(organizationId, userId, ppermission.ppermissionType.id, ppermission.dateRange.startDateTime, ppermission.dateRange.endDateTime);
            }
        }
        let approvements = await this.listPPermissionUserApprovementsAndStatus(organizationId, ppermission.id, args.trx);
        await dal_manager_1.dbManager.accessLogPacs.insertPPermissionHistoryContentLog(organizationId, ppermission.id, args.approveUserId, new Date(), args.status === dal_constants_1.DalConstants.PPermissionStatus.Approved ? dal_constants_1.DalConstants.PPermissionHistoryActionType.Approved : dal_constants_1.DalConstants.PPermissionHistoryActionType.Rejected, {
            typeId: ppermission.ppermissionType.id,
            startDate: ppermission.dateRange.startDateTime,
            endDate: ppermission.dateRange.endDateTime,
            status: approvements.status,
            userAndDeputyList: [
                {
                    userId: userId,
                    deputyUserId: null,
                },
            ],
            approvementList: approvements.userApprovementList
                ? approvements.userApprovementList.map((m) => {
                    return {
                        approverUserId: m.approverUserId,
                        organizationUnitId: m.organizationUnitId,
                        isHr: m.isHr,
                        order: m.order,
                        note: m.note,
                        isApproved: m.isApproved,
                    };
                })
                : null,
        });
        return Promise.resolve(true);
    }
    async approvePPermissionForUser(organizationId, ppermissionId, userId, args) {
        return this.dbClient.transaction(async (trx) => {
            let result = await this.approveAllApproveableStepsAsUser(organizationId, {
                ppermissionId,
                userId,
                approveUserId: args.approveUserId,
                note: args.note,
                status: args.status,
                isHR: args.isHR,
                approvedCount: 0,
            }, trx);
            return Promise.resolve(result);
        });
    }
    async approvePPermission(organizationId, ppermissionId, args) {
        const approvedUsers = [];
        const result = await this.dbClient.transaction(async (trx) => {
            const ppermission = await this.getPPermissionTrx(organizationId, ppermissionId, trx);
            const ppermissionType = await dal_manager_1.dbManager.accessPacs.getPPermissionType(organizationId, ppermission.ppermissionType.id, trx);
            const usersToAdd = [];
            const userIds = ppermission.userAndDeputyList.map((ud) => ud.user.id);
            const userPPermissionApprovementsMap = await dal_manager_1.dbManager.accessPacs.getUserPPermissionApprovementsBulk(organizationId, ppermissionId, userIds, trx);
            for (const userId of userIds) {
                approvedUsers.push(userId);
                await this.approveAllApproveableStepsAsUser(organizationId, {
                    ppermissionId,
                    userId: userId,
                    approveUserId: args.approveUserId,
                    note: args.note,
                    status: args.status,
                    isHR: args.isHR,
                    approvedCount: 0,
                }, trx);
                if (ppermissionType.type === app_enums_1.enums.PredefinedPPermissionType.Annual) {
                    const userPPermissionApprovements = userPPermissionApprovementsMap.get(userId) ?? [];
                    const hasPending = userPPermissionApprovements.some((upa) => upa.status === false);
                    if (!hasPending) {
                        usersToAdd.push(userId);
                    }
                }
            }
            if (usersToAdd.length > 0) {
                await dal_manager_1.dbManager.accessPacs2.updateRemainingAnnualPPermissionOfUsersByApproval(organizationId, usersToAdd, ppermission.ppermissionType.id, ppermission.dateRange.startDateTime, ppermission.dateRange.endDateTime, trx, false);
            }
            return true;
        });
        (0, business_pacs_ppermission_1.sendNextPPermissionNotification)({ organizationId, ppermissionId, userIds: approvedUsers, requesterUserId: args.approveUserId });
        return result;
    }
    async listUsersWithUnsignedPastPPermissions(organizationId, date) {
        let result = [];
        await this.dbClient
            .withSchema(organizationId)
            .from("ppermissions as p")
            .innerJoin("userPPermissions as up", (join) => {
            join.on("p.id", "up.ppermissionId");
        })
            .innerJoin("ppermissionTypes as pt", "pt.id", "p.ppermissionTypeId")
            .where("pt.userMustSign", true)
            .where("p.endDateTime", "<", date)
            .whereRaw('(p."fromIntegration" is null or p."fromIntegration" = false)')
            .where((builder) => {
            builder.where("up.signedByUser", false).orWhereNull("up.signedByUser");
        })
            .select("up.userId", "p.id")
            .then((rows) => {
            for (let row of rows) {
                result.push({
                    userId: row.userId,
                    ppermissionId: row.id,
                });
            }
        });
        return Promise.resolve(result);
    }
    async listPastUnpaidPPermissionsOfUser(organizationId, userId, date, trx) {
        let qb = this.dbClient
            .withSchema(organizationId)
            .from("ppermissions as p")
            .innerJoin("userPPermissions as up", (join) => {
            join.on("p.id", "up.ppermissionId");
        })
            .innerJoin("ppermissionTypes as pt", "pt.id", "p.ppermissionTypeId")
            .where("up.userId", userId)
            .where("p.organizationId", organizationId)
            .where("pt.isUnpaidLeave", true)
            .where("p.status", dal_constants_1.DalConstants.PPermissionStatus.Approved)
            .where("p.endDateTime", "<=", date)
            .whereRaw('(p."fromIntegration" is null or p."fromIntegration" = false)');
        if (trx)
            qb.transacting(trx);
        let results = await qb.select("p.id", "p.startDateTime", "p.endDateTime", "up.userId");
        return Promise.resolve(results);
    }
    async listUsersPastUnpaidPPermissions(organizationId, userIds, date, trx) {
        const result = (await trx.query(`
			SELECT	up."userId",
					json_agg(json_build_object(
						'permissionId', p.id,
						'startDateTime', p."startDateTime",
						'endDateTime', p."endDateTime"
					)) as "ppermissionDateInfo"
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissions}" as p
			INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userPPermissions}" as up
				ON p.id = up."ppermissionId"
			INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissionTypes}" as pt
				ON pt.id = p."ppermissionTypeId"
			WHERE	up."userId" = ANY($1::uuid[])
					AND pt."isUnpaidLeave" IS TRUE
					AND p.status = 1
					AND p."endDateTime" <= $2
					AND (p."fromIntegration" IS NULL OR p."fromIntegration" = false)
			GROUP BY up."userId"
			`, [userIds, date])).rows;
        return result;
    }
    async listUnapprovedPPermissionsOfUserForDay(organizationId, userId, day, trx) {
        let dayStart = (0, moment_1.default)(day).startOf("day");
        let qb = this.dbClient.raw(`SELECT pp.id, pp."startDateTime", pp."endDateTime", pp."ppermissionTypeId", 
		ppt."method", ppt."applyOnHolidays", ppt."applyOnlyOnWorkingHours", ppt."isUnpaidLeave", ppt."name", pp."note", uppa_2."status"
		FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissions}" AS pp
		INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissionTypes}" AS ppt ON pp."ppermissionTypeId" = ppt.id
		INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userPPermissions}" AS uppt ON pp.id = uppt."ppermissionId"
		INNER JOIN (
			SELECT uppa."ppermissionId",
			CASE WHEN bool_or(uppa."approvementDate" IS NOT NULL AND uppa."status" = false) THEN 2 
			WHEN bool_or(uppa."approvementDate" IS NULL) THEN 0 
			ELSE 1 END as status  
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userPPermissionApprovements}" as uppa
			WHERE uppa."userId" = ? 	
			GROUP BY uppa."ppermissionId"
		) as uppa_2 ON uppa_2."ppermissionId" = pp."id"
		WHERE uppt."userId" = ? AND ppt."deletedAt" IS NULL
		AND tstzrange(?, ?) && tstzrange(pp."startDateTime", pp."endDateTime")
		AND (uppa_2."status" <> 1)`, [userId, userId, dayStart.toDate(), dayStart.add(1, "day").toDate()]);
        if (trx)
            qb.transacting(trx);
        let dbResults = [];
        const elems = await qb;
        for (const elem of elems.rows) {
            dbResults.push(Object.assign({}, {
                id: elem.id,
                typeId: elem.ppermissionTypeId,
                typeName: elem.name,
                note: elem.note,
                startDateTime: elem.startDateTime,
                endDateTime: elem.endDateTime,
                status: elem.status,
                method: elem.method,
            }));
        }
        let results = [];
        for (const row of dbResults) {
            let entry = {
                id: row.id,
                type: row.method,
                typeId: row.typeId,
                typeName: row.typeName,
                note: row.note,
                range: {
                    startDateTime: row.startDateTime,
                    endDateTime: row.endDateTime,
                },
                status: row.status,
            };
            results.push(entry);
        }
        return Promise.resolve(results);
    }
    async getOrganizationPACSModuleSettings(organizationId, trx) {
        let settings;
        let qb = this.dbClient.withSchema(organizationId).from("organizationPACSModuleSettings").where("organizationId", organizationId);
        if (trx)
            qb.transacting(trx);
        await qb.first().then((row) => {
            if (row) {
                settings = {
                    id: row.id,
                    organizationId: row.organizationId,
                    annualLeavePPermissionTypeId: row.annualLeavePPermissionTypeId,
                    casualLeavePPermissionTypeId: row.casualLeavePPermissionTypeId,
                    deputySelectionEnabled: row.deputySelectionEnabled,
                    annualPPermissionAgeLimits: row.annualPPermissionAgeLimits,
                    ppermissionClaimProfile: row.ppermissionClaimProfile,
                    bufferMinutes: row.bufferMinutes,
                    middayHour: row.middayHour,
                };
            }
        });
        return Promise.resolve(settings);
    }
    async getOrganizationPACSModuleSettingsPoolClient(organizationId, trx) {
        const result = (await trx.query(`
			SELECT  id,
					"organizationId",
					"annualLeavePPermissionTypeId",
					"casualLeavePPermissionTypeId",
					"deputySelectionEnabled",
					"annualPPermissionAgeLimits",
					"ppermissionClaimProfile",
					"bufferMinutes",
					"middayHour"
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationPACSModuleSettings}"
		`)).rows[0];
        return result;
    }
    async upsertOrganizationPACSModuleSettings(organizationId, settings, trx) {
        let existingSettings = await this.getOrganizationPACSModuleSettings(organizationId, trx);
        if (existingSettings) {
            let qb = this.dbClient.withSchema(organizationId).from("organizationPACSModuleSettings").where("id", existingSettings.id);
            if (trx)
                qb.transacting(trx);
            await qb.update({
                annualLeavePPermissionTypeId: settings.annualLeavePPermissionTypeId,
                casualLeavePPermissionTypeId: settings.casualLeavePPermissionTypeId,
                annualPPermissionAgeLimits: settings.annualPPermissionAgeLimits,
                ppermissionClaimProfile: settings.ppermissionClaimProfile ? settings.ppermissionClaimProfile : dal_constants_1.DalConstants.PPermissionClaimProfile.Auto,
                middayHour: settings.middayHour ?? "12:00",
            });
        }
        else {
            let qb = this.dbClient.withSchema(organizationId).from("organizationPACSModuleSettings");
            if (trx)
                qb.transacting(trx);
            await qb.insert({
                id: uuid_1.default.v4(),
                organizationId: settings.organizationId,
                annualLeavePPermissionTypeId: settings.annualLeavePPermissionTypeId,
                casualLeavePPermissionTypeId: settings.casualLeavePPermissionTypeId,
                annualPPermissionAgeLimits: settings.annualPPermissionAgeLimits,
                ppermissionClaimProfile: settings.ppermissionClaimProfile ? settings.ppermissionClaimProfile : dal_constants_1.DalConstants.PPermissionClaimProfile.Auto,
                middayHour: settings.middayHour ?? "12:00",
            });
        }
    }
    async upsertScheduledJobsAndNotifications(organizationId, trx) {
        await trx.raw(`INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.notification}"
				(id, "createdT", type, "receiverFilterId", email, sms, "pushNotification", web, "settings")
				VALUES (uuid_generate_v4(), now(), UNNEST (?::int[]), null, true, false, true, true, null)`, [dal_constants_1.DalConstants.organizationPACSModuleNotificationTypesWithSingleInstances]);
        for (const entry of dal_constants_1.DalConstants.pacsModuleScheduledJobsAndIntervals) {
            await trx.raw(`INSERT INTO public.${dal_db_armon_schema_1.ArmonSchema.tableNames.scheduled_job}(
                id, "createdT", "organizationId", type, "createdByUserId", enabled, "interval", "firstExecutionDate", "nextExecutionDate", note, "notificationId") 
				SELECT uuid_generate_v4(), now(), '${organizationId}', ?, null, true, ?, 
				(select date_trunc('day', now()) + interval '1 day'), (select date_trunc('day', now()) + interval '1 day'), null, 
				(SELECT id FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.notification}" WHERE type = ${entry.notificationType})
			`, [entry.type, entry.interval]);
        }
        await trx.raw(`
			UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizations}"
			SET "settings" = jsonb_set ("settings", '{notification,enabledTypes}', 
										(REPLACE( REPLACE ((REPLACE( REPLACE ("settings"->'notification'->>'enabledTypes', '[', '{'), ']', '}')::int[] 
															|| ?::int[])::text,'{','['),'}',']'))::jsonb, true)
			WHERE id = '${organizationId}';
		`, [dal_constants_1.DalConstants.organizationPACSModuleNotificationTypes]);
    }
    async removeScheduledJobsAndNotifications(organizationId, trx) {
        await trx.raw(`DELETE FROM public."${dal_db_armon_schema_1.ArmonSchema.tableNames.scheduled_job}"
				WHERE type = ANY (?::int[]) AND "organizationId" = ?
			`, [dal_constants_1.DalConstants.pacsModuleScheduledJobsAndIntervals.map((p) => p.type), organizationId]);
        await trx.raw(`DELETE FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.notification}"
				WHERE type = ANY (?::int[])
			`, [dal_constants_1.DalConstants.organizationPACSModuleNotificationTypes]);
        await trx.raw(`
			UPDATE "${organizationId}".organizations
			SET "settings" = jsonb_set ("settings", '{notification,enabledTypes}', (REPLACE( REPLACE ((
				SELECT COALESCE(ARRAY_AGG(elem), '{}')
				FROM UNNEST(REPLACE( REPLACE ("settings"->'notification'->>'enabledTypes', '[', '{'), ']', '}')::int[] ) elem
				WHERE elem <> ALL(?::int[]))::text,'{','['),'}',']'))::jsonb, true)
			WHERE id = '${organizationId}';
		`, [dal_constants_1.DalConstants.organizationPACSModuleNotificationTypes]);
    }
    async uploadPPermissionAttachment(organizationId, fileName) {
        let id = uuid_1.default.v4();
        await this.dbClient.withSchema(organizationId).table("ppermissionAttachments").insert({
            id: id,
            ppermissionId: null,
            name: fileName,
        });
        return Promise.resolve(id);
    }
    async removePPermissionAttachment(organizationId, attachmentId) {
        let affectedRowCount = 0;
        await this.dbClient
            .withSchema(organizationId)
            .table("ppermissionAttachments")
            .where("id", attachmentId)
            .del()
            .then((result) => (affectedRowCount = result));
        return Promise.resolve(affectedRowCount > 0);
    }
    async attachFileToPPermission(organizationId, ppermissionId, attachmentId, trx) {
        let affectedRowCount = 0;
        let qb = this.dbClient.withSchema(organizationId).table("ppermissionAttachments").whereIn("id", attachmentId);
        if (trx)
            qb.transacting(trx);
        await qb
            .update({
            ppermissionId: ppermissionId,
        })
            .then((result) => (affectedRowCount = result));
        return Promise.resolve(affectedRowCount > 0);
    }
    async getAttachmentOfPermission(organizationId, attachmentId) {
        let result = "";
        await this.dbClient
            .withSchema(organizationId)
            .table("ppermissionAttachments")
            .where("id", attachmentId)
            .select("name")
            .first()
            .then((row) => (result = row.name));
        return Promise.resolve(result);
    }
    async finalizeMonthlyReportOfOrganization(organizationId, year, month, userId, note) {
        await this.dbClient.transaction(async (trx) => {
            await trx
                .withSchema(organizationId)
                .table("monthlySummaryReports")
                .where("year", year)
                .where("month", month)
                .where("organizationId", organizationId)
                .update({
                isFinalized: true,
                updatedById: userId,
                updatedDate: new Date(Date.now()),
                note: note,
            });
            let updatedMonths = await trx.withSchema(organizationId).table("monthlySummaryReports").where("organizationId", organizationId).where("year", year).where("month", month).first();
            let logResult = await dal_manager_1.dbManager.accessLogPacs.insertMonthlySummaryHistory(organizationId, [updatedMonths]);
            if (!logResult) {
                return Promise.reject("rethink write error for user monthly summary");
            }
        });
        return Promise.resolve(true);
    }
    async getMonthlySummaryReportId(organizationId, year, month) {
        let result = await this.dbClient
            .withSchema(organizationId)
            .from("monthlySummaryReports as ms")
            .where("ms.year", year)
            .where("ms.month", month)
            .where("ms.organizationId", organizationId)
            .first("id");
        return Promise.resolve(result ? result.id : null);
    }
    async upsertAllPPermissionsToAccepted(organizationId) {
        await this.dbClient.withSchema(organizationId).table("ppermissions").update({
            status: dal_constants_1.DalConstants.PPermissionStatus.Approved,
        });
    }
    async collectOrganizationUnitIds(organizationId, isOrganizationWide, authorizedOrganizationUnitIds, filterIds) {
        let result = [];
        let newFilter = [];
        if (filterIds && filterIds.length > 0) {
            if (!isOrganizationWide) {
                for (let id of filterIds) {
                    if (authorizedOrganizationUnitIds.indexOf(id) > -1)
                        newFilter.push(id);
                }
            }
            else {
                newFilter = filterIds;
            }
        }
        else {
            if (!isOrganizationWide)
                newFilter = authorizedOrganizationUnitIds;
        }
        let qb = this.dbClient.withSchema(organizationId).table("organizationUnits").where("organizationId", organizationId).whereNull("deletedAt");
        if (!isOrganizationWide || (isOrganizationWide && newFilter.length > 0))
            qb.whereIn("id", newFilter);
        result = await qb.select("id", "name");
        return Promise.resolve(result);
    }
    async updateAnnualPPermissionIntervals(organizationId, updateIntervals, trx) {
        let qb = this.dbClient.withSchema(organizationId).table("organizations").where("id", organizationId);
        if (trx)
            qb.transacting(trx);
        return await qb.update({
            annualPPermissionUpdateIntervals: JSON.stringify(updateIntervals),
        });
    }
    async uploadBulkPPermissions(organizationId, ppermissions, systemUser) {
        let ppermissionInserts = [];
        let userPPermissionInserts = [];
        let userApprovementInserts = [];
        for (const ppermission of ppermissions) {
            let ppermissionId = uuid_1.default.v4();
            ppermissionInserts.push({
                id: ppermissionId,
                organizationId: organizationId,
                ppermissionTypeId: ppermission.ppermissionType.id,
                startDateTime: ppermission.startDate,
                endDateTime: ppermission.endDate,
                note: "",
                status: ppermission.approved ? dal_constants_1.DalConstants.PPermissionStatus.Approved : dal_constants_1.DalConstants.PPermissionStatus.Waiting,
                requestDateTime: new Date(),
                fromIntegration: true,
            });
            userPPermissionInserts.push({
                id: uuid_1.default.v4(),
                userId: ppermission.userId,
                ppermissionId: ppermissionId,
                currentApprovementOrder: 1,
            });
            if (ppermission.approved) {
                userApprovementInserts.push({
                    id: uuid_1.default.v4(),
                    ppermissionId: ppermissionId,
                    userId: ppermission.userId,
                    organizationUnitId: null,
                    order: 0,
                    approverUserId: systemUser,
                    approvementDate: new Date(),
                    note: null,
                    status: true,
                });
            }
            else {
                let userApprovements = await this.createApprovementForIntegrationPPermission(organizationId, ppermission.userId, ppermissionId, ppermission.ppermissionType);
                userApprovementInserts = userApprovementInserts.concat(userApprovements);
            }
        }
        await this.dbClient.transaction(async (trx) => {
            await trx.batchInsert(`${organizationId}.ppermissions`, ppermissionInserts, 1000);
            await trx.batchInsert(`${organizationId}.userPPermissions`, userPPermissionInserts, 1000);
            await trx.batchInsert(`${organizationId}.userPPermissionApprovements`, userApprovementInserts, 1000);
        });
        return Promise.resolve();
    }
    async createApprovementForIntegrationPPermission(organizationId, userId, ppermissionId, ppermissionType) {
        let userApprovements = [];
        let approvementList = ppermissionType.approvementList;
        if (approvementList && approvementList.length > 0) {
            let organizationRootIndex = {};
            let currentApprovementOrder = 1;
            for (let approvementIndex = 0; approvementIndex < approvementList.length; approvementIndex++) {
                let currentApprovementRoot = approvementList[approvementIndex];
                if (!currentApprovementRoot) {
                    userApprovements.push({
                        id: uuid_1.default.v4(),
                        ppermissionId: ppermissionId,
                        userId: userId,
                        organizationUnitId: null,
                        order: currentApprovementOrder,
                        approverUserId: null,
                        approvementDate: null,
                        note: null,
                        status: false,
                    });
                    currentApprovementOrder++;
                    continue;
                }
                let organizationRootOrder = organizationRootIndex[currentApprovementRoot] || 0;
                let userOrganizationUnitList = await this.getUserOrganizationUnitList(organizationId, userId, currentApprovementRoot, null);
                let anyApprovement = false;
                for (let uo of userOrganizationUnitList) {
                    let ancestors = [];
                    if (uo.ancestorIds)
                        ancestors = uo.ancestorIds.split(",");
                    let targetOrganizationUnitId = uo.organizationUnitId;
                    if (ancestors.length < 1) {
                        if (userId == uo.managerUserId) {
                            userApprovements.push({
                                id: uuid_1.default.v4(),
                                ppermissionId: ppermissionId,
                                userId: userId,
                                organizationUnitId: targetOrganizationUnitId,
                                order: currentApprovementOrder,
                                approverUserId: userId,
                                approvementDate: new Date(),
                                note: null,
                                status: true,
                            });
                            anyApprovement = true;
                        }
                        else {
                            if (userApprovements.filter((a) => a.organizationUnitId == targetOrganizationUnitId && a.order == currentApprovementOrder).length == 0) {
                                userApprovements.push({
                                    id: uuid_1.default.v4(),
                                    ppermissionId: ppermissionId,
                                    userId: userId,
                                    organizationUnitId: targetOrganizationUnitId,
                                    order: currentApprovementOrder,
                                    approverUserId: null,
                                    approvementDate: null,
                                    note: null,
                                    status: false,
                                });
                                anyApprovement = true;
                            }
                        }
                    }
                    else {
                        if (userId == uo.managerUserId) {
                            organizationRootOrder += 1;
                        }
                        if (organizationRootOrder > ancestors.length) {
                            continue;
                        }
                        else {
                            if (organizationRootOrder > 0)
                                targetOrganizationUnitId = ancestors[ancestors.length - organizationRootOrder];
                        }
                        if (userApprovements.filter((a) => a.organizationUnitId == targetOrganizationUnitId && a.order == currentApprovementOrder).length == 0) {
                            userApprovements.push({
                                id: uuid_1.default.v4(),
                                ppermissionId: ppermissionId,
                                userId: userId,
                                organizationUnitId: targetOrganizationUnitId,
                                order: currentApprovementOrder,
                                approverUserId: null,
                                approvementDate: null,
                                note: null,
                                status: false,
                            });
                            anyApprovement = true;
                        }
                    }
                }
                if (anyApprovement) {
                    organizationRootIndex[currentApprovementRoot] = (organizationRootIndex[currentApprovementRoot] || 0) + 1;
                    currentApprovementOrder++;
                }
            }
        }
        else {
            const systemUserId = dal_constants_1.DalConstants.SystemUserId;
            userApprovements.push({
                id: uuid_1.default.v4(),
                ppermissionId: ppermissionId,
                userId: userId,
                organizationUnitId: null,
                order: 0,
                approverUserId: systemUserId,
                approvementDate: new Date(),
                note: "",
                status: true,
            });
        }
        return Promise.resolve(userApprovements);
    }
    async getUsersPPermissionApprovements(params) {
        const { rows, rowCount } = await (this._pgPool || params.trx).query(`	
			SELECT "userId", json_agg(json_build_object(
								'id', "id",
								'order', "order",  
								'approverUserId', "approverUserId",
								'approvementDate', "approvementDate",
								'note', note, 
								'status', status, 
								'organizationUnitId', "organizationUnitId")) as "data"
				FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userPPermissionApprovements}"
				WHERE "ppermissionId" = $1
				GROUP BY "userId"`, [params.ppermissionId]);
        return rowCount > 0 ? rows : null;
    }
    getLeavesUserFilter(organizationId, requestUserId, listAs, args) {
        let result = { query: "", bindings: args.bindings };
        result.bindings.push(requestUserId);
        if (listAs === dal_constants_1.DalConstants.PPermissionInsertRole.Self) {
            result.query = `
			SELECT $${result.bindings.length}::uuid AS id, '{}'::uuid[] AS "canApprove", false AS "editAsHr", false AS "editAsManager"`;
            return result;
        }
        result.query = `
		SELECT * FROM (
			WITH baseuser AS (
				SELECT 
					_u.id as "userId", _uo.id as "userOrganizationId", _uo."roleId" as "roleId"
				FROM "${organizationId}"."users" _u
				INNER JOIN "${organizationId}"."userOrganizations" _uo
					ON _uo."userId" = _u.id AND _uo."deletedAt" IS NULL AND NOT _uo."isDisabled" AND _u."deletedAt" IS NULL AND _u.id = $${result.bindings.length}
				INNER JOIN "${organizationId}"."userOrganizationProfiles" _uop
					ON _uop."userId" = _u.id AND _uop."deletedAt" is NULL
			),
			_observer AS (
				SELECT
					NULL as "unitId",
					false as "unitWide",
					COALESCE(POSITION('hp:r' IN _rg.permissions) > 0, false) as "read",
					COALESCE(POSITION('hp:w' IN _rg.permissions) > 0, false) as "edit",
					COALESCE(POSITION('hp:h' IN _rg.permissions) > 0, false) as "hrApprove",
					false as "managerApprove"
				FROM baseuser _bu
				INNER JOIN "${organizationId}"."roles" _rg
					ON _rg.id = _bu."roleId" AND _rg."deletedAt" IS NULL
				WHERE POSITION('hp:r' IN _rg.permissions) > 0
				UNION ALL
				SELECT
					_ouc.id as "unitId",
					true as "unitWide",
					COALESCE(POSITION('hp:r' IN _ru.permissions) > 0, false) as "read",
					COALESCE(POSITION('hp:w' IN _ru.permissions) > 0, false) as "editAsManager",
					COALESCE(POSITION('hp:h' IN _ru.permissions) > 0, false) as "hrApprove",
					COALESCE(_ou."managerUserId" = _bu."userId", false) as "managerApprove"
				FROM baseuser _bu
				INNER JOIN "${organizationId}"."userOrganizationOrganizationUnits" _uoou
					ON _uoou."userOrganizationId" = _bu."userOrganizationId" AND _uoou."deletedAt" IS NULL
				INNER JOIN "${organizationId}"."organizationUnits" _ou
					ON _ou.id = _uoou."organizationUnitId" AND _ou."deletedAt" IS NULL
				INNER JOIN "${organizationId}"."roles" _ru
					ON _ru.id = _uoou."roleId" AND _ru."deletedAt" IS NULL
				INNER JOIN "${organizationId}"."organizationUnits" _ouc
					ON _ouc."deletedAt" IS NULL AND (_ouc.id = _ou.id OR POSITION(_ou.id::text IN _ouc."ancestorIds") > 0)
				WHERE POSITION('hp:r' IN _ru.permissions) > 0
			)
			SELECT
				_people.id,
				BOOL_OR(_observer."unitWide") as "unitWide",
				CASE
					WHEN BOOL_OR(_observer."hrApprove") THEN array_remove(array_agg(CASE WHEN _observer."managerApprove" THEN _people."approvableUnits" ELSE null END), null) || ARRAY[NULL::uuid]
					ELSE array_remove(array_agg(CASE WHEN _observer."managerApprove" THEN _people."approvableUnits" ELSE null END), null)
				END as "canApprove",
				BOOL_OR(_observer."unitId" IS NULL AND _observer.edit) as "editAsHr",
				BOOL_OR(_observer."unitId" IS NOT NULL AND _observer.edit) as "editAsManager"
			FROM _observer
			INNER JOIN (
				SELECT __u.id,
				unnest(CASE WHEN __ou."parentId" IS NULL then ARRAY[__ou.id] ELSE STRING_TO_ARRAY(__ou."ancestorIds", ',')::uuid[] || __ou.id END) as "approvableUnits"
				FROM "${organizationId}"."users" __u
				INNER JOIN "${organizationId}"."userOrganizations" __uo
					ON __uo."userId" = __u.id AND __uo."deletedAt" IS NULL AND NOT __uo."isDisabled" AND __u."deletedAt" IS NULL
				INNER JOIN "${organizationId}"."userOrganizationProfiles" __uop
					ON __uop."userId" = __u.id AND __uop."deletedAt" is NULL
				LEFT JOIN "${organizationId}"."userOrganizationOrganizationUnits" __uoou
					ON __uoou."deletedAt" IS NULL AND __uoou."userOrganizationId" = __uo.id
				LEFT JOIN "${organizationId}"."organizationUnits" __ou
					ON __ou.id = __uoou."organizationUnitId" AND __ou."deletedAt" IS NULL
		`;
        if (args.userIds && args.userIds.length > 0) {
            result.bindings.push(args.userIds);
            result.query += `
			WHERE __u.id = ANY($${result.bindings.length})`;
        }
        result.query += `
			) _people
				ON _observer."unitId" IS NULL OR
					_observer."unitId" = _people."approvableUnits"
			GROUP BY _people.id
		) __sq`;
        if (listAs === dal_constants_1.DalConstants.PPermissionInsertRole.Manager) {
            result.query += `
			WHERE "unitWide"`;
        }
        return result;
    }
    async listLeaves(trx, organizationId, requestUserId, listAs, args) {
        const result = {
            pagination: {
                take: args.take,
                skip: args.skip,
                total: 0,
            },
            items: [],
        };
        const userFilter = (0, dal_access_psql_common_1.getReportUserFilterForPgClient)({
            bindingKeys: [],
            specificSelectItems: ["userId"],
            requesterUserId: requestUserId,
            organizationId,
            idBasedUserFilter: {
                applyOrganizationUnitFilterHierarchically: args.organizationUnitHierarchically,
                organizationUnitIds: args.organizationUnitIds ?? [],
                userGroupIds: args.userGroupIds ?? [],
                userIds: args.userIds ?? [],
            },
            requiredOrganizationUnitWidePermissions: ["i:b"],
            requiredOrganizationWidePermissions: ["i:b"],
        });
        const userResult = await trx.query(userFilter.query, userFilter.bindingKeys);
        const userIdsGathered = userResult.rows.map((row) => {
            return row.userId;
        });
        const userQuery = this.getLeavesUserFilter(organizationId, requestUserId, listAs, {
            bindings: [],
        });
        const bindings = userQuery.bindings;
        bindings.push(requestUserId);
        const requestUserBindIndex = bindings.length;
        const systemUserId = dal_constants_1.DalConstants.SystemUserId;
        let query = `
			WITH users AS (
				${userQuery.query}
			)
			SELECT 
			pp.id,
			pp."startDateTime",
			pp."endDateTime",
			ppt.id as "ppermissionTypeId",
			pp.note,
			CASE WHEN patt.data IS NOT NULL THEN patt.data ELSE '[]'::jsonb END as "attachmentInfoList",
			pp."requestUserId",
			pp."requestRole",
			pp."dynamicFormId",
			pp."dynamicFormData",
			upp.*,
			CASE
				WHEN upp.rejected > 0 THEN 2
				WHEN upp.approved > 0 THEN 1
				ELSE 0
			END as status
			FROM (
				SELECT upp_3."leaveId",
					COUNT(*) as "totalCount",
					COUNT(*) FILTER (WHERE upp_3.unauthorized) as "unauthorizedCount",
					COUNT(*) FILTER (WHERE NOT upp_3.unauthorized) as "authorizedCount",
					COUNT(*) FILTER (WHERE upp_3.visible) as "visibleCount",
					COUNT(*) FILTER (WHERE upp_3.visible AND upp_3.status = 2) as "rejected",
					COUNT(*) FILTER (WHERE upp_3.visible AND upp_3.status = 1) as "approved",
					COUNT(*) FILTER (WHERE upp_3.visible AND upp_3.status = 0) as "waiting",
					BOOL_OR(upp_3."isHrApproved") as "isHrApproved",
					BOOL_OR(upp_3."isApproved") as "isApproved",
					BOOL_OR(upp_3."approvedByMe") as "approvedByMe",
					BOOL_OR(upp_3."rejectedByMe") as "rejectedByMe",
					BOOL_OR(cardinality(upp_3."approveAs") > 0) as "waitingMyApproval",
					BOOL_AND(upp_3."editAsHr") as "editAsHr",
					BOOL_AND(upp_3."editAsManager") as "editAsManager",
					BOOL_AND(upp_3."sysApproved") as "sysApproved",
					array_agg(jsonb_build_object('id', "userId", 'status', "status", 'approveAs', "approveAs", 'approveInfo', "approveInfo", 'signedByUser', "signedByUser", 'note', "note")) FILTER (WHERE upp_3.visible) as "userInfo"
				FROM (
					SELECT
						upp_2."leaveId", upp_2."userId", upp_2."unauthorized", upp_2.visible, upp_2."editAsHr", upp_2."editAsManager",
						upp_2."approvedByMeCount" > 0 as "approvedByMe", upp_2."rejectedByMeCount" > 0 as "rejectedByMe",
						upp_2."isHrApproved" > 0 as "isHrApproved", (upp_2."acceptCount" + upp_2."rejectCount" > 0) as "isApproved", upp_2."sysApproved",
						CASE
							WHEN upp_2."rejectCount" > 0 THEN 2
							WHEN upp_2."totalCount" = upp_2."acceptCount" THEN 1
							ELSE 0
						END as status,
						CASE
							WHEN (upp_2."rejectCount" > 0) OR (upp_2."totalCount" = upp_2."acceptCount") THEN ARRAY[]::jsonb[]
							ELSE upp_2."approveAs"
						END as "approveAs",
						upp_2."approveInfo",
						upp_x."signedByUser",
						upp_x."note"
					FROM
					(
						SELECT uppa_1."ppermissionId" as "leaveId", uppa_1."userId",
							users.id IS NULL as unauthorized,`;
        if (userIdsGathered) {
            if (userIdsGathered.length === 0 && (args.userIds?.length > 0 || args.organizationUnitIds?.length > 0 || args.userGroupIds?.length > 0)) {
                bindings.push(null);
            }
            else {
                bindings.push(userIdsGathered);
            }
            query += `
								users.id = ANY($${bindings.length}) as visible,`;
        }
        else {
            query += `
								true as visible,
								`;
        }
        query += `
							BOOL_OR(users."editAsHr") as "editAsHr",
							BOOL_OR(users."editAsManager") as "editAsManager",
							BOOL_AND(uppa_1."approverUserId" = $`;
        bindings.push(systemUserId);
        query += `${bindings.length}) as "sysApproved",
							COUNT(*) as "totalCount",
							COUNT(*) FILTER (WHERE uppa_1."approvementDate" IS NOT NULL AND uppa_1."organizationUnitId" IS NULL) as "isHrApproved",
							COUNT(*) FILTER (WHERE uppa_1."approvementDate" IS NULL) as "waitingCount",
							COUNT(*) FILTER (WHERE uppa_1."approvementDate" IS NOT NULL AND uppa_1.status) as "acceptCount",
							COUNT(*) FILTER (WHERE uppa_1."approvementDate" IS NOT NULL AND NOT uppa_1.status) as "rejectCount",
							COUNT(*) FILTER (WHERE uppa_1."approvementDate" IS NOT NULL AND uppa_1.status AND uppa_1."approverUserId" = $${requestUserBindIndex}) as "approvedByMeCount",
							COUNT(*) FILTER (WHERE uppa_1."approvementDate" IS NOT NULL AND NOT uppa_1.status AND uppa_1."approverUserId" = $${requestUserBindIndex}) as "rejectedByMeCount",
							COALESCE(array_agg(
								jsonb_build_object('id', uppa_1."organizationUnitId", 'name', ou.name)
							) FILTER (WHERE uppa_1."approvementDate" IS NULL AND uppa_1.order = upp_1."currentApprovementOrder" AND array_position(users."canApprove", uppa_1."organizationUnitId") IS NOT NULL),ARRAY[]::jsonb[]) as "approveAs",
							array_agg(to_jsonb(uppa_1) || jsonb_build_object('organizationUnitName', ou.name)) as "approveInfo"
						FROM "${organizationId}"."userPPermissionApprovements" uppa_1
						INNER JOIN "${organizationId}"."userPPermissions" upp_1 ON upp_1."ppermissionId" = uppa_1."ppermissionId" AND upp_1."userId" = uppa_1."userId"
						LEFT JOIN users ON users.id = uppa_1."userId"
						LEFT JOIN "${organizationId}"."organizationUnits" ou ON ou.id = uppa_1."organizationUnitId"
						GROUP BY uppa_1."ppermissionId", uppa_1."userId", users.id
						ORDER BY uppa_1."ppermissionId", uppa_1."userId"
					) upp_2
					INNER JOIN "${organizationId}"."userPPermissions" upp_x
						ON upp_x."ppermissionId" = upp_2."leaveId" AND upp_x."userId" = upp_2."userId"
				) upp_3
				GROUP BY upp_3."leaveId"
			) upp
			INNER JOIN "${organizationId}"."ppermissions" pp
				ON pp.id = upp."leaveId"
			INNER JOIN "${organizationId}"."ppermissionTypes" ppt
				ON pp."ppermissionTypeId" = ppt."id" AND ppt."deletedAt" IS NULL
			LEFT JOIN (
				SELECT patt_1."ppermissionId", jsonb_agg(jsonb_build_object('id', patt_1.id,'name', patt_1.name)) as data
				FROM "${organizationId}"."ppermissionAttachments" patt_1
				GROUP BY patt_1."ppermissionId"
			) patt ON patt."ppermissionId" = pp.id
			WHERE upp."authorizedCount" > 0 AND upp."visibleCount" > 0`;
        if (args.ppermissionTypeId) {
            bindings.push(args.ppermissionTypeId);
            query += `
			AND ppt.id = $${bindings.length}`;
        }
        if (args.ppermissionIds && args.ppermissionIds.length > 0) {
            bindings.push(args.ppermissionIds);
            query += `
			AND pp.id = ANY($${bindings.length})`;
        }
        if (args.startDateTime || args.endDateTime) {
            bindings.push(args.startDateTime ? args.startDateTime : null);
            query += `AND tstzrange(pp."startDateTime", pp."endDateTime") && tstzrange($${bindings.length}`;
            bindings.push(args.endDateTime ? args.endDateTime : null);
            query += `, $${bindings.length})`;
        }
        if (args.status) {
            switch (args.status) {
                case dal_constants_1.DalConstants.PPermissionFilterStatus.ApprovedByMe:
                    query += `
					AND upp."approvedByMe"`;
                    break;
                case dal_constants_1.DalConstants.PPermissionFilterStatus.RejectedByMe:
                    query += `
					AND upp."rejectedByMe"`;
                    break;
                case dal_constants_1.DalConstants.PPermissionFilterStatus.WaitingMyApproval:
                    query += `
					AND upp."waitingMyApproval"`;
                    break;
                case dal_constants_1.DalConstants.PPermissionFilterStatus.Approved:
                    query += `
					AND upp."waiting" = 0 AND upp."rejected" = 0`;
                    break;
                case dal_constants_1.DalConstants.PPermissionFilterStatus.Rejected:
                    query += `
					AND upp."waiting" = 0 AND upp."rejected" > 0`;
                    break;
                case dal_constants_1.DalConstants.PPermissionFilterStatus.Waiting:
                    query += `
					AND upp."waiting" > 0 AND upp."rejected" = 0`;
                    break;
                default:
                    break;
            }
        }
        result.pagination.total = parseInt((await trx.query(`SELECT COUNT(*) as cnt FROM (${query}) sq`, bindings)).rows[0].cnt);
        let sortType;
        let sortOrder = ["ASC", "DESC"].includes(args.sortOrder) ? args.sortOrder : "DESC";
        switch (args.sortType) {
            case app_enums_1.enums.PPermissionFilterSortType.PPermissionStart:
                sortType = ' pp."startDateTime" ';
                break;
            case app_enums_1.enums.PPermissionFilterSortType.PPermissionEnd:
                sortType = ' pp."endDateTime" ';
                break;
            case app_enums_1.enums.PPermissionFilterSortType.PPermissionType:
                sortType = " ppt.name";
                break;
            case app_enums_1.enums.PPermissionFilterSortType.PPermissionApproval:
                sortType = " pp.status ";
                break;
            case app_enums_1.enums.PPermissionFilterSortType.PPermissionDuration:
                sortType = ' pp."endDateTime"::timestamp - pp."startDateTime"::timestamp ';
                break;
            default:
                sortType = ' pp."requestDateTime" ';
                break;
        }
        query += `
		ORDER BY ${sortType} ${sortOrder}`;
        if (args.skip) {
            bindings.push(args.skip);
            query += `
			OFFSET $${bindings.length}`;
        }
        if (args.take) {
            bindings.push(args.take);
            query += `
			LIMIT $${bindings.length}`;
        }
        const dbResult = await trx.query(query, bindings);
        const orgUnitIds = new Set();
        for (const row of dbResult.rows) {
            for (const upp of row.userInfo) {
                upp.approveInfo?.forEach((app) => {
                    if (!app.approverUserId) {
                        orgUnitIds.add(app.organizationUnitId);
                    }
                });
            }
        }
        const orgUnitIdList = Array.from(orgUnitIds);
        const managerOrDeputies = await this.listManagerOrDeputyOfOrganizationUnitBulk(organizationId, orgUnitIdList, new Date());
        const managerMap = new Map(managerOrDeputies.map((m) => [m.organizationUnitId, m.managerOrDeputyUserIds]));
        const userIds = new Set();
        const dynamicFormIds = new Set();
        const ppermissionTypeIds = new Set();
        for (const row of dbResult.rows) {
            if (row.dynamicFormId)
                dynamicFormIds.add(row.dynamicFormId);
            ppermissionTypeIds.add(row.ppermissionTypeId);
            for (const upp of row.userInfo) {
                userIds.add(upp.id);
                upp.approveInfo?.forEach((app) => {
                    if (app.approverUserId) {
                        userIds.add(app.approverUserId);
                        app.approverUsers = [app.approverUserId];
                    }
                    else {
                        const approvers = managerMap.get(app.organizationUnitId) || [];
                        app.approverUsers = approvers;
                        approvers.forEach((id) => userIds.add(id));
                    }
                });
            }
        }
        const [userInfoList, vacations, permissionTypes, dynamicForms] = await Promise.all([
            this.getBasicUserInfoList(organizationId, Array.from(userIds)),
            this.listAllVacations(organizationId),
            this.getPPermissionTypes(organizationId, Array.from(ppermissionTypeIds)),
            dal_manager_1.dbManager.accessSystem.getDynamicForms({ organizationId, dynamicFormIds: Array.from(dynamicFormIds), trx }),
        ]);
        const userMap = new Map(userInfoList.map((u) => [u.id, u]));
        const permissionTypeMap = new Map(permissionTypes.map((pt) => [pt.id, pt]));
        const dynamicFormMap = new Map(dynamicForms.map((df) => [df.id, df]));
        for (const row of dbResult.rows) {
            const permissionType = permissionTypeMap.get(row.ppermissionTypeId);
            const permissionUsageCount = permissionType.isDailyScheduled
                ? await (0, business_pacs_ppermission_1.calculateDailyVacationUsage)(organizationId, permissionType.id, row.startDateTime, row.endDateTime, vacations, row.userInfo[0]?.id, permissionType)
                : await (0, business_pacs_ppermission_1.estimateHourlyVacationUsage)(organizationId, permissionType.id, row.startDateTime, row.endDateTime, vacations, permissionType);
            const userPPermissionInfo = row.userInfo.map((ui) => {
                const basicUserInfo = userMap.get(ui.id);
                if (basicUserInfo) {
                    basicUserInfo.userCaptions = basicUserInfo.captionLines;
                }
                const upi = {
                    userAndDeputyInfo: {
                        user: basicUserInfo,
                        userCaptions: basicUserInfo?.captionLines,
                        deputyCaptions: null,
                        deputyUser: null,
                    },
                    userSigned: ui.signedByUser,
                    status: ui.status,
                    canListerApprove: ui.approveAs.length > 0 &&
                        ((listAs === dal_constants_1.DalConstants.PPermissionInsertRole.HR && !ui.approveAs[0].id) || (listAs === dal_constants_1.DalConstants.PPermissionInsertRole.Manager && ui.approveAs[0].id)),
                    approvementInfo: ui.approveInfo?.map((ai) => {
                        const canApprove = !ai.status &&
                            ui.approveAs.some((aa) => aa.id === ai.organizationUnitId) &&
                            ((listAs === dal_constants_1.DalConstants.PPermissionInsertRole.HR && !ai.organizationUnitId) ||
                                (listAs === dal_constants_1.DalConstants.PPermissionInsertRole.Manager && ai.organizationUnitId));
                        const ret = {
                            approverUsers: [],
                            isHR: ai.organizationUnitId === null,
                            order: ai.order,
                            status: ai.status,
                            note: ai.note,
                            approvementDate: ai.approvementDate,
                            approverUserId: ai.approverUserId,
                            organizationUnit: ai.organizationUnitId,
                            userOrganizationUnitName: ai.organizationUnitName,
                            canListerApprove: !!canApprove,
                        };
                        ret.approverUsers = ai.approverUsers.map((approver) => {
                            let basicApproverInfo = userInfoList.find((uil) => uil.id === approver);
                            if (basicApproverInfo) {
                                basicApproverInfo.userCaptions = userInfoList.find((uc) => uc.id === basicApproverInfo.id).captionLines;
                            }
                            return basicApproverInfo;
                        });
                        return ret;
                    }),
                };
                return upi;
            });
            let item = {
                id: row.id,
                dateRange: {
                    startDateTime: row.startDateTime,
                    endDateTime: row.endDateTime,
                },
                ppermissionType: permissionType,
                note: row.note,
                attachmentInfoList: row.attachmentInfoList,
                status: row.status,
                userCount: parseInt(row.totalCount),
                userPPermissionInfo,
                dynamicFormData: row.dynamicFormData,
                editableStatus: dal_constants_1.DalConstants.PPermissionEditableStatus.Editable,
                approvementInfo: {
                    acceptCount: parseInt(row.approved),
                    rejectCount: parseInt(row.rejected),
                    waitingCount: parseInt(row.waiting),
                    waitingMyApprovalCount: 0,
                },
                permissionUsageCount: permissionUsageCount,
            };
            item.approvementInfo.waitingMyApprovalCount = 0;
            item.userPPermissionInfo.forEach((upi) => {
                upi.approvementInfo.forEach((ai) => {
                    if (ai.canListerApprove) {
                        item.approvementInfo.waitingMyApprovalCount++;
                    }
                });
            });
            item.ppermissionType.dynamicForm = row.dynamicFormId ? dynamicFormMap.get(row.dynamicFormId) ?? null : null;
            if (row.sysApproved) {
                item.editableStatus = dal_constants_1.DalConstants.PPermissionEditableStatus.Editable;
            }
            else if (row.unauthorizedCount > 0 || row.visibleCount < row.authorizedCount) {
                item.editableStatus = dal_constants_1.DalConstants.PPermissionEditableStatus.PermissionHasUnlistedUsers;
            }
            else if (row.isHrApproved && !row.editAsHr) {
                item.editableStatus = dal_constants_1.DalConstants.PPermissionEditableStatus.AlreadyApprovedByHigherPrivilegedUser;
            }
            else if (row.isApproved && !(row.editAsHr || row.editAsManager)) {
                item.editableStatus = dal_constants_1.DalConstants.PPermissionEditableStatus.AlreadyApprovedByHigherPrivilegedUser;
            }
            else if (row.requestRole === dal_constants_1.DalConstants.PPermissionInsertRole.HR && !row.editAsHr) {
                item.editableStatus = dal_constants_1.DalConstants.PPermissionEditableStatus.CreatorHasHigherPrivilege;
            }
            else if (row.requestRole === dal_constants_1.DalConstants.PPermissionInsertRole.Manager && !(row.editAsHr || row.editAsManager)) {
                item.editableStatus = dal_constants_1.DalConstants.PPermissionEditableStatus.CreatorHasHigherPrivilege;
            }
            result.items.push(item);
        }
        return result;
    }
    async getFirstManagerOfUsersOrganizationUnitTree(trx, organizationId, userId) {
        const usersOrganizationUnits = await dal_manager_1.dbManager.accessUser.getUsersOrganizationUnits(organizationId, userId, trx);
        const managerUserIds = new Set();
        const pairsOfManagersAndOrganizationUnit = await dal_manager_1.dbManager.accessPacs.getPairOfOrganizationUnitAndManagerPg(organizationId, trx);
        for (const organizationUnits of usersOrganizationUnits) {
            if (organizationUnits.managerUserId && organizationUnits.managerUserId !== userId) {
                managerUserIds.add(organizationUnits.managerUserId);
                continue;
            }
            for (const ancestorId of organizationUnits.ancestorIds.split(",").reverse()) {
                const managerUserId = pairsOfManagersAndOrganizationUnit.find((elem) => elem.organizationUnitId === ancestorId)?.managerUserId;
                if (managerUserId && !managerUserIds.has(managerUserId) && managerUserId !== userId) {
                    managerUserIds.add(managerUserId);
                    break;
                }
            }
        }
        return Array.from(managerUserIds);
    }
    async getFirstHrOfUsersOrganizationUnitTree(trx, organizationId, userId) {
        const usersOrganizationUnitIds = await dal_manager_1.dbManager.accessUser.getUsersOrganizationUnits(organizationId, userId, trx);
        const hrIds = new Set();
        const pairOfHrAndOrganizationUnit = await dal_manager_1.dbManager.accessPacs.getPairOfOrganizationUnitAndHRPg(organizationId, trx);
        for (const organizationUnit of usersOrganizationUnitIds) {
            if (pairOfHrAndOrganizationUnit[organizationUnit.id]) {
                hrIds.add(pairOfHrAndOrganizationUnit[organizationUnit.id]);
                continue;
            }
            for (const ancestorId of organizationUnit.ancestorIds.split(",").reverse()) {
                if (pairOfHrAndOrganizationUnit[ancestorId] && pairOfHrAndOrganizationUnit[ancestorId] !== userId) {
                    hrIds.add(pairOfHrAndOrganizationUnit[ancestorId]);
                    break;
                }
            }
        }
        if (hrIds.size === 0) {
            const organizationWideHrs = await dal_manager_1.dbManager.accessPacs.getOrganizationWideHRsPg(organizationId, trx);
            organizationWideHrs.forEach((elem) => hrIds.add(elem));
        }
        return Array.from(hrIds);
    }
}
exports.PSQLDalAccessPacs = PSQLDalAccessPacs;
