"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PSQLDalAccessOrganizationUnit = void 0;
const lodash_1 = __importDefault(require("lodash"));
const uuid_1 = __importDefault(require("uuid"));
const api_error_1 = require("../../../api/api.error");
const app_enums_1 = require("../../../app.enums");
const app_logs_1 = require("../../../app.logs");
const dal_constants_1 = require("../../dal.constants");
const dal_manager_1 = require("../../dal.manager");
const dal_db_armon_schema_1 = require("../../db/armon/dal.db.armon.schema");
const dal_access_error_1 = require("../dal.access.error");
const dal_access_rdb_organizationUnit_1 = require("../rdb/dal.access.rdb.organizationUnit");
const predefined_roles_1 = require("../../db/predefined/predefined.roles");
class PSQLDalAccessOrganizationUnit extends dal_access_rdb_organizationUnit_1.RDBDalAccessOrganizationUnit {
    constructor(knex, pgPool) {
        super(knex, pgPool);
    }
    async getAncestorsOfOrganizationUnit(organizationId, organizationUnitId, trx) {
        const trxx = trx ?? this._pgPool;
        let result = [];
        const ancestorIdsResult = await trxx.query(`
			SELECT ou."ancestorIds"
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" ou
			WHERE ou.id = $1
			AND ou."deletedAt" IS NULL;
			`, [organizationUnitId]);
        if (ancestorIdsResult.rowCount > 0) {
            const ancestorIds = ancestorIdsResult.rows[0].ancestorIds?.split(",");
            const { rows } = await trxx.query(`
						SELECT ou."id", ou."name", ou."typeId", ou."parentId", ou."ancestorIds", ou."shortCode", ou."managerUserId", oud.value AS "typeDefinition"
						FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" AS ou
						INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUserDefinedListValues}" AS oud 
							ON ou."typeId" = oud.id
						WHERE ou.id = ANY ($1)
							AND ou."deletedAt" IS NULL;
						`, [ancestorIds]);
            for (let row of rows) {
                let item = {
                    id: row.id,
                    name: row.name,
                    typeId: row.typeId,
                    typeDefinition: JSON.parse(row.typeDefinition),
                    parentId: row.parentId,
                    ancestorIds: row.ancestorIds ? row.ancestorIds[0].split(",") : [],
                    shortCode: row.shortCode,
                };
                item = Object.assign(item, await this.getCurrentOrganizationUnitManagerDeputy(organizationId, row.id));
                result.push(item);
            }
        }
        return result;
    }
    async getOrganizationUnitBasicOfUser(organizationId, userId, trx) {
        const transaction = trx ?? this._pgPool;
        let result = [];
        const { rows, rowCount } = await transaction.query(`
					SELECT ou."id", ou."name", ou."typeId", ou."parentId", ou."ancestorIds", ou."shortCode", ou."managerUserId", oud.value AS "typeDefinition"
					FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" AS ou
					INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUserDefinedListValues}" AS oud 
						ON ou."typeId" = oud.id
					INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}" AS uoou 
						ON ou.id = uoou."organizationUnitId"
					LEFT JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" AS uop
						ON uop."userOrganizationId" = uoou."userOrganizationId"
					WHERE uop."userId" = $1
						AND ou."deletedAt" IS NULL 
						AND uoou."deletedAt" IS NULL;
					`, [userId]);
        for (const row of rows) {
            let item = {
                id: row.id,
                name: row.name,
                typeId: row.typeId,
                typeDefinition: JSON.parse(row.typeDefinition),
                parentId: row.parentId,
                ancestorIds: row.ancestorIds ? row.ancestorIds[0].split(",") : [],
                shortCode: row.shortCode,
            };
            item = Object.assign(item, await this.getCurrentOrganizationUnitManagerDeputy(organizationId, row.id));
            result.push(item);
        }
        return result;
    }
    async getCurrentOrganizationUnitManagerDeputy(organizationId, organizationUnitId, trx) {
        let now = new Date(Date.now());
        let qb = this.dbClient
            .withSchema(organizationId)
            .table("organizationUnits as ou")
            .leftJoin("userOrganizations as uo", (join) => {
            join.on("uo.userId", "ou.managerUserId").andOn(this.dbClient.raw('ou."organizationId" = ? ', organizationId));
        })
            .leftJoin("userOrganizationDeputies as oud", (join) => {
            join.on("uo.id", "oud.userOrganizationId").on(this.dbClient.raw('oud."startDateTime"  <= ? ', now)).on(this.dbClient.raw('oud."endDateTime" >= ? ', now));
        })
            .leftJoin("userOrganizations as uos", (join) => {
            join.on("uos.id", "oud.deputyUserOrganizationId");
        })
            .where("ou.organizationId", organizationId)
            .where("ou.id", organizationUnitId)
            .whereNull("ou.deletedAt");
        if (trx)
            qb.transacting(trx);
        return await qb
            .select(["ou.managerUserId", "uos.userId", "oud.startDateTime", "oud.endDateTime"])
            .first()
            .then(async (result) => {
            if (result)
                return {
                    manager: await dal_manager_1.dbManager.accessPacs.getBasicUserInfo(organizationId, result.managerUserId),
                    deputy: await dal_manager_1.dbManager.accessPacs.getBasicUserInfo(organizationId, result.userId),
                    deputyStartDateTime: result.startDateTime,
                    deputyEndDateTime: result.endDateTime,
                };
        });
    }
    async getIdNamePairsOfOrganizationUnits(organizationId, organizationUnitIds, trx) {
        let transactionScope = async (trx) => {
            return trx
                .withSchema(organizationId)
                .table(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits)
                .whereNull("deletedAt")
                .whereIn("id", organizationUnitIds)
                .where("organizationId", organizationId)
                .select("id", "name");
        };
        if (trx) {
            return transactionScope(trx);
        }
        else {
            return this.dbClient.transaction(transactionScope);
        }
    }
    async getOrganizationUnitBasic(params) {
        const trxx = params.trx ?? this._pgPool;
        let generatedQuery = `
		SELECT ou."id", ou."name", ou."typeId", ou."parentId", ou."ancestorIds", ou."shortCode", ou."managerUserId", oud.value AS "typeDefinition"
		FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" ou
		INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUserDefinedListValues}" oud 
		ON ou."typeId" = oud.id
		WHERE ou.id = $1 ${params.includingDeleted ? "" : ' AND ou."deletedAt" IS NULL'}
		`;
        const { rows, rowCount } = await trxx.query(generatedQuery, [params.organizationUnitId]);
        if (rowCount < 1) {
            (0, dal_access_error_1.throwDbAccessNotFoundError)("Organization unit is not found!");
        }
        const row = rows[0];
        let result = {
            id: row.id,
            name: row.name,
            typeId: row.typeId,
            typeDefinition: JSON.parse(row.typeDefinition),
            parentId: row.parentId,
            ancestorIds: row.ancestorIds ? row.ancestorIds.split(",") : [],
            shortCode: row.shortCode,
        };
        result = Object.assign(result, await this.getCurrentOrganizationUnitManagerDeputy(params.organizationId, params.organizationUnitId));
        return result;
    }
    async getOrganizationUnitDetailed(organizationId, organizationUnitId, hasOrganizationWideReadDetailedOrganizationUnit, userId) {
        if (!hasOrganizationWideReadDetailedOrganizationUnit) {
            let dbAuthResult = await dal_manager_1.dbManager.accessFunctions.dbFuncAuthorizeUserFor({
                organizationId: organizationId,
                userId: userId,
                organizationUnitIds: [organizationUnitId],
            });
            if (!dbAuthResult.result) {
                (0, dal_access_error_1.throwDbAccessAuthorizationError)(dbAuthResult.reason);
            }
        }
        let unit = await dal_manager_1.dbManager.accessFunctions.dbFuncGetOrganizationUnitDetailed(organizationId, organizationUnitId, hasOrganizationWideReadDetailedOrganizationUnit, userId);
        let result = {
            id: unit.id,
            name: unit.name,
            typeId: unit.typeId,
            typeDefinition: unit.typeDefinition,
            dynamicFormData: unit.dynamicFormData,
            parent: !unit.parent
                ? undefined
                : {
                    id: unit.parent.id,
                    name: unit.parent.name,
                    typeId: unit.parent.typeId,
                    typeDefinition: unit.parent.typeDefinition,
                    parentId: unit.parent.parentId,
                    ancestorIds: unit.parent.ancestorIds,
                    dynamicFormData: unit.parent.dynamicFormData,
                    shortCode: unit.parent.shortCode,
                },
            ancestorIds: unit.ancestorIds,
            shortCode: unit.shortCode,
            defaultAccessControlPoints: unit.defaultAccessControlPoints.map((acp) => {
                return {
                    id: acp.id,
                    name: acp.name,
                    location: acp.location,
                    remoteAvailable: acp.remoteAvailable,
                    isRemoteDefault: acp.isRemoteDefault,
                    type: acp.type,
                    managable: acp.managable,
                    defaultPrivileges: acp.defaultPrivileges,
                };
            }),
        };
        return Object.assign(result, await this.getCurrentOrganizationUnitManagerDeputy(organizationId, organizationUnitId, null));
    }
    async upsertOrganizationUnit(organizationId, requesterUserId, args, trx) {
        const id = args.id || uuid_1.default.v4();
        const now = new Date();
        let managerUserOrganizationId = null;
        if (args.managerUserId) {
            managerUserOrganizationId = await dal_manager_1.dbManager.accessUser.getUserOrganizationId({ organizationId, userId: args.managerUserId, trx });
        }
        if (args.id) {
            const { rows: unitRows, rowCount: unitCount } = await trx.query(`
				SELECT id, "managerUserId", "managerUserOrganizationId" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}"
				WHERE id = $1
			`, [args.id]);
            if (unitCount === 0) {
                (0, dal_access_error_1.throwDbAccessNotFoundError)("organization unit not found");
            }
            const unit = unitRows[0];
            await this.updateManagerStatus({
                organizationId,
                organizationUnitId: id,
                trx,
                oldManagerUserId: unit ? unit.managerUserId : null,
                oldManagerUserOrganizationId: unit ? unit.managerUserOrganizationId : null,
                newManagerUserId: args.managerUserId,
                requesterUserId,
            });
            if (!args.parentId) {
                await trx.query(`UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}"
					 SET "parentId" = $1,
						"ancestorIds" = $2,
						"typeId" = $3,
						"shortCode" = $4,
						name = $5,
						"managerUserId" = $6,
						"managerUserOrganizationId" = $7,
						"updatedAt" = $8,
						settings = jsonb_set(settings, '{"dynamicFormData"}', $9)
					WHERE id = $10`, [null, null, args.typeId, args.shortCode, args.name, args.managerUserId, managerUserOrganizationId, now, JSON.stringify(args.dynamicFormData), args.id]);
            }
            else {
                await trx.query(`UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}"
					 SET "parentId" = $1,
						 "typeId" = $2,
						 "shortCode" = $3,
						 name = $4,
						 "managerUserId" = $5,
						 "managerUserOrganizationId" = $6,
						 "updatedAt" = $7,
						 settings = jsonb_set(settings, '{"dynamicFormData"}', $8)
					WHERE id = $9`, [args.parentId, args.typeId, args.shortCode, args.name, args.managerUserId, managerUserOrganizationId, now, JSON.stringify(args.dynamicFormData), args.id]);
            }
        }
        else {
            await trx.query(`INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}"
			(id, "organizationId", "parentId", "typeId", "shortCode", name, "managerUserId", "managerUserOrganizationId", "createdAt", "updatedAt")
			VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10)`, [id, organizationId, args.parentId, args.typeId, args.shortCode, args.name, args.managerUserId, managerUserOrganizationId, now, now]);
            if (args.managerUserId) {
                await this.updateManagerStatus({
                    organizationId,
                    organizationUnitId: id,
                    trx,
                    oldManagerUserId: null,
                    oldManagerUserOrganizationId: null,
                    newManagerUserId: args.managerUserId,
                    requesterUserId,
                });
            }
            if (args.parentId) {
                const parent = await this.getOrganizationUnitBasic({ organizationId, organizationUnitId: args.parentId, trx });
                if (parent) {
                    await trx.query(`
						UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}"
						SET "ancestorIds" = $1
						WHERE id = $2
						`, [parent.ancestorIds && parent.ancestorIds.length > 0 ? parent.ancestorIds.join(",") + "," + parent.id : args.parentId, id]);
                }
            }
        }
        return id;
    }
    async updateManagerStatus(params) {
        let roles = await dal_manager_1.dbManager.accessUser.getTypedRoleIdsOfOrganizationPg({ organizationId: params.organizationId, trx: params.trx });
        if (params.oldManagerUserId && params.oldManagerUserId !== params.newManagerUserId) {
            if (!params.oldManagerUserOrganizationId) {
                params.oldManagerUserOrganizationId = await dal_manager_1.dbManager.accessUser.getUserOrganizationId({ organizationId: params.organizationId, userId: params.oldManagerUserId });
            }
            let newManagerUserOrganizationId = await dal_manager_1.dbManager.accessUser.getUserOrganizationId({ organizationId: params.organizationId, userId: params.newManagerUserId });
            let relatedUnit = (await params.trx.query(`SELECT * FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}"
					WHERE "id" = $1`, [params.organizationUnitId])).rows[0];
            let relatedUserOrganizationUnitRole = (await params.trx.query(`SELECT r.name as name FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}" uoou
					LEFT JOIN "${params.organizationId}"."roles" r ON r.id = uoou."roleId"
					WHERE uoou."userOrganizationId" = $1 AND uoou."organizationUnitId" = $2`, [newManagerUserOrganizationId, params.organizationUnitId])).rows[0];
            await params.trx.query(`UPDATE "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}"
				SET "roleId" = $1,
				"extraPermissions" = $2
				WHERE "userOrganizationId" = $3
				AND "organizationUnitId" = $4`, [roles.standartUserId, null, params.oldManagerUserOrganizationId, params.organizationUnitId]);
            let oldManagerLog = {
                oId: params.organizationId,
                o: params.requesterUserId,
                u: new Date(),
                c: dal_constants_1.DalConstants.UserActionCategory.Main,
                tg: ["Identity", "OrganizationUnit"],
                t: dal_constants_1.DalConstants.UserMainActionType.UpdateUserOrganizationUnitRole,
                d: {
                    c: JSON.stringify([{ f: relatedUnit?.name, o: predefined_roles_1.PredefinedRoles.UnitAdministrator.name, n: predefined_roles_1.PredefinedRoles.StandartUser.name, i: true }]),
                    id: params.oldManagerUserId,
                },
            };
            await params.trx.query(`INSERT INTO "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.UserActionHistory}"
				(id, "organizationId", "actionUtc", "userId", category, type, log)
				VALUES(uuid_generate_v4(), $1, now(), $2, $3, $4, $5);`, [
                params.organizationId,
                params.requesterUserId,
                dal_constants_1.DalConstants.UserActionCategory.Main,
                dal_constants_1.DalConstants.UserMainActionType.UpdateUserOrganizationUnitRole,
                JSON.stringify(oldManagerLog),
            ]);
            if (!relatedUserOrganizationUnitRole) {
                let userAddedToUnitLog = {
                    oId: params.organizationId,
                    o: params.requesterUserId,
                    u: new Date(),
                    c: dal_constants_1.DalConstants.UserActionCategory.Main,
                    tg: ["Identity", "OrganizationUnit"],
                    t: dal_constants_1.DalConstants.UserMainActionType.AssignUserToOrganizationUnit,
                    d: {
                        c: JSON.stringify([
                            {
                                f: "organizationUnitName",
                                o: null,
                                n: relatedUnit.name,
                                i: true,
                            },
                        ]),
                        id: params.newManagerUserId,
                    },
                };
                await params.trx.query(`INSERT INTO "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.UserActionHistory}"
					(id, "organizationId", "actionUtc", "userId", category, type, log)
					VALUES(uuid_generate_v4(), $1, now(), $2, $3, $4, $5);`, [
                    params.organizationId,
                    params.requesterUserId,
                    dal_constants_1.DalConstants.UserActionCategory.Main,
                    dal_constants_1.DalConstants.UserMainActionType.AssignUserToOrganizationUnit,
                    JSON.stringify(userAddedToUnitLog),
                ]);
            }
            let newManagerLog = {
                oId: params.organizationId,
                o: params.requesterUserId,
                u: new Date(),
                c: dal_constants_1.DalConstants.UserActionCategory.Main,
                tg: ["Identity", "OrganizationUnit"],
                t: dal_constants_1.DalConstants.UserMainActionType.UpdateUserOrganizationUnitRole,
                d: {
                    c: JSON.stringify([
                        { f: relatedUnit.name, o: relatedUserOrganizationUnitRole?.name ?? predefined_roles_1.PredefinedRoles.StandartUser.name, n: predefined_roles_1.PredefinedRoles.UnitAdministrator.name, i: true },
                    ]),
                    id: params.newManagerUserId,
                },
            };
            await params.trx.query(`INSERT INTO "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.UserActionHistory}"
				(id, "organizationId", "actionUtc", "userId", category, type, log)
				VALUES(uuid_generate_v4(), $1, now(), $2, $3, $4, $5);`, [
                params.organizationId,
                params.requesterUserId,
                dal_constants_1.DalConstants.UserActionCategory.Main,
                dal_constants_1.DalConstants.UserMainActionType.UpdateUserOrganizationUnitRole,
                JSON.stringify(newManagerLog),
            ]);
        }
        if (params.newManagerUserId) {
            const { rows: existsInOrganizationUnitRows, rowCount: existsInOrganizationUnitRowCount } = await params.trx.query(`
			SELECT uoou.id FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}" uoou
			INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo ON uoou."userOrganizationId" = uo.id
			WHERE uo."userId" = $1
			AND uoou."organizationUnitId" = $2
			AND uo."deletedAt" IS NULL
			AND uoou."deletedAt" IS NULL
			`, [params.newManagerUserId, params.organizationUnitId]);
            if (existsInOrganizationUnitRowCount > 0) {
                await params.trx.query(`UPDATE "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}"
				SET "roleId" = $1,
					"extraPermissions" = $2
				WHERE id = $3`, [roles.unitAdministratorId, null, existsInOrganizationUnitRows[0].id]);
            }
            else {
                await params.trx.query(`INSERT INTO "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}"
                        (id, "createdAt", "updatedAt", "deletedAt", "userOrganizationId", "organizationUnitId", "roleId", "extraPermissions")
                    SELECT uuid_generate_v4(), now(), now(), null, uo."id", $1, $2, null from "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" as uo 
                    WHERE uo."userId" = $3 AND uo."organizationId" = $4
					ON CONFLICT ("userOrganizationId", "organizationUnitId") WHERE "deletedAt" IS NULL
					DO UPDATE SET 
						"roleId" = EXCLUDED."roleId", 
						"extraPermissions" = EXCLUDED."extraPermissions";`, [params.organizationUnitId, roles.unitAdministratorId, params.newManagerUserId, params.organizationId]);
            }
        }
        else {
            await params.trx.query(`UPDATE "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}"
			SET "managerUserId" = $1,
				"managerUserOrganizationId" = $2
			WHERE id = $3`, [null, null, params.organizationUnitId]);
        }
    }
    async getChildrenOfOrganizationUnit(params) {
        const result = [];
        const trxx = params.trx ?? this._pgPool;
        const { rows, rowCount } = await trxx.query(`
					SELECT ou."id", ou."name", ou."typeId", ou."parentId", ou."ancestorIds", ou."shortCode", ou."managerUserId", oud.value AS "typeDefinition"
					FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" ou
					INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUserDefinedListValues}" oud ON ou."typeId" = oud.id
					WHERE ou."parentId" = $1
					AND ou."deletedAt" IS NULL
					`, [params.organizationUnitId]);
        for (const row of rows) {
            let item = {
                id: row.id,
                name: row.name,
                typeId: row.typeId,
                typeDefinition: JSON.parse(row.typeDefinition),
                parentId: row.parentId,
                ancestorIds: row.ancestorIds ? row.ancestorIds.split(",") : [],
                shortCode: row.shortCode,
            };
            item = Object.assign(item, await this.getCurrentOrganizationUnitManagerDeputy(params.organizationId, row.id));
            result.push(item);
        }
        return result;
    }
    async getSiblingsOfOrganizationUnit(organizationId, organizationUnitId) {
        return await this.dbClient.transaction(async (trx) => {
            let result = {
                siblings: [],
                ancestors: [],
                ancestorIds: [],
            };
            let current = await this.getOrganizationUnitBasic({ organizationId, organizationUnitId });
            if (current) {
                result.id = current.id;
                result.name = current.name;
                result.parentId = current.parentId;
                result.manager = current.manager;
                result.shortCode = current.shortCode;
                result.typeId = current.typeId;
                result.typeDefinition = current.typeDefinition;
                result.deputy = current.deputy;
                result.deputyStartDateTime = current.deputyStartDateTime;
                result.deputyEndDateTime = current.deputyEndDateTime;
                result.ancestorIds = current.ancestorIds;
                result.shortCode = current.shortCode;
            }
            result.siblings = await this.getChildrenOfOrganizationUnit({ organizationId, organizationUnitId: result.parentId });
            if (result.siblings) {
                let selfIndex = result.siblings.findIndex((x) => {
                    return x.id == result.id;
                });
                delete result.siblings[selfIndex];
                result.siblings.splice(selfIndex, 1);
            }
            if (current) {
                for (let element of result.ancestorIds) {
                    result.ancestors.push(await this.getOrganizationUnitBasic({ organizationId, organizationUnitId: element }));
                }
            }
            return Promise.resolve(result);
        });
    }
    async removeOrganizationUnit(organizationId, organizationUnitId, action, requestUserId, trx) {
        let organizationUnit = await this.getOrganizationUnitBasic({ organizationId, organizationUnitId: organizationUnitId, trx: trx });
        let permissionTypes = (await trx.query(`SELECT DISTINCT(ppt."name") as name
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissionTypeApprovements}" as ppta
				INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissionTypes}" as ppt 
					ON ppt."id" = ppta."ppermissionTypeId"
				WHERE "rootOrganizationUnitId" = $1 and "deletedAt" IS NULL;`, [organizationUnit.id])).rows;
        if (permissionTypes.length > 0) {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.BAD_REQUEST, "ERRORS.PACS.UNITHASPPERMISSIONTYPE", {
                permissionTypeNames: permissionTypes.map((elem) => elem.name).join(", "),
            });
        }
        let approvalPendingPermissions = (await trx.query(`SELECT "ppermissionId" as pids
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userPPermissionApprovements}" as uppa
				GROUP BY uppa."ppermissionId"
				HAVING bool_or(not uppa."status") 
					AND bool_or(uppa."organizationUnitId" = $1)
					AND bool_and(uppa."approvementDate" IS NULL OR uppa."status");`, [organizationUnit.id])).rows;
        if (approvalPendingPermissions.length > 0) {
            let permissionTypeNames = (await trx.query(`SELECT DISTINCT(ppt.name)
					FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissions}" as pp
					INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissionTypes}" as ppt
						ON	pp."ppermissionTypeId" = ppt."id"
					WHERE pp.id = ANY($1) AND ppt."deletedAt" IS NULL;`, [approvalPendingPermissions.map((row) => row.pids)])).rows
                .map((row) => row.name)
                .join(", ");
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.BAD_REQUEST, "ERRORS.PACS.UNITHASPPERMISSION", { permissionTypeNames: permissionTypeNames });
        }
        await trx.query(`UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.workPlans}"
			SET "organizationUnitId" = $1
			WHERE "organizationUnitId" = $2 AND "deletedAt" IS NULL;`, [organizationUnit.parentId, organizationUnit.id]);
        const affectedUnits = await trx.query(`UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}"
			SET "deletedAt" = now()
			WHERE id = $1;`, [organizationUnit.id]);
        await trx.query(`UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.visitorRegistrationPoints}"
			SET "organizationUnitId" = NULL
			WHERE "organizationUnitId" = $1;`, [organizationUnit.id]);
        if (affectedUnits.rowCount == 0)
            return;
        await trx.query(`UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}"
			SET "parentId" = $1, "ancestorIds" = $2
			WHERE "parentId" = $3 AND "deletedAt" IS NULL;`, [organizationUnit.parentId, organizationUnit.ancestorIds ? organizationUnit.ancestorIds.join(",") : null, organizationUnit.id]);
        let updateList = await trx.query(`SELECT id, "ancestorIds"
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}"
			WHERE "deletedAt" IS NULL AND "parentId" <> $1 AND "ancestorIds" like $2;`, [organizationUnit.id, "%" + organizationUnit.id + "%"]);
        for (const targetUnit of updateList.rows) {
            targetUnit.ancestorIds = targetUnit.ancestorIds
                .split(",")
                .filter((u) => u != organizationUnitId)
                .join(",");
            await trx.query(`UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}"
				SET "ancestorIds" = $1
				WHERE "deletedAt" IS NULL AND id = $2;`, [targetUnit.ancestorIds, targetUnit.id]);
        }
        let targetUserIds = (await trx.query(`SELECT uo."userId"
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo
				INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}" uoou
					ON uoou."organizationUnitId" = $1 AND uoou."userOrganizationId" = uo.id
					AND uo."deletedAt" IS NULL AND uoou."deletedAt" IS NULL`, [organizationUnitId])).rows.map((r) => r.userId);
        if (action === app_enums_1.enums.RemoveOrganizationUnitAction.RemoveAccessRights) {
            let acpList = (await trx.query(`SELECT "accessControlPointId"
					FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnitDefaultAccessControlPoints}"
					WHERE "deletedAt" IS NULL AND "organizationUnitId" = $1;`, [organizationUnitId])).rows.map((r) => r.accessControlPointId);
            if (acpList.length > 0) {
                await trx.query(`UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userAccessRights}"
					SET "deletedAt" = now()
					WHERE "deletedAt" IS NULL AND access = true AND "accessControlPointId" = ANY($1) AND "userId" = ANY($2);`, [acpList, targetUserIds]);
            }
        }
        await trx.query(`UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}"
			SET "deletedAt" = now()
			WHERE "deletedAt" IS NULL AND "organizationUnitId" = $1;`, [organizationUnitId]);
        if (action === app_enums_1.enums.RemoveOrganizationUnitAction.MakeUsersPassive) {
            for (const uid of targetUserIds) {
                const basicInfo = await trx.query(`SELECT
						uop.name, uop.surname, uo."roleId", uop."uniqueId", uop."phoneNumber", uop.email, uop.address,
						uop."employmentStartUtc", uop."employmentEndUtc", uop."previousServiceDuration", uop."extensionFields",
						uop."pacsEnabledRemainedAnnualPPermission", uop."birthDateUtc",uo."extraPermissions"
					FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" uop
					INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo
						ON uop."userId" = $1 AND uo."userId" = uop."userId" AND uo."deletedAt" IS NULL AND uop."deletedAt" IS NULL`, [uid]);
                await dal_manager_1.dbManager.accessIdentity.updateIdentity(organizationId, requestUserId, trx, {
                    id: uid,
                    organizationProfile: {
                        locale: "tr",
                        name: basicInfo.rows[0].name,
                        surname: basicInfo.rows[0].surname,
                        roleId: basicInfo.rows[0].roleId,
                        uniqueId: basicInfo.rows[0].uniqueId,
                        isDisabled: true,
                        phoneNumber: basicInfo.rows[0].phoneNumber,
                        email: basicInfo.rows[0].email,
                        address: basicInfo.rows[0].address,
                        employmentStartUtc: basicInfo.rows[0].employmentStartUtc,
                        employmentEndUtc: basicInfo.rows[0].employmentEndUtc,
                        previousServiceDuration: basicInfo.rows[0].previousServiceDuration,
                        extensionFields: basicInfo.rows[0].extensionFields,
                        pacsEnabledRemainedAnnualPPermission: basicInfo.rows[0].pacsEnabledRemainedAnnualPPermission,
                        birthDateUtc: basicInfo.rows[0].birthDateUtc,
                        extraPermissions: basicInfo.rows[0].extraPermissions,
                    },
                }, true);
            }
        }
    }
    async updateForUsers(organizationId, targetUserIds, addedAccessControlPoints, removedAccessControlPointIds, opTime, trx) {
        if (removedAccessControlPointIds.length > 0 && targetUserIds.length > 0) {
            await trx
                .withSchema(organizationId)
                .from(dal_db_armon_schema_1.ArmonSchema.tableNames.userAccessRights)
                .whereNull("deletedAt")
                .whereIn("userId", targetUserIds)
                .whereIn("accessControlPointId", removedAccessControlPointIds)
                .update({
                deletedAt: opTime,
                updatedAt: opTime,
            });
        }
        let addUserAccessRights = [];
        if (addedAccessControlPoints && addedAccessControlPoints.length > 0) {
            let addedAccessControlPointIds = addedAccessControlPoints.map((a) => a.id);
            let existingRights = await trx
                .withSchema(organizationId)
                .from(dal_db_armon_schema_1.ArmonSchema.tableNames.userAccessRights)
                .whereNull("deletedAt")
                .whereIn("userId", targetUserIds)
                .whereIn("accessControlPointId", addedAccessControlPointIds)
                .select("id", "accessControlPointId", "userId");
            for (const acpId of addedAccessControlPointIds) {
                let privileges = addedAccessControlPoints.find((a) => a.id == acpId);
                for (const userId of targetUserIds) {
                    if (!existingRights.some((r) => r.userId == userId && r.accessControlPointId == acpId)) {
                        addUserAccessRights.push({
                            id: uuid_1.default.v4(),
                            createdAt: opTime,
                            deletedAt: null,
                            updatedAt: opTime,
                            accessControlPointId: acpId,
                            userId: userId,
                            access: privileges.access,
                            grant: privileges.grant,
                            config: privileges.config,
                            read: privileges.read,
                            remoteAccess: privileges.remoteAccess,
                            snapshot: privileges.snapshot,
                        });
                    }
                }
                await trx
                    .withSchema(organizationId)
                    .from(dal_db_armon_schema_1.ArmonSchema.tableNames.userAccessRights)
                    .whereNull("deletedAt")
                    .whereIn("userId", targetUserIds)
                    .where("accessControlPointId", acpId)
                    .update({
                    access: privileges.access,
                    grant: privileges.grant,
                    config: privileges.config,
                    read: privileges.read,
                    remoteAccess: privileges.remoteAccess,
                    updatedAt: opTime,
                    snapshot: privileges.snapshot,
                });
            }
            if (addUserAccessRights.length > 0) {
                addUserAccessRights = lodash_1.default.uniqBy(addUserAccessRights, (a) => a.userId + a.accessControlPointId);
                await trx.batchInsert(`${organizationId}.userAccessRights`, addUserAccessRights, 250);
            }
        }
        return Promise.resolve();
    }
    async changeOrganizationUnitDefaultAccessControlPoints(organizationId, organizationUnitId, addedAccessControlPoints, removedAccessControlPointIds, applyToMembers, applyHierarchically, hasOrganizationWideReadDetailedOrganizationUnit, userId) {
        if (!hasOrganizationWideReadDetailedOrganizationUnit) {
            let dbAuthResult = await dal_manager_1.dbManager.accessFunctions.dbFuncAuthorizeUserFor({
                organizationId: organizationId,
                userId: userId,
                organizationUnitIds: [organizationUnitId],
                accessControlPointIds: removedAccessControlPointIds.concat(addedAccessControlPoints.map((a) => a.id)),
            });
            if (!dbAuthResult.result) {
                (0, dal_access_error_1.throwDbAccessAuthorizationError)(dbAuthResult.reason);
            }
        }
        let opTime = new Date();
        await this.dbClient.transaction(async (trx) => {
            let targetOrganizationUnitIds = [];
            if (!applyHierarchically) {
                targetOrganizationUnitIds = [organizationUnitId];
            }
            else {
                targetOrganizationUnitIds = (await trx
                    .withSchema(organizationId)
                    .from(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits)
                    .where("organizationId", organizationId)
                    .whereNull("deletedAt")
                    .andWhereRaw(` "id" = ? OR "ancestorIds" LIKE '%' || ? || '%' `, [organizationUnitId, organizationUnitId])
                    .select("id")).map((o) => o.id);
            }
            let targetUserIds = [];
            if (applyToMembers) {
                targetUserIds = (await trx
                    .withSchema(organizationId)
                    .from(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations + " as uo")
                    .innerJoin(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits + " as uoou", "uoou.userOrganizationId", "uo.id")
                    .whereNull("uoou.deletedAt")
                    .whereNull("uo.deletedAt")
                    .whereIn("uoou.organizationUnitId", targetOrganizationUnitIds)
                    .select("uo.userId")).map((u) => u.userId);
            }
            let indexOfCurrentUser = targetUserIds.indexOf(userId);
            if (indexOfCurrentUser > -1) {
                targetUserIds.splice(indexOfCurrentUser, 1);
            }
            for (let page = 0; page < targetUserIds.length; page += 1000) {
                let userChunk = targetUserIds.slice(page, page + 1000);
                await this.updateForUsers(organizationId, userChunk, addedAccessControlPoints, removedAccessControlPointIds, opTime, trx);
            }
            let addedDefaultAccessControlPoints = [];
            if (removedAccessControlPointIds.length > 0) {
                await trx
                    .withSchema(organizationId)
                    .from(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnitDefaultAccessControlPoints)
                    .whereNull("deletedAt")
                    .whereIn("organizationUnitId", targetOrganizationUnitIds)
                    .whereIn("accessControlPointId", removedAccessControlPointIds)
                    .update({
                    deletedAt: opTime,
                    updatedAt: opTime,
                });
            }
            if (addedAccessControlPoints && addedAccessControlPoints.length > 0) {
                let addedAccessControlPointIds = addedAccessControlPoints.map((a) => a.id);
                let existingControlPoints = await trx
                    .withSchema(organizationId)
                    .from(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnitDefaultAccessControlPoints)
                    .whereNull("deletedAt")
                    .whereIn("organizationUnitId", targetOrganizationUnitIds)
                    .whereIn("accessControlPointId", addedAccessControlPointIds)
                    .select();
                for (const acpId of addedAccessControlPointIds) {
                    let privileges = addedAccessControlPoints.find((a) => a.id == acpId);
                    for (const organizationUnitId of targetOrganizationUnitIds) {
                        let acpExistsForUnit = existingControlPoints.some((a) => a.accessControlPointId == acpId && a.organizationUnitId == organizationUnitId);
                        if (acpExistsForUnit) {
                            await trx
                                .withSchema(organizationId)
                                .from(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnitDefaultAccessControlPoints)
                                .where("organizationUnitId", organizationUnitId)
                                .where("accessControlPointId", acpId)
                                .update({
                                defaultPrivileges: {
                                    access: privileges.access,
                                    grant: privileges.grant,
                                    config: privileges.config,
                                    read: privileges.read,
                                    remoteAccess: privileges.remoteAccess,
                                    snapshot: privileges.snapshot,
                                },
                            });
                        }
                        else {
                            addedDefaultAccessControlPoints.push({
                                id: uuid_1.default.v4(),
                                createdAt: opTime,
                                updatedAt: opTime,
                                deletedAt: null,
                                accessControlPointId: acpId,
                                organizationUnitId: organizationUnitId,
                                defaultPrivileges: {
                                    access: privileges.access,
                                    grant: privileges.grant,
                                    config: privileges.config,
                                    read: privileges.read,
                                    remoteAccess: privileges.remoteAccess,
                                    snapshot: privileges.snapshot,
                                },
                            });
                        }
                    }
                }
                if (addedDefaultAccessControlPoints.length > 0) {
                    await trx.withSchema(organizationId).from(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnitDefaultAccessControlPoints).insert(addedDefaultAccessControlPoints);
                }
            }
            let affectedAcpIds = addedAccessControlPoints.map((a) => a.id).concat(removedAccessControlPointIds);
            if (affectedAcpIds.length > 0) {
                let acpIds = lodash_1.default.uniq(affectedAcpIds);
                for (const acpId of acpIds) {
                    let accessRights = await trx
                        .withSchema(organizationId)
                        .from(dal_db_armon_schema_1.ArmonSchema.tableNames.userAccessRights)
                        .where("accessControlPointId", acpId)
                        .whereNull("deletedAt")
                        .select("accessControlPointId as id", "access", "read", "config", "grant", "remoteAccess", "userId", "snapshot");
                    let countOfGrantUsers = accessRights.filter((a) => a.id == acpId && a.grant && a.read);
                    if (countOfGrantUsers.length < 1) {
                        (0, dal_access_error_1.throwDbAccessBadRequestErrorTr)("ERRORS.GENERAL.LASTUSERWITHREAD");
                    }
                    let countOfConfigUsers = accessRights.filter((a) => a.id == acpId && a.config && a.read);
                    if (countOfConfigUsers.length < 1) {
                        (0, dal_access_error_1.throwDbAccessBadRequestErrorTr)("ERRORS.GENERAL.LASTUSERWITHREAD");
                    }
                }
            }
        });
        return Promise.resolve();
    }
    async genericSearchOrganizationUnit(organizationId, options, underAuthorityOrganizationUnitIds) {
        let result = {
            total: 0,
            items: [],
        };
        let qb = this.dbClient.withSchema(organizationId).from("organizationUnits as ou").where("ou.organizationId", "=", organizationId).whereNull("ou.deletedAt");
        if (underAuthorityOrganizationUnitIds && underAuthorityOrganizationUnitIds.length > 0) {
            qb.where((qbOr) => {
                qbOr.whereIn("id", underAuthorityOrganizationUnitIds).orWhereRaw(`"ancestorIds" similar to '%(${underAuthorityOrganizationUnitIds.join("|")})%'`);
            });
        }
        let filterExists = options.filter && options.filter.trim().length > 0;
        let regexList = [];
        if (filterExists) {
            qb.whereWrapped((q) => {
                let filterTokens = options.filter.split(" ");
                for (const filterToken of filterTokens) {
                    let token = filterToken.trim();
                    regexList.push(new RegExp(options.filter.replace(/[-[\]{}()*+?.,\\^$|#\s]/g, "\\$&"), "ig"));
                    if (token.length > 0) {
                        token = "%" + token + "%";
                        q.orWhereRaw("UNACCENT(UPPER(ou.name)) ilike UNACCENT(UPPER(?))", token);
                        q.orWhereRaw('UNACCENT(UPPER(ou."shortCode")) ilike UNACCENT(UPPER(?))', token);
                    }
                }
            });
        }
        let columns = ["ou.id", "ou.name", { shortCode: "ou.shortCode" }];
        let qbCount = qb.clone();
        result.total = await qbCount.count("ou.id as c").then((rows) => {
            return parseInt(rows[0].c);
        });
        if (result.total === 0) {
            return Promise.resolve(result);
        }
        qb.orderByRaw("ou.name").column(columns).select().offset(options.skip).limit(options.take);
        result.items = await qb.then((rows) => {
            return rows.map((t) => {
                let name = t["name"];
                let shortCode = t["shortCode"];
                let captionLines = [name];
                if (shortCode) {
                    captionLines.push(shortCode);
                }
                let matchItem = null;
                if (filterExists) {
                    for (const regex of regexList) {
                        if (name.search(regex) >= 0) {
                            matchItem = name;
                            break;
                        }
                        else if (shortCode && shortCode.search(regex) >= 0) {
                            matchItem = shortCode;
                            break;
                        }
                    }
                }
                return {
                    id: t["id"],
                    captionLines: captionLines,
                    matchItem: matchItem,
                };
            });
        });
        return Promise.resolve(result);
    }
    async listOrganizationUnits(organizationId, args) {
        let result = {
            total: 0,
            items: [],
        };
        let qb = this.dbClient.withSchema(organizationId).from("organizationUnits as ou").whereIn("ou.id", args.organizationUnitIds).whereNull("ou.deletedAt");
        result.total = await qb
            .clone()
            .count("ou.id as c")
            .then((rows) => {
            return parseInt(rows[0].c);
        });
        if (result.total === 0) {
            return Promise.resolve(result);
        }
        if (args.take)
            qb.limit(args.take);
        if (args.skip)
            qb.offset(args.skip);
        result.items = await qb
            .orderByRaw("ou.name")
            .select("ou.id", "ou.name", "ou.shortCode")
            .then((rows) => {
            return rows.map((t) => {
                let name = t["name"];
                let shortCode = t["shortCode"];
                let captionLines = [name];
                if (shortCode) {
                    captionLines.push(shortCode);
                }
                return {
                    id: t["id"],
                    captionLines: captionLines,
                };
            });
        });
        return Promise.resolve(result);
    }
    async getOrganizationUnitsAtRootLevel(params) {
        let result = [];
        const trxx = params.trx ?? this._pgPool;
        const { rows } = await trxx.query(`
			SELECT ou.id, ou.name, ou."typeId", oud."value" as "typeDefinition", ou."ancestorIds", ou."managerUserId", ou."shortCode"
			FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" ou
			INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUserDefinedListValues}" oud ON ou."typeId" = oud.id
			WHERE ou."parentId" IS NULL
			AND ou."deletedAt" IS NULL
			`);
        for (const row of rows) {
            result.push({
                id: row.id,
                ancestorIds: row.ancestorIds ? row.ancestorIds.split(",") : [],
                name: row.name,
                typeId: row.typeId,
                typeDefinition: JSON.parse(row.typeDefinition),
                manager: await dal_manager_1.dbManager.accessPacs.getBasicUserInfo(params.organizationId, row.managerUserId),
                shortCode: row.shortCode,
            });
        }
        return result;
    }
    async upsertOrganizationUnitUsers(organizationId, organizationUnitId, userIds, standartUserId) {
        await this.dbClient.raw(`INSERT INTO "${organizationId}"."userOrganizationOrganizationUnits"
            (id, "createdAt", "updatedAt", "deletedAt", "userOrganizationId",
            "organizationUnitId", "roleId", "extraPermissions")
			SELECT uuid_generate_v4(), now(), now(), null, uo."id", ?, ?, null from "${organizationId}"."userOrganizations" as uo 
			where uo."userId" = ANY(?) AND uo."organizationId" = ?
			ON CONFLICT ("userOrganizationId", "organizationUnitId") WHERE "deletedAt" IS NULL
			DO UPDATE SET 
				"roleId" = EXCLUDED."roleId", 
				"extraPermissions" = EXCLUDED."extraPermissions";`, [organizationUnitId, standartUserId, userIds, organizationId]);
    }
    async getOrganizationUnitIdFromShortCode(organizationId, shortCode) {
        let organizationUnitId = "";
        await this.dbClient
            .withSchema(organizationId)
            .from("organizationUnits")
            .where("shortCode", shortCode)
            .where("organizationId", organizationId)
            .first("id")
            .then((result) => {
            if (result)
                organizationUnitId = result.id;
        });
        return Promise.resolve(organizationUnitId);
    }
    async listUserOrganizationUnitNames(organizationId, userId) {
        return await this.dbClient
            .withSchema(organizationId)
            .from("userOrganizations as uo")
            .innerJoin("userOrganizationOrganizationUnits as uoou", "uo.id", "uoou.userOrganizationId")
            .innerJoin("organizationUnits as ou", "ou.id", "uoou.organizationUnitId")
            .where("uo.userId", userId)
            .where("uo.organizationId", organizationId)
            .whereNull("uo.deletedAt")
            .whereNull("uoou.deletedAt")
            .whereNull("ou.deletedAt")
            .select("ou.name")
            .then((rows) => {
            if (rows)
                return rows.map((r) => r.name);
        });
    }
    async listUsersWithOrganizationUnits(organizationId, organizationUnitIds) {
        let qb = this.dbClient
            .withSchema(organizationId)
            .from("userOrganizations as uo")
            .innerJoin("userOrganizationOrganizationUnits as uoou", "uo.id", "uoou.userOrganizationId")
            .innerJoin("organizationUnits as ou", "ou.id", "uoou.organizationUnitId")
            .where("uo.organizationId", organizationId)
            .whereNull("uo.deletedAt")
            .whereNull("uoou.deletedAt")
            .whereNull("ou.deletedAt");
        if (organizationUnitIds) {
            qb.whereIn("uoou.organizationUnitId", organizationUnitIds);
            for (const orgUnitId of organizationUnitIds) {
                qb.orWhere("ou.ancestorIds", "like", "%" + orgUnitId + "%");
            }
        }
        let rows = await qb.select("ou.id", "uo.userId");
        let result = [];
        for (const row of rows) {
            let unit = result.find((r) => r.organizationUnitId == row.id);
            if (unit) {
                unit.userIds.push(row.userId);
            }
            else {
                result.push({
                    organizationUnitId: row.id,
                    userIds: [row.userId],
                });
            }
        }
        return Promise.resolve(result);
    }
    async listOrganizationUnitsOfUser(organizationId, userId) {
        let userOrganizationUnits = await this.dbClient
            .withSchema(organizationId)
            .from("users as u")
            .innerJoin("userOrganizations as uo", "uo.userId", "u.id")
            .innerJoin("userOrganizationOrganizationUnits as uoou", "uoou.userOrganizationId", "uo.id")
            .innerJoin("organizationUnits as ou", (join) => {
            join.on(this.dbClient.raw(`ou."ancestorIds" like '%' || uoou."organizationUnitId" || '%'`)).orOn("ou.id", "uoou.organizationUnitId");
        })
            .whereNull("u.deletedAt")
            .whereNull("uoou.deletedAt")
            .whereNull("uo.deletedAt")
            .where("uo.organizationId", organizationId)
            .where("u.id", userId)
            .select("ou.id");
        return userOrganizationUnits.map((u) => u.id);
    }
    async applyUserSelectionSessionWithOrganizationUnitRoleAndPrivileges(organizationId, requestUserId, params, trx) {
        const session = await dal_manager_1.dbManager.accessUser.validateUserSelectionSession(organizationId, requestUserId, params.sessionId, trx);
        if (session.type !== dal_constants_1.DalConstants.UserSelectionSessionType.AddUsersToOrganizationUnit) {
            throw (0, api_error_1.generateBadRequestApiError)({ message: "Invalid Session Type" });
        }
        const { rows } = await trx.query(`SELECT 
			COUNT(CASE WHEN action = 2 THEN 1 END)::integer AS added, 
			COUNT(CASE WHEN action = 3 THEN 1 END)::integer AS removed FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userSelectionSessionActions}"
		WHERE "sessionId" = $1
		GROUP BY "sessionId"`, [params.sessionId]);
        const addedUserCount = rows[0].added;
        const removedUserCount = rows[0].removed;
        const addUserOrganizationUnitWithSession = async (trx) => {
            const now = new Date();
            if (addedUserCount) {
                await trx.query(`INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}" 
					(id, "createdAt", "updatedAt", "deletedAt", "userOrganizationId", "organizationUnitId", "roleId", "extraPermissions") 
					SELECT uuid_generate_v4(), now(), now(), null, uo.id, $1, $2, null
					FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userSelectionSessionActions}" ussa
					INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo
					ON ussa."userId" = uo."userId"
					WHERE "sessionId" = $3 AND action = $4
					ON CONFLICT ("userOrganizationId", "organizationUnitId") WHERE "deletedAt" IS NULL
					DO UPDATE SET 
						"roleId" = EXCLUDED."roleId", 
						"extraPermissions" = EXCLUDED."extraPermissions";`, [session.relatedItemId, params.roleId, params.sessionId, dal_constants_1.DalConstants.UserSelectionSessionAction.Added]);
            }
            if (removedUserCount) {
                const sessionDeleteActions = (await trx.query(`
						SELECT id, "userId" 
						FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userSelectionSessionActions}"
						WHERE "sessionId" = $1 AND "action" = $2`, [session.id, dal_constants_1.DalConstants.UserSelectionSessionAction.Removed])).rows;
                const relatedUnit = (await trx.query(`
						SELECT id, "managerUserId", "managerUserOrganizationId" 
						FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}"
						WHERE id = $1`, [session.relatedItemId])).rows[0];
                for (const action of sessionDeleteActions) {
                    if (relatedUnit?.managerUserId === action.userId) {
                        await trx.query(`
							UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}"
							SET "managerUserId" = NULL,
								"managerUserOrganizationId" = NULL
							WHERE id = $1`, [session.relatedItemId]);
                    }
                }
                await trx.query(`UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}" uoou
					 SET "deletedAt" = $1,
						  "updatedAt" = $1
					 FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo
					 INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userSelectionSessionActions}" ussa
						 ON ussa."userId" = uo."userId"
					 WHERE uoou."deletedAt" IS NULL
						AND uoou."userOrganizationId" = uo.id
						AND ussa."sessionId" = $2 
						AND ussa.action = $3
						AND uoou."organizationUnitId" = $4`, [now, params.sessionId, dal_constants_1.DalConstants.UserSelectionSessionAction.Removed, session.relatedItemId]);
            }
        };
        if (addedUserCount || removedUserCount) {
            await addUserOrganizationUnitWithSession(trx);
        }
        if (params.acpAuthorize && addedUserCount > 0) {
            const scope = params.acpAuthorize.scope;
            await Promise.all(params.acpAuthorize.accessRights.map(async ({ id: acpId, ...accessRights }) => {
                const { rows } = await trx.query(`SELECT ussa."userId", uar.id as "accessRightId" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userSelectionSessionActions}" AS ussa
						 LEFT JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userAccessRights}" uar 
							ON uar."userId" = ussa."userId" AND uar."deletedAt" IS NULL AND uar."accessControlPointId" = $1
						 WHERE "ussa"."sessionId" = $2 AND "ussa"."action" = $3 `, [acpId, params.sessionId, dal_constants_1.DalConstants.UserSelectionSessionAction.Added]);
                if (scope === dal_constants_1.DalConstants.UserSelectionSessionAccessControlPointRightScope.InsertNewExpandCurrent ||
                    scope === dal_constants_1.DalConstants.UserSelectionSessionAccessControlPointRightScope.InsertNewOverrideCurrent) {
                    let query = `UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userAccessRights}"`;
                    if (scope === dal_constants_1.DalConstants.UserSelectionSessionAccessControlPointRightScope.InsertNewExpandCurrent) {
                        query += `
								SET "remoteAccess" = "remoteAccess" OR $1, 
									"read" = "read" OR $2, 
									"access" = "access" OR $3, 
									"config" = "config" OR $4, 
									"grant" = "grant" OR $5, 
									"snapshot" = "snapshot" OR $6`;
                    }
                    else {
                        query += `
								SET "remoteAccess" = $1, 
									"read" = $2, 
									"access" = $3, 
									"config" = $4, 
									"grant" = $5, 
									"snapshot" = $6`;
                    }
                    query += `
								WHERE "userId" = ANY($7)
								AND "deletedAt" IS NULL 
								AND "accessControlPointId" = $8`;
                    await trx.query(query, [
                        accessRights.remoteAccess,
                        accessRights.read,
                        accessRights.access,
                        accessRights.config,
                        accessRights.grant,
                        accessRights.snapshot,
                        rows.filter((row) => row.accessRightId).map((row) => row.userId),
                        acpId,
                    ]);
                }
                const bindings = [accessRights.remoteAccess, accessRights.read, accessRights.access, accessRights.config, accessRights.grant, accessRights.snapshot];
                let bindingIndex = 7;
                const insertQuery = [];
                const usersThatDontHaveAccessRightForAcp = rows.filter((row) => !row.accessRightId);
                if (usersThatDontHaveAccessRightForAcp.length > 0) {
                    for (const user of usersThatDontHaveAccessRightForAcp) {
                        app_logs_1.logger.debug(user.userId);
                        app_logs_1.logger.debug("acpId: " + acpId);
                        bindings.push(user.userId, acpId);
                        insertQuery.push(`(uuid_generate_v4(), now(), now(), $${bindingIndex++}, $${bindingIndex++}, $1, $2, $3, $4, $5, $6)`);
                    }
                    await trx.query(`INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userAccessRights}"
							(id, "createdAt", "updatedAt", "userId", "accessControlPointId", "remoteAccess", "read", "access", "config", "grant", "snapshot")
							VALUES ${insertQuery.join(",")}`, bindings);
                }
                const countOfGranters = (await trx.query(`SELECT COUNT(*) FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userAccessRights}"
							 WHERE "deletedAt" IS NULL 
							 AND "read" = TRUE 
							 AND "grant" = TRUE 
							 AND "accessControlPointId" = $1`, [acpId])).rows[0].count;
                if (countOfGranters < 1) {
                    throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.INVALID_ACCESS_CONTROL_POINT_RIGHTS, "ERRORS.GENERAL.LASTUSERWITHGRANT", null, true);
                }
                const countOfConfigers = (await trx.query(`SELECT COUNT(*) FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userAccessRights}"
							 WHERE "deletedAt" IS NULL 
							 AND "read" = TRUE 
							 AND "config" = TRUE 
							 AND "accessControlPointId" = $1`, [acpId])).rows[0].count;
                if (countOfConfigers < 1) {
                    throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.INVALID_ACCESS_CONTROL_POINT_RIGHTS, "ERRORS.GENERAL.LASTUSERWITHCONFIG", null, true);
                }
            }));
        }
        if (removedUserCount > 0 && params.removeDefaultAccessRights) {
            const defaultAccessControlPointIds = (await dal_manager_1.dbManager.accessOrganizationUnit.getOrganizationUnitDetailed(organizationId, session.relatedItemId, params.hasOrganizationWideReadDetailedOrganizationUnit, requestUserId)).defaultAccessControlPoints.map((acp) => acp.id);
            const { rows: removedUserRows } = await trx.query(`SELECT "userId" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userSelectionSessionActions}"
				 WHERE "sessionId" = $1 AND "action" = $2 `, [params.sessionId, dal_constants_1.DalConstants.UserSelectionSessionAction.Removed]);
            await trx.query(`UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userAccessRights}"
				SET "updatedAt" = now(),
					"deletedAt" = now()
				WHERE "userId" = ANY($1)
				AND "accessControlPointId" = ANY($2)
				AND "deletedAt" IS NULL`, [removedUserRows.map((user) => user.userId), defaultAccessControlPointIds]);
        }
        await trx.query(`DELETE FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userSelectionSessions}"
			 WHERE "id" = $1`, [session.id]);
    }
    async getAncestorAndChildOrganizationUnitIds(organizationId, userId, trx) {
        let requestedUserOrganizationUnitBasicInfo = [];
        let organizationUnitBasicsOfSuccessors = [];
        let organizationUnitBasicsOfAncestors = [];
        requestedUserOrganizationUnitBasicInfo = await dal_manager_1.dbManager.accessOrganizationUnit.getOrganizationUnitBasicOfUser(organizationId, userId, trx);
        for (let orgUnit of requestedUserOrganizationUnitBasicInfo) {
            let successors = await this.getAllSuccessorOrganizationUnitsBasic(organizationId, [orgUnit.id], trx);
            let ancestors = await dal_manager_1.dbManager.accessOrganizationUnit.getAncestorsOfOrganizationUnit(organizationId, orgUnit.id, trx);
            organizationUnitBasicsOfSuccessors = organizationUnitBasicsOfSuccessors.concat(successors);
            organizationUnitBasicsOfAncestors = organizationUnitBasicsOfAncestors.concat(ancestors);
        }
        return [...requestedUserOrganizationUnitBasicInfo, ...organizationUnitBasicsOfSuccessors, ...organizationUnitBasicsOfAncestors].map((info) => info.id);
    }
    async getAllSuccessorOrganizationUnitsBasic(organizationId, organizationUnitIds, trx) {
        const successorsDbResult = await trx.query(`
			SELECT ou."id", ou."name", ou."typeId", ou."parentId", ou."ancestorIds", ou."shortCode", ou."managerUserId", oud.value AS "typeDefinition"
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" ou
			INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUserDefinedListValues}" oud ON ou."typeId" = oud.id
			WHERE EXISTS (
      			SELECT 1 
      			FROM UNNEST(STRING_TO_ARRAY(ou."ancestorIds", ',')::UUID[]) AS ancestorId
      			WHERE ancestorId = ANY ($1)
   			)
			AND ou."deletedAt" IS NULL
		`, [organizationUnitIds]);
        if (successorsDbResult.rowCount === 0) {
            return [];
        }
        else {
            return successorsDbResult.rows;
        }
    }
    async getDefaultAccessControlPointRightsOfOrganizationUnits(organizationId, organizationUnitIds, trx) {
        const unitDefaultAccessControlPointRights = await trx.query(`
				SELECT "organizationUnitId", 
					ARRAY_AGG(
						JSONB_BUILD_OBJECT(
							'accessControlPointId', "accessControlPointId",
							'defaultPrivileges', "defaultPrivileges"
						)
					) AS "defaultAccessControlPointRights"
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnitDefaultAccessControlPoints}"
				WHERE "organizationUnitId" = ANY($1::UUID[])
					AND "deletedAt" IS NULL
				GROUP BY "organizationUnitId"
			`, [organizationUnitIds]);
        if (unitDefaultAccessControlPointRights.rowCount === 0) {
            return [];
        }
        else {
            return unitDefaultAccessControlPointRights.rows;
        }
    }
}
exports.PSQLDalAccessOrganizationUnit = PSQLDalAccessOrganizationUnit;
