"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getOrganizationUserFilterAccessRightActivationStatus = exports.getOrganizationIntegrationCredsById = exports.getOrganizationSecretById = exports.PSQLDalAccessOrganization = void 0;
const nodemailer_1 = __importDefault(require("nodemailer"));
const uuid_1 = __importDefault(require("uuid"));
const app_enums_1 = require("../../../app.enums");
const app_logs_1 = require("../../../app.logs");
const restapi_1 = require("../../../lib/es/models/restapi");
const dal_constants_1 = require("../../dal.constants");
const dal_manager_1 = require("../../dal.manager");
const dal_utils_1 = require("../../dal.utils");
const dal_db_armon_schema_1 = require("../../db/armon/dal.db.armon.schema");
const predefined_roles_1 = require("../../db/predefined/predefined.roles");
const dal_access_error_1 = require("../dal.access.error");
const dal_access_rdb_organization_1 = require("../rdb/dal.access.rdb.organization");
class PSQLDalAccessOrganization extends dal_access_rdb_organization_1.RDBDalAccessOrganization {
    constructor(knex, pgPool) {
        super(knex, pgPool);
    }
    async getOrganizationMobileCreditSettings(organizationId) {
        let mifareCredential;
        const result = await this.dbClient
            .withSchema(organizationId)
            .from(dal_db_armon_schema_1.ArmonSchema.tableNames.organizations)
            .where("id", organizationId)
            .whereNull("deletedAt")
            .first("mobileCreditApplicationSettings", "credentialTypes");
        if (result.credentialTypes) {
            mifareCredential = result.credentialTypes.find((elem) => elem.type === dal_constants_1.DalConstants.CredentialType.MiFare);
        }
        if (!result) {
            (0, dal_access_error_1.throwDbAccessNotFoundError)("organization id not found");
        }
        const mobileCreditSettings = {
            checkInterval: result.mobileCreditApplicationSettings.checkInterval,
            factor: result.mobileCreditApplicationSettings.factor,
            dataConverter: mifareCredential ? mifareCredential.dataConverter : undefined,
        };
        return Promise.resolve(mobileCreditSettings);
    }
    async getOrganizationBasic(organizationId) {
        return await this.dbClient
            .withSchema(organizationId)
            .from(dal_db_armon_schema_1.ArmonSchema.tableNames.organizations)
            .where("id", organizationId)
            .whereNull("deletedAt")
            .first("id", "name", "publicKey", "privateKey");
    }
    async disableModulesExcept(organizationId, modules, trx) {
        let notPassiveModules = await trx
            .withSchema(organizationId)
            .from(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationModules)
            .select("id", "module")
            .where("organizationId", organizationId)
            .whereNot("status", dal_constants_1.DalConstants.ArmonApplicationModuleStatus.Passive);
        let modulesShouldBeDisabled = notPassiveModules.filter((p) => modules.indexOf(p.module) < 0);
        await trx
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationModules)
            .update({
            status: dal_constants_1.DalConstants.ArmonApplicationModuleStatus.Passive,
            disableUtc: new Date(),
        })
            .whereIn("id", modulesShouldBeDisabled.map((m) => m.id))
            .whereNotIn("module", modules);
        return Promise.resolve(modulesShouldBeDisabled.map((m) => m.module));
    }
    async enableModules(organizationId, modules, trx) {
        let result = [];
        let currentModules = await trx
            .withSchema(organizationId)
            .table(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationModules)
            .where("organizationId", organizationId)
            .whereIn("module", modules.map((m) => m.type))
            .whereWrapped((w) => {
            w = w
                .where("status", dal_constants_1.DalConstants.ArmonApplicationModuleStatus.Active)
                .orWhere("status", dal_constants_1.DalConstants.ArmonApplicationModuleStatus.CredentialCountExceededLicence)
                .orWhere("status", dal_constants_1.DalConstants.ArmonApplicationModuleStatus.Passive);
        })
            .select("id", "status", "module");
        let newModules = modules.filter((c) => !currentModules.some((m) => m.module == c.type));
        for (const newModule of newModules) {
            await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationModules).insert({
                id: newModule.id,
                organizationId: organizationId,
                status: dal_constants_1.DalConstants.ArmonApplicationModuleStatus.Active,
                usageLimit: newModule.maxCredentialCount,
                enableUtc: new Date(),
                module: newModule.type,
                exceedPercentage: 0,
                licenseControlPolicy: dal_constants_1.DalConstants.LicenseControlPolicy.Instant,
            });
        }
        result = newModules.map((m) => m.type);
        for (const existingModule of modules) {
            let currentModule = currentModules.find((c) => c.module === existingModule.type);
            if (!currentModule) {
                continue;
            }
            let currentUsage = 0;
            switch (existingModule.type) {
                case dal_constants_1.DalConstants.ArmonApplicationModule.PACS:
                    currentUsage = await trx
                        .withSchema(organizationId)
                        .from(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations + " as uo")
                        .innerJoin(dal_db_armon_schema_1.ArmonSchema.tableNames.userWorkPlans + " as uwp", "uwp.userId", "uo.userId")
                        .whereNull("uo.deletedAt")
                        .where("uo.isDisabled", false)
                        .countDistinct("uwp.userId")
                        .then((rows) => {
                        return rows[0].count;
                    });
                    break;
                case dal_constants_1.DalConstants.ArmonApplicationModule.SocialArea:
                    currentUsage = await trx
                        .withSchema(organizationId)
                        .from(dal_db_armon_schema_1.ArmonSchema.tableNames.userRegionTicketUnits)
                        .where("organizationId", organizationId)
                        .count()
                        .then((rows) => {
                        return rows[0].count;
                    });
                    break;
                case dal_constants_1.DalConstants.ArmonApplicationModule.Visitor:
                    break;
                case dal_constants_1.DalConstants.ArmonApplicationModule.Base:
                    currentUsage = await trx
                        .withSchema(organizationId)
                        .from(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles + " as uop")
                        .innerJoin(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations + " as uo", "uo.id", "uop.userOrganizationId")
                        .whereNot("uo.isDisabled", true)
                        .whereNull("uo.deletedAt")
                        .count()
                        .then((rows) => {
                        return rows[0].count;
                    });
                    break;
                default:
                    app_logs_1.logger.error("Undefined module " + module);
                    break;
            }
            let status = existingModule.maxCredentialCount && currentUsage - 1 > existingModule.maxCredentialCount
                ? dal_constants_1.DalConstants.ArmonApplicationModuleStatus.CredentialCountExceededLicence
                : dal_constants_1.DalConstants.ArmonApplicationModuleStatus.Active;
            if (currentModule.status !== dal_constants_1.DalConstants.ArmonApplicationModuleStatus.Active && status === dal_constants_1.DalConstants.ArmonApplicationModuleStatus.Active) {
                result.push(existingModule.type);
            }
            await trx
                .withSchema(organizationId)
                .table(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationModules)
                .update({
                status: status,
                usageLimit: existingModule.maxCredentialCount,
            })
                .where("module", existingModule.type)
                .where("organizationId", organizationId);
        }
        return Promise.resolve(result);
    }
    async getScheduledJobParameters(organizationId) {
        let result = null;
        await this.dbClient
            .withSchema(organizationId)
            .table("organizations")
            .where("id", organizationId)
            .select("scheduledJobParameters")
            .first()
            .then((row) => (result = row.scheduledJobParameters));
        return Promise.resolve(result);
    }
    async checkDeputySelectionEnabled(organizationId) {
        let result = false;
        await this.dbClient
            .withSchema(organizationId)
            .table("organizationPACSModuleSettings")
            .where("organizationId", organizationId)
            .select("deputySelectionEnabled")
            .first()
            .then((row) => (result = row.deputySelectionEnabled));
        return Promise.resolve(result);
    }
    async getOrganizationAnnualPPermissionUpdateIntervals(organizationId, trx) {
        let intervals = [];
        let qb = this.dbClient.withSchema(organizationId).table("organizations").where("id", organizationId);
        if (trx)
            qb.transacting(trx);
        await qb
            .select("annualPPermissionUpdateIntervals")
            .first()
            .then((row) => {
            if (row && row.annualPPermissionUpdateIntervals) {
                intervals = JSON.parse(row.annualPPermissionUpdateIntervals);
            }
        });
        return Promise.resolve(intervals);
    }
    async listOrganizationPg(trx) {
        const trxx = trx ?? this._pgPool;
        const { rowCount, rows } = await trxx.query(`SELECT id FROM public."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationList}"`);
        return rowCount > 0 ? rows.map((row) => row.id) : null;
    }
    async listOrganizations(trx) {
        let transactionScope = async (trx) => {
            const dbResult = (await trx.query(`
				SELECT "id"
				FROM public."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationList}"
			`)).rows.map((r) => r.id);
            return dbResult;
        };
        if (trx) {
            return transactionScope(trx);
        }
        else {
            return dal_manager_1.dbManager.systemTransaction(transactionScope);
        }
    }
    async getOrganization(organizationId, trx) {
        let qb = this.dbClient.withSchema(organizationId).from(dal_db_armon_schema_1.ArmonSchema.tableNames.organizations).whereNull("deletedAt");
        if (trx)
            qb.transacting(trx);
        let organization = await qb.first();
        if (organization) {
            return Promise.resolve({
                id: organization.id,
                annualPPermissionUpdateIntervals: organization.annualPPermissionUpdateIntervals,
                availableCredentialTypes: organization.availableCredentialTypes,
                credentialTypes: organization.credentialTypes,
                emailAddresses: organization.emailAddresses,
                emergencyState: organization.emergencyState,
                hasEmergencyManagement: organization.hasEmergencyManagement,
                name: organization.name,
                privateKey: organization.privateKey,
                publicKey: organization.publicKey,
                scheduledJobParameters: organization.scheduledJobParameters,
                smtpOptions: organization.smtpOptions,
                uniqueIdFieldCaption: organization.uniqueIdFieldCaption,
                userOrganizationProfileSettings: organization.userOrganizationProfileSettings,
                usernamePolicy: organization.usernamePolicy,
                settings: organization.settings,
            });
        }
        return Promise.resolve(null);
    }
    async getOrganizationSummary(organizationId, trx) {
        let result = {
            passiveUserCount: 0,
            activeCredentialCount: 0,
            activeUserCount: 0,
            devices: {
                mainControlPanelCount: 0,
                extensionControlPanelCount: 0,
            },
            modules: [],
            multiFactorAuthenticationSettings: { option: app_enums_1.enums.MultifactorAuthenticationOption.DISABLE },
        };
        const organizationSettings = await dal_manager_1.dbManager.accessOrganization.getOrganizationSettings({ organizationId, trx });
        if (organizationSettings)
            result.multiFactorAuthenticationSettings = organizationSettings.multiFactorAuthenticationSettings;
        let users = (await trx.query(`
			SELECT uo."id", uo."isDisabled" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" AS uo
			INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" AS uop
				ON uop."userOrganizationId" = uo."id" AND uop."deletedAt" IS NULL
 			WHERE uo."deletedAt" IS NULL
		`, [])).rows;
        result.activeUserCount = users.filter((u) => !u.isDisabled).length - 1;
        result.passiveUserCount = users.filter((u) => u.isDisabled).length - 1;
        result.activeCredentialCount = await this.getOrganizationActiveCredentialCount(organizationId, trx);
        const modules = await trx.query(`
			SELECT module, status, "enableUtc", "disableUtc", "usageLimit"
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationModules}"
		`);
        for (const m of modules.rows) {
            const mResult = {
                currentUsage: 0,
                enabled: m.status === dal_constants_1.DalConstants.ArmonApplicationModuleStatus.Active,
                maxCount: m.usageLimit,
                module: m.module,
                disableUtc: m.disableUtc,
                enableUtc: m.enableUtc,
            };
            switch (m.module) {
                case dal_constants_1.DalConstants.ArmonApplicationModule.Base:
                    mResult.currentUsage = result.activeUserCount;
                    break;
                case dal_constants_1.DalConstants.ArmonApplicationModule.PACS:
                    mResult.currentUsage = await this.getOrganizationActiveUsersAssignedToWorkplanCount(organizationId, trx);
                    break;
                case dal_constants_1.DalConstants.ArmonApplicationModule.SocialArea:
                    mResult.currentUsage = await this.getOrganizationActiveUsersAssignedToRegionTicketCount(organizationId, trx);
                    break;
                case dal_constants_1.DalConstants.ArmonApplicationModule.Visitor:
                    mResult.currentUsage = null;
                    mResult.maxCount = null;
                    break;
                default:
                    app_logs_1.logger.error("Undefined organization module type!");
                    break;
            }
            result.modules.push(mResult);
        }
        return Promise.resolve(result);
    }
    async getOrganizationActiveCredentialCount(organizationId, trx) {
        let result = (await trx.query(`
			SELECT COUNT (*) FILTER (WHERE NOT uoc."type" = ANY($2::INTEGER[])) + 
			COUNT (DISTINCT (uoc."userId")) FILTER (WHERE uoc."type" = ANY($2::INTEGER[])) AS c 
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials}" AS uoc
			INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" AS uo
				ON uo."userId" = uoc."userId"
			INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.roles}" AS r
				ON r."id" = uo."roleId"
			WHERE uoc."deletedAt" IS NULL
				AND ("expiresOn" IS NULL OR "expiresOn" > now())
				AND NOT r."typeId" = ANY($1::UUID[])
				AND uo."deletedAt" IS NULL
				AND r."deletedAt" IS NULL
				AND uo."isDisabled" = FALSE
		`, [[predefined_roles_1.PredefinedRoles.OrganizationUnitVisitor.id, predefined_roles_1.PredefinedRoles.OrganizationVisitor.id], dal_constants_1.DalConstants.mobileGeneratedCredentialTypes])).rows[0];
        return parseInt(result.c);
    }
    async getOrganizationActiveUsersAssignedToRegionTicketCount(organizationId, trx) {
        let result = (await trx.query(`
			SELECT COUNT( DISTINCT(urtu."userId")) AS c
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userRegionTicketUnits}" AS urtu
			INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" AS uo
				ON uo."userId" = urtu."userId"
			WHERE uo."deletedAt" IS NULL AND
				uo."isDisabled" != TRUE;
		`)).rows[0];
        return parseInt(result.c);
    }
    async getOrganizationActiveUsersAssignedToWorkplanCount(organizationId, trx) {
        let result = (await trx.query(`
			SELECT COUNT( DISTINCT(uwp."userId")) AS c
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userWorkPlans}" AS uwp
			INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" AS uo
				ON uo."userId" = uwp."userId"
			WHERE uo."deletedAt" IS NULL AND
				uo."isDisabled" != TRUE;
		`)).rows[0];
        return parseInt(result.c);
    }
    async listOrganizationModules(organizationId, trx) {
        let result;
        let modules = (await (trx ?? this._pgPool).query(`
			SELECT "module", "status", "enableUtc" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationModules}"
		`)).rows;
        result = modules.map((r) => {
            return {
                module: r.module,
                enabled: r.status == dal_constants_1.DalConstants.ArmonApplicationModuleStatus.Active,
                enableUtc: r.enableUtc,
            };
        });
        return Promise.resolve(result);
    }
    async checkEmergencyManagementEnabled(organizationId) {
        let dbResult = await this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.organizations).where("id", organizationId).first("hasEmergencyManagement");
        return Promise.resolve(dbResult && dbResult.hasEmergencyManagement);
    }
    async getOrganizationForms(organizationId) {
        return await this.dbClient.withSchema(organizationId).from("organizationForms").where("organizationId", organizationId).select("type", "fields");
    }
    async getOrganizationUniqeIdFieldCaption(organizationId) {
        return await this.dbClient
            .withSchema(organizationId)
            .from("organizations")
            .where("id", organizationId)
            .first("uniqueIdFieldCaption")
            .then((r) => {
            return r.uniqueIdFieldCaption;
        });
    }
    async getTopFiveOrganizationsInformation() {
        return await dal_manager_1.dbManager.systemTransaction(async (trx) => {
            let orgIds = await this.listOrganizations(trx);
            let q = ``;
            for (let i = 0; i < orgIds.length && i < 5; i++) {
                q += `
				SELECT * FROM "${orgIds[i]}"."organizations" WHERE "deletedAt" IS NULL
				`;
                if (i !== orgIds.length - 1 && i < 4) {
                    q += `
					UNION ALL
					`;
                }
            }
            let organizations = (await trx.query(q)).rows;
            let items = [];
            for (const organization of organizations) {
                items.push({
                    id: organization.id,
                    name: organization.name,
                    emergencyState: dal_constants_1.DalConstants.EmergencyState[organization.emergencyState || dal_constants_1.DalConstants.EmergencyState.Normal],
                });
            }
            return items;
        });
    }
    async getAdministrators(organizationId) {
        let qb = this.dbClient
            .withSchema(organizationId)
            .from("organizations as o")
            .innerJoin("userOrganizations as uo", "uo.organizationId", "o.id")
            .leftJoin("users as u", "u.id", "uo.userId")
            .whereNull("uo.deletedAt")
            .whereNull("o.deletedAt")
            .where("uo.roleId", predefined_roles_1.PredefinedRoles.SystemAdministrator.id);
        if (organizationId) {
            qb.where("o.id", organizationId);
        }
        else {
            qb.limit(101);
        }
        let administrators = (await qb.select("o.id", "o.name as organizationName", "u.username", "u.id as userId"));
        let header = "Administrator users.";
        if (!organizationId && administrators.length > 100) {
            header += ` There are more than 100 administrator users. Please narrow down your query by specifying an organization id!`;
        }
        return Promise.resolve({
            header: header,
            items: administrators.map((admin) => {
                return {
                    organizationId: admin.id,
                    organizationName: admin.organizationName,
                    userId: admin.userId,
                    username: admin.username,
                };
            }),
        });
    }
    async createAdministrator(username, password, organizationId, name, surname, uniqueId) {
        let organizationCount = await this.dbClient
            .withSchema(organizationId)
            .from("organizations")
            .where("id", organizationId)
            .whereNull("deletedAt")
            .count()
            .then((c) => {
            return parseInt(c[0].count);
        });
        if (organizationCount === 0) {
            return Promise.resolve({
                error: "Organization is not found. Please check the organization Id that you specified.",
            });
        }
        let userCount = await this.dbClient
            .withSchema(organizationId)
            .from("users")
            .whereRaw("lower(username) = lower(?)", username)
            .whereNull("deletedAt")
            .count()
            .then((c) => {
            return parseInt(c[0].count);
        });
        if (userCount > 0) {
            return Promise.resolve({
                error: "The user is already exits. If you want to change password, please try the 'change_password' command.",
            });
        }
        let hash = (0, dal_utils_1.saltHashPassword)(password);
        await this.dbClient.transaction(async (trx) => {
            let opTime = new Date();
            let userId = uuid_1.default.v4();
            await trx.withSchema(organizationId).from("users").insert({
                id: userId,
                createdAt: opTime,
                updatedAt: opTime,
                accountEnabled: true,
                username: username,
                fromLDAP: false,
                salt: hash.salt,
                password: hash.passwordHash,
            });
            let userOrganizationId = uuid_1.default.v4();
            await trx.withSchema(organizationId).from("userOrganizations").insert({
                id: userOrganizationId,
                createdAt: opTime,
                updatedAt: opTime,
                userId: userId,
                organizationId: organizationId,
                roleId: predefined_roles_1.PredefinedRoles.SystemAdministrator.id,
                isDisabled: false,
            });
            let userOrganizationProfileId = uuid_1.default.v4();
            await trx.withSchema(organizationId).from("userOrganizationProfiles").insert({
                id: userOrganizationProfileId,
                createdAt: opTime,
                updatedAt: opTime,
                userId: userId,
                userOrganizationId: userOrganizationId,
                name: name,
                surname: surname,
                uniqueId: uniqueId,
            });
            await trx.withSchema(organizationId).from("userProfiles").insert({
                id: uuid_1.default.v4(),
                createdAt: opTime,
                updatedAt: opTime,
                userId: userId,
                name: name,
                surname: surname,
            });
        });
        return Promise.resolve({
            result: `Administrator account for ${username} is successfully created!`,
        });
    }
    async changeAdministratorPassword(username, password) {
        let userCount = await this.dbClient
            .withSchema("public")
            .from("userList")
            .whereRaw("lower(username) = lower(?)", username)
            .count()
            .then((c) => {
            return parseInt(c[0].count);
        });
        if (userCount === 0) {
            return Promise.resolve({
                error: "The user is not exits. Please check the username argument",
            });
        }
        let hash = (0, dal_utils_1.saltHashPassword)(password);
        let organizationIdsOfUser = (await dal_manager_1.dbManager.poolMain.query(`
            SELECT uom."organizationId" FROM public."userOrganizationMapping" as uom
                INNER JOIN public."userList" as ul ON ul.id = uom."userId"
                WHERE lower(ul.username) = lower($1)
        `, [username])).rows.map((r) => r.organizationId);
        for (const oid of organizationIdsOfUser) {
            await this.dbClient.withSchema(oid).from("users").whereRaw("lower(username) = lower(?)", username).update({
                updatedAt: new Date(),
                salt: hash.salt,
                password: hash.passwordHash,
            });
        }
        return Promise.resolve({
            result: `Password is successfully changed for ${username}!`,
        });
    }
    async getOrganizationUserProfileSettingsNew(organizationId) {
        const { rows, rowCount } = await this._pgPool.query(`
            SELECT "userOrganizationProfileSettings" 
            FROM "${organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.organizations}
            WHERE id = $1
        `, [organizationId]);
        if (rowCount === 0) {
            return dal_constants_1.DalConstants.defaultUserProfileSettings;
        }
        return rows[0].userProfileSettings;
    }
    async getOrganizationUserProfileSettingsPg(params) {
        const { rows } = await (params.trx ?? this._pgPool).query(`
			SELECT "userOrganizationProfileSettings" FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizations}"
			WHERE id = $1
		`, [params.organizationId]);
        return rows[0]?.userOrganizationProfileSettings ?? dal_constants_1.DalConstants.defaultUserProfileSettings;
    }
    async getOrganizationUserProfileSettings(organizationId) {
        let userProfileSettings = (await this.dbClient.withSchema(organizationId).from(dal_db_armon_schema_1.ArmonSchema.tableNames.organizations).where("id", organizationId).first("userOrganizationProfileSettings")).userOrganizationProfileSettings;
        if (!userProfileSettings) {
            userProfileSettings = dal_constants_1.DalConstants.defaultUserProfileSettings;
        }
        return Promise.resolve(userProfileSettings);
    }
    async getOrganizationUserDefinedList(params) {
        const { rows: listRows } = await params.trx.query(`SELECT * FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUserDefinedLists}"
	 		WHERE type = $1`, [params.type]);
        let list;
        if (listRows.length < 1) {
            const newList = {
                id: uuid_1.default.v4(),
                caption: "",
                organizationId: params.organizationId,
                type: params.type,
                valueType: params.type === dal_constants_1.DalConstants.OrganizationUserDefinedListType.OrganizationUnitType
                    ? dal_constants_1.DalConstants.OrganizationUserDefinedListDataType.JSON
                    : dal_constants_1.DalConstants.OrganizationUserDefinedListDataType.String,
            };
            switch (params.type) {
                case dal_constants_1.DalConstants.OrganizationUserDefinedListType.Brand:
                    {
                        newList.caption = "Marka";
                    }
                    break;
                case dal_constants_1.DalConstants.OrganizationUserDefinedListType.Model:
                    {
                        newList.caption = "Model";
                    }
                    break;
                case dal_constants_1.DalConstants.OrganizationUserDefinedListType.ColorCode:
                    {
                        newList.caption = "Renk";
                    }
                    break;
                case dal_constants_1.DalConstants.OrganizationUserDefinedListType.VehicleType:
                    {
                        newList.caption = "Araç Tipi";
                    }
                    break;
                case dal_constants_1.DalConstants.OrganizationUserDefinedListType.OrganizationUnitType:
                    {
                        newList.caption = "Organizasyon Birim Tipi";
                    }
                    break;
                default:
                    break;
            }
            await params.trx.query(`
				INSERT INTO "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUserDefinedLists}"
				(id, caption, "organizationId", type, "valueType") VALUES ($1, $2, $3, $4, $5)
			`, [newList.id, newList.caption, params.organizationId, params.type, newList.valueType]);
            list = {
                id: newList.id,
                caption: newList.caption,
                organizationId: params.organizationId,
                type: params.type,
                valueType: newList.valueType,
            };
        }
        else {
            list = listRows[0];
        }
        const qb = [];
        let q = `
		SELECT id, value FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUserDefinedListValues}"
		WHERE "userDefinedListId" = $1
		`;
        qb.push(list.id);
        if (params.relatedValueId) {
            q += ` AND "relatedValueId" = $2`;
            qb.push(params.relatedValueId);
        }
        const { rows: values } = await params.trx.query(q, qb);
        let result = {
            id: list.id,
            caption: list.caption,
            organizationId: params.organizationId,
            relatedListId: list.relatedListId,
            type: params.type,
            valueType: list.valueType,
            values: values.map((v) => {
                return {
                    id: v.id,
                    value: params.type === dal_constants_1.DalConstants.OrganizationUserDefinedListType.OrganizationUnitType ? JSON.parse(v.value) : v.value,
                };
            }),
        };
        return result;
    }
    async upsertOrganizationUserDefinedListValue(organizationId, userListId, value, relatedValueId, valueId) {
        let id = valueId ? valueId : uuid_1.default.v4();
        if (valueId) {
            await this.dbClient.withSchema(organizationId).from(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUserDefinedListValues).where("id", valueId).update({
                value: value,
                relatedValueId: relatedValueId,
            });
        }
        else {
            await this.dbClient.withSchema(organizationId).from(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUserDefinedListValues).insert({
                id: id,
                userDefinedListId: userListId,
                value: value,
                relatedValueId: relatedValueId,
            });
        }
        return Promise.resolve(id);
    }
    async deleteOrganizationUserDefinedListValue(params) {
        const { rows: userListRows, rowCount: userListRowCount } = await params.trx.query(`SELECT type FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUserDefinedLists}"
			WHERE id = $1`, [params.userDefinedListId]);
        let usage;
        if (userListRowCount > 0 &&
            userListRows[0].type !== dal_constants_1.DalConstants.OrganizationUserDefinedListType.VisitorRegistrationPoint &&
            userListRows[0].type !== dal_constants_1.DalConstants.OrganizationUserDefinedListType.OrganizationUnitType) {
            let q = `SELECT id FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials}"`;
            const qw = [`"deletedAt" IS NULL`];
            const qb = [];
            let qx = 1;
            switch (userListRows[0].type) {
                case dal_constants_1.DalConstants.OrganizationUserDefinedListType.Brand:
                    {
                        qw.push(`("extensionFields"->>'brand')::text ilike $${qx++}`);
                        qb.push("%" + params.userDefinedListValueId + "%");
                    }
                    break;
                case dal_constants_1.DalConstants.OrganizationUserDefinedListType.Model:
                    {
                        qw.push(`("extensionFields"->>'model')::text ilike $${qx++}`);
                        qb.push("%" + params.userDefinedListValueId + "%");
                    }
                    break;
                case dal_constants_1.DalConstants.OrganizationUserDefinedListType.ColorCode:
                    {
                        qw.push(`("extensionFields"->>'color')::text ilike $${qx++}`);
                        qb.push("%" + params.userDefinedListValueId + "%");
                    }
                    break;
                case dal_constants_1.DalConstants.OrganizationUserDefinedListType.VehicleType:
                    {
                        qw.push(`("extensionFields"->>'type')::text ilike $${qx++}`);
                        qb.push("%" + params.userDefinedListValueId + "%");
                    }
                    break;
                default:
                    break;
            }
            q = q + ` WHERE ${qw.join(" AND ")}`;
            usage = await params.trx.query(q, qb);
        }
        else if (userListRowCount > 0 && userListRows[0].type === dal_constants_1.DalConstants.OrganizationUserDefinedListType.OrganizationUnitType) {
            usage = await params.trx.query(`SELECT id FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}"
			WHERE "typeId" = $1`, [params.userDefinedListValueId]);
        }
        if (usage.rowCount > 0) {
            (0, dal_access_error_1.throwDbAccessConflictErrorTr)("ERRORS.GENERAL.VALUEINUSE");
        }
        await this.dbClient
            .withSchema(params.organizationId)
            .from(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUserDefinedListValues)
            .where("userDefinedListId", params.userDefinedListId)
            .where("id", params.userDefinedListValueId)
            .delete();
    }
    async getOrganizationLogoFile(organizationId) {
        let organization = await this.dbClient.withSchema(organizationId).from(dal_db_armon_schema_1.ArmonSchema.tableNames.organizations).where("id", organizationId).first();
        return Promise.resolve(organization ? organization.settings?.logo : null);
    }
    async getOrganizationLogoAndName(organizationId) {
        let organization = await this.dbClient.withSchema(organizationId).from(dal_db_armon_schema_1.ArmonSchema.tableNames.organizations).where("id", organizationId).first();
        return Promise.resolve(organization ? { logo: organization.logo, name: organization.name } : null);
    }
    async listAllOrganizationAuthMethods(trx, showHidden) {
        let result = [];
        const organizationsDbResult = await trx.query(`
				SELECT
					id,
					code,
					name,
					logo,
					alias
				FROM public."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationList}"
				ORDER BY name ASC
			`, []);
        const organizationAuthMethodsDbResult = await trx.query(`
				SELECT * 
				FROM "public"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationAuthenticationMethods}"
				WHERE "deletedAt" IS NULL
					${!showHidden ? ` AND ("isHidden" IS FALSE OR "isHidden" IS NULL)` : ``}
			`);
        for (const organization of organizationsDbResult.rows) {
            result.push({
                id: organization.id,
                code: organization.code,
                name: organization.name,
                logo: organization.logo,
                alias: organization.alias,
                authentications: organizationAuthMethodsDbResult.rows.filter((a) => a.organizationId == organization.id),
            });
        }
        return Promise.resolve(result);
    }
    async getAuthenticationMethodById(grantTypeId, trx) {
        const dbResult = await trx.query(`
				SELECT oam.*, 
					ol.id AS "orgId",
					ol.name AS "orgName",
					ol.code AS "orgCode",
					ol.logo AS "orgLogo",
					ol.alias AS "orgAlias"
				FROM 
				public."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationAuthenticationMethods}" oam
				LEFT JOIN public."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationList}" ol
					on ol.id = oam."organizationId"
				WHERE oam.id = $1
					AND oam."deletedAt" IS NULL
			`, [grantTypeId]);
        return dbResult.rows[0];
    }
    async searchOrganizationAuthByAliasOrName(alias, trx) {
        let result = [];
        let organizationDbResult = await trx.query(`
				SELECT o.code,
					o.logo,
					o.id,
					o.name,
					o.alias
				FROM public."organizationList" o
				WHERE o.alias ilike $1
					OR o.name ilike $1
			`, ["%" + alias + "%"]);
        if (organizationDbResult.rowCount) {
            const organization = organizationDbResult.rows[0];
            let authMethodsDbResult = await trx.query(`
					SELECT *
					FROM public."organizationAuthenticationMethods"
					WHERE "deletedAt" IS NULL
						AND "organizationId" = $1	
				`, [organization?.id]);
            for (const authMethod of authMethodsDbResult.rows) {
                if (authMethod.authenticationMethod === dal_constants_1.DalConstants.AuthenticationMethod.LDAP) {
                    delete authMethod.config;
                }
            }
            result.push({
                id: organization.id,
                code: organization.code,
                name: organization.name,
                logo: organization.logo,
                alias: organization.alias,
                authentications: authMethodsDbResult.rows,
            });
        }
        return result;
    }
    async getOrganizationSettings(params) {
        const { rows, rowCount } = await params.trx.query(`
            SELECT settings, "smsBalance" FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizations}"
            WHERE id = $1
        `, [params.organizationId]);
        rows[0].settings.notification["smsBalance"] = rows[0].smsBalance;
        if (!params.returnSmtpPassword && rows[0].settings.notification.smtpSettings?.config.auth?.pass) {
            delete rows[0].settings.notification.smtpSettings.config.auth.pass;
        }
        return rowCount ? rows[0].settings : null;
    }
    async getOrganizationSettingsIntegrations(params) {
        const { rows, rowCount } = await params.trx.query(`
            SELECT settings->'integrations' AS integrations
			FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizations}"
            WHERE id = $1
        	`, [params.organizationId]);
        return rowCount ? rows[0].integrations : null;
    }
    async setOrganizationAvailableMedium(params) {
        const organizationSettings = await this.getOrganizationSettings({ organizationId: params.organizationId, trx: params.trx, returnSmtpPassword: true });
        organizationSettings.notification.availableMediums = params.availableMediums;
        await params.trx.query(`
			UPDATE "${params.organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.organizations}
			SET settings = $2
			WHERE id = $1
			AND "deletedAt" IS NULL		
			`, [params.organizationId, organizationSettings]);
    }
    async setOrganizationSettings(params) {
        const organizationSettings = await this.getOrganizationSettings({ organizationId: params.organizationId, trx: params.trx, returnSmtpPassword: true });
        let smtpSettings = null;
        let smsSettings = null;
        const customNotificationSettings = {};
        if (params.settings.notification) {
            if (params.settings?.notification?.mediumSettings?.custom) {
                for (const customSetting of params.settings?.notification?.mediumSettings?.custom) {
                    customNotificationSettings[customSetting.type] = customSetting.mediums;
                }
            }
            if (params.settings.notification?.smtpSettings && params.settings.notification.smtpSettings.enabled) {
                smtpSettings = {
                    enabled: params.settings.notification.smtpSettings.enabled,
                    config: {
                        requireTLS: params.settings.notification.smtpSettings.config.requireTLS ?? true,
                        auth: {
                            user: params.settings.notification.smtpSettings.config.auth.user,
                            pass: null,
                        },
                        tls: {
                            rejectUnauthorized: false,
                        },
                        host: params.settings.notification.smtpSettings.config.host,
                        port: params.settings.notification.smtpSettings.config.port,
                        debug: false,
                        logger: false,
                        secure: params.settings.notification.smtpSettings.config.secure ?? false,
                    },
                    defaultFromAddress: params.settings.notification.smtpSettings.defaultFromAddress,
                };
                if (params.settings.notification?.smtpSettings?.config?.auth?.pass) {
                    smtpSettings.config.auth.pass = params.settings.notification.smtpSettings.config.auth.pass;
                }
                else {
                    smtpSettings.config.auth.pass = organizationSettings.notification.smtpSettings?.config?.auth?.pass;
                }
                const transporter = nodemailer_1.default.createTransport(smtpSettings.config);
                try {
                    const checkSmtp = await transporter.verify();
                    if (!checkSmtp) {
                        throw new Error("SMTP settings could not verified");
                    }
                    organizationSettings.notification.availableMediums.email = true;
                    organizationSettings.notification.mediumSettings.general.email = true;
                }
                catch (error) {
                    app_logs_1.logger.error(error.message);
                    (0, dal_access_error_1.throwDbAccessBadRequestErrorTr)("ERROR.SMTP.SMTPSETTINGSWRONG", null, false);
                }
                finally {
                    transporter.close();
                }
            }
            if (params.settings.notification?.smsSettings) {
                if (params.settings.notification?.smsSettings?.enabled) {
                    let smsConfig = params.settings.notification?.smsSettings.config;
                    organizationSettings.notification.availableMediums.sms = true;
                    organizationSettings.notification.mediumSettings.general.sms = true;
                    if (params.settings.notification?.smsSettings.type === restapi_1.SmsConfigType.Armon) {
                        smsSettings = {
                            type: restapi_1.SmsConfigType.Armon,
                            enabled: true,
                            config: { apiKey: process.env.ILETI_MERKEZI_API_KEY, privateKey: process.env.ILETI_MERKEZI_PRIVATE_KEY, header: "ARMON" },
                        };
                    }
                    else {
                        smsSettings = {
                            type: params.settings.notification?.smsSettings.type,
                            enabled: true,
                            config: smsConfig,
                        };
                    }
                    await params.trx.query(`
						UPDATE "${params.organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.notification}
						SET sms = true;

						UPDATE "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}"
						SET settings = jsonb_set(settings, '{notification,mediumSettings,general,sms}', 'true', true)
						WHERE settings -> 'notification' -> 'mediumSettings' -> 'general' ->> 'sms' IS NOT NULL;
						`);
                }
                else if (!params.settings.notification?.smsSettings?.enabled) {
                    organizationSettings.notification.availableMediums.sms = false;
                    organizationSettings.notification.mediumSettings.general.sms = false;
                    smsSettings = params.settings.notification?.smsSettings;
                    await params.trx.query(`
						UPDATE "${params.organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.notification}
						SET sms = false;

						UPDATE "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}"
						SET settings = jsonb_set(settings, '{notification,mediumSettings,general,sms}', 'false'::jsonb, true)
						WHERE settings -> 'notification' -> 'mediumSettings' -> 'general' ->> 'sms' IS NOT NULL;
						`);
                }
            }
        }
        let webRtcSettings = {
            twilio: organizationSettings.webRtc.twilio,
            iceServers: organizationSettings.webRtc.iceServers,
        };
        if (params.settings.webRtc) {
            if (params.settings.webRtc.twilio) {
                webRtcSettings.twilio = params.settings.webRtc.twilio;
                if (params.settings.webRtc.twilio.authToken) {
                    webRtcSettings.twilio.authToken = params.settings.webRtc.twilio.authToken;
                }
                else {
                    webRtcSettings.twilio.authToken = organizationSettings.webRtc.twilio.authToken;
                }
            }
            if (params.settings.webRtc.iceServers) {
                webRtcSettings.iceServers = params.settings.webRtc.iceServers;
                for (const iceServerSetting of webRtcSettings.iceServers) {
                    if (!iceServerSetting.credential) {
                        iceServerSetting.credential = organizationSettings.webRtc?.iceServers?.find((f) => f.urls === iceServerSetting.urls && f.username === iceServerSetting.username)?.credential;
                    }
                }
            }
        }
        const updatedOrganizationSettings = {
            locale: params.settings?.locale || organizationSettings.locale,
            logo: params.settings?.logo || organizationSettings.logo,
            userExpression: params.settings.userExpression || organizationSettings.userExpression,
            helpUrl: params.settings.helpUrl ?? organizationSettings.helpUrl,
            enableTriggerThrowErrorIfUserRoleChangeCauseTerminalHasNoWriteAdmin: organizationSettings.enableTriggerThrowErrorIfUserRoleChangeCauseTerminalHasNoWriteAdmin,
            notification: {
                enableSendingCreatedAccountEmail: organizationSettings.notification.enableSendingCreatedAccountEmail,
                availableMediums: organizationSettings.notification.availableMediums,
                mediumSettings: {
                    general: {
                        email: params.settings?.notification?.mediumSettings?.general?.email ?? organizationSettings.notification.mediumSettings.general.email,
                        web: params.settings?.notification?.mediumSettings?.general?.web ?? organizationSettings.notification.mediumSettings.general.web,
                        sms: params.settings?.notification?.mediumSettings?.general?.sms ?? organizationSettings.notification.mediumSettings.general.sms,
                        pushNotification: params.settings?.notification?.mediumSettings?.general?.pushNotification ?? organizationSettings.notification.mediumSettings.general.pushNotification,
                    },
                    custom: params.settings?.notification?.mediumSettings?.custom ? customNotificationSettings : organizationSettings.notification.mediumSettings.custom,
                },
                enabledTypes: params.settings?.notification?.enabledTypes || organizationSettings.notification.enabledTypes,
                smsSettings: smsSettings ?? organizationSettings.notification.smsSettings,
                smtpSettings: smtpSettings ?? organizationSettings.notification.smtpSettings,
            },
            webRtc: params.settings.webRtc ? webRtcSettings : organizationSettings.webRtc,
            passiveUserSettings: organizationSettings.passiveUserSettings
                ? organizationSettings.passiveUserSettings
                : {
                    anonymize: organizationSettings.passiveUserSettings?.anonymize || false,
                },
            checkMobileDeviceMismatchOnLogin: params.settings.checkMobileDeviceMismatchOnLogin ?? organizationSettings.checkMobileDeviceMismatchOnLogin,
            snapshotSettings: organizationSettings.snapshotSettings,
            multiFactorAuthenticationSettings: params.settings.hasOwnProperty("multiFactorAuthenticationSettings")
                ? params.settings.multiFactorAuthenticationSettings
                : organizationSettings.multiFactorAuthenticationSettings,
            integrations: params.settings?.integrations ?? organizationSettings.integrations,
        };
        if (params.settings.notification?.smtpSettings?.enabled === false) {
            delete updatedOrganizationSettings.notification.smtpSettings;
            updatedOrganizationSettings.notification.availableMediums.email = false;
            if (updatedOrganizationSettings?.notification?.mediumSettings) {
                Object.values(updatedOrganizationSettings.notification.mediumSettings).forEach((setting) => {
                    if (setting && typeof setting === "object" && "email" in setting) {
                        setting.email = false;
                    }
                });
            }
        }
        if (webRtcSettings?.twilio?.enabled) {
            try {
                const twilio = await Promise.resolve().then(() => __importStar(require("twilio")));
                const client = twilio.default(webRtcSettings.twilio.accountSid, webRtcSettings.twilio.authToken);
                const newToken = await client.tokens.create();
                const iceServers = newToken.iceServers.map((iceServer) => {
                    const { url, ...rest } = iceServer;
                    return rest;
                });
                updatedOrganizationSettings.webRtc.iceServers = iceServers;
            }
            catch (error) {
                if (error.message.includes("The requested resource")) {
                    (0, dal_access_error_1.throwProxyAuthRequiredErrorTr)("ERRORS.GENERAL.TWILIO_WEBRTC_ACCOUNT_WAS_NOT_FOUND");
                }
                else if (error.message.includes("accountSid must start with AC")) {
                    (0, dal_access_error_1.throwProxyAuthRequiredErrorTr)("ERRORS.GENERAL.TWILIO_WEBRTC_ACCOUNTSID_STARTS_WITH");
                }
                else if (error.message.includes("invalid username")) {
                    (0, dal_access_error_1.throwProxyAuthRequiredErrorTr)("ERRORS.GENERAL.TWILIO_WEBRTC_INVALID_USERNAME");
                }
                else if (error.message.includes("Authenticate")) {
                    (0, dal_access_error_1.throwProxyAuthRequiredErrorTr)("ERRORS.GENERAL.TWILIO_WEBRTC_AUTH_ERROR");
                }
                else {
                    throw error;
                }
            }
        }
        await params.trx.query(`
            UPDATE "${params.organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.organizations}
            SET settings = $2
            WHERE id = $1
            AND "deletedAt" IS NULL
            `, [params.organizationId, updatedOrganizationSettings]);
    }
    async removeOrganizationLogoFile(organizationId, trx) {
        await trx.query(`
            UPDATE "${organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.organizations}
            SET settings = settings || '{"logo":null}'::jsonb
            WHERE id = $1
            AND "deletedAt" IS NULL
            `, [organizationId]);
    }
    async getUniqueNotificationIdsOfOrganization(params) {
        const { rows, rowCount } = await params.trx.query(`
            SELECT id, type FROM "${params.organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.notification}
            WHERE type != ANY ($1::int[])
        `, [dal_constants_1.DalConstants.receiverFilterDeterminedWhenNotificationGenerated]);
        if (rowCount === 0) {
            return {};
        }
        const result = {};
        for (const row of rows) {
            result[row.type] = row.id;
        }
        return result;
    }
    async getOrganizationRedisPersistentData(params) {
        const { rows, rowCount } = await (params.trx ?? this._pgPool).query(`
			SELECT name as n FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizations}"
			WHERE id = $1
			AND "deletedAt" IS NULL
		`, [params.organizationId]);
        return rowCount > 0 ? rows[0] : null;
    }
    async getAllOrganizationRedisPersistentData(params) {
        const organizationIds = await dal_manager_1.dbManager.accessOrganization.listOrganizationPg(params.trx);
        const result = await Promise.all(organizationIds.map(async (organizationId) => {
            return (await (params.trx ?? this._pgPool).query(`SELECT id, name as n FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizations}"
					 WHERE id = $1
					 AND "deletedAt" IS NULL`, [organizationId])).rows[0];
        }));
        result;
    }
    async isCheckMobileDeviceMismatchOnLoginEnabled(organizationId, trx) {
        const dbResult = await trx.query(`
				SELECT "settings"
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizations}"
				WHERE id = $1
					AND "deletedAt" IS NULL
			`, [organizationId]);
        return (dbResult.rows[0]?.settings).checkMobileDeviceMismatchOnLogin ?? false;
    }
    async getOrganizationSnapshotSettings(params) {
        const { rows } = await (params.trx ?? this.pgPool).query(`SELECT 
				(settings->'snapshotSettings'->>'quality') AS "quality", 
				(settings->'snapshotSettings'->>'width') AS "width", 
				COALESCE((settings->'snapshotSettings'->>'discardSnapshot')::BOOLEAN, FALSE) AS "discardSnapshot"
			FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizations}"
			WHERE id = $1`, [params.organizationId]);
        return {
            quality: rows[0].quality ? parseInt(rows[0].quality) : 60,
            width: rows[0].width ? parseInt(rows[0].width) : undefined,
            discardSnapshot: rows[0].discardSnapshot,
        };
    }
    async getModuleLicenceSettings(organizationId, module, trx) {
        const moduleLimitDbResult = await trx.query(`
			SELECT "usageLimit", "exceedPercentage", "licenseControlPolicy", status
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationModules}"
			WHERE "module" = $1
			`, [module]);
        if (!moduleLimitDbResult.rowCount || moduleLimitDbResult.rows[0].status !== dal_constants_1.DalConstants.ArmonApplicationModuleStatus.Active) {
            return null;
        }
        return {
            exceedPercentage: moduleLimitDbResult.rows[0].exceedPercentage ?? 0,
            licenseControlPolicy: moduleLimitDbResult.rows[0].licenseControlPolicy ?? dal_constants_1.DalConstants.LicenseControlPolicy.Instant,
            usageLimit: moduleLimitDbResult.rows[0].usageLimit ?? 0,
        };
    }
    async baseModuleLicenceVacancyCheck(organizationId, licenceSettings, trx) {
        const activeUserCountDbResult = await trx.query(`
				SELECT COUNT(*) AS "activeUserCount"
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo
				INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" uop
					ON uo.id = uop."userOrganizationId"
				WHERE uo."isDisabled" IS FALSE
			`);
        if (activeUserCountDbResult.rows[0].activeUserCount - 1 < licenceSettings.usageLimit + (licenceSettings.usageLimit * licenceSettings.exceedPercentage) / 100) {
            return true;
        }
        else {
            return false;
        }
    }
}
exports.PSQLDalAccessOrganization = PSQLDalAccessOrganization;
async function getOrganizationSecretById(organizationId, secretId, trx) {
    const secretResult = await trx.query(`
		SELECT secret
		FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.secrets}"
		WHERE id = $1
	`, [secretId]);
    if (secretResult.rowCount) {
        return secretResult.rows[0].secret;
    }
}
exports.getOrganizationSecretById = getOrganizationSecretById;
async function getOrganizationIntegrationCredsById(organizationId, clientId, trx) {
    const clientResult = await trx.query(`
		SELECT *
		FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.integrationClients}"
		WHERE id = $1
	`, [clientId]);
    if (clientResult.rowCount) {
        return clientResult.rows[0];
    }
}
exports.getOrganizationIntegrationCredsById = getOrganizationIntegrationCredsById;
async function getOrganizationUserFilterAccessRightActivationStatus(organizationId, trx) {
    const userFilterAccessRightActivationDbResult = await trx.query(`
		SELECT "userFilterAccessRightActivation" 
		FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizations}"
		`);
    return userFilterAccessRightActivationDbResult.rows[0].userFilterAccessRightActivation ?? false;
}
exports.getOrganizationUserFilterAccessRightActivationStatus = getOrganizationUserFilterAccessRightActivationStatus;
