"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PSQLDalAccessLogPacs = void 0;
const dal_manager_1 = require("../../dal.manager");
const dal_db_armon_schema_1 = require("../../db/armon/dal.db.armon.schema");
const uuid_1 = __importDefault(require("uuid"));
const app_logs_1 = require("../../../app.logs");
const dal_constants_1 = require("../../dal.constants");
const moment_1 = __importDefault(require("moment"));
const Cursor = require("pg-cursor");
class PSQLDalAccessLogPacs {
    constructor(knex, pgPool) {
        this._dbClient = knex;
        this._pgPool = pgPool;
    }
    get pgPool() {
        return this._pgPool;
    }
    get dbClient() {
        return this._dbClient;
    }
    convertToNoSqlDBType(data) {
        data = Object.assign({
            originalId: data.id,
            versionDateTime: new Date(Date.now()),
        }, data);
        data.id = uuid_1.default.v4();
        return data;
    }
    async convertToNoSqlDBTypeArray(dataList) {
        return Promise.resolve(dataList.map((item) => this.convertToNoSqlDBType(item)));
    }
    async getHistoryOfMonthlyReportOfOrganization(organizationId, year, month, pagination) {
        let result = {
            items: [],
            pagination: {
                total: null,
            },
        };
        let currentMonthReportId = await dal_manager_1.dbManager.accessPacs.getMonthlySummaryReportId(organizationId, year, month);
        if (!currentMonthReportId) {
            return Promise.resolve(result);
        }
        let query = `
            SELECT * FROM "${organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.MonthlySummaryHistory}
            WHERE "originalId" = $1 AND "organizationId" = $2
        `;
        let queryParams = [currentMonthReportId, organizationId];
        const total = (await this._pgPool.query("SELECT COUNT(*) FROM (" + query + ")q2", queryParams)).rows[0].count;
        query += `
            LIMIT $3 OFFSET $4
        `;
        queryParams.push(pagination.take, pagination.skip);
        const { rows } = await this._pgPool.query(query, queryParams);
        let items = rows.map((row) => {
            return {
                isFinalized: row.log.isFinalized,
                note: row.log.note,
                updatedBy: null,
                updateDateTime: row.actionUtc,
                updateById: row.log.updatedById,
            };
        });
        result.items = items.sort((x, y) => (x.updateDateTime < y.updateDateTime ? 1 : -1));
        let systemUser = await dal_manager_1.dbManager.accessUser.getPredefinedSystemUserInfoForOrganization(organizationId);
        for (let item of result.items) {
            if (item.updateById) {
                item.updatedBy = await dal_manager_1.dbManager.accessPacs.getBasicUserInfo(organizationId, item.updateById);
            }
            else {
                item.updatedBy = systemUser;
            }
        }
        result.pagination = {
            total: parseInt(total),
            skip: pagination.skip,
            take: pagination.take,
        };
        return Promise.resolve(result);
    }
    async getApprovementOfMonthlyReportOfOrganization(organizationId, year, month) {
        let approved;
        let currentMonthReportId = await dal_manager_1.dbManager.accessPacs.getMonthlySummaryReportId(organizationId, year, month);
        if (!currentMonthReportId) {
            return Promise.resolve(approved);
        }
        const { rows } = await this._pgPool.query(`
            SELECT log -> 'isFinalized' as approved FROM "${organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.MonthlySummaryHistory}
            WHERE "originalId" = $1 AND "organizationId" = $2
            ORDER BY "actionUtc" DESC
            FETCH FIRST ROW ONLY
        `, [currentMonthReportId, organizationId]);
        if (rows && rows.length > 0) {
            approved = rows[0];
        }
        return Promise.resolve(approved);
    }
    async insertMonthlySummaryHistory(organizationId, dataList) {
        try {
            let summaries = [];
            for (const log of dataList) {
                summaries.push({
                    id: uuid_1.default.v4(),
                    actionUtc: new Date(),
                    originalId: log.id,
                    organizationId: log.organizationId,
                    log: log,
                });
            }
            await this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.MonthlySummaryHistory).insert(summaries);
            return Promise.resolve(true);
        }
        catch (error) {
            app_logs_1.logger.error(error);
            return Promise.resolve(false);
        }
    }
    async listMonthlyReportHistoryOp(organizationId, userMonthlySummaryId, pagination) {
        let result = [];
        let query = `
        SELECT "originalId", "log" 
        FROM "${organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.UserMonthlySummaryHistory}
        WHERE "originalId" = $1 AND "organizationId" = $2
        `;
        let queryParams = [userMonthlySummaryId, organizationId];
        const total = (await this._pgPool.query("SELECT COUNT(*) FROM (" + query + ")q2", queryParams)).rows[0].count;
        query += `
            LIMIT $3 OFFSET $4
        `;
        queryParams.push(pagination.take, pagination.skip);
        const { rows } = await this._pgPool.query(query, queryParams);
        result = rows.map((log) => {
            return {
                originalId: log.originalId,
                missedWorkDuration: log.log.data.m,
                normalWorkDuration: log.log.data.n,
                extraWorkDuration: log.log.data.e,
                note: log.log.note,
                updatedById: log.log.updatedByUserId,
                updatedDateTime: log.log.updatedAt,
                systemNotes: log.log.systemNotes,
                extensionValues: log.log.extensionValues,
                updatedBy: null,
                isControlled: log.log.isChecked,
            };
        });
        let systemUser = await dal_manager_1.dbManager.accessUser.getPredefinedSystemUserInfoForOrganization(organizationId);
        for (let item of result) {
            if (item.updatedById) {
                item.updatedBy = await dal_manager_1.dbManager.accessPacs.getBasicUserInfo(organizationId, item.updatedById);
            }
            else {
                item.updatedBy = systemUser;
            }
        }
        return Promise.resolve({
            pagination: {
                take: pagination.take,
                skip: pagination.skip,
                total: parseInt(total),
            },
            items: result,
        });
    }
    async insertPPermissionHistoryContentLog(organizationId, ppermissionId, actionUserId, actionUtc, actionType, content) {
        let logItem = {
            u: actionUtc,
            oId: organizationId,
            pid: ppermissionId,
            auid: actionUserId,
            t: actionType,
            pc: {
                tid: content.typeId,
                sd: content.startDate,
                ed: content.endDate,
                s: content.status,
                ul: content.userAndDeputyList
                    ? content.userAndDeputyList.map((m) => {
                        return {
                            u: m.userId,
                            d: m.deputyUserId || null,
                        };
                    })
                    : null,
                al: content.approvementList
                    ? content.approvementList.map((m) => {
                        return {
                            ou: m.organizationUnitId,
                            u: m.approverUserId,
                            h: m.organizationUnitId == null,
                            o: m.order,
                            a: m.isApproved === true
                                ? dal_constants_1.DalConstants.PPermissionStatus.Approved
                                : m.isApproved === false
                                    ? dal_constants_1.DalConstants.PPermissionStatus.Rejected
                                    : dal_constants_1.DalConstants.PPermissionStatus.Waiting,
                            n: m.note || null,
                        };
                    })
                    : null,
            },
            nt: null,
        };
        let result = {
            id: logItem.pid,
            actionUtc: logItem.u,
            organizationId: logItem.oId,
            log: logItem,
        };
        await this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.PPermissionHistory).insert(result);
    }
    async insertPPermissionHistoryNotificationLog(organizationId, ppermissionId, actionUtc, notification, trx) {
        const trxx = trx ?? this._pgPool;
        let logItem = {
            u: actionUtc.toISOString(),
            oId: organizationId,
            pid: ppermissionId,
            auid: dal_constants_1.DalConstants.SystemUserId,
            t: dal_constants_1.DalConstants.PPermissionHistoryActionType.Notification,
            pc: null,
            nt: {
                e: notification.eventId,
                m: notification.mediums,
                u: notification.receiverUserId,
                t: notification.type,
            },
        };
        let result = {
            id: logItem.pid,
            actionUtc: logItem.u,
            organizationId: logItem.oId,
            log: logItem,
        };
        const { rows, rowCount } = await trxx.query(`SELECT * FROM "${organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.PPermissionHistory}
                                WHERE id = $1`, [ppermissionId]);
        if (rowCount > 0 && rows.find((r) => r.log && r.log.nt && r.log.nt.m)) {
            let row = rows.find((r) => r.log && r.log.nt && r.log.nt.m);
            if (row.log.nt.m.includes(logItem.nt.m)) {
                return;
            }
            row.log.nt.m.push(...logItem.nt.m);
            await trxx.query(`UPDATE "${organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.PPermissionHistory}
                SET log = $1
                WHERE id = $3
                AND "actionUtc" = $2`, [row.log, row.actionUtc, ppermissionId]);
        }
        else {
            await trxx.query(`INSERT INTO "${organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.PPermissionHistory}(
	        id, "organizationId", "actionUtc", log) VALUES
            ($1, $2, $3, $4)`, [result.id, result.organizationId, result.actionUtc, result.log]);
        }
    }
    async getHistoryOfPPermission(organizationId, ppermissionId) {
        let logs = (await this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.PPermissionHistory).where("id", ppermissionId).orderBy("actionUtc", "asc").select("log")).map((r) => r.log);
        let resultItems = logs.map((l) => {
            return {
                id: l.id,
                actionUtc: l.u,
                organizationId: l.oId,
                actionUserId: l.auid,
                ppermissionId: l.pid,
                actionType: l.t,
                content: l.pc
                    ? {
                        typeId: l.pc.tid,
                        startDate: l.pc.sd,
                        endDate: l.pc.ed,
                        status: l.pc.s,
                        userAndDeputyList: l.pc.ul
                            ? l.pc.ul.map((m) => {
                                return {
                                    userId: m.u,
                                    deputyUserId: m.d,
                                };
                            })
                            : null,
                        approvementList: l.pc.al
                            ? l.pc.al.map((m) => {
                                return {
                                    organizationUnitId: m.ou || null,
                                    isHr: m.h,
                                    order: m.o,
                                    approverUserId: m.u,
                                    note: m.n,
                                    isApproved: m.h,
                                };
                            })
                            : null,
                    }
                    : null,
                notification: l.nt
                    ? {
                        eventId: l.nt.e,
                        mediums: l.nt.m,
                        receiverUserId: l.nt.u,
                        type: l.nt.t,
                    }
                    : null,
            };
        });
        return Promise.resolve(resultItems);
    }
    async getAccessTimesWithDirection(params) {
        const trxx = params.trx ?? this._pgPool;
        let query = `
		SELECT id, "actionUtc", (log->>'d')::smallint as direction
		FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs}" al
		CROSS JOIN LATERAL jsonb_array_elements(CASE WHEN log->>'rg' IS NULL THEN '[{}]'::jsonb ELSE log->'rg' END) regions(rg)
		WHERE log->>'o' = $1
			AND "actionUtc" >= $2 AND "actionUtc" <= $3
			AND log->>'d' IS NOT NULL
			AND (log->>'d')::integer <> 0
			AND (log->>'s')::boolean = true
			AND ((log->>'ir')::boolean = false OR (log->>'ir')::boolean IS NULL)
			AND (rg->>'i' = $4 OR (rg->>'i' IS NULL AND log->>'a' = ANY($5)))`;
        const queryParams = [params.userId, params.start, params.end, params.regionId, params.accessControlPointIds];
        if (params.allowUnreliableCheckins === false) {
            query += `
			AND (log->>'lr' IS NULL OR (log->>'lr')::boolean = true)`;
        }
        query += `
		ORDER BY "actionUtc" ASC
		`;
        const { rows: accesses } = await trxx.query(query, queryParams);
        return accesses;
    }
    async getAccessTimesFirstLast(params) {
        const trxx = params.trx ?? this._pgPool;
        const result = {
            idMin: null,
            actionUtcMin: null,
            idMax: null,
            actionUtcMax: null,
        };
        let query = `
		SELECT id, "actionUtc"
		FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs}" al
		CROSS JOIN LATERAL jsonb_array_elements(CASE WHEN log->>'rg' IS NULL THEN '[{}]'::jsonb ELSE log->'rg' END) regions(rg)
		WHERE log->>'o' = $1
			AND "actionUtc" >= $2 AND "actionUtc" <= $3
			AND (log->>'s')::boolean = true
			AND (rg->>'i' = $4 OR (rg->>'i' IS NULL AND log->>'a' = ANY($5)))
		`;
        const queryParams = [params.userId, params.start, params.end, params.regionId, params.accessControlPointIds];
        if (params.allowUnreliableCheckins === false) {
            query += `
			AND (log->>'lr' IS NULL OR (log->>'lr')::boolean = true)`;
        }
        const { rows: minAccess } = await trxx.query(query +
            `
		ORDER BY "actionUtc" ASC LIMIT 1`, queryParams);
        if (minAccess.length > 0) {
            result.idMin = minAccess[0].id;
            result.actionUtcMin = minAccess[0].actionUtc;
        }
        const { rows: maxAccess } = await trxx.query(query +
            `
		ORDER BY "actionUtc" DESC LIMIT 1`, queryParams);
        if (maxAccess.length > 0) {
            result.idMax = maxAccess[0].id;
            result.actionUtcMax = maxAccess[0].actionUtc;
        }
        return result;
    }
    async getAccessLogsById(options) {
        if ((0, moment_1.default)(options.dateRange.endDateTime).isAfter((0, moment_1.default)())) {
            options.dateRange.endDateTime = (0, moment_1.default)().toDate();
        }
        const client = await this._pgPool.connect();
        try {
            const cursor = client.query(new Cursor(`SELECT "log" FROM "${options.organizationId}".access_logs WHERE id = ANY($1::uuid[]) AND "actionUtc" >= $2 AND "actionUtc" <= $3
            AND "organizationId" = $4 ORDER BY "actionUtc";`, [options.accessLogIds, options.dateRange.startDateTime, options.dateRange.endDateTime, options.organizationId], {
                rowMode: "array",
            }));
            let rows = [];
            while (true) {
                rows = await new Promise((resolve, reject) => {
                    cursor.read(100, (err, rows) => {
                        if (err) {
                            reject(err);
                        }
                        else {
                            resolve(rows);
                        }
                    });
                });
                options.onData(rows);
                if (rows.length < 100) {
                    break;
                }
            }
            await new Promise((resolve, reject) => {
                cursor.close((err) => {
                    if (err) {
                        reject(err);
                    }
                    else {
                        resolve();
                    }
                });
            });
            client.release();
        }
        catch (error) {
            app_logs_1.logger.error(error);
            client?.release();
        }
    }
}
exports.PSQLDalAccessLogPacs = PSQLDalAccessLogPacs;
