"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PSQLDalAccessIdentity = void 0;
const lodash_1 = __importDefault(require("lodash"));
const luxon_1 = require("luxon");
const moment_1 = __importDefault(require("moment"));
const sharp_1 = __importDefault(require("sharp"));
const uuid_1 = __importDefault(require("uuid"));
const app_config_1 = require("../../../app.config");
const app_enums_1 = require("../../../app.enums");
const app_logs_1 = require("../../../app.logs");
const business_notification_1 = require("../../../business/business.notification");
const terminal_1 = require("../../../lib/es/models/terminal");
const dal_constants_1 = require("../../dal.constants");
const dal_logger_1 = require("../../dal.logger");
const dal_manager_1 = require("../../dal.manager");
const dal_utils_1 = require("../../dal.utils");
const dal_db_armon_schema_1 = require("../../db/armon/dal.db.armon.schema");
const predefined_roles_1 = require("../../db/predefined/predefined.roles");
const dal_access_error_1 = require("../dal.access.error");
const dal_memcache_1 = require("../dal.memcache");
const dal_access_rdb_identity_1 = require("../rdb/dal.access.rdb.identity");
const dal_access_psql_common_1 = require("./dal.access.psql.common");
const restapi_1 = require("../../../lib/es/models/restapi");
const api_error_1 = require("../../../api/api.error");
const Cursor = require("pg-cursor");
class PSQLDalAccessIdentity extends dal_access_rdb_identity_1.RDBDalAccessIdentity {
    constructor(knex, pgPool) {
        super(knex, pgPool);
    }
    async searchIdentityBasic(organizationId, hasReadBasicOrganizationWide, filter, jwt) {
        let result = {
            total: 0,
            skip: filter.skip,
            take: filter.take,
            items: [],
        };
        const queryParams = [];
        let queryFrom = ` FROM "${organizationId}"."mVW_SearchUsers" AS vw
                                WHERE vw."organizationId" = ?`;
        queryParams.push(organizationId);
        if (filter.identityStatus === dal_constants_1.DalConstants.IdentityStatusType.Enabled) {
            queryFrom += ` AND vw."isDisabledForUserOrganization" IS FALSE `;
        }
        else if (filter.identityStatus === dal_constants_1.DalConstants.IdentityStatusType.Disabled) {
            queryFrom += ` AND vw."isDisabledForUserOrganization" IS TRUE `;
        }
        if (filter.genericFilter) {
            filter.genericFilter = filter.genericFilter
                .trim()
                .replace(/[\r\n\t]/g, " ")
                .replace(/[&\/\\#,+()$~%.'":*?<>{}`]/g, "")
                .split(" ")
                .map((f) => f.trim() + ":*")
                .filter((f) => f.length > 2)
                .join(" & ");
            if (filter.genericFilter && filter.genericFilter.length) {
                queryFrom += ` AND vw."searchDocument" @@ to_tsquery('simple', unaccent(?)) `;
                queryParams.push(filter.genericFilter);
            }
            else {
                return Promise.resolve(result);
            }
        }
        if (filter.userGroupId) {
            queryFrom += ` AND vw."userGroupIds" LIKE '%'||?||'%' `;
            queryParams.push(filter.userGroupId);
        }
        if (filter.organizationUnitId) {
            if (filter.collectChildrenOfOrganizationUnit) {
                queryFrom += ` AND vw."organizationUnitHierarchies" LIKE '%'||?||'%' `;
                queryParams.push(filter.organizationUnitId);
            }
            else {
                queryFrom += ` AND vw."organizationUnitIds" LIKE '%'||?||'%' `;
                queryParams.push(filter.organizationUnitId);
            }
        }
        if (!hasReadBasicOrganizationWide) {
            let unitIdsOfQueryUser = jwt.organizationUnitIdsOfOrganization(organizationId);
            if (unitIdsOfQueryUser.length === 0) {
                result.total = filter.returnTotalCount ? 0 : null;
                return Promise.resolve(result);
            }
            else {
                queryFrom +=
                    " AND (" +
                        unitIdsOfQueryUser
                            .map((ui) => {
                            queryParams.push(ui);
                            return `vw."organizationUnitHierarchies" LIKE '%'||?||'%'`;
                        })
                            .join(" OR ") +
                        ") ";
            }
        }
        let query = `SELECT COUNT(*) AS "count" ` + queryFrom;
        let countResult = await this.dbClient.raw(query, queryParams);
        if (countResult && countResult.rowCount > 0) {
            result.total = parseInt(countResult.rows[0].count);
        }
        else {
            result.total = 0;
        }
        if (result.total === 0) {
            return Promise.resolve(result);
        }
        let selectQuery = `SELECT * ${queryFrom} ORDER BY ` +
            (filter.genericFilter && filter.genericFilter.length > 0 ? ` ts_rank(vw."searchDocument", to_tsquery('turkish', ?)) DESC, ` : "") +
            ` vw."userOrganizationProfileNameForOrder" ASC,
                        vw."userOrganizationProfileSurnameForOrder" ASC,
                        vw."userOrganizationProfileUniqueIdForOrder" ASC
                        LIMIT ${filter.take} OFFSET ${filter.skip}`;
        if (filter.genericFilter && filter.genericFilter.length) {
            queryParams.push(filter.genericFilter);
        }
        let users = await this.dbClient.raw(selectQuery, queryParams);
        users.rows.map((user) => {
            let item = {
                email: user.userOrganizationProfile.email,
                fullName: user.userOrganizationProfile.length > 0 ? user.userOrganizationProfile[0].name + " " + user.userOrganizationProfile[0].surname : "",
                id: user.userId,
                userGroups: user.userGroups,
                isDisabled: user.isDisabledForUserOrganization,
                hasUserAccount: user.user.length > 0 ? user.user[0].username != null : false,
                organizationUnits: user.organizationUnits,
                uniqueId: user.userOrganizationProfileUniqueIdForOrder,
            };
            result.items.push(item);
        });
        return Promise.resolve(result);
    }
    async collectUserIdsByFieldSelect(organizationId, request) {
        let result = {
            fieldName: request.fieldName,
            fieldType: request.fieldType,
            isExtensionField: request.isExtensionField,
            items: [],
        };
        let qb = this.dbClient
            .withSchema(organizationId)
            .table("userOrganizationProfiles as uop")
            .innerJoin("userOrganizations as uo", "uo.id", "uop.userOrganizationId")
            .whereNull("uop.deletedAt")
            .whereNull("uo.deletedAt")
            .where("uo.organizationId", organizationId);
        if (request.isExtensionField) {
            qb.where((where) => {
                for (let param of request.parameters) {
                    where.orWhereRaw(`uop."extensionFields"->>? = ?`, [request.fieldName, param]);
                }
            });
        }
        else {
            qb.whereIn("uop." + request.fieldName, request.parameters);
        }
        await qb.select("uop.userId", "uop." + (request.isExtensionField ? "extensionFields" : request.fieldName)).then((rows) => {
            result.items = rows.map((item) => {
                return {
                    parameter: request.isExtensionField ? (item.extensionFields ? item.extensionFields[request.fieldName] : "") : item[request.fieldName],
                    userId: item.userId,
                };
            });
        });
        return result;
    }
    async deleteUserOrganizationProfilePhoto(organizationId, userId, photoId) {
        await this.dbClient.withSchema(organizationId).table("userOrganizationProfilePhotos").where("id", photoId).del();
        return Promise.resolve(true);
    }
    async addUserOrganizationProfilePhoto(args) {
        let userOrganizationId = "";
        let id = uuid_1.default.v4();
        await this.dbClient
            .withSchema(args.organizationId)
            .table("userOrganizations")
            .where("userId", args.userId)
            .where("organizationId", args.organizationId)
            .select("id")
            .first()
            .then(async (userOrganizationId) => await this.dbClient.withSchema(args.organizationId).table("userOrganizationProfilePhotos").insert({
            id: id,
            userId: args.userId,
            userOrganizationId: userOrganizationId.id,
            imageType: args.type,
        }));
        return Promise.resolve(id);
    }
    async listUserOrganizationProfilePhotos(args) {
        let photoItemList = {
            items: [],
        };
        await this.dbClient
            .withSchema(args.organizationId)
            .table("userOrganizations")
            .where("userId", args.userId)
            .where("organizationId", args.organizationId)
            .select("id")
            .first()
            .then(async (userOrganizationId) => {
            await this.dbClient
                .withSchema(args.organizationId)
                .table("userOrganizationProfilePhotos")
                .where("userOrganizationId", userOrganizationId.id)
                .where("userId", args.userId)
                .select()
                .then(function (results) {
                results.forEach((element) => {
                    photoItemList.items.push({
                        id: element.id,
                        userId: element.userId,
                        organizationId: args.organizationId,
                        imageType: element.imageType,
                    });
                });
                return Promise.resolve(photoItemList);
            });
        });
        return Promise.resolve(photoItemList);
    }
    async upsertUserOrganizationThumbnail(args) {
        let affectedRowCount = 0;
        const base64Data = new Buffer(args.thumbnail.toString().replace(/^data:image\/\w+;base64,/, ""), "base64");
        let avatar = (await (0, sharp_1.default)(base64Data, { failOnError: false }).resize(256, 256).jpeg({ quality: 90 }).toFormat("jpeg").toBuffer()).toString("base64");
        avatar = "data:image/jpeg;base64," + avatar;
        await this.dbClient
            .withSchema(args.organizationId)
            .table("userOrganizations")
            .where("userId", args.userId)
            .where("organizationId", args.organizationId)
            .select("id")
            .first()
            .then(async (userOrganizationId) => {
            await this.dbClient
                .withSchema(args.organizationId)
                .table("userOrganizationProfiles")
                .where("userOrganizationId", userOrganizationId.id)
                .where("userId", args.userId)
                .update({
                thumbnail: avatar,
            })
                .then((result) => {
                affectedRowCount = result;
            });
            Promise.resolve();
        });
        if (affectedRowCount > 0)
            return Promise.resolve(true);
        else
            return Promise.resolve(false);
    }
    async geUserOrganizationThumbnail(args) {
        let result = await this.dbClient
            .withSchema(args.organizationId)
            .table("userOrganizations as uo")
            .innerJoin("userOrganizationProfiles as uop", "uop.userOrganizationId", "uo.id")
            .where("uo.userId", args.userId)
            .where("uo.organizationId", args.organizationId)
            .first("uop.thumbnail");
        return Promise.resolve(result && result.thumbnail ? result : null);
    }
    async getUserEmailAddressAndIsDisabled(organizationId, userId, trx) {
        const trxx = trx ?? this._pgPool;
        let { rows, rowCount } = await trxx.query(`
            SELECT uop."email", uo."isDisabled" FROM "${organizationId}"."userOrganizationProfiles" AS uop
            INNER JOIN "${organizationId}"."userOrganizations" AS uo
            ON uop."userOrganizationId" = uo.id
            WHERE uo."organizationId" = $1
            AND uo."userId" = $2`, [organizationId, userId]);
        if (rowCount === 0) {
            let { rows, rowCount } = await trxx.query(`
            SELECT up."email", uo."isDisabled" FROM "${organizationId}"."userProfiles" AS up
            INNER JOIN "${organizationId}"."userOrganizations" AS uo
            ON up."userId" = uo."userId"
                WHERE uo."userId" = $1
            `, [userId]);
            return {
                email: rows[0].email,
                isDisabled: rows[0].isDisabled,
            };
        }
        return rows[0];
    }
    async getUsersEmailAddress(organizationId, userIds) {
        let { rows } = await this._pgPool.query(`
        SELECT up."userId", uop.email as "organizationMail", up.email as "userMail" FROM "${organizationId}"."userOrganizationProfiles" AS uop
        INNER JOIN "${organizationId}"."userProfiles" up
        ON up."userId" = uop."userId"
        WHERE up."userId" = ANY($1::uuid[])
        `, [userIds]);
        return rows.map((r) => r.organizationMail || r.userMail).filter((r) => r);
    }
    prepareCountableSearchIdentityRequest(filter) {
        let qb = this.dbClient
            .withSchema(filter.organizationId)
            .from("userOrganizations as uo")
            .innerJoin("userOrganizationProfiles as uop", "uop.userOrganizationId", "uo.id")
            .where("uo.organizationId", "=", filter.organizationId)
            .whereNull("uo.deletedAt")
            .whereNull("uop.deletedAt");
        if (filter.hasMail) {
            qb.whereNotNull("uop.email");
        }
        if (filter.status && filter.status !== dal_constants_1.DalConstants.IdentityStatusType.All) {
            qb.where("uo.isDisabled", "=", filter.status === dal_constants_1.DalConstants.IdentityStatusType.Disabled);
        }
        if (filter.organizationUnits?.length) {
            qb.innerJoin("userOrganizationOrganizationUnits as uoou", "uo.id", "uoou.userOrganizationId")
                .innerJoin("organizationUnits as ou", "ou.id", "uoou.organizationUnitId")
                .whereNull("uoou.deletedAt")
                .whereNull("ou.deletedAt");
            let rqitems = [];
            for (const unit of filter.organizationUnits) {
                rqitems.push(`ou.id = '${unit.id}'`);
                if (unit.includeDescendants) {
                    rqitems.push(`ou."ancestorIds" like '%${unit.id}%'`);
                }
            }
            qb.andWhere(this.dbClient.raw("(" + rqitems.join(" or ") + ")"));
        }
        if (filter.workplans?.length) {
            qb.whereRaw(`uo."userId" IN (SELECT DISTINCT "userId" FROM "${filter.organizationId}"."userWorkPlans" uwp WHERE uwp."workPlanId" = ANY(?))`, [filter.workplans]);
        }
        if (filter.userGroupIds?.length) {
            qb.innerJoin("userGroupUserOrganizations as uguo", "uo.id", "uguo.userOrganizationId")
                .innerJoin("userGroups as ug", "ug.id", "uguo.userGroupId")
                .whereNull("uguo.deletedAt")
                .whereNull("ug.deletedAt")
                .whereIn("ug.id", filter.userGroupIds);
        }
        if (filter.credentialExtensionFilters?.length) {
            qb.innerJoin("userOrganizationCredentials as uoc", "uoc.userId", "uo.userId")
                .innerJoin("roles as r", "uo.roleId", "r.id")
                .where("uoc.organizationId", filter.organizationId)
                .whereNull("uoc.deletedAt");
            for (const filt of filter.credentialExtensionFilters) {
                let column = `uoc."extensionFields"->>'` + filt.name + "'";
                qb.whereRaw(`"${filter.organizationId}".unaccent('${column}')::TEXT ilike "${filter.organizationId}".unaccent('%${filt.value}%')`);
            }
        }
        if (filter.profileFilter) {
            let profileFilter = filter.profileFilter
                .trim()
                .replace(/[\r\n\t]/g, " ")
                .replace(/\s/g, "%");
            qb.whereRaw(` uop."name" || ' ' || uop."surname" ILIKE ? `, "%" + profileFilter + "%");
        }
        if (filter.permissions?.length) {
            const perms = [];
            for (const per of filter.permissions) {
                perms.push(`POSITION('${per}' in r.permissions) > 0`);
            }
            qb.innerJoin("roles as r", "uo.roleId", "r.id").whereRaw(perms.join(" AND "));
        }
        if (filter.permissionsOr?.length) {
            const perms = [];
            for (const per of filter.permissionsOr) {
                perms.push(`POSITION('${per}' in r.permissions) > 0`);
            }
            qb.innerJoin("roles as r", "uo.roleId", "r.id").whereRaw(`(${perms.join(" OR ")})`);
        }
        return qb;
    }
    prepareCountableSearchIdentityExactRequest(organizationId, filter) {
        let qb = this.dbClient
            .withSchema(organizationId)
            .from("userOrganizations as uo")
            .innerJoin("userOrganizationProfiles as uop", "uop.userOrganizationId", "uo.id")
            .where("uo.organizationId", organizationId)
            .whereNull("uo.deletedAt")
            .whereNull("uop.deletedAt");
        if (filter.createdDate) {
            if (filter.createdDate.createdAtStart) {
                qb.where("uo.createdAt", ">", filter.createdDate.createdAtStart);
            }
            if (filter.createdDate.createdAtEnd) {
                qb.where("uo.createdAt", "<", filter.createdDate.createdAtEnd);
            }
        }
        if (filter.user) {
            qb.innerJoin("users as u", "u.id", "uo.userId").whereNull("u.deletedAt");
            if (filter.user.username) {
                qb.whereRaw("lower(u.username) = lower(?)", filter.user.username);
            }
            if (filter.user.accountEnabled) {
                qb.where("u.accountEnabled", filter.user.accountEnabled);
            }
        }
        if (filter.userOrganization) {
            qb.where("uo.isDisabled", filter.userOrganization.isDisabled);
            if (filter.userOrganization.roleId) {
                qb.leftJoin("userOrganizationOrganizationUnits as uoou2", (join) => {
                    join.on("uoou2.userOrganizationId", "uo.id").andOnNull("uoou2.deletedAt");
                });
                qb.where((innerQb) => {
                    innerQb.where("uo.roleId", filter.userOrganization.roleId).orWhere("uoou2.roleId", filter.userOrganization.roleId);
                });
            }
        }
        if (filter.userOrganizationProfile) {
            for (const key of Object.keys(filter.userOrganizationProfile)) {
                if (key && key != "extensionFields") {
                    qb.whereRaw(`unaccent(upper(uop."${key}")) = unaccent(upper(?))`, filter.userOrganizationProfile[key]);
                }
            }
            if (filter.userOrganizationProfile.extensionFields?.length) {
                for (const filt of filter.userOrganizationProfile.extensionFields) {
                    if (filt.name === "birthDateUtc" || filt.name === "employmentStartUtc" || filt.name === "employmentEndUtc") {
                        qb.whereRaw(` uop."${filt.name}" > ?`, filt.value.start);
                        qb.whereRaw(` uop."${filt.name}" < ?`, filt.value.end);
                    }
                    else {
                        let column = `uop."extensionFields"->>'` + filt.name + "'";
                        qb.whereRaw("unaccent(upper(" + column + "))::TEXT = unaccent(upper(?))", filt.value);
                    }
                }
            }
        }
        if (filter.organizationUnit && filter.organizationUnit.id) {
            qb.innerJoin("userOrganizationOrganizationUnits as uoou", "uo.id", "uoou.userOrganizationId")
                .innerJoin("organizationUnits as ou", "ou.id", "uoou.organizationUnitId")
                .whereNull("uoou.deletedAt")
                .whereNull("ou.deletedAt");
            if (filter.organizationUnit.includeDescendants && filter.organizationUnit.id) {
                qb.whereRaw(`(ou."ancestorIds" like '%${filter.organizationUnit.id}%' or ou.id = '${filter.organizationUnit.id}' )`);
            }
            if (!filter.organizationUnit.includeDescendants && filter.organizationUnit.id) {
                qb.where("ou.id", filter.organizationUnit.id);
            }
        }
        if (filter.userGroup && filter.userGroup.id) {
            qb.innerJoin("userGroupUserOrganizations as uguo", "uo.id", "uguo.userOrganizationId")
                .innerJoin("userGroups as ug", "ug.id", "uguo.userGroupId")
                .whereNull("uguo.deletedAt")
                .whereNull("ug.deletedAt")
                .where("ug.id", filter.userGroup.id);
        }
        if (filter.userOrganizationCredential) {
            qb.innerJoin("userOrganizationCredentials as uoc", "uoc.userId", "uo.userId")
                .innerJoin("roles as r", "uo.roleId", "r.id")
                .where("uoc.organizationId", organizationId)
                .where("uoc.type", filter.userOrganizationCredential.type)
                .whereNull("uoc.deletedAt");
            if (filter.userOrganizationCredential.data) {
                qb.where("uoc.data", filter.userOrganizationCredential.data);
            }
            if (filter.userOrganizationCredential.extensionFields?.length) {
                for (const extensionField of filter.userOrganizationCredential.extensionFields) {
                    let fieldValue = extensionField.value;
                    if (fieldValue && fieldValue.id && fieldValue.value) {
                        let column = `uoc."extensionFields"->>'` + extensionField.name + "'";
                        qb.whereRaw("" + column + "::text ilike ?", "%" + fieldValue.id + "%");
                    }
                    else if (fieldValue) {
                        let column = `uoc."extensionFields"->>'` + extensionField.name + "'";
                        qb.whereRaw("" + column + "::text = ?", fieldValue);
                    }
                }
            }
            if (filter.userOrganizationCredential.customFields?.length) {
                for (const customField of filter.userOrganizationCredential.customFields) {
                    let fieldValue = customField.value;
                    if (fieldValue && fieldValue.id && fieldValue.value) {
                        let column = `uoc."customFields"->>'` + customField.name + "'";
                        qb.whereRaw("" + column + "::text ilike ?", "%" + fieldValue.id + "%");
                    }
                    else if (fieldValue) {
                        let column = `uoc."customFields"->>'` + customField.name + "'";
                        qb.whereRaw("" + column + "::text = ?", fieldValue);
                    }
                }
            }
        }
        if (filter.workplan?.id) {
            qb.innerJoin("userWorkPlans as uwp", "uo.userId", "uwp.userId").where("uwp.workPlanId", filter.workplan.id);
        }
        return qb;
    }
    async getSearchIdentityResultFromUserOrganizationIds(userOrganizationIds, organizationId, trx) {
        let organizationUnitsColumn = this.dbClient
            .raw(`
            to_json( ARRAY(select json_build_object('id', ou.id, 'name', ou.name, 'ancestorIds',
            ou."ancestorIds", 'shortCode', ou."shortCode")
            from "${organizationId}"."userOrganizationOrganizationUnits" as uoou
            inner join "${organizationId}"."organizationUnits" as ou on uoou."organizationUnitId" = ou.id
            where ou."deletedAt" is null and uoou."userOrganizationId" = uo.id and uoou."deletedAt" is null
            order by ou.name))
            `)
            .wrap("(", ') "organizationUnits"');
        let userGroupsColumn = this.dbClient
            .raw(`
            select to_json( ARRAY(select json_build_object('id', ug.id, 'name', ug.name)
            from "${organizationId}"."userGroupUserOrganizations" as uguo
            inner join "${organizationId}"."userGroups" as ug on uguo."userGroupId" = ug.id
            where
            uguo."userOrganizationId" = uo.id and
            uguo."deletedAt" is null and
            ug."deletedAt" is null
            order by ug.name))
            `)
            .wrap("(", ') "userGroups"');
        let forbiddanceColumn = this.dbClient
            .raw(`
                select to_json( ARRAY(select json_build_object('id', uof.id)
                from "${organizationId}"."userOrganizationForbiddances" as uof
                where
                uof."userId" = uo."userId" and
                (uof."startUtc" <= now()  or uof."startUtc" is null) and (uof."endUtc" >= now() or uof."endUtc" is null) and
                uof."deletedAt" is null
                ))
                `)
            .wrap("(", ') "forbiddance"');
        let now = new Date();
        let qbResultItems = this.dbClient
            .withSchema(organizationId)
            .from("userOrganizations as uo")
            .innerJoin("users as u", "u.id", "uo.userId")
            .innerJoin("userOrganizationProfiles as uop", "uop.userOrganizationId", "uo.id")
            .whereIn("uo.id", userOrganizationIds)
            .whereNull("uo.deletedAt")
            .whereNull("uop.deletedAt")
            .select("uo.userId", "uop.name", "uop.surname", "uop.uniqueId", "u.username", "uo.isDisabled", "uop.isAnonymized", forbiddanceColumn, organizationUnitsColumn, userGroupsColumn)
            .orderBy("uop.name")
            .orderBy("uop.surname");
        if (trx) {
            qbResultItems.transacting(trx);
        }
        return qbResultItems.then((items) => items.map((item) => {
            return {
                id: item.userId,
                fullname: [item.name, item.surname].join(" "),
                uniqueId: item.uniqueId,
                isDisabled: item.isDisabled,
                isAnonymized: item.isAnonymized,
                userCaptions: [],
                hasUserAccount: item.username ? true : false,
                hasForbiddance: item.forbiddance && item.forbiddance[0] && item.forbiddance[0].id ? true : false,
                organizationUnits: !item.organizationUnits
                    ? []
                    : item.organizationUnits.map((ou) => {
                        return {
                            id: ou.id,
                            name: ou.name,
                            ancestorIds: ou.ancestorIds ? ou.ancestorIds.split(",") : [],
                            shortCode: ou.shortCode,
                        };
                    }),
                userGroups: item.userGroups || [],
            };
        }));
    }
    async searchIdentity(filter) {
        let qb = this.prepareCountableSearchIdentityRequest(filter);
        let qbCount = await qb.clone().countDistinct("uo.id").first();
        let count = qbCount && qbCount.count ? parseInt(qbCount.count) : 0;
        let result = {
            pagination: {
                total: count,
                skip: 0,
                take: 0,
            },
            items: [],
        };
        if (count === 0) {
            return Promise.resolve(result);
        }
        result.pagination.take = filter.take || 0;
        result.pagination.skip = filter.skip || 0;
        qb.distinct("uo.id", "uop.name", "uop.surname").select().orderBy("uop.name").orderBy("uop.surname");
        if (filter.skip) {
            qb.offset(filter.skip);
        }
        if (filter.take) {
            qb.limit(filter.take);
        }
        let userOrganizationIds = (await qb).map((m) => m.id);
        result.items = await this.getSearchIdentityResultFromUserOrganizationIds(userOrganizationIds, filter.organizationId);
        let userCaptionList = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(filter.organizationId, result.items.map((u) => u.id));
        for (let index = 0; index < result.items.length; index++) {
            result.items[index].userCaptions = (0, dal_utils_1.loadSafeUserCaptions)(result.items[index].id, userCaptionList);
        }
        return Promise.resolve(result);
    }
    async searchIdentityForExcel(filter) {
        let result = [];
        let qb = this.prepareCountableSearchIdentityRequest(filter);
        let userOrganizationIds = (await qb.distinct("uo.id").limit(filter.take)).map((m) => m.id);
        userOrganizationIds = lodash_1.default.uniq(userOrganizationIds);
        let page = 1000;
        for (let pageIndex = 0; pageIndex < userOrganizationIds.length; pageIndex += page) {
            let ids = userOrganizationIds.slice(pageIndex, pageIndex + page);
            let qbResultItems = this.dbClient
                .withSchema(filter.organizationId)
                .from("userOrganizations as uo")
                .innerJoin("users as u", "u.id", "uo.userId")
                .innerJoin("userOrganizationProfiles as uop", "uop.userOrganizationId", "uo.id")
                .whereIn("uo.id", ids)
                .whereNull("uo.deletedAt")
                .whereNull("uop.deletedAt")
                .orderBy("uop.name")
                .orderBy("uop.surname")
                .select("uo.userId");
            let items = await qbResultItems;
            for (const item of items) {
                result.push(item.userId);
            }
        }
        return Promise.resolve(result);
    }
    async searchIdentityExact(organizationId, filter) {
        let qb = this.prepareCountableSearchIdentityExactRequest(organizationId, filter);
        let qbCount = await qb.clone().countDistinct("uo.id").first();
        let count = qbCount && qbCount.count ? parseInt(qbCount.count) : 0;
        let result = {
            pagination: {
                total: count,
                skip: 0,
                take: 0,
            },
            items: [],
        };
        if (count === 0) {
            return Promise.resolve(result);
        }
        result.pagination.take = filter.take || 0;
        result.pagination.skip = filter.skip || 0;
        qb.distinct("uo.id", "uop.name", "uop.surname").select().orderBy("uop.name").orderBy("uop.surname");
        if (filter.skip) {
            qb.offset(filter.skip);
        }
        if (filter.take) {
            qb.limit(filter.take);
        }
        let userOrganizationIds = (await qb).map((m) => m.id);
        result.items = await this.getSearchIdentityResultFromUserOrganizationIds(userOrganizationIds, organizationId);
        let userCaptionList = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, result.items.map((u) => u.id));
        for (let index = 0; index < result.items.length; index++) {
            result.items[index].userCaptions = (0, dal_utils_1.loadSafeUserCaptions)(result.items[index].id, userCaptionList);
        }
        return Promise.resolve(result);
    }
    async listIdentity(organizationId, userIds, take, skip) {
        let qb = this.dbClient
            .withSchema(organizationId)
            .from("userOrganizations as uo")
            .innerJoin("userOrganizationProfiles as uop", "uop.userOrganizationId", "uo.id")
            .where("uo.organizationId", organizationId)
            .whereIn("uo.userId", userIds)
            .whereNull("uo.deletedAt")
            .whereNull("uop.deletedAt");
        let qbCount = await qb.clone().countDistinct("uo.id").first();
        let count = qbCount && qbCount.count ? parseInt(qbCount.count) : 0;
        qb.distinct("uo.id", "uop.name", "uop.surname").select().orderBy("uop.name").orderBy("uop.surname");
        if (skip) {
            qb.offset(skip);
        }
        if (take) {
            qb.limit(take);
        }
        let result = {
            pagination: {
                total: count,
                skip: 0,
                take: 0,
            },
            items: [],
        };
        result.items = await this.getSearchIdentityResultFromUserOrganizationIds((await qb).map((m) => m.id), organizationId);
        return Promise.resolve(result);
    }
    async findCredentialStatus(organizationId, credentialData, credentialType) {
        let credentialSearchValues = await this.generateCredentialSearch(organizationId, credentialData, credentialType);
        let qb = this.dbClient
            .withSchema(organizationId)
            .table("userOrganizationCredentials")
            .where("organizationId", organizationId)
            .whereNull("deletedAt")
            .where((where) => {
            for (const credential of credentialSearchValues) {
                where = where.orWhereRaw("upper(unaccent(data)) = upper(unaccent(?))", credential);
            }
        });
        if (credentialType !== null && credentialType !== undefined)
            qb.where("type", credentialType);
        let credential = await qb.first();
        if (!credential)
            return Promise.resolve({
                existance: dal_constants_1.DalConstants.CredentialExistance.NotExits,
            });
        let visitor = await this.dbClient
            .withSchema(organizationId)
            .table("userOrganizations as uo")
            .innerJoin("organizationVisitorProfiles as ovp", (join) => {
            join.on("ovp.userOrganizationId", "=", "uo.id");
        })
            .where("uo.userId", credential.userId)
            .where("uo.organizationId", organizationId)
            .whereNull("ovp.deletedAt")
            .whereNull("uo.deletedAt")
            .first("ovp.id", "ovp.name");
        if (visitor)
            return Promise.resolve({
                existance: dal_constants_1.DalConstants.CredentialExistance.Visitor,
                visitorInfo: visitor.name,
                visitorProfileId: visitor.id,
                credentialType: credential.type,
            });
        return Promise.resolve({
            existance: dal_constants_1.DalConstants.CredentialExistance.Member,
            credentialType: credential.type,
            userId: credential.userId,
        });
    }
    async generateCredentialSearch(organizationId, credentialRawData, credentialType) {
        const organization = await dal_manager_1.dbManager.accessOrganization.getOrganization(organizationId);
        if (!organization) {
            return Promise.resolve([credentialRawData]);
        }
        let result = [];
        if (credentialType !== null && credentialType !== undefined) {
            const credentialConverter = organization.credentialTypes.find((c) => c.type === credentialType);
            let convertFunction = credentialConverter && credentialConverter.dataConverter ? new Function("r", credentialConverter.dataConverter) : dal_utils_1.createCredentialData;
            result.push(await convertFunction(credentialRawData, credentialType));
        }
        else {
            for (const cType of organization.availableCredentialTypes) {
                if ([terminal_1.CredentialType.MiFare, terminal_1.CredentialType.ProximityCard, terminal_1.CredentialType.VehiclePlate, terminal_1.CredentialType.UHFRfid].includes(cType)) {
                    let credentialConverter = organization.credentialTypes.find((c) => c.type == cType);
                    let convertFunction = credentialConverter && credentialConverter.dataConverter ? new Function("r", credentialConverter.dataConverter) : dal_utils_1.createCredentialData;
                    result.push(await convertFunction(credentialRawData, cType));
                }
                else {
                    result.push(credentialRawData);
                }
            }
        }
        return Promise.resolve(lodash_1.default.uniq(result));
    }
    async findMemberByCredential(organizationId, hasVisitorRead, credentialData, credentialType, organizationUnitIds) {
        let result = {
            roleId: null,
            existance: dal_constants_1.DalConstants.CredentialExistance.NotExits,
            member: null,
            visitorInfo: "",
            credentialType: credentialType,
        };
        let checkCredential = await this.findCredentialStatus(organizationId, credentialData, credentialType);
        result.existance = checkCredential.existance;
        result.credentialType = checkCredential.credentialType;
        if (checkCredential.existance == dal_constants_1.DalConstants.CredentialExistance.NotExits) {
            return Promise.resolve(result);
        }
        if (checkCredential.existance == dal_constants_1.DalConstants.CredentialExistance.Visitor) {
            if (hasVisitorRead)
                result.visitorInfo = checkCredential.visitorInfo;
            return Promise.resolve(result);
        }
        let credentialSearchValues = await this.generateCredentialSearch(organizationId, credentialData, credentialType);
        let qbCredential = this.dbClient
            .withSchema(organizationId)
            .from("userOrganizationCredentials as uoc")
            .innerJoin("userOrganizations as uo", "uoc.userId", "uo.userId")
            .innerJoin("roles as r", "uo.roleId", "r.id")
            .where("uo.organizationId", "=", organizationId)
            .where("uoc.organizationId", "=", organizationId)
            .whereNull("uoc.deletedAt")
            .whereNull("uo.deletedAt")
            .where((where) => {
            for (const credential of credentialSearchValues) {
                where = where.orWhereRaw("upper(unaccent(uoc.data)) = upper(unaccent(?))", credential);
            }
        });
        let credentialResult = await qbCredential.first("uoc.id", "uoc.userId", "uoc.type", "uo.id as userOrganizationId", "r.typeId");
        if (!credentialResult) {
            return Promise.resolve(null);
        }
        if (credentialResult.roleId === predefined_roles_1.PredefinedRoles.OrganizationVisitor.id || credentialResult.roleId === predefined_roles_1.PredefinedRoles.OrganizationUnitVisitor.id) {
            return Promise.resolve(result);
        }
        result.roleId = credentialResult.roleId;
        let findUserResult = await this.getSearchIdentityResultFromUserOrganizationIds([credentialResult.userOrganizationId], organizationId);
        if (findUserResult?.length) {
            result.member = findUserResult[0];
            if (organizationUnitIds?.length) {
                let canSee = false;
                for (const organizationUnitId of organizationUnitIds) {
                    if (result.member.organizationUnits.find((ou) => ou.id === organizationUnitId || ou.ancestorIds.includes(organizationUnitId))) {
                        canSee = true;
                        break;
                    }
                }
                if (!canSee) {
                    result.member = null;
                    result.existance = dal_constants_1.DalConstants.CredentialExistance.NotPermitted;
                }
            }
        }
        return Promise.resolve(result);
    }
    async getSelfDeputyInfo(organizationId, userId) {
        let deputySummary = {
            items: [],
        };
        await this.dbClient
            .withSchema(organizationId)
            .table("userOrganizationDeputies as uod")
            .where("uod.startDateTime", "<", new Date())
            .where("uod.endDateTime", ">", new Date())
            .innerJoin("organizationUnits as ou", "ou.id", "uod.organizationUnitId")
            .whereNull("ou.deletedAt")
            .innerJoin("userOrganizations as uo", "uo.id", "uod.deputyUserOrganizationId")
            .whereNull("uo.deletedAt")
            .where("uo.userId", userId)
            .where("uo.organizationId", organizationId)
            .innerJoin("userOrganizations as uo2", "uo2.id", "uod.userOrganizationId")
            .whereNull("uo2.deletedAt")
            .innerJoin("userOrganizationProfiles as uop", "uop.userOrganizationId", "uo2.id")
            .select("uop.name", "uop.surname", "ou.name as organizationUnitName", "uod.startDateTime", "uod.endDateTime")
            .then((rows) => {
            for (let row of rows) {
                deputySummary.items.push({
                    organizationUnitName: row.organizationUnitName,
                    principalFullName: (row.name ?? "") + " " + (row.surname ?? ""),
                    startDateTime: row.startDateTime,
                    endDateTime: row.endDateTime,
                });
            }
        });
        return Promise.resolve(deputySummary);
    }
    async getUserSummaryInfo(params) {
        const result = {
            thumbnail: "",
            name: "",
            surname: "",
            workPlanName: "",
        };
        let queryParamsIndex = 1;
        const query = `
			SELECT encode("uop"."thumbnail", 'escape') as thumbnail, "uop"."name", "uop"."surname", "wp"."name" AS "workPlanName" 
			FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" AS "uo" 
			INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" AS "uop" ON "uop"."userOrganizationId" = "uo"."id" 
			LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userWorkPlans}" AS "uwp" ON "uwp"."userId" = "uo"."userId" 
			LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.workPlans}" AS "wp" ON "wp"."id" = "uwp"."workPlanId" 
			WHERE "uo"."userId" = $${queryParamsIndex++} AND "uo"."organizationId" = $${queryParamsIndex++} 
			AND ("uwp"."startDateTime" <= $${queryParamsIndex} OR "uwp"."startDateTime" IS NULL) 
			AND ("uwp"."endDateTime" >= $${queryParamsIndex} OR "uwp"."endDateTime" IS NULL) 
			AND "uo"."deletedAt" IS NULL AND "uop"."deletedAt" IS NULL 
			ORDER BY "uwp"."startDateTime" ASC 
			LIMIT 1
		`;
        const { rows, rowCount } = await params.trx.query(query, [params.userId, params.organizationId, (0, moment_1.default)().toDate()]);
        if (rowCount > 0) {
            const row = rows[0];
            result.name = row.name;
            result.surname = row.surname;
            result.thumbnail = row.thumbnail;
            result.workPlanName = row.workPlanName ? row.workPlanName : "";
        }
        return result;
    }
    async insertIdentity(organizationId, requestUserId, params, trx) {
        const { insertIdentity, notifyUsersForChange } = params;
        const opTimeWithOffset = luxon_1.DateTime.fromJSDate(new Date()).toSQL({ includeOffset: true });
        const { userId, userProfileId, userOrganizationId, userOrganizationProfileId } = await this.insertUserInfo(organizationId, requestUserId, opTimeWithOffset, trx, insertIdentity);
        const { rows: existingOrganizationUnits } = await trx.query(`
			SELECT "organizationUnitId"
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}"
			WHERE "userOrganizationId" = $1
			AND "deletedAt" IS NULL;
		`, [userOrganizationId]);
        if (insertIdentity.organizationUnits?.length) {
            let roles = await dal_manager_1.dbManager.accessUser.getTypedRoleIdsOfOrganizationPg({ organizationId, trx });
            let newOrganizationUnits = insertIdentity.organizationUnits.filter((ou) => !existingOrganizationUnits.some((e) => e.organizationUnitId === ou.organizationUnitId));
            await trx.query(`
				INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}"
				(id, "createdAt", "updatedAt", "userOrganizationId", "organizationUnitId", "roleId", "extraPermissions", "source", "sourceId")
				SELECT gen_random_uuid(), $1, $1, $2, (sq->>'organizationUnitId')::UUID, (sq->>'roleId')::UUID, sq->>'extraPermissions', $4, $5
				FROM UNNEST ($3::JSONB[]) AS sq
				ON CONFLICT ("userOrganizationId", "organizationUnitId") WHERE "deletedAt" IS NULL
				DO UPDATE SET
					"roleId" = EXCLUDED."roleId", 
					"extraPermissions" = EXCLUDED."extraPermissions";
				`, [opTimeWithOffset, userOrganizationId, newOrganizationUnits, insertIdentity.source ?? dal_constants_1.DalConstants.SourceType.User, insertIdentity.sourceId]);
            await this.promoteUserToOrganizationUnitManager(organizationId, trx, {
                userId: userId,
                userOrganizationId: userOrganizationId,
                organizationUnitIds: newOrganizationUnits.filter((uo) => uo.roleId === roles.unitAdministratorId).map((uo) => uo.organizationUnitId),
                roles,
            });
        }
        if (insertIdentity.credentials?.length) {
            for (const credential of insertIdentity.credentials) {
                let extensionFields = credential.extensionFields;
                try {
                    await trx.query(`
						INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials}"
						(id, "createdAt", "updatedAt", "organizationId", "userId", type, data, "expiresOn", "credentialNumber", "extensionFields", "customFields", note, "groupNumber", "source", "sourceId")
						VALUES(uuid_generate_v4(), $1,  $1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13);`, [
                        opTimeWithOffset,
                        organizationId,
                        userId,
                        credential.type,
                        credential.data,
                        credential.expirationUtc,
                        credential.credentialNumber,
                        extensionFields,
                        credential.customFields,
                        credential.note,
                        credential.groupNumber,
                        insertIdentity.source ?? dal_constants_1.DalConstants.SourceType.User,
                        insertIdentity.sourceId,
                    ]);
                }
                catch (error) {
                    let message = error.message.split(",")[1];
                    if (error.message === app_enums_1.enums.ErrorCode.OneTimeAssignableCredential.toString()) {
                        (0, dal_access_error_1.throwDbAccessConflictErrorTr)("ERRORS.IDENTITY.NONRETURNABLE_CREDENTIAL");
                    }
                    else if (error.message.startsWith(app_enums_1.enums.ErrorCode.CredentialToInsertAlreadyExists.toString())) {
                        (0, dal_access_error_1.throwDbAccessConflictErrorTr)("ERRORS.IDENTITY.DUPLICATECREDENTIAL", { data: message });
                    }
                    throw error;
                }
            }
        }
        if (insertIdentity.userGroupIds?.length) {
            const { rows: existingGroupIds, rowCount } = await trx.query(`
				SELECT "userGroupId"
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupUserOrganizations}" 
				WHERE "userOrganizationId" = $1
				AND "deletedAt" IS NULL;
				`, [userOrganizationId]);
            let newUserGroupIds = insertIdentity.userGroupIds.filter((c) => !existingGroupIds.some((e) => e.userGroupId == c));
            await trx.query(`
				INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupUserOrganizations}"
				(id, "createdAt", "updatedAt", "userGroupId", "userOrganizationId", "source", "sourceId")
				SELECT gen_random_uuid(), $1, $1, UNNEST($2::UUID[]), $3, $4, $5
				ON CONFLICT ("userOrganizationId", "userGroupId") WHERE "deletedAt" is null
				DO NOTHING;`, [opTimeWithOffset, newUserGroupIds, userOrganizationId, insertIdentity.source ?? dal_constants_1.DalConstants.SourceType.User, insertIdentity.sourceId]);
        }
        if (insertIdentity.accessRights?.length) {
            let remoteRightAcpIds = insertIdentity.accessRights.filter((a) => a.remoteAccess).map((a) => a.accessControlPointId);
            if (remoteRightAcpIds) {
                const { rows: remoteAvailableAcps } = await trx.query(`
					SELECT id
					FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessControlPoints}" 
					WHERE "remoteAvailable" = $1
					AND id = ANY($2::uuid[]);
					`, [true, remoteRightAcpIds]);
                for (const acpId of remoteRightAcpIds) {
                    if (!remoteAvailableAcps.some((a) => a.id == acpId)) {
                        (0, dal_access_error_1.throwDbAccessBadRequestErrorTr)("ERRORS.GENERAL.ACCESSRIGHTERROR");
                    }
                }
            }
            const { rows: existingAccessRightIds } = await trx.query(`
				SELECT "accessControlPointId"
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userAccessRights}"
				WHERE "userId" = $1
				AND "deletedAt" IS NULL;
				`, [userId]);
            let newAccessRightIds = insertIdentity.accessRights.filter((c) => !existingAccessRightIds.some((e) => e.accessControlPointId == c.accessControlPointId));
            for (const accessRight of newAccessRightIds) {
                await trx.query(`
					INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userAccessRights}"
					("id", "createdAt", "updatedAt", "userId", "accessControlPointId", "remoteAccess", "access", "read", "grant", "config", "snapshot")
					VALUES(uuid_generate_v4(), $1, $1, $2, $3, $4, $5, $6, $7, $8, $9);`, [
                    opTimeWithOffset,
                    userId,
                    accessRight.accessControlPointId,
                    accessRight.remoteAccess,
                    accessRight.access,
                    accessRight.read,
                    accessRight.grant,
                    accessRight.config,
                    accessRight.snapshot,
                ]);
            }
        }
        await dal_manager_1.dbManager.accessLog.addUserActionHistoryItemPg(organizationId, requestUserId, trx, {
            oId: organizationId,
            o: requestUserId,
            u: luxon_1.DateTime.fromSQL(opTimeWithOffset).toJSDate(),
            c: dal_constants_1.DalConstants.UserActionCategory.Main,
            tg: ["Identity"],
            t: dal_constants_1.DalConstants.UserMainActionType.InsertIdentity,
            d: {
                id: userId,
                data: JSON.stringify(insertIdentity),
            },
        });
        const result = await this.getIdentityDetailed(organizationId, requestUserId, {
            userId,
            hasOrganizationWideRight: true,
            hasIdentityFullWrite: false,
            permittedUnitIds: null,
            hasOrganizationWideRightToSeePassiveUsers: false,
            trx,
        });
        const transaction = (await trx.query(`SELECT txid_current() as "id", now()::timestamp without time zone::text as "actionT"`)).rows[0];
        if (notifyUsersForChange) {
            setTimeout(async () => {
                const offlineTerminals = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
                    return await dal_manager_1.dbManager.accessDevice.getOfflineTerminals({ organizationId: organizationId, transaction: transaction, trx: trx });
                }, requestUserId, organizationId);
                if (offlineTerminals.length > 0) {
                    (0, business_notification_1.sendDeviceNotInformedNotification)({ organizationId, offlineTerminals: offlineTerminals, receiverUserId: requestUserId });
                }
            }, app_config_1.appConfig.offlineTerminalNotificationIntervalSeconds * 1000);
        }
        return result;
    }
    async insertUserInfo(organizationId, requestUserId, opTimeWithOffset, trx, insertIdentity) {
        let username = null;
        let salt = null;
        let hashedPassword = null;
        const orgSettings = await (0, dal_memcache_1.getCacheOrganizationSettings)(organizationId, trx);
        if (insertIdentity.userAccount) {
            let hash = (0, dal_utils_1.saltHashPassword)(insertIdentity.userAccount.password);
            username = insertIdentity.userAccount.username;
            salt = hash.salt;
            hashedPassword = hash.passwordHash;
        }
        if (username) {
            const { rowCount } = await trx.query(`SELECT id FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}"
				WHERE lower(username) = lower($1)
				AND "deletedAt" IS NULL`, [username]);
            if (rowCount > 0) {
                (0, dal_access_error_1.throwDbAccessConflictErrorTr)("ERRORS.IDENTITY.USERNAMEEXIST");
            }
        }
        const { rowCount } = await trx.query(`SELECT uop.id FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" AS uop
            INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" AS uo 
			ON uo.id = uop."userOrganizationId"
            WHERE uop."deletedAt" IS NULL 
			AND uo."deletedAt" IS NULL 
			AND uop."uniqueId" = $1`, [insertIdentity.organizationProfile.uniqueId]);
        if (rowCount) {
            (0, dal_access_error_1.throwDbAccessConflictErrorTr)("ERRORS.IDENTITY.UNIQUEIDEXIST");
        }
        const userId = uuid_1.default.v4();
        await trx.query(`
			INSERT INTO "public"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userList}"
			(id, username)
			VALUES($1, $2);`, [userId, username]);
        await trx.query(`
			INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}"
			(id, "createdAt", "updatedAt", "accountEnabled", username, salt, "hashedPassword", settings)
			VALUES($1, $2, $2, $3, $4, $5, $6, $7);`, [userId, opTimeWithOffset, true, username, salt, hashedPassword, JSON.stringify({ locale: orgSettings.locale })]);
        const userProfileId = uuid_1.default.v4();
        await trx.query(`
			INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userProfiles}"
			(id, "createdAt", "updatedAt", "userId", name, surname, email, "phoneNumber", "mfaSettings")
			VALUES($1, $2, $2, $3, $4, $5, $6, $7, $8);`, [
            userProfileId,
            opTimeWithOffset,
            userId,
            insertIdentity.organizationProfile.name,
            insertIdentity.organizationProfile.surname,
            insertIdentity.organizationProfile.email,
            insertIdentity.organizationProfile.phoneNumber,
            JSON.stringify({
                enabledStatus: orgSettings.multiFactorAuthenticationSettings.option === app_enums_1.enums.MultifactorAuthenticationOption.DISABLE ||
                    orgSettings.multiFactorAuthenticationSettings.option === app_enums_1.enums.MultifactorAuthenticationOption.OPTIONAL
                    ? false
                    : true,
            }),
        ]);
        const userOrganizationId = uuid_1.default.v4();
        await trx.query(`
			INSERT INTO "public"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationMapping}"
			(id, "userId", "organizationId")
			VALUES($1, $2, $3);`, [userOrganizationId, userId, organizationId]);
        await trx.query(`
			INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}"
			(id, "createdAt", "updatedAt", "userId", "organizationId", "roleId", "isDisabled", "extraPermissions", settings)
			VALUES($1, $2, $2, $3, $4, $5, $6, $7, $8);`, [
            userOrganizationId,
            opTimeWithOffset,
            userId,
            organizationId,
            insertIdentity.organizationProfile.roleId,
            insertIdentity.organizationProfile.isDisabled,
            insertIdentity.organizationProfile.extraPermissions,
            JSON.stringify({
                notification: {
                    mediumSettings: (await (0, dal_memcache_1.getCacheOrganizationSettings)(organizationId, trx)).notification.mediumSettings,
                },
            }),
        ]);
        const userOrganizationProfileId = uuid_1.default.v4();
        await trx.query(`
			INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}"
			(id, "createdAt", "updatedAt", "userId", "userOrganizationId", name, surname, "phoneNumber", email, address, "uniqueId", "employmentStartUtc", 
				"employmentEndUtc", "previousServiceDuration", "extensionFields", "pacsEnabledRemainedAnnualPPermission", "birthDateUtc")
			VALUES($1, $2, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14, $15, $16);`, [
            userOrganizationProfileId,
            opTimeWithOffset,
            userId,
            userOrganizationId,
            insertIdentity.organizationProfile.name,
            insertIdentity.organizationProfile.surname,
            insertIdentity.organizationProfile.phoneNumber,
            insertIdentity.organizationProfile.email,
            insertIdentity.organizationProfile.address,
            insertIdentity.organizationProfile.uniqueId,
            insertIdentity.organizationProfile.employmentStartUtc,
            insertIdentity.organizationProfile.employmentEndUtc,
            insertIdentity.organizationProfile.previousServiceDuration,
            insertIdentity.organizationProfile.extensionFields,
            insertIdentity.organizationProfile.pacsEnabledRemainedAnnualPPermission ? insertIdentity.organizationProfile.pacsEnabledRemainedAnnualPPermission * 24 : null,
            insertIdentity.organizationProfile.birthDateUtc,
        ]);
        return Promise.resolve({
            userId: userId,
            userProfileId: userProfileId,
            userOrganizationId: userOrganizationId,
            userOrganizationProfileId: userOrganizationProfileId,
        });
    }
    async removeFromOrganizationUnitManager(organizationId, trx, params) {
        if (params.organizationUnitIds?.length) {
            await trx.query(`UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}"
				 SET "roleId"= $1
				 WHERE "userOrganizationId" = $2
				 AND   "organizationUnitId" = ANY($3::uuid[])
				 AND "deletedAt" IS NULL;`, [params.roles.standartUserId, params.userOrganizationId, params.organizationUnitIds]);
            await trx.query(`UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}"
				 SET "managerUserId"= $1,
				 	 "managerUserOrganizationId"= $2
				 WHERE   id = ANY($3::uuid[])
				 AND "deletedAt" IS NULL;`, [null, null, params.organizationUnitIds]);
        }
    }
    async promoteUserToOrganizationUnitManager(organizationId, trx, params) {
        if (params.organizationUnitIds?.length) {
            await trx.query(`UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}"
				 SET "roleId"= $1
				 WHERE   "organizationUnitId" = ANY($2::uuid[])
				 AND "roleId" = $3
				 AND "userOrganizationId" != $4
				 AND "deletedAt" IS NULL;`, [params.roles.standartUserId, params.organizationUnitIds, params.roles.unitAdministratorId, params.userOrganizationId]);
            await trx.query(`UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}"
				 SET "managerUserId"= $1,
				 	 "managerUserOrganizationId"= $2
				 WHERE id = ANY($3::uuid[])
				 AND "deletedAt" IS NULL;`, [params.userId, params.userOrganizationId, params.organizationUnitIds]);
        }
    }
    async enableUser(organizationId, userIds, trx) {
        const userIdArray = Array.isArray(userIds) ? userIds : [userIds];
        if ((await (0, dal_memcache_1.getCacheOrganizationSettings)(organizationId, trx)).passiveUserSettings?.keepCredentials) {
            const credentials = (await trx.query(`SELECT id, "extensionFields" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials}" 
					 WHERE "userId" = ANY($1::uuid[])`, [userIdArray])).rows;
            const credentialsToEnabled = credentials.filter((credential) => credential.extensionFields?.keepCredentialsOnUserDisable === true);
            await trx.query(`UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials}" 
				 SET "deletedAt" = null,
				 	"extensionFields" = ("extensionFields")::jsonb - 'keepCredentialsOnUserDisable',
					"updatedAt" = now()
				 WHERE id = ANY($1::uuid[])`, [credentialsToEnabled.map((credential) => credential.id)]);
        }
    }
    async disableUsers(organizationId, userIds, trx, isScheduled) {
        const userIdArray = Array.isArray(userIds) ? userIds : [userIds];
        const opTimeWithOffset = luxon_1.DateTime.now().toSQL({ includeOffset: true });
        for (const uid of userIdArray) {
            await dal_manager_1.dbManager.accessSystem.deleteScheduledJobsOfDisabledUser({ organizationId, userId: uid, trx });
            await dal_manager_1.dbManager.accessRedisCache.removeUserBadge(organizationId, uid);
            const oldIdentity = await this.getIdentityDetailsForDisableUser(organizationId, uid, trx);
            if (!oldIdentity)
                continue;
            const removedAccessControlPointIds = oldIdentity.accessRights?.map((ar) => ar.accessControlPointId);
            if (removedAccessControlPointIds?.length) {
                await trx.query(`
					UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userAccessRights}"
					SET "deletedAt" = $1, "updatedAt" = $1
					WHERE "userId" = $2
						AND "accessControlPointId" = ANY($3::uuid[])
						AND "deletedAt" IS NULL;
				   	`, [opTimeWithOffset, uid, removedAccessControlPointIds]);
            }
            const removedOrganizationUnitIds = oldIdentity.organizationUnits?.map((ou) => ou.organizationUnitId);
            if (removedOrganizationUnitIds?.length) {
                const roles = await dal_manager_1.dbManager.accessUser.getTypedRoleIdsOfOrganizationPg({ organizationId, trx });
                const { rows: managerRoles } = await trx.query(`
					SELECT "organizationUnitId"
					FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}"
					WHERE "userOrganizationId" = $1
						AND "roleId" = $2
						AND "organizationUnitId" = ANY($3::uuid[])
						AND "deletedAt" IS NULL;	
					`, [oldIdentity.userOrganizationId, roles.unitAdministratorId, removedOrganizationUnitIds]);
                await this.removeFromOrganizationUnitManager(organizationId, trx, {
                    userOrganizationId: oldIdentity.userOrganizationId,
                    organizationUnitIds: managerRoles.map((mr) => mr.organizationUnitId),
                    roles,
                });
                await trx.query(`
					UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}"
					SET "deletedAt" = $1, "updatedAt" = $1
					WHERE "userOrganizationId" = $2
						AND "organizationUnitId" = ANY($3::uuid[])
						AND "deletedAt" IS NULL;
					`, [opTimeWithOffset, oldIdentity.userOrganizationId, removedOrganizationUnitIds]);
            }
            const removedUserGroupIds = oldIdentity.userGroups?.map((ug) => ug.id);
            if (removedUserGroupIds?.length) {
                await trx.query(`
					UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupUserOrganizations}"
					SET "deletedAt" = $1, "updatedAt" = $1
					WHERE "userOrganizationId" = $2
						AND "userGroupId" = ANY($3::uuid[])
						AND "deletedAt" IS NULL;
					`, [opTimeWithOffset, oldIdentity.userOrganizationId, removedUserGroupIds]);
            }
        }
        const keepCredentials = (await (0, dal_memcache_1.getCacheOrganizationSettings)(organizationId, trx)).passiveUserSettings?.keepCredentials;
        let setSourceString = "";
        let qParams = [opTimeWithOffset, userIdArray];
        if (isScheduled) {
            let sourceParams = [dal_constants_1.DalConstants.SourceType.System];
            setSourceString = ', "source" = $3';
            qParams = [...qParams, ...sourceParams];
        }
        await trx.query(`
			UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials}" 
			SET "deletedAt" = $1,
				"updatedAt" = $1 ${setSourceString}
				${keepCredentials
            ? `,"extensionFields" = CASE WHEN "extensionFields" IS NOT NULL THEN ("extensionFields"::jsonb || jsonb_build_object('keepCredentialsOnUserDisable', true))::json ELSE JSON_BUILD_OBJECT('keepCredentialsOnUserDisable', true) END`
            : ""}
			WHERE "userId" = ANY($2::uuid[])
			AND "deletedAt" IS NULL
			`, qParams);
        await trx.query(`
				UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.oAuthTokens}"
				SET "deletedAt" = $1, reason = 'user marked passive'
				WHERE "userId" = ANY ($2::UUID[])
					AND "deletedAt" IS NULL
			`, [opTimeWithOffset, userIdArray]);
        await trx.query(`
                UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" SET "isDisabled" = true WHERE "isDisabled" = false AND "deletedAt" IS NULL AND "userId" = ANY($1);
            `, [userIdArray]);
        if ((await (0, dal_memcache_1.getCacheOrganizationSettings)(organizationId, trx)).passiveUserSettings.anonymize) {
            await trx.query(`UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}"
				SET "updatedAt" = $1,
					"name" = '*****',
					"uniqueId" = "userId",
					"surname" = '*****',
					"phoneNumber" = '*****',
					"email" = '*****',
					"address" = '*****',
					"employmentStartUtc" = NULL,
					"employmentEndUtc" = NULL,
					"previousServiceDuration" = 0,
					"pacsEnabledRemainedAnnualPPermission" = 0,
					"extensionFields" = '{}'::jsonb,
					"birthDateUtc" = NULL,
					"isAnonymized" = true
				WHERE "userId" = ANY($2) AND "deletedAt" IS NULL;`, [opTimeWithOffset, userIdArray]);
            await trx.query(`UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}"
				SET "accountEnabled" = false,
					username = NULL,
					salt = NULL,
					"hashedPassword" = NULL,
					"notificationToken" = NULL,
					"notificationTokenType" = NULL,
					"publicKey" = NULL
				WHERE id = ANY($1);`, [userIdArray]);
            await trx.query(`UPDATE public."${dal_db_armon_schema_1.ArmonSchema.tableNames.userList}"
				SET username = id::text WHERE id = ANY($1);`, [userIdArray]);
            new Promise(async (res, rej) => {
                const client = await dal_manager_1.dbManager.poolMain.connect();
                while (true) {
                    const logIds = await client.query(`SELECT id FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs}"
						WHERE log->>'o' = ANY($1) AND log->>'on' <> '***** *****'
						ORDER BY "actionUtc" ASC
						LIMIT 10
						FOR UPDATE;`, [userIdArray]);
                    if (logIds.rows.length > 0) {
                        await client.query(`UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs}"
							SET log = log || '{"on":"***** *****"}'
							WHERE id = ANY($1);
							`, [logIds.rows.map((r) => r.id)]);
                    }
                    else {
                        break;
                    }
                }
                await client.query(`DELETE FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.UserActionHistory}"
					WHERE category = $1 AND type = $2 AND log->'d'->>'id' = ANY($3)`, [dal_constants_1.DalConstants.UserActionCategory.Main, dal_constants_1.DalConstants.UserMainActionType.UpdateProfile, userIdArray]);
                client.release();
                res();
            }).then(() => {
                app_logs_1.logger.info("Access log & history anonymization completed for:");
                app_logs_1.logger.info(JSON.stringify(userIdArray));
            });
        }
    }
    async updateIdentityInfo(organizationId, requestUserId, trx, updateIdentity, oldIdentity) {
        const opTimeWithOffset = luxon_1.DateTime.fromJSDate(new Date()).toSQL({ includeOffset: true });
        const visitorRoleIds = [predefined_roles_1.PredefinedRoles.OrganizationUnitVisitor.id, predefined_roles_1.PredefinedRoles.OrganizationVisitor.id];
        try {
            const { rows: userRows, rowCount } = await trx.query(`
				SELECT u.id, u.username, uo.id AS "userOrganizationId", uo."roleId", uop."uniqueId", uop.name, uop.surname, uo."isDisabled"
				FROM "${organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.users} u 
				INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo
					ON u.id = uo."userId"
				INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" uop
					ON uo.id = uop."userOrganizationId"
				WHERE u.id = $1
				AND uop."deletedAt" IS NULL
				AND uo."deletedAt" IS NULL
				AND u."deletedAt" IS NULL;
			`, [updateIdentity.id]);
            if (rowCount === 0) {
                (0, dal_access_error_1.throwDbAccessConflictErrorTr)("ERRORS.IDENTITY.USERNOTFOUND");
            }
            const user = userRows[0];
            if ((visitorRoleIds.includes(user.roleId) && !visitorRoleIds.includes(updateIdentity.organizationProfile.roleId)) ||
                (!visitorRoleIds.includes(user.roleId) && visitorRoleIds.includes(updateIdentity.organizationProfile.roleId))) {
                if (rowCount === 0) {
                    (0, dal_access_error_1.throwDbAccessConflictErrorTr)("ERRORS.IDENTITY.USERVISITORSWITCH");
                }
            }
            if (updateIdentity.userAccount && !updateIdentity.userAccount.username && !updateIdentity.userAccount.password) {
                await trx.query(`UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}"
					SET "updatedAt"		= $1,
						"accountEnabled"= FALSE,
						username		= NULL,
						salt			= NULL,
						"hashedPassword"= NULL
					WHERE id = $2;`, [opTimeWithOffset, user.id]);
                await trx.query(`UPDATE "public"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userList}"
					SET username = NULL
					WHERE id = $1;`, [user.id]);
            }
            else if (updateIdentity.userAccount) {
                const { rowCount } = await trx.query(`SELECT id FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}"
					WHERE lower(username) = lower($1)
					AND id != $2
					AND "deletedAt" IS NULL;`, [updateIdentity.userAccount.username, updateIdentity.id]);
                if (rowCount > 0) {
                    (0, dal_access_error_1.throwDbAccessConflictErrorTr)("ERRORS.IDENTITY.USERNAMEEXIST");
                }
                const hash = (0, dal_utils_1.saltHashPassword)(updateIdentity.userAccount.password);
                await trx.query(`UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}"
					SET "updatedAt"		= $1,
						"accountEnabled"= $2,
						username		= $3,
						salt			= $4,
						"hashedPassword"= $5
					WHERE id = $6;`, [opTimeWithOffset, true, updateIdentity.userAccount.username, hash.salt, hash.passwordHash, user.id]);
                await trx.query(`UPDATE "public"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userList}"
					SET username = $1
					WHERE id = $2;`, [updateIdentity.userAccount.username, user.id]);
            }
            const roles = await dal_manager_1.dbManager.accessUser.getTypedRoleIdsOfOrganizationPg({ organizationId, trx });
            if (updateIdentity.removedOrganizationUnitIds?.length) {
                const { rows: managerRoles, rowCount } = await trx.query(`SELECT "organizationUnitId" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}"
					WHERE "userOrganizationId" = $1
					AND "roleId" = $2
					AND "organizationUnitId" = ANY($3::uuid[])
					AND "deletedAt" IS NULL;`, [user.userOrganizationId, roles.unitAdministratorId, updateIdentity.removedOrganizationUnitIds]);
                await this.removeFromOrganizationUnitManager(organizationId, trx, {
                    userOrganizationId: user.userOrganizationId,
                    organizationUnitIds: managerRoles.map((mr) => mr.organizationUnitId),
                    roles,
                });
                let setSourceString = "";
                let qParams = [opTimeWithOffset, user.userOrganizationId, updateIdentity.removedOrganizationUnitIds];
                if (updateIdentity.source) {
                    let sourceParams = [updateIdentity.source, updateIdentity.sourceId];
                    setSourceString = ', "source" = $4, "sourceId" = $5';
                    qParams = [...qParams, ...sourceParams];
                }
                await trx.query(`UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}"
					SET "deletedAt" = $1,
						"updatedAt" = $1 ${setSourceString}
					WHERE "userOrganizationId" = $2
					AND "organizationUnitId" = ANY($3::uuid[])
					AND "deletedAt" IS NULL;`, qParams);
            }
            if (updateIdentity.updatedOrganizationUnits?.length) {
                const { rows: managerRoles } = await trx.query(`SELECT "organizationUnitId" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}"
					WHERE "userOrganizationId" = $1
					AND "roleId" = $2
					AND "organizationUnitId" = ANY($3::uuid[])
					AND "deletedAt" IS NULL;`, [
                    user.userOrganizationId,
                    roles.unitAdministratorId,
                    updateIdentity.updatedOrganizationUnits.filter((ou) => ou.roleId !== roles.unitAdministratorId).map((ou) => ou.organizationUnitId),
                ]);
                await this.removeFromOrganizationUnitManager(organizationId, trx, {
                    userOrganizationId: user.userOrganizationId,
                    organizationUnitIds: managerRoles.map((mr) => mr.organizationUnitId),
                    roles,
                });
                await this.promoteUserToOrganizationUnitManager(organizationId, trx, {
                    userId: user.id,
                    userOrganizationId: user.userOrganizationId,
                    organizationUnitIds: updateIdentity.updatedOrganizationUnits.filter((uo) => uo.roleId === roles.unitAdministratorId).map((uo) => uo.organizationUnitId),
                    roles,
                });
                let updateQ = `
					UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}"
						SET "updatedAt" = $2,
							"roleId" = (sq->>'roleId')::UUID ,
							"extraPermissions" = sq->>'extraPermissions'
				`;
                const qParams = [user.userOrganizationId, opTimeWithOffset, updateIdentity.updatedOrganizationUnits];
                let fromQ = `
					FROM UNNEST ($3::JSONB[]) AS sq
					WHERE "deletedAt" IS NULL
						AND "userOrganizationId" = $1
						AND "organizationUnitId" = (sq->>'organizationUnitId')::UUID
				`;
                if (updateIdentity.source) {
                    updateQ + `, "source" = $4, "sourceId" = $5`;
                    qParams.push(updateIdentity.source, updateIdentity.sourceId);
                }
                await trx.query(updateQ + fromQ, qParams);
            }
            if (updateIdentity.addedOrganizationUnits?.length) {
                await trx.query(`
					INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}" 
						(id, "createdAt", "updatedAt","userOrganizationId", "organizationUnitId", "roleId", "extraPermissions", "source", "sourceId")
					SELECT gen_random_uuid(), $1, $1, $2, (sq->>'organizationUnitId')::UUID, (sq->>'roleId')::UUID, sq->>'extraPermissions', $4, $5
					FROM UNNEST ($3::JSONB[]) AS sq
					ON CONFLICT ("userOrganizationId", "organizationUnitId") WHERE "deletedAt" IS NULL
						DO UPDATE SET
							"roleId" = EXCLUDED."roleId", 
							"extraPermissions" = EXCLUDED."extraPermissions";
					`, [opTimeWithOffset, user.userOrganizationId, updateIdentity.addedOrganizationUnits, updateIdentity.source ?? dal_constants_1.DalConstants.SourceType.User, updateIdentity.sourceId]);
                await this.promoteUserToOrganizationUnitManager(organizationId, trx, {
                    userId: user.id,
                    userOrganizationId: user.userOrganizationId,
                    organizationUnitIds: updateIdentity.addedOrganizationUnits.filter((uo) => uo.roleId === roles.unitAdministratorId).map((uo) => uo.organizationUnitId),
                    roles,
                });
            }
            if (updateIdentity.addedUserGroupIds?.length) {
                await trx.query(`INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupUserOrganizations}" 
						(id, "createdAt", "updatedAt","deletedAt", "userGroupId", "userOrganizationId", "source", "sourceId")
					 SELECT uuid_generate_v4(), $1, $1, NULL, sq, $3, $4, $5
					 FROM UNNEST($2::UUID[]) as sq
					 ON CONFLICT ("userOrganizationId", "userGroupId") WHERE "deletedAt" is null
					DO UPDATE SET 
						"sourceId" = $5, 
						"source" = $4;`, [opTimeWithOffset, updateIdentity.addedUserGroupIds, user.userOrganizationId, updateIdentity.source ?? dal_constants_1.DalConstants.SourceType.User, updateIdentity.sourceId]);
            }
            if (updateIdentity.removedUserGroupIds?.length) {
                let setSourceString = "";
                let qParams = [opTimeWithOffset, user.userOrganizationId, updateIdentity.removedUserGroupIds];
                if (updateIdentity.source) {
                    let sourceParams = [updateIdentity.source, updateIdentity.sourceId];
                    setSourceString = ', "source" = $4, "sourceId" = $5';
                    qParams = [...qParams, ...sourceParams];
                }
                await trx.query(`UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupUserOrganizations}"
					SET "deletedAt" = $1,
						"updatedAt" = $1 ${setSourceString}
					WHERE "userOrganizationId" = $2
					AND "userGroupId" = ANY($3::uuid[]);`, qParams);
            }
            if (updateIdentity.removedCredentialIds?.length) {
                let setSourceString = "";
                let qParams = [opTimeWithOffset, user.id, updateIdentity.removedCredentialIds];
                if (updateIdentity.source) {
                    let sourceParams = [updateIdentity.source, updateIdentity.sourceId];
                    setSourceString = ', "source" = $4, "sourceId" = $5';
                    qParams = [...qParams, ...sourceParams];
                }
                await trx.query(`UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials}"
					SET "deletedAt" = $1,
						"updatedAt" = $1 ${setSourceString}
					WHERE "userId" = $2
					AND "deletedAt" IS NULL
					AND id = ANY($3::uuid[])
					AND	"deletedAt" IS NULL;`, qParams);
            }
            if (updateIdentity.addedCredentials?.length) {
                for (const credential of updateIdentity.addedCredentials) {
                    const { rowCount } = await trx.query(`SELECT id FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials}"
						WHERE type = $1
						AND "deletedAt" is NULL
						AND data = $2;`, [credential.type, credential.data]);
                    if (rowCount > 0) {
                        (0, dal_access_error_1.throwDbAccessConflictErrorTr)("ERRORS.IDENTITY.DUPLICATECREDENTIAL");
                    }
                    let extensionFields = credential.extensionFields;
                    await trx.query(`INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials}"
					(id, "createdAt", "updatedAt", "organizationId", "userId", type, data, "expiresOn", "credentialNumber", "extensionFields", "customFields", note, "groupNumber", "source", "sourceId")
					VALUES (uuid_generate_v4(), $1, $1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13);`, [
                        opTimeWithOffset,
                        organizationId,
                        user.id,
                        credential.type,
                        credential.data,
                        credential.expirationUtc,
                        credential.credentialNumber,
                        extensionFields,
                        credential.customFields,
                        credential.note,
                        credential.groupNumber,
                        updateIdentity.source ?? dal_constants_1.DalConstants.SourceType.User,
                        updateIdentity.sourceId,
                    ]);
                }
            }
            if (updateIdentity.updatedCredentials?.length) {
                for (const credential of updateIdentity.updatedCredentials) {
                    let setSourceString = "";
                    let qParams = [
                        credential.expirationUtc,
                        credential.credentialNumber,
                        credential.extensionFields,
                        credential.note,
                        credential.customFields,
                        credential.groupNumber,
                        credential.id,
                        opTimeWithOffset,
                    ];
                    if (updateIdentity.source) {
                        let sourceParams = [updateIdentity.source, updateIdentity.sourceId];
                        setSourceString = ', "source" = $9, "sourceId" = $10';
                        qParams = [...qParams, ...sourceParams];
                    }
                    await trx.query(`UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials}"
						SET "updatedAt" = $8,
							"expiresOn" = $1,
							"credentialNumber" = $2,
							"extensionFields" = $3,
							note = $4,
							"customFields" = $5,
							"groupNumber" = $6 ${setSourceString}
						WHERE "id" = $7`, qParams);
                }
            }
            if (updateIdentity.removedAccessControlPointIds?.length) {
                await trx.query(`UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userAccessRights}"
					SET "deletedAt" = $1,
						"updatedAt" = $1
					WHERE "userId" = $2
					AND	"accessControlPointId" = ANY($3::uuid[])
					AND	"deletedAt" IS NULL;`, [opTimeWithOffset, user.id, updateIdentity.removedAccessControlPointIds]);
            }
            if (updateIdentity.updatedAccessRights?.length) {
                const remoteRightAcpIds = updateIdentity.updatedAccessRights.filter((a) => a.remoteAccess).map((a) => a.accessControlPointId);
                if (remoteRightAcpIds) {
                    const { rows: remoteAvailableAcps } = await trx.query(`SELECT id FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessControlPoints}"
						WHERE "remoteAvailable" = $1
						AND id = ANY($2::uuid[]);`, [true, remoteRightAcpIds]);
                    for (const acpId of remoteRightAcpIds) {
                        if (!remoteAvailableAcps.some((a) => a.id == acpId)) {
                            (0, dal_access_error_1.throwDbAccessBadRequestErrorTr)("ERRORS.GENERAL.ACCESSRIGHTERROR");
                        }
                    }
                }
                for (const updatedAccessRight of updateIdentity.updatedAccessRights) {
                    await trx.query(`UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userAccessRights}"
						SET "updatedAt" = $1,
							"remoteAccess" = $2,
							"access" = $3,
							"config" = $4,
							"grant" = $5,
							"read" = $6,
							"snapshot" = $7
						WHERE "userId" = $8
						AND "accessControlPointId" = $9
						AND "deletedAt" IS NULL;`, [
                        opTimeWithOffset,
                        updatedAccessRight.remoteAccess,
                        updatedAccessRight.access,
                        updatedAccessRight.config,
                        updatedAccessRight.grant,
                        updatedAccessRight.read,
                        updatedAccessRight.snapshot,
                        user.id,
                        updatedAccessRight.accessControlPointId,
                    ]);
                }
            }
            if (updateIdentity.addedAccessRights?.length) {
                let remoteRightAcpIds = updateIdentity.addedAccessRights.filter((a) => a.remoteAccess).map((a) => a.accessControlPointId);
                if (remoteRightAcpIds) {
                    const { rows: remoteAvailableAcps } = await trx.query(`SELECT id FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessControlPoints}"
						WHERE "remoteAvailable" = $1
						AND id = ANY($2::uuid[]);`, [true, remoteRightAcpIds]);
                    for (const acpId of remoteRightAcpIds) {
                        if (!remoteAvailableAcps.some((a) => a.id == acpId)) {
                            (0, dal_access_error_1.throwDbAccessBadRequestErrorTr)("ERRORS.GENERAL.ACCESSRIGHTERROR");
                        }
                    }
                }
                for (const newAccessRight of updateIdentity.addedAccessRights) {
                    await trx.query(`INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userAccessRights}" 
						("id", "createdAt", "updatedAt", "userId", "accessControlPointId", "remoteAccess", "access", "config", "grant", "read", "snapshot")
						VALUES (uuid_generate_v4(), $1, $1, $2, $3, $4, $5, $6, $7, $8, $9);`, [
                        opTimeWithOffset,
                        user.id,
                        newAccessRight.accessControlPointId,
                        newAccessRight.remoteAccess,
                        newAccessRight.access,
                        newAccessRight.config,
                        newAccessRight.grant,
                        newAccessRight.read,
                        newAccessRight.snapshot,
                    ]);
                }
            }
            let userEmploymentEnded = !user.isDisabled &&
                updateIdentity.organizationProfile &&
                updateIdentity.organizationProfile.employmentEndUtc &&
                (0, moment_1.default)(updateIdentity.organizationProfile.employmentEndUtc).isBefore((0, moment_1.default)(), "day");
            if (userEmploymentEnded) {
                updateIdentity.organizationProfile.isDisabled = true;
            }
            await trx.query(`UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}"
					SET "updatedAt" = $1,
						"roleId" = $2,
						"isDisabled" = $3,
						"extraPermissions" = $4
					WHERE "userId" = $5
					AND	"deletedAt" IS NULL;`, [
                opTimeWithOffset,
                updateIdentity.organizationProfile.roleId,
                updateIdentity.organizationProfile.isDisabled,
                updateIdentity.organizationProfile.extraPermissions,
                updateIdentity.id,
            ]);
            const userOrganizationProfileFields = {
                name: updateIdentity.organizationProfile.name,
                uniqueId: updateIdentity.organizationProfile.uniqueId,
                surname: updateIdentity.organizationProfile.surname,
                phoneNumber: updateIdentity.organizationProfile.phoneNumber,
                email: updateIdentity.organizationProfile.email,
                address: updateIdentity.organizationProfile.address,
                employmentStartUtc: updateIdentity.organizationProfile.employmentStartUtc,
                employmentEndUtc: updateIdentity.organizationProfile.employmentEndUtc,
                previousServiceDuration: updateIdentity.organizationProfile.previousServiceDuration,
                extensionFields: oldIdentity.organizationProfile.extensionFields ?? {},
                pacsEnabledRemainedAnnualPPermission: updateIdentity.organizationProfile.pacsEnabledRemainedAnnualPPermission
                    ? updateIdentity.organizationProfile.pacsEnabledRemainedAnnualPPermission * 24
                    : undefined,
                birthDateUtc: updateIdentity.organizationProfile.birthDateUtc,
            };
            if (updateIdentity.organizationProfile?.extensionFields) {
                Object.entries(updateIdentity.organizationProfile.extensionFields).forEach(([key, value]) => {
                    userOrganizationProfileFields.extensionFields[key] = value;
                });
            }
            const fieldsToUpdate = Object.entries(userOrganizationProfileFields)
                .filter(([_, value]) => value !== undefined)
                .map(([key, _], index) => `"${key}" = $${index + 1}`);
            const updateValues = Object.values(userOrganizationProfileFields).filter((value) => value !== undefined);
            fieldsToUpdate.push(`"updatedAt" = $${updateValues.length + 1}`);
            updateValues.push(opTimeWithOffset);
            updateValues.push(updateIdentity.id, user.userOrganizationId);
            const query = `
					UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}"
					SET ${fieldsToUpdate.join(", ")}
					WHERE "userId" = $${updateValues.length - 1}
					AND "userOrganizationId" = $${updateValues.length}
					AND "deletedAt" IS NULL;
				`;
            await trx.query(query, updateValues);
            if (userEmploymentEnded || (!user.isDisabled && updateIdentity.organizationProfile.isDisabled)) {
                await this.disableUsers(organizationId, user.id, trx);
            }
            else if (user.isDisabled && !updateIdentity.organizationProfile.isDisabled) {
                await this.enableUser(organizationId, user.id, trx);
            }
        }
        catch (error) {
            if (error.message === app_enums_1.enums.ErrorCode.OneTimeAssignableCredential.toString()) {
                (0, dal_access_error_1.throwDbAccessConflictErrorTr)("ERRORS.IDENTITY.NONRETURNABLE_CREDENTIAL");
            }
            else if (error.message === app_enums_1.enums.ErrorCode.CredentialToInsertAlreadyExists.toString()) {
                (0, dal_access_error_1.throwDbAccessConflictErrorTr)("ERRORS.IDENTITY.DUPLICATECREDENTIAL");
            }
            else if (error.message.includes("prevent_terminal_has_no_write_admin")) {
                const terminalIds = error.message.substring(error.message.indexOf("{") + 1, error.message.indexOf("}")).split(",");
                const terminalNames = (await dal_manager_1.dbManager.accessDevice.getDeviceBasics(organizationId, terminalIds)).map((terminal) => terminal.name).join(", ");
                (0, dal_access_error_1.throwDbAccessBadRequestErrorTr)("ERRORS.IDENTITY.PREVENT_TERMINAL_HAS_NO_WRITE_ADMIN_ON_USER_ROLE_CHANGE", { terminalNames }, false, true);
            }
            else if ((error?.message).startsWith("20002,")) {
                const credentialData = error.message.split(",")[1];
                const details = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
                    return this.getCredentialOwnerDetails({ organizationId, credentialData, trx });
                }, requestUserId, organizationId);
                if (details.userFullname) {
                    (0, dal_access_error_1.throwDbAccessConflictErrorTr)("ERRORS.IDENTITY.DUPLICATE_CREDENTIAL_USER", { fullname: details.userFullname, data: details.data });
                }
                else {
                    (0, dal_access_error_1.throwDbAccessConflictErrorTr)("ERRORS.IDENTITY.DUPLICATE_CREDENTIAL_VISITOR", { fullname: details.visitorFullname, data: details.data });
                }
            }
            else {
                throw error;
            }
        }
    }
    async getIdentityDetailed(organizationId, requesterUserId, params) {
        const transactionScope = async (trx) => {
            const userOrganization = (await trx.query(`
						SELECT "id", "roleId", "isDisabled"
						FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}"
						WHERE "userId" = $1 AND "deletedAt" IS NULL
						LIMIT 1;
					`, [params.userId])).rows[0];
            if (!userOrganization)
                (0, dal_access_error_1.throwDbAccessNotFoundError)("User is not found");
            if (userOrganization.isDisabled && params.hasOrganizationWideRightToSeePassiveUsers) {
                params.hasOrganizationWideRight = true;
            }
            let requireManagables = !params.hasIdentityFullWrite;
            if (!params.hasOrganizationWideRight) {
                if (userOrganization.isDisabled) {
                    (0, dal_access_error_1.throwDbAccessBadRequestErrorTr)("ERRORS.GENERAL.PASSIVE_USER_PERMISSION");
                }
                if (!(await dal_manager_1.dbManager.accessUser.checkUserUnderHierarchyOfAnotherUser(organizationId, requesterUserId, params.userId, trx)).length) {
                    (0, dal_access_error_1.throwDbAccessNotFoundError)("You do not have permissions to get details of that user");
                }
            }
            const organizationProfile = (await trx.query(`
						SELECT *
						FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}"
						WHERE "deletedAt" IS NULL AND
							"userOrganizationId" = $1
						LIMIT 1;
					`, [userOrganization.id])).rows[0];
            let userOrganizationProfile = {
                roleId: userOrganization.roleId,
                isDisabled: userOrganization.isDisabled,
                uniqueId: null,
                name: null,
                surname: null,
                email: null,
                phoneNumber: null,
                address: null,
                extensionFields: null,
                employmentStartUtc: null,
                employmentEndUtc: null,
                previousServiceDuration: 0,
                pacsEnabledRemainedAnnualPPermission: 0,
                locale: "tr",
                isAnonymized: false,
            };
            if (organizationProfile) {
                userOrganizationProfile.uniqueId = organizationProfile.uniqueId;
                userOrganizationProfile.name = organizationProfile.name;
                userOrganizationProfile.surname = organizationProfile.surname;
                userOrganizationProfile.email = organizationProfile.email;
                userOrganizationProfile.phoneNumber = organizationProfile.phoneNumber;
                userOrganizationProfile.address = organizationProfile.address;
                userOrganizationProfile.extensionFields = organizationProfile.extensionFields;
                userOrganizationProfile.employmentStartUtc = organizationProfile.employmentStartUtc;
                userOrganizationProfile.employmentEndUtc = organizationProfile.employmentEndUtc;
                userOrganizationProfile.previousServiceDuration = organizationProfile.previousServiceDuration;
                userOrganizationProfile.pacsEnabledRemainedAnnualPPermission = organizationProfile.pacsEnabledRemainedAnnualPPermission
                    ? organizationProfile.pacsEnabledRemainedAnnualPPermission / 24
                    : null;
                userOrganizationProfile.birthDateUtc = organizationProfile.birthDateUtc;
                userOrganizationProfile.isAnonymized = organizationProfile.isAnonymized;
            }
            let qx = 1;
            let qParams = [];
            let q = `
				SELECT uoou."organizationUnitId", uoou."roleId", ou."name", ou."typeId", oudlv.value AS "typeDefinition", uoou."source", uoou."sourceId",
					CASE WHEN ou."ancestorIds" = '[]' OR ou."ancestorIds" = '' OR ou."ancestorIds" IS NULL THEN ARRAY[]::UUID[]
						ELSE REGEXP_SPLIT_TO_ARRAY(ou."ancestorIds", E',')::UUID[] END AS "ancestorIds",
					ARRAY(
						SELECT "accessControlPointId" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnitDefaultAccessControlPoints}" 
							WHERE "organizationUnitId" = ou.id AND "deletedAt" IS NULL
						) AS "defaultAccessControlPointIds"	
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}" AS uoou
				INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" AS ou
					ON ou.id = uoou."organizationUnitId"
				INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUserDefinedListValues}" AS oudlv
					ON oudlv."id" = ou."typeId"
				WHERE ou."deletedAt" IS NULL
					AND uoou."deletedAt" IS NULL
					AND uoou."userOrganizationId" = $${qx++}
			`;
            qParams.push(userOrganization.id);
            if (!params.hasIdentityFullWrite && params.permittedUnitIds?.length) {
                let permittedOrganizationUnitIds = [];
                let units = (await dal_manager_1.dbManager.accessOrganizationUnit.getAllSuccessorOrganizationUnitsBasic(organizationId, params.permittedUnitIds, trx)).map((organizationUnit) => organizationUnit.id);
                permittedOrganizationUnitIds.push(...units, ...params.permittedUnitIds);
                permittedOrganizationUnitIds = [...new Set(permittedOrganizationUnitIds)];
                q += `
					AND uoou."organizationUnitId" = ANY ($${qx++}::UUID[])
				`;
                qParams.push(permittedOrganizationUnitIds);
            }
            q += `
				GROUP BY uoou."organizationUnitId", uoou."roleId", ou."id", oudlv."value", uoou."source", uoou."sourceId"
			`;
            const userOrganizationUnits = (await trx.query(q, qParams)).rows;
            let requesterAcp = [];
            if (requireManagables) {
                requesterAcp = (await trx.query(`
							SELECT acp.id
							FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userAccessRights}" AS uar
							INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessControlPoints}" AS acp
								ON acp.id = uar."accessControlPointId"
							WHERE uar."deletedAt" IS NULL
								AND acp."deletedAt" IS NULL
								AND uar."userId" = $1
						`, [requesterUserId])).rows.map((r) => r.id);
            }
            const userOrganizationAccessRights = (await trx.query(`
					SELECT acp.id as "accessControlPointId",
						uar."id",
						acp."name",
						uar."remoteAccess",
						acp."remoteAvailable",
						acp."isRemoteDefault",
						acp."accessControlPointType",
						acp."deviceId",
						uar."access",
						uar."read",
						uar."grant",
						uar."config",
						uar."snapshot",
						uar."remoteAccess",
						(CASE
							WHEN NOT $1 THEN TRUE
							WHEN acp."id" = ANY($2:: UUID[]) THEN TRUE
							ELSE FALSE
							END) AS "managable"
					FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userAccessRights}" AS uar
					INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessControlPoints}" AS acp
						ON acp.id = uar."accessControlPointId"
					WHERE uar."deletedAt" IS NULL
						AND acp."deletedAt" IS NULL
						AND uar."userId" = $3
			`, [requireManagables, requesterAcp, params.userId])).rows;
            const userOrganizationCredentials = (await trx.query(`
				SELECT id, "type", data, "credentialNumber", "expiresOn" AS "expirationUtc", "extensionFields", "source", "sourceId",
					"customFields", "note", "specialDataSecondary", "groupNumber"
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials}"
				WHERE "userId" = $1 AND "deletedAt" IS NULL
			`, [params.userId])).rows;
            const userGroups = (await trx.query(`
						SELECT ug.name, ug."colorCode", ug.id, uguo."source", uguo."sourceId"
						FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroups}" AS ug
						INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupUserOrganizations}" AS uguo
							ON ug.id = uguo."userGroupId" AND uguo."deletedAt" IS NULL
						INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" AS uo
							ON uo.id = uguo."userOrganizationId" AND uo."deletedAt" IS NULL
						WHERE ug."deletedAt" IS NULL
							AND uo."userId" = $1
					`, [params.userId])).rows;
            const user = (await trx.query(`
						SELECT * 
						FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}"
						WHERE id = $1
						LIMIT 1;
					`, [params.userId])).rows[0];
            const identityDetailed = {
                id: params.userId,
                username: user.username,
                publicKey: user.publicKey,
                organizationProfile: userOrganizationProfile,
                organizationUnits: userOrganizationUnits.map((u) => {
                    return {
                        organizationUnitId: u.organizationUnitId,
                        name: u.name,
                        ancestorIds: u.ancestorIds,
                        typeId: u.typeId,
                        typeDefinition: u.typeDefinition,
                        defaultAccessControlPointIds: u.defaultAccessControlPointIds,
                        roleId: u.roleId,
                        source: u.source,
                        sourceId: u.sourceId,
                    };
                }),
                accessRights: userOrganizationAccessRights.map((r) => {
                    return {
                        id: r.id,
                        accessControlPointId: r.accessControlPointId,
                        accessControlPointName: r.name,
                        remoteAccess: r.remoteAccess,
                        accessControlPointRemoteAvailable: r.remoteAvailable,
                        accessControlPointIsRemoteDefault: r.isRemoteDefault,
                        managable: r.managable,
                        isControlPanelSync: null,
                        deviceId: r.deviceId,
                        access: r.access,
                        read: r.read,
                        config: r.config,
                        grant: r.grant,
                        snapshot: r.snapshot,
                    };
                }),
                credentials: userOrganizationCredentials.map((c) => {
                    return {
                        id: c.id,
                        type: c.type,
                        data: c.data,
                        credentialNumber: c.credentialNumber,
                        expirationUtc: c.expirationUtc,
                        extensionFields: c.extensionFields,
                        customFields: c.customFields,
                        note: c.note,
                        specialDataSecondary: c.specialDataSecondary,
                        groupNumber: c.groupNumber,
                        source: c.source,
                        sourceId: c.sourceId,
                    };
                }),
                userGroups: userGroups.map((u) => {
                    return {
                        id: u.id,
                        name: u.name,
                        colorCode: u.colorCode,
                        source: u.source,
                        sourceId: u.sourceId,
                    };
                }),
            };
            return identityDetailed;
        };
        let identityDetailed = null;
        if (params.trx) {
            identityDetailed = await transactionScope(params.trx);
        }
        else {
            identityDetailed = await dal_manager_1.dbManager.organizationTransaction(transactionScope, requesterUserId, organizationId);
        }
        return identityDetailed;
    }
    async getIdentityDetailsForDisableUser(organizationId, userId, trx) {
        const transactionScope = async (trx) => {
            const userOrganization = (await trx.query(`
						SELECT "id", "roleId", "isDisabled"
						FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}"
						WHERE "userId" = $1 AND "deletedAt" IS NULL
						LIMIT 1;
					`, [userId])).rows[0];
            if (!userOrganization)
                (0, dal_access_error_1.throwDbAccessNotFoundError)("User is not found");
            const userOrganizationUnits = (await trx.query(`
						SELECT uoou."organizationUnitId", uoou."roleId", ou."name", ou."typeId", oudlv.value AS "typeDefinition",
							CASE WHEN ou."ancestorIds" = '[]' OR ou."ancestorIds" = '' OR ou."ancestorIds" IS NULL THEN ARRAY[]::UUID[]
								ELSE REGEXP_SPLIT_TO_ARRAY(ou."ancestorIds", E',')::UUID[] END AS "ancestorIds",
							ARRAY(
								SELECT "accessControlPointId" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnitDefaultAccessControlPoints}" 
									WHERE "organizationUnitId" = ou.id AND "deletedAt" IS NULL
								) AS "defaultAccessControlPointIds"	
						FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}" AS uoou
						INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" AS ou
							ON ou.id = uoou."organizationUnitId"
						INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUserDefinedListValues}" AS oudlv
							ON oudlv."id" = ou."typeId"
						WHERE ou."deletedAt" IS NULL
							AND uoou."deletedAt" IS NULL
							AND uoou."userOrganizationId" = $1
						GROUP BY uoou."organizationUnitId", uoou."roleId", ou."id", oudlv."value"
					`, [userOrganization.id])).rows;
            const userOrganizationAccessRights = (await trx.query(`
					SELECT acp.id as "accessControlPointId",
						uar."id",
						acp."name",
						uar."remoteAccess",
						acp."remoteAvailable",
						acp."isRemoteDefault",
						acp."accessControlPointType",
						acp."deviceId",
						uar."access",
						uar."read",
						uar."grant",
						uar."config",
						uar."snapshot",
						uar."remoteAccess"
					FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userAccessRights}" AS uar
					INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessControlPoints}" AS acp
						ON acp.id = uar."accessControlPointId"
					WHERE uar."deletedAt" IS NULL
						AND acp."deletedAt" IS NULL
						AND uar."userId" = $1
					`, [userId])).rows;
            const userGroups = (await trx.query(`
						SELECT ug.name, ug."colorCode", ug.id
						FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroups}" AS ug
						INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupUserOrganizations}" AS uguo
							ON ug.id = uguo."userGroupId" AND uguo."deletedAt" IS NULL
						INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" AS uo
							ON uo.id = uguo."userOrganizationId" AND uo."deletedAt" IS NULL
						WHERE ug."deletedAt" IS NULL
							AND uo."userId" = $1
					`, [userId])).rows;
            const identityDetailed = {
                id: userId,
                userOrganizationId: userOrganization.id,
                organizationUnits: userOrganizationUnits.map((u) => {
                    return {
                        organizationUnitId: u.organizationUnitId,
                        name: u.name,
                        ancestorIds: u.ancestorIds,
                        typeId: u.typeId,
                        typeDefinition: u.typeDefinition,
                        defaultAccessControlPointIds: u.defaultAccessControlPointIds,
                        roleId: u.roleId,
                    };
                }),
                accessRights: userOrganizationAccessRights.map((r) => {
                    return {
                        id: r.id,
                        accessControlPointId: r.accessControlPointId,
                        accessControlPointName: r.name,
                        remoteAccess: r.remoteAccess,
                        accessControlPointRemoteAvailable: r.remoteAvailable,
                        accessControlPointIsRemoteDefault: r.isRemoteDefault,
                        managable: r.managable,
                        isControlPanelSync: null,
                        deviceId: r.deviceId,
                        access: r.access,
                        read: r.read,
                        config: r.config,
                        grant: r.grant,
                        snapshot: r.snapshot,
                    };
                }),
                userGroups: userGroups.map((u) => {
                    return {
                        id: u.id,
                        name: u.name,
                        colorCode: u.colorCode,
                    };
                }),
            };
            return identityDetailed;
        };
        let identityDetails = null;
        if (trx) {
            identityDetails = await transactionScope(trx);
        }
        else {
            identityDetails = await dal_manager_1.dbManager.systemTransaction(transactionScope);
        }
        return identityDetails;
    }
    async logProfileUpdate(organizationId, requestUserId, oldIdentity, newIdentity) {
        let opTime = new Date();
        let roles = await dal_manager_1.dbManager.accessUser.listRolesOfOrganization(organizationId);
        let changedItems = [];
        Object.keys(newIdentity.organizationProfile).forEach((key) => {
            if (key == "extensionFields") {
                if (newIdentity.organizationProfile.extensionFields && oldIdentity.organizationProfile.extensionFields) {
                    Object.keys(newIdentity.organizationProfile.extensionFields).forEach((extensionKey) => {
                        if (newIdentity.organizationProfile.extensionFields[extensionKey] !== oldIdentity.organizationProfile.extensionFields[extensionKey]) {
                            if (newIdentity.organizationProfile.extensionFields[extensionKey] instanceof Array) {
                                if (!oldIdentity.organizationProfile.extensionFields[extensionKey]) {
                                    changedItems.push({
                                        f: key,
                                        o: null,
                                        n: newIdentity.organizationProfile.extensionFields[extensionKey],
                                        i: false,
                                    });
                                }
                                else {
                                    for (let index = 0; index < newIdentity.organizationProfile.extensionFields[extensionKey]?.length; index++) {
                                        if (newIdentity.organizationProfile.extensionFields[extensionKey][index] !== oldIdentity.organizationProfile.extensionFields[extensionKey][index]) {
                                            changedItems.push({
                                                f: key,
                                                o: oldIdentity.organizationProfile.extensionFields[extensionKey],
                                                n: newIdentity.organizationProfile.extensionFields[extensionKey],
                                                i: false,
                                            });
                                            break;
                                        }
                                    }
                                }
                            }
                            else {
                                changedItems.push({
                                    f: key,
                                    o: oldIdentity.organizationProfile.extensionFields[extensionKey],
                                    n: newIdentity.organizationProfile.extensionFields[extensionKey],
                                    i: false,
                                });
                            }
                        }
                    });
                }
            }
            else {
                if (newIdentity.organizationProfile[key] instanceof Date || oldIdentity.organizationProfile[key] instanceof Date) {
                    let oldDate = new Date(oldIdentity.organizationProfile[key]);
                    let newDate = new Date(newIdentity.organizationProfile[key]);
                    if (oldDate.getTime() !== newDate.getTime())
                        changedItems.push({
                            f: key,
                            o: (0, moment_1.default)(oldDate).format("YYYY-MM-DD"),
                            n: (0, moment_1.default)(newDate).format("YYYY-MM-DD"),
                            i: true,
                        });
                }
                else if (newIdentity.organizationProfile[key] != oldIdentity.organizationProfile[key]) {
                    if (key === "roleId") {
                        changedItems.push({
                            f: key,
                            o: roles.find((r) => r.id === oldIdentity.organizationProfile[key]).name,
                            n: roles.find((r) => r.id === newIdentity.organizationProfile[key]).name,
                            i: true,
                        });
                    }
                    else {
                        changedItems.push({
                            f: key,
                            o: oldIdentity.organizationProfile[key],
                            n: newIdentity.organizationProfile[key],
                            i: true,
                        });
                    }
                }
            }
        });
        if (changedItems.length > 0) {
            await dal_manager_1.dbManager.accessLog.addUserActionHistoryItem({
                oId: organizationId,
                o: requestUserId,
                u: opTime,
                c: dal_constants_1.DalConstants.UserActionCategory.Main,
                tg: ["Identity"],
                t: dal_constants_1.DalConstants.UserMainActionType.UpdateProfile,
                d: {
                    id: newIdentity.id,
                    c: JSON.stringify(changedItems),
                },
            });
        }
        changedItems = [];
        let addedOrganizationUnits = newIdentity.organizationUnits.filter((ou) => !oldIdentity.organizationUnits.map((oldou) => oldou.organizationUnitId).includes(ou.organizationUnitId));
        for (const addedUnit of addedOrganizationUnits) {
            changedItems.push({
                f: "organizationUnitName",
                o: null,
                n: addedUnit.name,
                i: true,
            });
        }
        if (changedItems.length > 0) {
            await dal_manager_1.dbManager.accessLog.addUserActionHistoryItem({
                oId: organizationId,
                o: requestUserId,
                u: opTime,
                c: dal_constants_1.DalConstants.UserActionCategory.Main,
                tg: ["Identity", "OrganizationUnit"],
                t: dal_constants_1.DalConstants.UserMainActionType.AssignUserToOrganizationUnit,
                d: {
                    id: newIdentity.id,
                    c: JSON.stringify(changedItems),
                },
            });
        }
        changedItems = [];
        let removedOrganizationUnits = oldIdentity.organizationUnits.filter((ou) => !newIdentity.organizationUnits.map((newou) => newou.organizationUnitId).includes(ou.organizationUnitId));
        for (const removedUnit of removedOrganizationUnits) {
            changedItems.push({
                f: "organizationUnitName",
                o: removedUnit.name,
                n: null,
                i: true,
            });
        }
        if (changedItems.length > 0) {
            await dal_manager_1.dbManager.accessLog.addUserActionHistoryItem({
                oId: organizationId,
                o: requestUserId,
                u: opTime,
                c: dal_constants_1.DalConstants.UserActionCategory.Main,
                tg: ["Identity", "OrganizationUnit"],
                t: dal_constants_1.DalConstants.UserMainActionType.RemoveUserFromOrganizationUnit,
                d: {
                    id: newIdentity.id,
                    c: JSON.stringify(changedItems),
                },
            });
        }
        changedItems = [];
        for (const unit of newIdentity.organizationUnits) {
            let oldUnit = oldIdentity.organizationUnits.find((ou) => ou.organizationUnitId === unit.organizationUnitId);
            if (!oldUnit || oldUnit.roleId === unit.roleId)
                continue;
            else {
                changedItems.push({
                    f: unit.name,
                    o: roles.find((r) => r.id === oldUnit.roleId).name,
                    n: roles.find((r) => r.id === unit.roleId).name,
                    i: true,
                });
            }
        }
        if (changedItems.length > 0) {
            await dal_manager_1.dbManager.accessLog.addUserActionHistoryItem({
                oId: organizationId,
                o: requestUserId,
                u: opTime,
                c: dal_constants_1.DalConstants.UserActionCategory.Main,
                tg: ["Identity", "OrganizationUnit"],
                t: dal_constants_1.DalConstants.UserMainActionType.UpdateUserOrganizationUnitRole,
                d: {
                    id: newIdentity.id,
                    c: JSON.stringify(changedItems),
                },
            });
        }
        changedItems = [];
        let addedUserGroups = newIdentity.userGroups.filter((ug) => !oldIdentity.userGroups.map((oldug) => oldug.id).includes(ug.id));
        for (const addedGroup of addedUserGroups) {
            changedItems.push({
                f: "userGroupName",
                o: null,
                n: addedGroup.name,
                i: true,
            });
        }
        if (changedItems.length > 0) {
            await dal_manager_1.dbManager.accessLog.addUserActionHistoryItem({
                oId: organizationId,
                o: requestUserId,
                u: opTime,
                c: dal_constants_1.DalConstants.UserActionCategory.Main,
                tg: ["Identity", "OrganizationUnit"],
                t: dal_constants_1.DalConstants.UserMainActionType.AssignUserToUserGroup,
                d: {
                    id: newIdentity.id,
                    c: JSON.stringify(changedItems),
                },
            });
        }
        changedItems = [];
        let removedUserGroups = oldIdentity.userGroups.filter((ug) => !newIdentity.userGroups.map((newug) => newug.id).includes(ug.id));
        for (const removedGroup of removedUserGroups) {
            changedItems.push({
                f: "userGroupName",
                o: removedGroup.name,
                n: null,
                i: true,
            });
        }
        if (changedItems.length > 0) {
            await dal_manager_1.dbManager.accessLog.addUserActionHistoryItem({
                oId: organizationId,
                o: requestUserId,
                u: opTime,
                c: dal_constants_1.DalConstants.UserActionCategory.Main,
                tg: ["Identity", "OrganizationUnit"],
                t: dal_constants_1.DalConstants.UserMainActionType.RemoveUserFromUserGroup,
                d: {
                    id: newIdentity.id,
                    c: JSON.stringify(changedItems),
                },
            });
        }
        let addedAccessRights = newIdentity.accessRights.filter((ar) => !oldIdentity.accessRights.map((oldar) => oldar.id).includes(ar.id));
        if (addedAccessRights.length > 0) {
            await dal_manager_1.dbManager.accessLog.addUserActionHistoryItem({
                oId: organizationId,
                o: requestUserId,
                u: opTime,
                c: dal_constants_1.DalConstants.UserActionCategory.Main,
                tg: ["Identity", "AccessRight"],
                t: dal_constants_1.DalConstants.UserMainActionType.AddAccessRight,
                d: {
                    id: newIdentity.id,
                    accessRightId: addedAccessRights.map((aar) => aar.id),
                },
            });
        }
        let removedAccessRights = oldIdentity.accessRights.filter((ar) => !newIdentity.accessRights.map((newar) => newar.id).includes(ar.id));
        if (removedAccessRights.length > 0) {
            await dal_manager_1.dbManager.accessLog.addUserActionHistoryItem({
                oId: organizationId,
                o: requestUserId,
                u: opTime,
                c: dal_constants_1.DalConstants.UserActionCategory.Main,
                tg: ["Identity", "AccessRight"],
                t: dal_constants_1.DalConstants.UserMainActionType.RemoveAccessRight,
                d: {
                    id: newIdentity.id,
                    removedAccessRightIds: removedAccessRights.map((rar) => rar.accessControlPointId),
                },
            });
        }
        let addedCredentials = newIdentity.credentials.filter((nc) => !oldIdentity.credentials.map((oldc) => oldc.id).includes(nc.id));
        for (const addedCredential of addedCredentials) {
            await dal_manager_1.dbManager.accessLog.addUserActionHistoryItem({
                oId: organizationId,
                o: requestUserId,
                u: opTime,
                c: dal_constants_1.DalConstants.UserActionCategory.Main,
                tg: ["Identity", "Credential"],
                t: dal_constants_1.DalConstants.UserMainActionType.AddCredential,
                d: {
                    id: newIdentity.id,
                    credentialId: addedCredential.id,
                    c: JSON.stringify([
                        {
                            f: addedCredential.type,
                            o: null,
                            n: dal_constants_1.DalConstants.credentialTypesWithMeaningfulData.includes(addedCredential.type) ? addedCredential.data : "*****",
                            i: true,
                        },
                    ]),
                },
            });
        }
        let removedCredentials = oldIdentity.credentials.filter((oc) => !newIdentity.credentials.map((newc) => newc.id).includes(oc.id));
        if (removedCredentials.length > 0) {
            await dal_manager_1.dbManager.accessLog.addUserActionHistoryItem({
                oId: organizationId,
                o: requestUserId,
                u: opTime,
                c: dal_constants_1.DalConstants.UserActionCategory.Main,
                tg: ["Identity", "Credential"],
                t: dal_constants_1.DalConstants.UserMainActionType.RemoveCredential,
                d: {
                    id: newIdentity.id,
                    removedCredentialIds: removedCredentials.map((rc) => rc.id),
                    c: JSON.stringify(removedCredentials.map((m) => {
                        return {
                            f: m.type,
                            o: dal_constants_1.DalConstants.credentialTypesWithMeaningfulData.includes(m.type) ? m.data : "*****",
                            n: null,
                            i: true,
                        };
                    })),
                },
            });
        }
        for (const newCredential of newIdentity.credentials) {
            const oldCredential = oldIdentity.credentials.find((oc) => oc.id === newCredential.id);
            if (!oldCredential)
                continue;
            if (!lodash_1.default.isEqual(newCredential, oldCredential)) {
                await dal_manager_1.dbManager.accessLog.addUserActionHistoryItem({
                    oId: organizationId,
                    o: requestUserId,
                    u: opTime,
                    c: dal_constants_1.DalConstants.UserActionCategory.Main,
                    tg: ["Identity", "Credential"],
                    t: dal_constants_1.DalConstants.UserMainActionType.UpdateCredential,
                    d: {
                        id: newIdentity.id,
                        credentialId: newCredential.id,
                        c: JSON.stringify([
                            {
                                f: newCredential.type,
                                o: dal_constants_1.DalConstants.credentialTypesWithMeaningfulData.includes(newCredential.type) ? newCredential.data : "*****",
                                n: dal_constants_1.DalConstants.credentialTypesWithMeaningfulData.includes(newCredential.type) ? newCredential.data : "*****",
                                i: true,
                            },
                        ]),
                    },
                });
            }
        }
        if (!oldIdentity.username && newIdentity.username) {
            await dal_manager_1.dbManager.accessLog.addUserActionHistoryItem({
                oId: organizationId,
                o: requestUserId,
                u: opTime,
                c: dal_constants_1.DalConstants.UserActionCategory.Main,
                tg: ["Identity", "Account"],
                t: dal_constants_1.DalConstants.UserMainActionType.AccountCreated,
                d: {
                    id: newIdentity.id,
                    c: JSON.stringify([
                        {
                            f: "username",
                            o: null,
                            n: newIdentity.username,
                            i: true,
                        },
                    ]),
                },
            });
        }
        if (oldIdentity.username && !newIdentity.username) {
            await dal_manager_1.dbManager.accessLog.addUserActionHistoryItem({
                oId: organizationId,
                o: requestUserId,
                u: opTime,
                c: dal_constants_1.DalConstants.UserActionCategory.Main,
                tg: ["Identity", "Account"],
                t: dal_constants_1.DalConstants.UserMainActionType.AccountDeleted,
                d: {
                    id: newIdentity.id,
                    c: JSON.stringify([
                        {
                            f: "username",
                            o: oldIdentity.username,
                            n: null,
                            i: true,
                        },
                    ]),
                },
            });
        }
    }
    async updateIdentity(organizationId, requestUserId, trx, updateIdentity, notifyUsersForChange) {
        let oldIdentity = await this.getIdentityDetailed(organizationId, requestUserId, {
            userId: updateIdentity.id,
            hasOrganizationWideRight: true,
            hasIdentityFullWrite: false,
            permittedUnitIds: null,
            hasOrganizationWideRightToSeePassiveUsers: false,
        });
        if (oldIdentity.organizationProfile.isDisabled && !updateIdentity.organizationProfile.isDisabled) {
            const licenceSettings = await dal_manager_1.dbManager.accessOrganization.getModuleLicenceSettings(organizationId, restapi_1.ArmonApplicationModule.Base, trx);
            if (!licenceSettings) {
                throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.NOT_FOUND, "ERRORS.GENERAL.MODULENOTFOUND", null, false, false);
            }
            else {
                if (licenceSettings.licenseControlPolicy === dal_constants_1.DalConstants.LicenseControlPolicy.Instant) {
                    const licenceCheckResult = await dal_manager_1.dbManager.accessOrganization.baseModuleLicenceVacancyCheck(organizationId, licenceSettings, trx);
                    if (!licenceCheckResult) {
                        (0, business_notification_1.sendOrganizationModuleLicenseLimitReachedNotification)({ organizationId, applicationModule: dal_constants_1.DalConstants.ArmonApplicationModule.Base });
                        throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.PAYMENT_REQUIRED, "ERRORS.IDENTITY.BASE_MODULE_LICENCE_EXCEEDED", null, true, true);
                    }
                }
            }
        }
        await this.updateIdentityInfo(organizationId, requestUserId, trx, updateIdentity, oldIdentity);
        let newIdentity = await this.getIdentityDetailed(organizationId, requestUserId, {
            userId: updateIdentity.id,
            hasOrganizationWideRight: true,
            hasIdentityFullWrite: false,
            permittedUnitIds: null,
            hasOrganizationWideRightToSeePassiveUsers: false,
            trx: trx,
        });
        await this.logProfileUpdate(organizationId, requestUserId, oldIdentity, newIdentity);
        const transaction = (await trx.query(`SELECT txid_current() as "id", now()::timestamp without time zone::text as "actionT"`)).rows[0];
        if (notifyUsersForChange) {
            setTimeout(async () => {
                const offlineTerminals = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
                    return await dal_manager_1.dbManager.accessDevice.getOfflineTerminals({ organizationId: organizationId, transaction: transaction, trx: trx });
                }, requestUserId, organizationId);
                if (offlineTerminals.length > 0) {
                    (0, business_notification_1.sendDeviceNotInformedNotification)({ organizationId, offlineTerminals: offlineTerminals, receiverUserId: requestUserId });
                }
            }, app_config_1.appConfig.offlineTerminalNotificationIntervalSeconds * 1000);
        }
        return { oldIdentity, newIdentity };
    }
    async upsertIdentityForbiddenStatus(organizationId, requestUserId, args) {
        let id = args.id || uuid_1.default.v4();
        let opTime = new Date();
        await this.dbClient.transaction(async (trx) => {
            if (args.id) {
                await trx
                    .withSchema(organizationId)
                    .table(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationForbiddances)
                    .where("id", args.id)
                    .update({
                    updatedAt: opTime,
                    startUtc: args.startUtc,
                    endUtc: args.endUtc,
                    regionId: args.regionId,
                    note: args.note ? args.note : "",
                    credentialIds: args.credentialIds ? JSON.stringify(args.credentialIds.sort()) : null,
                });
                await dal_manager_1.dbManager.accessLog.addUserActionHistoryItem({
                    oId: organizationId,
                    o: requestUserId,
                    u: opTime,
                    c: dal_constants_1.DalConstants.UserActionCategory.Main,
                    tg: ["Forbidden"],
                    t: dal_constants_1.DalConstants.UserMainActionType.UpdateForbiddenState,
                    d: {
                        id: requestUserId,
                        fid: id,
                        f: JSON.stringify(args),
                    },
                });
            }
            else {
                let existingForbids = await trx
                    .withSchema(organizationId)
                    .table(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationForbiddances)
                    .whereNull("deletedAt")
                    .where("userId", args.userId)
                    .where("startUtc", "<", args.endUtc)
                    .where("endUtc", ">", args.startUtc)
                    .select();
                if (args.credentialIds && args.credentialIds) {
                    let existingForbiddenCredentials = [].concat.apply([], existingForbids.filter((a) => a.credentialIds).map((a) => a.credentialIds));
                    for (const credentialId of args.credentialIds) {
                        if (existingForbiddenCredentials.some((a) => a == credentialId)) {
                            (0, dal_access_error_1.throwDbAccessConflictErrorTr)("ERRORS.IDENTITY.DUPLICATEFORBID");
                        }
                    }
                }
                else {
                    if (args.regionId && existingForbids.some((a) => a.regionId == args.regionId)) {
                        (0, dal_access_error_1.throwDbAccessConflictErrorTr)("ERRORS.IDENTITY.DUPLICATEFORBID");
                    }
                    else if (existingForbids.length > 0 && existingForbids.filter((a) => !a.regionId).length > 0) {
                        (0, dal_access_error_1.throwDbAccessConflictErrorTr)("ERRORS.IDENTITY.DUPLICATEFORBID");
                    }
                }
                await trx
                    .withSchema(organizationId)
                    .table(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationForbiddances)
                    .insert({
                    id: id,
                    organizationId: organizationId,
                    createdAt: opTime,
                    updatedAt: opTime,
                    deletedAt: null,
                    startUtc: args.startUtc,
                    endUtc: args.endUtc,
                    regionId: args.regionId,
                    note: args.note ? args.note : "",
                    userId: args.userId,
                    credentialIds: args.credentialIds ? JSON.stringify(args.credentialIds.sort()) : null,
                });
                await dal_manager_1.dbManager.accessLog.addUserActionHistoryItem({
                    oId: organizationId,
                    o: requestUserId,
                    u: opTime,
                    c: dal_constants_1.DalConstants.UserActionCategory.Main,
                    tg: ["Forbidden"],
                    t: dal_constants_1.DalConstants.UserMainActionType.AddForbiddenState,
                    d: {
                        id: requestUserId,
                        fid: id,
                        f: JSON.stringify(args),
                    },
                });
            }
        });
        return id;
    }
    async deleteIdentityForbiddenStatus(organizationId, requestUserId, forbidId) {
        let opTime = new Date();
        return await this.dbClient.transaction(async (trx) => {
            let state = await trx.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationForbiddances).where("id", forbidId).whereNull("deletedAt").first();
            if (state) {
                await this._pgPool.query(`
					DELETE FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationForbiddances}"
					WHERE id = $1
				`, [forbidId]);
                await dal_manager_1.dbManager.accessLog.addUserActionHistoryItem({
                    oId: organizationId,
                    o: requestUserId,
                    u: new Date(),
                    c: dal_constants_1.DalConstants.UserActionCategory.Main,
                    tg: ["Forbidden"],
                    t: dal_constants_1.DalConstants.UserMainActionType.RemoveForbiddenState,
                    d: {
                        id: requestUserId,
                        fid: forbidId,
                    },
                });
                return Promise.resolve(true);
            }
            return Promise.resolve(false);
        });
    }
    async getForbiddenStatus(organizationId, forbidId, includeDeleted) {
        const q = this.dbClient.withSchema(organizationId).table(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationForbiddances).where("id", forbidId).first();
        if (!includeDeleted) {
            q.whereNull("deletedAt");
        }
        return q.first();
    }
    async getForbiddenStatusFromHistory(params) {
        const { rows, rowCount } = await params.trx.query(`
			SELECT id, "userId", "regionId", "credentialIds", "startUtc" as "startDateISO", "endUtc" as "endDateISO"
			FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.zz_user_organization_forbiddances}"
			WHERE id = $1
			${!params.includeDeleted ? `AND "deletedAt" IS NULL` : ""}
			ORDER BY "actionT" DESC
			LIMIT 1
		`, [params.forbidId]);
        return rowCount > 0 ? rows[0] : null;
    }
    async listForbidden(params) {
        let result = {
            pagination: {
                take: params.filter.pagination.take,
                skip: params.filter.pagination.skip,
                total: 0,
            },
            items: [],
        };
        const trxx = params.trx ?? this._pgPool;
        const qb = [];
        const qw = [];
        const qj = [];
        let qx = 1;
        let query = `
			SELECT uop.name, uop.surname, uop."uniqueId", r.name as "regionName", uof.* FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationForbiddances}" AS uof
			INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" AS uo
				ON uo."userId" = uof."userId"
			INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" AS uop
				ON uo.id = uop."userOrganizationId"
			LEFT JOIN "${params.organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.regions} AS r
				ON r.id = uof."regionId"
			LEFT JOIN "${params.organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.region_administrators} ra
				ON r.id = ra."regionId" 
		`;
        if (!params.filter.hasOrganizationWideRight) {
            qj.push(`
				INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}" AS uoou
					ON uoou."userOrganizationId" = uo.id
				INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" AS ou
					ON ou.id = uoou."organizationUnitId"
			`);
        }
        if (params.filter.credentialData) {
            qj.push(`
            INNER JOIN "${params.organizationId}"."userOrganizationCredentials" as uoc
            ON uoc."userId" = uo."userId"
            `);
        }
        query += qj.join(" ");
        qw.push(`uof."deletedAt" IS NULL`, `uop."deletedAt" IS NULL`, `uo."deletedAt" IS NULL`);
        if (!params.filter.hasOrganizationWideRight) {
            qw.push(`uoou."deletedAt" IS NULL`, `ou."deletedAt" IS NULL`, `(ou."id" = ANY ($${qx++}::UUID[]) OR ou."ancestorIds" SIMILAR TO '%'||$${qx++}||'%' )`);
            qb.push(params.filter.permittedUnitIds);
            qb.push(params.filter.permittedUnitIds.join("|"));
        }
        if (params.filter.credentialData) {
            let credentialSearchValues = await this.generateCredentialSearch(params.organizationId, params.filter.credentialData, undefined);
            credentialSearchValues.push(params.filter.credentialData);
            const qOr = [];
            switch (params.filter.searchMethod) {
                case dal_constants_1.DalConstants.SearchMethod.StartsWith:
                    for (const credential of credentialSearchValues) {
                        qOr.push(`upper(unaccent(data)) ilike upper(unaccent($${qx++}))`);
                        qb.push(credential + "%");
                    }
                    break;
                case dal_constants_1.DalConstants.SearchMethod.Similar:
                    for (const credential of credentialSearchValues) {
                        qOr.push(`upper(unaccent(data)) ilike upper(unaccent($${qx++}))`);
                        qb.push("%" + credential + "%");
                    }
                    break;
                case dal_constants_1.DalConstants.SearchMethod.Exact:
                default:
                    for (const credential of credentialSearchValues) {
                        qOr.push(`upper(unaccent(data)) = upper(unaccent($${qx++}))`);
                        qb.push(credential);
                    }
            }
            qw.push(`(${qOr.join(" OR ")})`);
        }
        if (params.requesterUserId) {
            qw.push(`(ra."userId" IS NULL OR ra."userId" = $${qx++} AND "read" IS TRUE)`);
            qb.push(params.requesterUserId);
        }
        if (params.filter.startUtc) {
            qw.push(`uof."endUtc" >= $${qx++}`);
            qb.push(params.filter.startUtc);
        }
        if (params.filter.endUtc) {
            qw.push(`uof."startUtc" <= $${qx++}`);
            qb.push(params.filter.endUtc);
        }
        if (params.filter.regionId) {
            qw.push(`uof."regionId" = $${qx++}`);
            qb.push(params.filter.regionId);
        }
        if (params.filter.userId) {
            qw.push(`uof."userId" = $${qx++}`);
            qb.push(params.filter.userId);
        }
        if (params.filter.note) {
            qw.push(`uof.note like $${qx++}`);
            qb.push(`%${params.filter.note}%`);
        }
        query += `WHERE ${qw.join(" AND ")}`;
        result.pagination.total = parseInt((await trxx.query("SELECT COUNT(*) FROM (" + query + ")q1", qb)).rows[0].count);
        query += `
            ORDER BY uof."createdAt" DESC
            OFFSET $${qx++}
            LIMIT $${qx++}
            `;
        qb.push(params.filter.pagination.skip, params.filter.pagination.take);
        const { rows } = await trxx.query(query, qb);
        result.items = await Promise.all(rows.map(async (row) => {
            return {
                endUtc: row.endUtc,
                startUtc: row.startUtc,
                id: row.id,
                note: row.note,
                regionId: row.regionId,
                regionName: row.regionName,
                credentialIds: row.credentialIds,
                userId: row.userId,
            };
        }));
        return result;
    }
    async searchCredentialsForExcel(organizationId, requesterUserId, onData, filter) {
        const userFilter = (0, dal_access_psql_common_1.getReportUserFilterForPgClient)({
            bindingKeys: [],
            specificSelectItems: ["userId"],
            requesterUserId,
            organizationId,
            idBasedUserFilter: {
                applyOrganizationUnitFilterHierarchically: filter.filterUsersHierarchically,
                organizationUnitIds: filter.organizationUnitIds,
                userGroupIds: filter.userGroupIds,
                userIds: filter.userIds,
                userOrganizationStatus: filter.userOrganizationStatus,
            },
            requiredOrganizationUnitWidePermissions: ["i:b"],
            requiredOrganizationWidePermissions: ["i:b"],
        });
        let qx = userFilter.bindingKeys.length + 1;
        const qb = [...userFilter.bindingKeys];
        const qWhere = [];
        let qFrom = `FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials}" as uoc`;
        if (filter.userOrganizationStatus && filter.userOrganizationStatus !== app_enums_1.enums.IdentityStatusType.All) {
            qFrom += `
				INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" as uo ON uo."userId" = uoc."userId" 
				AND uo."isDisabled" = $${qx++} AND uo."deletedAt" IS NULL
			`;
            qb.push(filter.userOrganizationStatus === app_enums_1.enums.IdentityStatusType.Disabled);
        }
        qFrom += `
			INNER JOIN(
				${userFilter.query}
			) UFQ ON UFQ."userId" = uoc."userId" `;
        qWhere.push(`uoc."deletedAt" IS NULL`);
        if (filter.data) {
            const credentialSearchValues = await this.generateCredentialSearch(organizationId, filter.data, filter.credentialType);
            credentialSearchValues.push(filter.data);
            const credSearchVals = [];
            switch (filter.searchMethod) {
                case dal_constants_1.DalConstants.SearchMethod.StartsWith:
                    for (const credential of credentialSearchValues) {
                        credSearchVals.push(` upper(unaccent(uoc.data)) ilike upper(unaccent($${qx++})) `);
                        qb.push(credential + "%");
                    }
                    break;
                case dal_constants_1.DalConstants.SearchMethod.Similar:
                    for (const credential of credentialSearchValues) {
                        credSearchVals.push(` upper(unaccent(uoc.data)) ilike upper(unaccent($${qx++})) `);
                        qb.push("%" + credential + "%");
                    }
                    break;
                case dal_constants_1.DalConstants.SearchMethod.Exact:
                default:
                    for (const credential of credentialSearchValues) {
                        credSearchVals.push(` upper(unaccent(uoc.data)) = upper(unaccent($${qx++})) `);
                        qb.push(credential);
                    }
            }
            qWhere.push(`(${credSearchVals.join(" OR ")})`);
        }
        if (filter.credentialType !== undefined && filter.credentialType !== null) {
            qWhere.push(` uoc.type = $${qx++} `);
            qb.push(filter.credentialType);
        }
        if (filter.dateRange) {
            qWhere.push(`uoc."expiresOn" <= $${qx++}`, `uoc."expiresOn" >= $${qx++} `);
            qb.push(filter.dateRange.endDateTime);
            qb.push(filter.dateRange.startDateTime);
        }
        const q = `
			SELECT uoc."id", uoc."type", uoc."data", uoc."userId", uoc."expiresOn" ${qFrom}
			WHERE ${qWhere.join(" AND ")}
			ORDER BY uoc.data DESC`;
        const client = await this._pgPool.connect();
        const cursor = client.query(new Cursor(q, qb));
        let rows = [];
        while (true) {
            try {
                let result = [];
                rows = await new Promise((resolve, reject) => {
                    cursor.read(1000, async (err, rows) => {
                        if (err) {
                            reject(err);
                        }
                        else {
                            let users = lodash_1.default.uniq(rows.map((i) => i.userId));
                            let userCaptions = await dal_manager_1.dbManager.accessUser.getUserOrganizationProfile(organizationId, users);
                            let forbids = await this.getForbids(organizationId, users);
                            for (const item of rows) {
                                let userCaption = userCaptions.find((u) => u.userId == item.userId);
                                let userForbids = forbids.filter((f) => f.userId === item.userId);
                                let selectedCredentialUserForbids = [];
                                if (userForbids && userForbids.length > 0) {
                                    for (const forbid of userForbids) {
                                        if (forbid.credentialIds && forbid.credentialIds.length) {
                                            for (const forbiddenCredentialId of forbid.credentialIds) {
                                                if (forbiddenCredentialId === item.id) {
                                                    selectedCredentialUserForbids.push(forbid);
                                                }
                                            }
                                        }
                                        else {
                                            selectedCredentialUserForbids.push(forbid);
                                        }
                                    }
                                }
                                result.push({
                                    id: item.id,
                                    data: item.data,
                                    userId: item.userId,
                                    credentialType: item.type,
                                    expiresOn: item.expiresOn,
                                    userProfile: userCaption && userCaption.userOrganizationProfile ? userCaption.userOrganizationProfile : null,
                                    forbids: selectedCredentialUserForbids,
                                });
                            }
                            resolve(rows);
                        }
                    });
                });
                await onData(result);
            }
            catch (error) {
                app_logs_1.logger.error("Error while fetch credentials data with cursor!");
            }
            if (rows.length < 1000) {
                break;
            }
        }
        try {
            await new Promise((resolve, reject) => {
                cursor.close((err) => {
                    if (err) {
                        reject(err);
                    }
                    else {
                        resolve();
                    }
                });
            });
            client.release();
        }
        catch (error) {
            client?.release(error);
            app_logs_1.logger.error(error);
        }
    }
    async searchCredentials(organizationId, requesterUserId, filter, trx) {
        const result = {
            items: [],
            pagination: {
                take: filter.pagination.take,
                skip: filter.pagination.skip,
                total: 0,
            },
        };
        const filterUsers = await dal_manager_1.dbManager.accessFunctions.dbFuncCollectUsersForAccessReportFilter({
            organizationId: organizationId,
            userIds: filter.userIds || [],
            organizationUnitIds: filter.organizationUnitIds || [],
            userGroupIds: filter.userGroupIds || [],
            filterOrganizationUnitMembersHierarchically: filter.filterUsersHierarchically,
        });
        let qx = 1;
        const qb = [];
        let qFrom = `
		FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials}" uoc`;
        if (filter.userOrganizationStatus && filter.userOrganizationStatus !== app_enums_1.enums.IdentityStatusType.All) {
            qFrom += `
				INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" as uo ON uo."userId" = uoc."userId" 
			AND uo."isDisabled" = $${qx++} AND uo."deletedAt" IS NULL
			`;
            qb.push(filter.userOrganizationStatus === app_enums_1.enums.IdentityStatusType.Disabled);
        }
        const qWhere = [`uoc."deletedAt" IS NULL`];
        if (filterUsers.length > 0) {
            qWhere.push(`uoc."userId" = ANY($${qx++}::uuid[])`);
            qb.push(filterUsers.map((a) => a.userId));
        }
        if (filter.data) {
            const credentialSearchValues = await this.generateCredentialSearch(organizationId, filter.data, filter.credentialType);
            credentialSearchValues.push(filter.data);
            const credSearchVals = [];
            switch (filter.searchMethod) {
                case dal_constants_1.DalConstants.SearchMethod.StartsWith:
                    for (const credential of credentialSearchValues) {
                        credSearchVals.push(` upper(unaccent(uoc.data)) ilike upper(unaccent($${qx++})) `);
                        qb.push(`${credential}%`);
                    }
                    break;
                case dal_constants_1.DalConstants.SearchMethod.Similar:
                    for (const credential of credentialSearchValues) {
                        credSearchVals.push(` upper(unaccent(uoc.data)) ilike upper(unaccent($${qx++})) `);
                        qb.push(`%${credential}%`);
                    }
                    break;
                case dal_constants_1.DalConstants.SearchMethod.Exact:
                default:
                    for (const credential of credentialSearchValues) {
                        credSearchVals.push(` upper(unaccent(uoc.data)) = upper(unaccent($${qx++})) `);
                        qb.push(credential);
                    }
            }
            qWhere.push(`(${credSearchVals.join(" OR ")})`);
        }
        if (filter.credentialType !== undefined && filter.credentialType !== null) {
            qWhere.push(`uoc.type = $${qx++}`);
            qb.push(filter.credentialType);
        }
        if (filter.dateRange) {
            qWhere.push(`uoc."expiresOn" <= $${qx++} AND uoc."expiresOn" >= $${qx++} `);
            qb.push(filter.dateRange.endDateTime);
            qb.push(filter.dateRange.startDateTime);
        }
        qFrom += ` WHERE ${qWhere.join(" AND ")}`;
        app_logs_1.logger.debug(qFrom);
        const { rows: CountQueryRows } = await trx.query(`SELECT count(*)::integer as c ${qFrom}`, qb);
        result.pagination.total = CountQueryRows[0].c;
        if (result.pagination.total > 0) {
            const q = `
				SELECT uoc."id", uoc."type", uoc."data", uoc."userId", uoc."expiresOn" ${qFrom}
				ORDER BY uoc.data DESC
				LIMIT $${qx++}
				OFFSET $${qx++}`;
            qb.push(filter.pagination.take ?? 100);
            qb.push(filter.pagination.skip ?? 0);
            const { rows } = await trx.query(q, qb);
            const users = lodash_1.default.uniq(rows.map((i) => i.userId));
            const userCaptions = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, users);
            const forbids = await this.getForbids(organizationId, users);
            for (const item of rows) {
                const userCaption = userCaptions.find((u) => u.id == item.userId);
                let userForbids = forbids.filter((f) => f.userId === item.userId);
                const selectedCredentialUserForbids = [];
                if (userForbids && userForbids.length > 0) {
                    for (const forbid of userForbids) {
                        if (forbid.credentialIds && forbid.credentialIds.length) {
                            for (const credentialId of forbid.credentialIds) {
                                if (credentialId === item.id) {
                                    selectedCredentialUserForbids.push(forbid);
                                }
                            }
                        }
                        else {
                            selectedCredentialUserForbids.push(forbid);
                        }
                    }
                }
                result.items.push({
                    id: item.id,
                    data: item.data,
                    userId: item.userId,
                    credentialType: item.type,
                    expiresOn: item.expiresOn,
                    userCaption: userCaption ? userCaption.captionLines : [],
                    forbids: selectedCredentialUserForbids,
                });
            }
        }
        return result;
    }
    async getForbids(organizationId, userIds) {
        let qb = this.dbClient
            .withSchema(organizationId)
            .from(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationForbiddances + " as uof")
            .innerJoin(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations + " as uo", "uo.userId", "uof.userId")
            .innerJoin(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles + " as uop", "uo.id", "uop.userOrganizationId")
            .leftJoin(dal_db_armon_schema_1.ArmonSchema.tableNames.regions + " as r ", "r.id", "uof.regionId")
            .whereNull("uof.deletedAt")
            .whereNull("uop.deletedAt")
            .whereNull("uo.deletedAt")
            .where("uo.organizationId", organizationId)
            .whereIn("uof.userId", userIds);
        let rows = await qb.select("uop.name", "uop.surname", "uop.uniqueId", "r.name as regionName", "uof.*");
        let result = [];
        for (const row of rows) {
            result.push({
                endUtc: row.endUtc,
                startUtc: row.startUtc,
                id: row.id,
                note: row.note,
                regionId: row.regionId,
                regionName: row.regionName,
                credentialIds: row.credentialIds,
                userId: row.userId,
            });
        }
        return result;
    }
    async addAccessRightForUser(organizationId, requestUserId, trx, params) {
        const opTime = new Date();
        const requesterUserAccessRights = await this.getUserAccessRights({
            organizationId,
            userId: requestUserId,
            trx,
        });
        const apsThatRequesterUserDoesNotHaveGrantAuth = params.accessRights
            .map((accessRight) => accessRight.accessControlPointId)
            .filter((accessPointId) => !requesterUserAccessRights
            .filter((ap) => ap.grant)
            .map((ap) => ap.accessControlPointId)
            .includes(accessPointId));
        if (apsThatRequesterUserDoesNotHaveGrantAuth.length > 0) {
            (0, dal_access_error_1.throwDbAccessAuthorizationErrorTr)("ERRORS.IDENTITY.USER_HAS_NO_GRANT_FOR_AP", { ap: JSON.stringify(apsThatRequesterUserDoesNotHaveGrantAuth) });
        }
        let remoteAccessAccessControlPointIds = params.accessRights.filter((ar) => ar.remoteAccess).map((ar) => ar.accessControlPointId);
        if (remoteAccessAccessControlPointIds) {
            const remoteAccessAvailableAccessControlPointIds = (await trx.query(`
                    SELECT id FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessControlPoints}"
                    WHERE "remoteAvailable" = true AND
                    id = ANY($1:: UUID[])
                        `, [remoteAccessAccessControlPointIds])).rows.map((r) => r.id);
            for (const rightACPId of remoteAccessAccessControlPointIds) {
                if (!remoteAccessAvailableAccessControlPointIds.some((availableACPId) => availableACPId === rightACPId)) {
                    (0, dal_access_error_1.throwDbAccessBadRequestError)("Please make sure ACP's has remote access function before giving remote access right");
                    return false;
                }
            }
        }
        try {
            let qx = 1;
            const qb = [];
            const qv = [];
            const addedAccessRightIds = [];
            for (const accessRight of params.accessRights) {
                const newAccessRightId = uuid_1.default.v4();
                addedAccessRightIds.push(newAccessRightId);
                qv.push(`($${qx++}, $${qx++}, $${qx++}, $${qx++}, $${qx++}, $${qx++}, $${qx++}, $${qx++}, $${qx++}, $${qx++}, $${qx++})`);
                qb.push(newAccessRightId, opTime, opTime, params.userId, accessRight.accessControlPointId, accessRight.remoteAccess ? accessRight.remoteAccess : null, accessRight.read ? accessRight.read : null, accessRight.access ? accessRight.access : null, accessRight.grant ? accessRight.grant : null, accessRight.config ? accessRight.config : null, accessRight.snapshot ? accessRight.snapshot : null);
            }
            await trx.query(`
				INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userAccessRights}"
					("id", "createdAt", "updatedAt", "userId", "accessControlPointId", "remoteAccess", "read", "access", "grant", "config", "snapshot")
				VALUES ${qv.join(" , ")};
				`, qb);
            await dal_manager_1.dbManager.accessLog.addUserActionHistoryItem({
                oId: organizationId,
                o: requestUserId,
                u: opTime,
                c: dal_constants_1.DalConstants.UserActionCategory.Main,
                tg: ["Identity", "AccessRight"],
                t: dal_constants_1.DalConstants.UserMainActionType.AddAccessRight,
                d: {
                    id: params.userId,
                    accessRightId: addedAccessRightIds,
                },
            });
        }
        catch (err) {
            dal_logger_1.dbLogger.error(err);
            (0, dal_access_error_1.throwDbAccessBadRequestError)(err.message + " while adding access rights");
            return false;
        }
        return true;
    }
    async getUserAccessRights(params) {
        const fn = async (trx) => {
            try {
                let result = (await trx.query(`
                SELECT * FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessControlPoints}" AS acp
                INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userAccessRights}" AS uar
                ON acp."id" = uar."accessControlPointId"
                WHERE uar."deletedAt" IS NULL AND
                acp."deletedAt" IS NULL AND
                uar."userId" = $1 AND
                acp."organizationId" = $2
           `, [params.userId, params.organizationId])).rows;
                return result;
            }
            catch (err) {
                (0, dal_access_error_1.throwDbAccessBadRequestError)("DB Error while getting access rights of user!");
                dal_logger_1.dbLogger.error(err);
                return null;
            }
        };
        if (params.trx) {
            return fn(params.trx);
        }
        else {
            return dal_manager_1.dbManager.systemTransaction(async (trx) => {
                return fn(trx);
            });
        }
    }
    identityListExactQueryBuilder(organizationId, options, includeAccessRights) {
        let queryParamIndex = 1;
        let queryParams = [];
        let query = `
        SELECT uo."id" AS "id",
        uo."id" AS "publicKey",

        json_build_object(
            'userId', uop."userId",
            'name', uop."name", 
            'surname', uop."surname",
            'address', uop."address",
            'phoneNumber', uop."phoneNumber", 
            'email', uop."email", 
            'uniqueId', uop."uniqueId", 
            'extensionFields', uop."extensionFields",
            'roleId', r."id",
            'roleName', r."name",
            'username', u."username",
            'isDisabled', true,
			'employmentStartUtc', uop."employmentStartUtc",
            'birthDateUtc', uop."birthDateUtc",  
			'employmentEndUtc', uop."employmentEndUtc"
		) AS "organizationProfile",

		(
			SELECT COALESCE (JSON_AGG (
				JSON_BUILD_OBJECT (
					'organizationUnitId', uoou."organizationUnitId",
					'name', ou."name",
					'ancestorIds', null,
					'type', null,
					'defaultAccessControlPointIds', null
				)
			), '[]')
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}" uoou
			INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" ou
				ON uoou."organizationUnitId" = ou.id
			WHERE uoou."userOrganizationId" = uo.id
				AND uoou."deletedAt" IS NULL
				AND ou."deletedAt" IS NULL
		) AS "organizationUnits",
		${includeAccessRights
            ? `
			(
				SELECT COALESCE (JSON_AGG (
					JSON_BUILD_OBJECT (
						'id', uar."id",
						'accessControlPointId', acp."id",
						'accessControlPointName', acp."name",
						'read', uar."read",
						'access', uar."access",
						'config', uar."config",
						'grant', uar."grant",
						'remoteAccess', uar."remoteAccess",
						'snapshot', uar."snapshot",
						'isControlPanelSync', null
					)
				), '[]')
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userAccessRights}" uar
				INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessControlPoints}" acp
					ON uar."accessControlPointId" = acp.id
				WHERE uar."userId" = uo."userId"
					AND acp."deletedAt" IS NULL
					AND uar."deletedAt" IS NULL
			) AS "accessRights",
			`
            : `ARRAY[]::JSON[] AS "accessRights", `}
		(
			SELECT COALESCE (JSON_AGG (
				JSON_BUILD_OBJECT (
					'id', uoc."id",
					'type', uoc."type",
					'expirationUtc', uoc."expiresOn",
					'data', CASE WHEN uoc.type IN(${terminal_1.CredentialType.ProximityCard},${terminal_1.CredentialType.MiFare},${terminal_1.CredentialType.VehiclePlate},${terminal_1.CredentialType.UHFRfid},${terminal_1.CredentialType.MRZ}) THEN uoc."data" ELSE NULL END,
					'credentialNumber', uoc."credentialNumber",
					'extensionFields', uoc."extensionFields"
				)
			), '[]')
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials}" uoc
			WHERE uoc."userId" = uo."userId"
				AND uoc."deletedAt" IS NULL
		) AS "credentials",
		(
			SELECT COALESCE (JSON_AGG (
				JSON_BUILD_OBJECT (
					'id', ug."id",
					'name', ug."name",
					'colorCode', null
				)
			), '[]')
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupUserOrganizations}" uguo
			INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroups}" ug
				ON uguo."userGroupId" = ug.id
			WHERE uguo."userOrganizationId" = uo.id
				AND uguo."deletedAt" IS NULL
				AND ug."deletedAt" IS NULL
		) AS "userGroups"

        FROM "${organizationId}"."userOrganizationProfiles" AS uop 
        INNER JOIN "${organizationId}"."userOrganizations" AS uo ON uop."userOrganizationId" = uo."id" 
        AND uop."deletedAt" IS NULL AND uo."deletedAt" IS NULL
        INNER JOIN "${organizationId}"."users" AS u ON uo."userId" = u."id"  AND u."deletedAt" IS NULL
        INNER JOIN "${organizationId}"."roles" AS r ON r."id" = uo."roleId" AND r."deletedAt" IS NULL
		
		WHERE uo."organizationId" = $${queryParamIndex++}
        `;
        queryParams.push(organizationId);
        if (options.createdDate) {
            if (options.createdDate.createdAtStart) {
                query += ` AND  uo."createdAt" > $${queryParamIndex++} `;
                queryParams.push(options.createdDate.createdAtStart);
            }
            if (options.createdDate.createdAtEnd) {
                query += ` AND  uo."createdAt" < $${queryParamIndex++} `;
                queryParams.push(options.createdDate.createdAtEnd);
            }
        }
        if (options.userOrganization) {
            query += ` AND  uo."isDisabled" = $${queryParamIndex++} `;
            queryParams.push(options.userOrganization.isDisabled);
            if (options.userOrganization.roleId) {
                query += ` AND  uo."roleId" = $${queryParamIndex++} `;
                queryParams.push(options.userOrganization.roleId);
            }
        }
        if (options.user) {
            if (options.user.username) {
                query += ` AND lower(u."username") = lower($${queryParamIndex++})  `;
                queryParams.push(options.user.username);
            }
            if (options.user.accountEnabled) {
                query += ` AND u."accountEnabled" = $${queryParamIndex++}  `;
                queryParams.push(options.user.accountEnabled);
            }
        }
        if (options.userOrganizationProfile) {
            let rqitems = [];
            for (const key of Object.keys(options.userOrganizationProfile)) {
                if (key && key != "extensionFields") {
                    rqitems.push(` upper(uop."${key}") = upper($${queryParamIndex++}) `);
                    queryParams.push(options.userOrganizationProfile[key]);
                }
            }
            if (rqitems.length > 0) {
                query += " AND (" + rqitems.join(" AND ") + ")";
            }
            if (options.userOrganizationProfile.extensionFields?.length) {
                let extitems = [];
                for (const filt of options.userOrganizationProfile.extensionFields) {
                    if (filt.name === "birthDateUtc" || filt.name === "employmentStartUtc" || filt.name === "employmentEndUtc") {
                        extitems.push(` uop."${filt.name}" > $${queryParamIndex++}`);
                        extitems.push(` uop."${filt.name}" < $${queryParamIndex++}`);
                        queryParams.push(filt.value.start);
                        queryParams.push(filt.value.end);
                    }
                    else {
                        let column = `uop."extensionFields"->>'` + filt.name + "'";
                        extitems.push(` unaccent(upper(${column}))::TEXT = unaccent(upper($${queryParamIndex++})) `);
                        queryParams.push(filt.value);
                    }
                }
                if (extitems.length > 0) {
                    query += "  AND (" + extitems.join(" AND ") + ")";
                }
            }
        }
        if (options.organizationUnit && options.organizationUnit.id) {
            let rqitems = [];
            rqitems.push(`ou."id" = $${queryParamIndex++}`);
            queryParams.push(options.organizationUnit.id);
            if (options.organizationUnit) {
                rqitems.push(`ou."ancestorIds" like $${queryParamIndex++}`);
                queryParams.push("%" + options.organizationUnit.id + "%");
            }
            query += `
				AND uo.id in (
					SELECT uoou."userOrganizationId" 
					FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}" uoou
					INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" ou
						ON ou."id" = uoou."organizationUnitId"
					WHERE uoou."deletedAt" IS NULL
						AND ou."deletedAt" IS NULL
						AND ( ${rqitems.join(" OR ")} )
				)
			`;
        }
        if (options.userGroup && options.userGroup.id) {
            query += `
				AND uo.id in (
					SELECT uguo."userOrganizationId" 
					FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupUserOrganizations}" uguo
					WHERE uguo."deletedAt" IS NULL
						AND uguo."userGroupId" = $${queryParamIndex++}
				)
			`;
            queryParams.push(options.userGroup.id);
        }
        if (options.workplan?.id) {
            query += `
				AND uo."userId" in (
					SELECT uwp."userId" 
					FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userWorkPlans}" uwp
					WHERE uwp."workPlanId" = $${queryParamIndex++}
				)
			`;
            queryParams.push(options.workplan.id);
        }
        if (options.userOrganizationCredential) {
            const credentialWhereQ = [];
            credentialWhereQ.push(` uoc."deletedAt" IS NULL `);
            credentialWhereQ.push(` uoc.type = $${queryParamIndex++} `);
            queryParams.push(options.userOrganizationCredential.type);
            if (options.userOrganizationCredential.data) {
                credentialWhereQ.push(` uoc."data" =  $${queryParamIndex++} `);
                queryParams.push(options.userOrganizationCredential.data);
            }
            if (options.userOrganizationCredential.extensionFields?.length) {
                for (const extensionField of options.userOrganizationCredential.extensionFields) {
                    let fieldValue = extensionField.value;
                    credentialWhereQ.push(` uoc."extensionFields"->>'${extensionField.name}' ilike $${queryParamIndex++} `);
                    if (fieldValue && fieldValue.id && fieldValue.value) {
                        queryParams.push("%" + fieldValue.id + "%");
                    }
                    else if (fieldValue) {
                        queryParams.push(fieldValue);
                    }
                }
            }
            if (options.userOrganizationCredential.customFields?.length) {
                for (const customField of options.userOrganizationCredential.customFields) {
                    let fieldValue = customField.value;
                    credentialWhereQ.push(` uoc."customField"->>'${customField.name}' ilike $${queryParamIndex++} `);
                    if (fieldValue && fieldValue.id && fieldValue.value) {
                        queryParams.push("%" + fieldValue.id + "%");
                    }
                    else if (fieldValue) {
                        queryParams.push(fieldValue);
                    }
                }
            }
            query += `
			AND uo."userId" IN (
				SELECT uoc."userId"
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials}" uoc
				WHERE ${credentialWhereQ.join(` AND `)}
			)
		`;
        }
        return { query, queryParams };
    }
    async fetchAllDataIdentityExact(organizationId, options, onData) {
        const q = this.identityListExactQueryBuilder(organizationId, options, false);
        await this.handleCursor(onData, q.query, q.queryParams);
        return options;
    }
    async handleCursor(onData, query, queryParams) {
        const client = await this._pgPool.connect();
        const cursor = client.query(new Cursor(query, queryParams));
        let rows = [];
        while (true) {
            try {
                rows = await new Promise((resolve, reject) => {
                    cursor.read(1000, (err, rows) => {
                        if (err) {
                            reject(err);
                        }
                        else {
                            resolve(rows);
                        }
                    });
                });
                await onData(rows);
            }
            catch (error) {
                app_logs_1.logger.error(error);
            }
            if (rows.length < 1000) {
                break;
            }
        }
        try {
            await new Promise((resolve, reject) => {
                cursor.close((err) => {
                    if (err) {
                        reject(err);
                    }
                    else {
                        resolve();
                    }
                });
            });
            client.release();
        }
        catch (error) {
            client?.release(error);
            app_logs_1.logger.error(error);
            throw error;
        }
    }
    async fetchAllDataIdentity(options, organizationId, onData) {
        const q = this.identityListQueryBuilder(options, organizationId, false);
        await this.handleCursor(onData, q.query, q.queryParams);
        return options;
    }
    identityListQueryBuilder(options, organizationId, includeAccessRights) {
        let queryParamIndex = 1;
        let queryParams = [];
        let query = `
        SELECT uo."id" AS "id",
        uo."id" AS "publicKey",
        JSON_BUILD_OBJECT(
            'userId', uop."userId",
            'name', uop."name", 
            'surname', uop."surname",
            'address', uop."address",
            'phoneNumber', uop."phoneNumber", 
            'email', uop."email", 
            'uniqueId', uop."uniqueId", 
            'extensionFields', uop."extensionFields",
            'roleId', r."id",
            'roleName', r."name",
            'username', u."username",
            'isDisabled', true,
			'employmentStartUtc', uop."employmentStartUtc",
            'birthDateUtc', uop."birthDateUtc",  
			'employmentEndUtc', uop."employmentEndUtc"
        ) AS "organizationProfile",

		(
			SELECT COALESCE (JSON_AGG (
				JSON_BUILD_OBJECT (
					'organizationUnitId', uoou."organizationUnitId",
					'name', ou."name",
					'ancestorIds', null,
					'type', null,
					'defaultAccessControlPointIds', null,
					'roleId', r."id",
					'roleName', r."name"
				)
			), '[]')
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}" uoou
			INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" ou
				ON uoou."organizationUnitId" = ou.id
			INNER JOIN "${organizationId}"."roles" AS r ON r."id" = uoou."roleId"
			WHERE uoou."userOrganizationId" = uo.id
				AND uoou."deletedAt" IS NULL
				AND ou."deletedAt" IS NULL
		) AS "organizationUnits",
		${includeAccessRights
            ? `
			(
				SELECT COALESCE (JSON_AGG (
					JSON_BUILD_OBJECT (
						'id', uar."id",
						'accessControlPointId', acp."id",
						'accessControlPointName', acp."name",
						'read', uar."read",
						'access', uar."access",
						'config', uar."config",
						'grant', uar."grant",
						'remoteAccess', uar."remoteAccess",
						'snapshot', uar."snapshot",
						'isControlPanelSync', null
					)
				), '[]')
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userAccessRights}" uar
				INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessControlPoints}" acp
					ON uar."accessControlPointId" = acp.id
				WHERE uar."userId" = uo."userId"
					AND acp."deletedAt" IS NULL
					AND uar."deletedAt" IS NULL
			) AS "accessRights",
			`
            : `ARRAY[]::JSON[] AS "accessRights", `}
		(
			SELECT COALESCE (JSON_AGG (
				JSON_BUILD_OBJECT (
					'id', uoc."id",
					'type', uoc."type",
					'expirationUtc', uoc."expiresOn",
					'data', CASE WHEN uoc.type IN(${terminal_1.CredentialType.ProximityCard},${terminal_1.CredentialType.MiFare},${terminal_1.CredentialType.VehiclePlate},${terminal_1.CredentialType.UHFRfid},${terminal_1.CredentialType.MRZ}) THEN uoc."data" ELSE NULL END,
					'credentialNumber', uoc."credentialNumber",
					'extensionFields', uoc."extensionFields"
				)
			), '[]')
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials}" uoc
			WHERE uoc."userId" = uo."userId"
				AND uoc."deletedAt" IS NULL
		) AS "credentials",
		(
			SELECT COALESCE (JSON_AGG (
				JSON_BUILD_OBJECT (
					'id', ug."id",
					'name', ug."name",
					'colorCode', null
				)
			), '[]')
			FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupUserOrganizations}" uguo
			INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroups}" ug
				ON uguo."userGroupId" = ug.id
			WHERE uguo."userOrganizationId" = uo.id
				AND uguo."deletedAt" IS NULL
				AND ug."deletedAt" IS NULL
		) AS "userGroups"

        FROM "${organizationId}"."userOrganizationProfiles" AS uop 
        INNER JOIN "${organizationId}"."userOrganizations" AS uo ON uop."userOrganizationId" = uo."id" 
        AND uop."deletedAt" IS NULL AND uo."deletedAt" IS NULL
        INNER JOIN "${organizationId}"."users" AS u ON uo."userId" = u."id" 
        INNER JOIN "${organizationId}"."roles" AS r ON r."id" = uo."roleId"
		WHERE uo."organizationId" = $${queryParamIndex++}
        `;
        queryParams.push(organizationId);
        if (options.createdDate) {
            if (options.createdDate.createdAtStart) {
                query += ` AND  uo."createdAt" > $${queryParamIndex++} `;
                queryParams.push(options.createdDate.createdAtStart);
            }
            if (options.createdDate.createdAtEnd) {
                query += ` AND  uo."createdAt" < $${queryParamIndex++} `;
                queryParams.push(options.createdDate.createdAtEnd);
            }
        }
        if (options.userOrganization) {
            query += ` AND  uo."isDisabled" = $${queryParamIndex++} `;
            queryParams.push(options.userOrganization.isDisabled);
            if (options.userOrganization.roleId) {
                query += ` AND  uo."roleId" = $${queryParamIndex++} `;
                queryParams.push(options.userOrganization.roleId);
            }
        }
        if (options.user) {
            if (options.user.username) {
                query += ` AND lower(u."username") = lower($${queryParamIndex++})  `;
                queryParams.push(options.user.username);
            }
            if (options.user.accountEnabled) {
                query += ` AND u."accountEnabled" = $${queryParamIndex++}  `;
                queryParams.push(options.user.accountEnabled);
            }
        }
        if (options.userOrganizationProfile) {
            let rqitems = [];
            for (const key of Object.keys(options.userOrganizationProfile)) {
                if (key && key != "extensionFields") {
                    rqitems.push(` upper(uop."${key}") = upper($${queryParamIndex++}) `);
                    queryParams.push(options.userOrganizationProfile[key]);
                }
            }
            if (rqitems.length > 0) {
                query += " AND (" + rqitems.join(" AND ") + ")";
            }
            if (options.userOrganizationProfile.extensionFields?.length) {
                let extitems = [];
                for (const filt of options.userOrganizationProfile.extensionFields) {
                    if (filt.name === "birthDateUtc" || filt.name === "employmentStartUtc" || filt.name === "employmentEndUtc") {
                        extitems.push(` uop."${filt.name}" > $${queryParamIndex++}`);
                        extitems.push(` uop."${filt.name}" < $${queryParamIndex++}`);
                        queryParams.push(filt.value.start);
                        queryParams.push(filt.value.end);
                    }
                    else {
                        let column = `uop."extensionFields"->>'` + filt.name + "'";
                        extitems.push(` unaccent(upper(${column}))::TEXT = unaccent(upper($${queryParamIndex++})) `);
                        queryParams.push(filt.value);
                    }
                }
                if (extitems.length > 0) {
                    query += "  AND (" + extitems.join(" AND ") + ")";
                }
            }
        }
        if (options.organizationUnit && options.organizationUnit.id) {
            let rqitems = [];
            rqitems.push(`ou."id" = $${queryParamIndex++}`);
            queryParams.push(options.organizationUnit.id);
            if (options.organizationUnit) {
                rqitems.push(`ou."ancestorIds" like $${queryParamIndex++}`);
                queryParams.push("%" + options.organizationUnit.id + "%");
            }
            query += `
				AND uo.id in (
					SELECT uoou."userOrganizationId" 
					FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}" uoou
					INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" ou
						ON ou."id" = uoou."organizationUnitId"
					WHERE uoou."deletedAt" IS NULL
						AND ou."deletedAt" IS NULL
						AND ( ${rqitems.join(" OR ")} )
				)
			`;
        }
        if (options.organizationUnits?.length) {
            let rqitems = [];
            for (const unit of options.organizationUnits) {
                rqitems.push(`ou."id" = $${queryParamIndex++}`);
                queryParams.push(unit.id);
                if (unit.includeDescendants) {
                    rqitems.push(`ou."ancestorIds" like $${queryParamIndex++}`);
                    queryParams.push("%" + unit.id + "%");
                }
            }
            query += `
				AND uo.id in (
					SELECT uoou."userOrganizationId" 
					FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}" uoou
					INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" ou
						ON ou."id" = uoou."organizationUnitId"
					WHERE uoou."deletedAt" IS NULL
						AND ou."deletedAt" IS NULL
						AND ( ${rqitems.join(" OR ")} )
				)
			`;
        }
        if (options.userGroup && options.userGroup.id) {
            query += `
				AND uo.id in (
					SELECT uguo."userOrganizationId" 
					FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupUserOrganizations}" uguo
					WHERE uguo."deletedAt" IS NULL
						AND uguo."userGroupId" = $${queryParamIndex++}
				)
			`;
            queryParams.push(options.userGroup.id);
        }
        if (options.userGroupIds?.length) {
            query += `
				AND uo.id in (
					SELECT uguo."userOrganizationId" 
					FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupUserOrganizations}" uguo
					WHERE uguo."deletedAt" IS NULL
						AND uguo."userGroupId" = ANY($${queryParamIndex++}::UUID[])
				)
			`;
            queryParams.push(options.userGroupIds);
        }
        if (options.workplan?.id) {
            query += `
				AND uo."userId" in (
					SELECT uwp."userId" 
					FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userWorkPlans}" uwp
					WHERE uwp."workPlanId" = $${queryParamIndex++}
				)
			`;
            queryParams.push(options.workplan.id);
        }
        if (options.workplans?.length) {
            query += `
				AND uo."userId" in (
					SELECT uwp."userId"
					FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userWorkPlans}" uwp
					WHERE uwp."workPlanId" = ANY ($${queryParamIndex++}::UUID[])
				)
			`;
            queryParams.push(options.workplans);
        }
        if (options.profileFilter) {
            let profileFilter = options.profileFilter
                .trim()
                .replace(/[\r\n\t]/g, " ")
                .replace(/\s/g, "%");
            query += ` AND UNACCENT(UPPER(uop."profileText"::text)) ILIKE UNACCENT (UPPER($${queryParamIndex++}))
            `;
            queryParams.push("%" + profileFilter + "%");
        }
        if (options.status && options.status !== dal_constants_1.DalConstants.IdentityStatusType.All) {
            query += `
                AND  uo."isDisabled" = $${queryParamIndex++}
            `;
            if (options.status === dal_constants_1.DalConstants.IdentityStatusType.Disabled) {
                queryParams.push(`TRUE`);
            }
            else {
                queryParams.push(`FALSE`);
            }
        }
        if (options.userOrganizationCredential) {
            const credentialWhereQ = [];
            credentialWhereQ.push(` uoc."deletedAt" IS NULL `);
            credentialWhereQ.push(` uoc.type = $${queryParamIndex++} `);
            queryParams.push(options.userOrganizationCredential.type);
            if (options.userOrganizationCredential.data) {
                credentialWhereQ.push(` uoc."data" =  $${queryParamIndex++} `);
                queryParams.push(options.userOrganizationCredential.data);
            }
            if (options.userOrganizationCredential.extensionFields?.length) {
                for (const extensionField of options.userOrganizationCredential.extensionFields) {
                    let fieldValue = extensionField.value;
                    credentialWhereQ.push(` uoc."extensionFields"->>'${extensionField.name}' ilike $${queryParamIndex++} `);
                    if (fieldValue && fieldValue.id && fieldValue.value) {
                        queryParams.push("%" + fieldValue.id + "%");
                    }
                    else if (fieldValue) {
                        queryParams.push(fieldValue);
                    }
                }
            }
            if (options.userOrganizationCredential.customFields?.length) {
                for (const customField of options.userOrganizationCredential.customFields) {
                    let fieldValue = customField.value;
                    credentialWhereQ.push(` uoc."customField"->>'${customField.name}' ilike $${queryParamIndex++} `);
                    if (fieldValue && fieldValue.id && fieldValue.value) {
                        queryParams.push("%" + fieldValue.id + "%");
                    }
                    else if (fieldValue) {
                        queryParams.push(fieldValue);
                    }
                }
            }
            query += `
			AND uo."userId" IN (
				SELECT uoc."userId"
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials}" uoc
				WHERE ${credentialWhereQ.join(` AND `)}
			)
		`;
        }
        return { query, queryParams };
    }
    async getUserForbiddances(userId, organizationId) {
        return (await this._pgPool.query(`
        SELECT id, "userId", "regionId", "credentialIds", "startUtc" as "startDateISO", "endUtc" as "endDateISO" from "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationForbiddances}"
        WHERE "startUtc" < now() 
        AND "endUtc" > now()
        AND "deletedAt" IS NULL
        AND "userId" = $1
        AND "organizationId" = $2`, [userId, organizationId])).rows;
    }
    async getUserOrganizationCredential(organizationId, credentialData, credentialType) {
        let credentialSearchValues = await this.generateCredentialSearch(organizationId, credentialData, credentialType);
        let queryParams = [];
        let queryParamIndex = 1;
        let query = `SELECT * FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials}"
            WHERE "organizationId" = $${queryParamIndex++} AND "deletedAt" IS NULL `;
        queryParams.push(organizationId);
        if (credentialSearchValues?.length) {
            const orConditions = [];
            for (const credential of credentialSearchValues) {
                orConditions.push(` (upper(unaccent(data)) ilike upper(unaccent($${queryParamIndex++})))`);
                queryParams.push("%" + credential + "%");
            }
            query += ` AND  (${orConditions.join(" OR ")}) `;
        }
        if (credentialType !== null && credentialType !== undefined) {
            query += `
            AND type = $${queryParamIndex}`;
            queryParams.push(credentialType);
        }
        const { rows, rowCount } = await this._pgPool.query(query, queryParams);
        if (rowCount === 0) {
            return null;
        }
        else {
            return rows[0];
        }
    }
    async addAccessRightsToUser(params) {
        await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
            const userFilter = (0, dal_access_psql_common_1.getReportUserFilterForPgClient)({
                organizationId: params.organizationId,
                requesterUserId: params.requesterUserId,
                idBasedUserFilter: {
                    userIds: [params.userId],
                    applyOrganizationUnitFilterHierarchically: true,
                    userOrganizationStatus: dal_constants_1.DalConstants.IdentityStatusType.All,
                },
                requiredOrganizationUnitWidePermissions: ["i:b"],
                requiredOrganizationWidePermissions: ["i:b"],
                bindingKeys: [],
                specificSelectItems: ["userId"],
            });
            const { rows: UserFilter } = await trx.query(userFilter.query, userFilter.bindingKeys);
            const allowedUserIds = UserFilter.map((filter) => filter.userId);
            if (!allowedUserIds.includes(params.userId)) {
                (0, dal_access_error_1.throwDbAccessAuthorizationErrorTr)("ERRORS.IDENTITY.NOT_AUTH_FOR_SPECIFIED_USER");
            }
            const oldIdentity = await this.getIdentityDetailed(params.organizationId, params.requesterUserId, {
                userId: params.userId,
                hasOrganizationWideRight: true,
                hasIdentityFullWrite: false,
                permittedUnitIds: null,
                hasOrganizationWideRightToSeePassiveUsers: false,
                trx,
            });
            await this.addAccessRightsToUserImp({ ...params, trx });
            const newIdentity = await this.getIdentityDetailed(params.organizationId, params.requesterUserId, {
                userId: params.userId,
                hasOrganizationWideRight: true,
                hasIdentityFullWrite: false,
                permittedUnitIds: null,
                hasOrganizationWideRightToSeePassiveUsers: false,
                trx,
            });
            await this.logProfileUpdate(params.organizationId, params.requesterUserId, oldIdentity, newIdentity);
        }, params.requesterUserId, params.organizationId);
    }
    async addAccessRightsToUserImp(params) {
        const accessPointBasics = await dal_manager_1.dbManager.accessAccessControlPoint.getAccessControlPointBasics(params.organizationId, params.accessPointIds);
        if (params.accessRight.remoteAccess && accessPointBasics.some((ap) => ap.remoteAvailable === false)) {
            (0, dal_access_error_1.throwDbAccessBadRequestErrorTr)(`ERRORS.IDENTITY.REMOTE_ACCESS_UNABLE`, {
                ap: accessPointBasics
                    .filter((ap) => ap.remoteAvailable === false)
                    .map((ap) => ap.id)
                    .join(", "),
            });
            return;
        }
        const userAccessRights = await dal_manager_1.dbManager.accessAccessControlPoint.getUserAccessControlPointRights(params.organizationId, params.userId);
        const alreadyHasAccessRight = userAccessRights.filter((accessRight) => params.accessPointIds.includes(accessRight.accessControlPointId));
        if (alreadyHasAccessRight.length > 0) {
            (0, dal_access_error_1.throwDbAccessBadRequestErrorTr)(`ERRORS.IDENTITY.ALREADY_HAS_ACCESS_RIGHT`, { ap: alreadyHasAccessRight.map((ar) => ar.accessControlPointId).join(",") });
            return;
        }
        await params.trx.query(`
			INSERT INTO "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userAccessRights}"
			(id, "createdAt", "updatedAt", "userId", "accessControlPointId", "remoteAccess", "read", "access", "grant", "config", "snapshot")
			SELECT uuid_generate_v4(), now(), now(), $1, id, $2, $3, $4, $5, $6, $7 FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessControlPoints}"
			WHERE id = ANY($8::uuid[])
		`, [
            params.userId,
            params.accessRight.remoteAccess ?? false,
            params.accessRight.read ?? false,
            params.accessRight.access ?? false,
            params.accessRight.grant ?? false,
            params.accessRight.config ?? false,
            params.accessRight.snapshot ?? false,
            params.accessPointIds,
        ]);
    }
    async updateAcessRightsToUser(params) {
        await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
            const userFilter = (0, dal_access_psql_common_1.getReportUserFilterForPgClient)({
                organizationId: params.organizationId,
                requesterUserId: params.requesterUserId,
                idBasedUserFilter: {
                    userIds: [params.userId],
                    applyOrganizationUnitFilterHierarchically: true,
                    userOrganizationStatus: dal_constants_1.DalConstants.IdentityStatusType.All,
                },
                requiredOrganizationUnitWidePermissions: ["i:b"],
                requiredOrganizationWidePermissions: ["i:b"],
                bindingKeys: [],
                specificSelectItems: ["userId"],
            });
            const { rows: UserFilter } = await trx.query(userFilter.query, userFilter.bindingKeys);
            const allowedUserIds = UserFilter.map((filter) => filter.userId);
            if (!allowedUserIds.includes(params.userId)) {
                (0, dal_access_error_1.throwDbAccessAuthorizationErrorTr)("ERRORS.IDENTITY.NOT_AUTH_FOR_SPECIFIED_USER");
            }
            const oldIdentity = await this.getIdentityDetailed(params.organizationId, params.requesterUserId, {
                userId: params.userId,
                hasOrganizationWideRight: true,
                hasIdentityFullWrite: false,
                permittedUnitIds: null,
                hasOrganizationWideRightToSeePassiveUsers: false,
                trx,
            });
            await this.updateAcessRightsToUserImp({ ...params, trx });
            const newIdentity = await this.getIdentityDetailed(params.organizationId, params.requesterUserId, {
                userId: params.userId,
                hasOrganizationWideRight: true,
                hasIdentityFullWrite: false,
                permittedUnitIds: null,
                hasOrganizationWideRightToSeePassiveUsers: false,
                trx,
            });
            await this.logProfileUpdate(params.organizationId, params.requesterUserId, oldIdentity, newIdentity);
        }, params.requesterUserId, params.organizationId);
    }
    async updateAcessRightsToUserImp(params) {
        const [accessPointBasics, userAccessRights] = await Promise.all([
            dal_manager_1.dbManager.accessAccessControlPoint.getAccessControlPointBasics(params.organizationId, params.accessPointIds),
            this.getUserAccessRights({ organizationId: params.organizationId, userId: params.userId, trx: params.trx }),
        ]);
        const updatedAccessRights = userAccessRights
            .filter((uar) => params.accessPointIds.includes(uar.accessControlPointId))
            .map((uar) => {
            let updatedRights = {
                access: params.accessRight.access ?? uar.access,
                read: params.accessRight.read ?? uar.read,
                grant: params.accessRight.grant ?? uar.grant,
                config: params.accessRight.config ?? uar.config,
                snapshot: params.accessRight.snapshot ?? uar.snapshot,
                remoteAccess: params.accessRight.remoteAccess ?? uar.remoteAccess,
            };
            const isUserAlsoNeedReadRight = (updatedRights.grant || updatedRights.config || updatedRights.snapshot) && !updatedRights.read;
            if (isUserAlsoNeedReadRight) {
                (0, dal_access_error_1.throwDbAccessBadRequestErrorTr)(`ERRORS.IDENTITY.NEED_READ_PERMISSION`);
            }
            const isAllValuesAreFalse = !Object.values(updatedRights).reduce((result, value) => result || value);
            if (isAllValuesAreFalse) {
                (0, dal_access_error_1.throwDbAccessBadRequestErrorTr)(`ERRORS.IDENTITY.NO_ACCESS_RIGHTS_LEFT`, { ap: uar.accessControlPointId });
            }
            return { ...updatedRights, ...{ accessPointId: uar.accessControlPointId } };
        });
        const identityHasNoAccessRightForGivenAccessPoints = accessPointBasics.map((acp) => acp.id).some((apId) => !userAccessRights.map((uar) => uar.accessControlPointId).includes(apId));
        if (identityHasNoAccessRightForGivenAccessPoints) {
            (0, dal_access_error_1.throwDbAccessBadRequestErrorTr)(`ERRORS.IDENTITY.NO_ACCESS_RIGHT_FOR_GIVEN_ACCESS_POINT`, {
                ap: accessPointBasics
                    .map((acp) => acp.id)
                    .filter((apId) => !userAccessRights.map((uar) => uar.accessControlPointId).includes(apId))
                    .join(", "),
            });
        }
        const accessPointsHasNoRemoteAccessFeature = params.accessRight.remoteAccess && accessPointBasics.some((ap) => ap.remoteAvailable === false);
        if (accessPointsHasNoRemoteAccessFeature) {
            (0, dal_access_error_1.throwDbAccessBadRequestErrorTr)(`ERRORS.IDENTITY.REMOTE_ACCESS_UNABLE`, {
                ap: accessPointBasics
                    .filter((ap) => ap.remoteAvailable === false)
                    .map((ap) => ap.id)
                    .join(", "),
            });
            return;
        }
        await Promise.all(updatedAccessRights.map(async (accessRight) => {
            await params.trx.query(`UPDATE "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userAccessRights}"
					SET "updatedAt" = $1,
						"remoteAccess" = $2,
						read = $3,
						access = $4,
						"grant" = $5,
						config = $6,
						snapshot = $7
					WHERE "userId" = $8
				   AND "deletedAt" IS NULL
				   AND "accessControlPointId" = $9
				`, [
                new Date(),
                accessRight.remoteAccess,
                accessRight.read,
                accessRight.access,
                accessRight.grant,
                accessRight.config,
                accessRight.snapshot,
                params.userId,
                accessRight.accessPointId,
            ]);
        }));
        const { rows } = await params.trx.query(`
				SELECT "accessControlPointId", (COUNT(*) FILTER (WHERE "grant"))::SMALLINT "grantCount", (COUNT(*) FILTER (WHERE "config"))::SMALLINT "configCount"
				FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userAccessRights}"
				WHERE "accessControlPointId" = ANY($1::uuid[])
				GROUP BY "accessControlPointId"
				`, [params.accessPointIds]);
        const AccessPointsWithoutGrantOrConfigRights = rows.filter((row) => row.configCount === 0 || row.grantCount === 0);
        const AccessPointsWithoutAnyPermission = params.accessPointIds.filter((acp) => !rows.map((row) => row.accessControlPointId).includes(acp));
        if (AccessPointsWithoutGrantOrConfigRights.length > 0 || AccessPointsWithoutAnyPermission.length > 0) {
            (0, dal_access_error_1.throwDbAccessBadRequestErrorTr)(`ERRORS.IDENTITY.NO_USER_WITH_GRANT_OR_CONFIG_PERMISSION`, {
                ap: [...new Set(AccessPointsWithoutGrantOrConfigRights.map((a) => a.accessControlPointId).concat(AccessPointsWithoutAnyPermission))].join(", "),
            });
        }
    }
    async deleteAccessRightsFromUser(params) {
        await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
            const userFilter = (0, dal_access_psql_common_1.getReportUserFilterForPgClient)({
                organizationId: params.organizationId,
                requesterUserId: params.requesterUserId,
                idBasedUserFilter: {
                    userIds: [params.userId],
                    applyOrganizationUnitFilterHierarchically: true,
                    userOrganizationStatus: dal_constants_1.DalConstants.IdentityStatusType.All,
                },
                requiredOrganizationUnitWidePermissions: ["i:b"],
                requiredOrganizationWidePermissions: ["i:b"],
                bindingKeys: [],
                specificSelectItems: ["userId"],
            });
            const { rows: UserFilter } = await trx.query(userFilter.query, userFilter.bindingKeys);
            const allowedUserIds = UserFilter.map((filter) => filter.userId);
            if (!allowedUserIds.includes(params.userId)) {
                (0, dal_access_error_1.throwDbAccessAuthorizationErrorTr)("ERRORS.IDENTITY.NOT_AUTH_FOR_SPECIFIED_USER");
            }
            const oldIdentity = await this.getIdentityDetailed(params.organizationId, params.requesterUserId, {
                userId: params.userId,
                hasOrganizationWideRight: true,
                hasIdentityFullWrite: false,
                permittedUnitIds: null,
                hasOrganizationWideRightToSeePassiveUsers: false,
                trx,
            });
            await this.deleteAccessRightsFromUserImp({ ...params, trx });
            const newIdentity = await this.getIdentityDetailed(params.organizationId, params.requesterUserId, {
                userId: params.userId,
                hasOrganizationWideRight: true,
                hasIdentityFullWrite: false,
                permittedUnitIds: null,
                hasOrganizationWideRightToSeePassiveUsers: false,
                trx,
            });
            await this.logProfileUpdate(params.organizationId, params.requesterUserId, oldIdentity, newIdentity);
        }, params.requesterUserId, params.organizationId);
    }
    async deleteAccessRightsFromUserImp(params) {
        await params.trx.query(`UPDATE "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userAccessRights}"
			SET "deletedAt" = $1,
				"updatedAt" = $1
			WHERE "userId" = $2
			AND "deletedAt" IS NULL
			AND "accessControlPointId" = ANY($3::uuid[])
			`, [new Date(), params.userId, params.accessPointIds]);
        const { rows } = await params.trx.query(`
		SELECT "accessControlPointId", (COUNT(*) FILTER (WHERE "grant"))::SMALLINT "grantCount", (COUNT(*) FILTER (WHERE "config"))::SMALLINT "configCount"
		FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userAccessRights}"
		WHERE "accessControlPointId" = ANY($1::uuid[])
		AND "deletedAt" IS NULL
		GROUP BY "accessControlPointId"
		`, [params.accessPointIds]);
        const AccessPointsWithoutGrantOrConfigRights = rows.filter((row) => row.configCount === 0 || row.grantCount === 0);
        const AccessPointsWithoutAnyPermission = params.accessPointIds.filter((acp) => !rows.map((row) => row.accessControlPointId).includes(acp));
        if (AccessPointsWithoutGrantOrConfigRights.length > 0 || AccessPointsWithoutAnyPermission.length > 0) {
            (0, dal_access_error_1.throwDbAccessBadRequestErrorTr)(`ERRORS.IDENTITY.NO_USER_WITH_GRANT_OR_CONFIG_PERMISSION`, {
                ap: [...new Set(AccessPointsWithoutGrantOrConfigRights.map((a) => a.accessControlPointId).concat(AccessPointsWithoutAnyPermission))].join(", "),
            });
        }
    }
    async findTransformedMifareCredentialDatasOfUser(organizationId, params, trx) {
        let dbResult = (await trx.query(`
				SELECT id, type, data, "credentialNumber", "expiresOn",	"extensionFields", 
					"customFields", note, "specialData",	"specialDataSecondary",	"groupNumber"
				FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials}"
				WHERE "userId" = $1
					AND "deletedAt" IS NULL
					AND type = $2
			`, [params.userId, dal_constants_1.DalConstants.CredentialType.MiFare])).rows;
        const result = [];
        const organization = await dal_manager_1.dbManager.accessOrganization.getOrganization(organizationId);
        for (const c of dbResult) {
            const dataConverter = organization.credentialTypes.find((ct) => ct.type === c.type)?.dataConverter;
            if (!dataConverter) {
                if (c.type === dal_constants_1.DalConstants.CredentialType.MiFare || c.type === dal_constants_1.DalConstants.CredentialType.ProximityCard) {
                    result.push({
                        id: c.id,
                        type: c.type,
                        data: parseInt(c.data, 16).toString(),
                    });
                }
                else {
                    result.push({
                        id: c.id,
                        type: c.type,
                        data: c.data,
                    });
                }
            }
            else {
                result.push({
                    id: c.id,
                    type: c.type,
                    data: c.data,
                });
            }
        }
        return result;
    }
    async getCredentialDetails(params) {
        const { rows, rowCount } = await params.trx.query(`SELECT data, type, "userId" FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials}"
			WHERE id = $1
		`, [params.credentialId]);
        return rowCount > 0
            ? {
                data: rows[0].data,
                userId: rows[0].userId,
                type: rows[0].type,
            }
            : null;
    }
    async getUserAccessRight(params) {
        const { rows, rowCount } = await params.trx.query(`
			SELECT "userId", "accessControlPointId" as "accessPointId" FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userAccessRights}"
			WHERE id = $1
		`, [params.accessRightId]);
        return rowCount > 0 ? rows[0] : null;
    }
    async assignCredentialToUser(organizationId, requesterUserId, trx, params) {
        const addedCredentials = [];
        const qValues = [];
        const qb = [];
        let qx = 1;
        const userFilter = (0, dal_access_psql_common_1.getReportUserFilterForPgClient)({
            organizationId,
            requesterUserId,
            idBasedUserFilter: {
                userIds: [params.userId],
                applyOrganizationUnitFilterHierarchically: true,
                userOrganizationStatus: dal_constants_1.DalConstants.IdentityStatusType.All,
            },
            requiredOrganizationUnitWidePermissions: ["i:b"],
            requiredOrganizationWidePermissions: ["i:b"],
            bindingKeys: [],
            specificSelectItems: ["userId"],
        });
        const { rows: UserFilter } = await trx.query(userFilter.query, userFilter.bindingKeys);
        const allowedUserIds = UserFilter.map((filter) => filter.userId);
        if (!allowedUserIds.includes(params.userId)) {
            (0, dal_access_error_1.throwDbAccessAuthorizationErrorTr)("ERRORS.IDENTITY.NOT_AUTH_FOR_SPECIFIED_USER");
        }
        for (const credential of params.credentials) {
            const { rows: existCredentialRows } = await trx.query(`
				SELECT data FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials}"
				WHERE type = $1
				AND "deletedAt" IS NULL
				AND data = $2
			`, [credential.type, credential.data]);
            if (existCredentialRows.length > 0) {
                (0, dal_access_error_1.throwDbAccessConflictErrorTr)("ERRORS.IDENTITY.DUPLICATECREDENTIAL", { data: existCredentialRows[0].data });
            }
            const credentialDBItem = {
                id: uuid_1.default.v4(),
                organizationId,
                userId: params.userId,
                type: credential.type,
                data: credential.data,
                expiresOn: credential.expirationUtc,
                credentialNumber: credential.credentialNumber,
                note: credential.note,
                groupNumber: credential.groupNumber,
            };
            qValues.push(`($${qx++}, now(), now(), $${qx++},$${qx++},$${qx++},$${qx++},$${qx++},$${qx++},$${qx++},$${qx++})`);
            qb.push(credentialDBItem.id, organizationId, params.userId, credentialDBItem.type, credentialDBItem.data, credentialDBItem.expiresOn, credentialDBItem.credentialNumber, credentialDBItem.note, credentialDBItem.groupNumber);
            addedCredentials.push({ id: credentialDBItem.id, data: credentialDBItem.data, type: credentialDBItem.type });
        }
        await trx.query(`INSERT INTO "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials}"
		(id, "createdAt", "updatedAt", "organizationId", "userId", type, data, "expiresOn", "credentialNumber",  note, "groupNumber")
		VALUES ${qValues.join(" , ")}`, qb);
        for (const credential of addedCredentials) {
            dal_manager_1.dbManager.accessLog.addUserActionHistoryItem({
                oId: organizationId,
                o: requesterUserId,
                u: new Date(),
                c: dal_constants_1.DalConstants.UserActionCategory.Main,
                tg: ["Identity", "Credential"],
                t: dal_constants_1.DalConstants.UserMainActionType.AddCredential,
                d: {
                    id: params.userId,
                    credentialId: credential.id,
                    c: JSON.stringify([
                        {
                            f: credential.type,
                            o: null,
                            n: dal_constants_1.DalConstants.credentialTypesWithMeaningfulData.includes(credential.type) ? credential.data : "*****",
                            i: true,
                        },
                    ]),
                },
            });
        }
        return addedCredentials;
    }
    async unAssignCredentialFromUser(organizationId, requesterUserId, trx, params) {
        const userFilter = (0, dal_access_psql_common_1.getReportUserFilterForPgClient)({
            organizationId,
            requesterUserId,
            idBasedUserFilter: {
                userIds: [params.userId],
                applyOrganizationUnitFilterHierarchically: true,
                userOrganizationStatus: dal_constants_1.DalConstants.IdentityStatusType.All,
            },
            requiredOrganizationUnitWidePermissions: ["i:b"],
            requiredOrganizationWidePermissions: ["i:b"],
            bindingKeys: [],
            specificSelectItems: ["userId"],
        });
        const { rows: UserFilter } = await trx.query(userFilter.query, userFilter.bindingKeys);
        const allowedUserIds = UserFilter.map((filter) => filter.userId);
        if (!allowedUserIds.includes(params.userId)) {
            (0, dal_access_error_1.throwDbAccessAuthorizationErrorTr)("ERRORS.IDENTITY.NOT_AUTH_FOR_SPECIFIED_USER");
        }
        const { rows, rowCount } = await trx.query(`UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials}"
				SET "deletedAt" = now(),
					"updatedAt" = now()
			WHERE id = ANY($1::uuid[])
			AND "userId" = $2
			AND "deletedAt" IS NULL
			RETURNING id, data, type`, [params.credentialIds, params.userId]);
        if (rowCount > 0) {
            await dal_manager_1.dbManager.accessLog.addUserActionHistoryItem({
                oId: organizationId,
                o: requesterUserId,
                u: new Date(),
                c: dal_constants_1.DalConstants.UserActionCategory.Main,
                tg: ["Identity", "Credential"],
                t: dal_constants_1.DalConstants.UserMainActionType.RemoveCredential,
                d: {
                    id: params.userId,
                    removedCredentialIds: rows.map((row) => row.id),
                    c: JSON.stringify(rows.map((m) => {
                        return {
                            f: m.type,
                            o: dal_constants_1.DalConstants.credentialTypesWithMeaningfulData.includes(m.type) ? m.data : "*****",
                            n: null,
                            i: true,
                        };
                    })),
                },
            });
        }
        return rows.map((row) => row.id);
    }
    async getCredentialOwnerDetails(params) {
        const { rows } = await params.trx.query(`SELECT uop.name || ' ' || uop.surname as "userFullname", ovp.name "visitorFullname", uoc.data FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials}" uoc
			 INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo
			 	ON uo."userId" = uoc."userId"
			 LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" uop
				ON uo.id = uop."userOrganizationId"
			 LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorProfiles}" ovp
				ON uo.id = ovp."userOrganizationId"
			 WHERE uoc.data = $1
			 AND uoc."deletedAt" IS NULL`, [params.credentialData]);
        return rows[0];
    }
    async updateUsername(organizationId, userId, username, trx) {
        const { rows: userRows, rowCount } = await trx.query(`SELECT u.username, u."accountEnabled" 
				 FROM "${organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.users} u 
				 INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo
				 	ON u.id = uo."userId"
				 INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" uop
				 	ON uo.id = uop."userOrganizationId"
				 WHERE u.id = $1
				 AND uop."deletedAt" IS NULL
				 AND uo."deletedAt" IS NULL
				 AND u."deletedAt" IS NULL`, [userId]);
        if (rowCount === 0) {
            (0, dal_access_error_1.throwDbAccessConflictErrorTr)("ERRORS.IDENTITY.USERNOTFOUND");
        }
        else if (!userRows[0].accountEnabled) {
            (0, dal_access_error_1.throwDbAccessConflictErrorTr)("ERRORS.IDENTITY.NOUSERACCOUNTTOEDIT");
        }
        const user = userRows[0];
        const doesUsernameExists = (await trx.query(`SELECT count(*)::integer as cnt FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}"
				 WHERE lower(username) = lower($1)
				 AND id != $2
				 AND "deletedAt" IS NULL`, [username, userId])).rows[0].cnt;
        if (doesUsernameExists) {
            (0, dal_access_error_1.throwDbAccessConflictErrorTr)("ERRORS.IDENTITY.USERNAMEEXIST");
        }
        await trx.query(`UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}"
				SET "updatedAt" = $1,
					username = $2
			 WHERE id = $3;`, [luxon_1.DateTime.local().toSQL({ includeOffset: true }), username, userId]);
        await trx.query(`UPDATE "public"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userList}"
			 	SET username = $1
			 WHERE id = $2;`, [username, userId]);
        return { oldUsername: userRows[0].username };
    }
    async deleteUserAccount(organizationId, userId, trx) {
        const { rows, rowCount } = await trx.query(`SELECT "username", "accountEnabled" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}"
			 WHERE id = $1`, [userId]);
        if (rowCount === 0) {
            (0, dal_access_error_1.throwDbAccessConflictErrorTr)("ERRORS.IDENTITY.USERNOTFOUND");
        }
        else if (!rows[0].accountEnabled) {
            (0, dal_access_error_1.throwDbAccessConflictErrorTr)("ERRORS.IDENTITY.NOUSERACCOUNTTOEDIT");
        }
        await trx.query(`UPDATE "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}"
			 SET "updatedAt"	= $1,
				 username = $2,
				 "accountEnabled" = $3,
				 salt = $4,
				 "hashedPassword" = $5
			 WHERE id = $6`, [luxon_1.DateTime.local().toSQL({ includeOffset: true }), null, false, null, null, userId]);
        await trx.query(`UPDATE "public"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userList}"
			 SET username = $1
			 WHERE id = $2`, [null, userId]);
        return { deletedUsername: rows[0].username };
    }
}
exports.PSQLDalAccessIdentity = PSQLDalAccessIdentity;
