"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PSQLDalAccessAccessLog = void 0;
const lodash_1 = __importDefault(require("lodash"));
const uuid_1 = __importDefault(require("uuid"));
const api_error_1 = require("../../../api/api.error");
const app_enums_1 = require("../../../app.enums");
const restapi_1 = require("../../../lib/es/models/restapi");
const dal_constants_1 = require("../../dal.constants");
const dal_manager_1 = require("../../dal.manager");
const dal_db_armon_schema_1 = require("../../db/armon/dal.db.armon.schema");
const predefined_permissions_1 = require("../../db/predefined/predefined.permissions");
const predefined_roles_1 = require("../../db/predefined/predefined.roles");
const dal_access_error_1 = require("../dal.access.error");
const dal_access_rdb_accessLog_1 = require("../rdb/dal.access.rdb.accessLog");
class PSQLDalAccessAccessLog extends dal_access_rdb_accessLog_1.RDBDalAccessAccessLog {
    async filterVisitorAccessLogs(organizationId, filter, requesterUserId, authorizeForFilter, shouldReturnSnapshots, userAccessRights) {
        if (authorizeForFilter) {
            let authResultForFilter = await dal_manager_1.dbManager.accessFunctions.dbFuncAuthorizeUserFor({
                organizationId: organizationId,
                userId: requesterUserId,
                accessControlPointIds: filter.accessControlPointIds,
            });
            if (!authResultForFilter.result) {
                throw (0, api_error_1.generateForbiddenError)({ message: "Unauthorized filter items" });
            }
        }
        return await this.getAccessLogsReport(organizationId, {
            startUtc: filter.startUtc,
            endUtc: filter.endUtc,
            userIds: filter.userIds,
            accessControlPointIds: filter.accessControlPointIds,
            sortDateDesc: filter.sortDateDesc,
            accessResult: filter.accessResult,
            take: filter.take,
            skip: filter.skip,
            showOnlyVisitors: false,
            direction: filter.direction,
            showOnlyRemoteAccess: filter.showOnlyRemoteAccess,
            showOnlyManuallyInserted: filter.showOnlyManuallyInserted,
            showOnlyExitButtons: filter.showOnlyExitButtons,
            credentialTypes: filter.credentialTypes,
        }, shouldReturnSnapshots, userAccessRights);
    }
    async filterAccessLogs(organizationId, filter, requesterUserId, authorizeForFilter, shouldReturnSnapshots, userAccessRights) {
        if (authorizeForFilter) {
            let authResultForFilter = await dal_manager_1.dbManager.accessFunctions.dbFuncAuthorizeUserFor({
                organizationId: organizationId,
                userId: requesterUserId,
                organizationUnitIds: filter.organizationUnitIds,
                accessControlPointIds: filter.accessControlPointIds,
                userIds: filter.userIds,
                userGroupIds: filter.userGroupIds,
            });
            if (!authResultForFilter.result) {
                throw (0, api_error_1.generateForbiddenError)({ message: "Unauthorized filter items" });
            }
            if (lodash_1.default.isEmpty(filter.organizationUnitIds) && lodash_1.default.isEmpty(filter.userGroupIds) && lodash_1.default.isEmpty(filter.userIds)) {
                let userOrganizationUnits = await this.dbClient
                    .withSchema(organizationId)
                    .from("users as u")
                    .innerJoin("userOrganizations as uo", "uo.userId", "u.id")
                    .innerJoin("userOrganizationOrganizationUnits as uoou", "uoou.userOrganizationId", "uo.id")
                    .innerJoin("organizationUnits as ou", (join) => {
                    join.on(this.dbClient.raw(`ou."ancestorIds" like '%' || uoou."organizationUnitId" || '%'`)).orOn("ou.id", "uoou.organizationUnitId");
                })
                    .whereNull("u.deletedAt")
                    .whereNull("uoou.deletedAt")
                    .whereNull("uo.deletedAt")
                    .where("uo.organizationId", organizationId)
                    .where("u.id", requesterUserId)
                    .select("ou.id");
                filter.organizationUnitIds = userOrganizationUnits.map((u) => u.id);
            }
        }
        let filterUsers = await dal_manager_1.dbManager.accessFunctions.dbFuncCollectUsersForAccessReportFilter({
            organizationId: organizationId,
            userIds: filter.userIds,
            organizationUnitIds: filter.organizationUnitIds,
            userGroupIds: filter.userGroupIds,
            filterOrganizationUnitMembersHierarchically: filter.filterOrganizationUnitMembersHierarchically,
        });
        let filterUserIds = (filterUsers && filterUsers.length) > 0 ? filterUsers.map((f) => f.userId) : [];
        return await this.getAccessLogsReport(organizationId, {
            startUtc: filter.startUtc,
            endUtc: filter.endUtc,
            userIds: filterUserIds,
            accessControlPointIds: filter.accessControlPointIds,
            sortDateDesc: filter.sortDateDesc,
            accessResult: filter.accessResult,
            take: filter.take,
            skip: filter.skip,
            returnTotalCount: filter.returnTotalCount,
            showOnlyVisitors: filter.showOnlyVisitors,
            direction: filter.direction,
            showOnlyRemoteAccess: filter.showOnlyRemoteAccess,
            showOnlyManuallyInserted: filter.showOnlyManuallyInserted,
            showOnlyExitButtons: filter.showOnlyExitButtons,
            reasons: filter.reasons,
            credentialData: filter.credentialData,
        }, shouldReturnSnapshots, userAccessRights);
    }
    async getArventoAccessLogs(organizationId, filter, requesterUserId, authorizeForFilter, permittedUnitIds) {
        return this.dbClient.transaction(async (trx) => {
            let shouldIncludeUnkownUsers = false;
            if (authorizeForFilter) {
                let authResultForFilter = await dal_manager_1.dbManager.accessFunctions.dbFuncAuthorizeUserFor({
                    organizationId: organizationId,
                    userId: requesterUserId,
                    organizationUnitIds: filter.organizationUnitIds,
                    accessControlPointIds: filter.accessControlPointIds,
                    userIds: filter.userIds,
                    userGroupIds: filter.userGroupIds,
                    regionIds: filter.regionIds,
                });
                if (!authResultForFilter.result) {
                    throw (0, api_error_1.generateForbiddenError)({ message: "Unauthorized filter items" });
                }
                if (lodash_1.default.isEmpty(filter.organizationUnitIds) && lodash_1.default.isEmpty(filter.userGroupIds) && lodash_1.default.isEmpty(filter.userIds)) {
                    filter.organizationUnitIds = permittedUnitIds;
                    filter.filterOrganizationUnitMembersHierarchically = true;
                    if (await this.canUserSeeUnkownUserAccessLogs(organizationId, requesterUserId, permittedUnitIds))
                        shouldIncludeUnkownUsers = true;
                }
            }
            let filterUsers = await dal_manager_1.dbManager.accessFunctions.dbFuncCollectUsersForAccessReportFilter({
                organizationId: organizationId,
                userIds: filter.userIds,
                organizationUnitIds: filter.organizationUnitIds,
                userGroupIds: filter.userGroupIds,
                filterOrganizationUnitMembersHierarchically: filter.filterOrganizationUnitMembersHierarchically,
                transaction: trx,
            });
            let filterUserIds = filterUsers?.map((f) => f.userId) || [];
            if (shouldIncludeUnkownUsers && filterUserIds) {
                filterUserIds.push(dal_db_armon_schema_1.ArmonSchema.unknownCredentialOwnerId);
            }
            let report = await this.getArventoAccessLogsReport(organizationId, {
                pagination: filter.pagination,
                startUtc: filter.startUtc,
                endUtc: filter.endUtc,
                userIds: filterUserIds,
                organizationUnitIds: filter.organizationUnitIds,
                userGroupIds: filter.userGroupIds,
                accessControlPointIds: filter.accessControlPointIds,
                type: filter.type,
                region: filter.region,
                licencePlate: filter.licencePlate,
                isSynchronized: filter.isSynchronized,
                regionIds: filter.regionIds,
            });
            let userCaptions = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, report.items.map((r) => r.userId), trx);
            for (const item of report.items) {
                let captions;
                if (item.userId) {
                    captions = userCaptions.find((a) => a.id == item.userId);
                }
                if (captions) {
                    item.userCaptions = captions.captionLines;
                }
            }
            return report;
        });
    }
    async filterAccessLogsWithUserCaptions(organizationId, filter, requesterUserId, authorizeForFilter, permittedUnitIds) {
        return this.dbClient.transaction(async (trx) => {
            let shouldIncludeUnkownUsers = false;
            if (authorizeForFilter) {
                let authResultForFilter = await dal_manager_1.dbManager.accessFunctions.dbFuncAuthorizeUserFor({
                    organizationId: organizationId,
                    userId: requesterUserId,
                    organizationUnitIds: filter.organizationUnitIds,
                    accessControlPointIds: filter.accessControlPointIds,
                    userIds: filter.userIds,
                    userGroupIds: filter.userGroupIds,
                    regionIds: filter.regionIds,
                });
                if (!authResultForFilter.result) {
                    throw (0, api_error_1.generateForbiddenError)({ message: "Unauthorized filter items" });
                }
                if (lodash_1.default.isEmpty(filter.organizationUnitIds) && lodash_1.default.isEmpty(filter.userGroupIds) && lodash_1.default.isEmpty(filter.userIds)) {
                    filter.organizationUnitIds = permittedUnitIds;
                    filter.filterOrganizationUnitMembersHierarchically = true;
                    if (await this.canUserSeeUnkownUserAccessLogs(organizationId, requesterUserId, permittedUnitIds))
                        shouldIncludeUnkownUsers = true;
                }
            }
            let filterUsers = await dal_manager_1.dbManager.accessFunctions.dbFuncCollectUsersForAccessReportFilter({
                organizationId: organizationId,
                userIds: filter.userIds,
                organizationUnitIds: filter.organizationUnitIds,
                userGroupIds: filter.userGroupIds,
                filterOrganizationUnitMembersHierarchically: filter.filterOrganizationUnitMembersHierarchically,
                transaction: trx,
            });
            let filterUserIds = filterUsers?.map((f) => f.userId) || [];
            if (shouldIncludeUnkownUsers && filterUserIds) {
                filterUserIds.push(dal_db_armon_schema_1.ArmonSchema.unknownCredentialOwnerId);
            }
            let userAccessRights = await dal_manager_1.dbManager.accessAccessControlPoint.getUserAccessControlPointRights(organizationId, requesterUserId);
            let report = await this.getAccessLogsReport(organizationId, {
                startUtc: filter.startUtc,
                endUtc: filter.endUtc,
                userIds: filterUserIds,
                organizationUnitIds: filter.organizationUnitIds,
                userGroupIds: filter.userGroupIds,
                accessControlPointIds: filter.accessControlPointIds,
                sortDateDesc: filter.sortDateDesc,
                accessResult: filter.accessResult,
                take: filter.take,
                skip: filter.skip,
                returnTotalCount: filter.returnTotalCount,
                showOnlyVisitors: filter.showOnlyVisitors,
                direction: filter.direction,
                showOnlyRemoteAccess: filter.showOnlyRemoteAccess,
                showOnlyManuallyInserted: filter.showOnlyManuallyInserted,
                showOnlyExitButtons: filter.showOnlyExitButtons,
                reasons: filter.reasons,
                credentialData: filter.credentialData,
                credentialTypes: filter.credentialTypes,
                manualRecordCreatorUserId: filter.manualRecordCreatorUserId,
                regionIds: filter.regionIds,
            }, true, userAccessRights);
            let result = {
                items: [],
                skip: report.skip,
                take: report.take,
                total: report.total,
            };
            let userCaptions = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, report.items.filter((r) => r.identity).map((r) => r.identity.id), trx);
            for (const item of report.items) {
                let captions;
                if (item.identity) {
                    captions = userCaptions.find((a) => a.id == item.identity.id);
                }
                if (captions) {
                    result.items.push(lodash_1.default.assign({}, item, { userCaptions: captions.captionLines }));
                }
                else {
                    if (item.logUserInfo) {
                        result.items.push(lodash_1.default.assign({}, item, {
                            userCaptions: [
                                {
                                    text: [item.logUserInfo],
                                    color: [],
                                },
                            ],
                            identity: {
                                name: item.logUserInfo,
                            },
                        }));
                    }
                    else {
                        result.items.push(lodash_1.default.assign({}, item, { userCaptions: [] }));
                    }
                }
            }
            return result;
        });
    }
    async getAccessLogsReportNew(organizationId, filter, onData) {
        await dal_manager_1.dbManager.accessLog.getAccessLogsNew(organizationId, filter, async (rows) => {
            let logs = rows.map((i) => {
                return { ...i.log, iUtc: i.insertionUtc };
            });
            let accessControlPointIds = [];
            let userIds = [];
            let credentialIds = [];
            let statusSensorIds = [];
            let counterSensorIds = [];
            for (let index = 0; index < logs.length; index++) {
                let row = logs[index];
                accessControlPointIds.push(row.a);
                userIds.push(row.o);
                if (lodash_1.default.isArray(row.ci)) {
                    for (const cid of row.ci) {
                        credentialIds.push(cid);
                    }
                }
                else {
                    credentialIds.push(row.ci);
                }
                if (row.cx && lodash_1.default.isArray(row.cx)) {
                    for (const credential of row.cx) {
                        credentialIds.push(credential.i);
                    }
                }
                if (row.ss && lodash_1.default.isArray(row.ss)) {
                    statusSensorIds.push(...row.ss);
                }
                if (row.cs && lodash_1.default.isArray(row.cs)) {
                    counterSensorIds.push(...row.cs);
                }
            }
            accessControlPointIds = lodash_1.default.compact(lodash_1.default.uniq(accessControlPointIds));
            userIds = lodash_1.default.compact(lodash_1.default.uniq(userIds));
            credentialIds = lodash_1.default.compact(lodash_1.default.uniq(credentialIds));
            counterSensorIds = lodash_1.default.compact(lodash_1.default.uniq(counterSensorIds));
            statusSensorIds = lodash_1.default.compact(lodash_1.default.uniq(statusSensorIds));
            const statusSensorLogs = await dal_manager_1.dbManager.accessLog.listStatusSensorLogs(organizationId, statusSensorIds);
            const counterSensorLogs = await dal_manager_1.dbManager.accessLog.listCounterSensorLogs(organizationId, counterSensorIds);
            const accessControlPoints = await dal_manager_1.dbManager.accessAccessControlPoint.listAccessControlPointDetails(organizationId, accessControlPointIds);
            let organizationUnitsColumn = `
                (to_json( ARRAY(select json_build_object('id', ou.id, 'name', ou.name) 
                from "${organizationId}"."userOrganizationOrganizationUnits" AS uoou 
                inner join "${organizationId}"."organizationUnits" AS ou on uoou."organizationUnitId" = ou.id
                where uoou."userOrganizationId" = uo.id and uoou."deletedAt" is null
                order by ou.name))) "organizationUnits"
                `;
            let userGroupsColumn = `
                (to_json( ARRAY(select json_build_object(
                    'id', ug.id, 'name', ug.name) 
                    from "${organizationId}"."userGroupUserOrganizations" AS uguo
                    inner join "${organizationId}"."userGroups" AS ug on uguo."userGroupId" = ug.id
                    where uguo."userOrganizationId" = uo.id and uguo."deletedAt" is null
                    order by ug.name))) "userGroups"
                `;
            let userOrganizationProfiles = (await this._pgPool.query(` SELECT uo."userId", uop.name, uop.surname, uop."uniqueId", uop."extensionFields", ${organizationUnitsColumn}, ${userGroupsColumn}
                FROM "${organizationId}"."userOrganizations" AS uo
                INNER JOIN "${organizationId}".users AS u
                ON u.id = uo."userId"
                INNER JOIN "${organizationId}"."userOrganizationProfiles" AS uop
                ON uop."userOrganizationId" = uo.id
                WHERE u.id = ANY ($1)
                AND uo."deletedAt" IS NULL
                AND uop."deletedAt" IS NULL 
            `, [userIds])).rows;
            let visitorProfiles = (await this._pgPool.query(`
                    SELECT uo."userId", ovp."uniqueId", ovp.name, ovp.surname
                    FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" AS uo
                    INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorProfiles}" AS ovp
                    ON ovp."userOrganizationId" = uo.id
                    WHERE uo."organizationId" = $1
                    AND uo."userId" = ANY ($2)
                `, [organizationId, userIds])).rows;
            let credentialParamsIndex = 1;
            let credentialParams = [];
            let credentialInfoQb = `
                    SELECT uo."userId", uoc.data, uoc.id, uoc.type, uoc."expiresOn" FROM "${organizationId}"."userOrganizations" AS uo
                    INNER JOIN "${organizationId}"."userOrganizationCredentials" as uoc
                    ON uoc."userId" = uo."userId"
                    WHERE uo."deletedAt" IS NULL
                    AND uoc."deletedAt" IS NULL
                    AND uoc.id = ANY ($${credentialParamsIndex++})
            `;
            credentialParams.push(credentialIds);
            if (filter.credentialData) {
                credentialInfoQb += `AND upper(unaccent(uoc.data)) = upper(unaccent($${credentialParamsIndex++}))`;
                credentialParams.push(filter.credentialData);
            }
            let credentialInfo = (await this._pgPool.query(credentialInfoQb, credentialParams)).rows;
            let accessLogsReportItems = await Promise.all(logs.map(async (l) => {
                let accessControlPoint = accessControlPoints.find((t) => t.id === l.a);
                let identity = userOrganizationProfiles.find((t) => t.userId === l.o);
                if (!identity && visitorProfiles.length > 0) {
                    identity = visitorProfiles.find((t) => t.userId === l.o);
                }
                let manualRecordCreatorCaptions;
                if (l.m) {
                    manualRecordCreatorCaptions = (await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId, userId: l.m })).caption;
                }
                let credentialDatas = [];
                if (l.cx) {
                    for (const credential of l.cx) {
                        let credentialInfoFromDb = credentialInfo.find((c) => c.id == credential.i);
                        credentialDatas.push({
                            data: credential.d,
                            type: credential.t,
                            expiresOn: credentialInfoFromDb ? credentialInfoFromDb.expiresOn : null,
                        });
                    }
                }
                else {
                    let logCredentials = [];
                    if (lodash_1.default.isArray(l.ci)) {
                        logCredentials = credentialInfo.filter((c) => l.ci.some((r) => r == c.id));
                    }
                    else {
                        let credentialData = credentialInfo.find((c) => c.id === l.ci);
                        if (credentialData)
                            logCredentials.push(credentialData);
                    }
                    for (const credential of logCredentials) {
                        if (credential.type == dal_constants_1.DalConstants.CredentialType.VehiclePlate) {
                            credentialDatas.push({
                                type: credential.type,
                                data: credential.data,
                                expiresOn: credential.expiresOn,
                            });
                        }
                        else if (credential.type == dal_constants_1.DalConstants.CredentialType.MiFare || credential.type == dal_constants_1.DalConstants.CredentialType.ProximityCard) {
                            credentialDatas.push({
                                type: credential.type,
                                data: parseInt(credential.data, 16).toString(),
                                expiresOn: credential.expiresOn,
                            });
                        }
                        else {
                            credentialDatas.push({
                                type: credential.type,
                                expiresOn: credential.expiresOn,
                            });
                        }
                    }
                }
                let reader = null;
                if (l.re !== null && accessControlPoint) {
                    let wr = accessControlPoint.wiegandReaders.find((wr) => wr.deviceId === l.cid && wr.readerWiegandInputNumber === l.re);
                    if (wr) {
                        reader = {
                            id: wr.id,
                            name: wr.name,
                            location: wr.location,
                        };
                    }
                }
                return {
                    id: l.id,
                    utc: new Date(l.u),
                    insertionDate: new Date(l.iUtc),
                    accessRightId: l.i,
                    credentialType: null,
                    result: l.r,
                    direction: l.d,
                    isVisitor: l.ul === predefined_roles_1.PredefinedRoles.OrganizationVisitor.id || l.ul === predefined_roles_1.PredefinedRoles.OrganizationUnitVisitor.id || l.v,
                    isManuallyLogged: !(lodash_1.default.isUndefined(l.m) || lodash_1.default.isNull(l.m)),
                    accessControlPoint: accessControlPoint
                        ? {
                            id: l.a,
                            name: accessControlPoint.name,
                            location: accessControlPoint.location,
                            reader: reader,
                        }
                        : {},
                    identity: identity
                        ? {
                            id: identity.userId,
                            name: identity.name,
                            surname: identity.surname,
                            uniqueId: identity.uniqueId,
                            extensionFields: identity.extensionFields,
                            credentialData: identity.credentialData,
                            organizationUnits: identity.organizationUnits,
                            userGroups: identity.userGroups,
                        }
                        : null,
                    credentials: credentialDatas,
                    isExitButton: l.di !== null,
                    isRemoteAccess: l.ir !== null,
                    logAccessControlPointInfo: l.an || "",
                    logUserInfo: l.on || "",
                    unknownCredentialData: l.c,
                    snapshots: null,
                    counterSensorData: counterSensorLogs
                        ? counterSensorLogs
                            .filter((c) => c.l == l.id)
                            .map((c) => {
                            return {
                                deviceId: c.ci,
                                number: c.n,
                                utc: new Date(c.u),
                                reason: c.r,
                            };
                        })
                        : [],
                    statusSensorData: statusSensorLogs
                        ? statusSensorLogs
                            .filter((c) => c.l == l.id)
                            .map((c) => {
                            return {
                                deviceId: c.ci,
                                number: c.n,
                                utc: new Date(c.u),
                                reason: c.r,
                            };
                        })
                        : [],
                    mobileAuthenticated: l.ma,
                    location: l.lt && l.ln
                        ? {
                            latitude: l.lt,
                            longitude: l.ln,
                            isLocationReliable: l.lr !== null ? l.lr : undefined,
                        }
                        : undefined,
                    manualRecordCreatorCaptions: manualRecordCreatorCaptions,
                };
            }));
            let result = {
                items: [],
                skip: filter.skip || 0,
                take: filter.take || 1000,
                total: accessLogsReportItems.length,
            };
            let userCaptions = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, accessLogsReportItems.filter((r) => r.identity).map((r) => r.identity.id));
            for (const item of accessLogsReportItems) {
                let captions;
                if (item.identity) {
                    captions = userCaptions.find((a) => a.id == item.identity.id);
                }
                if (captions) {
                    result.items.push(lodash_1.default.assign({}, item, { userCaptions: captions.captionLines }));
                }
                else {
                    if (item.logUserInfo) {
                        result.items.push(lodash_1.default.assign({}, item, {
                            userCaptions: [
                                {
                                    text: [item.logUserInfo],
                                    color: [],
                                },
                            ],
                            identity: {
                                name: item.logUserInfo,
                            },
                        }));
                    }
                    else {
                        result.items.push(lodash_1.default.assign({}, item, { userCaptions: [] }));
                    }
                }
            }
            await onData(result);
        });
    }
    async getAccessLogsReport(organizationId, filter, shouldReturnSnapshots, userAccessRights) {
        let logResult = await dal_manager_1.dbManager.accessLog.getAccessLogsReport(organizationId, filter, shouldReturnSnapshots, userAccessRights);
        let logs = logResult.logs;
        let totalCount = logResult.totalCount;
        let accessControlPointIds = [];
        let userIds = [];
        let credentialIds = [];
        let statusSensorIds = [];
        let counterSensorIds = [];
        for (let index = 0; index < logs.length; index++) {
            let row = logs[index];
            accessControlPointIds.push(row.a);
            userIds.push(row.o);
            if (lodash_1.default.isArray(row.ci)) {
                for (const cid of row.ci) {
                    credentialIds.push(cid);
                }
            }
            else {
                credentialIds.push(row.ci);
            }
            if (row.cx && lodash_1.default.isArray(row.cx)) {
                for (const credential of row.cx) {
                    credentialIds.push(credential.i);
                }
            }
            if (row.ss && lodash_1.default.isArray(row.ss)) {
                statusSensorIds.push(...row.ss);
            }
            if (row.cs && lodash_1.default.isArray(row.cs)) {
                counterSensorIds.push(...row.cs);
            }
        }
        accessControlPointIds = lodash_1.default.compact(lodash_1.default.uniq(accessControlPointIds));
        userIds = lodash_1.default.compact(lodash_1.default.uniq(userIds));
        credentialIds = lodash_1.default.compact(lodash_1.default.uniq(credentialIds));
        counterSensorIds = lodash_1.default.compact(lodash_1.default.uniq(counterSensorIds));
        statusSensorIds = lodash_1.default.compact(lodash_1.default.uniq(statusSensorIds));
        let statusSensorLogs = await dal_manager_1.dbManager.accessLog.listStatusSensorLogs(organizationId, statusSensorIds);
        let counterSensorLogs = await dal_manager_1.dbManager.accessLog.listCounterSensorLogs(organizationId, counterSensorIds);
        let accessControlPoints = await dal_manager_1.dbManager.accessAccessControlPoint.listAccessControlPointDetails(organizationId, accessControlPointIds);
        let organizationUnitsColumn = this.dbClient
            .raw(`
            to_json( ARRAY(select json_build_object('id', ou.id, 'name', ou.name) 
            from "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}" as uoou 
            inner join "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" as ou on uoou."organizationUnitId" = ou.id
            where uoou."userOrganizationId" = uo.id and uoou."deletedAt" is null
            order by ou.name))
            `)
            .wrap("(", ') "organizationUnits"');
        let userGroupsColumn = this.dbClient
            .raw(`
                to_json( ARRAY(select json_build_object('id', ug.id, 'name', ug.name) 
                from "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupUserOrganizations}" as uguo
                inner join "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroups}" as ug on uguo."userGroupId" = ug.id
                where uguo."userOrganizationId" = uo.id and uguo."deletedAt" is null
                order by ug.name))
                `)
            .wrap("(", ') "userGroups"');
        let qbResultItems = this.dbClient
            .withSchema(organizationId)
            .from("userOrganizations as uo")
            .innerJoin("users as u", "u.id", "uo.userId")
            .innerJoin("userOrganizationProfiles as uop", "uop.userOrganizationId", "uo.id")
            .whereIn("u.id", userIds)
            .whereNull("uo.deletedAt")
            .whereNull("uop.deletedAt")
            .select("uo.userId", "uop.name", "uop.surname", "uop.uniqueId", "uop.extensionFields", organizationUnitsColumn, userGroupsColumn);
        let userOrganizationProfiles = await qbResultItems;
        let visitorProfiles = await this.dbClient
            .withSchema(organizationId)
            .from(dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations + " as uo")
            .innerJoin(dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorProfiles + " as ovp", "ovp.userOrganizationId", "uo.id")
            .whereIn("uo.userId", userIds)
            .where("uo.organizationId", organizationId)
            .select("uo.userId", "ovp.uniqueId", "ovp.name", "ovp.surname");
        let credentialInfoQb = this.dbClient
            .withSchema(organizationId)
            .from("userOrganizations as uo")
            .innerJoin("userOrganizationCredentials as uc ", "uc.userId", "uo.userId")
            .whereNull("uo.deletedAt")
            .whereNull("uc.deletedAt")
            .whereIn("uc.id", credentialIds);
        if (filter.credentialData) {
            credentialInfoQb = credentialInfoQb.whereRaw("upper(unaccent(uc.data)) = upper(unaccent(?))", filter.credentialData);
        }
        let credentialInfo = await credentialInfoQb.select("uo.userId", "uc.data", "uc.id", "uc.type", "uc.expiresOn");
        let resultItems = await Promise.all(logs.map(async (l) => {
            let accessControlPoint = accessControlPoints.find((t) => t.id === l.a);
            let identity = userOrganizationProfiles.find((t) => t.userId === l.o);
            if (!identity && visitorProfiles.length > 0) {
                identity = visitorProfiles.find((t) => t.userId === l.o);
            }
            let manualRecordCreatorCaptions;
            if (l.m) {
                manualRecordCreatorCaptions = (await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId, userId: l.m })).caption;
            }
            let credentialDatas = [];
            if (l.cx) {
                for (const credential of l.cx) {
                    let credentialInfoFromDb = credentialInfo.find((c) => c.id == credential.i);
                    credentialDatas.push({
                        data: credential.d,
                        type: credential.t,
                        expiresOn: credentialInfoFromDb ? credentialInfoFromDb.expiresOn : null,
                    });
                }
            }
            else {
                let logCredentials = [];
                if (lodash_1.default.isArray(l.ci)) {
                    logCredentials = credentialInfo.filter((c) => l.ci.some((r) => r == c.id));
                }
                else {
                    let credentialData = credentialInfo.find((c) => c.id === l.ci);
                    if (credentialData)
                        logCredentials.push(credentialData);
                }
                for (const credential of logCredentials) {
                    if (credential.type == dal_constants_1.DalConstants.CredentialType.VehiclePlate) {
                        credentialDatas.push({
                            type: credential.type,
                            data: credential.data,
                            expiresOn: credential.expiresOn,
                        });
                    }
                    else if (credential.type == dal_constants_1.DalConstants.CredentialType.MiFare || credential.type == dal_constants_1.DalConstants.CredentialType.ProximityCard) {
                        credentialDatas.push({
                            type: credential.type,
                            data: parseInt(credential.data, 16).toString(),
                            expiresOn: credential.expiresOn,
                        });
                    }
                    else {
                        credentialDatas.push({
                            type: credential.type,
                            expiresOn: credential.expiresOn,
                        });
                    }
                }
            }
            let reader = null;
            if (l.re !== null && accessControlPoint) {
                let wr = accessControlPoint.wiegandReaders.find((wr) => wr.deviceId === l.cid && wr.readerWiegandInputNumber === l.re);
                if (wr) {
                    reader = {
                        id: wr.id,
                        name: wr.name,
                        location: wr.location,
                    };
                }
            }
            return {
                id: l.id,
                utc: l.u,
                insertionDate: l.iUtc,
                accessRightId: l.i,
                credentialType: null,
                result: l.r,
                direction: l.d,
                isVisitor: l.ul === predefined_roles_1.PredefinedRoles.OrganizationVisitor.id || l.ul === predefined_roles_1.PredefinedRoles.OrganizationUnitVisitor.id || l.v,
                isManuallyLogged: !(lodash_1.default.isUndefined(l.m) || lodash_1.default.isNull(l.m)),
                accessControlPoint: accessControlPoint
                    ? {
                        id: l.a,
                        name: accessControlPoint.name,
                        location: accessControlPoint.location,
                        reader: reader,
                    }
                    : {},
                identity: identity
                    ? {
                        id: identity.userId,
                        name: identity.name,
                        surname: identity.surname,
                        uniqueId: identity.uniqueId,
                        extensionFields: identity.extensionFields,
                        credentialData: identity.credentialData,
                        organizationUnits: identity.organizationUnits,
                        userGroups: identity.userGroups,
                    }
                    : null,
                credentials: credentialDatas,
                isExitButton: l.di !== null,
                isRemoteAccess: l.ir === null || l.ir === undefined ? false : l.ir,
                logAccessControlPointInfo: l.an || "",
                logUserInfo: l.on || "",
                unknownCredentialData: l.c,
                snapshots: l.snapshots,
                counterSensorData: counterSensorLogs
                    ? counterSensorLogs
                        .filter((c) => c.l == l.id)
                        .map((c) => {
                        return {
                            deviceId: c.ci,
                            number: c.n,
                            utc: new Date(c.u),
                            reason: c.r,
                        };
                    })
                    : [],
                statusSensorData: statusSensorLogs
                    ? statusSensorLogs
                        .filter((c) => c.l == l.id)
                        .map((c) => {
                        return {
                            deviceId: c.ci,
                            number: c.n,
                            utc: new Date(c.u),
                            reason: c.r,
                        };
                    })
                    : [],
                mobileAuthenticated: l.ma || false,
                location: l.lt && l.ln
                    ? {
                        latitude: l.lt,
                        longitude: l.ln,
                        isLocationReliable: l.lr !== null ? l.lr : undefined,
                    }
                    : undefined,
                manualRecordCreatorCaptions: manualRecordCreatorCaptions,
            };
        }));
        let result = {
            total: totalCount,
            skip: filter.skip,
            take: filter.take,
            items: resultItems,
        };
        return result;
    }
    async getArventoAccessLogsReport(organizationId, filter) {
        let query = `
			SELECT 
				ala.id, 
				u."id" as "userId", 
				ala."actionUtc", 
				ala."type", 
				ala."data", 
				al."log", 
				acp."name" as "acpName",
				ala.type,
				ala."isSynchronized"
			FROM "${organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogsArvento} AS ala
			INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs}" AS al
				ON al.id = ala."accessLogId"
			INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessControlPoints}" AS acp
				ON acp.id = ala."accessControlPointId"
			INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}" AS u
				ON u.id = (al."log"->>'o')::uuid
			WHERE 1 = 1
		`;
        const queryParams = [];
        if (filter.startUtc) {
            queryParams.push(filter.startUtc);
            query += ` AND ala."actionUtc" >= $${queryParams.length}`;
        }
        else {
            queryParams.push(new Date(new Date().getTime() - 24 * 60 * 60 * 1000));
            query += ` AND ala."actionUtc" >= $${queryParams.length}`;
        }
        if (filter.endUtc) {
            queryParams.push(filter.endUtc);
            query += ` AND ala."actionUtc" <= $${queryParams.length}`;
        }
        else {
            queryParams.push(new Date());
            query += ` AND ala."actionUtc" <= $${queryParams.length}`;
        }
        if (filter.accessControlPointIds && filter.accessControlPointIds.length > 0) {
            queryParams.push(filter.accessControlPointIds);
            query += ` AND al.log->>'a' = ANY($${queryParams.length})`;
        }
        if (filter.regionIds?.length) {
            let regionAcpIds = [];
            await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                regionAcpIds.push(...(await dal_manager_1.dbManager.accessAccessControlPoint.listRegionsAccesControlPointIds(organizationId, filter.regionIds, trx)));
            });
            if (regionAcpIds.length > 0) {
                queryParams.push(regionAcpIds);
                query += ` AND al.log->>'a' = ANY($${queryParams.length})`;
            }
        }
        if (filter.userIds?.length === 0) {
            if (!filter.organizationUnitIds && !filter.userGroupIds) {
                filter.userIds = null;
            }
        }
        if (filter.userIds) {
            if (filter.userIds.length > 0) {
                queryParams.push(filter.userIds);
                query += ` AND al.log->>'o' = ANY($${queryParams.length})`;
            }
            else {
                query += ` AND 1 = 0`;
            }
        }
        if (filter.type) {
            queryParams.push(filter.type);
            query += ` AND ala."type" = $${queryParams.length}`;
            if (filter.type === restapi_1.ArventoAccessLogType.ArventoQr) {
                if (filter.licencePlate) {
                    queryParams.push(`%${filter.licencePlate}%`);
                    query += ` AND (ala."data"->>'licencePlate' IS NOT NULL AND ala."data"->>'licencePlate' ILIKE $${queryParams.length})`;
                }
            }
            else if (filter.type === restapi_1.ArventoAccessLogType.ArventoRegion) {
                if (filter.region) {
                    queryParams.push(`%${filter.region}%`);
                    query += ` AND (ala."data"->>'region' IS NOT NULL AND ala."data"->>'region' ILIKE $${queryParams.length})`;
                }
            }
        }
        else {
            if (filter.licencePlate) {
                queryParams.push(`%${filter.licencePlate}%`);
                query += ` AND (ala."data"->>'licencePlate' IS NOT NULL AND ala."data"->>'licencePlate' ILIKE $${queryParams.length})`;
            }
            if (filter.region) {
                queryParams.push(`%${filter.region}%`);
                query += ` AND (ala."data"->>'region' IS NOT NULL AND ala."data"->>'region' ILIKE $${queryParams.length})`;
            }
        }
        if (filter.isSynchronized !== undefined) {
            queryParams.push(filter.isSynchronized);
            query += ` AND ala."isSynchronized" = $${queryParams.length}`;
        }
        const countQuery = `
			SELECT COUNT(*)
			FROM (
				${query}
			) AS count_query
		`;
        const countResult = await this._pgPool.query(countQuery, queryParams);
        const totalCount = parseInt(countResult.rows[0].count, 10);
        if (totalCount === 0) {
            return {
                pagination: { total: 0, skip: filter.pagination?.skip, take: filter.pagination?.take },
                items: [],
            };
        }
        query += ` ORDER BY ala."actionUtc" DESC`;
        if (filter.pagination?.skip) {
            queryParams.push(filter.pagination.skip);
            query += ` OFFSET $${queryParams.length}`;
        }
        if (filter.pagination?.take) {
            queryParams.push(filter.pagination.take);
            query += ` LIMIT $${queryParams.length}`;
        }
        const { rows } = await this._pgPool.query(query, queryParams);
        let resultItems = rows.map((row) => {
            return {
                id: row.id,
                userId: row.userId,
                actionUtc: row.actionUtc,
                accessControlPointName: row.acpName,
                region: row.data?.region,
                licencePlate: row.data?.licencePlate,
                type: row.type,
                isSynchronized: row.isSynchronized,
            };
        });
        let result = {
            pagination: { total: totalCount, skip: filter.pagination?.skip, take: filter.pagination?.take },
            items: resultItems,
        };
        return result;
    }
    async upsertAccessRuleSet(params) {
        const id = params.ruleset.id || uuid_1.default.v4();
        const now = new Date();
        if (params.ruleset.id) {
            await params.trx.query(`UPDATE "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessRuleSets}"
				SET name = $1,
					"regionId" = $2,
					"isActive" = $3,
					"startUtc" = $4,
					"endUtc" = $5,
					"updatedAt" = $6,
					note = $7,
					"uncoveredAccessStatus" = $8
				WHERE id = $9`, [
                params.ruleset.name,
                params.ruleset.regionId,
                params.ruleset.isActive,
                params.ruleset.startUtc,
                params.ruleset.endUtc,
                now,
                params.ruleset.note,
                params.ruleset.uncoveredAccessStatus,
                params.ruleset.id,
            ]);
        }
        else {
            await params.trx.query(`INSERT INTO "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessRuleSets}"
				(id, "organizationId", name, "regionId", "isActive", "startUtc", "endUtc", "createdAt", "updatedAt", note, "uncoveredAccessStatus")
				VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11)`, [
                id,
                params.organizationId,
                params.ruleset.name,
                params.ruleset.regionId,
                params.ruleset.isActive,
                params.ruleset.startUtc,
                params.ruleset.endUtc,
                now,
                now,
                params.ruleset.note,
                params.ruleset.uncoveredAccessStatus,
            ]);
        }
        params.ruleset.id = id;
        await dal_manager_1.dbManager.accessLog.addUserActionHistoryItem({
            oId: params.organizationId,
            o: params.requesterUserId,
            u: new Date(),
            c: dal_constants_1.DalConstants.UserActionCategory.Main,
            tg: ["Access Rule"],
            t: dal_constants_1.DalConstants.UserMainActionType.UpsertAccessRuleSet,
            d: params.ruleset,
        });
        return id;
    }
    async upsertAccessRule(params) {
        const id = params.rule.id || uuid_1.default.v4();
        const isAddNewRuleRequest = !params.rule.id;
        const existingRules = await this.listAccessRules({
            organizationId: params.organizationId,
            accessRuleSetId: params.rule.accessRuleSetId,
            trx: params.trx,
        });
        if (isAddNewRuleRequest) {
            switch (params.rule.type) {
                case dal_constants_1.DalConstants.AccessRuleType.CountBased:
                    if (existingRules.items.find((er) => er.type === dal_constants_1.DalConstants.AccessRuleType.CountBased)) {
                        throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.ACCESS_RULE.MAX_ACCESS_COUNT_RULE");
                    }
                    break;
                case dal_constants_1.DalConstants.AccessRuleType.CapacityBased:
                    if (existingRules.items.find((er) => er.type === dal_constants_1.DalConstants.AccessRuleType.CapacityBased)) {
                        throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.ACCESS_RULE.MAX_ACCESS_CAPACITY_RULE");
                    }
                    break;
                default:
                    break;
            }
        }
        const ruleParameters = params.rule.parameters;
        const accessRuleSet = await this.getAccessRuleSet({
            organizationId: params.organizationId,
            accessRuleSetId: params.rule.accessRuleSetId,
            trx: params.trx,
        });
        if (params.rule.type === dal_constants_1.DalConstants.AccessRuleType.CapacityBased) {
            for (const param of ruleParameters) {
                if (!accessRuleSet.userGroups.map((ug) => ug.id).includes(param.groupId)) {
                    throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.ACCESS_RULE.USER_GROUP_NOT_MAPPED_WITH_RULE_SET");
                }
            }
            const totalCapacity = ruleParameters.length ? ruleParameters.map((item) => item.capacity).reduce((total, capacity) => total + capacity) : 0;
            if (accessRuleSet.region.capacity && totalCapacity > accessRuleSet.region.capacity) {
                throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.ACCESS_RULE.MAX_CAPACITY_EXCEEDED");
            }
        }
        const now = new Date();
        if (!isAddNewRuleRequest) {
            await params.trx.query(`UPDATE "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessRules}"
				SET "acceptStatus" = $1,
					type = $2,
					direction = $3,
					parameters = $4,
					"updatedAt" = $5
				WHERE id = $6`, [params.rule.acceptStatus, params.rule.type, params.rule.direction, JSON.stringify(params.rule.parameters), now, params.rule.id]);
            const targetRule = existingRules.items.find((f) => f.id === params.rule.id);
            if (params.rule.type === dal_constants_1.DalConstants.AccessRuleType.TimeCountBased &&
                (params.rule.parameters.startTime !== targetRule.parameters.startTime ||
                    params.rule.parameters.endTime !== targetRule.parameters.endTime)) {
                await params.trx.query(`UPDATE "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessRuleHistory}"
					SET "count" = 0, "lastUpdateAccessControlPointId" = NULL
					WHERE "accessRuleId" = $1 
						AND now() at time zone 'utc' <= to_timestamp("options"->>'endDateTime', 'YYYY-MM-DD"T"HH24:MI:SS"Z"') 
						AND now() at time zone 'utc' >= to_timestamp("options"->>'startDateTime', 'YYYY-MM-DD"T"HH24:MI:SS"Z"')`, [params.rule.id]);
            }
        }
        else {
            await params.trx.query(`INSERT INTO "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessRules}"
			(id, "accessRuleSetId", "acceptStatus", type, direction, parameters, "createdAt", "updatedAt")
			VALUES ($1, $2, $3, $4, $5, $6, $7, $8)`, [id, params.rule.accessRuleSetId, params.rule.acceptStatus, params.rule.type, params.rule.direction, JSON.stringify(params.rule.parameters), now, now]);
        }
        if (params.rule.type === dal_constants_1.DalConstants.AccessRuleType.CapacityBased) {
            for (const countBasedRuleDetails of ruleParameters) {
                const groupCapacityUsage = (await dal_manager_1.dbManager.accessRegion.getRegionStateReportNew(params.organizationId, {
                    pagination: {
                        take: 100000,
                        skip: 0,
                    },
                    regionId: accessRuleSet.region.id,
                    status: app_enums_1.enums.IdentityStatusType.Enabled,
                    userGroupIds: [countBasedRuleDetails.groupId],
                    userIds: [],
                    applyOrganizationUnitFilterHierarchically: false,
                    organizationUnitIds: [],
                    state: [app_enums_1.enums.AntiPassbackState.In],
                }, params.requesterUserId)).pagination.total;
                await params.trx.query(`
					UPDATE "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupAccessRuleSets}"
					SET "currentCapacityUsage" = $1, "lastActionDateISO" = now(), "lastUpdateAccessControlPointId" = NULL
					WHERE "accessRuleSetId" = $2
						AND "userGroupId" = $3
				`, [groupCapacityUsage, accessRuleSet.id, countBasedRuleDetails.groupId]);
            }
        }
        params.rule.id = id;
        await dal_manager_1.dbManager.accessLog.addUserActionHistoryItem({
            oId: params.organizationId,
            o: params.requesterUserId,
            u: new Date(),
            c: dal_constants_1.DalConstants.UserActionCategory.Main,
            tg: ["Access Rule"],
            t: dal_constants_1.DalConstants.UserMainActionType.UpsertAccessRule,
            d: params.rule,
        });
        return id;
    }
    async removeAccessRule(params) {
        const rule = await this.getAccessRule({ organizationId: params.organizationId, requesterUserId: params.requesterUserId, accessRuleId: params.ruleId, trx: params.trx });
        if (!rule) {
            (0, dal_access_error_1.throwDbAccessNotFoundErrorTr)("ERRORS.ACCESS_RULE.ACCESS_RULE_NOT_FOUND");
        }
        if (rule.type === dal_constants_1.DalConstants.AccessRuleType.CapacityBased) {
            await params.trx.query(`
				UPDATE "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupAccessRuleSets}"
				SET "currentCapacityUsage" = NULL ,"lastUpdateAccessControlPointId" = NULL ,"lastActionDateISO" = NULL
				WHERE "accessRuleSetId" = $1
			`, [rule.accessRuleSetId]);
        }
        await params.trx.query(`
			UPDATE "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessRules}"
			SET "deletedAt" = $1
			WHERE id = $2 `, [new Date(), params.ruleId]);
        await dal_manager_1.dbManager.accessLog.addUserActionHistoryItem({
            oId: params.organizationId,
            o: params.requesterUserId,
            u: new Date(),
            c: dal_constants_1.DalConstants.UserActionCategory.Main,
            tg: ["Access Rule"],
            t: dal_constants_1.DalConstants.UserMainActionType.RemoveAccessRule,
            d: {
                id: params.ruleId,
            },
        });
        return true;
    }
    async removeAccessRuleSet(params) {
        const ruleSet = await this.getAccessRuleSet({ organizationId: params.organizationId, accessRuleSetId: params.rulesetId, trx: params.trx });
        if (!ruleSet) {
            (0, dal_access_error_1.throwDbAccessNotFoundErrorTr)("ERRORS.ACCESS_RULE.RULE_SET_NOT_FOUND");
        }
        if (ruleSet.userGroups?.length) {
            await this.assignUserGroupsToRuleSet({
                organizationId: params.organizationId,
                accessRuleSetId: ruleSet.id,
                requesterUserId: params.requesterUserId,
                trx: params.trx,
                userGroupIds: [],
            });
        }
        const rules = (await this.listAccessRules({ organizationId: params.organizationId, accessRuleSetId: params.rulesetId, trx: params.trx })).items;
        if (rules?.length) {
            for (const r of rules) {
                await this.removeAccessRule({
                    organizationId: params.organizationId,
                    requesterUserId: params.requesterUserId,
                    ruleId: r.id,
                    trx: params.trx,
                });
            }
        }
        await params.trx.query(`UPDATE "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessRuleSets}"
		SET "deletedAt" = $1
		WHERE id = $2`, [new Date(), params.rulesetId]);
        await dal_manager_1.dbManager.accessLog.addUserActionHistoryItem({
            oId: params.organizationId,
            o: params.requesterUserId,
            u: new Date(),
            c: dal_constants_1.DalConstants.UserActionCategory.Main,
            tg: ["Access Rule"],
            t: dal_constants_1.DalConstants.UserMainActionType.RemoveAccessRuleSet,
            d: {
                id: params.rulesetId,
            },
        });
        return true;
    }
    async listAccessRuleSets(params) {
        let qx = 1;
        const qb = [];
        const qw = [];
        const qFrom = `FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessRuleSets}" ars
					   LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessRules}" ar
						   	ON ars.id = ar."accessRuleSetId" AND ar."deletedAt" IS NULL
					   LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.regions}" r
						   	ON r.id = ars."regionId"
					   LEFT JOIN "${params.organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.region_administrators} ra
						 	ON r.id = ra."regionId" 
						`;
        qw.push(`ars."deletedAt" IS NULL`);
        if (params.requesterUserId) {
            qw.push(`(ra."userId" IS NULL OR ra."userId" = $${qx++} AND "read" IS TRUE)`);
            qb.push(params.requesterUserId);
        }
        if (params.request.filter) {
            qw.push(`ars."name" ilike  $${qx++}`);
            qb.push(`%${params.request.filter}%`);
        }
        let qws = `
				WHERE ` + qw.join(" AND ");
        const totalCount = await params.trx.query(`SELECT COUNT(*)::SMALLINT as c ${qFrom} ${qws}`, qb);
        const result = {
            pagination: {
                skip: params.request.skip ?? 0,
                take: params.request.take ?? 100,
                total: totalCount.rowCount > 0 ? totalCount.rows[0].c : 0,
            },
            items: [],
        };
        if (result.pagination.total > 0) {
            const q = ` SELECT ars.id, 
			ars.name, 
			ars."isActive",
			ars."startUtc",
			ars."endUtc",
			ars.note,
			ars."uncoveredAccessStatus",
			jsonb_build_object('id', r.id, 'name', r.name, 'capacity', r.capacity) as region,
			(
				SELECT coalesce(jsonb_agg(jsonb_build_object('id', ug.id, 'name', ug.name)), '[]'::jsonb) 
				FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupAccessRuleSets}" ugars
				INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroups}" ug
						ON ug.id = ugars."userGroupId"
				WHERE ugars."accessRuleSetId" = ars.id
			) as "userGroups", 
			COUNT(DISTINCT ar.id)::integer as "ruleCount" 
			${qFrom}
			${qws}
			GROUP BY ars.id, ars.name, ars."uncoveredAccessStatus", r.name, r.id
			ORDER BY ars."name" DESC
			OFFSET $${qx++}
			LIMIT $${qx++}
			`;
            qb.push(result.pagination.skip, result.pagination.take);
            const { rows } = await params.trx.query(q, qb);
            result.items = rows;
        }
        return result;
    }
    async listAccessRules(params) {
        let qx = 1;
        const qb = [];
        const qFrom = `
		FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessRules}"
		WHERE "accessRuleSetId" = $${qx++}
		AND "deletedAt" IS NULL`;
        qb.push(params.accessRuleSetId);
        const totalCount = await params.trx.query(`SELECT COUNT(*)::INTEGER AS c` + qFrom, qb);
        const result = {
            pagination: {
                skip: params.pagination?.skip ?? 0,
                take: params.pagination?.take ?? 100,
                total: totalCount.rows[0].c,
            },
            items: [],
        };
        if (result.pagination.total > 0) {
            const q = `SELECT id, "acceptStatus", "accessRuleSetId", direction, parameters, type 
					  ${qFrom}
					  ORDER BY "createdAt" DESC
					  LIMIT $${qx++}
					  OFFSET $${qx++}
					  `;
            qb.push(result.pagination.take, result.pagination.skip);
            const { rows } = await params.trx.query(q, qb);
            result.items = rows;
        }
        return result;
    }
    async assignUserGroupsToRuleSet(params) {
        await params.trx.query(`
			DELETE FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupAccessRuleSets}"
			WHERE "accessRuleSetId" = $1 AND
			NOT "userGroupId" = ANY ($2::UUID[]) 
		`, [params.accessRuleSetId, params.userGroupIds]);
        const capacityBasedRuleDetails = (await params.trx.query(`
			SELECT id, parameters
			FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessRules}"
			WHERE "accessRuleSetId" = $1 AND
				"type" = $2 AND
				"deletedAt" IS NULL
		`, [params.accessRuleSetId, dal_constants_1.DalConstants.AccessRuleType.CapacityBased])).rows[0];
        if (capacityBasedRuleDetails && capacityBasedRuleDetails.parameters?.length) {
            const updatedCapacityBasedRuleDetails = capacityBasedRuleDetails.parameters.filter((p) => params.userGroupIds.includes(p.groupId));
            await params.trx.query(`
				UPDATE "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessRules}" 
				SET parameters = $2
				WHERE id = $1
			`, [capacityBasedRuleDetails.id, JSON.stringify(updatedCapacityBasedRuleDetails)]);
        }
        await params.trx.query(`
			UPDATE "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessRules}" ar
			SET parameters = ar2.parameters
			FROM (
				SELECT id, array_to_json (array_agg(elem)) AS parameters
				FROM  "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessRules}" arin, json_array_elements (arin.parameters) elem
				WHERE elem ->>'groupId' = ANY($3::TEXT[]) AND
					arin.type = $2 AND
					arin."accessRuleSetId" = $1 AND
					arin."deletedAt" IS NULL
  				GROUP  BY 1
			) ar2
			WHERE ar.id = ar2.id AND
				ar."accessRuleSetId" = $1 AND
				ar."type" = $2 AND 
				ar."deletedAt" IS NULL
		`, [params.accessRuleSetId, app_enums_1.enums.AccessRuleType.CapacityBased, params.userGroupIds]);
        await params.trx.query(`INSERT INTO "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupAccessRuleSets}"
			(id, "userGroupId", "accessRuleSetId")
				SELECT uuid_generate_v4(), id, $1
				FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroups}"
				WHERE id = ANY($2::uuid[])
			ON CONFLICT ON CONSTRAINT "userGroupAccessRuleSets_userGroupId_accessRuleSetId_key"
				DO NOTHING;
		`, [params.accessRuleSetId, params.userGroupIds]);
        await dal_manager_1.dbManager.accessLog.addUserActionHistoryItem({
            oId: params.organizationId,
            o: params.requesterUserId,
            u: new Date(),
            c: dal_constants_1.DalConstants.UserActionCategory.Main,
            tg: ["Access Rule"],
            t: dal_constants_1.DalConstants.UserMainActionType.AssignUserGroupToAccessRule,
            d: {
                id: params.accessRuleSetId,
                userGroupIds: params.userGroupIds,
            },
        });
    }
    async getAccessRule(params) {
        const { rows, rowCount } = await params.trx.query(`
			SELECT id, "acceptStatus", "accessRuleSetId", direction, parameters, type 
			FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessRules}"
			WHERE id = $1
			AND "deletedAt" IS NULL
		`, [params.accessRuleId]);
        if (rowCount === 0) {
            (0, dal_access_error_1.throwDbAccessNotFoundErrorTr)("ERRORS.ACCESS_RULE.RULE_NOT_FOUND");
        }
        return rows[0];
    }
    async getAccessRuleSet(params) {
        const { rows: accessRuleSetRows } = await params.trx.query(`	SELECT ars.id, 
					   ars.name, 
					   ars."isActive",
					   ars."startUtc",
					   ars."endUtc",
					   ars.note,
					   json_build_object('id', r.id, 'name', r.name, 'capacity', r.capacity) as region,
					   (
							SELECT coalesce(jsonb_agg(jsonb_build_object('id', ug.id, 'name', ug.name)), '[]'::jsonb) 
							FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupAccessRuleSets}" ugars
							INNER JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroups}" ug
									ON ug.id = ugars."userGroupId"
							WHERE ugars."accessRuleSetId" = ars.id
						) as "userGroups", 
					   COUNT(DISTINCT ar.id)::integer as "ruleCount" 
				FROM "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessRuleSets}" ars
				LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.accessRules}" ar
					ON ars.id = ar."accessRuleSetId" AND ar."deletedAt" IS NULL
				LEFT JOIN "${params.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.regions}" r
					ON r.id = ars."regionId"
				WHERE ars.id = $1
				AND ars."deletedAt" IS NULL
		    	GROUP BY ars.id, ars.name, ars."isActive", r.id, r.name`, [params.accessRuleSetId]);
        if (accessRuleSetRows.length === 0) {
            (0, dal_access_error_1.throwDbAccessNotFoundErrorTr)("ERRORS.ACCESS_RULE.RULE_SET_NOT_FOUND");
        }
        return accessRuleSetRows[0];
    }
    async canUserSeeUnkownUserAccessLogs(organizationId, requesterUserId, managingOrganizationUnitIds) {
        return await dal_manager_1.dbManager.pgTransactionMainDb(async (trx) => {
            let params = [];
            let query = `
                SELECT r.permissions FROM "${organizationId}"."roles" AS r
                INNER JOIN "${organizationId}"."userOrganizationOrganizationUnits" AS uoou ON r."id" = uoou."roleId"
                INNER JOIN "${organizationId}"."userOrganizations" AS uo ON uoou."userOrganizationId" = uo."id"
                WHERE uo."userId" = $1
                    AND uo."deletedAt" IS NULL
                    AND uoou."deletedAt" IS NULL 
                    AND r."deletedAt" IS NULL
            `;
            params.push(requesterUserId);
            if (managingOrganizationUnitIds && managingOrganizationUnitIds.length > 0) {
                query += `
                AND uoou."organizationUnitId" = ANY ($2::UUID[]);
                `;
                params.push(managingOrganizationUnitIds);
            }
            const { rows, rowCount } = await trx.query(query, params);
            if (rowCount < 1)
                return false;
            else
                for (const r of rows) {
                    if (r.permissions.includes(predefined_permissions_1.Permissions.accessLog.getUnknownUserAccess()))
                        return true;
                }
            return false;
        });
    }
}
exports.PSQLDalAccessAccessLog = PSQLDalAccessAccessLog;
