"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getPartitionBoundryExpression = exports.doesPartitionExists = exports.findTablesWithPartitions = exports.findForeignKeysReferencingGivenTable = void 0;
const findForeignKeysReferencingGivenTable = async (schema, tablename, trx) => {
    const foreignKeyReferenceDbResult = await trx.query(`
		SELECT 
			c.confrelid::regclass::text AS referenced_table,
			la.attrelid::regclass::text AS referencing_table, 
			c.conname::text AS reference_name,
		  	pg_get_constraintdef(c.oid) reference_definition
		FROM pg_constraint AS c
		JOIN pg_index AS i
			ON i.indexrelid = c.conindid
		JOIN pg_attribute AS la
			ON la.attrelid = c.conrelid
				AND la.attnum = any(c.conkey)
		JOIN pg_attribute AS ra
			ON ra.attrelid = c.confrelid
				AND ra.attnum = any (c.confkey)
				AND array_position (c.conkey, la.attnum) = array_position(c.confkey, ra.attnum)
		WHERE 
			(
				c.confrelid::regclass::text = quote_ident($1) || '.' || quote_ident($2)
				OR c.confrelid::regclass::text = quote_ident($2) -- This or clause is needed for public tables, since public schema name is truncated
			) 
			AND coninhcount = 0 
			AND c.contype = 'f'
		GROUP BY c.confrelid, la.attrelid, c.conname, c.oid

	`, [schema, tablename]);
    return foreignKeyReferenceDbResult.rows;
};
exports.findForeignKeysReferencingGivenTable = findForeignKeysReferencingGivenTable;
const findTablesWithPartitions = async (trx, organizationId) => {
    const queryParams = [];
    let partitionedTableFinderQ = `SELECT 
            REPLACE (par.relnamespace::regnamespace::text , '"', '') AS schema, 
            par.relname AS table_name, 
            col.column_name AS partition_key, 
            col.udt_name AS partition_key_type
        FROM   
            (SELECT 
                partrelid,
                partnatts,
                CASE partstrat 
                    WHEN 'l' THEN 'list' 
                    WHEN 'r' THEN 'range' END AS partition_strategy,
                UNNEST (partattrs) column_index
            FROM
                pg_partitioned_table) pt 
        JOIN pg_class par 
        	ON par.oid = pt.partrelid
        JOIN information_schema.columns col
        	ON quote_ident (col.table_schema::text) = par.relnamespace::regnamespace::text
            	AND col.table_name = par.relname
            	AND ordinal_position = pt.column_index
            	AND col.table_schema != 'organization' -- IGNORE TEMPLATE SCHEMA
`;
    if (organizationId) {
        partitionedTableFinderQ += `
		WHERE col.table_schema = $1
		`;
        queryParams.push(organizationId);
    }
    partitionedTableFinderQ += `
		ORDER BY col.table_schema::text, col.table_name
	`;
    const partitionedTablesWithPartitionKeysDbResult = await trx.query(partitionedTableFinderQ, queryParams);
    return partitionedTablesWithPartitionKeysDbResult.rows;
};
exports.findTablesWithPartitions = findTablesWithPartitions;
const doesPartitionExists = async (schemaName, partitionName, trx) => {
    const partitionCheckDbResult = await trx.query(`
		SELECT EXISTS (
			SELECT FROM pg_tables
			WHERE schemaname = $1
			AND tablename  = $2
		);
	`, [schemaName, partitionName]);
    if (partitionCheckDbResult.rows[0].exists) {
        return true;
    }
    else {
        return false;
    }
};
exports.doesPartitionExists = doesPartitionExists;
const getPartitionBoundryExpression = async (schemaName, partitionName, trx) => {
    const partitionBoundaryDbResult = await trx.query(`
		SELECT relname AS "partitionName", 
			pg_get_expr (relpartbound, oid, true) AS "partitionBoundryExpression"
		FROM pg_class
		WHERE relname = $1
			AND relnamespace::regnamespace::text = quote_ident($2)
		`, [schemaName, partitionName]);
    if (partitionBoundaryDbResult.rowCount) {
        return partitionBoundaryDbResult.rows[0];
    }
    else {
        return null;
    }
};
exports.getPartitionBoundryExpression = getPartitionBoundryExpression;
