"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.partitionRecovery = exports.createForwardPartitions = exports.createBackwardPartitions = void 0;
const pgmaintenance_db_utils_1 = require("./pgmaintenance.db.utils");
const luxon_1 = require("luxon");
const app_logs_1 = require("../../../app.logs");
const cli_queries_1 = require("../../../dal/access/psql/cli-queries");
const createBackwardPartitions = async (pool, months, orgId) => {
    try {
        const backwardPartitionsTemplateVariables = months.map((month) => {
            const startOfPartition = luxon_1.DateTime.fromFormat(month, "yyyyMM").startOf("month");
            const endOfPartition = luxon_1.DateTime.fromFormat(month, "yyyyMM").plus({ months: 1 });
            return {
                startOfPartition,
                endOfPartition,
                partitionPostfix: `_p${month}`,
                nextPartitionPostfix: `_p${luxon_1.DateTime.fromFormat(month, "yyyyMM").plus({ months: 1 }).toFormat("yyyyMM")}`,
                previousPartitionPostfix: `_p${luxon_1.DateTime.fromFormat(month, "yyyyMM").minus({ months: 1 }).toFormat("yyyyMM")}`,
            };
        });
        const tablesWithPartitions = await (0, cli_queries_1.systemTransaction)(pool, async (trx) => {
            return (0, pgmaintenance_db_utils_1.findTablesWithPartitions)(trx, orgId);
        });
        let hasEntriesInDefault = false;
        await (0, cli_queries_1.systemTransaction)(pool, async (trx) => {
            for (const bptv of backwardPartitionsTemplateVariables) {
                for (const twp of tablesWithPartitions) {
                    const defaultPartitionPostfix = "_default";
                    const defaultPartitionName = twp.table_name + defaultPartitionPostfix;
                    const defaultTableEntryCountForPartition = (await trx.query(`SELECT COUNT(*)::INTEGER AS c FROM "${twp.schema}"."${defaultPartitionName}"
							 WHERE "${twp.partition_key}" >= $1 AND "${twp.partition_key}" < $2`, [bptv.startOfPartition, bptv.endOfPartition])).rows[0].c;
                    if (defaultTableEntryCountForPartition > 0) {
                        app_logs_1.logger.info(`[partition-manager.ts] Table "${twp.table_name}" has ${defaultTableEntryCountForPartition} entries in the default partition for month ${bptv.partitionPostfix.substring(2, 6)}-${bptv.partitionPostfix.substring(6)}.`);
                        hasEntriesInDefault = true;
                    }
                }
            }
        });
        if (hasEntriesInDefault) {
            app_logs_1.logger.info(`[partition-manager.ts] Aborting partition creation for all months as entries exist in default partitions.`);
            return false;
        }
        for (const bptv of backwardPartitionsTemplateVariables) {
            for (const twp of tablesWithPartitions) {
                const partitionName = twp.table_name + bptv.partitionPostfix;
                try {
                    await (0, cli_queries_1.systemTransaction)(pool, async (trx) => {
                        const partitionCheckResult = await (0, pgmaintenance_db_utils_1.doesPartitionExists)(twp.schema, partitionName, trx);
                        if (!partitionCheckResult) {
                            let partitionStart = null;
                            let partitionEnd = null;
                            switch (twp.partition_key_type) {
                                case "timestamp":
                                case "timestamptz":
                                    partitionStart = bptv.startOfPartition.toFormat("yyyy-MM-dd HH:mm:ssZZ");
                                    partitionEnd = bptv.endOfPartition.toFormat("yyyy-MM-dd HH:mm:ssZZ");
                                    break;
                                case "date":
                                    partitionStart = bptv.startOfPartition.toFormat("yyyy-MM-dd");
                                    partitionEnd = bptv.endOfPartition.toFormat("yyyy-MM-dd");
                                    break;
                            }
                            const previousPartitionCheckDbResult = await (0, pgmaintenance_db_utils_1.getPartitionBoundryExpression)(twp.schema, twp.table_name + bptv.previousPartitionPostfix, trx);
                            if (previousPartitionCheckDbResult) {
                                partitionStart = previousPartitionCheckDbResult.partitionBoundryExpression.split("'")[3];
                            }
                            const nextPartitionCheckDbResult = await (0, pgmaintenance_db_utils_1.getPartitionBoundryExpression)(twp.schema, twp.table_name + bptv.nextPartitionPostfix, trx);
                            if (nextPartitionCheckDbResult) {
                                partitionEnd = nextPartitionCheckDbResult.partitionBoundryExpression.split("'")[1];
                            }
                            await trx.query(`
								CREATE TABLE IF NOT EXISTS "${twp.schema}"."${partitionName}"
								PARTITION OF "${twp.schema}"."${twp.table_name}"
								FOR VALUES FROM('${partitionStart}') TO ('${partitionEnd}');
							`);
                            app_logs_1.logger.info(`[partition-manager.ts] Partition ${partitionName} created successfully for schema ${twp.schema}!`);
                        }
                    });
                }
                catch (error) {
                    app_logs_1.logger.error(`[partition-manager.ts] Error while creating partition for backward months, Parameters are below:`);
                    app_logs_1.logger.error(`[partition-manager.ts] Organization Id: ${orgId}`);
                    app_logs_1.logger.error(`[partition-manager.ts] Table Info: ${JSON.stringify(twp, null, 4)}`);
                    app_logs_1.logger.error(`[partition-manager.ts] Partition Info: ${JSON.stringify(bptv, null, 4)}`);
                    app_logs_1.logger.error(error);
                }
            }
        }
        app_logs_1.logger.info(`[partition-manager.ts] Backward partition creation process completed successfully`);
        return true;
    }
    catch (error) {
        app_logs_1.logger.info(`[partition-manager.ts] Backward partition error: `, error);
        return false;
    }
};
exports.createBackwardPartitions = createBackwardPartitions;
const createForwardPartitions = async (pool, forwardPartitionCount) => {
    const forwardPartitionsTemplateVariables = [];
    const defaultPartitionPostfix = "_default";
    for (let i = 0; i <= forwardPartitionCount; i++) {
        forwardPartitionsTemplateVariables.push({
            startOfPartition: luxon_1.DateTime.now().plus({ months: i }).startOf("month"),
            endOfPartition: luxon_1.DateTime.now()
                .plus({ months: i + 1 })
                .startOf("month"),
            partitionPostfix: "_p" + luxon_1.DateTime.now().plus({ months: i }).toFormat("yyyyMM"),
            nextPartitionPostfix: "_p" +
                luxon_1.DateTime.now()
                    .plus({ months: i + 1 })
                    .toFormat("yyyyMM"),
            previousPartitionPostfix: "_p" +
                luxon_1.DateTime.now()
                    .plus({ months: i - 1 })
                    .toFormat("yyyyMM"),
        });
    }
    const tablesWithPartitions = await (0, cli_queries_1.systemTransaction)(pool, async (trx) => {
        return (0, pgmaintenance_db_utils_1.findTablesWithPartitions)(trx);
    });
    for (const twp of tablesWithPartitions) {
        for (const fptv of forwardPartitionsTemplateVariables) {
            const partitionName = twp.table_name + fptv.partitionPostfix;
            try {
                await (0, cli_queries_1.systemTransaction)(pool, async (trx) => {
                    const partitionCheckResult = await (0, pgmaintenance_db_utils_1.doesPartitionExists)(twp.schema, partitionName, trx);
                    if (!partitionCheckResult) {
                        let partitionStart = null;
                        let partitionEnd = null;
                        switch (twp.partition_key_type) {
                            case "timestamp":
                            case "timestamptz":
                                partitionStart = fptv.startOfPartition.toFormat("yyyy-MM-dd HH:mm:ssZZ");
                                partitionEnd = fptv.endOfPartition.toFormat("yyyy-MM-dd HH:mm:ssZZ");
                                break;
                            case "date":
                                partitionStart = fptv.startOfPartition.toFormat("yyyy-MM-dd");
                                partitionEnd = fptv.endOfPartition.toFormat("yyyy-MM-dd");
                                break;
                        }
                        const previousPartitionCheckDbResult = await (0, pgmaintenance_db_utils_1.getPartitionBoundryExpression)(twp.schema, twp.table_name + fptv.previousPartitionPostfix, trx);
                        if (previousPartitionCheckDbResult) {
                            partitionStart = previousPartitionCheckDbResult.partitionBoundryExpression.split("'")[3];
                        }
                        const nextPartitionCheckDbResult = await (0, pgmaintenance_db_utils_1.getPartitionBoundryExpression)(twp.schema, twp.table_name + fptv.nextPartitionPostfix, trx);
                        if (nextPartitionCheckDbResult) {
                            partitionEnd = nextPartitionCheckDbResult.partitionBoundryExpression.split("'")[1];
                        }
                        const defaulPartitionName = twp.table_name + defaultPartitionPostfix;
                        const defaultTableEntryCountForPartition = (await trx.query(`
							SELECT COUNT(*)::INTEGER AS c
							FROM "${twp.schema}"."${defaulPartitionName}"
							WHERE "${twp.partition_key}" > $1
								AND "${twp.partition_key}" < $2
						`, [partitionStart, partitionEnd])).rows[0].c;
                        if (!defaultTableEntryCountForPartition) {
                            await trx.query(`
								CREATE TABLE IF NOT EXISTS "${twp.schema}"."${partitionName}"
								PARTITION OF "${twp.schema}"."${twp.table_name}"
								FOR VALUES FROM('${partitionStart}') TO ('${partitionEnd}');
							`);
                            app_logs_1.logger.info(`[partition-manager.ts] Partition ${partitionName} created successfully for schema ${twp.schema}!`);
                        }
                        else {
                            app_logs_1.logger.info(`[partition-manager.ts] Partition ${partitionName} could not be created for schema ${twp.schema}, 
because there are ${defaultTableEntryCountForPartition} entries in default partition between ${partitionStart} and ${partitionEnd}!`);
                        }
                    }
                });
            }
            catch (error) {
                app_logs_1.logger.error(`[partition-manager.ts] Error while creating partition for forward months, Parameters are below:`);
                app_logs_1.logger.error(`[partition-manager.ts] Table Info: ${JSON.stringify(twp, null, 4)}`);
                app_logs_1.logger.error(`[partition-manager.ts] Partition Info: ${JSON.stringify(fptv, null, 4)}`);
                app_logs_1.logger.error(error);
            }
        }
    }
    app_logs_1.logger.info(`[partition-manager.ts] Forward partition creation process completed successfully`);
    return;
};
exports.createForwardPartitions = createForwardPartitions;
const partitionRecovery = async (trx) => {
    const partitionedTables = await (0, pgmaintenance_db_utils_1.findTablesWithPartitions)(trx);
    const defaultPartitionPostfix = "_default";
    for (const pt of partitionedTables) {
        const defaulPartitionName = pt.table_name + defaultPartitionPostfix;
        const defaultTableEntryCount = (await trx.query(`
			SELECT COUNT(*)::INTEGER AS c
			FROM "${pt.schema}"."${defaulPartitionName}"
		`)).rows[0].c;
        if (defaultTableEntryCount === 0) {
            continue;
        }
        else {
            app_logs_1.logger.info(`[partition-manager.ts] There are entries in table: "${pt.schema}"."${defaulPartitionName}"`);
            try {
                await trx.query(`SAVEPOINT partitionrecovery`);
                const partitionedTableDependentForeignKeys = await (0, pgmaintenance_db_utils_1.findForeignKeysReferencingGivenTable)(pt.schema, pt.table_name, trx);
                if (partitionedTableDependentForeignKeys.length) {
                    app_logs_1.logger.info(`[partition-manager.ts] Dropping foreign key constraints of tables: ${partitionedTableDependentForeignKeys.map((m) => m.referencing_table).join(", ")}`);
                    for (const ptfk of partitionedTableDependentForeignKeys) {
                        await trx.query(`
						ALTER TABLE IF EXISTS ${ptfk.referencing_table}
						DROP CONSTRAINT "${ptfk.reference_name}"
						`);
                    }
                }
                app_logs_1.logger.info(`[partition-manager.ts] Disabling triggers of table "${pt.schema}"."${pt.table_name}" and detach default partition`);
                await trx.query(`
						ALTER TABLE "${pt.schema}"."${pt.table_name}"
						DISABLE TRIGGER ALL;
	
						ALTER TABLE "${pt.schema}"."${pt.table_name}"
						DETACH PARTITION "${pt.schema}"."${defaulPartitionName}"
					`);
                const missingPartitionMonthsDbResult = await trx.query(`
						SELECT DATE_TRUNC ('month', "${pt.partition_key}") AS "missingPartitionMonth", COUNT(*) AS c
						FROM "${pt.schema}"."${defaulPartitionName}"
						GROUP BY DATE_TRUNC ('month', "${pt.partition_key}")
						ORDER BY DATE_TRUNC ('month', "${pt.partition_key}") ASC
					`);
                for (const row of missingPartitionMonthsDbResult.rows) {
                    const missingPartitionMonth = row.missingPartitionMonth;
                    const partitionToCreate = pt.table_name + "_p" + luxon_1.DateTime.fromJSDate(missingPartitionMonth).toFormat("yyyyMM");
                    let partitionStart = null;
                    let partitionEnd = null;
                    switch (pt.partition_key_type) {
                        case "timestamp":
                        case "timestamptz":
                            partitionStart = luxon_1.DateTime.fromJSDate(missingPartitionMonth).startOf("month").toFormat("yyyy-MM-dd HH:mm:ssZZ");
                            partitionEnd = luxon_1.DateTime.fromJSDate(missingPartitionMonth).plus({ months: 1 }).startOf("month").toFormat("yyyy-MM-dd HH:mm:ssZZ");
                            break;
                        case "date":
                            partitionStart = luxon_1.DateTime.fromJSDate(missingPartitionMonth).startOf("month").toFormat("yyyy-MM-dd");
                            partitionEnd = luxon_1.DateTime.fromJSDate(missingPartitionMonth).plus({ months: 1 }).startOf("month").toFormat("yyyy-MM-dd");
                            break;
                    }
                    const previousPartition = pt.table_name + "_p" + luxon_1.DateTime.fromJSDate(missingPartitionMonth).minus({ months: 1 }).toFormat("yyyyMM");
                    const nextPartition = pt.table_name + "_p" + luxon_1.DateTime.fromJSDate(missingPartitionMonth).plus({ months: 1 }).toFormat("yyyyMM");
                    const previousPartitionCheckDbResult = await (0, pgmaintenance_db_utils_1.getPartitionBoundryExpression)(pt.schema, previousPartition, trx);
                    if (previousPartitionCheckDbResult) {
                        partitionStart = previousPartitionCheckDbResult.partitionBoundryExpression.split("'")[3];
                    }
                    const nextPartitionCheckDbResult = await (0, pgmaintenance_db_utils_1.getPartitionBoundryExpression)(pt.schema, nextPartition, trx);
                    if (nextPartitionCheckDbResult) {
                        partitionEnd = nextPartitionCheckDbResult.partitionBoundryExpression.split("'")[1];
                    }
                    app_logs_1.logger.info(`[partition-manager.ts] There are ${row.c} entries in default partition for "${pt.schema}"."${partitionToCreate}" between ${partitionStart} and ${partitionEnd}`);
                    app_logs_1.logger.info(`[partition-manager.ts] Creating partition "${pt.schema}"."${partitionToCreate}"`);
                    await trx.query(`
						CREATE TABLE IF NOT EXISTS "${pt.schema}"."${partitionToCreate}"
						PARTITION OF "${pt.schema}"."${pt.table_name}"
						FOR VALUES FROM ('${partitionStart}') TO ('${partitionEnd}')
					`);
                }
                app_logs_1.logger.info(`[partition-manager.ts] All necessary partitions are created, copying data from "${pt.schema}"."${defaulPartitionName}" to table "${pt.schema}"."${pt.table_name}"`);
                await trx.query(`
					INSERT INTO "${pt.schema}"."${pt.table_name}"
					SELECT * FROM "${pt.schema}"."${defaulPartitionName}"
				`);
                app_logs_1.logger.info(`[partition-manager.ts] Delete data from "${pt.schema}"."${defaulPartitionName}"`);
                await trx.query(`
					DELETE FROM "${pt.schema}"."${defaulPartitionName}"
				`);
                app_logs_1.logger.info(`[partition-manager.ts] Re-enabling triggers of table "${pt.schema}"."${pt.table_name}" and attach default partition`);
                await trx.query(`
						ALTER TABLE "${pt.schema}"."${pt.table_name}"
						ATTACH PARTITION "${pt.schema}"."${defaulPartitionName}" DEFAULT;
					
						ALTER TABLE "${pt.schema}"."${pt.table_name}"
						ENABLE TRIGGER ALL;
					`);
                if (partitionedTableDependentForeignKeys.length) {
                    app_logs_1.logger.info(`[partition-manager.ts] Re-adding foreign key constraints of tables: ${partitionedTableDependentForeignKeys.map((m) => m.referencing_table).join(", ")}`);
                    for (const ptfk of partitionedTableDependentForeignKeys) {
                        await trx.query(`
							ALTER TABLE ${ptfk.referencing_table}
							ADD CONSTRAINT "${ptfk.reference_name}"
							${ptfk.reference_definition}
							`);
                    }
                }
                app_logs_1.logger.info(`[partition-manager.ts] Missing partitions created successfully for table: "${pt.schema}"."${pt.table_name}"`);
                await trx.query(`RELEASE SAVEPOINT partitionrecovery`);
            }
            catch (error) {
                app_logs_1.logger.error(`[partition-manager.ts] Error while partitionRecovery process. Schema name: ${pt.schema}, Table name: ${pt.table_name}`);
                app_logs_1.logger.error(error);
                await trx.query(`ROLLBACK TO SAVEPOINT partitionrecovery`);
            }
        }
    }
};
exports.partitionRecovery = partitionRecovery;
