"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const elastic_apm_node_1 = __importDefault(require("elastic-apm-node"));
if (process.env.ELASTIC_APM_SERVER_URL &&
    process.env.ELASTIC_APM_SECRET_TOKEN &&
    process.env.ELASTIC_APM_PREFIX &&
    ((process.env.ELASTIC_APM_VERIFY_SERVER_CERT === "true" && process.env.ELASTIC_APM_SERVER_CA_CERT_FILE) || process.env.ELASTIC_APM_VERIFY_SERVER_CERT === "false")) {
    elastic_apm_node_1.default.start({
        serviceName: `${process.env.ELASTIC_APM_PREFIX}-pgmaintenance`,
        serverUrl: process.env.ELASTIC_APM_SERVER_URL,
        secretToken: process.env.ELASTIC_APM_SECRET_TOKEN,
        environment: "production",
        serverCaCertFile: process.env.ELASTIC_APM_SERVER_CA_CERT_FILE ?? null,
        verifyServerCert: process.env.ELASTIC_APM_VERIFY_SERVER_CERT === "true",
    });
}
else {
    console.warn(`Elastic APM is not enabled. Please set ELASTIC_APM_SERVER_URL, ELASTIC_APM_SECRET_TOKEN, ELASTIC_APM_PREFIX, ELASTIC_APM_VERIFY_SERVER_CERT, and ELASTIC_APM_SERVER_CA_CERT_FILE environment variables.`);
}
const pg_1 = require("pg");
const app_logs_1 = require("../../app.logs");
const pgmaintenance_config_1 = __importDefault(require("./pgmaintenance.config"));
const partition_manager_1 = require("./db/partition-manager");
const luxon_1 = require("luxon");
const messageBroker_manager_1 = require("../../messageBroker/messageBroker.manager");
const messageBroker_server_to_pgmaintenance_sub_1 = require("../../messageBroker/messageBroker.server-to-pgmaintenance.sub");
const pool = new pg_1.Pool({
    user: pgmaintenance_config_1.default.PGSUPERUSER,
    application_name: "pg_maintenance_" + process.env.HOSTNAME,
    max: 100,
    keepAlive: true,
});
const forwardPartitionCount = 4;
async function start() {
    try {
        const onMessage = async (channel, msg) => {
            if (msg === null) {
                return;
            }
            try {
                const data = JSON.parse(msg.content.toString());
                let response;
                switch (data.e) {
                    case messageBroker_server_to_pgmaintenance_sub_1.amqpServerToPGMaintenanceSubEventNames.backwardPartitioning:
                        let partitionRes = await (0, partition_manager_1.createBackwardPartitions)(pool, data.d["months"], data.d["orgId"]);
                        response = { isSuccess: partitionRes };
                        break;
                    default:
                        app_logs_1.logger.warn(`[server-to-pgmaintenance.sub] There is no handler for type ${data.e}`);
                        app_logs_1.logger.warn(`[server-to-pgmaintenance.sub] Data from RabbitMQ: ${JSON.stringify(data, null, 4)}. Discarding message`);
                        break;
                }
                if (msg.properties.replyTo && response) {
                    channel.sendToQueue(msg.properties.replyTo, Buffer.from(JSON.stringify(response)), { correlationId: msg.properties.correlationId });
                }
                channel.ack(msg);
            }
            catch (error) {
                app_logs_1.logger.error("[server-to-pgmaintenance.sub] error while consuming msg %s", error.message || error);
                channel.nack(msg);
            }
        };
        await messageBroker_manager_1.messageBrokerManager.initAmqpForPgMaintenanceService(pgmaintenance_config_1.default, onMessage);
        await (0, partition_manager_1.createForwardPartitions)(pool, forwardPartitionCount);
        let now = luxon_1.DateTime.now();
        let targetDate = null;
        if (now.weekday > 6 || (now.weekday === 6 && now.hour > 5)) {
            targetDate = luxon_1.DateTime.now().plus({ weeks: 1 }).set({ weekday: 6, hour: 5, minute: 0, second: 0, millisecond: 0 });
        }
        else {
            targetDate = luxon_1.DateTime.now().set({ weekday: 6, hour: 5, minute: 0, second: 0, millisecond: 0 });
        }
        app_logs_1.logger.info(`Next run time for partition creation is ${targetDate.toISO()}`);
        app_logs_1.logger.info(`Milliseconds until next run: ${targetDate.diffNow().milliseconds}`);
        setTimeout(async () => {
            try {
                await (0, partition_manager_1.createForwardPartitions)(pool, forwardPartitionCount);
            }
            catch (error) {
                app_logs_1.logger.error(`Error while creating forward partitions`);
                app_logs_1.logger.error(error);
                process.exit(1);
            }
            setInterval(async () => {
                try {
                    await (0, partition_manager_1.createForwardPartitions)(pool, forwardPartitionCount);
                }
                catch (error) {
                    app_logs_1.logger.error(`Error while creating forward partitions`);
                    app_logs_1.logger.error(error);
                    process.exit(1);
                }
            }, 7 * 24 * 60 * 60 * 1000);
        }, targetDate.diffNow().milliseconds);
    }
    catch (error) {
        app_logs_1.logger.error(error || error.message);
        process.exit(1);
    }
}
start();
