"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.DefaultUserTask = void 0;
const app_logs_1 = require("../../../../app.logs");
const dal_constants_1 = require("../../../../dal/dal.constants");
const dal_db_armon_schema_1 = require("../../../../dal/db/armon/dal.db.armon.schema");
const integration_config_1 = __importDefault(require("../../integration.config"));
const integration_base_1 = require("../../integrations/integration.base");
const predefined_roles_1 = require("../../../../dal/db/predefined/predefined.roles");
class DefaultUserTask {
    constructor() {
        this.standartUserTypeId = predefined_roles_1.PredefinedRoles.StandartUser.id;
        this.standartUserOrganizationTypeId = predefined_roles_1.PredefinedRoles.StandartUserOrganization.id;
        this.unitAdministratorTypeId = predefined_roles_1.PredefinedRoles.UnitAdministrator.id;
    }
    static create() {
        return new DefaultUserTask();
    }
    async execute(trx, token, taskId, identityData) {
        app_logs_1.logger.info("Executing task for: " + identityData.id);
        let taskData = {
            uniqueId: identityData.id,
            taskid: taskId,
        };
        let extensionFieldsFromReq = {};
        let orgUnits;
        let orgGroups;
        let roles;
        try {
            let extensionFieldsOfOrg;
            let orgUnitShortCodes = identityData.units?.map((unit) => {
                return unit.orgUnitId;
            });
            extensionFieldsOfOrg = (await trx.query(`SELECT fields
					FROM "${integration_config_1.default.INTEGRATION_ORGANIZATION}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationForms}" of
					WHERE of."organizationId" = $1`, [integration_config_1.default.INTEGRATION_ORGANIZATION])).rows[0];
            if (extensionFieldsOfOrg) {
                extensionFieldsOfOrg.fields.forEach((field) => {
                    if (field.name in identityData.fields) {
                        extensionFieldsFromReq[field.name] = identityData.fields[field.name];
                        delete identityData.fields[field.name];
                    }
                });
            }
            orgUnits = (await trx.query(`SELECT "id", "shortCode"
					FROM "${integration_config_1.default.INTEGRATION_ORGANIZATION}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" ou
					WHERE ou."shortCode" = ANY($1::varchar[])`, [orgUnitShortCodes])).rows;
            orgGroups = (await trx.query(`SELECT "id", "name"
					FROM "${integration_config_1.default.INTEGRATION_ORGANIZATION}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroups}" og
					WHERE og."name" = ANY($1::varchar[])`, [identityData.groups])).rows;
            roles = (await trx.query(`SELECT "id", "typeId"
					FROM "${integration_config_1.default.INTEGRATION_ORGANIZATION}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.roles}" r
					WHERE r."typeId" = ANY($1::uuid[])`, [[this.standartUserTypeId, this.unitAdministratorTypeId, this.standartUserOrganizationTypeId]])).rows;
        }
        catch (error) {
            app_logs_1.logger.error("errr: ", error);
        }
        let userUnitsFromReq = [];
        let userGroupsFromReq = [];
        let standartUserRoleId = roles.find((row) => row.typeId === this.standartUserTypeId).id;
        let standartUserOrganizationRoleId = roles.find((row) => row.typeId === this.standartUserOrganizationTypeId).id;
        let unitAdministratorRoleId = roles.find((row) => row.typeId === this.unitAdministratorTypeId).id;
        identityData.units?.forEach((unit) => {
            let foundedUnit = orgUnits.find((row) => row.shortCode === unit.orgUnitId);
            if (foundedUnit) {
                userUnitsFromReq.push({
                    organizationUnitId: foundedUnit.id,
                    roleId: unit.isManager ? unitAdministratorRoleId : standartUserRoleId,
                });
            }
        });
        identityData.groups?.forEach((group) => {
            let foundedGroup = orgGroups.find((row) => row.name === group);
            if (foundedGroup) {
                userGroupsFromReq.push(foundedGroup);
            }
        });
        const user = (await trx.query(`SELECT uop.*, uo."isDisabled", uo."roleId"
                FROM "${integration_config_1.default.INTEGRATION_ORGANIZATION}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" uop
                INNER JOIN "${integration_config_1.default.INTEGRATION_ORGANIZATION}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo
					ON uop."userOrganizationId" = uo."id"
                WHERE "uniqueId" = $1 AND uo."deletedAt" IS NULL AND uop."deletedAt" IS NULL;`, [identityData.id])).rows[0];
        if (identityData.photo) {
            identityData.photo = identityData.photo.replace(/^data:image\/\w+;base64,/, "");
        }
        if (!user) {
            app_logs_1.logger.info("Will insert new user for " + identityData.id);
            const requestBody = {
                organizationProfile: {
                    uniqueId: identityData.id,
                    isDisabled: false,
                    name: identityData.name,
                    surname: identityData.surname,
                    previousServiceDuration: 0,
                    pacsEnabledRemainedAnnualPPermission: 0,
                    roleId: identityData.roleId ?? standartUserOrganizationRoleId,
                    extensionFields: extensionFieldsFromReq,
                },
                organizationUnits: userUnitsFromReq,
                accessRights: [],
                credentials: identityData.credentials,
                userAccount: {
                    username: identityData.account?.username,
                    password: identityData.account?.password,
                },
                userGroupIds: userGroupsFromReq.map((group) => {
                    return group.id;
                }),
                regions: [],
                addDefaultAccessRightsOfUnits: true,
                source: dal_constants_1.DalConstants.SourceType.Integration,
                sourceId: identityData.integrationClientId,
            };
            Object.keys(identityData.fields).forEach((field) => {
                if (field === "employmentStartUtc" && identityData.fields[field]) {
                    requestBody.organizationProfile[field] = new Date(identityData.fields[field]);
                }
                else if (field === "employmentEndUtc" && identityData.fields[field]) {
                    requestBody.organizationProfile[field] = new Date(identityData.fields[field]);
                }
                else {
                    requestBody.organizationProfile[field] = identityData.fields[field];
                }
            });
            let insertedUser;
            try {
                insertedUser = await token.httpRequest(taskData, "POST", `/u/v1/${integration_config_1.default.INTEGRATION_ORGANIZATION}/identity/add`, requestBody);
            }
            catch (error) {
                app_logs_1.logger.error("Could not insert user: " + error.status + " - " + error.message);
                throw error;
            }
            if (insertedUser?.status === 200) {
                app_logs_1.logger.info("User " + identityData.id + " inserted with id: " + insertedUser.data.id);
                if (identityData.photo) {
                    const thumbnail = await token.httpRequest(taskData, "POST", `/u/v1/${integration_config_1.default.INTEGRATION_ORGANIZATION}/identity/${insertedUser.data.id}/thumbnail/upsert`, {
                        data: identityData.photo,
                    });
                    if (thumbnail?.status === 200) {
                        app_logs_1.logger.info("Profile picture added");
                    }
                    else {
                        app_logs_1.logger.error("Could not added profile picture: " + thumbnail?.status + " - " + thumbnail?.data?.message);
                    }
                }
            }
        }
        else {
            app_logs_1.logger.info("Will update user for " + identityData.id + " with id " + user.userId);
            const existingUnits = (await trx.query(`SELECT "organizationUnitId" as id, uoou."roleId" as role, uoou."sourceId", uoou."source"
                    FROM "${integration_config_1.default.INTEGRATION_ORGANIZATION}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}" uoou
                    INNER JOIN "${integration_config_1.default.INTEGRATION_ORGANIZATION}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" ou
                        ON ou.id = uoou."organizationUnitId" AND uoou."deletedAt" IS NULL AND ou."deletedAt" IS NULL
                    WHERE uoou."userOrganizationId" = $1`, [user.userOrganizationId])).rows;
            const existingGroupIdsOfUser = (await trx.query(`SELECT "userGroupId", uguo."sourceId", uguo."source"
					FROM "${integration_config_1.default.INTEGRATION_ORGANIZATION}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupUserOrganizations}" as uguo
					INNER JOIN "${integration_config_1.default.INTEGRATION_ORGANIZATION}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" as uo
						ON uo."id" = uguo."userOrganizationId"
					WHERE	uo."userId" = $1 AND
						uo."isDisabled" = false AND
						uo."deletedAt" IS NULL AND
						uguo."deletedAt" IS  NULL
					;`, [user.userId])).rows;
            const userOrganizationCredentials = (await trx.query(`
				SELECT id, "type", data, "credentialNumber", "expiresOn" AS "expirationUtc", "extensionFields", "sourceId", "source",
					"customFields", "note", "specialDataSecondary", "groupNumber"
				FROM "${integration_config_1.default.INTEGRATION_ORGANIZATION}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials}"
				WHERE "userId" = $1 AND "deletedAt" IS NULL
			`, [user.userId])).rows;
            const updatedCredentials = [];
            if (identityData.credentials) {
                userOrganizationCredentials.forEach((orgCred) => {
                    if (orgCred.source == dal_constants_1.DalConstants.SourceType.Integration) {
                        const match = identityData.credentials.find((cred) => cred.data === orgCred.data && cred.type === orgCred.type);
                        if (match && (match.expirationUtc !== orgCred.expirationUtc || orgCred.sourceId !== identityData.integrationClientId)) {
                            updatedCredentials.push({
                                id: orgCred.id,
                                data: orgCred.data,
                                type: orgCred.type,
                                expirationUtc: match.expirationUtc,
                                source: dal_constants_1.DalConstants.SourceType.Integration,
                                sourceId: identityData.integrationClientId,
                            });
                        }
                    }
                });
            }
            let standartUserRoleId = roles.find((row) => row.typeId === this.standartUserTypeId).id;
            let unitAdministratorRoleId = roles.find((row) => row.typeId === this.unitAdministratorTypeId).id;
            const receivedOrganizations = [];
            identityData.units?.forEach((unit) => {
                let foundedUnit = orgUnits.find((row) => row.shortCode === unit.orgUnitId);
                if (foundedUnit) {
                    if (unit.isManager) {
                        app_logs_1.logger.info("User will be manager for " + unit.orgUnitId + " when user profile is syncronized");
                        receivedOrganizations.push({ id: foundedUnit.id, role: unitAdministratorRoleId });
                    }
                    else {
                        receivedOrganizations.push({ id: foundedUnit.id, role: standartUserRoleId });
                    }
                }
            });
            const requestBody = {
                addedOrganizationUnits: receivedOrganizations
                    .filter((o) => !existingUnits?.find((e) => e.id === o.id))
                    .map((o) => {
                    return {
                        organizationUnitId: o.id,
                        roleId: o.role,
                        extraPermissions: [],
                    };
                }),
                updatedOrganizationUnits: receivedOrganizations
                    .filter((o) => existingUnits?.find((e) => e.id === o.id &&
                    (e.role != o.role || e.sourceId != identityData.integrationClientId) &&
                    [unitAdministratorRoleId, standartUserRoleId].includes(e.role) &&
                    e.source == dal_constants_1.DalConstants.SourceType.Integration))
                    .map((o) => {
                    return {
                        organizationUnitId: o.id,
                        roleId: o.role,
                        extraPermissions: [],
                    };
                }),
                removedOrganizationUnitIds: existingUnits
                    ?.filter((e) => !receivedOrganizations.find((o) => o.id === e.id) && e.source == dal_constants_1.DalConstants.SourceType.Integration && e.sourceId == identityData.integrationClientId)
                    .map((e) => e.id),
                addedAccessRights: [],
                updatedAccessRights: [],
                removedAccessControlPointIds: [],
                addedCredentials: identityData.credentials?.filter((c) => !userOrganizationCredentials.find((ec) => ec.data === c.data && ec.type === c.type)),
                updatedCredentials: updatedCredentials,
                removedCredentialIds: userOrganizationCredentials
                    ?.filter((ec) => !identityData.credentials?.find((c) => ec.data === c.data && ec.type === c.type) &&
                    ec.source == dal_constants_1.DalConstants.SourceType.Integration &&
                    ec.sourceId == identityData.integrationClientId)
                    .map((ec) => ec.id),
                addedUserGroupIds: userGroupsFromReq
                    .filter((ug) => !existingGroupIdsOfUser.find((e) => e.userGroupId === ug.id && e.sourceId == identityData.integrationClientId))
                    .map((ug) => {
                    return ug.id;
                }),
                removedUserGroupIds: existingGroupIdsOfUser
                    ?.filter((e) => !userGroupsFromReq.find((ug) => ug.id === e.userGroupId && e.sourceId === identityData.integrationClientId) &&
                    e.source == dal_constants_1.DalConstants.SourceType.Integration &&
                    e.sourceId == identityData.integrationClientId)
                    .map((e) => e.userGroupId),
                removedRegionIds: [],
                addedRegions: [],
                updatedRegions: [],
                id: user.userId,
                organizationProfile: {
                    roleId: identityData.roleId ?? user.roleId,
                    isDisabled: !identityData.active,
                    uniqueId: identityData.id,
                    name: identityData.name,
                    surname: identityData.surname,
                    extensionFields: { ...user.extensionFields, ...extensionFieldsFromReq },
                    previousServiceDuration: 0,
                    pacsEnabledRemainedAnnualPPermission: 0,
                    locale: "tr",
                    isAnonymized: false,
                },
                userAccount: {
                    username: identityData.account?.username,
                    password: identityData.account?.password,
                },
                addDefaultAccessRightsOfUnits: true,
                removeDefaultAccessRightsOfUnits: true,
                source: dal_constants_1.DalConstants.SourceType.Integration,
                sourceId: identityData.integrationClientId,
            };
            if (!identityData.account) {
                delete requestBody.userAccount;
            }
            Object.keys(identityData.fields).forEach((field) => {
                if (field === "employmentStartUtc" && identityData.fields[field]) {
                    requestBody.organizationProfile[field] = new Date(identityData.fields[field]);
                }
                else if (field === "employmentEndUtc" && identityData.fields[field]) {
                    requestBody.organizationProfile[field] = new Date(identityData.fields[field]);
                }
                else {
                    requestBody.organizationProfile[field] = identityData.fields[field];
                }
            });
            let updatedUser;
            try {
                updatedUser = await token.httpRequest(taskData, "POST", `/u/v1/${integration_config_1.default.INTEGRATION_ORGANIZATION}/identity/update`, requestBody);
            }
            catch (error) {
                app_logs_1.logger.error("Could not update user: " + error.status + " - " + error.message);
                throw error;
            }
            if (updatedUser?.status === 200) {
                app_logs_1.logger.info("User " + identityData.id + " updated successfully");
                if (identityData.photo) {
                    const thumbnail = await token.httpRequest(taskData, "POST", `/u/v1/${integration_config_1.default.INTEGRATION_ORGANIZATION}/identity/${user.userId}/thumbnail/upsert`, {
                        data: identityData.photo,
                    });
                    if (thumbnail?.status === 200) {
                        app_logs_1.logger.info("Profile picture updated");
                    }
                    else {
                        app_logs_1.logger.error("Could not update profile picture: " + thumbnail?.status + " - " + thumbnail?.data?.message);
                    }
                }
            }
        }
        return integration_base_1.OperationStatus.Success;
    }
}
exports.DefaultUserTask = DefaultUserTask;
DefaultUserTask.id = "0a1e0a9f-c7e3-4576-808e-ecaf95abcda0";
