"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.migrate = void 0;
const path_1 = __importDefault(require("path"));
const app_logs_1 = require("../../../../app.logs");
const cli_config_1 = __importDefault(require("../../cli.config"));
const utils_1 = require("./utils");
const partition_manager_1 = require("../../../pgmaintenance/db/partition-manager");
async function migrateUpTo(client, name, dbuser, dbsuperuser) {
    const migration = require.call(this, path_1.default.resolve(__dirname, "..", "..", "..", "..", "dal", "db", "migrations", "postgresql", name));
    app_logs_1.logger.info("DB is migrating up to " + name);
    await migration.up(client, dbuser, dbsuperuser);
    app_logs_1.logger.info("DB is migrated up to " + name + " successfully");
    await client.query(`INSERT INTO public.migration
                    ("name", "migratedAt") VALUES ($1, now())`, [name]);
}
async function migrateDownTo(client, name, dbuser, dbsuperuser) {
    if (!name) {
        return;
    }
    const migration = require.call(this, path_1.default.resolve(__dirname, "..", "..", "..", "..", "dal", "db", "migrations", "postgresql", name));
    app_logs_1.logger.info("DB is migrating down to " + name);
    await migration.down(client, dbuser, dbsuperuser);
    app_logs_1.logger.info("DB is migrated down to " + name + " successfully");
    await client.query(`DELETE FROM public.migration
        WHERE "name" = $1`, [name]);
}
async function migrateTo(client, migrationFiles, migrateTo, nextMigrationIndex, migrationList, dbuser, dbsuperuser) {
    let currentMigration = "";
    let migrateToFileIndex = 0;
    if (nextMigrationIndex > 0) {
        migrateToFileIndex = migrationFiles.indexOf(migrateTo);
        currentMigration = migrationList[nextMigrationIndex - 1];
        if (currentMigration === migrateTo) {
            return;
        }
    }
    else {
        for (let index = migrationFiles.length - 1; migrationFiles[index] != migrateTo; index--) {
            await client.query("BEGIN");
            await migrateDownTo(client, migrationFiles[index], dbuser, dbsuperuser);
            await client.query("COMMIT");
        }
        return;
    }
    if (currentMigration < migrateTo) {
        for (let index = nextMigrationIndex; index <= migrateToFileIndex; index++) {
            await client.query("BEGIN");
            await migrateUpTo(client, migrationFiles[index], dbuser, dbsuperuser);
            await client.query("COMMIT");
        }
    }
    else {
        app_logs_1.logger.error("Should not reach here, there is an error in migration mechanism");
    }
}
async function migrate(trx, dbuser, dbsuperuser, targetMigrationName) {
    const migrationFiles = (0, utils_1.getMigrationFiles)();
    try {
        await trx.query(`
			SELECT pg_terminate_backend(pg_stat_activity.pid) FROM pg_stat_activity 
				WHERE pg_stat_activity.datname = '${cli_config_1.default.PGDATABASE}' AND pg_stat_activity.usename = '${cli_config_1.default.PGUSER}' AND pid <> pg_backend_pid();`);
        await trx.query("COMMIT");
        const migrationCheckResult = await (0, utils_1.checkFileSystemAgainstMigrationsTable)(trx, targetMigrationName);
        if (targetMigrationName && migrationCheckResult.isDbCompatible && migrationCheckResult.migrationNeeded) {
            if (migrationCheckResult.nextMigrationIndex < 0) {
                await migrateTo(trx, migrationCheckResult.migrationList, targetMigrationName, migrationCheckResult.nextMigrationIndex, migrationCheckResult.migrationList, dbuser, dbsuperuser);
            }
            else {
                await migrateTo(trx, migrationFiles, targetMigrationName, migrationCheckResult.nextMigrationIndex, migrationCheckResult.migrationList, dbuser, dbsuperuser);
            }
        }
        else if (migrationCheckResult.isDbCompatible && migrationCheckResult.migrationNeeded && migrationCheckResult.nextMigrationIndex != -1) {
            for (let index = migrationCheckResult.nextMigrationIndex; index < migrationFiles.length; index++) {
                await trx.query("BEGIN");
                await migrateUpTo(trx, migrationFiles[index], dbuser, dbsuperuser);
                await trx.query("COMMIT");
            }
        }
        await trx.query("BEGIN");
        try {
            await (0, partition_manager_1.partitionRecovery)(trx);
        }
        catch (partitionError) {
            await trx.query("ROLLBACK");
            app_logs_1.logger.error("Error while creating missing partitions (probably notification_instance and notification_event tables)");
            app_logs_1.logger.error(partitionError);
            await trx.query("BEGIN");
        }
    }
    catch (error) {
        await trx.query("ROLLBACK");
        app_logs_1.logger.error(error);
        await trx.query("BEGIN");
    }
}
exports.migrate = migrate;
