"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.runDbBackupService = void 0;
const child_process_1 = require("child_process");
const fs_1 = __importDefault(require("fs"));
const path_1 = __importDefault(require("path"));
const moment = require("moment");
const uuid = require("uuid");
const app_config_1 = require("../../../../app.config");
const app_logs_1 = require("../../../../app.logs");
const cli_config_1 = __importDefault(require("../../cli.config"));
const dal_db_armon_schema_1 = require("../../../../dal/db/armon/dal.db.armon.schema");
const app_constants_1 = require("../../../../app.constants");
var mainDbBackupFilePrefix = "";
async function runDbBackupService(trx) {
    console.info("Db maintenance job is starting...");
    try {
        app_config_1.appConfig.init(app_constants_1.ServiceNames.PGMaintenanceService);
        app_logs_1.appLogger.init(app_config_1.appConfig.dbLogDirectory);
        mainDbBackupFilePrefix += cli_config_1.default.PGDATABASE;
    }
    catch (error) {
        console.error(error);
        process.exit(1);
    }
    let organizationIds = [];
    let partitionedTableNames = [];
    try {
        organizationIds = (await trx.query(`SELECT id from public."organizationList";`)).rows.map((r) => r.id);
        partitionedTableNames = (await trx.query(` SELECT DISTINCT(relname) FROM pg_class
				WHERE relkind = 'p' AND oid IN (SELECT DISTINCT inhparent FROM pg_inherits);`)).rows.map((r) => r.relname);
    }
    catch (err) {
        app_logs_1.logger.error(err);
    }
    try {
        app_logs_1.logger.info("Archiving...");
        await archiveAndDetachExpiredData(partitionedTableNames, organizationIds, trx);
        app_logs_1.logger.info("Archived successfully...");
    }
    catch (error) {
        if (error.message && error.message.indexOf("pg_dump: no matching tables were found") >= 0) {
            app_logs_1.logger.info("There is no table found to archive");
        }
        else {
            app_logs_1.logger.error(error);
            process.exit(1);
        }
    }
    app_logs_1.logger.info("Db maintainance is initialized!");
    try {
        app_logs_1.logger.info("Start backup ops.");
        for (const oid of organizationIds) {
            dbBackup(oid);
        }
        dbBackup("public");
        app_logs_1.logger.info("Backed up successfully");
    }
    catch (error) {
        app_logs_1.logger.error(error);
        process.exit(1);
    }
    app_logs_1.logger.info("Db is maintained successfully :)");
}
exports.runDbBackupService = runDbBackupService;
async function archiveAndDetachExpiredData(partitionedTableNames, organizationIds, trx) {
    const optime = moment();
    const eventIndex = partitionedTableNames.indexOf(dal_db_armon_schema_1.ArmonSchema.tableNames.notificationEvent), instanceIndex = partitionedTableNames.indexOf(dal_db_armon_schema_1.ArmonSchema.tableNames.notificationInstance);
    if (instanceIndex > eventIndex && eventIndex > -1 && instanceIndex > -1) {
        partitionedTableNames[eventIndex] = dal_db_armon_schema_1.ArmonSchema.tableNames.notificationInstance;
        partitionedTableNames[instanceIndex] = dal_db_armon_schema_1.ArmonSchema.tableNames.notificationEvent;
    }
    const accessLogIndex = partitionedTableNames.indexOf(dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs), userRegionStateIndex = partitionedTableNames.indexOf(dal_db_armon_schema_1.ArmonSchema.tableNames.userRegionStates);
    if (userRegionStateIndex > accessLogIndex && accessLogIndex > -1 && userRegionStateIndex > -1) {
        partitionedTableNames[accessLogIndex] = dal_db_armon_schema_1.ArmonSchema.tableNames.userRegionStates;
        partitionedTableNames[userRegionStateIndex] = dal_db_armon_schema_1.ArmonSchema.tableNames.AccessLogs;
    }
    let tableSpecificNames = app_config_1.appConfig.db.log.backup && app_config_1.appConfig.db.log.backup.tableSpecific ? Object.keys(app_config_1.appConfig.db.log.backup.tableSpecific) : [];
    for (const oid of organizationIds) {
        const pathToArchiveDir = path_1.default.join(app_config_1.appConfig.db.log.backup.archiveDirectory, oid, optime.format("YYYY-MM"));
        app_logs_1.logger.info(pathToArchiveDir);
        fs_1.default.mkdirSync(pathToArchiveDir, { recursive: true });
        const backupFilePrefix = mainDbBackupFilePrefix + optime.format("YYYYMMDDHHmmss");
        let outFileNotPartitionedForOrganizationSchema = path_1.default.join(pathToArchiveDir, backupFilePrefix + ".non-partitioned.backup");
        let outFilePartitionedForOrganizationSchema = path_1.default.join(pathToArchiveDir, backupFilePrefix + ".partitioned.backup");
        let outFileMetadataForOrganizationSchema = path_1.default.join(pathToArchiveDir, backupFilePrefix + ".txt");
        let filesInOrganizationArchiveDir = fs_1.default.readdirSync(pathToArchiveDir).sort();
        if (filesInOrganizationArchiveDir.length > 0 && filesInOrganizationArchiveDir.filter((f) => f.startsWith(mainDbBackupFilePrefix + optime.format("YYYYMM"))).length > 0) {
            continue;
        }
        let archiveCommandForNotPartitionedTablesInOrganizationSchema = `pg_dump -U ${cli_config_1.default.PGSUPERUSER} -h ${cli_config_1.default.PGHOST} --schema "${oid}" --format=${app_config_1.appConfig.db.log.backup.format || "c"} -f "${outFileNotPartitionedForOrganizationSchema}" `;
        let archiveCommandForPartitionedTablesInOrganizationSchema = `pg_dump -U ${cli_config_1.default.PGSUPERUSER} -h ${cli_config_1.default.PGHOST} --schema "${oid}" --format=${app_config_1.appConfig.db.log.backup.format || "c"} -f "${outFilePartitionedForOrganizationSchema}" --load-via-partition-root -a`;
        let detachSqlForOrganizationSchema = "";
        let shouldArchivePartitionedTables = false;
        for (const tableName of partitionedTableNames) {
            if (!Object.values(dal_db_armon_schema_1.ArmonSchema.publicSchemaTableNames).includes(tableName) || tableName === dal_db_armon_schema_1.ArmonSchema.tableNames.notificationInstance) {
                let monthCount = app_config_1.appConfig.db.log.backup.expiredBeforeLastMonthsCount;
                if (tableSpecificNames.indexOf(tableName) >= 0 && app_config_1.appConfig.db.log.backup.tableSpecific[tableName] > 0) {
                    monthCount = app_config_1.appConfig.db.log.backup.tableSpecific[tableName];
                }
                const { rows } = await trx.query(`
					SELECT inhrelid::regclass::text AS child
					FROM   pg_catalog.pg_inherits
					WHERE  inhparent = '${oid}.${tableName}'::regclass
					AND inhrelid::regclass::text NOT ILIKE '%_default'
					ORDER BY inhrelid::regclass::text ASC;
				`);
                const partitionsToNotArchive = generatePartitionsToKeepForLastMonths(monthCount + 1);
                const partitionsToArchive = rows
                    .map((r) => r.child)
                    .filter((r) => {
                    for (const p of partitionsToNotArchive) {
                        if (r.endsWith(p)) {
                            return false;
                        }
                    }
                    if (r.slice(r.length - 6) > moment().format("YYYYMM")) {
                        return false;
                    }
                    return true;
                });
                archiveCommandForNotPartitionedTablesInOrganizationSchema += ` --exclude-table-data "${oid}.${tableName}*"`;
                if (partitionsToArchive.length > 0) {
                    shouldArchivePartitionedTables = true;
                    for (const partition of partitionsToArchive) {
                        const expiredMonth = partition.split(`${tableName}_p`)[1];
                        archiveCommandForPartitionedTablesInOrganizationSchema += ` -t "${oid}.${tableName}_p${expiredMonth}"`;
                        fs_1.default.appendFileSync(outFileMetadataForOrganizationSchema, `"${oid}"."${tableName}_p${expiredMonth}"\n`);
                        detachSqlForOrganizationSchema += `
						ALTER TABLE "${oid}"."${tableName}" DETACH PARTITION "${oid}"."${tableName}_p${expiredMonth}";
						DROP TABLE IF EXISTS "${oid}"."${tableName}_p${expiredMonth}";
					`;
                    }
                }
            }
        }
        archiveCommandForNotPartitionedTablesInOrganizationSchema += ` ${cli_config_1.default.PGDATABASE}`;
        (0, child_process_1.execSync)(archiveCommandForNotPartitionedTablesInOrganizationSchema, {
            env: {
                ...process.env,
                PGPASSFILE: cli_config_1.default.PGPASSFILE,
            },
            stdio: "pipe",
        });
        app_logs_1.logger.info("Archive for not-partitioned tables in " + oid + " schema is successfully saved at " + outFileNotPartitionedForOrganizationSchema);
        archiveCommandForPartitionedTablesInOrganizationSchema += ` ${cli_config_1.default.PGDATABASE}`;
        if (shouldArchivePartitionedTables) {
            (0, child_process_1.execSync)(archiveCommandForPartitionedTablesInOrganizationSchema, {
                env: {
                    ...process.env,
                    PGPASSFILE: cli_config_1.default.PGPASSFILE,
                },
            });
        }
        app_logs_1.logger.info("Archive for partitioned tables in " + oid + " schema is successfully saved at " + outFilePartitionedForOrganizationSchema);
        let detachSqlFile = path_1.default.join(pathToArchiveDir, uuid.v4() + ".sql");
        fs_1.default.writeFileSync(detachSqlFile, detachSqlForOrganizationSchema);
        let detachCommand = `psql -U ${cli_config_1.default.PGSUPERUSER} -h ${cli_config_1.default.PGHOST} -d ${cli_config_1.default.PGDATABASE} -f "${detachSqlFile}"`;
        (0, child_process_1.execSync)(detachCommand, {
            env: {
                ...process.env,
                PGPASSFILE: cli_config_1.default.PGPASSFILE,
            },
        });
        fs_1.default.unlinkSync(detachSqlFile);
    }
    const pathToArchiveDir = path_1.default.join(app_config_1.appConfig.db.log.backup.archiveDirectory, "public", optime.format("YYYY-MM"));
    fs_1.default.mkdirSync(pathToArchiveDir, { recursive: true });
    const backupFilePrefix = mainDbBackupFilePrefix + optime.format("YYYYMMDDHHmmss");
    let outFileNotPartitionedForPublicSchema = path_1.default.join(pathToArchiveDir, backupFilePrefix + ".non-partitioned.backup");
    let outFilePartitionedForPublicSchema = path_1.default.join(pathToArchiveDir, backupFilePrefix + ".partitioned.backup");
    let outFileMetadataForPublicSchema = path_1.default.join(pathToArchiveDir, backupFilePrefix + ".txt");
    let filesInPublicArchiveDir = fs_1.default.readdirSync(pathToArchiveDir).sort();
    if (filesInPublicArchiveDir.length > 0 && filesInPublicArchiveDir.filter((f) => f.startsWith(mainDbBackupFilePrefix + optime.format("YYYYMM"))).length > 0) {
        return;
    }
    let archiveCommandForNotPartitionedTablesInPublicSchema = `pg_dump -U ${cli_config_1.default.PGSUPERUSER} -h ${cli_config_1.default.PGHOST} --schema "public" -f "${outFileNotPartitionedForPublicSchema}" `;
    let archiveCommandForPartitionedTablesInPublicSchema = `pg_dump -U ${cli_config_1.default.PGSUPERUSER} -h ${cli_config_1.default.PGHOST} --schema "public"  --format=${app_config_1.appConfig.db.log.backup.format || "c"} -f "${outFilePartitionedForPublicSchema}" --load-via-partition-root -a`;
    let detachSqlForPublicSchema = "";
    let shouldArchivePartitionedTables = false;
    for (const tableName of partitionedTableNames) {
        if (Object.values(dal_db_armon_schema_1.ArmonSchema.publicSchemaTableNames).includes(tableName)) {
            let monthCount = app_config_1.appConfig.db.log.backup.expiredBeforeLastMonthsCount;
            if (tableSpecificNames.indexOf(tableName) >= 0 && app_config_1.appConfig.db.log.backup.tableSpecific[tableName] > 0) {
                monthCount = app_config_1.appConfig.db.log.backup.tableSpecific[tableName];
            }
            const { rows } = await trx.query(`
					SELECT inhrelid::regclass::text AS child
					FROM   pg_catalog.pg_inherits
					WHERE  inhparent = 'public.${tableName}'::regclass
					AND inhrelid::regclass::text NOT ILIKE '%_default'
					ORDER BY inhrelid::regclass::text ASC;
				`);
            const partitionsToNotArchive = generatePartitionsToKeepForLastMonths(monthCount + 1);
            const partitionsToArchive = rows
                .map((r) => r.child)
                .filter((r) => {
                for (const p of partitionsToNotArchive) {
                    if (r.endsWith(p)) {
                        return false;
                    }
                }
                if (r.slice(r.length - 6) > moment().format("YYYYMM")) {
                    return false;
                }
                return true;
            });
            archiveCommandForNotPartitionedTablesInPublicSchema += ` --exclude-table-data "${tableName}*"`;
            if (partitionsToArchive.length > 0) {
                shouldArchivePartitionedTables = true;
                for (const partition of partitionsToArchive) {
                    const expiredMonth = partition.split(`${tableName}_p`)[1];
                    archiveCommandForPartitionedTablesInPublicSchema += ` -t "public.${tableName}_p${expiredMonth}"`;
                    fs_1.default.appendFileSync(outFileMetadataForPublicSchema, `public."${tableName}_p${expiredMonth}"\n`);
                    detachSqlForPublicSchema += `
						ALTER TABLE "${tableName}" DETACH PARTITION "${tableName}_p${expiredMonth}";
						DROP TABLE IF EXISTS public."${tableName}_p${expiredMonth}";
					`;
                }
            }
        }
    }
    archiveCommandForNotPartitionedTablesInPublicSchema += ` ${cli_config_1.default.PGDATABASE}`;
    (0, child_process_1.execSync)(archiveCommandForNotPartitionedTablesInPublicSchema, {
        env: {
            ...process.env,
            PGPASSFILE: cli_config_1.default.PGPASSFILE,
        },
    });
    app_logs_1.logger.info("Archive for not-partitioned tables is successfully saved at " + outFileNotPartitionedForPublicSchema);
    archiveCommandForPartitionedTablesInPublicSchema += ` ${cli_config_1.default.PGDATABASE}`;
    if (shouldArchivePartitionedTables) {
        (0, child_process_1.execSync)(archiveCommandForPartitionedTablesInPublicSchema, {
            env: {
                ...process.env,
                PGPASSFILE: cli_config_1.default.PGPASSFILE,
            },
        });
    }
    app_logs_1.logger.info("Archive for partitioned tables in public schema is successfully saved at " + outFilePartitionedForPublicSchema);
    let detachSqlFile = path_1.default.join(pathToArchiveDir, uuid.v4() + ".sql");
    fs_1.default.writeFileSync(detachSqlFile, detachSqlForPublicSchema);
    let detachCommand = `psql -U ${cli_config_1.default.PGSUPERUSER} -h ${cli_config_1.default.PGHOST} -d ${cli_config_1.default.PGDATABASE} -f "${detachSqlFile}"`;
    (0, child_process_1.execSync)(detachCommand, {
        env: {
            ...process.env,
            PGPASSFILE: cli_config_1.default.PGPASSFILE,
        },
    });
    fs_1.default.unlinkSync(detachSqlFile);
}
function dbBackup(schemaName) {
    const pathToBackupDir = path_1.default.join(app_config_1.appConfig.db.main.backup.backupDirectory, schemaName);
    fs_1.default.mkdirSync(pathToBackupDir, { recursive: true });
    let mainDbBackupCommand = prepareMainDbBackupCommand(schemaName, pathToBackupDir);
    (0, child_process_1.execSync)(mainDbBackupCommand, {
        env: {
            ...process.env,
            PGPASSFILE: cli_config_1.default.PGPASSFILE,
        },
    });
    app_logs_1.logger.info("Db backed up successfully!");
    app_logs_1.logger.info("Removing older backups. Keeps last main " + app_config_1.appConfig.db.main.backup.keepLastBackupCount + "!");
    let filesInMainBackupDestPath = fs_1.default.readdirSync(pathToBackupDir).sort();
    app_logs_1.logger.info("Files at " + pathToBackupDir + ": " + filesInMainBackupDestPath.join(","));
    let mainDbFilesShouldBeKept = [];
    let filesShouldBeRemoved = [];
    if (filesInMainBackupDestPath.length > 0) {
        for (let index = filesInMainBackupDestPath.length - 1; index >= 0; index--) {
            let fileName = filesInMainBackupDestPath[index];
            if (fileName.indexOf(mainDbBackupFilePrefix) === 0) {
                if (mainDbFilesShouldBeKept.length >= app_config_1.appConfig.db.main.backup.keepLastBackupCount) {
                    filesShouldBeRemoved.push(fileName);
                }
                else {
                    mainDbFilesShouldBeKept.push(fileName);
                }
            }
        }
    }
    app_logs_1.logger.info("Main db backup files are kept: %s", mainDbFilesShouldBeKept.join(","));
    for (const fileName of filesShouldBeRemoved) {
        try {
            fs_1.default.unlinkSync(path_1.default.join(pathToBackupDir, fileName));
        }
        catch (err) {
            app_logs_1.logger.error("Armon Maintainance service cannot find backup file to remove");
            app_logs_1.logger.error(err);
        }
    }
    app_logs_1.logger.info("Expired backup files are removed: %s", filesShouldBeRemoved.join(","));
}
function prepareMainDbBackupCommand(schemaName, folderPath) {
    let outFile = path_1.default.join(folderPath, mainDbBackupFilePrefix + moment().format("YYYYMMDDHHmmss") + ".backup");
    return `pg_dump -U ${cli_config_1.default.PGSUPERUSER} -h ${cli_config_1.default.PGHOST} --schema "${schemaName}"  --format=c -f "${outFile}" ${cli_config_1.default.PGDATABASE}`;
}
function generatePartitionsToKeepForLastMonths(monthCount) {
    let mt = moment();
    let paramtItems = {};
    let month, year;
    for (let i = 0; i < monthCount; i++) {
        year = mt.format("YYYY");
        month = mt.format("MM");
        if (paramtItems[year]) {
            paramtItems[year].push(month);
        }
        else {
            paramtItems[year] = [month];
        }
        mt = mt.add(-1, "M");
    }
    let paramtYears = Object.keys(paramtItems);
    let paramtYearItems = [];
    for (const paramtYear of paramtYears) {
        for (const monthOfYear of paramtItems[paramtYear]) {
            paramtYearItems.push(paramtYear + monthOfYear);
        }
    }
    return paramtYearItems;
}
