"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PdfReportFirstInLastOut = void 0;
const sharp_1 = __importDefault(require("sharp"));
const fs_1 = __importDefault(require("fs"));
const path_1 = __importDefault(require("path"));
const uuid_1 = __importDefault(require("uuid"));
const pdfmake_1 = __importDefault(require("pdfmake/build/pdfmake"));
require("pdfmake/build/vfs_fonts");
const app_config_1 = require("../../../app.config");
const moment_1 = __importDefault(require("moment"));
const report_util_1 = require("../report.util");
const i18n_1 = __importDefault(require("i18n"));
const dal_manager_1 = require("../../../dal/dal.manager");
const business_pacs_report_1 = require("../../pacs/business.pacs.report");
const app_logs_1 = require("../../../app.logs");
const app_enums_1 = require("../../../app.enums");
const worker_threads_1 = require("worker_threads");
const generator_1 = require("../generator");
class PdfReportFirstInLastOut extends generator_1.ReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._options = this._request.filter;
    }
    async generateReport() {
        this.organization = await dal_manager_1.dbManager.accessOrganization.getOrganizationLogoAndName(this._request.organizationId);
        this._data = await (0, business_pacs_report_1.getDailySummaryReport)(this._request.organizationId, {
            date: this._options.date,
            organizationUnitIds: this._options.organizationUnitIds,
            applyOrganizationUnitFilterHierarchically: this._options.applyOrganizationUnitFilterHierarchically,
            userGroupIds: this._options.userGroupIds,
            userIds: this._options.userIds,
            workPlanIds: this._options.workPlanIds,
            status: this._options.status,
            sortingMethod: this._options.sortingMethod,
            workStatusFilter: this._options.workStatusFilter,
            pagination: {
                take: app_config_1.appConfig.reportExportRowLimit + 1,
                skip: 0,
            },
            hasOrganizationWide: this._options.hasOrganizationWide,
            permittedUnitIds: this._options.permittedUnitIds,
        }, this._request.requesterUserId, this._options.hasOrganizationWide, this._options.permittedUnitIds);
        if ((!this._data.items || this._data.items.length === 0) && this._isScheduled && this._reportTemplateId)
            return;
        let rows = [];
        let locale = this._locale;
        let employee = i18n_1.default.__({ phrase: "PDF-REPORT.employee", locale });
        let status = i18n_1.default.__({ phrase: "PDF-REPORT.status", locale });
        let title = i18n_1.default.__({ phrase: "EXCEL-REPORT.EXPECTED_WORKPLAN", locale });
        rows.push([
            { text: employee, style: "tableHeader" },
            { text: "Durum", style: "tableHeader" },
            { text: status, style: "tableHeader" },
            { text: title, style: "tableHeader" },
        ]);
        this._data.items.forEach((result) => {
            let rowObj = [];
            let entrancePenaltyTimeInMinutes;
            let entranceDateTimeText;
            let exitPenaltyTimeInMinutes;
            let dateText;
            let hasPermissionNow;
            let permissionNowType;
            let userId;
            let exitDateTimeHourText;
            let exitDateTimeText;
            let entranceDateTimeHourText;
            let statusText;
            let statusColor;
            let startOfWorkUtcHourText;
            let endOfWorkUtcHourText;
            let startOfWorkUtcText;
            let endOfWorkUtcText;
            if (result.entrancePenaltyTimeInSeconds && result.entrancePenaltyTimeInSeconds != 0) {
                entrancePenaltyTimeInMinutes = Math.round(result.entrancePenaltyTimeInSeconds / 60);
            }
            if (result.exitPenaltyTimeInSeconds && result.exitPenaltyTimeInSeconds != 0) {
                exitPenaltyTimeInMinutes = Math.round(result.exitPenaltyTimeInSeconds / 60);
            }
            let hasPermissionInDay = result.ppermissions && result.ppermissions.length > 0;
            if (hasPermissionInDay) {
                result.ppermissions.forEach((permission) => {
                    if ((0, moment_1.default)().isBetween(permission.startUtc, permission.endUtc)) {
                        hasPermissionNow = true;
                        permissionNowType = permission.ppermissionTypeName;
                    }
                });
            }
            if (result.user || result.userCaptions) {
                if (result.user)
                    userId = result.user.id;
                if (result.exitDateTime) {
                    exitDateTimeHourText = (0, moment_1.default)(result.exitDateTime).format("HH:mm");
                    exitDateTimeText = (0, moment_1.default)(result.exitDateTime).format("LLLL:ss");
                }
                if (result.entranceDateTime) {
                    entranceDateTimeHourText = (0, moment_1.default)(result.entranceDateTime).format("HH:mm");
                    entranceDateTimeText = (0, moment_1.default)(result.entranceDateTime).format("LLLL:ss");
                    statusText = i18n_1.default.__({ phrase: "PDF-REPORT.arrived", locale });
                    if (entrancePenaltyTimeInMinutes > 0) {
                        if (entrancePenaltyTimeInMinutes > result.entranceTolerance) {
                            statusColor = "#c80202";
                        }
                        else {
                            let percentageLate = Math.round((entrancePenaltyTimeInMinutes * 100) / result.entranceTolerance);
                            statusColor = this.getPercentageColor(percentageLate);
                        }
                    }
                    else {
                        statusColor = "#62A325";
                    }
                }
                else {
                    statusText = i18n_1.default.__({ phrase: "PDF-REPORT.not_arrived", locale });
                    statusColor = "#c80202";
                }
            }
            let workplan = result.workPlan;
            if (workplan.startOfWorkUtc) {
                startOfWorkUtcHourText = (0, moment_1.default)(workplan.startOfWorkUtc).format("HH:mm");
                startOfWorkUtcText = (0, moment_1.default)(workplan.startOfWorkUtc).format("LLLL");
            }
            if (result.workPlan.endOfWorkUtc) {
                endOfWorkUtcHourText = (0, moment_1.default)(workplan.endOfWorkUtc).format("HH:mm");
                endOfWorkUtcText = (0, moment_1.default)(workplan.endOfWorkUtc).format("LLLL");
            }
            statusText += "\n";
            rowObj.push((0, report_util_1.parseCaption)(result.userCaptions));
            let arrSecondCol = [];
            arrSecondCol.push({ text: statusText, fontSize: 10, color: statusColor, margin: [0, 15] });
            if (entrancePenaltyTimeInMinutes) {
                arrSecondCol.push({ text: i18n_1.default.__({ phrase: "PDF-REPORT.late_arrival", locale }) + ": ", style: "status" });
                arrSecondCol.push({ text: (0, report_util_1.formatTime)(entrancePenaltyTimeInMinutes, locale) + "\n", style: "status" });
            }
            if (exitPenaltyTimeInMinutes) {
                arrSecondCol.push({ text: i18n_1.default.__({ phrase: "PDF-REPORT.early_left", locale }) + ": ", style: "status" });
                arrSecondCol.push({ text: (0, report_util_1.formatTime)(exitPenaltyTimeInMinutes, locale) + "\n", style: "status" });
            }
            if (result.missingWorkInMinutes) {
                arrSecondCol.push({ text: i18n_1.default.__({ phrase: "PDF-REPORT.missing_work", locale }) + ": ", style: "status" });
                arrSecondCol.push({ text: (0, report_util_1.formatTime)(result.missingWorkInMinutes, locale) + "\n", style: "status" });
            }
            if (result.extraWorkInMinutes) {
                arrSecondCol.push({ text: i18n_1.default.__({ phrase: "PDF-REPORT.extra_working", locale }) + ": ", style: "status" });
                arrSecondCol.push({ text: (0, report_util_1.formatTime)(result.extraWorkInMinutes, locale) + "\n", style: "status" });
            }
            if (hasPermissionInDay) {
                if (hasPermissionNow) {
                    arrSecondCol.push({
                        text: [
                            { text: i18n_1.default.__({ phrase: "PDF-REPORT.day_off", locale }) + "\n", style: "status" },
                            { text: permissionNowType, style: "status" },
                        ],
                    });
                }
                else if (hasPermissionInDay) {
                    result.ppermissions.forEach((permission) => {
                        permission.ppermissionTypeName += "\n";
                        let interval = (0, moment_1.default)(permission.startUtc).locale(locale).format("DD/MM/YYYY HH:mm") + " - " + (0, moment_1.default)(permission.endUtc).locale(locale).format("DD/MM/YYYY HH:mm");
                        arrSecondCol.push({
                            text: [
                                { text: permission.ppermissionTypeName, style: "status" },
                                { text: interval, fontSize: 10, color: "black", margin: [0, 15] },
                            ],
                        });
                    });
                }
            }
            rowObj.push({
                text: arrSecondCol,
            });
            let arrThirdCol = [];
            if (result.entranceDateTime) {
                arrThirdCol.push({ text: entranceDateTimeHourText + "-", fontSize: 10, color: "black" });
            }
            else {
                arrThirdCol.push({ text: i18n_1.default.__({ phrase: "PDF-REPORT.no_record", locale }) + "-", fontSize: 10, color: "gray" });
            }
            if (result.exitDateTime) {
                arrThirdCol.push({ text: exitDateTimeHourText + "\n", fontSize: 10, color: "black" });
            }
            else {
                arrThirdCol.push({ text: i18n_1.default.__({ phrase: "PDF-REPORT.no_record", locale }) + "\n", fontSize: 10, color: "gray" });
            }
            if (result.warnings && result.warnings.length > 0) {
                result.warnings.forEach((warning) => {
                    if (warning == 1) {
                        arrThirdCol.push({ text: i18n_1.default.__({ phrase: "PDF-REPORT.first_in_last_out_warning_1", locale }), fontSize: 10, color: "#2680a1" });
                    }
                    else if (warning == 2) {
                        arrThirdCol.push({ text: i18n_1.default.__({ phrase: "PDF-REPORT.first_in_last_out_warning_2", locale }), fontSize: 10, color: "#2680a1" });
                    }
                });
            }
            rowObj.push({
                text: arrThirdCol,
            });
            let arrFourthCol = [];
            if (result.workPlan) {
                arrFourthCol.push({ text: result.workPlan.name + "\n", fontSize: 10, color: "gray" });
                if (startOfWorkUtcText && endOfWorkUtcText && startOfWorkUtcText != "Invalid date" && endOfWorkUtcText != "Invalid date") {
                    arrFourthCol.push({
                        text: i18n_1.default.__({ phrase: "PDF-REPORT.expected_workplan", locale }) +
                            ":" +
                            (0, moment_1.default)(new Date(startOfWorkUtcText)).locale(locale).format("DD/MM/YYYY HH:mm") +
                            "-" +
                            (0, moment_1.default)(new Date(endOfWorkUtcText)).locale(locale).format("DD/MM/YYYY HH:mm"),
                        fontSize: 10,
                        color: "gray",
                    });
                }
            }
            else {
                arrFourthCol.push({ text: "" });
            }
            rowObj.push({
                text: arrFourthCol,
            });
            rows.push(rowObj);
        });
        let header;
        if (this.organization == null) {
            header = {
                table: {
                    widths: ["*"],
                    headerRows: 1,
                    body: [
                        [
                            { text: i18n_1.default.__({ phrase: "PDF-REPORT.first_in_last_out_report", locale }) + "\n", style: "header" },
                            { text: (0, moment_1.default)(this._options.date).locale(locale).format("DD/MM/YYYY"), style: "date" },
                        ],
                    ],
                },
                layout: "noBorders",
            };
        }
        if (this.organization.logo == null) {
            header = {
                table: {
                    widths: [170, "*"],
                    headerRows: 1,
                    body: [
                        [
                            { text: this.organization.name, style: "organizationName" },
                            [
                                { text: i18n_1.default.__({ phrase: "PDF-REPORT.first_in_last_out_report", locale }) + "\n", style: "header" },
                                { text: (0, moment_1.default)(this._options.date).locale(locale).format("DD/MM/YYYY"), style: "date" },
                            ],
                        ],
                    ],
                },
                layout: "noBorders",
            };
        }
        else {
            let logoImage = await (0, sharp_1.default)(this.organization.logo, { failOnError: false }).resize(240, 60).toFormat("png").toBuffer();
            let logoNew = (await (0, sharp_1.default)({
                create: {
                    width: 240,
                    height: 60,
                    channels: 4,
                    background: { r: 1, g: 50, b: 91, alpha: 0.9 },
                },
                failOnError: false,
            })
                .png()
                .composite([{ input: logoImage }])
                .toBuffer()).toString("base64");
            header = {
                table: {
                    widths: [170, "*"],
                    headerRows: 1,
                    body: [
                        [
                            {
                                image: "data:image/png;base64," + logoNew,
                                width: report_util_1.ReportConstants.widthOrganizationLogo,
                                alignment: "left",
                            },
                            [
                                { text: i18n_1.default.__({ phrase: "PDF-REPORT.first_in_last_out_report", locale }) + "\n", style: "header" },
                                { text: (0, moment_1.default)(this._options.date).locale(locale).format("DD/MM/YYYY"), style: "date" },
                            ],
                        ],
                    ],
                },
                layout: "noBorders",
            };
        }
        this._dd = {
            footer: ((currentPage, pageCount) => {
                return {
                    columns: [
                        {
                            text: i18n_1.default.__({ phrase: "PDF-REPORT.report_create_date", locale }) + ": " + (0, moment_1.default)().locale(locale).format("DD/MM/YYYY HH:mm") + " ",
                            alignment: "left",
                            fontSize: "10",
                            margin: [40, 0],
                        },
                        { text: currentPage.toString() + " / " + pageCount, alignment: "right", fontSize: "10", margin: [40, 0] },
                    ],
                };
            }),
            content: [
                header,
                {
                    style: "tableExample",
                    layout: {
                        fillColor: function (rowIndex, node, columnIndex) {
                            return rowIndex % 2 === 0 ? report_util_1.ReportConstants.evenRowColor : null;
                        },
                    },
                    table: {
                        headerRows: 1,
                        dontBreakRows: true,
                        keepWithHeaderRows: 1,
                        widths: ["*", "*", "auto", "auto"],
                        body: rows,
                    },
                },
            ],
            styles: {
                header: {
                    fontSize: 18,
                    bold: true,
                    margin: [0, 0, 0, 10],
                    alignment: "left",
                },
                subheader: {
                    fontSize: 16,
                    bold: true,
                    margin: [0, 10, 0, 5],
                },
                tableExample: {
                    margin: [0, 5, 0, 15],
                },
                tableHeader: {
                    bold: true,
                    fontSize: 10,
                    color: "black",
                },
                status: {
                    margin: [0, 15],
                    fontSize: 10,
                    color: "gray",
                },
                center: {
                    alignment: "center",
                },
                date: {
                    fontSize: 11,
                    color: "black",
                    alignment: "right",
                },
                organizationName: {
                    fontSize: 11,
                    bold: true,
                    alignment: "left",
                },
            },
            defaultStyle: {},
        };
        this._fileId = uuid_1.default.v4();
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".pdf");
        let pdf = pdfmake_1.default.createPdf(this._dd);
        await new Promise((resolve, reject) => {
            pdf.getBuffer((result) => {
                fs_1.default.writeFileSync(this._filePath, result);
                resolve(this._filePath);
            });
        });
        if (!worker_threads_1.isMainThread) {
            app_logs_1.logger.debug("[Worker Thread] Pdf report is ready -> " + this._filePath);
        }
        return {
            format: app_enums_1.enums.ReportFormat.Excel,
            preparedReportId: this._fileId,
            filePath: this._filePath,
            notificationType: undefined,
            reportType: app_enums_1.enums.ReportCode.PacsFirstInLastOut,
            sendFileInAttachments: undefined,
        };
    }
    getPercentageColor(percentage) {
        if (percentage > 75) {
            return "#c84b02";
        }
        else if (percentage > 50) {
            return "#c87f02";
        }
        else if (percentage > 25) {
            return "#c8ae02";
        }
        else {
            return "#b8c802";
        }
    }
}
exports.PdfReportFirstInLastOut = PdfReportFirstInLastOut;
