"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PdfReportAccessControlPoints = void 0;
const fs_1 = __importDefault(require("fs"));
const path_1 = __importDefault(require("path"));
const uuid_1 = __importDefault(require("uuid"));
const pdfmake_1 = __importDefault(require("pdfmake/build/pdfmake"));
require("pdfmake/build/vfs_fonts");
const app_config_1 = require("../../../app.config");
const moment_1 = __importDefault(require("moment"));
const report_util_1 = require("../report.util");
const i18n_1 = __importDefault(require("i18n"));
const app_enums_1 = require("../../../app.enums");
const dal_manager_1 = require("../../../dal/dal.manager");
const app_logs_1 = require("../../../app.logs");
const worker_threads_1 = require("worker_threads");
const generator_1 = require("../generator");
class PdfReportAccessControlPoints extends generator_1.ReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._options = request.filter;
    }
    async generateReport() {
        let pdfRows = [];
        const organizationId = this._request.organizationId;
        pdfRows.push(this.createColumns(this._locale));
        let fetched = 0;
        const filter = this._options.filter;
        const filters = {
            filter,
        };
        this._organizationName = (await dal_manager_1.dbManager.accessRedisCache.getOrganizationCache({ organizationId: organizationId })).n;
        let accessControlPointInfos = await dal_manager_1.dbManager.accessAccessControlPoint.searchAccessControlPoint(organizationId, filters, true, this._request.requesterUserId);
        fetched += accessControlPointInfos.items?.length;
        if (fetched < app_config_1.appConfig.reportExportRowLimit + 1) {
            this.createRows(accessControlPointInfos, pdfRows);
        }
        const resFilePath = await this.createReportTemplate(pdfRows);
        if (!worker_threads_1.isMainThread) {
            app_logs_1.logger.debug("[Worker Thread] Pdf report is ready -> " + this._filePath);
        }
        return {
            preparedReportId: this._fileId,
            filePath: resFilePath,
            format: app_enums_1.enums.ReportFormat.Pdf,
            notificationType: undefined,
            reportType: app_enums_1.enums.ReportCode.AccessLogs,
            sendFileInAttachments: undefined,
            reportContainsEmptyData: this._request.reportTemplateId ? fetched === 0 : undefined,
        };
    }
    async createReportTemplate(rows) {
        await this.createDocumentDefinitionOfPdf(this._locale, rows);
        let res = await this.createPdf();
        return res;
    }
    createPdf() {
        this._fileId = uuid_1.default.v4();
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".pdf");
        let pdf = pdfmake_1.default.createPdf(this._dd);
        return new Promise((resolve, reject) => {
            pdf.getBuffer((result) => {
                fs_1.default.writeFileSync(this._filePath, result);
                app_logs_1.logger.debug("Report generated:" + this._filePath);
                resolve(this._filePath);
            });
        });
    }
    createColumns(locale) {
        let acpName = i18n_1.default.__({ phrase: "PDF-REPORT.ACCESS_CONTROL_POINT_NAME", locale });
        let acpLocation = i18n_1.default.__({ phrase: "PDF-REPORT.LOCATION", locale });
        let acpType = i18n_1.default.__({ phrase: "PDF-REPORT.ACCESS_CONTROL_POINT_TYPE", locale });
        return [
            { text: acpName, style: "tableHeader" },
            { text: acpLocation, style: "tableHeader" },
            { text: acpType, style: "tableHeader" },
        ];
    }
    async createDocumentDefinitionOfPdf(locale, rows) {
        let organization;
        organization = { text: this._organizationName, style: "organizationName" };
        this._dd = {
            footer: ((currentPage, pageCount) => {
                return {
                    columns: [
                        {
                            text: i18n_1.default.__({ phrase: "PDF-REPORT.report_create_date", locale }) + ": " + (0, moment_1.default)().locale(locale).format("DD/MM/YYYY HH:mm") + " ",
                            alignment: "left",
                            fontSize: "10",
                            margin: [40, 0],
                        },
                        { text: currentPage.toString() + " / " + pageCount, alignment: "right", fontSize: "10", margin: [40, 0] },
                    ],
                };
            }),
            content: [
                organization,
                { text: i18n_1.default.__({ phrase: "PDF-REPORT.ACCESS_CONTROL_POINTS", locale }) + "\n", style: "header" },
                {
                    text: { text: this._options.filter ? i18n_1.default.__({ phrase: "COMMON.SEARCH_HEADER", locale }) + " : " + this._options.filter : "", style: "filter" },
                },
                {
                    style: "tableStyle",
                    layout: {
                        fillColor: function (rowIndex, node, columnIndex) {
                            return rowIndex % 2 === 0 ? report_util_1.ReportConstants.evenRowColor : null;
                        },
                    },
                    table: {
                        headerRows: 1,
                        dontBreakRows: true,
                        keepWithHeaderRows: 1,
                        widths: ["*", "*", "*"],
                        body: rows,
                    },
                },
            ],
            styles: {
                header: {
                    fontSize: 18,
                    bold: true,
                    margin: [0, 0, 0, 10],
                    alignment: "center",
                },
                tableStyle: {
                    margin: [0, 5, 0, 15],
                },
                tableHeader: {
                    bold: true,
                    fontSize: 10,
                    color: "black",
                },
                status: {
                    margin: [0, 15],
                    fontSize: 10,
                    color: "gray",
                },
                date: {
                    fontSize: 11,
                    color: "black",
                    alignment: "right",
                },
                organizationName: {
                    fontSize: 11,
                    bold: true,
                    alignment: "left",
                },
                filter: {
                    fontSize: 10,
                    color: "black",
                    alignment: "left",
                },
                filterBold: {
                    fontSize: 10,
                    color: "black",
                    bold: true,
                    alignment: "left",
                },
            },
            defaultStyle: {},
        };
    }
    createRows(acpInfos, pdfRows) {
        acpInfos.items.forEach((acpInf) => {
            let rowObj = [];
            rowObj.push({
                text: acpInf.name,
            });
            rowObj.push({
                text: acpInf.location,
            });
            rowObj.push({
                text: (0, report_util_1.getAccessControlPointTypeName)(acpInf.type, this._locale),
            });
            pdfRows.push(rowObj);
        });
    }
}
exports.PdfReportAccessControlPoints = PdfReportAccessControlPoints;
