"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.VisitorsExcelReport = void 0;
const exceljs_1 = __importDefault(require("exceljs"));
const fs_1 = __importDefault(require("fs"));
const i18n_1 = __importDefault(require("i18n"));
const moment_1 = __importDefault(require("moment"));
const path_1 = __importDefault(require("path"));
const uuid_1 = __importDefault(require("uuid"));
const app_config_1 = require("../../../app.config");
const app_enums_1 = require("../../../app.enums");
const app_logs_1 = require("../../../app.logs");
const dal_constants_1 = require("../../../dal/dal.constants");
const dal_manager_1 = require("../../../dal/dal.manager");
const business_report_export_1 = require("../business.report.export");
const generator_1 = require("../generator");
const report_util_1 = require("../report.util");
class VisitorsExcelReport extends generator_1.ReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._fileId = uuid_1.default.v4();
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".xlsx");
        if (fs_1.default.existsSync(this._filePath)) {
            fs_1.default.unlinkSync(this._filePath);
        }
        this._wb = new exceljs_1.default.stream.xlsx.WorkbookWriter({
            filename: this._filePath,
            useStyles: true,
            useSharedStrings: true,
        });
        this._options = request.filter;
    }
    async generateReport() {
        this._organizationName = (await dal_manager_1.dbManager.accessOrganization.getOrganizationBasic(this._request.organizationId)).name;
        app_logs_1.logger.debug("Organization Name : " + this._organizationName);
        let { caption: requestedUserCaption } = await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId: this._request.organizationId, userId: this._request.requesterUserId });
        if (!requestedUserCaption) {
            throw new Error("Not found");
        }
        this._requestedUserCaption = requestedUserCaption[0].text[0];
        this._wb.creator = this._requestedUserCaption;
        app_logs_1.logger.debug("Generating visitors excel report...");
        app_logs_1.logger.debug("Requester User : " + this._requestedUserCaption);
        this._wsGeneral = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.GENERAL", locale: this._locale }), {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsGeneral.properties.defaultRowHeight = 15;
        this._generalRowIndex = 10;
        this._data = await (0, business_report_export_1.generateVisitorListReportData)(this._request.organizationId, this._request.requesterUserId, this._options);
        this.columnFields = this._data.organizationVisitorModuleSettings.visitorProfileFormFields.filter((item) => item.reportable);
        this.initGeneralSheet(this._wsGeneral, 9, 1);
        const filters = [];
        filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.ONLY_BLACKLIST", locale: this._locale }) + " : " + (this._options.showOnlyBlacklist ? "Evet" : "Hayır"));
        filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.ONLY_TEMPORARY_PERMITTED", locale: this._locale }) + " : " + (this._options.showOnlyTemporaryPermitted ? "Evet" : "Hayır"));
        this._options.visitorProfileFilterFields.forEach((field, index) => {
            let fieldName = this._data.organizationVisitorModuleSettings.visitorProfileFormFields.find((f) => f.name == field.name).caption;
            if (fieldName) {
                let valueOfCell;
                let fieldType = this._data.organizationVisitorModuleSettings.visitorProfileFormFields.find((f) => f.name == field.name).type;
                let value = this._options.visitorProfileFilterFields.find((f) => f.name == field.name).value;
                if (fieldType == dal_constants_1.DalConstants.FormFieldType.Checkbox) {
                    if (value) {
                        if (value == true) {
                            valueOfCell = "Evet";
                        }
                        else if (value == false) {
                            valueOfCell = "Hayır";
                        }
                    }
                }
                else if (fieldType == dal_constants_1.DalConstants.FormFieldType.Date || fieldType == dal_constants_1.DalConstants.FormFieldType.DateTime) {
                    if (value) {
                        valueOfCell = (0, moment_1.default)(value).locale(this._locale).format("DD/MM/YYYY HH:mm");
                    }
                }
                else {
                    if (value)
                        valueOfCell = this._options.visitorProfileFilterFields.find((f) => f.name == field.name).value;
                    else
                        valueOfCell = "";
                }
                filters.push(fieldName + " : " + valueOfCell);
            }
        });
        app_logs_1.logger.debug("\n\nApplied Filters:\n");
        app_logs_1.logger.debug(filters.join("\r\n"));
        app_logs_1.logger.debug("\n\n");
        this.addFilterToSheet(this._wsGeneral, 1, 1, filters.join("\r\n"));
        for (const item of this._data.items) {
            this.addGeneralRow(item);
        }
        this._wsGeneral.pageSetup.printArea = "A1:" + this._wsGeneral.getColumn(this._wsGeneral.columns.length).letter + this._generalRowIndex;
        this._wsGeneral.pageSetup.printTitlesRow = "5:5";
        this._wsGeneral.commit();
        await this._wb.commit();
        app_logs_1.logger.debug("File is ready -> " + this._filePath);
        return {
            format: app_enums_1.enums.ReportFormat.Excel,
            preparedReportId: this._fileId,
            filePath: this._filePath,
            notificationType: undefined,
            reportType: app_enums_1.enums.ReportCode.VisitorReport,
            sendFileInAttachments: undefined,
        };
    }
    initGeneralSheet(wsRaw, startRowIndex, startColIndex) {
        let columns = [];
        let data = [];
        if (this.columnFields) {
            this.columnFields.forEach((form, index) => {
                data.push({
                    value: form.caption,
                    cellWidth: 30
                });
            });
        }
        data = [
            ...data,
            {
                phrase: "EXCEL-REPORT.TEMP_START_DATE",
                cellWidth: 30
            },
            {
                phrase: "EXCEL-REPORT.TEMP_END_DATE",
                cellWidth: 30
            },
            {
                phrase: "EXCEL-REPORT.TEMP_NOTE",
                cellWidth: 30
            },
            {
                phrase: "EXCEL-REPORT.BLACKLIST_START_DATE",
                cellWidth: 30
            },
            {
                phrase: "EXCEL-REPORT.BLACKLIST_END_DATE",
                cellWidth: 30
            },
            {
                phrase: "EXCEL-REPORT.BLACKLIST_NOTE",
                cellWidth: 30
            }
        ];
        startColIndex = this.addInitCellsToRow(data, columns, startColIndex, startRowIndex, wsRaw);
        wsRaw.autoFilter = {
            from: "A" + startRowIndex,
            to: wsRaw.getColumn(columns.length).letter + startRowIndex,
        };
        wsRaw.columns = columns;
    }
    addGeneralRow(record) {
        let colIndex = 1;
        let rowData = [];
        let tempStartDate = "";
        let tempEndDate = "";
        let tempNote = "";
        let blacklistStartDate = "";
        let blacklistEndDate = "";
        let blacklistNote = "";
        this.columnFields.forEach((column) => {
            let isBlacklisted;
            let isBlacklistedFuture;
            let isTempPermitted;
            let isTempPermittedFuture;
            if (record.states) {
                let blacklistState = record.states.find((state) => state.state == dal_constants_1.DalConstants.OrganizationVisitorStates.Forbidden);
                if (blacklistState) {
                    isBlacklisted = true;
                    if (blacklistState.startUtc)
                        blacklistStartDate = (0, report_util_1.formatDateForExcel)(blacklistState.startUtc);
                    if (blacklistState.endUtc)
                        blacklistEndDate = (0, report_util_1.formatDateForExcel)(blacklistState.endUtc);
                    if (blacklistState.note)
                        blacklistNote = blacklistState.note;
                    if ((0, moment_1.default)().isBefore(blacklistState.startUtc)) {
                        isBlacklistedFuture = true;
                    }
                }
                let tempPermitState = record.states.find((state) => state.state == dal_constants_1.DalConstants.OrganizationVisitorStates.TemporaryPermit);
                if (tempPermitState) {
                    isTempPermitted = true;
                    if (tempPermitState.startUtc)
                        tempStartDate = (0, report_util_1.formatDateForExcel)(tempPermitState.startUtc);
                    if (tempPermitState.endUtc)
                        tempEndDate = (0, report_util_1.formatDateForExcel)(tempPermitState.endUtc);
                    if (tempPermitState.note)
                        tempNote = tempPermitState.note;
                    if ((0, moment_1.default)().isBefore(tempPermitState.startUtc)) {
                        isTempPermittedFuture = true;
                    }
                }
            }
            let result;
            if (column.name == "name") {
                if (!record.fields[column.name])
                    result = " ";
                else
                    result = record.fields[column.name];
                if (isBlacklisted) {
                    result += "\r\n" + i18n_1.default.__({ phrase: "EXCEL-REPORT.FORBIDDEN", locale: this._locale });
                }
                if (isBlacklistedFuture) {
                    result += "\r\n" + i18n_1.default.__({ phrase: "EXCEL-REPORT.STATE_FUTURE_DATE", locale: this._locale });
                }
                if (isTempPermitted) {
                    result += "\r\n" + i18n_1.default.__({ phrase: "EXCEL-REPORT.TEMPORARY_PERMITTED_USER", locale: this._locale });
                }
                if (isTempPermittedFuture) {
                    result += "\r\n" + i18n_1.default.__({ phrase: "EXCEL-REPORT.STATE_FUTURE_DATE", locale: this._locale });
                }
            }
            else if (column.name != "name") {
                if (!record.fields[column.name])
                    result = " ";
                else {
                    if (column.type == dal_constants_1.DalConstants.FormFieldType.Checkbox) {
                        if (record.fields[column.name] == "true") {
                            result = "Evet";
                        }
                        else if (record.fields[column.name] == "false") {
                            result = "Hayır";
                        }
                    }
                    else if (column.type == dal_constants_1.DalConstants.FormFieldType.Date) {
                        result = (0, report_util_1.formatDateForExcel)(record.fields[column.name]);
                    }
                    else
                        result = record.fields[column.name];
                }
            }
            rowData.push(result ?? "");
        });
        rowData.push(tempStartDate);
        rowData.push(tempEndDate);
        rowData.push(tempNote);
        rowData.push(blacklistStartDate);
        rowData.push(blacklistEndDate);
        rowData.push(blacklistNote);
        colIndex = this.addCellsToRow(rowData, colIndex, this._generalRowIndex, this._wsGeneral, generator_1.ReportGenerator.Constants.Styling.AllThin);
        this._wsGeneral.getRow(this._generalRowIndex).commit();
        this._generalRowIndex++;
    }
    addFilterToSheet(ws, startRowIndex, startColIndex, filter) {
        this.addFilterCell11(ws, startRowIndex, startColIndex, this._organizationName + " " + i18n_1.default.__({ phrase: "EXCEL-REPORT.VISITORS", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin, {
            bold: true,
        });
        ws.mergeCells(startRowIndex, startColIndex, startRowIndex, startColIndex + 3);
        this.addFilterCell11(ws, startRowIndex + 1, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.FILE_CREATION_DATE", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        this.addFilterCell11(ws, startRowIndex + 1, startColIndex + 1, (0, moment_1.default)().locale(this._locale).format("LL") + " " + (0, moment_1.default)().locale(this._locale).format("HH:mm:ss"), generator_1.ReportGenerator.Constants.Styling.AllThin, { italic: false }, generator_1.ReportGenerator.Constants.DateFormat);
        this.addFilterCell11(ws, startRowIndex + 2, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.REQUESTED_USER", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        this.addFilterCell11(ws, startRowIndex + 2, startColIndex + 1, this._requestedUserCaption, generator_1.ReportGenerator.Constants.Styling.AllThin);
        ws.mergeCells(startRowIndex + 1, startColIndex + 1, startRowIndex + 1, startColIndex + 3);
        ws.mergeCells(startRowIndex + 2, startColIndex + 1, startRowIndex + 2, startColIndex + 3);
        ws.mergeCells(startRowIndex + 3, startColIndex + 1, startRowIndex + 3, startColIndex + 3);
        this.addFilterCell11(ws, startRowIndex + 3, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.APPLIED_FILTERS", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        this.addFilterCell11(ws, startRowIndex + 3, startColIndex + 1, filter, generator_1.ReportGenerator.Constants.Styling.AllThin);
        let filterlength = filter.split("\r\n").length;
        let row = ws.getRow(startRowIndex + 3);
        row.height = filterlength * 25;
    }
}
exports.VisitorsExcelReport = VisitorsExcelReport;
