"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.VisitorCountReport = void 0;
const exceljs_1 = __importDefault(require("exceljs"));
const fs_1 = __importDefault(require("fs"));
const i18n_1 = __importDefault(require("i18n"));
const moment_1 = __importDefault(require("moment"));
const path_1 = __importDefault(require("path"));
const uuid_1 = __importDefault(require("uuid"));
const app_config_1 = require("../../../app.config");
const app_enums_1 = require("../../../app.enums");
const app_logs_1 = require("../../../app.logs");
const dal_constants_1 = require("../../../dal/dal.constants");
const dal_manager_1 = require("../../../dal/dal.manager");
const generator_1 = require("../generator");
class VisitorCountReport extends generator_1.ReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._fileId = uuid_1.default.v4();
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".xlsx");
        if (fs_1.default.existsSync(this._filePath)) {
            fs_1.default.unlinkSync(this._filePath);
        }
        this._wb = new exceljs_1.default.stream.xlsx.WorkbookWriter({
            filename: this._filePath,
            useStyles: true,
            useSharedStrings: true,
        });
        this._currentDate = (0, moment_1.default)();
        this._options = request.filter;
    }
    async getFilter(filter) {
        switch (this._options.filterType) {
            case dal_constants_1.DalConstants.VisitAnalysisVisitCountGenericFilterType.ByVisitorId:
                const query = `
                SELECT name FROM "${this._request.organizationId}"."organizationVisitorProfiles" as orgVP
                WHERE orgVP."id" = $1;
                `;
                const { rows } = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
                    return await trx.query(query, [filter]);
                }, this._request.requesterUserId, this._request.organizationId);
                if (rows[0]) {
                    return Promise.resolve(rows[0]);
                }
                else {
                    return Promise.resolve("");
                }
            case dal_constants_1.DalConstants.VisitAnalysisVisitCountGenericFilterType.ByVisitedPersonId:
                return Promise.resolve((await dal_manager_1.dbManager.accessUser.getBasicUserInfo(this._request.organizationId, filter)).fullname);
            default:
                return "";
        }
    }
    findVisitReason(value) {
        let out = "";
        if (!this._visitReasons) {
            return out;
        }
        for (const element of this._visitReasons.options.options) {
            if (element.value === value) {
                out = element.captionLines[0];
                break;
            }
        }
        return out;
    }
    async getVisitReasons(organizationId) {
        const { rows } = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
            return await trx.query(`
			SELECT  "settings" 
			FROM "${organizationId}"."organizationVisitorModuleSettings" as ovm
			WHERE ovm."organizationId" = $1;
			`, [organizationId]);
        }, this._request.requesterUserId, this._request.organizationId);
        let visitorModuleSettings = rows[0]["settings"];
        if (visitorModuleSettings && visitorModuleSettings.visitFormFields && visitorModuleSettings.visitFormFields.length > 0) {
            let reasons = visitorModuleSettings.visitFormFields.find((vff) => vff.name === "reason");
            return reasons;
        }
        return null;
    }
    addRow(row) {
        let colIndex = 1;
        let rowData = [
            this._options.filterType === dal_constants_1.DalConstants.VisitAnalysisVisitCountGenericFilterType.ByVisitReason
                ? row.name
                    ? this.findVisitReason(parseInt(row.name))
                    : ""
                : row.name?.toUpperCase() ?? "",
        ];
        if (this._options.filterType === dal_constants_1.DalConstants.VisitAnalysisVisitCountGenericFilterType.ByVisitorId ||
            this._options.filterType === dal_constants_1.DalConstants.VisitAnalysisVisitCountGenericFilterType.ByVisitedPersonId) {
            rowData.push(row.captionLines && row.captionLines.length >= 2 ? row.captionLines[1].text[0] : "");
        }
        rowData.push(row.count?.toString() ?? "");
        colIndex = this.addCellsToRow(rowData, colIndex, this._rowIndex, this._ws);
        this._ws.getRow(this._rowIndex).commit();
        this._rowIndex++;
    }
    addFilterToSheet(ws, startRowIndex, startColIndex, filter) {
        this.addFilterCell11(ws, startRowIndex, startColIndex, this._organizationName + " " + i18n_1.default.__({ phrase: "EXCEL-REPORT.VISITOR_LIST", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin, {
            bold: true,
        });
        this.addFilterCell11(ws, startRowIndex + 1, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.FILE_CREATION_DATE", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        this.addFilterCell11(ws, startRowIndex + 1, startColIndex + 1, this._currentDate.locale(this._locale).format("LL") + " " + this._currentDate.locale(this._locale).format("HH:mm:ss"), generator_1.ReportGenerator.Constants.Styling.AllThin, { italic: true }, generator_1.ReportGenerator.Constants.DateFormat);
        this.addFilterCell11(ws, startRowIndex + 2, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.APPLIED_FILTERS", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        ws.mergeCells(startRowIndex, startColIndex, startRowIndex, startColIndex + 2);
        ws.mergeCells(startRowIndex + 1, startColIndex + 1, startRowIndex + 1, startColIndex + 2);
        ws.mergeCells(startRowIndex + 2, startColIndex + 1, startRowIndex + 2, startColIndex + 2);
        this.addFilterCell11(ws, startRowIndex + 2, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.APPLIED_FILTERS", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        this.addFilterCell11(ws, startRowIndex + 2, startColIndex + 1, filter, generator_1.ReportGenerator.Constants.Styling.AllThin);
        let filterlength = filter.split("\r\n").length;
        let row = ws.getRow(startRowIndex + 2);
        row.height = filterlength * 15;
    }
    initSheet(ws, startRowIndex, startColIndex) {
        let columns = [];
        let data;
        switch (this._options.filterType) {
            case dal_constants_1.DalConstants.VisitAnalysisVisitCountGenericFilterType.ByVisitorId:
                data = [
                    {
                        phrase: "EXCEL-REPORT.VISITOR",
                        cellWidth: 40,
                    },
                    {
                        phrase: "EXCEL-REPORT.UNIQUE_ID",
                        cellWidth: 40,
                    },
                ];
                break;
            case dal_constants_1.DalConstants.VisitAnalysisVisitCountGenericFilterType.ByVisitedPersonId:
                data = [
                    {
                        phrase: "EXCEL-REPORT.VISITED",
                        cellWidth: 40,
                    },
                    {
                        phrase: "EXCEL-REPORT.UNIQUE_ID",
                        cellWidth: 40,
                    },
                ];
                break;
            case dal_constants_1.DalConstants.VisitAnalysisVisitCountGenericFilterType.ByVisitReason:
                data = [
                    {
                        phrase: "EXCEL-REPORT.VISIT_REASON",
                        cellWidth: 40,
                    },
                ];
                break;
            case dal_constants_1.DalConstants.VisitAnalysisVisitCountGenericFilterType.ByLicensePlate:
                data = [
                    {
                        phrase: "EXCEL-REPORT.LICENSE_PLATE",
                        cellWidth: 40,
                    },
                ];
                break;
        }
        data.push({
            phrase: "EXCEL-REPORT.VISIT_COUNT",
            cellWidth: 20,
        });
        startColIndex = this.addInitCellsToRow(data, columns, startColIndex, startRowIndex, ws);
        ws.autoFilter = {
            from: "A" + startRowIndex,
            to: ws.getColumn(columns.length).letter + startRowIndex,
        };
        ws.columns = columns;
        ws.views.push({ state: "frozen", ySplit: startRowIndex, activeCell: "A1", showGridLines: false });
    }
    async processVisitorCaptionLines(visitorCaptionLines, visitorUserIdProfileIdMapping, captionLines) {
        const chunkSize = 1000;
        for (let i = 0; i < visitorCaptionLines.length; i += chunkSize) {
            for (let j = i; j < Math.min(i + chunkSize, visitorCaptionLines.length); j++) {
                let vcl = visitorCaptionLines[j];
                let mapping = visitorUserIdProfileIdMapping.find((vm) => vm.visitorUserId === vcl.id);
                if (mapping) {
                    captionLines.push({
                        id: mapping.visitorProfileId,
                        captionLines: vcl.captionLines,
                    });
                }
            }
            await new Promise((resolve) => setImmediate(resolve));
        }
    }
    async processDbResultItems(dbResult, captionLines, allVisitorCountData) {
        const chunkSize = 1000;
        for (let i = 0; i < dbResult.items.length; i += chunkSize) {
            for (let j = i; j < Math.min(i + chunkSize, dbResult.items.length); j++) {
                let dbr = dbResult.items[j];
                let captions = [];
                if (captionLines && captionLines.length > 0) {
                    let captionLinesOfSingleUser = captionLines.find((cl) => cl.id === dbr.id);
                    if (captionLinesOfSingleUser && captionLinesOfSingleUser.captionLines) {
                        captions = captionLinesOfSingleUser.captionLines;
                    }
                }
                allVisitorCountData.items.push({
                    captionLines: captions,
                    name: dbr.name,
                    count: parseInt(dbr.count),
                });
            }
            await new Promise((resolve) => setImmediate(resolve));
        }
    }
    async generateVisitorCountReport() {
        this._organizationName = (await dal_manager_1.dbManager.accessOrganization.getOrganizationBasic(this._request.organizationId)).name;
        app_logs_1.logger.debug("Organization Name : " + this._organizationName);
        this._visitReasons = await this.getVisitReasons(this._request.organizationId);
        this._ws = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.VISITOR_LIST", locale: this._locale }), {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._ws.properties.defaultRowHeight = 15;
        this._rowIndex = 6;
        let { caption: requestedUserCaption } = await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId: this._request.organizationId, userId: this._request.requesterUserId });
        if (!requestedUserCaption) {
            throw new Error("Not found");
        }
        this._requestedUserCaption = requestedUserCaption[0].text[0];
        this._wb.creator = this._requestedUserCaption;
        let filters = [];
        let filter = this._options;
        if (filter.filterValue) {
            let filterValue = "";
            switch (this._options.filterType) {
                case dal_constants_1.DalConstants.VisitAnalysisVisitCountGenericFilterType.ByVisitorId:
                    filterValue = await this.getFilter(filter.filterValue);
                    filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.VISITOR", locale: this._locale }) + " : " + filterValue["name"]);
                    break;
                case dal_constants_1.DalConstants.VisitAnalysisVisitCountGenericFilterType.ByVisitedPersonId:
                    filterValue = await this.getFilter(filter.filterValue);
                    filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.VISITED", locale: this._locale }) + " : " + filterValue);
                    break;
                case dal_constants_1.DalConstants.VisitAnalysisVisitCountGenericFilterType.ByLicensePlate:
                    filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.LICENSE_PLATE", locale: this._locale }) + " : " + filter.filterValue);
                    break;
            }
        }
        if (filters.length === 0) {
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.NOT_EXISTING", locale: this._locale }));
        }
        app_logs_1.logger.debug("\n\nApplied Filters:\n");
        app_logs_1.logger.debug(filters.join("\r\n"));
        app_logs_1.logger.debug("\n\n");
        this.addFilterToSheet(this._ws, 1, 1, filters.join("\r\n"));
        this.initSheet(this._ws, 5, 1);
        app_logs_1.logger.debug("Generating report...");
        let option = Object.assign(this._options, {
            pagination: {
                take: undefined,
                skip: undefined,
            },
        });
        let dbResult = await dal_manager_1.dbManager.accessLog.getVisitCountsByGenericFilter(this._request.organizationId, option);
        let captionLines = [];
        switch (this._options.filterType) {
            case app_enums_1.enums.VisitAnalysisVisitCountGenericFilterType.ByVisitorId:
                let visitorUserIdProfileIdMapping = await dal_manager_1.dbManager.accessVisitor.getUserIdsByVisitorProfileIds(this._request.organizationId, dbResult.items.map((d) => d.id));
                let visitorCaptionLines = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(this._request.organizationId, visitorUserIdProfileIdMapping.map((v) => v.visitorUserId));
                await this.processVisitorCaptionLines(visitorCaptionLines, visitorUserIdProfileIdMapping, captionLines);
                break;
            case app_enums_1.enums.VisitAnalysisVisitCountGenericFilterType.ByVisitedPersonId:
                captionLines = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(this._request.organizationId, dbResult.items.map((d) => d.id));
                break;
            default:
                break;
        }
        let allVisitorCountData = {
            pagination: dbResult.pagination,
            filterType: this._options.filterType,
            items: [],
        };
        if (this._options.filterType !== app_enums_1.enums.VisitAnalysisVisitCountGenericFilterType.ByLicensePlate) {
            await this.processDbResultItems(dbResult, captionLines, allVisitorCountData);
        }
        else {
            allVisitorCountData.items = dbResult.items;
        }
        await this.onDataCallbackFunction(allVisitorCountData.items);
        app_logs_1.logger.debug("end of fetching");
        this._ws.pageSetup.printArea = "A6:" + this._ws.getColumn(this._ws.columns.length).letter + this._rowIndex;
        this._ws.pageSetup.printTitlesRow = "5:5";
        this._ws.commit();
        await this._wb.commit();
        app_logs_1.logger.debug("File is ready -> " + this._filePath);
        return {
            format: app_enums_1.enums.ReportFormat.Excel,
            preparedReportId: this._fileId,
            filePath: this._filePath,
            notificationType: undefined,
            reportType: app_enums_1.enums.ReportCode.VisitorCountReport,
            sendFileInAttachments: undefined,
        };
    }
    async onDataCallbackFunction(rows) {
        app_logs_1.logger.debug("Fetch data %s rows", rows.length);
        for (const row of rows) {
            this.addRow(row);
        }
        return Promise.resolve();
    }
    async generateReport() {
        return await this.generateVisitorCountReport();
    }
}
exports.VisitorCountReport = VisitorCountReport;
