"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.VisitRecordsExcelReport = void 0;
const i18n_1 = __importDefault(require("i18n"));
const luxon_1 = require("luxon");
const app_enums_1 = require("../../../app.enums");
const app_logs_1 = require("../../../app.logs");
const dal_manager_1 = require("../../../dal/dal.manager");
const generator_1 = require("../generator");
const excel_report_refactory_1 = require("../excel.report.refactory");
class VisitRecordsExcelReport extends generator_1.ReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._options = request.filter;
        this._visitorProfileFormFields = this._options.organizationVisitorModuleSettings.visitorProfileFormFields.filter((item) => item.reportable);
        this._visitFormFields = this._options.organizationVisitorModuleSettings.visitFormFields.filter((item) => item.reportable);
        this._reportFactory = new excel_report_refactory_1.ReportFactory(request.organizationId, locale);
    }
    async generateReport() {
        try {
            this._visitReasons = await this.getVisitReasons(this._request.organizationId);
            this._registrationPoints = await dal_manager_1.dbManager.accessVisitor.listRegistrationPointsDetailed(this._request.organizationId, this._request.requesterUserId);
            const sheetInfo = [
                {
                    sheetName: i18n_1.default.__({ phrase: "EXCEL-REPORT.GENERAL", locale: this._locale }),
                    headers: this.getHeaders(),
                },
            ];
            const reportInfoSheetData = {
                reportNamePhrase: "EXCEL-REPORT.VISIT_RECORDS_REPORT",
                filters: await this.getFilters(),
            };
            await this._reportFactory.initialize({
                requesterUserId: this._request.requesterUserId,
                userIds: null,
                reportInfoSheetData,
                sheetInfo,
            });
            await dal_manager_1.dbManager.accessVisitor.generateVisitRecordsExcelReport(this._request.organizationId, this._request.requesterUserId, this._options, this._options.organizationVisitorModuleSettings, this.onDataCallbackFunction.bind(this));
            const fileObj = await this._reportFactory.finalize(reportInfoSheetData);
            return {
                format: app_enums_1.enums.ReportFormat.Excel,
                preparedReportId: fileObj.fileId,
                filePath: fileObj.filePath,
                notificationType: undefined,
                reportType: app_enums_1.enums.ReportCode.TerminatedVisits,
                sendFileInAttachments: undefined,
            };
        }
        catch (error) {
            app_logs_1.logger.error(`Error generating VisitRecordsExcelReport: ${error.message}`, {
                stack: error.stack,
                organizationId: this._request.organizationId,
                requesterUserId: this._request.requesterUserId,
            });
            throw new Error("Failed to generate VisitRecordsExcelReport");
        }
    }
    findVisitReason(value) {
        if (!this._visitReasons)
            return "";
        const option = this._visitReasons.options.options.find((element) => element.value === value);
        return option ? option.captionLines[0] : "";
    }
    async getVisitReasons(organizationId) {
        const { rows } = await dal_manager_1.dbManager.organizationTransaction(async (trx) => trx.query(`
            SELECT "settings"
            FROM "${organizationId}"."organizationVisitorModuleSettings" AS ovm
            WHERE ovm."organizationId" = $1;
            `, [organizationId]), this._request.requesterUserId, this._request.organizationId);
        const visitorModuleSettings = rows[0]?.["settings"];
        if (visitorModuleSettings?.visitFormFields?.length) {
            return visitorModuleSettings.visitFormFields.find((vff) => vff.name === "reason") || null;
        }
        return null;
    }
    checkNull(value) {
        if (value === "true") {
            return i18n_1.default.__({ phrase: "EXCEL-REPORT.YES", locale: this._locale });
        }
        if (value === "false") {
            return i18n_1.default.__({ phrase: "EXCEL-REPORT.NO", locale: this._locale });
        }
        if (value?.startDate && value?.endDate) {
            const format = generator_1.ReportGenerator.Constants.LuxonDateTimeFormat;
            const startDate = luxon_1.DateTime.fromJSDate(new Date(value.startDate)).toFormat(format);
            const endDate = luxon_1.DateTime.fromJSDate(new Date(value.endDate)).toFormat(format);
            return `${startDate} - ${endDate}`;
        }
        if (value instanceof Object) {
            return `${value.name ?? ""} ${value.surname ?? ""}`;
        }
        return value ?? "";
    }
    findRegistrationPoint(regPointId) {
        const registrationPoint = this._registrationPoints.find((rp) => rp.id === regPointId);
        return registrationPoint ? registrationPoint.deviceName || registrationPoint.name || "" : "";
    }
    async getFilters() {
        let filters = {};
        let filter = this._options;
        if (filter.visitFilterFields?.length) {
            for (let element of filter.visitFilterFields) {
                switch (element.name) {
                    case "visitedPerson":
                        const visited = await dal_manager_1.dbManager.accessIdentity.listIdentity(this._request.organizationId, [element.value]);
                        filters.visitedName = visited.items[0]?.fullname || "";
                        break;
                    case "reason":
                        filters.visitReason = this.findVisitReason(element.value);
                        break;
                }
            }
        }
        if (filter.visitorProfileIds?.length) {
            const organizationVisitorModuleSettings = await dal_manager_1.dbManager.accessVisitor.getVisitAndVisitorFormSettings(this._request.organizationId);
            filters.visitors = await Promise.all(filter.visitorProfileIds.map(async (element) => {
                const visitorProfile = await dal_manager_1.dbManager.accessVisitor.getVisitor(this._request.organizationId, element, organizationVisitorModuleSettings);
                return visitorProfile.fields.name;
            }));
        }
        if (filter.visitorProfileFilterFields?.length) {
            filter.visitorProfileFilterFields.forEach((element) => {
                switch (element.name) {
                    case "name":
                        filters.visitorName = element.value;
                        break;
                    case "uniqueId":
                        filters.uniqueId = element.value;
                        break;
                    case "licencePlate":
                        filters.visitorLicensePlate = element.value;
                        break;
                    case "forbidden":
                        if (element.value)
                            filters.forbiddenUser = true;
                        break;
                }
            });
        }
        filters.visitStatus = filter.type;
        filters.dateTimeRange = { startDateTime: this._options.startUtc, endDateTime: this._options.endUtc };
        return filters;
    }
    getHeaders() {
        const baseHeaders = [
            { phrase: "EXCEL-REPORT.START_DATE", width: 30 },
            { phrase: "EXCEL-REPORT.END_DATE", width: 30 },
        ];
        const formData = [
            ...(this._visitorProfileFormFields ?? []).map((form) => ({ value: form.caption, width: 30 })),
            ...(this._visitFormFields ?? []).map((form) => ({ value: form.caption, width: 30 })),
        ];
        const additionalHeaders = [
            { phrase: "EXCEL-REPORT.VISITED", width: 30 },
            { phrase: "EXCEL-REPORT.VISIT_REASON", width: 30 },
            { phrase: "EXCEL-REPORT.NOTE", width: 30 },
            { phrase: "EXCEL-REPORT.VISITOR_REGISTRATION_POINT", width: 30 },
            { phrase: "EXCEL-REPORT.PREREGISTRATION_CREATOR", width: 30 },
            { phrase: "EXCEL-REPORT.VISIT_STARTER", width: 30 },
            { phrase: "EXCEL-REPORT.VISIT_TERMINATOR", width: 30 },
        ];
        return [...baseHeaders, ...formData, ...additionalHeaders];
    }
    async onDataCallbackFunction(rows) {
        app_logs_1.logger.debug("Fetch data %s rows", rows.length);
        const visitIds = rows.map((r) => r.visitId);
        this._visitLogs = await dal_manager_1.dbManager.accessLog.listMultipleUserActionHistory(this._request.organizationId, visitIds, "visitId", app_enums_1.enums.UserActionCategory.Visitor, [
            app_enums_1.enums.UserVisitorActionType.NewVisit,
            app_enums_1.enums.UserVisitorActionType.TerminateVisit,
            app_enums_1.enums.UserVisitorActionType.NewExpectedVisit,
            app_enums_1.enums.UserVisitorActionType.DeleteExpectedVisit,
        ]);
        const sheetInfo = [
            {
                sheetName: i18n_1.default.__({ phrase: "EXCEL-REPORT.GENERAL", locale: this._locale }),
                data: [],
            },
        ];
        for (const row of rows) {
            const rowData = [
                { value: row.visitStartUtc ? luxon_1.DateTime.fromJSDate(row.visitStartUtc).toFormat(generator_1.ReportGenerator.Constants.LuxonDateTimeFormat) : "" },
                { value: row.visitEndUtc ? luxon_1.DateTime.fromJSDate(row.visitEndUtc).toFormat(generator_1.ReportGenerator.Constants.LuxonDateTimeFormat) : "" },
                ...(this._visitorProfileFormFields?.map((form) => ({
                    value: this.checkNull(row.visitorProfileFields[form.name]),
                })) || []),
                ...(this._visitFormFields?.map((form) => ({
                    value: form.name === "reason" ? this.findVisitReason(parseInt(row.visitFields["reason"])) : this.checkNull(row.visitFields[form.name]),
                })) || []),
                { value: row.visitFields["visitedPerson"] ? `${row.visitFields["visitedPerson"].name} ${row.visitFields["visitedPerson"].surname}` : "" },
                { value: this.findVisitReason(parseInt(row.visitFields["reason"])) ?? "" },
                { value: row.visitFields["Note"] ?? "" },
                { value: row.visitFields["visitorRegistrationPointId"] ? this.findRegistrationPoint(row.visitFields["visitorRegistrationPointId"]) : "" },
            ];
            const visitLogsMap = this._visitLogs.reduce((acc, item) => {
                if (item.actionUser?.fullname && item.visitId === row.visitId) {
                    switch (item.action) {
                        case app_enums_1.enums.UserVisitorActionType.NewExpectedVisit:
                            acc.preregistrationCreator = item.actionUser.fullname;
                            break;
                        case app_enums_1.enums.UserVisitorActionType.NewVisit:
                            acc.visitStarter = item.actionUser.fullname;
                            break;
                        case app_enums_1.enums.UserVisitorActionType.TerminateVisit:
                            acc.visitFinisher = item.actionUser.fullname;
                            break;
                    }
                }
                return acc;
            }, { preregistrationCreator: "", visitStarter: "", visitFinisher: "" });
            rowData.push({ value: visitLogsMap.preregistrationCreator, alignment: { wrapText: true } }, { value: visitLogsMap.visitStarter, alignment: { wrapText: true } }, { value: visitLogsMap.visitFinisher, alignment: { wrapText: true } });
            sheetInfo[0].data.push({ cells: rowData });
        }
        this._reportFactory.appendData(sheetInfo);
        return Promise.resolve();
    }
}
exports.VisitRecordsExcelReport = VisitRecordsExcelReport;
