"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.TerminalList = void 0;
const fs_1 = __importDefault(require("fs"));
const path_1 = __importDefault(require("path"));
const exceljs_1 = __importDefault(require("exceljs"));
const moment_1 = __importDefault(require("moment"));
const app_config_1 = require("../../../app.config");
const dal_manager_1 = require("../../../dal/dal.manager");
const dal_constants_1 = require("../../../dal/dal.constants");
const uuid_1 = __importDefault(require("uuid"));
const i18n_1 = __importDefault(require("i18n"));
const enums_1 = require("../../../lib/access-model/v2/enums");
const app_logs_1 = require("../../../app.logs");
const report_util_1 = require("../report.util");
const app_enums_1 = require("../../../app.enums");
const generator_1 = require("../generator");
class TerminalList extends generator_1.ReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._wb = new exceljs_1.default.Workbook();
        this._currentDate = (0, moment_1.default)();
        this._options = request.filter;
    }
    addTerminalRow(row) {
        let colIndex = 1;
        const rowData = [
            row.name.toUpperCase(),
            row.location.toUpperCase(),
            (0, moment_1.default)(row.lastUpdateUtc, "YYYY-MM-DD HH:mm Z").format("DD/MM/YYYY HH:mm"),
            row.serialNumber,
            row.ip,
            row.temperature ? (row.temperature + " °C") : "",
            dal_constants_1.DalConstants.DevicePowerState[row.power] ? i18n_1.default.__({ phrase: "EXCEL-REPORT." + dal_constants_1.DalConstants.DevicePowerState[row.power].toUpperCase(), locale: this._locale }) : "",
            row.overallStorage,
            row.model ? row.model : "",
            row.hwVersion,
            row.swVersion,
            row.serviceVersion,
            row.libAccessVersion,
            row.onBoardComputerModel,
            row.onBoardComputerVersion,
        ];
        colIndex = this.addCellsToRow(rowData, colIndex, this._terminalRowIndex, this._wsTerminal);
        this._terminalRowIndex++;
    }
    addDryContactRow(row) {
        let colIndex = 1;
        const rowData = [
            row.name.toUpperCase(),
            (0, moment_1.default)(row.lastUpdateUtc, "YYYY-MM-DD HH:mm Z").format("DD/MM/YYYY HH:mm"),
            row.deviceName,
            row.accessControlPointName,
            row.relayName,
            row.timeout,
            dal_constants_1.DalConstants.AccessDirection[row.direction]
                ? i18n_1.default.__({ phrase: "EXCEL-REPORT." + dal_constants_1.DalConstants.AccessDirection[row.direction].toUpperCase(), locale: this._locale })
                : "",
            row.number ? row.number : "",
        ];
        colIndex = this.addCellsToRow(rowData, colIndex, this._dryContactRowIndex, this._wsDryContact);
        this._dryContactRowIndex++;
    }
    addReaderRow(row) {
        let colIndex = 1;
        let credType = report_util_1.ReportConstants.CredentialTypes.find((e) => e.type === row.credentialType);
        const rowData = [
            row.name,
            row.location,
            (0, moment_1.default)(row.lastUpdateUtc, "YYYY-MM-DD HH:mm Z").format("DD/MM/YYYY HH:mm"),
            row.deviceName,
            row.accessControlPointName,
            row.relayName,
            row.recurrentAttemptTimeout
                ? (row.recurrentAttemptTimeout + " " + i18n_1.default.__({ phrase: "EXCEL-REPORT.SECOND_ABBR", locale: this._locale }))
                : "",
            dal_constants_1.DalConstants.AccessDirection[row.direction] ? i18n_1.default.__({ phrase: "EXCEL-REPORT." + dal_constants_1.DalConstants.AccessDirection[row.direction].toUpperCase(), locale: this._locale }) : "",
            row.number,
            enums_1.EnumsV2.ReaderOutputType[row.readerOutputType]
                ? i18n_1.default.__({ phrase: "EXCEL-REPORT.READEROUTPUTTYPE_" + enums_1.EnumsV2.ReaderOutputType[row.readerOutputType].toUpperCase(), locale: this._locale })
                : "",
            dal_constants_1.DalConstants.CardDataFormat[row.dataFormat]
                ? i18n_1.default.__({ phrase: "EXCEL-REPORT.CARDDATAFORMAT_" + dal_constants_1.DalConstants.CardDataFormat[row.dataFormat].toUpperCase(), locale: this._locale })
                : "",
            credType ? i18n_1.default.__({ phrase: "EXCEL-REPORT." + credType.description, locale: this._locale }) : "",
        ];
        colIndex = this.addCellsToRow(rowData, colIndex, this._readerRowIndex, this._wsReader);
        this._readerRowIndex++;
    }
    addRelayRow(row) {
        let colIndex = 1;
        const rowData = [
            row.name,
            row.location,
            (0, moment_1.default)(row.lastUpdateUtc, "YYYY-MM-DD HH:mm Z").format("DD/MM/YYYY HH:mm"),
            row.deviceName,
            row.accessControlPointName,
            row.driveDuration ? row.driveDuration + " " + i18n_1.default.__({ phrase: "EXCEL-REPORT.SECOND_ABBR", locale: this._locale }) : "",
            dal_constants_1.DalConstants.AccessDirection[row.direction] ? i18n_1.default.__({ phrase: "EXCEL-REPORT." + dal_constants_1.DalConstants.AccessDirection[row.direction].toUpperCase(), locale: this._locale }) : "",
            row.number,
        ];
        colIndex = this.addCellsToRow(rowData, colIndex, this._relayRowIndex, this._wsRelay);
        this._relayRowIndex++;
    }
    addFilterToSheet(ws, startRowIndex, startColIndex, text, filter) {
        this.addFilterCell11(ws, startRowIndex, startColIndex, text, generator_1.ReportGenerator.Constants.Styling.AllThin, { bold: true });
        this.addFilterCell11(ws, startRowIndex + 1, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.FILE_CREATION_DATE", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin, { bold: true });
        this.addFilterCell11(ws, startRowIndex + 1, startColIndex + 1, this._currentDate.locale(this._locale).format("LL") + " " + this._currentDate.locale(this._locale).format("HH:mm:ss"), generator_1.ReportGenerator.Constants.Styling.AllThin, { italic: true }, generator_1.ReportGenerator.Constants.DateFormat);
        ws.mergeCells(startRowIndex, startColIndex, startRowIndex, startColIndex + 2);
        ws.mergeCells(startRowIndex + 1, startColIndex + 1, startRowIndex + 1, startColIndex + 2);
        ws.mergeCells(startRowIndex + 2, startColIndex + 1, startRowIndex + 2, startColIndex + 2);
        this.addFilterCell11(ws, startRowIndex + 2, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.APPLIED_FILTERS", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin, { bold: true });
        this.addFilterCell11(ws, startRowIndex + 2, startColIndex + 1, filter, generator_1.ReportGenerator.Constants.Styling.AllThin);
        let filterlength = filter.split("\r\n").length;
        let row = ws.getRow(startRowIndex + 2);
        row.height = filterlength * 20;
    }
    initTerminalSheet(wsRaw, startRowIndex, startColIndex) {
        let columns = [];
        const data = [
            {
                phrase: "EXCEL-REPORT.TERMINAL_NAME",
                cellWidth: 30,
            },
            {
                phrase: "EXCEL-REPORT.LOCATION",
                cellWidth: 30,
            },
            {
                phrase: "EXCEL-REPORT.LAST_UPDATE",
                cellWidth: 20,
            },
            {
                phrase: "EXCEL-REPORT.SERIAL_NUMBER",
                cellWidth: 20,
            },
            {
                phrase: "EXCEL-REPORT.IP",
                cellWidth: 20,
            },
            {
                phrase: "EXCEL-REPORT.TEMPERATURE",
                cellWidth: 15,
            },
            {
                phrase: "EXCEL-REPORT.POWER_STATUS",
                cellWidth: 20,
            },
            {
                phrase: "EXCEL-REPORT.STORAGE",
                cellWidth: 20,
            },
            {
                phrase: "EXCEL-REPORT.MODEL",
                cellWidth: 15,
            },
            {
                phrase: "EXCEL-REPORT.HARDWARE_VERSION",
                cellWidth: 20,
            },
            {
                phrase: "EXCEL-REPORT.SOFTWARE_VERSION",
                cellWidth: 20,
            },
            {
                phrase: "EXCEL-REPORT.SERVICE_VERSION",
                cellWidth: 20,
            },
            {
                phrase: "EXCEL-REPORT.ACCESS_LIB_VERSION",
                cellWidth: 20,
            },
            {
                phrase: "EXCEL-REPORT.ONBOARD_COMPUTER_MODEL",
                cellWidth: 20,
            },
            {
                phrase: "EXCEL-REPORT.ONBOARD_COMPUTER_VERSION",
                cellWidth: 20,
            },
        ];
        startColIndex = this.addInitCellsToRow(data, columns, startColIndex, startRowIndex, wsRaw);
        wsRaw.autoFilter = {
            from: "A" + (this._terminalRowIndex - 1),
            to: wsRaw.getColumn(columns.length).letter + this._terminalRowIndex,
        };
        wsRaw.columns = columns;
        wsRaw.views = [{ state: "frozen", ySplit: this._terminalRowIndex - 1, activeCell: "A1", showGridLines: false }];
    }
    initDryContactSheet(wsRaw, startRowIndex, startColIndex) {
        let columns = [];
        const data = [
            {
                phrase: "EXCEL-REPORT.DRY_CONTACT_NAME",
                cellWidth: 30,
            },
            {
                phrase: "EXCEL-REPORT.LAST_UPDATE",
                cellWidth: 20,
            },
            {
                phrase: "EXCEL-REPORT.DEVICE_NAME",
                cellWidth: 30,
            },
            {
                phrase: "EXCEL-REPORT.ACCESS_CONTROL_POINT_NAME",
                cellWidth: 30,
            },
            {
                phrase: "EXCEL-REPORT.RELAY_NAME",
                cellWidth: 30,
            },
            {
                phrase: "EXCEL-REPORT.TIMEOUT",
                cellWidth: 18,
            },
            {
                phrase: "EXCEL-REPORT.DIRECTION",
                cellWidth: 20,
            },
            {
                phrase: "EXCEL-REPORT.NUMBER",
                cellWidth: 20,
            },
        ];
        startColIndex = this.addInitCellsToRow(data, columns, startColIndex, startRowIndex, wsRaw);
        wsRaw.autoFilter = {
            from: "A" + (this._dryContactRowIndex - 1),
            to: wsRaw.getColumn(columns.length).letter + this._dryContactRowIndex,
        };
        wsRaw.columns = columns;
        wsRaw.views = [{ state: "frozen", ySplit: this._dryContactRowIndex - 1, activeCell: "A1", showGridLines: false }];
    }
    initReaderSheet(wsRaw, startRowIndex, startColIndex) {
        let columns = [];
        const data = [
            {
                phrase: "EXCEL-REPORT.READER_NAME",
                cellWidth: 30,
            },
            {
                phrase: "EXCEL-REPORT.LOCATION",
                cellWidth: 30,
            },
            {
                phrase: "EXCEL-REPORT.LAST_UPDATE",
                cellWidth: 20,
            },
            {
                phrase: "EXCEL-REPORT.DEVICE_NAME",
                cellWidth: 30,
            },
            {
                phrase: "EXCEL-REPORT.ACCESS_CONTROL_POINT_NAME",
                cellWidth: 30,
            },
            {
                phrase: "EXCEL-REPORT.RELAY_NAME",
                cellWidth: 30,
            },
            {
                phrase: "EXCEL-REPORT.RECURRENT_ATTEMPT_TIMEOUT",
                cellWidth: 15,
            },
            {
                phrase: "EXCEL-REPORT.DIRECTION",
                cellWidth: 15,
            },
            {
                phrase: "EXCEL-REPORT.NUMBER",
                cellWidth: 15,
            },
            {
                phrase: "EXCEL-REPORT.READER_OUTPUT_TYPE",
                cellWidth: 15,
            },
            {
                phrase: "EXCEL-REPORT.DATA_FORMAT",
                cellWidth: 25,
            },
            {
                phrase: "EXCEL-REPORT.CREDENTIAL_TYPE",
                cellWidth: 20,
            },
        ];
        startColIndex = this.addInitCellsToRow(data, columns, startColIndex, startRowIndex, wsRaw);
        wsRaw.autoFilter = {
            from: "A" + (this._readerRowIndex - 1),
            to: wsRaw.getColumn(columns.length).letter + this._readerRowIndex,
        };
        wsRaw.columns = columns;
        wsRaw.views = [{ state: "frozen", ySplit: this._readerRowIndex - 1, activeCell: "A1", showGridLines: false }];
    }
    initRelaySheet(wsRaw, startRowIndex, startColIndex) {
        let columns = [];
        const data = [
            {
                phrase: "EXCEL-REPORT.READER_NAME",
                cellWidth: 30,
            },
            {
                phrase: "EXCEL-REPORT.LOCATION",
                cellWidth: 30,
            },
            {
                phrase: "EXCEL-REPORT.LAST_UPDATE",
                cellWidth: 20,
            },
            {
                phrase: "EXCEL-REPORT.DEVICE_NAME",
                cellWidth: 30,
            },
            {
                phrase: "EXCEL-REPORT.ACCESS_CONTROL_POINT_NAME",
                cellWidth: 30,
            },
            {
                phrase: "EXCEL-REPORT.DRIVE_DURATION",
                cellWidth: 18,
            },
            {
                phrase: "EXCEL-REPORT.DIRECTION",
                cellWidth: 20,
            },
            {
                phrase: "EXCEL-REPORT.NUMBER",
                cellWidth: 20,
            },
        ];
        startColIndex = this.addInitCellsToRow(data, columns, startColIndex, startRowIndex, wsRaw);
        wsRaw.autoFilter = {
            from: "A" + (this._relayRowIndex - 1),
            to: wsRaw.getColumn(columns.length).letter + this._relayRowIndex,
        };
        wsRaw.columns = columns;
        wsRaw.views = [{ state: "frozen", ySplit: this._relayRowIndex - 1, activeCell: "A1", showGridLines: false }];
    }
    async generateTerminalsReport() {
        this._wsTerminal = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.TERMINALS", locale: this._locale }), {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsTerminal.properties.defaultRowHeight = 15;
        this._terminalRowIndex = 6;
        this._wsTerminal.headerFooter.oddHeader =
            "&L" +
                i18n_1.default.__({ phrase: "EXCEL-REPORT.TERMINALS_LIST", locale: this._locale }) +
                "\n" +
                "&R" +
                i18n_1.default.__({ phrase: "EXCEL-REPORT.FILE_CREATION_DATE", locale: this._locale }) +
                " : " +
                this._currentDate.locale(this._locale).format("LL") +
                " " +
                this._currentDate.locale(this._locale).format("HH:mm:ss") +
                "\n";
        this._wsTerminal.headerFooter.oddFooter = i18n_1.default.__({ phrase: "EXCEL-REPORT.PAGE", locale: this._locale }) + " &P / &N" + "&R" + "&D  &T";
        this._wsDryContact = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.DRY_CONTACTS", locale: this._locale }), {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsDryContact.properties.defaultRowHeight = 15;
        this._dryContactRowIndex = 6;
        this._wsDryContact.headerFooter.oddHeader =
            "&L" +
                i18n_1.default.__({ phrase: "EXCEL-REPORT.DRY_CONTACTS_LIST", locale: this._locale }) +
                "\n" +
                "&R" +
                i18n_1.default.__({ phrase: "EXCEL-REPORT.FILE_CREATION_DATE", locale: this._locale }) +
                " : " +
                this._currentDate.locale(this._locale).format("LL") +
                " " +
                this._currentDate.locale(this._locale).format("HH:mm:ss") +
                "\n";
        this._wsDryContact.headerFooter.oddFooter = i18n_1.default.__({ phrase: "EXCEL-REPORT.PAGE", locale: this._locale }) + " &P / &N" + "&R" + "&D  &T";
        this._wsReader = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.READERS", locale: this._locale }), {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsReader.properties.defaultRowHeight = 15;
        this._readerRowIndex = 6;
        this._wsReader.headerFooter.oddHeader =
            "&L" +
                i18n_1.default.__({ phrase: "EXCEL-REPORT.READERS_LIST", locale: this._locale }) +
                "\n" +
                "&R" +
                i18n_1.default.__({ phrase: "EXCEL-REPORT.FILE_CREATION_DATE", locale: this._locale }) +
                " : " +
                this._currentDate.locale(this._locale).format("LL") +
                " " +
                this._currentDate.locale(this._locale).format("HH:mm:ss") +
                "\n";
        this._wsReader.headerFooter.oddFooter = i18n_1.default.__({ phrase: "EXCEL-REPORT.PAGE", locale: this._locale }) + " &P / &N" + "&R" + "&D  &T";
        this._wsRelay = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.RELAYS", locale: this._locale }), {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsRelay.properties.defaultRowHeight = 15;
        this._relayRowIndex = 6;
        this._wsRelay.headerFooter.oddHeader =
            "&L" +
                i18n_1.default.__({ phrase: "EXCEL-REPORT.RELAYS_LIST", locale: this._locale }) +
                "\n" +
                "&R" +
                i18n_1.default.__({ phrase: "EXCEL-REPORT.FILE_CREATION_DATE", locale: this._locale }) +
                " : " +
                this._currentDate.locale(this._locale).format("LL") +
                " " +
                this._currentDate.locale(this._locale).format("HH:mm:ss") +
                "\n";
        this._wsRelay.headerFooter.oddFooter = i18n_1.default.__({ phrase: "EXCEL-REPORT.PAGE", locale: this._locale }) + " &P / &N" + "&R" + "&D  &T";
        let { caption: requestedUserCaption } = await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId: this._request.organizationId, userId: this._request.requesterUserId });
        if (!requestedUserCaption) {
            throw new Error("Not found");
        }
        this._requestedUserCaption = requestedUserCaption[0].text[0];
        this._wb.creator = this._requestedUserCaption;
        let filters = [];
        let filter = this._options;
        if (filter.name) {
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.TERMINAL_NAME", locale: this._locale }) + " : " + filter.name);
        }
        if (filter.brand) {
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.BRAND", locale: this._locale }) + " : " + dal_constants_1.DalConstants.DeviceBrand[filter.brand]);
        }
        if (filter.model) {
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.MODEL", locale: this._locale }) + " : " + filter.model);
        }
        if (filter.connectionStatus) {
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.CONNECTION_STATUS", locale: this._locale }) +
                " : " +
                i18n_1.default.__({ phrase: "EXCEL-REPORT." + dal_constants_1.DalConstants.TerminalConnectionStatus[filter.connectionStatus].toUpperCase(), locale: this._locale }));
        }
        if (filter.warningLevel) {
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.WARNING_STATUS", locale: this._locale }) +
                " : " +
                i18n_1.default.__({ phrase: "EXCEL-REPORT." + dal_constants_1.DalConstants.TerminalWarningLevel[filter.warningLevel].toUpperCase(), locale: this._locale }));
        }
        app_logs_1.logger.debug("\n\nApplied Filters:\n");
        app_logs_1.logger.debug(filters.join("\r\n"));
        app_logs_1.logger.debug("\n\n");
        app_logs_1.logger.debug("Requested User : " + this._requestedUserCaption);
        app_logs_1.logger.debug("Requested User ID : " + this._request.requesterUserId);
        app_logs_1.logger.debug("Organization ID : " + this._request.organizationId);
        app_logs_1.logger.debug("Generating report...");
        this.addFilterToSheet(this._wsTerminal, 1, 1, i18n_1.default.__({ phrase: "EXCEL-REPORT.TERMINALS_LIST", locale: this._locale }), filters.join("\r\n"));
        this.initTerminalSheet(this._wsTerminal, 5, 1);
        this.addFilterToSheet(this._wsDryContact, 1, 1, i18n_1.default.__({ phrase: "EXCEL-REPORT.DRY_CONTACTS_LIST", locale: this._locale }), filters.join("\r\n"));
        this.initDryContactSheet(this._wsDryContact, 5, 1);
        this.addFilterToSheet(this._wsReader, 1, 1, i18n_1.default.__({ phrase: "EXCEL-REPORT.READERS_LIST", locale: this._locale }), filters.join("\r\n"));
        this.initReaderSheet(this._wsReader, 5, 1);
        this.addFilterToSheet(this._wsRelay, 1, 1, i18n_1.default.__({ phrase: "EXCEL-REPORT.RELAYS_LIST", locale: this._locale }), filters.join("\r\n"));
        this.initRelaySheet(this._wsRelay, 5, 1);
        await dal_manager_1.dbManager.accessDevice.fetchTerminalData(this._request.organizationId, this._request.requesterUserId, this._options, this.onDataCallbackFunctionTerminal.bind(this));
        await dal_manager_1.dbManager.accessDevice.fetchDryContactData(this._request.organizationId, this._request.requesterUserId, this._options, this.onDataCallbackFunctionDryContact.bind(this));
        await dal_manager_1.dbManager.accessDevice.fetchReaderData(this._request.organizationId, this._request.requesterUserId, this._options, this.onDataCallbackFunctionReader.bind(this));
        await dal_manager_1.dbManager.accessDevice.fetchRelayData(this._request.organizationId, this._request.requesterUserId, this._options, this.onDataCallbackFunctionRelay.bind(this));
        app_logs_1.logger.debug("End of fetching...");
        this._wsTerminal.pageSetup.printArea = "A5:" + this._wsTerminal.getColumn(this._wsTerminal.columns.length).letter + this._terminalRowIndex;
        this._wsTerminal.pageSetup.printTitlesRow = "4:4";
        this._wsDryContact.pageSetup.printArea = "A5:" + this._wsDryContact.getColumn(this._wsDryContact.columns.length).letter + this._dryContactRowIndex;
        this._wsDryContact.pageSetup.printTitlesRow = "4:4";
        this._wsReader.pageSetup.printArea = "A5:" + this._wsReader.getColumn(this._wsReader.columns.length).letter + this._readerRowIndex;
        this._wsReader.pageSetup.printTitlesRow = "4:4";
        this._wsRelay.pageSetup.printArea = "A5:" + this._wsRelay.getColumn(this._wsRelay.columns.length).letter + this._relayRowIndex;
        this._wsRelay.pageSetup.printTitlesRow = "4:4";
        let fileId = uuid_1.default.v4();
        let filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, fileId + ".xlsx");
        if (fs_1.default.existsSync(filePath)) {
            fs_1.default.unlinkSync(filePath);
        }
        await this._wb.xlsx.writeFile(filePath);
        app_logs_1.logger.debug("File is ready : " + filePath);
        return {
            format: app_enums_1.enums.ReportFormat.Excel,
            preparedReportId: fileId,
            filePath: filePath,
            notificationType: undefined,
            reportType: app_enums_1.enums.ReportCode.TerminalList,
            sendFileInAttachments: undefined,
        };
    }
    async onDataCallbackFunctionTerminal(rows) {
        app_logs_1.logger.debug("Fetch terminal data %s rows", rows.length);
        for (const row of rows) {
            if (row.usedStorage && row.availableStorage) {
                row.overallStorage = row.usedStorage.toFixed(2) + " / " + row.availableStorage.toFixed(2) + "  GB";
            }
            Object.keys(row).forEach(function (key) {
                if (!row[key] && row[key] !== 0) {
                    row[key] = "";
                }
            });
            this.addTerminalRow(row);
        }
        return Promise.resolve();
    }
    async onDataCallbackFunctionDryContact(rows) {
        app_logs_1.logger.debug("Fetch dry contact data %s rows", rows.length);
        for (const row of rows) {
            if (row.timeout) {
                row.timeout = row.timeout + " " + i18n_1.default.__({ phrase: "EXCEL-REPORT.SECOND_ABBR", locale: this._locale });
            }
            Object.keys(row).forEach(function (key) {
                if (!row[key] && row[key] !== 0) {
                    row[key] = "";
                }
            });
            this.addDryContactRow(row);
        }
        return Promise.resolve();
    }
    async onDataCallbackFunctionReader(rows) {
        app_logs_1.logger.debug("Fetch reader data %s rows", rows.length);
        for (const row of rows) {
            Object.keys(row).forEach(function (key) {
                if (!row[key] && row[key] !== 0) {
                    row[key] = "";
                }
            });
            this.addReaderRow(row);
        }
        return Promise.resolve();
    }
    async onDataCallbackFunctionRelay(rows) {
        app_logs_1.logger.debug("Fetch relay data %s rows", rows.length);
        for (const row of rows) {
            Object.keys(row).forEach(function (key) {
                if (!row[key] && row[key] !== 0) {
                    row[key] = "";
                }
            });
            this.addRelayRow(row);
        }
        return Promise.resolve();
    }
    async generateReport() {
        return await this.generateTerminalsReport();
    }
}
exports.TerminalList = TerminalList;
