"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.SensorEventsExcelReport = void 0;
const i18n_1 = __importDefault(require("i18n"));
const app_logs_1 = require("../../../app.logs");
const app_enums_1 = require("../../../app.enums");
const generator_1 = require("../generator");
const business_report_export_1 = require("../business.report.export");
const report_util_1 = require("../report.util");
const excel_report_refactory_1 = require("../excel.report.refactory");
class SensorEventsExcelReport extends generator_1.ReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._options = request.filter;
    }
    async generateReport() {
        try {
            const data = await (0, business_report_export_1.getSensorEventsReport)({ organizationId: this._request.organizationId, filter: this._options, requesterUserId: this._request.requesterUserId });
            const headers = [
                { phrase: "PDF-REPORT.date", width: 30 },
                { phrase: "PDF-REPORT.sensor", width: 30 },
                { phrase: "PDF-REPORT.status", width: 45 },
                { phrase: "PDF-REPORT.terminal", width: 45 },
                { phrase: "PDF-REPORT.last_action", width: 60 },
            ];
            const generalSheetData = data.result.items.map((item) => ({
                cells: [
                    { value: item?.lastAction?.actionDateTime ? (0, report_util_1.formatDateForExcel)(item.lastAction.actionDateTime) : "" },
                    { value: this.createCellSensor(item) },
                    { value: this.createCellStatus(item) },
                    { value: this.createCellTerminal(item) },
                    { value: this.createCellForLastAction(item) },
                ],
            }));
            const sheetInfo = [
                {
                    sheetName: i18n_1.default.__({ phrase: "EXCEL-REPORT.GENERAL", locale: this._locale }),
                    headers,
                    data: generalSheetData,
                },
            ];
            const reportInfoSheetData = {
                reportNamePhrase: "PDF-REPORT.SENSOR_EVENTS",
                filters: {
                    sensorIds: this._options.sensorIds,
                    terminalIds: this._options.terminalDeviceIds,
                    dateTimeRange: {
                        startDateTime: new Date(this._options.dateRange.startDateTime),
                        endDateTime: new Date(this._options.dateRange.endDateTime),
                    },
                },
            };
            const reportFactory = new excel_report_refactory_1.ReportFactory(this._request.organizationId, this._locale);
            await reportFactory.initialize({ requesterUserId: this._request.requesterUserId, userIds: null, reportInfoSheetData, sheetInfo });
            reportFactory.appendData(sheetInfo);
            const fileObj = await reportFactory.finalize(reportInfoSheetData);
            return {
                format: app_enums_1.enums.ReportFormat.Excel,
                preparedReportId: fileObj.fileId,
                filePath: fileObj.filePath,
                notificationType: undefined,
                reportType: app_enums_1.enums.ReportCode.SensorEvents,
                sendFileInAttachments: undefined,
            };
        }
        catch (error) {
            app_logs_1.logger.error(`Error generating EntryByUnitReport: ${error.message}`, {
                stack: error.stack,
                organizationId: this._request.organizationId,
                requesterUserId: this._request.requesterUserId,
            });
            throw new Error("Failed to generate EntryByUnitReport");
        }
    }
    createCellSensor(record) {
        if (!record.input)
            return "";
        const { name, typeName } = record.input;
        return [name, typeName].filter(Boolean).join("\n");
    }
    createCellStatus(record) {
        const locale = this._locale;
        if (!record.lastAction)
            return "";
        const getState = (state) => state != null ? i18n_1.default.__({ phrase: this.getStateText(record.input.type, state), locale }) : i18n_1.default.__({ phrase: "PDF-REPORT.no_record", locale });
        return `${getState(record.lastAction.stateBeforeAction)} => ${getState(record.lastAction.state)}`;
    }
    createCellTerminal(record) {
        if (!record.terminal)
            return "";
        const { name, serialNumber, model } = record.terminal;
        return [name, serialNumber, model].filter(Boolean).join("\n");
    }
    createCellForLastAction(record) {
        const locale = this._locale;
        if (!record.input?.type || !record.lastAction?.state || !record.lastAction?.reason)
            return "";
        let result = [
            `${i18n_1.default.__({ phrase: "PDF-REPORT.NEW_STATE", locale })}: ${i18n_1.default.__({ phrase: this.getStateText(record.input.type, record.lastAction.state), locale })}`,
            `${i18n_1.default.__({ phrase: "PDF-REPORT.REASON", locale })}: ${i18n_1.default.__({ phrase: this.getReasonText(record.input.type, record.lastAction.reason), locale })}`,
        ].join("\n");
        const accessLog = record.lastAction.accessLog;
        if (accessLog?.user) {
            const users = accessLog.user
                .filter((user) => user.text && user.text !== "__avatar")
                .map((user) => user.text.filter(Boolean).join(" "))
                .join("\n");
            if (users)
                result += `\n${users}`;
        }
        const state = accessLog?.state ? i18n_1.default.__({ phrase: this.getStateText(record.input.type, accessLog.state), locale }) : "";
        const reason = accessLog?.reason ? i18n_1.default.__({ phrase: this.getReasonText(record.input.type, accessLog.reason), locale }) : "";
        return [result, state, reason].filter(Boolean).join("\n");
    }
    getStateText(type, state) {
        const stateMap = {
            [app_enums_1.enums.DryContactTypeV2.StatusSensor]: report_util_1.ReportConstants.StatusSensorStates,
            [app_enums_1.enums.DryContactTypeV2.CounterSensor]: report_util_1.ReportConstants.CounterSensorStates,
            [app_enums_1.enums.DryContactTypeV2.TamperSwitch]: report_util_1.ReportConstants.TamperSwitchStates,
        };
        return stateMap[type]?.find((item) => item.type === state)?.description || "";
    }
    getReasonText(type, reason) {
        const reasonMap = {
            [app_enums_1.enums.DryContactTypeV2.StatusSensor]: report_util_1.ReportConstants.StatusSensorReasons,
            [app_enums_1.enums.DryContactTypeV2.CounterSensor]: report_util_1.ReportConstants.CounterSensorReasons,
        };
        return reasonMap[type]?.find((item) => item.type === reason)?.description || "";
    }
}
exports.SensorEventsExcelReport = SensorEventsExcelReport;
