"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.RegionStateExcelReport = void 0;
const i18n_1 = __importDefault(require("i18n"));
const app_logs_1 = require("../../../app.logs");
const app_enums_1 = require("../../../app.enums");
const generator_1 = require("../generator");
const business_report_export_1 = require("../business.report.export");
const report_util_1 = require("../report.util");
const excel_report_refactory_1 = require("../excel.report.refactory");
class RegionStateExcelReport extends generator_1.ReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._options = request.filter;
    }
    async generateReport() {
        try {
            const data = await (0, business_report_export_1.generateRegionStateData)({
                organizationId: this._request.organizationId,
                requesterUserId: this._request.requesterUserId,
                regionId: this._options.regionId,
                filter: this._options,
            });
            const userIds = [...new Set(data.items.map((item) => item.userId))];
            const generalSheetData = data.items.map((item) => ({
                userId: item.userId,
                cells: [
                    { value: item.accessControlPoint ? item.accessControlPoint.name : "" },
                    { value: item?.lastActionUtc ? (0, report_util_1.formatDateForExcel)(item.lastActionUtc) : "" },
                    { value: this.getStateDescription(item.state) },
                ],
            }));
            const sheetInfo = [
                {
                    sheetName: i18n_1.default.__({ phrase: "EXCEL-REPORT.GENERAL", locale: this._locale }),
                    headers: [
                        { phrase: "PDF-REPORT.access_point", width: 40 },
                        { phrase: "PDF-REPORT.REGION_STATE_LAST_ACTION", width: 30 },
                        { phrase: "PDF-REPORT.status", width: 30 },
                    ],
                    data: generalSheetData,
                },
            ];
            const reportInfoSheetData = {
                reportNamePhrase: "PDF-REPORT.REGION_STATE",
                filters: {
                    regionId: this._options.regionId,
                    orgUnitIds: this._options.organizationUnitIds,
                    userIds: this._options.userIds,
                    userGroupIds: this._options.userGroupIds,
                    applyOrganizationUnitFilterHierarchically: this._options.applyOrganizationUnitFilterHierarchically,
                    regionStates: this._options.state,
                    userOrganizationStatus: this._options.status,
                },
            };
            const reportFactory = new excel_report_refactory_1.ReportFactory(this._request.organizationId, this._locale);
            await reportFactory.initialize({
                requesterUserId: this._request.requesterUserId,
                userIds,
                reportInfoSheetData,
                sheetInfo,
            });
            reportFactory.appendData(sheetInfo);
            const fileObj = await reportFactory.finalize(reportInfoSheetData);
            return {
                format: app_enums_1.enums.ReportFormat.Excel,
                preparedReportId: fileObj.fileId,
                filePath: fileObj.filePath,
                notificationType: undefined,
                reportType: app_enums_1.enums.ReportCode.RegionState,
                sendFileInAttachments: undefined,
            };
        }
        catch (error) {
            app_logs_1.logger.error(`Error generating RegionStateReport: ${error.message}`, {
                stack: error.stack,
                organizationId: this._request.organizationId,
                requesterUserId: this._request.requesterUserId,
            });
            throw new Error("Failed to generate RegionStateReport");
        }
    }
    getStateDescription(state) {
        let locale = this._locale;
        switch (state) {
            case app_enums_1.enums.AntiPassbackState.Unknown:
                return i18n_1.default.__({ phrase: "PDF-REPORT.REGION_STATE_UNSET", locale });
            case app_enums_1.enums.AntiPassbackState.In:
                return i18n_1.default.__({ phrase: "PDF-REPORT.REGION_STATE_IN", locale });
            case app_enums_1.enums.AntiPassbackState.Out:
                return i18n_1.default.__({ phrase: "PDF-REPORT.REGION_STATE_OUT", locale });
            default:
                return "";
        }
    }
}
exports.RegionStateExcelReport = RegionStateExcelReport;
