"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PacsMontlyWorkSummaryExtraWorkExcelReport = void 0;
const i18n_1 = __importDefault(require("i18n"));
const luxon_1 = require("luxon");
const excel_report_refactory_1 = require("../excel.report.refactory");
const dal_manager_1 = require("../../../dal/dal.manager");
const app_logs_1 = require("../../../app.logs");
const app_enums_1 = require("../../../app.enums");
const business_common_1 = require("../../business.common");
const generator_1 = require("../generator");
class PacsMontlyWorkSummaryExtraWorkExcelReport extends generator_1.ReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._options = request.filter;
    }
    async generateReport() {
        try {
            this._hsFmt = this._locale === "tr" ? generator_1.ReportGenerator.Constants.hsFmtTr : generator_1.ReportGenerator.Constants.hsFmtEng;
            const dateFilter = this._options.dateFilter;
            if (dateFilter.group === app_enums_1.enums.MonthlySummaryReportDateFilterGroup.Annually && dateFilter.year) {
                this.begin = luxon_1.DateTime.fromObject({ year: dateFilter.year, month: 1, day: 1 }).startOf("year");
                this.end = this.begin.endOf("year");
            }
            else if (dateFilter.group === app_enums_1.enums.MonthlySummaryReportDateFilterGroup.Monthly && dateFilter.month) {
                this.begin = luxon_1.DateTime.fromObject({ year: dateFilter.year, month: dateFilter.month, day: 1 }).startOf("month");
                this.end = this.begin.endOf("month");
            }
            else if (dateFilter.group === app_enums_1.enums.MonthlySummaryReportDateFilterGroup.Whole && dateFilter.range?.startDateTime && dateFilter.range?.endDateTime) {
                this.begin = luxon_1.DateTime.fromJSDate(new Date(dateFilter.range.startDateTime)).startOf("day");
                this.end = luxon_1.DateTime.fromJSDate(new Date(dateFilter.range.endDateTime)).endOf("day");
            }
            const data = await dal_manager_1.dbManager.accessPacs2.listMonthlyExtraWorks({
                organizationId: this._request.organizationId,
                requesterUserId: this._request.requesterUserId,
                pagination: {},
                workPlanIds: this._options.userFilter?.workPlanIds,
                startMonth: this.begin.toJSDate(),
                endMonth: this.end.toJSDate(),
                idBasedUserFilter: {
                    userIds: this._options.userFilter?.userIds,
                    userGroupIds: this._options.userFilter?.userGroupIds,
                    organizationUnitIds: this._options.userFilter?.organizationUnitIds,
                    applyOrganizationUnitFilterHierarchically: this._options.userFilter?.applyOrganizationUnitFilterHierarchically,
                },
                requiredOrganizationWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
                requiredOrganizationUnitWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
            });
            const generalSheetData = data.map((item) => {
                let cells = this.getRowData(item);
                return {
                    userId: item.user.id,
                    cells,
                };
            });
            const sheetInfo = [
                {
                    sheetName: i18n_1.default.__({ phrase: "EXCEL-REPORT.GENERAL", locale: this._locale }),
                    headers: [],
                    data: generalSheetData,
                },
            ];
            const reportInfoSheetData = {
                reportNamePhrase: "REPORT.PACS_EXCEL_MONTHLY_EXTRA_HOURS",
                filters: {
                    orgUnitIds: this._options.userFilter.organizationUnitIds,
                    applyOrganizationUnitFilterHierarchically: this._options.userFilter.applyOrganizationUnitFilterHierarchically,
                    userGroupIds: this._options.userFilter.userGroupIds,
                    userIds: this._options.userFilter.userIds,
                    workplanIds: this._options.userFilter.workPlanIds,
                    dateTimeRange: {
                        startDateTime: this.begin.toJSDate(),
                        endDateTime: this.end.toJSDate(),
                    },
                },
            };
            const userIds = data.map((item) => item.user.id);
            const reportFactory = new excel_report_refactory_1.ReportFactory(this._request.organizationId, this._locale);
            await reportFactory.initialize({
                requesterUserId: this._request.requesterUserId,
                userIds,
                reportInfoSheetData,
                sheetInfo,
                sheetVerticalOffset: 3,
            });
            const ws = reportFactory.getWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.GENERAL", locale: this._locale }));
            this.initGeneralSheet(ws, 1, reportFactory.getProfileInfoOfset() + 1);
            reportFactory.setSheetIndex(i18n_1.default.__({ phrase: "EXCEL-REPORT.GENERAL", locale: this._locale }), 5);
            reportFactory.appendData(sheetInfo);
            const fileObj = await reportFactory.finalize(reportInfoSheetData);
            return {
                format: app_enums_1.enums.ReportFormat.Excel,
                preparedReportId: fileObj.fileId,
                filePath: fileObj.filePath,
                notificationType: undefined,
                reportType: app_enums_1.enums.ReportCode.PacsMonthlyWorkingHoursSummaryExtraWork,
                sendFileInAttachments: undefined,
            };
        }
        catch (error) {
            app_logs_1.logger.error(`Error generating PacsMontlyWorkSummaryExtraWorkExcelReport: ${error.message}`, {
                stack: error.stack,
                organizationId: this._request.organizationId,
                requesterUserId: this._request.requesterUserId,
            });
            throw new Error("Failed to generate PacsMontlyWorkSummaryExtraWorkExcelReport");
        }
    }
    initGeneralSheet(wsRaw, startRowIndex, startColIndex) {
        let columns = [];
        for (let i = 1; i < startColIndex; i++) {
            columns.push({ width: 30 });
        }
        let startOfMonth = this.begin;
        let endOfMonth = this.end;
        let i = 1;
        let weekCount = 1;
        let currentDay = startOfMonth;
        let dayCountOfWeek = 0;
        while (currentDay <= endOfMonth) {
            this.addInitCell11(wsRaw, startRowIndex + 1, startColIndex, i++, generator_1.ReportGenerator.Constants.Styling.AllThin);
            columns.push({ width: 10 });
            this.addInitCell11(wsRaw, startRowIndex + 2, startColIndex, currentDay.setLocale(this._locale).toLocaleString(luxon_1.DateTime.DATE_FULL), generator_1.ReportGenerator.Constants.Styling.AllThin);
            columns.push({ width: 10 });
            this.addInitCell11(wsRaw, startRowIndex + 3, startColIndex, business_common_1.BusinessCommon.GetWeekDayName(currentDay.weekday), generator_1.ReportGenerator.Constants.Styling.AllThin);
            columns.push({ width: 10 });
            dayCountOfWeek++;
            if (currentDay.weekday === 7 || currentDay.hasSame(endOfMonth, "day")) {
                this.addInitCell11(wsRaw, startRowIndex, startColIndex - dayCountOfWeek + 1, weekCount + "." + i18n_1.default.__({ phrase: "PDF-REPORT.week", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
                wsRaw.mergeCells(startRowIndex, startColIndex - dayCountOfWeek + 1, startRowIndex, startColIndex++);
                this.addInitCell11(wsRaw, startRowIndex, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.WEEKEND_EXTRA_WORK", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
                columns.push({ width: 15 });
                wsRaw.mergeCells(startRowIndex, startColIndex, startRowIndex + 3, startColIndex++);
                this.addInitCell11(wsRaw, startRowIndex, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.WEEKDAY_EXTRA_WORK", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
                columns.push({ width: 15 });
                wsRaw.mergeCells(startRowIndex, startColIndex, startRowIndex + 3, startColIndex++);
                this.addInitCell11(wsRaw, startRowIndex, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.TOTAL_EXTRA_WORK", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
                columns.push({ width: 15 });
                wsRaw.mergeCells(startRowIndex, startColIndex, startRowIndex + 3, startColIndex);
                dayCountOfWeek = 0;
                weekCount++;
            }
            currentDay = currentDay.plus({ days: 1 });
            startColIndex++;
        }
        wsRaw.autoFilter = {
            from: "A" + (startRowIndex + 2),
            to: wsRaw.getColumn(2).letter + (startRowIndex + 2),
        };
        wsRaw.columns = columns;
        app_logs_1.logger.debug("Sheet initializing ended...");
    }
    getRowData(item) {
        let cells = [];
        let startOfMonth = this.begin;
        let endOfMonth = this.end;
        let currentDay = startOfMonth;
        let weekendTotal = 0;
        let weekdayTotal = 0;
        let weekTotal = 0;
        while (currentDay < endOfMonth) {
            let dayInfo = item.extraWorkList.find((i) => luxon_1.DateTime.fromJSDate(i.date).hasSame(currentDay, "day"));
            if (dayInfo) {
                const dayOfWeek = luxon_1.DateTime.fromJSDate(dayInfo.date).weekday;
                if (dayOfWeek === app_enums_1.enums.DayOfWeek.Sunday || dayOfWeek === app_enums_1.enums.DayOfWeek.Saturday) {
                    weekendTotal += dayInfo.extraWorkDuration;
                }
                else {
                    weekdayTotal += dayInfo.extraWorkDuration;
                }
                cells.push({
                    value: dayInfo.extraWorkDuration / 1440,
                    border: generator_1.ReportGenerator.Constants.Styling.Dotted,
                    numFmt: this._hsFmt,
                });
                weekTotal += dayInfo.extraWorkDuration;
            }
            else {
                cells.push({
                    value: 0,
                    border: generator_1.ReportGenerator.Constants.Styling.Dotted,
                    numFmt: this._hsFmt,
                });
            }
            if (currentDay.weekday === 7 || currentDay.hasSame(endOfMonth, "day")) {
                cells = [
                    ...cells,
                    {
                        value: weekendTotal / 1440,
                        border: generator_1.ReportGenerator.Constants.Styling.Dotted,
                        numFmt: this._hsFmt,
                    },
                    {
                        value: weekdayTotal / 1440,
                        border: generator_1.ReportGenerator.Constants.Styling.Dotted,
                        numFmt: this._hsFmt,
                    },
                    {
                        value: weekTotal / 1440,
                        border: generator_1.ReportGenerator.Constants.Styling.Dotted,
                        numFmt: this._hsFmt,
                    },
                ];
                weekTotal = 0;
                weekdayTotal = 0;
                weekendTotal = 0;
            }
            currentDay = currentDay.plus({ days: 1 });
        }
        return cells;
    }
}
exports.PacsMontlyWorkSummaryExtraWorkExcelReport = PacsMontlyWorkSummaryExtraWorkExcelReport;
