"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.SystemFeedbackExcelReport = void 0;
const path_1 = __importDefault(require("path"));
const exceljs_1 = __importDefault(require("exceljs"));
const app_config_1 = require("../../../app.config");
const dal_manager_1 = require("../../../dal/dal.manager");
const uuid_1 = __importDefault(require("uuid"));
const i18n_1 = __importDefault(require("i18n"));
const app_logs_1 = require("../../../app.logs");
const app_enums_1 = require("../../../app.enums");
const fs_1 = __importDefault(require("fs"));
const moment_1 = __importDefault(require("moment"));
const generator_1 = require("../generator");
class SystemFeedbackExcelReport extends generator_1.ReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._fileId = uuid_1.default.v4();
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".xlsx");
        if (fs_1.default.existsSync(this._filePath)) {
            fs_1.default.unlinkSync(this._filePath);
        }
        this._wb = new exceljs_1.default.stream.xlsx.WorkbookWriter({
            filename: this._filePath,
            useStyles: true,
            useSharedStrings: true,
        });
        this._options = request.filter;
    }
    async generateReport() {
        this._organizationName = (await dal_manager_1.dbManager.accessOrganization.getOrganizationBasic(this._request.organizationId)).name;
        app_logs_1.logger.debug("Organization Name : " + this._organizationName);
        let { caption: requestedUserCaption } = await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId: this._request.organizationId, userId: this._request.requesterUserId });
        if (!requestedUserCaption) {
            throw new Error("Not found");
        }
        this._requestedUserCaption = requestedUserCaption[0].text[0];
        this._wb.creator = this._requestedUserCaption;
        app_logs_1.logger.debug("Generating feedbacks excel report...");
        app_logs_1.logger.debug("Requester User : " + this._requestedUserCaption);
        this._wsGeneral = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.GENERAL", locale: this._locale }), {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsGeneral.properties.defaultRowHeight = 15;
        this._generalRowIndex = 6;
        this.initGeneralSheet(this._wsGeneral, 5, 1);
        this.addFilterToSheet(this._wsGeneral, 1, 1);
        let option = Object.assign(this._options, {
            pagination: {
                take: undefined,
                skip: undefined,
            },
        });
        this._data = await dal_manager_1.dbManager.accessSystem.listOrganizationFeedback(this._request.organizationId, option);
        for (let index = 0; index < this._data.items.length; index++) {
            const acp = this._data.items[index];
            this.addGeneralRow(acp);
        }
        this._wsGeneral.pageSetup.printArea = "A5:" + this._wsGeneral.getColumn(this._wsGeneral.columns.length).letter + this._generalRowIndex;
        this._wsGeneral.pageSetup.printTitlesRow = "5:5";
        this._wsGeneral.commit();
        await this._wb.commit();
        app_logs_1.logger.debug("File is ready -> " + this._filePath);
        return {
            format: app_enums_1.enums.ReportFormat.Excel,
            preparedReportId: this._fileId,
            filePath: this._filePath,
            notificationType: undefined,
            reportType: app_enums_1.enums.ReportCode.SystemFeedback,
            sendFileInAttachments: undefined,
        };
    }
    initGeneralSheet(wsRaw, startRowIndex, startColIndex) {
        let columns = [];
        const data = [
            {
                phrase: "FEEDBACK.TITLE.NUMBER",
                cellWidth: 20
            },
            {
                phrase: "FEEDBACK.TITLE.REQUESTDATE",
                cellWidth: 30
            },
            {
                phrase: "FEEDBACK.TITLE.REQUESTUSER",
                cellWidth: 30
            },
            {
                phrase: "FEEDBACK.TITLE.STATUS",
                cellWidth: 20
            },
            {
                phrase: "FEEDBACK.TITLE.TYPE",
                cellWidth: 20
            },
            {
                phrase: "FEEDBACK.TITLE.TITLE",
                cellWidth: 30
            },
            {
                phrase: "FEEDBACK.TITLE.NOTE",
                cellWidth: 50
            }
        ];
        startColIndex = this.addInitCellsToRow(data, columns, startColIndex, startRowIndex, wsRaw);
        wsRaw.autoFilter = {
            from: "A" + startRowIndex,
            to: wsRaw.getColumn(columns.length).letter + startRowIndex,
        };
        wsRaw.columns = columns;
        app_logs_1.logger.debug("Sheet initializing ended...");
    }
    addGeneralRow(item) {
        let colIndex = 1;
        const rowData = [
            item.caption.toString(),
            (0, moment_1.default)(item.createdAt).locale(this._locale).format("LL") + " " + (0, moment_1.default)(item.createdAt).locale(this._locale).format("HH:mm:ss"),
            item.requestUser.fullname,
            i18n_1.default.__({ phrase: "FEEDBACK.STATUS." + item.status, locale: this._locale }),
            i18n_1.default.__({ phrase: "FEEDBACK.TYPE." + item.type, locale: this._locale }),
            item.title,
            item.body
        ];
        colIndex = this.addCellsToRow(rowData, colIndex, this._generalRowIndex, this._wsGeneral);
        this._wsGeneral.getRow(this._generalRowIndex).commit();
        this._generalRowIndex++;
    }
    addFilterToSheet(ws, startRowIndex, startColIndex) {
        this.addFilterCell11(ws, startRowIndex, startColIndex, this._organizationName + " " + i18n_1.default.__({ phrase: "EXCEL-REPORT.FEEDBACK_REPORT", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin, {
            bold: true,
        });
        ws.mergeCells(startRowIndex, startColIndex, startRowIndex, startColIndex + 1);
    }
}
exports.SystemFeedbackExcelReport = SystemFeedbackExcelReport;
