"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.EntryByUnitReport = void 0;
const luxon_1 = require("luxon");
const dal_manager_1 = require("../../../dal/dal.manager");
const i18n_1 = __importDefault(require("i18n"));
const app_logs_1 = require("../../../app.logs");
const app_enums_1 = require("../../../app.enums");
const generator_1 = require("../../../business/report/generator");
const excel_report_refactory_1 = require("../excel.report.refactory");
class EntryByUnitReport extends generator_1.CustomReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._request = request;
        this._filter = request.filter;
        this._dateCount = 0;
    }
    async generateReport() {
        try {
            const data = await dal_manager_1.dbManager.accessReport.getEntryByUnitData(this._filter);
            let headers = [
                {
                    value: "",
                    width: 40,
                },
            ];
            data.items[0].dayResults.forEach((day) => {
                headers.push({
                    value: luxon_1.DateTime.fromISO(day.date.toString()).setLocale(this._locale).toLocaleString(luxon_1.DateTime.DATE_FULL),
                    width: 20,
                });
                this._dateCount++;
            });
            const generalSheetData = data.items.map((item) => {
                let cells = [
                    {
                        value: item.name === "totals" ? i18n_1.default.__({ phrase: "EXCEL-REPORT.SUM", locale: this._locale }) : this.generateLevelString(item.rootPath?.length) + item.name,
                    },
                ];
                item.dayResults.forEach((day) => {
                    cells.push({
                        value: (day.count === 0 ? "" : day.count).toString(),
                        alignment: {
                            horizontal: "center",
                        },
                    });
                });
                cells.push({
                    value: (Math.round((item.total / this._dateCount + Number.EPSILON) * 100) / 100).toString(),
                    alignment: {
                        horizontal: "center",
                    },
                });
                return {
                    cells,
                };
            });
            headers.push({
                phrase: "EXCEL-REPORT.AVG",
                width: 20,
            });
            const sheetInfo = [
                {
                    sheetName: i18n_1.default.__({ phrase: "EXCEL-REPORT.GENERAL", locale: this._locale }),
                    headers,
                    data: generalSheetData,
                },
            ];
            const reportInfoSheetData = {
                reportNamePhrase: "EXCEL-REPORT.ENTRY_BY_UNIT_REPORT",
                filters: {
                    regionIds: this._filter.regionIds,
                    orgUnitIds: this._filter.organizationUnitIds,
                    userGroupIds: this._filter.userGroupIds,
                    accessControlPointIds: this._filter.accessControlPointIds,
                    dateRange: {
                        startDate: new Date(this._filter.dateRange.startDateTime),
                        endDate: new Date(this._filter.dateRange.endDateTime),
                    },
                },
            };
            const reportFactory = new excel_report_refactory_1.ReportFactory(this._request.organizationId, this._locale);
            await reportFactory.initialize({
                requesterUserId: this._request.requesterUserId,
                userIds: null,
                reportInfoSheetData,
                sheetInfo,
            });
            reportFactory.appendData(sheetInfo);
            const fileObj = await reportFactory.finalize(reportInfoSheetData);
            return {
                format: app_enums_1.enums.ReportFormat.Excel,
                preparedReportId: fileObj.fileId,
                filePath: fileObj.filePath,
                notificationType: undefined,
                reportType: app_enums_1.enums.ReportCode.EntryByUnit,
                sendFileInAttachments: undefined,
            };
        }
        catch (error) {
            app_logs_1.logger.error(`Error generating EntryByUnitReport: ${error.message}`, {
                stack: error.stack,
                organizationId: this._request.organizationId,
                requesterUserId: this._request.requesterUserId,
            });
            throw new Error("Failed to generate EntryByUnitReport");
        }
    }
    generateLevelString(level) {
        return "   ".repeat(level - 1);
    }
}
exports.EntryByUnitReport = EntryByUnitReport;
