"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.EmployeesPermissionUsageReport = void 0;
const luxon_1 = require("luxon");
const dal_manager_1 = require("../../../dal/dal.manager");
const i18n_1 = __importDefault(require("i18n"));
const app_logs_1 = require("../../../app.logs");
const app_enums_1 = require("../../../app.enums");
const generator_1 = require("../generator");
const business_pacs_claim_1 = require("../../pacs/business.pacs.claim");
const excel_report_refactory_1 = require("../excel.report.refactory");
class EmployeesPermissionUsageReport extends generator_1.ReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._filter = request.filter;
        this._reportFactory = new excel_report_refactory_1.ReportFactory(request.organizationId, locale);
    }
    addRow(rows, ws, row) {
        let colIndex = 1;
        let rowIndex = row;
        switch (rows.ppermissionInfo.type) {
            case app_enums_1.enums.PredefinedPPermissionType.Annual:
                for (const item of rows.items) {
                    if (item.permissionClaimInfo) {
                        this.setDataCell11({
                            rowIndex: rowIndex,
                            colIndex: colIndex++,
                            value: item.user.fullname + "\n" + item.user.uniqueId + "\n",
                            ws: ws,
                            border: generator_1.ReportGenerator.Constants.Styling.Dotted,
                            newLineCount: 4,
                        });
                        let rowData = [
                            item.permissionClaimInfo.startDate.getFullYear(),
                            this._reportFactory.formatDateTime(item.permissionClaimInfo.startDate, this._locale),
                            this._reportFactory.formatDateTime(item.permissionClaimInfo.endDate, this._locale),
                            item.permissionClaimInfo.leftFromPreviousPeriod,
                            item.permissionClaimInfo.newlyClaimed,
                            item.permissionClaimInfo.usage,
                            item.permissionClaimInfo.remaining,
                            this._reportFactory.formatDateTime(item.permissionClaimInfo.nextPeriodStartDate, this._locale),
                            item.permissionClaimInfo.nextPeriodClaim,
                        ];
                        colIndex = this.addCellsToRow(rowData, colIndex, rowIndex, ws);
                        rowIndex++;
                        colIndex = 1;
                    }
                }
                break;
            default:
                for (const item of rows.items) {
                    if (item.permissionClaimInfo) {
                        this.setDataCell11({
                            rowIndex: rowIndex,
                            colIndex: colIndex++,
                            value: item.user.fullname + "\n" + item.user.uniqueId + "\n",
                            ws: ws,
                            border: generator_1.ReportGenerator.Constants.Styling.Dotted,
                            newLineCount: 4,
                        });
                        let rowData = [
                            item.permissionClaimInfo.startDate.getFullYear(),
                            this._reportFactory.formatDateTime(item.permissionClaimInfo.startDate, this._locale),
                            this._reportFactory.formatDateTime(item.permissionClaimInfo.endDate, this._locale),
                            item.permissionClaimInfo.newlyClaimed,
                            item.permissionClaimInfo.usage,
                            item.permissionClaimInfo.remaining,
                        ];
                        colIndex = this.addCellsToRow(rowData, colIndex, rowIndex, ws);
                        rowIndex++;
                        colIndex = 1;
                    }
                }
                break;
        }
        rows.items.forEach((item) => {
            if (!item.permissionClaimInfo) {
                this.setDataCell11({
                    ws: ws,
                    rowIndex: rowIndex,
                    colIndex: colIndex++,
                    value: item.user.fullname + "\n" + item.user.uniqueId + "\n",
                    border: generator_1.ReportGenerator.Constants.Styling.Dotted,
                    newLineCount: 4,
                });
                this.setDataCell11({
                    ws: ws,
                    rowIndex: rowIndex,
                    colIndex: colIndex,
                    value: i18n_1.default.__({ phrase: "ERRORS.PACS.EMPLOYEESTARTDATEMISSING", locale: this._locale }),
                    border: generator_1.ReportGenerator.Constants.Styling.Dotted,
                    alignment: "center",
                });
                if (this._permissionType.type === app_enums_1.enums.PredefinedPPermissionType.Annual) {
                    ws.mergeCells(rowIndex, colIndex, rowIndex, colIndex + 8);
                    rowIndex++;
                    colIndex = 1;
                }
                else {
                    ws.mergeCells(rowIndex, colIndex, rowIndex, colIndex + 5);
                    rowIndex++;
                    colIndex = 1;
                }
            }
        });
    }
    getHeaders() {
        let translationPermissionPhrase = "";
        if (this._permissionType.isDailyScheduled === true) {
            translationPermissionPhrase = "  (" + i18n_1.default.__({ phrase: "EXCEL-REPORT.DAY", locale: this._locale }) + ")";
        }
        else {
            translationPermissionPhrase = "  (" + i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR", locale: this._locale }) + ")";
        }
        let data = [
            {
                phrase: "EXCEL-REPORT.USER_LIST",
                width: 25,
            },
            {
                phrase: "EXCEL-REPORT.PERIOD",
                width: 25,
            },
            {
                phrase: "EXCEL-REPORT.PERIOD_START_DATE",
                width: 25,
            },
            {
                phrase: "EXCEL-REPORT.PERIOD_END_DATE",
                width: 25,
            },
        ];
        switch (this._permissionType.type) {
            case app_enums_1.enums.PredefinedPPermissionType.Annual:
                data = [
                    ...data,
                    {
                        value: i18n_1.default.__({ phrase: "EXCEL-REPORT.FROM_PREVIOUS", locale: this._locale }) + " " + translationPermissionPhrase,
                        width: 25,
                    },
                    {
                        value: i18n_1.default.__({ phrase: "EXCEL-REPORT.NEWLY_CLAIMED", locale: this._locale }) + " " + translationPermissionPhrase,
                        width: 25,
                    },
                    {
                        value: i18n_1.default.__({ phrase: "EXCEL-REPORT.USED", locale: this._locale }) + " " + translationPermissionPhrase,
                        width: 25,
                    },
                    {
                        value: i18n_1.default.__({ phrase: "EXCEL-REPORT.REMAINING", locale: this._locale }) + " " + translationPermissionPhrase,
                        width: 25,
                    },
                    {
                        value: i18n_1.default.__({ phrase: "EXCEL-REPORT.NEXT_PERIOD", locale: this._locale }) + " " + translationPermissionPhrase,
                        width: 25,
                    },
                    {
                        value: i18n_1.default.__({ phrase: "EXCEL-REPORT.NEXT_PERIOD_CLAIM", locale: this._locale }) + " " + translationPermissionPhrase,
                        width: 25,
                    },
                ];
                break;
            default:
                data = [
                    ...data,
                    {
                        value: i18n_1.default.__({ phrase: "EXCEL-REPORT.NEWLY_CLAIMED", locale: this._locale }) + " " + translationPermissionPhrase,
                        width: 25,
                    },
                    {
                        value: i18n_1.default.__({ phrase: "EXCEL-REPORT.USED", locale: this._locale }) + " " + translationPermissionPhrase,
                        width: 25,
                    },
                    {
                        value: i18n_1.default.__({ phrase: "EXCEL-REPORT.REMAINING", locale: this._locale }) + " " + translationPermissionPhrase,
                        width: 25,
                    },
                ];
                break;
        }
        return data;
    }
    async createExcelSheet(result) {
        const headers = this.getHeaders();
        const sheetName = i18n_1.default.__({ phrase: "EXCEL-REPORT.EMPLOYEES_PERMISSION_USAGE_REPORT", locale: this._locale });
        const sheetInfo = [
            {
                sheetName,
                headers,
            },
        ];
        let filters = {
            userIds: this._filter.userIds,
            orgUnitIds: this._filter.organizationUnitIds,
            applyOrganizationUnitFilterHierarchically: this._filter.applyOrganizationUnitFilterHierarchically,
            userGroupIds: this._filter.userGroupIds,
            permissionTypeName: this._permissionType.name,
        };
        const reportInfoSheetData = {
            reportNamePhrase: "EXCEL-REPORT.EMPLOYEES_PERMISSION_USAGE_REPORT",
            filters,
        };
        await this._reportFactory.initialize({
            requesterUserId: this._request.requesterUserId,
            reportInfoSheetData,
            userIds: null,
            sheetInfo,
        });
        this.addRow(result, this._reportFactory.getWorksheet(sheetName), this._reportFactory.getSheetIndex(sheetName));
        return await this._reportFactory.finalize(reportInfoSheetData);
    }
    async fetchUsersPermissionUsageData() {
        return await dal_manager_1.dbManager.systemTransaction(async (trx) => {
            this._permissionType = await dal_manager_1.dbManager.accessPacs.getPPermissionType(this._request.organizationId, this._filter.ppermissionTypeId);
            let result = {
                ppermissionInfo: null,
                items: [],
            };
            const users = await dal_manager_1.dbManager.accessUser.getFilteredUserIds(this._request.organizationId, {
                pagination: {
                    skip: 0,
                    take: 1000,
                },
                ppermissionTypeId: this._filter.ppermissionTypeId,
                userGroupIds: this._filter.userGroupIds,
                organizationUnitIds: this._filter.organizationUnitIds,
                userIds: this._filter.userIds,
                applyOrganizationUnitFilterHierarchically: this._filter.applyOrganizationUnitFilterHierarchically,
                userOrganizationStatus: this._filter.userOrganizationStatus,
            }, trx);
            const getUserCaptionsLines = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(this._request.organizationId, users.items.map((elem) => elem.id));
            const ppermissionTypeUsage = await (0, business_pacs_claim_1.getUsersPPermissionsTypeUsageOfBetweenDates)(this._request.organizationId, users.items.map((elem) => elem.id), this._filter.ppermissionTypeId, trx);
            let factor = ppermissionTypeUsage.permissionTypeInfo.isDailyScheduled
                ? app_enums_1.enums.PPermissionUsageFactor.DailyPermissionUsageFactor
                : app_enums_1.enums.PPermissionUsageFactor.HourlyPermissionUsageFactor;
            result.ppermissionInfo = ppermissionTypeUsage.permissionTypeInfo;
            for (const userPeriods of ppermissionTypeUsage.userPeriods) {
                const user = users.items.find((elem) => elem.id === userPeriods.userId);
                const basicUserInfo = {
                    id: user.id,
                    fullname: user.fullname,
                    uniqueId: user.uniqueId,
                    userCaptions: getUserCaptionsLines.find((elem) => elem.id === user.id).captionLines,
                    orgTitle: user.orgTitle,
                };
                if (ppermissionTypeUsage.permissionTypeInfo.type === app_enums_1.enums.PredefinedPPermissionType.Annual) {
                    if (userPeriods.periods.length > 0) {
                        const lastPeriod = userPeriods.periods[0];
                        result.items.push({
                            user: basicUserInfo,
                            permissionClaimInfo: {
                                usage: lastPeriod.usage !== null ? lastPeriod.usage / factor : null,
                                startDate: lastPeriod.periodStart,
                                endDate: lastPeriod.periodEnd,
                                newlyClaimed: lastPeriod.claim !== null ? lastPeriod.claim / factor : null,
                                leftFromPreviousPeriod: lastPeriod.fromPreviousPeriod !== null ? lastPeriod.fromPreviousPeriod / factor : null,
                                remaining: userPeriods.periods[0].remaining !== null ? userPeriods.periods[0].remaining / factor : null,
                                nextPeriodStartDate: luxon_1.DateTime.fromJSDate(new Date(userPeriods.periods[0].periodStart))
                                    .plus({ years: 1, days: userPeriods.periods[0].unpaidLeaveOffset })
                                    .toJSDate(),
                                nextPeriodClaim: userPeriods.periods[0].nextClaim,
                                ppermissionClaimProfile: ppermissionTypeUsage.ppermissionClaimProfile,
                                manuallyRemainedAnnualPermission: user.manuallyRemainedAnnualPermission,
                            },
                        });
                    }
                    else {
                        result.items.push({
                            user: basicUserInfo,
                        });
                    }
                }
                else {
                    if (userPeriods.periods.length > 0) {
                        const lastPeriod = userPeriods.periods[0];
                        result.items.push({
                            user: basicUserInfo,
                            permissionClaimInfo: {
                                usage: lastPeriod.usage !== null ? lastPeriod.usage / factor : null,
                                startDate: lastPeriod.periodStart,
                                endDate: lastPeriod.periodEnd,
                                newlyClaimed: lastPeriod.claim !== null ? lastPeriod.claim / factor : null,
                                leftFromPreviousPeriod: lastPeriod.fromPreviousPeriod !== null ? lastPeriod.fromPreviousPeriod / factor : null,
                                remaining: userPeriods.periods[0].remaining !== null ? userPeriods.periods[0].remaining / factor : null,
                                manuallyRemainedAnnualPermission: user.manuallyRemainedAnnualPermission,
                            },
                        });
                    }
                    else {
                        result.items.push({
                            user: basicUserInfo,
                        });
                    }
                }
            }
            result.items.sort((a, b) => a.user.fullname.toUpperCase().localeCompare(b.user.fullname.toUpperCase()));
            return result;
        });
    }
    async generateReport() {
        try {
            const result = await this.fetchUsersPermissionUsageData();
            const fileObj = await this.createExcelSheet(result);
            return {
                format: app_enums_1.enums.ReportFormat.Excel,
                preparedReportId: fileObj.fileId,
                filePath: fileObj.filePath,
                notificationType: undefined,
                reportType: app_enums_1.enums.ReportCode.PermissionClaimUsagesForEmployees,
                sendFileInAttachments: undefined,
            };
        }
        catch (error) {
            app_logs_1.logger.error(`Error generating EmployeePermissionUsageReport: ${error.message}`, {
                stack: error.stack,
                organizationId: this._request.organizationId,
                requesterUserId: this._request.requesterUserId,
            });
            throw new Error("Failed to generate EmployeePermissionUsageReport");
        }
    }
}
exports.EmployeesPermissionUsageReport = EmployeesPermissionUsageReport;
