"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.EmployeePermissionUsageReport = void 0;
const luxon_1 = require("luxon");
const dal_manager_1 = require("../../../dal/dal.manager");
const i18n_1 = __importDefault(require("i18n"));
const app_logs_1 = require("../../../app.logs");
const app_enums_1 = require("../../../app.enums");
const business_pacs_claim_1 = require("../../pacs/business.pacs.claim");
const moment_range_1 = require("moment-range");
const generator_1 = require("../generator");
const excel_report_refactory_1 = require("../excel.report.refactory");
class EmployeePermissionUsageReport extends generator_1.ReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._options = request.filter;
    }
    round(value, decimals) {
        const factor = Math.pow(10, decimals);
        return Math.round(value * factor) / factor;
    }
    async getUserEmploymentStartDate() {
        const userEmploymentInfo = await dal_manager_1.dbManager.accessUser.getUserOrganizationProfileEmploymentInfo(this._request.organizationId, this._options.userId);
        if (userEmploymentInfo.employmentStartUtc)
            return userEmploymentInfo.employmentStartUtc;
        const modules = await dal_manager_1.dbManager.accessOrganization.listOrganizationModules(this._request.organizationId);
        const pacsModule = modules.find((m) => m.module === app_enums_1.enums.ArmonApplicationModule.PACS);
        if (!pacsModule) {
            app_logs_1.logger.debug("PACS module is not enabled");
            throw new Error("PACS module is not enabled");
        }
        return pacsModule.enableUtc || new Date();
    }
    getHeaders() {
        const headers = [
            { phrase: "EXCEL-REPORT.PERIOD", width: 25 },
            { phrase: "EXCEL-REPORT.PERIOD_START_DATE", width: 25 },
            { phrase: "EXCEL-REPORT.PERIOD_END_DATE", width: 25 },
        ];
        const commonHeaders = [
            { phrase: "EXCEL-REPORT.NEWLY_CLAIMED", width: 25 },
            { phrase: "EXCEL-REPORT.USED", width: 25 },
            { phrase: "EXCEL-REPORT.REMAINING", width: 25 },
        ];
        if (this._permissionType.type === app_enums_1.enums.PredefinedPPermissionType.Annual) {
            headers.push({ phrase: "EXCEL-REPORT.FROM_PREVIOUS", width: 25 });
        }
        return headers.concat(commonHeaders);
    }
    async fetchPermissionUsageData() {
        var _a;
        this._permissionTypes = (await dal_manager_1.dbManager.accessPacs.listPPermissionTypes(this._request.organizationId, {
            take: 1000,
            skip: 0,
        })).items;
        this._permissionType = this._permissionTypes.find((permissionType) => permissionType.id === this._options.ppermissionTypeId);
        if (!this._options.startDay) {
            this._options.startDay = await this.getUserEmploymentStartDate();
        }
        (_a = this._options).endDay || (_a.endDay = new Date());
        const usage = await (0, business_pacs_claim_1.getPPermissionTypeUsageOfUserBetweenDates)(this._request.organizationId, this._options.userId, this._permissionType.id, this._options.startDay, this._options.endDay);
        return {
            pagination: { take: undefined, skip: undefined, total: usage.periods.length },
            ppermissionInfo: this._permissionType,
            items: this.buildReportData(usage),
        };
    }
    buildReportData(usage) {
        const factor = this._permissionType.isDailyScheduled ? app_enums_1.enums.PPermissionUsageFactor.DailyPermissionUsageFactor : app_enums_1.enums.PPermissionUsageFactor.HourlyPermissionUsageFactor;
        const filterRange = new moment_range_1.DateRange(this._options.startDay, this._options.endDay);
        return usage.periods
            .sort((a, b) => a.startDate - b.startDate)
            .filter((period) => new moment_range_1.DateRange(period.periodStart, period.periodEnd).overlaps(filterRange))
            .map((period) => ({
            usage: period.usage !== null ? period.usage / factor : null,
            rollOverToNextPeriod: period.remaining !== null ? period.remaining / factor : null,
            newlyClaimed: period.claim !== null ? period.claim / factor : null,
            leftFromPreviousPeriod: period.fromPreviousPeriod !== null ? period.fromPreviousPeriod / factor : null,
            startDate: period.periodStart,
            endDate: period.periodEnd,
        }));
    }
    async createExcelSheet(result) {
        const headers = this.getHeaders();
        const sheetInfo = [
            {
                sheetName: i18n_1.default.__({ phrase: "EXCEL-REPORT.EMPLOYEE_PERMISSION_USAGE_REPORT", locale: this._locale }),
                headers,
                data: result.items.map((row) => {
                    let rowData = [];
                    switch (this._permissionType.type) {
                        case app_enums_1.enums.PredefinedPPermissionType.Annual:
                            rowData = [
                                { value: row.startDate ? row.startDate.getFullYear() : "" },
                                { value: row.startDate ? luxon_1.DateTime.fromJSDate(row.startDate).setLocale(this._locale).toFormat("d MMMM yyyy") : "" },
                                { value: row.endDate ? luxon_1.DateTime.fromJSDate(row.endDate).setLocale(this._locale).toFormat("d MMMM yyyy") : "" },
                                { value: row.leftFromPreviousPeriod ? `${row.leftFromPreviousPeriod} ${i18n_1.default.__({ phrase: "EXCEL-REPORT.DAY", locale: this._locale })}` : "" },
                                { value: row.newlyClaimed ? `${row.newlyClaimed} ${i18n_1.default.__({ phrase: "EXCEL-REPORT.DAY", locale: this._locale })}` : "" },
                                { value: row.usage ? `${row.usage} ${i18n_1.default.__({ phrase: "EXCEL-REPORT.DAY", locale: this._locale })}` : "" },
                                { value: row.rollOverToNextPeriod ? `${row.rollOverToNextPeriod} ${i18n_1.default.__({ phrase: "EXCEL-REPORT.DAY", locale: this._locale })}` : "" },
                            ];
                            break;
                        case app_enums_1.enums.PredefinedPPermissionType.Casual:
                            rowData = [
                                { value: row.startDate ? row.startDate.getFullYear() : "" },
                                { value: row.startDate ? luxon_1.DateTime.fromJSDate(row.startDate).setLocale(this._locale).toFormat("d MMMM yyyy") : "" },
                                { value: row.endDate ? luxon_1.DateTime.fromJSDate(row.endDate).setLocale(this._locale).toFormat("d MMMM yyyy") : "" },
                                { value: row.newlyClaimed ? `${this.round(row.newlyClaimed, 1)} ${i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR", locale: this._locale })}` : "" },
                                { value: row.usage ? `${this.round(row.usage, 1)} ${i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR", locale: this._locale })}` : "" },
                                {
                                    value: row.rollOverToNextPeriod
                                        ? `${this.round(row.rollOverToNextPeriod, 1)} ${i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR", locale: this._locale })}`
                                        : "",
                                },
                            ];
                            break;
                        case app_enums_1.enums.PredefinedPPermissionType.Other:
                            rowData = [
                                {
                                    value: this._permissionType.name === "Görev İzni"
                                        ? row.startDate
                                            ? luxon_1.DateTime.fromJSDate(row.startDate).toFormat("MMMM yyyy")
                                            : ""
                                        : row.startDate
                                            ? row.startDate.getFullYear()
                                            : "",
                                },
                                { value: row.startDate ? luxon_1.DateTime.fromJSDate(row.startDate).setLocale(this._locale).toFormat("d MMMM yyyy") : "" },
                                { value: row.endDate ? luxon_1.DateTime.fromJSDate(row.endDate).setLocale(this._locale).toFormat("d MMMM yyyy") : "" },
                            ];
                            if (!this._permissionType.isDailyScheduled) {
                                rowData.push({ value: row.newlyClaimed ? `${this.round(row.newlyClaimed, 1)} ${i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR", locale: this._locale })}` : "" }, { value: row.usage ? `${this.round(row.usage, 1)} ${i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR", locale: this._locale })}` : "" }, {
                                    value: row.rollOverToNextPeriod
                                        ? `${this.round(row.rollOverToNextPeriod, 1)} ${i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR", locale: this._locale })}`
                                        : "",
                                });
                            }
                            else {
                                rowData.push({ value: row.newlyClaimed ? `${row.newlyClaimed} ${i18n_1.default.__({ phrase: "EXCEL-REPORT.DAY", locale: this._locale })}` : "" }, { value: row.usage ? `${row.usage} ${i18n_1.default.__({ phrase: "EXCEL-REPORT.DAY", locale: this._locale })}` : "" }, { value: row.rollOverToNextPeriod ? `${row.rollOverToNextPeriod} ${i18n_1.default.__({ phrase: "EXCEL-REPORT.DAY", locale: this._locale })}` : "" });
                            }
                            break;
                    }
                    return {
                        cells: rowData,
                    };
                }),
            },
        ];
        let filters = {
            userIds: [this._options.userId],
            permissionTypeName: this._permissionType.name,
            dateRange: {
                startDate: this._options.startDay,
                endDate: this._options.endDay,
            },
        };
        if (this._permissionType.type === app_enums_1.enums.PredefinedPPermissionType.Annual) {
            filters["nextPeriod"] = luxon_1.DateTime.fromJSDate(result.annualPermissionCurrentStatus?.nextPeriodStartDate).setLocale(this._locale).toFormat("d MMMM yyyy");
            filters["remainedAnnualPermission"] = result.annualPermissionCurrentStatus?.leftOverPermissionCount;
        }
        const reportInfoSheetData = {
            reportNamePhrase: "EXCEL-REPORT.EMPLOYEE_PERMISSION_USAGE_REPORT",
            filters,
        };
        const reportFactory = new excel_report_refactory_1.ReportFactory(this._request.organizationId, this._locale);
        await reportFactory.initialize({
            requesterUserId: this._request.requesterUserId,
            reportInfoSheetData,
            userIds: null,
            sheetInfo,
        });
        reportFactory.appendData(sheetInfo);
        return await reportFactory.finalize(reportInfoSheetData);
    }
    async generateReport() {
        try {
            const result = await this.fetchPermissionUsageData();
            const fileObj = await this.createExcelSheet(result);
            return {
                format: app_enums_1.enums.ReportFormat.Excel,
                preparedReportId: fileObj.fileId,
                filePath: fileObj.filePath,
                notificationType: undefined,
                reportType: app_enums_1.enums.ReportCode.EmployeePermissionUsage,
                sendFileInAttachments: undefined,
            };
        }
        catch (error) {
            app_logs_1.logger.error(`Error generating EmployeePermissionUsageReport: ${error.message}`, {
                stack: error.stack,
                organizationId: this._request.organizationId,
                requesterUserId: this._request.requesterUserId,
            });
            throw new Error("Failed to generate EmployeePermissionUsageReport");
        }
    }
}
exports.EmployeePermissionUsageReport = EmployeePermissionUsageReport;
