"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.DailyFirstEntranceLastExitReport = void 0;
const path_1 = __importDefault(require("path"));
const exceljs_1 = __importDefault(require("exceljs"));
const moment_1 = __importDefault(require("moment"));
const app_config_1 = require("../../../app.config");
const dal_manager_1 = require("../../../dal/dal.manager");
const dal_constants_1 = require("../../../dal/dal.constants");
const uuid_1 = __importDefault(require("uuid"));
const i18n_1 = __importDefault(require("i18n"));
const app_logs_1 = require("../../../app.logs");
const business_pacs_report_1 = require("../../pacs/business.pacs.report");
const app_enums_1 = require("../../../app.enums");
const generator_1 = require("../generator");
class DailyFirstEntranceLastExitReport extends generator_1.ReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._wb = new exceljs_1.default.Workbook();
        this._options = this._request.filter;
    }
    async generateReport() {
        let data = await (0, business_pacs_report_1.getDailySummaryReport)(this._request.organizationId, {
            date: this._options.date,
            organizationUnitIds: this._options.organizationUnitIds,
            applyOrganizationUnitFilterHierarchically: this._options.applyOrganizationUnitFilterHierarchically,
            userGroupIds: this._options.userGroupIds,
            userIds: this._options.userIds,
            workPlanIds: this._options.workPlanIds,
            status: this._options.status,
            sortingMethod: this._options.sortingMethod,
            workStatusFilter: this._options.workStatusFilter,
            pagination: {
                take: app_config_1.appConfig.reportExportRowLimit + 1,
                skip: 0,
            },
            hasOrganizationWide: this._options.hasOrganizationWide,
            permittedUnitIds: this._options.permittedUnitIds,
        }, this._request.requesterUserId, this._options.hasOrganizationWide, this._options.permittedUnitIds);
        this._items = data.items;
        this._workplans = data.workPlans;
        this._filePath = await this.generateDailyFirstInLastOutReport();
        return {
            format: app_enums_1.enums.ReportFormat.Excel,
            preparedReportId: this._fileId,
            filePath: this._filePath,
            notificationType: undefined,
            reportType: app_enums_1.enums.ReportCode.PacsFirstInLastOut,
            sendFileInAttachments: undefined,
        };
    }
    async generateDailyFirstInLastOutReport() {
        this._organizationBasic = await dal_manager_1.dbManager.accessOrganization.getOrganizationBasic(this._request.organizationId);
        this._wsGeneral = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.GENERAL", locale: this._locale }), {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsGeneral.properties.defaultRowHeight = 15;
        this._generalRowIndex = 6;
        this._wsGeneral.headerFooter.oddHeader =
            "&L" +
                this._organizationBasic.name +
                " " +
                i18n_1.default.__({ phrase: "EXCEL-REPORT.DAILY_FIRST_IN_LAST_OUT", locale: this._locale }) +
                "\n" +
                "&R" +
                i18n_1.default.__({ phrase: "EXCEL-REPORT.DATE", locale: this._locale }) +
                " : " +
                (0, moment_1.default)(this._options.date).format("DD/MM/YYYY") +
                "\n";
        this._wsGeneral.headerFooter.oddFooter = i18n_1.default.__({ phrase: "EXCEL-REPORT.PAGE", locale: this._locale }) + " &P / &N" + "&R" + "&D  &T";
        this.initGeneralSheet(this._wsGeneral, 5, 1);
        let { caption: requestedUserCaption } = await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId: this._request.organizationId, userId: this._request.requesterUserId });
        if (!requestedUserCaption) {
            throw new Error("Not found");
        }
        this._requestedUserCaption = requestedUserCaption[0].text[0];
        this._wb.creator = this._requestedUserCaption;
        app_logs_1.logger.debug("Generating report...");
        for (const item of this._items) {
            this.addGeneralRow(item);
        }
        let filters = [];
        if (this._options.status) {
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.USER_STATUS", locale: this._locale }) +
                " : " +
                i18n_1.default.__({ phrase: "EXCEL-REPORT." + dal_constants_1.DalConstants.IdentityStatusType[this._options.status].toUpperCase(), locale: this._locale }));
        }
        if (this._options.workStatusFilter) {
            this._options.workStatusFilter.forEach((workStatusFilter) => {
                filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.WORK_STATUS", locale: this._locale }) +
                    " : " +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT." + dal_constants_1.DalConstants.WorkStatusFilter[workStatusFilter].toUpperCase(), locale: this._locale }));
            });
        }
        if (this._options.organizationUnitIds && this._options.organizationUnitIds.length > 0) {
            let organizationUnitNames = await dal_manager_1.dbManager.accessOrganizationUnit.getIdNamePairsOfOrganizationUnits(this._request.organizationId, this._options.organizationUnitIds);
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.ORGANIZATION_UNITS", locale: this._locale }) + " : " + organizationUnitNames.map((oun) => oun.name).join(","));
            if (this._options.applyOrganizationUnitFilterHierarchically) {
                filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.INCLUDE_DESCENDANTS", locale: this._locale }) + " : " + i18n_1.default.__({ phrase: "EXCEL-REPORT.YES", locale: this._locale }));
            }
        }
        if (this._options.userGroupIds && this._options.userGroupIds.length > 0) {
            let userGroupNames = await dal_manager_1.dbManager.accessUserGroup.listUserGroupsByIds(this._request.organizationId, this._options.userGroupIds);
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.USER_GROUPS", locale: this._locale }) + " : " + userGroupNames.map((ugn) => ugn.name).join(","));
        }
        if (this._options.workPlanIds && this._options.workPlanIds.length > 0) {
            let workplanNames = this._workplans.filter((wp) => this._options.workPlanIds.includes(wp.id)).map((wp) => wp.name);
            filters.push(i18n_1.default.__({ phrase: "REPORT.WORKPLANS", locale: this._locale }) + " : " + workplanNames.join(","));
        }
        if (this._options.userIds && this._options.userIds.length > 0) {
            let userNames = this._items.filter((i) => this._options.userIds.includes(i.user.id)).map((i) => i.user.fullname);
            filters.push(i18n_1.default.__({ phrase: "PDF-REPORT.IDENTITES", locale: this._locale }) + " : " + userNames.join(","));
        }
        app_logs_1.logger.debug("\n\nApplied Filters:\n");
        app_logs_1.logger.debug(filters.join("\r\n"));
        app_logs_1.logger.debug("\n\n");
        this.addFilterToSheet(this._wsGeneral, 1, 1, filters.join("\r\n"), filters.length);
        this._wsGeneral.pageSetup.printArea = "A6:" + this._wsGeneral.getColumn(this._wsGeneral.columns.length).letter + this._generalRowIndex;
        this._wsGeneral.pageSetup.printTitlesRow = "5:5";
        this._fileId = uuid_1.default.v4();
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".xlsx");
        await this._wb.xlsx.writeFile(this._filePath);
        app_logs_1.logger.debug("File is ready");
        return this._filePath;
    }
    initGeneralSheet(wsRaw, startRowIndex, startColIndex) {
        let columns = [];
        this.addInitCell11(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.name_surname", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        columns.push({ width: 35 });
        this.addInitCell11(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.UNIQUE_ID", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell11(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.FIRST_START", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell11(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.LAST_LEAVE", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell11(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.STATUS", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell11(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.EXPECTED_ENTRANCE", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell11(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.EXPECTED_EXIT", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell11(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.ENTRANCE_PENALTY_TIME", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell11(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.MISSING_WORK", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        columns.push({ width: 20 });
        this.addInitCell11(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.OVERTIME", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        columns.push({ width: 20 });
        wsRaw.autoFilter = {
            from: "A" + startRowIndex,
            to: wsRaw.getColumn(columns.length).letter + startRowIndex,
        };
        wsRaw.columns = columns;
        wsRaw.views = [{ state: "frozen", ySplit: startRowIndex, activeCell: "A1", showGridLines: false }];
    }
    addGeneralRow(row) {
        let colIndex = 1;
        this.setDataCell11({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: row.user.fullname.toUpperCase(),
            ws: this._wsGeneral,
            border: generator_1.ReportGenerator.Constants.Styling.Dotted,
        });
        this.setDataCell11({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: row.user.uniqueId,
            ws: this._wsGeneral,
            border: generator_1.ReportGenerator.Constants.Styling.Dotted,
        });
        this.setDataCell11({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: row.entranceDateTime ? (0, moment_1.default)(row.entranceDateTime).format("HH:mm") : "-----",
            ws: this._wsGeneral,
            border: generator_1.ReportGenerator.Constants.Styling.Dotted,
            numFmt: row.entranceDateTime ? generator_1.ReportGenerator.Constants.TimeFormat : null,
        });
        this.setDataCell11({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: row.exitDateTime ? (0, moment_1.default)(row.exitDateTime).format("HH:mm") : "-----",
            ws: this._wsGeneral,
            border: generator_1.ReportGenerator.Constants.Styling.Dotted,
            numFmt: row.exitDateTime ? generator_1.ReportGenerator.Constants.TimeFormat : null,
        });
        let statusText = row.entranceDateTime
            ? i18n_1.default.__({ phrase: "EXCEL-REPORT.ARRIVED", locale: this._locale })
            : i18n_1.default.__({ phrase: "EXCEL-REPORT.NOT_ARRIVED", locale: this._locale });
        let colorText = undefined;
        let hasPermissionNow;
        let permissionNowType;
        let hasPermissionInDay = row.ppermissions && row.ppermissions.length > 0;
        if (hasPermissionInDay) {
            row.ppermissions.forEach((permission) => {
                if ((0, moment_1.default)().isBetween(permission.startUtc, permission.endUtc)) {
                    hasPermissionNow = true;
                    permissionNowType = permission.ppermissionTypeName;
                }
            });
        }
        if (hasPermissionInDay) {
            colorText = generator_1.ReportGenerator.Constants.ColorCodes.PERMISSION;
            if (hasPermissionNow) {
                statusText += "\n" + i18n_1.default.__({ phrase: "PDF-REPORT.day_off", locale: this._locale }) + "(" + permissionNowType + ")";
            }
            else if (hasPermissionInDay) {
                row.ppermissions.forEach((permission) => {
                    let interval = (0, moment_1.default)(permission.startUtc).locale(this._locale).format("DD/MM/YYYY HH:mm") + " - " + (0, moment_1.default)(permission.endUtc).locale(this._locale).format("DD/MM/YYYY HH:mm");
                    statusText += "\n" + permission.ppermissionTypeName + "\n(" + interval + ")";
                });
            }
        }
        else {
            colorText = this.findColorCodeByPerformance(row);
        }
        this.setDataCell11({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: statusText,
            ws: this._wsGeneral,
            border: generator_1.ReportGenerator.Constants.Styling.Dotted,
            numFmt: null,
            backgroundColorCode: colorText,
            isTextWhite: true,
        });
        this.setDataCell11({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: row.workPlan.startOfWorkUtc ? (0, moment_1.default)(row.workPlan.startOfWorkUtc).format("HH:mm") : "-----",
            ws: this._wsGeneral,
            border: generator_1.ReportGenerator.Constants.Styling.Dotted,
            numFmt: row.workPlan.startOfWorkUtc ? generator_1.ReportGenerator.Constants.TimeFormat : null,
        });
        this.setDataCell11({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: row.workPlan.endOfWorkUtc ? (0, moment_1.default)(row.workPlan.endOfWorkUtc).format("HH:mm") : "-----",
            ws: this._wsGeneral,
            border: generator_1.ReportGenerator.Constants.Styling.Dotted,
            numFmt: row.workPlan.endOfWorkUtc ? generator_1.ReportGenerator.Constants.TimeFormat : null,
        });
        this.setDataCell11({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: row.entrancePenaltyTimeInSeconds ? (Math.round(row.entrancePenaltyTimeInSeconds / 60) / (24 * 60)) : null,
            ws: this._wsGeneral,
            border: generator_1.ReportGenerator.Constants.Styling.Dotted,
            numFmt: row.entrancePenaltyTimeInSeconds
                ? '[<0.000694444]"";[<0.0415][m]" ' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '";[h] "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR_ABBREVATION", locale: this._locale }) +
                    ' " m "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '"'
                : null,
        });
        this.setDataCell11({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: row.missingWorkInMinutes ? (row.missingWorkInMinutes / (24 * 60)) : null,
            ws: this._wsGeneral,
            border: generator_1.ReportGenerator.Constants.Styling.Dotted,
            numFmt: row.missingWorkInMinutes
                ? '[<0.000694444]"";[<0.0415][m]" ' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '";[h] "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR_ABBREVATION", locale: this._locale }) +
                    ' " m "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '"'
                : null,
        });
        this.setDataCell11({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: row.extraWorkInMinutes ? (row.extraWorkInMinutes / (24 * 60)) : null,
            ws: this._wsGeneral,
            border: generator_1.ReportGenerator.Constants.Styling.Dotted,
            numFmt: row.extraWorkInMinutes
                ? '[<0.000694444]"";[<0.0415][m]" ' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '";[h] "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR_ABBREVATION", locale: this._locale }) +
                    ' " m "' +
                    i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: this._locale }) +
                    '"'
                : null,
        });
        this._generalRowIndex++;
    }
    findColorCodeByPerformance(employeeData) {
        let colorCode = null;
        if (employeeData.entranceDateTime) {
            if (employeeData.entrancePenaltyTimeInSeconds > 0) {
                let percentage = employeeData.entrancePenaltyTimeInSeconds / 60 / employeeData.entranceTolerance;
                if (percentage > 1) {
                    colorCode = generator_1.ReportGenerator.Constants.ColorCodes.NOT_TOLERATED_2;
                }
                else if (percentage > 0.75) {
                    colorCode = generator_1.ReportGenerator.Constants.ColorCodes.PER_75_100;
                }
                else if (percentage > 0.5) {
                    colorCode = generator_1.ReportGenerator.Constants.ColorCodes.PER_50_75;
                }
                else if (percentage > 0.25) {
                    colorCode = generator_1.ReportGenerator.Constants.ColorCodes.PER_25_50;
                }
                else if (percentage > 0) {
                    colorCode = generator_1.ReportGenerator.Constants.ColorCodes.PER_0_25;
                }
            }
            else {
                colorCode = generator_1.ReportGenerator.Constants.ColorCodes.ON_TIME_2;
            }
        }
        else {
            if (employeeData.workPlan.startOfWorkUtc) {
                colorCode = generator_1.ReportGenerator.Constants.ColorCodes.NOT_TOLERATED_2;
            }
        }
        return colorCode;
    }
    addFilterToSheet(ws, startRowIndex, startColIndex, filter, filterlength) {
        let border = {
            left: {
                style: "thin",
            },
            top: {
                style: "thin",
            },
            right: {
                style: "thin",
            },
            bottom: {
                style: "thin",
            },
        };
        this.addFilterCell11(ws, startRowIndex, startColIndex, this._organizationBasic.name + " " + i18n_1.default.__({ phrase: "EXCEL-REPORT.DAILY_FIRST_IN_LAST_OUT", locale: this._locale }), border, { bold: true });
        this.addFilterCell11(ws, startRowIndex + 1, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.DATE", locale: this._locale }), border);
        this.addFilterCell11(ws, startRowIndex + 1, startColIndex + 1, {
            formula: "DATE(" + (0, moment_1.default)(this._options.date).year() + "," + ((0, moment_1.default)(this._options.date).month() + 1) + "," + (0, moment_1.default)(this._options.date).date() + ")",
        }, border, { italic: true }, generator_1.ReportGenerator.Constants.DateFormat);
        ws.mergeCells(startRowIndex, startColIndex, startRowIndex, startColIndex + 2);
        ws.mergeCells(startRowIndex + 1, startColIndex + 1, startRowIndex + 1, startColIndex + 2);
        ws.mergeCells(startRowIndex + 2, startColIndex + 1, startRowIndex + 2, startColIndex + 2);
        this.addFilterCell11(ws, startRowIndex + 2, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.APPLIED_FILTERS", locale: this._locale }), border);
        this.addFilterCell11(ws, startRowIndex + 2, startColIndex + 1, filter, border);
        if (filterlength) {
            let row = ws.getRow(startRowIndex + 2);
            row.height = filterlength * 15;
        }
    }
}
exports.DailyFirstEntranceLastExitReport = DailyFirstEntranceLastExitReport;
