"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ExcelReportAccessLogs = void 0;
const lodash_1 = __importDefault(require("lodash"));
const app_config_1 = require("../../../app.config");
const i18n_1 = __importDefault(require("i18n"));
const report_util_1 = require("../report.util");
const dal_manager_1 = require("../../../dal/dal.manager");
const app_enums_1 = require("../../../app.enums");
const app_logs_1 = require("../../../app.logs");
const api_error_1 = require("../../../api/api.error");
const generator_1 = require("../generator");
const excel_report_refactory_1 = require("../excel.report.refactory");
class ExcelReportAccessLogs extends generator_1.ReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._filter = this._request.filter;
        this._reportFactory = new excel_report_refactory_1.ReportFactory(request.organizationId, locale);
        this._emptyData = true;
    }
    async generateReport() {
        try {
            const sheetInfo = [
                {
                    sheetName: i18n_1.default.__({ phrase: "EXCEL-REPORT.ACCESS_LOGS", locale: this._locale }),
                    headers: [
                        {
                            phrase: "EXCEL-REPORT.ACCESS_LOGS_ACCESS_TIME",
                            width: 30,
                        },
                        {
                            phrase: "EXCEL-REPORT.ACCESS_LOGS_ACCESS_POINT",
                            width: 30,
                        },
                        {
                            phrase: "EXCEL-REPORT.ACCESS_LOGS_RESULT",
                            width: 30,
                        },
                        {
                            phrase: "EXCEL-REPORT.ACCESS_LOGS_DESCRIPTION",
                            width: 45,
                        },
                        {
                            phrase: "PDF-REPORT.ACCESS_REASONS",
                            width: 30,
                        },
                        {
                            phrase: "EXCEL-REPORT.ACCESS_LOGS_CREDENTIAL_INFO",
                            width: 30,
                        },
                        {
                            phrase: "EXCEL-REPORT.ACCESS_LOGS_ORG_UNIT",
                            width: 30,
                        },
                        {
                            phrase: "EXCEL-REPORT.ACCESS_LOGS_USER_GROUP",
                            width: 30,
                        },
                    ],
                },
            ];
            const reportInfoSheetData = {
                reportNamePhrase: "EXCEL-REPORT.ACCESS_LOGS",
                filters: {
                    accessResult: this._filter.accessResult,
                    direction: this._filter.direction,
                    userIds: this._filter.userIds,
                    orgUnitIds: this._filter.organizationUnitIds,
                    applyOrganizationUnitFilterHierarchically: this._filter.filterOrganizationUnitMembersHierarchically,
                    userGroupIds: this._filter.userGroupIds,
                    credentialTypes: this._filter.credentialTypes,
                    regionIds: this._filter.regionIds,
                    accessControlPointIds: this._filter.accessControlPointIds,
                    accessReasons: this._filter.reasons,
                    credentialData: this._filter.credentialData,
                    manualRecordCreatorUserId: this._filter.manualRecordCreatorUserId,
                    showOnlyVisitors: this._filter.showOnlyVisitors,
                    showOnlyExitButtons: this._filter.showOnlyExitButtons,
                    showOnlyRemoteAccess: this._filter.showOnlyRemoteAccess,
                    showOnlyManuallyInserted: this._filter.showOnlyManuallyInserted,
                    dateTimeRange: { startDateTime: this._filter.startUtc, endDateTime: this._filter.endUtc },
                },
            };
            await this._reportFactory.initialize({
                requesterUserId: this._request.requesterUserId,
                userIds: [],
                reportInfoSheetData,
                sheetInfo,
            });
            await this.constructExcel();
            const fileObj = await this._reportFactory.finalize(reportInfoSheetData);
            return {
                format: app_enums_1.enums.ReportFormat.Excel,
                preparedReportId: fileObj.fileId,
                filePath: fileObj.filePath,
                notificationType: undefined,
                reportType: app_enums_1.enums.ReportCode.AccessLogs,
                sendFileInAttachments: undefined,
                reportContainsEmptyData: this._request.reportTemplateId ? this._emptyData : undefined,
            };
        }
        catch (error) {
            app_logs_1.logger.error(`Error generating AccessLogsReport: ${error.message}`, {
                stack: error.stack,
                organizationId: this._request.organizationId,
                requesterUserId: this._request.requesterUserId,
            });
            throw new Error("Failed to generate AccessLogsReport");
        }
    }
    async constructExcel() {
        if (this._filter.authorizeForFilter) {
            let authResultForFilter = await dal_manager_1.dbManager.accessFunctions.dbFuncAuthorizeUserFor({
                organizationId: this._request.organizationId,
                userId: this._request.requesterUserId,
                organizationUnitIds: this._filter.organizationUnitIds,
                accessControlPointIds: this._filter.accessControlPointIds,
                userIds: this._filter.userIds,
                userGroupIds: this._filter.userGroupIds,
            });
            if (!authResultForFilter.result) {
                throw (0, api_error_1.generateForbiddenError)({ message: "Unauthorized filter items" });
            }
            if (lodash_1.default.isEmpty(this._filter.organizationUnitIds) && lodash_1.default.isEmpty(this._filter.userGroupIds) && lodash_1.default.isEmpty(this._filter.userIds)) {
                if (this._filter.permittedUnitIds) {
                    this._filter.organizationUnitIds = this._filter.permittedUnitIds;
                }
                else {
                    const userOrganizationUnits = await dal_manager_1.dbManager.armondb
                        .withSchema(this._request.organizationId)
                        .from("users as u")
                        .innerJoin("userOrganizations as uo", "uo.userId", "u.id")
                        .innerJoin("userOrganizationOrganizationUnits as uoou", "uoou.userOrganizationId", "uo.id")
                        .innerJoin("organizationUnits as ou", (join) => {
                        join.on(dal_manager_1.dbManager.armondb.raw(`ou."ancestorIds" like '%' || uoou."organizationUnitId" || '%'`)).orOn("ou.id", "uoou.organizationUnitId");
                    })
                        .whereNull("u.deletedAt")
                        .whereNull("uoou.deletedAt")
                        .whereNull("uo.deletedAt")
                        .where("uo.organizationId", this._request.organizationId)
                        .where("u.id", this._request.requesterUserId)
                        .select("ou.id");
                    this._filter.organizationUnitIds = userOrganizationUnits.map((u) => u.id);
                }
            }
        }
        let filterUsers = await dal_manager_1.dbManager.accessFunctions.dbFuncCollectUsersForAccessReportFilter({
            organizationId: this._request.organizationId,
            userIds: this._filter.userIds,
            organizationUnitIds: this._filter.organizationUnitIds,
            userGroupIds: this._filter.userGroupIds,
            filterOrganizationUnitMembersHierarchically: this._filter.filterOrganizationUnitMembersHierarchically,
        });
        let filterUserIds = (filterUsers && filterUsers.length) > 0 ? filterUsers.map((f) => f.userId) : [];
        this._filter.userIds = filterUserIds;
        const option = Object.assign(this._filter, {
            take: app_config_1.appConfig.reportExportRowLimit,
            skip: undefined,
        });
        await dal_manager_1.dbManager.accessAccessLog.getAccessLogsReportNew(this._request.organizationId, option, async (rows) => {
            app_logs_1.logger.debug("[AccessLogsReport]Fetch data %s rows", rows.items.length);
            this._emptyData = rows.items.length === 0;
            await this.constructRows(rows);
        });
        (0, report_util_1.styleExcelWorksheet)(this._reportFactory.getWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.ACCESS_LOGS", locale: this._locale })), this._locale);
    }
    createCellForArray(record, key, subKey) {
        let result = "";
        if (record.identity == null)
            return result;
        const items = record.identity[key];
        if (items == null || items.length === 0)
            return result;
        items.forEach((item) => {
            result += item[subKey] + "\r\n";
        });
        return result;
    }
    createCellForStatus(record, locale) {
        let log = "";
        if (record.isVisitor) {
            log += i18n_1.default.__({ phrase: "PDF-REPORT.log_visitor", locale }) + "\r\n";
        }
        if (record.isRemoteAccess) {
            log += i18n_1.default.__({ phrase: "PDF-REPORT.log_remote_access", locale }) + "\r\n";
        }
        if (record.isManuallyLogged) {
            log += i18n_1.default.__({ phrase: "PDF-REPORT.log_manual", locale }, { manualCreator: record.manualRecordCreatorCaptions[0].text[0] }) + "\r\n";
        }
        if (record.isExitButton) {
            log += i18n_1.default.__({ phrase: "PDF-REPORT.log_exit_button", locale }) + "\r\n";
        }
        const translationKey = (0, report_util_1.getDescriptionAccessLogItem)(record.result, record.direction);
        const translatedReason = (0, report_util_1.getLocalizedTextOfAccessLogReason)(record.result, this._locale);
        const isSuccess = record.result === app_enums_1.enums.AccessLogReason.Success || (record.isManuallyLogged && record.result === app_enums_1.enums.AccessLogReason.Manual);
        let credentialDescription = "";
        if (record.credentials && record.credentials.length > 0) {
            let visibleCredentialData = [];
            record.credentials.forEach((credential) => {
                visibleCredentialData.push((0, report_util_1.getCredentialDataVisibility)(credential.type));
            });
            record.credentials.forEach((credential, index) => {
                let credentialType = report_util_1.ReportConstants.CredentialTypes.find((item) => item.type === credential.type);
                credentialDescription += "\r\n" + i18n_1.default.__({ phrase: "PDF-REPORT." + credentialType.description, locale }) + " ";
                if (credential.data && visibleCredentialData[index]) {
                    credentialDescription += "(" + credential.data + ")";
                }
                if (record.mobileAuthenticated) {
                    credentialDescription += "\r\n" + i18n_1.default.__({ phrase: "EXCEL-REPORT.MOBILE_AUTHENTICATED", locale });
                }
            });
        }
        return {
            statusColor: isSuccess ? "#62A420" : "#E62700",
            description: record.accessControlPoint != null && record.accessControlPoint.name != null
                ? i18n_1.default.__({ phrase: "PDF-REPORT." + translationKey, locale }, {
                    an: record.accessControlPoint.name,
                })
                : i18n_1.default.__({ phrase: "PDF-REPORT." + translationKey, locale }),
            reason: translatedReason,
            log: log,
            statusInfo: (0, report_util_1.getAccessLogStatusInfo)(record.direction, this._locale, isSuccess),
            credentialDescription: credentialDescription,
        };
    }
    async constructRows(rows) {
        const userIds = [...new Set(rows.items.map((item) => item?.identity?.id))];
        await this._reportFactory.appendToUserInfo(userIds);
        const sheetInfo = [
            {
                sheetName: i18n_1.default.__({ phrase: "EXCEL-REPORT.ACCESS_LOGS", locale: this._locale }),
                data: [],
            },
        ];
        rows.items.forEach((record) => {
            const userId = record?.identity?.id;
            const cellForStatus = this.createCellForStatus(record, this._locale);
            const cellForUserGroups = this.createCellForArray(record, "userGroups", "name");
            const cellForOrganizationUnits = this.createCellForArray(record, "organizationUnits", "name");
            sheetInfo[0].data.push({
                userId,
                cells: [
                    { value: this._reportFactory.formatDateTime(record.utc, this._locale) ?? "" },
                    { value: record?.accessControlPoint?.name ?? "" },
                    { value: cellForStatus.statusInfo ?? "" },
                    {
                        value: cellForStatus.description + "\r\n" + cellForStatus.log,
                        alignment: {
                            wrapText: true,
                        },
                    },
                    { value: cellForStatus.reason ?? "" },
                    {
                        value: cellForStatus.credentialDescription ?? "",
                        alignment: {
                            wrapText: true,
                        },
                    },
                    {
                        value: cellForOrganizationUnits ?? "",
                        alignment: {
                            wrapText: true,
                        },
                    },
                    {
                        value: cellForUserGroups ?? "",
                        alignment: {
                            wrapText: true,
                        },
                    },
                ],
            });
        });
        this._reportFactory.appendData(sheetInfo);
    }
}
exports.ExcelReportAccessLogs = ExcelReportAccessLogs;
