"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ReportFactory = void 0;
const exceljs_1 = __importDefault(require("exceljs"));
const path_1 = __importDefault(require("path"));
const uuid_1 = __importDefault(require("uuid"));
const fs_1 = __importDefault(require("fs"));
const i18n_1 = __importDefault(require("i18n"));
const app_logs_1 = require("../../app.logs");
const luxon_1 = require("luxon");
const app_enums_1 = require("../../app.enums");
const app_config_1 = require("../../app.config");
const dal_manager_1 = require("../../dal/dal.manager");
const excel_report_constants_1 = require("./excel.report.constants");
const dal_constants_1 = require("../../dal/dal.constants");
const report_util_1 = require("./report.util");
const restapi_1 = require("../../lib/es/models/restapi");
class ReportFactory {
    constructor(organizationId, locale) {
        this.sheets = {};
        this.formatDateTimeRange = (startDate, endDate, locale) => {
            if (!startDate || !endDate) {
                app_logs_1.logger.warn("Invalid date range provided for formatDateTimeRange");
                return "";
            }
            const start = luxon_1.DateTime.fromJSDate(typeof startDate === "string" ? new Date(startDate) : startDate).setLocale(locale);
            const end = luxon_1.DateTime.fromJSDate(typeof endDate === "string" ? new Date(endDate) : endDate).setLocale(locale);
            return `${start.toFormat("DD HH:mm:ss")} - ${end.toFormat("DD HH:mm:ss")}`;
        };
        this.formatDateTime = (date, locale) => {
            if (!date) {
                app_logs_1.logger.warn("Invalid date provided for formatDateTime");
                return "";
            }
            const formattedDate = luxon_1.DateTime.fromJSDate(typeof date === "string" ? new Date(date) : date).setLocale(locale);
            return formattedDate.toFormat("DD HH:mm:ss");
        };
        this.formatDate = (date, locale) => {
            if (!date) {
                app_logs_1.logger.warn("Invalid date provided for formatDate");
                return "";
            }
            const dateTime = luxon_1.DateTime.fromJSDate(typeof date === "string" ? new Date(date) : date).setLocale(locale);
            return `${dateTime.toFormat("D")}`;
        };
        this.formatDateRange = (startDate, endDate, locale) => {
            if (!startDate || !endDate) {
                app_logs_1.logger.warn("Invalid date range provided for formatDateRange");
                return "";
            }
            const start = luxon_1.DateTime.fromJSDate(typeof startDate === "string" ? new Date(startDate) : startDate).setLocale(locale);
            const end = luxon_1.DateTime.fromJSDate(typeof endDate === "string" ? new Date(endDate) : endDate).setLocale(locale);
            return `${start.toFormat("D")} - ${end.toFormat("D")}`;
        };
        this._fileId = uuid_1.default.v4();
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".xlsx");
        this._locale = locale ?? "tr";
        this._organizationId = organizationId;
        this._profileInfoOffset = 0;
        this._profileColumnsIndex = 0;
        this._sheetVerticalOffset = null;
        if (fs_1.default.existsSync(this._filePath)) {
            fs_1.default.unlinkSync(this._filePath);
        }
    }
    async initialize(data) {
        const { reportInfoSheetData, userIds, sheetInfo, requesterUserId } = data;
        const hasInfoSheet = Boolean(reportInfoSheetData);
        const hasProfileInfo = Boolean(userIds);
        this._wb = new exceljs_1.default.stream.xlsx.WorkbookWriter({
            filename: this._filePath,
            useStyles: true,
            useSharedStrings: true,
        });
        this._wb.creator = (await dal_manager_1.dbManager.accessUser.getBasicUserInfoList(this._organizationId, [requesterUserId]))[0].fullname;
        if (data?.sheetVerticalOffset) {
            this._sheetVerticalOffset = data.sheetVerticalOffset;
        }
        this.createWorksheets(sheetInfo.map((x) => x.sheetName), hasInfoSheet);
        if (hasProfileInfo) {
            this._usersInfo = (await dal_manager_1.dbManager.accessUser.getReportableUserInfoList(this._organizationId, userIds)).reduce((acc, user) => {
                const { id, ...userData } = user;
                acc[id] = userData;
                return acc;
            }, {});
        }
        if (data?.profileColumnsIndex) {
            this._profileColumnsIndex = data.profileColumnsIndex;
        }
        sheetInfo.forEach((sheet) => {
            let rowIndex = 1;
            if (this?._sheetVerticalOffset) {
                rowIndex += data.sheetVerticalOffset;
            }
            this.addHeaders(sheet.sheetName, sheet.headers, rowIndex, 1, hasProfileInfo);
        });
    }
    appendData(sheetInfo) {
        const hasProfileInfo = Boolean(this._usersInfo);
        sheetInfo.forEach((sheet) => {
            this.addRows(sheet.sheetName, sheet.data, this.sheets[sheet.sheetName]._rowIndex, 1, hasProfileInfo);
        });
    }
    async appendToUserInfo(userIds) {
        if (!this._usersInfo) {
            this._usersInfo = {};
        }
        const updatedUsers = (await dal_manager_1.dbManager.accessUser.getReportableUserInfoList(this._organizationId, userIds)).reduce((acc, user) => {
            const { id, ...userData } = user;
            acc[id] = userData;
            return acc;
        }, {});
        Object.assign(this._usersInfo, updatedUsers);
    }
    async finalize(reportInfoSheetData) {
        if (reportInfoSheetData) {
            await this.fillInfoSheet(reportInfoSheetData);
        }
        for (const sheetName in this.sheets) {
            this.sheets[sheetName]._ws.commit();
        }
        await this._wb.commit();
        return { filePath: this._filePath, fileId: this._fileId };
    }
    getProfileInfoOfset() {
        return this._profileInfoOffset;
    }
    getWorksheet(sheetName) {
        return this.sheets[sheetName]._ws;
    }
    getSheetIndex(sheetName) {
        return this.sheets[sheetName]._rowIndex;
    }
    setSheetIndex(sheetName, rowIndex) {
        this.sheets[sheetName]._rowIndex = rowIndex;
    }
    addNotetoSheet(sheetName, text) {
        const sheet = this.sheets[sheetName];
        const targetRow = sheet._rowIndex + 2;
        const targetColumn = 1;
        const existingNote = this.getCellValue(sheet._ws, targetRow, targetColumn);
        const updatedNote = existingNote ? `${existingNote}\n${text}` : text;
        this.addCell(sheet._ws, targetRow, targetColumn, updatedNote, {
            border: excel_report_constants_1.ExcelReportConstants.Styling.AllThin,
            font: {
                bold: true,
            },
            alignment: {
                wrapText: true,
            },
        });
        sheet._rowIndex++;
    }
    countMaxLines(texts) {
        if (texts.length === 0)
            return 0;
        return Math.max(...texts.map((text) => text.split("\n").length));
    }
    getCellValue(ws, row, column) {
        const cell = ws.getCell(row, column);
        return cell.value ? cell.value.toString() : null;
    }
    addCell(ws, row, col, value, options = {}) {
        const c = ws.getCell(row, col);
        c.value = options.formula ? { formula: options.formula } : value;
        c.alignment = options.alignment ?? undefined;
        c.font = options.font ?? undefined;
        c.border = options.border ?? undefined;
        c.numFmt = options.numFmt ?? undefined;
        if (options.fillColor) {
            c.fill = { type: "pattern", pattern: "solid", fgColor: { argb: options.fillColor } };
        }
        if (options.width)
            ws.getColumn(col).width = options.width;
        if (options.newLineCount) {
            ws.getRow(row).height = options.newLineCount * 15;
        }
        else if (options.height) {
            ws.getRow(row).height = options.height;
        }
    }
    createWorksheets(sheetNames, hasInfoSheet = false) {
        sheetNames.forEach((sheetName) => {
            let rowIndex = 1;
            if (this?._sheetVerticalOffset) {
                rowIndex += this._sheetVerticalOffset;
            }
            const ws = this._wb.addWorksheet(sheetName, {
                pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
            });
            this.sheets[sheetName] = {
                _ws: ws,
                _rowIndex: rowIndex,
            };
        });
        if (hasInfoSheet) {
            const ws = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.INFORMATION", locale: this._locale }), {
                pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
            });
            ws.properties.defaultRowHeight = 15;
            this.sheets[i18n_1.default.__({ phrase: "EXCEL-REPORT.INFORMATION", locale: this._locale })] = {
                _ws: ws,
                _rowIndex: 1,
            };
        }
    }
    addHeaders(sheetName, headers, startRowIndex, startColIndex, hasProfileInfo) {
        const sheet = this.sheets[sheetName];
        if (!sheet) {
            throw new Error(`Sheet "${sheetName}" not found`);
        }
        const columns = [];
        if (hasProfileInfo) {
            const profileInfoHeaders = this.getProfileInfoHeaders();
            this._profileInfoOffset = profileInfoHeaders.length;
            if (profileInfoHeaders) {
                headers = [...headers.slice(0, this._profileColumnsIndex), ...profileInfoHeaders, ...headers.slice(this._profileColumnsIndex)];
            }
            else {
                app_logs_1.logger.warn("Profile info headers are undefined or null");
            }
        }
        headers.forEach((header, index) => {
            if (!header) {
                app_logs_1.logger.warn(`Header at index ${index} is null or undefined`);
                return;
            }
            const headerValue = header.value || (header.phrase ? i18n_1.default.__({ phrase: header.phrase, locale: this._locale }) : "");
            this.addCell(sheet._ws, startRowIndex, index + startColIndex, headerValue, {
                border: header.border ?? excel_report_constants_1.ExcelReportConstants.Styling.AllThin,
                alignment: header.alignment ?? {
                    horizontal: "center",
                    vertical: "middle",
                    wrapText: true,
                },
                font: header.font ?? { bold: true },
                height: header.height,
                width: header.width ?? 30,
                fillColor: header.fillColor,
            });
            columns.push({ width: header.width ?? 30 });
        });
        const lastColumnLetter = sheet._ws.getColumn(columns.length)?.letter;
        if (lastColumnLetter) {
            sheet._ws.autoFilter = {
                from: "A" + startRowIndex,
                to: lastColumnLetter + startRowIndex,
            };
        }
        else {
            app_logs_1.logger.warn("Could not determine last column letter for autofilter");
        }
        sheet._rowIndex++;
        sheet._ws.columns = columns;
    }
    addRows(sheetName, data, startRowIndex, startColIndex, hasProfileInfo) {
        const sheet = this.sheets[sheetName];
        if (!sheet) {
            throw new Error(`Sheet "${sheetName}" not found`);
        }
        data.forEach((item, rowIndex) => {
            if (!item || !Array.isArray(item.cells)) {
                app_logs_1.logger.warn(`Invalid row data at index ${rowIndex}, skipping...`);
                return;
            }
            if (hasProfileInfo) {
                const profileInfoRow = this.getProfileInfoRow(item.userId);
                if (profileInfoRow && Array.isArray(profileInfoRow)) {
                    item.cells = [...item.cells.slice(0, this._profileColumnsIndex), ...profileInfoRow, ...item.cells.slice(this._profileColumnsIndex)];
                }
                else {
                    app_logs_1.logger.warn(`Profile info row for userId ${item.userId} is missing or invalid`);
                }
            }
            item.cells.forEach((cell, colIndex) => {
                if (!cell) {
                    app_logs_1.logger.warn(`Invalid cell data at row ${rowIndex}, column ${colIndex}, skipping...`);
                    return;
                }
                try {
                    this.addCell(sheet._ws, rowIndex + startRowIndex, colIndex + startColIndex, cell.value || "", {
                        border: cell.border ?? excel_report_constants_1.ExcelReportConstants.Styling.Dotted,
                        alignment: cell.alignment,
                        font: cell.font,
                        height: cell.height,
                        width: cell.width,
                        fillColor: cell.fillColor,
                        numFmt: cell.numFmt,
                        newLineCount: cell.newLineCount,
                        formula: cell.formula,
                    });
                }
                catch (error) {
                    app_logs_1.logger.error(`Error adding cell at row ${rowIndex}, column ${colIndex}: ${error}`);
                }
            });
            if (sheet._ws.getRow(sheet._rowIndex)) {
                sheet._ws.getRow(sheet._rowIndex).commit();
                sheet._rowIndex++;
            }
            else {
                app_logs_1.logger.warn(`Row ${sheet._rowIndex} is invalid or missing, skipping commit`);
            }
        });
    }
    async fillInfoSheet(reportInfoSheetData) {
        if (!reportInfoSheetData) {
            app_logs_1.logger.warn("Invalid report info sheet data provided");
            return;
        }
        const sheetName = i18n_1.default.__({ phrase: "EXCEL-REPORT.INFORMATION", locale: this._locale });
        const ws = this.sheets[sheetName]?._ws;
        if (!ws) {
            app_logs_1.logger.warn(`Sheet "${sheetName}" not found`);
            return;
        }
        ws.columns = [{ width: 30 }, { width: 30 }, { width: 30 }, { width: 30 }];
        const filter = (await this.generateFilterText(reportInfoSheetData.filters)) ?? "";
        let startRowIndex = 1;
        let startColIndex = 1;
        const organization = await dal_manager_1.dbManager.accessOrganization.getOrganizationBasic(this._organizationId);
        const reportName = organization.name + " " + (reportInfoSheetData.reportNameValue ?? i18n_1.default.__({ phrase: reportInfoSheetData.reportNamePhrase, locale: this._locale }));
        this.addCell(ws, startRowIndex, startColIndex, reportName, {
            border: excel_report_constants_1.ExcelReportConstants.Styling.AllThin,
            font: { bold: true },
            alignment: {
                horizontal: "center",
                vertical: "middle",
                wrapText: true,
            },
        });
        ws.mergeCells(startRowIndex, startColIndex, startRowIndex, startColIndex + 3);
        this.addCell(ws, startRowIndex + 1, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.FILE_CREATION_DATE", locale: this._locale }), {
            border: excel_report_constants_1.ExcelReportConstants.Styling.AllThin,
        });
        const formattedDate = luxon_1.DateTime.now().setLocale(this._locale).toFormat("DD HH:mm:ss");
        this.addCell(ws, startRowIndex + 1, startColIndex + 1, formattedDate, {
            border: excel_report_constants_1.ExcelReportConstants.Styling.AllThin,
            font: { italic: false },
            numFmt: excel_report_constants_1.ExcelReportConstants.DateFormat,
        });
        this.addCell(ws, startRowIndex + 2, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.REQUESTED_USER", locale: this._locale }), { border: excel_report_constants_1.ExcelReportConstants.Styling.AllThin });
        this.addCell(ws, startRowIndex + 2, startColIndex + 1, this._wb?.creator ?? "", { border: excel_report_constants_1.ExcelReportConstants.Styling.AllThin });
        ws.mergeCells(startRowIndex + 1, startColIndex + 1, startRowIndex + 1, startColIndex + 3);
        ws.mergeCells(startRowIndex + 2, startColIndex + 1, startRowIndex + 2, startColIndex + 3);
        ws.mergeCells(startRowIndex + 3, startColIndex + 1, startRowIndex + 3, startColIndex + 3);
        this.addCell(ws, startRowIndex + 3, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.APPLIED_FILTERS", locale: this._locale }), {
            border: excel_report_constants_1.ExcelReportConstants.Styling.AllThin,
            alignment: {
                vertical: "middle",
                wrapText: true,
            },
        });
        this.addCell(ws, startRowIndex + 3, startColIndex + 1, filter, {
            border: excel_report_constants_1.ExcelReportConstants.Styling.AllThin,
            alignment: {
                vertical: "middle",
                wrapText: true,
            },
        });
        const filterLength = filter.split("\r\n").length;
        const row = ws.getRow(startRowIndex + 3);
        if (row) {
            row.height = filterLength * 25;
        }
        else {
            app_logs_1.logger.warn(`Invalid row at index ${startRowIndex + 3}`);
        }
        startRowIndex += 6;
        ws.mergeCells(startRowIndex, startColIndex, startRowIndex, startColIndex + 3);
        this.addCell(ws, startRowIndex, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.FILE_NOTICE", locale: this._locale }), {
            height: 70,
            font: { size: 7 },
            alignment: {
                wrapText: true,
                vertical: "middle",
            },
        });
    }
    async parseFilters(filters) {
        let filterResultData = {};
        const translatePhrase = (key) => {
            if (!key)
                return "";
            const translated = i18n_1.default.__({ phrase: key, locale: this._locale });
            return translated ?? key;
        };
        const fetchUserNames = async (userIds) => (await dal_manager_1.dbManager.accessUser.getBasicUserInfoList(this._organizationId, userIds)).map((user) => user.fullname);
        const fetchSingleUserName = async (userId) => (await dal_manager_1.dbManager.accessUser.getBasicUserInfoList(this._organizationId, [userId]))[0]?.fullname;
        if (filters.userIds)
            filterResultData.users = await fetchUserNames(filters.userIds);
        if (filters.actionUserIds)
            filterResultData.actionUsers = await fetchUserNames(filters.actionUserIds);
        if (filters.modifiedByUserIds)
            filterResultData.modifiedByUsers = await fetchUserNames(filters.modifiedByUserIds);
        if (filters.manualRecordCreatorUserId)
            filterResultData.manualRecordCreator = await fetchSingleUserName(filters.manualRecordCreatorUserId);
        if (filters.targetUserId)
            filterResultData.targetUser = await fetchSingleUserName(filters.targetUserId);
        if (filters.actionUserId)
            filterResultData.actionUser = await fetchSingleUserName(filters.actionUserId);
        if (filters.employeeIds)
            filterResultData.employees = await fetchUserNames(filters.employeeIds);
        if (filters.userOrganizationStatus) {
            filterResultData.userOrganizationStatus = translatePhrase(`EXCEL-REPORT.${dal_constants_1.DalConstants.IdentityStatusType[filters.userOrganizationStatus].toUpperCase()}`);
        }
        if (filters.userMainActionType) {
            const actionTypeMap = {
                [dal_constants_1.DalConstants.UserMainActionType.UpdateIdentity]: "EXCEL-REPORT.AUDIT_UPDATE_IDENTITY",
                [dal_constants_1.DalConstants.UserMainActionType.InsertIdentity]: "EXCEL-REPORT.AUDIT_INSERT_IDENTITY",
                [dal_constants_1.DalConstants.UserMainActionType.AddAccessRight]: "EXCEL-REPORT.AUDIT_ADD_ACCESSRIGHT_IDENTITY",
                [dal_constants_1.DalConstants.UserMainActionType.RemoveAccessRight]: "EXCEL-REPORT.AUDIT_REMOVE_ACCESSRIGHT_IDENTITY",
                [dal_constants_1.DalConstants.UserMainActionType.AddCredential]: "EXCEL-REPORT.AUDIT_ADD_CREDENTIAL",
                [dal_constants_1.DalConstants.UserMainActionType.RemoveCredential]: "EXCEL-REPORT.AUDIT_REMOVE_CREDENTIAL",
                [dal_constants_1.DalConstants.UserMainActionType.UpdateProfile]: "EXCEL-REPORT.AUDIT_UPDATE_PROFILE",
                [dal_constants_1.DalConstants.UserMainActionType.UpdateCredential]: "EXCEL-REPORT.AUDIT_UPDATE_PROFILE",
                [dal_constants_1.DalConstants.UserMainActionType.InsertAccessControlPoint]: "EXCEL-REPORT.AUDIT_INSERT_ACP",
                [dal_constants_1.DalConstants.UserMainActionType.UpdateAccessControlPoint]: "EXCEL-REPORT.AUDIT_UPDATE_ACP",
                [dal_constants_1.DalConstants.UserMainActionType.UpdateAccessControlPointMapping]: "EXCEL-REPORT.AUDIT_UPDATE_ACP_MAPPING",
                [dal_constants_1.DalConstants.UserMainActionType.UpdateRelay]: "EXCEL-REPORT.AUDIT_UPDATE_RELAY",
                [dal_constants_1.DalConstants.UserMainActionType.UpdateStatusSensor]: "EXCEL-REPORT.AUDIT_UPDATED_STATUS_SENSOR",
                [dal_constants_1.DalConstants.UserMainActionType.UpdateCounterSensor]: "EXCEL-REPORT.AUDIT_UPDATED_COUNTER_SENSOR",
                [dal_constants_1.DalConstants.UserMainActionType.UpdateReader]: "EXCEL-REPORT.AUDIT_UPDATED_READER",
            };
            if (filters.userMainActionType in actionTypeMap) {
                filterResultData.userMainActionType = translatePhrase(actionTypeMap[filters.userMainActionType]);
            }
        }
        if (filters.regionIds) {
            filterResultData.regions = (await dal_manager_1.dbManager.accessRegion.getRegionIdName(this._organizationId, filters.regionIds)).map((region) => region.name);
        }
        if (filters.regionId) {
            filterResultData.region = (await dal_manager_1.dbManager.accessRegion.getRegionIdName(this._organizationId, [filters.regionId]))[0]?.name;
        }
        if (filters.roleId) {
            filterResultData.role = (await dal_manager_1.dbManager.accessUser.listRolesOfOrganization(this._organizationId)).find((role) => role.id === filters.roleId)?.name;
        }
        if (filters.orgUnitIds) {
            filterResultData.organizationUnits = (await dal_manager_1.dbManager.accessOrganizationUnit.getIdNamePairsOfOrganizationUnits(this._organizationId, filters.orgUnitIds)).map((ou) => ou.name);
        }
        if (filters.userGroupIds) {
            filterResultData.userGroups = (await dal_manager_1.dbManager.accessUserGroup.listUserGroupsByIds(this._organizationId, filters.userGroupIds)).map((ug) => ug.name);
        }
        if (filters.credentialTypes) {
            filterResultData.credentialTypes = filters.credentialTypes.map((type) => {
                const description = report_util_1.ReportConstants.CredentialTypes.find((cred) => cred.type === type)?.description;
                return description ? translatePhrase(`PDF-REPORT.${description}`) : "";
            });
        }
        if (filters.accessControlPointIds) {
            filterResultData.accessControlPoints = (await dal_manager_1.dbManager.accessAccessControlPoint.getAccessControlPointIdName(this._organizationId, filters.accessControlPointIds)).map((ap) => ap.name);
        }
        if (filters.accessControlPointId) {
            filterResultData.accessControlPoint = (await dal_manager_1.dbManager.accessAccessControlPoint.getAccessControlPointIdName(this._organizationId, [filters.accessControlPointId]))[0]?.name;
        }
        if (filters.arventoAccessLogType) {
            const arventoMap = {
                [restapi_1.ArventoAccessLogType.ArventoQr]: "EXCEL-REPORT.ARVENTO_QR_ACCESS_LOG",
                default: "EXCEL-REPORT.ARVENTO_REGION_ACCESS_LOG",
            };
            filterResultData.arventoAccessLogType = translatePhrase(arventoMap[filters.arventoAccessLogType] || arventoMap.default);
        }
        if (filters.accessResult) {
            const description = report_util_1.ReportConstants.AccessResultTypes.find((item) => item.type === filters.accessResult)?.description;
            if (description)
                filterResultData.accessResult = translatePhrase(description);
        }
        if (filters.direction) {
            const description = report_util_1.ReportConstants.Direction.find((item) => item.type === filters.direction)?.description;
            if (description)
                filterResultData.direction = translatePhrase(description);
        }
        if (filters.accessReasons?.length) {
            filterResultData.accessReasons = filters.accessReasons
                .map((reason) => {
                const description = report_util_1.ReportConstants.AccessLogReason.find((item) => item.type === reason)?.description;
                return description ? translatePhrase(description) : "";
            })
                .filter(Boolean);
        }
        if (filters.employeeStatusAtSegmentStatus) {
            const statusMap = {
                [dal_constants_1.DalConstants.EmployeeStatus.AtBreak]: "EXCEL-REPORT.AT_BREAK",
                [dal_constants_1.DalConstants.EmployeeStatus.AtHoliday]: "EXCEL-REPORT.AT_HOLIDAY",
                [dal_constants_1.DalConstants.EmployeeStatus.AtOffTimePermission]: "EXCEL-REPORT.AT_PERMISSION",
                [dal_constants_1.DalConstants.EmployeeStatus.AtOutOfWorkingHours]: "EXCEL-REPORT.AT_OUT_WORKING_HOURS",
                [dal_constants_1.DalConstants.EmployeeStatus.AtWorkingHours]: "EXCEL-REPORT.AT_WORKING_HOURS",
            };
            filterResultData.employeeStatusAtSegmentStatus = translatePhrase(statusMap[filters.employeeStatusAtSegmentStatus] || "EXCEL-REPORT.ALL");
        }
        if (filters.workStatusFilters) {
            filterResultData.workStatusFilters = filters.workStatusFilters.map((status) => translatePhrase(`EXCEL-REPORT.${dal_constants_1.DalConstants.WorkStatusFilter[status].toUpperCase()}`));
        }
        if (filters.workplanIds) {
            filterResultData.workPlans = (await dal_manager_1.dbManager.accessPacs.listWorkPlanByIds(this._organizationId, filters.workplanIds)).items.map((wp) => wp.name);
        }
        if (filters.regionStates) {
            const regionStateMap = {
                [app_enums_1.enums.AntiPassbackState.Unknown]: "PDF-REPORT.REGION_STATE_UNSET",
                [app_enums_1.enums.AntiPassbackState.In]: "PDF-REPORT.REGION_STATE_IN",
                [app_enums_1.enums.AntiPassbackState.Out]: "PDF-REPORT.REGION_STATE_OUT",
            };
            filterResultData.regionStates = filters.regionStates.map((state) => translatePhrase(regionStateMap[state] || ""));
        }
        if (filters.permissionDateRange) {
            const { startDate, endDate } = filters.permissionDateRange;
            filterResultData.permissionDate = [startDate, endDate]
                .filter(Boolean)
                .map((date) => luxon_1.DateTime.fromJSDate(date).setLocale(this._locale).toFormat("dd/MM/yyyy HH:mm"))
                .join(" - ");
        }
        if (filters.userCreatedDateRange) {
            filterResultData.userCreatedDate = `${luxon_1.DateTime.fromJSDate(filters.userCreatedDateRange.startDate).toLocaleString(luxon_1.DateTime.DATE_SHORT)} - ${luxon_1.DateTime.fromJSDate(filters.userCreatedDateRange.endDate).toLocaleString(luxon_1.DateTime.DATE_SHORT)}`;
        }
        if (filters.forbiddenInterval) {
            filterResultData.forbiddenInterval = {
                startDate: luxon_1.DateTime.fromJSDate(filters.forbiddenInterval.startDate).toFormat(excel_report_constants_1.ExcelReportConstants.DateTimeFormat),
                endDate: luxon_1.DateTime.fromJSDate(filters.forbiddenInterval.endDate).toFormat(excel_report_constants_1.ExcelReportConstants.DateTimeFormat),
            };
        }
        if (filters.dateTimeRange) {
            const startDate = filters.dateTimeRange?.startDateTime;
            const endDate = filters.dateTimeRange?.endDateTime;
            filterResultData.dateTimeRange = this.formatDateTimeRange(startDate, endDate, this._locale);
        }
        if (filters.date) {
            filterResultData.date = this.formatDate(filters.date, this._locale);
        }
        if (filters.dateRange) {
            const startDate = filters.dateRange?.startDate;
            const endDate = filters.dateRange?.endDate;
            filterResultData.dateRange = this.formatDateRange(startDate, endDate, this._locale);
        }
        if (filters.terminalIds) {
            filterResultData.terminals = (await dal_manager_1.dbManager.accessDevice.genericListTerminals(this._organizationId, filters.terminalIds, {
                skip: null,
                take: null,
            })).items.map((terminal) => terminal.captionLines[0]);
        }
        if (filters.sensorIds) {
            filterResultData.sensors = (await dal_manager_1.dbManager.accessReport.collectSensorEventsInfo(this._organizationId, filters.sensorIds, [])).sensorFilterResult.map((sensor) => sensor.name);
        }
        if (filters.visitStatus) {
            switch (filters.visitStatus) {
                case dal_constants_1.DalConstants.ListVisitState.All:
                    filterResultData.visitStatus = i18n_1.default.__({ phrase: "EXCEL-REPORT.VISIT_STATE_ALL", locale: this._locale });
                    break;
                case dal_constants_1.DalConstants.ListVisitState.OnlyActive:
                    filterResultData.visitStatus = i18n_1.default.__({ phrase: "EXCEL-REPORT.VISIT_STATE_ACTIVE", locale: this._locale });
                    break;
                case dal_constants_1.DalConstants.ListVisitState.OnlyTerminated:
                    filterResultData.visitStatus = i18n_1.default.__({ phrase: "EXCEL-REPORT.VISIT_STATE_TERMINATED", locale: this._locale });
                    break;
                default:
                    filterResultData.visitStatus = "";
                    break;
            }
        }
        return filterResultData;
    }
    async generateFilterText(filters) {
        const translatePhrase = (phrase) => i18n_1.default.__({ phrase, locale: this._locale });
        const formatList = (items) => items.join(", ");
        const addFilter = (key, value) => {
            if (value === undefined || value === null)
                return;
            if (Array.isArray(value)) {
                if (value.length > 0) {
                    filtersResult.push(`${translatePhrase(key)} : ${formatList(value)}`);
                }
            }
            else if (typeof value === "boolean") {
                filtersResult.push(`${translatePhrase(key)} : ${translatePhrase(value ? "EXCEL-REPORT.YES" : "EXCEL-REPORT.NO")}`);
            }
            else if (value !== "") {
                filtersResult.push(`${translatePhrase(key)} : ${value}`);
            }
        };
        const filterResultData = await this.parseFilters(filters);
        const filtersResult = [];
        const singleFilters = [
            { phrase: "EXCEL-REPORT.DATE", value: filterResultData?.dateTimeRange },
            { phrase: "EXCEL-REPORT.DATE", value: filterResultData?.dateRange },
            { phrase: "EXCEL-REPORT.DATE", value: filterResultData?.date },
            { phrase: "PDF-REPORT.STATUS", value: filterResultData?.accessResult },
            { phrase: "PDF-REPORT.DIRECTION", value: filterResultData?.direction },
            { phrase: "EXCEL-REPORT.REGION", value: filters?.arventoRegion },
            { phrase: "PDF-REPORT.licence_plate", value: filters?.licencePlate },
            { phrase: "EXCEL-REPORT.CREDENTIAL_DATA", value: filters?.credentialData },
            { phrase: "EXCEL-REPORT.EXPIRATION_DATE_RANGE", value: filterResultData?.expirationDateRange },
            { phrase: "EXCEL-REPORT.ARVENTO_ACCESS_LOG_TYPE", value: filterResultData?.arventoAccessLogType },
            { phrase: "REPORT.ACCESS_LOGS_MANUAL_LOG_CREATOR", value: filterResultData?.manualRecordCreator },
            { phrase: "EXCEL-REPORT.AUDIT_ACTION_TYPE", value: filterResultData?.userMainActionType },
            { phrase: "EXCEL-REPORT.USER_STATUS", value: filterResultData?.userOrganizationStatus },
            { phrase: "COMMON.SEARCH_HEADER", value: filters?.searchText },
            { phrase: "EXCEL-REPORT.WORKING_PERIOD", value: filterResultData?.employeeStatusAtSegmentStatus },
            { phrase: "EXCEL-REPORT.EMPLOYEE", value: filters?.employeeName },
            { phrase: "EXCEL-REPORT.PERMISSION_TYPE", value: filters?.permissionTypeName },
            { phrase: "EXCEL-REPORT.ANNUAL_PERMISSION_REMAINED", value: filters?.remainedAnnualPermission },
            { phrase: "EXCEL-REPORT.NEXT_PERIOD", value: filters?.nextPeriod },
            { phrase: "EXCEL-REPORT.FORBIDDEN_INTERVAL_START", value: filterResultData?.forbiddenInterval?.startDate },
            { phrase: "EXCEL-REPORT.FORBIDDEN_INTERVAL_END", value: filterResultData?.forbiddenInterval?.endDate },
            { phrase: "EXCEL-REPORT.NOTE", value: filters?.note },
            { phrase: "EXCEL-REPORT.PROFILE_FILTER", value: filters?.profileFilter },
            { phrase: "EXCEL-REPORT.ROLE", value: filterResultData?.role },
            { phrase: "EXCEL-REPORT.UNIQUE_ID", value: filters?.uniqueId },
            { phrase: "EXCEL-REPORT.NAME", value: filters?.name },
            { phrase: "EXCEL-REPORT.SURNAME", value: filters?.surname },
            { phrase: "EXCEL-REPORT.EMAIL", value: filters?.email },
            { phrase: "EXCEL-REPORT.PHONE_NUMBER", value: filters?.phoneNumber },
            { phrase: "EXCEL-REPORT.ADDRESS", value: filters?.address },
            { phrase: "EXCEL-REPORT.USER_CREATION_DATE", value: filterResultData?.userCreatedDate },
            { phrase: "EXCEL-REPORT.ACCEPTANCE_STATUS", value: filterResultData?.acceptanceStatus },
            { phrase: "EXCEL-REPORT.PERMISSION_START_DATE", value: filterResultData?.permissionDate },
            { phrase: "PDF-REPORT.REGION_STATE_REGION", value: filterResultData?.region },
            { phrase: "EXCEL-REPORT.TRANSACTION_PERSON", value: filterResultData?.actionUser },
            { phrase: "EXCEL-REPORT.TRANSACTION_OWNER", value: filterResultData?.targetUser },
            { phrase: "EXCEL-REPORT.TERMINAL_NAME", value: filters?.terminalName },
            { phrase: "EXCEL-REPORT.BRAND", value: filterResultData?.deviceBrand },
            { phrase: "EXCEL-REPORT.MODEL", value: filters?.deviceModel },
            { phrase: "EXCEL-REPORT.CONNECTION_STATUS", value: filterResultData?.connectionStatus },
            { phrase: "EXCEL-REPORT.WARNING_STATUS", value: filterResultData?.warningLevel },
            { phrase: "EXCEL-REPORT.VISITOR", value: filters?.visitorName },
            { phrase: "EXCEL-REPORT.VISITED", value: filters?.visitedName },
            { phrase: "EXCEL-REPORT.VISIT_REASON", value: filters?.visitReason },
            { phrase: "EXCEL-REPORT.LICENSE_PLATE", value: filters?.visitorLicensePlate },
            { phrase: "PDF-REPORT.access_control_point", value: filterResultData?.accessControlPoint },
            { phrase: "EXCEL-REPORT.VISIT_STATUS", value: filterResultData?.visitStatus },
        ];
        const listFilters = [
            { phrase: "PDF-REPORT.IDENTITES", value: filterResultData?.users },
            { phrase: "EXCEL-REPORT.ORGANIZATION_UNITS", value: filterResultData?.organizationUnits },
            { phrase: "EXCEL-REPORT.REGIONS", value: filterResultData?.regions },
            { phrase: "EXCEL-REPORT.USER_GROUPS", value: filterResultData?.userGroups },
            { phrase: "PDF-REPORT.ACCESS_CONTROL_POINTS", value: filterResultData?.accessControlPoints },
            { phrase: "EXCEL-REPORT.AUDIT_ACTION_OWNER_IDENTITIES", value: filterResultData?.actionUsers },
            { phrase: "EXCEL-REPORT.AUDIT_IDENTITIES", value: filterResultData?.modifiedByUsers },
            { phrase: "EXCEL-REPORT.AUDIT_TERMINAL", value: filterResultData?.terminals },
            { phrase: "PDF-REPORT.SENSOR", value: filterResultData?.sensors },
            { phrase: "EXCEL-REPORT.WORK_STATUS", value: filterResultData?.workStatusFilters },
            { phrase: "REPORT.WORKPLANS", value: filterResultData?.workPlans },
            { phrase: "EXCEL-REPORT.EMPLOYEES", value: filterResultData?.employees },
            { phrase: "EXCEL-REPORT.TICKET", value: filters?.regionTickets },
            { phrase: "PDF-REPORT.ACCESS_REASONS", value: filterResultData?.accessReasons },
            { phrase: "PDF-REPORT.STATUS", value: filterResultData?.regionStates },
            { phrase: "EXCEL-REPORT.CREDENTIAL_TYPE", value: filterResultData?.credentialTypes },
            { phrase: "EXCEL-REPORT.VISITOR", value: filters?.visitors },
        ];
        const booleanFilters = [
            { phrase: "EXCEL-REPORT.IS_SYNCHRONIZED", value: filters?.isSynchronized },
            { phrase: "EXCEL-REPORT.INCLUDE_DESCENDANTS", value: filters?.applyOrganizationUnitFilterHierarchically },
            { phrase: "PDF-REPORT.ACCESS_LOGS_ONLY_VISITORS", value: filters?.showOnlyVisitors },
            { phrase: "PDF-REPORT.ACCESS_LOGS_ONLY_EXIT_BUTTONS", value: filters?.showOnlyExitButtons },
            { phrase: "PDF-REPORT.ACCESS_LOGS_ONLY_REMOTE_ACCESS", value: filters?.showOnlyRemoteAccess },
            { phrase: "PDF-REPORT.ACCESS_LOGS_ONLY_MANUALLY_INSERTED", value: filters?.showOnlyManuallyInserted },
            { phrase: "EXCEL-REPORT.ONLY_BLACKLIST", value: filters?.showOnlyBlacklist },
            { phrase: "EXCEL-REPORT.ONLY_TEMPORARY_PERMITTED", value: filters?.showOnlyTemporaryPermitted },
            { phrase: "EXCEL-REPORT.USER_FORBIDDEN", value: filters?.forbiddenUser },
        ];
        [...singleFilters, ...listFilters, ...booleanFilters].forEach((filter) => {
            if (filter.value) {
                addFilter(filter.phrase, filter.value);
            }
        });
        if (typeof filters?.employeeStatusAtSegmentWorking === "boolean") {
            addFilter("EXCEL-REPORT.WORK_STATUS", translatePhrase(filters.employeeStatusAtSegmentWorking ? "EXCEL-REPORT.WORKING" : "EXCEL-REPORT.NOT_WORKING"));
        }
        if (typeof filters?.employeeStatusAtSegmentInRegion === "boolean") {
            addFilter("EXCEL-REPORT.STATE_OF_BEING_IN_REGION", translatePhrase(filters.employeeStatusAtSegmentInRegion ? "EXCEL-REPORT.IN_REGION" : "EXCEL-REPORT.NOT_IN_REGION"));
        }
        if (typeof filters?.userAccountEnabled === "boolean") {
            addFilter("EXCEL-REPORT.USER_ACCOUNT", translatePhrase(filters.userAccountEnabled ? "EXCEL-REPORT.ACTIVE" : "EXCEL-REPORT.PASSIVE"));
        }
        if (typeof filters?.userOrganizationDisabled === "boolean") {
            addFilter("EXCEL-REPORT.USER", translatePhrase(filters.userOrganizationDisabled ? "EXCEL-REPORT.PASSIVE" : "EXCEL-REPORT.ACTIVE"));
        }
        if (filters?.visitorProfileFilters && filters.visitorProfileFilters.length > 0) {
            filters.visitorProfileFilters.forEach((filter) => {
                filtersResult.push(filter);
            });
        }
        return filtersResult.length > 0 ? filtersResult.join("\r\n") : translatePhrase("EXCEL-REPORT.NOT_EXISTING");
    }
    getProfileInfoHeaders() {
        if (!this._usersInfo || Object.keys(this._usersInfo).length === 0) {
            app_logs_1.logger.warn("No user info available to generate headers");
            return [];
        }
        const firstUserId = Object.keys(this._usersInfo)[0];
        const userInfo = this._usersInfo[firstUserId];
        if (!userInfo || typeof userInfo !== "object") {
            app_logs_1.logger.warn("Invalid user info data structure");
            return [];
        }
        const headers = [{ phrase: "EXCEL-REPORT.name_surname" }, { phrase: "EXCEL-REPORT.UNIQUE_ID" }];
        if (userInfo.reportableFields) {
            Object.values(userInfo.reportableFields).forEach((entry) => {
                headers.push({
                    value: entry.caption ?? "UNKNOWN_HEADER",
                });
            });
        }
        return headers;
    }
    getProfileInfoRow(userId) {
        const userInfo = this._usersInfo[userId];
        let row = [];
        if (userInfo) {
            Object.keys(userInfo).forEach((key) => {
                const value = userInfo[key];
                if (typeof value === "string") {
                    row.push({ value });
                }
            });
            if (userInfo.reportableFields) {
                Object.values(userInfo.reportableFields).forEach((entry) => {
                    row.push({
                        value: entry.value ?? "",
                    });
                });
            }
        }
        else {
            for (let i = 0; i < this._profileInfoOffset; i++) {
                row.push({ value: "" });
            }
        }
        return row;
    }
}
exports.ReportFactory = ReportFactory;
