"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.EmployeeDay = void 0;
const moment_1 = __importDefault(require("moment"));
const moment_range_1 = require("moment-range");
const pacs_utils_1 = require("./pacs.utils");
const workplan_day_1 = require("./workplan.day");
const dal_manager_1 = require("../../dal/dal.manager");
const app_enums_1 = require("../../app.enums");
const lodash_1 = require("lodash");
const app_logs_1 = require("../../app.logs");
class EmployeeDay {
    constructor(params) {
        this._isLive = false;
        this._permissionUsages = [];
        this._remainingWorkDurations = [];
        this._physicalAccessRanges = [];
        this._params = params;
        this._holidayUsages = params.holidays.map((m) => {
            return {
                id: m.id,
                definedRange: m.range,
                usedDuration: 0,
            };
        });
    }
    segmentsPrinter(sgm) {
        return ("\n" +
            sgm
                .map((s) => {
                return ((s.isBufferSegment ? "X - " : "  - ") +
                    s.range.duration("minutes") +
                    " - " +
                    s.range.start.format("HH:mm") +
                    " - " +
                    s.range.end.format("HH:mm") +
                    " ::: " +
                    s.workPlanTimeRangeType +
                    "-" +
                    (s.employeeTimeRangeType ? s.employeeTimeRangeType : " ") +
                    "-" +
                    (s.attendanceTimeRangeType ? s.attendanceTimeRangeType : " ") +
                    " " +
                    (s.attendanceData ? (s.attendanceData.tolerance ? s.attendanceData.tolerance : "0") : "0") +
                    " " +
                    JSON.stringify(s.conflicts));
            })
                .join("\n"));
    }
    getDurationOfSegment(s) {
        return s.range.duration("minutes");
    }
    convertWorkPlanSegmentsToEmployeeSegments(wp) {
        return wp.workPlanDurationSegments.map((w) => {
            return Object.assign((0, lodash_1.cloneDeep)(w), {
                employeeTimeRangeType: w.workPlanTimeRangeType,
                employeeData: undefined,
                conflicts: [],
            });
        });
    }
    process() {
        this._serialized = null;
        this._summary = null;
        this._processedAt = this._processedAt ? this._processedAt : (0, moment_1.default)();
        if (!this._segmentsWithWorkPlan) {
            this.processWorkPlanInformation();
        }
        if (!this._segmentsWithOffTimePermissions) {
            this.processOffTimePermissions();
        }
        if (!this._segmentsWithDeclaredWorkPermissions) {
            this.processDeclaredWorkPermissions();
        }
        if (!this._segmentsWithAccessTimes) {
            this.processAccessTimes();
        }
        if (!this._segmentsWithShouldBeIgnoredWorkPermissions) {
            this.processShouldBeIgnoredWorkPermissions();
            this.calculateTolerances();
        }
        if (!this._segmentsWithFlexibleWorkingHours) {
            this.processFlexibleWorkingHours();
        }
        if (!this._segmentsWithBreaks) {
            this.processBreaks();
        }
        if (!this._segmentsWithFlexibleWorkingHoursOvertimes) {
            this.processFlexibleWorkingHoursOvertimes();
        }
        if (!this._segmentsWithExtraWorkPermissions) {
            this.processExtraWorkPermissions();
        }
        if (!this._resultSegments) {
            this.processExtraWorkingBreaks();
        }
        for (const s of this._resultSegments) {
            s.status = this.getStatusFromTypes(s.attendanceTimeRangeType, s.employeeTimeRangeType);
        }
        this.processAccountedAttendances();
        this.processConflicts();
    }
    addConflictToSegments(range, id_1, typeId_1, type_1, id_2, typeId_2, type_2) {
        for (const s of this._resultSegments) {
            if (s.range.overlaps(range)) {
                if (!s.conflicts.find((ci) => ci.id === id_1)) {
                    s.conflicts.push({
                        id: id_1,
                        typeId: typeId_1,
                        type: type_1,
                    });
                }
                if (!s.conflicts.find((ci) => ci.id === id_2)) {
                    s.conflicts.push({
                        id: id_2,
                        typeId: typeId_2,
                        type: type_2,
                    });
                }
            }
        }
        if (type_1 === app_enums_1.enums.SegmentConflictItemType.Holiday || type_2 === app_enums_1.enums.SegmentConflictItemType.Holiday) {
            for (const s of this._segmentsWithOffTimePermissions) {
                if (s.range.overlaps(range)) {
                    if (!s.conflicts.find((ci) => ci.id === id_1)) {
                        s.conflicts.push({
                            id: id_1,
                            typeId: typeId_1,
                            type: type_1,
                        });
                    }
                    if (!s.conflicts.find((ci) => ci.id === id_2)) {
                        s.conflicts.push({
                            id: id_2,
                            typeId: typeId_2,
                            type: type_2,
                        });
                    }
                }
            }
        }
    }
    processConflicts() {
        for (const h of this._params.holidays) {
            for (const ot of this._params.offTimePermissions) {
                let intersect = h.range.intersect(ot.range);
                if (intersect && !ot.applyOnHolidays) {
                    this.addConflictToSegments(intersect, h.id, undefined, app_enums_1.enums.SegmentConflictItemType.Holiday, ot.id, ot.typeId, app_enums_1.enums.SegmentConflictItemType.Permission);
                }
            }
        }
        for (const ot of this._params.offTimePermissions) {
            for (const dw of this._params.declaredWorkPermissions) {
                let intersect = ot.range.intersect(dw.range);
                if (intersect) {
                    this.addConflictToSegments(intersect, ot.id, ot.typeId, app_enums_1.enums.SegmentConflictItemType.Permission, dw.id, dw.typeId, app_enums_1.enums.SegmentConflictItemType.Permission);
                }
            }
            for (const sbi of this._params.shouldBeIgnoredWorkPermissions) {
                let intersect = ot.range.intersect(sbi.range);
                if (intersect) {
                    this.addConflictToSegments(intersect, ot.id, ot.typeId, app_enums_1.enums.SegmentConflictItemType.Permission, sbi.id, sbi.typeId, app_enums_1.enums.SegmentConflictItemType.Permission);
                }
            }
            for (const ew of this._params.extraWorkPermissions) {
                let intersect = ot.range.intersect(ew.range);
                if (intersect) {
                    this.addConflictToSegments(intersect, ot.id, ot.typeId, app_enums_1.enums.SegmentConflictItemType.Permission, ew.id, ew.typeId, app_enums_1.enums.SegmentConflictItemType.Permission);
                }
            }
        }
        for (const dw of this._params.declaredWorkPermissions) {
            for (const sbi of this._params.shouldBeIgnoredWorkPermissions) {
                let intersect = dw.range.intersect(sbi.range);
                if (intersect) {
                    this.addConflictToSegments(intersect, dw.id, dw.typeId, app_enums_1.enums.SegmentConflictItemType.Permission, sbi.id, sbi.typeId, app_enums_1.enums.SegmentConflictItemType.Permission);
                }
            }
            for (const ew of this._params.extraWorkPermissions) {
                let intersect = dw.range.intersect(ew.range);
                if (intersect) {
                    this.addConflictToSegments(intersect, dw.id, dw.typeId, app_enums_1.enums.SegmentConflictItemType.Permission, ew.id, ew.typeId, app_enums_1.enums.SegmentConflictItemType.Permission);
                }
            }
        }
        for (const sbi of this._params.shouldBeIgnoredWorkPermissions) {
            for (const ew of this._params.extraWorkPermissions) {
                let intersect = sbi.range.intersect(ew.range);
                if (intersect) {
                    this.addConflictToSegments(intersect, sbi.id, sbi.typeId, app_enums_1.enums.SegmentConflictItemType.Permission, ew.id, ew.typeId, app_enums_1.enums.SegmentConflictItemType.Permission);
                }
            }
        }
    }
    processAccountedAttendances() {
        for (const s of this._resultSegments) {
            if (s.isBufferSegment) {
                continue;
            }
            for (const wpAccessRanges of this._physicalAccessRanges) {
                if (s.workPlanId && s.workPlanId === wpAccessRanges.workPlanId) {
                    for (const a of wpAccessRanges.accessRanges) {
                        if (!a.range.intersect(this._params.offsetDayRange)) {
                            continue;
                        }
                        if (s.range.start.isSame(a.range.start, "second")) {
                            if (!s.attendanceData.accountedPhysicalAttendance) {
                                s.attendanceData.accountedPhysicalAttendance = {
                                    arange: {
                                        range: a.range,
                                        startId: a.startId,
                                        regionId: a.regionId,
                                    },
                                };
                            }
                            else {
                                s.attendanceData.accountedPhysicalAttendance.arange = Object.assign({}, s.attendanceData.accountedPhysicalAttendance.arange, {
                                    range: a.range,
                                    startId: a.startId,
                                    regionId: a.regionId,
                                });
                            }
                        }
                        else if (s.range.start.isSame(a.range.end, "second")) {
                            if (!s.attendanceData.accountedPhysicalAttendance) {
                                s.attendanceData.accountedPhysicalAttendance = {
                                    arange: {
                                        range: a.range,
                                        startId: a.endId,
                                        regionId: a.regionId,
                                    },
                                };
                            }
                            else {
                                s.attendanceData.accountedPhysicalAttendance.arange = Object.assign({}, s.attendanceData.accountedPhysicalAttendance.arange, {
                                    range: a.range,
                                    startId: a.endId,
                                    regionId: a.regionId,
                                });
                            }
                        }
                        if (s.range.end.isSame(a.range.start, "second")) {
                            if (!s.attendanceData.accountedPhysicalAttendance) {
                                s.attendanceData.accountedPhysicalAttendance = {
                                    arange: {
                                        range: a.range,
                                        endId: a.startId,
                                        regionId: a.regionId,
                                    },
                                };
                            }
                            else {
                                s.attendanceData.accountedPhysicalAttendance.arange = Object.assign({}, s.attendanceData.accountedPhysicalAttendance.arange, {
                                    range: a.range,
                                    endId: a.startId,
                                    regionId: a.regionId,
                                });
                            }
                        }
                        else if (s.range.end.isSame(a.range.end, "second")) {
                            if (!s.attendanceData.accountedPhysicalAttendance) {
                                s.attendanceData.accountedPhysicalAttendance = {
                                    arange: {
                                        range: a.range,
                                        endId: a.endId,
                                        regionId: a.regionId,
                                    },
                                };
                            }
                            else {
                                s.attendanceData.accountedPhysicalAttendance.arange = Object.assign({}, s.attendanceData.accountedPhysicalAttendance.arange, {
                                    range: a.range,
                                    endId: a.endId,
                                    regionId: a.regionId,
                                });
                            }
                        }
                    }
                }
            }
        }
    }
    processWorkPlanInformation() {
        this._inheritedWorkPlanDurationSegments = [];
        this._segmentsWithWorkPlan = [
            {
                range: this._params.bufferedDayRange,
                workPlanTimeRangeType: app_enums_1.enums.WorkPlanTimeRangeType.OutOfWorkingHours,
                employeeTimeRangeType: app_enums_1.enums.EmployeeTimeRangeType.OutOfWorkingHours,
                employeeData: {},
                isBufferSegment: false,
                conflicts: [],
            },
        ];
        if (this._params.workPlanDayItems && this._params.workPlanDayItems.length > 0) {
            for (const wpDi of this._params.workPlanDayItems) {
                for (const wpds of wpDi.workPlanDurationSegments) {
                    if (wpds.range.overlaps(wpDi.dateIntersection)) {
                        this._inheritedWorkPlanDurationSegments.push(wpds);
                    }
                }
                let segments = [];
                for (const s of this._segmentsWithWorkPlan) {
                    let wint = s.range.intersect(wpDi.dateIntersection);
                    if (!wint) {
                        segments.push(s);
                        continue;
                    }
                    else {
                        let subs = s.range.subtract(wpDi.dateIntersection);
                        if (!subs || subs.length === 0) {
                            segments.push(...this.convertWorkPlanSegmentsToEmployeeSegments(wpDi));
                        }
                        else if (subs.length === 1) {
                            if (wpDi.dateIntersection.start.isAfter(s.range.start)) {
                                segments.push(Object.assign((0, lodash_1.cloneDeep)(s), {
                                    range: subs[0],
                                    employeeTimeRangeType: s.workPlanTimeRangeType,
                                    employeeData: undefined,
                                }));
                                segments.push(...this.convertWorkPlanSegmentsToEmployeeSegments(wpDi));
                            }
                            else {
                                segments.push(...this.convertWorkPlanSegmentsToEmployeeSegments(wpDi));
                                segments.push(Object.assign((0, lodash_1.cloneDeep)(s), {
                                    range: subs[0],
                                    employeeTimeRangeType: s.workPlanTimeRangeType,
                                    employeeData: undefined,
                                }));
                            }
                        }
                        else {
                            segments.push(Object.assign((0, lodash_1.cloneDeep)(s), {
                                range: subs[0],
                                employeeTimeRangeType: s.workPlanTimeRangeType,
                                employeeData: undefined,
                            }));
                            segments.push(...this.convertWorkPlanSegmentsToEmployeeSegments(wpDi));
                            segments.push(Object.assign((0, lodash_1.cloneDeep)(s), {
                                range: subs[1],
                                employeeTimeRangeType: s.workPlanTimeRangeType,
                                employeeData: undefined,
                            }));
                        }
                    }
                }
                this._segmentsWithWorkPlan = segments;
            }
        }
    }
    processOffTimePermissions() {
        this._segmentsWithOffTimePermissions = (0, lodash_1.cloneDeep)(this._segmentsWithWorkPlan);
        this._permissionUsages = this._permissionUsages.filter((u) => u.type != app_enums_1.enums.PPermissionType.OffTime);
        if (this._params.offTimePermissions) {
            this._permissionUsages.push(...this._params.offTimePermissions.map((m) => {
                return {
                    id: m.id,
                    typeId: m.typeId,
                    type: app_enums_1.enums.PPermissionType.OffTime,
                    definedRange: m.range,
                    usedDuration: 0,
                };
            }));
            if (this._params.offTimePermissions.length > 0) {
                this.invertOffTimePermissions(this._params.offTimePermissions);
            }
        }
    }
    getStatusFromTypes(at, et) {
        let result = {
            inRegion: null,
            status: null,
            working: null,
        };
        switch (at) {
            case app_enums_1.enums.AttendanceTimeRangeType.Absent:
                result.working = false;
                result.inRegion = false;
                break;
            case app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance:
                result.working = false;
                result.inRegion = true;
                break;
            case app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance:
                result.working = true;
                result.inRegion = true;
                break;
            case app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance:
                result.working = true;
                result.inRegion = false;
                break;
            default:
                break;
        }
        switch (et) {
            case app_enums_1.enums.EmployeeTimeRangeType.Break:
                result.status = app_enums_1.enums.EmployeeStatus.AtBreak;
                break;
            case app_enums_1.enums.EmployeeTimeRangeType.FlexibleWorkingHours:
            case app_enums_1.enums.EmployeeTimeRangeType.WorkingHours:
                result.status = app_enums_1.enums.EmployeeStatus.AtWorkingHours;
                break;
            case app_enums_1.enums.EmployeeTimeRangeType.Holiday:
                result.status = app_enums_1.enums.EmployeeStatus.AtHoliday;
                break;
            case app_enums_1.enums.EmployeeTimeRangeType.OffTimePermission:
                result.status = app_enums_1.enums.EmployeeStatus.AtOffTimePermission;
                break;
            case app_enums_1.enums.EmployeeTimeRangeType.OutOfWorkingHours:
                result.status = app_enums_1.enums.EmployeeStatus.AtOutOfWorkingHours;
                break;
            default:
                break;
        }
        return result;
    }
    processDeclaredWorkPermissions() {
        this._segmentsWithDeclaredWorkPermissions = [];
        for (const e of this._segmentsWithOffTimePermissions) {
            this._segmentsWithDeclaredWorkPermissions.push(Object.assign((0, lodash_1.cloneDeep)(e), {
                attendanceTimeRangeType: app_enums_1.enums.AttendanceTimeRangeType.Absent,
                attendanceData: {},
                status: null,
            }));
        }
        this._permissionUsages = this._permissionUsages.filter((u) => u.type != app_enums_1.enums.PPermissionType.DeclaredWork);
        if (this._params.declaredWorkPermissions) {
            this._permissionUsages.push(...this._params.declaredWorkPermissions.map((m) => {
                return {
                    id: m.id,
                    typeId: m.typeId,
                    type: app_enums_1.enums.PPermissionType.DeclaredWork,
                    definedRange: m.range,
                    usedDuration: 0,
                };
            }));
            if (this._params.declaredWorkPermissions.length > 0) {
                this.invertDeclaredWorksPermissions(this._params.declaredWorkPermissions);
            }
        }
    }
    processAccessTimes() {
        this._segmentsWithAccessTimes = (0, lodash_1.cloneDeep)(this._segmentsWithDeclaredWorkPermissions);
        if (this._params.physicalAccessTimes && this._params.physicalAccessTimes.length > 0) {
            this.segmentizeAccessTimes(this._params.physicalAccessTimes);
            if (this._physicalAccessRanges.length > 0) {
                this.invertAccessTimes();
            }
        }
    }
    processShouldBeIgnoredWorkPermissions() {
        this._segmentsWithShouldBeIgnoredWorkPermissions = (0, lodash_1.cloneDeep)(this._segmentsWithAccessTimes);
        this._permissionUsages = this._permissionUsages.filter((u) => u.type != app_enums_1.enums.PPermissionType.ShouldBeIgnoredWork);
        if (this._params.shouldBeIgnoredWorkPermissions) {
            this._permissionUsages.push(...this._params.shouldBeIgnoredWorkPermissions.map((m) => {
                return {
                    id: m.id,
                    typeId: m.typeId,
                    type: app_enums_1.enums.PPermissionType.ShouldBeIgnoredWork,
                    definedRange: m.range,
                    usedDuration: 0,
                };
            }));
            if (this._params.shouldBeIgnoredWorkPermissions.length > 0) {
                this.invertShouldBeIgnoredWorksPermissions(this._params.shouldBeIgnoredWorkPermissions);
            }
        }
    }
    processFlexibleWorkingHours() {
        this._segmentsWithFlexibleWorkingHours = (0, lodash_1.cloneDeep)(this._segmentsWithShouldBeIgnoredWorkPermissions);
        if (this._params.weeklyRemainingWorkDurations) {
            this._remainingWorkDurations = this._params.weeklyRemainingWorkDurations.map((m) => {
                return {
                    workPlanId: m.workPlanId,
                    weekly: m.duration,
                    today: m.duration,
                    used: 0,
                };
            });
        }
        this.invertFlexibleWorkingHours();
    }
    processBreaks() {
        this._segmentsWithBreaks = (0, lodash_1.cloneDeep)(this._segmentsWithFlexibleWorkingHours);
        this.invertFixedBreaks();
        this.invertClaimBreak();
    }
    processFlexibleWorkingHoursOvertimes() {
        this._segmentsWithFlexibleWorkingHoursOvertimes = (0, lodash_1.cloneDeep)(this._segmentsWithBreaks);
        this.invertFlexibleWorkingHoursOvertimes();
    }
    processExtraWorkPermissions() {
        this._segmentsWithExtraWorkPermissions = (0, lodash_1.cloneDeep)(this._segmentsWithFlexibleWorkingHoursOvertimes);
        this._permissionUsages = this._permissionUsages.filter((u) => u.type != app_enums_1.enums.PPermissionType.ExtraWork);
        if (this._params.extraWorkPermissions) {
            for (const m of this._params.extraWorkPermissions) {
                if (m.range.intersect(this._params.offsetDayRange)) {
                    this._permissionUsages.push({
                        id: m.id,
                        typeId: m.typeId,
                        type: app_enums_1.enums.PPermissionType.ExtraWork,
                        definedRange: m.range,
                        usedDuration: 0,
                    });
                }
            }
        }
        this.invertExtraWorkPermissions();
    }
    processExtraWorkingBreaks() {
        this._resultSegments = (0, lodash_1.cloneDeep)(this._segmentsWithExtraWorkPermissions);
        this.invertExtraWorkBreaks();
    }
    async accessTimesUpdated() {
        this._serialized = null;
        this._summary = null;
        this._resultSegments = null;
        this._segmentsWithExtraWorkPermissions = null;
        this._segmentsWithFlexibleWorkingHoursOvertimes = null;
        this._segmentsWithBreaks = null;
        this._segmentsWithFlexibleWorkingHours = null;
        this._segmentsWithShouldBeIgnoredWorkPermissions = null;
        this._segmentsWithAccessTimes = null;
        this._params.physicalAccessTimes = [];
        for (const wpr of this._params.workPlanDayItems) {
            let accessTimes = await dal_manager_1.dbManager.accessPacs2.getAccessTimesForWorkPlanIntersection({
                organizationId: this._params.organizationId,
                accessCheckType: wpr.attendanceControlMethod,
                offsetDayRange: wpr.attendanceControlMethod === app_enums_1.enums.WorkPlanAccessCheckType.Region ? this._params.bufferedDayRange : this._params.offsetDayRange,
                regionId: wpr.regionId,
                trx: null,
                userId: this._params.employee.userId,
                allowMobileCheckins: wpr.mobileCheckinAllowed,
                allowUnreliableCheckins: wpr.unreliableCheckinAllowed,
            });
            this._params.physicalAccessTimes.push({
                workPlanId: wpr.workPlanId,
                accessTimes: accessTimes,
            });
        }
    }
    async permissionUpdated() {
        const allPermissions = await dal_manager_1.dbManager.accessPacs2.getEmployeePermissionsForDay({
            organizationId: this._params.organizationId,
            offsetDayRange: this._params.offsetDayRange,
            bufferedDayRange: this._params.bufferedDayRange,
            userId: this._params.employee.userId,
        });
        this._serialized = null;
        this._summary = null;
        this._resultSegments = null;
        this._segmentsWithExtraWorkPermissions = null;
        this._segmentsWithFlexibleWorkingHoursOvertimes = null;
        this._segmentsWithBreaks = null;
        this._segmentsWithFlexibleWorkingHours = null;
        this._segmentsWithShouldBeIgnoredWorkPermissions = null;
        this._segmentsWithAccessTimes = null;
        this._segmentsWithDeclaredWorkPermissions = null;
        this._segmentsWithOffTimePermissions = null;
        this._params.offTimePermissions = allPermissions.offTimePermissions;
        this._params.shouldBeIgnoredWorkPermissions = allPermissions.shouldBeIgnoredWorkPermissions;
        this._params.declaredWorkPermissions = allPermissions.declaredWorkPermissions;
        this._params.extraWorkPermissions = allPermissions.extraWorkPermissions;
    }
    summarize() {
        this._summarizedAt = this._summarizedAt ? this._summarizedAt : (0, moment_1.default)();
        if (!this._resultSegments) {
            this.process();
        }
        for (const permission of this._permissionUsages) {
            permission.usedDuration = 0;
        }
        for (const holiday of this._holidayUsages) {
            holiday.usedDuration = 0;
        }
        let s = {
            dayRange: {
                start: this._params.offsetDayRange.start.toISOString(),
                end: this._params.offsetDayRange.end.toISOString(),
            },
            workplanDuration: 0,
            earlyStart: 0,
            lateStart: 0,
            earlyLeave: 0,
            lateLeave: 0,
            toleratedEarlyStart: 0,
            toleratedLateStart: 0,
            toleratedEarlyLeave: 0,
            toleratedLateLeave: 0,
            usedFixedBreakDuration: 0,
            automaticallyUsedFixedBreakDuration: 0,
            expectedFixedBreakDuration: 0,
            usedClaimBreakDuration: 0,
            automaticallyUsedClaimBreakDuration: 0,
            expectedClaimBreakDuration: 0,
            expectedExtraWorkBreakDuration: 0,
            automaticallyUsedExtraWorkBreakDuration: 0,
            physicallyInRegionDuration: 0,
            normalWorkDuration: 0,
            expectedWorkDuration: 0,
            missingWorkDuration: 0,
            extraWorkDuration: 0,
            usedHolidayDuration: 0,
            expectedHolidayDuration: 0,
            accountedDeclaredWorkDuration: 0,
            accountedOffTimePermissionDuration: 0,
            expectedOffTimePermissionDuration: 0,
            accountedIgnoredWorkDuration: 0,
            hasConflicts: false,
            hasRejectedPermissions: false,
            hasWaitingPermissions: false,
            score: null,
            expectedExtraWork: 0,
            ignoredExtraWork: 0,
            usedExtraWork: 0,
            unpairedAccesses: this._params.workPlanDayItems.find((data) => data.ignoreUnpairedAccesses) ? [] : undefined,
        };
        if (this._fixedBreakUsages && this._fixedBreakUsages.length > 0) {
            for (const f of this._fixedBreakUsages) {
                s.expectedFixedBreakDuration += f.maxAvailableDuration;
            }
        }
        if (this._claimBreakUsages && this._claimBreakUsages.length > 0) {
            for (const c of this._claimBreakUsages) {
                s.expectedClaimBreakDuration += c.claimedBreakDuration;
            }
        }
        if (this._params.extraWorkPermissions && this._params.extraWorkPermissions.length > 0) {
            for (const ewp of this._params.extraWorkPermissions) {
                for (const wpi of this._params.workPlanDayItems) {
                    if (!wpi.permissionRequiredForExtraWorking) {
                        continue;
                    }
                    const intersection = wpi.dateIntersection.intersect(ewp.range)?.intersect(this._params.offsetDayRange);
                    if (intersection) {
                        s.expectedExtraWork += intersection.duration("minutes");
                    }
                }
            }
        }
        for (const par of this._physicalAccessRanges) {
            for (const wpr of this._params.workPlanDayItems) {
                if (wpr.workPlanId === par.workPlanId) {
                    for (const ar of par.accessRanges) {
                        const r = ar.range.intersect(wpr.dateIntersection)?.intersect(this._params.offsetDayRange);
                        if (r) {
                            let d = r.duration("seconds");
                            if (this._isLive) {
                                if (ar.range.contains(this._summarizedAt)) {
                                    d = new moment_range_1.DateRange(r.start, this._summarizedAt).duration("seconds");
                                }
                                else if (this._summarizedAt.isBefore(ar.range.start)) {
                                    continue;
                                }
                            }
                            s.physicallyInRegionDuration += d;
                            if ((ar.endId == null || ar.startId == null) && wpr.ignoreUnpairedAccesses) {
                                s.unpairedAccesses.push({ start: ar.range.start, end: ar.range.end });
                            }
                        }
                    }
                }
            }
        }
        for (const w of this._segmentsWithWorkPlan) {
            if (w.workPlanTimeRangeType === app_enums_1.enums.WorkPlanTimeRangeType.WorkingHours && !w.isBufferSegment) {
                s.workplanDuration += w.range.duration("seconds");
            }
        }
        for (const wp of this._params.workPlanDayItems) {
            if (wp.fixedBreaks) {
                for (const _break of wp.fixedBreaks) {
                    if (_break.executeAutomaticallyBelowDuration > 0) {
                        const brStart = (0, moment_1.default)(wp.date)
                            .set("hours", parseInt(_break.startTime.substr(0, 2)))
                            .set("minutes", parseInt(_break.startTime.substr(2, 2)));
                        const brEnd = brStart
                            .clone()
                            .set("hours", parseInt(_break.endTime.substr(0, 2)))
                            .set("minutes", parseInt(_break.endTime.substr(2, 2)));
                        const brRange = new moment_range_1.DateRange(brStart, brEnd);
                        const intersect = brRange.intersect(wp.dateIntersection)?.intersect(this._params.offsetDayRange);
                        if (!intersect || !brRange.isEqual(intersect))
                            continue;
                        let allSegmentsValid = true;
                        for (const ws of this._segmentsWithWorkPlan) {
                            if (!ws.range.intersect(brRange)) {
                                continue;
                            }
                            if (ws.workPlanId !== wp.workPlanId || ws.workPlanTimeRangeType != app_enums_1.enums.WorkPlanTimeRangeType.WorkingHours) {
                                allSegmentsValid = false;
                            }
                        }
                        if (allSegmentsValid) {
                            s.workplanDuration -= _break.executeAutomaticallyBelowDuration * 60;
                        }
                    }
                }
            }
        }
        if (s.workplanDuration < 0) {
            s.workplanDuration = 0;
        }
        let result = [];
        for (let i = 0; i < this._segmentsWithOffTimePermissions.length; i++) {
            const e = this._segmentsWithOffTimePermissions[i];
            if (e.employeeTimeRangeType !== app_enums_1.enums.EmployeeTimeRangeType.WorkingHours) {
                continue;
            }
            const resultItem = {
                workPlanId: e.workPlanId,
                expectedRange: e.range.clone(),
                earlyStart: 0,
                lateStart: 0,
                earlyLeave: 0,
                lateLeave: 0,
                toleratedEarlyStart: 0,
                toleratedLateStart: 0,
                toleratedEarlyLeave: 0,
                toleratedLateLeave: 0,
            };
            result.push(resultItem);
        }
        this._calculatedTolerances = result;
        for (const c of this._resultSegments) {
            if (c.isBufferSegment) {
                continue;
            }
            let d = c.range.duration("seconds");
            const wpr = this._params.workPlanDayItems.find((wpdi) => wpdi.workPlanId === c.workPlanId && wpdi.dateIntersection.intersect(c.range));
            if (this._isLive) {
                if (c.range.contains(this._summarizedAt)) {
                    d = new moment_range_1.DateRange(c.range.start, this._summarizedAt).duration("seconds");
                }
                else if (this._summarizedAt.isBefore(c.range.start) || c.isBufferSegment) {
                    continue;
                }
            }
            try {
                let toleranceItem;
                if (c.attendanceData?.untolerated) {
                    switch (c.attendanceData.untolerated) {
                        case app_enums_1.enums.ToleranceType.EarlyLeave:
                            toleranceItem = this._calculatedTolerances.find((s) => s.workPlanId === c.workPlanId && s.expectedRange.overlaps(c.range, { adjacent: true }) && c.range.end.isSameOrBefore(s.expectedRange.end));
                            toleranceItem.earlyLeave += d;
                            s.earlyLeave += d;
                            break;
                        case app_enums_1.enums.ToleranceType.EarlyStart:
                            toleranceItem = this._calculatedTolerances.find((s) => s.workPlanId === c.workPlanId && s.expectedRange.overlaps(c.range, { adjacent: true }) && c.range.start.isSameOrBefore(s.expectedRange.start));
                            toleranceItem.earlyStart += d;
                            s.earlyStart += d;
                            break;
                        case app_enums_1.enums.ToleranceType.LateLeave:
                            toleranceItem = this._calculatedTolerances.find((s) => s.workPlanId === c.workPlanId && s.expectedRange.overlaps(c.range, { adjacent: true }) && c.range.end.isSameOrAfter(s.expectedRange.end));
                            toleranceItem.lateLeave += d;
                            s.lateLeave += d;
                            break;
                        case app_enums_1.enums.ToleranceType.LateStart:
                            toleranceItem = this._calculatedTolerances.find((s) => s.workPlanId === c.workPlanId && s.expectedRange.overlaps(c.range, { adjacent: true }) && c.range.start.isSameOrAfter(s.expectedRange.start));
                            toleranceItem.lateStart += d;
                            s.lateStart += d;
                            break;
                        default:
                            break;
                    }
                }
                if (c.attendanceData?.tolerance) {
                    switch (c.attendanceData.tolerance) {
                        case app_enums_1.enums.ToleranceType.EarlyLeave:
                            toleranceItem = this._calculatedTolerances.find((s) => s.workPlanId === c.workPlanId && s.expectedRange.overlaps(c.range, { adjacent: true }) && c.range.end.isSameOrBefore(s.expectedRange.end));
                            toleranceItem.toleratedEarlyLeave += d;
                            s.toleratedEarlyLeave += d;
                            break;
                        case app_enums_1.enums.ToleranceType.EarlyStart:
                            toleranceItem = this._calculatedTolerances.find((s) => s.workPlanId === c.workPlanId && s.expectedRange.overlaps(c.range, { adjacent: true }) && c.range.start.isSameOrBefore(s.expectedRange.start));
                            toleranceItem.toleratedEarlyStart += d;
                            s.toleratedEarlyStart += d;
                            break;
                        case app_enums_1.enums.ToleranceType.LateLeave:
                            toleranceItem = this._calculatedTolerances.find((s) => s.workPlanId === c.workPlanId && s.expectedRange.overlaps(c.range, { adjacent: true }) && c.range.end.isSameOrAfter(s.expectedRange.end));
                            toleranceItem.toleratedLateLeave += d;
                            s.toleratedLateLeave += d;
                            break;
                        case app_enums_1.enums.ToleranceType.LateStart:
                            toleranceItem = this._calculatedTolerances.find((s) => s.workPlanId === c.workPlanId && s.expectedRange.overlaps(c.range, { adjacent: true }) && c.range.start.isSameOrAfter(s.expectedRange.start));
                            toleranceItem.toleratedLateStart += d;
                            s.toleratedLateStart += d;
                            break;
                        default:
                            break;
                    }
                }
            }
            catch (error) {
                app_logs_1.logger.error("Error in tolerance calculation : " +
                    error +
                    "\r\nOrganization ID: " +
                    this._params.organizationId +
                    "\r\nUser ID: " +
                    this._params.employee.userId +
                    "\r\nDate: " +
                    this._params.date +
                    "\r\nSegment: " +
                    JSON.stringify(c));
            }
            if (c.employeeTimeRangeType === app_enums_1.enums.EmployeeTimeRangeType.Break) {
                switch (c.attendanceTimeRangeType) {
                    case app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance:
                        s.extraWorkDuration += c.attendanceData.tolerance ? 0 : d;
                        s.usedExtraWork += !wpr.permissionRequiredForExtraWorking || c.attendanceData.tolerance ? 0 : d;
                        break;
                    case app_enums_1.enums.AttendanceTimeRangeType.Absent:
                        switch (c.employeeData?.break?.method) {
                            case app_enums_1.enums.BreakMethod.Fixed:
                                if (c.employeeData?.break?.breakUsedDurationType == app_enums_1.enums.BreakUsedDurationType.Manually) {
                                    s.usedFixedBreakDuration += d;
                                }
                                else if (c.employeeData?.break?.breakUsedDurationType == app_enums_1.enums.BreakUsedDurationType.Automatically) {
                                    s.automaticallyUsedFixedBreakDuration += d;
                                }
                                break;
                            case app_enums_1.enums.BreakMethod.Claim:
                                if (c.employeeData?.break?.breakUsedDurationType == app_enums_1.enums.BreakUsedDurationType.Manually) {
                                    s.usedClaimBreakDuration += d;
                                }
                                else if (c.employeeData?.break?.breakUsedDurationType == app_enums_1.enums.BreakUsedDurationType.Automatically) {
                                    s.automaticallyUsedClaimBreakDuration += d;
                                }
                                break;
                            default:
                                break;
                        }
                        break;
                    case app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance:
                    case app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance:
                        switch (c.employeeData?.break?.method) {
                            case app_enums_1.enums.BreakMethod.Fixed:
                                s.automaticallyUsedFixedBreakDuration += d;
                                break;
                            case app_enums_1.enums.BreakMethod.Claim:
                                s.automaticallyUsedClaimBreakDuration += d;
                                break;
                            case app_enums_1.enums.BreakMethod.ExtraWork:
                                s.automaticallyUsedExtraWorkBreakDuration += d;
                                s.expectedExtraWorkBreakDuration += d;
                                break;
                            default:
                                break;
                        }
                        break;
                    default:
                        break;
                }
            }
            else if (c.employeeTimeRangeType === app_enums_1.enums.EmployeeTimeRangeType.Holiday) {
                s.expectedHolidayDuration += d;
                switch (c.attendanceTimeRangeType) {
                    case app_enums_1.enums.AttendanceTimeRangeType.Absent:
                    case app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance:
                        s.usedHolidayDuration += d;
                        for (const holidayId of c.workPlanData.holidayIds) {
                            let holidayUsage = this._holidayUsages.find((hh) => hh.id === holidayId);
                            if (holidayUsage) {
                                holidayUsage.usedDuration += d;
                            }
                        }
                        break;
                    case app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance:
                        s.extraWorkDuration += c.attendanceData.tolerance ? 0 : d;
                        s.usedExtraWork += !wpr.permissionRequiredForExtraWorking || c.attendanceData.tolerance ? 0 : d;
                        break;
                    case app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance:
                        s.extraWorkDuration += c.attendanceData.tolerance ? 0 : d;
                        s.usedExtraWork += !wpr.permissionRequiredForExtraWorking || c.attendanceData.tolerance ? 0 : d;
                        s.accountedDeclaredWorkDuration += d;
                        let permissionUsage = this._permissionUsages.find((p) => p.id === c.attendanceData.declaredAttendance?.id);
                        if (permissionUsage) {
                            permissionUsage.usedDuration += d;
                        }
                        break;
                    default:
                        break;
                }
            }
            else if (c.employeeTimeRangeType === app_enums_1.enums.EmployeeTimeRangeType.OffTimePermission) {
                s.expectedOffTimePermissionDuration += d;
                switch (c.attendanceTimeRangeType) {
                    case app_enums_1.enums.AttendanceTimeRangeType.Absent:
                    case app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance:
                        s.accountedOffTimePermissionDuration += d;
                        let permissionUsage = this._permissionUsages.find((p) => p.id === c.employeeData.offTimePermissionId);
                        if (permissionUsage) {
                            permissionUsage.usedDuration += d;
                        }
                        break;
                    case app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance:
                    case app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance:
                        s.extraWorkDuration += c.attendanceData.tolerance ? 0 : d;
                        s.usedExtraWork += !wpr.permissionRequiredForExtraWorking || c.attendanceData.tolerance ? 0 : d;
                        break;
                    default:
                        break;
                }
            }
            else if (c.employeeTimeRangeType === app_enums_1.enums.EmployeeTimeRangeType.OutOfWorkingHours) {
                switch (c.attendanceTimeRangeType) {
                    case app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance:
                        s.extraWorkDuration += c.attendanceData.tolerance ? 0 : d;
                        s.usedExtraWork += !wpr.permissionRequiredForExtraWorking || c.attendanceData.tolerance ? 0 : d;
                        break;
                    case app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance:
                        s.extraWorkDuration += c.attendanceData.tolerance ? 0 : d;
                        s.usedExtraWork += !wpr.permissionRequiredForExtraWorking || c.attendanceData.tolerance ? 0 : d;
                        s.accountedDeclaredWorkDuration += d;
                        let permissionUsage = this._permissionUsages.find((p) => p.id === c.attendanceData.declaredAttendance?.id);
                        if (permissionUsage) {
                            permissionUsage.usedDuration += d;
                        }
                        break;
                    case app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance:
                        if (wpr.permissionRequiredForExtraWorking) {
                            s.ignoredExtraWork += d;
                        }
                        break;
                    default:
                        break;
                }
            }
            else if (c.employeeTimeRangeType === app_enums_1.enums.EmployeeTimeRangeType.WorkingHours || c.employeeTimeRangeType === app_enums_1.enums.EmployeeTimeRangeType.FlexibleWorkingHours) {
                s.expectedWorkDuration += d;
                if (c.attendanceTimeRangeType === app_enums_1.enums.AttendanceTimeRangeType.Absent) {
                    if (c.attendanceData.tolerance) {
                        s.normalWorkDuration += d;
                    }
                    else {
                        s.missingWorkDuration += d;
                    }
                }
                else if (c.attendanceTimeRangeType === app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance) {
                    s.missingWorkDuration += d;
                    s.accountedIgnoredWorkDuration += d;
                    let permissionUsage = this._permissionUsages.find((p) => p.id === c.attendanceData.ignoredPhysicalAttendance?.shouldBeIgnoredPermissionId);
                    if (permissionUsage) {
                        permissionUsage.usedDuration += d;
                    }
                }
                else if (c.attendanceTimeRangeType === app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance) {
                    if (c.attendanceData.tolerance) {
                        s.missingWorkDuration += d;
                    }
                    else {
                        s.normalWorkDuration += d;
                    }
                }
                else if (c.attendanceTimeRangeType === app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance) {
                    s.normalWorkDuration += d;
                    s.accountedDeclaredWorkDuration += d;
                    let permissionUsage = this._permissionUsages.find((p) => p.id === c.attendanceData.declaredAttendance?.id);
                    if (permissionUsage) {
                        permissionUsage.usedDuration += d;
                    }
                }
            }
            s.hasConflicts = s.hasConflicts || c.conflicts.length > 0;
            if (c.attendanceData.extraWorkPermissionId && c.attendanceData.extraWorkPermissionTypeId) {
                let permissionUsage = this._permissionUsages.find((p) => p.id === c.attendanceData.extraWorkPermissionId);
                if (permissionUsage) {
                    permissionUsage.usedDuration += d;
                }
            }
        }
        s.workplanDuration = Math.round(s.workplanDuration / 60);
        s.physicallyInRegionDuration = Math.round(s.physicallyInRegionDuration / 60);
        s.expectedWorkDuration = Math.round(s.expectedWorkDuration / 60);
        s.extraWorkDuration = Math.round(s.extraWorkDuration / 60);
        s.missingWorkDuration = Math.round(s.missingWorkDuration / 60);
        s.normalWorkDuration = Math.round(s.normalWorkDuration / 60);
        s.accountedDeclaredWorkDuration = Math.round(s.accountedDeclaredWorkDuration / 60);
        s.accountedIgnoredWorkDuration = Math.round(s.accountedIgnoredWorkDuration / 60);
        s.accountedOffTimePermissionDuration = Math.round(s.accountedOffTimePermissionDuration / 60);
        s.expectedHolidayDuration = Math.round(s.expectedHolidayDuration / 60);
        s.expectedOffTimePermissionDuration = Math.round(s.expectedOffTimePermissionDuration / 60);
        s.usedHolidayDuration = Math.round(s.usedHolidayDuration / 60);
        s.usedExtraWork = Math.round(s.usedExtraWork / 60);
        s.ignoredExtraWork = Math.round(s.ignoredExtraWork / 60);
        s.normalWorkDuration = s.normalWorkDuration < 0 ? 0 : s.normalWorkDuration;
        s.earlyStart = Math.round(s.earlyStart / 60);
        s.earlyLeave = Math.round(s.earlyLeave / 60);
        s.lateStart = Math.round(s.lateStart / 60);
        s.lateLeave = Math.round(s.lateLeave / 60);
        s.toleratedEarlyStart = Math.round(s.toleratedEarlyStart / 60);
        s.toleratedEarlyLeave = Math.round(s.toleratedEarlyLeave / 60);
        s.toleratedLateStart = Math.round(s.toleratedLateStart / 60);
        s.toleratedLateLeave = Math.round(s.toleratedLateLeave / 60);
        s.usedClaimBreakDuration = Math.round(s.usedClaimBreakDuration / 60);
        s.automaticallyUsedClaimBreakDuration = Math.round(s.automaticallyUsedClaimBreakDuration / 60);
        s.usedFixedBreakDuration = Math.round(s.usedFixedBreakDuration / 60);
        s.automaticallyUsedFixedBreakDuration = Math.round(s.automaticallyUsedFixedBreakDuration / 60);
        s.automaticallyUsedExtraWorkBreakDuration = Math.round(s.automaticallyUsedExtraWorkBreakDuration / 60);
        s.expectedExtraWorkBreakDuration = Math.round(s.expectedExtraWorkBreakDuration / 60);
        for (const item of this._calculatedTolerances) {
            item.earlyLeave = Math.round(item.earlyLeave / 60);
            item.earlyStart = Math.round(item.earlyStart / 60);
            item.lateLeave = Math.round(item.lateLeave / 60);
            item.lateStart = Math.round(item.lateStart / 60);
            item.toleratedEarlyLeave = Math.round(item.toleratedEarlyLeave / 60);
            item.toleratedEarlyStart = Math.round(item.toleratedEarlyStart / 60);
            item.toleratedLateLeave = Math.round(item.toleratedLateLeave / 60);
            item.toleratedLateStart = Math.round(item.toleratedLateStart / 60);
        }
        for (const pu of this._permissionUsages) {
            pu.usedDuration = Math.round(pu.usedDuration / 60);
        }
        for (const hu of this._holidayUsages) {
            hu.usedDuration = Math.round(hu.usedDuration / 60);
        }
        if (s.expectedWorkDuration > 0) {
            s.score = Math.ceil(((s.normalWorkDuration + s.extraWorkDuration) / s.expectedWorkDuration) * 100);
            if (s.score > 100) {
                s.score = 100;
            }
        }
        this._summary = s;
        this._summarizedAt = null;
    }
    serialize() {
        if (!this._summary) {
            this.summarize();
        }
        let serializedSummary = {
            r: {
                s: this._summary.dayRange.start,
                e: this._summary.dayRange.end,
            },
            wd: this._summary.workplanDuration,
            n: this._summary.normalWorkDuration,
            e: this._summary.extraWorkDuration,
            m: this._summary.missingWorkDuration,
            ew: this._summary.expectedWorkDuration,
            el: this._summary.earlyLeave,
            es: this._summary.earlyStart,
            ll: this._summary.lateLeave,
            ls: this._summary.lateStart,
            tel: this._summary.toleratedEarlyLeave,
            tes: this._summary.toleratedEarlyStart,
            tll: this._summary.toleratedLateLeave,
            tls: this._summary.toleratedLateStart,
            ao: this._summary.accountedOffTimePermissionDuration,
            eo: this._summary.expectedOffTimePermissionDuration,
            s: this._summary.score,
            dw: this._summary.accountedDeclaredWorkDuration,
            iw: this._summary.accountedIgnoredWorkDuration,
            ufb: Math.round(this._summary.usedFixedBreakDuration),
            afb: Math.round(this._summary.automaticallyUsedFixedBreakDuration),
            ucb: Math.round(this._summary.usedClaimBreakDuration),
            acb: Math.round(this._summary.automaticallyUsedClaimBreakDuration),
            aeb: Math.round(this._summary.automaticallyUsedExtraWorkBreakDuration),
            eeb: Math.round(this._summary.expectedExtraWorkBreakDuration),
            efb: this._summary.expectedFixedBreakDuration,
            ecb: this._summary.expectedClaimBreakDuration,
            uh: this._summary.usedHolidayDuration,
            eh: this._summary.expectedHolidayDuration,
            pr: this._summary.physicallyInRegionDuration,
            eew: this._summary.expectedExtraWork,
            iew: this._summary.ignoredExtraWork,
            uew: this._summary.usedExtraWork,
        };
        if (this._summary.unpairedAccesses) {
            serializedSummary.ua = this._summary.unpairedAccesses.map((item) => {
                return {
                    s: item.start,
                    e: item.end,
                };
            });
        }
        const workplanIds = this._params.workPlanDayItems
            .filter((wpdi) => wpdi.dateIntersection.intersect(this._params.offsetDayRange))
            .map((wpdi) => wpdi.workPlanId)
            .filter((val, index, self) => {
            return self.indexOf(val) === index;
        });
        const serializedAccessRanges = [];
        let start = undefined;
        let end = undefined;
        for (const par of this._physicalAccessRanges) {
            if (!workplanIds.includes(par.workPlanId)) {
                continue;
            }
            let filteredAccessRanges = par.accessRanges.filter((ar) => ar.range.start.isSameOrAfter(this._params.offsetDayRange.start) && ar.range.end.isSameOrBefore(this._params.offsetDayRange.end));
            if (filteredAccessRanges.length > 0) {
                if (!start || (start && start.isAfter(filteredAccessRanges[0].range.start))) {
                    start = filteredAccessRanges[0].range.start;
                }
                if (!end || (end && end.isBefore(filteredAccessRanges[filteredAccessRanges.length - 1].range.end))) {
                    end = filteredAccessRanges[filteredAccessRanges.length - 1].range.end;
                }
            }
            serializedAccessRanges.push({
                w: par.workPlanId,
                a: filteredAccessRanges.map((ar) => {
                    return {
                        r: ar.regionId,
                        s: ar.range.start.toDate(),
                        e: ar.range.end.toDate(),
                        si: ar.startId,
                        ei: ar.endId,
                    };
                }),
            });
        }
        this._serialized = {
            d: (0, moment_1.default)(this._params.date).toDate(),
            w: workplanIds,
            r: this._remainingWorkDurations
                .filter((wrwd) => workplanIds.includes(wrwd.workPlanId))
                .map((wrwd) => {
                return {
                    w: wrwd.workPlanId,
                    r: wrwd.weekly,
                };
            }),
            c: this._resultSegments
                .filter((s) => !s.isBufferSegment)
                .map((s) => {
                return {
                    wp: s.workPlanId,
                    s: s.range.start.toDate(),
                    e: s.range.end.toDate(),
                    wt: s.workPlanTimeRangeType,
                    et: s.employeeTimeRangeType,
                    at: s.attendanceTimeRangeType,
                    wd: {
                        h: s.workPlanData?.holidayIds,
                    },
                    ed: s.employeeData
                        ? {
                            p: s.employeeData.offTimePermissionId,
                            pt: s.employeeData.offTimePermissionTypeId,
                            b: s.employeeData.break
                                ? {
                                    m: s.employeeData.break.method,
                                    t: s.employeeData.break.type,
                                }
                                : undefined,
                        }
                        : undefined,
                    ad: s.attendanceData
                        ? {
                            i: s.attendanceData.ignoredPhysicalAttendance
                                ? {
                                    p: s.attendanceData.ignoredPhysicalAttendance.shouldBeIgnoredPermissionId,
                                    pt: s.attendanceData.ignoredPhysicalAttendance.shouldBeIgnoredPermissionTypeId,
                                    s: s.attendanceData.ignoredPhysicalAttendance.range.start.toDate(),
                                    e: s.attendanceData.ignoredPhysicalAttendance.range.end.toDate(),
                                }
                                : undefined,
                            a: s.attendanceData.accountedPhysicalAttendance
                                ? {
                                    s: s.attendanceData.accountedPhysicalAttendance.arange.range.start.toDate(),
                                    e: s.attendanceData.accountedPhysicalAttendance.arange.range.end.toDate(),
                                    si: s.attendanceData.accountedPhysicalAttendance.arange.startId,
                                    ei: s.attendanceData.accountedPhysicalAttendance.arange.endId,
                                    r: s.attendanceData.accountedPhysicalAttendance.arange.regionId,
                                }
                                : undefined,
                            d: s.attendanceData.declaredAttendance?.id,
                            dt: s.attendanceData.declaredAttendance?.typeId,
                            t: s.attendanceData.tolerance,
                            xp: s.attendanceData.extraWorkPermissionId,
                            xt: s.attendanceData.extraWorkPermissionTypeId,
                            ua: s.attendanceData.unpairedAccess,
                        }
                        : undefined,
                    es: {
                        r: s.status.inRegion,
                        s: s.status.status,
                        w: s.status.working,
                    },
                    cs: s.conflicts.map((conflict) => {
                        return {
                            t: conflict.type,
                            i: conflict.id,
                            ti: conflict.typeId,
                        };
                    }),
                };
            }),
            e: this._segmentsWithOffTimePermissions
                .filter((s) => !s.isBufferSegment)
                .map((s) => {
                return {
                    wp: s.workPlanId,
                    s: s.range.start.toDate(),
                    e: s.range.end.toDate(),
                    wt: s.workPlanTimeRangeType,
                    et: s.employeeTimeRangeType,
                    wd: {
                        h: s.workPlanData?.holidayIds,
                    },
                    ed: s.employeeData
                        ? {
                            p: s.employeeData.offTimePermissionId,
                            pt: s.employeeData.offTimePermissionTypeId,
                            b: s.employeeData.break
                                ? {
                                    m: s.employeeData.break.method,
                                    t: s.employeeData.break.type,
                                }
                                : undefined,
                        }
                        : undefined,
                    cs: s.conflicts.map((conflict) => {
                        return {
                            t: conflict.type,
                            i: conflict.id,
                            ti: conflict.typeId,
                        };
                    }),
                };
            }),
            ws: this._inheritedWorkPlanDurationSegments
                .filter((s) => !s.isBufferSegment)
                .map((iwpds) => {
                return {
                    wp: iwpds.workPlanId,
                    wt: iwpds.workPlanTimeRangeType,
                    s: iwpds.range.start.toDate(),
                    e: iwpds.range.end.toDate(),
                    wd: !iwpds.workPlanData
                        ? undefined
                        : {
                            h: iwpds.workPlanData.holidayIds,
                        },
                };
            }),
            t: this._calculatedTolerances
                .filter((ct) => ct.expectedRange.intersect(this._params.offsetDayRange) && workplanIds.includes(ct.workPlanId))
                .map((s) => {
                return {
                    wp: s.workPlanId,
                    s: s.expectedRange.start.toDate(),
                    e: s.expectedRange.end.toDate(),
                    u: {
                        es: s.earlyStart,
                        ls: s.lateStart,
                        el: s.earlyLeave,
                        ll: s.lateLeave,
                    },
                    t: {
                        es: s.toleratedEarlyStart,
                        ls: s.toleratedLateStart,
                        el: s.toleratedEarlyLeave,
                        ll: s.toleratedLateLeave,
                    },
                };
            }),
            fb: this._fixedBreakUsages.map((fbu) => {
                return {
                    d: {
                        wp: fbu.workPlanId,
                        t: fbu.break.type,
                        m: app_enums_1.enums.BreakMethod.Fixed,
                        s: (0, moment_1.default)(fbu.date)
                            .hour(parseInt(fbu.break.startTime.substr(0, 2)))
                            .minute(parseInt(fbu.break.startTime.substr(2, 2))),
                        e: (0, moment_1.default)(fbu.date)
                            .hour(parseInt(fbu.break.endTime.substr(0, 2)))
                            .minute(parseInt(fbu.break.endTime.substr(2, 2))),
                        ir: fbu.intersectionRatio,
                        al: fbu.maxAvailableDuration,
                        ex: fbu.break.executeAutomaticallyBelowDuration,
                    },
                    u: fbu.usedDuration,
                    a: fbu.automaticallyUsedDuration,
                    rs: fbu.usedRanges.map((ur) => {
                        return {
                            s: ur.start.toDate(),
                            e: ur.end.toDate(),
                        };
                    }),
                };
            }),
            cb: this._claimBreakUsages.map((cbu) => {
                return {
                    d: {
                        wp: cbu.workPlanId,
                        w: cbu.break.workDuration,
                        b: cbu.break.breakDuration,
                        t: cbu.break.type,
                        m: app_enums_1.enums.BreakMethod.Claim,
                        ex: cbu.break.executeAutomaticallyBelowDuration,
                    },
                    c: cbu.claimedBreakDuration,
                    u: cbu.usedDuration,
                    a: cbu.automaticallyUsedDuration,
                    rs: cbu.usedRanges.map((ur) => {
                        return {
                            s: ur.start.toDate(),
                            e: ur.end.toDate(),
                        };
                    }),
                };
            }),
            ewb: this._extraWorkBreakUsages.map((ebu) => {
                return {
                    d: {
                        wp: ebu.workPlanId,
                        mw: ebu.break.minWorkDuration,
                        b: ebu.break.breakDuration,
                        t: ebu.break.type,
                        m: app_enums_1.enums.BreakMethod.ExtraWork,
                    },
                    u: ebu.usedDuration,
                    a: ebu.automaticallyUsedDuration,
                    rs: ebu.usedRanges.map((ur) => {
                        return {
                            s: ur.start.toDate(),
                            e: ur.end.toDate(),
                        };
                    }),
                };
            }),
            ar: serializedAccessRanges,
            ex: {
                s: start,
                e: end,
                el: serializedSummary.el,
                ls: serializedSummary.ls,
                x: serializedSummary.e,
                m: serializedSummary.m,
            },
            p: this._permissionUsages
                ?.filter((pu) => pu.definedRange.intersect(this._params.offsetDayRange))
                .map((pu) => {
                return {
                    p: pu.id,
                    pt: pu.typeId,
                    r: {
                        s: pu.definedRange.start.toDate(),
                        e: pu.definedRange.end.toDate(),
                    },
                    u: pu.usedDuration,
                    t: pu.type,
                };
            }),
            h: this._holidayUsages
                .filter((hu) => hu.definedRange.intersect(this._params.offsetDayRange))
                .map((hu) => {
                return {
                    h: hu.id,
                    u: hu.usedDuration,
                    r: {
                        s: hu.definedRange.start.toDate(),
                        e: hu.definedRange.end.toDate(),
                    },
                };
            }),
            pt: this._processedAt.toDate(),
            s: serializedSummary,
        };
        return this._serialized;
    }
    invertExtraWorkBreaks() {
        const result = [];
        for (const wp of this._params.workPlanDayItems) {
            if (!wp.extraWorkBreaks || wp.extraWorkBreaks.length < 1) {
                continue;
            }
            for (const _break of wp.extraWorkBreaks) {
                const resultItem = {
                    break: _break,
                    date: wp.date,
                    automaticallyUsedDuration: 0,
                    usedDuration: 0,
                    usedRanges: [],
                    workPlanId: wp.workPlanId,
                    expectedBreakDuration: _break.breakDuration,
                };
                const hStart = parseInt(_break.periodStartTime.substr(0, 2));
                const mStart = parseInt(_break.periodStartTime.substr(2, 2));
                const hEnd = parseInt(_break.periodEndTime.substr(0, 2));
                const mEnd = parseInt(_break.periodEndTime.substr(2, 2));
                const brRanges = [];
                const startOfDay = wp.dateIntersection.start.clone().startOf("day");
                if (_break.periodStartTime > _break.periodEndTime) {
                    let brStart1 = startOfDay.clone().add(1, "days").startOf("day");
                    let brEnd1 = brStart1.clone().set("hours", hEnd).set("minutes", mEnd);
                    let brR1 = new moment_range_1.DateRange(brStart1, brEnd1);
                    let brR1Intersection = brR1.intersect(wp.dateIntersection);
                    if (brR1Intersection) {
                        brRanges.push(brR1Intersection);
                    }
                    let brStart2 = startOfDay.clone().set("hours", hStart).set("minutes", mStart);
                    let brEnd2 = brStart2.clone().add(1, "days").startOf("day");
                    let brR2 = new moment_range_1.DateRange(brStart2, brEnd2);
                    let brR2Intersection = brR2.intersect(wp.dateIntersection);
                    if (brR2Intersection) {
                        brRanges.push(brR2Intersection);
                    }
                }
                else {
                    let brStart = startOfDay.clone().set("hours", hStart).set("minutes", mStart);
                    let brEnd = brStart.clone().set("hours", hEnd).set("minutes", mEnd);
                    let br = new moment_range_1.DateRange(brStart, brEnd);
                    let brRIntersection = br.intersect(wp.dateIntersection);
                    if (brRIntersection) {
                        brRanges.push(brRIntersection);
                    }
                }
                if (brRanges.length === 0) {
                    continue;
                }
                let usedBreakDuration = 0;
                let attendanceDuration = 0;
                for (const c of this._resultSegments) {
                    if (c.workPlanId !== wp.workPlanId ||
                        !wp.dateIntersection.intersect(c.range) ||
                        c.isBufferSegment ||
                        !brRanges.find((r) => r.intersect(c.range) || c.attendanceData?.tolerance)) {
                        continue;
                    }
                    if ((c.employeeTimeRangeType === app_enums_1.enums.EmployeeTimeRangeType.Break ||
                        c.employeeTimeRangeType === app_enums_1.enums.EmployeeTimeRangeType.Holiday ||
                        c.employeeTimeRangeType === app_enums_1.enums.EmployeeTimeRangeType.OffTimePermission ||
                        c.employeeTimeRangeType === app_enums_1.enums.EmployeeTimeRangeType.OutOfWorkingHours) &&
                        (c.attendanceTimeRangeType === app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance ||
                            c.attendanceTimeRangeType === app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance) &&
                        !c.attendanceData.tolerance) {
                        attendanceDuration += c.range.duration("minutes");
                    }
                }
                if (attendanceDuration < _break.minWorkDuration) {
                    continue;
                }
                for (const br of brRanges) {
                    let segments = [];
                    for (const c of this._resultSegments) {
                        if (c.workPlanId !== wp.workPlanId || !c.range.intersect(br) || c.isBufferSegment || c.attendanceData?.tolerance || usedBreakDuration >= _break.breakDuration) {
                            segments.push(c);
                            continue;
                        }
                        if ((c.employeeTimeRangeType === app_enums_1.enums.EmployeeTimeRangeType.Break && c.attendanceTimeRangeType === app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance) ||
                            ((c.employeeTimeRangeType === app_enums_1.enums.EmployeeTimeRangeType.Holiday ||
                                c.employeeTimeRangeType === app_enums_1.enums.EmployeeTimeRangeType.OffTimePermission ||
                                c.employeeTimeRangeType === app_enums_1.enums.EmployeeTimeRangeType.OutOfWorkingHours) &&
                                (c.attendanceTimeRangeType === app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance ||
                                    c.attendanceTimeRangeType === app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance))) {
                            const segmentDuration = c.range.duration("minutes", true);
                            const availableDuration = _break.breakDuration - usedBreakDuration;
                            if (segmentDuration <= availableDuration) {
                                segments.push(Object.assign((0, lodash_1.cloneDeep)(c), {
                                    employeeTimeRangeType: app_enums_1.enums.EmployeeTimeRangeType.Break,
                                    attendanceTimeRangeType: app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance,
                                    employeeData: Object.assign({}, (0, lodash_1.cloneDeep)(c.employeeData), {
                                        break: {
                                            method: app_enums_1.enums.BreakMethod.ExtraWork,
                                            type: resultItem.break.type,
                                        },
                                    }),
                                    attendanceData: Object.assign((0, lodash_1.cloneDeep)(c.attendanceData), { untolerated: undefined }),
                                }));
                                resultItem.usedRanges.push(c.range);
                                usedBreakDuration += segmentDuration;
                            }
                            else {
                                let cint;
                                if (wp.workPlanDurationSegments.find((s) => s.workPlanId === c.workPlanId &&
                                    s.workPlanTimeRangeType === app_enums_1.enums.WorkPlanTimeRangeType.WorkingHours &&
                                    s.range.overlaps(c.range, { adjacent: true }) &&
                                    c.range.end.isSameOrBefore(s.range.end))) {
                                    cint = c.range.intersect(new moment_range_1.DateRange(c.range.start, c.range.start.clone().add(availableDuration, "minutes")));
                                }
                                else {
                                    cint = c.range.intersect(new moment_range_1.DateRange(c.range.end.clone().subtract(availableDuration, "minutes"), c.range.end));
                                }
                                segments.push(...(0, pacs_utils_1.splitDateRangeAccordingToIntersections)(c.range, cint, (0, lodash_1.cloneDeep)(c), Object.assign((0, lodash_1.cloneDeep)(c), {
                                    employeeTimeRangeType: app_enums_1.enums.EmployeeTimeRangeType.Break,
                                    attendanceTimeRangeType: app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance,
                                    employeeData: Object.assign({}, (0, lodash_1.cloneDeep)(c.employeeData), {
                                        break: {
                                            method: app_enums_1.enums.BreakMethod.ExtraWork,
                                            type: resultItem.break.type,
                                        },
                                    }),
                                    attendanceData: Object.assign((0, lodash_1.cloneDeep)(c.attendanceData), { untolerated: undefined }),
                                })));
                                resultItem.usedRanges.push(cint);
                                usedBreakDuration += availableDuration;
                            }
                        }
                        else {
                            segments.push(c);
                        }
                    }
                    this._resultSegments = segments;
                }
                resultItem.usedDuration = usedBreakDuration;
                resultItem.automaticallyUsedDuration = usedBreakDuration;
                result.push(resultItem);
            }
        }
        this._extraWorkBreakUsages = result;
    }
    invertExtraWorkPermissions() {
        for (const wpr of this._params.workPlanDayItems) {
            if (!wpr.permissionRequiredForExtraWorking)
                continue;
            for (let s of this._segmentsWithExtraWorkPermissions) {
                if (!s.range.intersect(wpr.dateIntersection))
                    continue;
                if ((s.employeeTimeRangeType === app_enums_1.enums.EmployeeTimeRangeType.Break ||
                    s.employeeTimeRangeType === app_enums_1.enums.EmployeeTimeRangeType.Holiday ||
                    s.employeeTimeRangeType === app_enums_1.enums.EmployeeTimeRangeType.OffTimePermission ||
                    s.employeeTimeRangeType === app_enums_1.enums.EmployeeTimeRangeType.OutOfWorkingHours) &&
                    (s.attendanceTimeRangeType === app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance ||
                        s.attendanceTimeRangeType === app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance) &&
                    !s.attendanceData.tolerance) {
                    s.attendanceTimeRangeType = app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance;
                }
            }
        }
        for (let p of this._params.extraWorkPermissions) {
            const segments = [];
            for (let s of this._segmentsWithExtraWorkPermissions) {
                if (!p.applicableDays.includes(s.range.start.isoWeekday())) {
                    segments.push(s);
                    continue;
                }
                const wpr = this._params.workPlanDayItems.find((wpdi) => wpdi.workPlanId === s.workPlanId);
                if (!wpr || !wpr.permissionRequiredForExtraWorking) {
                    segments.push(s);
                    continue;
                }
                if ((s.employeeTimeRangeType === app_enums_1.enums.EmployeeTimeRangeType.Break ||
                    s.employeeTimeRangeType === app_enums_1.enums.EmployeeTimeRangeType.Holiday ||
                    s.employeeTimeRangeType === app_enums_1.enums.EmployeeTimeRangeType.OffTimePermission ||
                    s.employeeTimeRangeType === app_enums_1.enums.EmployeeTimeRangeType.OutOfWorkingHours) &&
                    s.attendanceTimeRangeType === app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance &&
                    !s.attendanceData.tolerance) {
                    let pint = s.range.intersect(p.range);
                    if (!pint) {
                        s.attendanceTimeRangeType = app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance;
                        segments.push(s);
                        continue;
                    }
                    let x = Object.assign((0, lodash_1.cloneDeep)(s), {
                        range: undefined,
                        attendanceTimeRangeType: app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance,
                    });
                    let o = Object.assign((0, lodash_1.cloneDeep)(s), {
                        range: undefined,
                        attendanceTimeRangeType: app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance,
                        attendanceData: Object.assign({}, (0, lodash_1.cloneDeep)(s.attendanceData), {
                            extraWorkPermissionId: p.id,
                            extraWorkPermissionTypeId: p.typeId,
                        }),
                    });
                    segments.push(...(0, pacs_utils_1.splitDateRangeAccordingToIntersections)(s.range, pint, x, o));
                }
                else {
                    segments.push(s);
                }
            }
            this._segmentsWithExtraWorkPermissions = segments;
        }
    }
    invertFlexibleWorkingHours() {
        for (const remaining of this._remainingWorkDurations) {
            const workPlanDayItem = this._params.workPlanDayItems.find((wpdi) => wpdi.workPlanId === remaining.workPlanId);
            if (workPlanDayItem) {
                if (!workPlanDayItem.flexible) {
                    continue;
                }
                let segments = [];
                if (workPlanDayItem.workPlanType === app_enums_1.enums.WorkPlanType.HALF_FLEXIBLE) {
                    for (const s of this._segmentsWithFlexibleWorkingHours) {
                        if (s.workPlanId !== remaining.workPlanId || s.isBufferSegment) {
                            segments.push(s);
                            continue;
                        }
                        if (s.employeeTimeRangeType !== app_enums_1.enums.EmployeeTimeRangeType.WorkingHours) {
                            segments.push(s);
                            continue;
                        }
                        if (s.attendanceTimeRangeType !== app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance &&
                            s.attendanceTimeRangeType !== app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance) {
                            segments.push(s);
                            continue;
                        }
                        let d = s.range.duration("minutes");
                        if (remaining.used >= remaining.today) {
                            segments.push(s);
                        }
                        else if (d <= remaining.today - remaining.used) {
                            remaining.used += d;
                            segments.push(s);
                        }
                        else {
                            segments.push(s);
                            remaining.used = remaining.today;
                        }
                    }
                    this._segmentsWithFlexibleWorkingHours = segments;
                }
                segments = [];
                for (const s of this._segmentsWithFlexibleWorkingHours) {
                    if (s.workPlanId !== remaining.workPlanId || s.isBufferSegment) {
                        segments.push(s);
                        continue;
                    }
                    if (s.employeeTimeRangeType === app_enums_1.enums.EmployeeTimeRangeType.WorkingHours) {
                        segments.push(s);
                        continue;
                    }
                    if (s.attendanceTimeRangeType !== app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance &&
                        s.attendanceTimeRangeType !== app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance) {
                        segments.push(s);
                        continue;
                    }
                    let d = s.range.duration("minutes");
                    if (remaining.used >= remaining.today) {
                        segments.push(s);
                    }
                    else if (d <= remaining.today - remaining.used) {
                        remaining.used += d;
                        s.employeeTimeRangeType = app_enums_1.enums.EmployeeTimeRangeType.FlexibleWorkingHours;
                        segments.push(s);
                    }
                    else {
                        let r = new moment_range_1.DateRange(s.range.start.clone(), s.range.start.clone().add(remaining.today - remaining.used, "minutes"));
                        let dividedSegments = (0, pacs_utils_1.splitDateRangeAccordingToIntersections)(s.range, r, Object.assign((0, lodash_1.cloneDeep)(s), {
                            employeeTimeRangeType: app_enums_1.enums.EmployeeTimeRangeType.FlexibleWorkingHours,
                        }), Object.assign((0, lodash_1.cloneDeep)(s), {
                            employeeTimeRangeType: app_enums_1.enums.EmployeeTimeRangeType.FlexibleWorkingHours,
                        }));
                        segments.push(...dividedSegments);
                        remaining.used = remaining.today;
                    }
                }
                this._segmentsWithFlexibleWorkingHours = segments;
            }
        }
    }
    invertFlexibleWorkingHoursOvertimes() {
        for (const remaining of this._remainingWorkDurations) {
            let workPlanDayItem = this._params.workPlanDayItems.find((wpdi) => wpdi.workPlanId === remaining.workPlanId);
            if (workPlanDayItem) {
                if (!workPlanDayItem.flexible) {
                    continue;
                }
                let segments = [];
                if (workPlanDayItem.workPlanType === app_enums_1.enums.WorkPlanType.HALF_FLEXIBLE) {
                    for (const s of this._segmentsWithFlexibleWorkingHoursOvertimes) {
                        if (s.workPlanId !== remaining.workPlanId || s.isBufferSegment) {
                            segments.push(s);
                            continue;
                        }
                        if (s.employeeTimeRangeType !== app_enums_1.enums.EmployeeTimeRangeType.WorkingHours) {
                            segments.push(s);
                            continue;
                        }
                        if (s.attendanceTimeRangeType !== app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance &&
                            s.attendanceTimeRangeType !== app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance) {
                            segments.push(s);
                            continue;
                        }
                        let d = s.range.duration("minutes");
                        if (remaining.today === 0) {
                            segments.push(s);
                        }
                        else if (d <= remaining.today) {
                            remaining.today -= d;
                            remaining.weekly -= d;
                            segments.push(s);
                        }
                        else {
                            let r = new moment_range_1.DateRange(s.range.start.clone(), s.range.start.clone().add(remaining.today, "minutes"));
                            segments.push(...(0, pacs_utils_1.splitDateRangeAccordingToIntersections)(s.range, r, (0, lodash_1.cloneDeep)(s), (0, lodash_1.cloneDeep)(s)));
                            remaining.weekly = 0;
                            remaining.today = 0;
                        }
                    }
                    this._segmentsWithFlexibleWorkingHoursOvertimes = segments;
                }
                segments = [];
                for (const s of this._segmentsWithFlexibleWorkingHoursOvertimes) {
                    if (s.workPlanId !== remaining.workPlanId || s.isBufferSegment) {
                        segments.push(s);
                        continue;
                    }
                    if (s.employeeTimeRangeType !== app_enums_1.enums.EmployeeTimeRangeType.FlexibleWorkingHours) {
                        segments.push(s);
                        continue;
                    }
                    if (s.attendanceTimeRangeType !== app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance &&
                        s.attendanceTimeRangeType !== app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance) {
                        segments.push(s);
                        continue;
                    }
                    let d = s.range.duration("minutes");
                    if (remaining.today === 0) {
                        s.employeeTimeRangeType = app_enums_1.enums.EmployeeTimeRangeType.OutOfWorkingHours;
                        segments.push(s);
                    }
                    else if (d <= remaining.today) {
                        segments.push(s);
                        remaining.today -= d;
                        remaining.weekly -= d;
                    }
                    else {
                        let r = new moment_range_1.DateRange(s.range.start.clone(), s.range.start.clone().add(remaining.today, "minutes"));
                        let dividedSegments = (0, pacs_utils_1.splitDateRangeAccordingToIntersections)(s.range, r, Object.assign((0, lodash_1.cloneDeep)(s), {
                            employeeTimeRangeType: app_enums_1.enums.EmployeeTimeRangeType.OutOfWorkingHours,
                        }), (0, lodash_1.cloneDeep)(s));
                        segments.push(...dividedSegments);
                        remaining.weekly = 0;
                        remaining.today = 0;
                    }
                }
                this._segmentsWithFlexibleWorkingHoursOvertimes = segments;
            }
        }
    }
    invertFixedBreaks() {
        let result = [];
        for (const wp of this._params.workPlanDayItems) {
            if (!wp.fixedBreaks || wp.fixedBreaks.length < 1) {
                continue;
            }
            for (let _break of wp.fixedBreaks) {
                let resultItem = {
                    break: _break,
                    date: wp.date,
                    maxAvailableDuration: 0,
                    automaticallyUsedDuration: 0,
                    usedDuration: 0,
                    usedRanges: [],
                    intersectionRatio: 1,
                    workPlanId: wp.workPlanId,
                };
                let hStart = parseInt(_break.startTime.substr(0, 2));
                let mStart = parseInt(_break.startTime.substr(2, 2));
                let hEnd = parseInt(_break.endTime.substr(0, 2));
                let mEnd = parseInt(_break.endTime.substr(2, 2));
                let brRanges = [];
                let brDefinedDuration = 0;
                let brIntersectedDuration = 0;
                const startOfDay = wp.dateIntersection.start.clone().startOf("day");
                if (_break.startTime > _break.endTime) {
                    let brStart1 = startOfDay.clone().add(1, "days").startOf("day");
                    let brEnd1 = brStart1.clone().set("hours", hEnd).set("minutes", mEnd);
                    let brR1 = new moment_range_1.DateRange(brStart1, brEnd1);
                    let brR1Intersection = brR1.intersect(wp.dateIntersection);
                    if (brR1Intersection) {
                        brIntersectedDuration += brR1Intersection.duration("minutes");
                        brRanges.push(brR1Intersection);
                    }
                    brDefinedDuration += brR1.duration("minutes");
                    let brStart2 = startOfDay.clone().set("hours", hStart).set("minutes", mStart);
                    let brEnd2 = brStart2.clone().add(1, "days").startOf("day");
                    let brR2 = new moment_range_1.DateRange(brStart2, brEnd2);
                    let brR2Intersection = brR2.intersect(wp.dateIntersection);
                    if (brR2Intersection) {
                        brIntersectedDuration += brR2Intersection.duration("minutes");
                        brRanges.push(brR2Intersection);
                    }
                    brDefinedDuration += brR2.duration("minutes");
                }
                else {
                    let brStart = startOfDay.clone().set("hours", hStart).set("minutes", mStart);
                    let brEnd = brStart.clone().set("hours", hEnd).set("minutes", mEnd);
                    let br = new moment_range_1.DateRange(brStart, brEnd);
                    let brRIntersection = br.intersect(wp.dateIntersection)?.intersect(this._params.offsetDayRange);
                    if (brRIntersection) {
                        brIntersectedDuration += brRIntersection.duration("minutes");
                        brRanges.push(brRIntersection);
                    }
                    brDefinedDuration = br.duration("minutes");
                }
                if (brRanges.length === 0)
                    continue;
                let usedBreakDuration = 0;
                if (brDefinedDuration > 0) {
                    resultItem.intersectionRatio = brIntersectedDuration / brDefinedDuration;
                }
                let maxDuration = _break.allowedMaxBreakDuration > 0 ? _break.allowedMaxBreakDuration * resultItem.intersectionRatio : brIntersectedDuration;
                resultItem.maxAvailableDuration = maxDuration;
                for (const br of brRanges) {
                    let segments = [];
                    for (const c of this._segmentsWithBreaks) {
                        if (c.workPlanId !== wp.workPlanId || usedBreakDuration >= maxDuration || c.isBufferSegment) {
                            segments.push(c);
                            continue;
                        }
                        if ((c.employeeTimeRangeType === app_enums_1.enums.EmployeeTimeRangeType.WorkingHours || c.employeeTimeRangeType === app_enums_1.enums.EmployeeTimeRangeType.FlexibleWorkingHours) &&
                            c.attendanceTimeRangeType === app_enums_1.enums.AttendanceTimeRangeType.Absent) {
                            let cint = c.range.intersect(br);
                            if (cint) {
                                let cintDuration = cint.duration("minutes", true);
                                let availableDuration = maxDuration - usedBreakDuration;
                                if (cintDuration <= availableDuration) {
                                    usedBreakDuration += cintDuration;
                                }
                                else {
                                    cint = new moment_range_1.DateRange(cint.start, cint.start.clone().add(availableDuration, "minutes"));
                                    usedBreakDuration += availableDuration;
                                }
                                resultItem.usedRanges.push(cint);
                                segments.push(...(0, pacs_utils_1.splitDateRangeAccordingToIntersections)(c.range, cint, (0, lodash_1.cloneDeep)(c), Object.assign((0, lodash_1.cloneDeep)(c), {
                                    employeeTimeRangeType: app_enums_1.enums.EmployeeTimeRangeType.Break,
                                    employeeData: Object.assign({}, (0, lodash_1.cloneDeep)(c.employeeData), {
                                        break: {
                                            method: app_enums_1.enums.BreakMethod.Fixed,
                                            type: resultItem.break.type,
                                            breakUsedDurationType: app_enums_1.enums.BreakUsedDurationType.Manually,
                                        },
                                    }),
                                })));
                            }
                            else {
                                segments.push(c);
                            }
                        }
                        else {
                            segments.push(c);
                        }
                    }
                    if (_break.executeAutomaticallyBelowDuration * resultItem.intersectionRatio &&
                        usedBreakDuration < _break.executeAutomaticallyBelowDuration * resultItem.intersectionRatio) {
                        resultItem.automaticallyUsedDuration = _break.executeAutomaticallyBelowDuration * resultItem.intersectionRatio - usedBreakDuration;
                    }
                    this._segmentsWithBreaks = segments;
                    if (resultItem.automaticallyUsedDuration > 0) {
                        let autoUsedDurationToMark = resultItem.automaticallyUsedDuration;
                        segments = [];
                        for (const c of this._segmentsWithBreaks) {
                            if (c.workPlanId !== wp.workPlanId || autoUsedDurationToMark === 0 || c.isBufferSegment) {
                                segments.push(c);
                                continue;
                            }
                            if ((c.employeeTimeRangeType === app_enums_1.enums.EmployeeTimeRangeType.WorkingHours || c.employeeTimeRangeType === app_enums_1.enums.EmployeeTimeRangeType.FlexibleWorkingHours) &&
                                (c.attendanceTimeRangeType === app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance ||
                                    c.attendanceTimeRangeType === app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance)) {
                                let cint = c.range.intersect(br);
                                if (cint) {
                                    let cintDuration = cint.duration("minutes", true);
                                    if (cintDuration <= autoUsedDurationToMark) {
                                        autoUsedDurationToMark -= cintDuration;
                                    }
                                    else {
                                        cint = new moment_range_1.DateRange(cint.start, cint.start.clone().add(autoUsedDurationToMark, "minutes"));
                                        autoUsedDurationToMark = 0;
                                    }
                                    resultItem.usedRanges.push(cint);
                                    segments.push(...(0, pacs_utils_1.splitDateRangeAccordingToIntersections)(c.range, cint, (0, lodash_1.cloneDeep)(c), Object.assign((0, lodash_1.cloneDeep)(c), {
                                        employeeTimeRangeType: app_enums_1.enums.EmployeeTimeRangeType.Break,
                                        attendanceTimeRangeType: app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance,
                                        employeeData: Object.assign({}, (0, lodash_1.cloneDeep)(c.employeeData), {
                                            break: {
                                                method: app_enums_1.enums.BreakMethod.Fixed,
                                                type: resultItem.break.type,
                                                breakUsedDurationType: app_enums_1.enums.BreakUsedDurationType.Automatically,
                                            },
                                        }),
                                    })));
                                }
                                else {
                                    segments.push(c);
                                }
                            }
                            else if (c.employeeTimeRangeType === app_enums_1.enums.EmployeeTimeRangeType.OffTimePermission) {
                                let newAttendanceTimeRangeType = c.attendanceTimeRangeType;
                                if (c.attendanceTimeRangeType === app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance ||
                                    c.attendanceTimeRangeType === app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance) {
                                    newAttendanceTimeRangeType = app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance;
                                }
                                let cint = c.range.intersect(br);
                                if (cint) {
                                    let cintDuration = cint.duration("minutes", true);
                                    if (cintDuration <= autoUsedDurationToMark) {
                                        autoUsedDurationToMark -= cintDuration;
                                    }
                                    else {
                                        cint = new moment_range_1.DateRange(cint.start, cint.start.clone().add(autoUsedDurationToMark, "minutes"));
                                        autoUsedDurationToMark = 0;
                                    }
                                    resultItem.usedRanges.push(cint);
                                    segments.push(...(0, pacs_utils_1.splitDateRangeAccordingToIntersections)(c.range, cint, (0, lodash_1.cloneDeep)(c), Object.assign((0, lodash_1.cloneDeep)(c), {
                                        employeeTimeRangeType: app_enums_1.enums.EmployeeTimeRangeType.Break,
                                        attendanceTimeRangeType: newAttendanceTimeRangeType,
                                        employeeData: Object.assign({}, (0, lodash_1.cloneDeep)(c.employeeData), {
                                            offTimePermissionId: undefined,
                                            offTimePermissionTypeId: undefined,
                                            break: {
                                                method: app_enums_1.enums.BreakMethod.Fixed,
                                                type: resultItem.break.type,
                                                breakUsedDurationType: app_enums_1.enums.BreakUsedDurationType.Automatically,
                                            },
                                        }),
                                    })));
                                }
                                else {
                                    segments.push(c);
                                }
                            }
                            else {
                                segments.push(c);
                            }
                        }
                    }
                    this._segmentsWithBreaks = segments;
                }
                resultItem.usedDuration = usedBreakDuration;
                result.push(resultItem);
            }
        }
        this._fixedBreakUsages = result;
    }
    invertClaimBreak() {
        let result = [];
        for (const wp of this._params.workPlanDayItems) {
            let _break = wp.claimBreak;
            if (!_break) {
                continue;
            }
            let resultItem = {
                break: _break,
                automaticallyUsedDuration: 0,
                usedDuration: 0,
                usedRanges: [],
                claimedBreakDuration: 0,
                workPlanId: wp.workPlanId,
                date: wp.date,
            };
            let attendanceDuration = 0;
            let hasNoBufferedSegment = false;
            for (const c of this._segmentsWithBreaks) {
                if (c.workPlanId !== wp.workPlanId || !wp.dateIntersection.intersect(c.range) || c.isBufferSegment) {
                    continue;
                }
                if (c.attendanceTimeRangeType === app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance ||
                    c.attendanceTimeRangeType === app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance) {
                    attendanceDuration += c.range.duration("minutes");
                }
                hasNoBufferedSegment = true;
            }
            if (!hasNoBufferedSegment) {
                continue;
            }
            let claimCoefficient = attendanceDuration / _break.workDuration;
            resultItem.claimedBreakDuration = claimCoefficient * _break.breakDuration;
            if (resultItem.claimedBreakDuration > 0) {
                let segments = [];
                for (const c of this._segmentsWithBreaks) {
                    if (c.workPlanId !== wp.workPlanId || !wp.dateIntersection.intersect(c.range) || c.isBufferSegment || resultItem.usedDuration >= resultItem.claimedBreakDuration) {
                        segments.push(c);
                        continue;
                    }
                    if ((c.employeeTimeRangeType === app_enums_1.enums.EmployeeTimeRangeType.WorkingHours || c.employeeTimeRangeType === app_enums_1.enums.EmployeeTimeRangeType.FlexibleWorkingHours) &&
                        c.attendanceTimeRangeType === app_enums_1.enums.AttendanceTimeRangeType.Absent) {
                        let cDuration = c.range.duration("minutes", true);
                        let availableDuration = resultItem.claimedBreakDuration - resultItem.usedDuration;
                        let cint = c.range;
                        if (cDuration <= availableDuration) {
                            resultItem.usedDuration += cDuration;
                        }
                        else {
                            cint = new moment_range_1.DateRange(c.range.start.clone(), c.range.start.clone().add(availableDuration, "minutes"));
                            resultItem.usedDuration += availableDuration;
                        }
                        resultItem.usedRanges.push(cint);
                        segments.push(...(0, pacs_utils_1.splitDateRangeAccordingToIntersections)(c.range, cint, (0, lodash_1.cloneDeep)(c), Object.assign((0, lodash_1.cloneDeep)(c), {
                            employeeTimeRangeType: app_enums_1.enums.EmployeeTimeRangeType.Break,
                            employeeData: Object.assign({}, (0, lodash_1.cloneDeep)(c.employeeData), {
                                break: {
                                    method: app_enums_1.enums.BreakMethod.Claim,
                                    type: resultItem.break.type,
                                    breakUsedDurationType: app_enums_1.enums.BreakUsedDurationType.Manually,
                                },
                            }),
                        })));
                    }
                    else {
                        segments.push(c);
                    }
                }
                this._segmentsWithBreaks = segments;
            }
            if (_break.executeAutomaticallyBelowDuration * claimCoefficient && resultItem.usedDuration < _break.executeAutomaticallyBelowDuration * claimCoefficient) {
                resultItem.automaticallyUsedDuration = _break.executeAutomaticallyBelowDuration * claimCoefficient - resultItem.usedDuration;
            }
            if (resultItem.automaticallyUsedDuration > 0) {
                let autoUsedDurationToMark = resultItem.automaticallyUsedDuration;
                let segments = [];
                for (let index = this._segmentsWithBreaks.length - 1; index >= 0; index--) {
                    const c = this._segmentsWithBreaks[index];
                    if (c.workPlanId !== wp.workPlanId || !wp.dateIntersection.intersect(c.range) || c.isBufferSegment || autoUsedDurationToMark === 0) {
                        segments = [c, ...segments];
                        continue;
                    }
                    if ((c.employeeTimeRangeType === app_enums_1.enums.EmployeeTimeRangeType.WorkingHours || c.employeeTimeRangeType === app_enums_1.enums.EmployeeTimeRangeType.FlexibleWorkingHours) &&
                        (c.attendanceTimeRangeType === app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance ||
                            c.attendanceTimeRangeType === app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance)) {
                        let cint = c.range.clone();
                        if (cint) {
                            let cintDuration = cint.duration("minutes", true);
                            if (cintDuration <= autoUsedDurationToMark) {
                                autoUsedDurationToMark -= cintDuration;
                            }
                            else {
                                cint = new moment_range_1.DateRange(cint.end.clone().subtract(autoUsedDurationToMark, "minutes"), cint.end);
                                autoUsedDurationToMark = 0;
                            }
                            resultItem.usedRanges.push(cint);
                            segments = [
                                ...(0, pacs_utils_1.splitDateRangeAccordingToIntersections)(c.range, cint, (0, lodash_1.cloneDeep)(c), Object.assign((0, lodash_1.cloneDeep)(c), {
                                    employeeTimeRangeType: app_enums_1.enums.EmployeeTimeRangeType.Break,
                                    attendanceTimeRangeType: app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance,
                                    employeeData: Object.assign({}, (0, lodash_1.cloneDeep)(c.employeeData), {
                                        break: {
                                            method: app_enums_1.enums.BreakMethod.Claim,
                                            type: resultItem.break.type,
                                            breakUsedDurationType: app_enums_1.enums.BreakUsedDurationType.Manually,
                                        },
                                    }),
                                })),
                                ...segments,
                            ];
                        }
                        else {
                            segments = [c, ...segments];
                        }
                    }
                    else {
                        segments = [c, ...segments];
                    }
                }
                this._segmentsWithBreaks = segments;
            }
            result.push(resultItem);
        }
        this._claimBreakUsages = result;
    }
    invertDeclaredWorksPermissions(permissions) {
        for (let p of permissions) {
            let segments = [];
            for (let s of this._segmentsWithDeclaredWorkPermissions) {
                if (!p.applicableDays.includes(s.range.start.isoWeekday())) {
                    segments.push(s);
                    continue;
                }
                if (!s.range.intersect(p.range)) {
                    segments.push(s);
                    continue;
                }
                if (s.employeeTimeRangeType === app_enums_1.enums.EmployeeTimeRangeType.OffTimePermission || s.attendanceTimeRangeType === app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance) {
                    segments.push(s);
                    continue;
                }
                let foundSegment = this._inheritedWorkPlanDurationSegments.filter((is) => is.range.intersect(s.range) && is.workPlanId === s.workPlanId && is.isWorkingHour);
                let isWorkingHour = foundSegment && foundSegment.length > 0;
                const wp = this._params.workPlanDayItems.find((w) => w.workPlanId === s.workPlanId);
                if ((wp && wp.workPlanType === app_enums_1.enums.WorkPlanType.FULL_FLEXIBLE) || !p.applyOnlyOnWorkingHours || isWorkingHour) {
                    let x = (0, lodash_1.cloneDeep)(s);
                    let o = Object.assign((0, lodash_1.cloneDeep)(s), {
                        range: undefined,
                        attendanceTimeRangeType: app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance,
                        attendanceData: Object.assign({}, (0, lodash_1.cloneDeep)(s.attendanceData), {
                            declaredAttendance: {
                                id: p.id,
                                typeId: p.typeId,
                            },
                        }),
                    });
                    segments.push(...(0, pacs_utils_1.splitDateRangeAccordingToIntersections)(s.range, p.range, x, o));
                }
                else {
                    segments.push(s);
                    continue;
                }
            }
            this._segmentsWithDeclaredWorkPermissions = segments;
        }
    }
    invertAccessTimes() {
        for (let wpAccessRanges of this._physicalAccessRanges) {
            for (const a of wpAccessRanges.accessRanges) {
                let segments = [];
                const workPlanDayItem = this._params.workPlanDayItems.find((element) => element.workPlanId == wpAccessRanges.workPlanId);
                for (let s of this._segmentsWithAccessTimes) {
                    if (wpAccessRanges.workPlanId !== s.workPlanId) {
                        segments.push(s);
                        continue;
                    }
                    if (!s.range.intersect(a.range)) {
                        segments.push(s);
                        continue;
                    }
                    if (!workPlanDayItem.ignoreUnpairedAccesses || this._isLive || (a.endId != null && a.startId != null)) {
                        let x = (0, lodash_1.cloneDeep)(s);
                        let o = Object.assign((0, lodash_1.cloneDeep)(s), {
                            range: null,
                            attendanceTimeRangeType: workPlanDayItem &&
                                workPlanDayItem.attendanceControlMethod === app_enums_1.enums.WorkPlanAccessCheckType.FirstAndLastAccess &&
                                s.employeeTimeRangeType === app_enums_1.enums.EmployeeTimeRangeType.OffTimePermission
                                ? app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance
                                : app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance,
                            attendanceData: (0, lodash_1.cloneDeep)(s.attendanceData),
                        });
                        segments.push(...(0, pacs_utils_1.splitDateRangeAccordingToIntersections)(s.range, a.range, x, o));
                    }
                    else {
                        let x = (0, lodash_1.cloneDeep)(s);
                        let o = Object.assign((0, lodash_1.cloneDeep)(s), {
                            range: null,
                            attendanceData: Object.assign((0, lodash_1.cloneDeep)(s.attendanceData), {
                                unpairedAccess: true,
                            }),
                        });
                        segments.push(...(0, pacs_utils_1.splitDateRangeAccordingToIntersections)(s.range, a.range, x, o));
                    }
                }
                this._segmentsWithAccessTimes = segments;
            }
        }
    }
    invertShouldBeIgnoredWorksPermissions(permissions) {
        for (let p of permissions) {
            let segments = [];
            for (let s of this._segmentsWithShouldBeIgnoredWorkPermissions) {
                if (!p.applicableDays.includes(s.range.start.isoWeekday())) {
                    segments.push(s);
                    continue;
                }
                if (!s.range.intersect(p.range)) {
                    segments.push(s);
                    continue;
                }
                if (s.attendanceTimeRangeType === app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance || s.attendanceTimeRangeType === app_enums_1.enums.AttendanceTimeRangeType.DeclaredAttendance) {
                    segments.push(s);
                    continue;
                }
                if (s.attendanceTimeRangeType === app_enums_1.enums.AttendanceTimeRangeType.Absent) {
                    segments.push(s);
                    continue;
                }
                let x = (0, lodash_1.cloneDeep)(s);
                let o = Object.assign((0, lodash_1.cloneDeep)(s), {
                    range: undefined,
                    attendanceTimeRangeType: app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance,
                    attendanceData: Object.assign({}, (0, lodash_1.cloneDeep)(s.attendanceData), {
                        ignoredPhysicalAttendance: {
                            shouldBeIgnoredPermissionId: p.id,
                            shouldBeIgnoredPermissionTypeId: p.typeId,
                            range: p.range,
                        },
                    }),
                });
                segments.push(...(0, pacs_utils_1.splitDateRangeAccordingToIntersections)(s.range, p.range, x, o));
            }
            this._segmentsWithShouldBeIgnoredWorkPermissions = segments;
        }
    }
    invertOffTimePermissions(permissions) {
        let hasWorkHours = false;
        for (const s of this._segmentsWithOffTimePermissions) {
            if (s.employeeTimeRangeType === app_enums_1.enums.EmployeeTimeRangeType.WorkingHours) {
                hasWorkHours = true;
                break;
            }
        }
        for (let p of permissions) {
            let segments = [];
            for (let s of this._segmentsWithOffTimePermissions) {
                if (!p.applicableDays.includes(s.range.start.isoWeekday())) {
                    segments.push(s);
                    continue;
                }
                let pint = s.range.intersect(p.range);
                if (!pint) {
                    segments.push(s);
                    continue;
                }
                if (s.employeeTimeRangeType === app_enums_1.enums.EmployeeTimeRangeType.OffTimePermission) {
                    segments.push(s);
                    continue;
                }
                if ((s.employeeTimeRangeType === app_enums_1.enums.EmployeeTimeRangeType.Holiday && !p.applyOnHolidays) ||
                    (s.employeeTimeRangeType !== app_enums_1.enums.EmployeeTimeRangeType.Holiday && hasWorkHours && s.employeeTimeRangeType !== app_enums_1.enums.EmployeeTimeRangeType.WorkingHours)) {
                    segments.push(s);
                    continue;
                }
                let x = (0, lodash_1.cloneDeep)(s);
                let o = Object.assign((0, lodash_1.cloneDeep)(s), {
                    range: undefined,
                    employeeTimeRangeType: app_enums_1.enums.EmployeeTimeRangeType.OffTimePermission,
                    employeeData: Object.assign({}, (0, lodash_1.cloneDeep)(s.employeeData), {
                        offTimePermissionId: p.id,
                        offTimePermissionTypeId: p.typeId,
                    }),
                });
                segments.push(...(0, pacs_utils_1.splitDateRangeAccordingToIntersections)(s.range, pint, x, o));
            }
            this._segmentsWithOffTimePermissions = segments;
        }
    }
    calculateTolerances() {
        for (let i = 0; i < this._segmentsWithOffTimePermissions.length; i++) {
            const e = this._segmentsWithOffTimePermissions[i];
            if (e.employeeTimeRangeType !== app_enums_1.enums.EmployeeTimeRangeType.WorkingHours) {
                continue;
            }
            if (!e.workPlanId) {
                continue;
            }
            const workPlan = this._params.workPlanDayItems.find((w) => w.workPlanId === e.workPlanId && e.range.intersect(w.dateIntersection));
            for (let j = 0; j < this._segmentsWithShouldBeIgnoredWorkPermissions.length; j++) {
                let c = this._segmentsWithShouldBeIgnoredWorkPermissions[j];
                if (!e.range.overlaps(c.range, { adjacent: true })) {
                    continue;
                }
                if (e.workPlanId !== c.workPlanId) {
                    continue;
                }
                if (c.attendanceTimeRangeType === app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance) {
                    if (c.range.start.isSameOrBefore(e.range.start)) {
                        if (!workPlan.flexible) {
                            let diff = Math.round(e.range.start.diff(c.range.start, "seconds", true));
                            if (workPlan.tolerances && workPlan.tolerances.earlyStart && workPlan.tolerances.earlyStart * 60 > diff) {
                                c.attendanceData.tolerance = diff > 0 ? app_enums_1.enums.ToleranceType.EarlyStart : undefined;
                            }
                            else {
                                c.attendanceData.untolerated = diff > 0 ? app_enums_1.enums.ToleranceType.EarlyStart : undefined;
                            }
                        }
                    }
                    else {
                        let diff = Math.round(c.range.start.diff(e.range.start, "seconds", true));
                        let previous = this._segmentsWithShouldBeIgnoredWorkPermissions[j - 1];
                        if (workPlan.tolerances && workPlan.tolerances.lateStart && workPlan.tolerances.lateStart * 60 > diff) {
                            if (j > 0 && diff > 0) {
                                if (previous.attendanceTimeRangeType == app_enums_1.enums.AttendanceTimeRangeType.Absent ||
                                    previous.attendanceTimeRangeType == app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance) {
                                    previous.attendanceData.tolerance = app_enums_1.enums.ToleranceType.LateStart;
                                }
                            }
                        }
                        else {
                            previous.attendanceData.untolerated = app_enums_1.enums.ToleranceType.LateStart;
                        }
                    }
                    break;
                }
            }
            for (let j = this._segmentsWithShouldBeIgnoredWorkPermissions.length - 1; j >= 0; j--) {
                let c = this._segmentsWithShouldBeIgnoredWorkPermissions[j];
                if (!e.range.overlaps(c.range, { adjacent: true })) {
                    continue;
                }
                if (e.workPlanId !== c.workPlanId) {
                    continue;
                }
                if (c.attendanceTimeRangeType === app_enums_1.enums.AttendanceTimeRangeType.AccountedPhysicalAttendance) {
                    if (c.range.end.isSameOrAfter(e.range.end)) {
                        if (!workPlan.flexible) {
                            let diff = Math.round(c.range.end.diff(e.range.end, "seconds", true));
                            if (workPlan.tolerances && workPlan.tolerances.lateLeave && workPlan.tolerances.lateLeave * 60 > diff) {
                                c.attendanceData.tolerance = diff > 0 ? app_enums_1.enums.ToleranceType.LateLeave : undefined;
                            }
                            else {
                                c.attendanceData.untolerated = diff > 0 ? app_enums_1.enums.ToleranceType.LateLeave : undefined;
                            }
                        }
                    }
                    else {
                        let diff = Math.round(e.range.end.diff(c.range.end, "seconds", true));
                        let next = this._segmentsWithShouldBeIgnoredWorkPermissions[j + 1];
                        if (workPlan.tolerances && workPlan.tolerances.earlyLeave && workPlan.tolerances.earlyLeave * 60 > diff) {
                            if (j < this._segmentsWithShouldBeIgnoredWorkPermissions.length - 1 && diff > 0) {
                                if (next.attendanceTimeRangeType == app_enums_1.enums.AttendanceTimeRangeType.Absent ||
                                    next.attendanceTimeRangeType == app_enums_1.enums.AttendanceTimeRangeType.IgnoredPhysicalAttendance) {
                                    next.attendanceData.tolerance = app_enums_1.enums.ToleranceType.EarlyLeave;
                                }
                            }
                        }
                        else {
                            next.attendanceData.untolerated = app_enums_1.enums.ToleranceType.EarlyLeave;
                        }
                    }
                    break;
                }
            }
        }
    }
    segmentizeAccessTimes(params) {
        if (this._isLive) {
            for (const regionAccesses of params) {
                let acc = regionAccesses.accessTimes;
                if (acc.length === 2 && acc[0].id === acc[1].id) {
                    regionAccesses.accessTimes = [acc[0]];
                }
            }
        }
        this._physicalAccessRanges = [];
        for (const regionAccesses of params) {
            let ranges = [];
            let acs = null;
            let ace = null;
            for (let i = 0; i < regionAccesses.accessTimes.length; i++) {
                let a = regionAccesses.accessTimes[i];
                if (a.checkIn) {
                    if (ace) {
                        ranges.push({
                            range: new moment_range_1.DateRange(acs.time, ace.time),
                            startId: acs.id,
                            endId: ace.id,
                        });
                        acs = a;
                        ace = null;
                    }
                    else if (acs) {
                        continue;
                    }
                    else {
                        acs = a;
                    }
                }
                else {
                    ace = a;
                }
            }
            if (acs && ace) {
                ranges.push({
                    range: new moment_range_1.DateRange(acs.time, ace.time),
                    startId: acs.id,
                    endId: ace.id,
                });
            }
            else if (acs) {
                ranges.push({
                    range: new moment_range_1.DateRange(acs.time, this._params.offsetDayRange.end),
                    startId: acs.id,
                    endId: null,
                });
            }
            else if (ace) {
                ranges.push({
                    range: new moment_range_1.DateRange(this._params.offsetDayRange.start, ace.time),
                    startId: null,
                    endId: ace.id,
                });
            }
            let newRanges = [];
            for (const r of ranges) {
                if (r.range.contains(this._params.offsetDayRange.start, { excludeStart: true, excludeEnd: true })) {
                    newRanges.push({
                        range: new moment_range_1.DateRange(r.range.start, this._params.offsetDayRange.start),
                        startId: r.startId,
                        endId: null,
                        regionId: r.regionId,
                    });
                    newRanges.push({
                        range: new moment_range_1.DateRange(this._params.offsetDayRange.start, r.range.end),
                        startId: null,
                        endId: r.endId,
                        regionId: r.regionId,
                    });
                }
                else {
                    newRanges.push(r);
                }
            }
            ranges = newRanges;
            newRanges = [];
            for (const r of ranges) {
                if (r.range.contains(this._params.offsetDayRange.end, { excludeStart: true, excludeEnd: true })) {
                    newRanges.push({
                        range: new moment_range_1.DateRange(r.range.start, this._params.offsetDayRange.end),
                        startId: r.startId,
                        endId: null,
                        regionId: r.regionId,
                    });
                    newRanges.push({
                        range: new moment_range_1.DateRange(this._params.offsetDayRange.end, r.range.end),
                        startId: null,
                        endId: r.endId,
                        regionId: r.regionId,
                    });
                }
                else {
                    newRanges.push(r);
                }
            }
            ranges = newRanges;
            this._physicalAccessRanges.push({
                workPlanId: regionAccesses.workPlanId,
                accessRanges: ranges,
            });
        }
    }
    getSummary() {
        if (!this._summary) {
            this.summarize();
        }
        return this._summary;
    }
    getSerialized() {
        if (!this._serialized) {
            this.serialize();
        }
        return this._serialized;
    }
    setLive(isLive) {
        if (this._isLive != isLive) {
            this._summary = null;
            this._serialized = null;
        }
        if (!isLive) {
            this._serialized = null;
            this._summary = null;
            this._resultSegments = null;
            this._segmentsWithExtraWorkPermissions = null;
            this._segmentsWithFlexibleWorkingHoursOvertimes = null;
            this._segmentsWithBreaks = null;
            this._segmentsWithFlexibleWorkingHours = null;
            this._segmentsWithShouldBeIgnoredWorkPermissions = null;
            this._segmentsWithAccessTimes = null;
            this._segmentsWithDeclaredWorkPermissions = null;
            this._segmentsWithOffTimePermissions = null;
            this._segmentsWithWorkPlan = null;
        }
        this._isLive = isLive;
    }
    setProcessTime(time) {
        this._processedAt = time;
    }
    setSummarizedTime(time) {
        this._summarizedAt = time;
    }
    getDetailed() {
        return {
            calculatedDurationSegments: this._resultSegments,
            calculatedTolerances: this._calculatedTolerances,
            claimBreakUsages: this._claimBreakUsages,
            employeeDurationSegments: this._segmentsWithOffTimePermissions,
            fixedBreakUsages: this._fixedBreakUsages,
            holidayUsages: this._holidayUsages,
            permissionUsages: this._permissionUsages,
            summary: this._summary,
            workPlanDurationSegments: this._inheritedWorkPlanDurationSegments,
        };
    }
    static async createEmployeeDay(organizationId, userId, date, previousDays) {
        const params = await dal_manager_1.dbManager.accessPacs2.getEmployeeDayParams(organizationId, userId, date, previousDays);
        const holidays = [];
        const workPlanDayItems = [];
        for (const w of params.flexibleWorkPlanDayParams) {
            workPlanDayItems.push(new workplan_day_1.FlexibleWorkPlanDay(w));
            for (const holiday of w.holidays || []) {
                if (!holidays.find((h) => h.id === holiday.id)) {
                    holidays.push(holiday);
                }
            }
        }
        for (const w of params.fixedWorkPlanDayParams) {
            workPlanDayItems.push(new workplan_day_1.WorkPlanDay(w));
            for (const holiday of w.holidays || []) {
                if (!holidays.find((h) => h.id === holiday.id)) {
                    holidays.push(holiday);
                }
            }
        }
        let result = null;
        if (workPlanDayItems.length > 0) {
            result = new EmployeeDay(Object.assign({
                workPlanDayItems: workPlanDayItems,
                holidays: holidays || [],
            }, params.employeeDayParams));
        }
        return result;
    }
}
exports.EmployeeDay = EmployeeDay;
