"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.BusinessWorkPlanImport = exports.transformPPermissionHistoryItemsToLocalizedMessages = exports.getHistoryOfPPermissionOfUser = exports.decoratePPermissionListWithCaptions = exports.upsertPredefinedPPermissions = exports.disablePACSModule = exports.enablePACSModule = void 0;
const i18n_1 = __importDefault(require("i18n"));
const i18n_2 = __importDefault(require("i18n"));
const app_enums_1 = require("../../app.enums");
const app_util_1 = require("../../app.util");
const dal_constants_1 = require("../../dal/dal.constants");
const dal_manager_1 = require("../../dal/dal.manager");
const dal_utils_1 = require("../../dal/dal.utils");
const business_hooks_1 = require("../business.hooks");
const business_pacs_excel_1 = require("./business.pacs.excel");
const app_httpserver_1 = require("../../app.httpserver");
async function enablePACSModule(organizationId, trx) {
    await upsertPredefinedPPermissions(organizationId, trx);
    await dal_manager_1.dbManager.accessPacs.upsertScheduledJobsAndNotifications(organizationId, trx);
    let hook = business_hooks_1.armonHookManager.getModuleStatusChangeHook(organizationId);
    if (hook) {
        await hook(organizationId, app_enums_1.enums.ArmonApplicationModule.PACS, true);
    }
}
exports.enablePACSModule = enablePACSModule;
async function disablePACSModule(organizationId, trx) {
    let settings = await dal_manager_1.dbManager.accessPacs.getOrganizationPACSModuleSettings(organizationId, trx);
    if (settings) {
        if (settings.annualLeavePPermissionTypeId)
            await dal_manager_1.dbManager.accessPacs.deletePPermissionType(organizationId, settings.annualLeavePPermissionTypeId, trx);
        if (settings.casualLeavePPermissionTypeId)
            await dal_manager_1.dbManager.accessPacs.deletePPermissionType(organizationId, settings.casualLeavePPermissionTypeId, trx);
    }
    else {
        await dal_manager_1.dbManager.accessPacs.deletePPermissionTypeByType(organizationId, [app_enums_1.enums.PredefinedPPermissionType.Annual, app_enums_1.enums.PredefinedPPermissionType.Casual], trx);
    }
    await dal_manager_1.dbManager.accessPacs.upsertOrganizationPACSModuleSettings(organizationId, {
        organizationId: organizationId,
        annualLeavePPermissionTypeId: null,
        casualLeavePPermissionTypeId: null,
        middayHour: "12:00",
    }, trx);
    await dal_manager_1.dbManager.accessPacs.removeScheduledJobsAndNotifications(organizationId, trx);
    let hook = business_hooks_1.armonHookManager.getModuleStatusChangeHook(organizationId);
    if (hook) {
        await hook(organizationId, app_enums_1.enums.ArmonApplicationModule.PACS, true);
    }
}
exports.disablePACSModule = disablePACSModule;
async function upsertPredefinedPPermissions(organizationId, trx) {
    let ppermissionTypes = await dal_manager_1.dbManager.accessPacs.listPPermissionTypes(organizationId, {
        take: 1000,
        skip: 0,
    }, trx);
    let ppermissionTypeOne = {
        name: "Mazeret İzni",
        isPassive: false,
        description: "Mazeret İzni",
        hrCanInit: true,
        personnelCanInit: true,
        unitManagerCanInit: true,
        isDailyScheduled: false,
        minValue: 1,
        maxValue: 40,
        maxValueBoundaryInterval: 1,
        method: dal_constants_1.DalConstants.PPermissionType.OffTime,
        approvementList: [null],
        sendNotification: true,
        needsManagerDeputy: false,
        needsUserDeputy: false,
        userMustSign: false,
        type: app_enums_1.enums.PredefinedPPermissionType.Casual,
        showRemainingCount: true,
        isUnpaidLeave: false,
        notifyHR: false,
        noteIsMandatory: false,
        dynamicForm: null,
        applyOnHolidays: false,
        applyOnlyOnWorkingHours: false,
        notifyManagers: false,
        isNegativeBalanceAllowed: true,
    };
    let ppermissionTypeTwo = {
        name: "Yıllık İzin",
        isPassive: false,
        description: "Yıllık İzin",
        hrCanInit: true,
        personnelCanInit: true,
        unitManagerCanInit: true,
        isDailyScheduled: true,
        minValue: 1,
        maxValue: 14,
        maxValueBoundaryInterval: 1,
        method: dal_constants_1.DalConstants.PPermissionType.OffTime,
        approvementList: [null],
        sendNotification: true,
        needsManagerDeputy: true,
        needsUserDeputy: false,
        userMustSign: false,
        type: app_enums_1.enums.PredefinedPPermissionType.Annual,
        showRemainingCount: true,
        isUnpaidLeave: false,
        notifyHR: false,
        noteIsMandatory: false,
        dynamicForm: null,
        applyOnHolidays: false,
        applyOnlyOnWorkingHours: false,
        notifyManagers: false,
        isNegativeBalanceAllowed: true,
    };
    ppermissionTypeOne.id = null;
    await dal_manager_1.dbManager.systemTransaction(async (trx) => {
        if (ppermissionTypes.items.some((p) => p.type == app_enums_1.enums.PredefinedPPermissionType.Casual)) {
            ppermissionTypeOne.id = ppermissionTypes.items.find((p) => p.type == app_enums_1.enums.PredefinedPPermissionType.Casual).id;
        }
        else {
            ppermissionTypeOne.id = await dal_manager_1.dbManager.accessPacs.upsertPPermissionTypePg({ organizationId, args: ppermissionTypeOne, trx });
        }
        if (ppermissionTypes.items.some((p) => p.type == app_enums_1.enums.PredefinedPPermissionType.Annual)) {
            ppermissionTypeTwo.id = ppermissionTypes.items.find((p) => p.type == app_enums_1.enums.PredefinedPPermissionType.Annual).id;
        }
        else {
            ppermissionTypeTwo.id = await dal_manager_1.dbManager.accessPacs.upsertPPermissionTypePg({ organizationId, args: ppermissionTypeTwo, trx });
        }
    });
    let annualPPermissionAgeLimits = {
        youngerThanLimitYears: 18,
        youngerMinimumAmount: 480,
        olderThanLimitYears: 50,
        olderMinimumAmount: 480,
    };
    await dal_manager_1.dbManager.accessPacs.upsertOrganizationPACSModuleSettings(organizationId, {
        organizationId: organizationId,
        annualLeavePPermissionTypeId: ppermissionTypeTwo.id,
        casualLeavePPermissionTypeId: ppermissionTypeOne.id,
        annualPPermissionAgeLimits: annualPPermissionAgeLimits,
        middayHour: "12:00",
    }, trx);
    let updateIntervals = [];
    updateIntervals.push({
        lowerExperienceDuration: 5110,
        higherExperienceDuration: 100000,
        claim: 624,
    });
    updateIntervals.push({
        lowerExperienceDuration: 1825,
        higherExperienceDuration: 5109,
        claim: 480,
    });
    updateIntervals.push({
        lowerExperienceDuration: 365,
        higherExperienceDuration: 1824,
        claim: 336,
    });
    updateIntervals.push({
        lowerExperienceDuration: 0,
        higherExperienceDuration: 364,
        claim: 0,
    });
    await dal_manager_1.dbManager.accessPacs.updateAnnualPPermissionIntervals(organizationId, updateIntervals, trx);
}
exports.upsertPredefinedPPermissions = upsertPredefinedPPermissions;
async function decoratePPermissionListWithCaptions(organizationId, list) {
    let result = {
        items: [],
        pagination: list.pagination,
    };
    let userIds = [].concat.apply([], list.items.map((u) => u.userPPermissionInfo.map((a) => a.userAndDeputyInfo.user.id)));
    let deputyUserIds = [].concat.apply([], list.items.map((u) => u.userPPermissionInfo.filter((a) => a.userAndDeputyInfo.deputyUser).map((a) => a.userAndDeputyInfo.deputyUser.id)));
    let captions = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, userIds.concat(deputyUserIds));
    for (let i = 0; i < list.items.length; i++) {
        let ppermission = list.items[i];
        for (let k = 0; k < ppermission.userPPermissionInfo.length; k++) {
            ppermission.userPPermissionInfo[k].userAndDeputyInfo.userCaptions = (0, dal_utils_1.loadSafeUserCaptions)(ppermission.userPPermissionInfo[k].userAndDeputyInfo.user.id, captions);
            if (ppermission.userPPermissionInfo[k].userAndDeputyInfo.deputyUser) {
                ppermission.userPPermissionInfo[k].userAndDeputyInfo.deputyCaptions = (0, dal_utils_1.loadSafeUserCaptions)(ppermission.userPPermissionInfo[k].userAndDeputyInfo.deputyUser.id, captions);
            }
        }
        result.items.push(ppermission);
    }
    return Promise.resolve(result);
}
exports.decoratePPermissionListWithCaptions = decoratePPermissionListWithCaptions;
function getHistoryOfPPermissionOfUser(items, userId) {
    return items.filter((f) => {
        let exists = false;
        if (f.actionUserId === userId) {
            exists = true;
        }
        if (f.content) {
            if (f.content.userAndDeputyList) {
                exists = exists || f.content.userAndDeputyList.findIndex((t) => t.userId === userId) > -1;
            }
            if (f.content.approvementList) {
                exists = exists || f.content.approvementList.findIndex((t) => t.approverUserId === userId) > -1;
            }
        }
        if (f.notification) {
            exists = exists || f.notification.receiverUserId === userId;
        }
        return exists;
    });
}
exports.getHistoryOfPPermissionOfUser = getHistoryOfPPermissionOfUser;
async function transformPPermissionHistoryItemsToLocalizedMessages(organizationId, items, locale) {
    let result = [];
    let userIds = items.map((m) => m.actionUserId);
    userIds.concat(items.filter((m) => m.notification != null).map((m) => m.notification.receiverUserId));
    let userFullnames = await dal_manager_1.dbManager.accessUser.getUserOrganizationProfileFullNames(organizationId, userIds, true);
    for (const item of items) {
        let message = "";
        let userFullnameItem = userFullnames.find((u) => (item.notification != null ? u.id === item.notification.receiverUserId : u.id === item.actionUserId));
        let userFullname = userFullnameItem ? userFullnameItem.name : i18n_1.default.__({ phrase: dal_constants_1.DalConstants.SystemUserName, locale: locale });
        let notificationTypeLocaled = "";
        let notificationMediums = "";
        switch (item.actionType) {
            case app_enums_1.enums.PPermissionHistoryActionType.CreatedBySelf:
                message = i18n_1.default.__({ phrase: "PACS.PPERMISSIONHISTORYITEMMESSAGE.CREATEDBYSELF", locale: locale });
                break;
            case app_enums_1.enums.PPermissionHistoryActionType.CreatedByManager:
                message = (0, app_util_1.stringFormat)(i18n_1.default.__({ phrase: "PACS.PPERMISSIONHISTORYITEMMESSAGE.CREATEDBYMANAGER", locale: locale }), [userFullname]);
                break;
            case app_enums_1.enums.PPermissionHistoryActionType.CreatedBySystem:
                message = i18n_1.default.__({ phrase: "PACS.PPERMISSIONHISTORYITEMMESSAGE.CREATEDBYSYSTEM", locale: locale });
                break;
            case app_enums_1.enums.PPermissionHistoryActionType.Approved:
                message = (0, app_util_1.stringFormat)(i18n_1.default.__({ phrase: "PACS.PPERMISSIONHISTORYITEMMESSAGE.APPROVED", locale: locale }), [userFullname]);
                break;
            case app_enums_1.enums.PPermissionHistoryActionType.Rejected:
                message = (0, app_util_1.stringFormat)(i18n_1.default.__({ phrase: "PACS.PPERMISSIONHISTORYITEMMESSAGE.REJECTED", locale: locale }), [userFullname]);
                break;
            case app_enums_1.enums.PPermissionHistoryActionType.UpdatedBySelf:
                message = i18n_1.default.__({ phrase: "PACS.PPERMISSIONHISTORYITEMMESSAGE.UPDATEDBYSELF", locale: locale });
                break;
            case app_enums_1.enums.PPermissionHistoryActionType.UpdatedByManager:
                message = (0, app_util_1.stringFormat)(i18n_1.default.__({ phrase: "PACS.PPERMISSIONHISTORYITEMMESSAGE.UPDATEDBYMANAGER", locale: locale }), [userFullname]);
                break;
            case app_enums_1.enums.PPermissionHistoryActionType.UpdatedBySystem:
                message = i18n_1.default.__({ phrase: "PACS.PPERMISSIONHISTORYITEMMESSAGE.UPDATEDBYSYSTEM", locale: locale });
                break;
            case app_enums_1.enums.PPermissionHistoryActionType.Deleted:
                message = (0, app_util_1.stringFormat)(i18n_1.default.__({ phrase: "PACS.PPERMISSIONHISTORYITEMMESSAGE.DELETED", locale: locale }), [userFullname]);
                break;
            case app_enums_1.enums.PPermissionHistoryActionType.Notification:
                notificationTypeLocaled =
                    item.notification.type === app_enums_1.enums.NotificationType.PPermissionApprovementToOwner
                        ? i18n_1.default.__({ phrase: "PACS.PPERMISSIONHISTORYITEMMESSAGE.PPermissionApprovement", locale: locale })
                        : item.notification.type === app_enums_1.enums.NotificationType.PPermissionWaitForApprovementToApprover
                            ? i18n_1.default.__({ phrase: "PACS.PPERMISSIONHISTORYITEMMESSAGE.PPermissionNeedApprove", locale: locale })
                            : item.notification.type === app_enums_1.enums.NotificationType.PPermissionWaitForSignature
                                ? i18n_1.default.__({ phrase: "PACS.PPERMISSIONHISTORYITEMMESSAGE.PPermissionSign", locale: locale })
                                : i18n_1.default.__({ phrase: "PACS.PPERMISSIONHISTORYITEMMESSAGE.PPermissionOther", locale: locale });
                notificationMediums = item.notification.mediums
                    .map((m) => {
                    return m === app_enums_1.enums.NotificationMedium.Email
                        ? i18n_1.default.__({ phrase: "NOTIFICATION.MEDIUM.EMAIL", locale: locale })
                        : m === app_enums_1.enums.NotificationMedium.PushNotification
                            ? i18n_1.default.__({ phrase: "NOTIFICATION.MEDIUM.PUSHNOTIFICATION", locale: locale })
                            : m === app_enums_1.enums.NotificationMedium.SMS
                                ? i18n_1.default.__({ phrase: "NOTIFICATION.MEDIUM.SMS", locale: locale })
                                : m === app_enums_1.enums.NotificationMedium.Web
                                    ? i18n_1.default.__({ phrase: "NOTIFICATION.MEDIUM.WEB", locale: locale })
                                    : i18n_1.default.__({ phrase: "NOTIFICATION.MEDIUM.OTHER", locale: locale });
                })
                    .join(", ");
                message = (0, app_util_1.stringFormat)(i18n_1.default.__({ phrase: "PACS.PPERMISSIONHISTORYITEMMESSAGE.NOTIFICATION", locale: locale }), [
                    notificationTypeLocaled,
                    userFullname,
                    notificationMediums,
                ]);
                break;
            case app_enums_1.enums.PPermissionHistoryActionType.EmployeeListChanged:
                message = (0, app_util_1.stringFormat)(i18n_1.default.__({ phrase: "PACS.PPERMISSIONHISTORYITEMMESSAGE.EMPLOYEELISTCHANGED", locale: locale }), [userFullname]);
                break;
            case app_enums_1.enums.PPermissionHistoryActionType.UpdateDateTime:
                message = (0, app_util_1.stringFormat)(i18n_1.default.__({ phrase: "PACS.PPERMISSIONHISTORYITEMMESSAGE.UPDATEDATETIME", locale: locale }), [userFullname]);
                break;
            case app_enums_1.enums.PPermissionHistoryActionType.PPermissionSigned:
                message = (0, app_util_1.stringFormat)(i18n_1.default.__({ phrase: "PACS.PPERMISSIONHISTORYITEMMESSAGE.PPERMISSIONSIGNED", locale: locale }), [userFullname]);
                break;
            default:
                break;
        }
        let resultItem = {
            actionUtc: item.actionUtc,
            actionUserId: item.actionUserId,
            actionUserFullName: userFullname,
            logId: item.id,
            type: item.actionType,
            message: message,
        };
        result.push(resultItem);
    }
    return Promise.resolve(result);
}
exports.transformPPermissionHistoryItemsToLocalizedMessages = transformPPermissionHistoryItemsToLocalizedMessages;
var BusinessWorkPlanImport;
(function (BusinessWorkPlanImport) {
    async function downloadWorkPlanExcel(organizationId, userId, locale, params) {
        let workPlanExcel = new business_pacs_excel_1.WorkPlanImportUploadExcel(organizationId, userId, locale);
        let workPlanExcelResponse = {};
        let importTemplate = await dal_manager_1.dbManager.accessPacs.getWorkPlanShiftImport(organizationId, params.templateId);
        const timeout = setTimeout(() => {
            workPlanExcelResponse.errorMessage = i18n_2.default.__({ phrase: "VISITOR-EXCEL.TIMEOUT.WRITE", locale: locale });
            return workPlanExcelResponse;
        }, 10 * 1000);
        workPlanExcelResponse = await workPlanExcel.generateExcel({ dateRange: params.dateRange, days: params.days, type: params.type, userFilterId: params.userFilterId }, importTemplate);
        clearTimeout(timeout);
        return workPlanExcelResponse;
    }
    BusinessWorkPlanImport.downloadWorkPlanExcel = downloadWorkPlanExcel;
    async function validateWorkPlanExcel(organizationId, userId, params) {
        return new Promise(async (resolve, reject) => {
            let workPlanExcel = new business_pacs_excel_1.WorkPlanImportUploadExcel(organizationId, userId, params.locale);
            let workPlanExcelResponse = {};
            let importTemplate = await dal_manager_1.dbManager.accessPacs.getWorkPlanShiftImport(organizationId, params.templateId);
            const loadTimeout = setTimeout(() => {
                workPlanExcelResponse.errorMessage = i18n_2.default.__({ phrase: "VISITOR-EXCEL.TIMEOUT.LOAD", locale: params.locale });
                resolve(workPlanExcelResponse);
            }, 10 * 1000);
            await workPlanExcel.loadExcel(params.fileBuffer);
            clearTimeout(loadTimeout);
            const readTimeout = setTimeout(() => {
                workPlanExcelResponse.errorMessage = i18n_2.default.__({ phrase: "VISITOR-EXCEL.TIMEOUT.READ", locale: params.locale });
                resolve(workPlanExcelResponse);
            }, 10 * 1000);
            let validation = await workPlanExcel.read(importTemplate, params.type);
            clearTimeout(readTimeout);
            let failedRows = [];
            workPlanExcel.uploadExcel(params);
            if (validation.validationSuccess) {
                if (failedRows && failedRows.length > 0) {
                    validation.validationSuccess = false;
                }
                else {
                    app_httpserver_1.httpServer.sendVisitorListChangedNotification(organizationId);
                }
            }
            resolve(validation);
        });
    }
    BusinessWorkPlanImport.validateWorkPlanExcel = validateWorkPlanExcel;
})(BusinessWorkPlanImport = exports.BusinessWorkPlanImport || (exports.BusinessWorkPlanImport = {}));
