"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getUsersAnnualVacationStatusAtDateBulk = exports.getUsersAnnualVacationStatusAtDate = exports.getPPermissionClaimAndUsageBulk = exports.getPPermissionClaimAndUsage = exports.getUsersPPermissionsTypeUsageOfBetweenDates = exports.getPPermissionTypeUsageOfUsersBetweenDates = exports.getPPermissionTypeUsageOfUserBetweenDates = void 0;
const dal_manager_1 = require("../../dal/dal.manager");
const business_pacs_ppermission_1 = require("./business.pacs.ppermission");
const app_enums_1 = require("../../app.enums");
const moment_1 = __importDefault(require("moment"));
const moment_range_1 = require("moment-range");
const api_error_1 = require("../../api/api.error");
const business_hooks_1 = require("../business.hooks");
const luxon_1 = require("luxon");
function getPPermissionTypeUsageOfUserBetweenDates(organizationId, userId, ppermissionTypeId, startDateTime, endDateTime) {
    return dal_manager_1.dbManager.armondb.transaction(async (trx) => {
        let result = {
            type: null,
            ppermissionClaimProfile: null,
            periods: [],
        };
        let existingSettings = await dal_manager_1.dbManager.accessPacs.getOrganizationPACSModuleSettings(organizationId, trx);
        if (!existingSettings || !existingSettings.annualLeavePPermissionTypeId) {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.PACS.PACSMODULENOTENABLED");
        }
        result.type = await dal_manager_1.dbManager.accessPacs.getPPermissionType(organizationId, ppermissionTypeId);
        if (!result.type) {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.PACS.PERMISSIONTYPENOTFOUND", null, true);
        }
        let pacsModule = (await dal_manager_1.dbManager.accessOrganization.listOrganizationModules(organizationId)).find((m) => m.module === app_enums_1.enums.ArmonApplicationModule.PACS);
        if (!pacsModule) {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.PACS.PACSMODULENOTENABLED");
        }
        else if (!pacsModule.enableUtc) {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.PACS.MODULEENABLETIMEMISSING");
        }
        let pacsEnableTimestamp = (0, moment_1.default)(pacsModule.enableUtc);
        let userEmploymentInfo = await dal_manager_1.dbManager.accessUser.getUserOrganizationProfileEmploymentInfo(organizationId, userId, trx);
        if (!userEmploymentInfo.employmentStartUtc) {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.PACS.EMPLOYEESTARTDATEMISSING", null, true);
        }
        let employmentStartTimestamp = (0, moment_1.default)(userEmploymentInfo.employmentStartUtc);
        let filterRange = new moment_range_1.DateRange(startDateTime, endDateTime);
        if (employmentStartTimestamp.isAfter(filterRange.start)) {
            filterRange.start = employmentStartTimestamp;
        }
        if (filterRange.end.isAfter((0, moment_1.default)())) {
            filterRange.end = (0, moment_1.default)();
        }
        let vacations = await dal_manager_1.dbManager.accessPacs.listAllVacations(organizationId);
        let ppermissions = await dal_manager_1.dbManager.accessPacs2.getApprovedPPermissionsOfUserAtDate(organizationId, userId, new Date());
        if (result.type.type === app_enums_1.enums.PredefinedPPermissionType.Annual) {
            let updateIntervals = await dal_manager_1.dbManager.accessOrganization.getOrganizationAnnualPPermissionUpdateIntervals(organizationId, trx);
            if (!updateIntervals || updateIntervals.length < 1) {
                updateIntervals = (0, business_pacs_ppermission_1.getDefaultAnnualPPermissionIntervals)();
            }
            let unpaidPPermissions = await dal_manager_1.dbManager.accessPacs.listPastUnpaidPPermissionsOfUser(organizationId, userId, filterRange.end.toDate(), trx);
            let currentWorkingTime = 0;
            if (userEmploymentInfo.previousServiceDuration) {
                currentWorkingTime += userEmploymentInfo.previousServiceDuration;
            }
            let rolloverAmount = 0;
            if (employmentStartTimestamp.isSameOrBefore(pacsEnableTimestamp)) {
                rolloverAmount = userEmploymentInfo?.pacsEnabledRemainedAnnualPPermission ? userEmploymentInfo.pacsEnabledRemainedAnnualPPermission : 0;
            }
            else {
                rolloverAmount = userEmploymentInfo?.previousAnnualVacationRight ? userEmploymentInfo.previousAnnualVacationRight : 0;
            }
            rolloverAmount = rolloverAmount / 24;
            for (let startDate = (0, moment_1.default)(userEmploymentInfo.employmentStartUtc); startDate.isBefore(filterRange.end);) {
                let periodRange = new moment_range_1.DateRange(startDate, startDate.clone().add(1, "year"));
                let totalUnpaidOffset = 0;
                if (unpaidPPermissions) {
                    for (const ppermission of unpaidPPermissions) {
                        if (periodRange.contains(ppermission.startDateTime)) {
                            let duration = new moment_range_1.DateRange(ppermission.startDateTime, ppermission.endDateTime).duration("days");
                            totalUnpaidOffset += duration;
                            periodRange.end = periodRange.end.add(duration, "days");
                        }
                    }
                }
                let claim = 0;
                let targetInterval = updateIntervals.find((i) => currentWorkingTime >= i.lowerExperienceDuration && currentWorkingTime <= i.higherExperienceDuration);
                if (targetInterval)
                    claim = targetInterval.claim;
                currentWorkingTime += 365;
                let nextClaim = 0;
                let nextInterval = updateIntervals.find((i) => currentWorkingTime >= i.lowerExperienceDuration && currentWorkingTime <= i.higherExperienceDuration);
                if (nextInterval)
                    nextClaim = nextInterval.claim;
                if (existingSettings.annualPPermissionAgeLimits && userEmploymentInfo.birthDateUtc && existingSettings.ppermissionClaimProfile === app_enums_1.enums.PPermissionClaimProfile.Auto) {
                    let userAge = Math.abs((0, moment_1.default)(userEmploymentInfo.birthDateUtc).diff(periodRange.start, "year"));
                    if (existingSettings.annualPPermissionAgeLimits.youngerMinimumAmount &&
                        existingSettings.annualPPermissionAgeLimits.youngerThanLimitYears &&
                        userAge < existingSettings.annualPPermissionAgeLimits.youngerThanLimitYears) {
                        claim =
                            claim > 0 && claim < existingSettings.annualPPermissionAgeLimits.youngerMinimumAmount ? existingSettings.annualPPermissionAgeLimits.youngerMinimumAmount : claim;
                    }
                    else if (existingSettings.annualPPermissionAgeLimits.olderMinimumAmount &&
                        existingSettings.annualPPermissionAgeLimits.olderThanLimitYears &&
                        userAge >= existingSettings.annualPPermissionAgeLimits.olderThanLimitYears) {
                        claim = claim > 0 && claim < existingSettings.annualPPermissionAgeLimits.olderMinimumAmount ? existingSettings.annualPPermissionAgeLimits.olderMinimumAmount : claim;
                    }
                    if (existingSettings.annualPPermissionAgeLimits.youngerMinimumAmount &&
                        existingSettings.annualPPermissionAgeLimits.youngerThanLimitYears &&
                        userAge + 1 < existingSettings.annualPPermissionAgeLimits.youngerThanLimitYears) {
                        nextClaim =
                            nextClaim < existingSettings.annualPPermissionAgeLimits.youngerMinimumAmount ? existingSettings.annualPPermissionAgeLimits.youngerMinimumAmount : nextClaim;
                    }
                    else if (existingSettings.annualPPermissionAgeLimits.olderMinimumAmount &&
                        existingSettings.annualPPermissionAgeLimits.olderThanLimitYears &&
                        userAge + 1 >= existingSettings.annualPPermissionAgeLimits.olderThanLimitYears) {
                        nextClaim = nextClaim < existingSettings.annualPPermissionAgeLimits.olderMinimumAmount ? existingSettings.annualPPermissionAgeLimits.olderMinimumAmount : nextClaim;
                    }
                }
                let summationRange = periodRange.clone();
                if (summationRange.end.isBefore(pacsEnableTimestamp)) {
                    startDate = periodRange.end;
                    continue;
                }
                else if (summationRange.start.isBefore(pacsEnableTimestamp)) {
                    summationRange.start = (0, moment_1.default)(pacsModule.enableUtc);
                }
                if (summationRange.end.isAfter((0, moment_1.default)(endDateTime))) {
                    summationRange.end = (0, moment_1.default)(endDateTime);
                }
                let usage = 0;
                for (const row of ppermissions) {
                    if (row.ppermissionTypeId === result.type.id) {
                        let ppermRange = new moment_range_1.DateRange(row.startDateTime, new Date(row.endDateTime));
                        if (ppermRange.overlaps(summationRange)) {
                            let intersection = ppermRange.intersect(summationRange);
                            usage += await (0, business_pacs_ppermission_1.calculateDailyVacationUsage)(organizationId, ppermissionTypeId, intersection.start.toDate(), intersection.end.toDate(), vacations, userId, result.type);
                        }
                    }
                }
                let periodResult = {
                    periodStart: periodRange.start.toDate(),
                    periodEnd: periodRange.end.toDate(),
                    fromPreviousPeriod: existingSettings.ppermissionClaimProfile === app_enums_1.enums.PPermissionClaimProfile.Manual ? null : rolloverAmount,
                    claim: existingSettings.ppermissionClaimProfile === app_enums_1.enums.PPermissionClaimProfile.Manual ? null : periodRange.contains(pacsEnableTimestamp) ? 0 : claim / 24,
                    nextClaim: existingSettings.ppermissionClaimProfile === app_enums_1.enums.PPermissionClaimProfile.Manual ? null : nextClaim ? nextClaim / 24 : 0,
                    usage: usage,
                    remaining: 0,
                    unpaidLeaveOffset: totalUnpaidOffset,
                };
                periodResult.remaining =
                    existingSettings.ppermissionClaimProfile === app_enums_1.enums.PPermissionClaimProfile.Manual ? null : periodResult.claim + periodResult.fromPreviousPeriod - usage;
                rolloverAmount = periodResult.remaining;
                if (filterRange.overlaps(periodRange) && periodRange.end.isAfter(pacsEnableTimestamp)) {
                    result.periods.push(periodResult);
                }
                startDate = periodRange.end;
            }
        }
        else {
            let start;
            let end = filterRange.end;
            let interval = null;
            const hook = business_hooks_1.armonHookManager.getPPermissionsClaimWindowRangesHook(organizationId);
            if (result.type.maxValueBoundaryInterval === app_enums_1.enums.PPermissionMaxValueBoundaryInterval.Annual) {
                interval = "year";
            }
            else if (result.type.maxValueBoundaryInterval === app_enums_1.enums.PPermissionMaxValueBoundaryInterval.Monthly) {
                interval = "month";
            }
            if (hook) {
                const ppermissionWindowRanges = await hook(organizationId, userId, result.type.id, filterRange.start);
                switch (result.type.maxValueBoundaryInterval) {
                    case app_enums_1.enums.PPermissionMaxValueBoundaryInterval.Annual:
                        start = ppermissionWindowRanges.yearRange.start.clone();
                        break;
                    case app_enums_1.enums.PPermissionMaxValueBoundaryInterval.Monthly:
                        start = ppermissionWindowRanges.monthRange.start.clone();
                        break;
                    default:
                        break;
                }
            }
            else {
                start = filterRange.start.clone().startOf(interval);
                end = filterRange.end.clone().add(1, interval).startOf(interval);
            }
            while (start.isBefore(end, "day")) {
                let usage = 0;
                let periodRange = new moment_range_1.DateRange(start, start.clone().add(1, interval)).intersect(filterRange);
                for (const row of ppermissions) {
                    if (row.ppermissionTypeId === result.type.id) {
                        let ppermRange = new moment_range_1.DateRange(row.startDateTime, new Date(row.endDateTime));
                        if (ppermRange.overlaps(periodRange)) {
                            let intersection = ppermRange.intersect(periodRange);
                            if (result.type.isDailyScheduled) {
                                usage += await (0, business_pacs_ppermission_1.calculateDailyVacationUsage)(organizationId, ppermissionTypeId, intersection.start.toDate(), intersection.end.toDate(), vacations, userId, result.type);
                            }
                            else {
                                usage += await (0, business_pacs_ppermission_1.estimateHourlyVacationUsage)(organizationId, ppermissionTypeId, intersection.start.toDate(), intersection.end.toDate(), vacations, result.type);
                            }
                        }
                    }
                }
                let row = {
                    periodStart: periodRange.start.toDate(),
                    periodEnd: start.clone().add(1, interval).toDate(),
                    fromPreviousPeriod: 0,
                    claim: result.type.isDailyScheduled ? result.type.maxValue : result.type.maxValue * 60,
                    nextClaim: result.type.isDailyScheduled ? result.type.maxValue : result.type.maxValue * 60,
                    usage: usage,
                    remaining: 0,
                };
                row.remaining = row.claim - row.usage >= 0 ? row.claim - row.usage : 0;
                if (filterRange.overlaps(periodRange) && periodRange.end.isAfter(pacsEnableTimestamp)) {
                    result.periods.push(row);
                }
                start.add(1, interval);
            }
        }
        let reversedResult = [];
        for (const row of result.periods) {
            reversedResult.unshift(row);
        }
        result.periods = reversedResult;
        result.ppermissionClaimProfile = existingSettings.ppermissionClaimProfile;
        return Promise.resolve(result);
    });
}
exports.getPPermissionTypeUsageOfUserBetweenDates = getPPermissionTypeUsageOfUserBetweenDates;
function getPPermissionTypeUsageOfUsersBetweenDates(organizationId, userIds, ppermissionTypeId, startDateTime, endDateTime) {
    return dal_manager_1.dbManager.systemTransaction(async (trx) => {
        const results = [];
        const existingSettings = await dal_manager_1.dbManager.accessPacs.getOrganizationPACSModuleSettings(organizationId);
        if (!existingSettings || !existingSettings.annualLeavePPermissionTypeId) {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.PACS.PACSMODULENOTENABLED");
        }
        const type = await dal_manager_1.dbManager.accessPacs.getPPermissionType(organizationId, ppermissionTypeId);
        if (!type) {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.PACS.PERMISSIONTYPENOTFOUND", null, true);
        }
        const pacsModule = (await dal_manager_1.dbManager.accessOrganization.listOrganizationModules(organizationId)).find((m) => m.module === app_enums_1.enums.ArmonApplicationModule.PACS);
        if (!pacsModule) {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.PACS.PACSMODULENOTENABLED");
        }
        else if (!pacsModule.enableUtc) {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.PACS.MODULEENABLETIMEMISSING");
        }
        const pacsEnableDate = luxon_1.DateTime.fromJSDate(pacsModule.enableUtc);
        const vacations = await dal_manager_1.dbManager.accessPacs.listAllVacations(organizationId);
        let updateIntervals = await dal_manager_1.dbManager.accessOrganization.getOrganizationAnnualPPermissionUpdateIntervals(organizationId);
        if (!updateIntervals || updateIntervals.length < 1)
            updateIntervals = (0, business_pacs_ppermission_1.getDefaultAnnualPPermissionIntervals)();
        const now = luxon_1.DateTime.now();
        const globalStart = luxon_1.DateTime.fromJSDate(startDateTime);
        const globalEnd = luxon_1.DateTime.fromJSDate(endDateTime) > now ? now : luxon_1.DateTime.fromJSDate(endDateTime);
        const usersEmployementInfos = await dal_manager_1.dbManager.accessUser.getEmploymentsUserOrganizationProfileInfos(organizationId, userIds, trx);
        const usersPermissions = await dal_manager_1.dbManager.accessPacs2.getUsersApprovedPPermissionsAtDate(organizationId, userIds, new Date(), trx);
        for (const userId of userIds) {
            try {
                const userEmploymentInfo = usersEmployementInfos.find((e) => e.userId === userId).employeeProfileInfo;
                if (!userEmploymentInfo.employmentStartUtc) {
                    throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.PACS.EMPLOYEESTARTDATEMISSING", null, true);
                }
                const employmentStart = luxon_1.DateTime.fromJSDate(userEmploymentInfo.employmentStartUtc);
                const filterStart = employmentStart > globalStart ? employmentStart : globalStart;
                const filterRange = luxon_1.Interval.fromDateTimes(filterStart, globalEnd);
                const ppermissions = usersPermissions.find((p) => p.userId === userId).approvedPPermission;
                const result = {
                    userId,
                    type,
                    ppermissionClaimProfile: existingSettings.ppermissionClaimProfile,
                    periods: [],
                };
                if (type.type === app_enums_1.enums.PredefinedPPermissionType.Annual) {
                    const unpaidPPermissions = await dal_manager_1.dbManager.accessPacs.listPastUnpaidPPermissionsOfUser(organizationId, userId, filterRange.end.toJSDate());
                    let currentWorkingTime = userEmploymentInfo.previousServiceDuration || 0;
                    let rolloverAmount = 0;
                    if (employmentStart <= pacsEnableDate) {
                        rolloverAmount = userEmploymentInfo.pacsEnabledRemainedAnnualPPermission || 0;
                    }
                    else {
                        rolloverAmount = userEmploymentInfo.previousAnnualVacationRight || 0;
                    }
                    rolloverAmount /= 24;
                    for (let periodStart = employmentStart; periodStart < filterRange.end;) {
                        const nextYear = periodStart.plus({ year: 1 });
                        let periodEnd = nextYear;
                        let totalUnpaidOffset = 0;
                        for (const unpaid of unpaidPPermissions || []) {
                            const unpaidRange = luxon_1.Interval.fromDateTimes(luxon_1.DateTime.fromJSDate(unpaid.startDateTime), luxon_1.DateTime.fromJSDate(unpaid.endDateTime));
                            if (luxon_1.Interval.fromDateTimes(periodStart, periodEnd).overlaps(unpaidRange)) {
                                const overlap = unpaidRange.intersection(luxon_1.Interval.fromDateTimes(periodStart, periodEnd));
                                if (overlap) {
                                    const duration = overlap.toDuration("days").days;
                                    totalUnpaidOffset += duration;
                                    periodEnd = periodEnd.plus({ days: duration });
                                }
                            }
                        }
                        const targetInterval = updateIntervals.find((i) => currentWorkingTime >= i.lowerExperienceDuration && currentWorkingTime <= i.higherExperienceDuration);
                        const nextInterval = updateIntervals.find((i) => currentWorkingTime + 365 >= i.lowerExperienceDuration && currentWorkingTime + 365 <= i.higherExperienceDuration);
                        let claim = targetInterval?.claim || 0;
                        let nextClaim = nextInterval?.claim || 0;
                        if (existingSettings.annualPPermissionAgeLimits &&
                            userEmploymentInfo.birthDateUtc &&
                            existingSettings.ppermissionClaimProfile === app_enums_1.enums.PPermissionClaimProfile.Auto) {
                            const userAge = periodStart.diff(luxon_1.DateTime.fromJSDate(userEmploymentInfo.birthDateUtc), "years").years;
                            const nextAge = userAge + 1;
                            const ageLimits = existingSettings.annualPPermissionAgeLimits;
                            if (userAge < ageLimits.youngerThanLimitYears && claim < ageLimits.youngerMinimumAmount) {
                                claim = ageLimits.youngerMinimumAmount;
                            }
                            else if (userAge >= ageLimits.olderThanLimitYears && claim < ageLimits.olderMinimumAmount) {
                                claim = ageLimits.olderMinimumAmount;
                            }
                            if (nextAge < ageLimits.youngerThanLimitYears && nextClaim < ageLimits.youngerMinimumAmount) {
                                nextClaim = ageLimits.youngerMinimumAmount;
                            }
                            else if (nextAge >= ageLimits.olderThanLimitYears && nextClaim < ageLimits.olderMinimumAmount) {
                                nextClaim = ageLimits.olderMinimumAmount;
                            }
                        }
                        let summationStart = periodStart < pacsEnableDate ? pacsEnableDate : periodStart;
                        let summationEnd = periodEnd > globalEnd ? globalEnd : periodEnd;
                        const summationRange = luxon_1.Interval.fromDateTimes(summationStart, summationEnd);
                        let usage = 0;
                        for (const row of ppermissions) {
                            if (row.ppermissionTypeId === type.id) {
                                const permRange = luxon_1.Interval.fromDateTimes(luxon_1.DateTime.fromJSDate(new Date(row.startDateTime)), luxon_1.DateTime.fromJSDate(new Date(row.endDateTime)));
                                if (permRange.overlaps(summationRange)) {
                                    const overlap = permRange.intersection(summationRange);
                                    usage += await (0, business_pacs_ppermission_1.calculateDailyVacationUsage)(organizationId, ppermissionTypeId, overlap.start.toJSDate(), overlap.end.toJSDate(), vacations, userId, type);
                                }
                            }
                        }
                        const periodResult = {
                            periodStart: periodStart.toJSDate(),
                            periodEnd: periodEnd.toJSDate(),
                            fromPreviousPeriod: result.ppermissionClaimProfile === app_enums_1.enums.PPermissionClaimProfile.Manual ? null : rolloverAmount,
                            claim: result.ppermissionClaimProfile === app_enums_1.enums.PPermissionClaimProfile.Manual ? null : periodStart <= pacsEnableDate ? 0 : claim / 24,
                            nextClaim: result.ppermissionClaimProfile === app_enums_1.enums.PPermissionClaimProfile.Manual ? null : nextClaim / 24,
                            usage,
                            remaining: 0,
                            unpaidLeaveOffset: totalUnpaidOffset,
                        };
                        periodResult.remaining =
                            result.ppermissionClaimProfile === app_enums_1.enums.PPermissionClaimProfile.Manual ? null : periodResult.claim + periodResult.fromPreviousPeriod - usage;
                        rolloverAmount = periodResult.remaining;
                        if (filterRange.overlaps(luxon_1.Interval.fromDateTimes(periodStart, periodEnd)) && periodEnd > pacsEnableDate) {
                            result.periods.push(periodResult);
                        }
                        currentWorkingTime += 365;
                        periodStart = nextYear;
                    }
                }
                else {
                    let start;
                    let end = globalEnd;
                    let interval = null;
                    const hook = business_hooks_1.armonHookManager.getPPermissionsClaimWindowRangesHook(organizationId);
                    if (type.maxValueBoundaryInterval === app_enums_1.enums.PPermissionMaxValueBoundaryInterval.Annual) {
                        interval = "year";
                    }
                    else if (type.maxValueBoundaryInterval === app_enums_1.enums.PPermissionMaxValueBoundaryInterval.Monthly) {
                        interval = "month";
                    }
                    if (hook) {
                        const ppermissionWindowRanges = await hook(organizationId, userId, type.id, (0, moment_1.default)(filterRange.start.toJSDate()));
                        switch (type.maxValueBoundaryInterval) {
                            case app_enums_1.enums.PPermissionMaxValueBoundaryInterval.Annual:
                                start = luxon_1.DateTime.fromJSDate(ppermissionWindowRanges.yearRange.start.toDate());
                                break;
                            case app_enums_1.enums.PPermissionMaxValueBoundaryInterval.Monthly:
                                start = luxon_1.DateTime.fromJSDate(ppermissionWindowRanges.monthRange.start.toDate());
                                break;
                            default:
                                break;
                        }
                    }
                    else {
                        start = filterRange.start.startOf(interval);
                        end = globalEnd.plus({ [interval]: 1 }).startOf(interval);
                    }
                    while (start < end) {
                        let usage = 0;
                        const periodStart = start;
                        const periodEnd = start.plus({ [interval]: 1 });
                        const periodRange = luxon_1.Interval.fromDateTimes(periodStart, periodEnd).intersection(filterRange);
                        if (periodRange) {
                            for (const row of ppermissions) {
                                if (row.ppermissionTypeId === type.id) {
                                    const ppermRange = luxon_1.Interval.fromDateTimes(luxon_1.DateTime.fromJSDate(new Date(row.startDateTime)), luxon_1.DateTime.fromJSDate(new Date(row.endDateTime)));
                                    if (ppermRange.overlaps(periodRange)) {
                                        const intersection = ppermRange.intersection(periodRange);
                                        if (intersection) {
                                            if (type.isDailyScheduled) {
                                                usage += await (0, business_pacs_ppermission_1.calculateDailyVacationUsage)(organizationId, ppermissionTypeId, intersection.start.toJSDate(), intersection.end.toJSDate(), vacations, userId, type);
                                            }
                                            else {
                                                usage += await (0, business_pacs_ppermission_1.estimateHourlyVacationUsage)(organizationId, ppermissionTypeId, intersection.start.toJSDate(), intersection.end.toJSDate(), vacations, type);
                                            }
                                        }
                                    }
                                }
                            }
                            const row = {
                                periodStart: periodStart.toJSDate(),
                                periodEnd: periodEnd.toJSDate(),
                                fromPreviousPeriod: 0,
                                claim: type.isDailyScheduled ? type.maxValue : type.maxValue * 60,
                                nextClaim: type.isDailyScheduled ? type.maxValue : type.maxValue * 60,
                                usage: usage,
                                remaining: 0,
                            };
                            row.remaining = row.claim - row.usage >= 0 ? row.claim - row.usage : 0;
                            if (filterRange.overlaps(luxon_1.Interval.fromDateTimes(periodStart, periodEnd)) && periodEnd > pacsEnableDate) {
                                result.periods.push(row);
                            }
                        }
                        start = start.plus({ [interval]: 1 });
                    }
                }
                result.periods.reverse();
                results.push(result);
            }
            catch (e) {
                continue;
            }
        }
        return results;
    });
}
exports.getPPermissionTypeUsageOfUsersBetweenDates = getPPermissionTypeUsageOfUsersBetweenDates;
async function getUsersPPermissionsTypeUsageOfBetweenDates(organizationId, userIds, ppermissionTypeId, trx) {
    let result = {
        permissionTypeInfo: null,
        ppermissionClaimProfile: null,
        userPeriods: [],
    };
    let existingSettings = await dal_manager_1.dbManager.accessPacs.getOrganizationPACSModuleSettingsPoolClient(organizationId, trx);
    if (!existingSettings || !existingSettings.annualLeavePPermissionTypeId) {
        throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.PACS.PACSMODULENOTENABLED");
    }
    result.permissionTypeInfo = await dal_manager_1.dbManager.accessPacs.getPPermissionTypePoolClient(organizationId, ppermissionTypeId, trx);
    if (!result.permissionTypeInfo) {
        throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.PACS.PERMISSIONTYPENOTFOUND", null, true);
    }
    let pacsModule = (await dal_manager_1.dbManager.accessOrganization.listOrganizationModules(organizationId, trx)).find((m) => m.module === app_enums_1.enums.ArmonApplicationModule.PACS);
    if (!pacsModule) {
        throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.PACS.PACSMODULENOTENABLED");
    }
    else if (!pacsModule.enableUtc) {
        throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.PACS.MODULEENABLETIMEMISSING");
    }
    let pacsEnableTimestamp = (0, moment_1.default)(pacsModule.enableUtc);
    let userEmploymentInfo = await dal_manager_1.dbManager.accessUser.getEmploymentsUserOrganizationProfileInfos(organizationId, userIds, trx);
    userEmploymentInfo.filter((elem) => elem.employeeProfileInfo.employmentStartUtc);
    let vacations = await dal_manager_1.dbManager.accessPacs.listAllVacations(organizationId);
    let ppermissions = await dal_manager_1.dbManager.accessPacs2.getUsersApprovedPPermissionsAtDate(organizationId, userIds, new Date(), trx);
    if (result.permissionTypeInfo.type === app_enums_1.enums.PredefinedPPermissionType.Annual) {
        let updateIntervals = await dal_manager_1.dbManager.accessOrganization.getOrganizationAnnualPPermissionUpdateIntervals(organizationId);
        if (!updateIntervals || updateIntervals.length < 1) {
            updateIntervals = (0, business_pacs_ppermission_1.getDefaultAnnualPPermissionIntervals)();
        }
        let unpaidPPermissions = await dal_manager_1.dbManager.accessPacs.listUsersPastUnpaidPPermissions(organizationId, userIds, new Date(), trx);
        for (const employeeInfo of userEmploymentInfo) {
            result.userPeriods.push({ userId: employeeInfo.userId, periods: [] });
            if (!employeeInfo.employeeProfileInfo.employmentStartUtc) {
                continue;
            }
            let filterRange = new moment_range_1.DateRange((0, moment_1.default)(employeeInfo.employeeProfileInfo.employmentStartUtc), (0, moment_1.default)());
            const unPaidPPermForEmployee = unpaidPPermissions.find((elem) => elem.userId === employeeInfo.userId);
            const permissionForEmployee = ppermissions.find((elem) => elem.userId === employeeInfo.userId);
            let employmentStartTimestamp = (0, moment_1.default)(employeeInfo.employeeProfileInfo.employmentStartUtc);
            if (employmentStartTimestamp.isAfter(filterRange.start)) {
                filterRange.start = employmentStartTimestamp;
            }
            if (filterRange.end.isAfter((0, moment_1.default)())) {
                filterRange.end = (0, moment_1.default)();
            }
            let currentWorkingTime = 0;
            if (employeeInfo.employeeProfileInfo.previousServiceDuration) {
                currentWorkingTime += employeeInfo.employeeProfileInfo.previousServiceDuration;
            }
            let rolloverAmount = 0;
            if (employmentStartTimestamp.isSameOrBefore(pacsEnableTimestamp)) {
                rolloverAmount = employeeInfo.employeeProfileInfo.pacsEnabledRemainedAnnualPPermission ? employeeInfo.employeeProfileInfo.pacsEnabledRemainedAnnualPPermission : 0;
            }
            else {
                rolloverAmount = employeeInfo.employeeProfileInfo.previousAnnualVacationRight ? employeeInfo.employeeProfileInfo.previousAnnualVacationRight : 0;
            }
            rolloverAmount = rolloverAmount / 24;
            for (let startDate = (0, moment_1.default)(employeeInfo.employeeProfileInfo.employmentStartUtc); startDate.isBefore(filterRange.end);) {
                let periodRange = new moment_range_1.DateRange(startDate, startDate.clone().add(1, "year"));
                let totalUnpaidOffset = 0;
                if (unPaidPPermForEmployee) {
                    for (const ppermission of unPaidPPermForEmployee.ppermissionDateInfo) {
                        if (periodRange.contains(ppermission.startDateTime)) {
                            let duration = new moment_range_1.DateRange(ppermission.startDateTime, ppermission.endDateTime).duration("days");
                            totalUnpaidOffset += duration;
                            periodRange.end = periodRange.end.add(duration, "days");
                        }
                    }
                }
                let claim = 0;
                let targetInterval = updateIntervals.find((i) => currentWorkingTime >= i.lowerExperienceDuration && currentWorkingTime <= i.higherExperienceDuration);
                if (targetInterval)
                    claim = targetInterval.claim;
                currentWorkingTime += 365;
                let nextClaim = 0;
                let nextInterval = updateIntervals.find((i) => currentWorkingTime >= i.lowerExperienceDuration && currentWorkingTime <= i.higherExperienceDuration);
                if (nextInterval)
                    nextClaim = nextInterval.claim;
                if (existingSettings.annualPPermissionAgeLimits &&
                    employeeInfo.employeeProfileInfo.birthDateUtc &&
                    existingSettings.ppermissionClaimProfile === app_enums_1.enums.PPermissionClaimProfile.Auto) {
                    let userAge = Math.abs((0, moment_1.default)(employeeInfo.employeeProfileInfo.birthDateUtc).diff(periodRange.start, "year"));
                    if (existingSettings.annualPPermissionAgeLimits.youngerMinimumAmount &&
                        existingSettings.annualPPermissionAgeLimits.youngerThanLimitYears &&
                        userAge < existingSettings.annualPPermissionAgeLimits.youngerThanLimitYears) {
                        claim = claim < existingSettings.annualPPermissionAgeLimits.youngerMinimumAmount ? existingSettings.annualPPermissionAgeLimits.youngerMinimumAmount : claim;
                    }
                    else if (existingSettings.annualPPermissionAgeLimits.olderMinimumAmount &&
                        existingSettings.annualPPermissionAgeLimits.olderThanLimitYears &&
                        userAge >= existingSettings.annualPPermissionAgeLimits.olderThanLimitYears) {
                        claim = claim > 0 && claim < existingSettings.annualPPermissionAgeLimits.olderMinimumAmount ? existingSettings.annualPPermissionAgeLimits.olderMinimumAmount : claim;
                    }
                    if (existingSettings.annualPPermissionAgeLimits.youngerMinimumAmount &&
                        existingSettings.annualPPermissionAgeLimits.youngerThanLimitYears &&
                        userAge + 1 < existingSettings.annualPPermissionAgeLimits.youngerThanLimitYears) {
                        nextClaim =
                            nextClaim < existingSettings.annualPPermissionAgeLimits.youngerMinimumAmount ? existingSettings.annualPPermissionAgeLimits.youngerMinimumAmount : nextClaim;
                    }
                    else if (existingSettings.annualPPermissionAgeLimits.olderMinimumAmount &&
                        existingSettings.annualPPermissionAgeLimits.olderThanLimitYears &&
                        userAge + 1 >= existingSettings.annualPPermissionAgeLimits.olderThanLimitYears) {
                        nextClaim = nextClaim < existingSettings.annualPPermissionAgeLimits.olderMinimumAmount ? existingSettings.annualPPermissionAgeLimits.olderMinimumAmount : nextClaim;
                    }
                }
                let summationRange = periodRange.clone();
                if (summationRange.end.isBefore(pacsEnableTimestamp)) {
                    startDate = periodRange.end;
                    continue;
                }
                else if (summationRange.start.isBefore(pacsEnableTimestamp)) {
                    summationRange.start = (0, moment_1.default)(pacsModule.enableUtc);
                }
                if (summationRange.end.isAfter((0, moment_1.default)())) {
                    summationRange.end = (0, moment_1.default)();
                }
                let usage = 0;
                if (permissionForEmployee) {
                    for (const row of permissionForEmployee.approvedPPermission) {
                        if (row.ppermissionTypeId === result.permissionTypeInfo.id) {
                            let ppermRange = new moment_range_1.DateRange(row.startDateTime, new Date(row.endDateTime));
                            if (ppermRange.overlaps(summationRange)) {
                                let intersection = ppermRange.intersect(summationRange);
                                usage += await (0, business_pacs_ppermission_1.calculateDailyVacationUsage)(organizationId, ppermissionTypeId, intersection.start.toDate(), intersection.end.toDate(), vacations, permissionForEmployee.userId);
                            }
                        }
                    }
                }
                let periodResult = {
                    periodStart: periodRange.start.toDate(),
                    periodEnd: periodRange.end.toDate(),
                    fromPreviousPeriod: existingSettings.ppermissionClaimProfile === app_enums_1.enums.PPermissionClaimProfile.Manual ? null : rolloverAmount,
                    claim: existingSettings.ppermissionClaimProfile === app_enums_1.enums.PPermissionClaimProfile.Manual ? null : periodRange.contains(pacsEnableTimestamp) ? 0 : claim / 24,
                    nextClaim: existingSettings.ppermissionClaimProfile === app_enums_1.enums.PPermissionClaimProfile.Manual ? null : nextClaim ? nextClaim / 24 : 0,
                    usage: usage,
                    remaining: 0,
                    unpaidLeaveOffset: totalUnpaidOffset,
                };
                periodResult.remaining =
                    existingSettings.ppermissionClaimProfile === app_enums_1.enums.PPermissionClaimProfile.Manual ? null : periodResult.claim + periodResult.fromPreviousPeriod - usage;
                rolloverAmount = periodResult.remaining;
                if (filterRange.overlaps(periodRange) && periodRange.end.isAfter(pacsEnableTimestamp)) {
                    result.userPeriods.find((elem) => elem.userId === employeeInfo.userId).periods.push(periodResult);
                }
                startDate = periodRange.end;
            }
        }
    }
    else {
        for (const employeeInfo of userEmploymentInfo) {
            result.userPeriods.push({ userId: employeeInfo.userId, periods: [] });
            let interval = null;
            if (result.permissionTypeInfo.maxValueBoundaryInterval === app_enums_1.enums.PPermissionMaxValueBoundaryInterval.Annual) {
                interval = "year";
            }
            else if (result.permissionTypeInfo.maxValueBoundaryInterval === app_enums_1.enums.PPermissionMaxValueBoundaryInterval.Monthly) {
                interval = "month";
            }
            if (!employeeInfo.employeeProfileInfo.employmentStartUtc) {
                continue;
            }
            let filterRange = new moment_range_1.DateRange((0, moment_1.default)(employeeInfo.employeeProfileInfo.employmentStartUtc), (0, moment_1.default)());
            let start;
            let end = filterRange.end;
            const permissionForEmployee = ppermissions.find((elem) => elem.userId === employeeInfo.userId);
            const hook = business_hooks_1.armonHookManager.getPPermissionsClaimWindowRangesHook(organizationId);
            if (hook) {
                const ppermissionWindowRanges = await hook(organizationId, employeeInfo.userId, result.permissionTypeInfo.id, filterRange.start);
                switch (result.permissionTypeInfo.maxValueBoundaryInterval) {
                    case app_enums_1.enums.PPermissionMaxValueBoundaryInterval.Annual:
                        start = ppermissionWindowRanges.yearRange.start.clone();
                        break;
                    case app_enums_1.enums.PPermissionMaxValueBoundaryInterval.Monthly:
                        start = ppermissionWindowRanges.monthRange.start.clone();
                        break;
                    default:
                        break;
                }
            }
            else {
                start = filterRange.start.clone().startOf(interval);
                end = filterRange.end.clone().add(1, interval).startOf(interval);
            }
            let periodRange;
            while (start.isBefore(end, "day")) {
                periodRange = new moment_range_1.DateRange(start, start.clone().add(1, interval).endOf(interval)).intersect(filterRange);
                start.add(1, interval);
            }
            let usage = 0;
            if (permissionForEmployee) {
                for (const row of permissionForEmployee.approvedPPermission) {
                    if (row.ppermissionTypeId === result.permissionTypeInfo.id) {
                        let ppermRange = new moment_range_1.DateRange(row.startDateTime, new Date(row.endDateTime));
                        if (ppermRange.overlaps(periodRange)) {
                            let intersection = ppermRange.intersect(periodRange);
                            if (result.permissionTypeInfo.isDailyScheduled) {
                                usage += await (0, business_pacs_ppermission_1.calculateDailyVacationUsage)(organizationId, ppermissionTypeId, intersection.start.toDate(), intersection.end.toDate(), vacations, permissionForEmployee.userId);
                            }
                            else {
                                usage += await (0, business_pacs_ppermission_1.estimateHourlyVacationUsage)(organizationId, ppermissionTypeId, intersection.start.toDate(), intersection.end.toDate(), vacations);
                            }
                        }
                    }
                }
            }
            let row = {
                periodStart: periodRange.start.toDate(),
                periodEnd: start.toDate(),
                fromPreviousPeriod: 0,
                claim: result.permissionTypeInfo.isDailyScheduled ? result.permissionTypeInfo.maxValue : result.permissionTypeInfo.maxValue * 60,
                nextClaim: result.permissionTypeInfo.isDailyScheduled ? result.permissionTypeInfo.maxValue : result.permissionTypeInfo.maxValue * 60,
                usage: usage,
                remaining: 0,
            };
            row.remaining = row.claim - row.usage >= 0 ? row.claim - row.usage : 0;
            if (filterRange.overlaps(periodRange) && periodRange.end.isAfter(pacsEnableTimestamp)) {
                result.userPeriods.find((elem) => elem.userId === employeeInfo.userId).periods.push(row);
            }
        }
    }
    result.userPeriods.forEach((elem) => {
        if (elem.periods.length > 0) {
            elem.periods = [elem.periods[elem.periods.length - 1]];
        }
    });
    result.ppermissionClaimProfile = existingSettings.ppermissionClaimProfile;
    return Promise.resolve(result);
}
exports.getUsersPPermissionsTypeUsageOfBetweenDates = getUsersPPermissionsTypeUsageOfBetweenDates;
async function getPPermissionClaimAndUsage(organizationId, userId, time, ppermissionTypeId) {
    let result = [];
    let timestamp = (0, moment_1.default)(time).startOf("day");
    let existingSettings = await dal_manager_1.dbManager.accessPacs.getOrganizationPACSModuleSettings(organizationId);
    if (!existingSettings || !existingSettings.annualLeavePPermissionTypeId) {
        return Promise.reject("pacs module is not enabled");
    }
    let ppermissionTypes = (await dal_manager_1.dbManager.accessPacs.listPPermissionTypes(organizationId, {
        take: 1000,
        skip: 0,
    })).items;
    if (!ppermissionTypes)
        return Promise.resolve([]);
    if (ppermissionTypeId) {
        let ppermissionType = ppermissionTypes.find((ppt) => ppt.id === ppermissionTypeId);
        if (!ppermissionType) {
            return Promise.resolve([]);
        }
        ppermissionTypes = [ppermissionType];
    }
    let vacations = await dal_manager_1.dbManager.accessPacs.listAllVacations(organizationId);
    const defaultRanges = {
        weekRange: new moment_range_1.DateRange(timestamp.clone().startOf("week"), timestamp.clone().add(1, "week").startOf("week")),
        monthRange: new moment_range_1.DateRange(timestamp.clone().startOf("month"), timestamp.clone().add(1, "month").startOf("month")),
        yearRange: new moment_range_1.DateRange(timestamp.clone().startOf("year"), timestamp.clone().add(1, "year").startOf("year")),
    };
    const hook = business_hooks_1.armonHookManager.getPPermissionsClaimWindowRangesHook(organizationId);
    let ppermissions = await dal_manager_1.dbManager.accessPacs2.getApprovedPPermissionsOfUserAtDate(organizationId, userId, time);
    for (const type of ppermissionTypes) {
        let claim = 0;
        let usage = 0;
        let dueDate = new Date();
        let ppermissionWindowRanges = defaultRanges;
        if (hook) {
            ppermissionWindowRanges = await hook(organizationId, userId, type.id, timestamp);
        }
        let windowRange = ppermissionWindowRanges.weekRange;
        if (type.maxValueBoundaryInterval === app_enums_1.enums.PPermissionMaxValueBoundaryInterval.Annual) {
            windowRange = ppermissionWindowRanges.yearRange;
        }
        else if (type.maxValueBoundaryInterval === app_enums_1.enums.PPermissionMaxValueBoundaryInterval.Monthly) {
            windowRange = ppermissionWindowRanges.monthRange;
        }
        if (type.type === app_enums_1.enums.PredefinedPPermissionType.Annual) {
            let userClaim = await getUsersAnnualVacationStatusAtDate(organizationId, userId, timestamp.toDate());
            claim = userClaim.claim + userClaim.fromPreviousPeriod;
            dueDate = userClaim.periodEnd;
            usage = userClaim.usage;
        }
        else {
            dueDate = windowRange.end.toDate();
            if (type.isDailyScheduled) {
                claim = type.maxValue;
                for (const ppermission of ppermissions) {
                    if (ppermission.ppermissionTypeId === type.id) {
                        let range = new moment_range_1.DateRange(ppermission.startDateTime, new Date(ppermission.endDateTime)).intersect(windowRange);
                        if (range) {
                            usage += await (0, business_pacs_ppermission_1.calculateDailyVacationUsage)(organizationId, ppermission.ppermissionTypeId, range.start.toDate(), range.end.toDate(), vacations, userId, type);
                        }
                    }
                }
            }
            else {
                claim = type.maxValue * 60;
                for (const ppermission of ppermissions) {
                    if (ppermission.ppermissionTypeId === type.id) {
                        let range = new moment_range_1.DateRange(ppermission.startDateTime, new Date(ppermission.endDateTime)).intersect(windowRange);
                        if (range) {
                            usage += await (0, business_pacs_ppermission_1.estimateHourlyVacationUsage)(organizationId, ppermission.ppermissionTypeId, range.start.toDate(), range.end.toDate(), vacations, type);
                        }
                    }
                }
            }
        }
        result.push({
            type: type,
            claim: claim,
            usage: usage,
            left: claim - usage,
            dueDate: dueDate,
        });
    }
    return Promise.resolve(result);
}
exports.getPPermissionClaimAndUsage = getPPermissionClaimAndUsage;
async function getPPermissionClaimAndUsageBulk(organizationId, userIds, time, trx, ppermissionTypeId) {
    const resultMap = new Map();
    const timestamp = (0, moment_1.default)(time).startOf("day");
    const existingSettings = await dal_manager_1.dbManager.accessPacs.getOrganizationPACSModuleSettings(organizationId);
    if (!existingSettings || !existingSettings.annualLeavePPermissionTypeId) {
        throw new Error("pacs module is not enabled");
    }
    let ppermissionTypes = (await dal_manager_1.dbManager.accessPacs.listPPermissionTypes(organizationId, {
        take: 1000,
        skip: 0,
    })).items;
    if (!ppermissionTypes || ppermissionTypes.length === 0)
        return resultMap;
    if (ppermissionTypeId) {
        const filtered = ppermissionTypes.find((ppt) => ppt.id === ppermissionTypeId);
        if (!filtered)
            return resultMap;
        ppermissionTypes = [filtered];
    }
    const vacations = await dal_manager_1.dbManager.accessPacs.listAllVacations(organizationId);
    const defaultRanges = {
        weekRange: new moment_range_1.DateRange(timestamp.clone().startOf("week"), timestamp.clone().add(1, "week").startOf("week")),
        monthRange: new moment_range_1.DateRange(timestamp.clone().startOf("month"), timestamp.clone().add(1, "month").startOf("month")),
        yearRange: new moment_range_1.DateRange(timestamp.clone().startOf("year"), timestamp.clone().add(1, "year").startOf("year")),
    };
    const hook = business_hooks_1.armonHookManager.getPPermissionsClaimWindowRangesBulkHook(organizationId);
    const allPPermissions = await dal_manager_1.dbManager.accessPacs2.getUsersApprovedPPermissionsAtDate(organizationId, userIds, time, trx);
    const ppermissionsByUser = new Map();
    for (const userId of userIds) {
        ppermissionsByUser.set(userId, allPPermissions.filter((p) => p.userId === userId)?.[0]?.approvedPPermission ?? []);
    }
    const userClaims = await getUsersAnnualVacationStatusAtDateBulk(organizationId, userIds, timestamp.toDate());
    let rangesFromBulkHook;
    if (hook) {
        rangesFromBulkHook = await hook(organizationId, userIds, ppermissionTypes.map((t) => t.id), timestamp, trx);
    }
    for (const userId of userIds) {
        let userResult = [];
        try {
            const userPPermissions = ppermissionsByUser.get(userId);
            for (const type of ppermissionTypes) {
                let claim = 0;
                let usage = 0;
                let dueDate = new Date();
                let ppermissionWindowRanges = defaultRanges;
                if (hook) {
                    const rangesFromHook = rangesFromBulkHook?.get(userId)?.get(type.id);
                    if (rangesFromHook) {
                        ppermissionWindowRanges = rangesFromHook;
                    }
                }
                let windowRange = ppermissionWindowRanges.weekRange;
                if (type.maxValueBoundaryInterval === app_enums_1.enums.PPermissionMaxValueBoundaryInterval.Annual) {
                    windowRange = ppermissionWindowRanges.yearRange;
                }
                else if (type.maxValueBoundaryInterval === app_enums_1.enums.PPermissionMaxValueBoundaryInterval.Monthly) {
                    windowRange = ppermissionWindowRanges.monthRange;
                }
                if (type.type === app_enums_1.enums.PredefinedPPermissionType.Annual) {
                    const userClaim = userClaims.get(userId);
                    claim = userClaim.claim + userClaim.fromPreviousPeriod;
                    dueDate = userClaim.periodEnd;
                    usage = userClaim.usage;
                }
                else {
                    dueDate = windowRange.end.toDate();
                    const relevantPermissions = userPPermissions.filter((p) => p.ppermissionTypeId === type.id);
                    if (type.isDailyScheduled) {
                        claim = type.maxValue;
                        for (const ppermission of relevantPermissions) {
                            let range = new moment_range_1.DateRange(ppermission.startDateTime, new Date(ppermission.endDateTime)).intersect(windowRange);
                            if (range) {
                                usage += await (0, business_pacs_ppermission_1.calculateDailyVacationUsage)(organizationId, ppermission.ppermissionTypeId, range.start.toDate(), range.end.toDate(), vacations, userId, type);
                            }
                        }
                    }
                    else {
                        claim = type.maxValue * 60;
                        for (const ppermission of relevantPermissions) {
                            let range = new moment_range_1.DateRange(ppermission.startDateTime, new Date(ppermission.endDateTime)).intersect(windowRange);
                            if (range) {
                                usage += await (0, business_pacs_ppermission_1.estimateHourlyVacationUsage)(organizationId, ppermission.ppermissionTypeId, range.start.toDate(), range.end.toDate(), vacations, type);
                            }
                        }
                    }
                }
                if (userResult) {
                    userResult.push({
                        type: type,
                        claim: claim,
                        usage: usage,
                        left: claim - usage,
                        dueDate: dueDate,
                    });
                }
            }
        }
        catch (error) {
            userResult = null;
        }
        resultMap.set(userId, userResult);
    }
    return resultMap;
}
exports.getPPermissionClaimAndUsageBulk = getPPermissionClaimAndUsageBulk;
async function getUsersAnnualVacationStatusAtDate(organizationId, userId, until) {
    return dal_manager_1.dbManager.armondb.transaction(async (trx) => {
        let result = {
            type: undefined,
            periodStart: new Date(),
            periodEnd: new Date(),
            fromPreviousPeriod: 0,
            claim: 0,
            usage: 0,
            remaining: 0,
            unpaidLeaveOffset: 0,
        };
        let requestedTimestamp = (0, moment_1.default)(until);
        let existingSettings = await dal_manager_1.dbManager.accessPacs.getOrganizationPACSModuleSettings(organizationId, trx);
        if (!existingSettings || !existingSettings.annualLeavePPermissionTypeId) {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.PACS.PACSMODULENOTENABLED");
        }
        result.type = await dal_manager_1.dbManager.accessPacs.getPPermissionType(organizationId, existingSettings.annualLeavePPermissionTypeId);
        if (!result.type) {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.PACS.PERMISSIONTYPENOTFOUND", null, true);
        }
        let pacsModule = (await dal_manager_1.dbManager.accessOrganization.listOrganizationModules(organizationId)).find((m) => m.module === app_enums_1.enums.ArmonApplicationModule.PACS);
        if (!pacsModule) {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.PACS.PACSMODULENOTENABLED");
        }
        else if (!pacsModule.enableUtc) {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.PACS.MODULEENABLETIMEMISSING");
        }
        let pacsEnableTimestamp = (0, moment_1.default)(pacsModule.enableUtc);
        let userEmploymentInfo = await dal_manager_1.dbManager.accessUser.getUserOrganizationProfileEmploymentInfo(organizationId, userId, trx);
        if (!userEmploymentInfo.employmentStartUtc) {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.PACS.EMPLOYEESTARTDATEMISSING", null, true);
        }
        let employmentStartTimestamp = (0, moment_1.default)(userEmploymentInfo.employmentStartUtc);
        if (employmentStartTimestamp.isAfter(requestedTimestamp)) {
            return result;
        }
        let vacations = await dal_manager_1.dbManager.accessPacs.listAllVacations(organizationId);
        let ppermissions = await dal_manager_1.dbManager.accessPacs2.getApprovedPPermissionsOfUserAtDate(organizationId, userId, new Date());
        let updateIntervals = await dal_manager_1.dbManager.accessOrganization.getOrganizationAnnualPPermissionUpdateIntervals(organizationId, trx);
        if (!updateIntervals || updateIntervals.length < 1) {
            updateIntervals = (0, business_pacs_ppermission_1.getDefaultAnnualPPermissionIntervals)();
        }
        let unpaidPPermissions = await dal_manager_1.dbManager.accessPacs.listPastUnpaidPPermissionsOfUser(organizationId, userId, requestedTimestamp.toDate(), trx);
        let currentWorkingTime = 0;
        if (userEmploymentInfo.previousServiceDuration) {
            currentWorkingTime += userEmploymentInfo.previousServiceDuration;
        }
        let rolloverAmount = 0;
        if (employmentStartTimestamp.isSameOrBefore(pacsEnableTimestamp)) {
            rolloverAmount = userEmploymentInfo?.pacsEnabledRemainedAnnualPPermission ? userEmploymentInfo.pacsEnabledRemainedAnnualPPermission : 0;
        }
        else {
            rolloverAmount = userEmploymentInfo?.previousAnnualVacationRight ? userEmploymentInfo.previousAnnualVacationRight : 0;
        }
        rolloverAmount = rolloverAmount / 24;
        for (let startDate = (0, moment_1.default)(userEmploymentInfo.employmentStartUtc); startDate.isBefore(requestedTimestamp);) {
            let periodRange = new moment_range_1.DateRange(startDate, startDate.clone().add(1, "year"));
            let totalUnpaidOffset = 0;
            if (unpaidPPermissions) {
                for (const ppermission of unpaidPPermissions) {
                    if (periodRange.contains(ppermission.startDateTime)) {
                        let duration = new moment_range_1.DateRange(ppermission.startDateTime, ppermission.endDateTime).duration("days");
                        totalUnpaidOffset += duration;
                        periodRange.end = periodRange.end.add(duration, "days");
                    }
                }
            }
            let targetInterval = updateIntervals.find((i) => currentWorkingTime >= i.lowerExperienceDuration && currentWorkingTime <= i.higherExperienceDuration);
            currentWorkingTime += 365;
            let claim = 0;
            if (targetInterval)
                claim = targetInterval.claim;
            if (existingSettings.annualPPermissionAgeLimits && userEmploymentInfo.birthDateUtc && existingSettings.ppermissionClaimProfile === app_enums_1.enums.PPermissionClaimProfile.Auto) {
                let userAge = Math.abs((0, moment_1.default)(userEmploymentInfo.birthDateUtc).diff(periodRange.start, "year"));
                if (existingSettings.annualPPermissionAgeLimits.youngerMinimumAmount &&
                    existingSettings.annualPPermissionAgeLimits.youngerThanLimitYears &&
                    userAge < existingSettings.annualPPermissionAgeLimits.youngerThanLimitYears) {
                    claim = claim < existingSettings.annualPPermissionAgeLimits.youngerMinimumAmount ? existingSettings.annualPPermissionAgeLimits.youngerMinimumAmount : claim;
                }
                else if (existingSettings.annualPPermissionAgeLimits.olderMinimumAmount &&
                    existingSettings.annualPPermissionAgeLimits.olderThanLimitYears &&
                    userAge > existingSettings.annualPPermissionAgeLimits.olderThanLimitYears) {
                    claim = claim < existingSettings.annualPPermissionAgeLimits.olderMinimumAmount ? existingSettings.annualPPermissionAgeLimits.olderMinimumAmount : claim;
                }
            }
            let summationRange = periodRange.clone();
            if (summationRange.end.isBefore(pacsEnableTimestamp)) {
                startDate = periodRange.end;
                continue;
            }
            else if (summationRange.start.isBefore(pacsEnableTimestamp)) {
                summationRange.start = (0, moment_1.default)(pacsEnableTimestamp);
            }
            if (summationRange.end.isAfter(requestedTimestamp)) {
                summationRange.end = requestedTimestamp.clone();
            }
            let usage = 0;
            for (const row of ppermissions) {
                if (row.ppermissionTypeId === result.type.id) {
                    let ppermRange = new moment_range_1.DateRange(row.startDateTime, new Date(row.endDateTime));
                    if (ppermRange.overlaps(summationRange)) {
                        let intersection = ppermRange.intersect(summationRange);
                        usage += await (0, business_pacs_ppermission_1.calculateDailyVacationUsage)(organizationId, existingSettings.annualLeavePPermissionTypeId, intersection.start.toDate(), intersection.end.toDate(), vacations, userId, result.type);
                    }
                }
            }
            claim = periodRange.contains(pacsEnableTimestamp) ? 0 : claim / 24;
            if (periodRange.contains(requestedTimestamp)) {
                result.periodStart = periodRange.start.toDate();
                result.periodEnd = periodRange.end.toDate();
                result.claim = existingSettings.ppermissionClaimProfile === app_enums_1.enums.PPermissionClaimProfile.Manual ? null : claim;
                result.fromPreviousPeriod = existingSettings.ppermissionClaimProfile === app_enums_1.enums.PPermissionClaimProfile.Manual ? null : rolloverAmount;
                result.usage = usage;
                result.remaining = existingSettings.ppermissionClaimProfile === app_enums_1.enums.PPermissionClaimProfile.Manual ? null : claim + rolloverAmount - usage;
                result.unpaidLeaveOffset = totalUnpaidOffset;
                return Promise.resolve(result);
            }
            else {
                rolloverAmount = claim + rolloverAmount - usage;
                startDate = periodRange.end;
            }
        }
    });
}
exports.getUsersAnnualVacationStatusAtDate = getUsersAnnualVacationStatusAtDate;
async function getUsersAnnualVacationStatusAtDateBulk(organizationId, userIds, until) {
    return dal_manager_1.dbManager.armondb.transaction(async (trx) => {
        const resultMap = new Map();
        const requestedTimestamp = (0, moment_1.default)(until);
        const [existingSettings, pacsModule, vacations, updateIntervalsRaw, employmentInfos] = await Promise.all([
            dal_manager_1.dbManager.accessPacs.getOrganizationPACSModuleSettings(organizationId, trx),
            dal_manager_1.dbManager.accessOrganization.listOrganizationModules(organizationId),
            dal_manager_1.dbManager.accessPacs.listAllVacations(organizationId),
            dal_manager_1.dbManager.accessOrganization.getOrganizationAnnualPPermissionUpdateIntervals(organizationId, trx),
            dal_manager_1.dbManager.accessUser.getEmploymentsUserOrganizationProfileInfos(organizationId, userIds),
        ]);
        const [ppermissions, unpaidPPermissions] = await dal_manager_1.dbManager.systemTransaction(async (trx) => {
            const [approved, unpaid] = await Promise.all([
                dal_manager_1.dbManager.accessPacs2.getUsersApprovedPPermissionsAtDate(organizationId, userIds, until, trx),
                dal_manager_1.dbManager.accessPacs.listUsersPastUnpaidPPermissions(organizationId, userIds, until, trx),
            ]);
            return [approved, unpaid];
        });
        const unpaidPPermissionsMap = new Map(unpaidPPermissions.map((u) => [u.userId, u.ppermissionDateInfo]));
        const ppermissionsMap = new Map(ppermissions.map((u) => [u.userId, u.approvedPPermission]));
        if (!existingSettings || !existingSettings.annualLeavePPermissionTypeId) {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.PACS.PACSMODULENOTENABLED");
        }
        const permissionType = await dal_manager_1.dbManager.accessPacs.getPPermissionType(organizationId, existingSettings.annualLeavePPermissionTypeId);
        if (!permissionType) {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.PACS.PERMISSIONTYPENOTFOUND", null, true);
        }
        const pacsInfo = pacsModule.find((m) => m.module === app_enums_1.enums.ArmonApplicationModule.PACS);
        if (!pacsInfo?.enableUtc) {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.PACS.MODULEENABLETIMEMISSING");
        }
        const pacsEnableTimestamp = (0, moment_1.default)(pacsInfo.enableUtc);
        const updateIntervals = updateIntervalsRaw?.length > 0 ? updateIntervalsRaw : (0, business_pacs_ppermission_1.getDefaultAnnualPPermissionIntervals)();
        for (const userId of userIds) {
            const result = {
                type: permissionType,
                periodStart: new Date(),
                periodEnd: new Date(),
                fromPreviousPeriod: 0,
                claim: 0,
                usage: 0,
                remaining: 0,
                unpaidLeaveOffset: 0,
            };
            const userEmploymentInfo = employmentInfos.find((e) => e.userId === userId)?.employeeProfileInfo;
            if (!userEmploymentInfo?.employmentStartUtc)
                continue;
            const employmentStartTimestamp = (0, moment_1.default)(userEmploymentInfo.employmentStartUtc);
            if (employmentStartTimestamp.isAfter(requestedTimestamp)) {
                resultMap.set(userId, result);
                continue;
            }
            const ppermissions = ppermissionsMap.get(userId) ?? [];
            const unpaidPPermissions = unpaidPPermissionsMap.get(userId) ?? [];
            let currentWorkingTime = userEmploymentInfo.previousServiceDuration || 0;
            let rolloverAmount = 0;
            if (employmentStartTimestamp.isSameOrBefore(pacsEnableTimestamp)) {
                rolloverAmount = userEmploymentInfo?.pacsEnabledRemainedAnnualPPermission || 0;
            }
            else {
                rolloverAmount = userEmploymentInfo?.previousAnnualVacationRight || 0;
            }
            rolloverAmount /= 24;
            for (let startDate = (0, moment_1.default)(userEmploymentInfo.employmentStartUtc); startDate.isBefore(requestedTimestamp);) {
                let periodRange = new moment_range_1.DateRange(startDate, startDate.clone().add(1, "year"));
                let totalUnpaidOffset = 0;
                for (const unpaid of unpaidPPermissions) {
                    if (periodRange.contains(unpaid.startDateTime)) {
                        const unpaidDuration = new moment_range_1.DateRange(unpaid.startDateTime, unpaid.endDateTime).duration("days");
                        totalUnpaidOffset += unpaidDuration;
                        periodRange.end = periodRange.end.add(unpaidDuration, "days");
                    }
                }
                const targetInterval = updateIntervals.find((i) => currentWorkingTime >= i.lowerExperienceDuration && currentWorkingTime <= i.higherExperienceDuration);
                currentWorkingTime += 365;
                let claim = targetInterval?.claim || 0;
                if (existingSettings.annualPPermissionAgeLimits && userEmploymentInfo.birthDateUtc && existingSettings.ppermissionClaimProfile === app_enums_1.enums.PPermissionClaimProfile.Auto) {
                    const userAge = Math.abs((0, moment_1.default)(userEmploymentInfo.birthDateUtc).diff(periodRange.start, "year"));
                    const ageLimits = existingSettings.annualPPermissionAgeLimits;
                    if (ageLimits.youngerThanLimitYears && userAge < ageLimits.youngerThanLimitYears) {
                        claim = Math.max(claim, ageLimits.youngerMinimumAmount || 0);
                    }
                    else if (ageLimits.olderThanLimitYears && userAge > ageLimits.olderThanLimitYears) {
                        claim = Math.max(claim, ageLimits.olderMinimumAmount || 0);
                    }
                }
                let summationRange = periodRange.clone();
                if (summationRange.end.isBefore(pacsEnableTimestamp)) {
                    startDate = periodRange.end;
                    continue;
                }
                else if (summationRange.start.isBefore(pacsEnableTimestamp)) {
                    summationRange.start = (0, moment_1.default)(pacsEnableTimestamp);
                }
                if (summationRange.end.isAfter(requestedTimestamp)) {
                    summationRange.end = requestedTimestamp.clone();
                }
                let usage = 0;
                for (const row of ppermissions) {
                    if (row.ppermissionTypeId === permissionType.id) {
                        let ppermRange = new moment_range_1.DateRange(row.startDateTime, new Date(row.endDateTime));
                        if (ppermRange.overlaps(summationRange)) {
                            let intersection = ppermRange.intersect(summationRange);
                            usage += await (0, business_pacs_ppermission_1.calculateDailyVacationUsage)(organizationId, permissionType.id, intersection.start.toDate(), intersection.end.toDate(), vacations, userId, permissionType);
                        }
                    }
                }
                claim = periodRange.contains(pacsEnableTimestamp) ? 0 : claim / 24;
                if (periodRange.contains(requestedTimestamp)) {
                    result.periodStart = periodRange.start.toDate();
                    result.periodEnd = periodRange.end.toDate();
                    result.claim = existingSettings.ppermissionClaimProfile === app_enums_1.enums.PPermissionClaimProfile.Manual ? null : claim;
                    result.fromPreviousPeriod = existingSettings.ppermissionClaimProfile === app_enums_1.enums.PPermissionClaimProfile.Manual ? null : rolloverAmount;
                    result.usage = usage;
                    result.remaining = existingSettings.ppermissionClaimProfile === app_enums_1.enums.PPermissionClaimProfile.Manual ? null : claim + rolloverAmount - usage;
                    result.unpaidLeaveOffset = totalUnpaidOffset;
                    break;
                }
                else {
                    rolloverAmount = claim + rolloverAmount - usage;
                    startDate = periodRange.end;
                }
            }
            resultMap.set(userId, result);
        }
        return resultMap;
    });
}
exports.getUsersAnnualVacationStatusAtDateBulk = getUsersAnnualVacationStatusAtDateBulk;
