"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.sendSms = exports.sendPushNotification = exports.sendEmail = exports.sendWeb = exports.sendNotification = void 0;
const crypto_1 = __importDefault(require("crypto"));
const hcm_push_kit_1 = __importDefault(require("hcm-push-kit"));
const https_1 = __importDefault(require("https"));
const xml2js_1 = require("xml2js");
const app_config_1 = require("../../app.config");
const app_constants_1 = require("../../app.constants");
const app_enums_1 = require("../../app.enums");
const app_logs_1 = require("../../app.logs");
const app_util_1 = require("../../app.util");
const dal_memcache_1 = require("../../dal/access/dal.memcache");
const dal_constants_1 = require("../../dal/dal.constants");
const dal_manager_1 = require("../../dal/dal.manager");
const messageBroker_server_to_app_pub_1 = require("../../messageBroker/messageBroker.server-to-app.pub");
const messageBroker_server_to_app_sub_1 = require("../../messageBroker/messageBroker.server-to-app.sub");
const business_hooks_1 = require("../business.hooks");
const firebase_1 = require("./firebase");
async function sendNotification(params) {
    const sendNotificationOverrideHook = business_hooks_1.armonHookManager.getSendNotificationOverride(params.organizationId);
    if (sendNotificationOverrideHook) {
        try {
            const sendNotificationHookResult = await sendNotificationOverrideHook(params);
            if (sendNotificationHookResult.isImplemented) {
                return;
            }
        }
        catch (error) {
            app_logs_1.logger.error(`Error while running notification send hook override for notification instance with id [${params.notification.instanceId}]`);
            app_logs_1.logger.error(error);
            await dal_manager_1.dbManager.accessNotifications.setNotificationInstanceState({
                instanceId: params.notification.instanceId,
                organizationId: params.organizationId,
                state: 3,
                actionNote: {
                    errorCode: app_enums_1.enums.ErrorCode.CustomNotificationChannelError,
                    message: error,
                },
                trx: params.trx,
            });
            return;
        }
    }
    switch (params.medium) {
        case app_enums_1.enums.NotificationMedium.Web:
            return sendWeb(params);
        case app_enums_1.enums.NotificationMedium.Email:
            return sendEmail(params);
        case app_enums_1.enums.NotificationMedium.PushNotification:
            return sendPushNotification(params);
        case app_enums_1.enums.NotificationMedium.SMS:
            return sendSms(params);
        default:
            break;
    }
}
exports.sendNotification = sendNotification;
async function sendWeb(params) {
    try {
        const unreadNotificationCount = await dal_manager_1.dbManager.accessNotifications.getUnreadNotificationsCount({
            organizationId: params.organizationId,
            trx: params.trx,
            userId: params.notification.receiverUserId,
        });
        const content = params.notification.content;
        messageBroker_server_to_app_pub_1.amqpServerToAppPub.sendToExchange(params.organizationId + ".*." + params.notification.receiverUserId, {
            e: messageBroker_server_to_app_sub_1.amqpServerToAppSubUserEventNames.notification,
            p: {
                id: params.notification.instanceId,
                title: content.title,
                type: params.notification.type,
                unreadCount: unreadNotificationCount + 1,
            },
        });
        await dal_manager_1.dbManager.accessNotifications.setNotificationInstanceState({
            instanceId: params.notification.instanceId,
            organizationId: params.organizationId,
            state: 4,
            trx: params.trx,
        });
    }
    catch (error) {
        await dal_manager_1.dbManager.accessNotifications.setNotificationInstanceState({
            instanceId: params.notification.instanceId,
            organizationId: params.organizationId,
            state: 3,
            trx: params.trx,
            actionNote: {
                errorCode: app_enums_1.enums.ErrorCode.Unknown,
                message: error?.message ?? error,
            },
        });
        throw new Error(`o:[${params.organizationId}],niid[${params.notification.instanceId}], errorCode[${app_enums_1.enums.ErrorCode.Unknown}], web Failed! error: ${error}`);
    }
}
exports.sendWeb = sendWeb;
async function sendEmail(params) {
    try {
        if (!params.notification.content.to) {
            await dal_manager_1.dbManager.accessNotifications.setNotificationInstanceState({
                instanceId: params.notification.instanceId,
                organizationId: params.organizationId,
                state: 3,
                actionNote: {
                    errorCode: app_enums_1.enums.ErrorCode.UserhasNoEmailAddress,
                },
                trx: params.trx,
            });
            throw new Error(`o:[${params.organizationId}],niid[${params.notification.instanceId}],errorCode[${app_enums_1.enums.ErrorCode.UserhasNoEmailAddress}] Email Failed!`);
        }
        const smtpSettings = (await (0, dal_memcache_1.getCacheOrganizationSettings)(params.organizationId, params.trx)).notification.smtpSettings;
        if (!smtpSettings?.config) {
            await dal_manager_1.dbManager.accessNotifications.setNotificationInstanceState({
                instanceId: params.notification.instanceId,
                organizationId: params.organizationId,
                state: 3,
                actionNote: {
                    errorCode: app_enums_1.enums.ErrorCode.NoOrganizaitionSmtpSettings,
                },
                trx: params.trx,
            });
            (0, dal_memcache_1.invalidateCacheMailTransporter)(params.organizationId);
            throw new Error(`o:[${params.organizationId}],niid[${params.notification.instanceId}],errorCode[${app_enums_1.enums.ErrorCode.NoOrganizaitionSmtpSettings}] Email Failed!`);
        }
        const transporter = await (0, dal_memcache_1.getCacheMailTransporter)(params.organizationId, smtpSettings);
        const content = params.notification.content;
        const mailOptions = {
            ...content,
            from: smtpSettings.defaultFromAddress || smtpSettings.config.auth?.user,
        };
        await transporter.sendMail(mailOptions);
        await dal_manager_1.dbManager.accessNotifications.setNotificationInstanceState({
            instanceId: params.notification.instanceId,
            organizationId: params.organizationId,
            state: 4,
            trx: params.trx,
        });
    }
    catch (error) {
        if (!error.message?.includes("errorCode")) {
            await dal_manager_1.dbManager.accessNotifications.setNotificationInstanceState({
                instanceId: params.notification.instanceId,
                organizationId: params.organizationId,
                state: 3,
                actionNote: {
                    errorCode: app_enums_1.enums.ErrorCode.Unknown,
                    message: error?.message || error,
                },
                trx: params.trx,
            });
            throw new Error(`o:[${params.organizationId}],niid[${params.notification.instanceId}],errorCode[${app_enums_1.enums.ErrorCode.Unknown}] email Failed! error: ${error}`);
        }
        throw new Error(error?.message || error);
    }
}
exports.sendEmail = sendEmail;
async function sendPushNotification(params) {
    const { token, tokenType } = params.notification.content;
    if (!token || !tokenType) {
        await dal_manager_1.dbManager.accessNotifications.setNotificationInstanceState({
            instanceId: params.notification.instanceId,
            organizationId: params.organizationId,
            state: 3,
            actionNote: {
                errorCode: app_enums_1.enums.ErrorCode.UserHasNoNotificationToken,
            },
            trx: params.trx,
        });
        throw new Error(`o:[${params.organizationId}],niid[${params.notification.instanceId}],errorCode[${app_enums_1.enums.ErrorCode.UserHasNoNotificationToken}] pushNotification Failed!`);
    }
    switch (tokenType) {
        case dal_constants_1.DalConstants.NotificationTokenType.FireBase:
            await firebasePush({
                instanceId: params.notification.instanceId,
                organizationId: params.organizationId,
                content: params.notification.content,
                type: params.notification.type,
                notificationToken: token,
                userId: params.notification.receiverUserId,
                trx: params.trx,
            });
            break;
        case dal_constants_1.DalConstants.NotificationTokenType.HuaweiPush:
            await huaweiPush({
                instanceId: params.notification.instanceId,
                organizationId: params.organizationId,
                content: params.notification.content,
                type: params.notification.type,
                notificationToken: token,
                userId: params.notification.receiverUserId,
                trx: params.trx,
            });
            break;
        default:
            await dal_manager_1.dbManager.accessNotifications.setNotificationInstanceState({
                instanceId: params.notification.instanceId,
                organizationId: params.organizationId,
                state: 3,
                actionNote: {
                    errorCode: app_enums_1.enums.ErrorCode.InvalidPushNotificationMedium,
                    message: "Invalid Notification Token Type!",
                },
                trx: params.trx,
            });
            throw new Error(`o:[${params.organizationId}],niid[${params.notification.instanceId}],errorCode[${app_enums_1.enums.ErrorCode.InvalidPushNotificationMedium}] pushNotification Failed! Invalid Notification Token Type!`);
    }
}
exports.sendPushNotification = sendPushNotification;
async function huaweiPush(params) {
    if (!app_config_1.appConfig.huaweiPushSettings) {
        await dal_manager_1.dbManager.accessNotifications.setNotificationInstanceState({
            instanceId: params.instanceId,
            organizationId: params.organizationId,
            state: 3,
            actionNote: {
                errorCode: app_enums_1.enums.ErrorCode.NoHuaweiPushSettings,
            },
            trx: params.trx,
        });
        throw new Error(`o:[${params.organizationId}],niid[${params.instanceId}],errorCode[${app_enums_1.enums.ErrorCode.NoHuaweiPushSettings}] pushNotification Failed!`);
    }
    hcm_push_kit_1.default.init(app_config_1.appConfig.huaweiPushSettings);
    const mc = hcm_push_kit_1.default.messaging().messaging;
    const message = {
        notification: {
            title: params.content.title,
            body: params.content.body,
        },
        data: JSON.stringify({
            v: "V1",
            data: JSON.stringify(params.content.data),
            organizationId: params.organizationId,
            type: params.type,
            userId: params.userId,
            id: params.instanceId,
        }),
        android: {
            notification: {
                click_action: {
                    type: 1,
                    intent: "com.armongate.mobilemanager.Activity",
                },
                foreground_show: false,
            },
        },
        token: [params.notificationToken],
    };
    try {
        await mc.send(message);
        await dal_manager_1.dbManager.accessNotifications.setNotificationInstanceState({
            instanceId: params.instanceId,
            organizationId: params.organizationId,
            state: 4,
            trx: params.trx,
        });
    }
    catch (error) {
        switch (error) {
            case app_constants_1.pushNotificationMissingRegistrationError:
                await dal_manager_1.dbManager.accessNotifications.setNotificationInstanceState({
                    instanceId: params.instanceId,
                    organizationId: params.organizationId,
                    state: 3,
                    actionNote: {
                        errorCode: app_enums_1.enums.ErrorCode.MissingReceiver,
                    },
                    trx: params.trx,
                });
                break;
            case app_constants_1.pushNotificationInvalidRegistrationError:
                await dal_manager_1.dbManager.accessNotifications.setNotificationInstanceState({
                    instanceId: params.instanceId,
                    organizationId: params.organizationId,
                    state: 3,
                    actionNote: {
                        errorCode: app_enums_1.enums.ErrorCode.InvalidReceiver,
                    },
                    trx: params.trx,
                });
                break;
            case app_constants_1.pushNotificationInvalidServerResponseError:
                await dal_manager_1.dbManager.accessNotifications.setNotificationInstanceState({
                    instanceId: params.instanceId,
                    organizationId: params.organizationId,
                    state: 3,
                    actionNote: {
                        errorCode: app_enums_1.enums.ErrorCode.PushNotificationServerFailed,
                    },
                    trx: params.trx,
                });
                break;
            default:
                await dal_manager_1.dbManager.accessNotifications.setNotificationInstanceState({
                    instanceId: params.instanceId,
                    organizationId: params.organizationId,
                    state: 3,
                    actionNote: {
                        errorCode: app_enums_1.enums.ErrorCode.Unknown,
                        message: error?.message || error,
                    },
                    trx: params.trx,
                });
                break;
        }
        throw new Error(`o:[${params.organizationId}],niid[${params.instanceId}] pushNotification Failed! error: ${error}`);
    }
}
async function firebasePush(params) {
    if (!app_config_1.appConfig.firebasePrivateKey) {
        await dal_manager_1.dbManager.accessNotifications.setNotificationInstanceState({
            instanceId: params.instanceId,
            organizationId: params.organizationId,
            state: 3,
            actionNote: {
                errorCode: app_enums_1.enums.ErrorCode.NoFirebasePrivateKey,
            },
            trx: params.trx,
        });
        throw new Error(`o:[${params.organizationId}],niid[${params.instanceId}],errorCode[${app_enums_1.enums.ErrorCode.NoFirebasePrivateKey}] pushNotification Failed!`);
    }
    const data = params.content.data;
    const pushNotificationData = {
        notificationToken: params.notificationToken,
        data: {
            v: "V1",
            data: JSON.stringify(data),
            organizationId: params.organizationId,
            type: params.type.toString(),
            userId: params.userId,
            id: params.instanceId,
        },
        notification: {
            title: params.content.title,
            body: params.content.body,
        },
    };
    try {
        await (0, firebase_1.firebasePush)(pushNotificationData);
        await dal_manager_1.dbManager.accessNotifications.setNotificationInstanceState({
            instanceId: params.instanceId,
            organizationId: params.organizationId,
            state: 4,
            trx: params.trx,
        });
    }
    catch (error) {
        let errorCode;
        switch (error) {
            case firebase_1.firebaseMessagingErrorCodes.MissingRecevier:
                errorCode = app_enums_1.enums.ErrorCode.MissingReceiver;
                await dal_manager_1.dbManager.accessNotifications.setNotificationInstanceState({
                    instanceId: params.instanceId,
                    organizationId: params.organizationId,
                    state: 3,
                    actionNote: { errorCode },
                    trx: params.trx,
                });
                break;
            case firebase_1.firebaseMessagingErrorCodes.InvalidReceiver:
                errorCode = app_enums_1.enums.ErrorCode.InvalidReceiver;
                app_logs_1.logger.warn("The [object String] warning above is coming from 3rd party library. The warning is handled. You can ignore it.");
                await dal_manager_1.dbManager.accessNotifications.setNotificationInstanceState({
                    instanceId: params.instanceId,
                    organizationId: params.organizationId,
                    state: 3,
                    actionNote: { errorCode },
                    trx: params.trx,
                });
                break;
            case firebase_1.firebaseMessagingErrorCodes.ServerUnavailable:
                errorCode = app_enums_1.enums.ErrorCode.PushNotificationServerFailed;
                await dal_manager_1.dbManager.accessNotifications.setNotificationInstanceState({
                    instanceId: params.instanceId,
                    organizationId: params.organizationId,
                    state: 3,
                    actionNote: { errorCode },
                    trx: params.trx,
                });
                break;
            default:
                errorCode = app_enums_1.enums.ErrorCode.Unknown;
                await dal_manager_1.dbManager.accessNotifications.setNotificationInstanceState({
                    instanceId: params.instanceId,
                    organizationId: params.organizationId,
                    state: 3,
                    actionNote: {
                        errorCode: app_enums_1.enums.ErrorCode.Unknown,
                        message: error?.message || error,
                    },
                    trx: params.trx,
                });
                break;
        }
        throw new Error(`o:[${params.organizationId}],niid[${params.instanceId}] errorCode:[${errorCode}] pushNotification Failed! error: ${error}`);
    }
}
async function sendSms(params) {
    try {
        const smsSettings = (await (0, dal_memcache_1.getCacheOrganizationSettings)(params.organizationId, params.trx)).notification.smsSettings;
        if (smsSettings.type === app_enums_1.enums.SmsConfigType.Armon) {
            const balance = await dal_manager_1.dbManager.accessNotifications.getSmsBalance({ organizationId: params.organizationId, trx: params.trx });
            if (balance < 1) {
                await dal_manager_1.dbManager.accessNotifications.setNotificationInstanceState({
                    instanceId: params.notification.instanceId,
                    organizationId: params.organizationId,
                    state: 8,
                    actionNote: {
                        errorCode: app_enums_1.enums.ErrorCode.InsufficientSmsBalance,
                    },
                    trx: params.trx,
                });
                throw new Error(`o:[${params.organizationId}],niid[${params.notification.instanceId}],errorCode[${app_enums_1.enums.ErrorCode.InsufficientSmsBalance}] SMS Failed!`);
            }
        }
        const phoneNumber = params.notification.content.phoneNumber;
        if (!phoneNumber) {
            await dal_manager_1.dbManager.accessNotifications.setNotificationInstanceState({
                instanceId: params.notification.instanceId,
                organizationId: params.organizationId,
                state: 3,
                actionNote: {
                    errorCode: app_enums_1.enums.ErrorCode.UserHasNoPhoneNumber,
                },
                trx: params.trx,
            });
            throw new Error(`o:[${params.organizationId}],niid[${params.notification.instanceId}],errorCode[${app_enums_1.enums.ErrorCode.UserHasNoPhoneNumber}] SMS Failed!`);
        }
        if (!smsSettings) {
            await dal_manager_1.dbManager.accessNotifications.setNotificationInstanceState({
                instanceId: params.notification.instanceId,
                organizationId: params.organizationId,
                state: 3,
                actionNote: {
                    errorCode: app_enums_1.enums.ErrorCode.NoOrganizaitionSmsSettings,
                },
                trx: params.trx,
            });
            throw new Error(`o:[${params.organizationId}],niid[${params.notification.instanceId}],errorCode[${app_enums_1.enums.ErrorCode.NoOrganizaitionSmsSettings}] SMS Failed!`);
        }
        switch (smsSettings.type) {
            case app_enums_1.enums.SmsConfigType.IletiMerkezi:
                await sendByIletiMerkezi(smsSettings, params, phoneNumber, false);
                break;
            case app_enums_1.enums.SmsConfigType.NetGsm:
                await sendByNetGsm(smsSettings, params, phoneNumber);
                break;
            case app_enums_1.enums.SmsConfigType.MesajUssu:
                await sendByMesajUssu(smsSettings, params, phoneNumber);
                break;
            case app_enums_1.enums.SmsConfigType.TurkTelekom:
                await sendByTurkTelekom(smsSettings, params, phoneNumber);
                break;
            case app_enums_1.enums.SmsConfigType.Balaban:
                await sendByBalaban(smsSettings, params, phoneNumber);
                break;
            case app_enums_1.enums.SmsConfigType.Armon:
                await sendByIletiMerkezi(smsSettings, params, phoneNumber, true);
                break;
            default:
                throw new Error(`Sms Provider Settings[${smsSettings.type}] Does Not Implemented`);
        }
    }
    catch (error) {
        app_logs_1.logger.warn(`OrganizationId:[${params.organizationId}],notificationInstanceId[${params.notification.instanceId}],  response[${error.message || error}] SMS Failed!`);
        await dal_manager_1.dbManager.accessNotifications.setNotificationInstanceState({
            instanceId: params.notification.instanceId,
            organizationId: params.organizationId,
            state: 3,
            actionNote: {
                errorCode: app_enums_1.enums.ErrorCode.Unknown,
                message: error.message || error,
            },
            trx: params.trx,
        });
    }
}
exports.sendSms = sendSms;
async function sendByIletiMerkezi(smsSettings, params, phoneNumber, isArmon) {
    const config = smsSettings.config;
    const hmac = crypto_1.default.createHmac("sha256", config.privateKey).update(config.apiKey).digest("hex");
    const content = params.notification.content;
    const body = {
        request: {
            authentication: {
                key: config.apiKey,
                hash: hmac,
            },
            order: {
                sender: config.header.toUpperCase(),
                sendDateTime: [],
                iys: "0",
                message: {
                    text: (0, app_util_1.convertToEnglishCharString)(content.text),
                    receipents: {
                        number: [phoneNumber],
                    },
                },
            },
        },
    };
    const options = {
        hostname: "api.iletimerkezi.com",
        port: 443,
        path: "/v1/send-sms/json",
        method: "POST",
        headers: {
            "Content-Type": "application/json",
            "Content-Length": JSON.stringify(body).length,
        },
    };
    return new Promise((resolve, reject) => {
        try {
            const req = https_1.default.request(options, async (res) => {
                if (res.statusCode && res.statusCode >= 200 && res?.statusCode < 300) {
                    if (isArmon) {
                        await updateSmsBalance(params.organizationId, content.text.length, phoneNumber, smsSettings, params.trx);
                    }
                    await dal_manager_1.dbManager.accessNotifications.setNotificationInstanceState({
                        instanceId: params.notification.instanceId,
                        organizationId: params.organizationId,
                        state: 4,
                        trx: params.trx,
                    });
                    resolve(res.statusCode);
                }
                else {
                    await dal_manager_1.dbManager.accessNotifications.setNotificationInstanceState({
                        instanceId: params.notification.instanceId,
                        organizationId: params.organizationId,
                        state: 3,
                        actionNote: {
                            errorCode: app_enums_1.enums.ErrorCode.Unknown,
                            message: `OrganizationId:[${params.organizationId}],notificationInstanceId[${params.notification.instanceId}],errorCode[10${res.statusCode}] SMS Failed! Error: ${res.statusCode}`,
                        },
                        trx: params.trx,
                    });
                    reject(`OrganizationId:[${params.organizationId}],notificationInstanceId[${params.notification.instanceId}],errorCode[10${res.statusCode}] SMS Failed! Error: ${res.statusCode}`);
                }
            });
            req.on("error", (error) => {
                app_logs_1.logger.error("An error occured during SMS send request" + error);
            });
            req.write(JSON.stringify(body));
            req.end();
        }
        catch (error) {
            reject(error);
        }
    });
}
async function sendByNetGsm(smsSettings, params, phoneNumber) {
    const config = smsSettings.config;
    const content = params.notification.content;
    const text = (0, app_util_1.convertToEnglishCharString)(content.text);
    const url = new URL("https://api.netgsm.com.tr/sms/rest/v2/send");
    url.searchParams.append("usercode", config.usercode);
    url.searchParams.append("password", config.password);
    url.searchParams.append("gsmno", phoneNumber);
    url.searchParams.append("message", text);
    url.searchParams.append("msgheader", config.header);
    const options = {
        hostname: url.hostname,
        path: `${url.pathname}?${url.searchParams.toString()}`,
        method: "GET",
    };
    return new Promise((resolve, reject) => {
        try {
            const req = https_1.default.request(options, async (res) => {
                res.setEncoding("utf8");
                res.resume();
                let data = "";
                res.on("data", (chunk) => {
                    data += chunk;
                });
                res.on("end", async () => {
                    if (data.startsWith("00")) {
                        app_logs_1.logger.debug("sms sended data: " + data);
                        await dal_manager_1.dbManager.accessNotifications.setNotificationInstanceState({
                            instanceId: params.notification.instanceId,
                            organizationId: params.organizationId,
                            state: 4,
                            trx: params.trx,
                        });
                        resolve();
                    }
                    else {
                        await dal_manager_1.dbManager.accessNotifications.setNotificationInstanceState({
                            instanceId: params.notification.instanceId,
                            organizationId: params.organizationId,
                            state: 3,
                            actionNote: {
                                errorCode: app_enums_1.enums.ErrorCode.Unknown,
                                message: `OrganizationId:[${params.organizationId}],notificationInstanceId[${params.notification.instanceId}], response[${data}] netgsm SMS Failed!`,
                            },
                            trx: params.trx,
                        });
                        reject(`OrganizationId:[${params.organizationId}],notificationInstanceId[${params.notification.instanceId}],  response[${data}] netgsm SMS Failed!`);
                    }
                });
            });
            req.on("error", (error) => {
                app_logs_1.logger.error("An error occured during NetGSM SMS send request" + error);
            });
            req.end();
        }
        catch (error) {
            reject(error);
        }
    });
}
async function sendByMesajUssu(smsSettings, params, phoneNumber) {
    const config = smsSettings.config;
    const content = params.notification.content;
    const loginInfo = await (0, app_util_1.sendHttpsPostRequest)({
        hostname: "mesajussu.turkcell.com.tr",
        path: "/api/api/integration/v2/login",
        body: config.login,
    });
    const result = await (0, app_util_1.sendHttpsPostRequest)({
        hostname: "mesajussu.turkcell.com.tr",
        path: "/api/api/integration/v2/sms/send/oton",
        body: {
            message: (0, app_util_1.convertToEnglishCharString)(content.text),
            sender: config.header,
            receivers: [parseInt(phoneNumber)],
            etkFlag: false,
        },
        headers: {
            token: loginInfo.token,
        },
    });
    app_logs_1.logger.debug("mesajussu sms sended: " + JSON.stringify(result));
    await dal_manager_1.dbManager.accessNotifications.setNotificationInstanceState({
        instanceId: params.notification.instanceId,
        organizationId: params.organizationId,
        state: 4,
        trx: params.trx,
    });
}
async function sendByTurkTelekom(smsSettings, params, phoneNumber) {
    const config = smsSettings?.config;
    if (!config) {
        throw new Error("SMS settings configuration is missing.");
    }
    const content = params?.notification?.content;
    if (!content || !content.text) {
        throw new Error("SMS content is missing or invalid.");
    }
    if (!phoneNumber) {
        throw new Error("Phone number is missing.");
    }
    const soapXml = `<?xml version="1.0" encoding="utf-8"?>
<soap:Envelope xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" 
				xmlns:xsd="http://www.w3.org/2001/XMLSchema" 
				xmlns:soap="http://schemas.xmlsoap.org/soap/envelope/">
	<soap:Body>
	<sendSms xmlns="https://webservice.asistiletisim.com.tr/SmsProxy">
		<requestXml>
		<![CDATA[
			<SendSms>
			<Username>${config.username}</Username>
			<Password>${config.password}</Password>
			<UserCode>${config.usercode}</UserCode>
			<AccountId>${config.accountId}</AccountId>
			<Originator>${config.originator}</Originator>
			<SendDate></SendDate>
			<ValidityPeriod>60</ValidityPeriod>
			<MessageText>${(0, app_util_1.convertToEnglishCharString)(content.text)}</MessageText>
			<IsCheckBlackList>0</IsCheckBlackList>
			<IsEncryptedParameter>0</IsEncryptedParameter>
			<ReceiverList>
				<Receiver>${parseInt(phoneNumber)}</Receiver>
			</ReceiverList>
			</SendSms>
		]]>
		</requestXml>
	</sendSms>
	</soap:Body>
</soap:Envelope>`;
    const options = {
        hostname: "https://webservice.asistiletisim.com.tr",
        path: "/SmsProxy.asmx",
        method: "POST",
        headers: {
            "Content-Type": "text/xml; charset=utf-8",
            SOAPAction: "https://webservice.asistiletisim.com.tr/SmsProxy/sendSms",
        },
    };
    const result = await new Promise((resolve, reject) => {
        const req = https_1.default.request(options, (res) => {
            let data = "";
            res.on("data", (chunk) => {
                data += chunk;
            });
            res.on("end", () => {
                resolve({ body: data, statusCode: res.statusCode || 200 });
            });
        });
        req.on("error", (err) => {
            reject(err);
        });
        req.write(soapXml);
        req.end();
    });
    if (result.statusCode !== 200) {
        throw new Error(`Turk Telekom SMS send failed with status code: ${result.statusCode}`);
    }
    const parsedResponse = await (0, xml2js_1.parseStringPromise)(result.body);
    const errorCode = parsedResponse?.["soap:Envelope"]?.["soap:Body"]?.[0]?.["sendSmsResponse"]?.[0]?.["sendSmsResult"]?.[0]?.["ErrorCode"]?.[0];
    if (errorCode === "0") {
        app_logs_1.logger.debug("Turk Telekom SMS sent successfully.");
    }
    else {
        const errorMessage = handleTurkTelekomError(errorCode);
        throw new Error(`Turk Telekom SMS send failed: ${errorMessage}`);
    }
    await dal_manager_1.dbManager.accessNotifications.setNotificationInstanceState({
        instanceId: params.notification.instanceId,
        organizationId: params.organizationId,
        state: 4,
        trx: params.trx,
    });
}
async function sendByBalaban(smsSettings, params, phoneNumber) {
    const config = smsSettings.config;
    const content = params.notification.content;
    const requestBody = {
        Credential: {
            Username: config?.username,
            Password: config?.password,
        },
        Header: {
            From: config?.from,
            Route: 0,
        },
        Message: (0, app_util_1.convertToEnglishCharString)(content.text),
        To: [phoneNumber],
        DataCoding: "Default",
    };
    const result = await (0, app_util_1.sendHttpsPostRequest)({
        hostname: "gw.maradit.net",
        path: "/api/json/reply/Submit",
        body: requestBody,
        headers: {
            "Content-Type": "application/json",
        },
    });
    if (result.Response?.Status?.Code !== 200) {
        throw new Error(`Balaban SMS send failed with statusCode: ${result.Response?.Status?.Code}, description: ${result.Response?.Status?.Description}`);
    }
    app_logs_1.logger.debug("Balaban sms sended: " + JSON.stringify(result));
    await dal_manager_1.dbManager.accessNotifications.setNotificationInstanceState({
        instanceId: params.notification.instanceId,
        organizationId: params.organizationId,
        state: 4,
        trx: params.trx,
    });
}
async function updateSmsBalance(organizationId, messageLength, phoneNumber, smsSettings, trx) {
    const baseSmsCount = Math.ceil(messageLength / 155);
    const multiplier = (0, app_util_1.isPhoneNumberFromTurkey)(phoneNumber) ? 1 : smsSettings.internationalSMSMultiplier ?? 3;
    const usedSmsCount = baseSmsCount * multiplier;
    await dal_manager_1.dbManager.accessNotifications.setSmsBalance({ organizationId, used: usedSmsCount, trx });
}
function handleTurkTelekomError(errorCode) {
    const errorMessages = {
        "0": "NO_ERROR",
        "-1": "Girilen bilgilere sahip bir kullanıcı bulunamadı.",
        "-2": "Kullanıcı pasif durumda.",
        "-3": "Kullanıcı bloke durumda.",
        "-4": "Kullanıcı hesabı bulunamadı.",
        "-5": "Kullanıcı hesabı pasif durumda.",
        "-6": "Kayıt bulunamadı.",
        "-7": "Hatalı xml istek yapısı.",
        "-8": "Alınan parametrelerden biri veya birkaçı hatalı.",
        "-9": "Prepaid hesap bulunamadı.",
        "-10": "Operatör servisinde geçici kesinti.",
        "-11": "Başlangıç tarihi ile şu an ki zaman arasındaki fark 30 dakikadan az.",
        "-12": "Bitiş tarihi ile şu an ki zaman arasındaki fark 30 günden fazla.",
        "-13": "Geçersiz gönderici bilgisi.",
        "-14": "Hesaba ait SMS gönderim yetkisi bulunmuyor.",
        "-15": "Mesaj içeriği boş veya limit olan karakter sayısını aşıyor.",
        "-16": "Geçersiz alıcı bilgisi.",
        "-17": "Parametre adetleri ile şablon içerisindeki parametre adedi uyuşmuyor.",
        "-18": "Gönderim içerisinde birden fazla hata mevcut. MessageId kontrol edilmelidir.",
        "-19": "Mükerrer gönderim isteği.",
        "-20": "Bilgilendirme mesajı almak istemiyor.",
        "-21": "Numara karalistede.",
        "-22": "Yetkisiz IP Adresi.",
        "-23": "Kullanıcı yetkisi bulunmamaktadır.",
        "-24": "Belirtilen paket zaten onaylanmıştır.",
        "-25": "Belirtilen Id için onaylanmamış bir paket bulunamadı.",
        "-26": "Taahhüt süresi zaman aşımına uğradı.",
        "-27": "Taahhüt miktarı aşıldı.",
        "-28": "Kullanıcı gönderim limiti aşıldı.",
        "-29": "Başlangıç tarihi bitiş tarihinden büyük olamaz.",
        "-30": "Kullanıcının API kullanım yetkisi aktif değil.",
        "-31": "Mesaj metni içerisinde yasaklı kelime kullanıldığı için başarısız.",
        "-1000": "SYSTEM_ERROR",
    };
    return errorMessages[errorCode] || `Unknown error (Code: ${errorCode})`;
}
