"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.onNotificationMessageToSend = exports.onNotificationMessageReceived = void 0;
const app_constants_1 = require("../../app.constants");
const app_enums_1 = require("../../app.enums");
const app_logs_1 = require("../../app.logs");
const dal_manager_1 = require("../../dal/dal.manager");
const messageBroker_notification_sub_1 = require("../../messageBroker/messageBroker.notification.sub");
const generator_1 = require("./generator");
const sender_1 = require("./sender");
async function onNotificationMessageReceived(msg) {
    if (!msg) {
        return;
    }
    const amqpMessage = JSON.parse(msg.content.toString());
    try {
        app_logs_1.logger.debug(`nid[${amqpMessage.i}], o[${amqpMessage.o}] received!`);
        await (0, generator_1.generateNotificationInstance)(amqpMessage);
        app_logs_1.logger.debug(`nid[${amqpMessage.i}], o[${amqpMessage.o}] ready to send!`);
        await (0, messageBroker_notification_sub_1.sendToDelayedRetryQueue)(amqpMessage);
        (0, messageBroker_notification_sub_1.notificationAck)(msg);
    }
    catch (error) {
        (0, messageBroker_notification_sub_1.notificationAck)(msg);
        app_logs_1.logger.error("Notification [" + amqpMessage.i + "] failed. If you see this message, there is a bug in reAttempt mechanism." + error.message || JSON.stringify(error));
        return;
    }
}
exports.onNotificationMessageReceived = onNotificationMessageReceived;
async function onNotificationMessageToSend(msg) {
    if (!msg) {
        return;
    }
    const amqpMessage = JSON.parse(msg.content.toString());
    const attemptCount = ++amqpMessage.a;
    const organizationId = amqpMessage.o;
    let trx;
    try {
        const notificationInstancelist = await dal_manager_1.dbManager.systemTransaction(async (trx) => {
            await dal_manager_1.dbManager.accessNotifications.updateNotificationInstanceStates({
                organizationId,
                currentStates: [1, 3],
                newState: 2,
                eventId: amqpMessage.i,
                includeFailedNotifications: false,
                trx,
            });
            return dal_manager_1.dbManager.accessNotifications.getNotificationInstancesOfEventMarkedAsSending({ organizationId, notificationEventId: amqpMessage.i, trx });
        });
        if (notificationInstancelist.length === 0) {
            app_logs_1.logger.debug(`nid[${amqpMessage.i}], o[${amqpMessage.o}] no instance to to send. Have a nice day!`);
            (0, messageBroker_notification_sub_1.notificationAck)(msg);
            return;
        }
        app_logs_1.logger.debug(`nid[${amqpMessage.i}], o[${amqpMessage.o}] is sending...`);
        trx = await dal_manager_1.dbManager.poolMain.connect();
        const results = await Promise.allSettled(notificationInstancelist.map(async (notificationInstance) => await (0, sender_1.sendNotification)({
            organizationId,
            medium: notificationInstance.medium,
            notification: {
                content: notificationInstance.content,
                instanceId: notificationInstance.id,
                receiverUserId: notificationInstance.receiverUserId,
                type: notificationInstance.type,
            },
            message: amqpMessage,
            trx,
        })));
        trx.release();
        if (amqpMessage.a > 3) {
            app_logs_1.logger.debug(`nid[${amqpMessage.i}], o[${amqpMessage.o}] notification reattempts finished!`);
            if (results.length === 1) {
                switch (notificationInstancelist[0].type) {
                    case app_enums_1.enums.NotificationType.AccountCreated:
                        dal_manager_1.dbManager.systemTransaction(async (trx) => {
                            return dal_manager_1.dbManager.accessNotifications.anonymiseAccountCreatedNotification({ notificationEventId: amqpMessage.i, organizationId, trx });
                        });
                        break;
                    case app_enums_1.enums.NotificationType.SendingVerificationToken:
                        dal_manager_1.dbManager.systemTransaction(async (trx) => {
                            return dal_manager_1.dbManager.accessNotifications.anonymiseMFAVerificationNotification({ notificationEventId: amqpMessage.i, organizationId, trx });
                        });
                        break;
                    default:
                        break;
                }
            }
            (0, messageBroker_notification_sub_1.notificationNack)(msg, false, false);
            return;
        }
        await dal_manager_1.dbManager.systemTransaction(async (trx) => {
            return dal_manager_1.dbManager.accessNotifications.updateNotificationInstanceStates({
                organizationId,
                currentStates: [2],
                newState: 3,
                eventId: amqpMessage.i,
                trx,
            });
        });
        if (shouldServiceTryToSendNotificationAgain(results)) {
            const nextExecutionDate = await (0, messageBroker_notification_sub_1.sendToDelayedRetryQueue)(amqpMessage, msg.properties.priority);
            app_logs_1.logger.debug(`nid[${amqpMessage.i}], o[${amqpMessage.o}] attempt: [${attemptCount}] Retrying to send notification in ${nextExecutionDate}`);
        }
        else {
            app_logs_1.logger.debug(`nid[${amqpMessage.i}], o[${amqpMessage.o}] attempt: [${attemptCount}] sent.`);
            if (results.length === 1) {
                switch (notificationInstancelist[0].type) {
                    case app_enums_1.enums.NotificationType.AccountCreated:
                        dal_manager_1.dbManager.systemTransaction(async (trx) => {
                            return dal_manager_1.dbManager.accessNotifications.anonymiseAccountCreatedNotification({ notificationEventId: amqpMessage.i, organizationId, trx });
                        });
                        break;
                    case app_enums_1.enums.NotificationType.SendingVerificationToken:
                        dal_manager_1.dbManager.systemTransaction(async (trx) => {
                            return dal_manager_1.dbManager.accessNotifications.anonymiseMFAVerificationNotification({ notificationEventId: amqpMessage.i, organizationId, trx });
                        });
                        break;
                    default:
                        break;
                }
            }
        }
        (0, messageBroker_notification_sub_1.notificationAck)(msg);
    }
    catch (error) {
        trx?.release();
        app_logs_1.logger.debug(error);
        (0, messageBroker_notification_sub_1.notificationNack)(msg, false, false);
    }
}
exports.onNotificationMessageToSend = onNotificationMessageToSend;
function shouldServiceTryToSendNotificationAgain(PromiseSettledResult) {
    const notificationsNeedsRetry = PromiseSettledResult.filter((result) => result.status === "rejected" && !app_constants_1.safeNotificationErrorCodes.some((errorCode) => (result.reason?.message).includes(errorCode + "")));
    return notificationsNeedsRetry.length > 0;
}
