"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handleTerminalEmergency = exports.handleRegionEmergency = void 0;
const app_enums_1 = require("../app.enums");
const app_logs_1 = require("../app.logs");
const app_util_1 = require("../app.util");
const dal_manager_1 = require("../dal/dal.manager");
const messageBroker_server_to_device_pub_1 = require("../messageBroker/messageBroker.server-to-device.pub");
const business_notification_1 = require("./business.notification");
async function handleRegionEmergency(params) {
    try {
        const [deviceBasics] = await Promise.all([
            dal_manager_1.dbManager.accessDevice.getDeviceBasics(params.organizationId, params.relatedTerminalIds),
            dal_manager_1.dbManager.accessRegion.setEmergencyForRegion(params.organizationId, params.regionId, params.state),
        ]);
        (0, business_notification_1.handleRegionEmergencyNotifications)({
            organizationId: params.organizationId,
            regionId: params.regionId,
            regionName: params.regionName,
            state: params.state,
        });
        await Promise.allSettled(params.relatedTerminalIds.map((r) => handleTerminalEmergency({
            organizationId: params.organizationId,
            terminalId: r,
            state: params.state,
            terminalName: deviceBasics.find((db) => db.id === r).name,
            regionName: params.regionName,
        })));
        app_logs_1.logger.debug(`Region Emergency ${params.state === app_enums_1.enums.EmergencyState.Emergency ? "Started" : "End"} on ${params.regionName}`);
    }
    catch (error) {
        app_logs_1.logger.error(`Error while handling region emergency for ${params.regionName}!`);
        app_logs_1.logger.error(error);
    }
}
exports.handleRegionEmergency = handleRegionEmergency;
async function handleTerminalEmergency(params) {
    try {
        let terminalEmergencyPromise = new Promise((resolve, reject) => {
            const terminalEmergencyRPCMessageType = (0, app_util_1.appwebEmergencyCodeToDeviceEmergencyCode)(params.state);
            messageBroker_server_to_device_pub_1.amqpServerToDevicePub.sendToRpcQueue(params.terminalId, {
                e: terminalEmergencyRPCMessageType,
                p: null,
            }, 60 * 1000, (err, success) => {
                if (err) {
                    return reject(err);
                }
                resolve(null);
            });
        });
        await terminalEmergencyPromise;
        await Promise.all([
            dal_manager_1.dbManager.accessDevice.setEmergencyForDevices({ organizationId: params.organizationId, deviceIds: [params.terminalId], emergency: params.state }),
            dal_manager_1.dbManager.accessLog.insertSystemStatusLogForTerminalAction(params.organizationId, params.terminalId, { e: app_enums_1.enums.TerminalAction.Emergency }),
        ]);
        (0, business_notification_1.handleTerminalEmergencyNotifications)({
            organizationId: params.organizationId,
            terminalId: params.terminalId,
            terminalName: params.terminalName,
            state: params.state,
            regionName: params.regionName,
        });
        app_logs_1.logger.debug(`Terminal Emergency ${params.state === app_enums_1.enums.EmergencyState.Emergency ? "Started" : "End"} on ${params.terminalName}`);
    }
    catch (error) {
        app_logs_1.logger.error(`Error while handling terminal emergency for ${params.terminalName}!`);
        app_logs_1.logger.error(error);
    }
}
exports.handleTerminalEmergency = handleTerminalEmergency;
