"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.insertVirtualACPAccessLog = exports.deleteManualLog = exports.insertMultipleManualLogs = exports.insertManualLog = exports.insertMobileManualLog = void 0;
const dal_manager_1 = require("../dal/dal.manager");
const moment_1 = __importDefault(require("moment"));
const app_enums_1 = require("../app.enums");
const messageBroker_server_to_device_pub_1 = require("../messageBroker/messageBroker.server-to-device.pub");
const messagebroker_models_1 = require("../messageBroker/messagebroker.models");
const app_logs_1 = require("../app.logs");
const api_error_1 = require("../api/api.error");
const dal_constants_1 = require("../dal/dal.constants");
const api_securityhelper_1 = require("../api/api.securityhelper");
const business_device_1 = require("./business.device");
const messageBroker_server_to_app_pub_1 = require("../messageBroker/messageBroker.server-to-app.pub");
const ws_usernsp_1 = require("../ws/ws.usernsp");
const messageBroker_server_to_app_sub_1 = require("../messageBroker/messageBroker.server-to-app.sub");
const enums_1 = require("../lib/access-model/v2/enums");
const dal_access_psql_log_1 = require("../dal/access/psql/dal.access.psql.log");
async function processRegionsForManualLogInsert(organizationId, request, trx) {
    let regions = (await dal_manager_1.dbManager.accessRegion.listAccessControlPointRegions(organizationId, request.accessControlPointId)).map((r) => r.id);
    for (const region of regions) {
        let userStateRegion = await dal_manager_1.dbManager.accessRegion.getUserRegionLastState(organizationId, request.userId, region);
        if (!userStateRegion) {
            continue;
        }
        if ((0, moment_1.default)(userStateRegion.actionUtc).isAfter((0, moment_1.default)(request.actionUtc))) {
            continue;
        }
        let regionStateObject = {
            userId: request.userId,
            actionUtc: request.actionUtc,
            state: null,
            regionId: region,
        };
        if (request.direction === app_enums_1.enums.AccessDirection.Entrance) {
            regionStateObject.state = app_enums_1.enums.AntiPassbackState.In;
        }
        else {
            regionStateObject.state = app_enums_1.enums.AntiPassbackState.Out;
        }
        await dal_manager_1.dbManager.accessRegion.upsertUserRegionState(organizationId, regionStateObject, trx);
        let deviceId = (await trx.withSchema(organizationId).table("accessControlPoints").where("id", request.accessControlPointId).select("deviceId").first()).deviceId;
        try {
            let now = new Date();
            let regionNotification = {
                i: region,
                s: regionStateObject.state,
                ee: null,
                xe: null,
                o: regionStateObject.userId,
                t: (0, moment_1.default)(regionStateObject.actionUtc).format(),
            };
            messageBroker_server_to_device_pub_1.amqpServerToDevicePub.sendToRpcQueue(deviceId, {
                e: messagebroker_models_1.ServerToDeviceRPCMessageType.RegionStateChanged,
                p: regionNotification,
            }, 30000, (err, success) => {
                if (err) {
                    let changeItem = {
                        actionDateISO: now.toISOString(),
                        type: app_enums_1.enums.DeviceChangeItemType.RegionState,
                        data: {
                            actionDateISO: (0, moment_1.default)(regionStateObject.actionUtc).format(),
                            regionId: region,
                            state: regionStateObject.state,
                            userId: regionStateObject.userId,
                            entranceLockExpirationUtc: null,
                            exitLockExpirationUtc: null,
                        },
                    };
                    dal_manager_1.dbManager.accessDevice.addTerminalChange(deviceId, organizationId, changeItem);
                }
            });
        }
        catch (error) {
            app_logs_1.logger.error(error);
        }
    }
}
async function insertMobileManualLog(params) {
    const { organizationId, logRequest, requesterUserId, trx } = params;
    if ((0, moment_1.default)(logRequest.actionUtc).isAfter(new Date())) {
        return Promise.reject("invalid action date");
    }
    const userAcpInfo = await dal_manager_1.dbManager.accessAccessControlPoint.getUserMobileCheckinInformation(organizationId, logRequest.userId, params.accessControlPoint.id);
    if (!userAcpInfo.accessControlPointId) {
        throw (0, api_error_1.generateNotFoundApiError)({ message: "This organization does not have a mobile access control point" });
    }
    if (!userAcpInfo.hasAccessRight) {
        (0, api_securityhelper_1.throwAuthError)("This user does not have sufficient rights for mobile check-in");
    }
    const addedLogId = await dal_manager_1.dbManager.accessLog.addMobileCheckinAccessLog({
        organizationId,
        logRequest: {
            accessControlPointId: params.accessControlPoint.id,
            accessControlPointName: params.accessControlPoint.name,
            userId: logRequest.userId,
            reason: dal_constants_1.DalConstants.libEnumsV2.AccessLogReason.Manual,
            userRoleId: userAcpInfo.userRoleId,
            accessRightId: userAcpInfo.accessRightId,
            direction: logRequest.direction,
            actionUtc: new Date(logRequest.actionUtc),
            requesterUserId,
            isQrTriggered: params.isQrTriggered,
        },
        trx,
    });
    await dal_manager_1.dbManager.accessPacs2.onEmployeeMobileCheckin({ organizationId, userId: logRequest.userId, date: logRequest.actionUtc, trx });
    if (addedLogId && logRequest.direction && logRequest.direction === enums_1.EnumsV2.AccessDirection.Entrance) {
        await (0, dal_access_psql_log_1.assignAutoShift)(organizationId, {
            credentialOwnerUserId: logRequest.userId,
            generationTime: logRequest.actionUtc.toISOString(),
            redisCache: dal_manager_1.dbManager.accessRedisCache,
            logId: addedLogId,
        }, trx);
    }
    return addedLogId;
}
exports.insertMobileManualLog = insertMobileManualLog;
async function insertManualLog(organizationId, requestUserId, accessControlPoint, logRequest) {
    return await dal_manager_1.dbManager.transaction(async (trx) => {
        if ((0, moment_1.default)(logRequest.actionUtc).isAfter(new Date())) {
            return Promise.reject("invalid action date");
        }
        logRequest.accessControlPointName = accessControlPoint.name;
        let logInsertResult = await dal_manager_1.dbManager.accessLog.insertManualLog(organizationId, requestUserId, logRequest);
        await processRegionsForManualLogInsert(organizationId, logRequest, trx);
        dal_manager_1.dbManager.accessPacs2.onAccessLogsChanged({
            reason: app_enums_1.enums.RecalculateWorkReason.ManuelLogChange,
            acpId: logRequest.accessControlPointId,
            timestamp: logRequest.actionUtc,
            userId: logRequest.userId,
            direction: logRequest.direction,
            organizationId: organizationId,
        });
        return Promise.resolve(logInsertResult);
    });
}
exports.insertManualLog = insertManualLog;
async function insertMultipleManualLogs(organizationId, requestUserId, multiplelogsRequest) {
    return await dal_manager_1.dbManager.transaction(async (trx) => {
        let insertedIds = [];
        let nowMoment = (0, moment_1.default)();
        for (const dt of multiplelogsRequest.ranges) {
            if ((0, moment_1.default)(dt.startDateTime).isAfter(nowMoment) || (0, moment_1.default)(dt.endDateTime).isAfter(nowMoment))
                return Promise.reject("invalid action date");
        }
        let entranceAccessControlPoint = await dal_manager_1.dbManager.accessAccessControlPoint.getAccessControlPointBasic(organizationId, multiplelogsRequest.entrance.accessControlPointId);
        let exitAccessControlPoint = await dal_manager_1.dbManager.accessAccessControlPoint.getAccessControlPointBasic(organizationId, multiplelogsRequest.exit.accessControlPointId);
        if (!entranceAccessControlPoint || !exitAccessControlPoint)
            return Promise.reject("access control point not found");
        let entranceWiegandReaderInfo = await dal_manager_1.dbManager.accessAccessControlPoint.getWiegandInfoById(organizationId, multiplelogsRequest.entrance.accessControlPointId, multiplelogsRequest.entrance.wiegandReaderId);
        let exitWiegandReaderInfo = await dal_manager_1.dbManager.accessAccessControlPoint.getWiegandInfoById(organizationId, multiplelogsRequest.exit.accessControlPointId, multiplelogsRequest.exit.wiegandReaderId);
        if (!entranceWiegandReaderInfo || !exitWiegandReaderInfo)
            return Promise.reject("wiegand reader info not found");
        let entrancelogRequest = {
            accessControlPointId: multiplelogsRequest.entrance.accessControlPointId,
            userId: multiplelogsRequest.userId,
            credentialId: multiplelogsRequest.credentialId,
            wiegandReaderId: multiplelogsRequest.entrance.wiegandReaderId,
            direction: entranceWiegandReaderInfo.direction,
            actionUtc: undefined,
            accessControlPointName: entranceAccessControlPoint.name,
            deviceId: entranceWiegandReaderInfo.deviceId,
            wiegandReaderInputNumber: entranceWiegandReaderInfo.number,
        };
        let exitlogRequest = {
            accessControlPointId: multiplelogsRequest.exit.accessControlPointId,
            userId: multiplelogsRequest.userId,
            credentialId: multiplelogsRequest.credentialId,
            wiegandReaderId: multiplelogsRequest.exit.wiegandReaderId,
            direction: exitWiegandReaderInfo.direction,
            actionUtc: undefined,
            accessControlPointName: exitAccessControlPoint.name,
            deviceId: exitWiegandReaderInfo.deviceId,
            wiegandReaderInputNumber: exitWiegandReaderInfo.number,
        };
        let earliestEntranceTimestamp = nowMoment;
        let earliestExitTimestamp = nowMoment;
        for (const dt of multiplelogsRequest.ranges) {
            entrancelogRequest.actionUtc = dt.startDateTime;
            let entranceInsertResult = await dal_manager_1.dbManager.accessLog.insertManualLog(organizationId, requestUserId, entrancelogRequest, trx);
            await processRegionsForManualLogInsert(organizationId, entrancelogRequest, trx);
            if ((0, moment_1.default)(entrancelogRequest.actionUtc).isBefore(earliestEntranceTimestamp)) {
                earliestEntranceTimestamp = (0, moment_1.default)(entrancelogRequest.actionUtc);
            }
            insertedIds.push(entranceInsertResult);
            exitlogRequest.actionUtc = dt.endDateTime;
            let exitInsertResult = await dal_manager_1.dbManager.accessLog.insertManualLog(organizationId, requestUserId, exitlogRequest, trx);
            await processRegionsForManualLogInsert(organizationId, exitlogRequest, trx);
            if ((0, moment_1.default)(exitlogRequest.actionUtc).isBefore(earliestExitTimestamp)) {
                earliestExitTimestamp = (0, moment_1.default)(exitlogRequest.actionUtc);
            }
            insertedIds.push(exitInsertResult);
        }
        dal_manager_1.dbManager.accessPacs2.onAccessLogsChanged({
            reason: app_enums_1.enums.RecalculateWorkReason.ManuelLogChange,
            acpId: entrancelogRequest.accessControlPointId,
            timestamp: earliestEntranceTimestamp.toDate(),
            userId: entrancelogRequest.userId,
            direction: entrancelogRequest.direction,
            organizationId: organizationId,
        });
        dal_manager_1.dbManager.accessPacs2.onAccessLogsChanged({
            reason: app_enums_1.enums.RecalculateWorkReason.ManuelLogChange,
            acpId: exitlogRequest.accessControlPointId,
            timestamp: earliestExitTimestamp.toDate(),
            userId: exitlogRequest.userId,
            direction: exitlogRequest.direction,
            organizationId: organizationId,
        });
        return Promise.resolve(insertedIds);
    });
}
exports.insertMultipleManualLogs = insertMultipleManualLogs;
async function deleteManualLog(organizationId, logId) {
    await dal_manager_1.dbManager.transaction(async (trx) => {
        let logRequest = await dal_manager_1.dbManager.accessLog.getManualLogById(organizationId, logId);
        let regions = (await dal_manager_1.dbManager.accessRegion.listAccessControlPointRegions(organizationId, logRequest.accessControlPointId)).map((r) => r.id);
        await dal_manager_1.dbManager.accessLog.deleteManualLogById(organizationId, logId);
        dal_manager_1.dbManager.accessPacs2.onAccessLogsChanged({
            reason: app_enums_1.enums.RecalculateWorkReason.ManuelLogChange,
            acpId: logRequest.accessControlPointId,
            timestamp: logRequest.actionUtc,
            userId: logRequest.userId,
            direction: logRequest.direction,
            organizationId: organizationId,
        });
        for (const region of regions) {
            let userStateRegion = await dal_manager_1.dbManager.accessRegion.getUserRegionLastState(organizationId, logRequest.userId, region);
            if (!userStateRegion) {
                continue;
            }
            if ((0, moment_1.default)(userStateRegion.actionUtc).isAfter((0, moment_1.default)(logRequest.actionUtc))) {
                continue;
            }
            let regionStateObject = {
                userId: logRequest.userId,
                actionUtc: logRequest.actionUtc,
                state: null,
                regionId: region,
            };
            if (logRequest.direction === app_enums_1.enums.AccessDirection.Entrance) {
                regionStateObject.state = app_enums_1.enums.AntiPassbackState.Out;
            }
            else {
                regionStateObject.state = app_enums_1.enums.AntiPassbackState.In;
            }
            await dal_manager_1.dbManager.accessRegion.upsertUserRegionState(organizationId, regionStateObject, trx);
            let deviceId = (await trx.withSchema(organizationId).table("accessControlPoints").where("id", logRequest.accessControlPointId).select("deviceId").first()).deviceId;
            try {
                let now = new Date();
                let regionNotification = {
                    i: region,
                    s: regionStateObject.state,
                    ee: null,
                    xe: null,
                    o: regionStateObject.userId,
                    t: (0, moment_1.default)(regionStateObject.actionUtc).format(),
                };
                messageBroker_server_to_device_pub_1.amqpServerToDevicePub.sendToRpcQueue(deviceId, {
                    e: messagebroker_models_1.ServerToDeviceRPCMessageType.RegionStateChanged,
                    p: regionNotification,
                }, 30000, (err, success) => {
                    if (err) {
                        let changeItem = {
                            actionDateISO: now.toISOString(),
                            type: app_enums_1.enums.DeviceChangeItemType.RegionState,
                            data: {
                                actionDateISO: (0, moment_1.default)(regionStateObject.actionUtc).format(),
                                regionId: region,
                                state: regionStateObject.state,
                                userId: regionStateObject.userId,
                                entranceLockExpirationUtc: null,
                                exitLockExpirationUtc: null,
                            },
                        };
                        dal_manager_1.dbManager.accessDevice.addTerminalChange(deviceId, organizationId, changeItem);
                    }
                });
            }
            catch (error) {
                app_logs_1.logger.error(error);
            }
        }
    });
}
exports.deleteManualLog = deleteManualLog;
async function insertVirtualACPAccessLog(organizationId, params) {
    const { remoteAccessCommandStructure, direction, userId } = params;
    const actionUtc = new Date();
    try {
        let log = await (0, business_device_1.updateLogFromDevice)({
            organizationId: organizationId,
            deviceLog: {
                a: remoteAccessCommandStructure.accessControlPointId,
                d: direction,
                o: userId,
            },
            logReceiveMethod: app_enums_1.enums.LogReceiveMethod.Instant,
        });
        log.u = (0, moment_1.default)(actionUtc).toISOString();
        log.s = true;
        log.ir = params.qr === true ? false : true;
        log.r = app_enums_1.enums.AccessLogReason.Success;
        let insertedLog = await dal_manager_1.dbManager.accessLog.insertAccessLogFromDevice(organizationId, log);
        if (params.qr) {
            await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
                await dal_manager_1.dbManager.accessPacs2.onEmployeeMobileCheckin({ organizationId, userId: params.userId, date: actionUtc, trx });
            }, params.userId, organizationId);
        }
        messageBroker_server_to_app_pub_1.amqpServerToAppPub.sendToExchange(organizationId + "." + ws_usernsp_1.room.activity + "." + (log.a ? log.a : "*"), {
            e: messageBroker_server_to_app_sub_1.amqpServerToAppSubUserEventNames.newlog,
            p: insertedLog,
            oId: organizationId,
        });
        return true;
    }
    catch (err) {
        app_logs_1.logger.error("Error on remote access because no device id is present!");
        app_logs_1.logger.error(err);
        return false;
    }
}
exports.insertVirtualACPAccessLog = insertVirtualACPAccessLog;
