"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.notifyUsersAboutAperioBatteryStatus = exports.updateLogFromMobileDevice = exports.uploadDeviceSystemStatusLogs = exports.updateDeviceConnectionStatus = exports.terminalConnectionNotificationToSystemAdmins = exports.generateWarnings = exports.updateLogFromDevice = exports.updateLogFromDeviceWithClient = void 0;
const moment_1 = __importDefault(require("moment"));
const uuid_1 = __importDefault(require("uuid"));
const app_config_1 = require("../app.config");
const app_enums_1 = require("../app.enums");
const app_logs_1 = require("../app.logs");
const dal_access_cache_1 = require("../dal/access/access-interfaces/dal.access.cache");
const dal_memcache_1 = require("../dal/access/dal.memcache");
const cli_queries_1 = require("../dal/access/psql/cli-queries");
const dal_constants_1 = require("../dal/dal.constants");
const dal_manager_1 = require("../dal/dal.manager");
const dal_db_armon_schema_1 = require("../dal/db/armon/dal.db.armon.schema");
const business_notification_1 = require("./business.notification");
const restapi_1 = require("../lib/es/models/restapi");
async function updateLogFromDeviceWithClient(params) {
    const { logReceiveMethod, deviceLog, organizationId, redisCache, trx } = params;
    let accessLog = {
        rm: logReceiveMethod,
        id: deviceLog.id || uuid_1.default.v4(),
        u: deviceLog.u,
        oId: organizationId,
        a: deviceLog.a,
        r: deviceLog.r,
        o: deviceLog.o || dal_db_armon_schema_1.ArmonSchema.unknownCredentialOwnerId,
        i: deviceLog.i || null,
        ci: null,
        re: deviceLog.re || null,
        cid: deviceLog.cid || null,
        d: deviceLog.d ?? null,
        an: null,
        on: null,
        c: deviceLog.c || null,
        di: deviceLog.di || null,
        ir: deviceLog.ir || null,
        iq: deviceLog.iq || null,
        sid: deviceLog.sid || null,
        ul: null,
        s: deviceLog.s || deviceLog.r == app_enums_1.enums.AccessLogReason.Success || false,
        v: deviceLog.v || null,
        rg: deviceLog.rg || null,
        cx: deviceLog.cx || null,
        mp: deviceLog.mp || null,
        rx: deviceLog.rx || null,
        ss: deviceLog.ss || null,
        cs: deviceLog.cs || null,
        lt: deviceLog.lt || null,
        ln: deviceLog.ln || null,
        lr: deviceLog.lr !== undefined ? deviceLog.lr : null,
    };
    if (accessLog.a) {
        const accessControlPointBasic = await redisCache.getAccessPointCache({
            organizationId,
            accessPointId: accessLog.a,
            trx,
        });
        if (accessControlPointBasic) {
            accessLog.an = accessControlPointBasic.n;
        }
        else {
            app_logs_1.logger.error("unkown access device");
        }
    }
    if (accessLog.o === dal_db_armon_schema_1.ArmonSchema.unknownCredentialOwnerId && accessLog.cx && accessLog.cx.length > 0 && accessLog.cx[0].d) {
        const user = await (0, cli_queries_1.getUserInfoByCredentialData)({ organizationId, credentialData: accessLog.cx[0].d, trx });
        if (user) {
            accessLog.on = user.fullName;
            accessLog.o = user.userId;
            accessLog.ci = user.credentialId;
            accessLog.ul = user.roleId;
            if (accessLog.r == app_enums_1.enums.AccessLogReason.UnknownCredentialOwner) {
                accessLog.r = app_enums_1.enums.AccessLogReason.HasNotRight;
            }
        }
    }
    else if (accessLog.o) {
        const userBasicInfo = await redisCache.getUserNotificationCache({ organizationId, userId: accessLog.o, trx });
        if (userBasicInfo) {
            accessLog.on = userBasicInfo.f;
            accessLog.ul = userBasicInfo.r;
        }
    }
    return accessLog;
}
exports.updateLogFromDeviceWithClient = updateLogFromDeviceWithClient;
async function updateLogFromDevice(params) {
    const { logReceiveMethod, deviceLog, organizationId } = params;
    let accessLog = {
        rm: logReceiveMethod,
        id: deviceLog.id || uuid_1.default.v4(),
        u: deviceLog.u,
        oId: organizationId,
        a: deviceLog.a,
        r: deviceLog.r,
        o: deviceLog.o || dal_db_armon_schema_1.ArmonSchema.unknownCredentialOwnerId,
        i: deviceLog.i || null,
        ci: null,
        re: deviceLog.re || null,
        cid: deviceLog.cid || null,
        d: deviceLog.d ?? null,
        an: null,
        on: null,
        c: deviceLog.c || null,
        di: deviceLog.di || null,
        ir: deviceLog.ir || null,
        iq: deviceLog.iq || null,
        sid: deviceLog.sid || null,
        ul: null,
        s: deviceLog.s || deviceLog.r == app_enums_1.enums.AccessLogReason.Success || false,
        v: deviceLog.v || null,
        rg: deviceLog.rg || null,
        cx: deviceLog.cx || null,
        mp: deviceLog.mp || null,
        rx: deviceLog.rx || null,
        ss: deviceLog.ss || null,
        cs: deviceLog.cs || null,
        lt: deviceLog.lt || null,
        ln: deviceLog.ln || null,
        lr: deviceLog.lr !== undefined ? deviceLog.lr : null,
    };
    if (accessLog.a) {
        const accessControlPointBasic = await dal_manager_1.dbManager.accessRedisCache.getAccessPointCache({
            organizationId,
            accessPointId: accessLog.a,
        });
        if (accessControlPointBasic) {
            accessLog.an = accessControlPointBasic.n;
        }
        else {
            app_logs_1.logger.error("unkown access device");
        }
    }
    if (accessLog.o === dal_db_armon_schema_1.ArmonSchema.unknownCredentialOwnerId && accessLog.cx && accessLog.cx.length > 0 && accessLog.cx[0].d) {
        const user = await dal_manager_1.dbManager.accessUser.getUserInfoByCredentialData({ organizationId, credentialData: accessLog.cx[0].d });
        if (user) {
            accessLog.on = user.fullName;
            accessLog.o = user.userId;
            accessLog.ci = user.credentialId;
            accessLog.ul = user.roleId;
            if (accessLog.r == app_enums_1.enums.AccessLogReason.UnknownCredentialOwner) {
                accessLog.r = app_enums_1.enums.AccessLogReason.HasNotRight;
            }
        }
    }
    else if (accessLog.o) {
        const userBasicInfo = await dal_manager_1.dbManager.accessRedisCache.getUserNotificationCache({ organizationId, userId: accessLog.o });
        if (userBasicInfo) {
            accessLog.on = userBasicInfo.f;
            accessLog.ul = userBasicInfo.r;
        }
    }
    return accessLog;
}
exports.updateLogFromDevice = updateLogFromDevice;
function generateWarnings(deviceLog, totalStorage) {
    let warnings = [];
    if (deviceLog.ib) {
        warnings.push({
            level: app_enums_1.enums.TerminalWarningLevel.Medium,
            type: app_enums_1.enums.TerminalWarningType.OnBattery,
        });
    }
    if (deviceLog.b) {
        if (deviceLog.b < 20) {
            warnings.push({
                level: app_enums_1.enums.TerminalWarningLevel.Critical,
                type: app_enums_1.enums.TerminalWarningType.LowBattery,
            });
        }
        else if (deviceLog.b < 40) {
            warnings.push({
                level: app_enums_1.enums.TerminalWarningLevel.High,
                type: app_enums_1.enums.TerminalWarningType.LowBattery,
            });
        }
        else if (deviceLog.b < 60) {
            warnings.push({
                level: app_enums_1.enums.TerminalWarningLevel.Medium,
                type: app_enums_1.enums.TerminalWarningType.LowBattery,
            });
        }
    }
    if (totalStorage && deviceLog.df) {
        let usedStorageRatio = (deviceLog.df / totalStorage) * 100;
        if (usedStorageRatio < 25) {
            warnings.push({
                level: app_enums_1.enums.TerminalWarningLevel.Medium,
                type: app_enums_1.enums.TerminalWarningType.LowDiskSpace,
            });
        }
        else if (usedStorageRatio < 10) {
            warnings.push({
                level: app_enums_1.enums.TerminalWarningLevel.High,
                type: app_enums_1.enums.TerminalWarningType.LowDiskSpace,
            });
        }
    }
    if (deviceLog.t) {
        if (deviceLog.t < -20) {
            warnings.push({
                level: app_enums_1.enums.TerminalWarningLevel.High,
                type: app_enums_1.enums.TerminalWarningType.TemperatureLow,
            });
        }
        else if (deviceLog.t < -10) {
            warnings.push({
                level: app_enums_1.enums.TerminalWarningLevel.Medium,
                type: app_enums_1.enums.TerminalWarningType.TemperatureLow,
            });
        }
        if (deviceLog.t > 100) {
            warnings.push({
                level: app_enums_1.enums.TerminalWarningLevel.High,
                type: app_enums_1.enums.TerminalWarningType.TemperatureHigh,
            });
        }
        else if (deviceLog.t > 90) {
            warnings.push({
                level: app_enums_1.enums.TerminalWarningLevel.Medium,
                type: app_enums_1.enums.TerminalWarningType.TemperatureHigh,
            });
        }
    }
    return warnings;
}
exports.generateWarnings = generateWarnings;
async function terminalConnectionNotificationToSystemAdmins(terminalId, isConnected, organizationId) {
    if (!isConnected) {
        const fn = setTimeout(async () => (0, business_notification_1.handleTerminalConnectionNotifications)({ isConnected: false, organizationId, terminalId }), app_config_1.appConfig.sendConnectionEventNotificationForDeviceAfterInSec * 1000);
        (0, dal_memcache_1.setTerminalConnectionTimeout)(terminalId, fn);
    }
    else {
        const lastDisconnectionTime = await dal_manager_1.dbManager.accessLog.getTerminalLastConnectionTime({ organizationId, terminalId, type: dal_constants_1.DalConstants.SystemStatusLogType.TerminalDisconnected });
        if (lastDisconnectionTime) {
            const offlineTime = (new Date().getTime() - lastDisconnectionTime.getTime()) / 1000;
            if (offlineTime > app_config_1.appConfig.sendConnectionEventNotificationForDeviceAfterInSec) {
                (0, business_notification_1.handleTerminalConnectionNotifications)({ isConnected: true, organizationId, terminalId });
            }
            else {
                const timeout = (0, dal_memcache_1.getTerminalConnectionTimeout)(terminalId);
                clearTimeout(timeout);
            }
        }
    }
}
exports.terminalConnectionNotificationToSystemAdmins = terminalConnectionNotificationToSystemAdmins;
async function updateDeviceConnectionStatus(params) {
    try {
        if (!params.isRestart) {
            await terminalConnectionNotificationToSystemAdmins(params.deviceId, params.isConnected, params.organizationId);
            dal_manager_1.dbManager.accessLog.insertSystemStatusLogForConnection(params.organizationId, params.deviceId, params.isConnected);
        }
        let deviceHealth = await dal_manager_1.dbManager.accessDevice.getDeviceHealth(params.organizationId, params.deviceId);
        if (!deviceHealth) {
            deviceHealth = {
                temperature: null,
                availableStorage: null,
                isConnected: false,
                lastUpdateUtc: null,
                power: null,
                usedStorage: null,
                warnings: [],
                ip: null,
                onBoardComputerFreeStorage: null,
                deviceId: params.deviceId,
                onBoardComputerModel: null,
                onBoardComputerTotalStorage: 0,
                version: null,
                batteryLevel: 0,
            };
        }
        deviceHealth.isConnected = params.isConnected;
        await dal_manager_1.dbManager.accessDevice.updateDeviceHealth(params.organizationId, params.deviceId, deviceHealth);
        const deviceStatus = await dal_manager_1.dbManager.accessDevice.getDeviceStatus(params.organizationId);
        await dal_manager_1.dbManager.accessRedisCache.setValue(dal_access_cache_1.CacheConstantKeys.DeviceStatus + params.organizationId, JSON.stringify(deviceStatus));
    }
    catch (error) {
        app_logs_1.logger.error(error);
    }
}
exports.updateDeviceConnectionStatus = updateDeviceConnectionStatus;
async function uploadDeviceSystemStatusLogs(organizationId, deviceId, deviceLogs) {
    try {
        if (deviceLogs.length < 1) {
            return null;
        }
        let deviceHealth = await dal_manager_1.dbManager.accessDevice.getDeviceHealth(organizationId, deviceId);
        if (!deviceHealth) {
            deviceHealth = {
                temperature: null,
                availableStorage: null,
                isConnected: false,
                lastUpdateUtc: null,
                power: null,
                usedStorage: null,
                warnings: [],
                ip: null,
                onBoardComputerFreeStorage: null,
                deviceId: deviceId,
                onBoardComputerModel: null,
                onBoardComputerTotalStorage: 0,
                version: null,
                batteryLevel: 0,
            };
        }
        await dal_manager_1.dbManager.accessLog.uploadDeviceSystemStatusLogs(organizationId, deviceId, deviceLogs, deviceHealth.batteryLevel);
        let sortedLogs = deviceLogs.sort((x, y) => (x.u < y.u ? 1 : -1));
        let lastLog = sortedLogs[0];
        if (!deviceHealth.lastUpdateUtc || (deviceHealth.lastUpdateUtc && (0, moment_1.default)(lastLog.u).isAfter(deviceHealth.lastUpdateUtc))) {
            let warnings = generateWarnings(sortedLogs[0], deviceHealth.availableStorage);
            if (!deviceHealth.version) {
                deviceHealth.version = {};
            }
            deviceHealth = {
                temperature: lastLog.t || deviceHealth.temperature,
                availableStorage: lastLog.dt || deviceHealth.availableStorage,
                lastUpdateUtc: lastLog.u ? new Date(lastLog.u) : deviceHealth.lastUpdateUtc,
                power: lastLog.ib ? app_enums_1.enums.DevicePowerState.OnBattery : app_enums_1.enums.DevicePowerState.OnPower,
                usedStorage: lastLog.dt && lastLog.df ? lastLog.dt - lastLog.df : deviceHealth.usedStorage,
                warnings: warnings,
                mac: lastLog.mac ?? deviceHealth.mac,
                ip: lastLog.ip || deviceHealth.ip,
                onBoardComputerFreeStorage: lastLog.df || deviceHealth.onBoardComputerFreeStorage,
                onBoardComputerModel: lastLog.cm || deviceHealth.onBoardComputerModel,
                onBoardComputerTotalStorage: lastLog.dt || deviceHealth.onBoardComputerTotalStorage,
                isConnected: lastLog.ic === null || lastLog.ic === undefined ? deviceHealth.isConnected : lastLog.ic,
                batteryLevel: lastLog.b || deviceHealth.batteryLevel,
                emergencyStatus: deviceHealth.emergencyStatus || app_enums_1.enums.EmergencyState.Normal,
                version: {
                    hardwareHwVersion: lastLog.hhv || deviceHealth.version?.hardwareHwVersion,
                    hardwareSwVersion: lastLog.hsv || deviceHealth.version?.hardwareSwVersion,
                    onBoardComputerVersion: lastLog.cv || deviceHealth.version?.onBoardComputerVersion,
                    serviceVersion: lastLog.sv || deviceHealth.version?.serviceVersion,
                    libAccessVersion: lastLog.lav || deviceHealth.version?.libAccessVersion,
                    backendApiVersion: lastLog.bav || deviceHealth.version?.backendApiVersion,
                    operatingSystem: lastLog.os || deviceHealth.version?.operatingSystem,
                },
            };
        }
        await dal_manager_1.dbManager.accessDevice.updateDeviceHealth(organizationId, deviceId, deviceHealth);
        if (deviceHealth.warnings.length > 0) {
            let deviceStatus = await dal_manager_1.dbManager.accessDevice.getDeviceStatus(organizationId);
            await dal_manager_1.dbManager.accessRedisCache.setValue(dal_access_cache_1.CacheConstantKeys.DeviceStatus + organizationId, JSON.stringify(deviceStatus));
        }
        deviceHealth.deviceId = deviceId;
        return deviceHealth;
    }
    catch (error) {
        app_logs_1.logger.error(error);
    }
    return Promise.resolve(null);
}
exports.uploadDeviceSystemStatusLogs = uploadDeviceSystemStatusLogs;
async function updateLogFromMobileDevice(organizationId, deviceLog) {
    let now = new Date().getTime();
    let listAcpRegions = [];
    let accessLog = {
        id: uuid_1.default.v4(),
        rm: deviceLog.rm,
        u: deviceLog.u,
        oId: deviceLog.oId,
        a: deviceLog.a,
        c: deviceLog.c,
        r: null,
        o: deviceLog.o,
        i: null,
        ci: null,
        re: deviceLog.re || null,
        cid: null,
        d: deviceLog.d ?? null,
        an: null,
        on: null,
        di: null,
        ir: false,
        iq: false,
        sid: null,
        ul: null,
        s: null,
        v: deviceLog.v,
        rg: [],
        cx: deviceLog.cx || [],
        mp: deviceLog.mp || null,
        rx: deviceLog.rx || null,
        ss: deviceLog.ss || null,
        cs: deviceLog.cs || null,
        ma: true,
        lt: deviceLog.lt || null,
        ln: deviceLog.ln || null,
        lr: deviceLog.lr !== undefined ? deviceLog.lr : null,
    };
    if (areDataProvidedCorrectly(accessLog)) {
        const accessControlPointBasic = await dal_manager_1.dbManager.accessAccessControlPoint.getAccessControlPointBasic(organizationId, accessLog.a);
        if (accessControlPointBasic) {
            accessLog.an = accessControlPointBasic.name;
            accessLog.cid = accessControlPointBasic.deviceId;
            listAcpRegions = await dal_manager_1.dbManager.accessRegion.listAccessControlPointRegions(organizationId, accessControlPointBasic.id);
            if (isAcpStayOpen(accessControlPointBasic)) {
                accessLog.r = app_enums_1.enums.AccessLogReason.StateOpened;
                accessLog.s = true;
                return accessLog;
            }
            else if (isAcpDisabled(accessControlPointBasic)) {
                accessLog.r = app_enums_1.enums.AccessLogReason.StateDisabled;
                accessLog.s = false;
                return accessLog;
            }
        }
        else {
            accessLog.r = app_enums_1.enums.AccessLogReason.UnknownAccessPoint;
            accessLog.s = false;
            return accessLog;
        }
        const credential = await dal_manager_1.dbManager.accessIdentity.getUserOrganizationCredential(organizationId, accessLog.c, app_enums_1.enums.CredentialType.MiFare);
        if (!credential) {
            accessLog.r = app_enums_1.enums.AccessLogReason.UnknownCredentialOwner;
            accessLog.s = false;
            return accessLog;
        }
        const user = await dal_manager_1.dbManager.accessIdentity.findMemberByCredential(organizationId, false, accessLog.c, app_enums_1.enums.CredentialType.MiFare);
        if (isUserExistAndMember(user)) {
            const { isDisabled } = await dal_manager_1.dbManager.accessIdentity.getUserEmailAddressAndIsDisabled(organizationId, user.member.id);
            accessLog.on = user.member.fullname;
            accessLog.o = user.member.id;
            accessLog.ci = credential.id;
            accessLog.ul = user.roleId;
            accessLog.i = (await dal_manager_1.dbManager.accessIdentity.getUserAccessRights({ organizationId, userId: user.member.id }))?.find((r) => r.accessControlPointId === deviceLog.a)?.id;
            if (isDisabled) {
                accessLog.r = app_enums_1.enums.AccessLogReason.UserDisabled;
                accessLog.s = false;
                return accessLog;
            }
            let userForbiddances = await dal_manager_1.dbManager.accessIdentity.getUserForbiddances(user.member.id, organizationId);
            if (isUserForbidden(userForbiddances, listAcpRegions, accessLog.ci)) {
                accessLog.r = app_enums_1.enums.AccessLogReason.Forbidden;
                accessLog.s = false;
                return accessLog;
            }
            let acpIdsUserHasAccessRight = await dal_manager_1.dbManager.accessAccessControlPoint.getAcpIdsUserHasAccessRight(organizationId, user.member.id, { access: true });
            if (!doesUserHasAccessRight(acpIdsUserHasAccessRight, accessLog)) {
                accessLog.r = app_enums_1.enums.AccessLogReason.HasNotRight;
                accessLog.s = false;
                return accessLog;
            }
            if (isCredentialExpired(credential)) {
                accessLog.r = app_enums_1.enums.AccessLogReason.CredentialExpired;
                accessLog.s = false;
                return accessLog;
            }
            accessLog.r === null ? (accessLog.r = app_enums_1.enums.AccessLogReason.Success) : null;
            accessLog.s === null ? (accessLog.s = true) : null;
            accessLog.cx = [{ i: credential.id, d: credential.data, t: credential.type }];
        }
        else {
            accessLog.r = app_enums_1.enums.AccessLogReason.UnknownCredentialOwner;
            accessLog.s = false;
            return accessLog;
        }
        let userOrganizationAccessRules = await dal_manager_1.dbManager.accessUserGroup.getUserOrganizationAccessRules(organizationId, user.member.id);
        for (let ruleSet of userOrganizationAccessRules) {
            if (!ruleSet.isActive) {
                continue;
            }
            for (let rule of ruleSet.rules) {
                if (rule.direction === accessLog.d || rule.direction === app_enums_1.enums.AccessDirection.All) {
                    if (rule.type === app_enums_1.enums.AccessRuleType.CountBased) {
                        let parameters = rule.parameters;
                        let userAccess = (await dal_manager_1.dbManager.accessLog.getUserDailyAccessLogs(organizationId, accessLog.o)).filter((log) => {
                            return log.rg.some((r) => listAcpRegions.map((r) => r.id).includes(r.i));
                        });
                        let userAccessCount = userAccess ? userAccess.length : 0;
                        if (userAccessCount < parameters.maximumAccessCount) {
                            if (rule.acceptStatus === app_enums_1.enums.AccessRuleAcceptStatus.Accept) {
                            }
                            else if (rule.acceptStatus === app_enums_1.enums.AccessRuleAcceptStatus.Reject) {
                                accessLog.r = app_enums_1.enums.AccessLogReason.RuleReject;
                                accessLog.s = false;
                                return accessLog;
                            }
                        }
                    }
                    else if (rule.type === app_enums_1.enums.AccessRuleType.TimeBased) {
                        let parameters = rule.parameters;
                        if (parameters.days ? (parameters.days.includes(new Date().getDay()) ? true : false) : true) {
                            let startTime = parameters.startTime.split(":").map((s) => parseInt(s));
                            let endTime = parameters.endTime.split(":").map((s) => parseInt(s));
                            let sTime = new Date().setHours(startTime[0], startTime[1], 0, 0);
                            let eTime = new Date().setHours(endTime[0], endTime[1], 0, 0);
                            if (now >= sTime && now < eTime) {
                                if (rule.acceptStatus === app_enums_1.enums.AccessRuleAcceptStatus.Reject) {
                                    accessLog.r = app_enums_1.enums.AccessLogReason.RuleReject;
                                    accessLog.s = false;
                                    return accessLog;
                                }
                            }
                        }
                    }
                }
            }
        }
        for (let region of listAcpRegions) {
            let currentState = await dal_manager_1.dbManager.accessRegion.getUserRegionLastState(organizationId, user.member.id, region.id);
            if (currentState) {
                if (region.antiPassback) {
                    if (currentState.state === accessLog.d.valueOf()) {
                        accessLog.r = app_enums_1.enums.AccessLogReason.AntiPassbackReject;
                        accessLog.s = false;
                        return accessLog;
                    }
                    if ((currentState.state === app_enums_1.enums.AntiPassbackState.In && currentState.exitLockExpirationUtc && new Date(currentState.exitLockExpirationUtc).getTime() > now) ||
                        (currentState.state === app_enums_1.enums.AntiPassbackState.Out &&
                            currentState.entranceLockExpirationUtc &&
                            new Date(currentState.entranceLockExpirationUtc).getTime() > now)) {
                        accessLog.r = app_enums_1.enums.AccessLogReason.AntiPassbackTimeOutReject;
                        accessLog.s = false;
                        return accessLog;
                    }
                }
            }
            else {
                currentState = {
                    regionId: region.id,
                    actionUtc: new Date(accessLog.u),
                    state: accessLog.d === app_enums_1.enums.libEnumsV2.AccessDirection.All
                        ? restapi_1.AntiPassbackState.Unknown
                        : accessLog.d === app_enums_1.enums.libEnumsV2.AccessDirection.Entrance
                            ? restapi_1.AntiPassbackState.In
                            : restapi_1.AntiPassbackState.Out,
                    userId: accessLog.o,
                    entranceLockExpirationUtc: !region.antiPassback || accessLog.d === app_enums_1.enums.libEnumsV2.AccessDirection.All || accessLog.d === app_enums_1.enums.libEnumsV2.AccessDirection.Exit
                        ? undefined
                        : new Date(new Date(accessLog.u).setSeconds(new Date(accessLog.u).getSeconds() + region.antiPassbackLockDuration)),
                    exitLockExpirationUtc: !region.antiPassback || accessLog.d === app_enums_1.enums.libEnumsV2.AccessDirection.All || accessLog.d === app_enums_1.enums.libEnumsV2.AccessDirection.Entrance
                        ? undefined
                        : new Date(new Date(accessLog.u).setSeconds(new Date(accessLog.u).getSeconds() + region.antiPassbackLockDuration)),
                };
            }
            accessLog.rg.push({
                i: currentState.regionId,
                s: accessLog.d.valueOf(),
                ee: currentState.entranceLockExpirationUtc ? currentState.entranceLockExpirationUtc.toString() : undefined,
                xe: currentState.exitLockExpirationUtc ? currentState.exitLockExpirationUtc.toString() : undefined,
            });
        }
    }
    return accessLog;
}
exports.updateLogFromMobileDevice = updateLogFromMobileDevice;
const notifyUsersAboutAperioBatteryStatus = async (organizationId, deviceBatteryState) => {
    if (deviceBatteryState && deviceBatteryState.batteryLevel) {
        const { deviceInfo, hubDeviceId } = await dal_manager_1.dbManager.systemTransaction(async (trx) => {
            const deviceInfo = await dal_manager_1.dbManager.accessRedisCache.getDeviceNotificationInfoCache({ organizationId, deviceId: deviceBatteryState.lockDeviceId, trx });
            const hubDeviceId = (await trx.query(`SELECT "hubDeviceId" FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.aperioLocks}" 
						WHERE "lockDeviceId" = $1`, [deviceBatteryState.lockDeviceId])).rows[0]?.hubDeviceId;
            return { deviceInfo, hubDeviceId };
        });
        if (deviceInfo?.b === dal_constants_1.DalConstants.DeviceBrand.AssaAbloy && dal_constants_1.DalConstants.batteryPoweredDeviceModels.includes(deviceInfo?.m)) {
            try {
                await (0, business_notification_1.sendAssaAbloyBatteryStatusChangedNotification)(organizationId, {
                    hubDeviceId,
                    lockDeviceId: deviceBatteryState.lockDeviceId,
                    deviceName: deviceInfo.n,
                    state: deviceBatteryState.batteryLevel,
                    timestamp: deviceBatteryState.timestamp,
                });
            }
            catch (error) {
                app_logs_1.logger.error(`Error while sending battery level notification to notification service for device ${deviceBatteryState.lockDeviceId}`);
            }
        }
    }
};
exports.notifyUsersAboutAperioBatteryStatus = notifyUsersAboutAperioBatteryStatus;
const isAcpStayOpen = (accessControlPointBasic) => accessControlPointBasic.states && accessControlPointBasic.states.find((s) => s.state === app_enums_1.enums.AccessControlPointState.StayOpened);
const isAcpDisabled = (accessControlPointBasic) => accessControlPointBasic.states && accessControlPointBasic.states.find((s) => s.state === app_enums_1.enums.AccessControlPointState.Disabled);
const areDataProvidedCorrectly = (accessLog) => accessLog.c && accessLog.c.length > 0 && accessLog.a;
const isUserExistAndMember = (user) => user && user.member && user.existance === dal_constants_1.DalConstants.CredentialExistance.Member;
const isUserForbidden = (userForbiddances, listAcpRegions, credentialId) => {
    if (!userForbiddances || userForbiddances.length === 0) {
        return false;
    }
    for (const { regionId, credentialIds } of userForbiddances) {
        if (!regionId && !credentialIds) {
            return true;
        }
        else if (!regionId && credentialIds?.includes(credentialId)) {
            return true;
        }
        else if (!credentialIds && listAcpRegions?.map((r) => r.id).includes(regionId)) {
            return true;
        }
        else if (listAcpRegions?.map((r) => r.id).includes(regionId) && credentialIds?.includes(credentialId)) {
            return true;
        }
    }
    return false;
};
const doesUserHasAccessRight = (acpIdsUserHasAccessRight, accessLog) => acpIdsUserHasAccessRight && acpIdsUserHasAccessRight.includes(accessLog.a);
const isCredentialExpired = (credential) => new Date(credential.expiresOnISO) < new Date();
