"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.AccessLogsImportUploadExcel = void 0;
const exceljs_1 = __importDefault(require("exceljs"));
const fs_1 = __importDefault(require("fs"));
const i18n_1 = __importDefault(require("i18n"));
const luxon_1 = require("luxon");
const path_1 = __importDefault(require("path"));
const uuid_1 = __importDefault(require("uuid"));
const app_config_1 = require("../app.config");
const app_enums_1 = require("../app.enums");
const dal_memcache_1 = require("../dal/access/dal.memcache");
const dal_manager_1 = require("../dal/dal.manager");
const messageBroker_notification_pub_1 = require("../messageBroker/messageBroker.notification.pub");
const messageBroker_server_to_pgmaintenance_pub_1 = require("../messageBroker/messageBroker.server-to-pgmaintenance.pub");
const app_logs_1 = require("./../app.logs");
const pacs_models_1 = require("./pacs2/pacs.models");
class AccessLogsImportUploadExcel {
    constructor(organizationId, userId, locale) {
        this._uniqueIdOldestDateMap = new Map();
        this._usersWithLogsForToday = [];
        this.pacsCalculationMaxDay = 60;
        this._fileId = uuid_1.default.v4();
        this._requestedUserId = userId;
        this._organizationId = organizationId;
        this._locale = locale;
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".xlsx");
        if (fs_1.default.existsSync(this._filePath)) {
            fs_1.default.unlinkSync(this._filePath);
        }
    }
    async loadExcel(data) {
        this._wbReader = new exceljs_1.default.Workbook();
        this._uniqueIdOldestDateMap.clear();
        this._userIds = [];
        this._startDates = [];
        this._months = new Set();
        await this._wbReader.xlsx.load(data);
    }
    async read() {
        this._organizationName = (await dal_manager_1.dbManager.accessOrganization.getOrganizationBasic(this._organizationId)).name;
        let { caption: requestedUserCaption } = await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId: this._organizationId, userId: this._requestedUserId });
        if (!requestedUserCaption) {
            throw new Error("Not found");
        }
        this._requestedUserCaption = requestedUserCaption[0].text[0];
        return await this.validateExcel();
    }
    async uploadExcel(trx, defaultAcpId, userId, pacsEnabled) {
        try {
            let partitionRes = await messageBroker_server_to_pgmaintenance_pub_1.amqpServerToPgMaintenancePub.sendBackwardsPartitionRequest([...this._months], this._organizationId);
            if (partitionRes?.["isSuccess"] === false) {
                throw Error("Related partitions could not be created");
            }
            await dal_manager_1.dbManager.accessLog.excelUploadManualLog(this._organizationId, this._requestedUserId, this._users, this._wbReader.getWorksheet(1), this._accessControlPoints, trx, defaultAcpId, this._dateTimeCombined);
            let event = await this.createAccessLogExcelUploadedNotification(userId, true, trx);
            (0, messageBroker_notification_pub_1.publishToNotificationService)(event);
            if (pacsEnabled && this._oldestDate.diffNow("days").days > -this.pacsCalculationMaxDay) {
                this.pacsCalculation(trx);
            }
        }
        catch (err) {
            app_logs_1.logger.error(err);
            let event = await this.createAccessLogExcelUploadedNotification(userId, false, trx);
            (0, messageBroker_notification_pub_1.publishToNotificationService)(event);
        }
    }
    async createAccessLogExcelUploadedNotification(userId, isSuccessful, trx) {
        app_logs_1.logger.debug(`sending notification user[${userId}]`);
        const notificationId = await (0, dal_memcache_1.getCacheUniqueNotificationIdOfOrganization)(this._organizationId, app_enums_1.enums.NotificationType.AccessLogExcelImportResultNotification, trx);
        const instanceData = {
            s: isSuccessful,
            u: userId,
            t: new Date().toString(),
            receiverUserIds: [userId],
        };
        const notificationEventId = await dal_manager_1.dbManager.accessNotifications.addNotificationEvent({
            createdT: new Date(),
            notificationId,
            organizationId: this._organizationId,
            instanceData,
            trx: trx,
        });
        return {
            v: "1",
            n: 1,
            a: 0,
            i: notificationEventId,
            o: this._organizationId,
            d: instanceData,
        };
    }
    async pacsCalculation(trx) {
        this._uniqueIdOldestDateMap.forEach((value, key) => {
            this._userIds.push(this._users.find((user) => user.uniqueId == key).userId);
            this._startDates.push(value);
        });
        let userIdsWithLogForToday = new Set();
        this._usersWithLogsForToday.forEach((uniqueId) => {
            userIdsWithLogForToday.add(this._users.find((user) => user.uniqueId === uniqueId).userId);
        });
        dal_manager_1.dbManager.accessPacs2.insertRecalculateWorkQueueAfterAccessLogImport(this._organizationId, this._userIds, this._startDates, Array.from(userIdsWithLogForToday), trx);
    }
    async validateExcel() {
        await this.getAccessControlPoints();
        let errorMessage;
        let errorCells = [];
        let errorDates = [];
        let errorUsers = [];
        let uniqueIds = [];
        const sheet = this._wbReader.getWorksheet(1);
        const hiddenSheet = this._wbReader.getWorksheet("HiddenData");
        if (!hiddenSheet) {
            return {
                validationSuccess: false,
                errorMessage: i18n_1.default.__({ phrase: "ACCESS_LOGS_EXCEL.ERROR.MISSING_HIDDENSHEET", locale: this._locale }),
                errorCells,
                errorDates,
                errorUsers,
            };
        }
        if (!sheet) {
            return {
                validationSuccess: false,
                errorMessage: i18n_1.default.__({ phrase: "ACCESS_LOGS_EXCEL.ERROR.MISSING_DATASHEET", locale: this._locale }),
                errorCells,
                errorDates,
                errorUsers,
            };
        }
        this._dateTimeCombined = hiddenSheet.getCell("A1").text === "true";
        sheet.eachRow((row, rowNumber) => {
            let date;
            let time;
            let faultyDate = false;
            let uniqueId = row.getCell(1).value;
            row.eachCell({ includeEmpty: true }, (cell, colNumber) => {
                let cellValue = cell.text;
                if (rowNumber > 1 && colNumber === 1) {
                    uniqueIds.push(cellValue);
                }
                if (this._dateTimeCombined) {
                    if (rowNumber > 1 && colNumber === 2) {
                        try {
                            let dateTime = luxon_1.DateTime.fromJSDate(new Date(cell.value.toString()), { zone: "Etc/Utc" }).setZone("local", { keepLocalTime: true });
                            if (!dateTime.isValid) {
                                errorMessage = errorMessage ?? i18n_1.default.__({ phrase: "ACCESS_LOGS_EXCEL.ERROR.INCORRECT_DATETIME", locale: this._locale });
                                errorDates.push(`B${rowNumber}`);
                            }
                            if (dateTime > luxon_1.DateTime.now()) {
                                errorMessage = errorMessage ?? i18n_1.default.__({ phrase: "ACCESS_LOGS_EXCEL.ERROR.FUTURE_DATETIME", locale: this._locale });
                                errorDates.push(`B${rowNumber}`);
                            }
                            if (this._uniqueIdOldestDateMap.get(uniqueId)) {
                                if (dateTime < this._uniqueIdOldestDateMap.get(uniqueId)) {
                                    this._uniqueIdOldestDateMap.set(uniqueId, dateTime.toJSDate());
                                }
                            }
                            else {
                                this._uniqueIdOldestDateMap.set(uniqueId, dateTime.toJSDate());
                            }
                            if (this._oldestDate) {
                                if (dateTime < this._oldestDate) {
                                    this._oldestDate = dateTime;
                                }
                            }
                            else {
                                this._oldestDate = dateTime;
                            }
                            this._months.add(`${dateTime.year}${dateTime.month < 10 ? "0" + dateTime.month : dateTime.month}`);
                            if (dateTime.hasSame(luxon_1.DateTime.now(), "day")) {
                                this._usersWithLogsForToday.push(row.getCell(1).text);
                            }
                        }
                        catch (err) {
                            errorMessage = errorMessage ?? i18n_1.default.__({ phrase: "ACCESS_LOGS_EXCEL.ERROR.INCORRECT_DATETIME", locale: this._locale });
                            errorDates.push(`B${rowNumber}`);
                        }
                    }
                }
                else {
                    if (rowNumber > 1 && colNumber === 2) {
                        try {
                            date = luxon_1.DateTime.fromJSDate(cell.value, { zone: "Etc/Utc" });
                            if (!date) {
                                date = luxon_1.DateTime.fromISO(cell.text);
                            }
                            if (!date.isValid) {
                                errorMessage = errorMessage ?? i18n_1.default.__({ phrase: "ACCESS_LOGS_EXCEL.ERROR.INCORRECT_DATE", locale: this._locale });
                                errorDates.push(`B${rowNumber}`);
                            }
                        }
                        catch (err) {
                            faultyDate = true;
                            app_logs_1.logger.error(err);
                            errorMessage = errorMessage ?? i18n_1.default.__({ phrase: "ACCESS_LOGS_EXCEL.ERROR.INCORRECT_DATE", locale: this._locale });
                            errorDates.push(`B${rowNumber}`);
                        }
                    }
                    if (rowNumber > 1 && colNumber === 3 && !faultyDate) {
                        try {
                            time = luxon_1.DateTime.fromJSDate(cell.value, { zone: "Etc/UTC" });
                            if (!time) {
                                time = luxon_1.DateTime.fromISO(cell.text).toUTC();
                            }
                            if (!time.isValid) {
                                errorMessage = errorMessage ?? i18n_1.default.__({ phrase: "ACCESS_LOGS_EXCEL.ERROR.INCORRECT_TIME", locale: this._locale });
                                errorCells.push(`C${rowNumber}`);
                            }
                            else {
                                const combinedDateTime = time
                                    .set({
                                    year: date.toUTC().year,
                                    month: date.toUTC().month,
                                    day: date.toUTC().day,
                                })
                                    .setZone("local", { keepLocalTime: true });
                                if (combinedDateTime > luxon_1.DateTime.now()) {
                                    errorMessage = errorMessage ?? i18n_1.default.__({ phrase: "ACCESS_LOGS_EXCEL.ERROR.FUTURE_DATETIME", locale: this._locale });
                                    errorDates.push(`B${rowNumber}-C${rowNumber}`);
                                }
                                if (this._uniqueIdOldestDateMap.get(uniqueId)) {
                                    if (combinedDateTime < this._uniqueIdOldestDateMap.get(uniqueId)) {
                                        this._uniqueIdOldestDateMap.set(uniqueId, combinedDateTime.toJSDate());
                                    }
                                }
                                else {
                                    this._uniqueIdOldestDateMap.set(uniqueId, combinedDateTime.toJSDate());
                                }
                                if (this._oldestDate) {
                                    if (combinedDateTime < this._oldestDate) {
                                        this._oldestDate = combinedDateTime;
                                    }
                                }
                                else {
                                    this._oldestDate = combinedDateTime;
                                }
                                this._months.add(`${combinedDateTime.year}${combinedDateTime.month < 10 ? "0" + combinedDateTime.month : combinedDateTime.month}`);
                                if (combinedDateTime.hasSame(luxon_1.DateTime.now(), "day")) {
                                    this._usersWithLogsForToday.push(row.getCell(1).text);
                                }
                            }
                        }
                        catch (err) {
                            app_logs_1.logger.error(err);
                            errorMessage = errorMessage ?? i18n_1.default.__({ phrase: "ACCESS_LOGS_EXCEL.ERROR.INCORRECT_TIME", locale: this._locale });
                            errorCells.push(`C${rowNumber}`);
                        }
                    }
                }
                if (rowNumber > 1 && colNumber === 4 - +this._dateTimeCombined) {
                    if (!(cellValue === i18n_1.default.__({ phrase: "DIRECTIONTYPE.ENTRANCE", locale: this._locale }) ||
                        cellValue === i18n_1.default.__({ phrase: "DIRECTIONTYPE.EXIT", locale: this._locale }) ||
                        cellValue === "")) {
                        errorMessage = errorMessage ?? i18n_1.default.__({ phrase: "ACCESS_LOGS_EXCEL.ERROR.INCORRECT_DIRECTION", locale: this._locale });
                        if (this._dateTimeCombined) {
                            errorCells.push(`C${rowNumber}`);
                        }
                        else {
                            errorCells.push(`D${rowNumber}`);
                        }
                    }
                }
                if (rowNumber > 1 && colNumber === 5 - +this._dateTimeCombined) {
                    if (!this._accessControlPoints.some((acp) => acp.name.trim() === cellValue.trim())) {
                        errorMessage = errorMessage ?? i18n_1.default.__({ phrase: "ACCESS_LOGS_EXCEL.ERROR.UNKNOWN_ACP", locale: this._locale });
                        if (this._dateTimeCombined) {
                            errorCells.push(`D${rowNumber}`);
                        }
                        else {
                            errorCells.push(`E${rowNumber}`);
                        }
                    }
                }
            });
        });
        let userResponse = await this.checkUsersOnDb(uniqueIds);
        errorMessage = errorMessage?.length > 0 ? errorMessage : userResponse.errorMessage;
        if (userResponse.errorMessage?.length > 0) {
            userResponse.errorIndexes.forEach((index) => {
                errorUsers.push(`${this.getColumnLetter(1)}${index}`);
            });
        }
        return {
            validationSuccess: errorMessage?.length > 0 ? false : true,
            errorMessage,
            errorCells,
            errorDates,
            errorUsers,
        };
    }
    async getAccessControlPoints() {
        this._accessControlPoints = await dal_manager_1.dbManager.accessAccessControlPoint.listOrganizationAccessControlPoints(this._organizationId);
    }
    async checkUsersOnDb(uniqueIds) {
        let errorMessage;
        let errorIndexes = [];
        this._users = await dal_manager_1.dbManager.accessUser.listUserIdsFromUniqueIds(this._organizationId, uniqueIds);
        uniqueIds.forEach((id, idIndex) => {
            let index = this._users.findIndex((user) => user.uniqueId === id);
            if (!id) {
                errorMessage = i18n_1.default.__({ phrase: "ACCESS_LOGS_EXCEL.ERROR.MISSING_USER_ID", locale: this._locale });
                errorIndexes.push(idIndex + 2);
            }
            if (index < 0) {
                errorMessage = errorMessage?.length ? errorMessage : i18n_1.default.__({ phrase: "ACCESS_LOGS_EXCEL.ERROR.MISMATCHED_UNIQUE_ID", locale: this._locale });
                errorIndexes.push(idIndex + 2);
            }
        });
        return { errorMessage, errorIndexes };
    }
    async generateExcel(locale, dateTimeCombined) {
        this._organizationName = (await dal_manager_1.dbManager.accessOrganization.getOrganizationBasic(this._organizationId)).name;
        this.createWorkbook();
        this._sheetNames = {
            [pacs_models_1.PacsModels.AccessLogExcelSheetType.Input]: null,
        };
        await this.createSheets(locale, dateTimeCombined);
        for (let sheetName in this._sheetNames) {
            this._wbWriter.getWorksheet(this._sheetNames[sheetName]).commit();
        }
        await this._wbWriter.commit();
        return {
            filePath: this._filePath,
            filename: this._organizationName
                .toLocaleLowerCase()
                .replace(new RegExp(/[&\/\\#,+()$~%.'":*?<>{}`\r\n\t]/g), "")
                .replace(new RegExp(/[\s]+/g), "_")
                .replace(new RegExp(/ç/g), "c")
                .replace(new RegExp(/ğ/g), "g")
                .replace(new RegExp(/ı/g), "i")
                .replace(new RegExp(/ö/g), "o")
                .replace(new RegExp(/ş/g), "s")
                .replace(new RegExp(/ü/g), "u") +
                "_" +
                i18n_1.default.__({ phrase: "ACCESS_LOGS_EXCEL.FILENAME", locale: this._locale }) +
                ".xlsx",
        };
    }
    async createSheets(locale, dateTimeCombined) {
        let headers;
        if (dateTimeCombined) {
            headers = [
                i18n_1.default.__({ phrase: "ACCESS_LOGS_EXCEL.EXCEL_HEADERS.UNIQUE_ID", locale: locale }),
                i18n_1.default.__({ phrase: "ACCESS_LOGS_EXCEL.EXCEL_HEADERS.DATETIME", locale: locale }),
                i18n_1.default.__({ phrase: "ACCESS_LOGS_EXCEL.EXCEL_HEADERS.DIRECTION", locale: locale }),
                i18n_1.default.__({ phrase: "ACCESS_LOGS_EXCEL.EXCEL_HEADERS.ACCESS_CONTROL_POINT_NAME", locale: locale }),
            ];
        }
        else {
            headers = [
                i18n_1.default.__({ phrase: "ACCESS_LOGS_EXCEL.EXCEL_HEADERS.UNIQUE_ID", locale: locale }),
                i18n_1.default.__({ phrase: "ACCESS_LOGS_EXCEL.EXCEL_HEADERS.DATE", locale: locale }),
                i18n_1.default.__({ phrase: "ACCESS_LOGS_EXCEL.EXCEL_HEADERS.TIME", locale: locale }),
                i18n_1.default.__({ phrase: "ACCESS_LOGS_EXCEL.EXCEL_HEADERS.DIRECTION", locale: locale }),
                i18n_1.default.__({ phrase: "ACCESS_LOGS_EXCEL.EXCEL_HEADERS.ACCESS_CONTROL_POINT_NAME", locale: locale }),
            ];
        }
        this._headers = headers;
        for (let sheetName in this._sheetNames) {
            this.addSheetToWorkbook(sheetName);
            const hiddenSheet = this._wbWriter.addWorksheet("HiddenData");
            hiddenSheet.getCell("A1").value = dateTimeCombined;
            hiddenSheet.state = "veryHidden";
            this._wbWriter.getWorksheet(this._sheetNames[sheetName]).properties.defaultRowHeight = 65;
            this._wbWriter.getWorksheet(this._sheetNames[sheetName]).properties.defaultColWidth = 100;
            const headerRow = this._wbWriter.getWorksheet(this._sheetNames[sheetName]).addRow(this._headers);
            this._wbWriter.getWorksheet(this._sheetNames[sheetName]).columns.forEach((col) => {
                col.width = 30;
                col.eachCell((cell) => {
                    cell.numFmt = "@";
                });
            });
            this._wbWriter.getWorksheet(this._sheetNames[sheetName]).getRow(1);
            headerRow.eachCell((cell) => {
                cell.fill = {
                    type: "pattern",
                    pattern: "solid",
                    fgColor: { argb: "FFFF00" },
                };
                cell.font = {
                    bold: true,
                };
                cell.border = {
                    top: { style: "thin", color: { argb: "FF000000" } },
                    left: { style: "thin", color: { argb: "FF000000" } },
                    bottom: { style: "thin", color: { argb: "FF000000" } },
                    right: { style: "thin", color: { argb: "FF000000" } },
                };
                cell.alignment = {
                    horizontal: "center",
                };
            });
            let directionRange;
            let directionFormula;
            if (dateTimeCombined) {
                directionRange = `$C$2:$C$1048576`;
                directionFormula = `=AND(C2<>"Giriş", C2<>"Çıkış", C2<>"")`;
            }
            else {
                directionRange = `$D$2:$D$1048576`;
                directionFormula = `=AND(D2<>"Giriş", D2<>"Çıkış", D2<>"")`;
            }
            this._wbWriter.getWorksheet(this._sheetNames[sheetName]).addConditionalFormatting({
                ref: directionRange,
                rules: [
                    {
                        type: "expression",
                        formulae: [`${directionFormula}`],
                        priority: 3,
                        style: {
                            fill: {
                                type: "pattern",
                                pattern: "solid",
                                bgColor: { argb: "FFFF0000" },
                            },
                        },
                    },
                ],
            });
        }
    }
    getColumnLetter(col) {
        let letter = "";
        while (col > 0) {
            const temp = (col - 1) % 26;
            letter = String.fromCharCode(temp + 65) + letter;
            col = Math.floor((col - temp) / 26);
        }
        return letter;
    }
    addSheetToWorkbook(key) {
        let translatedName = i18n_1.default.__({ phrase: `ACCESS_LOGS_EXCEL.SHEET_NAME`, locale: this._locale });
        let ws = this._wbWriter.addWorksheet(translatedName, {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._sheetNames[key] = ws.id;
    }
    createWorkbook() {
        const options = {
            filename: this._filePath,
            useSharedStrings: true,
            useStyles: true,
        };
        this._wbWriter = new exceljs_1.default.stream.xlsx.WorkbookWriter(options);
        this._wbWriter.creator = this._requestedUserCaption;
        this._wbWriter.company = this._organizationName;
    }
}
exports.AccessLogsImportUploadExcel = AccessLogsImportUploadExcel;
