"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.logger = exports.appLogger = exports.initHttpLogger = exports.httpLogger = void 0;
const path_1 = __importDefault(require("path"));
const winston_1 = __importDefault(require("winston"));
const app_config_1 = require("./app.config");
const express_winston_1 = __importDefault(require("express-winston"));
const os_1 = __importDefault(require("os"));
class AppLogger {
    constructor() {
        this.logger = winston_1.default.createLogger({
            exitOnError: false,
            format: winston_1.default.format.combine(winston_1.default.format.splat(), winston_1.default.format.timestamp({
                format: "YYYY-MM-DD HH:mm:ss",
            }), winston_1.default.format.printf((error) => {
                return `[${error.timestamp}][${error.level}]${error.message}`;
            })),
            transports: [
                new winston_1.default.transports.Console({
                    level: "debug",
                }),
            ],
        });
        exports.logger = this.logger;
    }
    init(logDirectory) {
        this.logger.add(new (require("winston-daily-rotate-file"))({
            filename: path_1.default.join(logDirectory, "%DATE%.log"),
            timestamp: () => new Date().toLocaleTimeString(),
            datePattern: "YYYY-MM-DD",
            prepend: true,
            level: process.env.LOG_LEVEL ?? app_config_1.appConfig.log.app.level,
            handleExceptions: true,
            humanReadableUnhandledException: true,
            zippedArchive: process.env.LOG_LEVEL ?? app_config_1.appConfig.log.app.level,
            keepLastDayCount: process.env.LOG_LAST_DAY_COUNT ? parseInt(process.env.LOG_LAST_DAY_COUNT) : app_config_1.appConfig.log.app.keepLastDayCount,
        }));
        exports.logger = this.logger;
    }
}
const httpFormat = winston_1.default.format.combine(winston_1.default.format.timestamp({
    format: "YYYY-MM-DD HH:mm:ss",
}), winston_1.default.format.printf((log) => {
    const meta = log.meta;
    const requestMethod = meta?.req?.method;
    const requestUrl = meta?.req?.url;
    const statusCode = meta?.res?.statusCode;
    const responseTime = meta?.responseTime;
    return `[${log.timestamp}][${log.level}]` + (meta ? `[${requestMethod} ${requestUrl} ${statusCode} ${responseTime}ms]${os_1.default.EOL}${JSON.stringify(meta, null, 4)}` : log.message);
}));
const defaultTransports = [
    new winston_1.default.transports.Console({
        level: "debug",
    }),
];
exports.httpLogger = express_winston_1.default.logger({
    format: httpFormat,
    transports: defaultTransports,
});
function initHttpLogger(directory, level, keepLastDayCount, zippedArchive) {
    exports.httpLogger = express_winston_1.default.logger({
        meta: true,
        format: httpFormat,
        transports: [
            new winston_1.default.transports.Console({
                level: level,
            }),
            new (require("winston-daily-rotate-file"))({
                filename: path_1.default.join(directory, "%DATE%.http.log"),
                timestamp: () => new Date().toLocaleTimeString(),
                datePattern: "YYYY-MM-DD",
                prepend: true,
                level: level,
                handleExceptions: true,
                humanReadableUnhandledException: true,
                zippedArchive: zippedArchive,
                maxFiles: keepLastDayCount,
            }),
        ],
    });
}
exports.initHttpLogger = initHttpLogger;
exports.appLogger = new AppLogger();
exports.logger = exports.appLogger.logger;
