"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.App = void 0;
const elastic_apm_node_1 = __importDefault(require("elastic-apm-node"));
if (process.env.ELASTIC_APM_SERVER_URL &&
    process.env.ELASTIC_APM_SECRET_TOKEN &&
    process.env.ELASTIC_APM_PREFIX &&
    ((process.env.ELASTIC_APM_VERIFY_SERVER_CERT === "true" && process.env.ELASTIC_APM_SERVER_CA_CERT_FILE) || process.env.ELASTIC_APM_VERIFY_SERVER_CERT === "false")) {
    elastic_apm_node_1.default.start({
        serviceName: `${process.env.ELASTIC_APM_PREFIX}-webapp`,
        serverUrl: process.env.ELASTIC_APM_SERVER_URL,
        secretToken: process.env.ELASTIC_APM_SECRET_TOKEN,
        environment: "production",
        serverCaCertFile: process.env.ELASTIC_APM_SERVER_CA_CERT_FILE ?? null,
        verifyServerCert: process.env.ELASTIC_APM_VERIFY_SERVER_CERT === "true",
    });
}
else {
    console.warn(`Elastic APM is not enabled. Please set ELASTIC_APM_SERVER_URL, ELASTIC_APM_SECRET_TOKEN, ELASTIC_APM_PREFIX, ELASTIC_APM_VERIFY_SERVER_CERT, and ELASTIC_APM_SERVER_CA_CERT_FILE environment variables.`);
}
const cluster_1 = __importDefault(require("cluster"));
const os_1 = __importDefault(require("os"));
const jsonwebtoken_1 = __importDefault(require("jsonwebtoken"));
const fs_1 = __importDefault(require("fs"));
const path_1 = __importDefault(require("path"));
const app_auth_manager_1 = require("./app.auth.manager");
const app_config_1 = require("./app.config");
const app_constants_1 = require("./app.constants");
const app_extensions_1 = require("./app.extensions");
const app_httpserver_1 = require("./app.httpserver");
const app_logs_1 = require("./app.logs");
const app_swagger_1 = require("./app.swagger");
const app_timeServer_1 = require("./app.timeServer");
const business_hooks_1 = require("./business/business.hooks");
const dal_manager_1 = require("./dal/dal.manager");
const messageBroker_event_sub_1 = require("./messageBroker/messageBroker.event.sub");
const messageBroker_manager_1 = require("./messageBroker/messageBroker.manager");
const app_rate_limiter_1 = require("./app.rate-limiter");
const dal_utils_1 = require("./dal/dal.utils");
const business_notification_1 = require("./business/business.notification");
class App {
    static main(args) {
        let app = new App();
        app.start(args);
    }
    constructor() {
        this.listenDbNotifications = async () => {
            const client = await (0, dal_utils_1.listenClientWrapperWithRetry)("PG-APP-WEB", dal_manager_1.dbManager.poolMain);
            client.on("notification", async (msg) => {
                switch (msg.channel) {
                    case "sms_balance_notification":
                        {
                            const data = JSON.parse(msg.payload);
                            (0, business_notification_1.handleSMSBalanceNotifications)({ organizationId: data.organizationId, threshold: data.threshold });
                        }
                        break;
                    default:
                        return;
                }
            });
            client.on("error", (err) => {
                app_logs_1.logger.error("Error on PG-APP-WEB listener: " + err);
            });
            client.on("end", () => {
                app_logs_1.logger.info("pg listener for PG-APP-WEB ended, restarting...");
                client.release();
                this.listenDbNotifications();
            });
            await Promise.all([client.query("LISTEN sms_balance_notification")]);
            app_logs_1.logger.info("LISTEN PG-APP-WEB connections started.");
        };
    }
    async start(args) {
        try {
            await app_config_1.appConfig.init(app_constants_1.ServiceNames.WebApplication);
            app_logs_1.appLogger.init(app_config_1.appConfig.appLogDirectory);
            if (cluster_1.default.isMaster) {
                await this.startMaster();
            }
            else {
                await this.startChild();
            }
            app_logs_1.logger.info("armon-web-app is started");
        }
        catch (error) {
            app_logs_1.logger.error(error || error.message);
            process.exit(1);
        }
    }
    async startMaster() {
        try {
            let readyWorkerCount = 0;
            app_logs_1.logger.info("Extensions are initializing...");
            app_extensions_1.appExtensions.init();
            app_logs_1.logger.info("DBs are initializing...");
            await dal_manager_1.dbManager.init({
                main: app_config_1.appConfig.db.main,
                log: app_config_1.appConfig.db.log,
                environment: app_config_1.appConfig.nodeEnv,
                logDirectory: app_config_1.appConfig.dbLogDirectory,
                redis: app_config_1.appConfig.db.redis,
                targetMigration: app_config_1.appConfig.db.targetMigration ? app_config_1.appConfig.db.targetMigration : null,
            }, false, false, true, false);
            if (!app_config_1.appConfig.disableMessageQueue) {
                app_logs_1.logger.info("Message queues are initializing...");
                await messageBroker_manager_1.messageBrokerManager.initForMaster();
            }
            app_logs_1.logger.info("Hooks are initializing...");
            business_hooks_1.armonHookManager.init();
            app_logs_1.logger.info("Http server is initializing...");
            app_httpserver_1.httpServer.init();
            app_httpserver_1.httpServer.app.use(App.loggerMiddleware);
            app_logs_1.logger.info("Swagger is initializing...");
            let swaggerPromise = new Promise((resolve, reject) => {
                (0, app_swagger_1.initSwagger)(app_httpserver_1.httpServer.app, async (err) => {
                    if (err) {
                        return reject(err);
                    }
                    resolve(null);
                });
            });
            await swaggerPromise;
            if (app_config_1.appConfig.ntp) {
                app_logs_1.logger.info("Ntp is starting...");
                (0, app_timeServer_1.startTimeServer)(app_config_1.appConfig.ntp.port, app_config_1.appConfig.ntp.remote.serverDomain, app_config_1.appConfig.ntp.remote.serverPort);
            }
            cluster_1.default.on("exit", (worker, code, signal) => {
                app_logs_1.logger.error("child process %d died (%s). restarting...", worker.process.pid, signal || code);
                cluster_1.default.fork();
            });
            let childCountInfoLog = `There are #${os_1.default.cpus().length} cpus`;
            if (app_config_1.appConfig.maxCpuCore > 0) {
                childCountInfoLog += `but max available cpu count config is set as ${app_config_1.appConfig.maxCpuCore}`;
            }
            let childProcessCount = app_config_1.appConfig.maxCpuCore > 0 ? app_config_1.appConfig.maxCpuCore : os_1.default.cpus().length;
            childCountInfoLog += `. So there will be ${childProcessCount} child process will be ran.`;
            app_logs_1.logger.info(childCountInfoLog);
            for (let i = 0; i < childProcessCount; i++) {
                const worker = cluster_1.default.fork();
                worker.on("message", ((message, handle) => {
                    if (message.msg === "READY") {
                        readyWorkerCount++;
                        if (readyWorkerCount === childProcessCount) {
                            messageBroker_event_sub_1.amqpEventSub.updateDeviceConnectionStatusesAndNotifyDevices();
                        }
                    }
                }).bind(this));
            }
            if (!fs_1.default.existsSync(path_1.default.join(app_config_1.appConfig.configDirectory, "openvpn-client.conf"))) {
                fs_1.default.copyFileSync(path_1.default.join(app_config_1.appConfig.assetsDirectory, "openvpn", "openvpn-client.conf"), path_1.default.join(app_config_1.appConfig.configDirectory, "openvpn-client.conf"));
            }
            if (!fs_1.default.existsSync(path_1.default.join(app_config_1.appConfig.configDirectory, "openvpn-server.conf"))) {
                fs_1.default.copyFileSync(path_1.default.join(app_config_1.appConfig.assetsDirectory, "openvpn", "openvpn-server.conf"), path_1.default.join(app_config_1.appConfig.configDirectory, "openvpn-server.conf"));
            }
            await this.listenDbNotifications();
        }
        catch (error) {
            app_logs_1.logger.error("Error while starting app. %s", error || error.message);
            process.exit(1);
        }
        app_logs_1.logger.info("Master process is ready");
    }
    async startChild() {
        try {
            app_extensions_1.appExtensions.init();
            try {
                await dal_manager_1.dbManager.init({
                    main: app_config_1.appConfig.db.main,
                    log: app_config_1.appConfig.db.log,
                    environment: app_config_1.appConfig.nodeEnv,
                    logDirectory: app_config_1.appConfig.dbLogDirectory,
                    redis: app_config_1.appConfig.db.redis,
                }, false, false, true, false);
            }
            catch (err) {
                app_logs_1.logger.error(`Error while trying to initialize database connection for child process.`);
                app_logs_1.logger.error(err);
                app_logs_1.logger.error(`Terminating process due to database unavailability`);
                process.exit(1);
            }
            if (!app_config_1.appConfig.disableMessageQueue) {
                await messageBroker_manager_1.messageBrokerManager.initForChild();
            }
            app_httpserver_1.httpServer.init();
            app_httpserver_1.httpServer.app.use(App.loggerMiddleware);
            app_httpserver_1.httpServer.app.use("/u", app_rate_limiter_1.rateLimiterMiddleware);
            app_httpserver_1.httpServer.app.use("/v", app_rate_limiter_1.rateLimiterMiddleware);
            app_httpserver_1.httpServer.app.use("/p", app_rate_limiter_1.rateLimiterMiddleware);
            app_httpserver_1.httpServer.app.use("/s", app_rate_limiter_1.rateLimiterMiddleware);
            let swaggerPromise = new Promise((resolve, reject) => {
                (0, app_swagger_1.initSwagger)(app_httpserver_1.httpServer.app, async (err) => {
                    if (err) {
                        return reject(err);
                    }
                    resolve(null);
                });
            });
            app_logs_1.logger.info("Hooks are initializing...");
            business_hooks_1.armonHookManager.init();
            await swaggerPromise;
            await app_auth_manager_1.authManager.init(app_httpserver_1.httpServer.app);
            await app_httpserver_1.httpServer.startDeviceServer();
            await app_httpserver_1.httpServer.startApiServer();
            process.send({ msg: "READY" });
        }
        catch (error) {
            app_logs_1.logger.error("Error while starting child process %j", error);
            return;
        }
        app_logs_1.logger.info("Child process %s is ready", cluster_1.default.worker.id);
    }
}
exports.App = App;
App.loggerMiddleware = (req, res, next) => {
    if (req.method !== "OPTIONS") {
        const startHrtime = process.hrtime();
        res.on("finish", () => {
            const elapsedHrTime = process.hrtime(startHrtime);
            const elapsedTimeInMillis = elapsedHrTime[0] * 1000 + elapsedHrTime[1] / 1e6;
            let userId = null;
            if (req.rawHeaders.includes("Authorization") || req.rawHeaders.includes("authorization")) {
                let authHeader = req.rawHeaders[req.rawHeaders.indexOf("Authorization") + 1]?.split(" ")[1];
                if (!authHeader) {
                    authHeader = req.rawHeaders[req.rawHeaders.indexOf("authorization") + 1]?.split(" ")[1];
                }
                userId = jsonwebtoken_1.default.decode(authHeader)?.u;
            }
            else {
                userId = "Not Authorized Yet";
            }
            const logObject = {
                remoteAddress: req.ip,
                requestMethod: req.method,
                userAgent: req.header(app_constants_1.ArmonHeaders.UserAgent),
                clientUniqueId: req.header(app_constants_1.ArmonHeaders.MobileInstallId) ?? req.header(app_constants_1.ArmonHeaders.UserAgentId),
                sessionID: req.sessionID,
                url: req.url,
                requestBody: req.body,
                requesterUserId: userId,
                responseLength: res.getHeader("Content-Length"),
                responseStatus: res.statusCode,
                responseTime: `${elapsedTimeInMillis.toFixed(2)} ms`,
            };
            const checkSensitiveUrls = [
                "/usernamepass",
                "/identity/changepassword",
                "/changemypassword",
                "/identity/add",
                "/identity/update",
                "/camera/upsert",
                "/organization/settings",
                "/verifyToken",
            ];
            if (checkSensitiveUrls.find((f) => logObject.url.startsWith(f))) {
                if (logObject.requestBody.password) {
                    logObject.requestBody.password = "********";
                }
                if (logObject.requestBody.userAccount?.password) {
                    logObject.requestBody.userAccount.password = "********";
                }
                if (logObject.requestBody.notification?.smtpSettings?.config?.auth?.pass) {
                    logObject.requestBody.notification.smtpSettings.config.auth.pass = "********";
                }
                if (logObject.requestBody.newPassword || logObject.requestBody.oldPassword) {
                    logObject.requestBody.newPassword = "********";
                    logObject.requestBody.oldPassword = "********";
                }
            }
            else if (logObject.url.search("/visit/newvisit") && logObject.requestBody.visitFields?.find((f) => f.name === "cameraSnapshot")) {
                const index = logObject.requestBody.visitFields.findIndex((f) => f.name === "cameraSnapshot");
                logObject.requestBody.visitFields[index].value = "REDACTED";
            }
            dal_manager_1.dbManager.accessLog.addAuditLog(logObject);
        });
    }
    next();
};
App.main(process.argv);
process.on("uncaughtException", (exception) => {
    console.log("!!!!!!!!!##############uncaughtException!!!!!!!!!##############");
    console.log(exception);
});
process.on("unhandledRejection", (reason, p) => {
    console.log("!!!!!!!!!##############unhandledRejection!!!!!!!!!##############");
    console.log("Unhandled Rejection at: Promise ", p, " reason: ", reason);
});
