"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.appConfig = exports.AppConfigModels = void 0;
const fs_1 = __importDefault(require("fs"));
const i18n_1 = __importDefault(require("i18n"));
const os_1 = __importDefault(require("os"));
const path_1 = __importDefault(require("path"));
const app_constants_1 = require("./app.constants");
const app_logs_1 = require("./app.logs");
var AppConfigModels;
(function (AppConfigModels) {
    AppConfigModels.environmentVariables = {
        nodeEnv: "NODE_ENV",
        disableMessageQueue: "ARMON_DISABLE_MESSAGE_QUEUE",
        doNotLoadCacheAtStart: "ARMON_DO_NOT_LOAD_CACHE_AT_START",
        firebaseProjectId: "ARMON_FIREBASE_PROJECT_ID",
        firebasePriveteKey: "ARMON_FIREBASE_PRIVATE_KEY",
        firebaseClientEmail: "ARMON_FIREBASE_CLIENT_EMAIL",
        jwtSecret: "ARMON_JWT_SECRET",
        configDirectory: "ARMON_CONFIG_DIRECTORY",
        varDirectory: "ARMON_VAR_DIRECTORY",
        tmpDirectory: "ARMON_TMP_DIRECTORY",
        huaweiPushPrivateKey: "ARMON_HUAWEI_PRIVATE_KEY",
        huaweiAppId: "ARMON_HUAWEI_APP_ID",
        amqpManagementApiAuthentication: "AMQP_MANAGEMENT_AUTH",
        offlineTerminalNotificationIntervalSeconds: "OFFLINE_TERMINAL_NOTIFICATION_INTERVAL_SECONDS",
    };
})(AppConfigModels = exports.AppConfigModels || (exports.AppConfigModels = {}));
class AppConfig {
    constructor() {
        try {
            this._isWinPlatform = process.platform === "win32";
            let _nodeEnv = process.env[AppConfigModels.environmentVariables.nodeEnv];
            if (_nodeEnv && _nodeEnv !== "production" && _nodeEnv !== "development") {
                throw new Error("process.env.NODE_ENV should production or development but the value is " + _nodeEnv);
            }
            this._nodeEnv = _nodeEnv;
            let _disableMessageQueue = process.env[AppConfigModels.environmentVariables.disableMessageQueue];
            if (_disableMessageQueue !== undefined && _disableMessageQueue !== null) {
                _disableMessageQueue = _disableMessageQueue.toLowerCase();
                this._disableMessageQueue = _disableMessageQueue === "true";
            }
            else {
                this._disableMessageQueue = false;
            }
            let _doNotLoadCacheAtStart = process.env[AppConfigModels.environmentVariables.doNotLoadCacheAtStart];
            if (_doNotLoadCacheAtStart !== undefined && _doNotLoadCacheAtStart !== null) {
                _doNotLoadCacheAtStart = _doNotLoadCacheAtStart.toLowerCase();
                this._doNotLoadCacheAtStart = _doNotLoadCacheAtStart === "true";
            }
            else {
                this._doNotLoadCacheAtStart = false;
            }
            this._firebaseProjectId = process.env[AppConfigModels.environmentVariables.firebaseProjectId] || "1044630179574";
            this._firebasePriveteKey =
                process.env[AppConfigModels.environmentVariables.firebasePriveteKey] ||
                    "-----BEGIN PRIVATE KEY-----\nMIIEvQIBADANBgkqhkiG9w0BAQEFAASCBKcwggSjAgEAAoIBAQC1z6eK9Oe7qkZn\nP4eiMUs1gj7qgD10REcrk/2MRsJF2uu0hwFQO+st/H0b6me64zuNdco1bCBZdk2d\nqizUJ+fXt6WkfVFJ04YjCHqI4yTt4Y/64XLlky+2FBrmfhhQ41BKz9nVwjua5YnG\nhcNYfsfG5s+Jg+vfr8L27a1bvJWb4EJdR6ptB/fZztGNN02MnO/aFyvcZkipewlJ\nDNNi6/11gRqi/6sfAXwSvzsb/rLxqeuthsiKWrEpICJivK2Onkx9vpVr0jCopzuq\naKjvFI/7B42VUSsx5a9L0+Q2ZTWECAXEqtecpvaZrReIwbzeQHc2XB4KVEV28mHz\n2HOQp+9XAgMBAAECggEANLan26SlWaaFciMb1r1XEoUYa9JRvxea0PpADb9oTWur\nYnSwP0yYJAWsWn8soyWGGz43KiUW/glMeLWh+MM3N7bR6m7BNlgDcL3hFTZSezwm\njDV4pDJ4J2EN6Z1OvIkjuwUnohAGSyNZPnuDB3H+x18ddkAPEBR8QS7pUAtOD1rI\njyvw0cFhBOApN7p6SZ4QMgBjJMcW+nd1J1/EnGSGSmtvhGN83514rXIUC9DzTh2r\nCfNex6gvptgRK47ZlWvU9GSGh9afOITjxmP4TXozoMCMGhf4rZjSnVYUe3agDXRF\nMdPvb//l0Ys56LZMiXRtLSSzqMvMBtBvoW9WN/5rqQKBgQD1Xa81Vz7PBY8SiJhA\nkTt/TOAZuuaOAsSYrXi4+x6hpxnv+Mq0ZgI5M+uamzJ2g5skzuxlWdVThRn47CcS\nB7wjMaux7BZS57dVpsruKogk6oR21s2mzjNCno1Uc1C4UeTedbCI66v4DasqgYXb\nleOmL1vb35HIB3eZjQbIYQuUyQKBgQC9sNWoKDveULoTiIW40u5b6ty4sodPjMPn\ndW6OHqFsA7s3aY72zVYMQqDedVibDBeILgnzfNgEIoya5O5/et+tYu4rifUfwsgH\nb5iL9u+hEfOGzQJg3+vXi9jxrHi2LxhK6Jj6RBqK/H1ld7NWoDKjglgwr7OO1zE+\nRnwz+4YTHwKBgQCEsT95h08uQgFXsos0D9avR3HCOHNKxXBwaQ/qadPs9CNwxYlJ\n+gilytBVcizeLtoOE+GABg759fdPyNZwR8KU/d2ugNCp0SoypdsE5fduj21Hn27k\nEItkMb6YtywmIhO1wsvPoDQGF85DVHyBfhP44Y+53rRGCtPBNKVAZEsREQKBgD4G\n8vZ3oYwCrbXxcHiNUK/xzpb+mcxjZTPsbGhAj9w4JCHsfHkrj7WWm+/iQ0T/4ToA\ng/SAoEpJ2dxUe8Bnt3iE3wiRpfvqv2nfDezGpHjw28b6hoqS7hWffyMXmKlKIRFt\nmYhMeLl18rMZPyVWYPiiiJ/y9C4DI/EPL1H4L1FVAoGAOX2jTM4XRpiPbgSgXMfs\nFNN+XyCjZGPJrG065K4x/9f+sKIDcD3GW5Yoa8zZJioUEbwbHYALwzMFxTm77mJ5\nqK2ZN+MtfxknMc4TE1z9zPAWU013EO8Bog0BesImkYQWkLFG5cupBuz/67heFTNm\n5g1LSFMhwLr9bmnsAo5Q+Tc=\n-----END PRIVATE KEY-----\n";
            this._firebaseClientEmail = process.env[AppConfigModels.environmentVariables.firebaseClientEmail] || "firebase-adminsdk-hrat1@armongate-mobile.iam.gserviceaccount.com";
            this._jwtSecret = process.env[AppConfigModels.environmentVariables.jwtSecret] || "f70a7fb9263f412cac76d2aeb1cb463566274ecf431a4ff691b4b8df03f872b";
            this._offlineTerminalNotificationIntervalSeconds = +process.env[AppConfigModels.environmentVariables.offlineTerminalNotificationIntervalSeconds] || 15 * 60;
            this._huaweiPushPrivateKey = process.env[AppConfigModels.environmentVariables.huaweiPushPrivateKey];
            this._huaweiAppId = process.env[AppConfigModels.environmentVariables.huaweiAppId];
            this._amqpManagementApiAuth = process.env[AppConfigModels.environmentVariables.amqpManagementApiAuthentication];
            if (process.env[AppConfigModels.environmentVariables.varDirectory]) {
                this._varDirectory = process.env[AppConfigModels.environmentVariables.varDirectory];
            }
            if (process.env[AppConfigModels.environmentVariables.configDirectory]) {
                this._configDirectory = process.env[AppConfigModels.environmentVariables.configDirectory];
            }
            if (process.env[AppConfigModels.environmentVariables.tmpDirectory]) {
                this._tmpDirectory = process.env[AppConfigModels.environmentVariables.tmpDirectory];
            }
            if (this._isWinPlatform) {
                this._varDirectory = this._varDirectory || path_1.default.resolve(process.env.APPDATA, "armon", "webapp");
                this._configDirectory = this._configDirectory || this._varDirectory;
                this._preparedReportDirectory = path_1.default.resolve(this._varDirectory, "preparedReports");
                if (this._nodeEnv != "development") {
                    try {
                        require("child_process").execSync("net session", {
                            stdio: "pipe",
                        });
                    }
                    catch (err) {
                        app_logs_1.logger.error("Please run as administrator");
                        process.exit(1);
                    }
                }
            }
            else {
                this._varDirectory = this._varDirectory || path_1.default.resolve(path_1.default.sep, "var", "opt", "armon", "webapp");
                this._configDirectory = this._configDirectory || path_1.default.resolve(path_1.default.sep, "etc", "armon", "webapp");
                this._preparedReportDirectory = path_1.default.resolve(this._varDirectory, "preparedReports");
            }
            if (!this._assetsDirectory) {
                if (this._nodeEnv == "development") {
                    this._assetsDirectory = path_1.default.join(__dirname, "..", "assets");
                }
                else {
                    this._assetsDirectory = path_1.default.join(__dirname, "assets");
                }
            }
            this._httpServer = {
                hostBaseRedirectUrl: "http://localhost:4200",
                socketNspPrefix: "/socket/v1/",
                port: 3024,
                staticDirectory: path_1.default.join(this._assetsDirectory, "web", "public"),
                hostname: "api.armongate.com",
            };
            this._log = {
                app: {
                    keepLastDayCount: 60,
                    level: this._nodeEnv === "development" ? "debug" : "info",
                    zippedArchive: false,
                },
                db: {
                    keepLastDayCount: 60,
                    level: this._nodeEnv === "development" ? "debug" : "info",
                    zippedArchive: false,
                },
                httpRequests: {
                    keepLastDayCount: 30,
                    level: this._nodeEnv === "development" ? "debug" : "info",
                    zippedArchive: false,
                },
            };
            this._tmpDirectory = this._tmpDirectory || path_1.default.resolve(os_1.default.tmpdir(), "armon");
            if (!fs_1.default.existsSync(this._tmpDirectory)) {
                fs_1.default.mkdirSync(this._tmpDirectory);
            }
            if (!fs_1.default.existsSync(this._preparedReportDirectory)) {
                fs_1.default.mkdirSync(this._preparedReportDirectory, { recursive: true });
            }
            if (!fs_1.default.existsSync(this._varDirectory)) {
                fs_1.default.mkdirSync(this._varDirectory, { recursive: true });
            }
            this._organizationalFileStorageDirectory = path_1.default.join(this._varDirectory, "organizationFiles");
            if (!fs_1.default.existsSync(this._organizationalFileStorageDirectory)) {
                fs_1.default.mkdirSync(this._organizationalFileStorageDirectory, { recursive: true });
            }
            if (!fs_1.default.existsSync(this._configDirectory)) {
                fs_1.default.mkdirSync(this._configDirectory, { recursive: true });
            }
            this._serverName = process.env["ARMON_SERVER_NAME"] || process.env["SERVER_NAME"] || "cloud";
            this._extensionDirectory = path_1.default.join(__dirname, "extension", this._serverName);
            if (!fs_1.default.existsSync(this._extensionDirectory)) {
                fs_1.default.mkdirSync(this._extensionDirectory);
            }
            this._organizationBasedHookDirectory = path_1.default.join(this._extensionDirectory, "organization");
            if (!fs_1.default.existsSync(this._organizationBasedHookDirectory)) {
                fs_1.default.mkdirSync(this._organizationBasedHookDirectory);
            }
            this._serverBasedHookDirectory = path_1.default.join(this._extensionDirectory, "server");
            if (!fs_1.default.existsSync(this._serverBasedHookDirectory)) {
                fs_1.default.mkdirSync(this._serverBasedHookDirectory);
            }
            this._defaultLogDirectory = path_1.default.join(this._varDirectory, "log");
            if (!fs_1.default.existsSync(this._defaultLogDirectory)) {
                fs_1.default.mkdirSync(this._defaultLogDirectory);
            }
        }
        catch (error) {
            console.error(JSON.stringify(error));
            app_logs_1.logger.error(error);
            process.exit(1);
        }
    }
    configureTranslation() {
        i18n_1.default.configure({
            locales: ["tr", "en"],
            directory: path_1.default.join(this._assetsDirectory, "i18n"),
            logDebugFn: function (msg) {
                app_logs_1.logger.info(msg);
            },
            logWarnFn: function (msg) {
                app_logs_1.logger.info(msg);
            },
            logErrorFn: function (msg) {
                app_logs_1.logger.error(msg);
            },
            register: global,
            objectNotation: true,
            defaultLocale: "tr",
        });
    }
    init(serviceName) {
        let errors = [];
        let _configFilePath = path_1.default.join(this._configDirectory, "config.json");
        app_logs_1.logger.info("Config will be loaded from " + _configFilePath);
        if (fs_1.default.existsSync(_configFilePath)) {
            errors = this.parseConfigFile(_configFilePath, serviceName);
        }
        else {
            app_logs_1.logger.info("Config file is not found at " + _configFilePath + ". Default settings will be used");
        }
        if (errors.length > 0) {
            throw errors;
        }
        this.configureTranslation();
        (0, app_logs_1.initHttpLogger)(this.httpLogDirectory, this.log.httpRequests.level, this.log.httpRequests.keepLastDayCount, this.log.httpRequests.zippedArchive);
    }
    parseConfigFile(configPath, serviceName) {
        let _config = require(configPath);
        let errors = [];
        if (typeof _config !== "object") {
            errors.push();
        }
        this._maxCpuCore = _config.maxCpuCore;
        if (!this._maxCpuCore) {
            app_logs_1.logger.info("No max cpu core settings from configuration, default to 2");
            this._maxCpuCore = 2;
        }
        this._reportSendByMailFileSizeLimit = _config.reportSendByMailFileSizeLimit;
        if (!this._reportSendByMailFileSizeLimit) {
            app_logs_1.logger.info("No report file size limit from configuration, default to 1.5 mb");
            this._reportSendByMailFileSizeLimit = 1.5;
        }
        this._reportExportRowLimit = _config.reportExportRowLimit;
        if (!this._reportExportRowLimit) {
            app_logs_1.logger.info("No excel report limit from configuration, default to 10000");
            this._reportExportRowLimit = 10000;
        }
        this._reportMinRowCountToSendByNotificationForExcel = _config.reportMinRowCountToSendByNotificationForExcel;
        if (!this._reportMinRowCountToSendByNotificationForExcel) {
            app_logs_1.logger.info("No min row count to send by notification for excel from configuration, default to 5000");
            this._reportMinRowCountToSendByNotificationForExcel = 5000;
        }
        this._reportMinRowCountToSendByNotificationForPdf = _config.reportMinRowCountToSendByNotificationForPdf;
        if (!this._reportMinRowCountToSendByNotificationForPdf) {
            app_logs_1.logger.info("No min row count to send by notification for excel from configuration, default to 5000");
            this._reportMinRowCountToSendByNotificationForPdf = 5000;
        }
        this._reportPdfExcelTimeoutInSeconds = _config.reportPdfExcelTimeoutInSeconds;
        if (!this._reportPdfExcelTimeoutInSeconds) {
            app_logs_1.logger.info("No excel/pdf report timeout limit from configuration, default to 15 seconds");
            this._reportPdfExcelTimeoutInSeconds = 15;
        }
        this._allowedMaxConnectionPerUserSocket = _config.allowedMaxConnectionPerUserSocket;
        if (!this._allowedMaxConnectionPerUserSocket) {
            app_logs_1.logger.info("No allowedMaxConnectionPerUserSocket configuration, default 10");
            this._allowedMaxConnectionPerUserSocket = 10;
        }
        this._sendConnectionEventNotificationForDeviceAfterInSec = _config.sendConnectionEventNotificationForDeviceAfterInSec;
        if (!this._sendConnectionEventNotificationForDeviceAfterInSec) {
            app_logs_1.logger.info("No send connection event notification for devices timeout limit from configuration, default to 10 min");
            this._sendConnectionEventNotificationForDeviceAfterInSec = 60 * 10;
        }
        if (_config.assetsDirectory) {
            this._assetsDirectory = _config.assetsDirectory;
        }
        else {
            this._assetsDirectory = path_1.default.join(__dirname, "assets");
        }
        this._localConfigApiServerPort = _config.localConfigApiServerPort;
        if (!this.localConfigApiServerPort) {
            this._localConfigApiServerPort = 3025;
        }
        if (_config.extensionDirectory) {
            this._extensionDirectory = _config.extensionDirectory;
        }
        if (_config.terminalServer) {
            this._terminalServer = _config.terminalServer;
            if (this._terminalServer.ssl && this._terminalServer.ssl.customDirectory) {
                this._terminalServer.ssl.caFileName = path_1.default.join(this._terminalServer.ssl.customDirectory, this._terminalServer.ssl.caFileName);
                this._terminalServer.ssl.certFileName = path_1.default.join(this._terminalServer.ssl.customDirectory, this._terminalServer.ssl.certFileName);
                this._terminalServer.ssl.keyFileName = path_1.default.join(this._terminalServer.ssl.customDirectory, this._terminalServer.ssl.keyFileName);
            }
        }
        if (_config.httpServer) {
            this._httpServer = _config.httpServer;
            if (this._httpServer.ssl && this._httpServer.ssl.customDirectory) {
                this._httpServer.ssl.certFileName = path_1.default.join(this._httpServer.ssl.customDirectory, this._httpServer.ssl.certFileName);
                this._httpServer.ssl.keyFileName = path_1.default.join(this._httpServer.ssl.customDirectory, this._httpServer.ssl.keyFileName);
            }
            if (!this._httpServer.staticDirectory) {
                app_logs_1.logger.warn(` You didn't set "staticDirectory". Unsubscribe urls will not work. To set:
				nano /etc/armon/webapp/config.json
				-- under httpServer
				"staticDirectory": "/opt/armon/webapp/assets/public"`);
            }
            else {
                app_logs_1.logger.debug("staticDirectory: " + this._httpServer.staticDirectory);
            }
            if (!this._httpServer.apiBasePath) {
                app_logs_1.logger.warn(` You didn't set "apiBasePath". Unsubscribe urls will not work. To set:
				nano /etc/armon/webapp/config.json
				-- under httpServer
				"apiBasePath": "API_URL"
				--note: "API_URL" can be found in user_agent configuration.json file or check to requested api from web client`);
            }
            else {
                app_logs_1.logger.debug("apiBasePath: " + this._httpServer.apiBasePath);
            }
        }
        if (!_config?.httpServer?.hostBaseRedirectUrl && process.env.DOMAIN_NAME) {
            this.httpServer.hostBaseRedirectUrl = process.env.HTTP_PROTOCOL + "://" + process.env.DOMAIN_NAME + "/app/#";
        }
        if (!_config?.httpServer?.apiBasePath && process.env.DOMAIN_NAME) {
            this.httpServer.apiBasePath = process.env.HTTP_PROTOCOL + "://" + process.env.DOMAIN_NAME + ":" + this.httpServer.port;
        }
        if (_config.amqpClient) {
            this._amqpClient = _config.amqpClient;
            this._amqpClient.managementApiPort = _config.amqpClient.managementApiPort || 15672;
            if (this._amqpClient.ssl && this._amqpClient.ssl.customDirectory) {
                this._amqpClient.ssl.caFileName = path_1.default.join(this._amqpClient.ssl.customDirectory, this._amqpClient.ssl.caFileName);
                this._amqpClient.ssl.certFileName = path_1.default.join(this._amqpClient.ssl.customDirectory, this._amqpClient.ssl.certFileName);
                this._amqpClient.ssl.keyFileName = path_1.default.join(this._amqpClient.ssl.customDirectory, this._amqpClient.ssl.keyFileName);
            }
        }
        if (_config.log) {
            this._log = _config.log;
        }
        if (_config.db) {
            this._db = _config.db;
        }
        if (_config.ntp) {
            this._ntp = _config.ntp;
        }
        if (_config.deviceCertFolderPath) {
            this._deviceCertFolderPath = _config.deviceCertFolderPath;
        }
        this._deviceUpdateFolderPath = _config.deviceUpdateFolderPath;
        if (this._deviceUpdateFolderPath && !fs_1.default.existsSync(this._deviceUpdateFolderPath)) {
            errors.push("Given device update folder path doesnt exist: " + _config.deviceUpdateFolderPath);
        }
        if (_config.smtp) {
            this._smtp = _config.smtp;
        }
        if (_config.base64Logo) {
            this._base64Logo = _config.base64Logo;
        }
        this._huaweiPushSettings = _config.huaweiPushSettings;
        if (!_config.huaweiPushSettings) {
            this._huaweiPushSettings = {
                appId: this._huaweiAppId,
                appSecret: this._huaweiPushPrivateKey,
                authUrl: "https://oauth-login.cloud.huawei.com/oauth2/v2/token",
                pushUrl: "https://push-api.cloud.huawei.com/v1",
            };
        }
        if (!_config.rateLimits) {
            this._rateLimits = {
                auth: {
                    byIP: {
                        maxAttempts: 100,
                        windowDurationInMinutes: 60 * 3,
                        blockDurationInMinutes: 60 * 12,
                    },
                    byUsername: {
                        maxAttempts: 100,
                        windowDurationInMinutes: 60 * 3,
                        blockDurationInMinutes: 60 * 12,
                    },
                    byUsernameAndIP: {
                        maxAttempts: 30,
                        windowDurationInMinutes: 60 * 24,
                        blockDurationInMinutes: 60 * 24,
                    },
                },
                generic: {
                    maxAttempts: 500,
                    windowDurationInMinutes: 1,
                    blockDurationInMinutes: 5,
                },
                customUserParams: {},
            };
        }
        else {
            this._rateLimits = _config.rateLimits;
        }
        if ([app_constants_1.ServiceNames.WebApplication, app_constants_1.ServiceNames.MonitorService].includes(serviceName) && !this._amqpManagementApiAuth) {
            errors.push(`[${serviceName}] You should declare AMQP_MANAGEMENT_AUTH as ENVIRONMENT VARIABLE`);
        }
        return errors;
    }
    get isWinPlatform() {
        return this._isWinPlatform;
    }
    get isCloudServer() {
        return this._serverName === "cloud";
    }
    get disableMessageQueue() {
        return this._disableMessageQueue;
    }
    get doNotLoadCacheAtStart() {
        return this._doNotLoadCacheAtStart;
    }
    get firebaseProjectId() {
        return this._firebaseProjectId;
    }
    get firebasePrivateKey() {
        return this._firebasePriveteKey;
    }
    get firebaseClientEmail() {
        return this._firebaseClientEmail;
    }
    get offlineTerminalNotificationIntervalSeconds() {
        return this._offlineTerminalNotificationIntervalSeconds;
    }
    get jwtSecret() {
        return this._jwtSecret;
    }
    get reportExportRowLimit() {
        return this._reportExportRowLimit;
    }
    get reportSendByMailFileSizeLimit() {
        return this._reportSendByMailFileSizeLimit;
    }
    get reportMinRowCountToSendByNotificationForExcel() {
        return this._reportMinRowCountToSendByNotificationForExcel;
    }
    get reportMinRowCountToSendByNotificationForPdf() {
        return this._reportMinRowCountToSendByNotificationForPdf;
    }
    get allowedMaxConnectionPerUserSocket() {
        return this._allowedMaxConnectionPerUserSocket;
    }
    get sendConnectionEventNotificationForDeviceAfterInSec() {
        return this._sendConnectionEventNotificationForDeviceAfterInSec;
    }
    get reportPdfExcelTimeoutInSeconds() {
        return this._reportPdfExcelTimeoutInSeconds;
    }
    get extensionDirectory() {
        return this._extensionDirectory;
    }
    get terminalServer() {
        return this._terminalServer;
    }
    get httpServer() {
        return this._httpServer;
    }
    get localConfigApiServerPort() {
        return this._localConfigApiServerPort;
    }
    get amqpClient() {
        return this._amqpClient;
    }
    get log() {
        return this._log;
    }
    get db() {
        return this._db;
    }
    get nodeEnv() {
        return this._nodeEnv;
    }
    get maxCpuCore() {
        return this._maxCpuCore;
    }
    get ntp() {
        return this._ntp;
    }
    get varDirectory() {
        return this._varDirectory;
    }
    get tmpDirectory() {
        return this._tmpDirectory;
    }
    get preparedReportDirectory() {
        return this._preparedReportDirectory;
    }
    get serverName() {
        return this._serverName;
    }
    get organizationBasedHookDirectory() {
        return this._organizationBasedHookDirectory;
    }
    get serverBasedHookDirectory() {
        return this._serverBasedHookDirectory;
    }
    get dbLogDirectory() {
        return this._log.db.customDirectory || this._defaultLogDirectory;
    }
    get httpLogDirectory() {
        return this.log.httpRequests.customDirectory || this._defaultLogDirectory;
    }
    get appLogDirectory() {
        return this._log.app.customDirectory || this._defaultLogDirectory;
    }
    get deviceCertFolderPath() {
        return this._deviceCertFolderPath || this._varDirectory;
    }
    get hostBaseRedirectUrl() {
        return this._httpServer.hostBaseRedirectUrl;
    }
    get organizationalFileStorageDirectory() {
        return this._organizationalFileStorageDirectory;
    }
    get deviceUpdateFolderPath() {
        return this._deviceUpdateFolderPath;
    }
    get assetsDirectory() {
        return this._assetsDirectory;
    }
    get smtp() {
        return this._smtp;
    }
    get base64Logo() {
        return this._base64Logo;
    }
    get huaweiPushSettings() {
        return this._huaweiPushSettings;
    }
    get amqpManagementApiAuth() {
        return this._amqpManagementApiAuth;
    }
    get configDirectory() {
        return this._configDirectory;
    }
    get rateLimits() {
        return this._rateLimits;
    }
}
exports.appConfig = new AppConfig();
